"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 *
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TelemetryTest = void 0;
const path = __importStar(require("path"));
const telemetry_1 = require("../telemetry");
const basePropUtils = __importStar(require("../utils/basePropUtils"));
const errorUtils = __importStar(require("../utils/errorUtils"));
const projectUtils = __importStar(require("../utils/projectUtils"));
const versionUtils = __importStar(require("../utils/versionUtils"));
class TelemetryTest extends telemetry_1.Telemetry {
    /** Run at the beginning of each test. */
    static async startTest(options) {
        TelemetryTest.hasTestTelemetryProviders = false;
        TelemetryTest.testTelemetryProvidersRan = false;
        if (TelemetryTest.isEnabled()) {
            telemetry_1.Telemetry.reset();
        }
        // Ensure that we don't actually fire events when testing
        telemetry_1.Telemetry.isTest = true;
        await telemetry_1.Telemetry.setup(options);
    }
    /** Run at the end of each test where telemetry was fired. */
    static endTest(finalCallback) {
        var _a;
        (_a = telemetry_1.Telemetry.client) === null || _a === void 0 ? void 0 : _a.flush({
            callback: _ => {
                if (TelemetryTest.hasTestTelemetryProviders) {
                    expect(TelemetryTest.testTelemetryProvidersRan).toBe(true);
                }
                if (finalCallback) {
                    finalCallback();
                }
            },
        });
    }
    /** Sets that the telemetry provider has run. */
    static setTestTelemetryProvidersRan() {
        TelemetryTest.testTelemetryProvidersRan = true;
    }
    /** Retrieves the value of a common property.*/
    static getCommonProperty(key) {
        var _a;
        return (_a = TelemetryTest.client) === null || _a === void 0 ? void 0 : _a.commonProperties[key];
    }
    /** Retrieves the version of the specified tool/package. */
    static getVersion(key) {
        return key in TelemetryTest.versionsProp
            ? TelemetryTest.versionsProp[key]
            : null;
    }
    /** Retrieves the value of the preserveErrorMessages option. */
    static getPreserveErrorMessages() {
        return TelemetryTest.options.preserveErrorMessages;
    }
    /** Adds a telemetry processor, usually for verifying the envelope. */
    static addTelemetryProcessor(telemetryProcessor) {
        var _a;
        (_a = TelemetryTest.client) === null || _a === void 0 ? void 0 : _a.addTelemetryProcessor(telemetryProcessor);
        TelemetryTest.hasTestTelemetryProviders = true;
    }
}
exports.TelemetryTest = TelemetryTest;
test('setup() verify session id is valid and a common property', async () => {
    await TelemetryTest.startTest();
    const sessionId = TelemetryTest.getSessionId();
    expect(sessionId).toBeDefined();
    expect(sessionId).toHaveLength(32);
    expect(sessionId).toBe(basePropUtils.getSessionId());
    expect(TelemetryTest.getCommonProperty('sessionId')).toBe(sessionId);
    TelemetryTest.endTest();
});
test('setup() verify static common property values with async sources', async () => {
    await TelemetryTest.startTest();
    const props = {
        deviceId: basePropUtils.deviceId,
        deviceLocale: basePropUtils.deviceLocale,
    };
    for (const key in props) {
        if (!(key in Object.prototype)) {
            const value = TelemetryTest.getCommonProperty(key);
            expect(value).toBeDefined();
            expect(value).toBe(await props[key]());
        }
    }
    TelemetryTest.endTest();
});
test('setup() verify static common property values with sync sources', async () => {
    await TelemetryTest.startTest();
    const props = {
        deviceArchitecture: () => basePropUtils.deviceArchitecture(),
        devicePlatform: () => basePropUtils.devicePlatform(),
        deviceNumCPUs: () => basePropUtils.deviceNumCPUs().toString(),
        deviceTotalMemory: () => basePropUtils.deviceTotalMemory().toString(),
        ciCaptured: () => basePropUtils.captureCI().toString(),
        ciType: () => basePropUtils.ciType(),
        isMsftInternal: () => basePropUtils.isMsftInternal().toString(),
        sampleRate: () => basePropUtils.sampleRate().toString(),
        isTest: () => 'true',
    };
    for (const key in props) {
        if (!(key in Object.prototype)) {
            const value = TelemetryTest.getCommonProperty(key);
            expect(value).toBeDefined();
            expect(value).toBe(props[key]());
        }
    }
    TelemetryTest.endTest();
});
test('setup() verify other common property values are defined', async () => {
    await TelemetryTest.startTest();
    const props = ['deviceDiskFreeSpace'];
    for (const key of props) {
        const value = TelemetryTest.getCommonProperty(key);
        expect(value).toBeDefined();
    }
    TelemetryTest.endTest();
});
test('setup() verify tool versions are populated', async () => {
    await TelemetryTest.startTest();
    const props = {
        node: versionUtils.getNodeVersion,
        npm: versionUtils.getNpmVersion,
        yarn: versionUtils.getYarnVersion,
        VisualStudio: versionUtils.getVisualStudioVersion,
    };
    for (const key in props) {
        if (!(key in Object.prototype)) {
            const value = await props[key]();
            expect(value).toBe(TelemetryTest.getVersion(key));
        }
    }
    TelemetryTest.endTest();
});
test('tryUpdateVersionsProp() returns true for adding a new version', async () => {
    await TelemetryTest.startTest();
    const name = 'test';
    const version = '1.0';
    expect(await TelemetryTest.tryUpdateVersionsProp(name, async () => version)).toBe(true);
    expect(TelemetryTest.getVersion(name)).toBe(version);
    TelemetryTest.endTest();
});
test('tryUpdateVersionsProp() returns false for adding an existing version with refresh is false', async () => {
    await TelemetryTest.startTest();
    const name = 'test';
    const version = '1.0';
    expect(await TelemetryTest.tryUpdateVersionsProp(name, async () => version)).toBe(true);
    let getValueCalled = false;
    expect(await TelemetryTest.tryUpdateVersionsProp(name, async () => {
        getValueCalled = true;
        return version;
    })).toBe(false);
    expect(getValueCalled).toBe(false);
    TelemetryTest.endTest();
});
test('tryUpdateVersionsProp() returns true for adding an existing version with refresh is true', async () => {
    await TelemetryTest.startTest();
    const name = 'test';
    const version = '1.0';
    expect(await TelemetryTest.tryUpdateVersionsProp(name, async () => version)).toBe(true);
    let getValueCalled = false;
    expect(await TelemetryTest.tryUpdateVersionsProp(name, async () => {
        getValueCalled = true;
        return version;
    }, true)).toBe(true);
    expect(getValueCalled).toBe(true);
    TelemetryTest.endTest();
});
/** Returns the CommandStartInfo for our fake 'test-command'. */
function getTestCommandStartInfo() {
    return {
        commandName: 'test-command',
        args: {
            testArg1: 'true',
            testArg2: '10',
            testArg3: 'testValue',
        },
        options: {
            testArg0: 'unsetArg',
            testArg1: true,
            testArg2: 10,
            testArg3: 'testValue',
        },
        defaultOptions: {
            testArg0: 'unsetArg',
            testArg1: false,
            testArg2: 0,
            testArg3: 'defaultValue',
        },
    };
}
/** Returns the CommandEndInfo for our fake 'test-command'. */
function getTestCommandEndInfo(resultCode) {
    return {
        resultCode,
    };
}
function getTestCommandProjectInfo() {
    return {
        id: projectUtils.getProjectId('test-app-project'),
        platforms: ['windows'],
        rnwLang: 'cpp',
        usesTS: true,
        usesRNConfig: false,
        jsEngine: 'Chakra',
        rnwSource: 'Source',
        dependencies: [
            {
                id: projectUtils.getProjectId('test-module-project'),
                platforms: ['android', 'windows'],
                rnwLang: 'cpp',
            },
        ],
    };
}
function getExtraProps() {
    return {
        extraProp1: true,
        extraProp2: 1234,
        extraProp3: 'test',
        extraProp4: ['test'],
        extraProp5: {
            nestedProp1: true,
            nestedProp2: 1234,
        },
    };
}
/** Asynchronously waits the number in ms. */
async function promiseDelay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}
/** The body of the fake 'test-command' which will throw the provided error. */
async function testCommandBody(errorToThrow) {
    await promiseDelay(100);
    if (errorToThrow) {
        throw errorToThrow;
    }
}
/** Runs the complete 'test-command' with the right Telemetry setup and cleanup. */
async function runTestCommandE2E(commandBody) {
    TelemetryTest.startCommand(getTestCommandStartInfo());
    TelemetryTest.setProjectInfo(getTestCommandProjectInfo());
    let errorCode = 'Success';
    let caughtError;
    try {
        await commandBody();
    }
    catch (ex) {
        caughtError = ex instanceof Error ? ex : new Error(String(ex));
        errorCode =
            caughtError instanceof errorUtils.CodedError
                ? caughtError.type
                : 'Unknown';
        TelemetryTest.trackException(caughtError);
    }
    TelemetryTest.endCommand(getTestCommandEndInfo(errorCode), getExtraProps());
}
/** Verifies the contents of events fired during the 'test-command'. */
function verifyTestCommandTelemetryProcessor(caughtErrors, expectedResultCode, expectedError) {
    return (envelope, _) => {
        var _a, _b, _c, _d;
        try {
            // Processor has run, so the test can (potentially) pass
            TelemetryTest.setTestTelemetryProvidersRan();
            // Verify roleInstance has been removed
            expect(envelope.tags['ai.cloud.roleInstance']).toBeUndefined();
            const properties = (_a = envelope.data.baseData) === null || _a === void 0 ? void 0 : _a.properties;
            expect(properties).toBeDefined();
            // Verify basics
            expect(properties.commandName).toBe('test-command');
            // Verify versions info
            const versions = JSON.parse(properties.versions);
            expect(versions).toBeDefined();
            // Verify project info
            const project = JSON.parse(properties.project);
            expect(project).toStrictEqual(getTestCommandProjectInfo());
            expect(Object.keys(versions).length).toBeGreaterThan(0);
            for (const key of Object.keys(versions)) {
                expect(versions[key]).toBe(TelemetryTest.getVersion(key));
            }
            if (envelope.data.baseType === 'ExceptionData') {
                // Verify event name
                expect(properties.eventName).toBe(telemetry_1.CodedErrorEventName);
                // Verify exception info
                const exceptions = (_b = envelope.data.baseData) === null || _b === void 0 ? void 0 : _b.exceptions;
                expect(exceptions).toBeDefined();
                expect(exceptions.length).toBe(1);
                expect(exceptions[0].message).toBeDefined();
                expect(exceptions[0].message).not.toBe('');
                expect(exceptions[0].message).toBe(TelemetryTest.getPreserveErrorMessages()
                    ? errorUtils.sanitizeErrorMessage((expectedError === null || expectedError === void 0 ? void 0 : expectedError.message) || 'None')
                    : '[Removed]');
                // Verify coded error info
                const codedError = JSON.parse(properties.codedError);
                expect(codedError).toBeDefined();
                expect(codedError.type).toBe(expectedError instanceof errorUtils.CodedError
                    ? expectedError.type
                    : 'Unknown');
                expect(codedError.data).toStrictEqual((_c = expectedError.data) !== null && _c !== void 0 ? _c : {});
            }
            else {
                // Verify event name
                expect((_d = envelope.data.baseData) === null || _d === void 0 ? void 0 : _d.name).toBe(telemetry_1.CommandEventName);
                expect(properties.eventName).toBe(telemetry_1.CommandEventName);
                // Verify command info
                const expectedInfo = getTestCommandStartInfo();
                const command = JSON.parse(properties.command);
                expect(command).toBeDefined();
                expect(command.args).toStrictEqual(expectedInfo.args);
                expect(command.options).toStrictEqual(expectedInfo.options);
                expect(command.defaultOptions).toStrictEqual(expectedInfo.defaultOptions);
                expect(command.durationInSecs).toBeGreaterThan(0);
                expect(command.resultCode).toBe(expectedResultCode !== null && expectedResultCode !== void 0 ? expectedResultCode : 'Success');
                // Verify extra props
                const extraProps = getExtraProps();
                expect(JSON.parse(properties.extraProps)).toStrictEqual(extraProps);
            }
        }
        catch (ex) {
            caughtErrors.push(ex instanceof Error ? ex : new Error(String(ex)));
        }
        return true;
    };
}
test('Telemetry run test command end to end, verify event fires', async () => {
    await TelemetryTest.startTest();
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(verifyTestCommandTelemetryProcessor(caughtErrors));
    await runTestCommandE2E(testCommandBody);
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
const testTelemetryOptions = [
    { preserveErrorMessages: false },
    { preserveErrorMessages: true },
];
test.each(testTelemetryOptions)('Telemetry run test command end to end with CodedError, verify events fire %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new errorUtils.CodedError('MSBuildError', 'test error');
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(verifyTestCommandTelemetryProcessor(caughtErrors, expectedError.type, expectedError));
    await runTestCommandE2E(() => testCommandBody(expectedError));
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
test.each(testTelemetryOptions)('Telemetry run test command end to end with CodedError (with error in message), verify events fire %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new errorUtils.CodedError('MSBuildError', 'error FOO2020: test error');
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(verifyTestCommandTelemetryProcessor(caughtErrors, expectedError.type, expectedError));
    await runTestCommandE2E(() => testCommandBody(expectedError));
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
test.each(testTelemetryOptions)('Telemetry run test command end to end with CodedError (with data), verify events fire %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new errorUtils.CodedError('MSBuildError', 'test error', { foo: 42 });
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(verifyTestCommandTelemetryProcessor(caughtErrors, expectedError.type, expectedError));
    await runTestCommandE2E(() => testCommandBody(expectedError));
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
test.each(testTelemetryOptions)('Telemetry run test command end to end with Error, verify events fire %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new Error('error FOO2020: test error');
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(verifyTestCommandTelemetryProcessor(caughtErrors, 'Unknown', expectedError));
    await runTestCommandE2E(() => testCommandBody(expectedError));
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
test.each(testTelemetryOptions)('Telemetry run test command end to end with Error (no message), verify events fire %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new Error();
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(verifyTestCommandTelemetryProcessor(caughtErrors, 'Unknown', expectedError));
    await runTestCommandE2E(() => testCommandBody(expectedError));
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
function b(s) {
    throw new Error('hello ' + s);
}
function a(s) {
    b(s);
}
/** Verifies the contents of an exception's message and stack frames */
function getVerifyStackTelemetryProcessor(caughtErrors, expectedError) {
    return (envelope, _) => {
        try {
            // Processor has run, so the test can (potentially) pass
            TelemetryTest.setTestTelemetryProvidersRan();
            if (envelope.data.baseType === 'ExceptionData') {
                const data = envelope.data.baseData;
                expect(data.exceptions).toBeDefined();
                expect(data.exceptions.length).toBe(1);
                expect(data.exceptions[0].message).toBeDefined();
                expect(data.exceptions[0].message).not.toBe('');
                expect(data.exceptions[0].message).toBe(TelemetryTest.getPreserveErrorMessages()
                    ? errorUtils.sanitizeErrorMessage(expectedError.message || 'None')
                    : '[Removed]');
                const stack = data.exceptions[0].parsedStack;
                expect(stack).toBeDefined();
                expect(stack.length).toBeGreaterThan(2);
                const filename = path.relative(process.cwd(), __filename);
                expect(stack[0].method).toEqual('b');
                expect(stack[1].method).toEqual('b');
                expect(stack[2].method).toEqual('a');
                expect(stack[0].fileName).toEqual(`[project_dir]\\???.ts(${filename.length})`);
                expect(stack[1].fileName).toEqual(`[project_dir]\\???.ts(${filename.length})`);
                expect(stack[2].fileName).toEqual(`[project_dir]\\???.ts(${filename.length})`);
            }
        }
        catch (ex) {
            caughtErrors.push(ex instanceof Error ? ex : new Error(String(ex)));
        }
        return true;
    };
}
test.each(testTelemetryOptions)('Telemetry run test command end to end with Error, verify sanitized message and stack %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new Error('hello world');
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(getVerifyStackTelemetryProcessor(caughtErrors, expectedError));
    await runTestCommandE2E(async () => {
        await promiseDelay(100);
        a('world');
    });
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
test.each(testTelemetryOptions)('Telemetry run test command end to end with Error, verify sanitized message with path and stack %s', async (options) => {
    await TelemetryTest.startTest(options);
    const expectedError = new Error(`hello ${process.cwd()}`);
    // AI eats errors thrown in telemetry processors
    const caughtErrors = [];
    TelemetryTest.addTelemetryProcessor(getVerifyStackTelemetryProcessor(caughtErrors, expectedError));
    await runTestCommandE2E(async () => {
        await promiseDelay(100);
        a(process.cwd());
    });
    TelemetryTest.endTest(() => {
        // Check if any errors were thrown
        expect(caughtErrors).toHaveLength(0);
    });
});
//# sourceMappingURL=telemetry.test.js.map