"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 *
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.findRepoPackageSync = exports.findRepoPackage = exports.findPackage = exports.enumerateRepoPackagesSync = exports.enumerateRepoPackages = exports.WritableNpmPackage = exports.NpmPackage = void 0;
const lodash_1 = __importDefault(require("lodash"));
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const find_repo_root_1 = __importDefault(require("@react-native-windows/find-repo-root"));
const getMonorepoPackages = require('get-monorepo-packages');
/**
 * Represents an NPM package
 */
class NpmPackage {
    /**
     * Create a NpmPackage from a given path to a package
     */
    static async fromPath(pkgPath) {
        return WritableNpmPackage.fromPath(pkgPath);
    }
    constructor(pkgPath, pkgJson) {
        this.pkgPath = pkgPath;
        this.pkgJson = pkgJson;
    }
    get path() {
        return this.pkgPath;
    }
    get json() {
        return lodash_1.default.cloneDeep(this.pkgJson);
    }
}
exports.NpmPackage = NpmPackage;
/**
 * Represents an NPM package with methods to modify its on-disk package.json
 */
class WritableNpmPackage extends NpmPackage {
    /**
     * Create a WritableNpmPackage from a given path to a package
     */
    static async fromPath(pkgPath) {
        const jsonPath = path_1.default.join(pkgPath, 'package.json');
        try {
            return new WritableNpmPackage(pkgPath, await fs_1.default.readJsonFile(jsonPath));
        }
        catch (ex) {
            if (ex.code === 'ENOENT') {
                return null;
            }
            throw ex;
        }
    }
    /**
     * Assign properties to the package. Uses lodash merge semantics to assign
     * properties (i.e. deep merge instead of shallow)
     *
     * @param props key/values to merge into the package.json
     */
    async mergeProps(props) {
        await this.setJson(lodash_1.default.merge(this.json, props));
    }
    /**
     * Assign properties to the package. Uses Object.assign semantics to assign
     * properties (i.e. shallow merge)
     *
     * @param props key/values to merge into the package.json
     */
    async assignProps(props) {
        await this.setJson(Object.assign(this.json, props));
    }
    /**
     * Set new JSON for the package from the given object
     */
    async setJson(jsonObj) {
        this.pkgJson = jsonObj;
        await fs_1.default.writeFile(path_1.default.join(this.path, 'package.json'), JSON.stringify(this.json, null /*replacer*/, 2 /*space*/) + '\n');
    }
}
exports.WritableNpmPackage = WritableNpmPackage;
/**
 * Finds monorepo-local packages matching a given predicate. The root package
 * is not included.
 *
 * @param pred predicate describing whether to match a package
 */
async function enumerateRepoPackages(pred = async () => true) {
    const repoRoot = await (0, find_repo_root_1.default)();
    const allPackages = getMonorepoPackages(repoRoot).map(pkg => new WritableNpmPackage(pkg.location, pkg.package));
    const filteredPackages = [];
    for (const pkg of allPackages) {
        if (await pred(pkg)) {
            filteredPackages.push(pkg);
        }
    }
    return filteredPackages;
}
exports.enumerateRepoPackages = enumerateRepoPackages;
/**
 * Synchronously Finds monorepo-local packages matching a given predicate. The
 * root package is not included.
 *
 * @param pred predicate describing whether to match a package
 */
function enumerateRepoPackagesSync(pred = () => true) {
    const repoRoot = find_repo_root_1.default.sync();
    const allPackages = getMonorepoPackages(repoRoot).map(pkg => new WritableNpmPackage(pkg.location, pkg.package));
    return allPackages.filter(pred);
}
exports.enumerateRepoPackagesSync = enumerateRepoPackagesSync;
/**
 * Finds a package with a given name (local or dependency)
 */
async function findPackage(name, opts = {}) {
    const resolvePaths = require.resolve.paths(`${name}/package.json`);
    let pkgJsonPath;
    try {
        pkgJsonPath = require.resolve(`${name}/package.json`, {
            paths: [opts.searchPath || process.cwd(), ...resolvePaths],
        });
    }
    catch (ex) {
        if (ex.code === 'MODULE_NOT_FOUND') {
            return null;
        }
        else {
            throw ex;
        }
    }
    return new NpmPackage(path_1.default.dirname(pkgJsonPath), await fs_1.default.readJsonFile(pkgJsonPath));
}
exports.findPackage = findPackage;
/**
 * Finds a monorepo-local package with a given name
 */
async function findRepoPackage(name) {
    const packages = await enumerateRepoPackages(async (p) => p.json.name === name);
    if (packages.length === 0) {
        return null;
    }
    else {
        return packages[0];
    }
}
exports.findRepoPackage = findRepoPackage;
/**
 * Synchronously a monorepo-local package with a given name
 */
function findRepoPackageSync(name) {
    const packages = enumerateRepoPackagesSync(p => p.json.name === name);
    if (packages.length === 0) {
        return null;
    }
    else {
        return packages[0];
    }
}
exports.findRepoPackageSync = findRepoPackageSync;
//# sourceMappingURL=packageUtils.js.map