/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateAliases = exports.createAliasMap = void 0;
const AliasManaging_1 = require("./AliasManaging");
const ObjectTypes_1 = require("./ObjectTypes");
function translateObjectMembersDefinition(type, aliases, baseAliasName, prefix, options) {
    return type.properties
        .map((prop) => {
        let propType = prop.typeAnnotation;
        if (prop.optional && propType.type !== 'NullableTypeAnnotation') {
            propType = { type: 'NullableTypeAnnotation', typeAnnotation: propType };
        }
        return `${prefix}${(0, ObjectTypes_1.translateField)(propType, aliases, `${baseAliasName}_${prop.name}`, options)} ${prop.name};`;
    })
        .join('\n');
}
function translateObjectMembersReflection(type, aliasCppName, prefix) {
    return type.properties
        .map((prop) => {
        return `${prefix}{L"${prop.name}", &${aliasCppName}::${prop.name}},`;
    })
        .join('\n');
}
function createAliasMap(nativeModuleAliases) {
    const aliases = { types: {}, jobs: Object.keys(nativeModuleAliases) };
    for (const aliasName of aliases.jobs) {
        aliases.types[aliasName] = nativeModuleAliases[aliasName];
    }
    return aliases;
}
exports.createAliasMap = createAliasMap;
function generateSingleAlias(aliases, aliasName, aliasCode, options) {
    const aliasCppName = (0, AliasManaging_1.getAliasCppName)(aliasName);
    const aliasType = aliases.types[aliasName];
    const definition = `
struct ${aliasCppName} {
${translateObjectMembersDefinition(aliasType, aliases, aliasName, '    ', options)}
};
`;
    const reflection = `
inline winrt::Microsoft::ReactNative::FieldMap GetStructInfo(${aliasCppName}*) noexcept {
    winrt::Microsoft::ReactNative::FieldMap fieldMap {
${translateObjectMembersReflection(aliasType, aliasCppName, '        ')}
    };
    return fieldMap;
}
`;
    aliasCode[aliasName] = { definition, reflection };
}
function generateNestedAliasesInCorrectOrder(aliases, aliasCode, aliasOrder, options) {
    // retrieve and clean all ungenerated aliases
    const jobs = aliases.jobs;
    aliases.jobs = [];
    // generate each one in its found order
    for (const aliasName of jobs) {
        // generate a new struct and all fields will be examined
        // new anonymous objects could be found
        // they will be stored in aliases.jobs
        generateSingleAlias(aliases, aliasName, aliasCode, options);
        // nested C++ structs must be put before the current C++ struct
        // as they will be used in the current C++ struct
        // the order will be perfectly and easily ensured by doing this recursively
        generateNestedAliasesInCorrectOrder(aliases, aliasCode, aliasOrder, options);
        // all referenced C++ structs are generated
        // put the current one following them
        aliasOrder.push(aliasName);
    }
}
function generateAliases(aliases, options) {
    const aliasCode = {};
    const aliasOrder = [];
    generateNestedAliasesInCorrectOrder(aliases, aliasCode, aliasOrder, options);
    // aliasOrder now has the correct order of C++ struct code
    let customTypes = '';
    let customReflection = '';
    for (const aliasName of aliasOrder) {
        customTypes = `${customTypes}${aliasCode[aliasName].definition}`;
        customReflection = `${customReflection}${aliasCode[aliasName].reflection}`;
    }
    return [customTypes, customReflection];
}
exports.generateAliases = generateAliases;
//# sourceMappingURL=AliasGen.js.map