"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const os_1 = require("os");
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const child_process_1 = __importDefault(require("child_process"));
const chalk_1 = __importDefault(require("chalk"));
const os_2 = __importDefault(require("os"));
const shelljs_1 = __importDefault(require("shelljs"));
const version_1 = __importDefault(require("./version"));
const checkRequirements = __importStar(require("./checkRequirements"));
const commandWithProgress_1 = require("./commandWithProgress");
const child_process_2 = require("child_process");
const vsInstalls_1 = require("./vsInstalls");
const telemetry_1 = require("@react-native-windows/telemetry");
class MSBuildTools {
    /**
     * @param version is something like 17.0 for 2022
     * @param installationPath  Path to installation root
     * @param installationVersion is the full version e.g. 17.3.32929.385
     */
    constructor(version, installationPath, installationVersion) {
        this.version = version;
        this.installationPath = installationPath;
        this.installationVersion = installationVersion;
    }
    /**
     * @returns directory where x86 msbuild can be found
     */
    msbuildPath() {
        return path_1.default.join(this.installationPath, 'MSBuild/Current/Bin');
    }
    cleanProject(slnFile) {
        const cmd = `"${path_1.default.join(this.msbuildPath(), 'msbuild.exe')}" "${slnFile}" /t:Clean`;
        const results = child_process_1.default.execSync(cmd).toString().split(os_1.EOL);
        results.forEach(result => console.log(chalk_1.default.white(result)));
    }
    async restorePackageConfigs(slnFile, buildArch, buildType) {
        const text = 'Restoring NuGet packages ';
        const spinner = (0, commandWithProgress_1.newSpinner)(text);
        await (0, commandWithProgress_1.commandWithProgress)(spinner, text, path_1.default.join(this.msbuildPath(), 'msbuild.exe'), [
            slnFile,
            '/t:Restore',
            '/p:RestoreProjectStyle=PackagesConfig',
            '/p:RestorePackagesConfig=true',
            `/p:Platform=${buildArch}`,
            `/p:Configuration=${buildType}`,
        ], true, 'MSBuildError');
    }
    async buildProject(slnFile, buildType, buildArch, msBuildProps, verbose, target, buildLogDirectory, singleproc) {
        (0, commandWithProgress_1.newSuccess)(`Found Solution: ${slnFile}`);
        (0, commandWithProgress_1.newInfo)(`Build configuration: ${buildType}`);
        (0, commandWithProgress_1.newInfo)(`Build platform: ${buildArch}`);
        const verbosityOption = verbose ? 'normal' : 'minimal';
        const logPrefix = path_1.default.join(buildLogDirectory || os_2.default.tmpdir(), `msbuild_${process.pid}_${target}`);
        const errorLog = logPrefix + '.err';
        const warnLog = logPrefix + '.wrn';
        const localBinLog = target === 'build' ? '' : ':deploy.binlog';
        const binlog = buildLogDirectory ? `:${logPrefix}.binlog` : localBinLog;
        const args = [
            `/clp:NoSummary;NoItemAndPropertyList;Verbosity=${verbosityOption}`,
            '/nologo',
            `/p:Configuration=${buildType}`,
            `/p:Platform=${buildArch}`,
            '/p:AppxBundle=Never',
            `/bl${binlog}`,
            `/flp1:errorsonly;logfile=${errorLog}`,
            `/flp2:warningsonly;logfile=${warnLog}`,
        ];
        // Building projects in parallel increases compiler memory usage and
        // doesn't lead to dramatic performance gains (See #4739). Only enable
        // parallel builds on machines with >16GB of memory to avoid OOM errors
        const highMemory = (0, os_1.totalmem)() > 16 * 1024 * 1024 * 1024;
        const enableParallelBuilds = singleproc === false || highMemory;
        if (enableParallelBuilds) {
            args.push('/maxCpuCount');
        }
        if (target === 'build') {
            args.push('/restore', '/p:RestorePackagesConfig=true');
        }
        else {
            args.push(`/t:Deploy`);
        }
        Object.keys(msBuildProps).forEach(key => {
            args.push(`/p:${key}=${msBuildProps[key]}`);
        });
        try {
            checkRequirements.isWinSdkPresent('10.0');
        }
        catch (e) {
            (0, commandWithProgress_1.newError)(e.message);
            throw e;
        }
        if (verbose) {
            console.log(`Running MSBuild with args ${args.join(' ')}`);
        }
        const progressName = target === 'deploy' ? 'Deploying Solution' : 'Building Solution';
        const spinner = (0, commandWithProgress_1.newSpinner)(progressName);
        try {
            await (0, commandWithProgress_1.commandWithProgress)(spinner, progressName, path_1.default.join(this.msbuildPath(), 'msbuild.exe'), [slnFile].concat(args), verbose, 'MSBuildError');
        }
        catch (e) {
            let error = e;
            if (e instanceof telemetry_1.CodedError) {
                const origCodedError = e;
                if (origCodedError.type === 'MSBuildError') {
                    // Try to parse msbuild errors from errorLog
                    const errorLogContents = (await fs_1.default.readFile(errorLog))
                        .toString()
                        .split(os_1.EOL)
                        .filter(s => s)
                        .map(s => s.trim());
                    if (errorLogContents.length > 0) {
                        const firstMessage = errorLogContents[0];
                        error = new telemetry_1.CodedError('MSBuildError', firstMessage, origCodedError.data);
                        // Hide error messages in a field that won't automatically get reported
                        // with telemetry but is still available to be parsed and sanitized
                        error.msBuildErrorMessages = errorLogContents;
                    }
                }
            }
            throw error;
        }
        // If we have no errors, delete the error log when we're done
        if ((await fs_1.default.stat(errorLog)).size === 0) {
            await fs_1.default.unlink(errorLog);
        }
    }
    static findAvailableVersion(buildArch, verbose, prerelease) {
        // https://aka.ms/vs/workloads
        const requires = [
            'Microsoft.Component.MSBuild',
            getVCToolsByArch(buildArch),
        ];
        const minVersion = process.env.VisualStudioVersion || '17.0';
        const vsInstallation = (0, vsInstalls_1.findLatestVsInstall)({
            requires,
            minVersion,
            verbose,
            prerelease,
        });
        if (!vsInstallation) {
            if (process.env.VisualStudioVersion != null) {
                throw new telemetry_1.CodedError('NoMSBuild', `MSBuild tools not found for version ${process.env.VisualStudioVersion} (from environment). Make sure all required components have been installed`, { VisualStudioVersionFromEnv: process.env.VisualStudioVersion });
            }
            else {
                throw new telemetry_1.CodedError('NoMSBuild', `Could not find MSBuild with VCTools for Visual Studio ${minVersion} or later. Make sure all required components have been installed`, { minVersion: minVersion });
            }
        }
        const toolsPath = path_1.default.join(vsInstallation.installationPath, 'MSBuild/Current/Bin');
        if (fs_1.default.existsSync(toolsPath)) {
            if (verbose) {
                (0, commandWithProgress_1.newSuccess)(`Found compatible MSBuild at ${toolsPath} (${vsInstallation.installationVersion})`);
            }
            return new MSBuildTools(minVersion, vsInstallation.installationPath, vsInstallation.installationVersion);
        }
        else {
            throw new telemetry_1.CodedError('NoMSBuild', `MSBuild path '${toolsPath} does not exist'`);
        }
    }
    static getAllAvailableUAPVersions() {
        const results = [];
        const programFilesFolder = process.env['ProgramFiles(x86)'] || process.env.ProgramFiles;
        // No Program Files folder found, so we won't be able to find UAP SDK
        if (!programFilesFolder) {
            return results;
        }
        let uapFolderPath = path_1.default.join(programFilesFolder, 'Windows Kits', '10', 'Platforms', 'UAP');
        if (!shelljs_1.default.test('-e', uapFolderPath)) {
            // Check other installation folder from reg
            const sdkFolder = getSDK10InstallationFolder();
            if (sdkFolder) {
                uapFolderPath = path_1.default.join(sdkFolder, 'Platforms', 'UAP');
            }
        }
        // No UAP SDK exists on this machine
        if (!shelljs_1.default.test('-e', uapFolderPath)) {
            return results;
        }
        shelljs_1.default
            .ls(uapFolderPath)
            .filter(uapDir => shelljs_1.default.test('-d', path_1.default.join(uapFolderPath, uapDir)))
            .map(version_1.default.tryParse)
            .forEach(version => version && results.push(version));
        return results;
    }
    evaluateMSBuildProperties(solutionFile, projectFile, propertyNames, extraMsBuildProps) {
        const spinner = (0, commandWithProgress_1.newSpinner)('Running Eval-MsBuildProperties.ps1');
        try {
            const msbuildEvalScriptPath = path_1.default.resolve(__dirname, '..', '..', 'src', 'powershell', 'Eval-MsBuildProperties.ps1');
            let command = `${commandWithProgress_1.powershell} -ExecutionPolicy Unrestricted -NoProfile "${msbuildEvalScriptPath}" -SolutionFile '${solutionFile}' -ProjectFile '${projectFile}' -MSBuildPath '${this.msbuildPath()}'`;
            if (propertyNames && propertyNames.length > 0) {
                command += ` -PropertyNames '${propertyNames.join(',')}'`;
            }
            if (extraMsBuildProps) {
                command += " -ExtraMSBuildProps '";
                for (const extraProp in extraMsBuildProps) {
                    if (!(extraProp in Object.prototype)) {
                        command += `,${extraProp}=${extraMsBuildProps[extraProp]}`;
                    }
                }
                command += "'";
            }
            const commandOutput = (0, child_process_2.execSync)(command).toString();
            spinner.succeed();
            const properties = JSON.parse(commandOutput);
            spinner.succeed();
            return properties;
        }
        catch (e) {
            spinner.fail('Running Eval-MsBuildProperties.ps1 failed: ' + e.message);
            throw e;
        }
    }
}
exports.default = MSBuildTools;
function getVCToolsByArch(buildArch) {
    switch (buildArch) {
        case 'x86':
        case 'x64':
            return 'Microsoft.VisualStudio.Component.VC.Tools.x86.x64';
        case 'ARM64':
            return 'Microsoft.VisualStudio.Component.VC.Tools.ARM64';
    }
}
function getSDK10InstallationFolder() {
    const folder = '';
    const execString = 'reg query "HKLM\\SOFTWARE\\Microsoft\\Microsoft SDKs\\Windows\\v10.0" /s /v InstallationFolder /reg:32';
    let output;
    try {
        output = (0, child_process_2.execSync)(execString).toString();
    }
    catch (e) {
        return folder;
    }
    const re = /\\Microsoft SDKs\\Windows\\v10.0\s*InstallationFolder\s+REG_SZ\s+(.*)/gim;
    const match = re.exec(output);
    if (match) {
        return match[1];
    }
    return folder;
}
//# sourceMappingURL=msbuildtools.js.map