"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseMsBuildProps = exports.getAppProjectFile = exports.getAppSolutionFile = exports.buildSolution = void 0;
const path_1 = __importDefault(require("path"));
const msbuildtools_1 = __importDefault(require("./msbuildtools"));
const version_1 = __importDefault(require("./version"));
const commandWithProgress_1 = require("./commandWithProgress");
const telemetry_1 = require("@react-native-windows/telemetry");
async function buildSolution(buildTools, slnFile, buildType, buildArch, msBuildProps, verbose, target, buildLogDirectory, singleproc) {
    const minVersion = new version_1.default(10, 0, 19041, 0);
    const allVersions = msbuildtools_1.default.getAllAvailableUAPVersions();
    if (!allVersions.some(v => v.gte(minVersion))) {
        throw new telemetry_1.CodedError('MinSDKVersionNotMet', 'Must have a minimum Windows SDK version 10.0.19041.0 installed');
    }
    await buildTools.buildProject(slnFile, buildType, buildArch, msBuildProps, verbose, target, buildLogDirectory, singleproc);
}
exports.buildSolution = buildSolution;
const configErrorString = 'Error: ';
function getAppSolutionFile(options, config) {
    // Use the solution file if specified
    if (options.sln) {
        return path_1.default.join(options.root, options.sln);
    }
    // Check the answer from react-native config
    const windowsAppConfig = config.project.windows;
    if (!windowsAppConfig) {
        throw new telemetry_1.CodedError('NoWindowsConfig', "Couldn't determine Windows app config");
    }
    const configSolutionFile = windowsAppConfig.solutionFile;
    if (configSolutionFile.startsWith(configErrorString)) {
        (0, commandWithProgress_1.newError)(configSolutionFile.substr(configErrorString.length) +
            ' Optionally, use --sln {slnFile}.');
        return null;
    }
    else {
        return path_1.default.join(windowsAppConfig.folder, windowsAppConfig.sourceDir, configSolutionFile);
    }
}
exports.getAppSolutionFile = getAppSolutionFile;
function getAppProjectFile(options, config) {
    // Use the project file if specified
    if (options.proj) {
        return path_1.default.join(options.root, options.proj);
    }
    // Check the answer from react-native config
    const windowsAppConfig = config.project.windows;
    const configProject = windowsAppConfig.project;
    if (typeof configProject === 'string' &&
        configProject.startsWith(configErrorString)) {
        (0, commandWithProgress_1.newError)(configProject.substr(configErrorString.length) +
            ' Optionally, use --proj {projFile}.');
        return null;
    }
    else {
        const configProjectFile = configProject.projectFile;
        if (configProjectFile.startsWith(configErrorString)) {
            (0, commandWithProgress_1.newError)(configProjectFile.substr(configErrorString.length) +
                ' Optionally, use --proj {projFile}.');
            return null;
        }
        return path_1.default.join(windowsAppConfig.folder, windowsAppConfig.sourceDir, configProjectFile);
    }
}
exports.getAppProjectFile = getAppProjectFile;
function parseMsBuildProps(options) {
    const result = {};
    if (options.msbuildprops) {
        const props = options.msbuildprops.split(',');
        for (const prop of props) {
            const propAssignment = prop.split('=');
            result[propAssignment[0]] = propAssignment[1];
        }
    }
    return result;
}
exports.parseMsBuildProps = parseMsBuildProps;
//# sourceMappingURL=build.js.map