"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.installScriptsAndDependencies = exports.copyProjectTemplateAndReplace = void 0;
const chalk_1 = __importDefault(require("chalk"));
const path_1 = __importDefault(require("path"));
const username_1 = __importDefault(require("username"));
const uuid_1 = __importDefault(require("uuid"));
const child_process_1 = __importDefault(require("child_process"));
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const semver_1 = __importDefault(require("semver"));
const lodash_1 = __importDefault(require("lodash"));
const find_up_1 = __importDefault(require("find-up"));
const configUtils_1 = require("../commands/config/configUtils");
const generator_common_1 = require("../generator-common");
const telemetry_1 = require("@react-native-windows/telemetry");
const package_utils_1 = require("@react-native-windows/package-utils");
const windowsDir = 'windows';
const bundleDir = 'Bundle';
function pascalCase(str) {
    const camelCase = lodash_1.default.camelCase(str);
    return camelCase[0].toUpperCase() + camelCase.substr(1);
}
function resolveRnwPath(subpath) {
    return require.resolve(path_1.default.join('react-native-windows', subpath), {
        paths: [process.cwd()],
    });
}
// Existing high cyclomatic complexity
// eslint-disable-next-line complexity
async function copyProjectTemplateAndReplace(srcRootPath, destPath, newProjectName, namespace, options) {
    if (!srcRootPath) {
        throw new telemetry_1.CodedError('CopyProjectTemplateNoSourcePath', 'Need a path to copy from');
    }
    if (!destPath) {
        throw new telemetry_1.CodedError('CopyProjectTemplateNoDestPath', 'Need a path to copy to');
    }
    if (!newProjectName) {
        throw new telemetry_1.CodedError('CopyProjectTemplateNoProjectName', 'Need a project name');
    }
    const projectType = options.projectType;
    const language = options.language;
    // React-native init only allows alphanumerics in project names, but other
    // new project tools (like create-react-native-module) are less strict.
    if (projectType === 'lib') {
        newProjectName = pascalCase(newProjectName);
    }
    // Similar to the above, but we want to retain namespace separators
    if (projectType === 'lib') {
        namespace = namespace.split(/[.:]+/).map(pascalCase).join('.');
    }
    // Checking if we're overwriting an existing project and re-uses their projectGUID
    const existingProjectPath = path_1.default.join(destPath, windowsDir, newProjectName, newProjectName + (language === 'cs' ? '.csproj' : '.vcxproj'));
    let existingProjectGuid;
    if (fs_1.default.existsSync(existingProjectPath)) {
        console.log('Found existing project, extracting ProjectGuid.');
        existingProjectGuid = (0, configUtils_1.findPropertyValue)((0, configUtils_1.readProjectFile)(existingProjectPath), 'ProjectGuid', existingProjectPath).replace(/[{}]/g, '');
    }
    (0, generator_common_1.createDir)(path_1.default.join(destPath, windowsDir));
    (0, generator_common_1.createDir)(path_1.default.join(destPath, windowsDir, newProjectName));
    if (projectType === 'app') {
        (0, generator_common_1.createDir)(path_1.default.join(destPath, windowsDir, newProjectName, bundleDir));
        (0, generator_common_1.createDir)(path_1.default.join(destPath, windowsDir, newProjectName, 'BundleBuilder'));
    }
    const namespaceCpp = toCppNamespace(namespace);
    if (options.experimentalNuGetDependency) {
        console.log('Using experimental NuGet dependency.');
    }
    const experimentalPropsPath = path_1.default.join(destPath, windowsDir, 'ExperimentalFeatures.props');
    let existingUseHermes = null;
    if (fs_1.default.existsSync(experimentalPropsPath)) {
        existingUseHermes = (0, configUtils_1.tryFindPropertyValueAsBoolean)((0, configUtils_1.readProjectFile)(experimentalPropsPath), 'UseHermes');
    }
    if (existingUseHermes === false) {
        console.warn('Hermes is now the default JS engine and will be enabled for this project. Support for Chakra will be deprecated in the future. To disable Hermes and keep using Chakra for now, see https://microsoft.github.io/react-native-windows/docs/hermes#disabling-hermes.');
    }
    options.useHermes = true;
    if (options.useWinUI3) {
        throw new telemetry_1.CodedError('IncompatibleOptions', 'Experimental WinUI 3 project has been deprecated.');
    }
    const projDir = 'proj';
    const srcPath = path_1.default.join(srcRootPath, `${language}-${projectType}`);
    const sharedPath = path_1.default.join(srcRootPath, `shared-${projectType}`);
    const projectGuid = existingProjectGuid || uuid_1.default.v4();
    const rnwVersion = require(resolveRnwPath('package.json')).version;
    const nugetVersion = options.nuGetTestVersion || rnwVersion;
    const packageGuid = uuid_1.default.v4();
    const currentUser = username_1.default.sync(); // Gets the current username depending on the platform.
    let mainComponentName = newProjectName;
    const appJsonPath = await (0, find_up_1.default)('app.json', { cwd: destPath });
    if (appJsonPath) {
        const appJson = await fs_1.default.readJsonFile(appJsonPath);
        mainComponentName = appJson.name;
    }
    // We should prefer putting new, necessary PackageReference dependencies into the appropriate
    // external property sheets, but this is here if we "must" inject the dependency into the project file
    const csNugetPackages = [];
    const cppNugetPackages = [];
    const templateVars = {
        useMustache: true,
        regExpPatternsToRemove: [],
        name: newProjectName,
        namespace: namespace,
        namespaceCpp: namespaceCpp,
        languageIsCpp: language === 'cpp',
        rnwVersion: await (0, telemetry_1.getVersionOfNpmPackage)('react-native-windows'),
        mainComponentName: mainComponentName,
        // Visual Studio is very picky about the casing of the guids for projects, project references and the solution
        // https://www.bing.com/search?q=visual+studio+project+guid+casing&cvid=311a5ad7f9fc41089507b24600d23ee7&FORM=ANAB01&PC=U531
        // we therefore have to precariously use the right casing in the right place or risk building in VS breaking.
        projectGuidLower: `{${projectGuid.toLowerCase()}}`,
        projectGuidUpper: `{${projectGuid.toUpperCase()}}`,
        // packaging and signing variables:
        packageGuid: packageGuid,
        currentUser: currentUser,
        useExperimentalNuget: options.experimentalNuGetDependency,
        nuGetTestFeed: options.nuGetTestFeed,
        nuGetADOFeed: nugetVersion.startsWith('0.0.0-'),
        // cpp template variables
        useWinUI3: options.useWinUI3,
        useHermes: options.useHermes,
        cppNugetPackages: cppNugetPackages,
        // cs template variables
        csNugetPackages: csNugetPackages,
        // autolinking template variables
        autolinkPropertiesForProps: '',
        autolinkProjectReferencesForTargets: '',
        autolinkCsUsingNamespaces: '',
        autolinkCsReactPackageProviders: '',
        autolinkCppIncludes: '',
        autolinkCppPackageProviders: '\n    UNREFERENCED_PARAMETER(packageProviders);', // CODESYNC: vnext\local-cli\runWindows\utils\autolink.js
    };
    const commonMappings = projectType === 'app'
        ? [
            // app common mappings
            {
                from: path_1.default.join(srcRootPath, options.useDevMode
                    ? 'metro.devMode.config.js'
                    : 'metro.config.js'),
                to: 'metro.config.js',
            },
            {
                from: path_1.default.join(srcRootPath, '_gitignore'),
                to: path_1.default.join(windowsDir, '.gitignore'),
            },
            {
                from: path_1.default.join(srcRootPath, 'b_gitignore'),
                to: path_1.default.join(windowsDir, newProjectName, '.gitignore'),
            },
            {
                from: path_1.default.join(srcRootPath, 'index.windows.bundle'),
                to: path_1.default.join(windowsDir, newProjectName, bundleDir, 'index.windows.bundle'),
            },
            {
                from: path_1.default.join(srcPath, projDir, 'MyApp.sln'),
                to: path_1.default.join(windowsDir, newProjectName + '.sln'),
            },
        ]
        : [
            // lib common mappings
            {
                from: path_1.default.join(srcRootPath, '_gitignore'),
                to: path_1.default.join(windowsDir, '.gitignore'),
            },
            {
                from: path_1.default.join(srcPath, projDir, 'MyLib.sln'),
                to: path_1.default.join(windowsDir, newProjectName + '.sln'),
            },
        ];
    for (const mapping of commonMappings) {
        await (0, generator_common_1.copyAndReplaceWithChangedCallback)(mapping.from, destPath, mapping.to, templateVars, options.overwrite);
    }
    if (language === 'cs') {
        const csMappings = projectType === 'app'
            ? [
                // cs app mappings
                {
                    from: path_1.default.join(srcPath, projDir, 'MyApp.csproj'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.csproj'),
                },
            ]
            : [
                // cs lib mappings
                {
                    from: path_1.default.join(srcPath, projDir, 'MyLib.csproj'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.csproj'),
                },
            ];
        for (const mapping of csMappings) {
            await (0, generator_common_1.copyAndReplaceWithChangedCallback)(mapping.from, destPath, mapping.to, templateVars, options.overwrite);
        }
    }
    else {
        const cppMappings = projectType === 'app'
            ? [
                // cpp app mappings
                {
                    from: path_1.default.join(srcPath, projDir, 'MyApp.vcxproj'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.vcxproj'),
                },
                {
                    from: path_1.default.join(srcPath, projDir, 'MyApp.vcxproj.filters'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.vcxproj.filters'),
                },
            ]
            : [
                // cpp lib mappings
                {
                    from: path_1.default.join(srcPath, projDir, 'MyLib.vcxproj'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.vcxproj'),
                },
                {
                    from: path_1.default.join(srcPath, projDir, 'MyLib.vcxproj.filters'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.vcxproj.filters'),
                },
                {
                    from: path_1.default.join(srcPath, projDir, 'MyLib.def'),
                    to: path_1.default.join(windowsDir, newProjectName, newProjectName + '.def'),
                },
            ];
        for (const mapping of cppMappings) {
            await (0, generator_common_1.copyAndReplaceWithChangedCallback)(mapping.from, destPath, mapping.to, templateVars, options.overwrite);
        }
    }
    // shared proj
    if (fs_1.default.existsSync(path_1.default.join(sharedPath, projDir))) {
        const sharedProjMappings = [];
        sharedProjMappings.push({
            from: path_1.default.join(sharedPath, projDir, 'NuGet_Config'),
            to: path_1.default.join(windowsDir, 'NuGet.Config'),
        });
        if (fs_1.default.existsSync(path_1.default.join(sharedPath, projDir, 'ExperimentalFeatures.props'))) {
            sharedProjMappings.push({
                from: path_1.default.join(sharedPath, projDir, 'ExperimentalFeatures.props'),
                to: path_1.default.join(windowsDir, 'ExperimentalFeatures.props'),
            });
        }
        for (const mapping of sharedProjMappings) {
            await (0, generator_common_1.copyAndReplaceWithChangedCallback)(mapping.from, destPath, mapping.to, templateVars, options.overwrite);
        }
    }
    // shared assets
    if (fs_1.default.existsSync(path_1.default.join(sharedPath, 'assets'))) {
        await (0, generator_common_1.copyAndReplaceAll)(path_1.default.join(sharedPath, 'assets'), destPath, path_1.default.join(windowsDir, newProjectName, 'Assets'), templateVars, options.overwrite);
    }
    // shared src
    if (fs_1.default.existsSync(path_1.default.join(sharedPath, 'src'))) {
        await (0, generator_common_1.copyAndReplaceAll)(path_1.default.join(sharedPath, 'src'), destPath, path_1.default.join(windowsDir, newProjectName), templateVars, options.overwrite);
    }
    // src
    if (fs_1.default.existsSync(path_1.default.join(srcPath, 'src'))) {
        await (0, generator_common_1.copyAndReplaceAll)(path_1.default.join(srcPath, 'src'), destPath, path_1.default.join(windowsDir, newProjectName), templateVars, options.overwrite);
    }
    if (projectType === 'app') {
        console.log(chalk_1.default.white.bold('To run your app on UWP:'));
        console.log(chalk_1.default.white('   npx react-native run-windows'));
    }
}
exports.copyProjectTemplateAndReplace = copyProjectTemplateAndReplace;
function toCppNamespace(namespace) {
    return namespace.replace(/\./g, '::');
}
async function installScriptsAndDependencies(options) {
    var _a;
    const projectPackage = await package_utils_1.WritableNpmPackage.fromPath(process.cwd());
    if (!projectPackage) {
        throw new Error(`The current directory '${process.cwd()}' is not the root of an npm package`);
    }
    await projectPackage.mergeProps({
        scripts: { windows: 'react-native run-windows' },
    });
    const rnwPackage = await (0, package_utils_1.findPackage)('react-native-windows');
    if (!rnwPackage) {
        throw new Error('Could not locate the package for react-native-windows');
    }
    const rnPackage = await (0, package_utils_1.findPackage)('react-native');
    if (!rnPackage) {
        throw new Error('Could not locate the package for react-native');
    }
    // We add an exclusionList from metro config. This will be hoisted, but add
    // an explicit dep because we require it directly.
    const cliPackage = await (0, package_utils_1.findPackage)('@react-native-community/cli', {
        searchPath: rnPackage.path,
    });
    const metroConfigPackage = await (0, package_utils_1.findPackage)('metro-config', {
        searchPath: (cliPackage === null || cliPackage === void 0 ? void 0 : cliPackage.path) || rnPackage.path,
    });
    if (metroConfigPackage) {
        await projectPackage.mergeProps({
            devDependencies: {
                'metro-config': `^${metroConfigPackage.json.version}`,
            },
        });
    }
    const rnPeerDependency = rnwPackage.json.peerDependencies['react-native'];
    if (!semver_1.default.satisfies(rnPackage.json.version, rnPeerDependency) &&
        ((_a = projectPackage.json.dependencies) === null || _a === void 0 ? void 0 : _a['react-native'])) {
        console.log(chalk_1.default.green('Installing a compatible version of react-native:'));
        console.log(chalk_1.default.white(`    ${rnPeerDependency}`));
        // Patch package.json to have proper react-native version and install
        await projectPackage.mergeProps({
            dependencies: { 'react-native': rnPeerDependency },
        });
        // Install dependencies using correct package manager
        const isYarn = fs_1.default.existsSync(path_1.default.join(process.cwd(), 'yarn.lock'));
        child_process_1.default.execSync(isYarn ? 'yarn' : 'npm i', options.verbose ? { stdio: 'inherit' } : {});
    }
}
exports.installScriptsAndDependencies = installScriptsAndDependencies;
//# sourceMappingURL=index.js.map