"use strict";
/**
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const dependencyConfig_1 = require("../commands/config/dependencyConfig");
const projectConfig_utils_1 = require("./projectConfig.utils");
const generator_common_1 = require("../generator-common");
function project(name, setup) {
    return [name, setup];
}
const projects = [
    // Nothing but a react-native.config.js, with nulls
    project('BlankLib'),
    // Nothing but a windows folder
    project('MissingProjectFilesLib', async (folder) => {
        const windowsDir = path_1.default.join(folder, 'windows');
        await (0, projectConfig_utils_1.tryMkdir)(windowsDir);
    }),
    // New C++ app project based on the template
    project('SimpleCppApp', async (folder) => {
        await (0, projectConfig_utils_1.ensureCppAppProject)(folder, 'SimpleCppApp');
    }),
    // New C++ project based on the template
    project('SimpleCppLib', async (folder) => {
        const windowsDir = path_1.default.join(folder, 'windows');
        await (0, projectConfig_utils_1.tryMkdir)(windowsDir);
        const replacements = {
            name: 'SimpleCppLib',
            namespace: 'SimpleCppLib',
            useMustache: true,
            projectGuidUpper: projectConfig_utils_1.testProjectGuid,
            projectGuidLower: projectConfig_utils_1.testProjectGuid.toLowerCase(),
        };
        await (0, generator_common_1.copyAndReplace)(path_1.default.join(projectConfig_utils_1.templateRoot, 'cpp-lib/proj/MyLib.sln'), path_1.default.join(windowsDir, 'SimpleCppLib.sln'), replacements, null);
        const projDir = path_1.default.join(windowsDir, 'SimpleCppLib');
        await (0, projectConfig_utils_1.tryMkdir)(projDir);
        await (0, generator_common_1.copyAndReplace)(path_1.default.join(projectConfig_utils_1.templateRoot, 'cpp-lib/proj/MyLib.vcxproj'), path_1.default.join(projDir, 'SimpleCppLib.vcxproj'), replacements, null);
    }),
    // New C# app project based on the template
    project('SimpleCSharpApp', async (folder) => {
        await (0, projectConfig_utils_1.ensureCSharpAppProject)(folder, 'SimpleCSharpApp');
    }),
    // New C# project based on the template
    project('SimpleCSharpLib', async (folder) => {
        const windowsDir = path_1.default.join(folder, 'windows');
        await (0, projectConfig_utils_1.tryMkdir)(windowsDir);
        const replacements = {
            name: 'SimpleCSharpLib',
            namespace: 'SimpleCSharpLib',
            useMustache: true,
            projectGuidUpper: projectConfig_utils_1.testProjectGuid,
            projectGuidLower: projectConfig_utils_1.testProjectGuid.toLowerCase(),
        };
        await (0, generator_common_1.copyAndReplace)(path_1.default.join(projectConfig_utils_1.templateRoot, 'cs-lib/proj/MyLib.sln'), path_1.default.join(windowsDir, 'SimpleCSharpLib.sln'), replacements, null);
        const projDir = path_1.default.join(windowsDir, 'SimpleCSharpLib');
        await (0, projectConfig_utils_1.tryMkdir)(projDir);
        await (0, generator_common_1.copyAndReplace)(path_1.default.join(projectConfig_utils_1.templateRoot, 'cs-lib/proj/MyLib.csproj'), path_1.default.join(projDir, 'SimpleCSharpLib.csproj'), replacements, null);
    }),
    project('WithIndirectDependency'),
];
// Tests that given userConfig is null, the result will always be null
test.each(projects)('dependencyConfig - %s (userConfig is null)', async (name, setup) => {
    const folder = path_1.default.resolve('src/e2etest/projects/', name);
    if (setup !== undefined) {
        await setup(folder);
    }
    const userConfig = null;
    expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toBeNull();
});
// Tests the result given a windows project config in react-native.config.js
test.each(projects)('dependencyConfig - %s (Use react-native.config.js)', async (name, setup) => {
    const folder = path_1.default.resolve('src/e2etest/projects/', name);
    const rnc = require(path_1.default.join(folder, 'react-native.config.js'));
    if (setup !== undefined) {
        await setup(folder);
    }
    const userConfig = rnc.dependency.platforms.windows;
    if (name === 'BlankLib') {
        expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toMatchSnapshot();
    }
    else if (name.endsWith('App')) {
        expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toMatchSnapshot({
            folder: expect.stringContaining(name),
            projects: expect.arrayContaining([
                expect.objectContaining({
                    projectFile: expect.stringMatching(/Error: .*\.(?:cs|vcx)proj is type '\w+'/),
                }),
            ]),
        });
    }
    else {
        expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toMatchSnapshot({
            folder: expect.stringContaining(name),
        });
    }
});
// Tests the result of ignoring the windows project config in react-native.config.js
test.each(projects)('dependencyConfig - %s (Ignore react-native.config.js)', async (name, setup) => {
    const folder = path_1.default.resolve('src/e2etest/projects/', name);
    if (setup !== undefined) {
        await setup(folder);
    }
    const userConfig = {};
    if (name === 'BlankLib') {
        expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toMatchSnapshot();
    }
    else if (name.endsWith('App')) {
        expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toMatchSnapshot({
            folder: expect.stringContaining(name),
            projects: expect.arrayContaining([
                expect.objectContaining({
                    projectFile: expect.stringMatching(/Error: .*\.(?:cs|vcx)proj is type '\w+'/),
                }),
            ]),
        });
    }
    else {
        expect((0, dependencyConfig_1.dependencyConfigWindows)(folder, userConfig)).toMatchSnapshot({
            folder: expect.stringContaining(name),
        });
    }
});
//# sourceMappingURL=dependencyConfig.test.js.map