"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.runWindowsCommand = void 0;
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const telemetry_1 = require("@react-native-windows/telemetry");
const build = __importStar(require("../../utils/build"));
const chalk_1 = __importDefault(require("chalk"));
const deploy = __importStar(require("../../utils/deploy"));
const commandWithProgress_1 = require("../../utils/commandWithProgress");
const telemetryHelpers_1 = require("../../utils/telemetryHelpers");
const pathHelpers = __importStar(require("../../utils/pathHelpers"));
const info = __importStar(require("../../utils/info"));
const msbuildtools_1 = __importDefault(require("../../utils/msbuildtools"));
const runWindowsOptions_1 = require("./runWindowsOptions");
const autolinkWindows_1 = require("../autolinkWindows/autolinkWindows");
/**
 * Sanitizes the given option for telemetry.
 * @param key The key of the option.
 * @param value The unsanitized value of the option.
 * @returns The sanitized value of the option.
 */
// eslint-disable-next-line complexity
function optionSanitizer(key, value) {
    // Do not add a default case here.
    // Strings risking PII should just return true if present, false otherwise.
    // All others should return the value (or false if undefined).
    switch (key) {
        case 'root':
        case 'target':
        case 'sln':
        case 'proj':
        case 'buildLogDirectory':
            return value === undefined ? false : true; // Strip PII
        case 'msbuildprops':
            return value === undefined ? 0 : value.split(',').length; // Convert to count
        case 'release':
        case 'arch':
        case 'singleproc':
        case 'emulator':
        case 'device':
        case 'remoteDebugging':
        case 'logging':
        case 'packager':
        case 'bundle':
        case 'launch':
        case 'autolink':
        case 'build':
        case 'deploy':
        case 'deployFromLayout':
        case 'info':
        case 'directDebugging':
        case 'telemetry':
            return value === undefined ? false : value; // Return value
    }
}
/**
 * Get the extra props to add to the `run-windows` telemetry event.
 * @returns The extra props.
 */
async function getExtraProps() {
    const extraProps = {
        phase: runWindowsPhase,
        hasRunRnwDependencies,
        msBuildProps: evaluateMSBuildPropsCallback
            ? evaluateMSBuildPropsCallback()
            : {},
    };
    return extraProps;
}
let runWindowsPhase = 'None';
let hasRunRnwDependencies = false;
let evaluateMSBuildPropsCallback;
/**
 * The function run when calling `react-native run-windows`.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function runWindows(args, config, options) {
    await (0, telemetryHelpers_1.startTelemetrySession)('run-windows', config, options, (0, telemetryHelpers_1.getDefaultOptions)(config, runWindowsOptions_1.runWindowsOptions), optionSanitizer);
    // https://github.com/yarnpkg/yarn/issues/8334 - Yarn on Windows breaks apps that read from the environment variables
    // Yarn will run node via CreateProcess and pass npm_config_* variables in lowercase without unifying their value
    // with their possibly existing uppercase counterparts. This breaks programs that read from the environment block
    // and write to a case-insensitive dictionary since they expect to encounter each variable only once.
    // The values of the lowercase variables are the one npm actually wants to use, plus they are seeded from the
    // uppercase variable values one if there are no overrides.
    delete process.env.NPM_CONFIG_CACHE;
    delete process.env.NPM_CONFIG_PREFIX;
    if (process.env.LocalAppData) {
        hasRunRnwDependencies = fs_1.default.existsSync(path_1.default.join(process.env.LocalAppData, 'rnw-dependencies.txt')); // CODESYNC \vnext\scripts\rnw-dependencies.ps1
    }
    let runWindowsError;
    if (options.info) {
        runWindowsPhase = 'Info';
        try {
            const output = await info.getEnvironmentInfo();
            console.log(output.trimEnd());
            console.log('  Installed UWP SDKs:');
            const sdks = msbuildtools_1.default.getAllAvailableUAPVersions();
            sdks.forEach(version => console.log('    ' + version));
        }
        catch (ex) {
            runWindowsError =
                ex instanceof Error ? ex : new Error(String(ex));
            telemetry_1.Telemetry.trackException(runWindowsError);
            (0, commandWithProgress_1.newError)('Unable to print environment info.\n' + runWindowsError.toString());
        }
        await (0, telemetryHelpers_1.endTelemetrySession)(runWindowsError, getExtraProps);
        (0, commandWithProgress_1.setExitProcessWithError)(options.logging, runWindowsError);
        return;
    }
    try {
        await runWindowsInternal(args, config, options);
    }
    catch (ex) {
        runWindowsError =
            ex instanceof Error ? ex : new Error(String(ex));
        telemetry_1.Telemetry.trackException(runWindowsError);
        if (!hasRunRnwDependencies) {
            const rnwDependenciesPath = path_1.default.join(pathHelpers.resolveRnwRoot([process.cwd(), __dirname]), 'scripts/rnw-dependencies.ps1');
            (0, commandWithProgress_1.newError)(`It is possible your installation is missing required software dependencies. Dependencies can be automatically installed by running ${rnwDependenciesPath} from an elevated PowerShell prompt.\nFor more information, go to http://aka.ms/rnw-deps`);
        }
    }
    await (0, telemetryHelpers_1.endTelemetrySession)(runWindowsError, getExtraProps);
    (0, commandWithProgress_1.setExitProcessWithError)(options.logging, runWindowsError);
}
/**
 * Performs build deploy and launch of RNW apps.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function runWindowsInternal(args, config, options) {
    const verbose = options.logging === true;
    if (verbose) {
        (0, commandWithProgress_1.newInfo)('Verbose: ON');
    }
    // Get the solution file
    let slnFile;
    runWindowsPhase = 'FindSolution';
    try {
        slnFile = build.getAppSolutionFile(options, config);
    }
    catch (e) {
        (0, commandWithProgress_1.newError)(`Couldn't get app solution information. ${e.message}`);
        throw e;
    }
    let buildTools;
    runWindowsPhase = 'FindBuildTools';
    try {
        buildTools = msbuildtools_1.default.findAvailableVersion(options.arch, verbose);
    }
    catch (error) {
        (0, commandWithProgress_1.newWarn)('No public VS release found');
        // Try prerelease
        try {
            (0, commandWithProgress_1.newInfo)('Trying pre-release VS');
            buildTools = msbuildtools_1.default.findAvailableVersion(options.arch, verbose, true);
        }
        catch (e) {
            (0, commandWithProgress_1.newError)(e.message);
            throw error;
        }
    }
    // Set up the callback to capture MSBuild properties after the command completes
    evaluateMSBuildPropsCallback = () => {
        const projectFile = build.getAppProjectFile(options, config);
        if (projectFile) {
            if (verbose) {
                (0, commandWithProgress_1.newInfo)('Gathering MSBuild data for telemetry.');
            }
            const msBuildPropertiesJsonPath = path_1.default.resolve(path_1.default.dirname(projectFile), 'Generated Files', 'msbuildproperties.g.json');
            if (fs_1.default.existsSync(msBuildPropertiesJsonPath)) {
                if (verbose) {
                    (0, commandWithProgress_1.newInfo)('Loading properties from msbuildproperties.g.json');
                }
                return fs_1.default.readJsonFileSync(msBuildPropertiesJsonPath);
            }
            if (verbose) {
                (0, commandWithProgress_1.newInfo)('Unable to find msbuildproperties.g.json');
            }
        }
        return {};
    };
    // Restore packages.config files for dependencies that don't support PackageReference.
    runWindowsPhase = 'RestorePackagesConfig';
    const buildType = deploy.getBuildConfiguration(options);
    try {
        await buildTools.restorePackageConfigs(slnFile, options.arch, buildType);
    }
    catch (e) {
        (0, commandWithProgress_1.newError)(`Couldn't restore found packages.config instances. ${e.message}`);
        throw e;
    }
    if (options.autolink) {
        runWindowsPhase = 'Autolink';
        try {
            const autolinkArgs = [];
            const autolinkConfig = config;
            const autolinkOptions = {
                logging: options.logging,
                check: false,
                proj: options.proj,
                sln: options.sln,
                telemetry: options.telemetry,
            };
            await (0, autolinkWindows_1.autolinkWindowsInternal)(autolinkArgs, autolinkConfig, autolinkOptions);
        }
        catch (e) {
            (0, commandWithProgress_1.newError)(`Autolinking failed. ${e.message}`);
            throw e;
        }
    }
    else {
        (0, commandWithProgress_1.newInfo)('Autolink step is skipped');
    }
    if (options.build) {
        runWindowsPhase = 'Build';
        if (!slnFile) {
            (0, commandWithProgress_1.newError)('Visual Studio Solution file not found. Maybe run "npx react-native-windows-init" first?');
            throw new telemetry_1.CodedError('NoSolution', 'Cannot find solution file');
        }
        // Get build/deploy options
        const msBuildProps = build.parseMsBuildProps(options);
        // Disable the autolink check since we just ran it
        msBuildProps.RunAutolinkCheck = 'false';
        try {
            await build.buildSolution(buildTools, slnFile, buildType, options.arch, msBuildProps, verbose, 'build', options.buildLogDirectory, options.singleproc);
        }
        catch (e) {
            (0, commandWithProgress_1.newError)(`Build failed with message ${e.message}. Check your build configuration.`);
            if (e.logfile) {
                console.log('See', chalk_1.default.bold(e.logfile));
            }
            throw e;
        }
    }
    else {
        (0, commandWithProgress_1.newInfo)('Build step is skipped');
    }
    if (shouldLaunchPackager(options)) {
        await deploy.startServerInNewWindow(options, verbose);
    }
    if (options.deploy) {
        runWindowsPhase = 'Deploy';
        if (!slnFile) {
            (0, commandWithProgress_1.newError)('Visual Studio Solution file not found. Maybe run "npx react-native-windows-init" first?');
            throw new telemetry_1.CodedError('NoSolution', 'Cannot find solution file');
        }
        try {
            if (options.device || options.emulator || options.target) {
                await deploy.deployToDevice(options, verbose, config);
            }
            else {
                await deploy.deployToDesktop(options, verbose, config, buildTools);
            }
        }
        catch (e) {
            (0, commandWithProgress_1.newError)(`Failed to deploy${e ? `: ${e.message}` : ''}`);
            throw e;
        }
    }
    else {
        (0, commandWithProgress_1.newInfo)('Deploy step is skipped');
    }
}
function shouldLaunchPackager(options) {
    return (options.packager === true &&
        options.launch === true &&
        options.release !== true);
}
/*
// Example of running the Windows Command
runWindows({
  root: 'C:\\github\\hack\\myapp',
  debug: true,
  arch: 'x86',
  nugetPath: 'C:\\github\\react\\react-native-windows\\local-cli\\runWindows\\.nuget\\nuget.exe'
});
*/
/**
 * Starts the app on a connected Windows emulator or mobile device.
 */
exports.runWindowsCommand = {
    name: 'run-windows',
    description: 'Builds your app and starts it on a connected Windows desktop, emulator or device',
    func: runWindows,
    options: runWindowsOptions_1.runWindowsOptions,
};
//# sourceMappingURL=runWindows.js.map