"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.initCommand = exports.initWindowsInternal = exports.InitWindows = void 0;
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const chalk_1 = __importDefault(require("chalk"));
const glob_1 = require("glob");
const lodash_1 = __importDefault(require("lodash"));
const perf_hooks_1 = require("perf_hooks");
const util_1 = __importDefault(require("util"));
const glob = util_1.default.promisify(glob_1.glob);
const telemetry_1 = require("@react-native-windows/telemetry");
const commandWithProgress_1 = require("../../utils/commandWithProgress");
const pathHelpers = __importStar(require("../../utils/pathHelpers"));
const telemetryHelpers_1 = require("../../utils/telemetryHelpers");
const generator_common_1 = require("../../generator-common");
const initWindowsOptions_1 = require("./initWindowsOptions");
class InitWindows {
    constructor(config, options) {
        this.config = config;
        this.options = options;
        this.templates = new Map();
        this.rnwPath = pathHelpers.resolveRnwRoot(this.config.root);
    }
    verboseMessage(message) {
        verboseMessage(message, !!this.options.logging);
    }
    async loadTemplates() {
        const templatesRoot = path_1.default.join(this.rnwPath, 'templates');
        for (const file of await glob('**/template.config.js', {
            cwd: templatesRoot,
        })) {
            const templateName = path_1.default.dirname(file).replace('\\', '/');
            const templateConfig = require(path_1.default.join(templatesRoot, file));
            this.templates.set(templateName, templateConfig);
        }
        if (this.templates.size === 0) {
            throw new telemetry_1.CodedError('NoTemplatesFound', `No templates were found in ${templatesRoot}.`);
        }
    }
    getDefaultTemplateName() {
        for (const [name, config] of this.templates) {
            if (config.isDefault) {
                return name;
            }
        }
        throw new telemetry_1.CodedError('NoDefaultTemplate', 'No template specified and no default template found.');
    }
    pascalCase(str) {
        const camelCase = lodash_1.default.camelCase(str);
        return camelCase[0].toUpperCase() + camelCase.substr(1);
    }
    isValidProjectName(name) {
        if (name.match(/^[a-z][a-z0-9]*$/gi)) {
            return true;
        }
        return false;
    }
    getReactNativeProjectName(projectDir) {
        this.verboseMessage('Looking for project name in package.json...');
        const pkgJsonPath = path_1.default.join(projectDir, 'package.json');
        if (!fs_1.default.existsSync(pkgJsonPath)) {
            throw new telemetry_1.CodedError('NoPackageJson', 'Unable to find package.json. This should be run from within an existing react-native project.');
        }
        let name = fs_1.default.readJsonFileSync(pkgJsonPath).name;
        if (!name) {
            const appJsonPath = path_1.default.join(projectDir, 'app.json');
            if (fs_1.default.existsSync(appJsonPath)) {
                this.verboseMessage('Looking for project name in app.json...');
                name = fs_1.default.readJsonFileSync(pkgJsonPath).name;
            }
        }
        if (!name) {
            throw new telemetry_1.CodedError('NoProjectName', 'Please specify name in package.json or app.json');
        }
        return name;
    }
    async run(spinner) {
        var _a, _b;
        var _c, _d;
        await this.loadTemplates();
        spinner.info();
        (_a = (_c = this.options).template) !== null && _a !== void 0 ? _a : (_c.template = this.getDefaultTemplateName());
        spinner.info(`Using template '${this.options.template}'...`);
        if (!this.templates.has(this.options.template.replace('\\', '/'))) {
            throw new telemetry_1.CodedError('InvalidTemplateName', `Unable to find template '${this.options.template}'.`);
        }
        const templateConfig = this.templates.get(this.options.template);
        if (this.options.name && !this.isValidProjectName(this.options.name)) {
            throw new telemetry_1.CodedError('InvalidProjectName', `The specified name is not a valid identifier`);
        }
        if (!this.options.name) {
            const projectName = this.getReactNativeProjectName(this.config.root);
            this.options.name = this.isValidProjectName(projectName)
                ? projectName
                : this.pascalCase(projectName);
        }
        (_b = (_d = this.options).namespace) !== null && _b !== void 0 ? _b : (_d.namespace = this.options.name);
        if (templateConfig.preInstall) {
            spinner.info(`Running ${this.options.template} preInstall()...`);
            await templateConfig.preInstall(this.config, this.options);
        }
        // Get template files to copy and copy if available
        if (templateConfig.getFileMappings) {
            const fileMappings = await templateConfig.getFileMappings(this.config, this.options);
            for (const fileMapping of fileMappings) {
                const targetDir = path_1.default.join(this.config.root, path_1.default.dirname(fileMapping.to));
                if (!(await fs_1.default.exists(targetDir))) {
                    await fs_1.default.mkdir(targetDir, { recursive: true });
                }
                await (0, generator_common_1.copyAndReplaceWithChangedCallback)(fileMapping.from, this.config.root, fileMapping.to, fileMapping.replacements, this.options.overwrite);
            }
        }
        if (templateConfig.postInstall) {
            spinner.info(`Running ${this.options.template} postInstall()...`);
            await templateConfig.postInstall(this.config, this.options);
        }
        spinner.succeed();
    }
}
exports.InitWindows = InitWindows;
/**
 * Logs the given message if verbose is True.
 * @param message The message to log.
 * @param verbose Whether or not verbose logging is enabled.
 */
function verboseMessage(message, verbose) {
    if (verbose) {
        console.log(message);
    }
}
/**
 * Sanitizes the given option for telemetry.
 * @param key The key of the option.
 * @param value The unsanitized value of the option.
 * @returns The sanitized value of the option.
 */
function optionSanitizer(key, value) {
    // Do not add a default case here.
    // Strings risking PII should just return true if present, false otherwise.
    // All others should return the value (or false if undefined).
    switch (key) {
        case 'name':
        case 'namespace':
            return value === undefined ? false : true; // Strip PII
        case 'logging':
        case 'template':
        case 'overwrite':
        case 'telemetry':
            return value === undefined ? false : value; // Return value
    }
}
/**
 * Get the extra props to add to the `init-windows` telemetry event.
 * @returns The extra props.
 */
async function getExtraProps() {
    const extraProps = {};
    return extraProps;
}
/**
 * The function run when calling `react-native init-windows`.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function initWindows(args, config, options) {
    await (0, telemetryHelpers_1.startTelemetrySession)('init-windows', config, options, (0, telemetryHelpers_1.getDefaultOptions)(config, initWindowsOptions_1.initOptions), optionSanitizer);
    let initWindowsError;
    try {
        await initWindowsInternal(args, config, options);
    }
    catch (ex) {
        initWindowsError =
            ex instanceof Error ? ex : new Error(String(ex));
        telemetry_1.Telemetry.trackException(initWindowsError);
    }
    await (0, telemetryHelpers_1.endTelemetrySession)(initWindowsError, getExtraProps);
    (0, commandWithProgress_1.setExitProcessWithError)(options.logging, initWindowsError);
}
/**
 * Initializes a new RNW project from a given template.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function initWindowsInternal(args, config, options) {
    const startTime = perf_hooks_1.performance.now();
    const spinner = (0, commandWithProgress_1.newSpinner)('Running init-windows...');
    try {
        const codegen = new InitWindows(config, options);
        await codegen.run(spinner);
        const endTime = perf_hooks_1.performance.now();
        console.log(`${chalk_1.default.green('Success:')} init-windows completed. (${Math.round(endTime - startTime)}ms)`);
    }
    catch (e) {
        spinner.fail();
        const endTime = perf_hooks_1.performance.now();
        console.log(`${chalk_1.default.red('Error:')} ${e.toString()}. (${Math.round(endTime - startTime)}ms)`);
        throw e;
    }
}
exports.initWindowsInternal = initWindowsInternal;
/**
 * Initializes a new RNW project from a given template.
 */
exports.initCommand = {
    name: 'init-windows',
    description: 'Initializes a new RNW project from a given template.',
    func: initWindows,
    options: initWindowsOptions_1.initOptions,
};
//# sourceMappingURL=initWindows.js.map