"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.codegenCommand = exports.codegenWindowsInternal = exports.CodeGenWindows = void 0;
const path_1 = __importDefault(require("path"));
const chalk_1 = __importDefault(require("chalk"));
const perf_hooks_1 = require("perf_hooks");
const telemetry_1 = require("@react-native-windows/telemetry");
const codegen_1 = require("@react-native-windows/codegen");
const commandWithProgress_1 = require("../../utils/commandWithProgress");
const telemetryHelpers_1 = require("../../utils/telemetryHelpers");
const codegenWindowsOptions_1 = require("./codegenWindowsOptions");
class CodeGenWindows {
    areChangesNeeded() {
        return this.changesNecessary;
    }
    constructor(root, options) {
        this.root = root;
        this.options = options;
        this.changesNecessary = false;
    }
    async run(spinner) {
        var _a, _b;
        const verbose = this.options.logging;
        verboseMessage('', verbose);
        verboseMessage('Loading codegenConfig from package.json');
        const pkgJson = require(path_1.default.join(this.root, 'package.json'));
        if (!pkgJson.codegenConfig) {
            spinner.info(`No ${chalk_1.default.bold('codegenConfig')} specified in package.json - ${chalk_1.default.yellow('Skipping codegen-windows')}`);
            return;
        }
        const codegenConfigType = pkgJson.codegenConfig.type;
        if (codegenConfigType !== 'modules' && codegenConfigType !== 'all') {
            spinner.info(`${chalk_1.default.bold('codegenConfig.type')} in package.json is not ${chalk_1.default.bold('modules')} or ${chalk_1.default.bold('all')} - ${chalk_1.default.yellow('Skipping codegen-windows')}`);
            return;
        }
        if (!pkgJson.codegenConfig.windows) {
            spinner.info(`No ${chalk_1.default.bold('codegenConfig.windows')} specified in package.json - ${chalk_1.default.yellow('Skipping codegen-windows')}`);
            return;
        }
        if (!pkgJson.codegenConfig.windows.namespace) {
            throw new telemetry_1.CodedError('InvalidCodegenConfig', `Missing ${chalk_1.default.bold('codegenConfig.windows.namespace')} value in package.json`);
        }
        let cppStringType = 'std::string';
        if (pkgJson.codegenConfig.windows.cppStringType) {
            switch (pkgJson.codegenConfig.windows.cppStringType) {
                case 'std::string':
                case 'std::wstring':
                    cppStringType = pkgJson.codegenConfig.windows.cppStringType;
                    break;
                default:
                    throw new telemetry_1.CodedError('InvalidCodegenConfig', `Value of ${chalk_1.default.bold('codegenConfig.windows.cppStringType')} in package.json should be either 'std::string' or 'std::wstring'`);
            }
        }
        let separateDataTypes = false;
        if (pkgJson.codegenConfig.windows.separateDataTypes !== undefined) {
            switch (pkgJson.codegenConfig.windows.separateDataTypes) {
                case true:
                case false:
                    separateDataTypes = pkgJson.codegenConfig.windows.separateDataTypes;
                    break;
                default:
                    throw new telemetry_1.CodedError('InvalidCodegenConfig', `Value of ${chalk_1.default.bold('codegenConfig.windows.separateDataTypes')} in package.json should be either true or false`);
            }
        }
        if (!pkgJson.codegenConfig.name) {
            throw new telemetry_1.CodedError('InvalidCodegenConfig', `Missing ${chalk_1.default.bold('codegenConfig.name')} value in package.json`);
        }
        const projectName = pkgJson.codegenConfig.name.replace(/[^a-zA-Z]/g, '');
        const projectNamespace = pkgJson.codegenConfig.windows.namespace;
        const jsRootDir = pkgJson.codegenConfig.jsSrcsDir
            ? path_1.default.join(this.root, pkgJson.codegenConfig.jsSrcsDir)
            : this.root;
        const codegenOutputDir = (_a = pkgJson.codegenConfig.windows.outputDirectory) !== null && _a !== void 0 ? _a : 'codegen';
        const generators = (_b = pkgJson.codegenConfig.windows.generators) !== null && _b !== void 0 ? _b : [
            'modulesWindows',
        ];
        const jsRootPathRelative = path_1.default.relative(process.cwd(), jsRootDir);
        const options = {
            files: [
                `${jsRootPathRelative}${jsRootPathRelative ? '/' : ''}**/*Native*.[jt]s`,
            ],
            cppStringType,
            separateDataTypes,
            libraryName: projectName,
            methodOnly: false,
            modulesCxx: generators.indexOf('modulesCxx') !== -1,
            modulesTypeScriptTypes: generators.indexOf('modulesTypeScriptTypes') !== -1,
            modulesWindows: generators.indexOf('modulesWindows') !== -1,
            namespace: projectNamespace,
            outputDirectory: path_1.default.join(this.root, codegenOutputDir),
            test: !!this.options.check,
        };
        verboseMessage(`Run codegen with options: \n${JSON.stringify(options, null, 2)}`, verbose);
        this.changesNecessary = (0, codegen_1.runCodeGen)(options);
        spinner.succeed();
    }
}
exports.CodeGenWindows = CodeGenWindows;
/**
 * Logs the given message if verbose is True.
 * @param message The message to log.
 * @param verbose Whether or not verbose logging is enabled.
 */
function verboseMessage(message, verbose) {
    if (verbose) {
        console.log(message);
    }
}
/**
 * Sanitizes the given option for telemetry.
 * @param key The key of the option.
 * @param value The unsanitized value of the option.
 * @returns The sanitized value of the option.
 */
function optionSanitizer(key, value) {
    // Do not add a default case here.
    // Strings risking PII should just return true if present, false otherwise.
    // All others should return the value (or false if undefined).
    switch (key) {
        case 'logging':
        case 'check':
        case 'telemetry':
            return value === undefined ? false : value; // Return value
    }
}
/**
 * Get the extra props to add to the `codegen-windows` telemetry event.
 * @returns The extra props.
 */
async function getExtraProps() {
    const extraProps = {};
    return extraProps;
}
/**
 * The function run when calling `react-native codegen-windows`.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function codegenWindows(args, config, options) {
    await (0, telemetryHelpers_1.startTelemetrySession)('codegen-windows', config, options, (0, telemetryHelpers_1.getDefaultOptions)(config, codegenWindowsOptions_1.codegenOptions), optionSanitizer);
    let codegenWindowsError;
    try {
        await codegenWindowsInternal(args, config, options);
    }
    catch (ex) {
        codegenWindowsError =
            ex instanceof Error ? ex : new Error(String(ex));
        telemetry_1.Telemetry.trackException(codegenWindowsError);
    }
    await (0, telemetryHelpers_1.endTelemetrySession)(codegenWindowsError, getExtraProps);
    (0, commandWithProgress_1.setExitProcessWithError)(options.logging, codegenWindowsError);
}
/**
 * Performs codegen for RNW native modules and apps.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function codegenWindowsInternal(args, config, options) {
    const startTime = perf_hooks_1.performance.now();
    const spinner = (0, commandWithProgress_1.newSpinner)(options.check
        ? 'Checking codegen-windows files...'
        : 'Running codegen-windows...');
    try {
        const codegen = new CodeGenWindows(config.root, options);
        await codegen.run(spinner);
        const endTime = perf_hooks_1.performance.now();
        if (!codegen.areChangesNeeded()) {
            console.log(`${chalk_1.default.green('Success:')} No codegen-windows changes necessary. (${Math.round(endTime - startTime)}ms)`);
        }
        else if (options.check) {
            const codegenCommand = 'npx react-native codegen-windows';
            console.log(`${chalk_1.default.yellow('Warning:')} Codegen-windows changes were necessary but ${chalk_1.default.bold('--check')} specified. Run '${chalk_1.default.bold(`${codegenCommand}`)}' to apply the changes. (${Math.round(endTime - startTime)}ms)`);
            throw new telemetry_1.CodedError('NeedCodegen', `Codegen-windows changes were necessary but --check was specified. Run '${codegenCommand}' to apply the changes`);
        }
        else {
            console.log(`${chalk_1.default.green('Success:')} Codegen-windows changes completed. (${Math.round(endTime - startTime)}ms)`);
        }
    }
    catch (e) {
        spinner.fail();
        const endTime = perf_hooks_1.performance.now();
        console.log(`${chalk_1.default.red('Error:')} ${e.toString()}. (${Math.round(endTime - startTime)}ms)`);
        throw e;
    }
}
exports.codegenWindowsInternal = codegenWindowsInternal;
/**
 * Performs codegen for RNW native modules.
 */
exports.codegenCommand = {
    name: 'codegen-windows',
    description: 'Runs Windows-specific codegen',
    func: codegenWindows,
    options: codegenWindowsOptions_1.codegenOptions,
};
//# sourceMappingURL=codegenWindows.js.map