"use strict";

// Copyright (c) 2015-present Mattermost, Inc. All Rights Reserved.
// See LICENSE.txt for license information.

import { NativeEventEmitter } from "react-native";
import isURL from "validator/es/lib/isURL";
import { validateAPIClientConfiguration, validateRequestOptions, validateUploadRequestOptions } from "../schemas";
import { ApiClientEvents } from "./NativeApiClient";
const NativeApiClient = require("./NativeApiClient").default;
const Emitter = new NativeEventEmitter(NativeApiClient);
const CLIENTS = {};
const DEFAULT_API_CLIENT_CONFIG = {
  sessionConfiguration: {
    allowsCellularAccess: true,
    waitsForConnectivity: false,
    httpMaximumConnectionsPerHost: 10,
    cancelRequestsOnUnauthorized: false
  }
};
const generateTaskId = () => Math.random().toString(36).slice(-10) + Math.random().toString(36).slice(-10);

/**
 * Configurable client for consuming a REST API
 */
class APIClient {
  constructor(baseUrl, config = {}) {
    this.baseUrl = removeTrailingSlashes(baseUrl);
    this.config = Object.assign({}, DEFAULT_API_CLIENT_CONFIG, config);
    validateAPIClientConfiguration(config);
  }
  onClientError = callback => {
    if (this.onClientErrorSubscription) {
      this.onClientErrorSubscription.remove();
    }
    this.onClientErrorSubscription = Emitter.addListener(ApiClientEvents.CLIENT_ERROR, event => {
      if (event.serverUrl === this.baseUrl && callback) {
        callback(event);
      }
    });
  };
  getHeaders = () => {
    return NativeApiClient.getClientHeadersFor(this.baseUrl);
  };
  addHeaders = headers => {
    this.config.headers = {
      ...this.config.headers,
      ...headers
    };
    return NativeApiClient.addClientHeadersFor(this.baseUrl, headers);
  };
  importClientP12 = (path, password) => {
    return NativeApiClient.importClientP12For(this.baseUrl, path, password);
  };
  invalidate = () => {
    this.onClientErrorSubscription?.remove();
    delete CLIENTS[this.baseUrl];
    return NativeApiClient.invalidateClientFor(this.baseUrl);
  };
  head = (endpoint, options) => {
    validateRequestOptions(options);
    return NativeApiClient.head(this.baseUrl, endpoint);
  };
  get = (endpoint, options) => {
    validateRequestOptions(options);
    return NativeApiClient.get(this.baseUrl, endpoint, options);
  };
  put = (endpoint, options) => {
    validateRequestOptions(options);
    return NativeApiClient.put(this.baseUrl, endpoint, options);
  };
  post = (endpoint, options) => {
    validateRequestOptions(options);
    return NativeApiClient.post(this.baseUrl, endpoint, options);
  };
  patch = (endpoint, options) => {
    validateRequestOptions(options);
    return NativeApiClient.patch(this.baseUrl, endpoint, options);
  };
  delete = (endpoint, options) => {
    validateRequestOptions(options);
    return NativeApiClient.methodDelete(this.baseUrl, endpoint, options);
  };
  upload = (endpoint, fileUrl, options) => {
    validateUploadRequestOptions(options);
    const taskId = generateTaskId();
    const promise = new Promise((resolve, reject) => {
      const uploadSubscription = Emitter.addListener(ApiClientEvents.UPLOAD_PROGRESS, e => {
        if (e.taskId === taskId && promise.onProgress) {
          promise.onProgress(e.fractionCompleted, e.bytesRead);
        }
      });
      NativeApiClient.upload(this.baseUrl, endpoint, fileUrl, taskId, options).then(response => resolve(response)).catch(error => reject(error)).finally(() => {
        uploadSubscription.remove();
        delete promise.progress;
      });
    });
    promise.progress = fn => {
      promise.onProgress = fn;
      return promise;
    };
    promise.cancel = () => NativeApiClient.cancelRequest(taskId);
    return promise;
  };
  download = (endpoint, filePath, options) => {
    validateRequestOptions(options);
    const taskId = generateTaskId();
    const promise = new Promise((resolve, reject) => {
      const downloadSubscription = Emitter.addListener(ApiClientEvents.DOWNLOAD_PROGRESS, e => {
        if (e.taskId === taskId && promise.onProgress) {
          promise.onProgress(e.fractionCompleted, e.bytesRead);
        }
      });
      NativeApiClient.download(this.baseUrl, endpoint, filePath, taskId, options).then(response => resolve(response)).catch(error => reject(error)).finally(() => {
        downloadSubscription.remove();
        delete promise.progress;
      });
    });
    promise.progress = fn => {
      promise.onProgress = fn;
      return promise;
    };
    promise.cancel = () => NativeApiClient.cancelRequest(taskId);
    return promise;
  };
}
async function getOrCreateAPIClient(baseUrl, config = {}, clientErrorEventHandler) {
  if (!isValidBaseURL(baseUrl)) {
    throw new Error(`"${baseUrl}" is not a valid API base URL`);
  }
  let created = false;
  let client = CLIENTS[baseUrl];
  if (!client) {
    client = new APIClient(baseUrl, config);
    if (clientErrorEventHandler) {
      client.onClientError(clientErrorEventHandler);
    }
    await NativeApiClient.createClientFor(client.baseUrl, client.config);
    CLIENTS[baseUrl] = client;
    created = true;
  }
  return {
    client,
    created
  };
}
const isValidBaseURL = baseUrl => {
  return isURL(baseUrl, {
    protocols: ["http", "https"],
    require_protocol: true,
    require_valid_protocol: true,
    require_host: true,
    require_tld: false
  });
};
const removeTrailingSlashes = baseUrl => {
  return baseUrl.replace(/\/+$/, "");
};
export { getOrCreateAPIClient };
//# sourceMappingURL=index.js.map