// Adapted from the type defintions for https://github.com/jgm/commonmark.js provided by https://github.com/DefinitelyTyped/DefinitelyTyped

export type NodeType =
    'text' |'softbreak' | 'linebreak' | 'emph' | 'strong' | 'html_inline' | 'link' | 'image' | 'code' | 'document' | 'paragraph' |
    'block_quote' | 'item' | 'list' | 'heading' | 'code_block' | 'html_block' | 'thematic_break' | 'custom_inline' | 'custom_block' | 
    'at_mention' | 'channel_link' | 'emoji' | 'hashtag' | 'latex_inline' | 'table' | 'table_row' | 'table_cell' | 'mention_highlight' |
    'highlight_without_notification' | 'search_highlight' | 'checkbox' | 'edited_indicator';

export class Node {
    constructor(nodeType: NodeType, sourcepos?: Position);

    /**
     * (read-only): a String, one of text, softbreak, linebreak, emph, strong, html_inline, link, image, code, document, paragraph,
     * block_quote, item, list, heading, code_block, html_block, thematic_break.
     */
    readonly type: NodeType;
    /**
     * (read-only): a Node or null.
     */
    readonly firstChild: Node | null;
    /**
     * (read-only): a Node or null.
     */
    readonly lastChild: Node | null;
    /**
     * (read-only): a Node or null.
     */
    readonly next: Node | null;
    /**
     * (read-only): a Node or null.
     */
    readonly prev: Node | null;
    /**
     * (read-only): a Node or null.
     */
    readonly parent: Node | null;
    /**
     * (read-only): an Array with the following form: [[startline, startcolumn], [endline, endcolumn]]
     */
    readonly sourcepos: Position;
    /**
     * (read-only): true if the Node can contain other Nodes as children.
     */
    readonly isContainer: boolean;
    /**
     *  the literal String content of the node or null.
     */
    literal: string | null;
    /**
     * link or image destination (String) or null.
     */
    destination: string | null;
    /**
     *  link or image title (String) or null.
     */
    title: string | null;
    /**
     * fenced code block info string (String) or null.
     */
    info: string | null;
    /**
     * heading level (Number).
     */
    level: number;
    /**
     * either Bullet or Ordered (or undefined).
     */
    listType: 'bullet' | 'ordered';
    /**
     * true if list is tight
     */
    listTight: boolean;
    /**
     * a Number, the starting number of an ordered list.
     */
    listStart: number;
    /**
     * a String, either ) or . for an ordered list.
     */
    listDelimiter: ')' | '.';
    /**
     * If this is an at-mention, the name of the mentioned user/group. Otherwise, undefined.
     */
    mentionName?: string;
    /**
     * If this is a channel link, the name of the linked channel. Otherwise, undefined.
     */
    channelName?: string;
    /**
     * If this is an emoji, the name of the emoji. Otherwise, undefined.
     */
    emoji?: string;
    /**
     * If this is inline Latex, the Latex content of this Node. Otherwise, undefined.
     */
    latexCode?: string;
    /**
     * If this is a checkbox, whether or not it is checked. Otherwise, false.
     */
    isChecked: boolean;
    /**
     * If this is a table row, an array containing the alignment of each cell
     * in the column. Otherwise, an empty array.
     */
    alignColumns: Array<'' | 'left' | 'center' | 'right'>;
    /**
     * If this is a table row or table cell, whether or not this is the first row
     * of the table. Otherwise, false.
     */
    isHeading: boolean;
    /**
     * If this is a table cell, the alignment of this cell. Otherwise, an empty string.
     */
    align: '' | 'left' | 'center' | 'right';
    /**
     * used only for CustomBlock or CustomInline.
     */
    onEnter: string;
    /**
     * used only for CustomBlock or CustomInline.
     */
    onExit: string;

    /**
     * Append a Node child to the end of the Node's children.
     */
    appendChild(child: Node): void;

    /**
     *  Prepend a Node child to the beginning of the Node's children.
     */
    prependChild(child: Node): void;

    /**
     *  Remove the Node from the tree, severing its links with siblings and parents, and closing up gaps as needed.
     */
    unlink(): void;

    /**
     * Insert a Node sibling after the Node.
     */
    insertAfter(sibling: Node): void;

    /**
     * Insert a Node sibling before the Node.
     */
    insertBefore(sibling: Node): void;

    /**
     * Returns a NodeWalker that can be used to iterate through the Node tree rooted in the Node
     */
    walker(): NodeWalker;
}

/**
 * Instead of converting Markdown directly to HTML, as most converters do, commonmark.js parses Markdown to an AST (abstract syntax tree), and then renders this AST as HTML.
 * This opens up the possibility of manipulating the AST between parsing and rendering. For example, one could transform emphasis into ALL CAPS.
 */
export class Parser {
    /**
     * Constructs a new Parser
     */
    constructor(options?: ParserOptions);

    parse(input: string): Node;
}

export class HtmlRenderer {
    constructor(options?: HtmlRenderingOptions)

    render(root: Node): string;

    /**
     * Let's you override the softbreak properties of a renderer. So, to make soft breaks render as hard breaks in HTML:
     * writer.softbreak = "<br />";
     */
    softbreak: string;
    /**
     * Override the function that will be used to escape (sanitize) the html output. Return value is used to add to the html output
     * @param input the input to escape
     * @param isAttributeValue indicates wheter or not the input value will be used as value of an html attribute.
     */
    escape: (input: string, isAttributeValue: boolean) => string;
}

export class XmlRenderer {
    constructor(options?: XmlRenderingOptions)

    render(root: Node): string;
}
export interface NodeWalkingStep {
    /**
     * a boolean, which is true when we enter a Node from a parent or sibling, and false when we reenter it from a child
     */
    entering: boolean;
    /**
     * The node belonging to this step
     */
    node: Node;
}

export interface NodeWalker {
    /**
     * Returns an object with properties entering and node. Returns null when we have finished walking the tree.
     */
    next(): NodeWalkingStep;
    /**
     * Resets the iterator to resume at the specified node and setting for entering. (Normally this isn't needed unless you do destructive updates to the Node tree.)
     */
    resumeAt(node: Node, entering?: boolean): void;
}

export type Position = [[number, number], [number, number]];

export interface ListData {
    type?: string | undefined;
    tight?: boolean | undefined;
    delimiter?: string | undefined;
    bulletChar?: string | undefined;
}

export interface ParserOptions {
    /**
     *  if true, straight quotes will be made curly, -- will be changed to an en dash, --- will be changed to an em dash, and ... will be changed to ellipses.
     */
    smart?: boolean | undefined;
    time?: boolean | undefined;

    /**
     * A filter function for URLs allowed during autolinking. Can be used to disallow certain things from being
     * matched as URLs based on, for example, the protocol used.
     */
    urlFilter?: (url: string) => boolean;
    /**
     * The minimum number of characters in a valid hashtag. If not provided, defaults to 3.
     */
    minimumHashtagLength?: number;
}

export interface HtmlRenderingOptions extends XmlRenderingOptions {
    /**
     *  if true, raw HTML will not be passed through to HTML output (it will be replaced by comments), and potentially unsafe URLs in links and images
     *  (those beginning with javascript:, vbscript:, file:, and with a few exceptions data:) will be replaced with empty strings.
     */
    safe?: boolean | undefined;
    /**
     *  if true, straight quotes will be made curly, -- will be changed to an en dash, --- will be changed to an em dash, and ... will be changed to ellipses.
     */
    smart?: boolean | undefined;
    /**
     *  if true, source position information for block-level elements will be rendered in the data-sourcepos attribute (for HTML) or the sourcepos attribute (for XML).
     */
    sourcepos?: boolean | undefined;

    /**
     * A raw string to be used for a softbreak.
     * For example, `{ softbreak: "<br/>" }` treats a softbreak as `<br/>`.
     */
    softbreak?: string | undefined;
}

export interface XmlRenderingOptions {
    time?: boolean | undefined;
    sourcepos?: boolean | undefined;
}
