'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var logger$1 = require('@azure/logger');
var instrumentation = require('@opentelemetry/instrumentation');
var api = require('@opentelemetry/api');
var core = require('@opentelemetry/core');

// Copyright (c) Microsoft Corporation.
/**
 * The \@azure/logger configuration for this package.
 */
const logger = logger$1.createClientLogger("opentelemetry-instrumentation-azure-sdk");

// Copyright (c) Microsoft Corporation.
class OpenTelemetrySpanWrapper {
    constructor(span) {
        this._span = span;
    }
    setStatus(status) {
        if (status.status === "error") {
            if (status.error) {
                this._span.setStatus({ code: api.SpanStatusCode.ERROR, message: status.error.toString() });
                this.recordException(status.error);
            }
            else {
                this._span.setStatus({ code: api.SpanStatusCode.ERROR });
            }
        }
        else if (status.status === "success") {
            this._span.setStatus({ code: api.SpanStatusCode.OK });
        }
    }
    setAttribute(name, value) {
        if (value !== null && value !== undefined) {
            this._span.setAttribute(name, value);
        }
    }
    end() {
        this._span.end();
    }
    recordException(exception) {
        this._span.recordException(exception);
    }
    isRecording() {
        return this._span.isRecording();
    }
    /**
     * Allows getting the wrapped span as needed.
     * @internal
     *
     * @returns The underlying span
     */
    unwrap() {
        return this._span;
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
const SDK_VERSION = "1.0.0-beta.5";
/**
 * @internal
 *
 * Cached values of environment variables that were fetched.
 */
const environmentCache = new Map();
/**
 * Converts an environment variable to Boolean.
 * the strings "false" and "0" are treated as falsy values.
 *
 * @internal
 */
function envVarToBoolean(key) {
    var _a;
    if (!environmentCache.has(key)) {
        loadEnvironmentVariable(key);
    }
    const value = ((_a = environmentCache.get(key)) !== null && _a !== void 0 ? _a : "").toLowerCase();
    return value !== "false" && value !== "0" && Boolean(value);
}
function loadEnvironmentVariable(key) {
    var _a;
    if (typeof process !== "undefined" && process.env) {
        const rawValue = (_a = process.env[key]) !== null && _a !== void 0 ? _a : process.env[key.toLowerCase()];
        environmentCache.set(key, rawValue);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * Converts our TracingSpanKind to the corresponding OpenTelemetry SpanKind.
 *
 * By default it will return {@link SpanKind.INTERNAL}
 * @param tracingSpanKind - The core tracing {@link TracingSpanKind}
 * @returns - The OpenTelemetry {@link SpanKind}
 */
function toOpenTelemetrySpanKind(tracingSpanKind) {
    const key = (tracingSpanKind || "internal").toUpperCase();
    return api.SpanKind[key];
}
/**
 * Converts core-tracing's TracingSpanLink to OpenTelemetry's Link
 *
 * @param spanLinks - The core tracing {@link TracingSpanLink} to convert
 * @returns A set of {@link Link}s
 */
function toOpenTelemetryLinks(spanLinks = []) {
    return spanLinks.reduce((acc, tracingSpanLink) => {
        const spanContext = api.trace.getSpanContext(tracingSpanLink.tracingContext);
        if (spanContext) {
            acc.push({
                context: spanContext,
                attributes: toOpenTelemetrySpanAttributes(tracingSpanLink.attributes),
            });
        }
        return acc;
    }, []);
}
/**
 * Converts core-tracing's span attributes to OpenTelemetry attributes.
 *
 * @param spanAttributes - The set of attributes to convert.
 * @returns An {@link SpanAttributes} to set on a span.
 */
function toOpenTelemetrySpanAttributes(spanAttributes) {
    const attributes = {};
    for (const key in spanAttributes) {
        // Any non-nullish value is allowed.
        if (spanAttributes[key] !== null && spanAttributes[key] !== undefined) {
            attributes[key] = spanAttributes[key];
        }
    }
    return attributes;
}
/**
 * Converts core-tracing span options to OpenTelemetry options.
 *
 * @param spanOptions - The {@link InstrumenterSpanOptions} to convert.
 * @returns An OpenTelemetry {@link SpanOptions} that can be used when creating a span.
 */
function toSpanOptions(spanOptions) {
    const { spanAttributes, spanLinks, spanKind } = spanOptions || {};
    const attributes = toOpenTelemetrySpanAttributes(spanAttributes);
    const kind = toOpenTelemetrySpanKind(spanKind);
    const links = toOpenTelemetryLinks(spanLinks);
    return {
        attributes,
        kind,
        links,
    };
}

// Copyright (c) Microsoft Corporation.
// While default propagation is user-configurable, Azure services always use the W3C implementation.
const propagator = new core.W3CTraceContextPropagator();
class OpenTelemetryInstrumenter {
    startSpan(name, spanOptions) {
        let ctx = (spanOptions === null || spanOptions === void 0 ? void 0 : spanOptions.tracingContext) || api.context.active();
        let span;
        if (envVarToBoolean("AZURE_TRACING_DISABLED")) {
            // disable only our spans but not any downstream spans
            span = api.trace.wrapSpanContext(api.INVALID_SPAN_CONTEXT);
        }
        else {
            // Create our span
            span = api.trace
                .getTracer(spanOptions.packageName, spanOptions.packageVersion)
                .startSpan(name, toSpanOptions(spanOptions), ctx);
            if (envVarToBoolean("AZURE_HTTP_TRACING_CHILDREN_DISABLED") &&
                name.toUpperCase().startsWith("HTTP")) {
                // disable downstream spans
                ctx = core.suppressTracing(ctx);
            }
        }
        return {
            span: new OpenTelemetrySpanWrapper(span),
            tracingContext: api.trace.setSpan(ctx, span),
        };
    }
    withContext(tracingContext, callback, ...callbackArgs) {
        return api.context.with(tracingContext, callback, 
        /** Assume caller will bind `this` or use arrow functions */ undefined, ...callbackArgs);
    }
    parseTraceparentHeader(traceparentHeader) {
        return propagator.extract(api.context.active(), { traceparent: traceparentHeader }, api.defaultTextMapGetter);
    }
    createRequestHeaders(tracingContext) {
        const headers = {};
        propagator.inject(tracingContext || api.context.active(), headers, api.defaultTextMapSetter);
        return headers;
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * The instrumentation module for the Azure SDK. Implements OpenTelemetry's {@link Instrumentation}.
 */
class AzureSdkInstrumentation extends instrumentation.InstrumentationBase {
    constructor(options = {}) {
        super("@azure/opentelemetry-instrumentation-azure-sdk", SDK_VERSION, Object.assign({}, options));
    }
    /**
     * Entrypoint for the module registration.
     *
     * @returns The patched \@azure/core-tracing module after setting its instrumenter.
     */
    init() {
        const result = new instrumentation.InstrumentationNodeModuleDefinition("@azure/core-tracing", ["^1.0.0-preview.14", "^1.0.0"], (moduleExports) => {
            if (typeof moduleExports.useInstrumenter === "function") {
                moduleExports.useInstrumenter(new OpenTelemetryInstrumenter());
            }
            return moduleExports;
        });
        // Needed to support 1.0.0-preview.14
        result.includePrerelease = true;
        return result;
    }
}
/**
 * Enables Azure SDK Instrumentation using OpenTelemetry for Azure SDK client libraries.
 *
 * When registered, any Azure data plane package will begin emitting tracing spans for internal calls
 * as well as network calls
 *
 * Example usage:
 * ```ts
 * const openTelemetryInstrumentation = require("@opentelemetry/instrumentation");
 * openTelemetryInstrumentation.registerInstrumentations({
 *   instrumentations: [createAzureSdkInstrumentation()],
 * })
 * ```
 *
 * @remarks
 *
 * As OpenTelemetry instrumentations rely on patching required modules, you should register
 * this instrumentation as early as possible and before loading any Azure Client Libraries.
 */
function createAzureSdkInstrumentation(options = {}) {
    return new AzureSdkInstrumentation(options);
}

exports.createAzureSdkInstrumentation = createAzureSdkInstrumentation;
exports.logger = logger;
//# sourceMappingURL=index.js.map
