/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
import * as appInsights from 'applicationinsights';
import * as errorUtils from './utils/errorUtils';
import * as projectUtils from './utils/projectUtils';
export interface TelemetryOptions {
    setupString: string;
    preserveErrorMessages: boolean;
    populateNpmPackageVersions: boolean;
}
export interface CommandStartInfo {
    commandName: string;
    args: Record<string, any>;
    options: Record<string, any>;
    defaultOptions: Record<string, any>;
}
export interface CommandEndInfo {
    resultCode: errorUtils.CodedErrorType;
}
interface CommandInfo {
    startTime?: number;
    endTime?: number;
    startInfo?: CommandStartInfo;
    endInfo?: CommandEndInfo;
}
export declare const CommandEventName = "RNWCLI.Command";
export declare const CodedErrorEventName = "RNWCLI.CodedError";
export declare const EventNamesWeTrack: string[];
export declare const NpmPackagesWeTrack: string[];
export declare const NuGetPackagesWeTrack: string[];
/**
 * The Telemetry class is responsible for reporting telemetry for RNW CLI.
 */
export declare class Telemetry {
    protected static client?: appInsights.TelemetryClient;
    protected static options: TelemetryOptions;
    protected static isTest: boolean;
    protected static commandInfo: CommandInfo;
    protected static versionsProp: Record<string, string>;
    protected static projectProp?: projectUtils.AppProjectInfo | projectUtils.DependencyProjectInfo;
    protected static getDefaultSetupString(): string;
    protected static reset(): void;
    static isEnabled(): boolean;
    static getSessionId(): string;
    /** Sets up the Telemetry static to be used elsewhere. */
    static setup(options?: Partial<TelemetryOptions>): Promise<void>;
    /** Sets up Telemetry.client. */
    private static setupClient;
    /** Sets up any base properties that all telemetry events require. */
    private static setupBaseProperties;
    /** Sets up any telemetry processors. */
    private static setupTelemetryProcessors;
    /**
     * Performs the processing necessary (mostly PII sanitization) for all events.
     * @param envelope The ApplicationInsights event envelope.
     * @param _contextObjects An optional context object.
     * @returns Whether to kee
     */
    private static basicTelemetryProcessor;
    /**
     * Performs the processing necessary (mostly PII sanitization) for error events.
     * @param envelope
     * @param _contextObjects
     * @returns
     */
    private static errorTelemetryProcessor;
    /** Tries to update the version of the named package/tool by calling getValue(). */
    static tryUpdateVersionsProp(name: string, getValue: () => Promise<string | null>, forceRefresh?: boolean): Promise<boolean>;
    /** Populates the versions property of tools we care to track. */
    static populateToolsVersions(refresh?: boolean): Promise<void>;
    /** Populates the versions property of npm packages we care to track. */
    static populateNpmPackageVersions(refresh?: boolean): Promise<void>;
    /** Populates the versions property of nuget packages we care to track. */
    static populateNuGetPackageVersions(projectFile: string, refresh?: boolean): Promise<void>;
    static setProjectInfo(info: projectUtils.AppProjectInfo | projectUtils.DependencyProjectInfo): void;
    static startCommand(info: CommandStartInfo): void;
    static endCommand(info: CommandEndInfo, extraProps?: Record<string, any>): void;
    private static trackCommandEvent;
    static trackException(error: Error, extraProps?: Record<string, any>): void;
}
export {};
