/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.translateArgs = exports.translateSpecArgs = void 0;
const AliasManaging_1 = require("./AliasManaging");
function decorateType(type, target) {
    switch (target) {
        case 'method-arg':
            return `${type} &&`;
        case 'callback-arg':
            return `${type} const &`;
        default:
            return type;
    }
}
function translateUnionReturnType(type, target, options) {
    const memberType = type.memberType;
    switch (type.memberType) {
        case 'StringTypeAnnotation':
            return options.cppStringType;
        case 'NumberTypeAnnotation':
            return 'double';
        case 'ObjectTypeAnnotation':
            return decorateType('::React::JSValue', target);
        default:
            throw new Error(`Unknown enum/union member type in translateReturnType: ${memberType}`);
    }
}
function translateFunction(param, aliases, baseAliasName, target, options) {
    // TODO: type.returnTypeAnnotation
    switch (target) {
        case 'spec':
            return `Callback<${param.params
                .map((p) => translateSpecFunctionParam(p, aliases, `${baseAliasName}_${p.name}`, options))
                .join(', ')}>`;
        case 'template':
            return `std::function<void(${param.params
                .map((p) => translateCallbackParam(p, aliases, `${baseAliasName}_${p.name}`, options))
                .join(', ')})>`;
        default:
            return `std::function<void(${param.params
                .map((p) => translateCallbackParam(p, aliases, `${baseAliasName}_${p.name}`, options))
                .join(', ')})> const &`;
    }
}
function translateArray(param, aliases, baseAliasName, target, options) {
    if (param.elementType) {
        switch (target) {
            case 'spec':
            case 'template':
                return `std::vector<${translateNullableParamType(param.elementType, aliases, `${baseAliasName}_element`, 'template', 'template', options)}>`;
            default:
                return `std::vector<${translateNullableParamType(param.elementType, aliases, `${baseAliasName}_element`, 'template', 'template', options)}> const &`;
        }
    }
    else {
        return decorateType('::React::JSValueArray', target);
    }
}
function translateParam(param, aliases, baseAliasName, target, options) {
    // avoid: Property 'type' does not exist on type 'never'
    const paramType = param.type;
    switch (param.type) {
        case 'StringTypeAnnotation':
            return options.cppStringType;
        case 'NumberTypeAnnotation':
        case 'FloatTypeAnnotation':
        case 'DoubleTypeAnnotation':
            return 'double';
        case 'Int32TypeAnnotation':
            return 'int';
        case 'BooleanTypeAnnotation':
            return 'bool';
        case 'FunctionTypeAnnotation':
            return translateFunction(param, aliases, baseAliasName, target, options);
        case 'ArrayTypeAnnotation':
            return translateArray(param, aliases, baseAliasName, target, options);
        case 'GenericObjectTypeAnnotation':
            return decorateType('::React::JSValue', target);
        case 'ObjectTypeAnnotation':
            return decorateType((0, AliasManaging_1.getAnonymousAliasCppName)(aliases, baseAliasName, param), target);
        case 'ReservedTypeAnnotation': {
            // avoid: Property 'name' does not exist on type 'never'
            const name = param.name;
            // (#6597)
            // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
            if (name !== 'RootTag')
                throw new Error(`Unknown reserved function: ${name} in translateParam`);
            return 'double';
        }
        case 'TypeAliasTypeAnnotation':
            return decorateType((0, AliasManaging_1.getAliasCppName)(param.name), target);
        case 'MixedTypeAnnotation':
            return '';
        case 'EnumDeclaration':
        case 'UnionTypeAnnotation':
            return translateUnionReturnType(param, target, options);
        default:
            throw new Error(`Unhandled type in translateParam: ${paramType}`);
    }
}
function translateNullableParamType(paramType, aliases, baseAliasName, nullableTarget, target, options) {
    switch (paramType.type) {
        case 'NullableTypeAnnotation':
            return `std::optional<${translateParam(paramType.typeAnnotation, aliases, baseAliasName, nullableTarget, options)}>`;
        default:
            return translateParam(paramType, aliases, baseAliasName, target, options);
    }
}
function translateSpecFunctionParam(param, aliases, baseAliasName, options) {
    return translateNullableParamType(param.typeAnnotation, aliases, baseAliasName, 'spec', 'spec', options);
}
function translateCallbackParam(param, aliases, baseAliasName, options) {
    return translateNullableParamType(param.typeAnnotation, aliases, baseAliasName, 'template', 'callback-arg', options);
}
function translateFunctionParam(param, aliases, baseAliasName, options) {
    return translateNullableParamType(param.typeAnnotation, aliases, baseAliasName, 'template', 'method-arg', options);
}
function translateSpecArgs(params, aliases, baseAliasName, options) {
    return params.map(param => {
        const translatedParam = translateSpecFunctionParam(param, aliases, `${baseAliasName}_${param.name}`, options);
        return `${translatedParam}`;
    });
}
exports.translateSpecArgs = translateSpecArgs;
function translateArgs(params, aliases, baseAliasName, options) {
    return params.map(param => {
        const translatedParam = translateFunctionParam(param, aliases, `${baseAliasName}_${param.name}`, options);
        return `${translatedParam} ${param.name}`;
    });
}
exports.translateArgs = translateArgs;
//# sourceMappingURL=ParamTypes.js.map