"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.findLatestVsInstall = exports.enumerateVsInstalls = void 0;
const telemetry_1 = require("@react-native-windows/telemetry");
const child_process_1 = require("child_process");
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const semver_1 = __importDefault(require("semver"));
/**
 * Helper to run vswhere in JSON mode
 *
 * @param args Arguments to pass to vsWhere
 * @param verbose enable verbose logging
 */
function vsWhere(args, verbose) {
    // This path is maintained and VS has promised to keep it valid.
    const vsWherePath = path_1.default.join(process.env['ProgramFiles(x86)'] || process.env.ProgramFiles, '/Microsoft Visual Studio/Installer/vswhere.exe');
    if (verbose) {
        console.log('Looking for vswhere at: ' + vsWherePath);
    }
    if (!fs_1.default.existsSync(vsWherePath)) {
        throw new telemetry_1.CodedError('NoVSWhere', `Unable to find vswhere at ${vsWherePath}`);
    }
    const system32 = `${process.env.SystemRoot}\\System32`;
    const cmdline = `${system32}\\cmd.exe /c ${system32}\\chcp.com 65001>nul && "${vsWherePath}" ${args.join(' ')} -format json -utf8`;
    const json = JSON.parse((0, child_process_1.execSync)(cmdline).toString());
    for (const entry of json) {
        entry.prerelease = entry.catalog.productMilestoneIsPreRelease;
    }
    return json;
}
/**
 * Enumerate the installed versions of Visual Studio
 */
function enumerateVsInstalls(opts) {
    const args = [];
    if (opts.minVersion) {
        // VS 2019 ex: minVersion == 16.7 => [16.7,17.0)
        // VS 2022 ex: minVersion == 17.0 => [17.0,18.0)
        // Try to parse minVersion as both a Number and SemVer
        const minVersionNum = Number(opts.minVersion);
        const minVersionSemVer = semver_1.default.parse(opts.minVersion);
        let minVersion;
        let maxVersion;
        if (minVersionSemVer) {
            minVersion = minVersionSemVer.toString();
            maxVersion = (minVersionSemVer.major + 1).toFixed(1);
        }
        else if (!Number.isNaN(minVersionNum)) {
            minVersion = minVersionNum.toFixed(1);
            maxVersion = (Math.floor(minVersionNum) + 1).toFixed(1);
        }
        else {
            // Unable to parse minVersion and determine maxVersion,
            // caller will throw error that version couldn't be found.
            return [];
        }
        const versionRange = `[${minVersion},${maxVersion}` + (opts.prerelease ? ']' : ')');
        if (opts.verbose) {
            console.log(`Looking for VS installs with version range: ${versionRange}`);
        }
        args.push(`-version ${versionRange}`);
    }
    if (opts.requires) {
        args.push(`-requires ${opts.requires.join(' ')}`);
    }
    if (opts.latest) {
        args.push('-latest');
    }
    if (opts.prerelease) {
        args.push('-prerelease');
    }
    return vsWhere(args, opts.verbose);
}
exports.enumerateVsInstalls = enumerateVsInstalls;
/**
 * Find the latest available VS installation that matches the given constraints
 */
function findLatestVsInstall(opts) {
    let installs = enumerateVsInstalls({ ...opts, latest: true });
    if (opts.prerelease && installs.length > 0) {
        installs = installs.filter(x => x.prerelease === 'True');
    }
    if (installs.length > 0) {
        return installs[0];
    }
    else {
        return null;
    }
}
exports.findLatestVsInstall = findLatestVsInstall;
//# sourceMappingURL=vsInstalls.js.map