"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.startServerInNewWindow = exports.deployToDesktop = exports.deployToDevice = exports.getBuildConfiguration = void 0;
const child_process_1 = require("child_process");
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const http_1 = __importDefault(require("http"));
const path_1 = __importDefault(require("path"));
const glob_1 = __importDefault(require("glob"));
const xml_parser_1 = __importDefault(require("xml-parser"));
const winappdeploytool_1 = __importDefault(require("./winappdeploytool"));
const commandWithProgress_1 = require("./commandWithProgress");
const build = __importStar(require("./build"));
const configUtils = __importStar(require("../commands/config/configUtils"));
const telemetry_1 = require("@react-native-windows/telemetry");
const version_1 = __importDefault(require("./version"));
function pushd(pathArg) {
    const cwd = process.cwd();
    process.chdir(pathArg);
    return () => process.chdir(cwd);
}
function getBuildConfiguration(options) {
    return options.release
        ? options.bundle
            ? 'ReleaseBundle'
            : 'Release'
        : options.bundle
            ? 'DebugBundle'
            : 'Debug';
}
exports.getBuildConfiguration = getBuildConfiguration;
function shouldDeployByPackage(options, config) {
    if (options.deployFromLayout) {
        // Force deploy by layout
        return false;
    }
    let hasAppxSigningEnabled = null;
    let hasPackageCertificateKeyFile = null;
    // TODO: These two properties should really be determined by
    // getting the actual values msbuild used during the build,
    // but for now we'll try to get them manually
    // Check passed in msbuild property overrides
    if (options.msbuildprops) {
        const msbuildprops = build.parseMsBuildProps(options);
        if ('AppxSigningEnabled' in msbuildprops) {
            hasAppxSigningEnabled =
                msbuildprops.AppxSigningEnabled.toLowerCase() === 'true';
        }
        if ('PackageCertificateKeyFile' in msbuildprops) {
            hasPackageCertificateKeyFile = true;
        }
    }
    // If at least one override wasn't set, we need to parse the project file
    if (hasAppxSigningEnabled === null || hasPackageCertificateKeyFile === null) {
        const projectFile = build.getAppProjectFile(options, config);
        if (projectFile) {
            const projectContents = configUtils.readProjectFile(projectFile);
            // Find AppxSigningEnabled
            if (hasAppxSigningEnabled === null) {
                const appxSigningEnabled = configUtils.tryFindPropertyValue(projectContents, 'AppxSigningEnabled');
                if (appxSigningEnabled !== null) {
                    hasAppxSigningEnabled = appxSigningEnabled.toLowerCase() === 'true';
                }
            }
            // Find PackageCertificateKeyFile
            if (hasPackageCertificateKeyFile === null) {
                const packageCertificateKeyFile = configUtils.tryFindPropertyValue(projectContents, 'PackageCertificateKeyFile');
                if (packageCertificateKeyFile !== null) {
                    hasPackageCertificateKeyFile = true;
                }
            }
        }
    }
    return (hasAppxSigningEnabled === true && hasPackageCertificateKeyFile === true);
}
function shouldLaunchApp(options) {
    return options.launch === true;
}
function getAppPackage(options, projectName) {
    const configuration = getBuildConfiguration(options);
    const packageFolder = options.arch === 'x86'
        ? `{*_x86_${configuration}_*,*_Win32_${configuration}_*}`
        : `*_${options.arch}_${configuration}_*`;
    const appPackageGlob = `${options.root}/windows/{*/AppPackages,AppPackages/*}/${packageFolder}`;
    const appPackageCandidates = glob_1.default.sync(appPackageGlob);
    let appPackage;
    if (appPackageCandidates.length === 1 || !projectName) {
        appPackage = appPackageCandidates[0];
    }
    else if (appPackageCandidates.length > 1) {
        const filteredAppPackageCandidates = appPackageCandidates.filter(x => x.includes(projectName));
        if (filteredAppPackageCandidates.length >= 1) {
            appPackage = filteredAppPackageCandidates[0];
        }
    }
    if (!appPackage && options.release) {
        // in the latest vs, Release is removed
        (0, commandWithProgress_1.newWarn)('No package found in *_Release_* folder, removing the _Release_ prefix and checking again');
        const rootGlob = `${options.root}/windows/{*/AppPackages,AppPackages/*}`;
        const newGlob = `${rootGlob}/*_${options.arch === 'x86' ? '{Win32,x86}' : options.arch}_Test`;
        const result = glob_1.default.sync(newGlob);
        if (result.length > 1 && projectName) {
            const newFilteredGlobs = result.filter(x => x.includes(projectName));
            if (newFilteredGlobs.length >= 1) {
                (0, commandWithProgress_1.newWarn)(`More than one app package found: ${result}`);
            }
            appPackage = newFilteredGlobs[0];
        }
        else if (result.length === 1) {
            // we're good
            appPackage = result[0];
        }
    }
    if (!appPackage) {
        throw new telemetry_1.CodedError('NoAppPackage', `Unable to find app package using search path: "${appPackageGlob}"`);
    }
    return appPackage;
}
function getWindowsStoreAppUtils(options) {
    const popd = pushd(options.root);
    const windowsStoreAppUtilsPath = path_1.default.resolve(__dirname, '..', '..', 'src', 'powershell', 'WindowsStoreAppUtils.ps1');
    (0, child_process_1.execSync)(`${commandWithProgress_1.powershell} -NoProfile Unblock-File '${windowsStoreAppUtilsPath}'`);
    popd();
    return windowsStoreAppUtilsPath;
}
function getAppxManifestPath(options, projectName) {
    const configuration = getBuildConfiguration(options);
    // C++ x86 manifest would go under windows/Debug whereas x64 goes under windows/x64/Debug
    // If we've built both, this causes us to end up with two matches, so we have to carefully select the right folder
    let archFolder;
    if (options.arch !== 'x86') {
        archFolder = `${options.arch}/${configuration}`;
    }
    else {
        archFolder = `${configuration}`;
    }
    const appxManifestGlob = `windows/{*/bin/${options.arch}/${configuration},${archFolder}/*,target/${options.arch}/${configuration}}/AppxManifest.xml`;
    const globs = glob_1.default.sync(path_1.default.join(options.root, appxManifestGlob));
    let appxPath;
    if (globs.length === 1 || !projectName) {
        appxPath = globs[0];
    }
    else {
        const filteredGlobs = globs.filter(x => x.includes(projectName));
        appxPath = filteredGlobs[0];
        if (filteredGlobs.length > 1) {
            (0, commandWithProgress_1.newWarn)(`More than one appxmanifest for ${projectName}: ${filteredGlobs.join(',')}`);
            (0, commandWithProgress_1.newWarn)(`Choosing ${appxPath}`);
        }
    }
    if (!appxPath) {
        throw new telemetry_1.CodedError('NoAppxManifest', `Unable to find AppxManifest from "${options.root}", using search path: "${appxManifestGlob}" `);
    }
    return appxPath;
}
function getAppxRecipePath(options, projectName) {
    const appxManifestPath = path_1.default.dirname(getAppxManifestPath(options, projectName));
    const appxRecipeGlob = `*.build.appxrecipe`;
    const globs = glob_1.default.sync(path_1.default.join(appxManifestPath, appxRecipeGlob));
    let appxRecipePath;
    if (globs.length === 1 || !projectName) {
        appxRecipePath = globs[0];
    }
    else {
        const filteredGlobs = globs.filter(x => x.includes(projectName));
        appxRecipePath = filteredGlobs[0];
        if (filteredGlobs.length > 1) {
            (0, commandWithProgress_1.newWarn)(`More than one appxrecipe for ${projectName}: ${filteredGlobs.join(',')}`);
            (0, commandWithProgress_1.newWarn)(`Choosing ${appxRecipePath}`);
        }
    }
    if (!appxRecipePath) {
        throw new telemetry_1.CodedError('DeployRecipeFailure', `Unable to find AppxRecipe from "${appxManifestPath}", using search path: "${appxRecipeGlob}" `);
    }
    return appxRecipePath;
}
function parseAppxManifest(appxManifestPath) {
    return (0, xml_parser_1.default)(fs_1.default.readFileSync(appxManifestPath, 'utf8'));
}
function getAppxManifest(options, projectName) {
    return parseAppxManifest(getAppxManifestPath(options, projectName));
}
function handleResponseError(e) {
    if (e.message.indexOf('Error code -2146233088')) {
        throw new telemetry_1.CodedError('NoDevice', `No Windows Mobile device was detected: ${e.message}`);
    }
    else {
        throw new telemetry_1.CodedError('AppDidNotDeploy', `Unexpected error deploying app: ${e.message}`);
    }
}
// Errors: 0x80073d10 - bad architecture
async function deployToDevice(options, verbose, config) {
    const windowsConfig = config.project.windows;
    const projectName = windowsConfig && windowsConfig.project && windowsConfig.project.projectName
        ? windowsConfig.project.projectName
        : path_1.default.parse(options.proj).name;
    const appPackageFolder = getAppPackage(options);
    const deployTarget = options.target
        ? options.target
        : options.emulator
            ? 'emulator'
            : 'device';
    const deployTool = new winappdeploytool_1.default();
    const appxManifest = getAppxManifest(options, projectName);
    const shouldLaunch = shouldLaunchApp(options);
    const identity = appxManifest.root.children.filter(x => {
        return x.name === 'mp:PhoneIdentity';
    })[0];
    const appName = identity.attributes.PhoneProductId;
    const device = deployTool.findDevice(deployTarget);
    try {
        await deployTool.uninstallAppPackage(appName, device, verbose);
    }
    catch (e) {
        (0, commandWithProgress_1.newWarn)('Failed to uninstall app from ' + device.name);
    }
    const appxFile = glob_1.default.sync(path_1.default.join(appPackageFolder, '*.appx'))[0];
    try {
        await deployTool.installAppPackage(appxFile, device, shouldLaunch, false, verbose);
    }
    catch (e) {
        if (e.message.includes('Error code 2148734208 for command')) {
            await deployTool.installAppPackage(appxFile, device, shouldLaunch, true, verbose);
        }
        else {
            handleResponseError(e);
        }
    }
}
exports.deployToDevice = deployToDevice;
async function deployToDesktop(options, verbose, config, buildTools) {
    const windowsConfig = config.project.windows;
    const slnFile = windowsConfig && windowsConfig.solutionFile && windowsConfig.sourceDir
        ? path_1.default.join(windowsConfig.sourceDir, windowsConfig.solutionFile)
        : options.sln;
    const projectName = windowsConfig && windowsConfig.project && windowsConfig.project.projectName
        ? windowsConfig.project.projectName
        : path_1.default.parse(options.proj).name;
    const windowsStoreAppUtils = getWindowsStoreAppUtils(options);
    const appxManifestPath = getAppxManifestPath(options, projectName);
    const appxManifest = parseAppxManifest(appxManifestPath);
    const identity = appxManifest.root.children.filter(x => {
        return x.name === 'Identity';
    })[0];
    const appName = identity.attributes.Name;
    const vsVersion = version_1.default.fromString(buildTools.installationVersion);
    const args = [];
    if (options.remoteDebugging) {
        args.push('--remote-debugging');
    }
    if (options.directDebugging) {
        args.push('--direct-debugging', options.directDebugging.toString());
    }
    await (0, commandWithProgress_1.runPowerShellScriptFunction)('Enabling Developer Mode', windowsStoreAppUtils, 'EnableDevMode', verbose, 'EnableDevModeFailure');
    const appPackageFolder = getAppPackage(options, projectName);
    if (shouldDeployByPackage(options, config)) {
        // Deploy by package
        await (0, commandWithProgress_1.runPowerShellScriptFunction)('Removing old version of the app', windowsStoreAppUtils, `Uninstall-App ${appName}`, verbose, 'RemoveOldAppVersionFailure');
        const script = glob_1.default.sync(path_1.default.join(appPackageFolder, 'Add-AppDevPackage.ps1'))[0];
        await (0, commandWithProgress_1.runPowerShellScriptFunction)('Installing new version of the app', windowsStoreAppUtils, `Install-App "${script}" -Force`, verbose, 'InstallAppFailure');
    }
    else {
        // Deploy from layout
        // If we have DeployAppRecipe.exe, use it (start in 16.8.4, earlier 16.8 versions have bugs)
        const appxRecipe = getAppxRecipePath(options, projectName);
        const ideFolder = `${buildTools.installationPath}\\Common7\\IDE`;
        const deployAppxRecipeExePath = `${ideFolder}\\DeployAppRecipe.exe`;
        if (vsVersion.gte(version_1.default.fromString('16.8.30906.45')) &&
            fs_1.default.existsSync(deployAppxRecipeExePath)) {
            await (0, commandWithProgress_1.commandWithProgress)((0, commandWithProgress_1.newSpinner)('Deploying'), `Deploying ${appxRecipe}`, deployAppxRecipeExePath, [appxRecipe], verbose, 'DeployRecipeFailure');
        }
        else {
            // Install the app package's dependencies before attempting to deploy.
            await (0, commandWithProgress_1.runPowerShellScriptFunction)('Installing dependent framework packages', windowsStoreAppUtils, `Install-AppDependencies ${appxManifestPath} ${appPackageFolder} ${options.arch}`, verbose, 'InstallAppDependenciesFailure');
            await build.buildSolution(buildTools, slnFile, 
            /* options.release ? 'Release' : */ 'Debug', options.arch, { DeployLayout: 'true' }, verbose, 'deploy', options.buildLogDirectory);
        }
    }
    const appFamilyName = (0, child_process_1.execSync)(`${commandWithProgress_1.powershell} -NoProfile -c $(Get-AppxPackage -Name ${appName}).PackageFamilyName`)
        .toString()
        .trim();
    if (!appFamilyName) {
        throw new telemetry_1.CodedError('AppDidNotDeploy', 'Fail to check the installed app, maybe developer mode is off on Windows');
    }
    const loopbackText = 'Verifying loopbackExempt';
    const loopbackSpinner = (0, commandWithProgress_1.newSpinner)(loopbackText);
    await (0, commandWithProgress_1.commandWithProgress)(loopbackSpinner, loopbackText, 'CheckNetIsolation', `LoopbackExempt -a -n=${appFamilyName}`.split(' '), verbose, 'CheckNetIsolationFailure');
    if (shouldLaunchApp(options)) {
        await (0, commandWithProgress_1.runPowerShellScriptFunction)('Starting the app', windowsStoreAppUtils, `Start-Locally ${appName} ${args}`, verbose, 'AppStartupFailure');
    }
    else {
        (0, commandWithProgress_1.newInfo)('Skip the step to start the app');
    }
}
exports.deployToDesktop = deployToDesktop;
function startServerInNewWindow(options, verbose) {
    return new Promise(resolve => {
        http_1.default
            .get('http://localhost:8081/status', res => {
            if (res.statusCode === 200) {
                (0, commandWithProgress_1.newSuccess)('React-Native Server already started');
            }
            else {
                (0, commandWithProgress_1.newError)('React-Native Server not responding');
            }
            resolve();
        })
            .on('error', () => {
            launchServer(options, verbose);
            resolve();
        });
    });
}
exports.startServerInNewWindow = startServerInNewWindow;
function launchServer(options, verbose) {
    (0, commandWithProgress_1.newSuccess)('Starting the React-Native Server');
    const opts = {
        cwd: options.root,
        detached: true,
        stdio: verbose ? 'inherit' : 'ignore',
    };
    (0, child_process_1.spawn)('cmd.exe', ['/C', 'start npx react-native start'], opts);
}
//# sourceMappingURL=deploy.js.map