"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.autolinkCommand = exports.autolinkWindowsInternal = exports.AutoLinkWindows = void 0;
// Types in this file are inaccurate compared to usage in terms of falsiness.
// We should try to rewrite some of this to do automated schema validation to
// guarantee correct types
/* eslint-disable @typescript-eslint/no-unnecessary-condition */
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const chalk_1 = __importDefault(require("chalk"));
const perf_hooks_1 = require("perf_hooks");
const xmldom_1 = require("@xmldom/xmldom");
const formatter = require('xml-formatter');
const telemetry_1 = require("@react-native-windows/telemetry");
const autolinkWindowsOptions_1 = require("./autolinkWindowsOptions");
const commandWithProgress_1 = require("../../utils/commandWithProgress");
const vstools = __importStar(require("../../utils/vstools"));
const generatorCommon = __importStar(require("../../generator-common"));
const configUtils = __importStar(require("../config/configUtils"));
const pathHelpers = __importStar(require("../../utils/pathHelpers"));
const telemetryHelpers_1 = require("../../utils/telemetryHelpers");
class AutoLinkWindows {
    areChangesNeeded() {
        return this.changesNecessary;
    }
    getWindowsConfig() {
        return this.windowsAppConfig;
    }
    getSolutionFile() {
        return path_1.default.join(this.getWindowsConfig().folder, this.getWindowsConfig().sourceDir, this.getWindowsConfig().solutionFile);
    }
    constructor(projectConfig, dependenciesConfig, options) {
        this.projectConfig = projectConfig;
        this.dependenciesConfig = dependenciesConfig;
        this.options = options;
        this.changesNecessary = false;
        if (!('windows' in this.projectConfig) ||
            this.projectConfig.windows === null) {
            throw new telemetry_1.CodedError('NoWindowsConfig', 'Windows auto-link only supported on Windows app projects');
        }
        this.windowsAppConfig = projectConfig.windows;
    }
    async run(spinner) {
        const verbose = this.options.logging;
        verboseMessage('', verbose);
        verboseMessage('Parsing project...', verbose);
        const rnwRoot = resolveRnwRoot(this.windowsAppConfig);
        const templateRoot = resolveTemplateRoot(this.windowsAppConfig);
        this.fixUpForSlnOption();
        this.fixUpForProjOption();
        verboseMessage('Found Windows app project, config:', verbose);
        verboseMessage(this.windowsAppConfig, verbose);
        this.validateRequiredAppProperties();
        const solutionFile = this.getSolutionFile();
        const windowsAppProjectConfig = this.windowsAppConfig.project;
        this.validateRequiredProjectProperties();
        const projectFile = this.getProjectFile();
        const projectDir = path_1.default.dirname(projectFile);
        const projectLang = windowsAppProjectConfig.projectLang;
        verboseMessage('Parsing dependencies...', verbose);
        this.changesNecessary =
            (await this.ensureXAMLDialect()) || this.changesNecessary;
        // Generating cs/cpp files for app code consumption
        if (projectLang === 'cs') {
            this.changesNecessary =
                (await this.generateCSAutolinking(templateRoot, projectLang, projectDir)) || this.changesNecessary;
        }
        else if (projectLang === 'cpp') {
            this.changesNecessary =
                (await this.generateCppAutolinking(templateRoot, projectLang, projectDir)) || this.changesNecessary;
        }
        // Generating props for app project consumption
        let propertiesForProps = '';
        let csModuleNames = [];
        if (projectLang === 'cpp') {
            csModuleNames = this.getCSModules();
            if (csModuleNames.length > 0) {
                propertiesForProps += `
    <!-- Set due to dependency on C# module(s): ${csModuleNames.join()} -->
    <ConsumeCSharpModules Condition="'$(ConsumeCSharpModules)'==''">true</ConsumeCSharpModules>`;
            }
        }
        this.changesNecessary =
            (await this.generateAutolinkProps(templateRoot, projectDir, propertiesForProps)) || this.changesNecessary;
        // Generating targets for app project consumption
        this.changesNecessary =
            (await this.generateAutolinkTargets(projectDir, templateRoot)) ||
                this.changesNecessary;
        // Generating project entries for solution
        this.changesNecessary =
            this.updateSolution(rnwRoot, solutionFile) || this.changesNecessary;
        spinner.succeed();
    }
    /**
     * Handles the --proj command-line option by consuming its value into the windowsAppConfig
     */
    fixUpForProjOption() {
        if (this.options.proj) {
            const projFile = path_1.default.join(this.windowsAppConfig.folder, this.options.proj);
            const projectContents = configUtils.readProjectFile(projFile);
            this.windowsAppConfig.project = {
                projectFile: path_1.default.relative(path_1.default.join(this.windowsAppConfig.folder, this.windowsAppConfig.sourceDir), projFile),
                projectName: configUtils.getProjectName(projFile, projectContents),
                projectLang: configUtils.getProjectLanguage(projFile),
                projectGuid: configUtils.getProjectGuid(projectContents),
            };
        }
    }
    /**
     * Handles the --sln command-line option by consuming its value into the windowsAppConfig
     */
    fixUpForSlnOption() {
        if (this.options.sln) {
            const slnFile = path_1.default.join(this.windowsAppConfig.folder, this.options.sln);
            this.windowsAppConfig.solutionFile = path_1.default.relative(path_1.default.join(this.windowsAppConfig.folder, this.windowsAppConfig.sourceDir), slnFile);
        }
    }
    /** Validates the all of the required app (solution) properties are present and valid */
    validateRequiredAppProperties() {
        const alwaysRequired = [
            'folder',
            'sourceDir',
            'solutionFile',
            'project',
        ];
        alwaysRequired.forEach(item => {
            if (!(item in this.windowsAppConfig) ||
                this.windowsAppConfig[item] === null) {
                throw new telemetry_1.CodedError('IncompleteConfig', `${item} is required but not specified by react-native config`, { item: item });
            }
            else if (typeof this.windowsAppConfig[item] === 'string' &&
                this.windowsAppConfig[item].startsWith('Error: ')) {
                throw new telemetry_1.CodedError('InvalidConfig', `${item} invalid. ${this.windowsAppConfig[item]}`, { item: item });
            }
        });
    }
    /** @return the full path to the project file (.vcxproj or .csproj) */
    getProjectFile() {
        const windowsAppConfig = this.getWindowsConfig();
        return path_1.default.join(windowsAppConfig.folder, windowsAppConfig.sourceDir, windowsAppConfig.project.projectFile);
    }
    /** Validates that all of the required app _project_ properties are present and valid */
    validateRequiredProjectProperties() {
        const windowsAppProjectConfig = this.windowsAppConfig.project;
        const projectRequired = [
            'projectFile',
            'projectName',
            'projectLang',
            'projectGuid',
        ];
        projectRequired.forEach(item => {
            if (!(item in windowsAppProjectConfig) ||
                windowsAppProjectConfig[item] === null) {
                throw new telemetry_1.CodedError('IncompleteConfig', `project.${item} is required but not specified by react-native config`, { item: item });
            }
            else if (typeof windowsAppProjectConfig[item] === 'string' &&
                windowsAppProjectConfig[item].startsWith('Error: ')) {
                throw new telemetry_1.CodedError('InvalidConfig', `project.${item} invalid. ${windowsAppProjectConfig[item]}`, { item: item });
            }
        });
    }
    async generateCppAutolinking(templateRoot, projectLang, projectDir) {
        const { cppPackageProviders, cppIncludes } = this.getCppReplacements();
        const cppFileName = 'AutolinkedNativeModules.g.cpp';
        const srcCppFile = path_1.default.join(templateRoot, `${projectLang}-app`, 'src', cppFileName);
        const destCppFile = path_1.default.join(projectDir, cppFileName);
        verboseMessage(`Calculating ${chalk_1.default.bold(path_1.default.basename(destCppFile))}...`, this.options.logging);
        const cppContents = generatorCommon.resolveContents(srcCppFile, {
            useMustache: true,
            autolinkCppIncludes: cppIncludes,
            autolinkCppPackageProviders: cppPackageProviders,
        });
        return await this.updateFile(destCppFile, cppContents);
    }
    getCppReplacements() {
        let cppIncludes = '';
        let cppPackageProviders = '';
        const windowsDependencies = this.getWindowsDependencies();
        for (const dependencyName of Object.keys(windowsDependencies)) {
            windowsDependencies[dependencyName].projects.forEach(project => {
                if (project.directDependency) {
                    cppIncludes += `\n\n// Includes from ${dependencyName}`;
                    project.cppHeaders.forEach(header => {
                        cppIncludes += `\n#include <${header}>`;
                    });
                    cppPackageProviders += `\n    // IReactPackageProviders from ${dependencyName}`;
                    project.cppPackageProviders.forEach(packageProvider => {
                        cppPackageProviders += `\n    packageProviders.Append(winrt::${packageProvider}());`;
                    });
                }
            });
        }
        if (cppPackageProviders === '') {
            // There are no windows dependencies, this would result in warning. C4100: 'packageProviders': unreferenced formal parameter.
            // therefore add a usage.
            cppPackageProviders = '\n    UNREFERENCED_PARAMETER(packageProviders);'; // CODESYNC: vnext\local-cli\generator-windows\index.js
        }
        return { cppPackageProviders, cppIncludes };
    }
    generateCSAutolinking(templateRoot, projectLang, projectDir) {
        const { csUsingNamespaces, csReactPackageProviders } = this.getCsReplacements();
        const csFileName = 'AutolinkedNativeModules.g.cs';
        const srcCsFile = path_1.default.join(templateRoot, `${projectLang}-app`, 'src', csFileName);
        const destCsFile = path_1.default.join(projectDir, csFileName);
        verboseMessage(`Calculating ${chalk_1.default.bold(path_1.default.basename(destCsFile))}...`, this.options.logging);
        const csContents = generatorCommon.resolveContents(srcCsFile, {
            useMustache: true,
            autolinkCsUsingNamespaces: csUsingNamespaces,
            autolinkCsReactPackageProviders: csReactPackageProviders,
        });
        return this.updateFile(destCsFile, csContents);
    }
    getCsReplacements() {
        let csUsingNamespaces = '';
        let csReactPackageProviders = '';
        const windowsDependencies = this.getWindowsDependencies();
        for (const dependencyName of Object.keys(windowsDependencies)) {
            windowsDependencies[dependencyName].projects.forEach(project => {
                if (project.directDependency) {
                    csUsingNamespaces += `\n\n// Namespaces from ${dependencyName}`;
                    project.csNamespaces.forEach(namespace => {
                        csUsingNamespaces += `\nusing ${namespace};`;
                    });
                    csReactPackageProviders += `\n            // IReactPackageProviders from ${dependencyName}`;
                    project.csPackageProviders.forEach(packageProvider => {
                        csReactPackageProviders += `\n            packageProviders.Add(new ${packageProvider}());`;
                    });
                }
            });
        }
        return { csUsingNamespaces, csReactPackageProviders };
    }
    getWindowsDependencies() {
        if (!this.windowsDependencies) {
            this.windowsDependencies = {};
            for (const dependencyName of Object.keys(this.dependenciesConfig)) {
                const windowsDependency = this.dependenciesConfig[dependencyName].platforms.windows;
                if (windowsDependency) {
                    verboseMessage(`${chalk_1.default.bold(dependencyName)} has Windows implementation, config:`, this.options.logging);
                    verboseMessage(windowsDependency, this.options.logging);
                    let dependencyIsValid = false;
                    const hasValidSourceDir = 'sourceDir' in windowsDependency &&
                        windowsDependency.sourceDir !== undefined &&
                        !windowsDependency.sourceDir.startsWith('Error: ');
                    const hasProjectsInProjectsArray = 'projects' in windowsDependency &&
                        Array.isArray(windowsDependency.projects) &&
                        windowsDependency.projects.length > 0;
                    if (hasValidSourceDir && hasProjectsInProjectsArray) {
                        // Module is source-based and has projects
                        dependencyIsValid = true;
                        // Validate each source project
                        windowsDependency.projects.forEach(project => {
                            const itemsToCheck = [
                                'projectFile',
                                'directDependency',
                            ];
                            itemsToCheck.forEach(item => {
                                dependencyIsValid = !!(dependencyIsValid &&
                                    item in project &&
                                    project[item] !== '' &&
                                    !project[item].toString().startsWith('Error: '));
                            });
                        });
                    }
                    if (dependencyIsValid) {
                        verboseMessage(`Adding ${chalk_1.default.bold(dependencyName)}.`, this.options.logging);
                        this.windowsDependencies[dependencyName] = windowsDependency;
                    }
                    else {
                        verboseMessage(`Invalid dependency configuration for dependency ${dependencyName}`, this.options.logging);
                    }
                }
            }
        }
        return this.windowsDependencies;
    }
    /**
     * Updates the target file with the expected contents if it's different.
     * @param filePath Path to the target file to update.
     * @param expectedContents The expected contents of the file.
     * @return Whether any changes were necessary.
     */
    async updateFile(filePath, expectedContents) {
        const fileName = chalk_1.default.bold(path_1.default.basename(filePath));
        verboseMessage(`Reading ${fileName}...`, this.options.logging);
        const actualContents = fs_1.default.existsSync(filePath)
            ? (await fs_1.default.readFile(filePath)).toString()
            : '';
        const contentsChanged = expectedContents !== actualContents;
        if (contentsChanged) {
            verboseMessage(chalk_1.default.yellow(`${fileName} needs to be updated.`), this.options.logging);
            if (!this.options.check) {
                verboseMessage(`Writing ${fileName}...`, this.options.logging);
                await fs_1.default.writeFile(filePath, expectedContents, {
                    encoding: 'utf8',
                    flag: 'w',
                });
            }
        }
        else {
            verboseMessage(`No changes to ${fileName}.`, this.options.logging);
        }
        return contentsChanged;
    }
    generateAutolinkTargets(projectDir, templateRoot) {
        let projectReferencesForTargets = '';
        const windowsDependencies = this.getWindowsDependencies();
        for (const dependencyName of Object.keys(windowsDependencies)) {
            windowsDependencies[dependencyName].projects.forEach(project => {
                if (project.directDependency) {
                    const dependencyProjectFile = path_1.default.join(windowsDependencies[dependencyName].folder, windowsDependencies[dependencyName].sourceDir, project.projectFile);
                    const relDependencyProjectFile = path_1.default.relative(projectDir, dependencyProjectFile);
                    projectReferencesForTargets += `
    <!-- Projects from ${dependencyName} -->
    <ProjectReference Include="$(ProjectDir)${relDependencyProjectFile}">
      <Project>${project.projectGuid}</Project>
    </ProjectReference>`;
                }
            });
        }
        const targetFileName = 'AutolinkedNativeModules.g.targets';
        const srcTargetFile = path_1.default.join(templateRoot, `shared-app`, 'src', targetFileName);
        const destTargetFile = path_1.default.join(projectDir, targetFileName);
        verboseMessage(`Calculating ${chalk_1.default.bold(path_1.default.basename(destTargetFile))}...`, this.options.logging);
        const targetContents = generatorCommon.resolveContents(srcTargetFile, {
            useMustache: true,
            autolinkProjectReferencesForTargets: projectReferencesForTargets,
        });
        return this.updateFile(destTargetFile, targetContents);
    }
    generateAutolinkProps(templateRoot, projectDir, propertiesForProps) {
        const propsFileName = 'AutolinkedNativeModules.g.props';
        const srcPropsFile = path_1.default.join(templateRoot, `shared-app`, 'src', propsFileName);
        const destPropsFile = path_1.default.join(projectDir, propsFileName);
        verboseMessage(`Calculating ${chalk_1.default.bold(path_1.default.basename(destPropsFile))}...`, this.options.logging);
        const propsContents = generatorCommon.resolveContents(srcPropsFile, {
            useMustache: true,
            autolinkPropertiesForProps: propertiesForProps,
        });
        return this.updateFile(destPropsFile, propsContents);
    }
    getCSModules() {
        const csModuleNames = [];
        const windowsDependencies = this.getWindowsDependencies();
        for (const dependencyName of Object.keys(windowsDependencies)) {
            windowsDependencies[dependencyName].projects.forEach(project => {
                if (project.directDependency && project.projectLang === 'cs') {
                    csModuleNames.push(project.projectName);
                }
            });
        }
        return csModuleNames;
    }
    updateSolution(rnwRoot, solutionFile) {
        const projectsForSolution = [];
        const windowsDependencies = this.getWindowsDependencies();
        for (const dependencyName of Object.keys(windowsDependencies)) {
            // Process dependency projects
            windowsDependencies[dependencyName].projects.forEach(project => {
                const dependencyProjectFile = path_1.default.join(windowsDependencies[dependencyName].folder, windowsDependencies[dependencyName].sourceDir, project.projectFile);
                projectsForSolution.push({
                    projectFile: dependencyProjectFile,
                    projectName: project.projectName,
                    projectLang: project.projectLang,
                    projectGuid: project.projectGuid,
                });
            });
        }
        const csModuleNames = this.getCSModules();
        if (csModuleNames.length > 0) {
            // Add managed projects
            projectsForSolution.push({
                projectFile: path_1.default.join(rnwRoot, 'Microsoft.ReactNative.Managed/Microsoft.ReactNative.Managed.csproj'),
                projectName: 'Microsoft.ReactNative.Managed',
                projectLang: 'cs',
                projectGuid: '{F2824844-CE15-4242-9420-308923CD76C3}',
            });
            projectsForSolution.push({
                projectFile: path_1.default.join(rnwRoot, 'Microsoft.ReactNative.Managed.CodeGen//Microsoft.ReactNative.Managed.CodeGen.csproj'),
                projectName: 'Microsoft.ReactNative.Managed.CodeGen',
                projectLang: 'cs',
                projectGuid: '{ADED4FBE-887D-4271-AF24-F0823BCE7961}',
                projectTypeGuid: vstools.dotNetCoreProjectTypeGuid,
            });
        }
        verboseMessage(`Calculating ${chalk_1.default.bold(path_1.default.basename(solutionFile))} changes...`, this.options.logging);
        let changesNecessary = false;
        projectsForSolution.forEach(project => {
            const contentsChanged = vstools.addProjectToSolution(solutionFile, project, this.options.logging, this.options.check);
            changesNecessary = changesNecessary || contentsChanged;
        });
        return changesNecessary;
    }
    getExperimentalFeaturesPropsXml() {
        const experimentalFeaturesProps = path_1.default.join(path_1.default.dirname(this.getSolutionFile()), 'ExperimentalFeatures.props');
        if (fs_1.default.existsSync(experimentalFeaturesProps)) {
            const experimentalFeaturesContents = configUtils.readProjectFile(experimentalFeaturesProps);
            return {
                path: experimentalFeaturesProps,
                content: experimentalFeaturesContents,
            };
        }
        return undefined;
    }
    async ensureXAMLDialect() {
        var _a, _b;
        let changesNeeded = false;
        const useWinUI3FromConfig = this.getWindowsConfig().useWinUI3;
        const experimentalFeatures = this.getExperimentalFeaturesPropsXml();
        if (experimentalFeatures) {
            const useWinUI3FromExperimentalFeatures = ((_a = configUtils
                .tryFindPropertyValue(experimentalFeatures.content, 'UseWinUI3')) === null || _a === void 0 ? void 0 : _a.toLowerCase()) === 'true';
            // Check if WinUI2xVersion is specified in experimental features
            const targetWinUI2xVersion = configUtils.tryFindPropertyValue(experimentalFeatures.content, 'WinUI2xVersion');
            // Check if WinUI3Version is specified in experimental features
            const targetWinUI3xVersion = configUtils.tryFindPropertyValue(experimentalFeatures.content, 'WinUI3Version');
            // Use the UseWinUI3 value in react-native.config.js, or if not present, the value from ExperimentalFeatures.props
            changesNeeded = await this.updatePackagesConfigXAMLDialect(useWinUI3FromConfig !== undefined
                ? useWinUI3FromConfig
                : useWinUI3FromExperimentalFeatures, targetWinUI2xVersion, targetWinUI3xVersion);
            if (useWinUI3FromConfig !== undefined) {
                // Make sure ExperimentalFeatures.props matches the value that comes from react-native.config.js
                const node = experimentalFeatures.content.getElementsByTagName('UseWinUI3');
                const newValue = useWinUI3FromConfig ? 'true' : 'false';
                changesNeeded = ((_b = node.item(0)) === null || _b === void 0 ? void 0 : _b.textContent) !== newValue || changesNeeded;
                if (!this.options.check && changesNeeded) {
                    node.item(0).textContent = newValue;
                    const experimentalFeaturesOutput = new xmldom_1.XMLSerializer().serializeToString(experimentalFeatures.content);
                    await this.updateFile(experimentalFeatures.path, experimentalFeaturesOutput);
                }
            }
        }
        return changesNeeded;
    }
    getPackagesConfigXml() {
        const projectFile = this.getProjectFile();
        const packagesConfig = path_1.default.join(path_1.default.dirname(projectFile), 'packages.config');
        if (fs_1.default.existsSync(packagesConfig)) {
            return {
                path: packagesConfig,
                content: configUtils.readProjectFile(packagesConfig),
            };
        }
        return undefined;
    }
    async updatePackagesConfigXAMLDialect(useWinUI3, targetWinUI2xVersion, targetWinUI3xVersion) {
        let changed = false;
        const packagesConfig = this.getPackagesConfigXml();
        if (packagesConfig) {
            // if we don't have a packages.config, then this is a C# project, in which case we use <PackageReference> and dynamically pick the right XAML package.
            const project = this.getWindowsConfig();
            const winUIPropsPath = path_1.default.join(resolveRnwRoot(project), 'PropertySheets/WinUI.props');
            const winuiPropsContents = configUtils.readProjectFile(winUIPropsPath);
            // Use the given WinUI2xVersion, otherwise fallback to WinUI.props
            const winui2xVersion = targetWinUI2xVersion !== null && targetWinUI2xVersion !== void 0 ? targetWinUI2xVersion : configUtils.tryFindPropertyValue(winuiPropsContents, 'WinUI2xVersion');
            // Use the given WinUI3Version, otherwise fallback to WinUI.props
            const winui3Version = targetWinUI3xVersion !== null && targetWinUI3xVersion !== void 0 ? targetWinUI3xVersion : configUtils.tryFindPropertyValue(winuiPropsContents, 'WinUI3Version');
            const dialects = [
                { id: 'Microsoft.WindowsAppSDK', version: winui3Version },
                { id: 'Microsoft.UI.Xaml', version: winui2xVersion },
            ];
            const keepPkg = useWinUI3 ? dialects[0] : dialects[1];
            const removePkg = useWinUI3 ? dialects[1] : dialects[0];
            changed = this.updatePackagesConfig(packagesConfig, [removePkg], [keepPkg]);
            if (!this.options.check && changed) {
                const serializer = new xmldom_1.XMLSerializer();
                const output = serializer.serializeToString(packagesConfig.content);
                const formattedXml = formatter(output, { indentation: '  ' });
                await this.updateFile(packagesConfig.path, formattedXml);
            }
        }
        return changed;
    }
    updatePackagesConfig(packagesConfig, removePkgs, keepPkgs) {
        let changed = false;
        const packageElements = packagesConfig.content.documentElement.getElementsByTagName('package');
        const nodesToRemove = [];
        for (let i = 0; i < packageElements.length; i++) {
            const packageElement = packageElements.item(i);
            const idAttr = packageElement.getAttributeNode('id');
            const id = idAttr.value;
            const keepPkg = keepPkgs.find(pkg => pkg.id === id);
            if (removePkgs.find(pkg => pkg.id === id)) {
                nodesToRemove.push(packageElement);
                changed = true;
            }
            else if (keepPkg) {
                changed =
                    changed || keepPkg.version !== packageElement.getAttribute('version');
                packageElement.setAttribute('version', keepPkg.version);
                keepPkgs = keepPkgs.filter(pkg => pkg.id !== keepPkg.id);
            }
        }
        nodesToRemove.forEach(pkg => packagesConfig.content.documentElement.removeChild(pkg));
        keepPkgs.forEach(keepPkg => {
            const newPkg = packagesConfig.content.createElement('package');
            Object.entries(keepPkg).forEach(([attr, value]) => {
                newPkg.setAttribute(attr, value);
            });
            newPkg.setAttribute('targetFramework', 'native');
            packagesConfig.content.documentElement.appendChild(newPkg);
            changed = true;
        });
        return changed;
    }
    /** @return The CLI command to invoke autolink-windows independently */
    getAutolinkWindowsCommand() {
        const folder = this.windowsAppConfig.folder;
        const autolinkCommand = 'npx react-native autolink-windows';
        const autolinkArgs = `--sln "${path_1.default.relative(folder, this.getSolutionFile())}" --proj "${path_1.default.relative(folder, this.getProjectFile())}"`;
        return `${autolinkCommand} ${autolinkArgs}`;
    }
}
exports.AutoLinkWindows = AutoLinkWindows;
/**
 * Locates the react-native-windows directory
 * @param config project configuration
 */
function resolveRnwRoot(projectConfig) {
    return pathHelpers.resolveRnwRoot(projectConfig.folder);
}
/**
 * Locates the react-native-windows directory containing template files
 * @param config project configuration
 */
function resolveTemplateRoot(projectConfig) {
    const rnwPackage = resolveRnwRoot(projectConfig);
    return path_1.default.join(rnwPackage, 'template');
}
/**
 * Logs the given message if verbose is True.
 * @param message The message to log.
 * @param verbose Whether or not verbose logging is enabled.
 */
function verboseMessage(message, verbose) {
    if (verbose) {
        console.log(message);
    }
}
/**
 * Sanitizes the given option for telemetry.
 * @param key The key of the option.
 * @param value The unsanitized value of the option.
 * @returns The sanitized value of the option.
 */
function optionSanitizer(key, value) {
    // Do not add a default case here.
    // Strings risking PII should just return true if present, false otherwise.
    // All others should return the value (or false if undefined).
    switch (key) {
        case 'sln':
        case 'proj':
            return value === undefined ? false : true; // Strip PII
        case 'logging':
        case 'check':
        case 'telemetry':
            return value === undefined ? false : value; // Return value
    }
}
/**
 * Get the extra props to add to the `autolink-windows` telemetry event.
 * @returns The extra props.
 */
async function getExtraProps() {
    const extraProps = {};
    return extraProps;
}
/**
 * The function run when calling `react-native autolink-windows`.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function autolinkWindows(args, config, options) {
    await (0, telemetryHelpers_1.startTelemetrySession)('autolink-windows', config, options, (0, telemetryHelpers_1.getDefaultOptions)(config, autolinkWindowsOptions_1.autolinkOptions), optionSanitizer);
    let autolinkWindowsError;
    try {
        await autolinkWindowsInternal(args, config, options);
    }
    catch (ex) {
        autolinkWindowsError =
            ex instanceof Error ? ex : new Error(String(ex));
        telemetry_1.Telemetry.trackException(autolinkWindowsError);
    }
    await (0, telemetryHelpers_1.endTelemetrySession)(autolinkWindowsError, getExtraProps);
    (0, commandWithProgress_1.setExitProcessWithError)(options.logging, autolinkWindowsError);
}
/**
 * Performs auto-linking for RNW native modules and apps.
 * @param args Unprocessed args passed from react-native CLI.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 */
async function autolinkWindowsInternal(args, config, options) {
    const startTime = perf_hooks_1.performance.now();
    const spinner = (0, commandWithProgress_1.newSpinner)(options.check ? 'Checking auto-linked files...' : 'Auto-linking...');
    try {
        const autolink = new AutoLinkWindows(config.project, config.dependencies, options);
        await autolink.run(spinner);
        const endTime = perf_hooks_1.performance.now();
        if (!autolink.areChangesNeeded()) {
            console.log(`${chalk_1.default.green('Success:')} No auto-linking changes necessary. (${Math.round(endTime - startTime)}ms)`);
        }
        else if (options.check) {
            const autolinkCommand = autolink.getAutolinkWindowsCommand();
            console.log(`${chalk_1.default.yellow('Warning:')} Auto-linking changes were necessary but ${chalk_1.default.bold('--check')} specified. Run '${chalk_1.default.bold(`${autolinkCommand}`)}' to apply the changes. (${Math.round(endTime - startTime)}ms)`);
            throw new telemetry_1.CodedError('NeedAutolinking', `Auto-linking changes were necessary but --check was specified. Run '${autolinkCommand}' to apply the changes`);
        }
        else {
            console.log(`${chalk_1.default.green('Success:')} Auto-linking changes completed. (${Math.round(endTime - startTime)}ms)`);
        }
    }
    catch (e) {
        spinner.fail();
        const endTime = perf_hooks_1.performance.now();
        console.log(`${chalk_1.default.red('Error:')} ${e.toString()}. (${Math.round(endTime - startTime)}ms)`);
        throw e;
    }
}
exports.autolinkWindowsInternal = autolinkWindowsInternal;
/**
 * Performs auto-linking for RNW native modules and apps.
 */
exports.autolinkCommand = {
    name: 'autolink-windows',
    description: 'Runs Windows-specific autolinking',
    func: autolinkWindows,
    options: autolinkWindowsOptions_1.autolinkOptions,
};
//# sourceMappingURL=autolinkWindows.js.map