"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getProjectFileFromConfig = exports.configToProjectInfo = exports.usesTypeScript = exports.usesReactNativeConfig = exports.getProjectId = void 0;
const crypto_1 = require("crypto");
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
/**
 * Gets a unique, telemetry-safe project ID based on the project name.
 * @param projectName The project name.
 * @returns The telemetry-safe project ID.
 */
function getProjectId(projectName) {
    const hash = (0, crypto_1.createHash)('sha256');
    hash.update(projectName);
    return hash.digest('hex');
}
exports.getProjectId = getProjectId;
/**
 * Checks that a given file exits in the path specified.
 * @param fileName The file to check for.
 * @param projectRoot The root path to look in.
 * @returns Whether the file exists.
 */
async function fileExists(fileName, projectRoot) {
    try {
        const reactNativeConfigPath = path_1.default.resolve(projectRoot, fileName);
        const stats = await fs_1.default.stat(reactNativeConfigPath);
        return stats.isFile();
    }
    catch (_a) { }
    return false;
}
/**
 * Gets whether the project at the given path has a react-native.config.js file.
 * @param projectRoot The project root path to look in.
 * @returns Whether the project at the given path has a react-native.config.js file.
 */
async function usesReactNativeConfig(projectRoot) {
    return fileExists('./react-native.config.js', projectRoot);
}
exports.usesReactNativeConfig = usesReactNativeConfig;
/**
 * Gets whether the project at the given path is using TypeScript.
 * @param projectRoot The project root path to look in.
 * @returns Whether the project at the given path is using TypeScript.
 */
async function usesTypeScript(projectRoot) {
    return fileExists('./tsconfig.json', projectRoot);
}
exports.usesTypeScript = usesTypeScript;
/**
 * Get the list of keys in the object for which the value is defined.
 * @param obj The object to search.
 * @returns The list of keys.
 */
function getDefinedKeys(obj) {
    return Object.keys(obj).filter(value => obj[value] !== undefined && obj[value] !== null && value);
}
/**
 * Given a react-native CLI config, determine the language of the RNW dependency if possible.
 * @param config Dependency config passed from react-native CLI.
 * @returns The language of the RNW dependency.
 */
function getDependencyRnwLang(config) {
    if (config) {
        let cppCount = 0;
        let csCount = 0;
        for (const project of config.projects) {
            switch (project.projectLang) {
                case 'cpp':
                    cppCount++;
                    break;
                case 'cs':
                    csCount++;
                    break;
            }
        }
        if (cppCount > 0 && csCount > 0) {
            return 'cpp+cs';
        }
        else if (cppCount > 0) {
            return 'cpp';
        }
        else if (csCount > 0) {
            return 'cs';
        }
    }
    return null;
}
/**
 * Returns true if the item is a boolean with a value of true or a string with a value of 'true'.
 * @param item The item to parse.
 * @returns The boolean value.
 */
function parseBoolean(item) {
    if (typeof item === 'boolean') {
        return item;
    }
    else if (typeof item === 'string') {
        return item.toLowerCase() === 'true';
    }
    return false;
}
/**
 * Calculate the project telemetry info from a react-native CLI config.
 * @param config Config passed from react-native CLI.
 * @returns The calculated project info.
 */
async function configToProjectInfo(config) {
    var _a, _b, _c;
    try {
        const id = getProjectId(require(path_1.default.join(config.root, 'package.json')).name);
        const platforms = getDefinedKeys(config.project);
        if ('windows' in config.project && config.project.windows !== null) {
            const rnwLang = (_a = config.project.windows.project) === null || _a === void 0 ? void 0 : _a.projectLang;
            const usesTS = await usesTypeScript(config.project.windows.folder);
            const usesRNConfig = await usesReactNativeConfig(config.project.windows.folder);
            const jsEngine = parseBoolean((_b = config.project.windows.experimentalFeatures) === null || _b === void 0 ? void 0 : _b.UseHermes)
                ? 'Hermes'
                : 'Chakra';
            const rnwSource = parseBoolean((_c = config.project.windows.experimentalFeatures) === null || _c === void 0 ? void 0 : _c.UseExperimentalNuget)
                ? 'NuGet'
                : 'Source';
            const dependencies = [];
            for (const dependencyName in config.dependencies) {
                if (!Object.prototype.hasOwnProperty(dependencyName)) {
                    const dependencyId = getProjectId(dependencyName);
                    const dependencyPlatforms = getDefinedKeys(config.dependencies[dependencyName].platforms);
                    if (dependencyPlatforms.length > 0) {
                        const dependencyRnwLang = getDependencyRnwLang(config.dependencies[dependencyName].platforms.windows);
                        const dependencyInfo = {
                            id: dependencyId,
                            platforms: dependencyPlatforms,
                            rnwLang: dependencyRnwLang,
                        };
                        dependencies.push(dependencyInfo);
                    }
                }
            }
            const appInfo = {
                id,
                platforms,
                rnwLang,
                usesTS,
                usesRNConfig,
                jsEngine,
                rnwSource,
                dependencies,
            };
            return appInfo;
        }
        // Probably just a dependency project, return minimal info
        const dependencyInfo = {
            id,
            platforms,
            rnwLang: null,
        };
        return dependencyInfo;
    }
    catch (_d) { }
    return null;
}
exports.configToProjectInfo = configToProjectInfo;
/**
 * Gets the full path to the app's native project file from a react-native CLI config.
 * @param config Config passed from react-native CLI.
 * @returns The full path to the app's native project file
 */
function getProjectFileFromConfig(config) {
    try {
        if ('windows' in config.project && config.project.windows !== null) {
            return path_1.default.join(config.project.windows.folder, config.project.windows.sourceDir, config.project.windows.project.projectFile);
        }
    }
    catch (_a) { }
    return null;
}
exports.getProjectFileFromConfig = getProjectFileFromConfig;
//# sourceMappingURL=projectUtils.js.map