/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.translateField = exports.translateFieldOrReturnType = void 0;
const AliasManaging_1 = require("./AliasManaging");
function translateUnionReturnType(type, options) {
    const memberType = type.memberType;
    switch (type.memberType) {
        case 'StringTypeAnnotation':
            return options.cppStringType;
        case 'NumberTypeAnnotation':
            return 'double';
        case 'ObjectTypeAnnotation':
            return '::React::JSValue';
        default:
            throw new Error(`Unknown enum/union member type in translateReturnType: ${memberType}`);
    }
}
function translateFieldOrReturnType(type, aliases, baseAliasName, callerName, options) {
    // avoid: Property 'type' does not exist on type 'never'
    const returnType = type.type;
    switch (type.type) {
        case 'StringTypeAnnotation':
            return options.cppStringType;
        case 'NumberTypeAnnotation':
        case 'FloatTypeAnnotation':
        case 'DoubleTypeAnnotation':
            return 'double';
        case 'Int32TypeAnnotation':
            return 'int';
        case 'BooleanTypeAnnotation':
            return 'bool';
        case 'ArrayTypeAnnotation':
            if (type.elementType) {
                return `std::vector<${translateFieldOrReturnType(type.elementType, aliases, `${baseAliasName}_element`, callerName, options)}>`;
            }
            else {
                return `::React::JSValueArray`;
            }
        case 'GenericObjectTypeAnnotation':
            return '::React::JSValue';
        case 'ObjectTypeAnnotation':
            return (0, AliasManaging_1.getAnonymousAliasCppName)(aliases, baseAliasName, type);
        case 'ReservedTypeAnnotation': {
            // avoid: Property 'name' does not exist on type 'never'
            const name = type.name;
            // (#6597)
            // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
            if (name !== 'RootTag')
                throw new Error(`Unknown reserved function: ${name} in ${callerName}`);
            return 'double';
        }
        case 'TypeAliasTypeAnnotation':
            return (0, AliasManaging_1.getAliasCppName)(type.name);
        case 'NullableTypeAnnotation':
            return `std::optional<${translateFieldOrReturnType(type.typeAnnotation, aliases, baseAliasName, callerName, options)}>`;
        case 'MixedTypeAnnotation':
            return '';
        case 'EnumDeclaration':
        case 'UnionTypeAnnotation':
            return translateUnionReturnType(type, options);
        default:
            throw new Error(`Unhandled type in ${callerName}: ${returnType}`);
    }
}
exports.translateFieldOrReturnType = translateFieldOrReturnType;
function translateField(type, aliases, baseAliasName, options) {
    return translateFieldOrReturnType(type, aliases, baseAliasName, 'translateField', options);
}
exports.translateField = translateField;
//# sourceMappingURL=ObjectTypes.js.map