"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getVersionsOfNuGetPackages = exports.getVersionOfNpmPackage = exports.getVisualStudioVersion = exports.getYarnVersion = exports.getNpmVersion = exports.getNodeVersion = void 0;
const envinfo_1 = __importDefault(require("envinfo"));
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const xmldom_1 = require("@xmldom/xmldom");
const xpath_1 = __importDefault(require("xpath"));
const msbuildSelect = xpath_1.default.useNamespaces({
    msbuild: 'http://schemas.microsoft.com/developer/msbuild/2003',
});
/**
 * Gets the version of node being used.
 * @returns The version of node being used.
 */
async function getNodeVersion() {
    return process.version.slice(1);
}
exports.getNodeVersion = getNodeVersion;
/**
 * Gets the version of npm installed, if available.
 * @returns The version of npm installed, if available.
 */
async function getNpmVersion() {
    try {
        const info = await envinfo_1.default.helpers.getnpmInfo();
        return info[1];
    }
    catch (_a) { }
    return null;
}
exports.getNpmVersion = getNpmVersion;
/**
 * Gets the version of yarn installed, if available.
 * @returns The version of yarn installed, if available.
 */
async function getYarnVersion() {
    try {
        const info = await envinfo_1.default.helpers.getYarnInfo();
        return info[1];
    }
    catch (_a) { }
    return null;
}
exports.getYarnVersion = getYarnVersion;
/**
 * Gets the latest version of Visual Studio installed, if available.
 * @returns The latest version of Visual Studio installed, if available.
 */
async function getVisualStudioVersion() {
    try {
        const info = await envinfo_1.default.helpers.getVisualStudioInfo();
        const versions = info[1];
        return versions.sort().slice(-1)[0].split(' ')[0];
    }
    catch (_a) { }
    return null;
}
exports.getVisualStudioVersion = getVisualStudioVersion;
/**
 * Gets the version installed of the specified npm package.
 * @param pkgName The npm package name.
 * @returns The version installed, if available.
 */
async function getVersionOfNpmPackage(pkgName) {
    try {
        const pkgJsonPath = require.resolve(`${pkgName.trim()}/package.json`, {
            paths: [process.cwd(), __dirname],
        });
        const pkgJson = await fs_1.default.readJsonFile(pkgJsonPath);
        if (pkgJson.name === pkgName) {
            return pkgJson.version;
        }
    }
    catch (_a) { }
    return null;
}
exports.getVersionOfNpmPackage = getVersionOfNpmPackage;
/**
 * Reads and parses an XML file into a Document.
 * @param filePath The path to the XML file.
 * @returns The parsed Document.
 */
async function readXmlFile(filePath) {
    const contents = await fs_1.default.readFile(filePath, 'utf-8');
    return new xmldom_1.DOMParser().parseFromString(contents);
}
/**
 * Gets the versions of the specified NuGet packages referenced in a packages.config file.
 * @param projectDoc The XML document of the packages.config file.
 * @param nugetPackages The NuGet package names to look for.
 * @returns The mapping of NuGet package names and their versions.
 */
function getVersionsFromPackagesConfig(packagesConfigDoc, nugetPackages) {
    const versions = {};
    for (const pkgName of nugetPackages) {
        const version = xpath_1.default.select1(`//packages/package[@id='${pkgName}']/@version`, packagesConfigDoc);
        if (version) {
            const versionValue = version.nodeValue;
            if (versionValue !== null) {
                versions[pkgName] = versionValue;
            }
        }
    }
    return versions;
}
/**
 * Gets the versions of the specified NuGet packages referenced in a project file.
 * @param projectDoc The XML document of the project file.
 * @param nugetPackages The NuGet package names to look for.
 * @returns The mapping of NuGet package names and their versions.
 */
function getVersionsFromProjectFile(projectDoc, nugetPackages) {
    const versions = {};
    for (const pkgName of nugetPackages) {
        const version = msbuildSelect(`//msbuild:ItemGroup/msbuild:PackageReference[@Include='${pkgName}']/msbuild:Version`, projectDoc, true);
        if (version) {
            const versionValue = version.textContent;
            if (versionValue !== null) {
                versions[pkgName] = versionValue;
            }
        }
    }
    return versions;
}
/**
 * Gets the versions of the specified NuGet packages referenced in a project file.
 * @param projectFile Path to the native project file.
 * @param nugetPackages The NuGet package names to look for.
 * @returns The mapping of NuGet package names and their versions.
 */
async function getVersionsOfNuGetPackages(projectFile, nugetPackages) {
    try {
        // First check for the presence of a packages.config file
        const packagesConfigFile = path_1.default.join(path_1.default.dirname(projectFile), 'packages.config');
        const packagesConfigDoc = await readXmlFile(packagesConfigFile);
        return getVersionsFromPackagesConfig(packagesConfigDoc, nugetPackages);
    }
    catch (_a) { }
    try {
        const projectDoc = await readXmlFile(projectFile);
        return getVersionsFromProjectFile(projectDoc, nugetPackages);
    }
    catch (_b) { }
    return {};
}
exports.getVersionsOfNuGetPackages = getVersionsOfNuGetPackages;
//# sourceMappingURL=versionUtils.js.map