/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
interface ProjectInfo {
    id: string | null;
    platforms: Array<string>;
    rnwLang: 'cpp' | 'cs' | 'cpp+cs' | null;
}
export interface DependencyProjectInfo extends ProjectInfo {
}
export interface AppProjectInfo extends ProjectInfo {
    usesTS: boolean;
    usesRNConfig: boolean;
    jsEngine: string;
    rnwSource: string;
    dependencies: Array<DependencyProjectInfo>;
}
/**
 * Gets a unique, telemetry-safe project ID based on the project name.
 * @param projectName The project name.
 * @returns The telemetry-safe project ID.
 */
export declare function getProjectId(projectName: string): string;
/**
 * Gets whether the project at the given path has a react-native.config.js file.
 * @param projectRoot The project root path to look in.
 * @returns Whether the project at the given path has a react-native.config.js file.
 */
export declare function usesReactNativeConfig(projectRoot: string): Promise<boolean>;
/**
 * Gets whether the project at the given path is using TypeScript.
 * @param projectRoot The project root path to look in.
 * @returns Whether the project at the given path is using TypeScript.
 */
export declare function usesTypeScript(projectRoot: string): Promise<boolean>;
/**
 * Calculate the project telemetry info from a react-native CLI config.
 * @param config Config passed from react-native CLI.
 * @returns The calculated project info.
 */
export declare function configToProjectInfo(config: Record<string, any>): Promise<AppProjectInfo | DependencyProjectInfo | null>;
/**
 * Gets the full path to the app's native project file from a react-native CLI config.
 * @param config Config passed from react-native CLI.
 * @returns The full path to the app's native project file
 */
export declare function getProjectFileFromConfig(config: Record<string, any>): string | null;
export {};
