/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
export type OptionSanitizer = (key: string, value: any) => any;
export type YargsOptionsType = Record<string, any>;
export interface CommanderOptionsType extends Record<string, any> {
    logging?: boolean;
    telemetry?: boolean;
}
/**
 * Converts a raw Commander arg name ('--no-bad-dog') into an options property name ('badDog').
 * @param name The raw Commander arg name.
 * @returns The options name.
 */
export declare function commanderNameToOptionName(name: string): string;
/**
 * Converts a yargs-parsed options object into an options object suitable for telemetry.
 * @param options The options object as parsed by yargs.
 * @param sanitizer Function to sanitize the option values for telemetry.
 * @returns An options object suitable for telemetry.
 */
export declare function yargsOptionsToOptions(options: YargsOptionsType, sanitizer?: OptionSanitizer): Record<string, any>;
/**
 * Converts a Commander-parsed options object into an options object suitable for telemetry.
 * @param options The options object as parsed by Commander.
 * @param sanitizer Function to sanitize the option values for telemetry.
 * @returns An options object suitable for telemetry.
 */
export declare function commanderOptionsToOptions(options: CommanderOptionsType, sanitizer?: OptionSanitizer): Record<string, any>;
/**
 * Checks whether the given command-line args contains the given option key.
 * @param key The option key to look for.
 * @param argv The raw command-line args.
 * @returns Whether the given command-line args contains the given option key.
 */
export declare function argsContainsOption(key: string, argv: string[]): boolean;
/**
 * Filters down a given set of options to only include those that were present in the command-line args.
 * @param options The full set of options.
 * @param argv The raw command-line args.
 * @returns The filtered set of options.
 */
export declare function optionsToArgs(options: Record<string, any>, argv: string[]): Record<string, any>;
