"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSessionId = exports.isCliTest = exports.isMsftInternal = exports.ciType = exports.isCI = exports.captureCI = exports.sampleRate = exports.deviceDiskFreeSpace = exports.deviceTotalMemory = exports.deviceNumCPUs = exports.deviceLocale = exports.devicePlatform = exports.nodeArchitecture = exports.deviceArchitecture = exports.deviceId = void 0;
const child_process_1 = require("child_process");
const os_1 = require("os");
const ci_info_1 = __importDefault(require("ci-info"));
const crypto_1 = require("crypto");
const os_locale_1 = __importDefault(require("os-locale"));
const DeviceIdRegPath = 'HKEY_LOCAL_MACHINE\\SOFTWARE\\Microsoft\\SQMClient';
const DeviceIdRegKey = 'MachineId';
/**
 * Gets a telemetry-safe stable device ID.
 * @returns A telemetry-safe stable device ID.
 */
async function deviceId() {
    try {
        let regCommand = `${process.env.windir}\\System32\\reg.exe query ${DeviceIdRegPath} /v ${DeviceIdRegKey}`;
        if (deviceArchitecture() === 'x64') {
            // Ensure we query the correct registry
            regCommand += ' /reg:64';
        }
        const output = (0, child_process_1.execSync)(regCommand).toString();
        const result = output.match(/\{([0-9A-Fa-f-]{36})\}/);
        if (result && result.length > 1) {
            return `s:${result[1]}`;
        }
    }
    catch (_a) { }
    return '';
}
exports.deviceId = deviceId;
/**
 * Gets the device architecture, like x86/x64/arm64.
 * @returns The device architecture.
 */
function deviceArchitecture() {
    const nodeArch = nodeArchitecture();
    // Check if we're running x86 node on x64 hardware
    if (nodeArch === 'x86' && process.env.PROCESSOR_ARCHITEW6432 === 'AMD64') {
        return 'x64';
    }
    return nodeArch;
}
exports.deviceArchitecture = deviceArchitecture;
/**
 * Gets the node architecture, like x86/x64/arm64.
 * @returns The node architecture.
 */
function nodeArchitecture() {
    return process.arch === 'ia32' ? 'x86' : process.arch;
}
exports.nodeArchitecture = nodeArchitecture;
/**
 * Gets the device platform, like darwin/linux/win32.
 * @returns The device platform.
 */
function devicePlatform() {
    return (0, os_1.platform)();
}
exports.devicePlatform = devicePlatform;
/**
 * Gets the device locale.
 * @returns The device locale.
 */
async function deviceLocale() {
    return await (0, os_locale_1.default)();
}
exports.deviceLocale = deviceLocale;
/**
 * Gets the device's number of CPUs.
 * @returns The device's number of CPUs.
 */
function deviceNumCPUs() {
    return (0, os_1.cpus)().length;
}
exports.deviceNumCPUs = deviceNumCPUs;
/**
 * Gets the device's total memory in bytes.
 * @returns The device's total memory in bytes.
 */
function deviceTotalMemory() {
    return (0, os_1.totalmem)();
}
exports.deviceTotalMemory = deviceTotalMemory;
/**
 * Gets the free space of the give drive in bytes.
 * @param drivePath A path on the drive to check.
 * @returns The free space of the give drive in bytes.
 */
function deviceDiskFreeSpace(drivePath) {
    try {
        const out = (0, child_process_1.execSync)(`dir /-C ${drivePath !== null && drivePath !== void 0 ? drivePath : process.cwd()}`)
            .toString()
            .split('\r\n');
        const line = out[out.length - 2];
        const result = line.match(/(\d+) [^\d]+(\d+) /);
        if (result && result.length > 2) {
            return Number(result[2]);
        }
    }
    catch (_a) { }
    return -1;
}
exports.deviceDiskFreeSpace = deviceDiskFreeSpace;
/**
 * Gets the telemetry sample rate.
 * @returns The telemetry sample rate.
 */
function sampleRate() {
    return 100;
}
exports.sampleRate = sampleRate;
/**
 * Gets whether or not telemetry events are captured when running in CI.
 * @returns Whether or not telemetry events are captured when running in CI.
 */
function captureCI() {
    // Only capture events in CI if running tests
    return isCliTest();
}
exports.captureCI = captureCI;
/**
 * Gets the whether the process is currently running in CI.
 * @returns Whether the process is currently running in CI.
 */
function isCI() {
    return ci_info_1.default.isCI;
}
exports.isCI = isCI;
/**
 * Gets the type of CI the process is running under.
 * @returns The type of CI the process is running under.
 */
function ciType() {
    var _a;
    return ci_info_1.default.isCI ? (_a = ci_info_1.default.name) !== null && _a !== void 0 ? _a : 'Unknown' : 'None';
}
exports.ciType = ciType;
/**
 * Gets whether the process is running on a Microsoft owned machine.
 * @returns Whether the process is running on a Microsoft owned machine.
 */
function isMsftInternal() {
    return (process.env.UserDNSDomain !== undefined &&
        process.env.UserDNSDomain.toLowerCase().endsWith('corp.microsoft.com'));
}
exports.isMsftInternal = isMsftInternal;
/**
 * Gets whether the process is running as part of our CLI tests.
 * @returns Whether the process is running as part of our CLI tests.
 */
function isCliTest() {
    return (process.env.RNW_CLI_TEST !== undefined &&
        process.env.RNW_CLI_TEST.toLowerCase().trim() === 'true');
}
exports.isCliTest = isCliTest;
let sessionId;
/**
 * Gets a stable session ID for correlating telemetry events.
 * @returns A stable session ID for correlating telemetry events.
 */
function getSessionId() {
    return (sessionId !== null && sessionId !== void 0 ? sessionId : (sessionId = (0, crypto_1.randomBytes)(16).toString('hex').padStart(32, '0')));
}
exports.getSessionId = getSessionId;
//# sourceMappingURL=basePropUtils.js.map