"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Telemetry = exports.NuGetPackagesWeTrack = exports.NpmPackagesWeTrack = exports.EventNamesWeTrack = exports.CodedErrorEventName = exports.CommandEventName = void 0;
const appInsights = __importStar(require("applicationinsights"));
const basePropUtils = __importStar(require("./utils/basePropUtils"));
const versionUtils = __importStar(require("./utils/versionUtils"));
const errorUtils = __importStar(require("./utils/errorUtils"));
// This is our key with the AI backend
const RNWSetupString = '795006ca-cf54-40ee-8bc6-03deb91401c3';
// Environment variable to override the default setup string
const ENV_SETUP_OVERRIDE = 'RNW_TELEMETRY_SETUP';
// Environment variable to override the http proxy (such as http://localhost:8888 for Fiddler debugging)
const ENV_PROXY_OVERRIDE = 'RNW_TELEMETRY_PROXY';
exports.CommandEventName = 'RNWCLI.Command';
exports.CodedErrorEventName = 'RNWCLI.CodedError';
// These are the event names we're tracking
exports.EventNamesWeTrack = [
    exports.CommandEventName,
    exports.CodedErrorEventName,
];
// These are NPM packages we care about, in terms of capturing versions used
// and getting more details about when reporting errors
exports.NpmPackagesWeTrack = [
    '@react-native-community/cli',
    '@react-native-windows/cli',
    '@react-native-windows/telemetry',
    'react',
    'react-native',
    'react-native-windows',
    'react-native-windows-init',
];
// These are NPM packages we care about, in terms of capturing versions used
exports.NuGetPackagesWeTrack = [
    'Microsoft.UI.Xaml',
    'Microsoft.Windows.CppWinRT',
    'Microsoft.WinUI',
];
/**
 * The Telemetry class is responsible for reporting telemetry for RNW CLI.
 */
class Telemetry {
    static getDefaultSetupString() {
        var _a;
        // Enable overriding the default setup string via an environment variable
        return (_a = process.env[ENV_SETUP_OVERRIDE]) !== null && _a !== void 0 ? _a : RNWSetupString;
    }
    static reset() {
        // Reset client
        if (Telemetry.client) {
            Telemetry.client.flush();
            Telemetry.client = undefined;
        }
        // Reset local members
        Telemetry.options = {
            setupString: Telemetry.getDefaultSetupString(),
            preserveErrorMessages: false,
            populateNpmPackageVersions: true,
        };
        Telemetry.commandInfo = {};
        Telemetry.versionsProp = {};
        Telemetry.projectProp = undefined;
    }
    static isEnabled() {
        return Telemetry.client !== undefined;
    }
    static getSessionId() {
        return basePropUtils.getSessionId();
    }
    /** Sets up the Telemetry static to be used elsewhere. */
    static async setup(options) {
        if (Telemetry.client) {
            // Bail since we've already setup
            return;
        }
        // Bail if we're in CI and not capturing CI
        if (!this.isTest && basePropUtils.isCI() && !basePropUtils.captureCI()) {
            return;
        }
        // Save off options for later
        Object.assign(Telemetry.options, options);
        Telemetry.setupClient();
        await Telemetry.setupBaseProperties();
        Telemetry.setupTelemetryProcessors();
    }
    /** Sets up Telemetry.client. */
    static setupClient() {
        var _a;
        appInsights.Configuration.setInternalLogging(false, false);
        Telemetry.client = new appInsights.TelemetryClient(Telemetry.options.setupString);
        // Allow overriding the proxy server via an environment variable
        const proxyServer = process.env[ENV_PROXY_OVERRIDE];
        if (proxyServer) {
            Telemetry.client.config.proxyHttpUrl = proxyServer;
            Telemetry.client.config.proxyHttpsUrl = proxyServer;
        }
        Telemetry.client.config.disableAppInsights = Telemetry.isTest;
        Telemetry.client.config.disableStatsbeat = true;
        // Despite trying to disable the statsbeat, it might still be running: https://github.com/microsoft/ApplicationInsights-node.js/issues/943
        // So we want to disable it, and despite the method's typing, getStatsbeat() _can_ return undefined
        (_a = Telemetry.client.getStatsbeat()) === null || _a === void 0 ? void 0 : _a.enable(false);
        Telemetry.client.channel.setUseDiskRetryCaching(!Telemetry.isTest);
    }
    /** Sets up any base properties that all telemetry events require. */
    static async setupBaseProperties() {
        Telemetry.client.commonProperties.deviceId =
            await basePropUtils.deviceId();
        Telemetry.client.commonProperties.deviceArchitecture =
            basePropUtils.deviceArchitecture();
        Telemetry.client.commonProperties.nodeArchitecture =
            basePropUtils.nodeArchitecture();
        Telemetry.client.commonProperties.devicePlatform =
            basePropUtils.devicePlatform();
        Telemetry.client.commonProperties.deviceLocale =
            await basePropUtils.deviceLocale();
        Telemetry.client.commonProperties.deviceNumCPUs = basePropUtils
            .deviceNumCPUs()
            .toString();
        Telemetry.client.commonProperties.deviceTotalMemory = basePropUtils
            .deviceTotalMemory()
            .toString();
        Telemetry.client.commonProperties.deviceDiskFreeSpace = basePropUtils
            .deviceDiskFreeSpace()
            .toString();
        Telemetry.client.commonProperties.ciCaptured = basePropUtils
            .captureCI()
            .toString();
        Telemetry.client.commonProperties.ciType = basePropUtils.ciType();
        Telemetry.client.commonProperties.isMsftInternal = basePropUtils
            .isMsftInternal()
            .toString();
        Telemetry.client.commonProperties.sampleRate = basePropUtils
            .sampleRate()
            .toString();
        Telemetry.client.commonProperties.isTest = Telemetry.isTest.toString();
        Telemetry.client.commonProperties.sessionId = Telemetry.getSessionId();
        Telemetry.client.config.samplingPercentage = basePropUtils.sampleRate();
        await Telemetry.populateToolsVersions();
        if (Telemetry.options.populateNpmPackageVersions) {
            await Telemetry.populateNpmPackageVersions();
        }
    }
    /** Sets up any telemetry processors. */
    static setupTelemetryProcessors() {
        Telemetry.client.addTelemetryProcessor(Telemetry.basicTelemetryProcessor);
        Telemetry.client.addTelemetryProcessor(Telemetry.errorTelemetryProcessor);
    }
    /**
     * Performs the processing necessary (mostly PII sanitization) for all events.
     * @param envelope The ApplicationInsights event envelope.
     * @param _contextObjects An optional context object.
     * @returns Whether to kee
     */
    static basicTelemetryProcessor(envelope, _contextObjects) {
        var _a;
        delete envelope.tags['ai.cloud.roleInstance'];
        // Filter out "legacy" events from older stable branches
        const properties = (_a = envelope.data.baseData) === null || _a === void 0 ? void 0 : _a.properties;
        if ((properties === null || properties === void 0 ? void 0 : properties.eventName) &&
            exports.EventNamesWeTrack.includes(properties.eventName)) {
            return true;
        }
        return false;
    }
    /**
     * Performs the processing necessary (mostly PII sanitization) for error events.
     * @param envelope
     * @param _contextObjects
     * @returns
     */
    static errorTelemetryProcessor(envelope, _contextObjects) {
        if (envelope.data.baseType === 'ExceptionData') {
            const data = envelope.data.baseData;
            if (data === null || data === void 0 ? void 0 : data.exceptions) {
                for (const exception of data.exceptions) {
                    for (const frame of exception.parsedStack) {
                        errorUtils.sanitizeErrorStackFrame(frame);
                    }
                    // Exception message must never be blank, or AI will reject it
                    exception.message = exception.message || '[None]';
                    // CodedError has non-PII information in its 'type' member, plus optionally some more info in its 'data'.
                    // The message may contain PII information. This can be sanitized, but for now delete it.
                    // Note that the type of data.exceptions[0] is always going to be ExceptionDetails. It is not the original thrown exception.
                    // https://github.com/microsoft/ApplicationInsights-node.js/issues/707
                    if (Telemetry.options.preserveErrorMessages) {
                        exception.message = errorUtils.sanitizeErrorMessage(exception.message);
                    }
                    else {
                        exception.message = '[Removed]';
                    }
                }
            }
        }
        return true;
    }
    /** Tries to update the version of the named package/tool by calling getValue(). */
    static async tryUpdateVersionsProp(name, getValue, forceRefresh) {
        if (!Telemetry.client) {
            return true;
        }
        if (forceRefresh === true || !Telemetry.versionsProp[name]) {
            const value = await getValue();
            if (value) {
                Telemetry.versionsProp[name] = value;
                return true;
            }
        }
        return false;
    }
    /** Populates the versions property of tools we care to track. */
    static async populateToolsVersions(refresh) {
        await Telemetry.tryUpdateVersionsProp('node', versionUtils.getNodeVersion, refresh);
        await Telemetry.tryUpdateVersionsProp('npm', versionUtils.getNpmVersion, refresh);
        await Telemetry.tryUpdateVersionsProp('yarn', versionUtils.getYarnVersion, refresh);
        await Telemetry.tryUpdateVersionsProp('VisualStudio', versionUtils.getVisualStudioVersion, refresh);
    }
    /** Populates the versions property of npm packages we care to track. */
    static async populateNpmPackageVersions(refresh) {
        for (const npmPackage of exports.NpmPackagesWeTrack) {
            await Telemetry.tryUpdateVersionsProp(npmPackage, async () => await versionUtils.getVersionOfNpmPackage(npmPackage), refresh);
        }
    }
    /** Populates the versions property of nuget packages we care to track. */
    static async populateNuGetPackageVersions(projectFile, refresh) {
        const nugetVersions = await versionUtils.getVersionsOfNuGetPackages(projectFile, exports.NuGetPackagesWeTrack);
        for (const nugetPackage of exports.NuGetPackagesWeTrack) {
            await Telemetry.tryUpdateVersionsProp(nugetPackage, async () => nugetVersions[nugetPackage], refresh);
        }
    }
    static setProjectInfo(info) {
        if (!Telemetry.client) {
            return;
        }
        Telemetry.projectProp = info;
    }
    static startCommand(info) {
        if (!Telemetry.client) {
            return;
        }
        if (Telemetry.commandInfo.startInfo) {
            return;
        }
        Telemetry.commandInfo.startTime = Date.now();
        Telemetry.commandInfo.startInfo = info;
        // Set common command props
        Telemetry.client.commonProperties.commandName = info.commandName;
    }
    static endCommand(info, extraProps) {
        if (!Telemetry.client) {
            return;
        }
        if (!Telemetry.commandInfo.startInfo) {
            return;
        }
        Telemetry.commandInfo.endTime = Date.now();
        Telemetry.commandInfo.endInfo = info;
        Telemetry.trackCommandEvent(extraProps);
    }
    static trackCommandEvent(extraProps) {
        var _a, _b, _c, _d;
        const props = {
            eventName: exports.CommandEventName,
        };
        // Set command props
        props.command = {
            options: (_a = Telemetry.commandInfo.startInfo) === null || _a === void 0 ? void 0 : _a.options,
            defaultOptions: (_b = Telemetry.commandInfo.startInfo) === null || _b === void 0 ? void 0 : _b.defaultOptions,
            args: (_c = Telemetry.commandInfo.startInfo) === null || _c === void 0 ? void 0 : _c.args,
            durationInSecs: (Telemetry.commandInfo.endTime - Telemetry.commandInfo.startTime) /
                1000,
            resultCode: (_d = Telemetry.commandInfo.endInfo) === null || _d === void 0 ? void 0 : _d.resultCode,
        };
        // Set remaining common props
        props.project = Telemetry.projectProp;
        props.versions = Telemetry.versionsProp;
        // Set extra props
        props.extraProps = {};
        Object.assign(props.extraProps, extraProps);
        // Fire event
        Telemetry.client.trackEvent({ name: props.eventName, properties: props });
        Telemetry.client.flush();
    }
    static trackException(error, extraProps) {
        var _a, _b;
        if (!Telemetry.client) {
            return;
        }
        const props = {
            eventName: exports.CodedErrorEventName,
        };
        // Save off CodedError info
        const codedError = error instanceof errorUtils.CodedError
            ? error
            : null;
        props.codedError = {
            type: (_a = codedError === null || codedError === void 0 ? void 0 : codedError.type) !== null && _a !== void 0 ? _a : 'Unknown',
            data: (_b = codedError === null || codedError === void 0 ? void 0 : codedError.data) !== null && _b !== void 0 ? _b : {},
        };
        // Copy msBuildErrorMessages into the codedError.data object
        if (error.msBuildErrorMessages) {
            // Always grab MSBuild error codes if possible
            props.codedError.data.msBuildErrors = error.msBuildErrorMessages
                .map(errorUtils.tryGetErrorCode)
                .filter((msg) => msg);
            // Grab sanitized MSBuild error messages if we're preserving them
            if (Telemetry.options.preserveErrorMessages) {
                props.codedError.data.msBuildErrorMessages = error.msBuildErrorMessages
                    .map(errorUtils.sanitizeErrorMessage)
                    .filter((msg) => msg);
            }
        }
        // Copy miscellaneous system error fields into the codedError.data object
        const syscallExceptionFieldsToCopy = ['errno', 'syscall', 'code'];
        for (const f of syscallExceptionFieldsToCopy) {
            if (error[f]) {
                props.codedError.data[f] = error[f];
            }
        }
        // Set remaining common props
        props.project = Telemetry.projectProp;
        props.versions = Telemetry.versionsProp;
        // Set extra props
        props.extraProps = {};
        Object.assign(props.extraProps, extraProps);
        // Fire event
        Telemetry.client.trackException({
            exception: error,
            properties: props,
        });
        Telemetry.client.flush();
    }
}
exports.Telemetry = Telemetry;
Telemetry.client = undefined;
Telemetry.options = {
    setupString: Telemetry.getDefaultSetupString(),
    preserveErrorMessages: false,
    populateNpmPackageVersions: true,
};
Telemetry.isTest = basePropUtils.isCliTest();
Telemetry.commandInfo = {};
Telemetry.versionsProp = {};
Telemetry.projectProp = undefined;
//# sourceMappingURL=telemetry.js.map