"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.endTelemetrySession = exports.startTelemetrySession = exports.getDefaultOptions = void 0;
const lodash_1 = require("lodash");
const telemetry_1 = require("@react-native-windows/telemetry");
const commandWithProgress_1 = require("./commandWithProgress");
/**
 * Calculates a the default values of a given react-native CLI command's options.
 * @param config Config passed from react-native CLI.
 * @param commandOptions The options definition for the command.
 * @returns The default options for the command.
 */
function getDefaultOptions(config, commandOptions) {
    const result = {
        logging: false,
    };
    for (const option of commandOptions) {
        let key = option.name.startsWith('--')
            ? option.name.slice('--'.length)
            : option.name;
        key = key.trim();
        let value = option.default;
        if (value !== undefined && value.name) {
            value = value(config);
        }
        if (key.endsWith(' [string]')) {
            // Option is a string
            key = key.slice(0, -' [string]'.length);
            // Option is a number
        }
        else if (key.endsWith(' [number]')) {
            key = key.slice(0, -' [number]'.length);
        }
        else {
            // Option is a boolean
            // Note: Commander ignores the default property for booleans
            if (key.startsWith('no-')) {
                key = key.slice('no-'.length);
                // Commander always defaults to true for flags that start with --no
                value = true;
            }
            else {
                // Commander always defaults to undefined for flags
                value = undefined;
            }
        }
        key = (0, lodash_1.camelCase)(key);
        result[key] = value;
    }
    return result;
}
exports.getDefaultOptions = getDefaultOptions;
/**
 * Sets up and starts the telemetry gathering for the CLI command.
 * @param commandName The name of the CLI command.
 * @param config Config passed from react-native CLI.
 * @param options Options passed from react-native CLI.
 * @param defaultOptions Default options for the command.
 * @param optionSanitizer Function to sanitize the option values for telemetry.
 */
async function startTelemetrySession(commandName, config, options, defaultOptions, optionSanitizer) {
    const verbose = options.logging === true;
    if (!options.telemetry) {
        if (verbose) {
            (0, commandWithProgress_1.newInfo)('Telemetry is disabled');
        }
        return;
    }
    if (verbose) {
        (0, commandWithProgress_1.newInfo)(`Running ${(0, telemetry_1.nodeArchitecture)()} node on a ${(0, telemetry_1.deviceArchitecture)()} machine`);
    }
    if ((0, telemetry_1.deviceArchitecture)() !== (0, telemetry_1.nodeArchitecture)()) {
        (0, commandWithProgress_1.newWarn)('This version of node was built for a different architecture than this machine and may cause unintended behavior');
    }
    await telemetry_1.Telemetry.setup();
    const sanitizedOptions = (0, telemetry_1.commanderOptionsToOptions)(options, optionSanitizer);
    const sanitizedDefaultOptions = (0, telemetry_1.commanderOptionsToOptions)(defaultOptions, optionSanitizer);
    const sanitizedArgs = (0, telemetry_1.optionsToArgs)(sanitizedOptions, process.argv);
    const startInfo = {
        commandName,
        args: sanitizedArgs,
        options: sanitizedOptions,
        defaultOptions: sanitizedDefaultOptions,
    };
    telemetry_1.Telemetry.startCommand(startInfo);
    const projectInfo = await (0, telemetry_1.configToProjectInfo)(config);
    if (projectInfo) {
        telemetry_1.Telemetry.setProjectInfo(projectInfo);
    }
    const projectFile = (0, telemetry_1.getProjectFileFromConfig)(config);
    if (projectFile) {
        await telemetry_1.Telemetry.populateNuGetPackageVersions(projectFile);
    }
}
exports.startTelemetrySession = startTelemetrySession;
/**
 * Ends the gathering of telemetry for the CLI command.
 * @param error The error (if any) thrown during the command.
 * @param getExtraProps Function to get any extra command-specific telemetry properties.
 */
async function endTelemetrySession(error, getExtraProps) {
    if (!telemetry_1.Telemetry.isEnabled()) {
        // Bail early so don't waste time here
        return;
    }
    const endInfo = {
        resultCode: 'Success',
    };
    if (error) {
        endInfo.resultCode =
            error instanceof telemetry_1.CodedError ? error.type : 'Unknown';
    }
    telemetry_1.Telemetry.endCommand(endInfo, getExtraProps ? await getExtraProps() : undefined);
}
exports.endTelemetrySession = endTelemetrySession;
//# sourceMappingURL=telemetryHelpers.js.map