"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.optionsToArgs = exports.argsContainsOption = exports.commanderOptionsToOptions = exports.yargsOptionsToOptions = exports.commanderNameToOptionName = void 0;
const lodash_1 = require("lodash");
/**
 * Converts a raw Commander arg name ('--no-bad-dog') into an options property name ('badDog').
 * @param name The raw Commander arg name.
 * @returns The options name.
 */
function commanderNameToOptionName(name) {
    if (name.startsWith('--')) {
        name = name.slice('--'.length);
    }
    if (name.startsWith('no-')) {
        name = name.slice('no-'.length);
    }
    if (name.endsWith(' [string]')) {
        name = name.slice(0, name.length - ' [string]'.length);
    }
    if (name.endsWith(' [number]')) {
        name = name.slice(0, name.length - ' [number]'.length);
    }
    return (0, lodash_1.camelCase)(name);
}
exports.commanderNameToOptionName = commanderNameToOptionName;
/**
 * Converts a yargs-parsed options object into an options object suitable for telemetry.
 * @param options The options object as parsed by yargs.
 * @param sanitizer Function to sanitize the option values for telemetry.
 * @returns An options object suitable for telemetry.
 */
function yargsOptionsToOptions(options, sanitizer) {
    const result = {};
    for (const key of Object.keys(options)) {
        if (!(key in Object.prototype) &&
            key !== '$0' &&
            key !== '_' &&
            key.includes('-') !== (0, lodash_1.camelCase)(key) in options) {
            result[key] = sanitizer ? sanitizer(key, options[key]) : options[key];
        }
    }
    return result;
}
exports.yargsOptionsToOptions = yargsOptionsToOptions;
/**
 * Converts a Commander-parsed options object into an options object suitable for telemetry.
 * @param options The options object as parsed by Commander.
 * @param sanitizer Function to sanitize the option values for telemetry.
 * @returns An options object suitable for telemetry.
 */
function commanderOptionsToOptions(options, sanitizer) {
    const result = {};
    for (const key of Object.keys(options)) {
        if (!(key in Object.prototype)) {
            result[key] = sanitizer ? sanitizer(key, options[key]) : options[key];
        }
    }
    return result;
}
exports.commanderOptionsToOptions = commanderOptionsToOptions;
/**
 * Checks whether the given command-line args contains the given option key.
 * @param key The option key to look for.
 * @param argv The raw command-line args.
 * @returns Whether the given command-line args contains the given option key.
 */
function argsContainsOption(key, argv) {
    const kebabKey = (0, lodash_1.kebabCase)(key);
    return (argv.includes(`--${key}`) ||
        argv.includes(`--no-${key}`) ||
        argv.includes(`--${kebabKey}`) ||
        argv.includes(`--no-${kebabKey}`));
}
exports.argsContainsOption = argsContainsOption;
/**
 * Filters down a given set of options to only include those that were present in the command-line args.
 * @param options The full set of options.
 * @param argv The raw command-line args.
 * @returns The filtered set of options.
 */
function optionsToArgs(options, argv) {
    const result = {};
    for (const key of Object.keys(options)) {
        if (!(key in Object.prototype) && argsContainsOption(key, argv)) {
            result[key] = options[key];
        }
    }
    return result;
}
exports.optionsToArgs = optionsToArgs;
//# sourceMappingURL=optionUtils.js.map