"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.setExitProcessWithError = exports.newInfo = exports.newSuccess = exports.newWarn = exports.newError = exports.commandWithProgress = exports.runPowerShellScriptFunction = exports.powershell = exports.newSpinner = void 0;
const child_process_1 = require("child_process");
const ora_1 = __importDefault(require("ora"));
const cli_spinners_1 = __importDefault(require("cli-spinners"));
const chalk_1 = __importDefault(require("chalk"));
const telemetry_1 = require("@react-native-windows/telemetry");
function setSpinnerText(spinner, prefix, text) {
    text = prefix + spinnerString(text);
    spinner.text = text.length > 45 ? text.slice(0, 110) + '...' : text;
}
function spinnerString(msg) {
    msg = msg.trim();
    const lastLineIndex = msg.lastIndexOf('\n');
    if (lastLineIndex > 0) {
        msg = msg.slice(lastLineIndex + 1);
    }
    msg = msg.trim();
    return msg;
}
function newSpinner(text) {
    const options = {
        text: text,
        indent: 1,
        // ora turns off emoji rendering by default on windows, since the default terminal
        // doesn't render them.  Turn them back on when running in the new windows terminal
        ...(process.env.WT_SESSION && { spinner: cli_spinners_1.default.dots }),
        // By default, ora process.stderr as the output stream, however,the VS Code debugger
        // Uses stdout to match success patterns
        stream: process.stdout,
    };
    return (0, ora_1.default)(options).start();
}
exports.newSpinner = newSpinner;
exports.powershell = `${process.env.SystemRoot}\\System32\\WindowsPowerShell\\v1.0\\powershell.exe`;
async function runPowerShellScriptFunction(taskDescription, script, funcName, verbose, errorCategory) {
    try {
        const printException = verbose ? '$_;' : '';
        const importScript = script ? `Import-Module "${script}"; ` : '';
        await commandWithProgress(newSpinner(taskDescription), taskDescription, exports.powershell, [
            '-NoProfile',
            '-ExecutionPolicy',
            'RemoteSigned',
            `${importScript}try { ${funcName} -ErrorAction Stop; $lec = $LASTEXITCODE; } catch { $lec = 1; ${printException} }; exit $lec`,
        ], verbose, errorCategory);
    }
    catch (_a) {
        // The error output from the process will be shown if verbose is set.
        // We don't capture the process output if verbose is set, but at least we have the task name in text, so throw that.
        throw new telemetry_1.CodedError(errorCategory, taskDescription);
    }
}
exports.runPowerShellScriptFunction = runPowerShellScriptFunction;
function commandWithProgress(spinner, taskDoingName, command, args, verbose, errorCategory) {
    return new Promise((resolve, reject) => {
        const spawnOptions = verbose ? { stdio: 'inherit' } : {};
        if (verbose) {
            spinner.stop();
        }
        const cp = (0, child_process_1.spawn)(command, args, spawnOptions);
        let firstErrorLine = null;
        if (!verbose) {
            cp.stdout.on('data', chunk => {
                const text = chunk.toString();
                setSpinnerText(spinner, taskDoingName + ': ', text);
            });
            cp.stderr.on('data', chunk => {
                const text = chunk.toString();
                if (!firstErrorLine) {
                    firstErrorLine = text;
                }
                if (text) {
                    setSpinnerText(spinner, taskDoingName + ': ERROR: ', firstErrorLine);
                }
                reject(new telemetry_1.CodedError(errorCategory, firstErrorLine, {
                    taskName: taskDoingName,
                }));
            });
        }
        cp.on('error', e => {
            if (verbose) {
                console.error(chalk_1.default.red(e.toString()));
            }
            spinner.fail(e.toString());
            const ce = new telemetry_1.CodedError(errorCategory, e.message);
            ce.stack = e.stack;
            reject(ce);
        }).on('close', code => {
            if (code === 0) {
                spinner.succeed(taskDoingName);
                resolve();
            }
            else {
                spinner.fail();
                reject(new telemetry_1.CodedError(errorCategory, `${taskDoingName} - exit error code ${code}`, {
                    taskName: taskDoingName,
                    taskExitCode: code,
                }));
            }
        });
    });
}
exports.commandWithProgress = commandWithProgress;
function newError(text) {
    newSpinner(text).fail(text);
}
exports.newError = newError;
function newWarn(text) {
    newSpinner(text).warn(text);
}
exports.newWarn = newWarn;
function newSuccess(text) {
    newSpinner(text).succeed(text);
}
exports.newSuccess = newSuccess;
function newInfo(text) {
    newSpinner(text).info(text);
}
exports.newInfo = newInfo;
/**
 * Sets the process exit code and offers some information at the end of a CLI command.
 * @param loggingIsEnabled Is verbose logging enabled.
 * @param error The error caught during the process, if any.
 */
function setExitProcessWithError(loggingIsEnabled, error) {
    if (error) {
        const errorType = error instanceof telemetry_1.CodedError ? error.type : 'Unknown';
        process.exitCode = telemetry_1.CodedErrors[errorType];
        if (loggingIsEnabled) {
            console.log(`Command failed with error ${chalk_1.default.bold(errorType)}: ${error.message}`);
            if (telemetry_1.Telemetry.isEnabled()) {
                console.log(`Your telemetry sessionId was ${chalk_1.default.bold(telemetry_1.Telemetry.getSessionId())}`);
            }
        }
        else {
            console.log(`Command failed. Re-run the command with ${chalk_1.default.bold('--logging')} for more information.`);
        }
    }
}
exports.setExitProcessWithError = setExitProcessWithError;
//# sourceMappingURL=commandWithProgress.js.map