"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.copyAndReplaceAll = exports.copyAndReplaceWithChangedCallback = exports.createDir = exports.copyAndReplace = exports.resolveContents = void 0;
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const chalk_1 = __importDefault(require("chalk"));
const prompts_1 = __importDefault(require("prompts"));
const path_1 = __importDefault(require("path"));
const mustache_1 = __importDefault(require("mustache"));
const telemetry_1 = require("@react-native-windows/telemetry");
function walk(current) {
    if (!fs_1.default.lstatSync(current).isDirectory()) {
        return [current];
    }
    const files = fs_1.default
        .readdirSync(current)
        .map(child => walk(path_1.default.join(current, child)));
    const result = [];
    return result.concat.apply([current], files);
}
/**
 * Get a source file and replace parts of its contents.
 * @param srcPath Path to the source file.
 * @param replacements e.g. {'TextToBeReplaced': 'Replacement'}
 * @return The contents of the file with the replacements applied.
 */
function resolveContents(srcPath, replacements) {
    let content = fs_1.default.readFileSync(srcPath, 'utf8');
    if (content.includes('\r\n')) {
        // CRLF file, make sure multiline replacements are also CRLF
        for (const key of Object.keys(replacements)) {
            if (typeof replacements[key] === 'string') {
                replacements[key] = replacements[key].replace(/(?<!\r)\n/g, '\r\n');
            }
        }
    }
    else {
        // LF file, make sure multiline replacements are also LF
        for (const key of Object.keys(replacements)) {
            if (typeof replacements[key] === 'string') {
                replacements[key] = replacements[key].replace(/\r\n/g, '\n');
            }
        }
    }
    if (replacements.useMustache) {
        content = mustache_1.default.render(content, replacements);
        (replacements.regExpPatternsToRemove || []).forEach(regexPattern => {
            content = content.replace(new RegExp(regexPattern, 'g'), '');
        });
    }
    else {
        Object.keys(replacements).forEach(regex => {
            content = content.replace(new RegExp(regex, 'g'), replacements[regex]);
        });
    }
    return content;
}
exports.resolveContents = resolveContents;
// Binary files, don't process these (avoid decoding as utf8)
const binaryExtensions = ['.png', '.jar', '.keystore', '.ico', '.rc'];
/**
 * Copy a file to given destination, replacing parts of its contents.
 * @param srcPath Path to a file to be copied.
 * @param destPath Destination path.
 * @param replacements: e.g. {'TextToBeReplaced': 'Replacement'}
 * @param contentChangedCallback
 *        Used when upgrading projects. Based on if file contents would change
 *        when being replaced, allows the caller to specify whether the file
 *        should be replaced or not.
 *        If null, files will be overwritten.
 *        Function(path, 'identical' | 'changed' | 'new') => 'keep' | 'overwrite'
 */
async function copyAndReplace(srcPath, destPath, replacements, contentChangedCallback) {
    if (fs_1.default.lstatSync(srcPath).isDirectory()) {
        if (!fs_1.default.existsSync(destPath)) {
            fs_1.default.mkdirSync(destPath);
        }
        // Not recursive
        return;
    }
    const extension = path_1.default.extname(srcPath);
    if (binaryExtensions.includes(extension)) {
        // Binary file
        let shouldOverwrite = 'overwrite';
        if (contentChangedCallback) {
            const newContentBuffer = fs_1.default.readFileSync(srcPath);
            let contentChanged = 'identical';
            try {
                const origContentBuffer = fs_1.default.readFileSync(destPath);
                if (Buffer.compare(origContentBuffer, newContentBuffer) !== 0) {
                    contentChanged = 'changed';
                }
            }
            catch (err) {
                if (err.code === 'ENOENT') {
                    contentChanged = 'new';
                }
                else {
                    throw err;
                }
            }
            shouldOverwrite = await contentChangedCallback(destPath, contentChanged);
        }
        if (shouldOverwrite === 'overwrite') {
            copyBinaryFile(srcPath, destPath, err => {
                if (err) {
                    throw err;
                }
            });
        }
    }
    else {
        // Text file
        const srcPermissions = fs_1.default.statSync(srcPath).mode;
        const content = resolveContents(srcPath, replacements);
        let shouldOverwrite = 'overwrite';
        if (contentChangedCallback) {
            // Check if contents changed and ask to overwrite
            let contentChanged = 'identical';
            try {
                const origContent = fs_1.default.readFileSync(destPath, 'utf8');
                if (content !== origContent) {
                    // logger.info('Content changed: ' + destPath);
                    contentChanged = 'changed';
                }
            }
            catch (err) {
                if (err.code === 'ENOENT') {
                    contentChanged = 'new';
                }
                else {
                    throw err;
                }
            }
            shouldOverwrite = await contentChangedCallback(destPath, contentChanged);
        }
        if (shouldOverwrite === 'overwrite') {
            fs_1.default.writeFileSync(destPath, content, {
                encoding: 'utf8',
                mode: srcPermissions,
            });
        }
    }
}
exports.copyAndReplace = copyAndReplace;
/**
 * Same as 'cp' on Unix. Don't do any replacements.
 */
function copyBinaryFile(srcPath, destPath, cb) {
    let cbCalled = false;
    const srcPermissions = fs_1.default.statSync(srcPath).mode;
    const readStream = fs_1.default.createReadStream(srcPath);
    readStream.on('error', err => {
        done(err);
    });
    const writeStream = fs_1.default.createWriteStream(destPath, {
        mode: srcPermissions,
    });
    writeStream.on('error', err => {
        done(err);
    });
    writeStream.on('close', () => {
        done();
    });
    readStream.pipe(writeStream);
    function done(err) {
        if (!cbCalled) {
            cb(err);
            cbCalled = true;
        }
    }
}
function createDir(destPath) {
    if (!fs_1.default.existsSync(destPath)) {
        fs_1.default.mkdirSync(destPath);
    }
}
exports.createDir = createDir;
async function copyAndReplaceWithChangedCallback(srcPath, destRoot, relativeDestPath, replacements, alwaysOverwrite) {
    if (!replacements) {
        replacements = {};
    }
    const contentChangedCallback = alwaysOverwrite
        ? (_, contentChanged) => alwaysOverwriteContentChangedCallback(srcPath, relativeDestPath, contentChanged)
        : (_, contentChanged) => upgradeFileContentChangedCallback(srcPath, relativeDestPath, contentChanged);
    await copyAndReplace(srcPath, path_1.default.join(destRoot, relativeDestPath), replacements, contentChangedCallback);
}
exports.copyAndReplaceWithChangedCallback = copyAndReplaceWithChangedCallback;
async function copyAndReplaceAll(srcPath, destPath, relativeDestDir, replacements, alwaysOverwrite) {
    for (const absoluteSrcFilePath of walk(srcPath)) {
        const filename = path_1.default.relative(srcPath, absoluteSrcFilePath);
        const relativeDestPath = path_1.default.join(relativeDestDir, filename);
        await copyAndReplaceWithChangedCallback(absoluteSrcFilePath, destPath, relativeDestPath, replacements, alwaysOverwrite);
    }
}
exports.copyAndReplaceAll = copyAndReplaceAll;
async function alwaysOverwriteContentChangedCallback(absoluteSrcFilePath, relativeDestPath, contentChanged) {
    if (contentChanged === 'new') {
        console.log(`${chalk_1.default.bold('new')} ${relativeDestPath}`);
        return 'overwrite';
    }
    if (contentChanged === 'changed') {
        console.log(`${chalk_1.default.bold('changed')} ${relativeDestPath} ${chalk_1.default.yellow('[overwriting]')}`);
        return 'overwrite';
    }
    if (contentChanged === 'identical') {
        return 'keep';
    }
    throw new telemetry_1.CodedError('Autolinking', `Unknown file changed state: ${relativeDestPath}, ${contentChanged}`);
}
async function upgradeFileContentChangedCallback(absoluteSrcFilePath, relativeDestPath, contentChanged) {
    if (contentChanged === 'new') {
        console.log(`${chalk_1.default.bold('new')} ${relativeDestPath}`);
        return 'overwrite';
    }
    if (contentChanged === 'changed') {
        console.log(`${chalk_1.default.bold(relativeDestPath)} ` +
            `has changed in the new version.\nDo you want to keep your ${relativeDestPath} or replace it with the ` +
            'latest version?\nMake sure you have any changes you made to this file saved somewhere.\n' +
            `You can see the new version here: ${absoluteSrcFilePath}`);
        const { shouldReplace } = await (0, prompts_1.default)([
            {
                name: 'shouldReplace',
                type: 'confirm',
                message: `Do you want to replace ${relativeDestPath}?`,
                initial: false,
            },
        ]);
        return shouldReplace ? 'overwrite' : 'keep';
    }
    if (contentChanged === 'identical') {
        return 'keep';
    }
    throw new telemetry_1.CodedError('Autolinking', `Unknown file changed state: ${relativeDestPath}, ${contentChanged}`);
}
//# sourceMappingURL=index.js.map