"use strict";
/**
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const projectConfig_1 = require("../commands/config/projectConfig");
const autolinkWindows_1 = require("../commands/autolinkWindows/autolinkWindows");
const projectConfig_utils_1 = require("./projectConfig.utils");
function project(name, setup) {
    return [name, setup];
}
const projects = [
    // Nothing but a react-native.config.js, with nulls
    project('BlankApp'),
    // Nothing but a windows folder
    project('MissingProjectFilesApp', async (folder) => {
        const windowsDir = path_1.default.join(folder, 'windows');
        await (0, projectConfig_utils_1.tryMkdir)(windowsDir);
    }),
    // New C++ project based on the template
    project('SimpleCppApp', async (folder) => {
        await (0, projectConfig_utils_1.ensureCppAppProject)(folder, 'SimpleCppApp');
    }),
    // New C# project based on the template
    project('SimpleCSharpApp', async (folder) => {
        await (0, projectConfig_utils_1.ensureCSharpAppProject)(folder, 'SimpleCSharpApp');
    }),
    project('WithWinUI3', async (folder) => {
        await (0, projectConfig_utils_1.ensureWinUI3Project)(folder);
    }),
    project('WithHermes', async (folder) => {
        await (0, projectConfig_utils_1.ensureHermesProject)(folder);
    }),
    project('WithExperimentalNuget', async (folder) => {
        await (0, projectConfig_utils_1.ensureExperimentalNuGetProject)(folder);
    }),
    project('WithIndirectDependency'),
];
// Tests that given userConfig is null, the result will always be null
test.each(projects)('projectConfig - %s (userConfig is null)', async (name, setup) => {
    const folder = path_1.default.resolve('src/e2etest/projects/', name);
    if (setup !== undefined) {
        await setup(folder);
    }
    const userConfig = null;
    expect((0, projectConfig_1.projectConfigWindows)(folder, userConfig)).toBeNull();
});
// Tests the result given a windows project config in react-native.config.js
test.each(projects)('projectConfig - %s (Use react-native.config.js)', async (name, setup) => {
    const folder = path_1.default.resolve('src/e2etest/projects/', name);
    const rnc = require(path_1.default.join(folder, 'react-native.config.js'));
    if (setup !== undefined) {
        await setup(folder);
    }
    const userConfig = rnc.project.windows;
    if (name === 'BlankApp') {
        expect((0, projectConfig_1.projectConfigWindows)(folder, userConfig)).toMatchSnapshot();
    }
    else {
        expect((0, projectConfig_1.projectConfigWindows)(folder, userConfig)).toMatchSnapshot({
            folder: expect.stringContaining(name),
        });
    }
});
// Tests the result of ignoring the windows project config in react-native.config.js
test.each(projects)('projectConfig - %s (Ignore react-native.config.js)', async (name, setup) => {
    const folder = path_1.default.resolve('src/e2etest/projects/', name);
    if (setup !== undefined) {
        await setup(folder);
    }
    const userConfig = {};
    if (name === 'BlankApp') {
        expect((0, projectConfig_1.projectConfigWindows)(folder, userConfig)).toMatchSnapshot();
    }
    else {
        expect((0, projectConfig_1.projectConfigWindows)(folder, userConfig)).toMatchSnapshot({
            folder: expect.stringContaining(name),
        });
    }
});
test('useWinUI3=true in react-native.config.js, UseWinUI3=false in ExperimentalFeatures.props', async () => {
    const folder = path_1.default.resolve('src/e2etest/projects/WithWinUI3');
    // Create project with UseWinUI3 == false in ExperimentalFeatures.props
    await (0, projectConfig_utils_1.ensureCppAppProject)(folder, 'WithWinUI3', false, false, false);
    const rnc = require(path_1.default.join(folder, 'react-native.config.js'));
    const config = (0, projectConfig_1.projectConfigWindows)(folder, rnc.project.windows);
    // Set useWinUI3=true in react-native.config.js
    config.useWinUI3 = true;
    const experimentalFeaturesPropsFile = path_1.default.join(folder, 'windows/ExperimentalFeatures.props');
    // Verify starting props file
    const startingExperimentalFeatures = (await fs_1.default.readFile(experimentalFeaturesPropsFile)).toString();
    expect(startingExperimentalFeatures.replace(/\r/g, '')).toMatchSnapshot();
    // Run Autolink to sync the files
    const al = new autolinkWindows_1.AutoLinkWindows({ windows: config }, {}, {
        check: false,
        logging: false,
    });
    const exd = await al.ensureXAMLDialect();
    expect(exd).toBeTruthy();
    // Verify ending props file
    const finalExperimentalFeatures = (await fs_1.default.readFile(experimentalFeaturesPropsFile)).toString();
    expect(finalExperimentalFeatures.replace(/\r/g, '')).toMatchSnapshot();
});
//# sourceMappingURL=projectConfig.test.js.map