"use strict";
/**
 * Copyright (c) Microsoft Corporation.
 * Licensed under the MIT License.
 * @format
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getExperimentalFeatures = exports.getProjectGuid = exports.getProjectNamespace = exports.getProjectName = exports.getProjectType = exports.getOutputType = exports.getConfigurationType = exports.importProjectExists = exports.findPropertyValue = exports.tryFindPropertyValueAsBoolean = exports.tryFindPropertyValue = exports.readProjectFile = exports.getProjectLanguage = exports.findAppProjectFiles = exports.findDependencyProjectFiles = exports.isRnwDependencyProject = exports.findSolutionFiles = exports.isRnwSolution = exports.findWindowsFolder = exports.findFiles = void 0;
const fs_1 = __importDefault(require("@react-native-windows/fs"));
const path_1 = __importDefault(require("path"));
const glob_1 = __importDefault(require("glob"));
const xmldom_1 = require("@xmldom/xmldom");
const xpath_1 = __importDefault(require("xpath"));
const telemetry_1 = require("@react-native-windows/telemetry");
const msbuildSelect = xpath_1.default.useNamespaces({
    msbuild: 'http://schemas.microsoft.com/developer/msbuild/2003',
});
/**
 * Search for files matching the pattern under the target folder.
 * @param folder The absolute path to target folder.
 * @param filenamePattern The pattern to search for.
 * @return  Return the array of relative file paths.
 */
function findFiles(folder, filenamePattern) {
    const files = glob_1.default.sync(path_1.default.join('**', filenamePattern), {
        cwd: folder,
        ignore: [
            'node_modules/**',
            '**/Debug/**',
            '**/Release/**',
            '**/Generated Files/**',
            '**/packages/**',
        ],
    });
    return files;
}
exports.findFiles = findFiles;
/**
 * Search for the windows sub-folder under the target folder.
 * @param folder The absolute path to the target folder.
 * @return The absolute path to the windows folder, if it exists.
 */
function findWindowsFolder(folder) {
    const winDir = 'windows';
    const joinedDir = path_1.default.join(folder, winDir);
    if (fs_1.default.existsSync(joinedDir)) {
        return joinedDir;
    }
    return null;
}
exports.findWindowsFolder = findWindowsFolder;
/**
 * Checks if the target file path is a RNW solution file by checking if it contains the string "ReactNative".
 * @param filePath The absolute file path to check.
 * @return Whether the path is to a RNW solution file.
 */
function isRnwSolution(filePath) {
    return (fs_1.default
        .readFileSync(filePath)
        .toString()
        .search(/ReactNative/) > 0);
}
exports.isRnwSolution = isRnwSolution;
/**
 * Search for the RNW solution files under the target folder.
 * @param winFolder The absolute path to target folder.
 * @return Return the array of relative file paths.
 */
function findSolutionFiles(winFolder) {
    // First search for all potential solution files
    const allSolutions = findFiles(winFolder, '*.sln');
    if (allSolutions.length === 0) {
        // If there're no solution files, return 0
        return [];
    }
    else if (allSolutions.length === 1) {
        // If there is exactly one solution file, assume it's it
        return [allSolutions[0]];
    }
    const solutionFiles = [];
    // Try to find any solution file that appears to be a React Native solution
    for (const solutionFile of allSolutions) {
        if (isRnwSolution(path_1.default.join(winFolder, solutionFile))) {
            solutionFiles.push(path_1.default.normalize(solutionFile));
        }
    }
    return solutionFiles;
}
exports.findSolutionFiles = findSolutionFiles;
/**
 * Checks if the target file path is a RNW lib project file.
 * @param filePath The absolute file path to check.
 * @return Whether the path is to a RNW lib project file.
 */
function isRnwDependencyProject(filePath) {
    const projectContents = readProjectFile(filePath);
    const projectLang = getProjectLanguage(filePath);
    if (projectLang === 'cs') {
        return importProjectExists(projectContents, 'Microsoft.ReactNative.Uwp.CSharpLib.targets');
    }
    else if (projectLang === 'cpp') {
        return (importProjectExists(projectContents, 'Microsoft.ReactNative.Uwp.CppLib.targets') ||
            importProjectExists(projectContents, 'Microsoft.ReactNative.Composition.CppLib.targets'));
    }
    return false;
}
exports.isRnwDependencyProject = isRnwDependencyProject;
/**
 * Search for the RNW lib project files under the target folder.
 * @param winFolder The absolute path to target folder.
 * @return Return the array of relative file paths.
 */
function findDependencyProjectFiles(winFolder) {
    // First, search for all potential project files
    const allCppProj = findFiles(winFolder, '*.vcxproj');
    const allCsProj = findFiles(winFolder, '*.csproj');
    const allProjects = allCppProj.concat(allCsProj);
    if (allProjects.length === 0) {
        // If there're no project files, return 0
        return [];
    }
    const dependencyProjectFiles = [];
    // Try to find any project file that appears to be a dependency project
    for (const projectFile of allProjects) {
        // A project is marked as a RNW dependency iff either:
        // - If the project has the standard native module imports, or
        // - If we only have a single project (and it doesn't have the standard native module imports),
        // pick it and hope for the best. This enables autolinking for modules that were written
        // before the standard native module template existed.
        if (allProjects.length === 1 ||
            isRnwDependencyProject(path_1.default.join(winFolder, projectFile))) {
            dependencyProjectFiles.push(path_1.default.normalize(projectFile));
        }
    }
    return dependencyProjectFiles;
}
exports.findDependencyProjectFiles = findDependencyProjectFiles;
function getReactNativeProjectType(value) {
    switch (value) {
        case 'App-WinAppSDK':
            return value;
        default:
            return 'unknown';
    }
}
/**
 * Checks if the target file path is a RNW app project file.
 * @param filePath The absolute file path to check.
 * @return Whether the path is to a RNW app project file.
 */
function isRnwAppProject(filePath) {
    const projectContents = readProjectFile(filePath);
    const rnProjectType = getReactNativeProjectType(tryFindPropertyValue(projectContents, 'ReactNativeProjectType'));
    if (rnProjectType !== 'unknown') {
        return true;
    }
    const projectLang = getProjectLanguage(filePath);
    if (projectLang === 'cs') {
        return importProjectExists(projectContents, 'Microsoft.ReactNative.Uwp.CSharpApp.targets');
    }
    else if (projectLang === 'cpp') {
        return (importProjectExists(projectContents, 'Microsoft.ReactNative.Uwp.CppApp.targets') ||
            importProjectExists(projectContents, 'Microsoft.ReactNative.Composition.CppApp.targets'));
    }
    return false;
}
/**
 * Search for the RNW app project files under the target folder.
 * @param winFolder The absolute path to target folder.
 * @return Return the array of relative file paths.
 */
function findAppProjectFiles(winFolder) {
    // First, search for all potential project files
    const allCppProj = findFiles(winFolder, '*.vcxproj');
    const allCsProj = findFiles(winFolder, '*.csproj');
    const allProjects = allCppProj.concat(allCsProj);
    if (allProjects.length === 0) {
        // If there're no project files, return 0
        return [];
    }
    const appProjectFiles = [];
    // Try to find any project file that appears to be an app project
    for (const projectFile of allProjects) {
        if (isRnwAppProject(path_1.default.join(winFolder, projectFile))) {
            appProjectFiles.push(path_1.default.normalize(projectFile));
        }
    }
    return appProjectFiles;
}
exports.findAppProjectFiles = findAppProjectFiles;
/**
 * Returns the programming language of the project file.
 * @param projectPath The project file path to check.
 * @return The language string: cpp, cs, or null if unknown.
 */
function getProjectLanguage(projectPath) {
    if (projectPath.endsWith('.vcxproj')) {
        return 'cpp';
    }
    else if (projectPath.endsWith('.csproj')) {
        return 'cs';
    }
    return null;
}
exports.getProjectLanguage = getProjectLanguage;
/**
 * Reads in the contents of the target project file.
 * @param projectPath The target project file path.
 * @return The project file contents.
 */
function readProjectFile(projectPath) {
    const projectContents = fs_1.default.readFileSync(projectPath, 'utf8').toString();
    return new xmldom_1.DOMParser().parseFromString(projectContents, 'application/xml');
}
exports.readProjectFile = readProjectFile;
/**
 * Search for the given property in the project contents and return its value.
 * @param projectContents The XML project contents.
 * @param propertyName The property to look for.
 * @return The value of the tag if it exists.
 */
function tryFindPropertyValue(projectContents, propertyName) {
    const nodes = msbuildSelect(`//msbuild:PropertyGroup/msbuild:${propertyName}`, projectContents);
    if (nodes.length > 0) {
        // Take the last one
        return nodes[nodes.length - 1].textContent;
    }
    else {
        const noNamespaceNodes = xpath_1.default.select(`//PropertyGroup/${propertyName}`, projectContents);
        if (noNamespaceNodes.length > 0) {
            return noNamespaceNodes[noNamespaceNodes.length - 1]
                .textContent;
        }
    }
    return null;
}
exports.tryFindPropertyValue = tryFindPropertyValue;
/**
 * Search for the given property in the project contents and return its value.
 * @param projectContents The XML project contents.
 * @param propertyName The property to look for.
 * @return The value of the tag if it exists.
 */
function tryFindPropertyValueAsBoolean(projectContents, propertyName) {
    const rawValue = tryFindPropertyValue(projectContents, propertyName);
    switch (rawValue) {
        case 'true':
            return true;
        case 'false':
            return false;
        default:
            return null;
    }
}
exports.tryFindPropertyValueAsBoolean = tryFindPropertyValueAsBoolean;
function findPropertyValue(projectContents, propertyName, filePath) {
    const res = tryFindPropertyValue(projectContents, propertyName);
    if (!res) {
        throw new telemetry_1.CodedError('NoPropertyInProject', `Couldn't find property ${propertyName} from ${filePath}`, { propertyName: propertyName });
    }
    return res;
}
exports.findPropertyValue = findPropertyValue;
/**
 * Search for the given import project in the project contents and return if it exists.
 * @param projectContents The XML project contents.
 * @param projectName The project to look for.
 * @return If the target exists.
 */
function importProjectExists(projectContents, projectName) {
    const nodes = msbuildSelect(`//msbuild:Import[contains(@Project,'${projectName}')]`, projectContents);
    return nodes.length > 0;
}
exports.importProjectExists = importProjectExists;
/**
 * Gets the configuration type of the project from the project contents.
 * @param projectContents The XML project contents.
 * @return The project configuration type.
 */
function getConfigurationType(projectContents) {
    var _a;
    const configurationType = (_a = tryFindPropertyValue(projectContents, 'ConfigurationType')) === null || _a === void 0 ? void 0 : _a.toLowerCase();
    switch (configurationType) {
        case 'application':
        case 'dynamiclibrary':
        case 'generic':
        case 'staticlibrary':
            return configurationType;
        default:
            return 'unknown';
    }
}
exports.getConfigurationType = getConfigurationType;
/**
 * Gets the output type of the project from the project contents.
 * @param projectContents The XML project contents.
 * @return The project output type.
 */
function getOutputType(projectContents) {
    var _a;
    const outputType = (_a = tryFindPropertyValue(projectContents, 'OutputType')) === null || _a === void 0 ? void 0 : _a.toLowerCase();
    switch (outputType) {
        case 'appcontainerexe':
        case 'exe':
        case 'library':
        case 'module':
        case 'winexe':
        case 'winmdobj':
            return outputType;
        default:
            return 'unknown';
    }
}
exports.getOutputType = getOutputType;
/**
 * Gets the type of the project from the project contents.
 * @param projectPath The project file path to check.
 * @param projectContents The XML project contents.
 * @return The project type.
 */
function getProjectType(projectPath, projectContents) {
    switch (getProjectLanguage(projectPath)) {
        case 'cpp':
            return getConfigurationType(projectContents);
        case 'cs':
            return getOutputType(projectContents);
        default:
            return 'unknown';
    }
}
exports.getProjectType = getProjectType;
/**
 * Gets the name of the project from the project contents.
 * @param projectPath The project file path to check.
 * @param projectContents The XML project contents.
 * @return The project name.
 */
function getProjectName(projectPath, projectContents) {
    const name = tryFindPropertyValue(projectContents, 'ProjectName') ||
        path_1.default.parse(projectPath).name ||
        '';
    return name;
}
exports.getProjectName = getProjectName;
/**
 * Gets the namespace of the project from the project contents.
 * @param projectContents The XML project contents.
 * @return The project namespace.
 */
function getProjectNamespace(projectContents) {
    return tryFindPropertyValue(projectContents, 'RootNamespace');
}
exports.getProjectNamespace = getProjectNamespace;
/**
 * Gets the guid of the project from the project contents.
 * @param projectContents The XML project contents.
 * @return The project guid.
 */
function getProjectGuid(projectContents) {
    return tryFindPropertyValue(projectContents, 'ProjectGuid');
}
exports.getProjectGuid = getProjectGuid;
function getExperimentalFeatures(solutionDir) {
    const propsFile = path_1.default.join(solutionDir, 'ExperimentalFeatures.props');
    if (!fs_1.default.existsSync(propsFile)) {
        return undefined;
    }
    const result = {};
    const propsContents = readProjectFile(propsFile);
    const nodes = msbuildSelect(`//msbuild:PropertyGroup/msbuild:*`, propsContents);
    for (const node of nodes) {
        const propertyNode = node;
        result[propertyNode.nodeName] = propertyNode.textContent;
    }
    return result;
}
exports.getExperimentalFeatures = getExperimentalFeatures;
//# sourceMappingURL=configUtils.js.map