"use strict";

// Copyright (c) 2015-present Mattermost, Inc. All Rights Reserved.
// See LICENSE.txt for license information.

import { NativeEventEmitter } from "react-native";
import isURL from "validator/es/lib/isURL";
import NativeWebSocketClient, { WebSocketReadyState, WebSocketEvents } from "./NativeWebSocketClient";
const Emitter = new NativeEventEmitter(NativeWebSocketClient);
const CLIENTS = {};
const CREATING_CLIENT = {};

/**
 * Configurable WebSocket client
 */
class WebSocketClient {
  constructor(url) {
    this.url = url;
    this.readyState = WebSocketReadyState.CLOSED;
    this.onReadyStateSubscription = Emitter.addListener(WebSocketEvents.READY_STATE_EVENT, event => {
      if (event.url === this.url) {
        this.readyState = event.message;
      }
    });
  }
  open = () => NativeWebSocketClient.connectFor(this.url);
  close = () => {
    this.readyState = WebSocketReadyState.CLOSED;
    return NativeWebSocketClient.disconnectFor(this.url);
  };
  send = data => NativeWebSocketClient.sendDataFor(this.url, data);
  onOpen = callback => {
    if (this.onWebSocketOpenSubscription) {
      this.onWebSocketOpenSubscription.remove();
    }
    this.onWebSocketOpenSubscription = Emitter.addListener(WebSocketEvents.OPEN_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onClose = callback => {
    if (this.onWebSocketCloseSubscription) {
      this.onWebSocketCloseSubscription.remove();
    }
    this.onWebSocketCloseSubscription = Emitter.addListener(WebSocketEvents.CLOSE_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onError = callback => {
    if (this.onWebSocketErrorSubscription) {
      this.onWebSocketErrorSubscription.remove();
    }
    this.onWebSocketErrorSubscription = Emitter.addListener(WebSocketEvents.ERROR_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onMessage = callback => {
    if (this.onWebSocketMessageSubscription) {
      this.onWebSocketMessageSubscription.remove();
    }
    this.onWebSocketMessageSubscription = Emitter.addListener(WebSocketEvents.MESSAGE_EVENT, event => {
      if (event.url === this.url && callback) {
        callback(event);
      }
    });
  };
  onClientError = callback => {
    if (this.onWebSocketClientErrorSubscription) {
      this.onWebSocketClientErrorSubscription.remove();
    }
    this.onWebSocketClientErrorSubscription = Emitter.addListener(WebSocketEvents.ERROR_EVENT, event => {
      if (event.url === this.url) {
        callback(event);
      }
    });
  };
  invalidate = () => {
    this.onReadyStateSubscription.remove();
    this.onWebSocketOpenSubscription?.remove();
    this.onWebSocketCloseSubscription?.remove();
    this.onWebSocketErrorSubscription?.remove();
    this.onWebSocketMessageSubscription?.remove();
    this.onWebSocketClientErrorSubscription?.remove();
    delete CLIENTS[this.url];
    return NativeWebSocketClient.invalidateClientFor(this.url);
  };
}
async function getOrCreateWebSocketClient(url, config = {}, clientErrorEventHandler) {
  if (!isValidWebSocketURL(url)) {
    throw new Error(`"${url}" is not a valid WebSocket URL`);
  }
  let created = false;
  let client = CLIENTS[url];
  if (!client) {
    if (CREATING_CLIENT[url]) {
      throw new Error(`Already creating a client for url "${url}"`);
    }
    CREATING_CLIENT[url] = true;
    created = true;
    client = new WebSocketClient(url);
    if (clientErrorEventHandler) {
      client.onClientError(clientErrorEventHandler);
    }
    await NativeWebSocketClient.ensureClientFor(url, config);
    CLIENTS[url] = client;
    delete CREATING_CLIENT[url];
  }
  return {
    client,
    created
  };
}
const isValidWebSocketURL = url => {
  return isURL(url, {
    protocols: ["ws", "wss"],
    require_protocol: true,
    require_valid_protocol: true,
    require_host: true
  });
};
export { getOrCreateWebSocketClient };
//# sourceMappingURL=index.js.map