// Copyright (c) 2015-present Mattermost, Inc. All Rights Reserved.
// See LICENSE.txt for license information.

/* eslint-disable max-lines */

import React from 'react';
import Svg, {
    Path,
    G,
    Defs,
    LinearGradient,
    Stop,
} from 'react-native-svg';

import {useIsTablet} from '@hooks/device';
import {changeOpacity, makeStyleSheetFromTheme} from '@utils/theme';

type Props = {
    theme: Theme;
}

const getStyleSheet = makeStyleSheetFromTheme((theme: Theme) => ({
    backgroundGraphic: {
        backgroundColor: changeOpacity(theme.centerChannelColor, 0.06),
        position: 'absolute',
        flex: 1,
    },
}));

const Background = ({
    theme,
}: Props) => {
    const styles = getStyleSheet(theme);

    const isTablet = useIsTablet();

    return (
        <Svg
            viewBox='0 0 414 896'
            style={styles.backgroundGraphic}
        >
            <G>
                <Path
                    d='M476.396 575.017l.118-.395a1.278 1.278 0 00-.473-.869 1.454 1.454 0 00-1.529-.181 1.315 1.315 0 00-.724 1.287l.751.751.395.119.652-.652.81-.06zM423.733 576.163l-.751-.75-.395-.119a1.682 1.682 0 00-.474.118l-.77.771-.099.375c.034.353.204.679.474.909a1.22 1.22 0 00.948.276 1.393 1.393 0 001.067-1.58zM483.605 565.555l-.335.139-.277.276a1.397 1.397 0 00-.553 1.185l.118.396.613.612.296.059.099.099.316-.04.79-.79.119-.316.217-.217a1.4 1.4 0 00-1.324-1.323l-.079-.08zM460.118 578.534l-.829.829c-.027.15-.033.303-.02.455a1.255 1.255 0 00.533.928 1.487 1.487 0 001.245.217l.849-.849c.027-.137.033-.276.02-.415a1.338 1.338 0 00-1.442-1.165l-.099-.099-.257.099zM486.786 576.203l-.533.533-.079.316v.395a1.28 1.28 0 00.693 1.056 1.281 1.281 0 001.262-.048l.652-.652.04-.277a1.42 1.42 0 00-.297-.768 1.416 1.416 0 00-.671-.476l-.158-.158-.435.039-.138.138-.336-.098zM434.044 573.714l.257-.257.04-.237-.001-.277a1.336 1.336 0 00-.987-1.185 1.515 1.515 0 00-1.6.573l-.395.395c-.01.092-.01.185 0 .277a1.34 1.34 0 001.086 1.086 1.43 1.43 0 001.6-.375zM505.019 590.129l.118.118.139.336a1.219 1.219 0 00.79.79l.118-.118a1.78 1.78 0 00.632-.632l.119-.119.079-.316v-.197l-.039-.237-.494-.494a1.586 1.586 0 00-1.601-.218 1.384 1.384 0 00.139 1.087zM454.192 559.847l.672-.672.079-.316.099-.099a1.422 1.422 0 00-1.166-1.165h-.237l-.434.079-.593.592c-.12.164-.201.354-.237.554l.04.237.118.395.553.553a1.344 1.344 0 001.087.138l.019-.296zM468.81 558.661a1.315 1.315 0 00-.573-.968 1.427 1.427 0 00-1.205-.177l-.77.77-.079.316-.099.099.85.849.355.119.217.02 1.008-1.008.099-.336.197.316zM455.141 581.576l-.119-.119-.138-.336-.415-.415a1.348 1.348 0 00-.711-.355l-.119.118-.355.079-.712.711c-.105.229-.152.48-.138.731l.059.297.554.553.296.099c.406.02.804-.122 1.106-.395a1.315 1.315 0 00.692-.968zM523.567 577.23l-.474.474a1.445 1.445 0 00-.178.849c.018.163.058.322.118.474l.494.494a1.311 1.311 0 001.561.099l.829-.83v-.316a1.424 1.424 0 00-1.086-1.284h-.237l-.553.079-.474-.039zM460.771 557.417l.237.04.454-.06.988-.988v-.276a1.402 1.402 0 00-.406-.898 1.402 1.402 0 00-.898-.406l-.553.553a1.32 1.32 0 00-.336 1.008 1.391 1.391 0 00.514 1.027zM491.843 554.553c.257.125.547.166.83.118l.849-.849.099-.376-.099-.098a1.537 1.537 0 00-.948-1.225l-.237-.04-.455.06-.572.573a1.423 1.423 0 00-.356.711l.02.256.158.435.573.573.138-.138zM458.499 558.622l-.079-.356-.415-.415a1.372 1.372 0 00-1.324-.296 1.671 1.671 0 00-1.066 1.067l.019.257.415.414a1.453 1.453 0 001.64.218 1.38 1.38 0 00.81-.889zM493.502 544.873l-1.027 1.028a1.307 1.307 0 00.533 1.007 1.428 1.428 0 001.106.237l.988-.988.02-.375a1.312 1.312 0 00-1.205-1.205l-.257.02-.158.276zM480.919 559.708l-.296-.296-.336-.099-.119-.118-.217.02-.435.434a1.46 1.46 0 00-.434 1.185 1.424 1.424 0 001.501 1.265l.138-.139.395-.197.514-.514c.116-.232.157-.494.119-.75l-.415-.415-.415-.376zM444.474 525.159l-.711-.711a1.03 1.03 0 00-1.008-.138 1.272 1.272 0 00-.458.292c-.13.13-.23.286-.292.459l-.139.138-.118.395a1.261 1.261 0 00.906 1.087c.161.046.33.059.496.039.299-.022.583-.14.81-.336a1.371 1.371 0 00.514-1.225zM457.867 552.893l.256.257a1.316 1.316 0 001.502.751l.494-.494a1.51 1.51 0 00.434-.988l-.039-.237-.435-.434a1.61 1.61 0 00-.834-.337 1.611 1.611 0 00-.885.159 1.313 1.313 0 00-.73 1.323h.237zM467.783 552.854l.297-.296.118-.395a1.4 1.4 0 00-.518-.951 1.4 1.4 0 00-1.042-.294 1.43 1.43 0 00-1.008.534 1.339 1.339 0 00-.316 1.027l.435.434a1.364 1.364 0 00.968.534 1.434 1.434 0 001.145-.474l-.079-.119zM462.311 550.661l.87-.869a1.404 1.404 0 00-.238-.672l-.592-.592a1.403 1.403 0 00-1.383.158 1.373 1.373 0 00-.533 1.165c.038.341.208.653.474.869l.257.257a1.331 1.331 0 001.224.119l-.079-.435zM456.563 547.718l.534.099.296-.06.316-.118.593-.593.079-.316a1.399 1.399 0 00-1.383-1.382 1.363 1.363 0 00-.929.059 1.424 1.424 0 00-.691.889l-.099.099.04.316.573.572c.211.131.446.218.691.257l-.02.178zM432.01 559.827a1.34 1.34 0 00-.731-1.086 1.401 1.401 0 00-1.284.098l-.514.514-.059.257.019.217.889.889a1.291 1.291 0 001.298-.166 1.281 1.281 0 00.48-1.217l-.098.494zM492.257 528.419l.099.375.514.514c.194.135.417.223.652.256l.118-.118.356-.119.395-.395c.135-.17.217-.376.237-.592l-.02-.218-.119-.395-.493-.493a1.512 1.512 0 00-.613-.257l-.118.118-.395.119-.514.513a1.124 1.124 0 00-.237.593l.138.099zM487.972 543.688l-.395.395a1.116 1.116 0 00-.237.593l.019.257.198.474.553.553.375.099.217.019a1.337 1.337 0 001.205-1.442.585.585 0 00-.078-.316l-.731-.731-.316-.079-.099-.098-.119.118-.592.158zM473.195 535.392l-.296-.297.099-.098-.87-.869-.375-.099a1.324 1.324 0 00-.932.453 1.316 1.316 0 00-.312.989 1.318 1.318 0 001.126 1.126 1.25 1.25 0 001.071-.225 1.251 1.251 0 00.489-.98zM473.018 544.696l-.435.434-.138.336-.099.099-.019.257a1.415 1.415 0 00.651 1.086 1.487 1.487 0 001.265.079l.849-.849.119-.356v-.435l-.85-.849-.355-.118-.119-.119-.395.395-.474.04zM482.124 575.926l-.118.119a1.392 1.392 0 00-.691 1.323 1.45 1.45 0 001.916 1.215c.276-.095.517-.27.691-.504l.138-.138.099-.375.119-.119-.06-.257-.138-.138a1.422 1.422 0 00-1.422-.711l-.099.099-.435-.514zM507.863 520.122l.276-.039.553-.553a1.425 1.425 0 00.474-1.186l-.039-.237-.711-.711a1.478 1.478 0 00-1.047-.059l-.83.83a1.685 1.685 0 00-.059.612 1.368 1.368 0 001.383 1.343zM495.241 622.347a1.318 1.318 0 00.928 1.126l.336-.02.355-.118c.307-.07.584-.236.79-.475a1.194 1.194 0 00.277-.948l-.237-.237a1.287 1.287 0 00-1.541-.553 1.508 1.508 0 00-.908 1.225zM409.767 561.921a1.399 1.399 0 001.126-1.521 1.602 1.602 0 00-.138-.494l-.435-.435a1.307 1.307 0 00-1.323-.138l-.83.83.118.118-.335-.059c-.1.23-.147.48-.139.731l.889.889.356-.04.711.119zM514.145 533.594l-.652.652a1.4 1.4 0 001.442 1.245 1.284 1.284 0 001.244-1.245l-.474-.474a1.369 1.369 0 00-1.758-.454l.198.276zM480.623 552.854a1.256 1.256 0 00-.751-1.067l-.257-.019-.454.019-1.027 1.028-.099.375c.028.305.162.59.379.806.216.217.501.351.806.379l.099-.099.355-.079.277-.276-.04.039a1.38 1.38 0 00.498-.462c.124-.193.197-.415.214-.644zM482.025 574.563l.415-.415a1.589 1.589 0 00-.672-1.225l-.217-.217a1.51 1.51 0 00-.968-.573 1.845 1.845 0 00-1.876.771l-.139.138-.098.375-.099.099.039.237.692.691a1.447 1.447 0 00.79.791l.513.513a1.392 1.392 0 001.087-.415l.257-.256.276-.514zM440.8 605.714l.632-.632.079-.316v-.434a1.312 1.312 0 00-.613-.889 1.282 1.282 0 00-1.046-.138l-.198.197a1.287 1.287 0 00-.81 1.087l.119.118.098.336.692.691a1.37 1.37 0 001.047.06v-.08zM452.613 588.884l.158.159.296.098.099.099a1.622 1.622 0 001.165-.454 1.403 1.403 0 00.356-1.067l-.83-.83a1.284 1.284 0 00-1.106-.158l-.79.79a1.407 1.407 0 00-.119.751l.04.277.336.335h.395zM461.877 569.111l-1.027-1.027-.297-.02-.493.059-.435.435a1.366 1.366 0 00-.375.375l-.06.297.06.573.652.651a1.45 1.45 0 001.047.257l.888-.889c.073-.23.087-.474.04-.711zM451.624 570.039l-.118.119a1.397 1.397 0 00.553.988 1.535 1.535 0 001.225.237l.375-.376-.079.079.83-.829.138-.415.138-.138-1.047-1.047-.415-.06-.098-.098a1.417 1.417 0 00-.988.592 1.732 1.732 0 00-.81.455l-.257.256.553.237zM479.616 549.437l.099.098.316.04.099.099a1.483 1.483 0 001.283-1.6l-.888-.889-.435-.119-.257-.02-.217.02-.751.751a1.31 1.31 0 00-.197.869c.023.192.077.378.158.553l.454.454.336-.256zM475.309 570.138v-.237a1.391 1.391 0 00-1.185-1.185l-.276-.04-.119.119-.256.059-.554.553c-.18.304-.256.657-.217 1.008l.04.276.908.909a1.647 1.647 0 001.245-.415l.276-.276.138-.771zM464.978 572.884l-.04-.316-.197-.514-.158-.158a1.778 1.778 0 00-.375-.375l-.909-.909-.356-.118c-.25-.011-.5.033-.732.129a1.72 1.72 0 00-.611.424l-.316.316a1.306 1.306 0 00-.474.988l.612.612c.22.258.507.45.83.553l.276.04.178.177.336.02a1.51 1.51 0 001.936-.869zM470.232 571.343l-1.007-1.007a1.389 1.389 0 00-1.034.408 1.393 1.393 0 00-.408 1.034l.435.435a1.397 1.397 0 001.56.217c.245-.127.446-.325.576-.569.131-.243.185-.52.155-.794a.501.501 0 00-.277.276zM477.107 565.457l-.771.77a.94.94 0 000 .237 1.372 1.372 0 001.324 1.324l.098-.099.356-.119.652-.652c.089-.239.129-.495.118-.75l-.059-.257-.592-.593-.316-.079-.257.02-.099.099-.454.099zM429.244 588.667a1.313 1.313 0 00-.553-.988 1.54 1.54 0 00-1.343-.197l-.415.415a1.456 1.456 0 00-.336 1.086c.055.407.268.776.593 1.027l.316.316.237.04a1.524 1.524 0 001.185-.474 1.307 1.307 0 00.316-1.225zM499.843 572.667l.454-.455-.118-.118a1.423 1.423 0 00-.593-1.106 1.37 1.37 0 00-1.146-.198l-.316.316a1.26 1.26 0 00-.77 1.284c.016.172.062.339.138.494l.77.77c.201.079.419.107.633.079l.691-.691.257-.375zM504.426 567.965l-.77.771a1.788 1.788 0 001.106 1.541l.099.098h.316l.573-.572a1.485 1.485 0 00-.059-1.877l-.554-.553-.375-.099-.138-.138-.237.039-.119.119.158.671zM473.235 575.887v-.237a1.502 1.502 0 00-.593-1.067 1.49 1.49 0 00-1.224-.198l-1.008 1.008v.277a1.503 1.503 0 00.593 1.066 1.613 1.613 0 001.225.198l.928-.929.079-.118zM504.367 581.497l-.139-.139a1.2 1.2 0 00-1.244-.533l-.928.928-.099.376a1.28 1.28 0 00.691 1.047 1.38 1.38 0 001.316.032 1.347 1.347 0 00.664-.792 1.35 1.35 0 00.055-.524l-.316-.395zM526.925 595.877l-.553.553c-.026.15-.033.303-.02.454a1.399 1.399 0 00.968 1.205 1.641 1.641 0 001.699-.513l.434-.435v-.316a1.33 1.33 0 00-.731-1.086 1.48 1.48 0 00-1.521.059l-.276.079zM459.506 574.563l-.889.889.02.336a1.31 1.31 0 00.672 1.067 1.572 1.572 0 001.501 0l.711-.712.04-.316-.06-.256-.177-.455-.277-.276a1.537 1.537 0 00-1.541-.277zM486.352 623.177l-.474-.474a1.312 1.312 0 00-.396-.396l-.276-.079-.415.139-.711.711-.138.375a1.189 1.189 0 00.474.869c.271.222.619.328.968.297a1.315 1.315 0 00.968-1.442zM501.522 597.22a1.568 1.568 0 00-1.383-1.382h-.237l-.435.118-.632.632-.158.435-.138.138.04.316.908.909.375.059.08-.276v.276h.474l.908-.908.06-.297.138-.02zM486.312 570.83l-.119.118.04.237.829.83.356.118.217.02.928-.928.159-.356.118-.118a1.438 1.438 0 00-.553-.988 1.39 1.39 0 00-1.126-.257l-.81.81-.039.514zM515.034 580.114a1.315 1.315 0 00-1.087-1.087l-.276.04-.474.237a1.285 1.285 0 00-.652 1.244c.016.172.063.339.138.494l.632.632.277.04a1.286 1.286 0 001.205-1.205v-.237l.237-.158zM456.602 526.7l.455-.454.099-.336v-.198l-.02-.335-.81-.81-.336-.099-.118-.119a1.397 1.397 0 00-1.324.929l-.039.276.079.514.79.79c.256.104.536.138.81.099l.138-.138.276-.119zM440.424 594.238l.02-.297-.099-.533-.632-.632-.375-.099-.04-.039a1.387 1.387 0 00-1.086.414 1.28 1.28 0 00-.356.988l.553.553a1.456 1.456 0 001.265.316 1.48 1.48 0 00.75-.671zM443.19 577.566l.454-.455.296-.099a1.427 1.427 0 00.988-1.461l-1.047-1.047a1.725 1.725 0 00-1.403.296c-.223.126-.407.31-.533.533l-.119.119a1.32 1.32 0 00-.523.532 1.32 1.32 0 00-.148.732l.079.356.731.731.316.079h.237l.138-.139.534-.177zM476.336 530.631v.316l.81.81.316.119.119.118a1.42 1.42 0 001.067-.434c.133-.158.233-.34.294-.537.061-.197.082-.404.061-.609l-.889-.889-.336-.099-.118-.118-.988.987-.119.356-.217-.02zM527.814 551.886l.158.158.198.435.375.375c.188.154.412.256.652.296h.197l.336-.099.692-.691c.043-.154.056-.315.039-.474a1.388 1.388 0 00-.425-.986 1.408 1.408 0 00-.997-.397 1.395 1.395 0 00-1.284.884 1.395 1.395 0 00-.099.538l.158-.039zM511.478 569.329a1.253 1.253 0 00-.691.967l.118.119.119.395.593.593c.32.173.692.222 1.047.138l.395-.395-.079.079.296-.296.059-.376a1.277 1.277 0 00-.375-.849 1.277 1.277 0 00-.849-.375l-.218.019-.415-.019zM514.54 593.842l-.415.415a1.28 1.28 0 00-.257.613l.04.237.177.454.553.553.376.099.217.02a1.32 1.32 0 00.939-.385 1.304 1.304 0 00.384-.939l-.079-.434-.651-.652-.316-.079-.218.217-.75-.119zM471.26 554.197l.652.652a1.51 1.51 0 001.383-.158 1.422 1.422 0 00.553-1.185 1.236 1.236 0 00-.573-.929l-.257-.256-.316-.08a1.298 1.298 0 00-.988.435c-.232.29-.346.657-.316 1.027l-.138.494zM438.034 556.647l.297.296.414.138c.225.095.469.136.712.119l.849-.85.099-.335-.04-.317-.197-.553.118-.118a1.507 1.507 0 00-1.56-.692l-.83.83-.099.336-.118.118.039.316.376.376-.06.336zM505.987 547.046l-.415.415c-.122.178-.203.38-.237.593v.237l.138.375.474.474a.757.757 0 00-.079.316v.198l1.087 1.086a1.275 1.275 0 001.586-.745c.079-.196.111-.408.093-.618a1.326 1.326 0 00-.514-.948l.079-.316.099-.099a1.395 1.395 0 00-1.047-1.244l-.257-.02-.415.099-.592.197zM503.103 551.234c-.43 0-.842.171-1.146.474-.27.313-.418.713-.415 1.126l.632.632c.393.233.86.303 1.304.198l.79-.79.099-.336.099-.099-.04-.395-1.027-1.027-.296.217zM486.371 584.124a1.447 1.447 0 00-.548-.968 1.455 1.455 0 00-1.072-.296 1.198 1.198 0 00-.869.474 1.367 1.367 0 00-.316.987l.751.751.395.118.869-.869.79-.197zM447.753 588.964l.869-.87.079-.316-.02-.256-.908-.909-.395-.079h-.237a1.253 1.253 0 00-1.126 1.363 1.198 1.198 0 00.829 1.067l.514.118.395-.118zM492.831 567.235l.454.454a1.43 1.43 0 001.62.158c.245-.127.446-.325.576-.569.131-.243.185-.52.155-.794a1.85 1.85 0 00-.099-.454l-.652-.652a1.426 1.426 0 00-1.334.187 1.427 1.427 0 00-.582 1.215l-.138.455zM507.231 561.052l-.81.81.019.256a1.535 1.535 0 001.68 1.363l.138-.138.395-.158.513-.514.139-.335.098-.099v-.316l-.039-.237-.79-.79a2.97 2.97 0 00-1.008-.139l-.118.119-.217.178z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M451.348 569.447l.771-.771v-.355a1.484 1.484 0 00-1.186-1.185l-.098-.099-.455.02-.711.711c-.09.239-.131.495-.118.75l.059.297.553.553c.397.222.865.278 1.304.158l-.119-.079zM490.46 565.18l.79-.79c.18-.191.297-.432.336-.692l-.02-.217-.118-.434-.534-.534-.414-.138-.238-.039-1.027 1.027-.098.415.098.098a1.284 1.284 0 001.225 1.304zM453.264 574.721a1.3 1.3 0 00.316-.988l-.691-.691a1.114 1.114 0 00-.691-.217l-.731-.731a1.647 1.647 0 00-1.087.454l-.454.454-.02.218a1.316 1.316 0 001.067 1.066l.257.02.296-.059a1.38 1.38 0 001.047.375c.219.087.456.121.691.099zM498.223 538.651l.198-.197c.125-.036.245-.089.355-.158.194-.106.354-.263.463-.455.108-.193.16-.412.15-.632l-.119-.118-.119-.356-.651-.652-.356-.118-.118-.119c-.332.026-.648.15-.909.356l-.099.098c-.29.083-.542.264-.712.513a1.29 1.29 0 00-.216.851l.415.414a1.537 1.537 0 001.718.573zM517.424 557.417l-.415.415a1.666 1.666 0 00-.237.553l.02.257.118.395.573.573a1.18 1.18 0 001.087.138l.77-.77.079-.317a1.426 1.426 0 00-1.185-1.185h-.237l-.395.079-.178-.138zM455.161 507.915a.506.506 0 00-.04-.237l-1.007-1.008a1.491 1.491 0 00-1.186.435 1.416 1.416 0 00-.395 1.145l.06.257.711.711.296.099.099.099a1.475 1.475 0 001.165-.415 1.46 1.46 0 00.297-1.086zM441.906 591.018l.118-.119a1.197 1.197 0 00.672-1.027l-.02-.217-.138-.376-.553-.553a1.341 1.341 0 00-1.027-.158l-.85.85-.059.375a1.286 1.286 0 001.205 1.205l.652.02zM450.36 549.575l-.019-.296-.158-.475-.593-.592-.375-.099-.218-.02a1.282 1.282 0 00-1.224 1.423c-.004.11.023.22.079.316l.612.612.336.099a1.476 1.476 0 001.56-.968zM491.882 592.677l-.335-.336a1.291 1.291 0 00-1.443-.296 1.308 1.308 0 00-.75 1.304 1.28 1.28 0 00.75 1.027l.119.118.217.02.988-.988v-.355l-.099-.099.553-.395zM473.255 591.354l-.099-.415-.671-.672a1.426 1.426 0 00-1.917 1.403l.652.651a1.48 1.48 0 001.62.159 1.434 1.434 0 00.415-1.126zM446.37 557.358l-.316-.316a1.563 1.563 0 00-1.086-.06l-.909.909v.316a1.371 1.371 0 001.343 1.343 1.4 1.4 0 001.344-1.343l-.474-.474.098-.375zM448.524 612.055l.711-.711v-.276a1.335 1.335 0 00-1.363-1.363 1.338 1.338 0 00-1.324 1.323l.929.929c.257.125.547.166.829.118l.218-.02zM466.618 543.155l-.534.533-.098.415-.119.118a1.483 1.483 0 00.948 1.225l.237.04.455-.06 1.007-1.007-.02-.217a1.394 1.394 0 00-1.323-1.324l-.138.139-.415.138zM467.27 523.303l-.474-.474-.376-.139-.118-.118-.316.039-.731.731-.138.375c.043.361.219.692.494.929.282.214.635.313.987.276a1.367 1.367 0 001.21-.96c.054-.175.073-.359.054-.541l-.592-.118zM447.081 539.856l-.869-.869-.336-.099-.217-.02-.237.04-.454.454c-.17.172-.28.394-.316.632l-.119.119-.039.276v.237l1.027 1.028.355.039.218-.02a1.452 1.452 0 00.987-1.817zM488.505 556.943l-.317-.079-.098-.099-.316.04-1.225 1.224a1.51 1.51 0 00.573 1.047 1.433 1.433 0 001.146.237l.73-.731c.107-.227.142-.483.099-.731l-.059-.296-.632-.632.099.02zM439.299 546.375l.375-.376.059-.256-.039-.474-.988-.988-.434-.079-.257.02c-.299.09-.571.252-.79.474l.256.257-.217-.218-.237.474-.059.297.039.316.771.77s1.066.277 1.699-.237l-.178.02zM484.079 514.038l.534-.533a1.233 1.233 0 00-.471-1.71 1.233 1.233 0 00-.635-.147l-.257.257a1.313 1.313 0 00-1.087 1.086l.02.217.138.376.237.237c.191.18.432.297.692.336l.296-.02.533-.099zM463.457 540.073l.672-.671.079-.356.098-.099a1.48 1.48 0 00-1.6-1.323l-.889.889-.118.355v.237l.099.099.059.257.514.514a1.676 1.676 0 001.303.197l-.217-.099zM481.196 535.254l-.455.454-.118.355-.02.218.968.968.375.098.119.119.889-.889c.165-.272.235-.592.197-.909l-1.047-1.047h-.237l-.138.139-.533.494zM486.292 604.036l.119.118.138.375.849.85a1.432 1.432 0 001.055-.346 1.417 1.417 0 00.486-.997l-.118-.119-.119-.356-.909-.908c-.4-.028-.796.099-1.106.355a1.458 1.458 0 00-.395 1.028zM485.996 638.505a1.4 1.4 0 00.355-1.145l-.02-.257-.651-.652a1.587 1.587 0 00-1.502.237 1.38 1.38 0 00-.474 1.146 1.178 1.178 0 00.415.81l.672.671a1.511 1.511 0 001.205-.81zM469.403 609.112l-.573-.573-.454-.257-.178-.177a1.447 1.447 0 00-1.324 1.046l-.079.356.04.277.118.316.593.592.277.079a1.426 1.426 0 001.461-1.106l-.138-.138.257-.415zM482.44 593.665l-.118.395a1.338 1.338 0 00.731 1.047c.223.1.468.14.711.118a1.371 1.371 0 001.363-1.363l-.474-.474a.952.952 0 00-.395-.395l-.119-.118a1.059 1.059 0 00-.553 0l-.533.533-.613.257zM461.285 596.332l.177-.178a1.56 1.56 0 00.415-.415l.04-.277-.02-.493-.613-.613-.454-.217-.158-.158-.316.039-.849.85-.099.375-.119.119a1.402 1.402 0 001.146 1.145h.277l.474-.118.099-.059zM479.082 595.048l.237-.237.02-.218a1.504 1.504 0 00-.139-.454l.099-.375.118-.119a1.418 1.418 0 00-.632-1.027 1.307 1.307 0 00-1.126-.138l-.75.75-.079.316-.099.099.119.119a.789.789 0 00-.158.158l-.218.217v.237a1.267 1.267 0 00.437.868 1.26 1.26 0 00.926.297c.255-.027.496-.13.692-.296.198-.019.388-.087.553-.197zM498.934 577.487a1.233 1.233 0 00-.632 1.224c.016.172.063.339.138.494l.652.652a1.309 1.309 0 001.166-.296 1.462 1.462 0 00.494-1.245 1.285 1.285 0 00-1.186-1.185l-.296.059-.336.297zM479.28 596.47l.336.336a1.341 1.341 0 002.074-.929l-.138-.138-.119-.435-.533-.533-.356-.118-.118-.119-.415.02-.593.592-.217.455-.04.276.139.138-.02.455zM470.114 624.046l-.06-.297-.316-.316a1.394 1.394 0 00-.968-.968 1.348 1.348 0 00-.852.14c-.26.14-.468.361-.59.631l-.019.257.039.474 1.087 1.086.355.119.218.019a1.31 1.31 0 001.106-1.145zM461.699 609.389l.454-.455a1.394 1.394 0 00.434-1.145 1.147 1.147 0 00-.078-.356l-.87-.869-.513.079-.613.612-.158.395-.138.139.02.256a1.317 1.317 0 00.513.909 1.127 1.127 0 00.949.435zM501.147 603.877l-.869.87a1.397 1.397 0 00.553.987 1.537 1.537 0 001.225.237l.987-.987v-.198a1.365 1.365 0 00-1.442-1.244l-.118.118-.336.217zM474.598 605.655l.198.198a1.578 1.578 0 001.165-.455c.25-.304.37-.694.336-1.086l-.672-.672a1.345 1.345 0 00-1.086-.138l-.691.691a1.342 1.342 0 00-.119.712l.079.316.494.493.296-.059zM476.85 616.954a1.42 1.42 0 001.264.119l.751-.751.079-.276.099-.099a1.43 1.43 0 00-1.166-1.166h-.277l-.434.435a1.45 1.45 0 00-.316 1.027 1.247 1.247 0 000 .711zM490.638 581.872l-.099.375a1.281 1.281 0 00.573 1.008c.252.167.547.256.85.257l.375.098.217-.019.356-.119.533-.533c.112-.174.186-.369.217-.573l-.118-.119-.119-.395-.513-.513a1.193 1.193 0 00-.613-.257l-.098.099-.119-.119a1.436 1.436 0 00-1.086.375l-.474.474.118-.039zM477.956 582.662l-.474.474a1.4 1.4 0 00-.257.968 1.315 1.315 0 00.869 1.146 1.45 1.45 0 001.502-.237l.513-.514-.02-.257a1.44 1.44 0 00-.573-1.047 1.477 1.477 0 00-1.244-.217l-.316-.316zM467.349 585.131a1.423 1.423 0 001.284.099l.493-.494c.174-.263.263-.573.257-.889a1.475 1.475 0 00-.257-.612l-.691-.691a1.42 1.42 0 00-1.146.395 1.277 1.277 0 00-.395 1.106 1.4 1.4 0 00.455 1.086zM451.289 596.45l.671-.672.099-.335.099-.099v-.316l-.85-.85-.375-.098-.118-.119-1.087 1.087-.039.395.019.296a1.368 1.368 0 001.66 1.067l-.079-.356zM493.838 606.445a1.353 1.353 0 00-.944.462 1.349 1.349 0 00-.321 1l.633.632a1.397 1.397 0 001.935-.316l.376-.375v-.356a1.41 1.41 0 00-.622-.872 1.404 1.404 0 00-1.057-.175zM468.83 580.529a1.513 1.513 0 001.62.237 1.34 1.34 0 00.79-1.304l-.395-.395a1.456 1.456 0 00-1.701-.13 1.512 1.512 0 00-.472.446l-.138.138-.079.356.039.237.356.355-.02.06zM496.9 587.067l-.593.593-.099.296.02.217a1.398 1.398 0 00.593 1.106 1.37 1.37 0 001.145.198l.83-.83c.111-.219.153-.467.119-.711l-.06-.296-.691-.691a1.483 1.483 0 00-1.047-.06l-.217.178zM471.852 582.84l.099.375.533.533c.189.128.406.209.632.238l.119-.119.355-.119.514-.513c.127-.189.209-.406.237-.632l.099-.099-.138-.375-.396-.395a1.14 1.14 0 00-.651-.257l-.119.118-.375.099-.494.494a1.193 1.193 0 00-.257.612l-.158.04zM473.037 586.237l-.178.178-.158.356-.118.118v.277a1.511 1.511 0 00.593 1.067 1.434 1.434 0 001.244.177l.77-.77a1.71 1.71 0 00.06-.612 1.4 1.4 0 00-.988-1.225l-.217-.02-.435.04-.316.316-.257.098zM474.638 581.793l.098.375.415.415a1.484 1.484 0 001.62.237 1.389 1.389 0 00.77-1.323l-1.185-1.186a1.297 1.297 0 00-.612-.257l-.119.119-.375.138-.514.514a1.693 1.693 0 00-.217.612l.119.356zM483.685 580.371l.138.375.514.514c.181.124.392.199.612.217l.099-.099.395-.118.533-.534c.124-.162.205-.352.237-.553l-.019-.217-.119-.395-.514-.514a1.502 1.502 0 00-.612-.257l-.119.119-.395.118-.533.534c-.119.17-.193.367-.217.572v.238zM464.444 589.378l.672-.671.099-.336-.04-.395-.355-.356a1.516 1.516 0 00-.552-1.099 1.512 1.512 0 00-1.186-.323l-.929.928-.02.494c.043.393.225.758.514 1.027a1.344 1.344 0 00.583.923 1.342 1.342 0 001.076.184l.138-.376zM466.973 593.448l-.237.237-.02.217a1.206 1.206 0 00.633.987 1.531 1.531 0 001.54.001 1.315 1.315 0 00.593-1.265 1.175 1.175 0 00-.474-.869 1.449 1.449 0 00-1.106-.277l-.415.415c-.229.121-.41.316-.514.554zM491.073 598.702l.02.336.059.177.178.336.138.178.198.158.177.099.376.099.197.039.158-.039.198-.04.336-.178.316-.316.079-.197.118-.356.02-.217-.02-.296-.059-.218-.178-.336-.118-.158-.198-.158-.198-.079-.355-.118-.217-.02h-.158l-.178.059-.336.178-.178.138-.158.198-.099.178-.098.375-.02.178zM457.077 613.3l.415-.415a1.21 1.21 0 00.217-.573v-.237l-.118-.395-.593-.592a1.344 1.344 0 00-1.067-.159l-.77.771-.079.316.099.099v.237l.849.849.356.119.217.019.118-.118.356.079zM449.926 600.223l-.83-.83-.335-.098-.119-.119a1.403 1.403 0 00-1.054.408 1.379 1.379 0 00-.315.484 1.384 1.384 0 00-.093.57l1.008 1.007a1.393 1.393 0 001.284-.296 1.342 1.342 0 00.454-1.126zM436.237 620.589a1.401 1.401 0 00-.613-1.008 1.377 1.377 0 00-1.185-.158l-.79.791a1.335 1.335 0 00-.119.671c.053.346.228.662.494.889l.178.178a1.395 1.395 0 001.126-.375 1.43 1.43 0 00.909-.988zM486.845 609.171l-.888-.889a1.633 1.633 0 00-.435-.039l-.119.119-.375.138-.573.573-.158.395-.039.237.711.711a1.361 1.361 0 001.343-.198 1.518 1.518 0 00.533-1.047zM457.314 619.502l-.079-.316-.771-.77-.316-.079-.75.751a1.277 1.277 0 00-.257.612l.02.257.217.454.533.534.376.098.118.119a1.314 1.314 0 00.909-1.66zM476.633 589.2l.968.968.356.119.217.02h.237l.889-.889.099-.336-.099-.099a1.397 1.397 0 00-.553-.988l-.909-.908-.277-.079-.197-.198-.257-.059-.474.039-.573.573-.217.455-.079.276a1.423 1.423 0 00.672 1.106h.197zM464.781 600.065l.415-.415.118-.355a1.39 1.39 0 00-.164-.541 1.376 1.376 0 00-.369-.427 1.336 1.336 0 00-1.126-.257l-.79.79-.099.336-.119.118a1.463 1.463 0 00.553.988 1.485 1.485 0 001.127.257l.118-.119.336-.375zM515.389 611.562l.138.138.257.059a1.281 1.281 0 00.988-.434 1.406 1.406 0 00.335-1.047l-.059-.297-.77-.77-.297-.059-.434.039-.593.593-.217.454-.158.158.099.099a1.232 1.232 0 00.711 1.067zM448.819 535.392l.415.415a1.257 1.257 0 001.284-.06c.217-.118.396-.296.514-.513l.138-.139.138-.375.02-.217v-.237a1.748 1.748 0 00-.652-1.205l-.296-.297-.237-.039c-.311.004-.613.1-.869.277l-.771.77-.059.257.04.276.118.119.217.968zM424.562 539.856l.632-.632.06-.296a1.43 1.43 0 00-1.166-1.166l-.099-.099-.276.04-.534.533a1.455 1.455 0 00-.296 1.008 1.196 1.196 0 00.573.968 1.43 1.43 0 001.244.138l-.138-.494zM470.015 562.237l-.059.296.019.494.613.612.454.218.277.039c.374.024.741.111 1.086.257l.257.02.435-.04.553-.553.197-.435.158-.158a1.213 1.213 0 00-.195-.588 1.219 1.219 0 00-.457-.419l-.987-.988-.297-.099h-.355l-.139-.138a1.36 1.36 0 00-1.14.362 1.375 1.375 0 00-.42 1.12zM475.112 561.249l.474.474a1.537 1.537 0 001.64-.573l.197-.197.04-.593-.158-.158-.02-.217-.119-.435-.533-.533-.415-.138-.237-.04-.316.04-1.126 1.126-.02.573.652.652-.059.019zM463.536 561.881l.652-.652.118-.434v-.237a1.252 1.252 0 00-.632-.948 1.34 1.34 0 00-1.066-.119l-.83.83-.059.335a1.34 1.34 0 001.205 1.205l.138-.138.474.158zM443.289 565.22l-.692-.692a1.113 1.113 0 00-1.027-.158 1.253 1.253 0 00-.458.293c-.13.129-.23.286-.293.458l-.138.138-.099.376c.029.344.192.664.455.889a1.177 1.177 0 00.928.256 1.389 1.389 0 001.274-.992c.055-.184.072-.378.05-.568zM437.718 539.975l-.276-.277a1.314 1.314 0 00-.949.316 1.387 1.387 0 00-.454.968 1.289 1.289 0 001.165 1.442l.396-.119.493-.493.158-.356.02-.217-.118-.356-.178-.178-.257-.73zM429.916 592.065l-.395.118-.494.494-.158.356a.951.951 0 000 .474l.118.395.85.849a1.37 1.37 0 00.948-.316 1.18 1.18 0 00.435-.909 1.38 1.38 0 00-.344-1.005 1.37 1.37 0 00-.96-.456zM429.56 531.263l.079.356.257.257a1.34 1.34 0 001.067.711l.395-.04.928-.928v-.316l-.099-.336-.79-.79a.282.282 0 00-.118-.119 1.337 1.337 0 00-1.501 1.067l-.218.138zM438.429 556.607l.099.296.613.613.434.118a1.388 1.388 0 001.369-.936c.076-.213.101-.44.073-.664l-.751-.75a1.158 1.158 0 00-.395-.079 1.31 1.31 0 00-1.303 1.303l-.099.099h-.04zM440.662 529.92l.711.711.395.079a1.198 1.198 0 001.086-.454 1.454 1.454 0 00.237-1.304l-.928-.928a1.418 1.418 0 00-1.383.75l-.158.158-.04.435.02.257.06.296zM438.272 569.249l-.257-.059a1.248 1.248 0 00-.405-.37 1.24 1.24 0 00-.524-.163 1.75 1.75 0 00-1.264.474l-.474.474-.02.257.099.099.099.335.118.119a1.312 1.312 0 00.968.533c.133.311.381.559.692.692l.118.118.395-.039.909-.909a1.649 1.649 0 00-.415-1.442l-.039-.119zM429.244 581.477l-.415-.415-.277-.039a1.343 1.343 0 00-1.323.77 1.476 1.476 0 00.276 1.699l.395.395.297.059a1.338 1.338 0 001.323-.77 1.455 1.455 0 00-.276-1.699zM457.452 514.315l.119.118.355-.079a1.306 1.306 0 00.791-.79l.118-.118-.099-.099a1.806 1.806 0 00-.474-.751l-.474-.474h-.237l-.118.119-.316.118-.139.138a1.653 1.653 0 00-.474 1.541 1.285 1.285 0 00.948.277zM419.011 559.274l-.968-.968a1.427 1.427 0 00-1.442.849l-.02.218.001.434.572.573c.161.127.341.227.534.296l.217.02.415-.059.553-.553a1.255 1.255 0 00.316-1.027l-.178.217zM420.236 544.538a1.273 1.273 0 00-1.067.395 1.365 1.365 0 00-.375 1.165l1.086 1.086.119-.118.375-.099.968-.968v-.237l-.118-.118-.139-.376-.711-.711-.138-.019zM440.563 562.02l-.099-.099-.376.059-.335.336a1.332 1.332 0 00-.573.573l-.02.217.039.356.712.711c.219.143.47.231.73.257l.297-.02.889-.889a1.486 1.486 0 00-.178-1.166 1.398 1.398 0 00-1.086-.335zM448.405 494.324a1.365 1.365 0 001.284.257l.237-.237a1.341 1.341 0 00.395-1.541l-.83-.829-.336-.06a1.4 1.4 0 00-1.422.869l-.138.139-.02.573.83.829zM417.728 552.38l.158-.158.039-.474-.987-.988a.507.507 0 00-.139-.138 1.399 1.399 0 00-1.521 1.086l-.019.217.256.257a1.18 1.18 0 00.929.494 1.362 1.362 0 001.284-.296zM420.196 521.268c.153-.244.241-.523.257-.81l-1.205-1.205a1.509 1.509 0 00-1.363.731l-.158.158-.019.454.039.237.395.395c.137.227.327.417.553.553l.257.02.454-.059.573-.573.217.099zM418.537 554.81a1.312 1.312 0 00-.375.019l-.395.395a1.375 1.375 0 00-.514 1.265 1.566 1.566 0 00.889 1.165l.277.04.296-.297a1.479 1.479 0 00.474-1.58 1.371 1.371 0 00-.652-1.007zM410.952 517.989l.059.336.83.829a1.363 1.363 0 001.067-.355 1.323 1.323 0 00.434-1.067l-.099-.296-.691-.692a1.178 1.178 0 00-.356-.079 1.31 1.31 0 00-1.362 1.008v.237l.118.079zM423.358 532.962l-.692.691.395.395a1.454 1.454 0 001.087.652 1.422 1.422 0 001.481-1.284v-.237l-.138-.415-.573-.572a1.412 1.412 0 00-.711-.237l-.217.019-.376.139-.256.849zM426.893 550.879c.094-.145.161-.306.198-.475.191-.15.363-.322.513-.513.24-.038.466-.14.652-.296l.889.888c.138.061.285.101.435.119a1.31 1.31 0 001.323-.889l.02-.217v-.356l-1.146-1.145-.118.118-.336.099-.297.296-.296-.296a1.279 1.279 0 00-1.106-.79l-.336-.336-.138.138-.277.277-.493.494a1.815 1.815 0 00-.475.474l-.335.336a1.253 1.253 0 00-.652.651v.316l.039.237.139.415.533.534.434.118.218.02c.208-.03.408-.097.592-.198l.02-.019zM383.178 562.948l-.375.138-.435.435a1.123 1.123 0 00-.316.987 1.2 1.2 0 00.573.85l.415.415c.167.018.336.002.497-.047.16-.05.309-.132.437-.241a1.234 1.234 0 00.429-.878 1.317 1.317 0 00-.217-.849 1.403 1.403 0 00-1.008-.81zM412.77 554.454l.355-.119a1.286 1.286 0 00.988-1.343l-.415-.415a1.533 1.533 0 00-.889-.612l-.257.257a1.562 1.562 0 00-.474 1.659 1.318 1.318 0 001.185.948l-.493-.375zM414.409 544.676l-.553-.553a1.393 1.393 0 00-1.423 1.422c-.023.388.104.77.356 1.067a1.34 1.34 0 00.948.474l.395-.119.158-.158a1.23 1.23 0 00.79-.79 1.284 1.284 0 00-.276-1.185l-.395-.158zM411.327 549.693l-.888-.889a1.795 1.795 0 00-.711.119l-.534.533a1.372 1.372 0 00.352 1.881c.183.128.395.208.616.233.336.024.669-.082.929-.296.101-.083.194-.175.276-.277a1.361 1.361 0 00.296-1.165l-.336-.139zM407.357 554.928l.178-.494v-.316l-.059-.336-.909-.908a1.374 1.374 0 00-1.383.657 1.375 1.375 0 00-.177.508 1.367 1.367 0 00-.079.948 1.394 1.394 0 00.79.791l.098.098h.316l.455-.454c.17-.184.304-.398.395-.632l.375.138zM415.239 581.141a1.31 1.31 0 00-1.187.54 1.31 1.31 0 00-.136 1.297l.691.691.237.04.257-.06.513-.513a1.32 1.32 0 00.06-1.284 1.343 1.343 0 00-.435-.711zM394.498 516.428l.395-.039.494-.494a1.31 1.31 0 00.355-.593l.02-.217-.039-.395-.514-.513a1.34 1.34 0 00-.533-.336l-.119-.119-.415.059-.494.494c-.142.16-.25.349-.316.553l-.019.218.039.395.514.513c.148.153.331.268.533.336l.099.138zM408.819 523.243l.395.395c.109.177.258.326.434.435l.257.02.494-.099.553-.553.178-.375.118-.119a1.31 1.31 0 00-1.205-1.442l-.296.02-1.067 1.066.119.119.02.533zM398.211 536.458l-.099-.098-.218.217-.118-.119-.356.158-.73.731a1.334 1.334 0 00.699 1.329c.157.084.329.136.505.153a1.368 1.368 0 001.363-.889 1.394 1.394 0 00-1.046-1.482zM407.278 538.138l.79.79.257.059a1.376 1.376 0 001.165-.454 1.394 1.394 0 00.297-1.245l-.119-.118-.118-.356-.83-.83-.237-.039-.119.118-.375.099-.849.85v.237l-.02.217.099.099.059.573zM439.555 534.424l.119.118a1.366 1.366 0 001.165.889 1.451 1.451 0 001.403-.81 1.478 1.478 0 00-.218-1.639l-.138-.138-.375-.178-.119-.119-.256-.019-.238.237a1.367 1.367 0 00-.928 1.284l-.415.375zM388.867 499.697l-.02-.257-.098-.454-.336-.336a1.313 1.313 0 00-1.047-.652h-.277l-.592.593a1.417 1.417 0 00-.237 1.027l.928.929c.198.07.403.117.612.138a1.37 1.37 0 001.067-.988zM487.576 529.426a1.313 1.313 0 001.245-.731l.059-.296-.079-.395a1.317 1.317 0 00-.296-.849 1.262 1.262 0 00-.909-.435l-.217.217a1.256 1.256 0 00-.79 1.422 1.426 1.426 0 00.987 1.067zM413.5 603.383a1.452 1.452 0 00-1.343-1.343l-.513.04-.435.434a1.293 1.293 0 00-.395 1.265l.83.829c.197.158.439.248.691.257l.909-.909.039-.355.217-.218zM403.347 495.687l.771.77a1.424 1.424 0 001.501-1.224 1.284 1.284 0 00-.988-1.462l-.316.039-.276.277a1.369 1.369 0 00-.712 1.659l.02-.059zM416.602 532.014a1.345 1.345 0 00-1.165.573l-.159.158-.039.474v.198l1.205 1.204a1.308 1.308 0 001.402-.888l.119-.119-.02-.336-.454-.454a1.34 1.34 0 00-.889-.81zM438.291 534.384l-.553-.553a1.593 1.593 0 00-1.324.454l-.217.218a1.723 1.723 0 00-.79.79 1.898 1.898 0 00.434 1.975l.158.158.356.119.118.118.257.02.138-.138.336-.139.138-.138a.594.594 0 00.158-.158 1.337 1.337 0 00.791-.79l.316-.316c.078-.389 0-.794-.218-1.126l-.197-.198.099-.296zM461.936 580.272l-.988-.988a1.508 1.508 0 00-.988.435 1.257 1.257 0 00-.276 1.027 1.318 1.318 0 00.987.988l.099.098.356-.039.671-.672c.208-.299.293-.667.237-1.027l-.098.178zM447.338 565.753l.593-.593a1.487 1.487 0 00-.257-1.205 1.312 1.312 0 00-1.007-.533l-.119.119-.296.098-.375.376a1.366 1.366 0 00-.356 1.066l.691.692c.215.135.459.217.712.237a.759.759 0 00.296-.02l.296-.296-.178.059zM429.442 553.269l-.257.059-.435.198-.474.474-.079.276v.514l.079.276.218.217c.103.157.238.292.395.396l.197.197.573.02.296-.059.237-.237c.267-.255.429-.6.455-.968l-.139-.139-.138-.375-.474-.474a1.608 1.608 0 00-.454-.375zM428.611 563.541l.119.118c.397.02.786-.114 1.086-.375a1.457 1.457 0 00.415-1.166l-.731-.73-.039-.237-1.087-1.087-.237.04-.375.138-.494.494-.158.395a1.387 1.387 0 00.415 1.086c.014.306.116.601.296.849l.435.435.355.04zM413.065 532.468l.119-.118.099-.297.098-.098a1.481 1.481 0 00-1.461-1.462l-.099-.099-.375.138-.475.474-.197.435-.158.158.099.099.039.276.553.553c.219.214.505.347.81.376.2-.002.399-.028.593-.079l.494-.494-.139.138zM432.74 540.212l-.138-.139a1.37 1.37 0 00-1.402.929l-.297.296.02.257.533.533c.267.244.608.391.968.415h.277l.81-.81a1.562 1.562 0 00-.198-1.303l-.454-.455-.119.277zM433.668 550.859l-.296-.02-.533.099-.158.158a1.784 1.784 0 00-.474.237c-.187.029-.367.09-.534.178l-.81.81a1.733 1.733 0 00.336 1.402l.356.356c.208.309.526.527.889.612l.098.099.277-.04.514-.513a1.62 1.62 0 00.691-.692l.059-.256.395-.395a1.549 1.549 0 00.022-1.187 1.544 1.544 0 00-.832-.848zM433.036 545.407l-.908.908a1.392 1.392 0 00.545 1.514c.157.11.334.186.521.225l.218-.02.138-.138a1.428 1.428 0 00.454-1.521 1.312 1.312 0 00-1.185-.948l.217-.02zM428.434 537.644l.928.928a1.396 1.396 0 001.501-1.106l-.059-.376-.652-.651a1.526 1.526 0 00-.711-.237l-.296.019-.968.968-.02.257.099.099.178.099z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M443.19 588.746a1.4 1.4 0 00-1.067.395 1.59 1.59 0 00-.435 1.304l.455.454a1.47 1.47 0 001.05.494 1.474 1.474 0 001.083-.415l.474-.474a1.499 1.499 0 00-.257-1.244 1.398 1.398 0 00-1.303-.514zM439.397 516.488l-.632-.633a1.503 1.503 0 00-1.185.396 1.41 1.41 0 00-.395 1.106l.316.316a1.255 1.255 0 001.047 1.047c.18.008.36-.012.533-.059l.79-.791a2.18 2.18 0 00.158-.632l-.118-.118-.099-.336-.138-.138-.277-.158zM435.526 511.154l.889.889a1.733 1.733 0 001.699-.829l.118-.119.059-.296-.019-.218-.415-.414a1.361 1.361 0 00-.968-.494 1.784 1.784 0 00-.889.178l-.138.138.039-.04-.415.415v.237l.099.099-.059.454zM438.054 543.234l-.158-.158a1.484 1.484 0 00-1.126.415 1.504 1.504 0 00-.415 1.165l1.008 1.007.276.04a1.42 1.42 0 001.146-.395 1.448 1.448 0 00.395-1.185l-.909-.909-.217.02zM448.86 513.505l-.139.138a1.205 1.205 0 00-.731 1.126l.06.296 1.106 1.107a1.33 1.33 0 00.638-.111c.2-.09.375-.229.508-.403.185-.224.295-.5.316-.79a1.345 1.345 0 00-.719-1.316 1.335 1.335 0 00-.506-.146l-.533.099zM466.973 493.613l.553.553c.141.07.297.104.455.099a1.368 1.368 0 001.363-.731 1.59 1.59 0 00-.218-1.758l-.335-.336-.297-.059a1.362 1.362 0 00-1.225.513 1.572 1.572 0 00-.197 1.541l-.099.178zM434.399 556.528l1.067 1.067.336.059a1.432 1.432 0 001.165-.494 1.561 1.561 0 00.257-1.481l-.711-.711-.296-.099h-.277l-.474.118-.178.178a1.627 1.627 0 00-.889 1.363zM486.885 538.335l-.395.119-.198.197a.952.952 0 00-.395.395l-.178.178.079.474-.098.099.138.375.731.731a1.248 1.248 0 00.948-.316 1.342 1.342 0 00-.731-2.351l.099.099zM463.852 518.977a1.567 1.567 0 00-1.58 1.106l-.02.257.04.434.987.988.316.039.139-.138.375-.138.514-.514.118-.316-.198-.197.277.039.079-.474-.039-.276-.929-.929-.079.119zM435.407 529.604l.257.02.099-.099.355-.119.85-.849.039-.237-.119-.119-.098-.335-.83-.83a1.315 1.315 0 00-1.067.355 1.46 1.46 0 00-.434 1.067l.138.336.81.81zM449.353 502.739a1.257 1.257 0 00-1.185.948v.277l.158.513c.08.237.227.445.423.6.197.154.434.248.683.27.166-.001.332-.021.494-.06l.533-.533.099-.296a1.312 1.312 0 00-.889-1.482l-.257-.02-.059-.217zM386.754 551.274l-.81-.81-.296-.02-.218.02-.336.098-.809.81a1.421 1.421 0 00.691 1.482l.198.197.513.001.296-.06.593-.592c.144-.24.225-.512.237-.791l-.118-.118.059-.217zM450.578 578.672l-.296-.06-.553.001-.297.098-.434.435-.158.355-.139.139a1.486 1.486 0 00.218 1.165 1.314 1.314 0 00.928.494l.277-.039.118-.119a1.487 1.487 0 00.534-1.205 1.48 1.48 0 00-.198-1.264zM434.143 572.667l-.04-.316a1.51 1.51 0 00-1.244-1.245l-.316.079-.751.751a1.651 1.651 0 00.059 1.402c.121.239.315.433.553.554a1.313 1.313 0 001.146.869l.336-.02 1.047-1.047v-.237l-.79-.79zM393.984 532.508h.316l.336-.099.81-.81a1.566 1.566 0 00-.257-1.126 1.448 1.448 0 00-1.067-.553l-.217.217-.336.138-.632.633v.237l.02.217.118.355.711.712.198.079zM423.673 485.356h.277l.434-.119.475-.474a1.51 1.51 0 00.375-.612l.118-.119-.039-.355-.79-.79a1.06 1.06 0 00-.474-.119 1.396 1.396 0 00-1.521 1.205 1.34 1.34 0 001.145 1.383zM438.073 504.418a1.314 1.314 0 00.85.849h.237l.395-.039.573-.573a1.355 1.355 0 00.336-1.007l-.83-.83-.356-.118a1.31 1.31 0 00-1.422 1.027l.02.217.197.474zM462.746 505.583l.415.415c.156.158.345.279.553.356l.257-.02.474-.079.928-.928a1.334 1.334 0 00-.699-1.329 1.345 1.345 0 00-.506-.153l-.296.02-1.067 1.067.02.217-.079.434zM416.325 541.496l.454-.455a1.336 1.336 0 00.08-1.382 1.266 1.266 0 00-.947-.728 1.252 1.252 0 00-1.128.392l-.573.573a1.382 1.382 0 00.297 1.047c.246.283.593.46.968.493l.849.06zM413.105 574.662l.593-.593c.119-.213.193-.448.217-.691l-.039-.316-.277-.514-.454-.454-.336-.06-.573.139-.118-.119a1.627 1.627 0 00-.988 1.422l.118.119.119.356.83.829.296.02.217-.02.336-.099.059-.019zM415.219 506.097l.414.415c.151.155.343.265.554.316l.217.02.395-.04.395-.395.217.217.218.02.098.099h.237l.81-.81a1.362 1.362 0 00-.039-1.264 1.316 1.316 0 00-1.047-.613 1.35 1.35 0 00-1.027.316l-.316-.316a1.372 1.372 0 00-1.423.83l-.138.138.04.435.395.632zM418.834 509.594a1.586 1.586 0 00.276 1.224c.254.323.622.535 1.028.593l.256-.02.376-.375c.294-.353.436-.807.395-1.264l-1.106-1.107-.395-.039-.119.118-.336.099-.138.138.02-.019-.257.256v.396zM448.444 531.777a1.453 1.453 0 00-1.422 1.422 1.367 1.367 0 001.224 1.462l.376-.138.494-.494.177-.376.316-.316-.138-.375-.494-.494-.533-.691zM446.666 570.612l-.099-.335-.632-.632-.316-.119-.138-.138-.138.138-.375.138-.494.494-.158.395-.119.119a1.257 1.257 0 002.331.711l.198-.474-.06-.297zM432.839 522.453l.217-.019.237-.238a1.502 1.502 0 00.455-1.6 1.427 1.427 0 00-1.225-.948l-.454.02-.652.652a1.462 1.462 0 00-.189.67 1.461 1.461 0 00.592 1.217c.19.14.411.231.644.266l.375-.02zM429.204 507.203l.731.731.277.04a1.537 1.537 0 001.619-1.422v-.237l-.079-.435-.77-.77-.316-.079h-.237l-.257.059-.612.612a3.46 3.46 0 00-.297.968l-.098.099.039.434zM427.98 563.699l-.869-.87-.336-.059a1.456 1.456 0 00-1.442.849l-.079.475.019.217.672.672c.218.128.46.209.711.237l.297-.02.474-.474a1.602 1.602 0 00.395-1.225l.158.198zM430.35 524.468v-.197l-.079-.356-.336-.336a1.512 1.512 0 00-.632-.434l-.237-.04-.415.059-.968.968-.02.257.02.217.138.376.494.493.395.158.099.099a1.283 1.283 0 001.541-1.264zM433.353 562.711a1.264 1.264 0 00-.909-.474l-.395.118-.415.415a.86.86 0 00-.316.632l-.909.909a1.85 1.85 0 00.277 1.146l.612.612c.273.02.545-.045.779-.186.233-.142.417-.353.525-.604l.158-.158-.04-.277c.163-.107.299-.25.398-.418.1-.167.16-.355.175-.55a1.464 1.464 0 00.06-1.165zM405.599 512.28l-.198-.198a1.136 1.136 0 00-.079-.355 1.258 1.258 0 00-.889-.889l-.099-.099-.375.059-1.027 1.028c-.047.329.023.665.197.948l.099.099a1.285 1.285 0 001.166 1.165l.138-.138a1.594 1.594 0 001.067-1.62zM427.762 496.951c.11.164.251.305.415.415l.138.138.415-.059.573-.573a1.342 1.342 0 00.336-1.047l-1.087-1.086a1.447 1.447 0 00-1.442.849l-.039.237v.435l.691.691zM367.949 549.417h-.277l-.908.908a1.593 1.593 0 00.217 1.245 1.427 1.427 0 001.086.573l.257-.02.909-.909a1.422 1.422 0 00-.7-1.621 1.467 1.467 0 00-.584-.176zM447.516 576.519l-.118-.119a1.286 1.286 0 00-.869-.869l-.218-.02-.395.079-.533.534a1.337 1.337 0 00-.356.987l.85.85.355.118.099.099a1.314 1.314 0 001.383-.988l-.198-.671zM408.245 561.308l-.276-.039-.514.039-.987.988a1.279 1.279 0 00.699 1.308c.158.079.33.125.506.134h.316l1.047-1.047a1.345 1.345 0 00-.791-1.383zM457.689 527.787a1.455 1.455 0 00-.434.197 1.286 1.286 0 00-.553 1.383 1.229 1.229 0 001.165.928 1.175 1.175 0 001.126-.533.414.414 0 00.138-.138l.139-.138.019-.218-.039-.276-.217-.455-.455-.454-.355-.079-.099-.099-.435-.118zM453.304 545.842a1.593 1.593 0 00-.435.039l-.671.672a1.412 1.412 0 00-.059 1.323c.101.213.253.398.443.539.189.142.41.235.643.271h.237l.336-.336a1.503 1.503 0 00.454-1.56 1.447 1.447 0 00-.948-.948zM415.219 566.543l-.336.336a1.46 1.46 0 00-.257 1.047l.909.909.316.079a1.372 1.372 0 001.183-2.205 1.337 1.337 0 00-.887-.502l-.098.099-.316.079-.119.119-.395.039zM469.502 573.733l-.751-.75-.276-.04a1.371 1.371 0 00-1.365 1.22 1.368 1.368 0 001.068 1.487l.119-.119.395-.118.533-.534c.155-.228.244-.495.257-.77l.02-.376zM404.73 544.241l.513.514.395.197.139.139a1.51 1.51 0 001.363-.731l.138-.139.039-.474-1.126-1.126a1.4 1.4 0 00-1.521 1.087v.237l.06.296zM385.154 540.113l-.889.889-.02.336.079.316.81.81c.362.008.714-.119.988-.356a1.34 1.34 0 00.454-.889 1.423 1.423 0 00-1.284-1.521l-.138.415zM397.954 562.869l-.356.118-.474.475-.138.335-.257.257.435.435c.123.201.292.371.494.494l.316.316.217.019.257-.059.434-.197.534-.534.099-.336.019-.217a1.476 1.476 0 00-1.58-1.106zM421.974 524.942l-.237.237-.039.316.098.099.099.336 1.067 1.067a1.57 1.57 0 001.126-.376 1.459 1.459 0 00.415-1.086l-.711-.711a1.428 1.428 0 00-.712-.237 1.273 1.273 0 00-.276 0l-.751.75-.079-.395zM403.209 571.64l-.554-.554-.474-.039-.237.039-.829.83-.178.415-.02.257c.043.303.166.589.356.829l.553.554.336.019.276-.079.474-.237.218-.217s.454-1.008.059-1.719l.02-.098zM378.932 522.038l-.534-.533a1.172 1.172 0 00-.711-.316 1.256 1.256 0 00-1.323 1.086l.02.218.118.118a1.346 1.346 0 001.086 1.087l.119.118.395-.079.336-.336a1.5 1.5 0 00.434-.632l.02-.296.04-.435zM401.115 546.789l-1.047-1.047a1.48 1.48 0 00-1.541 1.344l.02.217.118.395.988.988.237-.04.514-.513a1.666 1.666 0 00.395-1.225l.316-.119zM399.356 528.478l.968.968.119-.118.375-.099.85-.85-.119-.118-.099-.336-.434-.435a1.565 1.565 0 00-.83-.355l-1.047 1.047-.02.257.119.118.118-.079zM468.001 535.135h.237l.355-.079.85-.849a1.423 1.423 0 00-1.047-1.758l-.395.079-.85.849c-.086.397-.023.812.178 1.165.157.263.392.47.672.593zM501.917 541.278a1.479 1.479 0 00-1.067-.553h-.237l-.671.672a1.622 1.622 0 000 1.501 1.288 1.288 0 001.027.672c.318.033.637-.059.889-.257l.691-.691a1.484 1.484 0 00-.632-1.344zM470.134 552.637l-.277.039-.928.929a1.4 1.4 0 00.155.877c.147.267.376.479.655.604l.355.119.296.019.317-.079.79-.79a1.403 1.403 0 00-.13-.956 1.403 1.403 0 00-.72-.644l-.138.139-.375-.257zM457.136 537.189l.276.277a1.315 1.315 0 001.304-1.304 1.376 1.376 0 00-.287-1.011 1.358 1.358 0 00-.918-.51l-.356.119-.197.197a1.152 1.152 0 00-.376.375.536.536 0 00-.138.139 1.455 1.455 0 00-.099.573l.751.75.04.395zM456.168 558.503l-.197-.197a1.266 1.266 0 00-.415-.415l-.178-.178-.494-.059-.257.059-.928.929-.02.335.099.099.118.356.85.849c.293.02.584-.05.835-.203.252-.152.449-.377.567-.646l.158-.158-.039-.474-.099-.297zM457.946 540.725l-.513-.513a1.152 1.152 0 00-.455.059l-.355-.158-.119-.119a1.408 1.408 0 00-1.106.435 1.245 1.245 0 00-.336 1.086l1.067 1.067.118-.118.435.434a1.27 1.27 0 00.932-.282c.129-.107.235-.238.312-.387.078-.148.125-.31.139-.477a1.174 1.174 0 00-.198-.75l.079-.277zM444.059 518.107a1.283 1.283 0 001.106.83c.172.017.346.004.514-.039l.651-.652a1.33 1.33 0 00-.118-1.225 1.509 1.509 0 00-1.106-.711 1.286 1.286 0 00-1.363 1.047l.019.257.297.493zM459.349 540.745l.256-.257a1.339 1.339 0 00.336-1.244 1.39 1.39 0 00-.908-.909l-.257-.02-.415.06-.928.928-.04.395.04.277.948.948.118-.119.85-.059zM484.988 554.493a.834.834 0 00-.315 0l-.297.297a1.527 1.527 0 00-.987.987 1.343 1.343 0 00.533 1.521l.158.158.474.04h.197l.356-.119 1.047-1.047a1.287 1.287 0 00-1.166-1.837zM469.107 560.301l-.099-.099-.099-.296-.237-.237a1.245 1.245 0 00-.433-.447 1.249 1.249 0 00-.594-.185.78.78 0 00-.375.02l-.968.967-.02.534.079.316.81.81.237.039a1.284 1.284 0 00.988-.355 1.198 1.198 0 00.711-1.067zM470.371 520.478l.889.889c.024.04.058.074.099.098a1.401 1.401 0 001.067-.355 1.52 1.52 0 00.434-1.185l-.099-.099-.118-.316-.771-.771a1.37 1.37 0 00-1.481 1.205v.198l-.02.336zM467.645 546.948l.296-.297a1.451 1.451 0 00-.237-1.224 1.423 1.423 0 00-1.007-.534l-.099.099-.316.118-.395.396a1.25 1.25 0 00-.316 1.027l.711.711c.211.141.457.223.711.237.099.012.2.005.296-.02l.692-.691-.336.178zM479.141 546.632a1.447 1.447 0 00.316-1.225l-.928-.929a1.4 1.4 0 00-1.422.83l-.099.099-.04.276.415.415c.237.275.568.451.928.494.265.105.557.119.83.04zM446.903 527.075l.277.277a1.387 1.387 0 001.086-.415 1.31 1.31 0 00.395-.79c.069-.111.122-.23.158-.356l.02-.217-.059-.375-.514-.514a1.201 1.201 0 00-.513-.316l-.119-.118-.395.039-.474.474c-.15.166-.265.361-.336.573l-.237.237c-.084.384-.021.787.178 1.126l.474.474.059-.099zM445.541 539.738l.454.454c.251.243.58.39.929.415a1.395 1.395 0 001.283-.652 1.477 1.477 0 00.001-1.541l-.593-.592-.277-.04a1.507 1.507 0 00-1.145.395 1.468 1.468 0 00-.395 1.185l-.257.376zM446.133 550.602a1.39 1.39 0 00.336-1.244l-.474-.475a1.237 1.237 0 00-.83-.395c-.23-.021-.462.02-.672.119l-.77.77a1.485 1.485 0 00.178 1.205 1.306 1.306 0 001.027.593 1.446 1.446 0 001.205-.573zM454.508 568.459l-.138-.336-.77-.77-.336-.02-.119.119-.375.099-.83.829-.019.257.138.138.158.395.514.514.375.138h.276a1.364 1.364 0 001.306-.86c.068-.172.102-.357.097-.542l-.277.039zM471.675 528.103a1.376 1.376 0 00.286 1.011 1.37 1.37 0 00.919.51l.691-.692a1.335 1.335 0 00.306-1.494 1.35 1.35 0 00-.306-.442l-.395-.395-.336-.059a1.375 1.375 0 00-.908.537 1.355 1.355 0 00-.257 1.024zM441.867 548.35a1.453 1.453 0 00.513-1.541 1.303 1.303 0 00-.365-.701 1.303 1.303 0 00-.701-.365l-.218.019-.118.119a1.315 1.315 0 00-.534 1.007 1.652 1.652 0 00.455 1.205l.592.593.218-.02.316-.118-.158-.198zM453.086 521.801l.81.81.217.02a1.386 1.386 0 001.14-.401 1.392 1.392 0 00.401-1.14l-.632-.632a1.625 1.625 0 00-.691-.257l-.297.02-.671.672a1.448 1.448 0 00-.218 1.007l-.059-.099zM444.671 545.782l.395-.039.514-.514c.16-.158.276-.356.336-.573l.019-.217-.059-.375-.494-.494a1.36 1.36 0 00-.553-.356l-.118-.118-.396.039-.493.494c-.168.164-.29.368-.356.593l-.02.217.06.375.493.494c.156.158.345.279.554.356l.118.118zM448.227 545.19l.534.533.256.059a1.417 1.417 0 001.126-.415 1.562 1.562 0 00.435-1.185l-.85-.849a2.77 2.77 0 00-.612-.178 1.4 1.4 0 00-1.343.79l-.138.138-.04.435.296.296.336.376zM444.098 542.839l.395-.04.376-.375a1.428 1.428 0 00.493-1.56 1.307 1.307 0 00-1.165-.968l-.356.079-.73.731c-.158.156-.28.344-.356.553l-.099.098.059.415.494.494c.151.156.343.265.553.316l.336.257zM444.236 533.693l.415-.059.494-.494c.159-.155.281-.344.356-.553l.019-.218-.059-.375-.415-.415a1.344 1.344 0 00-.533-.336l-.217-.019-.395.039-.514.514a1.657 1.657 0 00-.356.553l-.019.217.079.395.513.514c.149.153.332.268.534.336l.098-.099zM449.847 554.157l-1.008-1.007-.375-.02-.217.02-.198.198a1.509 1.509 0 00-1.554.851 1.5 1.5 0 00-.145.61l.079.316.731.731c.147.052.3.085.455.099a1.487 1.487 0 001.126-.336 1.331 1.331 0 001.011-.391 1.335 1.335 0 00.391-1.011l-.296-.06zM454.271 552.38l.455.454a1.312 1.312 0 001.066-.474 1.425 1.425 0 00.277-1.501 1.341 1.341 0 00-1.146-.83 1.23 1.23 0 00-.928.336c-.148.13-.266.292-.344.473a1.33 1.33 0 00-.111.574l.119.118a1.278 1.278 0 00.612.85zM463.595 529.545l.316.039.198-.039.355-.119.178-.099.198-.158.118-.158.178-.336.059-.217v-.355l-.138-.376-.079-.158-.198-.197-.158-.119-.335-.178-.178-.059-.336-.02-.198.04-.375.099-.158.118-.197.158-.119.158-.178.336-.059.178-.02.178.04.197.118.356.099.178.158.197.178.138.336.178.197.04zM472.148 565.516l-.316-.316a1.452 1.452 0 00-.415-.415l-.375.059-.415.059-.573.573a1.343 1.343 0 00-.316 1.028l.988.987.217.02.119-.118.356-.119.849-.849-.099-.099-.02-.81zM458.084 570.336l-.217.019-.355.119-.81.81a1.43 1.43 0 00.195 1.118 1.43 1.43 0 00.95.62l.099.099.316-.079.79-.79a1.541 1.541 0 00-.079-1.343 1.394 1.394 0 00-.889-.573zM475.803 587.304a1.346 1.346 0 00-1.119.382 1.338 1.338 0 00-.382 1.119l.79.791c.179.13.392.205.612.217.345.018.684-.095.948-.316l.455-.455a1.539 1.539 0 00-.198-1.185 1.364 1.364 0 00-1.106-.553zM473.215 535.411l-.889.889a1.61 1.61 0 00-.099.415l-.118.119.079.395 1.007 1.007.692-.691a1.43 1.43 0 00.039-1.383 1.536 1.536 0 00-.711-.751zM478.332 566.326l-.336.02-1.047 1.047.099.335.395.395c.158.161.356.276.573.336l.256.02.475-.119.553-.553.158-.355.118-.119a1.342 1.342 0 00-1.244-1.007zM451.763 542.128l1.521-1.521-.118-.119-.138-.375-.83-.83a1.28 1.28 0 00-1.047.376 1.055 1.055 0 00-.198.197l-.908.909-.198.197-.178.178-.039.474.039.237.929.929a1.425 1.425 0 001.205-.494l-.04-.158zM459.645 554.869a1.311 1.311 0 00-1.067.355 1.424 1.424 0 00-.454 1.047l1.126 1.126a1.395 1.395 0 001.066-.355 1.447 1.447 0 00.435-1.067l-1.106-1.106zM480.366 507.757l.336-.336a1.342 1.342 0 00-1.264-1.541h-.277l-1.047 1.047-.039.435.039.276.929.928a1.2 1.2 0 001.323-.809zM394.003 560.42l.415-.415c.141-.179.231-.393.263-.618a1.295 1.295 0 00-.085-.666 1.29 1.29 0 00-.514-.514l-.573-.573-.217-.02a1.709 1.709 0 00-1.304.435l-.454.454c-.058.308-.016.626.118.909l1.205 1.205.297-.02.118-.118.731-.059zM394.26 585.092l-.948-.948a1.478 1.478 0 00-1.422.869l-.039.276.493.494a1.375 1.375 0 00.968.494c.194.169.436.272.692.296a1.397 1.397 0 00.316-1.185l-.06-.296zM424.049 544.083l.177.178.514.039.277-.039.948-.948c.104-.359.249-.703.434-1.027l.158-.158.02-.455-1.027-1.027a1.404 1.404 0 00-1.126.454l-.474.474.059-.059-.592.593-.079.355-.139.139a1.366 1.366 0 00.85 1.481zM423.93 538.908l.494-.494a1.616 1.616 0 00-.276-1.738l-.217-.218-.573-.098-.257.019-.257-.019-.415.059-.968.968-.039.316.138.138.178.415.691.691.553.119.297-.02.177-.178.474.04zM422.587 550.404l-.534-.533-.415-.178-.138-.138a1.438 1.438 0 00-1.047.454 1.288 1.288 0 00-.296 1.047l.829.83.336.099.099.099a1.314 1.314 0 001.383-.988v-.237l-.217-.455zM422.449 570.929l-.395.118-.415.415a1.121 1.121 0 00-.336 1.007 1.285 1.285 0 00.573.85l.415.415c.336.024.668-.082.928-.297a1.223 1.223 0 00.435-.869 1.407 1.407 0 00-.669-1.459 1.396 1.396 0 00-.536-.18zM395.762 522.71l.375.138c.345-.028.664-.192.889-.454a1.377 1.377 0 00.296-1.047 1.302 1.302 0 00-.384-.939 1.318 1.318 0 00-.939-.385l-.751.751-.118.395.553.553.079.988zM395.248 470.047l.771-.771.118-.355a1.179 1.179 0 00-.098-.454l-.771-.771-.395-.118a1.34 1.34 0 00-.889.454 1.23 1.23 0 00-.296.968 1.393 1.393 0 001.56 1.047zM405.085 530.019l-.118-.356-.277-.276a1.363 1.363 0 00-1.165-.534 2.303 2.303 0 00-.395.079l-.731.731-.059.296-.099.099.019.297 1.127 1.126h.237a1.371 1.371 0 001.303-1.304l.158-.158zM392.423 506.413l-.829-.829a1.122 1.122 0 00-.435-.04 1.338 1.338 0 00-.968.534 1.477 1.477 0 00-.217 1.244l.849.849.395.04a1.369 1.369 0 001.205-1.442l.099-.099-.099-.257zM394.438 533.08l-.494-.493-.296-.099h-.395a1.258 1.258 0 00-1.007.612 1.312 1.312 0 00-.02 1.324l.908.908a1.535 1.535 0 001.284-.968l.04-.237-.02-.454-.138-.138.138-.455zM390.606 493.889l.257.02a1.283 1.283 0 001.047.375 1.633 1.633 0 001.145-.671l.139-.138.098-.336-.612-.613a1.482 1.482 0 00-1.047-.375 1.453 1.453 0 00-.849-.494l-.218.02-.395.119-.158.158-.75.75a1.651 1.651 0 00.671 1.343l.672-.158zM397.52 480.339l.257.256.217.02h.276a1.309 1.309 0 001.205-.968 1.48 1.48 0 00-.553-1.62l-.375-.375h-.277a1.314 1.314 0 00-1.205.968 1.483 1.483 0 00.455 1.719zM380.314 551.155l-.217.02-.237.237a1.23 1.23 0 00-.79.79l-.099.099.118.118c.154.261.372.479.633.633l.118.118.336.059.257-.019.494-.494a1.588 1.588 0 00.236-1.62 1.254 1.254 0 00-.849.059zM411.189 500.685l.671.671.316.079a1.34 1.34 0 001.265-1.066l.019-.218-.078-.434-.593-.593a1.395 1.395 0 00-.573-.217l-.119.119-.414.098-.573.573a1.559 1.559 0 00-.158 1.067l.237-.079zM412.276 515.401a1.397 1.397 0 00.988-.553 1.495 1.495 0 00.177-1.205l-.77-.77-.316-.079h-.198l-.849.849-.099.375-.099.099.02.257.889.889.336.099-.079.039zM389.46 501.396l.119-.119.356-.118.414-.415c.2-.187.326-.439.356-.711l.02-.217-.099-.376-.691-.691a1.39 1.39 0 00-.751-.158l-.296.059-.889.889a1.455 1.455 0 00.355 1.106 1.374 1.374 0 001.106.751zM393.016 569.783l-.475-.474a1.871 1.871 0 00-.849-.178 1.84 1.84 0 00-.474.118l-.375.376a1.31 1.31 0 00-.139 1.56l.83.83.336.02a1.54 1.54 0 001.284-1.087v-.237l-.06-.573-.078-.355zM413.52 507.322l-.039.277.039.434 1.008 1.008.257-.02a1.366 1.366 0 001.303-1.304l-.039-.237-.415-.415a1.225 1.225 0 00-.988-.355 1.396 1.396 0 00-1.126.612z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M416.009 538.414a1.594 1.594 0 00-.119.83l.85.849.375.099h.198a1.507 1.507 0 001.244-.929l.04-.237-.04-.474-.573-.573a1.202 1.202 0 00-.711-.355h-.237l-.435.158-.573.573-.019.059zM412.335 505.031c.135-.003.269-.03.395-.079l.395-.396a1.435 1.435 0 00.316-1.343 1.538 1.538 0 00-1.047-1.047h-.277l-.494.494a1.424 1.424 0 00-.237 1.62 1.34 1.34 0 00.949.751zM425.668 540.212l-.849-.85-.296-.059a1.509 1.509 0 00-.988.514 1.421 1.421 0 00-.257 1.125l.988.988.375.02a1.395 1.395 0 001.225-1.225l-.04-.237-.158-.276zM410.992 527.431l.296-.296.079-.356.02-.217-.02-.218-.316-.316a1.4 1.4 0 00-1.185-.474 1.454 1.454 0 00-1.264 1.502l.138.138.198.395.493.494c.238.095.495.136.751.118l.415-.415.395-.355zM404.651 510.305a1.09 1.09 0 00-.119.513 1.875 1.875 0 00-.336.652 1.204 1.204 0 00-.592.395l-.87-.869a2.282 2.282 0 00-.474-.039 1.284 1.284 0 00-1.106 1.106l-.02.217.079.356.652.651.376.139.237.039.691-.691.711.711a1.368 1.368 0 001.245.612l.316.08.237-.001.138-.138.277-.276.395-.395a1.36 1.36 0 00.296-.613l.119-.316.118-.118c.246-.211.402-.508.434-.83l-.039-.277-.968-.968-.434-.079-.257.02a1.866 1.866 0 00-.514.277 1.28 1.28 0 00-.592-.158zM445.995 491.4l.375-.177.435-.435a1.063 1.063 0 00.138-1.007 1.326 1.326 0 00-.29-.441 1.326 1.326 0 00-.441-.29l-.158-.158-.375-.099a1.244 1.244 0 00-.869.437 1.265 1.265 0 00-.296.926c.037.305.16.592.355.83a1.315 1.315 0 001.126.414zM418.102 504.477l-.237.237a1.286 1.286 0 00-.77 1.482l.494.494c.258.276.611.445.988.474l.237-.04.276-.276a1.533 1.533 0 00.198-1.699 1.317 1.317 0 00-1.324-.77l.138.098zM418.024 514.394l.375.375.375.138a1.418 1.418 0 00.972-.499 1.435 1.435 0 00.332-1.041 1.566 1.566 0 00-.513-1.028 1.393 1.393 0 00-1.028-.316l-.375.178-.158.158a1.37 1.37 0 00-.099 2.114l.119-.079zM420.275 508.942l.573.573.297.098.098.099c.241-.02.472-.101.672-.237l.83-.83a1.398 1.398 0 00-.139-1.363 1.307 1.307 0 00-1.165-.572 1.219 1.219 0 00-.869.434 1.814 1.814 0 00-.277.277 1.365 1.365 0 00-.118 1.185l.098.336zM423.357 503.134l-.079.553.02.297.83.829.296.06h.197a1.398 1.398 0 001.383-1.383 1.345 1.345 0 00-.099-.929 1.31 1.31 0 00-.79-.79l-.316.04-.671.671c-.123.22-.223.452-.297.692l-.474-.04zM411.465 478.521a1.279 1.279 0 001.252-1.37 1.29 1.29 0 00-.224-.645l-.514-.513-.257-.06-.257.02-.81.81a1.262 1.262 0 00.158 1.264c.161.229.389.401.652.494zM442.143 539.145l-.376.099-.513.513a1.345 1.345 0 00-.277.632l.02.218.119.395.513.513c.17.135.376.218.593.237l.237.001.355-.119.534-.533c.136-.179.225-.39.257-.613l-.02-.217-.099-.375-.533-.534a1.4 1.4 0 00-.573-.217h-.237zM426.933 534.681l-.395-.395a1.352 1.352 0 00-.553-.238l-.277.04-.474.158-.553.553-.099.376-.019.217a1.302 1.302 0 00.384.939 1.315 1.315 0 00.939.384.503.503 0 00.297-.059l.75-.751.079-.316.099-.098-.119-.119-.059-.691zM435.328 519.925l.218-.218.217-.019.751-.751.098-.375a1.316 1.316 0 10-2.627-.178 1.307 1.307 0 00.303 1.091 1.316 1.316 0 001.04.45zM425.609 519.411l-.335-.138-.119-.119h-.277a1.346 1.346 0 00-1.086.652 1.378 1.378 0 00-.099 1.245l.85.849.355.119.217.019.237-.039.83-.83.119-.355.019-.218v-.474l-.711-.711zM394.813 528.478l-.118-.119a1.4 1.4 0 00-1.304-.711 1.534 1.534 0 00-1.264 1.027 1.456 1.456 0 00.494 1.561l.138.138.395.119.217.02.257-.06.138-.138a1.419 1.419 0 00.711-1.422l.336-.415zM450.34 554.77l.158.158.198.435.316.316a1.435 1.435 0 001.166.494l.237-.04.711-.711c.146-.336.174-.713.079-1.067l-.929-.928a1.71 1.71 0 00-.612-.06 1.4 1.4 0 00-1.324 1.403zM348.195 541.002a1.226 1.226 0 00-1.126.889v.316l.138.375c.044.309.19.594.415.81a1.237 1.237 0 00.948.276l.237-.237a1.308 1.308 0 00.573-1.521 1.43 1.43 0 00-1.185-.908zM409.708 456.298a1.392 1.392 0 001.995 1.008l.434-.435a1.314 1.314 0 00.139-1.323l-.396-.395-.434-.237-.217.217.079-.316a1.511 1.511 0 00-.731-.138l-1.008 1.007.04.356.099.256zM436.671 561.012l-.651-.652a1.424 1.424 0 00-1.363 1.363 1.284 1.284 0 001.244 1.245l.494-.494a1.317 1.317 0 00.454-1.719l-.178.257zM417.827 527.194a1.362 1.362 0 001.066-.751l.04-.237-.04-.474-.888-.889-.356-.079-.099-.098a1.31 1.31 0 00-1.185 1.185l.099.099.059.335.474.474c.214.242.509.397.83.435zM396.117 528.359l.375.178.237.237a1.589 1.589 0 001.225-.671c.083-.061.156-.135.217-.217.309-.237.514-.584.573-.968a1.873 1.873 0 00-.731-1.877l-.138-.138-.395-.079h-.198l-.237.039-.553.553a1.81 1.81 0 01-.138.139 1.447 1.447 0 00-.79.79l-.316.316c-.023.202.001.407.07.598.068.192.179.366.325.508l.415.415.059.177zM365.361 486.818l.632.632.336.099.098.098.139-.138a1.272 1.272 0 001.081-1.093 1.24 1.24 0 00-.054-.527l-.119-.118a1.283 1.283 0 00-1.106-.83l-.356.119-.671.671a1.363 1.363 0 00-.079 1.067l.099.02zM382.171 498.769l-.356.355a1.374 1.374 0 00.455 1.166 1.304 1.304 0 001.066.355l.711-.711a1.45 1.45 0 00.198-1.106l-.79-.79a1.17 1.17 0 00-.751-.119l-.276.04-.336.336.079.474zM401.826 508.27l1.027-1.027.02-.296-.059-.494-.415-.415a1.484 1.484 0 00-.395-.395l-.297-.059-.573.059-.651.652a1.507 1.507 0 00-.257 1.047l.889.889c.228.08.475.093.711.039zM401.016 498.018l-.119-.119a1.413 1.413 0 00-.552.172 1.446 1.446 0 00-.692 1.586l1.442 1.443.395.158.237.039.257-.02.908-.908.099-.415a1.293 1.293 0 00-.573-1.008 1.65 1.65 0 00-.434-.79l-.119-.118-.849-.02zM421.283 526.226l-.118.119-.059.296a1.453 1.453 0 001.56 1.324l.869-.87.139-.454.019-.257-.691-.691.138.138a1.279 1.279 0 00-.849-.217 1.499 1.499 0 00-.593.158l-.454.454h.039zM400.66 521.683h.237a1.399 1.399 0 001.186-1.185l.039-.277-.099-.099-.059-.257-.553-.553a1.792 1.792 0 00-1.027-.237l-.277.04-.81.81c-.027.455.114.905.395 1.264l.297.296.671.198zM398.073 511.391l.316-.04.513-.197.158-.158c.152-.095.281-.223.376-.375.177-.058.344-.145.493-.257l.87-.869a1.736 1.736 0 00-.534-1.363l-.335-.336a1.515 1.515 0 00-.968-.494l-.277.04-.434.434a2.07 2.07 0 00-.573.81v.277l-.178.177-.04.316a1.534 1.534 0 00.613 2.035zM399.495 516.606l1.007-1.007a1.398 1.398 0 00-1.323-1.561l-.217.02-.158.158a1.398 1.398 0 00-.218 1.56 1.316 1.316 0 001.344.751.513.513 0 00-.435.079zM404.512 522.769l-.257.02a1.37 1.37 0 00-1.303 1.304l.118.118.099.336.652.652c.238.096.495.137.751.118l.276-.039.771-.771.098-.296-.098-.099v-.237l-1.107-1.106zM382.685 475.42a1.397 1.397 0 00.987-.553 1.53 1.53 0 00.237-1.344l-.494-.493a1.402 1.402 0 00-1.066-.316 1.423 1.423 0 00-1.047.572l-.454.455a1.553 1.553 0 00.493 1.205 1.284 1.284 0 001.344.474zM397.797 546.197l.474.474.118-.119a1.392 1.392 0 001.106-.592 1.305 1.305 0 00.218-1.126l-.336-.336a1.312 1.312 0 00-1.284-.77 1.397 1.397 0 00-.494.138l-.672.672a1.471 1.471 0 00-.079.671l.672.672.277.316zM402.477 550.839l-.79-.79-.098.099a1.79 1.79 0 00-1.561 1.086l-.02.296.692.692a1.339 1.339 0 001.027.316c.3-.028.587-.137.829-.316l.475-.474.098-.376.139-.138-.04-.237-.118-.119-.633-.039zM394.892 519.549h.277a1.453 1.453 0 001.047-.573 1.432 1.432 0 00.217-1.244l-1.007-1.007h-.277a1.453 1.453 0 00-1.047.572 1.593 1.593 0 00-.217 1.245l1.007 1.007zM388.927 550.602l.138-.138a1.2 1.2 0 00.534-1.245l-.85-.849-.395-.119c-.219.016-.43.088-.613.21a1.275 1.275 0 00-.434.482 1.174 1.174 0 00-.178.81 1.354 1.354 0 00.393.95 1.334 1.334 0 00.951.393l.454-.494zM374.23 573.042l-.573-.573a1.887 1.887 0 00-.474-.039 1.421 1.421 0 00-1.225.948 1.566 1.566 0 00.514 1.699l.434.434h.316a1.334 1.334 0 001.107-.711 1.596 1.596 0 00-.04-1.541l-.059-.217zM396.315 505.919l-.968-.967-.336-.02a1.394 1.394 0 00-1.106.672 1.619 1.619 0 000 1.501l.711.711.316.039.277-.039.434-.198.198-.197a1.539 1.539 0 00.474-1.502zM347.444 532.113l.474-.474c.16-.1.295-.235.395-.396a.54.54 0 00.158-.158l-.138-.454-.178-.375-.494-.494-.395-.118a1.312 1.312 0 00-1.165 1.402 1.367 1.367 0 001.343 1.067zM373.242 547.56a1.512 1.512 0 001.363-1.363v-.237l-.098-.454-.534-.534-.415-.178-.256-.019h-.316l-.87.869-.059.336.178.177h-.277l-.019.494.908.909.277.079.118-.079zM399.811 532.686l-.217-.02-.968.968-.119.355-.118.119v.237l.81.81.375.138.119.119a1.331 1.331 0 001.248-1.058c.041-.192.04-.391-.004-.582l-.909-.909-.217-.177zM390.132 561.328a1.315 1.315 0 001.086-1.086l-.059-.297-.217-.454a1.279 1.279 0 00-1.225-.671 1.403 1.403 0 00-.494.138l-.731.731-.039.276a1.284 1.284 0 001.205 1.205h.237l.237.158zM444.296 503.529l.435.435.336.099.217.019.296-.019.81-.81.119-.356.118-.118a1.457 1.457 0 00-.908-1.344l-.297-.059-.494.059-.79.791c-.124.25-.159.536-.099.809l.119.119.138.375zM376.936 486.541l.297.02.553-.119.612-.612.119-.395.138-.138a1.318 1.318 0 00-.395-1.107 1.286 1.286 0 00-.988-.355l-.454.454a1.373 1.373 0 00-.336 1.245c.016.381.179.742.454 1.007zM393.589 489.484l.454.455a.848.848 0 00.099.336 1.48 1.48 0 001.442 1.007l.099-.099.316-.118.75-.751c.054-.237.057-.482.009-.721a1.752 1.752 0 00-.285-.662 1.505 1.505 0 00-.652-.454l-.217-.217a1.316 1.316 0 00-1.245-.692l-.336.099-.731.731-.079.316v.237l.139.138.237.395zM440.128 523.184l-.316.039-.79.791-.138.335-.099.099a1.439 1.439 0 00.415 1.087c.311.268.716.403 1.126.375l.928-.928.099-.336-.099-.099v-.237l-.849-.85-.356-.118.079-.158zM418.222 574.405l-.158.158-.356.158-.395.395a1.392 1.392 0 00-.296.652l.099.336.809.81c.156.032.315.046.474.039a1.422 1.422 0 00.869-2.357 1.423 1.423 0 00-.888-.468l-.158.277zM400.937 557.95a1.288 1.288 0 00-.948-.671h-.237l-.395.118-.573.573a1.371 1.371 0 00-.158 1.067l.83.829.375.06a1.343 1.343 0 001.245-1.245l-.139-.138v-.593zM376.482 560.637l-.415-.415a1.287 1.287 0 00-.612-.257l-.237.04-.455.178-.553.553-.099.375-.019.217c-.005.178.027.354.094.519a1.31 1.31 0 001.249.825l.276-.04.751-.751.119-.316-.218-.217.119-.711zM416.621 517.85l-.652.652a1.4 1.4 0 00.119 1.383 1.257 1.257 0 001.185.553 1.221 1.221 0 00.948-.553l.237-.237.079-.316a1.569 1.569 0 00-.434-.988 1.352 1.352 0 00-1.047-.335l-.435-.159zM414.547 484.645l-.316.316-.158.395a1.783 1.783 0 00-.118.711l.849.85.336.098h.316l.533-.217.119.119a1.508 1.508 0 00.75-1.541l-.829-.83-.356-.118-.099-.099-.335.02-.356.355-.336-.059zM423.318 552.676l-.415-.415a1.118 1.118 0 00-.593-.237h-.197l-.415.099-.474.474a.843.843 0 00-.336-.099h-.197l-.139.139-.237.039-.81.81a1.287 1.287 0 001.363 1.679c.366-.03.705-.2.949-.474l.316.079.098.099a1.45 1.45 0 001.265-1.027l.019-.257-.098-.415-.099-.494zM419.189 549.733a1.557 1.557 0 00-.454-1.165 1.47 1.47 0 00-1.126-.415l-.652.652a1.684 1.684 0 00-.178 1.284l.791.79.335.098.099.099.395-.039.81-.81.099-.336-.119-.158zM386.438 532.508a1.48 1.48 0 001.284-1.6 1.345 1.345 0 00-.455-.889 1.316 1.316 0 00-.987-.316l-.751.75-.138.376.968.968.079.711zM382.092 493.87l.119.316.237.237.513.316.317.079.237-.04.967-.967.079-.395v-.238a1.224 1.224 0 00-1.069-1.193 1.225 1.225 0 00-1.34.878l-.119.513.059.494zM403.327 539.244l-.671.671a1.482 1.482 0 00-.198 1.62 1.428 1.428 0 001.383.751l.415-.099.573-.573a1.408 1.408 0 00-.139-1.323 1.477 1.477 0 00-1.363-1.047zM409.332 553.703l-.79-.79h-.277a1.542 1.542 0 00-1.027.559 1.543 1.543 0 00-.336 1.12l.119.119.138.415.514.513.355.119.099.098.296.02.237-.039.791-.791c.103-.325.15-.666.138-1.007l-.119-.118-.138-.218zM401.648 497.741l.849.85h.356a1.426 1.426 0 001.185-1.185v-.475l-.039-.237-.692-.691a1.368 1.368 0 00-.731-.138l-.296.059-.533.533a1.76 1.76 0 00-.198 1.304l.099-.02zM405.461 536.834l.118.118.119.356.434.434c.191.18.433.297.692.336h.237l.434-.118.534-.534.158-.395.039-.237-.039-.237-.909-.908-.395-.079-.099-.099a1.313 1.313 0 00-1.323 1.363zM396.493 499.539a1.284 1.284 0 00.968.336l.691-.692a.977.977 0 00.217-.691l.672-.671.079-.277a1.563 1.563 0 00-.435-1.106l-.454-.455-.217-.019-.099-.099a1.336 1.336 0 00-1.185.988l-.02.256.079.277a1.257 1.257 0 00-.415 1.047c-.103.37-.06.766.119 1.106zM431.851 545.012l.178.178.138.375a1.28 1.28 0 001.107.632l.118-.118.336-.139.652-.652.118-.355.119-.119a1.399 1.399 0 00-.356-.908l-.099-.099a1.31 1.31 0 00-1.323-.968l-.415.415a1.598 1.598 0 00-.568.787 1.601 1.601 0 00-.005.971zM412.75 563.877l-.415-.415a1.337 1.337 0 00-.553-.237l-.217.02-.415.098-.573.573a1.344 1.344 0 00-.178 1.087l.909.908.316.079a1.344 1.344 0 001.264-1.067l.02-.217-.079-.434-.079-.395zM463.082 502.324l.276-.039.909-.909a1.488 1.488 0 00-.435-1.185 1.421 1.421 0 00-1.126-.415 1.585 1.585 0 00-.256.059l-.988.988a1.56 1.56 0 00.454 1.166 1.363 1.363 0 001.166.335zM380.235 488.102l.138.138a1.26 1.26 0 001.008.691l.217-.019.375-.139.573-.573c.178-.311.234-.677.158-1.027l-.849-.849-.395-.079a1.313 1.313 0 00-1.205 1.205l.099.099-.119.553zM421.52 496.991h.277l.494-.178.573-.573.118-.395.02-.217a1.311 1.311 0 00-1.422-1.225l-.316.04-.731.73-.079.317-.099.098a1.424 1.424 0 001.165 1.403zM377.884 537.979c.13-.092.244-.205.336-.335a1.231 1.231 0 00.316-1.423 1.254 1.254 0 00-1.284-.77 1.338 1.338 0 00-1.047.731l-.118.118-.02.218 1.086 1.086.336.02.099.099.296.256zM379.445 519.352c.136-.003.269-.03.395-.079l.593-.593a1.565 1.565 0 00-.158-1.343 1.539 1.539 0 00-1.205-.612l-.573.573a1.479 1.479 0 00-.197 1.619 1.39 1.39 0 001.145.435zM413.777 492.922l.316-.316c.157-.342.178-.73.06-1.087l-.909-.908a1.668 1.668 0 00-.316 0 1.366 1.366 0 00-1.103 1.413 1.372 1.372 0 001.241 1.293l.612-.613.099.218zM359.04 494.403l.73.731h.277a1.328 1.328 0 00.923-.462 1.342 1.342 0 00.321-.98 1.3 1.3 0 00-.384-.939 1.318 1.318 0 00-.939-.384l-.138.138-.376.177-.513.514a1.596 1.596 0 00-.119.83l.218.375zM427.723 513.347l-.533-.534-.415-.098-.217-.02a1.56 1.56 0 00-1.245.928l-.039.237.059.455.849.849.257.059.218-.019a1.342 1.342 0 001.303-1.304l-.039-.237-.198-.316zM447.575 514.236l.474-.474.139-.376.118-.118-.02-.297-.928-.928-.375-.138a1.452 1.452 0 00-.929.494 1.284 1.284 0 00-.296.968 1.38 1.38 0 00.942 1.221c.174.057.357.078.539.062l.336-.414zM431.338 493.929l.869-.869.119-.356.039-.237-.039-.237-.455-.454a1.139 1.139 0 00-.632-.316l-.138-.139-.237-.039-.257.02-1.007 1.007-.06.375.119.119a1.454 1.454 0 001.679 1.126zM413.638 535.095l.079-.316-.02-.335-.79-.79-.316-.119-.118-.119-.218.02a1.342 1.342 0 00-1.284 1.679l.731.731c.225.093.468.133.711.119l.316-.04.652-.652.257-.178zM424.839 486.008l.375.375.237.04.454-.02.988-.988.099-.454v-.237a2.283 2.283 0 00-.474-.79l-.257.257.217-.218-.454-.257-.277-.039-.335.02-.79.79s-.297 1.047.197 1.699l.02-.178zM456.642 531.164l.434.435a1.261 1.261 0 001.897-1.067l-.04-.237-.138-.138a1.315 1.315 0 00-1.086-1.086l-.218.019-.375.139-.415.414a1.283 1.283 0 00-.355.712l.019.296.277.513zM430.844 510.226l.691.691.336.099.099.099a1.454 1.454 0 001.324-1.561l-.04-.237-.751-.751-.355-.118-.217-.02-.771.771a1.676 1.676 0 00-.197 1.303l-.119-.276zM435.467 528.004l-.475-.474-.355-.119h-.198l-.256.02-.119.119-.355.158-.455.454-.099.375-.118.119.889.889c.265.166.577.242.889.217l1.047-1.047-.02-.257-.119-.118-.256-.336zM366.605 532.271l-.217.02-.356.118-.474.474-.158.356-.118.118a1.387 1.387 0 00.323 1.049 1.392 1.392 0 00.981.492l.217-.02.355-.118.85-.85a1.676 1.676 0 00-.376-1.126 1.413 1.413 0 00-1.027-.513zM332.156 531.52a1.391 1.391 0 001.125.415l.257-.059.672-.672a1.594 1.594 0 00-.237-1.501 1.256 1.256 0 00-1.126-.494c-.315.028-.61.168-.83.395l-.296.296-.158.356-.119.118a1.432 1.432 0 00.712 1.146zM361.726 515.302l.296-.059.415-.415.237-.434.079-.277a1.428 1.428 0 00-1.007-1.323l-.376-.06-.276.04-.297.099-.79.79a1.369 1.369 0 001.107 1.462l.118-.119.099.099.395.197zM376.995 528.517l-.375-.138a1.343 1.343 0 00-1.067.711c-.106.229-.154.48-.138.731a1.367 1.367 0 001.363 1.363l.375-.178.198-.197a.988.988 0 00.395-.395l.138-.138c.046-.188.053-.383.02-.573l-1.047-1.047.138-.139zM374.546 507.263l.178.177c.1.172.243.315.415.415l.256.059.494-.019.593-.593.217-.454.059-.257-.019-.336-.85-.849-.355-.119-.218-.019a1.419 1.419 0 00-1.264 1.027v.276l.119.474.375.218zM375.672 525.14l.356.355c.154-.034.306-.08.454-.138l.375.099.217.019a1.36 1.36 0 001.028-.592 1.322 1.322 0 00.138-1.126l-.751-.751-.316-.079-.217.218-.514-.514a1.225 1.225 0 00-1.121.732 1.22 1.22 0 00.213 1.322c.011.16.058.316.138.455zM392.996 545.229a1.345 1.345 0 00-1.225-.672 1.425 1.425 0 00-.494.139l-.651.651a1.338 1.338 0 00.276 1.186 1.5 1.5 0 001.225.513 1.286 1.286 0 001.205-1.205l-.04-.276-.296-.336zM374.269 525.357l-.335.336a1.307 1.307 0 00-.125 1.283 1.305 1.305 0 001.033.771h.237l.435-.118.533-.534.139-.375.118-.119-.02-.414-.592-.593-.455-.217-.296-.06-.138.139-.534-.099zM346.773 515.836c.11.004.22-.024.315-.079l.317-.316a1.405 1.405 0 00.869-1.067 1.284 1.284 0 00-.751-1.422l-.158-.158-.474.039-1.146 1.146-.098.375-.02.217a1.284 1.284 0 001.146 1.265zM361.549 507.579l.573.573a1.339 1.339 0 001.126.454c.122-.007.242-.034.355-.079l.968-.968-.079-.513-.612-.613-.395-.158-.139-.138-.256.019a1.24 1.24 0 00-.929.494 1.167 1.167 0 00-.612.929zM366.487 547.204l-.474-.474-.376-.177-.118-.119a.334.334 0 00-.198 0 1.338 1.338 0 00-.968.533 1.405 1.405 0 00-.276 1.225l.691.691.336.139.099.099a1.401 1.401 0 001.343-1.344l-.119-.118.06-.455zM365.163 520.596l-.197.198a1.527 1.527 0 00.434 1.185c.3.261.69.396 1.087.375l.75-.75a1.331 1.331 0 00.158-1.067l-.809-.81a1.38 1.38 0 00-.731-.138l-.297.059-.711.711.316.237zM353.805 522.67a1.421 1.421 0 00-.138 1.245l.77.77.296.06.099.098a1.454 1.454 0 001.264-1.027v-.276l-.513-.514a1.424 1.424 0 00-1.008-.336 1.258 1.258 0 00-.77-.02zM388.71 536.873l-.376-.098a1.307 1.307 0 00-1.007.533c-.18.254-.276.558-.277.869a2.111 2.111 0 00-.098.375l.118.119.119.355.533.534c.159.128.351.21.553.237h.237l.395-.119.533-.533c.122-.178.203-.38.238-.593l.138-.138a1.54 1.54 0 00-.375-1.086l-.475-.474-.256.019zM388.097 524.172l-.553-.553a1.417 1.417 0 00-1.762.085 1.42 1.42 0 00-.371.507 1.45 1.45 0 00.237 1.502l.414.414h.277a1.449 1.449 0 001.047-.573 1.376 1.376 0 00.237-1.224l.474-.158zM385.766 513.544a1.367 1.367 0 00-.138 1.284l.553.553c.258.187.572.277.889.257.22-.015.432-.09.612-.217l.771-.77a1.523 1.523 0 00-.067-.622 1.523 1.523 0 00-.308-.544 1.278 1.278 0 00-1.107-.395 1.28 1.28 0 00-1.205.454zM374.684 497.248l.652.651.336.099.119.119.316-.04.829-.829.119-.356.019-.217-.019-.257-.929-.929-.414-.059-.297.02a1.397 1.397 0 00-1.086 1.64l.355.158zM364.097 539.797a1.36 1.36 0 00-.401-.982 1.36 1.36 0 00-.982-.401l-.731.731a1.426 1.426 0 00.316 1.936l.356.356c.11.025.223.032.335.019.185-.031.361-.099.517-.201a1.356 1.356 0 00.603-.904 1.379 1.379 0 00-.013-.554zM390.369 515.144a1.513 1.513 0 00-.237 1.62 1.34 1.34 0 001.304.79l.375-.375a1.367 1.367 0 00.376-1.087 1.634 1.634 0 00-.652-1.086l-.139-.138-.335-.099-.099-.099-.217.02-.356.355-.02.099zM383.455 543.076l-.612-.612-.297-.099-.197-.198a1.42 1.42 0 00-1.081.553 1.424 1.424 0 00-.262 1.186l.829.829c.224.121.479.169.731.139l.296-.06.633-.632c.149-.329.17-.703.059-1.047l-.099-.059zM387.979 518.088l-.376.098-.533.534c-.135.17-.217.376-.237.592l.02.218.099.375.513.513c.178.138.389.227.612.257l.119-.118.375-.099.514-.513a1.52 1.52 0 00.257-.652l-.119-.119-.099-.375-.493-.494a1.205 1.205 0 00-.613-.257l-.039.04zM384.561 519.214l-.514-.514h-.276a1.449 1.449 0 00-1.047.573 1.496 1.496 0 00-.217 1.244l.77.771c.204.065.418.092.632.079a1.402 1.402 0 001.205-.968l.04-.237-.02-.415-.336-.336-.237-.197zM389.006 520.853l-.395.119-.731.731a1.499 1.499 0 00-.237 1.619 1.426 1.426 0 001.284.771l1.106-1.106c.141-.176.23-.389.257-.613l-.119-.118-.099-.376-.533-.533a1.118 1.118 0 00-.593-.237l.06-.257zM390.31 529.94l-.395.118-.534.534a1.286 1.286 0 00-.256.612l.118.118.119.356.533.533c.162.124.352.206.553.237l.217-.019.396-.119.513-.513c.142-.176.23-.389.257-.613l-.02-.217-.099-.375-.493-.494a1.721 1.721 0 00-.573-.218l-.336.06zM381.519 510.601l.672.672.335.098.099.099.395-.039.336-.336a1.516 1.516 0 001.109-.524 1.519 1.519 0 00.353-1.175l-.099-.099-.118-.316-.731-.731a1.734 1.734 0 00-.455-.019c-.4.021-.775.199-1.047.493-.184.029-.36.095-.517.195a1.361 1.361 0 00-.613.892c-.036.183-.035.371.004.553l.277.237zM377.431 513.07l-.356-.355a1.28 1.28 0 00-.968.652 1.37 1.37 0 00-.02 1.521 1.233 1.233 0 001.225.632c.353-.035.679-.205.909-.474a1.404 1.404 0 00.257-1.087l-.139-.138a1.14 1.14 0 00-.908-.751zM371.879 537.071l-.316.039-.177.06-.356.158-.336.335-.099.178-.098.375-.02.178.02.178.039.198.178.335.316.316.178.099.355.119.218.02.316-.04.197-.04.336-.177.158-.119.158-.197.099-.178.118-.356.04-.197-.02-.178-.039-.198-.178-.335-.119-.158-.217-.178-.178-.099-.375-.099-.198-.039zM357.736 502.897l.415.415a.892.892 0 00.553.237h.237l.395-.119.573-.572a1.354 1.354 0 00.178-1.087l-.771-.77-.316-.079-.099-.099h-.237l-.849.849-.099.376-.118.118.098.099.04.632zM370.853 495.944l.829-.83.119-.355a1.384 1.384 0 00-.093-.57 1.379 1.379 0 00-.315-.484 1.381 1.381 0 00-1.054-.408l-.119.119-.316.118-.77.77a1.419 1.419 0 00.277 1.304 1.365 1.365 0 001.442.336zM350.684 481.978a1.293 1.293 0 001.008-.573 1.425 1.425 0 00.178-1.205l-.791-.79a1.557 1.557 0 00-.671-.118 1.366 1.366 0 00-.869.474l-.277.276a1.395 1.395 0 00.375 1.126 1.342 1.342 0 001.047.81zM361.469 532.745l.889-.889c.033-.143.046-.289.04-.435l-.119-.118-.138-.376-.593-.592-.375-.139-.237-.039-.81.81a1.342 1.342 0 00.198 1.343 1.418 1.418 0 001.145.435zM351.514 503.075l.296-.059.751-.751.079-.316-.731-.731a1.21 1.21 0 00-.613-.257l-.256.02-.455.178-.553.553-.099.375-.019.218a1.312 1.312 0 001.6.77zM381.559 522.769l-1.047 1.047-.099.376-.119.118 1.008 1.008.336.098.118.119a1.401 1.401 0 00.988-.553l.849-.85.04-.276.197-.198.04-.276-.02-.455-.573-.572-.434-.238-.178-.177a1.418 1.418 0 00-1.126.691l.02.138zM371.366 511.312l.355.119a1.323 1.323 0 001.008-.534 1.404 1.404 0 00.276-1.106l-.79-.79-.336-.099-.118-.118a1.412 1.412 0 00-1.008.533 1.509 1.509 0 00-.276 1.106l.889.889zM358.723 561.249l-.158.158-.059.257a1.338 1.338 0 00.435.988 1.392 1.392 0 001.047.336l.276-.04.79-.79.059-.257.099-.099-.019-.414-.633-.633-.454-.217-.277-.039a1.232 1.232 0 00-1.106.75zM435.704 495.648l-.494.494a1.305 1.305 0 00.039 1.303c.119.217.297.396.514.514l.138.138.375.138.237.04.237-.04a1.506 1.506 0 001.205-.612l.435-.435a1.395 1.395 0 00-.257-.849l-1.145-1.146a1 1 0 00-.297.06l-.118.118-.869.277zM431.555 471.311l.632.632.297.059.099.099a1.42 1.42 0 001.165-1.165l-.02-.297-.513-.513a1.306 1.306 0 00-1.008-.336 1.233 1.233 0 00-.968.573 1.37 1.37 0 00-.138 1.244l.454-.296zM408.681 516.586l-.257-.256-.514.039-.276.079-.415.415-.217.454-.04.277a4.353 4.353 0 01-.276 1.106l-.139.138.06.455.533.533.415.217.178.178a1.286 1.286 0 001.027-.632l1.185-1.185.02-.376.118-.118a1.305 1.305 0 00-.411-.969 1.317 1.317 0 00-.991-.355zM409.55 521.604l-.494.494a1.48 1.48 0 00.553 1.619l.198.198.592.039.178-.177h.237l.415-.139.533-.533.158-.395.04-.237-.04-.316-1.126-1.126-.572-.02-.277.079-.454.455.059.059zM409.036 510.028l.632.632.435.119.217.02a1.314 1.314 0 001.106-1.699l-.829-.83-.356-.039a1.314 1.314 0 00-1.205 1.205l.138.138-.138.454zM405.955 489.721l.375-.177.415-.415a1.118 1.118 0 00.158-1.027 1.203 1.203 0 00-.731-.731l-.158-.158-.376-.099a1.235 1.235 0 00-.869.434c-.214.26-.32.593-.296.929a1.42 1.42 0 001.482 1.244zM362.951 483.697l-.395-.04a1.338 1.338 0 00-.691.692 1.385 1.385 0 00-.099 1.086 1.301 1.301 0 00.61.815 1.32 1.32 0 001.01.133l.276-.276.336-.613.059-.414-.77-.771-.336-.612zM374.25 535.214l-.277.277-.335.612-.04.395c.037.156.104.304.198.435l.296.296.612.336.395.039c.318-.14.571-.394.712-.711.15-.314.185-.67.098-1.007a1.422 1.422 0 00-1.659-.672zM352.324 478.442l.533.534a1.287 1.287 0 001.244.296.96.96 0 00.376-.178l.493-.494v-.316l-.098-.099-.099-.296-.79-.79-.356-.079-.098-.099-.119.119a1.42 1.42 0 00-.817.583 1.42 1.42 0 00-.23.977l-.039-.158zM369.588 498.947l.613.612.316.079.434-.04a1.48 1.48 0 00.771-.77 1.69 1.69 0 00-.02-1.284l-.672-.672-.276-.079a1.322 1.322 0 00-1.063.529 1.318 1.318 0 00-.201 1.17l-.099.099.197.356zM362.141 473.287l.178.177.435.158h.316c.127-.005.253-.025.375-.059a1.316 1.316 0 00.849-.849 1.5 1.5 0 00-.256-1.324l-.711-.711-.257.02a1.447 1.447 0 00-1.106 1.106v.277l.098.414.178.178-.099.613z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M373.737 510.996h-.238a1.294 1.294 0 00-1.066-.158 1.638 1.638 0 00-.948.948l-.119.119-.039.355.77.771a1.231 1.231 0 001.067.158c.268.195.598.286.928.257l.119-.119.375-.178.494-.493.118-.396-.118-.118a1.563 1.563 0 00-.869-1.185l-.474.039zM369.766 525.515l-.237-.237-.218.02-.276.039a1.312 1.312 0 00-.968 1.205 1.476 1.476 0 00.869 1.462l.336.336.296-.06a1.363 1.363 0 00.968-1.205 1.48 1.48 0 00-.77-1.56zM372.294 452.743l.395-.395a1.315 1.315 0 00.533-1.008l-.217-.217a1.702 1.702 0 00-.612-.612l-.099-.099-.336-.02-.77.77a1.645 1.645 0 00.079 1.62 1.34 1.34 0 001.027-.039zM352.324 508.428l-.652-.652-.316-.039-.099.098a1.426 1.426 0 00-1.047 1.324l.138.138.158.395.593.593c.194.078.404.112.612.099l.968-.968a1.314 1.314 0 00-.059-1.087l-.296.099zM348.274 494.225a1.32 1.32 0 00-.79.791 1.39 1.39 0 00.04 1.224l.75.751.336.02 1.067-1.067.039-.356-.02-.217-1.027-1.027-.375-.02-.02-.099zM373.44 503.312l-.79.79a1.364 1.364 0 00-.217.77l.849.85.316.118c.25.055.509.041.751-.039l.296-.099.672-.671.059-.297a1.478 1.478 0 00-.593-1.027 1.339 1.339 0 00-1.343-.395zM356.135 436.979l.455.455c.277.075.569.081.849.02.164-.067.323-.147.474-.238l.356-.355a1.287 1.287 0 00-.178-1.561l-.494-.493-.336-.099-.118-.119a1.473 1.473 0 00-1.066.558 1.48 1.48 0 00-.317.825l.04.237.177.533.158.237zM348.669 502.403l-.019-.256-.079-.514-.554-.553-.414-.138-.257-.02-.138.138a1.37 1.37 0 00-1.028 1.541l.712.711a1.172 1.172 0 001.046.138 1.359 1.359 0 00.731-1.047zM339.919 472.437a1.196 1.196 0 00-.04-.829l-.849-.85h-.395l-.119.119a1.476 1.476 0 00-1.086 1.086v.237l.158.435.592.592c.23.127.488.195.751.198l.257-.059.81-.81-.079-.119zM350.308 504.398a1.1 1.1 0 00-.355.158l-.415.415a1.366 1.366 0 00-.04 1.344 1.515 1.515 0 001.265.79l.77-.771a1.507 1.507 0 00-.119-1.659 1.316 1.316 0 00-1.106-.277zM330.121 472.674l.454.455.514.197.336.02a1.323 1.323 0 00.79-.79 1.374 1.374 0 00.039-1.146l-.81-.81c-.127 0-.254.02-.375.059a1.289 1.289 0 00-.909 1.423l.158.158-.197.434zM346.792 482.472l-.02.375.79.79a1.428 1.428 0 001.63.021 1.43 1.43 0 00.583-1.522l-.948-.948a1.261 1.261 0 00-.771.02l-1.264 1.264zM356.61 497.821c.059-.172.079-.354.059-.534.096-.219.156-.453.178-.691a1.26 1.26 0 00.514-.514l.652.652.375.059c.159.012.319-.001.474-.039a1.338 1.338 0 00.909-1.304l-.119-.119-.158-.355-.731-.731-.415-.059-.217.02-.533.533-.356-.356a1.366 1.366 0 00-1.343-.355l-.099.099-.335-.02-.554.553c-.096.074-.182.16-.256.257l-.475.474-.158.355-.118.119a1.424 1.424 0 00-.158.672l-.059.335a1.2 1.2 0 00-.257.889l1.146 1.146.414-.02.158-.158c.158-.117.298-.257.415-.415a1.2 1.2 0 001.047-.493zM320.125 524.547l-.711.711a1.12 1.12 0 00.059 1.008 1.286 1.286 0 00.849.612l.139.138.395.04a1.234 1.234 0 00.927-1.081 1.246 1.246 0 00-.058-.5 1.506 1.506 0 00-.494-.73 1.413 1.413 0 00-1.106-.198zM344.778 506.097l.256-.257a1.307 1.307 0 00.435-1.619l-.573-.573a1.533 1.533 0 00-1.067-.237l-.691.691a1.679 1.679 0 00.138 1.719 1.345 1.345 0 001.462.474l.04-.198zM342.822 496.398l-.257-.257-.414-.059a1.44 1.44 0 00-.843.693 1.423 1.423 0 00-.106 1.085 1.49 1.49 0 00.81.81 1.344 1.344 0 001.067.118l.435-.434a1.255 1.255 0 00.335-1.087 1.257 1.257 0 00-.691-1.007l-.336.138zM341.736 502.186l-.553-.553h-.514a1.822 1.822 0 00-.612.375l-.474.474a1.343 1.343 0 00.395 1.304 1.284 1.284 0 002.014-.276 1.39 1.39 0 00.257-.257 1.37 1.37 0 00-.098-1.205l-.415.138zM339.898 508.487l-.039-.553-.632-.632-.514-.197h-.316l-.118.118a1.397 1.397 0 00-1.047 1.64c-.026.317.066.633.256.889.119.154.271.279.446.365.175.086.367.13.562.128l.691-.691a2.53 2.53 0 00.139-.731l.572-.336zM356.531 530.177a1.313 1.313 0 00-.592 2.173l.494.494h.276l.948-.949a1.283 1.283 0 00-.395-1.224 1.362 1.362 0 00-.731-.494zM314.179 477.04l.375-.178.514-.514a1.25 1.25 0 00.138-.652l-.928-.928a1.233 1.233 0 00-.632-.119l-.929.929a1.399 1.399 0 00-.118.632l.928.928c.186.085.389.125.593.119l.059-.217zM330.002 478.324l.316.316c.184.092.388.133.593.118l.276-.079.869-.869.04-.395-.118-.119a1.342 1.342 0 00-1.66-.908c-.073.058-.14.124-.197.197l-.731.731-.02.336.217.217.415.455zM324.767 494.462l-.632.633-.297.612-.019.375a1.371 1.371 0 001.145 1.004c.179.023.36.011.534-.036a1.343 1.343 0 00.928-1.323 1.307 1.307 0 00-.517-1.042 1.314 1.314 0 00-1.142-.223zM333.755 492.744l.435.434.355.079.099.099a.609.609 0 01.158-.158 1.424 1.424 0 00.889-.889 1.463 1.463 0 00-.158-1.264l-.948-.948-.356-.04-.217.02-.553.553.079-.079-.316.316-.04.356.02.217.434.435.119.869zM362.694 477.869l.139.138a1.365 1.365 0 001.422.396 1.397 1.397 0 001.007-1.245 1.477 1.477 0 00-.77-1.442l-.138-.138h-.395l-.218.02-.415.414a1.372 1.372 0 00-.414 1.521l.098.099-.316.237zM302.999 463.41l-.83-.83a1.487 1.487 0 00-1.245-.217 1.571 1.571 0 01-.158.158l-.79.79a1.339 1.339 0 00.138 1.047l.712.711.098-.099c.206.015.413-.005.613-.059a1.37 1.37 0 001.462-1.501zM405.855 456.2a1.315 1.315 0 00.929-1.126l-.079-.316-.257-.257a1.31 1.31 0 00-.652-.652 1.314 1.314 0 00-.988-.079l-.375.375a1.315 1.315 0 00-.237 1.62 1.452 1.452 0 001.659.435zM362.912 551.53a1.4 1.4 0 00-1.699-.829 1.31 1.31 0 00-.455.217l-.434.435a1.306 1.306 0 00.099 1.323l.513.514.454.138.099-.099.02.336c.241.063.494.056.731-.02l.869-.869-.118-.118-.099-.336.02-.692zM315.108 454.54l.434.435.297.059a1.396 1.396 0 00.948-1.659 1.283 1.283 0 00-1.442-1.007l-.514.513a1.345 1.345 0 00-.079 1.778l.356-.119zM340.392 483.776a1.307 1.307 0 00-.908.908v.277l.118.434 1.047 1.047.375.02a1.314 1.314 0 00.988-1.343l-.81-.81a1.42 1.42 0 00-.81-.533zM361.45 478.245l-.415-.099h-.237l-.118.119a1.582 1.582 0 00-.988.987.624.624 0 00-.198.198 1.565 1.565 0 00-.375 1.047 1.96 1.96 0 001.126 1.718l.119.119.375.019.988-.987c.299-.251.495-.602.553-.988l.118-.118.04-.277a1.233 1.233 0 00-.593-.988l-.197-.197-.198-.553zM399.91 512.794l-.612-.613-.336-.019-.257.256a1.352 1.352 0 00-.77.771 1.288 1.288 0 00.099 1.047l.177.177a1.337 1.337 0 001.265.593l.81-.81.118-.316a1.429 1.429 0 00-.158-1.067l-.336-.019zM381.125 504.398l.138-.138.059-.296-.099-.099a1.538 1.538 0 00-.652-1.047 1.386 1.386 0 00-1.125-.138l-.771.77a1.431 1.431 0 00.04 1.106l.79.791c.255.053.518.046.77-.02.069-.063.135-.129.198-.198l.454-.454.198-.277zM359.928 499.085l-.869.869-.02.257.04.276.158.474.435.435c.098.149.226.276.375.375h.316l.533-.178.652-.652c.156-.334.184-.713.079-1.066l-.889-.889a1.45 1.45 0 00-.81.099zM362.813 508.961l.099.099a1.433 1.433 0 00.79-.79 1.516 1.516 0 000-1.264l-1.205-1.205-.435-.079h-.237l-.81.81-.039.237v.395a1.284 1.284 0 00.81.81c.144.28.372.508.652.652l.118.118.257.217zM337.173 485.376v-.316l-.099-.099a1.482 1.482 0 00-1.817-.988l-.889.889-.02.454.02.257.889.889c.276.114.583.128.869.04.191-.054.372-.141.533-.257l.573-.573-.059-.296zM358.368 485.712l-.237.039a1.373 1.373 0 00-.987 1.343v.277l.869.869c.34.106.703.113 1.047.02l.889-.889.059-.297a1.71 1.71 0 00-.652-1.165l-.118-.118-.87-.079zM363.05 495.331l-.316.08-.533.533a1.115 1.115 0 00-.376.375c-.16.1-.295.235-.395.395l-.533.534-.04.355.099.099a1.738 1.738 0 00.79 1.225l.119.118.316.04c.307.226.689.325 1.066.276l.791-.79c.211-.262.347-.576.395-.909l-.04-.276.02-.257-.02-.336a1.508 1.508 0 00-1.343-1.462zM360.521 490.433l-.612.612-.04.316a1.431 1.431 0 001.041 1.207c.182.049.372.062.559.038l.652-.652a1.365 1.365 0 00-.099-1.561 1.342 1.342 0 00-1.462-.474c.014.172.001.346-.039.514zM353.489 484.862l.435.435.336.059.098.099a.699.699 0 00.139-.138 1.394 1.394 0 001.027-1.541l-.81-.81-.316-.079a1.308 1.308 0 00-.731.02.837.837 0 00-.197.197l-.751.751-.02.336.376.375.414.296zM385.351 527.352c-.359.162-.647.45-.81.81a1.652 1.652 0 00.06 1.363l.493.494a1.46 1.46 0 001.107.118c.193-.074.368-.188.514-.335.147-.146.261-.321.335-.514l.178-.178.02-.257a1.618 1.618 0 00-.244-.608 1.572 1.572 0 00-.468-.458 1.411 1.411 0 00-1.185-.435zM356.175 461.158l-.118-.119-.356-.079-.099.099a.466.466 0 00-.138.138 1.452 1.452 0 00-.968.771 1.43 1.43 0 00.02 1.165l.336.336a1.201 1.201 0 001.402.494c.164-.044.318-.118.455-.217l.632-.632-.099-.099a1.64 1.64 0 00-.04-.672l-.533-.533-.494-.652zM350.664 457.543l.434.434.336.06.099.098.118-.118a1.727 1.727 0 001.324-1.324l-.04-.316-.691-.691a1.45 1.45 0 00-1.067-.118 1.845 1.845 0 00-.77.493l-.553.553-.04.396.02.217.277.276.553.04zM364.432 486.64a.99.99 0 00-.257.059 1.408 1.408 0 00-.849.85 1.573 1.573 0 00.04 1.264l.711.711.316.04a.99.99 0 00.257-.06 1.393 1.393 0 00.849-.849 1.459 1.459 0 00-.059-1.245l-.613-.612-.296-.059-.099-.099zM363.978 455.014l-.138.139a1.087 1.087 0 00-.277 1.303l.889.889.395.04a1.335 1.335 0 00.889-.889 1.394 1.394 0 00-.02-.85 1.339 1.339 0 00-1.639-.928l-.099.296zM373.795 430.066l.455.454c.153 0 .306-.02.454-.059a1.393 1.393 0 00.988-1.185 1.543 1.543 0 00-.81-1.561l-.277-.276h-.237a.657.657 0 00-.296.059 1.365 1.365 0 00-.929.928 1.568 1.568 0 00.356 1.501l.296.139zM365.677 500.329l.948.948c.109-.009.216-.036.316-.079a1.39 1.39 0 00.909-.908 1.627 1.627 0 00-.296-1.482l-.178-.178-.435-.197-.276-.04-.297.02-.711.711c-.151.389-.144.822.02 1.205zM408.345 464.733l-.474.474c-.127.14-.216.31-.257.494l-.02.257 1.146 1.146.395.039c.311-.133.558-.38.691-.691.161-.311.196-.671.099-1.008a1.344 1.344 0 00-1.58-.711zM379.939 454.817a1.511 1.511 0 00-1.067 1.58l.04.237.178.415.553.553.434.079.257-.019 1.106-1.107-.019-.335-.277-.04.257-.059-.079-.474-.593-.593-.474-.158-.257-.02-.059-.059zM356.966 474.748h.197l1.008-1.007.02-.375-1.067-1.067-.375-.059-.218.019a1.509 1.509 0 00-.81.81 1.622 1.622 0 00-.039 1.146l.83.83.454-.297zM360.6 444.762a1.259 1.259 0 00-.771 1.324l.534.533a1.231 1.231 0 001.343.395.954.954 0 00.435-.237l.691-.691-.02-.296a1.34 1.34 0 00-1.382-1.067l-.139.138-.691-.099zM319.315 512.359l-.414-.415-.376-.019-.098.098-.316.079-.81.81-.06.375.099.099c.059.282.199.54.402.744.204.203.462.343.744.402h.276l.514-.158.632-.632.02-.257a1.34 1.34 0 00-.04-.83l-.395-.395-.178.099zM388.71 515.342l-.297.02-.513.237-.613.612-.019.415a1.331 1.331 0 00.612 1.007 1.194 1.194 0 001.027.158l.573-.572a1.395 1.395 0 00.079-1.304 1.423 1.423 0 00-.849-.573zM371.149 515.164a1.482 1.482 0 00-1.62-.711l-.751.751-.098.335a1.76 1.76 0 00.573 1.284c.191.188.443.3.711.317l.217.217a1.362 1.362 0 001.383.434c.087-.07.166-.149.237-.237l.711-.711v-.316l-1.363-1.363zM319.414 492.23l.316-.079.81-.81.059-.335a1.247 1.247 0 00-.632-.949 1.34 1.34 0 00-1.165-.138.443.443 0 00-.138.138l-.79.79-.04.356 1.106 1.106.395.04.079-.119zM330.456 437.651l.87-.869c.094-.218.141-.454.138-.692l-1.146-1.145a1.935 1.935 0 00-.474.039 1.428 1.428 0 00-.988 1.699 1.366 1.366 0 001.6.968zM350.664 450.353a1.229 1.229 0 001.067.474l.988-.988a1.325 1.325 0 00-.06-1.047l-.691-.691-.217-.02-.376.02a1.256 1.256 0 00-.948 1.461l.139.139.098.652zM374.151 442.669l.336.335c.201.104.426.152.652.139l1.027-1.028.02-.375v-.197a1.35 1.35 0 00-1.133-.989 1.36 1.36 0 00-.527.04.86.86 0 00-.197.198l-.731.731-.02.336.316.316.257.494zM343.553 492.665l.652-.652a1.4 1.4 0 00-.415-1.324 1.372 1.372 0 00-1.264-.316 1.255 1.255 0 00-.771.771l-.138.138-.039.316.099.099a1.386 1.386 0 00.632.869c.335.173.725.208 1.086.099h.158zM352.146 524.962l.296-.296.06-.415a1.855 1.855 0 00-.02-.731l-.849-.85-.336-.019-.316.079-.395.395a1.622 1.622 0 00-.415 1.679l.829.83.376.019.098.099.593-.593.079-.197zM329.844 459.992l.415.415c.208.079.428.126.651.138l.889-.889.336.02 1.225-1.225a1.281 1.281 0 00-.474-1.145 1.168 1.168 0 00-1.205-.217 1.426 1.426 0 00-.85.651l-.316-.039a1.428 1.428 0 00-1.027 1.304l.139.138.158.395.059.454zM334.525 462.027a1.514 1.514 0 001.237 1.249c.206.035.417.027.62-.024l.652-.652a1.622 1.622 0 00-.059-1.324l-.771-.77-.335-.02-.099.099-.395.119-.81.809-.059.336.019.178zM370.082 472.22a1.48 1.48 0 00-.69 2.293 1.325 1.325 0 00.888.482c.173.018.347.001.513-.049l.277-.277.335-.612.06-.415-.514-.513-.612-.336-.257-.573zM382.191 509.179l-.474-.474-.553-.198-.336-.019-1.067 1.066v.395l.139.139a1.282 1.282 0 001.56.849 1.254 1.254 0 00.948-.948v-.514l-.217-.296zM352.185 469.059l.652-.652a1.452 1.452 0 00-.138-1.639 1.4 1.4 0 00-1.482-.454 1.315 1.315 0 00-.415.177l-.553.553a1.4 1.4 0 00.415 1.245 1.455 1.455 0 001.521.77zM343.355 456.121l.435.434.356.079h.197l.138-.138a1.537 1.537 0 001.008-1.916l-.948-.948-.356-.04-.316.04-.948.948c-.037.329-.023.662.039.988l.139.138.256.415zM362.26 509.357l-.553-.553-.296-.06-.099-.099-.356.079a1.43 1.43 0 00-1.027 1.304l.099.099.079.435.553.553.415.178c.246.063.504.063.75 0 .069-.063.135-.129.198-.198l.553-.553a1.71 1.71 0 00-.059-1.324l-.257.139zM350.546 472.003l-.751-.751a1.403 1.403 0 00-.731-.178l-.237.04-.829.829-.06.415v.237l1.067 1.067.415-.02a1.285 1.285 0 001.126-1.639zM366.941 506.532a1.426 1.426 0 00-1.027-.119l-.79.79a.924.924 0 00-.06.692l-.652.652-.019.296c.092.401.332.754.671.988l.119.118.316.079.099.099.217-.02.119-.118a1.316 1.316 0 00.928-1.205l-.02-.257-.197-.198a1.23 1.23 0 00.177-1.086c.093-.225.133-.468.119-.711zM323.088 469.237l-.474-.474a1.304 1.304 0 00-1.225-.395l-1.027 1.027-.059.375.118.119c.075.325.265.612.534.81l.098.099a1.285 1.285 0 001.501.671l.613-.612a1.507 1.507 0 00-.079-1.62zM337.864 446.797l.395.395c.183.092.388.133.592.119l1.008-1.008a1.478 1.478 0 00-.06-1.086l-.77-.771-.336-.019a1.394 1.394 0 00-1.027 1.303l.138.138.158.396-.098.533zM301.162 517.436a1 1 0 00-.178.178l-.592.592-.059.296a1.443 1.443 0 00.632 1.107 1.419 1.419 0 001.204.138.57.57 0 00.159-.158l.612-.613.059-.335a1.477 1.477 0 00-.652-1.047 1.313 1.313 0 00-1.185-.158zM385.134 514.492l-.178-.177a1.231 1.231 0 00-1.165-.455l-.948.948a1.31 1.31 0 00.039 1.028l.83.829.375-.019.099-.099a1.258 1.258 0 00.928-1.403l-.118-.118.138-.534zM342.96 514.117l-.276.079-.454.257-.554.553-.039.435.118.118a1.317 1.317 0 001.66.909 2.83 2.83 0 00.197-.198l.711-.711.04-.316a1.479 1.479 0 00-1.403-1.126zM377.272 465.03a2.706 2.706 0 00-.355.355 1.253 1.253 0 00-.04 1.462 1.287 1.287 0 001.422.474 1.203 1.203 0 00.889-.889l.119-.119v-.237l-1.008-1.007-.336.059-.217.218-.474-.316zM379.603 483.677a1.076 1.076 0 00-.376.178l-.572.573a1.395 1.395 0 00.434 1.264 1.49 1.49 0 001.304.355l.672-.671a1.48 1.48 0 00-.139-1.6 1.418 1.418 0 00-1.323-.099zM351.355 516.547l-.118.118-.079.277c-.078.363-.022.742.158 1.067l.632.632.296.059.297-.059a1.37 1.37 0 00.948-1.66 1.368 1.368 0 00-1.62-.987l-.652.652.138-.099zM404.65 503.984l-.395-.395-.336-.06a.439.439 0 00-.276.04 1.37 1.37 0 00-.948 1.659 1.368 1.368 0 001.62.988l.948-.948a1.512 1.512 0 00-.04-.83l-.118-.119-.455-.335zM333.558 499.381l.514.514.414.019.218-.019a1.506 1.506 0 001.086-1.087l-.02-.257-.118-.434-.553-.553-.455-.139-.237-.039c-.042.05-.088.096-.138.138a1.427 1.427 0 00-1.047 1.561l.158.158.178.138zM313.923 502.541l-.336.613-.059.375.098.099.099.296.929.929.395.039a1.33 1.33 0 00.473-.246 1.328 1.328 0 00.494-.922 1.392 1.392 0 00-.729-1.345 1.391 1.391 0 00-1.048-.114l-.316.276zM334.051 519.194l-.869.869-.039.355v.198l.731.731c.202.124.434.192.671.198l.119.118.257-.02.829-.83.099-.296.119-.118-.04-.356-.138-.138a1.485 1.485 0 00-.742-.667 1.485 1.485 0 00-.997-.044zM342.901 475.262l-.019.336.079.316.79.79.355.079h.198l.118-.119a1.448 1.448 0 00.83-.829 1.395 1.395 0 00.02-1.166l-.83-.829a1.508 1.508 0 00-.731.019c-.1.027-.199.06-.296.099l-.652.652.138.652zM341.933 525.574l-.276-.276-.257.019-.454.099-.672.672-.04.237-.019.454.059.257c.152.274.378.5.652.652l.237-.237-.079.316.494.138h.316l.296-.099.771-.77a2.232 2.232 0 00-.1-.874 2.224 2.224 0 00-.434-.766l-.494.178zM301.636 487.805l-.218-.217-.316-.119a1.144 1.144 0 00-.77-.019 1.255 1.255 0 00-.869 1.462l.296.296a1.285 1.285 0 001.343.75l.751-.75c.14-.225.209-.486.198-.751l-.079-.276-.336-.376zM331.227 503.174l-.652-.652-.375-.02-.119.119a1.478 1.478 0 00-.968 1.797l.751.751.355.039.218-.019.829-.83a1.597 1.597 0 00-.079-1.304l.04.119zM323.01 486.581l.454.454.375.02.099.099 1.007-1.008.02-.375.099-.099-.83-.83a1.42 1.42 0 00-.908-.039l-.297.691-.375.652.296.297.06.138zM389.618 468.486l.751-.75.099-.376v-.197a1.43 1.43 0 00-1.051-1.231 1.419 1.419 0 00-.569-.033l-.85.849-.079.356-.019.217c.057.406.27.774.592 1.027a1.34 1.34 0 001.126.138zM423.516 462.244a1.312 1.312 0 00-1.186-.158l-.158.158-.75.751a1.622 1.622 0 00.513 1.422 1.377 1.377 0 001.205.257 1.31 1.31 0 00.751-.553l.296-.296.099-.336.02-.217a1.392 1.392 0 00-.79-1.028zM397.756 484.052l-.592.593-.178.494v.276a1.425 1.425 0 001.185 1.185l.099-.098.375-.02.257-.059.454-.455.218-.494.039-.276a1.422 1.422 0 00-1.304-1.304l-.118.119-.435.039zM380.137 474.215l.395.039a1.34 1.34 0 00.889-.888 1.15 1.15 0 00-.02-.731 1.345 1.345 0 00-1.107-1.034 1.348 1.348 0 00-.533.026l-.474.474a.923.923 0 00-.256.494l-.02.217c-.006.196.027.391.099.573l.533.534.494.296zM386.576 494.285a.945.945 0 00-.494-.257h-.276l-.474.118-.593.593-.118.474v.277l1.047 1.047.395.039.118-.119a1.422 1.422 0 001.027-1.303l-.158-.158-.197-.435-.277-.276zM382.131 477.237l-.118-.118h-.198a1.183 1.183 0 00-.414.217l-.356-.04-.217.02a1.404 1.404 0 00-.85.85 1.228 1.228 0 00.079 1.106l.731.731.336.019.237-.237.237.237.237-.039a1.264 1.264 0 00.772-.594 1.248 1.248 0 00.117-.967 1.173 1.173 0 00-.435-.632l-.158-.553zM361.094 461.099a1.258 1.258 0 001.324.375c.163-.044.318-.118.454-.218l.553-.553a1.34 1.34 0 00-.513-1.106 1.37 1.37 0 00-1.304-.237 1.232 1.232 0 00-.909 1.423l.178.177.217.139zM383.475 476.763l.237-.237a1.235 1.235 0 00-.139-1.284 1.332 1.332 0 00-1.145-.553l-.139.138-.414.218-.474.474-.06.375v.197l.119.395.612.613.474.118h.277l.454-.454h.198zM412.296 480.496c-.074.058-.14.125-.198.198l-.336.336a1.458 1.458 0 00-.632 1.225 1.312 1.312 0 001.008 1.244h.276l.455-.138 1.145-1.146.06-.375-.119-.119a1.313 1.313 0 00-1.659-1.225zM399.515 491.578l-.573-.573a1.29 1.29 0 00-1.205-.217.734.734 0 00-.335.138l-.751.751.217.217.178.494.612.613.415.059h.237a.99.99 0 00.257-.059 1.234 1.234 0 00.731-.731c.142-.203.218-.444.217-.692zM386.556 453.888l.474.475.395.118h.198a.574.574 0 00.118-.118 1.316 1.316 0 00.79-.791 1.393 1.393 0 00.02-1.244l-.75-.751-.356-.039h-.198a1.367 1.367 0 00-.948 1.659l.119.119.138.572zM393.391 479.608l.04-.277.099-.099a1.404 1.404 0 00-.672-1.027 1.307 1.307 0 00-1.126-.138l-.79.79a1.323 1.323 0 00.059 1.086l.692.692c.246.063.504.063.75 0l.217-.217.593-.593.138-.217zM404.039 475.242a1.341 1.341 0 00-.158-1.264l-.731-.731h-.316a1.426 1.426 0 00-1.047 1.284l.059.296.553.554a1.51 1.51 0 001.047.098c.214-.028.418-.109.593-.237zM366.981 468.407l.395.04a1.38 1.38 0 00.81-.81c.125-.286.153-.605.079-.909.027-.123.034-.25.02-.375l-.929-.928a1.145 1.145 0 00-.592-.119l-.948.948c-.085.207-.119.43-.099.652l.099.099-.02.217a1.482 1.482 0 00.592.988l.139.138.454.059zM370.161 480.714l.553.553c.319.137.675.158 1.008.059a1.366 1.366 0 001.007-1.007 1.457 1.457 0 00-.533-1.442l-.336-.336-.158.158a1.391 1.391 0 00-.85.849 1.37 1.37 0 00.02 1.245l-.711-.079zM374.605 490.65a1.501 1.501 0 00-.138-1.284l-.494-.494a1.396 1.396 0 00-.928-.059 1.23 1.23 0 00-.553.355l-.415.415-.079.316a1.317 1.317 0 00.593 1.067 1.477 1.477 0 001.165.178c.333-.059.634-.234.849-.494zM388.768 504.379l-.651-.652-.376-.02-1.145 1.146-.04.355 1.205 1.205h.395l.178-.177a1.428 1.428 0 00.77-1.838l-.336-.019zM390.467 460.565a1.397 1.397 0 001.699.988l.83-.83a1.373 1.373 0 00.041-1.048 1.39 1.39 0 00-.713-.769l-.336-.336a1.808 1.808 0 00-.335.059 1.369 1.369 0 00-1.186 1.936zM369.825 490.057a1.501 1.501 0 00-.079-1.62 1.347 1.347 0 00-1.422-.513l-.573.573a1.293 1.293 0 00-.118 1.106 1.475 1.475 0 00.908.909l.099.098.375.02.633-.632.177.059zM370.872 461.277l.593.592.296.02.138-.138a1.51 1.51 0 00.988-.791 1.59 1.59 0 00-.02-1.165l-.672-.672-.237-.039a1.491 1.491 0 00-.73.02.823.823 0 00-.198.197l-.672.672c-.078.35-.022.717.158 1.027l.356.277zM372.255 485.85c.093-.211.134-.442.118-.672l-.909-.909a1.09 1.09 0 00-.632-.118l-.908.909a1.42 1.42 0 00-.119.671l.889.889c.205.092.427.139.652.138l.909-.908zM374.625 484.862l.138.138.356.079.217.02c.084-.027.17-.047.257-.059a1.404 1.404 0 00.849-.85 1.5 1.5 0 00-.019-1.244l-.692-.692a1.545 1.545 0 00-.612.06 1.367 1.367 0 00-1.008 1.205l.119.118.118.395.356.356-.079.474zM369.983 484.171l.909-.909a1.477 1.477 0 00-.119-1.62 1.285 1.285 0 00-1.402-.494l-1.126 1.126a1.092 1.092 0 00-.119.632l.929.929c.192.086.403.12.612.099l.316.237zM366.803 475.499l.79-.79c.095-.197.136-.415.118-.632l-.928-.929a1.136 1.136 0 00-.593-.118l-.138.138-.375.178-.514.513c-.102.202-.15.426-.138.652l.928.929c.2.082.416.123.632.118l.218-.059zM379.228 492.783l-.652-.652-.336-.019-.098-.099-.376.138-.395.395a1.545 1.545 0 00-.962.756 1.54 1.54 0 00-.104 1.22l.612.612.336.099.098.098c.161-.007.32-.033.475-.079.366-.153.656-.444.809-.809a1.363 1.363 0 00.782-.753 1.369 1.369 0 00-.011-1.085l-.178.178zM382.843 489.445l.118.118.218-.019a1.257 1.257 0 00.829-.83 1.474 1.474 0 00-.276-1.501 1.393 1.393 0 00-1.363-.376 1.26 1.26 0 00-.711.711 1.375 1.375 0 00-.04 1.146l.119.119a1.285 1.285 0 001.106.632zM383.435 464.793l.316-.079.178-.099.277-.237.158-.158.118-.237.04-.198.059-.375-.039-.198-.04-.158-.099-.178-.217-.296-.158-.118-.237-.119-.198-.079-.375-.02h-.197l-.297.099-.197.079-.297.257-.118.158-.138.217-.04.198-.059.375.02.217.059.139.099.177.217.297.178.138.197.119.217.059.356.039.217-.019zM404.216 495.371l-.336-.336a1.42 1.42 0 00-.592-.118l-1.008 1.007a1.344 1.344 0 00.079 1.067l.771.77.335.02 1.087-1.087.039-.395v-.197l-.237-.237-.138-.494zM392.759 504.853l-.829.829-.059.376.118.118a1.402 1.402 0 00.573.974 1.388 1.388 0 001.106.231l.751-.75.098-.336.02-.217a1.396 1.396 0 00-.553-1.225 1.283 1.283 0 00-1.225 0zM415.377 514.433a1.449 1.449 0 00-.926 1.438c.01.208.063.411.155.597l.79.79c.225.026.452.013.672-.04.308-.137.554-.383.691-.691l.178-.178a1.415 1.415 0 00-.612-1.007 1.26 1.26 0 00-.948-.909zM394.497 466.926l-.711.711-.079.277.06.414.948.948.415.06h.237l.908-.909a1.407 1.407 0 00-.434-1.304 1.483 1.483 0 00-1.344-.197zM410.261 493.949a.86.86 0 00-.198.197l-.73.731-.02.336.79.79c.205.092.427.139.652.139l1.047-1.047.039-.395-.118-.119a1.313 1.313 0 00-1.462-.632zM376.838 480.754l.296-.297.099-.336.296-.296.06-.375-.02-.217-1.027-1.028-.356-.039a1.306 1.306 0 00-.79.79l-.612.612-.139.455-.019.296a.52.52 0 00-.079.276l-.02.257.138.455.573.572.474.158h.277a1.338 1.338 0 00.908-.908l-.059-.375zM389.776 490.334l-.415-.415-.355-.039-.217.019a1.57 1.57 0 00-.791.79 1.401 1.401 0 00-.019 1.166l.77.77.356.04.098.098a1.51 1.51 0 00.81-.809c.155-.359.162-.763.02-1.126l-.138-.139-.119-.355zM391.396 438.461l.138-.138.02-.257a1.507 1.507 0 00-.652-.889 1.333 1.333 0 00-1.066-.118.823.823 0 00-.198.197l-.751.751-.019.296.099.099.098.415.593.592.474.119.296-.02.099-.099a1.317 1.317 0 00.869-.948zM329.331 518.344l.454-.454a1.309 1.309 0 00-.277-1.264 1.225 1.225 0 00-.652-.336l-.138-.138-.415-.06-.217.02-.138.138a1.711 1.711 0 00-.968.968l-.158.158.099.099v.237c.057.303.209.58.434.79l.751.751.257-.02a.638.638 0 00.177-.178l.139-.138.652-.573zM338.338 541.298l-.593-.592h-.316a1.394 1.394 0 00-1.027 1.303l.711.711c.33.167.709.209 1.067.119a1.49 1.49 0 00.81-.81 1.402 1.402 0 00-.139-1.244l-.513.513zM351.632 492.408l.257.02.494-.139.612-.612.099-.494v-.276a3.64 3.64 0 01.059-1.126l-.019-.257-.139-.415-.573-.573-.434-.118h-.277a1.305 1.305 0 00-.849.849l-.178.178-.118.435-.119.118-.257.494-.059.296.079.356-.118.118a1.306 1.306 0 00.521.908 1.316 1.316 0 001.019.238zM349.676 487.608l.356-.356a1.565 1.565 0 00-.869-1.501l-.316-.039-.573.098-.316.316-.395.198-.534.533-.059.415v.237l.079.316.81.81.217.02.296.059.573-.099.751-.751-.02-.256zM352.521 498.828l-.612-.613-.455-.019-.237.039a1.323 1.323 0 00-.869 1.333c.013.184.067.363.158.524l.79.79.376-.019a1.316 1.316 0 00.968-1.403l-.158-.158.039-.474zM359.711 518.147l-.691.691a1.232 1.232 0 00.059 1.047c.195.305.499.524.849.612l.139.139.414.019c.304-.127.545-.368.672-.671.159-.297.194-.645.099-.968a1.401 1.401 0 00-.615-.71 1.402 1.402 0 00-.926-.159zM328.955 457.089l-.277-.277a1.29 1.29 0 00-.968.217 1.406 1.406 0 00-.533.929 1.32 1.32 0 00.195 1.001 1.308 1.308 0 00.852.559l.375-.098.889-.889-.099-.376-.177-.178-.257-.888zM316.786 508.329l-.395.079-.77.771a1.39 1.39 0 00-.02.494l.079.395.751.75a1.2 1.2 0 00.968-.217c.283-.201.475-.507.533-.849a1.396 1.396 0 00-1.146-1.423zM321.666 447.745l.04.356.256.257a1.403 1.403 0 00.909.908c.131.02.264.02.395 0l.929-.928.039-.316-.099-.099-.079-.356-.79-.79-.217-.019a1.338 1.338 0 00-1.6.928l.217.059zM328.283 473.721l.06.296.276.514.218.217.414.138a1.4 1.4 0 001.067-.276c.32-.306.511-.723.533-1.166l-.928-.928-.395-.118a1.342 1.342 0 00-1.501 1.106l.158-.158.098.375zM332.748 447.33l.711.711a.963.963 0 00.375.099 1.264 1.264 0 001.126-.336 1.39 1.39 0 00.356-1.303l-.929-.929a1.429 1.429 0 00-1.442.613l-.178.177-.079.435v.237l.06.296z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M327.039 486.304l-.178-.178a1.2 1.2 0 00-.888-.612 1.723 1.723 0 00-1.284.375l-.494.494-.02.257-.099.099.099.336.119.118c.207.322.534.55.908.632.133.311.381.559.692.692l.118.118.415-.02.889-.889a1.598 1.598 0 00-.277-1.461v.039zM316.964 497.682l-.434-.434-.257-.06a1.374 1.374 0 00-1.402.652 1.501 1.501 0 00.138 1.719l.415.414.276.08c.272.041.55.001.8-.114.249-.116.459-.304.603-.538a1.51 1.51 0 00-.139-1.719zM350.782 433.207l.218.019.375-.059a1.226 1.226 0 00.909-.672l.118-.118a1.477 1.477 0 00-.395-.79l-.474-.474-.237-.04-.099.099-.355.079-.139.138a1.57 1.57 0 00-.612 1.482c.191.18.432.297.691.336zM308.688 474.748l-.987-.987a1.455 1.455 0 00-1.502.75l-.138.138v.435l.02.217.434.435c.094.176.239.321.415.415l.138.138.415-.02.534-.533a1.364 1.364 0 00.414-1.008l.257.02zM311.137 460.249a1.318 1.318 0 00-1.086.297 1.42 1.42 0 00-.474 1.106l.118.118.119.316.948.948.375-.098.988-.988-.02-.217-.118-.395-.711-.711-.139-.376zM329.943 479.292l-.198-.001-.375.02-.316.316c-.244.13-.443.33-.573.573l-.119.119-.02.375.712.711c.206.147.442.248.691.296a.739.739 0 00.296.02l.79-.79a1.677 1.677 0 00-.039-1.027 1.395 1.395 0 00-.849-.612zM343.02 412.011l.454.454c.223.186.487.315.771.376h.513l.336-.336a1.367 1.367 0 00.514-1.501l-.83-.83-.316-.079a1.453 1.453 0 00-1.501.751l-.158.158-.04.553.257.454zM308.016 467.756l.158-.159.059-.454-.019-.257-.988-.987a1.366 1.366 0 00-1.58.948l-.02.217.237.237c.221.302.544.512.909.593a1.48 1.48 0 001.244-.138zM313.093 436.96c.171-.223.274-.491.296-.771l-.019-.217-.119-.356-.829-.829a1.481 1.481 0 00-1.423.593l-.158.158-.059.454.02.257.138.375.138.138c.124.237.317.429.553.553l.139.139.474-.04.553-.553.296.099zM308.589 470.264a1.08 1.08 0 00-.376-.02l-.375.376a1.422 1.422 0 00-.612 1.205 1.566 1.566 0 00.79 1.264l.099.099.276.039.099.099.178-.178a1.54 1.54 0 00.612-1.56 1.34 1.34 0 00-.691-1.324zM304.125 432.93l.059.336.83.829a1.35 1.35 0 001.106-.276 1.442 1.442 0 00.533-1.008l-.948-.948a.954.954 0 00-.356-.118 1.31 1.31 0 00-1.461.908v.237l.237.04zM315.325 448.832l-.81.81-.02.217.06.296.118.119a1.46 1.46 0 001.027.711c.361.059.731-.026 1.031-.236.299-.211.503-.53.57-.89v-.237l-.079-.435-.573-.573a1.352 1.352 0 00-.692-.296h-.237l-.375.099-.02.415zM317.221 467.044a.969.969 0 00.257-.454 2.97 2.97 0 00.514-.513c.245-.028.478-.124.671-.277l.87.869c.137.072.283.125.434.158a1.312 1.312 0 001.383-.79l.118-.118.02-.376-1.145-1.146-.356.079-.138.139-.277-.277a1.345 1.345 0 00-.968-.968l-.356-.355-.118.118a1.644 1.644 0 00-.257.257l-.454.454a1.288 1.288 0 00-.494.494l-.434.435a1.175 1.175 0 00-.652.652l-.02.296.02.257.098.415.534.533.415.178.237.039a2.18 2.18 0 00.573-.138 1.267 1.267 0 00-.475.039zM272.677 475.4l-.395.079-.434.435a1.093 1.093 0 00-.376.968c.009.176.058.348.143.502.086.154.206.286.351.387l.435.434a1.25 1.25 0 00.938-.224 1.25 1.25 0 00.504-.823 1.292 1.292 0 00-.138-.889 1.313 1.313 0 00-1.028-.869zM302.881 469.415l.355-.079a1.229 1.229 0 001.166-1.166l-.395-.395a1.456 1.456 0 00-.771-.77l-.118-.118-.139.138a1.541 1.541 0 00-.612 1.6 1.316 1.316 0 001.067 1.066l-.553-.276zM305.33 459.814l-.573-.572a1.423 1.423 0 00-1.62 1.185c-.058.394.033.796.257 1.126a1.37 1.37 0 00.909.553l.414-.099.139-.138a1.344 1.344 0 00.592-2.015l-.118-.04zM301.853 464.555l-.889-.888a1.332 1.332 0 00-.691.059l-.415.415a1.39 1.39 0 00-.198 1.382 1.283 1.283 0 001.502.553c.101-.082.194-.175.276-.276a1.347 1.347 0 00.415-1.126v-.119zM297.448 469.435l.217-.494.059-.297-.059-.335-.928-.929a1.423 1.423 0 00-1.66 1.027 1.315 1.315 0 00-.118.909 1.31 1.31 0 00.672.909l.098.099.297.059.454-.455c.182-.17.324-.379.415-.612l.553.119zM303.077 496.201a1.34 1.34 0 00-1.224.434 1.285 1.285 0 00-.237 1.265l.928.928.257-.02.415-.415a1.315 1.315 0 00.138-1.284 1.339 1.339 0 00-.277-.908zM287.868 429.987l.414-.02.475-.474c.216-.102.391-.277.493-.494l.119-.118-.04-.396-.474-.474a1.307 1.307 0 00-.454-.454l-.138-.138-.395.039-.475.474a1.04 1.04 0 00-.454.455l-.118.118v.395l.494.494c.112.174.26.322.434.435l.119.158zM301.596 437.987l.316.316c.1.184.251.334.435.435l.276.039.494-.059.948-.948a1.283 1.283 0 00-1.086-1.521l-.296-.02-.119.118-.296.099-.751.751v.197l.079.593zM289.883 450.254l-.415-.415-.218-.02-.395.119-.73.731a1.33 1.33 0 00.203 1.012 1.352 1.352 0 00.863.568 1.401 1.401 0 001.442-.77 1.34 1.34 0 00-.75-1.225zM298.752 452.683l.79.791.257.059a1.337 1.337 0 001.205-.336 1.362 1.362 0 00.395-1.185l-.02-.217-.079-.356-.988-.988-.217.02-.356.079-.849.849-.079.475-.059.809zM331.286 451.755l.119.119a1.452 1.452 0 002.409.207 1.458 1.458 0 00.31-1.069 1.458 1.458 0 00-.171-.541l-.573-.573-.612.376-.217.019a1.392 1.392 0 00-1.126 1.126l-.139.336zM283.581 412.939l.02-.256-.079-.474-.316-.317a1.506 1.506 0 00-1.027-.75h-.237l-.277.079-.415.415c-.222.288-.34.643-.335 1.007l.81.81c.192.103.398.176.612.217a1.395 1.395 0 001.244-.731zM379.583 450.866a1.287 1.287 0 001.324-.612l.079-.316-.02-.376a1.254 1.254 0 00-.257-.888 1.223 1.223 0 00-.849-.494l-.099-.099-.119.118a1.285 1.285 0 00-.908 1.344 1.446 1.446 0 00.849 1.323zM299.483 518.246a1.399 1.399 0 00-1.166-1.482h-.513l-.415.415a1.349 1.349 0 00-.414.556 1.348 1.348 0 00-.08.688l.059.297.593.592.237-.237-.217.218c.192.162.423.271.671.316l.79-.79.079-.356.376-.217zM298.456 410.075l.75.751a1.427 1.427 0 001.6-1.087 1.281 1.281 0 00-.869-1.54l-.099-.099-.296.02-.158.158a1.37 1.37 0 00-.869 1.58l-.059.217zM308.589 447.39a1.307 1.307 0 00-1.205.454l-.178.178-.079.474-.099.099.119.118.118.395.83.83a1.288 1.288 0 001.482-.771l.118-.118-.02-.336-.336-.336a1.35 1.35 0 00-.257-.583 1.351 1.351 0 00-.493-.404zM330.002 451.459l-.553-.553a1.525 1.525 0 00-1.343.355 1.173 1.173 0 00-.217.217 1.535 1.535 0 00-.791.791 1.87 1.87 0 00.277 2.014l.533.534.257.02.119-.119.355-.118.277-.277a1.41 1.41 0 00.928-.652l.435-.434c.059-.189.079-.389.059-.586a1.442 1.442 0 00-.178-.56l-.178-.178.02-.454zM349.716 499.302l-1.106-1.106a1.231 1.231 0 00-1.007.335 1.26 1.26 0 00-.396.988l.099.099a1.263 1.263 0 00.85 1.086h.553l.671-.671a1.288 1.288 0 00.317-.988l.019.257zM336.383 483.598l.494-.494a1.544 1.544 0 00-.139-1.245 1.374 1.374 0 00-.948-.592l-.118-.119-.316.119-.356.355a1.354 1.354 0 00-.435 1.028l.771.77c.193.161.424.27.671.316l.297.02.296-.296-.217.138zM319.591 469.652l-.256.02-.455.138-.75.751-.04.513.059.257.237.237c.094.153.223.281.376.375l.197.198.593.079.296-.059.514-.514c.278-.245.46-.581.513-.948l-.118-.119-.119-.395-.513-.513a1.11 1.11 0 00-.534-.02zM317.913 479.785l.138.139a1.373 1.373 0 001.13-.392c.146-.146.258-.324.325-.52.068-.195.091-.404.066-.609l-.099-.336-.513-.514-.04-.237-.967-.968-.218.02-.415.099-.493.494-.178.375a1.257 1.257 0 00.316 1.106c-.025.312.051.624.217.889l.435.435.296.019zM305.014 447.528l.336-.336a1.51 1.51 0 00-1.205-1.679l-.218.02-.395.079-1.066 1.066.019.257.138.139.159.355.158.158c.21.225.487.377.79.435.198.018.398.005.592-.04l.474-.474.218.02zM323.958 456.911l-.257-.059a1.37 1.37 0 00-1.462.79l-.316.316-.02.257.119.118.138.375.138.139c.255.251.578.423.929.494l.296.019.099-.098.276-.079.613-.613a1.617 1.617 0 00-.099-1.323l-.435-.435-.019.099zM324.016 467.598l-.296-.06-.553.079-.138.139c-.18.041-.353.108-.514.197a2.88 2.88 0 00-.533.138l-.81.81a1.701 1.701 0 00.197 1.423l.356.355c.152.344.426.619.77.771l.297.019.276-.276c.291-.176.535-.42.711-.711l.06-.257.197-.198c.028-.109.068-.215.119-.316a1.508 1.508 0 00-.139-2.113zM323.858 462.106l-.908.909a1.357 1.357 0 00.05 1.11c.086.172.207.323.354.445.148.121.32.211.504.262l.218.02.158-.158a1.37 1.37 0 00.592-1.462 1.34 1.34 0 00-1.086-1.087l.118-.039zM320.639 454.738l.257.059c.339.065.69.002.985-.177.296-.178.515-.46.615-.791l-.04-.355-.652-.652a1.36 1.36 0 00-.691-.296l-.257-.02-.119.118-.316.119-.888.889 1.106 1.106zM330.279 506.137a1.309 1.309 0 00-1.087.296 1.56 1.56 0 00-.553 1.264l.435.435a1.508 1.508 0 001.58.541c.2-.051.388-.141.553-.265l.494-.493a1.475 1.475 0 00-.178-1.284 1.373 1.373 0 00-1.244-.494zM332.629 433.819l-.651-.652a1.54 1.54 0 00-1.205.296 1.344 1.344 0 00-.475 1.067l.297.296a1.228 1.228 0 001.047 1.047c.177.023.358.017.533-.019l.77-.771c.134-.18.222-.39.257-.612l-.099-.099-.079-.355-.138-.139-.257-.059zM329.211 428.189l.909.909a1.786 1.786 0 001.758-.692l.119-.118.098-.296.099-.099v-.198l-.099-.098-.098-.297-.119-.118a1.415 1.415 0 00-.928-.573 1.517 1.517 0 00-.869.118l-1.028 1.028.119.118.039.316zM328.876 460.21a1.368 1.368 0 00-1.165.296 1.479 1.479 0 00-.514 1.146l.079.316.692.691a.717.717 0 00.178.178 1.424 1.424 0 001.659-1.462l-.079-.316-.85-.849zM342.289 431.666l-.118.118a1.118 1.118 0 00-.83 1.067v.276l.948.949a1.337 1.337 0 001.185-.395c.201-.213.332-.482.376-.771a1.353 1.353 0 00-.598-1.37 1.33 1.33 0 00-.489-.19l-.474.316zM361.943 413.433l.633.632c.143.071.296.117.454.139a1.39 1.39 0 001.402-.613 1.538 1.538 0 00-.079-1.778l-.454-.454-.296-.099a1.378 1.378 0 00-1.264.435 1.595 1.595 0 00-.317 1.501l-.079.237zM324.234 473.346l.098.098.08.277.671.672.316.079a1.458 1.458 0 001.225-.395 1.57 1.57 0 00.375-1.442l-.928-.929-.257-.02-.474.079-.178.178a1.646 1.646 0 00-.928 1.403zM378.082 459.597l-.375.099-.197.198c-.16.101-.294.236-.396.395a1.224 1.224 0 00-.197.197l.059.455.079.395.711.711c.335.046.675-.039.948-.237a1.368 1.368 0 00-.533-2.43l-.099.217zM356.788 438.382a1.533 1.533 0 00-1.679.968l-.039.237-.02.454.987.988.297.059h.237l.395-.118.731-.731-.198-.198.277.079.098-.494v-.237l-.078-.316-.731-.731-.277.04zM327.434 446.441l.118.119.237.04.217-.02.356-.079.849-.85.04-.237-.099-.099-.079-.355-.849-.849a1.306 1.306 0 00-1.087.256 1.31 1.31 0 00-.533 1.008l-.099.099.119.355.434.435.376.177zM343.691 420.939a1.26 1.26 0 00-1.284.85l-.019.296.098.494a1.233 1.233 0 00.988.988c.164.033.333.026.494-.02l.928-.928a1.28 1.28 0 00-.081-.914 1.28 1.28 0 00-.669-.627l-.237-.04-.218-.099zM277.241 464.042l-.81-.81-.336-.059-.099.099-.356.079-.829.829a1.529 1.529 0 00.553 1.541l.198.198.533.059.276-.04.633-.632c.177-.213.287-.474.315-.75l-.118-.119.04-.395zM338.496 496.734l-.297-.099-.533-.02-.296.06-.83.829a1.315 1.315 0 00.099 1.166 1.332 1.332 0 00.869.593h.237l.237-.237a1.318 1.318 0 00.632-1.146 1.462 1.462 0 00-.118-1.146zM323.049 489.86l-.454-.455-.001-.316a1.447 1.447 0 00-1.244-1.244l-.099-.099-.336.059-.75.751a1.849 1.849 0 00-.06 1.403c.139.237.336.434.573.572l.099.099a1.254 1.254 0 001.008 1.008.943.943 0 00.335.019l1.047-1.047.04-.237-.02-.217-.138-.296zM286.011 445.948l.316.039.355-.079.81-.81a1.355 1.355 0 00-.158-1.145 1.442 1.442 0 00-.439-.413 1.442 1.442 0 00-.568-.2l-.119-.118-.335.098-.81.81-.02.218-.099.098.079.395.711.712.277.395zM319.553 401.502l.256.02.455-.099.474-.474a1.46 1.46 0 00.494-.494l.118-.118v-.356l-.691-.691a1.404 1.404 0 00-1.269-.051 1.396 1.396 0 00-.785.999c.003.284.097.559.268.786.17.228.408.395.68.478zM332.313 421.69a1.248 1.248 0 00.829.83l.139.138.415-.02.553-.553c.259-.256.408-.603.414-.968l-.118-.118-.138-.336-.474-.474-.336-.138-.099-.099a1.317 1.317 0 00-.908.166c-.272.164-.476.42-.574.723l.001.237.296.612zM356.808 424.93l.316.316c.114.196.278.36.474.474l.257.02.474-.079.928-.929a1.288 1.288 0 00-1.067-1.541.757.757 0 00-.296-.019l-1.067 1.066v.198l-.019.494zM307.601 456.713l.454-.454a1.368 1.368 0 00.218-1.363 1.285 1.285 0 00-1.027-.83 1.209 1.209 0 00-1.028.316l-.592.593a1.316 1.316 0 00.197 1.027 1.282 1.282 0 00.909.593l.869.118zM301.439 489.504l.612-.612c.136-.198.237-.419.296-.652l-.02-.336-.256-.533-.455-.455-.316-.079-.573.06-.118-.119a1.565 1.565 0 00-1.087 1.324l.099.098.119.356.829.83.297.059.217-.02.355-.079.001.158zM309.359 421.374l.415.415c.112.186.268.342.454.454l.119.119.415-.02.375-.375.336.336h.237l.118.118.277-.039.632-.633a1.287 1.287 0 00.079-1.224 1.434 1.434 0 00-.909-.693 1.436 1.436 0 00-1.125.199l-.336-.336a1.421 1.421 0 00-1.482.692l-.158.157v.435l.553.395zM312.718 425.246a1.504 1.504 0 00.158 1.225c.23.343.584.584.988.671l.256.02.158-.158a1.677 1.677 0 00.494-1.205l-.118-.119-.099-.335-.514-.514-.395-.079-.098.099-.356.079-.672.671.198-.355zM340.294 449.898a1.434 1.434 0 00-1.067.273 1.43 1.43 0 00-.553.952c-.03.172-.024.348.017.518.04.17.115.33.22.47.203.281.508.472.849.533l.395-.079.988-.988-.099-.415-.474-.474-.276-.79zM335.296 488.359l-.04-.356-.849-.849-.237-.04h-.238l-.375.099-.513.513-.158.356-.139.138a1.232 1.232 0 001.589 1.442c.277-.093.513-.282.663-.533l.237-.474.06-.296zM325.616 439.172l.218-.02.138-.138a1.508 1.508 0 00.593-1.541 1.4 1.4 0 00-1.107-1.106l-.454-.02-.474.474a1.457 1.457 0 00-.14 1.332 1.446 1.446 0 001.009.881l.217.138zM323.246 423.725l.81.81.257.059c.197.03.399.02.592-.029a1.492 1.492 0 00.931-.695 1.51 1.51 0 00.196-.56l.039-.237-.059-.415-.948-.948-.316-.079-.119-.119-.276.04-.593.592a2.8 2.8 0 00-.395.949l-.099.098-.02.534zM317.241 479.904l-.849-.849-.336-.099a1.453 1.453 0 00-1.501.75l-.119.435v.276l.652.652c.204.151.441.252.691.297a.763.763 0 00.297.019l.335-.335a1.677 1.677 0 00.494-1.205l.336.059zM322.93 441.029l-.099-.099.06-.533-.336-.336a1.317 1.317 0 00-.553-.553l-.138-.138h-.435l-.968.968-.039.237v.237l.118.395.771.77a1.286 1.286 0 001.619-.948zM322.693 479.39a1.4 1.4 0 00-.869-.553l-.415.099-.415.415c-.188.153-.326.36-.395.593l-.869.869a1.526 1.526 0 00.178 1.165l.829.83a1.344 1.344 0 001.383-.672l.158-.158v-.276a1.281 1.281 0 00.632-.909 1.56 1.56 0 00-.217-1.403zM299.285 426.787l-.177-.178-.06-.375a1.198 1.198 0 00-.889-.889l-.118-.119-.375.059-1.028 1.028c-.077.328-.028.674.139.967l.099.099a1.287 1.287 0 001.165 1.166h.198l.138-.139a1.562 1.562 0 00.908-1.619zM322.693 413.335c.112.174.26.322.435.434l.138.138.415-.019.553-.554a1.22 1.22 0 00.415-1.007l-1.087-1.086a1.396 1.396 0 00-1.501.75l-.138.139-.02.414.79.791zM258.692 460.704a.584.584 0 00-.277-.04l-.908.909a1.56 1.56 0 00.118 1.264 1.501 1.501 0 001.007.652l.257.019.099-.098.296-.099.613-.612a1.482 1.482 0 00-.119-1.225 1.282 1.282 0 00-1.086-.77zM335.691 494.403l-.099-.099a1.284 1.284 0 00-.869-.869l-.118-.118-.395.039-.534.533c-.254.26-.402.605-.414.968l.118.119.158.356.454.454.356.158a1.314 1.314 0 001.481-.85l.119-.118-.257-.573zM297.764 475.874l-.158-.158-.533.02-.968.968a1.288 1.288 0 00.577 1.361c.149.093.316.154.49.18l.296.02.118-.119.297-.099.731-.731a1.367 1.367 0 00-.85-1.442zM349.992 446.56a1.188 1.188 0 00-.454.178 1.23 1.23 0 00-.671 1.303 1.307 1.307 0 001.066 1.067 1.201 1.201 0 001.166-.454l.494-.494-.04-.277-.178-.493-.454-.455-.336-.099-.118-.118-.475-.158zM343.987 464.259a1.479 1.479 0 00-.415-.02l-.651.652a1.45 1.45 0 00-.178 1.324 1.499 1.499 0 00.948.948l.118.119.139-.139a1.482 1.482 0 00.573-1.521 1.397 1.397 0 00-.534-1.363zM304.302 481.662l-.336.336a1.597 1.597 0 00-.316 1.027l.731.731a1.37 1.37 0 00.83-2.608l-.099-.098-.316.039-.118.119-.376.454zM357.795 493.396l-.731-.731a.523.523 0 00-.178-.178 1.304 1.304 0 00-1.013.209 1.312 1.312 0 00-.547.877 1.341 1.341 0 001.027 1.581h.237l.415-.099.395-.395c.18-.212.291-.474.316-.751v-.197l.079-.316zM295.69 458.511l1.028 1.027a1.455 1.455 0 001.422-.593l.158-.158.059-.454-.02-.257-.987-.987a1.342 1.342 0 00-1.581.948l-.079.474zM276.628 452.842l-.889.888-.04.317.04.316 1.067 1.066a1.33 1.33 0 001.007-.256c.14-.103.257-.233.346-.382.088-.15.145-.316.168-.488a1.369 1.369 0 00-1.126-1.6l-.573.139zM287.552 476.566l-.119.118-.375.099-.849.849-.02.218.099.098.079.316.118.119c.129.208.305.384.514.514l.434.434.257-.02.435-.158.533-.533.119-.356.118-.118a1.477 1.477 0 00-1.343-1.58zM314.219 441.167l-.06.297.099.098.079.356.929.928a1.524 1.524 0 001.145-.276 1.457 1.457 0 00.494-1.047l-.079-.316-.533-.534a1.234 1.234 0 00-.672-.316l-.296-.019-.119.118-.217-.059-.77.77zM291.898 485.731l-.553-.553-.455-.099-.256.02-.633.632-.217.415-.039.237c.019.308.114.607.276.869l.553.553.336.06.277-.04.493-.217.198-.198a2.147 2.147 0 00.237-1.698l-.217.019zM271.907 434.253l-.534-.533a1.195 1.195 0 00-.671-.355 1.256 1.256 0 00-1.423.948l.119.118.02.218a1.285 1.285 0 00.869 1.264l.217.019.375-.019.336-.336c.236-.124.429-.317.553-.553l.04-.316.099-.455zM291.917 460.782l-1.008-1.007a1.45 1.45 0 00-1.639 1.205l-.02.217.079.395 1.007 1.008h.238l.118-.119.336-.138.138-.138a1.566 1.566 0 00.494-1.205l.257-.218zM291.699 442.392l.988.988h.198l.375-.099.948-.948-.099-.099-.079-.316-.434-.435a1.459 1.459 0 00-.81-.415l-1.047 1.047-.039.237.019.218-.02-.178zM359.554 454.817l.217.02.375-.02.869-.869c.062-.179.087-.368.074-.557a1.422 1.422 0 00-.982-1.261l-.218-.019-.355.039-.85.85a1.462 1.462 0 00.079 1.185c.171.306.455.532.791.632zM392.819 463.825a1.4 1.4 0 00-1.008-.652.517.517 0 00-.256-.02l-.237.04-.573.572a1.58 1.58 0 00-.268.743c-.024.266.021.534.129.778a1.308 1.308 0 00.968.731c.312.057.634.001.909-.158l.672-.672a1.425 1.425 0 00-.336-1.362zM360.206 472.457h-.277l-.316.079-.751.75c-.064.295-.031.602.093.876.125.274.335.5.599.646l.256.256.277.04.316-.04.79-.79a1.392 1.392 0 00-.731-1.679l-.118.119-.138-.257zM348.551 455.943l.276.276a1.284 1.284 0 001.186-.434c.159-.194.268-.425.316-.672a1.376 1.376 0 00-.573-1.412 1.374 1.374 0 00-.494-.208l-.375.099-.198.198a1.283 1.283 0 00-.395.395.533.533 0 00-.138.138c-.1.168-.155.358-.158.553l.434.435.119.632zM345.765 477.099l-.178-.178a1 1 0 00-.395-.395l-.197-.197-.494-.099-.277.039-.296.099-.731.731-.059.296.118.119.079.355.85.85a1.455 1.455 0 001.481-.731l.06-.257v-.474l.039-.158zM349.045 459.558l-.514-.514a1.936 1.936 0 00-.474.04l-.395-.395a1.334 1.334 0 00-1.146.355 1.252 1.252 0 00-.415 1.047l.099.099.119.316 1.382 1.383a1.236 1.236 0 001.462-1.027 2.073 2.073 0 00-.296-1.126l.178-.178zM337.054 435.913a1.278 1.278 0 00.968.968c.177.023.358.016.533-.02l.652-.652a1.341 1.341 0 000-1.225 1.402 1.402 0 00-1.067-.79 1.26 1.26 0 00-1.442.889v.277l.356.553zM350.466 459.676l.139-.138a1.405 1.405 0 00.454-1.205 1.403 1.403 0 00-.909-.909l-.138-.138-.454.02-.929.928-.079.395v.277l1.047 1.047.217-.02.652-.257zM374.743 475.459l-.296-.019-.197.197a1.425 1.425 0 00-1.107.83 1.317 1.317 0 00.376 1.56l.177.178.455.099h.197l.356-.079 1.067-1.067a1.32 1.32 0 00-.126-1.084 1.32 1.32 0 00-.902-.615zM358.447 479.904l-.099-.099-.059-.296-.237-.237a1.308 1.308 0 00-.968-.731h-.395l-.869.869-.06.533.06.336.849.85.237.039a1.336 1.336 0 001.027-.237 1.257 1.257 0 00.514-1.027zM363.129 440.417l1.007 1.007a1.36 1.36 0 001.087-.296 1.462 1.462 0 00.533-1.126l-.099-.099-.079-.316-.79-.79a1.396 1.396 0 00-1.58 1.067l-.119.118.04.435zM358.17 466.551l.316-.316a1.578 1.578 0 00-.138-1.245 1.474 1.474 0 00-.968-.612l-.099-.099-.316.119-.316.316a1.342 1.342 0 00-.414 1.007l.691.691c.193.161.424.27.671.316h.317l.474-.474-.218.297zM369.569 467.321c.16-.15.283-.335.358-.541.076-.206.102-.426.076-.644l-.948-.948a1.421 1.421 0 00-1.482.691l-.079.277.415.414c.226.289.549.486.909.554.223.143.486.212.751.197zM339.286 444.94l.277.277a1.368 1.368 0 001.106-.316c.239-.189.406-.454.474-.751a1.36 1.36 0 00.257-.257l.119-.118-.02-.415-.494-.494a1.4 1.4 0 00-.435-.435l-.118-.118h-.395l-.494.494a1.25 1.25 0 00-.454.454l-.257.257c-.126.377-.098.789.079 1.146l.217.217.138.059zM336.758 457.503l.375.376a1.418 1.418 0 001.726.358 1.43 1.43 0 00.486-.398 1.453 1.453 0 00.139-1.521l-.633-.632a1.415 1.415 0 00-1.145.316 1.422 1.422 0 00-.514 1.146l-.434.355zM336.461 468.388a1.475 1.475 0 00.435-1.225l-.455-.455a1.338 1.338 0 00-.81-.454 1.068 1.068 0 00-.651.059l-.771.771a1.308 1.308 0 00.079 1.185 1.348 1.348 0 00.968.691 1.363 1.363 0 001.205-.572zM343.296 486.877l-.119-.316-.79-.79-.296-.059-.218.019-.355.079-.988.988v.237l.139.415.513.513.375.139.277.039a1.401 1.401 0 00.993-.409 1.393 1.393 0 00.409-.993l.06.138zM363.8 448.12a1.381 1.381 0 00.197 1.032 1.377 1.377 0 00.87.588l.257-.02.533-.533a1.4 1.4 0 00.178-1.955l-.395-.396-.336-.098a1.393 1.393 0 00-1.304 1.382zM332.452 465.761a1.54 1.54 0 00.612-1.521 1.338 1.338 0 00-1.067-1.067l-.099-.099-.118.119a1.287 1.287 0 00-.593.948 1.564 1.564 0 00.356 1.225l.632.632.356-.079-.079-.158zM345.805 440.278l.098.297.85.849a1.547 1.547 0 001.225-.316 1.458 1.458 0 00.493-1.047l-.77-.77a1.66 1.66 0 00-.691-.297l-.297-.019-.671.671a1.313 1.313 0 00-.316.988l.079-.356zM335.395 463.449h.395l.494-.493c.196-.115.36-.278.474-.475l.02-.217-.02-.375-.494-.494a1.193 1.193 0 00-.454-.454l-.119-.119h-.395l-.474.474a1.432 1.432 0 00-.474.474l-.138.139.039.395.455.454c.125.187.286.348.474.474l.217.217zM338.99 463.173l.533.533a.734.734 0 00.178.178 1.424 1.424 0 001.659-1.462l-.652-.652a1.89 1.89 0 00-.592-.237 1.396 1.396 0 00-1.423.672l-.138.138-.079.395v.237l.178.178.336.02zM335.079 460.486l.414-.019.277-.277a1.483 1.483 0 00.612-1.521 1.346 1.346 0 00-1.067-1.067h-.197l-.336.06-.75.75c-.19.108-.347.265-.455.455l-.118.118v.395l.494.494c.123.177.277.331.454.454l.672.158zM335.987 451.36h.396l.493-.494a1.31 1.31 0 00.455-.454l.118-.119v-.395l-.513-.513a1.455 1.455 0 00-.435-.435l-.217-.02h-.395l-.494.494c-.188.11-.345.266-.454.454l-.04.238.02.375.494.494c.107.19.264.347.454.454l.118-.079zM339.859 472.259l-.099-.335-.85-.85-.395-.079-.098-.099-.316.079a1.572 1.572 0 00-1.838 1.324l.04.316.889.889.474.158a1.479 1.479 0 001.126-.257 1.328 1.328 0 001.011-.391 1.346 1.346 0 00.391-1.011l-.335.256zM344.422 470.857l.632.632a1.284 1.284 0 001.126-.375 1.48 1.48 0 00.376-1.482 1.394 1.394 0 00-.988-.988 1.283 1.283 0 00-.988.277 1.29 1.29 0 00-.513.988l.118.118c-.01.295.073.585.237.83zM355.602 448.891l.316.039.218-.019.375-.06.178-.098.217-.139.118-.158.198-.316.079-.197.04-.158-.02-.218-.079-.355-.079-.198-.158-.197-.139-.138-.316-.198-.217-.059-.296-.06h-.237l-.356.079-.178.099-.217.139-.118.158-.198.316-.079.197-.039.158.019.217.079.356.099.178.138.217.139.138.335.218.178.059zM361.134 485.435l-.415-.415c-.114-.16-.255-.3-.415-.415l-.138-.138-.415.02-.553.553a1.359 1.359 0 00-.415 1.007l.099.099.118.316.909.909.217-.02.356-.079.849-.849v-.198l-.197-.79zM346.694 489.05l-.118.118-.356.079-.83.83a1.395 1.395 0 00.467 1.567c.154.119.332.204.521.251l.217-.139.316-.039.79-.79a1.39 1.39 0 00.02-1.324 1.308 1.308 0 00-1.027-.553zM362.931 507.46a1.43 1.43 0 00-1.145.316 1.45 1.45 0 00-.435 1.106l.119.119.118.316.435.435c.183.137.391.238.612.296.344.043.691-.05.968-.257l.454-.454a1.39 1.39 0 00-.079-1.185 1.387 1.387 0 00-1.047-.692zM364.669 455.587l-.098.099-.297.02-.691.691c-.064.132-.11.271-.138.415l-.02.217.039.395 1.107 1.107.276-.04.514-.513a1.39 1.39 0 00.263-.671 1.39 1.39 0 00-.105-.712 1.476 1.476 0 00-.85-1.008zM367.277 486.877l-.316-.04-1.225 1.225.099.099.059.336.316.316c.108.19.265.347.455.454l.257.02.493-.059.929-.929a1.282 1.282 0 00-1.067-1.422zM342.842 460.348l1.461-1.462.04-.237-.02-.217-.099-.375-.849-.85a1.377 1.377 0 00-1.086.296l-1.324 1.324-.197.198-.178.177-.079.474.02.257.928.929a1.434 1.434 0 001.225-.395l.158-.119zM350.269 474.65l-.79-.791a1.356 1.356 0 00-1.087.257 1.523 1.523 0 00-.553 1.027l.119.119.079.316.81.81a1.459 1.459 0 001.106-.277c.3-.261.485-.63.514-1.027l-.099-.099-.099-.335zM374.25 428.545l.335-.336a1.48 1.48 0 00-.177-1.086 1.453 1.453 0 00-.929-.573l-.296-.02-.099.099-.336.138-.711.711-.079.435.04.276.099.297.73.73a1.175 1.175 0 001.423-.671zM283.68 473.761l.514-.514a1.286 1.286 0 00.276-1.264 1.2 1.2 0 00-.494-.494l-.592-.593a1.68 1.68 0 00-1.343.316l-.475.474c-.077.301-.063.617.04.909l1.166 1.166h.908zM281.862 498.374l-.889-.889a1.366 1.366 0 00-1.481.73l-.099.099-.059.257-.099.099.593.592c.213.296.531.501.888.573a1.256 1.256 0 001.087-.336 1.4 1.4 0 00.434-1.185l-.375.06zM314.989 460.15l.356.356.493.099h.277l.948-.948c.145-.35.331-.681.553-.988l.158-.158.04-.435-.02-.256-.869-.87a1.29 1.29 0 00-1.146.356l-1.205 1.205-.099.336-.138.138a1.37 1.37 0 00.652 1.165zM315.306 454.856l.513-.513a1.561 1.561 0 00-.138-1.719l-.217-.217-.553-.158h-.277l-.237-.04h-.435l-.967.968-.04.316v.237l.138.415.731.731.553.158h.277l.276-.039.099-.099.277-.04zM313.014 466.076l-1.067-1.066a1.362 1.362 0 00-1.086.375 1.33 1.33 0 00-.375 1.007l.118.119.138.336.455.454.336.138a1.337 1.337 0 001.481-.849l.02-.217-.02-.297zM311.138 486.64l-.415.099-.415.415a1.052 1.052 0 00-.395.948 1.319 1.319 0 00.494.928l.415.415a1.227 1.227 0 00.948-.198c.258-.217.433-.517.493-.849a1.389 1.389 0 00-.168-1.132 1.393 1.393 0 00-.957-.626zM294.821 433.108l-.099-.376a1.18 1.18 0 00-.849-.493 1.48 1.48 0 00-1.047.296 1.31 1.31 0 00-.276 1.857l.256.257.692.098.375-.098.533-.534.02-.256.395-.751zM250.849 462.106l-.276-.277-.692-.098-.375.098a1.358 1.358 0 00-.336.336l-.277.277-.098.691.118.395c.214.276.525.46.869.514a1.34 1.34 0 00.968-.297 1.364 1.364 0 00.099-1.639zM295.907 421.315l-.513.513a1.318 1.318 0 00.178 1.284l.711.712 1.422-1.423.039-.237v-.197l-.059-.336-.276-.277a1.34 1.34 0 00-1.798-.414l.296.375zM282.731 445l-.138.533.02.336.296.296a1.254 1.254 0 00.948.514 1.438 1.438 0 001.186-.474l.414-.415-.019-.297-.099-.098a1.65 1.65 0 00-.277-.277 1.313 1.313 0 00-1.837-.217l-.197-.198-.297.297zM304.263 428.604l-.059.257v.434l.098.336.257.257a1.247 1.247 0 001.067.513 1.416 1.416 0 001.145-.711l.119-.118.079-.316-.099-.099-.019-.217-.277-.277a1.503 1.503 0 00-1.482-.533l-.651.652-.178-.178zM273.389 453.079l-.158-.158a1.457 1.457 0 00-.238-1.067 1.684 1.684 0 00-1.205-.612l-.335.098-.771.771a1.345 1.345 0 00.237 1.066c-.099.319-.078.664.06.968l.889.889.395-.039.138-.139a1.591 1.591 0 00.79-1.264l.198-.513zM258.297 454.54l-.553-.553a1.367 1.367 0 00-1.462-.474 1.455 1.455 0 00-1.067 1.344v.197l.455.454a1.333 1.333 0 001.461.475 1.54 1.54 0 001.166-1.443zM327.098 430.698l.395.395c.168.1.354.165.548.189.194.023.391.006.578-.051l.533-.533c.173-.249.262-.547.257-.85l-.138-.335-.77-.771a1.596 1.596 0 00-1.482.652 1.29 1.29 0 00.079 1.304zM267.956 432.12l.119-.356.118-.118v-.237l-.435-.435a1.45 1.45 0 00-1.6-.494l-.454.455-.118.355-.139.139c-.014.204.027.409.119.592l.612.613.336.098.119.119a1.32 1.32 0 001.007-.454l.316-.277zM279.749 423.211a1.37 1.37 0 00-1.008-.533 1.433 1.433 0 00-1.126.494l-.454.454v.237l.099.099.099.296.77.771.276.039.376-.099.928-.928.02-.257-.02-.217-.098-.336.138-.02zM280.322 449.977l-.712-.711a1.67 1.67 0 00-.829.079l-.711.711-.04.237.02.336c.021.259.117.506.277.711l.632.632.217.02.296-.059a1.502 1.502 0 00.731-.929 1.239 1.239 0 00.119-1.027zM335.908 409.957l-.079.434-.019.218c.004.295.109.581.296.809.088.146.21.268.355.356l.317.316a1.29 1.29 0 001.402-.731l.415-.415-.02-.375-.138-.138a.561.561 0 00-.217-.218 1.458 1.458 0 00-1.601-.493l-.572.572-.139-.335zM272.242 426.53l.81-.81.04-.237v-.435l-.336-.335a1.423 1.423 0 00-1.818-.395l-.671.671a1.39 1.39 0 00.257 1.047 1.424 1.424 0 001.718.494zM297.033 407.586c.295-.032.571-.156.79-.355l.514-.514.02-.257-.119-.118-.119-.356-.138-.138a1.586 1.586 0 00-1.442-.534l-.671.672-.139.375-.118.119c-.054.265-.027.541.079.79l.849.849.257-.019.237-.514zM270.959 428.762l-.652-.652a1.403 1.403 0 00-1.284.454c-.16.204-.265.445-.306.701-.042.255-.018.517.069.761l.81.81a1.429 1.429 0 001.501-.711 1.29 1.29 0 00-.138-1.363zM293.3 398.519l-.237.237-.099.297.02.572.197.198c.245.306.6.504.988.553.197.016.396-.009.583-.073.187-.065.358-.168.504-.302l.454-.455-.04-.276-.355-.356a1.307 1.307 0 00-1.679-.335l-.356.355.02-.415zM290.534 417.759l-.257-.019-.098.098-.356.119-.928.928a1.451 1.451 0 00.256 1.245 1.426 1.426 0 001.936.237l.613-.613.039-.276-.118-.119a1.578 1.578 0 00-.316-.711l-.415-.415-.356-.474zM241.269 407.883l-.276-.277-.395-.039-.139-.139a1.124 1.124 0 00-.928.415 1.316 1.316 0 00-.257 1.008l-.118.118.098.375a1.237 1.237 0 00.838.504 1.232 1.232 0 00.94-.267c.24-.184.413-.44.494-.73.04-.344-.052-.69-.257-.968zM267.304 424.317l.257.257a1.287 1.287 0 001.66-.158l.296-.296a1.432 1.432 0 00.039-1.501l-.671-.672a1.618 1.618 0 00-1.561.731 1.344 1.344 0 00.099 1.521l-.119.118zM275.739 418.925l-.118-.119.118-.118-.118-.395a1.426 1.426 0 00-1.996-.257 1.46 1.46 0 00-.572 1.007c-.039.365.06.731.276 1.027l.395.396a1.267 1.267 0 001.126-.06 1.309 1.309 0 00.711-.987l.178-.494zM269.971 419.991l.098-.296.119-.118-.415-.415a1.177 1.177 0 00-.494-.494l-.217-.217-.277-.04a1.41 1.41 0 00-.648.294 1.406 1.406 0 00-.418.575 1.254 1.254 0 00.158 1.264c.206.271.512.449.849.494l.375.06a1.418 1.418 0 001.067-.553l-.197-.554zM263.452 420.466l.514-.198.454-.454.079-.316v-.554l-.316-.316a1.372 1.372 0 00-1.419-.586 1.369 1.369 0 00-.497.211 1.42 1.42 0 00-.652.652 1.283 1.283 0 00-.118 1.106l.711.711.316.079c.248-.013.494-.06.731-.138l.197-.197zM249.171 443.874a1.316 1.316 0 00-1.177-.519 1.312 1.312 0 00-1.055.736l-.099.099-.059.256.967.968.257.02a1.311 1.311 0 00.771-2.113l.395.553zM283.502 385.205l.434.435.297.099.118.118c.23.013.46-.027.672-.118l.592-.593.079-.316.119-.119a1.539 1.539 0 00-.138-.651l-1.087-1.087a1.482 1.482 0 00-.652.138l-.573.573-.079.316-.118.119c-.013.209.021.418.099.612l.237.474zM287.986 400.435l-.118.119-.06.257c-.001.203.039.405.119.592l.711.712.316.039.119-.119.375-.138.138-.138a1.342 1.342 0 00.257-1.877l-.652-.652h-.237l-.316.08-.375.375-.277.75zM271.037 401.383l-.632-.632-.691-.059-.356.119a1.31 1.31 0 00-.423 1.727 1.311 1.311 0 001.687.564 1.284 1.284 0 00.415-1.719zM275.877 409.186l-.257.257v.198l.059.335.316.317a1.425 1.425 0 001.126.572 1.389 1.389 0 001.107-.592l.632-.632.039-.277-.099-.099-.098-.375-.83-.83-.257-.059-.118.119-.356.118-.691.691-.573.257zM300.174 430.797l-.119.118a1.345 1.345 0 00.119 1.462 1.505 1.505 0 001.541.514 1.537 1.537 0 001.066-1.265l-.118-.118-.138-.376-.573-.572a1.418 1.418 0 00-1.6.178l-.217.217.039-.158zM292.174 369.857l.672-.672v-.197a1.34 1.34 0 00-.217-1.245l-.672-.672-.237.04a1.334 1.334 0 00-.929.494l-.375.375.02.257.118.118c.067.206.167.4.297.573a1.34 1.34 0 001.323.929z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M335.968 463.193a1.31 1.31 0 001.383.434l.434-.434a1.4 1.4 0 00.415-.81 1.165 1.165 0 00-.277-.949l-.395-.395a1.283 1.283 0 00-1.58.356 1.479 1.479 0 00.02 1.798zM231.471 457.464a1.394 1.394 0 00.178-1.877 1.178 1.178 0 00-.375-.375l-.395-.395a1.31 1.31 0 00-1.205.573l-.198.197-.079.277.02.494.099-.099-.297.099c.027.248.116.486.257.691l.533.534.395.039.218-.02.513-.513.336.375zM304.836 377.976l-.138.138.059.296.553.553a1.422 1.422 0 001.896.277 1.286 1.286 0 00.435-1.699l-.632-.632a1.34 1.34 0 00-1.699.553l-.474.514zM286.663 412.07a1.367 1.367 0 00-1.205-.494l-.494.494-.138.138-.534.534v.276l.119.119.118.355.158.158a1.31 1.31 0 001.62.474l.652-.651a1.456 1.456 0 00-.296-1.403zM299.365 429.75l-.06-.415-.276-.277a1.59 1.59 0 00-1.225-.671l-.277-.04a1.756 1.756 0 00-1.145.04 1.953 1.953 0 00-1.185 1.659l.118.118.119.356.474.474.355.158.119.119.118-.119c.341.196.742.259 1.126.178h.198l.276-.04a1.392 1.392 0 00.692-.928l.178-.178.395-.434zM281.013 478.087l.118-.316.001-.237-.613-.613a1.315 1.315 0 00-.968-.454 1.31 1.31 0 00-.948.474l-.119.119a1.202 1.202 0 00-.098 1.363l.79.79.336-.02c.359-.045.685-.229.908-.514l.593-.592zM282.08 457.503l.118.119.296-.059.119-.119a1.475 1.475 0 00.731-1.007 1.346 1.346 0 00-.257-1.087l-.494-.494-.257-.019a1.43 1.43 0 00-1.027.434l-.296.297.118.118c.047.249.149.485.297.691l.533.534.119.592zM279.393 435.853l-.177-.177-.435-.198-.257-.059-.276.039-1.008 1.008-.099.336a1.244 1.244 0 00-.079.513l.83.83.316.079h.316a1.48 1.48 0 001.027-.316l.158-.158.178-.376.138-.138v-.276l-.098-.099a1.099 1.099 0 00-.534-1.008zM271.215 442.096l.119.118a1.365 1.365 0 00.988.553 1.419 1.419 0 001.165-.454l-.02.02.336-.336.04-.237.316-.316.079-.277v-.237l-.079-.434-.889-.889-.277-.04-.138.138-.375.139-.119.118a1.399 1.399 0 00-.533 1.047c-.219.215-.371.49-.435.79l.099.099-.277.198zM283.957 409.759l.414-.415a1.476 1.476 0 00.257-2.034l.119.118-.415-.415-.257-.019h-.237l-.415.138-.652.652-.118.355-.04.237c.01.307.114.603.297.85.116.158.256.298.415.415l.098.098.316.079.218-.059zM291.324 429.572l-.158-.158a1.371 1.371 0 00-1.62-.435l-.652.652-.158.356-.019.217c.003.36.128.708.355.988l.731.731.296-.06.099-.098c.21-.083.401-.209.561-.368.159-.16.285-.351.368-.561l-.099-.099.296-1.165zM284.015 437.394l-.75-.75a1.443 1.443 0 00-.474-.198 2.141 2.141 0 00-.514-.276l-.217-.218-.237-.039-.099-.099-.336.099-.118.118a1.736 1.736 0 00-.85 1.205l.079.316a1.35 1.35 0 00.138 1.087l.652.652c.318.11.662.124.988.039l.178-.178.276-.079.218-.217a1.45 1.45 0 001.066-1.462zM287.651 433.286l-.435-.435-.316.04-.119.118a1.435 1.435 0 00-.7.844 1.438 1.438 0 00.108 1.092l.573.573a1.453 1.453 0 001.442-.652 1.43 1.43 0 00-.119-1.541l-.434-.039zM290.357 424.653l-.119.119-.039.237.039.355.296.297c.202.277.499.471.834.544.335.072.685.019.984-.149l.77-.771v-.316a1.283 1.283 0 00-.296-.691l-.652-.652h-.237l-.316.119-.415.414-.849.494zM262.168 469.75a1.312 1.312 0 00-1.007-.533 1.54 1.54 0 00-1.245.533l-.316.317-.059.256c-.024.382.088.76.316 1.067a1.5 1.5 0 001.047.573l.257-.02.296-.296a1.529 1.529 0 00.731-1.047c.09-.277.083-.577-.02-.85zM313.429 418.688l-.099-.099-.079-.356-.277-.276a1.454 1.454 0 00-1.086-.612 1.416 1.416 0 00-1.067.434l-.316.316a1.284 1.284 0 00.04 1.501c.098.138.218.258.355.356l.119.119.355.078.237-.039.119-.119a1.65 1.65 0 00.612-.296l.494-.494.593-.513zM314.831 412.228l-.138.139-.04.237-.098.098.059.336.257.257a1.73 1.73 0 001.738.711l1.027-1.027a1.48 1.48 0 00-.296-1.047 1.74 1.74 0 00-.632-.632l-.119-.119-.257-.059h-.237l-.375.099-.454.454-.435.553zM292.628 435.537l-.178-.177a1.428 1.428 0 00-1.691-.442 1.468 1.468 0 00-.521.363l-.494.493.099.099.059.297.296.296a1.506 1.506 0 001.047.573 1.57 1.57 0 001.166-.494l.434-.435.02-.217-.059-.296-.178-.06zM322.002 423.705l-.119-.119a1.177 1.177 0 00-1.323.218l-.455.454-.019.296.098.415a1.375 1.375 0 001.146.514c.287-.033.56-.142.79-.316a1.288 1.288 0 00.512-1.39 1.303 1.303 0 00-.235-.467l-.395.395zM348.768 423.922l-.138.139.02.217a1.3 1.3 0 00.316.316 1.371 1.371 0 001.461.514 1.617 1.617 0 001.245-1.245l-.02-.217-.494-.494a1.417 1.417 0 00-1.225-.553 1.53 1.53 0 00-1.244.889l.079.434zM280.282 441.72l-.415.415-.02.217.02.297.336.336a1.364 1.364 0 001.146.553 1.654 1.654 0 001.283-.81l.159-.158.039-.514-1.067-1.067a1.567 1.567 0 00-1.481.731zM328.876 468.625l-.277-.277-.296-.02a.985.985 0 00-.534-.059l-.888.889-.04.316-.059.296.118.395c.217.259.518.433.85.494.352.04.706-.06.987-.276a1.338 1.338 0 00.139-1.758zM327.928 438.56a1.538 1.538 0 00-1.897-.435l-.829.83-.02.257.079.434.948.948.297.02.217-.02.336-.138.731-.731.059-.257-.02-.493-.079-.277.178-.138zM301.004 424.317l.829.83.277.04.217-.02.336-.099.829-.83.06-.256v-.198l-.099-.375-.119-.119a1.335 1.335 0 00-.99-.56 1.336 1.336 0 00-1.064.402l-.138.139-.139.335-.118.119-.02.257.139.335zM330.318 416.89a1.337 1.337 0 00-1.54-.237l-.554.553a1.261 1.261 0 00.119 1.383c.087.146.209.268.355.355l.119.119.316.079.415-.415a1.313 1.313 0 00.494-1.639l-.158-.158.434-.04zM252.37 402.747l.119-.119.019-.257.099-.099-.118-.355-1.008-1.008-.099.099-.375.059-.138.139a1.45 1.45 0 00-.632 1.501l.75.751.316.079.297.019.256-.059c.28-.041.542-.165.751-.355l.138-.139-.375-.256zM274.593 468.546l-.79-.79-.356-.079-.257.019-.375.099-.118-.118a1.464 1.464 0 00-.712.948 1.289 1.289 0 00.237 1.027l.534.533a1.415 1.415 0 001.224-.395 1.455 1.455 0 00.613-1.244zM268.075 452.94l.138-.138.02-.257-.099-.099c.081-.063.154-.136.217-.217a1.48 1.48 0 00.06-1.758l-.613-.612-.316.039-.118.119a1.734 1.734 0 00-.988.987 1.351 1.351 0 00-.04.791l-.118.118a1.283 1.283 0 00.079 1.422l.217.218.297.098.217.218.099-.099.296-.099.454-.454.198-.277zM271.156 395.576l.731.731.237.039.099.099.335-.059.119-.119a1.341 1.341 0 00.671-.948 1.41 1.41 0 00-.276-1.145l-.652-.652h-.237l-.336.098-.829.83-.06.257.119.118.099.376-.02.375zM326.091 386.193l.494.494.256.059.316.04a1.26 1.26 0 00.712-.119l.79-.79v-.355l-.119-.119a1.196 1.196 0 00-.217-.415 1.397 1.397 0 00-1.916-.336 1.374 1.374 0 00-.316 1.541zM321.527 409.641a1.229 1.229 0 00-.059 1.165l.593.593.355.118.257.02c.36-.02.698-.175.948-.435l.139-.138.118-.355.119-.119.019-.257-.118-.118-.158-.356-.099-.099a1.285 1.285 0 00-1.699-.355l-.158.158-.257.178zM337.133 428.802l-.119.118-.059.257c-.012.217.028.434.118.632l.692.691.316.04.118-.119.356-.118.138-.138a1.334 1.334 0 00.508-1.42 1.33 1.33 0 00-.231-.477l-.336-.336-.217.02-.099.099-.296.099-.356.355-.533.297zM279.413 418.214l.276.276a1.486 1.486 0 001.087-.849 1.336 1.336 0 00-.158-1.304 1.176 1.176 0 00-.988-.474l-.316.079-.118.118a1.453 1.453 0 00-.573.889 1.4 1.4 0 00.276 1.067l.514.198zM252.588 437.967l.138.138.415-.098c.243-.038.473-.132.672-.277l.237-.237.177-.533.02-.336-.099-.336-.632-.632a1.592 1.592 0 00-1.718.217l-.514.514-.02.257.119.355.731.731.474.237zM305.014 393.68l-.119.355-.039.237c-.013.218.028.434.118.632l.593.593.336.099.098.296.81.81.435.079h.237a1.424 1.424 0 00.889-.889 1.318 1.318 0 00-.237-1.185 1.452 1.452 0 00-.909-.553l-.099-.297-.118-.118a1.396 1.396 0 00-1.561-.494l-.632.632.198-.197zM304.895 398.697a1.596 1.596 0 00-.711 1.027 1.423 1.423 0 00.237 1.146l.711.711a1.644 1.644 0 001.185-.553l.474-.474v-.237l-.099-.099-.098-.336-.909-.908-.237-.04h-.198l-.355.079v-.316zM308.095 435.636a1.45 1.45 0 00-2.014-.197 1.309 1.309 0 00-.257 1.837l.276.276.711.079.376-.099.77-.77.099-.691.039-.435zM277.99 460.249l.237-.237.099-.336v-.592l-.948-.948h-.277l-.217.019-.375.139-.158.158a1.285 1.285 0 00-.237 1.777 1.313 1.313 0 001.284.455l.494-.178.098-.257zM304.619 417.779l.572.573a1.535 1.535 0 001.462-.711 1.458 1.458 0 00-.118-1.581l-.613-.612h-.237a1.542 1.542 0 00-1.047.849 1.568 1.568 0 00-.019 1.482zM313.508 404.9l-.277.276v.198l.04.355.316.316c.553.553 1.264.751 2.133.237l.613-.612.079-.277-.02-.217-.079-.355-.83-.83-.415-.059-.217.019a3.087 3.087 0 00-.948.396l-.138.138-.257.415zM270.643 441.72l.414-.414-.019-.297-.336-.335a1.455 1.455 0 00-1.62-.475l-.849.85-.02.454c.037.243.132.473.276.672l.514.513a1.649 1.649 0 001.166-.533l.474-.435zM301.438 417.285l.533-.533.079-.277a1.5 1.5 0 00-.079-.79l-.632-.632-.276-.079h-.237l-.435.079-.889.889-.02.257.04.237.138.375.119.118a1.282 1.282 0 001.659.356zM274.968 445.059a1.424 1.424 0 00-.256-1.008l-.257-.257-.375-.059-.139-.138a1.17 1.17 0 00-.691.178l-.099-.099-.276-.079-.099-.099-.375.375a1.678 1.678 0 00-.672.988l-.099.099.04.316.533.533a1.311 1.311 0 001.462.435l.178-.178-.297.296a1.306 1.306 0 001.087-.256 1.41 1.41 0 00.335-1.047zM293.991 390.717a1 1 0 00.178-.178.938.938 0 00.277-.277 1.231 1.231 0 00-.059-1.244l-.731-.731-.237-.039h-.237l-.356.078-.138.139a1.808 1.808 0 00-.573.81 1.284 1.284 0 00-.099 1.639l.613.613a1.563 1.563 0 001.362-.81zM320.224 396.406l-.099.335-.138.139c-.012.209.021.418.099.612l.612.612.356.119.118.118a1.334 1.334 0 001.008-.454l.474-.474v-.237l-.099-.099-.099-.296-.118-.119a1.401 1.401 0 00-1.6-.493l-.632.632.118-.395zM241.269 387.694l-.731-.731-.099.099-.296.059a1.634 1.634 0 00-.77 1.008 1.418 1.418 0 00.296 1.165l.316.316.217.02.316-.039.119-.119a1.396 1.396 0 00.454-2.153l.178.375zM274.159 464.911l.158-.158c.111-.188.171-.401.174-.62a1.247 1.247 0 00-.154-.624l-.573-.573-.356-.079-.118-.119c-.364.013-.709.16-.968.415l-.139.138-.138.336-.118.119v.276l.474.474a1.315 1.315 0 001.679.376l.375-.376-.296.415zM259.244 425.424l-.75-.751-.316-.039h-.237l-.395.118-.139.138a1.32 1.32 0 00-.502 1.389c.043.17.12.329.226.468l.296.296.316.119h.237l.099-.099.296-.099.119-.118a1.459 1.459 0 00.75-1.422zM317.34 439.864a2.124 2.124 0 00-.434-.198 1.284 1.284 0 00-1.383.513 1.29 1.29 0 00-.014 1.376 1.284 1.284 0 001.239.6l.572-.573.198-.474.039-.277-.81-.809.593-.158zM300.866 448.634l-.534-.533h-.237a1.451 1.451 0 00-.928.928 1.537 1.537 0 00.138 1.363l.652.652a1.422 1.422 0 001.462-.711 1.373 1.373 0 00-.553-1.699zM260.015 434.135l-.099-.099-.296.02a1.455 1.455 0 00-.929.533l-.178.178.099.099.06.296.217.217a1.41 1.41 0 001.975.316 1.41 1.41 0 00.316-1.975l-.494-.494-.671.909zM290.93 479.311l.237-.237-.001-.197-.039-.316-.296-.297a1.337 1.337 0 00-1.42-.508 1.349 1.349 0 00-.477.232 1.304 1.304 0 00-.551.848 1.318 1.318 0 00.216.989l.612.612.257.02.237-.039c.283-.037.55-.154.77-.336l.119-.119.336-.652zM269.615 411.339l-.158.158-.039.277.019.217.099.415.158.158a1.485 1.485 0 001.442.533l1.008-1.007-.02-.454-.316-.316a1.368 1.368 0 00-1.817-.396l-.139.139-.237.276zM259.62 394.193l-.691-.099-.375.099-.771.771-.039.316.098.375a1.349 1.349 0 001.897.237 1.425 1.425 0 00.257-1.956l-.376.257zM251.324 418.925l-.554-.553-.237-.04-.118.119-.613.217-.869.869c-.054.239-.026.49.079.711l.632.632.435.198.138.138.336-.019.514-.514a1.48 1.48 0 00.257-1.758zM294.426 412.406l-.059.257.059.375.277.277a1.314 1.314 0 001.027.553 1.436 1.436 0 001.086-.375l.553-.554-.138-.138a1.254 1.254 0 00-.276-.671l-.672-.672-.316-.119-.119-.118h-.237l-1.185 1.185zM248.202 428.604l.02-.257-.929-.928-.276-.04-.257.02-.434.158-.178.178a1.702 1.702 0 00-.395.83l.217.217-.316.039.039.514.633.632.513.158.296.02a2.066 2.066 0 001.304-1.067l-.237-.474zM268.924 377.383l.079-.276.02-.336a1.567 1.567 0 00-.277-.751 1.228 1.228 0 00-1.679-.257l-.296.297a1.228 1.228 0 00-.217 1.521l.73.73c.262.035.527.001.771-.098l.217-.218.652-.612zM265.644 410.095v-.237l-.098-.099-.119-.355-.138-.139a1.484 1.484 0 00-2.035-.256l-.573.572-.039.277.099.099.118.355.613.613.316.118.118.119a1.788 1.788 0 001.205-.534l.533-.533zM277.753 396.88l-.138.138-.02.257.02.217.079.356.85.849.256.02h.198l.355-.119.751-.75.059-.297a1.422 1.422 0 00-.276-.869l-.751-.039-.731-.099-.316.316-.336.02zM318.703 452.486l.81.81.375-.059.139-.139a1.431 1.431 0 00.746-1.419 1.449 1.449 0 00-.173-.536l-.81-.81-.356.04-.138.138a1.537 1.537 0 00-.731.928c-.093.354-.043.73.138 1.047zM336.718 481.879a1.418 1.418 0 00-.296-1.165l-.573-.573-.257-.02a1.62 1.62 0 00-1.067 1.067 1.285 1.285 0 00.988 1.699l.217.217.356-.039.138-.139c.293-.273.469-.647.494-1.047zM307.048 465.721l-.415-.415-.276-.04-.514.001-.178.177a1.46 1.46 0 00-.573 1.6l.158.356.613.612.553.04.276-.079a1.426 1.426 0 00.652-1.719l-.296-.533zM309.873 445.711l.118.395a1.308 1.308 0 001.166.493c.244-.032.475-.127.672-.276a1.347 1.347 0 00.54-1.413 1.355 1.355 0 00-.224-.484l-.277-.276-.296-.02a.985.985 0 00-.534-.059l-.217.217a1.261 1.261 0 00-.474.277l-.198.197-.276.949zM293.359 459.222l.059-.257c.074-.188.101-.391.079-.592l-.928-.929-.277-.039-.493.059-1.067 1.067.118.118.099.376.138.138a1.426 1.426 0 001.581.513l.513-.513.178.059zM307.779 448.674l-.336-.336a1.358 1.358 0 00-.336-.336l-.098-.375-.139-.139a1.427 1.427 0 00-1.066-.553 1.29 1.29 0 00-1.028.514l-.118.118-.099.297-.118.118v.198l.098.099.099.335.455.455v.197l.375.376a1.266 1.266 0 00.83.512 1.248 1.248 0 00.948-.236c.206-.166.352-.395.415-.652a.977.977 0 00.118-.592zM315.246 423.231a1.292 1.292 0 00.119 1.383c.098.137.218.257.355.355l.139.139.217-.02a1.367 1.367 0 00.869-.869 1.543 1.543 0 00-.237-1.304 1.285 1.285 0 00-1.679-.336l-.198.198.415.454z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M308.747 449.958a1.314 1.314 0 001.244-1.877l-.81-.81-.256-.02-.356.079-.83.83-.039.276.039.514 1.008 1.008zM315.602 477.988l-.533-.534a1.4 1.4 0 00-1.363-.138 1.286 1.286 0 00-.771 1.403l1.265 1.264.315.04.099.098.395-.118.139-.138a1.286 1.286 0 00.454-1.877zM300.668 470.047l.455-.454a1.379 1.379 0 00-.218-1.205l-.691-.692h-.237l-.948.948-.099.336v.237l.079.395.336.336a1.31 1.31 0 00.908.514 1.17 1.17 0 00.415-.415zM331.128 444.407l-.138.138-.06.257-.019.217.039.395.257.257a1.348 1.348 0 00.988.553 1.433 1.433 0 001.165-.454l.573-.573v-.198l-.099-.336-.118-.118a1.426 1.426 0 00-1.916-.296l-.415.414-.257-.256zM309.616 460.032l.297-.059a1.597 1.597 0 00.75-.988 1.314 1.314 0 00-.296-1.126l-.573-.573-.277-.039a1.333 1.333 0 00-1.007.454l-.415.415.02.257c.037.242.132.473.276.671l.751.751.474.237zM317.518 468.407a1.422 1.422 0 001.126-.572l.118-.119.119-.316.099-.099v-.237l-.099-.099-.099-.296-.118-.118a1.503 1.503 0 00-1.561-.494l-.849.849c-.043.378.055.758.276 1.067a1.346 1.346 0 00.988.434zM310.565 431.35l.079.395c.258.309.626.508 1.027.553.311-.001.615-.097.869-.277a.963.963 0 00.375-.099l.573-.572.079-.317.04-.236a1.489 1.489 0 00-.119-.593l-.573-.573-.335-.099-.119-.118a1.335 1.335 0 00-.632.118l-.257.257a1.451 1.451 0 00-.81.81l-.02.217-.177.534zM300.214 438.777l-.198.198c-.018.347.087.69.296.968a1.426 1.426 0 001.344.513c.257-.066.491-.2.679-.387.188-.188.322-.422.387-.68l.099-.098-.296-.297a1.398 1.398 0 00-1.047-.572 1.36 1.36 0 00-1.146.474l-.118-.119zM292.549 446.481a1.49 1.49 0 001.165-.573l.297-.296.039-.237a1.34 1.34 0 00-.277-.909 1.369 1.369 0 00-.454-.454l-.375-.376-.336.06-.099.098a1.29 1.29 0 00-.849.85 1.286 1.286 0 00.237 1.146c.128.3.359.546.652.691zM284.845 464.654l.138-.336.119-.118v-.237l-.099-.099-.118-.316-.869-.869-.257-.02h-.198l-.355.119-.474.474-.178.375-.158.158v.277l.118.118.158.395.198.198a1.392 1.392 0 00.993.429 1.403 1.403 0 001.002-.41l-.02-.138zM326.328 450.471a1.366 1.366 0 00-.316 1.896l.572.573.297.02a1.348 1.348 0 00.97-.392 1.345 1.345 0 00.393-.971l-.455-.454a1.363 1.363 0 00-1.461-.672zM291.364 441.819a1.51 1.51 0 001.502-.672 1.317 1.317 0 00-.04-1.501l-.513-.513a1.408 1.408 0 00-1.107.276c-.33.286-.536.69-.573 1.126l.119.119.099.335.671.672-.158.158zM318.189 432.851l.554.553a1.454 1.454 0 001.086.612 1.313 1.313 0 001.086-.454l.317-.316-.099-.099-.02-.217a1.532 1.532 0 00-.277-.672 1.144 1.144 0 00-.217-.217l-.494-.494-.217.02a1.48 1.48 0 00-.928.494l-.791.79zM295.156 442.175l.415.415.316.118.119.119c.217.012.434-.029.632-.119l.573-.573.098-.336a1.11 1.11 0 00-.118-.632l-.573-.573-.316-.079-.119-.118a1.254 1.254 0 00-.671.119l-.573.572-.079.316-.099.099c-.035.222 0 .45.099.652l.296.02zM297.962 444.427l.099.098.019.376.316.316a1.435 1.435 0 001.088.567 1.435 1.435 0 001.125-.488l.395-.395-.02-.257a1.598 1.598 0 00-.296-.573 1.347 1.347 0 00-1.462-.474l-.79.79-.474.04zM296.934 439.804l.909.909a1.48 1.48 0 001.482-.691 1.292 1.292 0 00-.04-1.502l-.514-.513-.335-.06-.336-.098-.119-.119a1.348 1.348 0 00-.632.119l-.494.494-.098.335-.119.119c-.003.216.037.431.119.632l.177.375zM303.868 433.779l.435.435.316.079.118.119c.218.012.435-.029.632-.119l.573-.573.099-.336.119-.118a1.4 1.4 0 00-.139-.613l-.572-.572-.336-.099-.119-.119a1.402 1.402 0 00-.632.119l-.593.592-.079.316-.118.119c-.012.217.028.434.118.632l.178.138zM292.371 451.637l.336-.336.02-.257-.099-.099-.099-.336-.77-.77a1.534 1.534 0 00-2.232-.296l-.593.592.02.336c.091.132.205.245.336.336.234.319.579.538.968.612a1.369 1.369 0 001.995.06l.118.158zM296.697 453.711l.02.217.138.138a1.252 1.252 0 001.087.494 1.45 1.45 0 001.284-.81 1.373 1.373 0 00-.139-1.402 1.338 1.338 0 00-.889-.494 1.504 1.504 0 00-1.066.395l-.119.119a1.205 1.205 0 00-.316 1.343zM319.869 445.395l.197.276.158.119.316.197.198.04.257.059.197-.04.356-.079.197-.079.257-.256.178-.336.079-.198.02-.257v-.197l-.099-.375-.099-.178-.178-.257-.158-.118-.316-.198-.217-.059-.237-.04-.217.02-.356.079-.178.099-.138.098-.119.159-.197.316-.079.197-.02.257v.197l.099.376.099.178zM298.811 475.815l.119-.119.059-.296a1.272 1.272 0 00-.099-.612l-.592-.593-.376-.099-.138-.138a1.394 1.394 0 00-.988.474l-.454.455v.237l1.126 1.126.276.039.099.099.356-.119.375-.375.237-.079zM285.833 468.566l-.672-.672-.217.02-.336.098-.118.119a1.4 1.4 0 00-.707.893 1.406 1.406 0 00.193 1.122l.632.632.336-.02.138-.138a1.415 1.415 0 00.948-.948 1.482 1.482 0 00-.197-1.106zM285.062 493.08a1.367 1.367 0 00-1.047-.534 1.283 1.283 0 00-1.086.494l-.139.138-.098.336-.119.119v.237l.099.099c.032.222.12.433.257.612.223.27.541.447.889.494l.256.02.119-.119a1.421 1.421 0 00.731-.928c.164-.295.213-.64.138-.968zM321.843 456.496l-.572-.573h-.316l-.099.099a1.039 1.039 0 00-.277.276l-.75.751v.237l.059.415.79.79.296.02a1.397 1.397 0 00.968-.968 1.45 1.45 0 00-.099-1.047zM302.327 480.951l-.553-.553h-.237l-.297.099-1.106 1.106-.039.276a1.33 1.33 0 00.118.632l.711.712.316.039.119-.119.375-.098.139-.138a1.311 1.311 0 00.454-1.956z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M302.525 444.96l.257.257.316-.04.118-.118.138.138.218-.02.375-.099.849-.849.02-.296-.099-.099-.098-.336-.139-.138a1.316 1.316 0 00-.968-.533l-.296.019-.197-.197-.277-.04-.474.04-.178.178-.276.039-.889.889-.04.276.04.475.197.197a1.42 1.42 0 001.185.553l.218-.296zM298.555 460.269v-.198l-.099-.335-.138-.139a1.513 1.513 0 00-.968-.573 1.535 1.535 0 00-1.106.396l-.613.612.099.099.079.355.138.139a1.343 1.343 0 00.988.553 1.426 1.426 0 001.086-.376l.534-.533zM347.287 443.34l.217-.02.158-.158a1.29 1.29 0 00.593-.908 1.336 1.336 0 00-.277-1.067l-.652-.652h-.237l-.099.099-.276.079-.613.612-.098.297-.06.256.079.514.494.494c.196.237.469.398.771.454zM250.355 414.164l.119.119.217.019.099-.098a1.318 1.318 0 001.086-.731 1.31 1.31 0 00.099-.731l-.079-.395-.316-.316a1.683 1.683 0 00-1.185-.672l-.257.02-.277.276a1.318 1.318 0 00-.632.633l-.158.158-.019.256-.099.099.691.691.711.672zM232.221 430.915l.316-.316v-.237l-.414-.415a1.4 1.4 0 00-1.581-.474l-.73.731a1.364 1.364 0 00.296 1.047 1.312 1.312 0 00.987.513 1.419 1.419 0 001.126-.572v-.277zM282.633 425.7l.928.929.277.039.513-.079.178-.178a3.097 3.097 0 011.067-.355l.849-.85.04-.276-.06-.455-.158-.158a1.37 1.37 0 00-1.086-.533l-.138-.138-.455.02-.099-.099-.553-.079-.316.079-.256.257a1.316 1.316 0 00-.731 1.876zM286.525 422.243h.355l.099-.098a1.568 1.568 0 001.087-1.363l-.06-.297-1.442-1.442h-.237l-.434.079-.554.553-.197.395-.039.237-.001.277.316.316.257.494.85.849zM276.963 428.841l.139-.138.039-.277v-.237l-.118-.434-.158-.159a1.322 1.322 0 00-1.047-.454 1.295 1.295 0 00-.968.455l-.139.138-.118.355-.099.099v.237l.553.553a1.281 1.281 0 001.66.356l.158-.158.098-.336zM261.655 442.372l-.277-.276-.375-.059-.139-.139a1.097 1.097 0 00-.948.435 1.31 1.31 0 00-.235.476 1.303 1.303 0 00-.022.531l-.118.119.118.395a1.171 1.171 0 00.85.494c.33.032.661-.059.928-.257a1.425 1.425 0 00.218-1.719zM256.558 401.956l-.296-.296a1.257 1.257 0 00-.948.158 1.398 1.398 0 00.296 2.509l.375-.059.593-.356.435-.435-.079-.395-.158-.158-.218-.968zM240.182 451.992l-.395.04-.612.375-.277.277c-.047.16-.067.327-.059.493l.059.415.356.593.276.276a1.37 1.37 0 001.008-.138 1.278 1.278 0 00.592-.83 1.362 1.362 0 00-.948-1.501zM250.04 392.04l.02.375.237.237a1.359 1.359 0 00.928.929c.129.034.262.048.395.039l.297-.098.73-.731.06-.297-.04-.355-.928-.928a1.397 1.397 0 00-1.679.809l-.02.02zM254.504 418.47l.04.316.237.514.217.217a.945.945 0 00.415.178 1.258 1.258 0 001.067-.198 1.512 1.512 0 00.632-1.106l-.06-.296-.553-.553a1.048 1.048 0 00-.375-.138 1.279 1.279 0 00-1.354.503 1.29 1.29 0 00-.206.464l-.06.099zM261.121 392.534l.712.711.375.138a1.288 1.288 0 001.165-.256 1.395 1.395 0 00.435-1.265l-1.007-1.007a1.512 1.512 0 00-1.502.514l-.178.177-.118.435v.237l.118.316zM252.232 430.935l-.178-.178a1.253 1.253 0 00-.77-.77 1.926 1.926 0 00-1.324.256l-.829.83.059.336.119.118a1.223 1.223 0 00.77.771c.133.311.38.558.691.691l.139.138.414.02.889-.889a1.564 1.564 0 00-.177-1.481l.197.158zM241.328 441.483l-.612-.612a1.403 1.403 0 00-1.442.533 1.513 1.513 0 000 1.739l.593.592a1.342 1.342 0 001.442-.533 1.453 1.453 0 00.019-1.719zM280.282 379.931l.119.119.375-.02a1.24 1.24 0 00.568-.173 1.24 1.24 0 00.42-.42l.217-.217a1.452 1.452 0 00-.336-.81l-.632-.632h-.197l-.336.059-.138.139a1.62 1.62 0 00-.731 1.442c.156.245.394.427.671.513zM234.829 417.838l-.098-.335-.751-.751a1.454 1.454 0 00-1.56.612l-.159.158-.039.435.02.217.118.356.178.177c.104.181.254.331.435.435l.138.138h.395l.237-.039.336-.138.158-.158a1.386 1.386 0 00.494-.968l.098-.139zM238.642 403.557a1.289 1.289 0 00-1.106.197 1.333 1.333 0 00-.573 1.087l.099.098.079.316.77.771.217.02.376-.06.849-.849.039-.237.02-.217-.079-.396-.711-.711.02-.019zM255.729 424.159l-.218-.019h-.355l-.316.316a1.255 1.255 0 00-.573.572l-.138.139-.04.355.02.218.573.573c.179.181.404.31.651.375l.317.039.316-.079.474-.474a1.423 1.423 0 00.019-1.165 1.406 1.406 0 00-.73-.85zM274.277 358.163l.178.415.158.158c.194.219.448.377.731.454a1.9 1.9 0 00.494.02l.237-.237a1.345 1.345 0 00.632-1.462l-.652-.652-.316-.118a1.476 1.476 0 00-1.561.651l-.138.139-.119.553.356.079zM234.79 410.846l.158-.158.099-.455v-.237l-.059-.296-.81-.81a1.398 1.398 0 00-1.679.81l-.119.118.139.139a1.337 1.337 0 00.73.731 1.373 1.373 0 001.541.158zM242.376 380.583c.203-.195.335-.452.375-.731v-.197l-.099-.376-.83-.829a1.589 1.589 0 00-1.461.474l-.158.158-.099.454-.02.257.119.395.118.119a1.4 1.4 0 00.553.553l.158.158h.474l.257-.06.494-.493.119.118zM235.244 413.473l-.395-.04-.356.356a1.318 1.318 0 00-.711 1.145 1.565 1.565 0 00.691 1.324l.356.356.257-.06a1.477 1.477 0 00.731-1.481 1.316 1.316 0 00-.573-1.6zM233.802 375.842l.02.336.81.81a1.394 1.394 0 001.738-1.146l-.039-.316-.672-.671-.375-.139a1.339 1.339 0 00-1.521.771l-.04.237.079.118zM243.561 392.672l-.929.929.06.296.098.099a1.432 1.432 0 00.889.889 1.399 1.399 0 001.679-1.008l.02-.217-.039-.435-.573-.573a1.513 1.513 0 00-.672-.355h-.237l-.375.059.079.316zM244.054 410.905c.137-.099.257-.219.356-.356.19-.151.362-.323.513-.514.251 0 .496-.069.711-.197l.04.277.671.671c.128.081.268.141.415.178a1.284 1.284 0 001.442-.652l.139-.138.059-.376-1.225-1.224h-.197l-.356.039-.138.139-.277-.277a1.294 1.294 0 00-.344-.624 1.294 1.294 0 00-.624-.344l-.375-.375-.119.118a.874.874 0 00-.257.257l-.948.948a2.083 2.083 0 00-.493.494l-.415.415a1.127 1.127 0 00-.397.255 1.127 1.127 0 00-.255.397l-.059.296-.099.099v.237l.079.434.355.356-.513-.514.434.435c.207-.017.412-.05.613-.099a1.28 1.28 0 001.264-.355zM198.956 415.586l-.395.04-.434.434a1.12 1.12 0 00-.435.909 1.227 1.227 0 00.415.968l.415.415a1.242 1.242 0 00.96-.133 1.258 1.258 0 00.581-.776 1.627 1.627 0 00-.079-.908 1.325 1.325 0 00-.377-.623 1.33 1.33 0 00-.651-.326zM229.555 412.09l.356-.04a1.31 1.31 0 001.185-1.185l-.02-.217-.276-.277a1.8 1.8 0 00-.77-.77l-.119-.118-.138.138a1.65 1.65 0 00-.751 1.58 1.346 1.346 0 001.067 1.067l-.534-.178zM232.794 402.727l-.572-.573a1.449 1.449 0 00-1.541 2.173 1.284 1.284 0 00.849.652l.415-.06.079-.079a1.304 1.304 0 00.79-.79 1.327 1.327 0 00-.079-1.224l.059-.099zM228.943 407.171l-.81-.81a1.182 1.182 0 00-.711 0l-.316.317a1.304 1.304 0 00-.316 1.343 1.25 1.25 0 00.908.908 1.192 1.192 0 00.968-.138.77.77 0 00.257-.257 1.282 1.282 0 00.494-1.086l-.474-.277zM224.122 411.675l.277-.474.059-.296-.02-.336-.928-.929a1.397 1.397 0 00-1.738.909 1.205 1.205 0 00-.218.889 1.26 1.26 0 00.593.948l.099.099.316.079.099.099.691-.692c.204-.134.379-.309.514-.513l.256.217zM227.56 438.816a1.316 1.316 0 00-1.264.317 1.307 1.307 0 00-.356 1.264l.869.869.257-.02.316-.316a1.203 1.203 0 00.257-1.244c.078-.29.05-.599-.079-.87zM217.861 371.595l.415.02.217-.02.316-.118.119-.119a1.65 1.65 0 00.494-.494l.138-.138v-.395l-.474-.474a1.43 1.43 0 00-.454-.455l-.139-.138h-.395l-.217.02-.336.336a1.362 1.362 0 00-.474.474l-.118.118-.04.395.119.119.118.316.138.138c.123.178.277.332.455.455l.118-.04zM230.84 380.662l.296.296c.107.178.257.327.434.435l.277.079.494-.02.948-.948a1.289 1.289 0 00-.473-1.403 1.29 1.29 0 00-.475-.217l-.316-.039-.316.079-.751.75v.198l-.118.79zM218.177 391.941l-.119-.118-.118.118-.218-.019-.395.079-.731.731c-.048.171-.061.35-.039.526a1.335 1.335 0 00.988 1.133 1.342 1.342 0 001.481-.652 1.31 1.31 0 00-.354-1.537 1.333 1.333 0 00-.495-.261zM226.829 395.102l.968.968a1.422 1.422 0 001.225-.237 1.4 1.4 0 00.474-1.185l-.099-.099-.059-.376-.988-.987-.277-.198-.375.06-1.126 1.126.099.098.158.83zM259.284 396.781l.099.099a1.394 1.394 0 001.093 1.101 1.398 1.398 0 001.435-.588 1.421 1.421 0 00.06-1.639l-.711-.711h-.237c-.279.054-.535.19-.735.391-.201.2-.337.456-.391.735l-.613.612zM215.055 354.133l.06-.257-.06-.454-.316-.316a1.333 1.333 0 00-.889-.889l-.138-.138-.257.059-.513.514a1.394 1.394 0 00-.395.987l.81.81a3 3 0 00.592.237 1.337 1.337 0 001.106-.553zM307.483 399.823a1.255 1.255 0 001.343-.513l.099-.297v-.395a1.165 1.165 0 00-.178-.889 1.256 1.256 0 00-.81-.573h-.197l-.099.099a1.288 1.288 0 00-1.027 1.264 1.446 1.446 0 00.869 1.304zM222.167 460.486a1.365 1.365 0 00-1.046-1.56 1.044 1.044 0 00-.514-.04l-.395.395a1.347 1.347 0 00-.593 1.186l.059.296.534.533c.177.185.403.315.652.376l.098.098.889-.889.079-.355.237-.04zM230.029 352.592l.751.751a1.422 1.422 0 001.679-.968 1.287 1.287 0 00-.731-1.6l-.099-.099h-.316l-.158.158a1.343 1.343 0 00-.987 1.501l-.139.257zM237.061 390.598a1.335 1.335 0 00-1.244.375l-.178.178-.118.474-.119.119.119.118.079.395.908.909a1.393 1.393 0 001.561-.652l.118-.118.04-.316-.336-.336a1.399 1.399 0 00-.83-1.146zM258.02 396.465l-.573-.573a1.643 1.643 0 00-1.383.237.999.999 0 00-.197.198c-.36.162-.648.45-.81.809a1.871 1.871 0 00.119 2.015l.533.534.276.039.376-.099.138-.138a.478.478 0 00.138-.138 1.467 1.467 0 00.968-.573l.534-.533a1.404 1.404 0 00-.02-1.166l-.198-.197.099-.415zM273.764 445.711l-1.107-1.107a1.265 1.265 0 00-1.027.277 1.194 1.194 0 00-.474.948l.099.099a1.224 1.224 0 00.79 1.146l.375.019.652-.652a1.45 1.45 0 00.395-.987l.297.257zM261.773 428.979l.592-.592a1.483 1.483 0 00-.039-1.265 1.349 1.349 0 00-.909-.671l-.118-.119-.316.079-.277.277a1.283 1.283 0 00-.533 1.007l.039.277.494.494c.196.165.425.286.672.355l.296.06.296-.099-.197.197zM246.207 413.69h-.276l-.455.138-.75.751-.099.494.039.276.119.316c.11.14.236.266.375.376l.218.217.553.119.316-.04.316-.118.296-.297a1.45 1.45 0 00.593-.869v-.237l-.079-.395-.514-.514a1.48 1.48 0 00-.652-.217zM243.679 423.685l.118.119a1.275 1.275 0 001.106-.198 1.446 1.446 0 00.949-.987l-.099-.099-.079-.356-.375-.375v-.237l-1.107-1.106h-.237l-.395.079-.889.889a1.373 1.373 0 00.218 1.126c-.034.307.028.617.177.889l.435.434.178-.178zM233.467 390.48l.454-.455a1.506 1.506 0 00-1.086-1.758l-.119-.118-.375.059-1.186 1.185v.277l.119.118.138.375.158.159c.2.226.461.39.751.474.186.055.383.061.573.019l.316-.316.257-.019zM251.58 401.364l-.158-.158a1.342 1.342 0 00-1.521.652l-.316.316-.02.257.02.217.099.375.158.158c.231.269.543.456.889.534a.59.59 0 00.276.039l.296-.059.712-.711a1.71 1.71 0 00.019-1.324l-.375-.375-.079.079z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M250.771 412.011l-.297-.059h-.553l-.158.158a1.538 1.538 0 00-.513.158 1.333 1.333 0 00-.553.079l-.81.81a1.672 1.672 0 00.079 1.462l.355.355a1.39 1.39 0 00.79.79l.277.04.257-.257c.314-.12.587-.325.79-.593l.375-.375c.066-.078.139-.151.218-.217a1.514 1.514 0 00.219-1.874 1.516 1.516 0 00-.476-.477zM251.067 406.539l-.296.06-.711.711a1.391 1.391 0 00.275 1.585c.136.135.297.241.475.311l.119.119.138-.139a1.418 1.418 0 00.691-1.402 1.394 1.394 0 00-1.086-1.087l.395-.158zM247.787 398.124l.929.929a1.372 1.372 0 001.679-.81l.118-.119v-.355l-.632-.632a1.387 1.387 0 00-.671-.356l-.277-.04-.336.099-.987.988.177.296zM253.832 450.945a1.372 1.372 0 00-1.106.198 1.628 1.628 0 00-.652 1.205l.019.257.297.296a1.371 1.371 0 00.908.632 1.463 1.463 0 001.186-.197l.493-.494a1.613 1.613 0 00-.059-1.284 1.419 1.419 0 00-1.086-.613zM262.109 379.062l-.652-.652a1.503 1.503 0 00-1.224.198 1.45 1.45 0 00-.573 1.007l.02.257.158.158a1.263 1.263 0 00.948 1.146c.176.026.355.033.533.019l.77-.77c.127-.179.221-.38.277-.593l-.099-.098-.039-.356-.138-.138.019-.178zM259.166 373.156l.928.928a1.784 1.784 0 001.817-.553l.415-.415.02-.217-.099-.099-.059-.296-.119-.119a1.404 1.404 0 00-.869-.632 1.6 1.6 0 00-.909.04l-1.027 1.027.099.099-.197.237zM256.183 405.058a1.526 1.526 0 00-1.185.197 1.498 1.498 0 00-.593 1.106l.079.316.869.87a1.405 1.405 0 001.185-.198 1.505 1.505 0 00.593-1.106l-.079-.316-.869-.869zM271.867 377.758c-.242.04-.466.155-.64.328a1.182 1.182 0 00-.328.64v.277l.85.849a1.341 1.341 0 001.204-.296c.216-.202.367-.463.435-.751a1.312 1.312 0 00-.948-1.62l-.573.573zM293.024 361.146l.632.632.414.177a1.369 1.369 0 001.462-.513 1.621 1.621 0 00.099-1.758l-.672-.672a1.406 1.406 0 00-1.284.296 1.54 1.54 0 00-.454 1.482l-.197.356zM250.632 417.838l.099.099.059.296.672.672.316.119a1.394 1.394 0 001.244-.297 1.475 1.475 0 00.494-1.402l-1.086-1.087-.494.059-.178.178a1.537 1.537 0 00-1.126 1.363zM305.31 408.515l-.395.039-.198.198a1.28 1.28 0 00-.494.296.857.857 0 00-.197.197v.475l-.099.098.059.415.711.711a1.201 1.201 0 00.968-.138 1.34 1.34 0 00.593-.79 1.311 1.311 0 00-.948-1.501zM285.833 385.6a1.488 1.488 0 00-1.739.83l-.158.158-.019.454.987.988.297.059.118.119.395-.079.593-.593-.198-.197.178.178.158-.475.02-.256-.06-.297-.73-.73.158-.159zM255.926 391.21l.118.119.237.039.119.119.375-.059.988-.988-.04-.356-.849-.849a1.372 1.372 0 00-1.106.198 1.362 1.362 0 00-.612.967l.098.099.079.356.435.434.158-.079zM274.218 367.151a1.252 1.252 0 00-1.343.75l-.04.277.08.513a1.221 1.221 0 00.987.988c.168.043.342.056.514.04l.928-.929a1.34 1.34 0 00-.632-1.62l-.138-.138-.356.119zM204.507 404.702l-.829-.83-.316-.079-.099-.098-.356.039-.829.83a1.505 1.505 0 00.434 1.58l.198.198.533.098h.277l.316-.118.375-.375a1.43 1.43 0 00.356-.712l-.119-.118.059-.415zM262.8 442.254l-.197-.198-.553-.079-.297.059-.829.83a1.42 1.42 0 00.039 1.185 1.254 1.254 0 00.731.731h.277l.118-.119a1.418 1.418 0 00.731-1.086 1.536 1.536 0 00-.02-1.323zM248.005 434.135l-.474-.474.039-.316a1.51 1.51 0 00-1.067-1.422l-.316.039-.731.731a1.71 1.71 0 00-.177 1.402c.09.254.247.48.454.652a1.317 1.317 0 001.008 1.008l.335.02.099-.099.296-.099.791-.79-.099-.099-.158-.553zM214.582 387.398l.316.079.099.099.355-.04.81-.81a1.416 1.416 0 00-.079-1.145 1.252 1.252 0 00-.948-.711l-.138-.139-.099.099-.356.079-.948.948.099.099.059.375.869.87-.039.197zM251.778 345.777l.237.04.474-.079.454-.454c.209-.13.384-.306.514-.514l.118-.119.02-.375-.751-.751a1.238 1.238 0 00-.434-.197 1.367 1.367 0 00-1.699.948 1.37 1.37 0 001.067 1.501zM262.8 366.953a1.288 1.288 0 00.692.968l.138.138.415.02.118-.118.356-.158.158-.158c.279-.234.456-.567.494-.929l-.02-.217-.118-.356-.81-.81a1.288 1.288 0 00-1.541.79l-.119.119.237.711zM286.959 372.188l.296.296c.11.2.275.364.474.474l.257.059.454-.019.257-.06.79-.79a1.27 1.27 0 00-.122-1.015 1.286 1.286 0 00-.826-.604.664.664 0 00-.276-.04l-.099.099-.336.099-.75.75-.001.198-.118.553zM235.224 399.902l.356-.355a1.487 1.487 0 00.316-1.344 1.284 1.284 0 00-.929-.928 1.168 1.168 0 00-1.066.237l-.474.474a1.366 1.366 0 00.098 1.086c.196.316.508.543.869.632l.83.198zM226.533 432.061l.553-.553c.159-.2.28-.428.355-.672v-.316l-.197-.553-.474-.474-.316-.079-.573.02-.099-.099a1.618 1.618 0 00-1.225 1.225l-.098.098.079.356.829.83.316.079.099.098.356-.039.395.079zM239.945 364.76l.119.356.158.158c.11.188.267.344.454.454l.138.139.415.019.138-.138.336-.138.474.474.237.04.297-.02.968-.968a1.322 1.322 0 00.158-1.225 1.286 1.286 0 00-.87-.869 1.37 1.37 0 00-1.047.138l-.355-.355a1.48 1.48 0 00-1.541.592l-.158.158-.059.415.138.77zM242.988 368.869a1.506 1.506 0 00.059 1.245c.161.372.457.668.83.829l.256.02.257-.257a1.626 1.626 0 00.612-1.165v-.198l-.098-.335-.771-.771-.415-.099-.098-.098-.356.039-.691.692.415.098zM268.509 395.655a1.437 1.437 0 00-1.087.183 1.416 1.416 0 00-.631.904c-.085.341-.036.701.138 1.007a1.288 1.288 0 00.829.593l.395-.04.593-.355.415-.415-.079-.395-.356-.593-.217-.889zM260.292 433.661l-.02-.336-1.007-1.007-.217-.02-.396.079-.928.928a1.258 1.258 0 00.909 1.502 1.232 1.232 0 001.284-.415l.256-.455.119-.276zM254.583 383.941l.237.04.138-.139a1.532 1.532 0 00.711-1.501 1.394 1.394 0 00-1.106-1.106l-.435-.04-.454.455a1.567 1.567 0 00-.296 1.323 1.583 1.583 0 001.205.968zM253.576 368.237l.987.988a1.595 1.595 0 001.837-1.166l.04-.237-.02-.415-1.185-1.185-.296.02-.573.573a3.6 3.6 0 00-.494.889l-.02.217-.276.316zM243.027 423.745l-1.126-1.126a1.426 1.426 0 00-1.561.612l-.158.435-.019.256.059.297.454.454c.182.17.408.287.652.336l.277.039.257-.256a1.66 1.66 0 00.612-1.166l.553.119zM251.876 385.462l.02-.217v-.356l-.336-.336a1.4 1.4 0 00-.553-.553l-.138-.138-.455-.02-.236.04-.83.829-.04.237-.019.218.079.395.888.889a1.295 1.295 0 001.028-.137 1.286 1.286 0 00.592-.851zM248.479 423.666a1.455 1.455 0 00-.731-.731l-.395.079-.415.415c-.208.113-.38.285-.493.493l-.87.87a1.429 1.429 0 00.079 1.185l.751.75a1.368 1.368 0 001.442-.572l.158-.158.04-.277a1.31 1.31 0 00.79-.79 1.568 1.568 0 00-.356-1.264zM229.476 369.304l-.178-.178a.616.616 0 00-.019-.375 1.152 1.152 0 00-.889-.889l-.119-.119h-.355l-.277.04-.869.869c-.085.32-.065.66.059.968a1.315 1.315 0 00.968 1.363l.217.02.139-.139a1.536 1.536 0 001.323-1.56zM253.536 357.412l.138.336.158.158c.097.174.241.318.415.415l.158.158.415.02.119-.119.335-.138.178-.178a1.364 1.364 0 00.494-.968l-.118-.118-.099-.336-.751-.751a1.402 1.402 0 00-1.56.612l-.139.139-.059.415.316.355zM186.236 399.783l-.257-.059-.297.059-.691.692a1.541 1.541 0 00.02 1.284 1.334 1.334 0 00.829.829.671.671 0 00.277.04l.316-.079.612-.612a1.508 1.508 0 00-.019-1.245 1.446 1.446 0 00-.79-.909zM260.291 439.785l-.118-.119a1.284 1.284 0 00-.751-.988l-.118-.118h-.395l-.119.118-.336.139-.178.177c-.269.23-.439.556-.474.909l.119.119.118.355.81.81a1.257 1.257 0 001.521-.731l.119-.118-.198-.553zM223.925 418.115l-.178-.178h-.513l-.297.059-.809.81a1.313 1.313 0 00.948 1.62l.316.039.296-.098.751-.751a1.402 1.402 0 00-.514-1.501zM278.366 393.186c-.155.034-.306.08-.454.138a1.256 1.256 0 00-.454 1.965c.17.2.397.342.651.406a1.208 1.208 0 001.225-.356.17.17 0 00.119-.119l.296-.296v-.276l-.138-.494-.455-.455-.335-.138-.119-.118-.336-.257zM270.938 410.332l-.414-.059-.731.731a1.501 1.501 0 00-.277 1.303 1.567 1.567 0 00.968.968l.119.119.138-.138a1.51 1.51 0 00.691-1.482 1.395 1.395 0 00-.494-1.442zM229.97 424.436l-.336.336a1.48 1.48 0 00-.414 1.007l.079.277.829.829a1.366 1.366 0 001.44-1.935 1.37 1.37 0 00-.689-.672l-.099-.099h-.355l-.455.257zM282.297 440.496l-.908-.909a1.371 1.371 0 00-1.516.975 1.366 1.366 0 00.765 1.632l.138.139.415-.06.553-.553c.196-.21.332-.47.395-.75l-.099-.099.257-.375zM223.274 400.633l1.007 1.008a1.479 1.479 0 001.462-.475l.178-.177.079-.435.019-.257-.987-.987a1.342 1.342 0 00-1.66.829l-.138.138.04.356zM204.784 393.403l-.593.356-.415.415-.079.316v.316l.711.711c.351.076.716.019 1.028-.158.308-.191.528-.497.612-.85a1.396 1.396 0 00-1.007-1.679l-.257.573zM213.614 418.036l-.218.02-.395.039-1.086 1.086.099.099.059.297.118.118c.108.225.289.406.514.514l.454.454h.277l.434-.118.534-.534.158-.355.118-.119a1.505 1.505 0 00-1.066-1.501zM243.422 384.632l-.355.356-.06.296-.019.218.059.336.928.928a1.544 1.544 0 001.186-.198 1.49 1.49 0 00.592-.987l-.099-.099-.039-.277-.534-.533a1.364 1.364 0 00-.651-.375l-.316-.04-.297.099-.336.336-.059-.06zM217.288 427.478l-.553-.553-.474-.119h-.237l-1.245 1.245c-.006.314.062.625.198.909l.355-.119-.237.237.514.514.316.039h.316l.494-.178.197-.197a1.975 1.975 0 00.356-1.659v-.119zM201.623 374.44l-.415-.415a1.117 1.117 0 00-.652-.415 1.228 1.228 0 00-1.461.83l-.02.217v.237a1.286 1.286 0 00.77 1.324l.119.118h.395l.316-.316a1.4 1.4 0 00.553-.553l.079-.277.316-.75zM219.323 402.608l-.079-.355-.83-.83a1.482 1.482 0 00-1.738 1.067l-.02.217.039.395 1.008 1.008h.237l.119-.119.335-.099.139-.138a1.648 1.648 0 00.612-1.165l.178.019zM220.627 384.237l.869.87.237.039.375-.059.85-.849-.04-.356-.336-.336a1.31 1.31 0 00-.79-.474l-.533.533-.652.376-.039.237-.099.098.158-.079zM287.236 402.213l.118.119h.395l.85-.85a1.399 1.399 0 00-.751-1.896l-.118-.119-.376.02-.869.869a1.592 1.592 0 00-.02 1.166c.169.313.442.557.771.691zM319.651 413.887a1.452 1.452 0 00-.948-.711l-.257-.059-.237.04-.573.573a1.68 1.68 0 00-.256 1.481 1.284 1.284 0 00.869.869c.311.077.64.042.928-.099l.652-.651c.089-.237.12-.492.089-.743a1.57 1.57 0 00-.267-.7zM286.465 419.774h-.276l-.316.04-.751.75a1.43 1.43 0 00.553 1.58l.415.415.336-.019.296-.099.573-.573c.124-.314.134-.661.029-.982a1.448 1.448 0 00-.602-.776l.04-.04-.297-.296zM276.174 402.411l.296.296a1.304 1.304 0 001.205-.336c.174-.185.302-.408.375-.652a1.33 1.33 0 00-.11-1.035 1.35 1.35 0 00-.818-.644l-.415.06-.198.197a.84.84 0 00-.375.375.57.57 0 00-.138.139 1.178 1.178 0 00-.198.513l.04.277.197.197.139.613zM271.67 423.27l-.178-.177a1.231 1.231 0 00-.296-.494l-.178-.178-.494-.138h-.276l-.317.079-.73.731-.079.316-.099.098.059.376.849.849a1.4 1.4 0 001.541-.592l.158-.158.04-.475v-.237zM276.371 406.045l-.592-.592a.951.951 0 00-.474 0l-.396-.395a1.301 1.301 0 00-1.148.206 1.314 1.314 0 00-.531 1.038l1.719 1.719a1.254 1.254 0 001.342-.443c.102-.133.176-.284.219-.446a1.101 1.101 0 00-.079-.751l-.06-.336zM266.376 381.511a1.253 1.253 0 00.988.988c.167.043.341.057.513.04l.454-.455a1.363 1.363 0 00.099-1.205 1.42 1.42 0 00-.948-.948 1.256 1.256 0 00-1.501.79l-.02.257.415.533zM277.872 406.401l.139-.138a1.373 1.373 0 00.533-1.205 1.319 1.319 0 00-.731-1.047l-.138-.139-.455-.019-.237.039-.77.771-.119.395.001.276.078.316.83.83.869-.079zM300.767 424.12l-.297-.059-.296.296a1.572 1.572 0 00-1.165.731 1.315 1.315 0 00.256 1.6l.178.178.455.138.197.197.395-.039 1.047-1.047a1.254 1.254 0 00-.198-1.726 1.255 1.255 0 00-.572-.269zM284.135 427.201l.098-.098-.059-.297-.217-.217a1.367 1.367 0 00-.869-.869l-.376-.02-.968.968-.118.514.039.316.139.335.77.771a1.32 1.32 0 001.047-.178 1.145 1.145 0 00.514-1.225zM292.075 388.247l.988.988a1.341 1.341 0 001.126-.178 1.5 1.5 0 00.612-1.086l-.059-.336-.79-.79a1.34 1.34 0 00-1.66.948l-.118.119-.099.335zM284.905 413.789l.217-.218a1.335 1.335 0 00-.059-1.244 1.228 1.228 0 00-.889-.691.276.276 0 00-.099-.099l-.316.079-.316.316a1.344 1.344 0 00-.533.968l.039.276.514.514c.182.17.408.287.652.336l.296.059.099-.099.296-.099.099-.098zM296.302 415.389a1.4 1.4 0 00.514-1.146l-.119-.316-.711-.711a1.367 1.367 0 00-1.521.573l-.079.276v.198l.178.178c.151.333.418.599.751.75a1.223 1.223 0 00.987.198zM267.739 390.776l.276.276a1.392 1.392 0 001.146-.197 1.46 1.46 0 00.534-.731 2.22 2.22 0 00.256-.257l.119-.118.039-.395-.493-.494a1.293 1.293 0 00-.435-.435l-.138-.138h-.395l-.218.02-.375.375c-.184.115-.339.27-.454.454l-.257.257a1.56 1.56 0 00.02 1.165l.138.139.237.079zM264.302 403.023l.217.217c.211.281.51.483.849.573a1.429 1.429 0 001.383-.434 1.512 1.512 0 00.237-1.502l-.513-.513a1.445 1.445 0 00-1.599.709 1.45 1.45 0 00-.179.594l-.395.356zM263.097 413.828a1.394 1.394 0 00.533-1.086l-.04-.237-.316-.316a1.28 1.28 0 00-.632-.632 1.536 1.536 0 00-.691.019l-.652.652a1.373 1.373 0 00-.02 1.205 1.352 1.352 0 00.83.83 1.365 1.365 0 00.988-.435zM268.391 432.831l-.079-.355-.356-.356-.375-.375-.316-.079-.099-.099-.375.059-.988.988v.237l.079.435.909.908a1.427 1.427 0 001.58-1.225l.02-.138zM292.016 395.892a1.35 1.35 0 00.111 1.035 1.326 1.326 0 00.817.644l.099.099h.277l.316-.316a1.37 1.37 0 00.316-1.896l-.612-.613a1.344 1.344 0 00-1.324 1.047zM259.264 410.905a1.508 1.508 0 00.771-1.442 1.372 1.372 0 00-.968-1.166l-.218-.019-.118.118a1.349 1.349 0 00-.672.909 1.56 1.56 0 00.257 1.244l-.197-.197.612.612.336-.059h.197zM274.732 386.588l.079.316.75.751a1.508 1.508 0 001.225-.198c.325-.251.538-.62.593-1.027l-.099-.099-.04-.276-.533-.533a1.332 1.332 0 00-.652-.336l-.296-.06-.277.079-.395.395c-.25.263-.397.606-.415.968l.06.02zM262.445 408.831l.414.02.119-.119.316-.119.138-.138c.185-.115.34-.27.455-.454l.138-.138.02-.376-.02-.217-.356-.356a1.282 1.282 0 00-.454-.454l-.138-.138h-.395l-.218.02-.335.335a1.674 1.674 0 00-.494.494l-.119.119v.395l.119.118.118.316.119.119c.098.197.257.357.454.454l.119.119zM266.039 408.831l.731.73a1.372 1.372 0 001.166-.217 1.48 1.48 0 00.612-1.086l-.02-.257-.592-.593a1.45 1.45 0 00-2.015.277l-.158.158-.119.395v.237l.04.277.355.079zM262.366 405.868l.415.019.118-.118.257-.059a1.535 1.535 0 00.731-1.482 1.315 1.315 0 00-1.047-1.047l-.119-.118-.355.039-.692.691a1.094 1.094 0 00-.454.455l-.138.138v.395l.019.217.356.356c.111.188.267.344.454.454l.455.06zM264.025 396.821l.415.019.217-.019.356-.356c.184-.115.339-.27.454-.454l.118-.119.04-.395-.119-.119.099-.414-.138-.139a1.299 1.299 0 00-.435-.434l-.138-.138h-.395l-.217.019-.336.139-.119.118a1.37 1.37 0 00-.474.474l-.118.119-.02.415.474.474c.115.184.27.339.454.454l-.118.356zM266.237 418.036l-.079-.356-.79-.79-.395-.079-.119-.118-.316.039a1.536 1.536 0 00-1.916 1.165l.02.297.731.731.434.158c.393.096.807.046 1.166-.139a1.398 1.398 0 001.6-1.205l-.336.297zM270.84 416.95l.632.632a1.256 1.256 0 001.126-.297 1.512 1.512 0 00.534-1.442 1.394 1.394 0 00-1.506-1.022c-.173.024-.34.082-.489.173-.318.235-.536.58-.613.968l.119.118c-.048.305.022.616.197.87zM283.798 395.971l.316.079h.198l.375-.059.178-.059.217-.139.178-.138.217-.296.079-.198.04-.158v-.197l-.059-.376-.04-.197-.138-.217-.158-.158-.296-.218-.198-.079-.296-.099h-.198l-.375.06-.198.079-.217.138-.158.119-.217.296-.079.197-.04.158-.02.218.06.375.059.178.138.217.139.138.316.237.177.099zM286.267 432.851l-.296-.296a1.179 1.179 0 00-.435-.435l-.138-.138-.415-.02-.217.02-.356.158-.158.158a1.364 1.364 0 00-.493.968l.098.336 1.028 1.027.474-.158.849-.85v-.197l.059-.573zM271.61 435.261h-.197l-.376.059-.829.83a1.394 1.394 0 00.829 1.896l.119.119.336-.02.869-.869a1.565 1.565 0 00.158-1.344 1.45 1.45 0 00-.909-.671zM286.169 455.034a1.407 1.407 0 00-1.146.237 1.418 1.418 0 00-.394.473c-.096.184-.15.387-.159.594l.771.77a1.31 1.31 0 001.58.158l.513-.513a1.369 1.369 0 000-1.185 1.34 1.34 0 00-1.165-.534zM292.253 403.399l-.316-.001-.573.573a.984.984 0 00-.158.395l-.019.218v.395l.138.138-.158-.158.75.751.257-.02.514-.514a1.335 1.335 0 00.257-1.323 1.512 1.512 0 00-.692-.454zM292.213 434.688l-.316-.039-.099.098-.316.079-.849.85.059.336.296.296c.111.187.267.344.455.454l.158.158.494-.02.948-.948a1.309 1.309 0 00-.83-1.264zM269.991 406.381l1.6-1.6v-.237l-.079-.395-.83-.829a1.395 1.395 0 00-1.126.217.748.748 0 00-.177.178l-.316.079-1.087 1.086-.138.454.02.257.059.296.731.731a1.333 1.333 0 001.284-.296l.059.059zM276.371 421.216l-.81-.81a1.371 1.371 0 00-1.106.158 1.492 1.492 0 00-.593.988l-.098.099.079.355.79.79a1.46 1.46 0 001.126-.177c.317-.236.535-.581.612-.968v-.435zM303.967 377.264l.336-.335a1.388 1.388 0 00-.099-1.087 1.37 1.37 0 00-.869-.671l-.297-.02-.098.099-.316.079-.633.632-.098.414v.277l.079.316.81.81a1.425 1.425 0 001.185-.514zM210.098 414.915l.514-.514a1.263 1.263 0 00.375-1.244 1.203 1.203 0 00-.356-.633l-.731-.73a1.606 1.606 0 00-1.323.217l-.652.652a1.42 1.42 0 00-.039.908l1.086 1.087c.09.027.183.041.277.039l.098.099.751.119zM206.206 439.231l-.099-.336-.711-.711a1.426 1.426 0 00-1.56.613l-.178.178.098.098.06.257a1.386 1.386 0 00.869.632 1.305 1.305 0 001.106-.237 1.369 1.369 0 00.534-1.126l-.119.632zM242.414 403.734l.277.277.494.138h.276l.316-.079.751-.751c.147-.344.347-.664.593-.948l.158-.158.118-.434-.02-.257-.869-.869a1.22 1.22 0 00-1.185.276l-1.185 1.185-.119.356-.118.118a1.251 1.251 0 00.513 1.146zM243.146 398.618l.494-.494a1.54 1.54 0 000-1.738l-.218-.217-.533-.218-.257-.02-.257-.256-.414-.02-.237.039-.83.83-.079.316v.237l.099.415.79.79.533.217h.277l.296-.019.336.138zM239.906 409.72l-1.067-1.067a1.459 1.459 0 00-1.106.277 1.36 1.36 0 00-.455.967l.119.119.119.355.671.672a1.285 1.285 0 001.541-.711l.04-.237.138-.375zM236.35 429.947l-.375.06-.415.414a1.143 1.143 0 00-.474.949 1.312 1.312 0 00.435.948l.414.415a1.23 1.23 0 001.521-.929 1.396 1.396 0 00-1.106-1.857zM221.08 375.289l-.276-.277a1.313 1.313 0 00-.988-.078 1.392 1.392 0 00-.75.75 1.291 1.291 0 00-.141 1.021 1.275 1.275 0 00.654.796l.395.04.652-.217.435-.435.019-.415-.138-.138.138-1.047zM193.445 420.11l-.395-.04-.691.218-.375.375a1.432 1.432 0 00-.158.434l-.04.396.217.691.277.277a1.253 1.253 0 001.004.098 1.251 1.251 0 00.754-.671 1.395 1.395 0 00-.593-1.778zM217.11 364.128l-.079.356-.02.217a1.367 1.367 0 00.731 1.087l.375.138.316-.04 1.047-1.047.06-.335-.909-.909a1.37 1.37 0 00-1.817.395l.296.138zM215.214 390.875l-.02.296.119.553.217.217c.089.108.189.207.296.297a1.325 1.325 0 001.087.098 1.668 1.668 0 00.909-.908l.098-.099.02-.296-.435-.435a.887.887 0 00-.276-.276 1.313 1.313 0 00-1.758.572l-.257-.019z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M227.738 367.21l.059.257.178.414.237.238c.074.096.16.182.256.256a1.255 1.255 0 001.186.04 1.422 1.422 0 00.731-1.126l-.099-.099-.079-.316-.672-.672a1.42 1.42 0 00-1.561.139l-.631.632.395.237zM210.078 402.411l-.177-.178a1.255 1.255 0 00-.771-.77 1.643 1.643 0 00-1.323-.06l-.79.79-.04.356.099.099a1.26 1.26 0 00.77.77c.049.337.217.646.474.869l.139.139.395.118.118.119.277-.04.573-.573a1.669 1.669 0 00.177-1.481l.079-.158zM197.001 410.115l-.612-.613a1.345 1.345 0 00-1.541.198 1.452 1.452 0 00-.395 1.659l.612.612a1.343 1.343 0 001.541-.197 1.452 1.452 0 00.395-1.659zM249.309 359.467l.138.138.356.079a1.339 1.339 0 001.106-.356l.375-.375a1.815 1.815 0 00-.118-.869l-.79-.79-.356-.001a1.623 1.623 0 00-1.146 1.146 1.29 1.29 0 00.435 1.028zM196.29 385.699v-.355l-.751-.751a1.397 1.397 0 00-1.659.237l-.158.158-.139.415-.019.217.019.376.158.158c.101.171.244.314.415.414l.158.158.395.119.119.118.375-.059.138-.138a1.253 1.253 0 00.771-.77l.178-.297zM203.244 372.662a1.375 1.375 0 00-1.146-.04c-.199.064-.38.175-.528.322a1.327 1.327 0 00-.322.528l.02.336.968.968.395.039.119-.118.256-.06.85-.849v-.395l-.81-.81.198.079zM215.056 396.682l-.138-.138-.356-.079-.276.039a1.455 1.455 0 00-.711.435l-.139.138-.138.336-.02.217.02.257.356.356c.137.214.319.396.533.533l.296.296.297-.019.474-.474c.23-.335.328-.743.276-1.146a1.507 1.507 0 00-.474-.751zM248.597 336.908l.099.415.158.158c.14.247.345.452.593.592.155.075.322.122.494.139l.098.098.119-.118a1.347 1.347 0 00.968-1.245l.118-.118-.039-.237-.751-.751a1.42 1.42 0 00-1.659.237l-.178.178-.237.514.217.138zM197.831 378.825l.138-.138.198-.395.158-.158v-.316l-.79-.791a1.37 1.37 0 00-1.798.415l-.237.237.02.218a1.302 1.302 0 00.75.75 1.4 1.4 0 001.561.178zM212.349 351.19c.248-.124.449-.325.573-.573l.139-.138v-.356l-.83-.83a1.509 1.509 0 00-1.541.119l-.178.178-.197.395-.059.257v.395l.118.118c.081.249.223.474.415.652l.178.178.454.099.257.019.276-.079.297-.296.098-.138zM197.633 381.393l-.375-.138-.198-.198-.099.099a1.398 1.398 0 00-.968.968 1.54 1.54 0 00.376 1.442l.493.494h.277a1.537 1.537 0 001.067-1.265 1.367 1.367 0 00-.573-1.402zM205.12 344.552l-.059.336.039.316.238.514.276.276a1.414 1.414 0 001.126.099 1.471 1.471 0 00.81-.81l.099-.098.039-.316-.81-.81a1.311 1.311 0 00-1.659.395l-.158.158.059-.06zM210.73 363.161l-.968.967-.039.316a1.46 1.46 0 00.77 1.008 1.397 1.397 0 001.778-.119l.138-.138.059-.454-.454-.455a1.225 1.225 0 00-.533-.533l-.139-.138-.375-.02-.237-.434zM206.818 381.057c.153-.094.282-.222.376-.375.227-.086.44-.206.632-.356.235.061.484.047.711-.039l-.04.276.672.672c.09.132.204.246.336.336a1.23 1.23 0 001.541-.316l.138-.139.138-.335.02-.218.02-.256.099-.099-.968-.968-.396-.04-.098.099.019-.257v-.197a1.4 1.4 0 00-.73-1.166l-.988-.987h-.198a.979.979 0 00-.375.098l-.375.139c-.231.076-.44.204-.613.375l-1.126 1.126a.252.252 0 00-.098.099 1.056 1.056 0 00-.81.454l-.198.198-.099.098-.138.139-.02.454.869.869c.195.045.395.058.593.04l.099.098a1.293 1.293 0 001.007.178zM161.899 374.973l-.415-.02-.355.119-.139.138a1.057 1.057 0 00-.711.711 1.254 1.254 0 00.198 1.027l.336.336c.147.076.309.122.474.136a1.27 1.27 0 00.921-.3c.126-.108.229-.24.303-.389a1.61 1.61 0 00.158-.869 1.482 1.482 0 00-.77-.889zM192.478 378.805l.355.04a1.26 1.26 0 001.422-.83l.099-.099-.237-.237a1.478 1.478 0 00-.592-.908l-.238-.237-.217.019a1.626 1.626 0 00-1.106 1.344 1.368 1.368 0 00.711 1.343l-.197-.435zM197.732 370.548l-.573-.573a1.38 1.38 0 00-1.085-.095 1.385 1.385 0 00-.811.728 1.449 1.449 0 00-.118 1.106c.149.345.425.621.77.77l.415.02.138-.138a1.338 1.338 0 00.988-.553 1.294 1.294 0 00.217-1.205l.059-.06zM193.07 373.788v-.277l-.671-.671a1.283 1.283 0 00-.692-.178h-.197l-.277.079a1.452 1.452 0 00-.612 1.244 1.308 1.308 0 00.691 1.087c.297.158.645.194.968.099.122-.028.241-.068.356-.119a1.311 1.311 0 00.75-.948l-.316-.316zM187.282 377.126l.593-.592.059-.336-.059-.297-.237-.513-.356-.356a1.4 1.4 0 00-1.896.474 1.325 1.325 0 00-.425.926 1.323 1.323 0 00.366.951l.553.553.296-.02c.201-.153.38-.332.533-.533l.573-.257zM184.161 404.386a1.284 1.284 0 00-1.303 0 1.31 1.31 0 00-.632 1.146l.059.257.948.948.276.039h.198l.158-.158a1.287 1.287 0 00.553-1.185 1.341 1.341 0 00-.257-1.047zM190.601 336.631l.375.139.119.118.336-.059.118-.119a1.49 1.49 0 00.474-.474l.139-.138.118-.395.02-.217-.04-.316-.118-.119a1.207 1.207 0 00-.454-.454l-.159-.158-.375-.099-.217-.02-.316.04-.119.118a1.315 1.315 0 00-.454.454l-.158.159-.099.375-.02.217.06.336.118.118c.112.175.26.324.435.435l.217.059zM201.09 348.543l.277.276c.124.177.278.331.454.455l.178.178.474.078.296-.019.297-.099.572-.573a1.283 1.283 0 00-.553-1.778l-.296-.296-.336.02-.968.968-.395.79zM186.136 356.543l-.553-.553-.375-.02-.632.237-.593.593a1.312 1.312 0 001.253 2.079c.344-.052.654-.238.861-.519a1.344 1.344 0 00.039-1.817zM193.801 361.64l.099.098.059.257.691.691a1.368 1.368 0 001.265.04 1.457 1.457 0 00.75-1.027l.138-.139.02-.375-.118-.118-.06-.257-.829-.83-.376-.02h-.197l-1.146 1.146-.296.534zM224.972 370.924a1.373 1.373 0 00.731 1.284 1.459 1.459 0 001.6-.218 1.483 1.483 0 00.435-1.58l-.692-.691a1.4 1.4 0 00-1.363.809l-.711.396zM192.004 319.07l.177-.177.06-.455-.02-.256-.138-.139a1.395 1.395 0 00-.711-1.027l-.178-.178h-.277l-.375.376c-.312.2-.538.51-.632.869v.276l.533.534c.12.168.267.314.435.434a1.402 1.402 0 001.126-.257zM271.077 385.186a1.233 1.233 0 001.422-.158l.237-.238.059-.375c.142-.281.17-.607.079-.908a1.247 1.247 0 00-.711-.712l-.316-.316a1.315 1.315 0 00-1.304.988 1.453 1.453 0 00.534 1.719zM173.91 424.08a1.371 1.371 0 00-.652-1.758 1.18 1.18 0 00-.494-.177l-.119-.119-.237.237a1.375 1.375 0 00-.948.948v.277l.04.276.336.336c.125.224.31.408.533.534l.217.217.119-.119.395-.118.632-.632.178.098zM206.898 321.085l.02.218.711.711a1.421 1.421 0 001.837-.534 1.286 1.286 0 00-.316-1.738l-.415-.415h-.237a1.369 1.369 0 00-1.284 1.284l-.316.474zM204.804 359.684a1.29 1.29 0 00-1.284.059l-.198.198-.217.414a.397.397 0 00-.119.119l-.138.138-.02.375.04.238.829.829a1.287 1.287 0 001.66-.276l.118-.119.119-.316.118-.119-.118-.118a1.288 1.288 0 00-.79-1.422zM223.807 370.311l-.474-.474a1.56 1.56 0 00-1.403-.099 1.147 1.147 0 00-.276.079 1.71 1.71 0 00-1.028.514 1.957 1.957 0 00-.375 1.995l.711.711.375.02.277-.277a1.391 1.391 0 001.067-.316l.434-.434c.227-.336.313-.748.237-1.146l-.039-.277.494-.296zM227.521 421.868v-.316l-.909-.909a1.26 1.26 0 00-1.067 0 1.285 1.285 0 00-.75.751 1.289 1.289 0 00.493 1.284l.139.138.336.099.098.099.257-.02.356-.356c.306-.193.526-.497.612-.849l.435.079zM219.817 402.786l.573-.573a1.5 1.5 0 00.256-1.244 1.394 1.394 0 00-.79-.79l-.237-.237-.316-.001-.158.159a1.304 1.304 0 00-.79.79v.276l.474.474c.118.217.297.396.514.514l.197.197h.316l-.039.435zM208.261 384.277l-.257-.059h-.474l-.257.059-.711.711-.198.474-.039.277.039.316c.026.178.079.352.159.514l.158.158.513.237.296.059.336-.059.218-.218c.339-.141.609-.411.75-.75l.138-.139.02-.414-.474-.475a1.489 1.489 0 00-.217-.691zM203.46 393.383l.257.257a1.452 1.452 0 001.146.079 1.61 1.61 0 00.889-.889l.098-.098v-.356l-.276-.277.138-.138v-.237l-.039-.276-.929-.929-.415-.02-.237.04-.731.731a1.35 1.35 0 00-.059 1.165 1.591 1.591 0 00-.039.909l.355.355-.158-.316zM201.367 358.696l.474-.474a1.455 1.455 0 00-.632-1.936l-.138-.138-.395-.04-.218.02-.257.059-.948.949-.02.217.04.395.158.158c.119.275.338.494.612.612.176.101.371.162.573.178l.297-.059.118-.119.336.178zM216.479 373.492l-.158-.159a1.424 1.424 0 00-1.66.277l-.632.632.02.375.118.119c.166.323.428.585.751.751l.178.177h.316l.593-.592a1.79 1.79 0 00.335-1.284l-.434-.435.573.139zM213.1 383.704l-.198-.197-.553-.119-.138.138a5.055 5.055 0 01-1.086-.019l-.672.671a1.845 1.845 0 00-.257 1.442l.336.336c.031.189.1.371.202.533.101.163.234.304.391.415l.177.178.218.02.138-.139a1.89 1.89 0 00.928-.375l.593-.593a1.509 1.509 0 00-.079-2.291zM214.661 378.469l-.099-.098h-.316l-.494.493a1.407 1.407 0 00-.277 1.061 1.406 1.406 0 00.574.935l.237.237.217-.02a1.396 1.396 0 001.106-1.106 1.37 1.37 0 00-.731-1.363l-.217-.139zM213.476 369.501l.908.909a1.37 1.37 0 001.818-.435l.118-.118.099-.336-.198-.237-.336-.336a1.393 1.393 0 00-.533-.533l-.276-.277-.336.02-1.008 1.007-.256.336zM206.937 422.263a1.386 1.386 0 00-1.126-.059 1.59 1.59 0 00-.968.968l-.138.138.098.099.04.276c.155.354.437.636.79.791a1.565 1.565 0 001.205.059l.494-.494a1.53 1.53 0 00.237-1.264 1.425 1.425 0 00-.632-.514zM231.867 354.35l-.652-.652a1.367 1.367 0 00-1.245-.098 1.42 1.42 0 00-.829.829l-.02.257.118.118a1.26 1.26 0 00.672 1.344.81.81 0 00.494.138l.355-.079.494-.494c.191-.121.353-.283.474-.474l.119-.118.039-.356-.098-.099.079-.316zM230.405 347.911l.908.908a1.812 1.812 0 001.897-.118l.474-.474v-.316a1.311 1.311 0 00-.751-.751 1.704 1.704 0 00-.909-.158l-.098-.099-1.028 1.027-.493-.019zM219.955 378.232a1.323 1.323 0 00-1.185-.079 1.534 1.534 0 00-.889.889v.316l.869.87a1.395 1.395 0 001.185.078 1.532 1.532 0 00.889-.888v-.316l-.869-.87zM241.763 355.318l-.119.119a1.142 1.142 0 00-1.106.75l-.079.277v.237l.692.691a1.417 1.417 0 001.264 0c.261-.122.47-.331.592-.592a1.312 1.312 0 00-.533-1.798l-.711.316zM266.158 344.157l.692.692a1.51 1.51 0 001.54-.158 1.615 1.615 0 00.494-1.719l-.671-.671a1.365 1.365 0 00-1.344 0 1.563 1.563 0 00-.77 1.323l.059.533zM211.54 389.215v.316l.77.771a1.398 1.398 0 001.284.019 1.56 1.56 0 00.83-1.264l-.04-.276-.178-.455-.592-.592-.494-.06-.257.02a1.51 1.51 0 00-1.323 1.521zM266.949 393.107l-.415-.02-.277.079a1.033 1.033 0 00-.513.198.752.752 0 00-.178.178l-.118.434-.119.118-.04.396.218.217.099.296.276.277c.306.154.661.183.988.079a1.255 1.255 0 00.711-.711 1.34 1.34 0 00-.632-1.541zM253.378 366.262a1.506 1.506 0 00-1.877.414l-.158.158-.158.435-.02.217.04.316.988.988.414.02.218-.02.592-.592-.197-.198.178.178.493-.494.02-.336-.79-.79.257-.296zM222.997 364.681l.435.435.355.039.218-.019.987-.988.04-.356-.751-.75a1.414 1.414 0 00-1.126-.099 1.471 1.471 0 00-.81.81l-.118.118-.02.376.652.652.138-.218zM246.425 345.599a1.234 1.234 0 00-1.462.435l-.178.178-.059.494a1.398 1.398 0 001.284 1.284l.316-.08.593-.592a1.343 1.343 0 00-.237-1.699l-.178-.178-.079.158zM169.801 365.649l-1.126-1.125-.356-.04-.217.02-.672.671a1.456 1.456 0 00.04 1.62l.237.237.494.217.257.06.355-.04.375-.375c.243-.132.442-.331.573-.573l.119-.118-.079-.554zM217.683 415.942l-.217-.217-.514-.198-.296-.02-.316.079-.593.593a1.4 1.4 0 00-.237 1.146 1.243 1.243 0 00.731.731l.158.158c.217-.071.414-.192.575-.354.162-.161.283-.358.354-.575a1.489 1.489 0 00.355-1.343zM205.199 404.564l-.04-.237-.296-.296a.868.868 0 00.099-.297 1.512 1.512 0 00-.692-1.639l-.335-.06-.593.593a1.9 1.9 0 00-.494 1.324c.02.268.124.524.296.73l-.098.099a1.28 1.28 0 00.691 1.245l.237.237.336-.02 1.027-1.027-.138-.652zM183.826 351.229l.316.316.356.04.098.099.672-.672a1.39 1.39 0 00.198-1.155 1.398 1.398 0 00-.751-.899l-.257-.257-.336-.02-1.086 1.086-.02.376.02.217.711.711.079.158zM229.673 319.525l.178.178.454.059.257-.059.277-.079c.225-.085.428-.22.592-.395l.139-.139.098-.336.119-.118v-.237l-.474-.474a1.325 1.325 0 00-.336-.336 1.397 1.397 0 00-1.877.494 1.432 1.432 0 00.573 1.442zM235.383 342.735a1.336 1.336 0 00.454 1.087l.158.158.375.099.119.118.375-.059.158-.158c.338-.144.606-.413.751-.751l.118-.118-.039-.356-.81-.81a1.286 1.286 0 00-1.699.395l-.138.138.178.257zM257.625 353.521l.099.098.079.277c.12.191.282.353.474.474l.158.158.494.099.257-.02.296-.099.573-.573a1.304 1.304 0 00.112-1.027 1.316 1.316 0 00-.665-.79l-.297-.296-.316.039-.968.968-.296.692zM200.833 368.277l.099.098.158-.158a1.366 1.366 0 00.416-1.849 1.377 1.377 0 00-.495-.482 1.228 1.228 0 00-1.086-.019l-.593.592a1.235 1.235 0 00-.138 1.047 1.35 1.35 0 00.77.77l.869.001zM184.833 397.512l.593-.593c.207-.131.382-.306.513-.513l.079-.316-.079-.593-.671-.672-.593-.118a1.542 1.542 0 00-1.442.928l-.119.119-.019.375 1.145 1.146.376.02.217.217zM213.752 335.209l.02.375.138.139c.114.185.269.341.455.454l.138.138.395.119.217.019.356-.039.671.672.277.039h.237l.435-.118.158-.158a1.29 1.29 0 00.434-1.146 1.229 1.229 0 00-.711-.988 1.307 1.307 0 00-1.047-.099l-.296-.296a1.4 1.4 0 00-1.64.217l-.158.158-.158.395.079.119zM215.688 339.95a1.68 1.68 0 00-.257 1.205c.11.405.372.752.731.968l.158.158.158-.158a1.642 1.642 0 00.929-.929v-.355l-1.146-1.146-.355-.039-.218.019-.513.514.513-.237zM234.197 371.97a1.454 1.454 0 00-1.936.672 1.37 1.37 0 00-.098 1.007 1.311 1.311 0 00.731.731l.395.04.652-.217.73-.731.04-.395-.217-.692-.297-.415zM217.268 406.994l.079-.356-.059-.336-.257-.533-.533-.533-.415-.02-.889.889a1.311 1.311 0 00.533 1.718 1.288 1.288 0 001.324-.138l.375-.375-.158-.316zM223.511 357.254l.138.138h.237a1.479 1.479 0 001.027-1.264 1.37 1.37 0 00-.77-1.363 1.035 1.035 0 00-.415-.138h-.237l-.138.138a1.446 1.446 0 00-.573 1.205 1.395 1.395 0 00.731 1.284zM226.138 341.827l-.001.197.83.83c.711.395 1.462.158 2.054-.672l.158-.158.08-.434-.119-.119-.04-.277-.888-.888-.257-.06h-.237l-.395.119-.158.158a2.982 2.982 0 00-.711.711l-.139.138-.177.455zM202.808 393.285l.02-.297-.889-.889a1.428 1.428 0 00-1.659.237l-.593.593-.019.296.039.237.178.415c.118.217.296.395.513.514l.198.197h.198l.098-.098a1.669 1.669 0 00.929-.929l.987-.276zM220.41 358.143l.118-.118.079-.396v-.197l-.059-.296a1.456 1.456 0 00-.395-.672l-.158-.158-.415-.138h-.237l-.277.039-.928.929-.04.395.04.237.731.731a1.255 1.255 0 001.541-.356zM208.123 394.489a1.399 1.399 0 00-.712-.711l-.414-.019-.376.098-.217.02a1.29 1.29 0 00-.514.514l-.296.059-.652.652a1.649 1.649 0 00-.197 1.185l.869.869a1.344 1.344 0 001.501-.197l.376-.376a1.347 1.347 0 00.869-.671 1.4 1.4 0 00-.237-1.423zM202.433 337.204l.02-.296c.039-.121.059-.248.059-.376a1.258 1.258 0 00-.573-1.125l-.138-.139-.356-.079-.276-.039-.099-.099-.731.731a1.452 1.452 0 00-.178.928 1.258 1.258 0 00.613 1.521l.217.218a1.483 1.483 0 001.442-1.245zM228.646 331.298l.04.355.138.139c.098.173.241.317.415.414l.158.158.395.119.138.138.356-.079.158-.158a1.26 1.26 0 00.77-.77l-.02-.336-.75-.751a1.404 1.404 0 00-1.66.237l.139-.138-.139.415-.138.257zM153.247 356.642l-.177-.178-.297-.02-.474.474a1.43 1.43 0 00-.316 1.225 1.494 1.494 0 00.85.85l.177.177.297.02.592-.593a1.613 1.613 0 00.277-1.224 1.47 1.47 0 00-.929-.731zM215.767 412.841a1.263 1.263 0 00-.494-1.126l-.158-.158-.395-.079h-.198l-.336.059-.138.138a1.394 1.394 0 00-.75.751l-.119.118.02.376.81.81a1.285 1.285 0 001.679-.376l.118-.118-.039-.395zM185.583 383.329l-.197-.198-.494-.138-.296.02-.316.079-.593.592a1.31 1.31 0 00.132 1.469c.111.136.247.247.401.329l.297.099.098.098h.316l.731-.73a1.401 1.401 0 00-.079-1.62zM244.351 371.891a1.932 1.932 0 00-.474.04 1.175 1.175 0 00-1.028 1.027 1.258 1.258 0 001.936 1.264l.593-.592.079-.277v-.513l-.83-.83-.276-.119zM233.111 386.806l-.395-.119-.218.02-.158.158a1.427 1.427 0 00.119 2.37l.138.139.217-.02a1.425 1.425 0 001.028-1.264 1.404 1.404 0 00-.731-1.284zM189.969 390.875l-.316.316c-.313.2-.538.51-.633.869l.001.276.75.751c.156.083.326.135.502.153a1.357 1.357 0 00.984-.3c.136-.112.249-.25.331-.406a1.298 1.298 0 00.146-1.008 1.304 1.304 0 00-.62-.809l-.118-.119-.336-.059-.099-.099-.592.435zM237.041 418.826l-.099-.099-.019-.217-.672-.672a1.337 1.337 0 00-1.485.147 1.353 1.353 0 00-.332.406 1.322 1.322 0 00-.146 1.009 1.32 1.32 0 00.62.809l.138.138.435.04.217-.02.277-.079.217-.218c.251-.118.454-.321.573-.572l.118-.119.158-.553zM189.06 366.183l.119.118.079.277.612.612a1.51 1.51 0 001.541-.118l.474-.475.04-.237.02-.296-.159-.158-.217-.415-.316-.316a1.396 1.396 0 00-1.837.415l-.138.138-.218.455zM172.744 354.844l-.652.178-.632.632-.059.336.573.573a1.37 1.37 0 001.224-2.45l-.454.731zM175.648 380.781l-.119-.119-.415-.02-.217.02-.968.968-.02.336c.077.23.206.439.376.612l.494.494.256.059.474-.039.257-.06.751-.75a1.45 1.45 0 00-.869-1.501zM212.508 355.298l-.691.692-.04.355.099.099.039.237.672.672a1.516 1.516 0 001.166.099 1.435 1.435 0 00.829-.83l.02-.296-.395-.395a1.278 1.278 0 00-.514-.514l-.296-.296-.336.019-.257.257-.296-.099zM176.912 390.855l-.79-.79-.257-.06-.217.02-.099.099-.296.099-.83.829a1.73 1.73 0 000 .909l.356-.04-.238.238.257.454.435.435.296.059.514-.04.197-.197a2.225 2.225 0 00.751-1.541l-.079-.474zM174.087 335.683l-.02-.296-.217-.217a1.482 1.482 0 00-.573-.573 1.255 1.255 0 00-1.62.474l-.138.138-.02.217a1.312 1.312 0 00.415 1.482l.138.138.375.099.218.02.296-.059c.254-.073.485-.209.672-.396l.217-.217.257-.81zM184.754 367.17l-.02-.375-.79-.79a1.48 1.48 0 00-1.956.612l-.138.139-.04.395.119.118.079.277 1.027 1.027h.356l.138-.138c.41-.173.736-.499.909-.909l.316-.356zM190.325 349.629l1.126 1.126.355.04.218-.02.948-.948.039-.356-.276-.276a1.479 1.479 0 00-.652-.652l-.711.276-.692.218-.296.296-.059.296zM250.869 382.756l.119.118.395.079.099.099.256-.02.553-.553a1.443 1.443 0 00.084-1.613 1.456 1.456 0 00-.38-.421l-.138-.139-.356-.079h-.197l-.277.04-.533.533a1.45 1.45 0 00-.277 1.146 1.198 1.198 0 00.652.81zM279.611 401.739a1.404 1.404 0 00-.85-.849l-.158-.158-.256-.02-.257.257a1.565 1.565 0 00-.593 1.382 1.26 1.26 0 00.692 1.008c.278.15.601.192.908.118l.534-.533a1.494 1.494 0 00-.02-1.205zM245.911 399.685l-.158-.158-.336-.02-.257.059-.474.237-.197.198a1.397 1.397 0 00.158 1.659l.434.435.336.059.296-.02.514-.198.217-.217a1.395 1.395 0 00-.158-1.817l-.375-.217zM240.044 380.346l.297.296a1.338 1.338 0 001.778-.592 1.372 1.372 0 00-.514-1.857l-.395-.04-.277.079a1.039 1.039 0 00-.513.198.417.417 0 00-.139.138 1.283 1.283 0 00-.395.395l-.039.277.099.099.098 1.007zM230.74 399.606l-.059-.257a2.012 2.012 0 00-.158-.553l-.533-.534-.257-.059h-.316l-1.047 1.047-.04.356.02.217.731.731a1.487 1.487 0 001.64-.217l.177-.178.158-.474-.316-.079zM239.393 383.921l-.613-.612-.474-.119-.454-.454a1.443 1.443 0 00-.596-.16 1.44 1.44 0 00-.609.101 1.358 1.358 0 00-.81.81l.04.316 1.6 1.6a1.247 1.247 0 001.422-.123c.131-.107.238-.24.316-.391a2.43 2.43 0 00.079-1.185l.099.217zM235.441 357.728a1.292 1.292 0 00.692 1.205c.155.076.322.123.493.138l.238-.039.138-.138a1.323 1.323 0 00.415-1.166 1.46 1.46 0 00-.771-1.086 1.285 1.285 0 00-1.639.415l-.178.177.612.494zM240.696 384.514l.138-.138a1.311 1.311 0 00.79-1.028 1.448 1.448 0 00-.454-1.204l-.158-.159-.415-.138h-.237l-.296.059-.85.85-.079.276v.316l.968.968.593.198zM258.85 407.211l-.297-.296h-.276a1.627 1.627 0 00-1.324.454 1.312 1.312 0 00-.098 1.6l.177.178.435.237.237.237.356.039.098.099.929-.928a1.288 1.288 0 00-.237-1.62zM241.96 406.322l.119-.118v-.317l-.099-.098a1.344 1.344 0 00-.711-.988l-.336-.336-.316.04-.652.652-.217.493-.04.317.06.335.77.771a1.201 1.201 0 001.067.079 1.148 1.148 0 00.355-.83zM258.77 370.193l-.019.217.059.257.691.691a1.291 1.291 0 001.106.079 1.389 1.389 0 00.889-.889l.02-.335-.83-.83a1.394 1.394 0 00-1.837.533l-.138.138.059.139zM245.852 393.462l.336-.335a1.515 1.515 0 00.257-1.205 1.366 1.366 0 00-.81-.81l-.218-.218-.335-.019-.277.276a1.48 1.48 0 00-.77.771v.276l.474.474c.125.224.31.408.533.534l.296.098.336-.019.119-.119.059.296zM256.519 397.729a1.32 1.32 0 00.543-.326 1.32 1.32 0 00.326-.543l-.099-.099-.039-.316-.711-.711a1.367 1.367 0 00-1.62.198l-.435.434v.277l.099.099c.142.339.412.609.751.75a1.202 1.202 0 001.185.237zM234.612 367.052l.276.277a1.522 1.522 0 001.166.059c.274-.136.496-.358.632-.632.102-.083.194-.175.276-.277l.139-.138.099-.376.019-.217-.039-.355-.139-.139a1.617 1.617 0 00-.414-.415l-.158-.158-.376-.098-.118-.119-.356.04-.118.118a1.46 1.46 0 00-.455.454l-.276.277a1.427 1.427 0 00-.277 1.106l.139.139-.02.454zM228.37 378.153l-.099.099.099.099c.14.317.394.57.711.711.226.12.48.174.735.156.255-.017.5-.105.707-.255a1.474 1.474 0 00.632-1.382l-.691-.692a1.327 1.327 0 00-1.186-.079 1.526 1.526 0 00-.888.889l-.02.454zM224.716 388.445a1.443 1.443 0 00.79-1.027l.039-.237-.276-.277a1.285 1.285 0 00-1.265-.829l-.098-.099-.613.612a1.487 1.487 0 00-.296 1.166 1.373 1.373 0 00.711.948c.356.044.716-.047 1.008-.257zM225.348 408.08v-.356l-1.126-1.126-.356-.039-.217.02-.869.869-.04.237-.019.415.039.237.81.81a1.395 1.395 0 001.522-.33c.128-.132.228-.289.295-.461l-.039-.276zM256.973 377.719a1.37 1.37 0 00.533 1.876l.257.257.198-.197a1.397 1.397 0 00.77-1.798l-.632-.632a1.368 1.368 0 00-1.126.494zM221.614 384.633a1.479 1.479 0 001.165-1.166 1.336 1.336 0 00-.671-1.343l-.119-.119a1.429 1.429 0 00-.79.79 1.705 1.705 0 00-.059 1.284l.296.297.356.039-.178.218zM242.356 364.642l.019.296.85.85a1.506 1.506 0 001.758-.237c.14-.14.247-.309.316-.494l.02-.297-.474-.474a1.402 1.402 0 00-.534-.533l-.197-.198-.297.02-.375.376a1.396 1.396 0 00-.612.849l-.474-.158zM225.19 383.349l.395.118.217.02.316-.04.138-.138c.188-.11.344-.267.455-.454l.138-.138.138-.376v-.197l-.039-.356-.119-.118a1.372 1.372 0 00-.454-.455l-.139-.138-.395-.118-.118-.119-.316.039-.119.119a1.56 1.56 0 00-.493.494l-.139.138-.099.375-.019.218.039.316.119.118c.113.186.269.341.454.455l.04.237zM228.706 384.218l.711.711a1.48 1.48 0 001.185.079 1.424 1.424 0 00.889-.889v-.277l-.395-.395a1.487 1.487 0 00-.435-.434 1.34 1.34 0 00-1.56.177l-.593.593v.277l-.099.098.297.06zM225.822 380.465l.375.098.217.02.198.198.256-.02a1.485 1.485 0 001.047-1.245 1.373 1.373 0 00-.671-1.343l-.119-.118-.375-.06-.672.119-.118.118a1.496 1.496 0 00-.455.455l-.138.138-.118.395-.02.217.059.336.119.119c.106.179.255.328.434.434l-.019.139zM229.555 372.069l.395.119.118.118.336-.059.119-.119c.185-.113.341-.269.454-.454l.138-.138.139-.375-.099-.099-.04-.356-.138-.138a1.088 1.088 0 00-.415-.415l-.158-.158-.395-.118-.118-.119-.336.059-.119.119a1.25 1.25 0 00-.454.454l-.158.158-.099.375-.02.218.06.336.118.118c.116.183.272.338.454.454l.218.02zM226.671 393.127v-.356l-1.165-1.166-.336-.019a1.561 1.561 0 00-2.153.652l-.119.118-.039.316.02.217.454.455c.09.131.204.245.336.336.353.183.762.226 1.145.118a1.369 1.369 0 001.837-.77l.02.099zM231.452 393.206l.632.632a1.289 1.289 0 001.165-.02 1.422 1.422 0 00.85-1.284 1.317 1.317 0 00-.711-1.185 1.263 1.263 0 00-1.511.227 1.331 1.331 0 00-.307.484l-.098.099a1.054 1.054 0 00-.02 1.047zM248.992 375.862l.277.158.178.059.395.04.197-.04.237-.079.178-.099.316-.237.119-.158.059-.138.079-.197.02-.376-.02-.217-.079-.237-.099-.178-.257-.296-.158-.119-.276-.158-.198-.039-.355-.04-.218.02-.237.079-.177.099-.316.237-.119.158-.059.138-.059.217-.04.356.02.217.079.237.099.178.237.277.177.138zM242.731 412.307l-.277-.276a1.465 1.465 0 00-.415-.415l-.158-.158-.395-.119h-.237l-.355.04-.158.158a1.449 1.449 0 00-.771.77l.02.336 1.047 1.047.375.02.099.099.948-.949.277-.553zM227.896 411.221l-.139-.139-.355-.039-.79.79a1.418 1.418 0 00-.271 1.12 1.422 1.422 0 00.646.954l.118.119.297.059.098.099.534-.534a1.4 1.4 0 00.454-1.244 1.37 1.37 0 00-.592-1.185zM237.535 433.779a1.392 1.392 0 00-1.975.87l-.099.098.04.356.414.415c.098.197.258.357.455.454a1.32 1.32 0 001.007.099l.514-.514a1.413 1.413 0 00.276-1.145 1.366 1.366 0 00-.632-.633zM255.452 385.087l-.099-.099-.296-.059-.217.019-.415.415c-.111.087-.21.186-.296.297l-.139.138-.098.375v.237l.948.948.257.06.276-.277a1.423 1.423 0 00.553-1.264 1.449 1.449 0 00-.474-.79zM248.064 415.508l-.217-.218-.336.02-.968.968-.02.336v.197l.158.158c.116.184.271.339.455.455l.197.197.454.099.277-.04.296-.098.178-.178a1.32 1.32 0 00-.054-1.546 1.313 1.313 0 00-.42-.35zM233.15 382.815l.238-.039 1.501-1.502.039-.395-.829-.829a1.408 1.408 0 00-1.126-.06.835.835 0 00-.178.178h-.316l-.257.059-.434.238-.178.177a.746.746 0 00-.198.198l-.197.197-.218.415-.158.158-.019.297.079.276.592.593a1.364 1.364 0 001.284.019l.375.02zM235.817 398.677l-.79-.79a1.291 1.291 0 00-1.106-.079 1.366 1.366 0 00-.81.81l-.119.119-.019.335.829.83c.349.178.752.213 1.126.099a1.37 1.37 0 00.81-.81l.118-.119-.039-.395zM272.973 362.429l.336-.335a1.223 1.223 0 00.158-1.067 1.35 1.35 0 00-.77-.77l-.277-.277-.336.02-.987.987-.079.277.019.336.81.81a1.347 1.347 0 001.126.019zM172.961 376.988l.375-.375a1.345 1.345 0 00.652-1.126 1.292 1.292 0 00-.217-.692l-.731-.731a1.568 1.568 0 00-1.344-.118l-.474.474a1.704 1.704 0 00-.256.889l.019.217 1.304 1.304.672.158zM163.401 399.705l-.04-.356-.711-.711a1.425 1.425 0 00-1.64.217l-.513.514v.276l.099.099c.153.331.419.598.75.751a1.317 1.317 0 001.126.019 1.35 1.35 0 00.516-.397c.138-.172.232-.374.274-.59l.139.178zM206.918 373.689l.474.474.435.237.177.178h.316l.731-.731c.222-.302.489-.568.79-.79l.178-.178.198-.395.059-.257-.02-.296-.158-.158-.237-.435-.257-.257a1.484 1.484 0 00-1.224 0l-.139.139-.414.217-.534.138-.612.613a1.283 1.283 0 00.237 1.501zM208.774 368.83l.474-.474a1.565 1.565 0 00.415-1.68l-1.086-1.086-.435-.119-.217-.019-.296.059-.85.849-.039.237-.02.415.039.237 1.225 1.225.296.059.494.297zM203.085 378.707l-.079-.277-.632-.632a1.287 1.287 0 00-1.558.325 1.286 1.286 0 00-.259.505l-.119.118.04.356.79.79a1.318 1.318 0 001.659-.356l.158-.158v-.671zM194.868 397.749l-.396-.04-.355.119-.237.039a1.145 1.145 0 00-.731.731 1.199 1.199 0 00.198 1.027l.414.415a1.248 1.248 0 00.968.06 1.206 1.206 0 00.692-.692 1.397 1.397 0 00-.553-1.659zM185.189 353.382l.138-.375a1.318 1.318 0 00-.934-1.178 1.326 1.326 0 00-.528-.047 1.3 1.3 0 00-.947.376 1.29 1.29 0 00-.376.948l.158.355.494.494.375.139.099-.099.099.099.118-.119.356-.158.355-.356.593-.079zM132.624 352.849l-.77-.77-.395-.119a1.273 1.273 0 00-.474.079l-.771.771-.118.395c.027.346.181.67.434.908.272.222.619.328.968.297a1.4 1.4 0 001.126-1.561zM192.695 344.177l-.356.119-.296.296a1.286 1.286 0 00-.553 1.146 1.4 1.4 0 00.079.395l.711.711.316.079.099.099.296-.02.79-.79.119-.316.118-.119v-.237a1.37 1.37 0 00-1.303-1.303l-.02-.06zM168.832 356.424l-.81.81a1.935 1.935 0 00-.039.474 1.365 1.365 0 00.494.968 1.563 1.563 0 001.264.237l.849-.849c.027-.137.033-.276.02-.415a1.342 1.342 0 00-1.402-1.205l-.099-.099-.277.079zM195.559 354.903l-.493.494-.079.316-.04.395a1.26 1.26 0 00.612 1.008 1.431 1.431 0 001.324.059l.81-.81a1.514 1.514 0 00-.929-1.284l-.138-.138-.454.02-.138.138-.475-.198z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M156.329 358.064l.039-.277a1.216 1.216 0 00.395-.987 1.763 1.763 0 00-.632-1.186l-.138-.138-.336-.099-.099-.098-.257.019-.098.099-.336.138-.138.139a1.332 1.332 0 00-.395 1.027 1.448 1.448 0 00-.494.849l.118.119.119.395.889.889a1.652 1.652 0 001.363-.612v-.277zM142.935 350.795l.139-.139.039-.237.02-.256a1.315 1.315 0 00-.948-1.225 1.457 1.457 0 00-1.62.513l-.375.376-.02.296a1.369 1.369 0 00.948 1.225 1.508 1.508 0 001.817-.553zM213.357 369.343l.119.119.138.336a1.31 1.31 0 00.79.79l.099.099.118-.119c.264-.134.479-.348.613-.612l.118-.119.079-.355.099-.099-.02-.257-.494-.494a1.673 1.673 0 00-1.58-.276 1.27 1.27 0 00-.138.486c-.014.169.006.34.059.501zM163.46 337.58l.671-.672.079-.316a1.394 1.394 0 00-1.047-1.284l-.217-.02-.435.079-.592.593a1.39 1.39 0 00-.217.573l.118.118.099.415.553.553a1.281 1.281 0 001.067.198l-.079-.237zM178.195 336.75a1.315 1.315 0 00-.533-1.008 1.49 1.49 0 00-1.205-.217l-.77.77-.099.336-.099.099.85.85.355.118.119.119.237-.04.889-.889.098-.336.158.198zM163.776 359.308l-.119-.118-.118-.356-.415-.415a1.38 1.38 0 00-.711-.395l-.376.099-.691.692a1.716 1.716 0 00-.119.75l.02.297.988.987c.405.042.81-.079 1.126-.336a1.453 1.453 0 00.415-1.205zM232.34 357.116l-.454.454a1.478 1.478 0 00-.217.85c.019.171.066.337.138.493l.356.356a1.28 1.28 0 001.56.178l.83-.83a.998.998 0 00.019-.336 1.446 1.446 0 00-1.066-1.303l-.217-.02-.573.059-.376.099zM170.136 335.328l.237.039.455-.019 1.007-1.008v-.237a1.368 1.368 0 00-1.304-1.304l-.355.356a1.21 1.21 0 00-.292.453 1.213 1.213 0 00-.064.534 1.368 1.368 0 00.316 1.186zM201.288 333.412c.246.135.532.184.809.138l.85-.85.118-.355v-.198a1.582 1.582 0 00-.289-.75 1.584 1.584 0 00-.619-.514l-.139-.138-.474.039-.572.573c-.18.197-.308.434-.376.692l.02.256.138.455.672.671-.138-.019zM167.707 336.572a.73.73 0 00-.059-.375l-.395-.395a1.402 1.402 0 00-1.324-.336 1.594 1.594 0 00-1.067 1.067v.276l.396.395a1.506 1.506 0 001.639.257 1.285 1.285 0 00.81-.889zM203.223 323.811l-.316.119-.612.612-.099.296a1.38 1.38 0 00.494 1.008 1.523 1.523 0 001.126.296l.79-.79.02-.375a1.314 1.314 0 00-1.205-1.205l-.138.138-.06-.099zM190.206 338.251l-.316-.316-.336-.099-.118-.118-.218.02-.217.217a1.566 1.566 0 00-.494 1.165 1.426 1.426 0 001.383 1.383l.237-.039.79-.791a1.14 1.14 0 00.138-.73l-.414-.415-.435-.277zM172.882 344.157c.153.089.32.149.494.178l.652.336c.109.212.282.385.494.494l-.633.632-.138.336a4.62 4.62 0 00-.059.454c.038.285.169.55.373.753.203.204.468.335.753.373l.217.02.356-.079.671-.672.139-.375.019-.217-.77-.771.336-.336a1.366 1.366 0 00.612-1.244l.099-.336-1.106-1.106a1.37 1.37 0 00-.613-.336l-.336-.336a1.062 1.062 0 00-.79-.474l-.316.04-.968.968-.059.414.02.257c.057.19.144.37.257.534.058.185.16.355.296.493zM154.808 302.576l-.712-.711a1.032 1.032 0 00-.987-.158 1.19 1.19 0 00-.731.731l-.158.158-.119.395a1.288 1.288 0 001.265 1.264c.299-.022.583-.14.81-.336a1.252 1.252 0 00.632-1.343zM167.371 330.705l.256.257a1.318 1.318 0 001.482.77l.494-.493c.273-.255.448-.598.494-.968l-.455-.455a1.567 1.567 0 00-1.699-.237 1.346 1.346 0 00-.79 1.304l.218-.178zM177.267 330.962l.316-.316.139-.375a1.457 1.457 0 00-.425-.998 1.457 1.457 0 00-.998-.425 1.567 1.567 0 00-1.027.514 1.172 1.172 0 00-.335 1.007l.177.376.158.158a1.23 1.23 0 00.968.573 1.259 1.259 0 001.126-.455l-.099-.059zM171.874 328.611l.573-.572.099-.297a1.566 1.566 0 00-.217-.691l-.593-.593a1.45 1.45 0 00-1.383.119 1.37 1.37 0 00-.573 1.165 1.237 1.237 0 00.435.869l.257.257a1.427 1.427 0 001.205.178l.197-.435zM166.126 325.431l.533.099.316-.04.316-.118.593-.593.099-.296-.119-.119a1.371 1.371 0 00-1.363-1.363 1.512 1.512 0 00-.928.06 1.219 1.219 0 00-.79.79l.039.316.672.672c.206.138.433.245.672.316l-.04.276zM141.296 336.868a1.282 1.282 0 00-.674-1.113 1.289 1.289 0 00-1.301.047l-.514.513-.059.257.019.257.791.79a1.257 1.257 0 001.264-.119c.22-.155.386-.376.474-.632zM202.473 307.297l.099.376.514.513c.19.133.405.227.632.277l.217-.02.375-.099.514-.513c.145-.174.235-.387.257-.613l-.099-.098-.119-.396-.493-.493a1.344 1.344 0 00-.593-.277l-.217.02-.376.099-.533.533a1.675 1.675 0 00-.237.553l.059.138zM197.732 322.429l-.355.355c-.132.157-.215.35-.237.553v.277l.177.454.593.593.395.118.119.119a1.32 1.32 0 00.939-.385 1.304 1.304 0 00.384-.939l-.059-.296-.751-.751-.316-.079-.099-.098-.118.118-.672-.039zM183.154 313.776l-.968-.967-.376-.099a1.315 1.315 0 00-1.325 1.194 1.312 1.312 0 001.088 1.413 1.38 1.38 0 001.181-.37 1.373 1.373 0 00.4-1.171zM182.759 322.982l-.494.493-.138.336-.119.119.02.257a1.34 1.34 0 001.857 1.224l.118-.118.356-.158.454-.455.138-.375.099-.099-.02-.256-.849-.85-.356-.118-.098-.099h-.237l-.218.02-.118.118-.395-.039zM190.957 354.449l-.119.119a1.43 1.43 0 00-.711 1.303 1.475 1.475 0 00.988 1.265 1.537 1.537 0 001.6-.455l.138-.138.118-.356.02-.217-.039-.276-.237-.237a1.423 1.423 0 00-1.423-.751l-.335-.257zM218.256 299.376l.256-.02.435-.197.316-.316a1.372 1.372 0 00.514-1.146l-.06-.257-.711-.711a1.537 1.537 0 00-1.066-.118l-.929.928a3.64 3.64 0 00-.079.632 1.369 1.369 0 001.324 1.205zM202.611 401.265a1.316 1.316 0 00.889 1.166h.316l.356-.119a1.37 1.37 0 00.829-.434 1.283 1.283 0 00.297-.968l-.237-.237a1.288 1.288 0 00-1.521-.613 1.48 1.48 0 00-.929 1.205zM119.034 338.271a1.397 1.397 0 001.185-1.462 1.382 1.382 0 00-.118-.514l-.435-.434a1.283 1.283 0 00-1.304-.198l-.731.731.198.198-.296-.059a1.195 1.195 0 00-.158.711l1.007 1.007.356-.039.296.059zM224.241 313.184l-.651.652a1.452 1.452 0 001.362 1.363 1.26 1.26 0 001.245-1.245l-.079-.316-.277-.276a1.342 1.342 0 00-1.738-.474l.138.296zM190.147 331.357a1.342 1.342 0 00-.731-1.086l-.257-.06-.454.02-.889.889-.099.375a1.31 1.31 0 001.086 1.284l.099-.098.336-.06.454-.454a1.34 1.34 0 00.455-.81zM190.858 353.126l.178-.376.138-.138a1.618 1.618 0 00-.652-1.245l-.197-.197a1.762 1.762 0 00-.988-.593 1.873 1.873 0 00-1.896.711l-.139.139-.098.375.098.099.04.237.672.671c.155.365.445.655.81.81l.414.415a1.343 1.343 0 001.107-.355l.414-.415.099-.138zM148.684 383.013l.632-.632.079-.316v-.237a1.456 1.456 0 00-.572-.929 1.275 1.275 0 00-.509-.181 1.27 1.27 0 00-.538.043l-.119.118a1.232 1.232 0 00-.869 1.067l.118.119.099.335.613.613a1.39 1.39 0 001.066.079v-.079zM161.01 366.538l.455.455a1.447 1.447 0 001.165-.415 1.395 1.395 0 00.395-1.067l-.809-.81a1.313 1.313 0 00-1.107-.197l-.79.79c-.122.23-.176.49-.158.75l.04.277.335.336.474-.119zM170.868 347.041l-.869-.869-.257-.059-.297-.02-.493.059-.415.415a2.11 2.11 0 00-.395.395l-.06.297.06.573.652.651a1.332 1.332 0 001.007.297l.889-.889c.146-.258.208-.555.178-.85zM160.595 347.674l-.118.118a1.321 1.321 0 00.513.988 1.454 1.454 0 001.225.276l1.422-1.422.178-.415.039-.237-.276-.276-.178-.376-.494-.493-.395-.079a1.39 1.39 0 00-1.027.553c-.3.064-.575.215-.79.434l-.257.257.158.672zM189.198 327.92l.099.099.316.079a1.426 1.426 0 001.343-1.541l-.118-.118-.158-.356-.494-.494-.435-.118-.158-.158-.217.019-.652.652a1.287 1.287 0 00-.237.83c.006.198.053.393.138.573l.455.454.118.079zM184.26 348.227a1.344 1.344 0 00-1.165-1.166l-.277-.039-.098-.099-.257.059-.553.553a1.704 1.704 0 00-.257 1.008l.059.296 1.008 1.008a1.624 1.624 0 001.244-.376l.297-.296-.001-.948zM173.87 350.913l-.04-.316-.197-.514-.514-.513a1.66 1.66 0 00-.276-.514l-.534-.533-.316-.119-.118-.118a1.727 1.727 0 00-1.363.533l-.119.119-.118.316a1.44 1.44 0 00-.358.441c-.088.17-.141.356-.156.547l.711.711c.198.272.473.479.79.592l.277.04.178.178.316.039a1.531 1.531 0 001.837-.889zM179.163 349.53l-1.007-1.007a1.428 1.428 0 00-1.6 1.284l.553.553a1.396 1.396 0 001.561.257 1.37 1.37 0 00.77-1.324l-.277.237zM186.275 343.92l-.79.791v.237a1.37 1.37 0 001.304 1.303l.098.099.356-.118.652-.652a1.34 1.34 0 00.138-.731l-.059-.296-.751-.751-.336-.099-.217.02-.118.118-.277.079zM137.563 365.689a1.258 1.258 0 00-.514-.988 1.63 1.63 0 00-1.343-.276l-.494.494a1.4 1.4 0 00-.355 1.066 1.523 1.523 0 00.553 1.067l.454.454a1.596 1.596 0 001.205-.454c.21-.151.369-.362.458-.605.088-.244.1-.508.036-.758zM208.695 351.763l.375-.376v-.237a1.355 1.355 0 00-.52-1.088 1.376 1.376 0 00-.559-.265 1.38 1.38 0 00-.62.01l-.335.336a1.257 1.257 0 00-.79 1.264c.009.173.056.341.138.494l.77.77c.209.077.43.11.652.099l.119-.118.335-.139.316-.316.119-.434zM213.436 347.18l-.454.454-.118.316-.119.119a1.79 1.79 0 001.067 1.58l.098.099.317.039.671-.671c.25-.282.378-.652.356-1.028a1.648 1.648 0 00-.336-.849l-.553-.553-.375-.138-.119-.119-.257.02-.118.118-.06.613zM182.028 354.173l-.02-.257a1.341 1.341 0 00-.553-1.067 1.508 1.508 0 00-1.225-.237l-.908.909c.01.092.009.184 0 .276a1.445 1.445 0 00.553 1.067 1.553 1.553 0 001.224.237l.929-.928zM212.824 360.553a1.179 1.179 0 00-1.245-.573l-.869.869-.138.376a1.367 1.367 0 00.672 1.066c.251.145.539.213.829.198a1.34 1.34 0 001.343-1.343l-.592-.593zM235.126 375.822l-.573.573a1.784 1.784 0 00-.059.454 1.475 1.475 0 00.928 1.245 1.59 1.59 0 001.719-.454l.434-.435v-.316a1.372 1.372 0 00-.671-1.146 1.598 1.598 0 00-1.541.04l-.237.039zM168.339 352.414l-.988.988-.02.336a1.324 1.324 0 00.652 1.086 1.511 1.511 0 001.502.04l.711-.711.039-.316-.039-.277-.178-.454-.178-.178a1.54 1.54 0 00-1.501-.514zM193.702 401.838l-.474-.474a1.212 1.212 0 00-.395-.395l-.177-.178-.435.158-.711.711-.099.376a1.145 1.145 0 00.415.888 1.263 1.263 0 00.948.316 1.365 1.365 0 00.928-1.402zM209.643 376.336a1.532 1.532 0 00-1.363-1.363h-.237l-.454.099-.533.533-.198.395-.02.257v.316l.909.909.355.079.198-.198-.02.296h.514l.257-.059.908-.908-.316-.356zM195.243 349.53l-.099.099.02.257.829.83.376.138.256-.02 1.087-1.086a1.43 1.43 0 00-.494-1.008 1.523 1.523 0 00-1.126-.296l-.81.81-.039.276zM223.708 359.704a1.31 1.31 0 00-1.086-1.087l-.297.059-.454.218a1.196 1.196 0 00-.691 1.205c.013.176.053.349.118.513l.711.711.297.06a1.289 1.289 0 001.323-1.087l-.118-.118.197-.474zM166.897 304.512l.434-.435.119-.355v-.316l-.81-.81-.336-.138-.118-.119a1.455 1.455 0 00-1.364.889l-.059.296.059.534.771.77c.247.136.532.184.81.138l.118-.118.376-.336zM148.664 371.536l.04-.316-.099-.533-.652-.652-.395-.119-.138-.138a1.343 1.343 0 00-1.126.375 1.28 1.28 0 00-.356.988l.632.632a1.43 1.43 0 001.245.336c.344-.079.647-.283.849-.573zM151.943 354.943l.454-.455.316-.078a1.515 1.515 0 001.047-1.443l-.968-.967a1.835 1.835 0 00-1.402.256c-.229.134-.42.324-.553.553a1.316 1.316 0 00-.692 1.245l.06.336.73.731.316.079h.238l.138-.139.316-.118zM186.492 309.016l.02.336 1.126 1.125a1.192 1.192 0 001.067-.395 1.415 1.415 0 00.414-1.125l-.908-.909-.356-.119-.099-.098-.256.019-.692.692-.138.375-.178.099zM237.298 331.851l.06.257.177.415.396.395c.196.148.427.244.671.276h.198l.335-.059.81-.81c.044-.154.057-.315.04-.474a1.414 1.414 0 00-.397-.997 1.388 1.388 0 00-.986-.425 1.395 1.395 0 00-.997.397 1.402 1.402 0 00-.425.986l.118.039zM220.449 348.78a1.254 1.254 0 00-.711.948v.237l.098.375.593.593c.304.196.672.267 1.027.198l.85-.85.059-.375a1.315 1.315 0 00-1.245-1.245l-.138.139-.533-.02zM222.74 373.393l-.395.395a1.277 1.277 0 00-.257.612l.02.257.158.434.553.554.375.098.119.119a1.28 1.28 0 00.968-.375 1.28 1.28 0 00.375-.968l-.04-.277-.849-.849-.336-.099-.099-.099-.118.119-.474.079zM180.705 332.424l.652.652a1.396 1.396 0 001.382-.119 1.25 1.25 0 00.573-1.165 1.082 1.082 0 00-.533-.929l-.257-.257-.316-.079-.099-.098a1.274 1.274 0 00-.987.434c-.13.136-.228.298-.289.475a1.297 1.297 0 00-.067.552l-.059.534zM147.42 333.866l.296.296.415.178c.226.087.468.128.711.118l.296-.098.632-.632.099-.336v-.316l-.178-.573.119-.119a1.595 1.595 0 00-1.541-.75l-.79.79-.119.355v.317l.139.138.138.336-.217.296zM215.688 326.359l-.435.435c-.131.167-.22.364-.257.573v.237l.119.395.474.474c-.08.065-.152.138-.217.217l.237.237.039.237.751.751a1.288 1.288 0 001.606-.689c.083-.193.115-.405.093-.615a1.357 1.357 0 00-.455-.968l.079-.316a1.397 1.397 0 00-1.027-1.264l-.237-.039-.434.079-.336.256zM212.626 330.449a1.5 1.5 0 00-1.146.434 1.56 1.56 0 00-.434 1.106l.059.257.474.474c.387.246.855.331 1.304.237l.79-.79.098-.336.099-.098-.039-.396-1.027-1.027-.178.139zM194.887 362.963a1.453 1.453 0 00-1.56-1.324c-.347.024-.672.18-.909.435a1.306 1.306 0 00-.316.988l.158.355.494.494.375.138.099-.099.375-.177.494-.494.79-.316zM156.091 366.519l.336-.139.632-.632-.099-.415v-.237l-.908-.908-.415-.099-.217.02a1.257 1.257 0 00-1.245 1.244 1.2 1.2 0 00.79 1.067l.514.118.612-.019zM201.86 346.113l.04.237.356.356a1.45 1.45 0 002.2-.339c.147-.243.22-.523.21-.807l-.119-.435-.593-.592a1.442 1.442 0 00-1.339.154 1.447 1.447 0 00-.636 1.189l-.119.237zM216.439 340.385l-.79.79v.276a1.538 1.538 0 001.639 1.403l.237-.04.415-.138.514-.514.118-.355.02-.336-.039-.237-.771-.77a2.791 2.791 0 00-1.007-.178l-.119.118-.217-.019z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M160.339 347.061l.77-.77c.016-.111.022-.224.02-.336a1.452 1.452 0 00-1.067-1.304l-.099-.098-.454-.02-.83.83a1.647 1.647 0 00-.138.73l.059.297.553.553a1.534 1.534 0 001.284.217l-.098-.099zM199.51 343.921l.356-.119.434-.435c.178-.187.3-.419.356-.671l-.02-.257-.099-.415-.533-.533-.415-.178-.217-.02-.257.02-.889.889-.099.415a1.294 1.294 0 00.404.957 1.282 1.282 0 00.979.347zM162.116 352.276a1.223 1.223 0 00.336-.968l-.691-.691a1.06 1.06 0 00-.672-.237l-.75-.751-.277-.079-.099-.099a1.565 1.565 0 00-1.106.435l-.593.592.099.099a1.399 1.399 0 00.968 1.205h.277l.256-.059a1.26 1.26 0 001.047.415 1.374 1.374 0 001.205.138zM208.122 317.727a.842.842 0 00.198-.197.75.75 0 00.355-.119 1.198 1.198 0 00.652-1.086l-.118-.119-.099-.336-.652-.651-.375-.139-.119-.118a1.702 1.702 0 00-.928.335 1.29 1.29 0 00-.968 1.324l.434.435a1.481 1.481 0 001.62.671zM226.75 337.066l-.414.415a1.937 1.937 0 00-.238.553v.237l.119.395.573.573a1.199 1.199 0 001.067.198l.79-.791.099-.296a1.43 1.43 0 00-1.067-1.304h-.237l-.415.06-.277-.04zM166.048 285.687l-.04-.277-.988-.987a1.53 1.53 0 00-1.224.395 1.391 1.391 0 00-.415 1.126l.039.276 1.008 1.008a1.492 1.492 0 001.185-.395 1.43 1.43 0 00.435-1.146zM150.324 368.257l.118-.119a1.305 1.305 0 00.712-1.027l-.119-.118-.138-.376-.553-.553a1.257 1.257 0 00-1.028-.158l-.849.85-.059.375a1.286 1.286 0 001.205 1.205l.098-.099.613.02zM159.943 327.15v-.277l-.138-.494-.573-.573-.395-.118-.119-.119a1.324 1.324 0 00-.939.385 1.315 1.315 0 00-.384.939l.04.316.928.928a1.397 1.397 0 001.58-.987zM200.142 371.536l-.336-.336a1.281 1.281 0 00-2.034.18 1.278 1.278 0 00-.198.749 1.224 1.224 0 00.731 1.047l.118.118.237.039.119.119 1.185-1.185.02-.336-.02-.217.178-.178zM181.554 369.62l-.079-.395-.593-.593a1.477 1.477 0 00-1.323.138 1.48 1.48 0 00-.633 1.186l.02.217.415.415a1.513 1.513 0 001.64.217 1.365 1.365 0 00.553-1.185zM155.795 334.893l-.316-.316a1.65 1.65 0 00-1.086-.099l-.83.83-.019.296a1.395 1.395 0 001.343 1.344 1.371 1.371 0 001.343-1.344l-.356-.355-.079-.356zM156.329 389.67l.612-.613a.757.757 0 00.02-.296c0-.177-.034-.351-.102-.514a1.328 1.328 0 00-.727-.728 1.334 1.334 0 00-.514-.101 1.398 1.398 0 00-1.344 1.343l.929.928c.247.136.532.185.81.139l.098-.099.218-.059zM176.418 321.263l-.514.514-.138.414-.119.119a1.506 1.506 0 00.909 1.264l.237.04.454-.06 1.027-1.027.02-.217a1.395 1.395 0 00-1.323-1.323l-.237.039-.316.237zM177.682 301.431l-.494-.494-.375-.139-.138-.138-.297.02-.316.118-.829.83-.139.375c.043.362.211.697.475.949.271.221.618.327.967.296a1.4 1.4 0 001.383-1.383l-.237-.434zM156.98 317.352l-.849-.849-.375-.099-.218-.217-.493.493a1.47 1.47 0 00-.316.632l-.119.119-.04.277v.237l1.008 1.007.375.059.119-.118a1.454 1.454 0 00.908-1.541zM197.87 335.683l-.316-.079-.099-.099-.335.02-1.028 1.027.001.198a1.443 1.443 0 00.513 1.067 1.532 1.532 0 001.146.276l.75-.75c.118-.218.172-.465.159-.712l-.04-.316-.652-.652-.099.02zM149.02 323.653l.533-.533-.019-.454-.06-.257-.829-.83-.455-.099-.256-.019a1.852 1.852 0 00-.79.474l.256.256-.237-.237-.237.474-.059.297v.316l.79.79a2.111 2.111 0 001.679-.178h-.316zM194.848 292.601l.434-.435c.134-.227.203-.487.198-.75a1.258 1.258 0 00-1.205-1.205l-.257.256a1.287 1.287 0 00-1.086 1.087l.118.118.119.356.434.435c.187.192.428.323.692.375l.296-.02.257-.217zM173.357 318.083l.671-.672.079-.355.119-.119a1.51 1.51 0 00-1.462-1.462l-.138.139-.356.158-.494.494-.118.355-.04.237.099.099.059.257.514.513c.387.246.855.331 1.304.237l-.237.119zM191.214 313.777l-.475.474-.118.355-.119.119.968.968.356.118.118.119.791-.79c.179-.253.263-.561.237-.87l-1.047-1.046h-.237l-.119.118-.355.435zM194.157 382.776l.118.118.119.356.474.474.355.158.119.119a1.438 1.438 0 001.066-.317 1.44 1.44 0 00.358-.441c.088-.17.141-.356.156-.546l-.099-.099-.118-.356-.85-.849a1.47 1.47 0 00-.597.052 1.467 1.467 0 00-.529.284 1.347 1.347 0 00-.572 1.047zM192.912 417.088a1.393 1.393 0 00.415-1.126l-.02-.257-.85-.85a1.587 1.587 0 00-1.481.218 1.28 1.28 0 00-.514 1.106c.033.316.164.613.376.849l.671.672a1.516 1.516 0 001.403-.612zM177.188 387.26l-.928-.929-.277-.079a1.368 1.368 0 00-1.343.988l-.059.375v.277l1.007 1.007a1.454 1.454 0 001.482-1.086l-.139-.139.099.099.158-.513zM190.68 372.227l-.139.376a1.37 1.37 0 00.692 1.086c.228.106.48.154.731.138a1.371 1.371 0 001.363-1.363l-.158-.355-.217-.218a1.152 1.152 0 00-.376-.375l-.138-.138a1.041 1.041 0 00-.553-.04l-.514.514-.691.375zM169.425 374.242l.198-.197c.163-.111.304-.252.414-.415l.158-.158-.019-.494-.079-.277-.415-.415-.454-.217-.158-.158-.336.02-.119.118-.355.158-.455.455-.138.336-.02.217a1.484 1.484 0 001.027 1.304h.277l.454-.099.02-.178zM187.263 373.511l.256-.257a.97.97 0 00-.158-.474l.099-.375.119-.118a1.533 1.533 0 00-.573-1.047 1.359 1.359 0 00-1.146-.158l-.731.73-.099.336-.098.099.217.217a.624.624 0 00-.158.158l-.217.218-.02.256a1.258 1.258 0 001.264 1.265c.259-.014.504-.119.692-.297a.727.727 0 00.553-.553zM207.55 356.661a1.261 1.261 0 00-.652 1.205c.006.169.039.336.098.494l.652.652a1.432 1.432 0 001.227-.314 1.424 1.424 0 00.492-1.167 1.284 1.284 0 00-1.205-1.205l-.277.079-.335.256zM187.46 374.934l.316.316a1.433 1.433 0 001.304.158 1.34 1.34 0 00.79-1.027v-.238l-.138-.414-.533-.534-.376-.138-.118-.119h-.395l-.277.079-.415.415-.237.474-.158.158.237.237v.633zM177.425 402.233l-.039-.316-.336-.336a1.487 1.487 0 00-.968-.968 1.317 1.317 0 00-1.442.731l-.059.257.039.474.218.217.138.336.711.711.355.119.119.118a1.28 1.28 0 001.185-.838c.06-.161.087-.333.079-.505zM169.465 387.319l.553-.553a1.434 1.434 0 00.494-1.126 1.149 1.149 0 00-.079-.356l-.869-.869-.534.059-.612.613-.158.395-.04.237v.237a1.343 1.343 0 00.475.948c.2.224.472.371.77.415zM209.071 382.993l-.474.474-.178.375-.138.139a1.447 1.447 0 00.513 1.027c.358.244.796.343 1.225.276l.75-.75.139-.336a1.368 1.368 0 00-1.344-1.343l-.118.118-.375.02zM182.482 383.981l.297.296a1.457 1.457 0 001.165-.415c.261-.3.396-.689.375-1.086l-.73-.731a1.349 1.349 0 00-1.087-.178l-.79.79a1.707 1.707 0 00-.158.751l.059.296.593.593.276-.316zM184.398 395.339a1.434 1.434 0 001.245.138l.751-.75.059-.297a1.4 1.4 0 00-1.027-1.264l-.297-.02-.434.435a1.698 1.698 0 00-.356 1.027c-.05.244-.029.498.059.731zM199.234 360.671l-.119.395a1.309 1.309 0 00.533 1.008c.255.18.558.276.87.276a.95.95 0 00.355.119l.218-.02.375-.099.533-.533c.124-.162.205-.352.237-.553l-.118-.119-.079-.395-.494-.493a1.004 1.004 0 00-.573-.257h-.198l-.118-.119a1.4 1.4 0 00-1.087.336l-.158.158-.177.296zM186.532 361.086l-.534.534a1.48 1.48 0 00-.296.967 1.4 1.4 0 00.83 1.186 1.427 1.427 0 001.501-.198l.415-.415.019-.257a1.445 1.445 0 00-.553-1.066 1.535 1.535 0 00-1.244-.257l-.138-.494zM175.845 363.2a1.514 1.514 0 001.284.178l.494-.494c.2-.259.298-.582.276-.909a1.54 1.54 0 00-.217-.612l-.77-.77a1.334 1.334 0 00-1.146.355 1.26 1.26 0 00-.435 1.067 1.373 1.373 0 00.514 1.185zM159.351 374.005l.671-.672.099-.335-.02-.336-.849-.85-.356-.118-.098-.099h-.237l-.948.948-.08.395.02.297a1.369 1.369 0 001.64 1.126l.158-.356zM201.702 385.344a1.363 1.363 0 00-1.281.851 1.363 1.363 0 00-.101.531l.632.632a1.36 1.36 0 001.026.269 1.374 1.374 0 00.91-.545l.276-.277.02-.335a1.402 1.402 0 00-.526-.846 1.39 1.39 0 00-.956-.28zM177.485 358.676a1.474 1.474 0 001.619.277 1.29 1.29 0 00.791-1.264l-.474-.474a1.333 1.333 0 00-1.067-.395 1.536 1.536 0 00-1.106.632l-.139.138-.098.336-.099.098.02.257.335.336.218.059zM205.337 366.064l-.612.612-.099.297v.197a1.544 1.544 0 00.573 1.126 1.418 1.418 0 001.146.237l.77-.77c.113-.226.16-.479.138-.731l-.039-.277-.791-.79a1.506 1.506 0 00-1.046-.098l-.04.197zM180.388 361.106l.079.395.534.533c.178.138.389.227.612.257l.395-.118.514-.514c.132-.167.22-.364.257-.573l-.02-.217-.079-.395-.514-.514a1.088 1.088 0 00-.632-.276l-.217.019-.375.099-.494.494a1.148 1.148 0 00-.277.593l.217.217zM181.494 364.504l-.138.138-.158.356-.118.118v.277a1.34 1.34 0 00.553 1.066 1.451 1.451 0 001.224.237l.672-.671c.065-.204.092-.419.079-.632a1.369 1.369 0 00-.948-1.225l-.138-.138-.415.019-.316.316-.297.139zM183.292 360.059l.099.375.533.534a1.427 1.427 0 001.6.256 1.284 1.284 0 00.79-1.264l-.118-.118-.119-.316-.849-.85a1.118 1.118 0 00-.593-.237l-.118.119-.375.098-.514.514c-.122.178-.203.38-.237.593l-.099.296zM192.34 358.953l.118.395.514.514c.185.124.393.212.612.256l.217-.019.376-.099.454-.454c.112-.169.192-.356.237-.554v-.237l-.099-.375-.533-.533a1.671 1.671 0 00-.612-.257l-.218.02-.375.098-.533.534a1.28 1.28 0 00-.257.573l.099.138zM172.843 367.388l.671-.672.099-.336-.039-.395-.336-.336a1.506 1.506 0 00-.44-1.14 1.508 1.508 0 00-1.141-.44l-.296.099-.731.731a1.931 1.931 0 00-.039.474 1.516 1.516 0 00.494 1.047 1.317 1.317 0 00.548.942 1.31 1.31 0 001.071.204l.139-.178zM175.233 371.516l-.277.277a1.277 1.277 0 00.612 1.007 1.51 1.51 0 001.521.059 1.386 1.386 0 00.652-1.244 1.173 1.173 0 00-.474-.909 1.255 1.255 0 00-1.066-.276l-.119.118a1.193 1.193 0 00-.565.361 1.194 1.194 0 00-.284.607zM199.154 377.541l.02.296.039.198.178.336.138.177.178.178.178.099.356.119.217.019h.158l.178-.059.355-.158.158-.118.158-.198.119-.198.118-.355.02-.178-.02-.336-.039-.197-.178-.336-.118-.158-.178-.178-.198-.118-.355-.119-.218-.02-.138.02-.217.059-.316.158-.178.139-.178.177-.099.178-.118.356-.02.217zM164.724 391.072l.414-.415c.132-.166.22-.363.257-.573l-.118-.118-.079-.395-.593-.593a1.45 1.45 0 00-1.067-.197l-.77.77-.099.336.968.968.356.118.118.119.119-.119.494.099zM157.968 377.798l-1.166-1.166a1.458 1.458 0 00-1.438.818 1.459 1.459 0 00-.142.565l1.106 1.106a1.39 1.39 0 001.304-.237 1.339 1.339 0 00.336-1.086zM143.666 397.71a1.369 1.369 0 00-.573-1.008 1.26 1.26 0 00-1.185-.197l-.889.888c-.103.202-.15.427-.138.652.037.342.198.659.454.889l.277.277a1.346 1.346 0 001.119-.382 1.338 1.338 0 00.382-1.119h.553zM194.611 387.852l-.889-.889a1.117 1.117 0 00-.434-.039l-.119.118-.375.139-.435.434-.158.395-.138.139.81.81a1.366 1.366 0 001.343-.158 1.51 1.51 0 00.395-.949zM164.763 397.275l-.059-.296-.751-.751-.316-.079-.099-.099-.118.119-.336.138-.395.395c-.122.178-.203.38-.237.593v.276l.178.455.553.553.355.118.218.02a1.283 1.283 0 001.007-1.442zM185.011 367.585l1.066 1.067.356.118.217.02.217-.02.889-.889.099-.335a1.31 1.31 0 00-.533-1.008l-.909-.908-.257-.06-.355-.355-.455.019-.572.573-.237.435-.08.277a1.349 1.349 0 00.672 1.106l-.118-.04zM172.823 378.074l.415-.414.099-.336.118-.119a1.448 1.448 0 00-.513-1.027 1.393 1.393 0 00-1.107-.277l-.79.791-.118.316-.119.118a1.43 1.43 0 00.494 1.008 1.523 1.523 0 001.126.296l.118-.118.277-.238zM223.037 391.112l.296.296a1.313 1.313 0 001.402-1.402l-.039-.277-.929-.928-.434.039-.257.059-.415.415-.217.454-.039.277a1.2 1.2 0 00.632 1.067zM158.857 312.947l.414.415a1.31 1.31 0 001.285-.02c.217-.118.395-.297.513-.514l.138-.138.158-.395.119-.119v-.237a1.733 1.733 0 00-.632-1.224l-.435-.435a1.51 1.51 0 00-.869.237l-.632.632-.04.277.04.276.119.119-.178 1.126zM134.482 316.7l.651-.652.06-.296.098-.099a1.417 1.417 0 00-1.046-1.284h-.277l-.415.415a1.316 1.316 0 00-.336 1.007 1.237 1.237 0 00.553.988c.177.123.38.201.594.228.213.028.429.004.631-.07l-.513-.237zM179.223 340.424l-.059.257.02.494.612.612.454.217.277.04c.374.05.739.149 1.086.296l.138.138.435-.039.869-.869.06-.257a1.407 1.407 0 00-.613-1.047l-1.185-1.185h-.356l-.118-.119a1.383 1.383 0 00-.62.039 1.38 1.38 0 00-.541.303 1.394 1.394 0 00-.459 1.12zM184.339 339.575l.494.493a1.595 1.595 0 001.758-.414l.119-.316.02-.573-.06-.257v-.237l-.138-.415-.514-.514-.414-.177-.139-.139-.296.02-1.165 1.166-.02.572.059.257.454.455-.158.079zM172.744 339.871l.612-.612.139-.415.019-.257a1.319 1.319 0 00-1.108-1.145 1.284 1.284 0 00-.551.039l-.83.829-.079.356a1.317 1.317 0 001.205 1.205l.218-.02.375.02zM152.397 342.597l-.691-.691a1.174 1.174 0 00-1.027-.198 1.51 1.51 0 00-.751.751l-.02.019-.138.376a1.232 1.232 0 00.435.869c.25.219.576.332.908.316a1.399 1.399 0 001.284-1.442z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M148.447 320.038l-.296-.296a1.4 1.4 0 00-.988-.118 1.535 1.535 0 00-.751.75 1.338 1.338 0 00.455 1.837l.395.04.671-.198.632-.632.06-.415-.139-.138-.039-.83zM119.053 363.733l-.395-.039-.651.177-.396.396a1.082 1.082 0 00-.177.414l-.06.415.178.652.296.296c.302.165.656.207.988.119a1.257 1.257 0 00.711-.711 1.37 1.37 0 00-.494-1.719zM144.872 308.72l-.08.355-.019.218a1.258 1.258 0 00.671 1.106l.356.355.316-.039.948-.948.04-.356-.099-.099-.04-.237-.651-.652a1.371 1.371 0 00-1.838.336l.396-.039zM141.968 335.367l-.039.316.079.553.217.217c.083.126.191.233.316.317a1.254 1.254 0 001.067.118 1.654 1.654 0 00.908-.909l.099-.098.039-.277-.987-.988a1.316 1.316 0 00-1.778.514l-.099.099.178.138zM155.301 312.315l.02.256.197.435.218.217c.08.103.173.196.276.277a1.234 1.234 0 001.166.059c.208-.102.388-.256.523-.445.134-.19.219-.41.247-.641l-.059-.336-.672-.672a1.486 1.486 0 00-1.58.079l-.632.632.296.139zM136.378 346.725l-.158-.158a1.336 1.336 0 00-.652-.889 1.763 1.763 0 00-1.324-.098l-.77.77-.059.336.098.099a1.255 1.255 0 00.652.889 1.237 1.237 0 00.435.869l.158.158.395.118.217.02.257-.02.435-.434a1.702 1.702 0 00.237-1.462l.079-.198zM123.024 353.916l-.612-.613a1.31 1.31 0 00-1.541.119 1.506 1.506 0 00-.455 1.639l.613.613a1.316 1.316 0 001.541-.119 1.508 1.508 0 00.454-1.639zM177.366 305.421l.119.118.375.099a1.294 1.294 0 001.106-.316l.593-.593c.058-.291.03-.593-.079-.869l-.791-.79-.355-.039a1.58 1.58 0 00-.745.42 1.58 1.58 0 00-.42.745 1.333 1.333 0 00.197 1.225zM123.281 329.481l-.04-.356-.731-.731a1.397 1.397 0 00-1.659.158l-.158.158-.178.415-.02.217.02.376.138.138c.109.177.258.326.435.434l.158.158.395.119.119.119.355-.04.139-.138c.347-.147.623-.424.77-.771l.257-.256zM130.688 316.739a1.239 1.239 0 00-1.106-.118 1.367 1.367 0 00-.869.869l-.099.099-.02.336.988.987.375.06 1.106-1.107.06-.414-.04-.237-.671-.672.276.197zM141.593 341.195l-.119-.119-.375-.099-.119-.118-.276.039a1.42 1.42 0 00-.711.395l-.139.139-.138.335-.118.119-.001.276.356.356a1.4 1.4 0 00.553.553l.296.296.297-.019.573-.573c.251-.319.371-.722.336-1.126a1.32 1.32 0 00-.415-.454zM177.405 282.744l.079.434.139.139c.12.262.33.472.592.592l.494.178.099.099.118-.119a1.307 1.307 0 001.008-1.205l-.02-.257-.751-.75a1.45 1.45 0 00-1.679.177l-.158.158-.256.494.335.06zM125.059 322.685l.177-.177.218-.415.158-.158.02-.297-.159-.158-.177-.414-.336-.336a1.396 1.396 0 00-1.837.335l-.158.159-.02.217c.083.353.304.658.612.849a1.402 1.402 0 001.502.395zM140.624 295.623c.258-.126.467-.334.593-.593l.118-.118v-.395l-.81-.81a1.532 1.532 0 00-1.56.059l-.494.494-.039.237-.02.415c.064.269.201.515.395.711l.158.158.454.138.257.02.296-.059.178-.178.474-.079zM124.762 325.234l-.375-.139-.118-.118h-.198a1.397 1.397 0 00-.968.968 1.593 1.593 0 00.316 1.462l.296.296.257-.02a1.482 1.482 0 001.166-1.165 1.402 1.402 0 00-.376-1.284zM133.651 288.69l-.059.335.04.316.197.514.297.296c.343.191.747.241 1.126.139a1.49 1.49 0 00.809-.81l.099-.099.04-.316-.869-.869a1.287 1.287 0 00-1.68.336l-.158.157.158.001zM138.609 307.475l-.948.948-.039.316-.099.099a1.515 1.515 0 00.711 1.027 1.426 1.426 0 001.896-.513l.158-.158.06-.415-.02-.257-.316-.316a1.752 1.752 0 00-.534-.533l-.138-.139-.415-.059h-.316zM133.908 325.174c.146-.087.268-.209.355-.355.228-.092.446-.205.652-.336.241.064.495.057.731-.02l-.02.296.099.099.059.257.376.375c.089.12.196.227.316.316a1.26 1.26 0 001.56-.217l.158-.158.139-.336.118-.118.02-.257.099-.099-.968-.968-.376-.059.02-.257-.099-.099a1.416 1.416 0 00-.671-1.185l-.79-.79a2.042 2.042 0 00-.376.099l-.375.138-.138.138a1.539 1.539 0 00-.593.356l-.355.355a1.188 1.188 0 00-.83.435l-.454.454-.04.435.04.237.829.83c.193.056.392.083.593.079a1.195 1.195 0 00-.079.355zM89.305 317.431l-.376-.059-.375.098-.237.04a1.064 1.064 0 00-.731.731 1.258 1.258 0 00.158 1.027l.356.356a1.24 1.24 0 00.956.14 1.24 1.24 0 00.762-.595c.157-.268.22-.581.178-.889a1.34 1.34 0 00-.691-.849zM119.685 322.448l.376.06a1.288 1.288 0 001.462-.791l.118-.118v-.237l-.118-.119a1.43 1.43 0 00-.554-.948l-.237-.237h-.237a1.618 1.618 0 00-1.224 1.225 1.396 1.396 0 00.671 1.383l-.257-.218zM125.355 314.33l-.573-.573a1.423 1.423 0 00-1.935.553 1.509 1.509 0 00-.139 1.126 1.296 1.296 0 00.751.75l.415.06.138-.139a1.458 1.458 0 001.007-.493 1.377 1.377 0 00.257-1.205l.079-.079zM120.416 317.53l.02-.297-.77-.77a1.292 1.292 0 00-.692-.178l-.098-.099-.277.079-.099.099a1.432 1.432 0 00-.652 1.205 1.262 1.262 0 00.652 1.126 1.169 1.169 0 00.949.119.807.807 0 00.375-.099 1.305 1.305 0 00.849-.85l-.257-.335zM114.589 320.552l.593-.593.059-.336-.02-.296-.237-.514-.336-.335a1.397 1.397 0 00-1.916.375 1.316 1.316 0 00-.454.81 1.234 1.234 0 00.336 1.047l.573.573.296-.02c.228-.1.435-.24.612-.415l.494-.296zM110.5 347.595a1.37 1.37 0 00-1.323-.02 1.396 1.396 0 00-.672 1.106l1.185 1.185.178-.177a1.284 1.284 0 00.593-1.146c.128-.301.142-.638.039-.948zM119.429 280.393l.375.138.119.119.336-.02c.233-.067.44-.205.592-.395l.138-.138.139-.376.019-.217-.039-.316-.119-.118a1.362 1.362 0 00-.454-.455l-.158-.158-.355-.118-.119-.119-.356.04-.098.098a1.34 1.34 0 00-.553.356l-.158.158-.119.356-.118.118.019.336.139.138c.097.174.241.317.414.415l.356.158zM129.484 292.541l.099.099.039.277c.06.205.176.39.336.533l.178.178.474.118h.276l.297-.098.572-.573a1.344 1.344 0 00-.474-1.818l-.316-.118-.316.039-.849.85-.316.513zM114.234 299.969l-.119-.119-.098.099-.138-.138-.376-.02-.652.217-.276.277a1.346 1.346 0 00-.154 1.028 1.347 1.347 0 00.628.829 1.369 1.369 0 001.6-.218 1.302 1.302 0 00.244-1.073 1.311 1.311 0 00-.659-.882zM121.701 305.342l.928.928a1.312 1.312 0 001.264.079 1.36 1.36 0 00.79-.987l.139-.139.019-.375-.098-.099-.06-.257-.849-.849-.375-.02h-.198l-.257.059-1.007 1.008-.296.652zM152.477 315.811a1.396 1.396 0 00.691 1.324 1.453 1.453 0 002.173-1.778l-.889-.889a1.421 1.421 0 00-1.422.751l-.553.592zM121.523 262.733l.158-.158.079-.474-.02-.257-.119-.118a1.342 1.342 0 00-.671-1.067l-.178-.178-.296-.019-.376.375a1.38 1.38 0 00-.75.751l-.02.256.533.534c.124.177.278.33.455.454a1.37 1.37 0 001.205-.099zM197.989 331.851a1.282 1.282 0 001.462-.118l.217-.218.099-.375c.141-.282.17-.607.079-.909a1.168 1.168 0 00-.692-.691l-.336-.336a1.365 1.365 0 00-1.343.948 1.507 1.507 0 00.514 1.699zM99.38 367.012a1.424 1.424 0 00-.573-1.797l-.494-.178-.119-.118-.099.098-.237.04a1.364 1.364 0 00-.928.928l-.04.316.158.158.198.435c.116.219.295.398.514.513l.118.119.217-.02.376-.098.632-.633.276.237zM136.338 265.301l.02.217.711.712a1.421 1.421 0 001.876-.455 1.28 1.28 0 00.206-.924 1.28 1.28 0 00-.482-.814l-.415-.415-.257-.02a1.373 1.373 0 00-1.363 1.166l-.296.533zM132.763 303.801a1.247 1.247 0 00-.645-.166 1.244 1.244 0 00-.639.186l-.514.513-.138.139-.119.118-.039.395.039.237.731.731a1.313 1.313 0 001.817.119l.119-.316.118-.119v-.237a1.42 1.42 0 00-.089-.926 1.417 1.417 0 00-.641-.674zM151.331 315.179l-.573-.573a1.82 1.82 0 00-1.402-.178l-.296.099a1.597 1.597 0 00-1.028.474 1.932 1.932 0 00-.434 1.976l.849.849.376.02.256-.257a1.425 1.425 0 001.087-.296l.316-.316a1.452 1.452 0 00.296-1.126l-.02-.297.573-.375zM153.089 366.834l-.02-.335-.889-.889a1.161 1.161 0 00-.541-.143 1.158 1.158 0 00-.545.123 1.221 1.221 0 00-.731.731 1.316 1.316 0 00.415 1.323l.138.139.336.098.118.119.257-.02.336-.336c.319-.187.553-.492.652-.849l.474.039zM146.116 347.456l.573-.573a1.475 1.475 0 00.276-1.224 1.218 1.218 0 00-.652-.889l-.237-.237-.316-.04-.158.158a1.433 1.433 0 00-.79.79l-.02.297.376.375c.137.214.319.396.533.533l.276.277h.317l-.178.533zM135.272 328.513l-.257-.06-.474-.039-.869.869-.218.454-.039.277.02.336c.007.185.054.367.138.533l.217.217.494.257.296.06.356-.04.197-.198c.335-.148.602-.416.751-.75l.138-.138.02-.415-.435-.435a1.76 1.76 0 00-.335-.928zM130.116 337.422l.316.316a1.222 1.222 0 001.106.118 1.448 1.448 0 00.889-.889l.039-.355-.355-.356.138-.138.02-.257-.04-.277-.928-.928-.415-.059-.869.869a1.316 1.316 0 00-.079 1.146 1.51 1.51 0 00-.099.928l.336.336-.059-.454zM129.365 302.695l.474-.474a1.485 1.485 0 00-.554-1.976l-.138-.138-.395-.039-.217-.02-1.245 1.244.04.395.138.139c.149.265.367.483.632.632.176.101.371.161.573.177l.316-.039.376.099zM143.824 318.103l-.178-.178a1.366 1.366 0 00-1.639.217l-.652.652.02.375.118.119c.166.322.428.585.751.75l.178.178.296.02.672-.672a1.651 1.651 0 00.355-1.264l-.494-.494.573.297zM140.131 328.157l-.217-.217-.553-.119-.178-.02a1.623 1.623 0 00-.534.02 1.722 1.722 0 00-.553-.079l-.671.672a1.785 1.785 0 00-.316 1.422l.336.336a1.394 1.394 0 00.513.948l.415.415.138-.138c.332-.035.651-.15.929-.336l.434-.435.237-.237a1.48 1.48 0 00.02-2.232zM141.889 322.962l-.297-.02-.592.593a1.423 1.423 0 00.217 2.034l.336.336a1.336 1.336 0 001.106-1.106 1.375 1.375 0 00-.672-1.383l-.098-.454zM141.158 313.974l-.099.099.79.79a1.395 1.395 0 001.837-.336l.139-.138.099-.336-.099-.099v-.237l-.336-.336a1.406 1.406 0 00-.533-.533l-.297-.296-.335.02-1.008 1.007-.158.395zM132.467 366.44a1.29 1.29 0 00-1.107-.119 1.559 1.559 0 00-.967.968l-.06.257v.197l.139.139a1.417 1.417 0 00.691.889 1.518 1.518 0 001.185.118l.494-.494a1.447 1.447 0 00.296-1.244 1.309 1.309 0 00-.671-.711zM160.022 299.534l-.671-.672a1.52 1.52 0 00-1.225-.158 1.435 1.435 0 00-.83.83l-.039.237.039.237a1.227 1.227 0 00.593 1.343c.147.1.317.161.494.178h.197l.336-.059.494-.494c.182-.132.342-.292.474-.474l.119-.119.039-.355-.099-.099.079-.395zM158.817 293.035l.099.099.059.257.632.632a1.737 1.737 0 001.877-.059l.691-.692.02-.296-.098-.099a1.373 1.373 0 00-.771-.77 1.568 1.568 0 00-.869-.198h-.198l-.256.06-.949.948-.237.118zM147.361 323.14l-.198-.198a1.342 1.342 0 00-1.185-.118 1.443 1.443 0 00-.869.869l-.119.118v.316l.85.85a1.35 1.35 0 001.185.118 1.424 1.424 0 00.889-.889v-.316l-.474-.474-.079-.276zM169.939 300.996l-.119.118a1.144 1.144 0 00-1.145.712l-.079.276-.099.099.099.099.059.257.415.414a1.363 1.363 0 001.244.059c.259-.141.471-.353.613-.612a1.315 1.315 0 00-.474-1.817l-.514.395zM194.65 290.704l.02.218.573.572a1.445 1.445 0 001.541-.118 1.57 1.57 0 00.553-1.659l-.158-.158-.217-.218a1.49 1.49 0 00-1.324-.059 1.594 1.594 0 00-.81 1.324l-.178.098zM138.353 333.609l-.099.099-.019.296.691.691a1.36 1.36 0 001.264.079 1.563 1.563 0 00.85-1.244l.118.198-.158-.475-.593-.592-.474-.079h-.276a1.575 1.575 0 00-.8.326c-.23.182-.405.425-.504.701zM193.564 339.594l-.395-.039-.277.079c-.19.01-.374.072-.533.178a.808.808 0 00-.178.177l-.158.435-.118.119-.04.395.158.632.277.276c.302.165.655.207.987.119a1.255 1.255 0 00.712-.711 1.368 1.368 0 00-.435-1.66zM181.06 312.275a1.483 1.483 0 00-1.896.316l-.178.178-.138.415-.138.138.019.296 1.008 1.008.395.039.612-.612.336-.336.079-.276.02-.297-.04-.276-.217-.455-.277-.276.415.138zM150.758 309.51l.434.434.356.04.217-.02.237-.039.85-.85.039-.355-.849-.85a1.426 1.426 0 00-1.126-.138 1.475 1.475 0 00-.81.81l-.118.118-.04.356.395.395.415.099zM174.897 291.376a1.26 1.26 0 00-1.502.355l-.177.178-.079.514a1.257 1.257 0 00.632 1.264c.155.083.321.143.494.178l.316-.04.691-.691a1.314 1.314 0 00-.178-1.719l-.158-.158-.039.119zM97.562 308.423l-1.126-1.126-.375-.059-.77.77a1.426 1.426 0 000 1.62l.236.237.474.237.159.158.335-.019.297-.099.177-.178c.253-.134.46-.34.593-.593l.119-.118-.119-.83zM143.489 360.513l-.218-.217-.513-.197-.297-.06-.335.06-.593.592a1.432 1.432 0 00-.296 1.126c.083.353.303.659.612.85l.277.276c.22-.064.42-.183.582-.346.163-.162.281-.362.346-.582a1.479 1.479 0 00.435-1.502zM131.439 348.661l-.414-.415c.045-.103.084-.208.118-.316a1.54 1.54 0 00-.632-1.659l-.118-.118-.297-.06-.099-.098-.572.572a1.754 1.754 0 00-.553 1.304c0 .266.09.523.256.731l-.098.099a1.288 1.288 0 00.651 1.284l.316.316.317-.04.908-.908.217-.692zM112.1 294.556l.336.336.376.059.237-.039.513-.514a1.364 1.364 0 00.257-1.126 1.434 1.434 0 00-.711-.948l-.257-.257-.336-.019-1.066 1.066-.06.375.672.672.039.395zM159.074 264.728l.178.178.454.059.257-.019.198-.198c.23-.077.439-.205.612-.375l.138-.139.139-.335.098-.099.02-.257-.099-.099-.118-.355-.139-.139a1.384 1.384 0 00-.335-.335 1.426 1.426 0 00-1.897.474 1.34 1.34 0 00.494 1.639zM163.993 288.057a1.29 1.29 0 00.395 1.107l.158.158.375.138.118.118.396-.039.158-.158a1.263 1.263 0 00.731-.731l.118-.119v-.355l-.81-.81a1.312 1.312 0 00-1.718.336l-.139.138.218.217zM185.801 299.672l.257.257c.099.196.258.355.454.455l.178.177.454.099.257.02.296-.099.573-.573a1.342 1.342 0 00-.454-1.837l-.198-.197-.336.019-.75.751-.731.928zM128.476 312.236l.118.118.139-.138a1.424 1.424 0 00.485-1.86 1.339 1.339 0 00-.466-.491 1.282 1.282 0 00-1.106-.079l-.592.593a1.395 1.395 0 00-.178 1.047 1.282 1.282 0 00.75.75l.85.06zM111.33 340.839l.257-.059.434-.435c.201-.139.375-.313.514-.514l.079-.316-.04-.592-.671-.672-.573-.138.059.375-.079-.395a1.569 1.569 0 00-1.462.869l-.138.138-.039.356 1.145 1.146.376.059.138.178zM142.541 279.8v.356l.158.158c.11.176.258.325.434.435l.158.158.376.138.217.02.375-.06.692.692.257.059h.237l.434-.119.158-.158a1.338 1.338 0 00.494-1.125 1.315 1.315 0 00-.671-1.028 1.4 1.4 0 00-1.048-.138l-.237-.237a1.423 1.423 0 00-1.639.138l-.454.455.059.256zM144.378 284.521c-.24.346-.339.77-.277 1.186.084.405.326.76.672.987l.355.356.237-.04a1.652 1.652 0 00.929-.928l.118-.119v-.355l-1.145-1.146-.376-.059-.533.533.02-.415zM161.642 317.194a1.452 1.452 0 00-1.956.612 1.3 1.3 0 00-.138 1.008c.063.161.159.307.282.429.122.123.268.219.429.282l.395.039.672-.158.672-.671.039-.395-.178-.692-.217-.454zM143.291 351.664l.079-.356-.039-.316-.237-.553-.534-.533-.415-.02-.888.889a1.282 1.282 0 00.474 1.738 1.313 1.313 0 001.343-.079l.375-.375-.158-.395zM151.528 302.102l.158.158h.237a1.424 1.424 0 001.146-1.146 1.366 1.366 0 00-.691-1.402l-.395-.158-.119-.119-.138.139a1.334 1.334 0 00-.613 1.165 1.368 1.368 0 00.415 1.363zM154.768 286.773v.198l.019.257.632.632a1.537 1.537 0 002.075-.613l.158-.158.098-.414v-.237l-.039-.277-.889-.889-.296-.059-.218.02-.395.079-.158.158a3.019 3.019 0 00-.711.711l-.138.138-.138.454zM129.563 337.421l.019-.296-.691-.691a1.451 1.451 0 00-1.679.178l-.593.592-.039.277.02.257.177.414c.131.208.306.384.514.514l.316.316.119-.119a1.645 1.645 0 00.928-.928l.909-.514zM148.348 302.833l.139-.138.098-.376-.098-.098-.04-.316a1.424 1.424 0 00-.375-.652l-.178-.178-.415-.138-.217-.02-.277.039-.928.929-.02.415.119.118.079.277.553.553a1.281 1.281 0 001.56-.415zM134.738 338.725a1.294 1.294 0 00-.711-.711l-.415-.059-.336.099-.257.019a1.037 1.037 0 00-.493.494l-.277.04-.553.553a1.563 1.563 0 00-.257 1.165l.869.869a1.335 1.335 0 001.541-.158l.356-.355a1.295 1.295 0 00.928-.652 1.65 1.65 0 00-.395-1.304zM131.261 281.242a.642.642 0 000-.276.67.67 0 00.099-.375 1.175 1.175 0 00-.533-1.127l-.139-.138-.355-.079-.277-.039h-.197l-.257.059-.553.553c-.16.297-.235.632-.217.968a1.287 1.287 0 00.533 1.561l.138.138.217-.02a1.48 1.48 0 001.541-1.225zM157.672 276.363l.04.356.118.118c.064.204.18.388.336.534l.158.158.395.118.138.139.376-.06.138-.138a1.35 1.35 0 00.77-.77l.119-.119-.02-.336-.751-.75a1.42 1.42 0 00-1.659.158l-.178.177-.177.376.197.039zM81.324 298.744l-.276-.277h-.317l-.572.573a1.43 1.43 0 00-.34 1.187 1.424 1.424 0 00.715 1.006l.276.276.297.02.691-.691a1.4 1.4 0 00.062-1.749 1.396 1.396 0 00-.477-.404l-.059.059zM141.672 357.353l.099-.099a1.254 1.254 0 00-.475-1.145l-.138-.139-.375-.098-.119-.119-.336.059-.138.138a1.397 1.397 0 00-.75.751l-.119.119.02.375.81.81a1.282 1.282 0 001.679-.296l.118-.119-.276-.237zM112.634 326.695l-.198-.197-.474-.158h-.276l-.317.079-.612.612a1.29 1.29 0 00-.157 1.012c.044.17.123.33.232.468.108.139.244.254.399.338l.217.217h.317l.711-.711a1.396 1.396 0 00.158-1.66zM171.815 317.53a1.363 1.363 0 00-.493.019 1.228 1.228 0 00-1.028 1.027 1.26 1.26 0 00.633 1.344 1.202 1.202 0 001.264 0 .394.394 0 00.118-.119l.652-.652.02-.494-.83-.829-.336-.296zM159.983 332.009a1.067 1.067 0 00-.375-.138l-.099-.099-.158.158a1.596 1.596 0 00-.632 1.185 1.626 1.626 0 00.652 1.205l.138.138.217-.019c.285-.064.546-.207.753-.413.206-.207.349-.468.413-.753a1.395 1.395 0 00-.909-1.264zM116.743 334.439l-.297.296a1.429 1.429 0 00-.652.849v.277l.731.731a1.432 1.432 0 002.167-1.512 1.425 1.425 0 00-.626-.898l-.118-.118-.356-.079-.849.454zM162.709 364.128l-.79-.79a1.396 1.396 0 00-2.049.959 1.315 1.315 0 00.607 1.352l.158.158.415.06.217-.02.277-.079.158-.158c.246-.126.447-.326.572-.573l.119-.119.316-.79zM116.782 309.944l.06.257.612.613a1.505 1.505 0 001.56-.06l.633-.632.019-.296-.138-.138-.198-.435-.553-.553a1.368 1.368 0 00-1.837.336l-.158.158v.75zM100.92 297.717l-.672.197-.632.632-.06.297.297.296.06.296.276.277a1.29 1.29 0 001.007.138c.165-.066.315-.165.441-.29.125-.126.224-.276.29-.441a1.36 1.36 0 00-.494-1.876l-.513.474zM102.836 323.772l-.138-.138-.375-.06-.218.02-1.007 1.008-.02.335.099.099c.047.234.164.447.336.612l.652.652h.474l.256-.059.751-.751a1.48 1.48 0 00-.81-1.718zM140.644 299.712l-.711.711-.059.336.889.889a1.373 1.373 0 001.165.138 1.414 1.414 0 00.83-.83l.02-.296-.395-.395a1.306 1.306 0 00-.534-.533l-.296-.297h-.316l-.257.257-.336.02zM103.685 333.866l-1.027-1.027-.968.968a1.622 1.622 0 00-.039.908l.336-.02-.238.237.257.455.435.434.296.06.534-.02.197-.198a2.12 2.12 0 00.79-1.501l-.573-.296zM102.994 278.655l-.02-.297-.217-.217a1.34 1.34 0 00-.553-.553 1.258 1.258 0 00-1.659.395l-.257.257a1.308 1.308 0 00.355 1.501l.138.138.376.099.118.119.297-.06c.26-.06.499-.19.691-.375l.198-.197.533-.81zM112.436 310.497v-.355l-.79-.79a1.48 1.48 0 00-1.975.553l-.139.138-.059.415v.197l1.106 1.107.376-.02.118-.119a1.649 1.649 0 00.929-.928l.434-.198zM118.698 293.213v.198l1.007 1.007.395.04.257-.06.731-.731.059-.375-.098-.099-.06-.296a1.479 1.479 0 00-.652-.652l-.711.277-.711.197-.296.297.079.197zM177.88 328.611l.138.139.375.098h.198l.257-.019.533-.534a1.387 1.387 0 00.335-1.064 1.41 1.41 0 00-.552-.97l-.139-.139-.355-.079-.119-.118-.256.059-.712.711a1.508 1.508 0 00-.335 1.126c.102.335.328.617.632.79zM205.891 348.721a1.432 1.432 0 00-.712-.949l-.177-.177-.257-.02-.356.356a1.675 1.675 0 00-.632 1.382 1.256 1.256 0 00.632 1.027c.283.157.613.206.929.139l.533-.534a1.46 1.46 0 00.04-1.224zM172.309 345.362l-.178-.177-.296-.02-.296.059-.455.218-.197.197a1.39 1.39 0 00.098 1.679l.455.454.316.079h.316l.513-.197.218-.217a1.452 1.452 0 00-.04-1.857l-.454-.218zM167.193 325.826l.277.277a1.263 1.263 0 001.001.133 1.259 1.259 0 00.777-.647 1.355 1.355 0 00.188-1.026 1.336 1.336 0 00-.603-.85l-.395-.04-.277.079a.896.896 0 00-.513.158.609.609 0 01-.158.158 1.896 1.896 0 00-.395.395l-.04.277.099.098.039.988zM157.178 344.651l-.04-.237a1.397 1.397 0 00-.158-.553l-.197-.197-.455-.257-.158-.158h-.316l-.276.079-.593.276-.553.554-.04.355.119.119.039.237.573.572a1.427 1.427 0 001.837.139l.178-.178.178-.454-.138-.297zM166.383 329.362l-.553-.553a.977.977 0 00-.474-.158l-.454-.454a1.53 1.53 0 00-1.205-.099 1.358 1.358 0 00-.81.81l.02.336 1.6 1.6a1.263 1.263 0 001.31-.014 1.258 1.258 0 00.586-1.172c.007-.099 0-.199-.02-.296zM163.44 303.031a1.26 1.26 0 001.106 1.382h.237l.139-.138a1.29 1.29 0 00.434-1.146 1.34 1.34 0 00-.711-1.106 1.253 1.253 0 00-1.659.316l-.198.198.652.494zM167.667 330.014l.138-.138a1.292 1.292 0 00.909-.909 1.317 1.317 0 00-.415-1.205l-.158-.158-.414-.138-.238-.04-.296.06-1.027 1.027v.316l.178.177.237.435.454.454.632.119zM184.951 353.382l-.197-.197h-.316a1.504 1.504 0 00-1.304.395 1.339 1.339 0 00-.197 1.62l.177.178.415.217.257.257.375.059.099.099.138-.139.316-.079.553-.553a1.34 1.34 0 00-.316-1.857zM168.122 351.842l.039-.316-.197-.198a1.341 1.341 0 00-.672-1.027l-.336-.336-.316.04-.652.651-.237.475-.079.316.06.335.454.455-.039-.04.177.178a1.265 1.265 0 001.027.119 1.318 1.318 0 00.771-.652zM186.215 316.325l-.118.118.039.277.692.691a1.382 1.382 0 001.106.119 1.456 1.456 0 00.889-.889l.02-.336-.83-.83a1.37 1.37 0 00-1.837.455l-.138.138.177.257zM172.487 339.14l.316-.316a1.505 1.505 0 00.296-1.205 1.371 1.371 0 00-.691-.928l-.217-.218-.336-.019-.277.079a1.335 1.335 0 00-.77.77l-.099.099v.276l.376.376c.126.211.302.387.513.513l.218.218h.316l.138-.139.217.494zM182.937 343.901a1.483 1.483 0 00.869-.869l-.099-.099-.019-.336-.712-.711a1.367 1.367 0 00-1.619.158l-.514.513.02.297a1.331 1.331 0 00.687.873 1.334 1.334 0 001.11.055l.277.119zM162.255 312.315l.296.296a1.431 1.431 0 001.126.099c.281-.143.51-.371.652-.652.096-.075.182-.161.257-.257l.138-.138.138-.375.02-.218-.02-.336-.138-.138a1.621 1.621 0 00-.415-.415l-.158-.158-.375-.138-.119-.119-.355.04-.119.119c-.218.055-.412.18-.553.355l-.257.257a1.597 1.597 0 00-.336 1.126l.119.118.099.534zM155.637 323.199l-.118.118c.064.161.16.306.283.428.122.123.267.219.428.283a1.365 1.365 0 001.442-.019 1.508 1.508 0 00.671-1.383l-.731-.731a1.483 1.483 0 00-1.185-.118 1.588 1.588 0 00-.889.889l.099.533zM151.489 333.234a1.35 1.35 0 00.908-.909l.139-.138-.02-.218-.158-.158a1.396 1.396 0 00-.652-.651 1.27 1.27 0 00-.632-.158l-.099-.099-.237.039-.474.474a1.422 1.422 0 00-.356 1.146 1.34 1.34 0 00.692.968c.322.014.639-.092.889-.296zM151.43 352.968v-.356l-1.087-1.086-.375-.06-1.126 1.126-.039.435.039.237.83.83a1.373 1.373 0 001.067.009 1.37 1.37 0 00.75-.76l-.059-.375zM184.201 323.87a1.37 1.37 0 00.454 1.877l.099.099.257.059.098-.099.277-.079a1.399 1.399 0 00.83-1.778l-.632-.632a1.404 1.404 0 00-1.383.553zM148.645 329.362a1.483 1.483 0 001.145-1.146 1.313 1.313 0 00-.612-1.363l-.217-.217a1.32 1.32 0 00-.909.672 1.561 1.561 0 00-.079 1.264l.79.79.356.04-.474-.04zM170.097 310.201v.316l.829.83a1.505 1.505 0 001.264.158 1.43 1.43 0 00.83-.83l.02-.296-.296-.296a1.623 1.623 0 00-.534-.534l-.178-.178h-.276l-.376.376a1.364 1.364 0 00-.73.731l-.553-.277zM152.239 328.256l.376.138.118.119.316-.04.119-.118c.177-.124.33-.278.454-.455l.158-.158.119-.355.019-.218-.019-.335-.119-.119a1.457 1.457 0 00-.454-.454l-.158-.158-.356-.119-.118-.118-.336.019a1.123 1.123 0 00-.573.376l-.158.158-.119.355-.019.218.019.335.099.099c.068.213.19.404.356.553l.276.277zM156.428 329.955a1.397 1.397 0 001.185.118 1.517 1.517 0 00.889-.889l.039-.276-.375-.376a2.38 2.38 0 00-.454-.454 1.369 1.369 0 00-1.541.118l-.593.593-.019.296.869.87zM152.971 325.392l.375.098.336.336.256.02a1.424 1.424 0 001.146-1.146 1.313 1.313 0 00-.612-1.363l-.138-.138-.336-.059-.099-.099-.336.059-.355.04-.119.118a1.369 1.369 0 00-.454.454l-.139.139-.138.375-.02.217.06.336.118.119c.113.173.261.321.435.434l.02.06zM157.02 317.154l.395.119.218.02.316-.04.118-.118c.186-.114.341-.269.455-.455l.138-.138.138-.375.138-.218-.019-.335-.119-.119a1.228 1.228 0 00-.434-.434l-.158-.159-.376-.138-.118-.118-.356.039a1.45 1.45 0 00-.474.474l-.138.139-.139.375-.019.217.059.336.118.119c.116.171.264.318.435.434l-.178.375zM153.365 338.034l.02-.375-1.067-1.067-.335-.02a1.54 1.54 0 00-1.775.108c-.165.132-.3.297-.398.485l-.099.098-.079.317.988.987a1.526 1.526 0 001.165.178 1.345 1.345 0 001.539-.266c.131-.133.232-.291.298-.465l-.257.02zM158.107 338.35l.632.632a1.317 1.317 0 001.185.04 1.486 1.486 0 00.889-1.245 1.341 1.341 0 00-.672-1.225 1.253 1.253 0 00-1.027-.118 1.404 1.404 0 00-.79.79 1.232 1.232 0 00-.217 1.126zM176.28 321.678l.276.158.198.079.355.039.218-.019.256-.06.178-.098.296-.218.139-.177.079-.119.079-.198.019-.375.001-.197-.06-.257-.099-.178-.237-.316-.158-.119-.276-.158-.198-.079-.355-.039-.218.02-.237.079-.197.079-.296.217-.139.178-.079.118-.059.218-.04.355.02.217.06.257.098.178.218.296.158.119zM168.616 357.866l-.099-.098-.04-.277a1.557 1.557 0 00-.414-.415l-.158-.158-.376-.138-.138-.138-.375.059-.158.158a1.286 1.286 0 00-.751.75l-.119.119.02.336 1.027 1.027.376.059.217-.019.83-.83.158-.435zM153.839 356.207l-.138-.138-.355-.04-.099-.099-.257.06-.711.711a1.41 1.41 0 00-.3 1.098 1.397 1.397 0 00.616.956l.118.119.317.079.237-.04.612-.612c.172-.155.307-.349.392-.565.086-.215.121-.448.102-.68a1.368 1.368 0 00-.534-.849zM162.59 379.102a1.36 1.36 0 00-1.145-.079 1.389 1.389 0 00-.85.849l-.118.118.039.356.593.593c.116.195.279.357.474.474.304.157.655.199.988.118l.513-.513a1.365 1.365 0 00.091-1.689 1.362 1.362 0 00-.446-.405l-.139.178zM182.404 331.16l-.119-.119-.296-.059-.218.02-.316.316a1.083 1.083 0 00-.296.296l-.138.138-.099.376-.02.217.929.928.257.059.177-.177a1.45 1.45 0 00.613-1.245 1.511 1.511 0 00-.474-.75zM173.85 361.244l-.316-.316-.317.04-.75.75-.059.336.158.158c.067.213.19.404.355.553l.178.178.474.119h.277l.296-.099.415-.415-.316.316a1.344 1.344 0 00-.395-1.62zM160.2 327.999l.217-.02 1.62-1.62.04-.395-.81-.81a1.347 1.347 0 00-1.126-.098l-.178.178-.296-.02-.257.059-.454.217-.198.198c-.06.071-.126.137-.197.197l-.672.672v.316l.039.277.83.829c.2.102.42.161.644.174a1.63 1.63 0 00.66-.095l.138-.059zM162.254 343.96l-.829-.83a1.374 1.374 0 00-1.107-.158 1.54 1.54 0 00-.829.83l-.099.099-.04.355.83.83a1.254 1.254 0 001.126.138 1.475 1.475 0 00.81-.81l.138-.454zM200.774 309.194l.336-.336c.208-.322.285-.71.217-1.087a1.452 1.452 0 00-.671-.869l-.297-.296h-.316l-1.007 1.007-.059.257-.02.336.869.869c.288.163.629.205.948.119zM100.189 319.782l.474-.475a1.391 1.391 0 00.731-1.086 1.341 1.341 0 00-.198-.711l-.731-.731a1.619 1.619 0 00-1.363-.178l-.474.474c-.183.253-.28.558-.276.87l.118.118.04.237 1.125 1.126.554.356zM89.858 342.202l-.02-.336-.711-.711a1.338 1.338 0 00-1.66.158l-.316.316v.277a1.282 1.282 0 00.751.75 1.232 1.232 0 001.106.079 1.422 1.422 0 00.889-.889l-.04.356zM134.343 317.905l.968.968h.316l.276-.079.553-.553c.223-.301.489-.568.79-.79l.178-.178.198-.395.158-.158v-.277l-.158-.158-.237-.434-.257-.257a1.28 1.28 0 00-1.205-.059l-.138.138-.395.197-.554.158-.572.573a1.283 1.283 0 00.079 1.304zM136.417 313.184l.119-.119.316-.118.138-.138a1.595 1.595 0 00.474-1.66l-1.106-1.106-.415-.138h-.237l-.276.039-.988.988-.04.435 1.581 1.58.434.237zM130.352 322.922l.119-.118-.079-.277-.869-.869a1.289 1.289 0 00-1.102-.051 1.285 1.285 0 00-.735.821l-.119.119.04.355.77.771a1.285 1.285 0 001.679-.296l.158-.158.138-.297zM121.404 341.629l-.395-.039-.355.079-.119.118a1.11 1.11 0 00-.75.751 1.164 1.164 0 00.158 1.027l.395.395c.302.165.655.207.987.118.303-.127.545-.368.672-.671a1.395 1.395 0 00-.593-1.778zM118.975 292.067l.276-.276a1.253 1.253 0 00.02-.968 1.35 1.35 0 00-.77-.771 1.301 1.301 0 00-1.004-.053 1.28 1.28 0 00-.735.686l-.02.414.257.652.534.534h.395l.138-.139.909-.079zM72.356 267.533l.02-.414-.257-.652-.375-.376a1.505 1.505 0 00-.455-.138h-.395l-.651.257-.277.276a1.287 1.287 0 00.691 1.719c.317.111.663.104.975-.02.312-.123.569-.354.724-.652zM129.819 287.346l-.355-.079-.237.237a1.288 1.288 0 00-1.027.751 2.032 2.032 0 00-.099.375l.099.099.059.296 1.047 1.047.336.02.908-.909a1.369 1.369 0 00-.494-1.797l-.237-.04zM103.033 287.267h-.316l-.533.139-.237.237a2.179 2.179 0 00-.296.296 1.38 1.38 0 00-.02 1.086 1.573 1.573 0 001.027.79l.296.02.435-.434c.101-.084.193-.176.276-.277a1.312 1.312 0 00-.691-1.719l-.099.099-.098-.099.256-.138z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M127.469 298.151l-.158.158-.415.218-.494.494a1.205 1.205 0 00.079 1.185 1.346 1.346 0 001.166.652l.099-.099.316-.079.671-.672a1.481 1.481 0 00-.257-1.56l-.177-.178-.415-.178-.139-.138-.276.197zM91.32 282.744a1.175 1.175 0 00.75-.751c.161-.433.161-.91 0-1.343l-.79-.79-.336-.02a1.303 1.303 0 00-.79.79 1.399 1.399 0 00-.81.533l-.138.139-.119.395.119.118.04.277.572.573a1.654 1.654 0 001.501.079zM82.569 270.398l.612-.613a1.422 1.422 0 00-.276-1.54 1.531 1.531 0 00-1.7-.277l-.612.612a1.343 1.343 0 00.277 1.502 1.484 1.484 0 001.699.316zM136.654 319.189l-.118.118-.06.376a1.293 1.293 0 00.435 1.027l.375.375c.29.01.576-.058.83-.197l.77-.771v-.355l-.098-.099a1.656 1.656 0 00-1.304-.988c-.333.056-.632.24-.83.514zM106.885 268.047l.356-.039.77-.771a1.424 1.424 0 00-.355-1.62l-.158-.158-.415-.138-.119-.119-.375.06-.158.158a2.516 2.516 0 00-.435.434l-.158.158-.079.395v.237l.079.356.158.158c.197.316.508.543.87.632l.019.257zM120.377 274.072a1.394 1.394 0 00-.02-1.126 1.53 1.53 0 00-.968-.731l-.099-.099-.355.04-.988.987v.396l1.106 1.106h.435l.83-.83.059.257zM97.206 287.524l.139-.138.059-.376-.099-.098-.079-.277a1.227 1.227 0 00-.553-.553l-.138-.138-.376-.099-.217-.02-.237.04-.375.375a1.65 1.65 0 00-.533.533 1.155 1.155 0 00-.08.277l.02.296.672.672a1.482 1.482 0 001.165.217c.29-.154.514-.406.632-.711zM159.134 316.937l-.435.119-.138.138c-.265.132-.48.347-.612.612-.068.15-.108.311-.119.474l.198.198a1.283 1.283 0 001.303.869h.198l.257-.059.513-.514c.09-.067.17-.147.237-.237a1.452 1.452 0 00-.336-1.639l-.158-.158-.533-.218-.375.415zM113.838 269.114l.178.177.435.158.257.06.296-.02.711-.711a.835.835 0 00.178-.178 1.401 1.401 0 00-.553-1.778l-.138-.138-.218.02a1.269 1.269 0 00-.456.294c-.13.129-.23.285-.294.456a1.399 1.399 0 00-.396 1.66zM142.402 281.716c.139.249.344.454.593.593l.099.099.375-.02.85-.849c.098-.256.128-.532.087-.802a1.575 1.575 0 00-.324-.739l-.178-.178-.415-.178-.237-.039-.395.039-.119.119c-.248.101-.47.256-.652.454l-.177.178-.079.435.019.256.079.277.198.197.276.158zM111.27 269.074c.055-.118.088-.245.099-.375l.158-.158-.138-.138a1.285 1.285 0 00-.948-.949 1.571 1.571 0 00-1.403.455l-.316.316.02.257a1.559 1.559 0 001.343.987 1.345 1.345 0 001.185-.395zM148.605 274.013l-.355-.04-.297.059-.493.297-.297.296a1.351 1.351 0 000 1.106 1.38 1.38 0 00.81.81l.099.099h.316l.573-.573a.874.874 0 00.257-.257 1.29 1.29 0 00-.514-1.62l-.257-.256.158.079zM129.286 279.859l-.316.001a1.417 1.417 0 00-.889.888 1.403 1.403 0 00-.015 1.054 1.4 1.4 0 00.726.764l.138.138.435.04.533-.534c.226-.122.412-.307.534-.533l.039-.237v-.395l-1.185-1.186zM112.238 278.26c.088.146.21.268.356.355.105.225.245.431.415.612-.053.24-.025.49.079.712h-.277l-.671.671c-.127.083-.234.19-.316.316a1.282 1.282 0 00.395 1.541l.138.138.356.119.118.118.257-.019.415-.455.809-.81.02-.375-.118-.118.257-.02.098-.099a1.281 1.281 0 001.126-.81c.096-.075.182-.161.257-.257l.237-.237a.949.949 0 00-.118-.355l-.257-.257a2.033 2.033 0 00-.494-.494l-.514-.513a1.256 1.256 0 00-.513-.791l-.297-.098-.237-.04h-.434l-.237.04-.83.829a2.066 2.066 0 00-.02.613 1.285 1.285 0 000-.316zM115.36 233.103v-.395l-.139-.376-.138-.138a1.001 1.001 0 00-.711-.711 1.225 1.225 0 00-1.007.257l-.415.415a1.237 1.237 0 00-.038.964 1.228 1.228 0 00.67.695c.28.12.589.155.889.099a1.411 1.411 0 00.889-.81zM113.522 263.741l-.02.375a1.26 1.26 0 00.949 1.383l.098.098.238-.039.118-.119c.361-.111.67-.35.869-.671l.138-.138-.118-.119a1.65 1.65 0 00-1.403-1.007 1.394 1.394 0 00-1.323.809l.454-.572zM122.214 268.521l.573-.573a1.422 1.422 0 00-.731-1.876 1.65 1.65 0 00-1.146-.04 1.449 1.449 0 00-.77.771l.02.414.138.139a1.251 1.251 0 00.592.948 1.234 1.234 0 001.205.138l.119.079zM118.481 263.958l.276-.039.692-.692c.079-.221.113-.456.098-.691l-.474-.474a1.433 1.433 0 00-1.284-.534 1.317 1.317 0 00-1.047.771 1.154 1.154 0 000 .948l.119.356a1.427 1.427 0 001.007.691l.613-.336zM114.846 258.466l.593.593.335.02.297-.059.474-.277.217-.217c.044-.049.09-.095.138-.139a1.393 1.393 0 00-.592-1.856 1.486 1.486 0 00-.85-.376 1.333 1.333 0 00-1.047.455l-.217.217-.02.217.06.296.098.099c.133.206.308.381.514.514v.513zM87.527 257.261a1.281 1.281 0 00-.62-1.725 1.284 1.284 0 00-.664-.112l-.85.85-.04.276.416.415a1.418 1.418 0 001.185.474c.202-.016.397-.077.573-.178zM155.439 259.039l-.078.395-.119.119.079.316.119.119c.12.203.29.373.493.493l.139.139.395.079.098.099.356-.079.119-.119c.189-.109.345-.265.454-.454l.138-.139.079-.395.02-.217-.079-.316-.119-.119a1.379 1.379 0 00-.474-.474l-.138-.138-.375-.099-.218.02-.335.059-.119.119a1.176 1.176 0 00-.435.434v.158zM144.298 270.338h-.197l-.178.178c-.164.125-.31.271-.434.435l-.178.178-.06.493.02.257.79.79a1.34 1.34 0 001.758-.691l.099-.296-.059-.336-1.126-1.126-.435.118zM135.31 255.938l.099-.099-.099-.098.139-.139-.02-.415-.277-.592-.276-.277a1.335 1.335 0 00-1.487.224 1.342 1.342 0 00-.311.428 1.341 1.341 0 00.395 1.58 1.365 1.365 0 001.837-.612zM130.688 263.997l-.099-.098-.869.869a1.375 1.375 0 00.06 1.244 1.336 1.336 0 001.066.672l.119.118h.395l1.106-1.106.02-.375-1.264-1.264-.534-.06zM123.577 295.643a1.377 1.377 0 00-.742.221c-.22.144-.395.348-.503.589a1.54 1.54 0 00.297 1.6 1.454 1.454 0 001.62.316l.869-.87a1.431 1.431 0 00-.889-1.323l-.099-.099-.553-.434zM173.08 259.257l.277.079.454.02.257-.06.118-.118a1.346 1.346 0 00.889-.889l.158-.158v-.277l-.415-.415c-.21-.3-.529-.505-.889-.572h-.276l-.553.553a2.293 2.293 0 00-.336.533 1.398 1.398 0 00.316 1.304zM112.495 342.636a1.316 1.316 0 00.257 1.442l.218.218.375.059c.295.099.614.099.908 0a1.247 1.247 0 00.712-.711l.098-.099-.098-.099a1.262 1.262 0 00-1.146-1.145 1.396 1.396 0 00-1.324.335zM67.082 248.313a1.395 1.395 0 001.718-.77c.06-.152.1-.312.119-.474v-.237l-.257-.257a1.341 1.341 0 00-1.087-.771l-.316.001-.493.493c-.215.122-.392.3-.514.514l.119.118.158.356.395.395.335.138-.177.494zM172.092 274.23l-.751.75a1.423 1.423 0 00.672 1.818 1.257 1.257 0 001.679-.455l.118-.118.06-.296-.02-.257a1.338 1.338 0 00-1.264-1.264l-.494-.178zM133.494 274.704a1.233 1.233 0 00-.158-1.264l-.198-.198-.415-.178-.237-.237h-.395l-.889.889a1.312 1.312 0 00.375 1.64l.139.138.316.079.118.119.218-.02a1.484 1.484 0 001.126-.968zM124.111 294.438l.454-.455a.394.394 0 00.119-.118 1.705 1.705 0 00.019-1.403l-.197-.197a1.652 1.652 0 00-.573-.968 1.93 1.93 0 00-2.015-.237l-.83.829.02.376.277.276a1.426 1.426 0 00.415 1.047l.533.534a1.41 1.41 0 001.126.177l.276-.039.376.178zM72.909 301.668l.336-.06.81-.81a1.255 1.255 0 00-.929-1.679l-.099.099a1.313 1.313 0 00-1.264.593l-.118.118-.08.356.1.099.039.237.356.355a1.361 1.361 0 00.888.573l-.04.119zM91.399 292.62l.592.593c.185.112.39.186.604.217.214.03.432.017.641-.039a1.492 1.492 0 00.81-.81l.118-.119v-.316l-.355-.355a1.372 1.372 0 00-.87-.672h-.276l-.375.375a1.317 1.317 0 00-.534.534 1 1 0 00-.177.177l-.1.099.04.316H91.4zM109.138 279.899l.039-.276-.039-.475-.83-.829-.474-.158-.277-.04-.335.06a2.238 2.238 0 00-.514.197l-.217.217-.237.514-.04.316.099.336.217.217a1.347 1.347 0 00.751.751l.118.118.415-.02.474-.474a1.53 1.53 0 00.85-.454zM99.734 275.711l-.217.218a1.284 1.284 0 000 1.145 1.418 1.418 0 00.869.87l.099.098h.356l.296-.296.138.138h.237l.277-.079.79-.79.02-.217.019-.415-.889-.889a1.358 1.358 0 00-1.145.04c-.295-.1-.614-.1-.909 0l-.356.355.415-.178zM134.106 271.366l.454.454a1.45 1.45 0 001.896-.751l.139-.138.019-.375-.019-.218-.968-.968-.257-.059-.217.02-.376.059-.158.158a1.401 1.401 0 00-.632.632 1.604 1.604 0 00-.099.573l.079.316.099.099.04.198zM120.397 287.326l.178-.177a1.422 1.422 0 00-.395-1.62l-.514-.514h-.197l-.395.04-.139.138a1.651 1.651 0 00-.731.731 1.14 1.14 0 00-.078.277l.039.316.593.592a1.641 1.641 0 001.303.237l.494-.494-.158.474zM110.007 284.679l.217-.217.059-.533-.039-.237a1.242 1.242 0 00-.06-.534c.04-.182.04-.37 0-.553l-.79-.79a1.653 1.653 0 00-1.442-.138l-.336.336a1.509 1.509 0 00-.908.632l-.099.099-.079.276.138.138c.068.335.217.647.435.909l.671.672a1.521 1.521 0 001.13.464 1.515 1.515 0 001.103-.524zM115.359 285.884l-.019-.296-.692-.691a1.389 1.389 0 00-1.607.001c-.156.11-.288.251-.388.414l-.138.138.02.217a1.45 1.45 0 001.264.948 1.371 1.371 0 001.304-.829.555.555 0 00.256.098zM124.229 284.087l-.672.672a.659.659 0 00-.158.158 1.453 1.453 0 00.534 1.797l.118.119.376.059.098.099.474-.474c.215-.138.397-.32.534-.534l.178-.177-.04-.317-1.027-1.027-.415-.375zM71.21 281.203a1.397 1.397 0 00-.02-1.126 1.561 1.561 0 00-1.066-.83l-.257-.02-.099-.098-.257.059a1.537 1.537 0 00-.79.79 1.45 1.45 0 000 1.185l.514.514a1.504 1.504 0 001.264.158 1.37 1.37 0 00.711-.632zM140.526 301.49l.672-.672a1.504 1.504 0 00.019-1.244 1.442 1.442 0 00-.928-.731h-.237l-.217.02a1.284 1.284 0 00-1.284.77c-.07.163-.11.337-.119.513l.81.81c.127.186.288.347.474.474l.119.119h.355l.099-.099.237.04zM146.906 299.534l-.217.02-.672.672a.38.38 0 00-.118.118 1.728 1.728 0 00.237 1.857l.573.573.296-.02a1.35 1.35 0 00.77-.77c.113-.282.147-.59.099-.889l-1.146-1.146.178-.415zM115.774 291.317l.178-.178a1.398 1.398 0 000-1.185 1.563 1.563 0 00-.988-.791h-.316l-.592.593a.448.448 0 00-.178.178 1.435 1.435 0 00-.02 1.205c.101.189.241.354.411.484.171.129.368.22.577.266l.098.099.336-.02.593-.592-.099-.059zM140.289 311.327l-.139-.138a1.116 1.116 0 00-.829-1.067l-.158-.158-.218.02-.889.889a1.452 1.452 0 00.099 1.244c.152.25.362.461.613.612a1.364 1.364 0 001.777-.671l-.256-.731zM153.03 334.893l-.553.553c-.117.094-.223.2-.316.316a1.372 1.372 0 00.256 1.521 1.562 1.562 0 001.719.375l.691-.691a1.373 1.373 0 00-.098-1.323 1.567 1.567 0 00-1.383-.672l-.316-.079zM104.456 283.514l-.099-.099-.296.02-.672.672a2.79 2.79 0 00-.237.237 1.421 1.421 0 00.643 1.807c.232.128.494.193.76.188l.276-.04.435-.197.454-.455.02-.493-.04-.277a1.59 1.59 0 00-1.244-1.363zM104.278 339.002v-.395l-.178-.178a1.234 1.234 0 00-.237-.514l-.217-.217-.435-.079-.119-.118-.395-.001-.316.119-.395.395a1.264 1.264 0 00-.019.968 1.363 1.363 0 001.965.645 1.36 1.36 0 00.523-.566l-.177-.059zM130.135 323.693a1.507 1.507 0 00-.513-1.857l-.178-.178-.415-.099-.257-.019-1.185 1.185v.395l.731.731.197-.198-.177.178.434.237.158.158.297-.02.711-.711.197.198zM129.661 293.272l-.414.415v.356l1.126 1.126h.355l.77-.771a1.448 1.448 0 000-1.145 1.475 1.475 0 00-.809-.81l-.099-.099-.376.02-.671.671.118.237zM150.363 315.278a1.288 1.288 0 00-.513-1.462l-.277-.079-.494-.02a1.31 1.31 0 00-1.185.751 2.373 2.373 0 00-.118.513l.098.297.712.711a1.317 1.317 0 001.679-.375l.158-.158-.06-.178zM125.098 240.234l1.126-1.126.039-.356-.81-.81a1.57 1.57 0 00-1.619.159l-.198.197-.198.474-.039.277.059.335.376.376c.128.244.328.444.572.573l.119.118.573-.217zM78.183 291.455l.217-.217.158-.514.02-.296-.099-.336-.592-.593a1.366 1.366 0 00-1.166-.177 1.32 1.32 0 00-.73.731l-.06.256.119.119a1.484 1.484 0 001.67.917 1.48 1.48 0 00.621-.285l-.158.395zM88.673 278.2l.454-.454.336.059a1.475 1.475 0 001.56-.81l.1-.098.039-.316-.711-.712a1.84 1.84 0 00-1.363-.414 1.725 1.725 0 00-.692.335l-.099-.098a1.316 1.316 0 00-1.204.77l-.1.336.06.336 1.007 1.007.613.059zM140.447 253.252l-.316.316-.04.355.79.79a1.425 1.425 0 001.126.139 1.404 1.404 0 00.849-.85l.139-.138-.04-.356-1.067-1.066h-.395l-.829.829-.217-.019zM175.134 296.828l-.059.257-.04.474.04.237.217.217c.118.217.296.396.513.514l.119.118.336.099.118.119.218-.02.493-.494c.132-.09.246-.204.336-.336a1.456 1.456 0 00-1.679-2.098c-.37.108-.683.358-.869.696l.257.217zM152.397 304.156a1.324 1.324 0 00-1.066.514l-.158.158-.079.395v.237l.079.356.158.158c.194.314.496.545.849.652l.099.098.375-.059.711-.711a1.284 1.284 0 00-.513-1.659l-.139-.138-.316-.001zM143.153 327.071l-.277.276a1.168 1.168 0 00-.454.455l-.178.177-.039.474.019.257.771.771a1.325 1.325 0 001.023.06 1.3 1.3 0 00.755-.693l.197-.197v-.198l-.039-.316-1.186-1.185-.592.119zM124.525 271.346v.237l.257.257a1.374 1.374 0 001.264.513 1.281 1.281 0 001.067-.75c.078-.173.114-.362.104-.551a1.183 1.183 0 00-.164-.536l-.592-.592a1.481 1.481 0 00-1.087-.06 1.296 1.296 0 00-.75.751l-.099.731zM94.341 257.4l.593.592c.14.201.314.374.514.514l.316.04.553-.08.691-.691.08-.593.098-.098a1.597 1.597 0 00-1.027-1.383l-.119-.119h-.355l-1.146 1.146-.02.375-.178.297zM158.403 281.993l-.336.059-.158.158a1.471 1.471 0 00-.454.455l-.138.138-.079.395v.237l.079.356a.657.657 0 00-.237.237l-.415.414-.04.277.02.257.158.395.158.158a1.29 1.29 0 001.166.375 1.371 1.371 0 00.967-.77 1.417 1.417 0 000-1.067l.395-.395a1.455 1.455 0 00-.316-1.62l-.158-.158-.414-.138-.198.237zM153.859 284.285a1.58 1.58 0 00-1.224-.159 1.484 1.484 0 00-.85.85l-.158.158.237.237c.238.381.613.656 1.047.77l.217.02.316-.039 1.146-1.146.04-.356-.791-.79.02.455zM123.142 304.927a1.451 1.451 0 00-.77-1.877 1.428 1.428 0 00-1.027-.039 1.35 1.35 0 00-.731.731v.395l.237.632.731.731h.395l.652-.257.513-.316zM87.092 290.408l.316.04.336-.06.534-.296.513-.514v-.395l-.04-.237-.928-.928a1.254 1.254 0 00-1.66.632 1.283 1.283 0 00.218 1.323l.356.356.355.079zM137.109 293.253l-.238.237.139.138a1.536 1.536 0 001.343.948 1.426 1.426 0 001.324-.849c.057-.14.097-.286.118-.435l-.375-.375a1.584 1.584 0 00-1.245-.494 1.536 1.536 0 00-1.066.83zM152.694 294.852l-.099-.098-.671.671a.812.812 0 00-.178.178 1.564 1.564 0 00.83 2.015l.158.158.414.059.218-.02.276-.079.889-.888.02-.297-.02-.217-.138-.375-.158-.158a3.17 3.17 0 00-.731-.731l-.119-.119-.691-.099zM99.774 275.04h.277l.355-.356a2.65 2.65 0 00.237-.237 1.452 1.452 0 00-.355-1.659l-.573-.573h-.277l-.257.059-.395.198c-.219.115-.398.294-.513.513a1.32 1.32 0 00-.1.297l.218.217a1.623 1.623 0 001.027.79l.356.751zM136.022 290.23l.119.119.375.059.099.099.316-.079a1.34 1.34 0 00.533-.533l.158-.159.099-.414v-.237l-.079-.277-.929-.928-.414.019-.87.87a1.282 1.282 0 00.593 1.461zM98.925 280.433a1.345 1.345 0 00.73-.731l.001-.395-.138-.336-.139-.139a1.09 1.09 0 00-.513-.513l-.06-.257-.671-.672a1.54 1.54 0 00-1.185-.118l-.751.751a.38.38 0 00-.119.118 1.34 1.34 0 00.317 1.501l.355.356a1.222 1.222 0 00.77.77 1.395 1.395 0 001.403-.335zM155.716 270.891l.257-.019c.129.033.262.046.395.039a1.223 1.223 0 001.087-.652l.118-.118.059-.375.02-.257-.849-.85a1.482 1.482 0 00-.968-.098 1.288 1.288 0 00-1.482.73l-.138.139.118.118a1.587 1.587 0 001.383 1.343zM163.341 296.65l-.336.06-.138.138c-.172.114-.32.262-.435.434l-.158.158-.099.415v.237l.08.356.158.158a1.355 1.355 0 00.888.652l.099.098.356-.039.75-.751a1.42 1.42 0 00-.335-1.639l-.158-.158-.396-.119h-.276zM132.98 223.147l.079-.276.099-.099-.04-.316-.474-.474a1.53 1.53 0 00-1.264-.198 1.468 1.468 0 00-.849.85l-.178.177.02.336.691.692a1.395 1.395 0 001.206.208c.204-.059.391-.163.549-.305.157-.143.28-.319.359-.516l-.198-.079zM81.028 289.401h.198a1.265 1.265 0 001.086-.573l.138-.139.06-.414-.1-.099-.098-.336-.158-.158a1.483 1.483 0 00-.73-.731l-.1.099-.395.039-.71.712a1.284 1.284 0 00.493 1.639l.118.119.198-.158zM108.525 257.202l.197-.197.119-.514-.04-.276-.098-.297-.593-.592a1.317 1.317 0 00-1.033-.045 1.305 1.305 0 00-.745.716l-.197.198.039.316.79.79a1.397 1.397 0 001.561-.099zM123.913 315.06c.01-.166-.01-.334-.06-.493a1.202 1.202 0 00-1.027-1.027 1.25 1.25 0 00-1.264.75 1.321 1.321 0 00.138 1.284l.593.593.276.039.514-.039.83-.83v-.277zM108.288 304.808a1.18 1.18 0 00.118-.395l-.02-.217-.256-.257a1.406 1.406 0 00-1.245-.494 1.422 1.422 0 00-1.126.771l-.138.138.138.138a1.454 1.454 0 001.343.948 1.36 1.36 0 001.186-.632zM101.196 262.16l-.316-.316a1.39 1.39 0 00-.928-.572l-.277.039-.533.533a.557.557 0 00-.217.218 1.336 1.336 0 00.244 1.473c.12.129.266.232.427.305a1.34 1.34 0 001.778-.593l.119-.119.039-.355-.099-.099-.237-.514zM76.623 310.952l.889-.889a1.345 1.345 0 00-.224-1.487 1.37 1.37 0 00-2.225.302l-.14.138-.019.415.494.494c.115.254.319.458.573.573l.099.098.553.356zM125.888 259.514h-.237l-.809.809a1.53 1.53 0 00.236 1.541l.159.158.414.178.158.158.277-.04.81-.809c.151-.306.181-.658.082-.985-.1-.327-.32-.603-.616-.773l-.138-.139-.336-.098zM136.101 242.466l-.257-.652-.652-.652-.296-.019-.593.592a1.367 1.367 0 00-.019 1.047 1.312 1.312 0 00.731.731 1.36 1.36 0 001.057.025c.169-.066.324-.165.454-.292.131-.126.235-.277.306-.444l-.731-.336zM110.421 247.128l.02-.217v-.395l-1.087-1.087-.336-.019-.098.098a1.616 1.616 0 00-.613.415l-.612.612.039.474.909.909a1.483 1.483 0 001.778-.79zM138.333 282.151l-.672-.672-.355-.039-.771.77a.697.697 0 00-.158.158 1.462 1.462 0 000 1.186 1.378 1.378 0 00.81.809l.099.099h.276l.514-.513c.219-.116.398-.295.514-.514.044-.095.077-.194.098-.296l.099-.099-.059-.336-.257-.257-.138-.296zM100.445 249.044l.474-.474.178-.415.059-.257-1.205-1.204a1.84 1.84 0 00-.908.039l.06.336-.218-.217-.573.572-.119.316-.059.297.099.533.197.198a2.136 2.136 0 001.581.632l.434-.356zM155.302 242.506l.296-.06.296-.296c.227-.136.417-.326.553-.553a1.228 1.228 0 00-.572-1.6l-.139-.138-.217-.02a1.283 1.283 0 00-1.442.533l-.138.139-.06.375.02.217.059.296c.105.247.26.469.455.652l.296.099.593.356zM124.703 255.365l.356-.039.81-.81a1.45 1.45 0 00-.751-1.896l-.138-.139-.395-.039-.237.039-.929.929-.059.256-.099.099.059.375.119.119a1.68 1.68 0 001.067.79l.197.316zM142.462 259.652l-1.106 1.106-.02.375 1.067 1.067h.355l.119-.119.276-.079c.283-.123.508-.349.632-.632l-.335-.691-.257-.691-.277-.277-.454-.059zM113.542 322.271l-.138.138-.059.375-.099.099.711.711a1.4 1.4 0 001.608-.001 1.39 1.39 0 00.387-.414l.139-.138.059-.375-.81-.81a1.514 1.514 0 00-1.146-.198c-.281.124-.511.34-.652.613zM96.495 352.276a1.39 1.39 0 00.85-.849l.157-.158v-.237l-.296-.297a1.65 1.65 0 00-1.442-.493 1.305 1.305 0 00-.968.73 1.37 1.37 0 00-.06.929l.514.513-.217-.217a1.427 1.427 0 001.462.079zM96.297 318.498l.06-.257v-.316l-.08-.277-.553-.553a1.4 1.4 0 00-1.66.277l-.434.434-.059.336.06.296.236.514.198.197a1.422 1.422 0 001.817-.276l-.138-.138.118.118.435-.355zM115.182 311.347l-.276.276a1.314 1.314 0 00.138 1.284c.131.207.307.382.513.514a1.372 1.372 0 001.818-.632v-.395l-.178-.178a1.279 1.279 0 00-.217-.533l-.139-.139a1.161 1.161 0 00-.493-.256l-.277-.04-.889.099zM95.389 303.366l.158-.158c.195-.02.38-.095.533-.217l.198-.197.217-.455.04-.276-.04-.316-1.047-1.047-.375-.02-.85.849a1.399 1.399 0 00.316 1.62l.178.178.474.118.198-.079zM111.586 310.912l.613-.612a1.28 1.28 0 00.079-.474l.514-.514a1.377 1.377 0 00-.04-1.185 1.235 1.235 0 00-.928-.652l-.099-.099-.316.04-.968.968a.697.697 0 01-.158.158l-.415.415a1.233 1.233 0 00-.038.964 1.228 1.228 0 00.67.695c.233.115.499.143.751.079.097.093.211.167.335.217zM137.464 305.184a1.286 1.286 0 00-1.146.79 1.682 1.682 0 00-.118.474l.553.553a1.317 1.317 0 001.185.316 1.444 1.444 0 001.027-.83 1.253 1.253 0 00-.513-1.619l-.277-.079-.711.395zM111.172 312.354l.138.139a1.371 1.371 0 001.067.711 1.346 1.346 0 001.165-.534l.158-.158.099-.414v-.238l-.059-.296-.593-.592-.375-.178-.158-.158-.316.039-.869.869-.257.81zM89.66 331.93c.039-.097.072-.196.099-.296l.099-.099-.04-.277a1.455 1.455 0 00-.533-1.284 1.317 1.317 0 00-1.6-.019l-.198.197-.177.415-.12.118-.019.376v.197l.158.158.099.297.573.573a1.318 1.318 0 001.66-.356zM89.423 315.021l.099.099h.316l.099-.099a1.313 1.313 0 00.849-.849c.051-.115.091-.234.118-.356l-.059-.336-.652-.652-.474-.197-.336-.02-.335.099-.415.415a1.283 1.283 0 00.73 1.758l.06.138zM126.599 329.323l-.217.019-.731.731a.394.394 0 00-.118.119 1.28 1.28 0 000 1.145 1.533 1.533 0 00.869.87l.099.098.335-.019.731-.731a1.394 1.394 0 00-.651-1.798l-.139-.138-.178-.296zM102.5 318.024l.336.335a1.359 1.359 0 001.225.158 1.471 1.471 0 00.809-.809l.099-.099-.02-.336-.276-.277a1.285 1.285 0 00-.869-.632l-.099-.099-.079-.316-.395.395a1.514 1.514 0 00-.514.514.885.885 0 00-.099.296l.04.316.237.238-.395.316zM99.004 328.987a1.43 1.43 0 001.047.691l.336-.059.731-.731a1.37 1.37 0 00-.316-1.58l-.514-.514h-.276a1.282 1.282 0 00-.751.751 1.23 1.23 0 00-.257 1.442zM128.16 304.927l-.277.277a1.377 1.377 0 000 1.145c.155.272.38.498.652.652l.395.395.395.079.218.02.316-.079.138-.138a1.58 1.58 0 00.435-.435l.138-.138.079-.395.02-.218-.06-.335-.138-.139a1.299 1.299 0 00-.454-.454l-.277-.277c-.346-.2-.752-.27-1.146-.197l-.138.138-.296.099zM116.604 299.534l-.099-.099-.118.119a1.393 1.393 0 00-.819 1.451c.035.252.141.488.305.682a1.476 1.476 0 001.422.514l.771-.771a1.428 1.428 0 00-.02-1.205 1.338 1.338 0 00-.869-.869l-.573.178zM106.154 296.512a1.368 1.368 0 001.067.75h.237l.296-.296c.294-.124.528-.358.652-.652.075-.209.108-.43.099-.652l-.731-.73a1.431 1.431 0 00-1.185-.198 1.265 1.265 0 00-.83.83 1.316 1.316 0 00.395.948zM86.54 298.547l.355-.04L88 297.401v-.395l-1.126-1.126-.415.02-.948.948a1.396 1.396 0 00.928 1.758l.1-.059zM118.955 328.039a1.359 1.359 0 00-1.055-.059 1.373 1.373 0 00-.783.71l-.039.277.395.395c.076.163.183.308.316.429a1.36 1.36 0 00.987.343c.179-.012.354-.059.514-.14l.632-.632a1.425 1.425 0 00-.967-1.323zM109.671 293.233a1.481 1.481 0 001.303.987 1.307 1.307 0 001.304-.75l.119-.119-.099-.098a1.396 1.396 0 00-.79-.791 1.643 1.643 0 00-1.264.04l-.79.79-.001.356.218-.415zM131.024 312.571l-.315.04-.83.83a1.473 1.473 0 00-.04 1.264 1.456 1.456 0 00.83.829l.296-.019.395-.395c.219-.116.398-.295.514-.514a.863.863 0 00.099-.296l-.02-.297-.375-.375a1.597 1.597 0 00-.909-.553l.355-.514zM111.192 296.729l-.099.375v.238l.079.316.119.118c.116.195.279.358.474.474l.138.138.395.079.099.099.336-.059.138-.138c.189-.109.346-.266.454-.455l.139-.138.098-.375-.118-.119-.04-.316-.118-.118a1.477 1.477 0 00-.494-.494l-.138-.139-.376-.098-.217.019-.316.04-.119.118a1.271 1.271 0 00-.474.475l.04-.04zM110.026 300.818a.835.835 0 00-.178.178 1.508 1.508 0 00.02 1.205 1.557 1.557 0 00.987.79h.238l.474-.474c.181-.118.336-.273.454-.454a1.344 1.344 0 00-.296-1.521l-.573-.573-.296.019-.83.83zM114.135 297.144l-.079.395-.099.099.02.257c.099.272.275.51.507.685.231.176.508.28.797.302a1.396 1.396 0 001.304-.79l.118-.118v-.554l-.099-.335-.059-.336-.118-.119a1.373 1.373 0 00-.475-.474l-.138-.138-.395-.079-.217.02-.336.059-.118.118a1.727 1.727 0 00-.455.455l-.158.553zM122.747 300.305l-.079.395-.02.217.079.316.139.138c.108.189.265.346.454.455l.138.138.376.099.118-.119.336-.059.119-.119c.18-.104.329-.254.434-.434l.158-.158.079-.395v-.237l-.079-.316-.118-.119a1.31 1.31 0 00-.455-.454l-.158-.158-.395-.079-.099.098-.355.079-.119.119a1.75 1.75 0 00-.454.454l-.099.139zM101.552 298.863l.355-.04.771-.77.178-.376.118-.118v-.316a1.534 1.534 0 00-.283-1.766 1.538 1.538 0 00-.527-.348l-.099-.099-.335-.019-.593.592a2.806 2.806 0 00-.336.336 1.437 1.437 0 00-.137.569c-.007.197.026.393.098.577a1.354 1.354 0 00.069 1.085 1.367 1.367 0 00.839.693h-.118zM101.769 303.623l-.632.632a1.353 1.353 0 00.099 1.166 1.45 1.45 0 001.343.75 1.34 1.34 0 001.146-.79 1.311 1.311 0 000-1.027 1.446 1.446 0 00-.314-.476 1.446 1.446 0 00-.476-.314l-.099-.099a1.427 1.427 0 00-1.067.158zM120.278 319.92l-.118.276-.04.237v.356l.02.217.099.218.099.177.276.277.158.118.138.06.218.059.375-.02.217-.019.218-.099.177-.099.277-.277.118-.158.139-.296.039-.197-.02-.376-.019-.217-.099-.217-.099-.178-.276-.277-.158-.118-.139-.059-.217-.06-.375.02-.218.02-.217.099-.178.098-.276.277-.119.158zM83.576 316.246l.178-.178c.177-.108.326-.257.435-.435l.158-.158.099-.375v-.237l-.08-.356-.158-.158a1.37 1.37 0 00-.888-.651l-.316.039-1.047 1.047-.02.375 1.047 1.047.592.04zM83.577 301.272l.118-.118.02-.375-.81-.81a1.424 1.424 0 00-2.015.513l-.04.356.653.652a1.338 1.338 0 001.856.069c.172-.152.3-.346.376-.563l-.158.276zM61.71 312.433a1.282 1.282 0 00-.02-1.165 1.309 1.309 0 00-.969-.731l-.335.059-.415.415c-.19.108-.346.265-.455.454a1.343 1.343 0 00-.02 1.008l.534.533a1.367 1.367 0 001.166.178 1.37 1.37 0 00.513-.751zM111.507 327.031l.04-.316-.889-.889-.375-.059-.869.869-.04.277.415.415a1.477 1.477 0 001.284.493c.221-.214.372-.489.434-.79zM80.673 321.717a.851.851 0 00.197-.197l-.098-.099-.04-.316-.968-.968-.336-.02-.276.277a1.365 1.365 0 00-.474.474l-.158.158-.06.494.04.276.77.771a1.315 1.315 0 001.403-.85zM112.278 304.611l1.64 1.639.415-.019.81-.81a1.206 1.206 0 00-.02-1.126l-.178-.178-.02-.296-1.106-1.106-.434-.198-.139-.138-.335.019-.751.751a1.431 1.431 0 00.059 1.324l.059.138zM96.653 308.344l.85-.849c.137-.356.137-.75 0-1.106a1.451 1.451 0 00-.83-.83l-.1-.099h-.355l-.85.85a1.394 1.394 0 00.02 1.126 1.372 1.372 0 00.81.809l.1.099h.355zM135.291 342.953l.336.335a1.385 1.385 0 001.086.099c.347-.147.623-.423.771-.77l.079-.277.098-.099-.039-.355-.711-.711-.376-.178-.296-.059-.316.039-.83.83a1.258 1.258 0 00.198 1.146zM114.076 244.066l.494.494a1.284 1.284 0 001.165.573c.251-.013.493-.102.691-.257l.731-.731a1.676 1.676 0 00.02-1.363l-.494-.494a1.428 1.428 0 00-.889-.177l-.217.019-.929.929a.723.723 0 00-.078.276l-.001.198-.493.533zM90.707 236.184l.336-.059.73-.731a1.395 1.395 0 00-.335-1.639l-.494-.494-.276.039a1.4 1.4 0 00-.75.751 1.286 1.286 0 00.039 1.106 1.308 1.308 0 001.027.711l-.277.316zM119.587 277.825l-.376.375-.217.455-.04.276.02.297.751.75c.302.222.568.489.79.79l.178.178.415.138.237.04.296-.02.751-.75a1.264 1.264 0 00-.06-1.205l-.474-.475-.197-.513-.632-.632a1.309 1.309 0 00-1.442.296zM124.447 279.326l.493.494a1.543 1.543 0 001.699.277l.218-.218.296-.493.059-.257.158-.158.099-.415v-.237l-.059-.296-.85-.85-.237-.039-.415.019-.237.04-1.165 1.165-.079.554.02.414zM114.095 274.348h.237l.83-.829a1.268 1.268 0 00-.099-1.126 1.28 1.28 0 00-.869-.632l-.099-.099-.375.059-.692.691a1.28 1.28 0 00.475 1.62l.158.158.434.158zM94.698 267.514v-.395l-.139-.336-.138-.139a1.224 1.224 0 00-.73-.73 1.323 1.323 0 00-1.027.276l-.416.415a1.256 1.256 0 00.252 1.387c.12.119.262.212.42.272a1.394 1.394 0 001.778-.75zM96.08 242.545l-.256-.257a1.293 1.293 0 00-.909.395 1.347 1.347 0 00-.375 1.008 1.313 1.313 0 001.323 1.323l.376-.177.493-.494.139-.376-.534-.533-.256-.889zM93.571 295.149l-.355.158-.494.494-.158.395a.92.92 0 00.079.474l.75.751.395.157a1.27 1.27 0 00.928-.376 1.28 1.28 0 00.376-.927 1.397 1.397 0 00-1.52-1.126z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M87.19 234.683l.1.375.276.277a1.314 1.314 0 001.146.593c.127-.001.254-.021.375-.06l.731-.731-.02-.177.099-.099-.02-.296-1.244-1.245a1.341 1.341 0 00-1.304 1.304l-.138.059zM98.47 259l.83.83c.14.045.287.071.435.078.377-.025.731-.195.987-.474a1.632 1.632 0 00.297-1.244l-.949-.948-.395-.04a1.344 1.344 0 00-1.323 1.324l.118.474zM97.996 232.253l.494.494.316.119.375.019a1.253 1.253 0 001.067-.553 1.483 1.483 0 00.079-1.343l-.79-.79a1.503 1.503 0 00-1.304.909l-.138.138-.02.454.158.158-.237.395zM99.576 271.603l-.256-.06a1.293 1.293 0 00-.988-.434 1.785 1.785 0 00-1.205.612l-.119.119-.138.336v.237l.474.474a1.427 1.427 0 001.027.434 1.4 1.4 0 00.81.534h.237l.376-.099.908-.909a1.696 1.696 0 00-.553-1.383l-.573.139zM91.834 284.68l-.455-.455h-.276a1.37 1.37 0 00-1.264.909 1.508 1.508 0 00.454 1.639l.395.395.296.02a1.367 1.367 0 001.245-.889 1.48 1.48 0 00-.395-1.619zM113.147 215.048l.119-.118.355-.119a1.27 1.27 0 00.5-.31 1.27 1.27 0 00.31-.5l-.099-.098a1.65 1.65 0 00-.632-.633l-.118-.118-.336-.099-.099-.099-.257.02-.474.474a1.59 1.59 0 00-.355 1.58 1.33 1.33 0 001.086.02zM79.408 263.642l-.671-.672-.297-.098-.098-.099a1.429 1.429 0 00-1.324 1.007l-.02.218.06.414.592.593c.158.138.348.233.553.277l.218-.02.395-.079.572-.573a1.48 1.48 0 00.238-1.067l-.218.099zM79.19 248.886a1.362 1.362 0 00-1.038.536 1.376 1.376 0 00-.245 1.143l1.086 1.087.138-.139.336-.138.474-.474.119-.356.118-.118v-.237l-1.106-1.107.119-.197zM101.157 264.175l-.119.119-.355.118-.395.395a1.51 1.51 0 00-.415.731l.099.099.059.375.691.692c.225.119.477.18.731.177l.277-.039.889-.889a1.43 1.43 0 00-.316-1.146 1.318 1.318 0 00-1.146-.632zM101.532 195.571l.474.474c.232.177.519.268.81.257l.494-.099.356-.355a1.315 1.315 0 00.256-1.561l-.829-.829a.974.974 0 00-.336-.02 1.394 1.394 0 00-1.343.987l-.139.139.06.573.197.434zM77.433 256.886l.158-.158-.02-.454-1.066-1.067h-.237a1.4 1.4 0 00-1.324 1.324l.356.355a1.346 1.346 0 001.007.415 1.406 1.406 0 001.126-.415zM76.663 225.774c.147-.243.21-.527.178-.809l-.85-.85-.375-.099-.099-.098a1.394 1.394 0 00-1.284.849l-.158.158.02.454.553.553c.174.203.41.342.672.395l.256.02.455-.138.553-.553.079.118zM78.52 259.237c-.14.006-.28.025-.415.059l-.395.395a1.401 1.401 0 00-.376 1.324 1.54 1.54 0 001.047 1.047l.257.019.395-.395a1.451 1.451 0 00.316-1.619 1.34 1.34 0 00-.83-.83zM67.22 223.364l.119.316.81.81a1.407 1.407 0 001.027-.474 1.365 1.365 0 00.316-1.106l-.77-.77-.376-.06a1.314 1.314 0 00-1.205 1.205l.118.119-.039-.04zM81.147 236.935l-.316.316-.119.316-.217.217.533.534a1.485 1.485 0 001.126.533 1.423 1.423 0 001.383-1.383l-.04-.237-.177-.415-.494-.493a1.538 1.538 0 00-.731-.178l-.138.138-.356.158-.454.494zM44.129 270.911l-.356.158-.415.415a1.145 1.145 0 00-.217 1.008 1.314 1.314 0 00.73.731l.159.158.375.138a1.26 1.26 0 001.264-1.264 1.37 1.37 0 00-.316-.83 1.309 1.309 0 00-1.224-.514zM72.731 259.454l.336-.138a1.282 1.282 0 00.85-1.442l-.416-.415a1.48 1.48 0 00-.967-.533l-.356.355a1.595 1.595 0 00-.296 1.679 1.287 1.287 0 001.264.83l-.415-.336zM73.364 249.577l-.297-.296-.375-.138a1.425 1.425 0 00-1.422 1.422c.005.386.161.755.434 1.027a1.398 1.398 0 001.027.395l.356-.158.158-.158a1.254 1.254 0 00.632-.948 1.23 1.23 0 00-.395-1.146h-.118zM70.815 254.891l-.691-.691c-.247.02-.489.087-.711.197l-.573.573a1.37 1.37 0 00.06 1.403 1.315 1.315 0 001.125.612 1.261 1.261 0 00.89-.415.81.81 0 00.256-.257 1.253 1.253 0 00.197-1.185l-.553-.237zM67.378 260.501l.138-.533v-.316l-1.007-1.008-.415.02a1.423 1.423 0 00-1.383 1.383c-.106.301-.1.631.02.928a1.224 1.224 0 00.79.79l.1.099.296-.02.572-.573c.143-.204.25-.431.316-.671l.573-.099zM77.848 285.746a1.37 1.37 0 00-1.126.652 1.34 1.34 0 000 1.304l.77.77h.237l1.186-1.185a1.341 1.341 0 00-.06-1.284 1.369 1.369 0 00-1.007-.257zM50.687 223.503l.375-.099.494-.494c.16-.179.269-.397.316-.632l-.119-.119-.098-.375-.514-.514a1.154 1.154 0 00-.593-.276h-.197l-.395.079-.514.514a1.613 1.613 0 00-.276.592l.118.119.08.395.513.513c.155.141.347.237.553.277l.257.02zM65.66 228.797l.395.395c.174.119.368.206.572.256l.277.001.454-.139.573-.573.099-.375.118-.118a1.285 1.285 0 00-1.323-1.324.61.61 0 00-.316.04l-.731.73-.099.336.119.119-.139.652zM56.435 243.039l-.217.217-.237.04-.731.731-.119.355a1.338 1.338 0 00.838 1.266c.167.067.346.1.525.097a1.308 1.308 0 001.225-1.027 1.255 1.255 0 00-1.284-1.679zM65.6 243.789l.79.791h.277c.221.006.44-.042.637-.142.197-.101.365-.249.49-.431a1.425 1.425 0 00.138-1.245l-1.205-1.205h-.237l-.83.83-.138.375-.119.119v.474l.119.118.079.316zM97.364 236.836l.118.119a1.287 1.287 0 001.245.77 1.396 1.396 0 001.304-.948 1.427 1.427 0 00-.376-1.6l-.138-.138-.395-.119-.119-.118-.276.039-.119.119a1.453 1.453 0 00-.79 1.383l-.454.493zM43.417 207.384l-.02-.257-.158-.435-.335-.335a1.473 1.473 0 00-1.126-.534l-.257.02-.711.711a1.452 1.452 0 00-.139 1.047l.83.83c.198.065.404.098.612.098a1.368 1.368 0 001.304-1.145zM144.615 227.039a1.32 1.32 0 001.185-.87l.02-.296-.099-.375a1.425 1.425 0 00-.395-.83 1.293 1.293 0 00-.948-.355l-.237.237a1.252 1.252 0 00-.652 1.481 1.503 1.503 0 001.126 1.008zM78.4 308.068a1.37 1.37 0 00-1.421-1.225 1.34 1.34 0 00-.514.079l-.435.434a1.29 1.29 0 00-.237 1.304l.731.731.119-.118-.08.316c.218.122.462.19.712.197l1.007-1.007v-.356l.119-.355zM57.422 201.952l.75.75a1.421 1.421 0 001.364-1.363 1.287 1.287 0 00-1.126-1.363l-.296.059-.178.178a1.398 1.398 0 00-.573 1.719l.06.02zM74.292 236.757a1.227 1.227 0 00-1.106.672l-.06.257-.02.454-.098.099.968.968.375.099a1.29 1.29 0 001.324-1.047l-.04-.356-.316-.316a1.34 1.34 0 00-1.027-.83zM96.06 236.876l-.395-.198-.237-.237a1.624 1.624 0 00-1.264.593l-.217.217c-.312.236-.535.57-.632.948a1.872 1.872 0 00.651 1.916l.139.139.355.118.119.119.237-.04.533-.533.158-.158a1.43 1.43 0 00.79-.79l.317-.316a1.429 1.429 0 00-.317-1.106l-.414-.415.177-.257zM124.249 280.196l-.632-.633-.316-.079-.119-.118-.217.02a1.29 1.29 0 00-.529.155 1.29 1.29 0 00-.419.358 1.232 1.232 0 00-.178 1.047l.099.099a1.228 1.228 0 00.968.968l.098.099.336-.099.672-.672c.173-.32.222-.693.138-1.047l.099-.098zM108.268 267.217l.375-.375a1.473 1.473 0 00-.395-1.185 1.251 1.251 0 00-1.047-.415l-.711.711a1.259 1.259 0 00-.217 1.086l.79.791c.228.112.477.172.731.177l.296-.059.316-.316-.138-.415zM89.186 256.59l-1.086 1.086-.08.277.06.494.099.296.217.217c.099.161.234.297.395.395l.316.079.553-.039.297-.099.454-.454c.218-.291.336-.644.336-1.008l-.89-.889a1.315 1.315 0 00-.67-.355zM89.443 266.921l.119.119a1.37 1.37 0 001.027-.475 1.556 1.556 0 00.316-1.224l-1.857-1.857h-.237l-.138.138-.375.178-.494.494-.119.395a1.396 1.396 0 00.514 1.027c.053.308.19.595.395.83l.612.612.237-.237zM70.815 237.548l.099-.099.099-.297a1.445 1.445 0 00-.426-1.036 1.453 1.453 0 00-1.036-.425l-.119.118-.375.138-.494.494-.138.415-.158.158.85.85c.242.172.531.268.829.276.196-.018.39-.065.573-.138l.612-.612-.316.158zM91.142 243.296a.507.507 0 00-.237-.04 1.451 1.451 0 00-1.324 1.047l-.059.296v.198l.02.257.553.553c.29.216.647.321 1.007.296l.277-.039.908-.909a1.646 1.646 0 00-.335-1.284l-.376-.375h-.434zM93.177 253.785h-.316l-.514.158-.178.178a1.618 1.618 0 00-.355.355c-.179.063-.35.142-.514.237l-.79.791a1.7 1.7 0 00.474 1.382l.336.336a1.349 1.349 0 00.928.534l.277-.04.434-.435c.29-.161.53-.401.692-.691l.02-.296.177-.178.06-.336a1.477 1.477 0 00-.252-1.653 1.482 1.482 0 00-.48-.342zM92.07 248.491l-.928.928a1.455 1.455 0 001.244 1.64l.454-.454a1.456 1.456 0 00.297-1.561 1.342 1.342 0 00-1.284-.81l.217.257zM86.599 241.162l.83.83h.236a1.37 1.37 0 001.304-1.304v-.197l-.099-.336-.671-.672a1.427 1.427 0 00-.75-.158l-.297.02-.573.573-.099.296-.099.099v.237l.119.119.099.493zM106.431 290.507a1.454 1.454 0 00-1.047.494 1.703 1.703 0 00-.296 1.323l.494.494a1.62 1.62 0 001.047.375 1.417 1.417 0 001.066-.513l.317-.317.059-.256a1.591 1.591 0 00-.395-1.225 1.392 1.392 0 00-1.245-.375zM95.39 218.979l-.14-.138-.315-.119-.119-.118-.217-.02a1.408 1.408 0 00-1.126.533 1.405 1.405 0 00-.296 1.126l.316.316a1.228 1.228 0 001.244.85c.17-.006.336-.04.494-.099l.613-.612c.082-.201.123-.416.118-.632l-.118-.119-.139-.336-.316-.316v-.316zM90.964 214.041l.889.889a1.818 1.818 0 001.6-1.008l.118-.118.04-.316-.573-.573a1.282 1.282 0 00-1.007-.375c-.312 0-.615.096-.87.276l.1-.099-.159.158-.158.356-.118.119v.237l.118.118.02.336zM96.692 245.804l-.256-.019a1.365 1.365 0 00-1.087.533 1.395 1.395 0 00-.296 1.205l.909.909h.276a1.404 1.404 0 001.087-.494 1.545 1.545 0 00.375-1.126l-1.008-1.008zM104.495 215.088l-.138.138a1.176 1.176 0 00-.612 1.205l.079.276.691.692.375.138a1.351 1.351 0 001.087-.612c.149-.252.23-.538.237-.83a1.367 1.367 0 00-1.343-1.343l-.376.336zM120.476 193.536l.553.554c.156.032.315.046.474.039a1.37 1.37 0 001.265-.869 1.542 1.542 0 00-.395-1.699l-.297-.296-.257-.059-.296-.02a1.473 1.473 0 00-1.185.632 1.616 1.616 0 000 1.541l.138.177zM94.46 259.375l.969.968a1.341 1.341 0 001.442-.573 1.618 1.618 0 00.118-1.501l-.711-.711-.296-.059h-.277l-.454.177-.198.198a1.615 1.615 0 00-.592 1.501zM144.832 235.947l-.375.178-.198.198a1.305 1.305 0 00-.395.395l-.158.158.099.454.177.376.494.493.376.139c.34-.019.662-.159.908-.395.227-.265.352-.601.356-.949a1.401 1.401 0 00-1.284-1.047zM120.001 219.019a1.509 1.509 0 00-1.461 1.264l-.02.257.079.434.494.494.415.178.158.158.316-.04.217-.019.79-.79.099-.336-.198-.198.297.02.019-.494-1.086-1.086-.119.158zM92.623 232.332l.119.119h.237l.118-.119.356-.158.454-.454.158-.355.02-.218v-.237l-.118-.118-.139-.336-.81-.81a1.36 1.36 0 00-1.301.994c-.055.19-.069.39-.042.586l.81.81.138.296zM103.903 204.302a1.28 1.28 0 00-1.086 1.086l.039.277.198.474a1.254 1.254 0 001.185.751 1.06 1.06 0 00.474-.119l.632-.632.079-.276a1.285 1.285 0 00-1.027-1.383h-.237l-.257-.178zM46.558 258.881l-.356-.355-.336-.099-.118-.119h-.316l-1.166 1.166a1.453 1.453 0 00.83 1.383l.158.158.533-.02.79-.79c.127-.243.181-.517.159-.79l-.139-.139-.04-.395zM112.772 279.741l-.316-.039-.513.079-.692.691-.138.375-.119.119a1.417 1.417 0 00.317 1.145c.259.255.604.403.967.415l.692-.691a1.425 1.425 0 00.395-1.225 1.507 1.507 0 00-.593-.869zM96.337 275.83l-.455-.454-.079-.316a1.48 1.48 0 00-1.382-1.107l-.119.119-.296.099-.75.75c-.115.478-.037.982.217 1.403.12.238.314.432.553.553l.118.118a1.252 1.252 0 001.205.731c.12.007.24-.007.356-.039l.73-.731.1-.296v-.238l-.139-.138-.06-.454zM51.813 239.523l.335-.02.336-.138.79-.79a1.458 1.458 0 00-.355-1.107 1.569 1.569 0 00-1.106-.434l-.218.02-.79.79-.118.355-.1.099v.237l.83.83.356.158h.04zM76.623 189.625l.237-.039.435-.158.494-.494c.166-.174.276-.395.316-.632l-.06-.336-.81-.81-.454-.059a1.396 1.396 0 00-1.383 1.383 1.37 1.37 0 001.225 1.145zM92.86 207.127a1.315 1.315 0 00.85.849l.217-.019.395-.079.573-.573a1.317 1.317 0 00.217-1.047l-.83-.83-.375-.098a1.255 1.255 0 00-1.225 1.224l.119.119.06.454zM117.533 205.803l.415.415c.181.123.381.217.592.277l.257-.02.474-.158.573-.573.138-.375.119-.119a1.335 1.335 0 00-.393-.95 1.334 1.334 0 00-.95-.393l-.297.059-.75.751-.099.296-.099.099.119.119-.099.572zM74.963 246.2l.652-.652a1.401 1.401 0 00-.06-1.363 1.309 1.309 0 00-1.145-.632 1.193 1.193 0 00-.968.493l-.257.257-.098.296a1.34 1.34 0 00.375 1.008c.277.261.646.403 1.027.395l.474.198zM75.102 279.524l.296-.296.198-.395c.086-.229.14-.468.158-.712l-.099-.296-.632-.632-.336-.099-.336-.02-.553.158.1.099-.12-.118a1.562 1.562 0 00-.83 1.501l1.186 1.185h.316l.119-.118.335-.139.198-.118zM70.242 211.058l.415.415c.172.138.376.233.593.276l.217.02.415-.099.395-.395.296.099h.237l.119-.119h.237l.83-.829a1.342 1.342 0 00-.158-1.225 1.283 1.283 0 00-1.107-.514 1.376 1.376 0 00-.968.415l-.296-.098-.099-.099a1.393 1.393 0 00-1.323.968l-.04.237.06.415.177.533zM74.252 214.238a1.59 1.59 0 00.376 1.166c.292.288.677.463 1.086.493l.237-.039.474-.474c.245-.38.337-.839.257-1.284l-.77-.77.059-.099-.119-.119-.395.04-.118.118-.317.119-.69.691-.08.158zM105.957 233.3a1.454 1.454 0 00-1.442 1.442 1.281 1.281 0 00.415.929 1.344 1.344 0 00.948.395l.375-.178.494-.494.138-.375-.099-.099-.158-.355-.494-.494-.177-.771zM108.012 272.057l-1.067-1.067h-.237l-.138.138-.376.178-.494.494-.118.395-.02.217a1.254 1.254 0 001.264 1.265 1.227 1.227 0 001.107-.751l.158-.514-.079-.355zM89.482 225.597l.356-.356a1.48 1.48 0 00.296-1.6 1.43 1.43 0 00-1.343-.83 1.077 1.077 0 00-.435.079l-.573.573a1.538 1.538 0 00.06 1.324 1.561 1.561 0 001.639.81zM84.326 210.801l.81.81h.277a1.538 1.538 0 001.52-1.521l-.137-.138-.119-.435-.75-.751-.317-.039-.256.02-.77.77a3.019 3.019 0 00-.238.988l.02.217-.04.079zM88.791 267.138l-.869-.869a1.094 1.094 0 00-.355-.039 1.476 1.476 0 00-1.343 1.027l-.04.474.04.237.671.672c.223.125.476.186.731.177l.277-.039.533-.533a1.569 1.569 0 00.276-1.265l.08.158zM87.21 227.868l-.118-.118-.119-.356-.414-.415a1.456 1.456 0 00-.672-.395h-.237l-.435.119-.81.81-.04.237v.237l.91.908.414.099a1.366 1.366 0 001.521-1.126zM94.025 265.617a1.259 1.259 0 00-.968-.375l-.355.158-.435.435a1.06 1.06 0 00-.237.671l-.849.85c-.017.406.125.802.395 1.106l.691.691a1.343 1.343 0 001.225-.908l.02-.257-.04-.277a1.279 1.279 0 00.435-1.027 1.51 1.51 0 00.118-1.067zM61.274 218.288l-.197-.198-.099-.375a1.311 1.311 0 00-1.087-.691l-.355.118-.652.652-.138.336-.138.138c.009.331.112.653.296.928l.099.099a1.37 1.37 0 001.284 1.047l.276-.276a1.54 1.54 0 00.711-1.778zM81.403 200.371l.415.415c.154.136.336.237.534.296h.236l.415-.098.573-.573a1.432 1.432 0 00.218-1.087l-1.087-1.086a1.429 1.429 0 00-1.323 1.007l-.02.218.079.434-.04.474zM27.693 259.02l-.276.039-.89.889a1.425 1.425 0 001.443 1.679l.257-.02.869-.869a1.399 1.399 0 00-.356-1.185 1.429 1.429 0 00-1.047-.533zM109.592 277.983l-.139-.138a1.316 1.316 0 00-.987-.751l-.218.02-.375.099-.553.553c-.198.31-.275.683-.217 1.047l.849.849.375.059a1.284 1.284 0 001.205-1.205l-.099-.098.159-.435zM68.88 266.743h-.277l-.494.099-.592.593-.138.375-.02.217a1.288 1.288 0 00.81 1.234c.163.064.338.095.513.09a.62.62 0 00.316-.04l.613-.612.118-.316.099-.099a1.398 1.398 0 00-.948-1.541zM114.787 228.342c-.13.093-.243.207-.336.336a1.25 1.25 0 00-.432.635c-.075.251-.069.52.017.768a1.285 1.285 0 001.264.829 1.233 1.233 0 001.067-.671l.119-.119.138-.138v-.198l-1.205-1.205-.356-.039-.276-.198zM112.159 246.812a.995.995 0 00-.395.079l-.592.593a1.527 1.527 0 00.099 1.323 1.448 1.448 0 001.165.691l.217-.019.237-.237a1.512 1.512 0 00.277-1.62 1.443 1.443 0 00-.409-.523 1.443 1.443 0 00-.599-.287zM76.385 271.247l-.316.316c-.178.333-.227.72-.138 1.087l.909.908.316.04a1.374 1.374 0 00.02-2.746l-.119.119-.296.098-.138.139-.238.039zM131.103 272.906l-.731-.731h-.276a1.355 1.355 0 00-.951.393 1.334 1.334 0 00-.393.951 1.367 1.367 0 001.343 1.343l.139-.138.395-.158.513-.514c.14-.246.202-.528.178-.81l-.099-.099-.118-.237zM63.645 250.052l.514.513.395.158.138.138a1.585 1.585 0 001.284-.849l.158-.158-.02-.454-1.007-1.008h-.237a1.343 1.343 0 00-1.304 1.304l.118.118-.04.238zM43.871 248.017l-.494.494-.138.375-.138.138v.316l.85.85.394.158c.35-.02.68-.167.929-.415a1.4 1.4 0 00.232-1.516 1.395 1.395 0 00-1.299-.815l-.336.415zM58.983 269.331l-.85.849-.118.356-.118.118.02.218.454.454c.17.178.39.302.632.356l.118.118.277.04h.237l.257-.06.849-.849.079-.356-.02-.217a1.452 1.452 0 00-1.817-1.027zM78.895 229.192l-.1.296-.098.099v.316l1.245 1.244a1.473 1.473 0 001.086-.494 1.364 1.364 0 00.316-1.106l-.75-.75a1.283 1.283 0 00-.731-.178l-.277.039-.632.632-.06-.098zM64.929 277.529l-.534-.534-.454-.02-.988.988-.118.435-.02.257c.066.299.217.574.435.79l.237-.237-.217.217.533.533.336-.019.75-.751s.376-1.047-.079-1.699l.119.04zM35.792 230.614l-.553-.553a1.475 1.475 0 00-.77-.217 1.229 1.229 0 00-1.185 1.185l.02.217.157.158a1.287 1.287 0 001.087 1.087l.217-.02.376-.099.355-.355c.19-.184.327-.416.395-.672l-.02-.296-.079-.435zM60.386 252.955l-.672-.671-.356-.119a1.482 1.482 0 00-1.461 1.462l.118.118.138.376.455.454.336.138.138.139h.197l.237-.04.514-.514a1.73 1.73 0 00.296-1.283l.06-.06zM56.751 234.98l.455.454.355.158.119.118h.237l.118-.118.356-.158.454-.454.138-.376.119-.118-.79-.79a1.283 1.283 0 00-.87-.277l-1.066 1.067v.237l.118.118.257.139zM125.691 234.624l.118-.119.376-.098.849-.85a1.395 1.395 0 00-1.225-1.62l-.217.02-.336.099-.849.849a1.502 1.502 0 00.277 1.146c.25.313.61.518 1.007.573zM160.102 237.39a1.365 1.365 0 00-1.126-.455l-.257.02-.751.751a1.618 1.618 0 00.158 1.501 1.232 1.232 0 001.916.217l.692-.691a1.51 1.51 0 00-.632-1.343zM129.621 251.869l-.256.059-.771.771-.079.276a1.344 1.344 0 00.929 1.363l.355.079h.277l.316-.079.79-.79a1.423 1.423 0 00-1.027-1.541l-.119.119-.415-.257zM115.123 237.804l.375.139a1.32 1.32 0 001.106-.632c.101-.232.161-.479.178-.731a1.371 1.371 0 00-1.363-1.363l-.375.177-.198.198a1.283 1.283 0 00-.395.395.256.256 0 00-.138.138 1.31 1.31 0 00-.04.553l.692.692.158.434zM116.308 259.099l-.178-.178a2.234 2.234 0 00-.415-.415l-.277-.079h-.513l-.909.909-.079.276v.316l1.205 1.205a1.394 1.394 0 001.324-.968l.039-.276-.099-.455-.098-.335zM116.288 241.261l-.513-.513a1.17 1.17 0 00-.455.098l-.375-.138-.099-.099a1.28 1.28 0 00-1.066.514 1.32 1.32 0 00-.218 1.126l.751.75.316.079.099.099.099-.099.513.514a1.287 1.287 0 001.284-1.284 1.141 1.141 0 00-.277-.711.895.895 0 00-.059-.336zM100.149 220.224a1.256 1.256 0 001.166.73c.175-.02.347-.06.513-.118l.652-.652a1.314 1.314 0 00-.217-1.205 1.424 1.424 0 00-1.186-.592 1.284 1.284 0 00-1.204 1.204l.039.277.237.356zM117.691 241.123l.316-.316c.142-.182.235-.398.27-.626a1.33 1.33 0 00-.073-.678 1.29 1.29 0 00-.889-.889h-.237l-.434.079-.533.534-.158.355-.119.119v.434l.573.573.454.257.158.158.139-.138.533.138zM144.575 252.205l-.296.059-.316.316a1.487 1.487 0 00-.968.968 1.31 1.31 0 00.652 1.481l.257.06h.474l.098.098 1.028-1.027.158-.355.118-.119a1.295 1.295 0 00-.283-1 1.321 1.321 0 00-.922-.481zM129.365 259.593l-.494-.494a1.364 1.364 0 00-1.086-.534 1.252 1.252 0 00-.375.06l-.87.869.04.553 1.027 1.027.257.02a1.372 1.372 0 00.968-.454 1.28 1.28 0 00.533-1.047zM126.599 219.848l.889.889h.198a1.365 1.365 0 001.027-.474 1.544 1.544 0 00.296-1.205l-.987-.988a1.372 1.372 0 00-1.344 1.344l.02.217-.099.217zM126.54 246.456l.198-.197a1.397 1.397 0 00-1.423-1.62l-.829.83a1.412 1.412 0 00-.218 1.086l.692.691c.22.131.475.193.731.178l.296-.02.593-.592-.04-.356zM137.958 244.994a1.333 1.333 0 00.198-1.224l-.949-.948a1.363 1.363 0 00-1.323.968l-.02.296.514.513a1.361 1.361 0 00.988.395c.194.049.398.049.592 0zM103.902 228.757l.376.178a1.337 1.337 0 001.027-.514 1.51 1.51 0 00.336-.849l.118-.356-.119-.118-.078-.395-.534-.534a1.063 1.063 0 00-.533-.256h-.237l-.376.098-.513.514c-.154.162-.251.37-.277.593v.197l-.138.139a1.477 1.477 0 00.316 1.106l.415.414.217-.217zM103.824 241.518l.533.533c.277.216.617.334.968.336a1.427 1.427 0 001.205-.77 1.586 1.586 0 00-.119-1.541l-.513-.514-.257-.019a1.59 1.59 0 00-1.086.533c-.137.165-.235.36-.287.568a1.442 1.442 0 00-.01.637l-.434.237zM105.522 252.264a1.394 1.394 0 00.217-1.284l-.474-.474a1.419 1.419 0 00-.889-.296 1.083 1.083 0 00-.632.197l-.75.751a1.426 1.426 0 00.276 1.185 1.35 1.35 0 001.087.454 1.308 1.308 0 001.165-.533zM115.656 269.193l-1.007-1.008-.336.02-.83.83-.118.355-.119.119v.237l.948.948.396.079h.276a1.357 1.357 0 00.929-.537 1.374 1.374 0 00.256-1.043h-.395zM128.654 227.315a1.393 1.393 0 001.383 1.383l.276-.079.533-.534c.115-.139.202-.3.254-.474.052-.173.07-.355.052-.535a1.383 1.383 0 00-.503-.926l-.375-.376a.872.872 0 00-.317-.039 1.397 1.397 0 00-1.303 1.58zM101.078 250.466a1.512 1.512 0 00.336-1.6 1.316 1.316 0 00-1.225-.869l-.652.652a1.454 1.454 0 00-.415 1.086 1.648 1.648 0 00.573 1.126l.119.119.355.118.099.099h.237l.356-.356.217-.375zM109.552 223.108l.909.908.119-.118a1.46 1.46 0 001.145-.514 1.405 1.405 0 00.297-1.126l-.83-.83a1.548 1.548 0 00-.731-.177l-.277.039-.691.692a1.587 1.587 0 00-.138 1.047l.197.079zM103.587 247.602l.395-.079.513-.514c.148-.166.244-.372.277-.592l.02-.218-.099-.375-.514-.513a1.282 1.282 0 00-.573-.297l-.217.02-.395.079-.494.494c-.16.178-.269.397-.316.632v.198l.099.375.494.494c.187.13.394.23.612.296h.198zM107.597 247.187h.276a1.389 1.389 0 001.087-.494 1.475 1.475 0 00.276-1.224l-.77-.771a1.797 1.797 0 00-.632-.118 1.422 1.422 0 00-1.264.908l-.04.238v.395l1.067 1.066zM102.717 244.758l.395-.079.435-.435a1.48 1.48 0 00.316-1.62 1.347 1.347 0 00-1.245-.849l-.118.118-.336.099-.691.691a1.096 1.096 0 00-.277.593l-.02.217.099.376.514.513c.163.146.36.248.573.297l.355.079zM101.927 235.631l.415-.098.514-.514c.148-.166.244-.372.276-.593l-.079-.395-.533-.533a2.252 2.252 0 00-.553-.276l-.217.019-.395.079-.514.514c-.142.18-.243.389-.296.612l.019.218.079.395.455.454c.174.119.368.206.573.257l.256-.139zM109.592 255.424l-.652-.652-.336-.138-.099-.099-.414.02-.336.336a1.554 1.554 0 00-1.143.458 1.557 1.557 0 00-.457 1.142l.928.928.494.06a1.648 1.648 0 001.067-.435 1.373 1.373 0 00.955-.524 1.363 1.363 0 00.269-1.056l-.276-.04zM113.819 253.212l.474.474a1.282 1.282 0 001.027-.593 1.445 1.445 0 00.099-1.521 1.337 1.337 0 00-1.205-.691 1.21 1.21 0 00-.928.415 1.427 1.427 0 00-.316 1.067l.118.118a1.203 1.203 0 00.731.731zM120.792 229.567l.297-.02.197-.039.356-.158.158-.119.178-.177.118-.158.138-.376.04-.197v-.158l-.079-.198-.138-.336-.099-.178-.198-.158-.178-.138-.355-.118-.198-.04h-.316l-.197.04-.356.158-.158.118-.197.198-.099.178-.138.335-.02.218v.158l.039.197.158.356.119.158.178.178.177.098.356.158.217.02zM132.921 264.491l-.415-.415a1.597 1.597 0 00-.553-.276h-.237l-.376.099-.592.592a1.27 1.27 0 00-.226.499c-.037.181-.034.368.009.548l.77.771.296.098.119.119h.237l.849-.849.119-.356.118-.119-.118-.118v-.593zM119.409 270.694l-.118.118-.356.158-.454.455-.138.336-.119.118a1.447 1.447 0 00.325 1.098 1.456 1.456 0 001.018.522l.099-.099.316-.118.79-.791a1.387 1.387 0 00-.197-1.303 1.426 1.426 0 00-1.166-.494zM138.768 285.786a1.417 1.417 0 00-1.067.513 1.395 1.395 0 00-.237 1.185l.79.791c.197.117.422.179.652.177.344-.034.666-.187.908-.434l.435-.435a1.415 1.415 0 00-.316-1.145 1.388 1.388 0 00-1.165-.652zM130.906 234.506l-.277.079-.711.711a1.293 1.293 0 00-.059.414l.118.119.099.375.869.87.237.039.277-.079.632-.632a1.452 1.452 0 00-.119-1.383 1.452 1.452 0 00-1.066-.513zM139.183 264.59l-.317.04-.75.75-.099.297-.118.118.73.731c.182.123.382.216.593.277l.257-.02.474-.158.553-.553.119-.356.019-.217a1.31 1.31 0 00-1.461-.909zM110.283 243.276l.949-.948.138-.375.118-.119v-.237l-1.224-1.225a1.511 1.511 0 00-1.028.514.787.787 0 00-.177.178l-1.265 1.264v.474l.06.257.75.75.277.079a1.397 1.397 0 001.146-.632l.256.02zM119.468 255.069a1.345 1.345 0 00-1.007.454 1.456 1.456 0 00-.356 1.106l1.166 1.166a1.28 1.28 0 001.047-.454 1.492 1.492 0 00.316-1.107l-1.166-1.165zM135.271 206.199l.316-.317a1.371 1.371 0 00-.355-1.027 1.408 1.408 0 00-1.027-.395l-.297.02-1.047 1.047v.434l.06.257.75.751.297.059a1.202 1.202 0 001.303-.829zM54.637 267.277l.415-.415a1.342 1.342 0 00.04-1.304 1.282 1.282 0 00-.514-.513l-.533-.534h-.237a1.678 1.678 0 00-1.265.553l-.296.297-.04.237c-.039.313.039.629.218.888l.85.85.256.059.296-.02.119-.118.691.02zM57.383 291.791l-.81-.81a1.452 1.452 0 00-1.343.988l-.02.296.415.415a1.377 1.377 0 00.988.395 1.35 1.35 0 001.027-.514 1.397 1.397 0 00.198-1.225l-.455.455zM82.904 248.017l.178.178.494.019.277-.079.77-.77.06-.257c.07-.374.182-.738.335-1.086l.04-.237-.02-.455-.534-.533-.414-.217-.178-.178a1.343 1.343 0 00-1.067.553l-1.205 1.205-.04.356a1.397 1.397 0 001.305 1.501zM82.253 242.861l.494-.494a1.563 1.563 0 00-.455-1.679l-.217-.217-.573-.059-.257.059h-.237l-.414.099-.534.533-.197.395-.04.237v.316l1.166 1.166.553.039.296-.059.277-.079.138-.257zM82.134 254.476l-.632-.632-.415-.138-.138-.138a1.331 1.331 0 00-.988.553 1.28 1.28 0 00-.198 1.066l.83.83.355.079a1.285 1.285 0 001.206-1.205l-.119-.118.099-.297zM84.01 274.882l-.355.158-.435.434a1.092 1.092 0 00-.217 1.008 1.35 1.35 0 00.75.75l.139.139.375.138c.34-.019.663-.159.909-.395.219-.251.332-.576.316-.909a1.4 1.4 0 00-.938-1.249 1.396 1.396 0 00-.544-.074zM79.645 198.297l-.395.04a1.279 1.279 0 00-.711.711 1.312 1.312 0 00.099 1.047 1.34 1.34 0 001.777.632l.277-.277.198-.671-.02-.415-.672-.672-.553-.395z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M100.228 246.772l-.296.297-.198.671.04.395a.97.97 0 00.336.336l.335.336.652.217.415-.019a1.26 1.26 0 00.711-.712 1.365 1.365 0 00-.099-1.007 1.395 1.395 0 00-1.896-.514zM68.228 195.097l.533.533a1.342 1.342 0 001.304.04c.103-.081.195-.174.276-.277l.553-.553v-.197l-.059-.297-1.007-1.007-.336.02-.257.256a1.43 1.43 0 00-.73 1.719l-.277-.237zM88.95 212.105l.217.217.513.237.316.04a2.62 2.62 0 00.415-.138 1.347 1.347 0 00.77-.771 1.647 1.647 0 00-.276-1.264l-.632-.632-.277-.04-.098.099a1.926 1.926 0 00-.356.118 1.326 1.326 0 00-.646.774 1.316 1.316 0 00.093 1.004l-.04.356zM76.9 188.203l.158.158.454.099.316-.04c.132-.027.258-.074.376-.138a1.258 1.258 0 00.691-.968 1.37 1.37 0 00-.494-1.244l-.573-.573-.276.039a1.428 1.428 0 00-.79 1.344l.158.158.177.414.159.158-.356.593zM95.408 222.989l-.257.059a1.322 1.322 0 00-1.086.06 1.73 1.73 0 00-.81 1.047l-.118.118.059.336.79.79a1.33 1.33 0 001.106-.04c.29.158.629.2.948.119l.909-.909.04-.395-.139-.138a1.65 1.65 0 00-1.047-1.047h-.395zM94.124 238.101l-.118-.119-.237.04-.178.177a1.313 1.313 0 00-.75 1.344 1.45 1.45 0 001.204 1.205l.119.118.099-.099.217-.019.197-.198a1.369 1.369 0 00.731-1.363 1.453 1.453 0 00-1.284-1.086zM83.062 166.118l.455-.454a1.256 1.256 0 00.335-1.086l-.474-.474a1.728 1.728 0 00-.79-.395h-.197l-.336.059-.79.79a1.676 1.676 0 00.395 1.58 1.227 1.227 0 001.402-.02zM73.798 224.569l-.513-.513h-.198l-.316.039-.099.099a1.484 1.484 0 00-.77 1.482l1.027 1.027a1.3 1.3 0 00.632 0l.968-.968a1.34 1.34 0 00-.257-1.087l-.474-.079zM67.16 211.374a1.315 1.315 0 00-.69.889 1.423 1.423 0 00.296 1.165l.71.711.317-.039.928-.928.02-.218-.02-.375-1.047-1.047-.098.099-.356.039-.06-.296zM93.59 215.601l-.73.731a1.061 1.061 0 00-.06.81l.85.849.336.06c.26.015.52-.039.75-.158a.992.992 0 00.218-.218l.652-.651-.02-.297-.099-.099a1.564 1.564 0 00-.83-.829 1.534 1.534 0 00-1.066-.198zM64.336 153.753l.395.197.139.139c.286.036.576-.019.83-.158a1.27 1.27 0 00.355-.356l.356-.355a1.314 1.314 0 00-.475-1.502l-.474-.474-.355-.039-.217.019a1.026 1.026 0 00-.317.119 1.422 1.422 0 00-.77 1.482l.553.553-.02.375zM69.077 219.335l-1.007-1.008-.435-.039-.257.02-.158.157a1.37 1.37 0 00-.71 1.699l.592.593a1.38 1.38 0 001.047-.059 1.402 1.402 0 00.928-1.363zM54.933 191.522a1.536 1.536 0 00-.217-.81l-.691-.692-.218-.019-.375.059-.138.138a1.453 1.453 0 00-.77 1.324l.059.256.889.889c.256.074.527.088.79.04l.869-.869.02-.257-.218-.059zM71.052 220.994c-.099.071-.186.158-.256.257l-.415.415a1.339 1.339 0 00.217 1.323 1.537 1.537 0 001.403.533l.79-.79a1.535 1.535 0 00-.435-1.62 1.316 1.316 0 00-1.303-.118zM45.334 193.576l.434.435.573.098.316-.039a1.394 1.394 0 00.79-.79 1.371 1.371 0 00-.178-1.126l-.671-.672-.257.02a.954.954 0 00-.356.118 1.372 1.372 0 00-.651 1.6l.256.257-.257.099zM63.763 199.759l-.158.158.04.356.83.829a1.558 1.558 0 001.264-.039 1.399 1.399 0 00.592-1.857l-.632-.632-.276-.079-.1-.099a1.17 1.17 0 00-.73.178l-.435.434-.395.751zM75.97 213.231a1.484 1.484 0 00-.038-.514c.048-.24.068-.486.06-.731a1.31 1.31 0 00.394-.592l.573.573.375-.02c.149-.042.294-.095.435-.158a1.255 1.255 0 00.652-1.442l-.988-.988-.237-.039-.375.019-.534.534-.316-.316a1.34 1.34 0 00-1.382-.079.191.191 0 00-.099.098.973.973 0 00-.336.02l-.652.652a.943.943 0 00-.118.356l-.08.395-.118.118c-.06.232-.073.474-.04.711l-.019.336.198.198a1.167 1.167 0 00-.12.908l1.028 1.028h.237l.435-.119.158-.158a2.08 2.08 0 00.415-.415c.193-.082.362-.211.493-.375zM45.156 246.417l-.277.276-.138.376-.158.158a1.087 1.087 0 00.257.967 1.167 1.167 0 00.948.435l.138.138.395-.039a1.264 1.264 0 00.63-.734 1.26 1.26 0 00-.077-.965 1.617 1.617 0 00-.632-.632 1.377 1.377 0 00-1.086.02zM65.956 223.68l.257-.256a1.288 1.288 0 00.158-1.66l-.573-.572a1.562 1.562 0 00-1.106-.04l-.573.573a1.62 1.62 0 00.434 1.659 1.315 1.315 0 001.522.178l-.119.118zM62.243 214.515h-.237l-.119-.119-.415.02a1.416 1.416 0 00-.7.844 1.425 1.425 0 00.108 1.092c.155.353.437.635.79.79.361.106.75.071 1.086-.099l.435-.434a1.29 1.29 0 00.118-1.106 1.23 1.23 0 00-.85-.85l-.216-.138zM62.242 220.401l-.533-.533-.316.04a1.65 1.65 0 00-.514.513l-.356.356a1.427 1.427 0 00.652 1.205 1.29 1.29 0 001.304.079c.31-.133.559-.381.691-.692.058-.106.105-.219.139-.335a1.453 1.453 0 00-.336-1.166l-.731.533zM61.59 226.94l-.099-.534-.651-.651-.534-.099-.316.039-.138.139a1.395 1.395 0 00-.711 1.817c.027.315.167.61.395.83a1.37 1.37 0 001.106.316l.889-.889c.04-.249.04-.502 0-.751l.06-.217zM81.996 245.133a1.31 1.31 0 00-.731 1.086 1.29 1.29 0 00.553 1.186l.296.296.257-.059.85-.85a1.342 1.342 0 00-.633-1.106 1.397 1.397 0 00-.592-.553zM30.478 200.845l.79-.79a1.619 1.619 0 00-.02-.691l-.908-.909a1.06 1.06 0 00-.652-.02l-.928.929a1.23 1.23 0 00.02.652l.928.928c.206.062.426.063.632 0l.138-.099zM46.163 199.225l.316.317c.201.053.414.046.612-.02l.909-.909.02-.217-.04-.395-.138-.139a1.322 1.322 0 00-.765-.669 1.32 1.32 0 00-1.013.077 1.219 1.219 0 00-.217.217l-.455.454-.04.238-.098.098.04.316.592.593.277.039zM44.128 215.996l-.889.889-.177.652.039.395a1.368 1.368 0 001.817.632 1.397 1.397 0 00.692-1.481 1.26 1.26 0 00-1.482-1.087zM52.78 212.717l.277.277h.355l.277-.277a1.42 1.42 0 00.77-1.007 1.315 1.315 0 00-.395-1.185l-.928-.929-.395.04-.988.987-.02.218.02.375 1.264 1.264-.237.237zM78.302 192.608l.138.138a1.425 1.425 0 001.482.139 1.455 1.455 0 00.75-1.423 1.425 1.425 0 00-1.026-1.264l-.119-.118-.395.039-.573.573a1.397 1.397 0 00-.138 1.6l.197.198-.316.118zM16.948 189.527l-.81-.81a1.365 1.365 0 00-1.264 0 .556.556 0 00-.178.177l-.612.613-.02.217a1.373 1.373 0 00.336 1.008l.69.691.258.02.118-.119c.213-.026.42-.086.613-.178a1.37 1.37 0 00.869-1.619zM116.624 163.254c.232-.118.422-.305.545-.534.124-.229.175-.491.146-.75l-.474-.474a1.314 1.314 0 00-.652-.652 1.2 1.2 0 00-.968.099l-.336.336a1.338 1.338 0 00.06 1.64 1.476 1.476 0 001.679.335zM92.189 265.005a1.397 1.397 0 00-1.817-.474 1.254 1.254 0 00-.356.355l-.415.415a1.315 1.315 0 00.336 1.284l.494.494.494.059-.099-.098.04.316c.25.013.501-.034.73-.139l.534-.533.118-.395.02-.217-.356-.356.277-.711zM27.18 178.544l.257.256h.316l.099-.098a1.457 1.457 0 00.632-1.818 1.313 1.313 0 00-1.6-.731l-.633.633a1.34 1.34 0 00.237 1.777l.692-.019zM57.442 202.603a1.29 1.29 0 00-.691 1.087l.06.257.197.434.869.869h.197l.395-.079a1.28 1.28 0 00.751-1.501l-.672-.672a1.38 1.38 0 00-1.106-.395zM77.137 193.221l-.415-.02-.237.039a.394.394 0 01-.119.119 1.64 1.64 0 00-.888 1.086 1.152 1.152 0 00-.178.178c-.197.341-.26.741-.178 1.126a1.928 1.928 0 001.442 1.442l.099.099.375-.06.85-.849s.098-.099.02-.217a1.51 1.51 0 00.374-1.087l.119-.118-.02-.257a1.334 1.334 0 00-.83-.83l-.197-.197-.217-.454zM121.365 219.986l-.336-.335-.237-.04-.355.04-.178.178a1.319 1.319 0 00-.613.888 1.236 1.236 0 00.277.988l.198.198a1.313 1.313 0 001.343.355l.829-.83.04-.315a1.539 1.539 0 00-.336-1.008l-.632-.119zM101.334 215.246l.139-.139-.02-.335a1.456 1.456 0 00-.869-.87 1.325 1.325 0 00-1.126.06 1.622 1.622 0 00-.138.138l-.613.612a1.344 1.344 0 00.257 1.087l.533.533.257.02a1.82 1.82 0 00.731-.178l.533-.533.316-.395zM79.546 213.962l-.691.691-.04.277.139.138.059.296.77.771c.158.088.333.142.514.158l.316-.04.474-.276.454-.455.06-.296c.066-.363.01-.738-.159-1.067l-.592-.592-.277-.04-.098-.099a1.344 1.344 0 00-.928.534zM84.189 223.147l.118-.118a1.315 1.315 0 00.692-.889 1.453 1.453 0 00-.237-1.225l-1.186-1.185-.237-.04h-.434l-1.067 1.067-.02.217.099.415.099.099a1.253 1.253 0 00.81.81c.153.273.379.498.652.652l.098.098.613.099zM54.697 204.796l-.08-.316a1.537 1.537 0 00-1.975-.632l-.75.75-.02.218.06.454 1.027 1.027c.294.057.6.016.869-.118.163-.111.304-.252.414-.415l.435-.435.02-.533zM75.517 201.122l-.158.158a1.395 1.395 0 00-.75 1.501l.059.297.869.869c.359.05.724-.012 1.047-.178a.824.824 0 00.197-.198l.672-.671v-.316l-.099-.099a1.685 1.685 0 00-.928-.928l-.119-.119-.79-.316zM81.877 209.695l-.75.751a1.027 1.027 0 00-.356.355c-.165.109-.306.25-.415.415l-.296.296v.356l.118.118a1.755 1.755 0 001.047 1.047h.316c.345.169.741.197 1.106.079l.692-.691c.18-.289.256-.63.217-.968l-.178-.178-.02-.296-.197-.198a1.537 1.537 0 00-1.284-1.086zM78.52 205.389l-.672.671v.316a1.395 1.395 0 001.797.929l.573-.573a1.45 1.45 0 00-.375-1.521 1.397 1.397 0 00-1.521-.178.81.81 0 00.197.356zM70.519 201.181l.395.395h.356l.276-.276a1.372 1.372 0 00.692-1.719l-.79-.79-.336-.059c-.252.017-.5.077-.731.177a1.018 1.018 0 00-.198.198l-.632.632.06.336.651.652.257.454zM109.789 237.014a1.381 1.381 0 00-.711.909 1.624 1.624 0 00.297 1.323l.493.494a1.506 1.506 0 001.857-1.027l.04-.277-.257-.256a1.574 1.574 0 00-.359-.55 1.574 1.574 0 00-.55-.359 1.397 1.397 0 00-.81-.257zM68.78 177.398l-.118-.119-.336.02-.256.257a1.588 1.588 0 00-.89.889 1.51 1.51 0 00.238 1.146l.336.335a1.282 1.282 0 001.481.218 1.04 1.04 0 00.356-.356l.474-.474v-.237l-.099-.099a1.343 1.343 0 00-.178-.652l-.513-.513-.494-.415zM62.697 174.869l.415.415.335-.02.119-.118a.394.394 0 00.118-.119 1.698 1.698 0 001.008-1.6l-1.008-1.007a1.478 1.478 0 00-1.086.099 1.617 1.617 0 00-.632.632l-.296.296-.02.217v.396l.691.691.356.118zM81.462 201.063a1.478 1.478 0 00-.849.849 1.454 1.454 0 00.277 1.225l.691.691.296-.02.178-.177a1.403 1.403 0 00.85-.85 1.563 1.563 0 00-.277-1.224l-.691-.692-.297.02-.178.178zM75.3 169.931l-.119.118a1.12 1.12 0 00-.04 1.344l.554.553.276.079.139.138.395-.039a1.289 1.289 0 00.71-1.028 1.62 1.62 0 00-.137-.849 1.34 1.34 0 00-1.818-.593l.04.277zM80.317 143.58l.237.237.217.02c.158-.023.311-.069.455-.139a1.39 1.39 0 00.75-1.343 1.54 1.54 0 00-1.106-1.383l-.118-.118h-.198l-.237.039c-.1.027-.2.06-.296.099a1.458 1.458 0 00-.77 1.087 1.591 1.591 0 00.651 1.402l.415.099zM85.472 214.159l.672.672h.276a1.49 1.49 0 00.317-.119 1.45 1.45 0 00.75-1.027 1.678 1.678 0 00-.573-1.402l-.177-.178-.474-.119h-.277l-.316.079-.711.711a1.54 1.54 0 00.513 1.383zM120.772 171.116l-.296.296-.079.277a1.168 1.168 0 00-.139.533l.04.277.711.711.316.079.296.099.395-.04c.311-.133.559-.38.692-.691a1.373 1.373 0 00-.099-1.008 1.338 1.338 0 00-1.334-.696 1.353 1.353 0 00-.503.163zM90.905 166.691a1.536 1.536 0 00-.75 1.778l.987.988.454-.02.237-.039.81-.81.06-.257v-.237l-.06-.336-.296.02.197-.198-.197-.434-.593-.593-.474-.079-.276.04s-.06.138-.1.177zM72.099 190.633l1.106-1.106-.04-.395-1.086-1.087-.375.02-.119.118a1.408 1.408 0 00-.79.79 1.459 1.459 0 00.158 1.146l.534.534.237.039.118.119.257-.178zM70.084 160.489a1.309 1.309 0 00-.513 1.462l.592.592a1.231 1.231 0 001.403.138c.14-.095.26-.215.355-.355l.553-.553-.098-.099-.08-.277-.098-.098a1.339 1.339 0 00-1.521-.81l-.158.158-.435-.158zM41.778 234.347h-.197l-.356.04-1.126 1.126.02.375.118.119a1.428 1.428 0 001.363.889l.297-.06.454-.257.415-.414.059-.297-.02-.256a1.561 1.561 0 00-.415-.652l-.612-.613zM110.836 224.589l-.276.079-.83.83-.02.257.06.414-.119.119a1.31 1.31 0 00.83.83 1.23 1.23 0 001.047-.06l.553-.553a1.43 1.43 0 00-.177-1.317 1.42 1.42 0 00-1.186-.599h.118zM94.322 228.184l-.139-.138-.256-.059-.237-.237a1.485 1.485 0 00-1.719-.376l-.73.731-.02.336c.096.484.38.91.79 1.185.233.134.501.196.77.178l.237.237a1.309 1.309 0 001.403.138c.087-.069.167-.149.237-.237l.592-.592-.059-.336-.711-.711-.158-.119zM38.44 214.811l1.007-1.007v-.356l-.118-.119a1.372 1.372 0 00-.81-.809 1.463 1.463 0 00-1.185.079l-.139.138-.513.514-.06.256.04.356.988.988.217.019.375-.019.198-.04zM39.17 159.165l.672-.671.099-.297c.076-.223.082-.464.02-.691l-.81-.81-.356-.079-.099-.099a2.258 2.258 0 00-.474.158 1.426 1.426 0 00-.652 1.837 1.371 1.371 0 001.6.652zM61.353 167.798a1.203 1.203 0 001.146.276l.593-.593.158-.355.138-.138a1.285 1.285 0 00-.237-1.027l-.534-.534-.256-.02h-.198l-.375.06-.099.098a1.315 1.315 0 00-.672 1.62l.435.435-.099.178zM83.003 155.906l.296.296c.21.075.438.082.652.02l.692-.691.098-.297.119-.118-.04-.395-.138-.138a1.306 1.306 0 00-.768-.691 1.307 1.307 0 00-1.03.078 1.009 1.009 0 00-.217.217l-.593.593.1.099.039.316.494.494.296.217zM62.223 210.781l.533-.533a1.371 1.371 0 00-.652-1.205 1.281 1.281 0 00-1.304-.079 1.175 1.175 0 00-.632.869l-.118.119.02.296.118.118a1.286 1.286 0 00.75.751c.174.061.359.085.542.071.184-.013.363-.064.526-.15l.217-.257zM76.78 240.748l.278-.277v-.435a2.94 2.94 0 00-.159-.711l-.632-.632-.335-.059-.336.059-.613.613.257.256-.355-.158a1.54 1.54 0 00-.08 1.699l.692.691.217.02.356-.039.73-.731-.02-.296zM42.687 181.171l.414.415c.208.054.425.054.632 0l.593-.593.138-.336.316-.039.81-.81.178-.415v-.237a1.259 1.259 0 00-.672-1.027 1.23 1.23 0 00-1.224 0c-.34.141-.61.411-.75.75l-.317.04-.118.119a1.39 1.39 0 00-.77 1.442l.453.454.317.237zM47.664 182.277a1.56 1.56 0 001.467.976c.204-.007.404-.057.588-.146l.652-.652a1.65 1.65 0 00-.316-1.264l-.652-.652-.336.059-.118.118-.356.158-.79.791v.355l-.139.257zM84.504 185.694a1.454 1.454 0 00-.572 1.956c.138.319.392.573.71.711a1.32 1.32 0 001.028-.079l.296-.296.218-.652-.04-.395-.652-.652-.652-.217-.336-.376zM103.27 219.75l-.454-.455-.573-.099-.336.06-.75.75-.059.297.098.098.06.415.158.158a1.283 1.283 0 001.679.534c.21-.114.385-.282.507-.487.122-.205.186-.44.184-.679l-.079-.513-.435-.079zM66.311 185.912l.573-.573a1.482 1.482 0 00-.434-1.58 1.366 1.366 0 00-1.56-.178 1.139 1.139 0 00-.317.316l-.395.395-.02.217a1.51 1.51 0 00.632 1.185 1.427 1.427 0 001.521.218zM55.17 174.771l.415.415h.554l.177-.178a1.515 1.515 0 00.806-1.486 1.513 1.513 0 00-.173-.569l-.949-.948-.375.02-.711.711-.119.395-.138.138c.023.341.103.675.237.988l.138.138.139.376zM83.734 223.641l-.454-.454-.296-.02c-.087.071-.166.15-.237.237a1.422 1.422 0 00-.77 1.481l.098.099.198.435.533.533.435.119c.25.013.501-.034.73-.139l.218-.217.276-.277a1.644 1.644 0 00-.316-1.264l-.415-.533zM65.264 188.973l-.75-.75a1.151 1.151 0 00-.75-.04l-.81.81-.06.257v.435l1.185 1.185.396-.079.118-.119a1.283 1.283 0 00.671-1.699zM87.803 220.006a1.19 1.19 0 00-1.007.06l-.296.296-.119.356-.158.158c-.058.23-.037.474.06.691l-.415.415.039.276.099.099c.161.383.466.688.85.849l.335.02.257-.257a.391.391 0 00.118-.118 1.372 1.372 0 00.712-1.343l-.356-.356a1.343 1.343 0 00-.889-1.797l.77.651zM37.768 191.502l-.198-.198a.892.892 0 00-.276-.276 1.256 1.256 0 00-1.265-.158l-.908.908-.02.218.02.375.138.138c.149.288.383.523.672.672l.098.099c.181.239.44.408.732.477.293.068.6.032.868-.102l.613-.612a1.508 1.508 0 00-.474-1.541zM48.099 166.711l.395.395c.195.043.397.043.592 0l1.008-1.007a1.396 1.396 0 00-.257-1.047l-.652-.652-.316.04-.118.118a1.428 1.428 0 00-.77 1.482l.453.454-.335.217zM25.185 242.881a.56.56 0 00-.178.178l-.474.474v.316l.098.099a1.433 1.433 0 00.805.931 1.421 1.421 0 001.23-.043l.198-.197.671-.672v-.316l-.098-.098a1.555 1.555 0 00-.87-.87 1.478 1.478 0 00-1.382.198zM107.201 224.352l-.177-.178a1.32 1.32 0 00-1.205-.256l-.949.948c-.052.36.032.726.237 1.027l.672.672h.197l.396-.08a1.289 1.289 0 00.652-1.56l-.119-.119.296-.454zM65.66 231.918l-.751.75-.099.336-.118.119.059.414.118.119a1.318 1.318 0 00.768.691 1.307 1.307 0 001.03-.079c.08-.064.153-.137.217-.217l.593-.592-.02-.336-.118-.119a1.454 1.454 0 00-1.68-1.086zM90.332 177.378a1.311 1.311 0 00-.336.336 1.257 1.257 0 102.41.592l.099-.098-1.166-1.166h-.197l-.218.02-.316.118-.336.336.06-.138zM96.001 195.176c-.116.079-.217.18-.296.296l-.395.395a1.593 1.593 0 00.652 1.166 1.57 1.57 0 001.363.099l.573-.573a1.452 1.452 0 00-.435-1.581 1.37 1.37 0 00-1.462.198zM74.37 232.728l-.118.118-.039.277c-.01.374.116.739.356 1.027l.414.415.297.019a.84.84 0 00.197-.197 1.353 1.353 0 00-.112-2.496 1.353 1.353 0 00-1.033.047l-.494.493.533.297zM124.368 210.465l-.356-.355-.336-.02-.099.099a.989.989 0 00-.177.177 1.376 1.376 0 00-.712.77 1.358 1.358 0 00.04 1.048 1.367 1.367 0 001.297.796c.18-.007.357-.049.52-.125l.613-.612.079-.277v-.237a1.537 1.537 0 00-.178-.809l-.356-.356-.335-.099zM53.669 219.177l.356.355.217.02.415-.02.158-.158a1.448 1.448 0 00.79-1.304l-1.008-1.007-.454-.059-.138.138a.445.445 0 00-.158.158 1.369 1.369 0 00-.731 1.719l.158.158h.395zM34.983 225.932l-.217.652.039.395 1.106 1.107.395-.04a1.342 1.342 0 00.731-.731 1.536 1.536 0 00-.079-1.027 1.423 1.423 0 00-1.877-.612l-.098.256zM57.837 238.555l-.73.731-.02.217.059.376.869.869c.22.089.46.109.691.059l.1.099.987-.988.02-.336-.1-.099-.118-.355-.138-.138a1.512 1.512 0 00-1.62-.435zM58.371 193.734l.02.336 1.126 1.126h.316l.257-.257a1.419 1.419 0 00.83-.83 1.339 1.339 0 00-.198-1.145l-.573-.573h-.237a1.173 1.173 0 00-.711.158 1.009 1.009 0 00-.218.217l-.474.474-.04.237-.098.257zM66.687 243.454l-.257-.257-.256.059-.415.178-.75.751v.237l.058.454.178.178c.203.232.461.409.75.514l.238-.237-.02.335.514.04.296-.06.79-.79.08-.276a2.345 2.345 0 00-.83-1.501l-.376.375zM19.95 213.547l-.316-.079a1.452 1.452 0 00-.77.138 1.26 1.26 0 00-.573 1.6l.198.198a1.254 1.254 0 001.441.494l.771-.771a1.87 1.87 0 00.06-.77l-.81-.81zM51.99 223.266l-.375-.376-.237-.039-.376.059-.118.119a1.478 1.478 0 00-.652 1.955l.889.889.355-.039 1.146-1.146a1.65 1.65 0 00-.316-1.264l-.316-.158zM41.027 208.668l.415.415.395-.04.83-.83.06-.256.118-.119-.02-.375-.672-.672-.276-.079a1.541 1.541 0 00-.929.138l-.158.751-.237.711.415.415.06-.059zM102.954 178.346l.85-.849.019-.376-.118-.118a1.429 1.429 0 00-1.253-1.017 1.418 1.418 0 00-.564.069l-.85.849-.02.375.119.119c.153.366.444.657.81.81.304.176.666.226 1.007.138zM135.113 165.941a1.385 1.385 0 00-1.185.079.56.56 0 00-.178.178l-.632.632a1.652 1.652 0 00.79 1.304 1.29 1.29 0 001.225.039 1.31 1.31 0 00.652-.652l.257-.257.02-.375-.119-.118a1.342 1.342 0 00-.83-.83zM113.918 192.154l-.612.612-.06.494.04.277a1.366 1.366 0 001.442.849l.355-.079.632-.632.139-.533-.02-.297a1.456 1.456 0 00-1.6-.928v.197l-.099-.098-.217.138zM94.737 185.773l.395-.039a1.237 1.237 0 00.692-1.047 1.342 1.342 0 00-.139-.731 1.31 1.31 0 00-1.798-.652l-.394.395c-.1.168-.155.358-.158.553v.198l.098.099c.025.184.092.36.198.513l.454.455.652.256zM105.048 204.46l-.178-.178a1.233 1.233 0 00-.572-.177l-.257.059-.455.217-.414.415-.079.276-.02.494 1.185 1.186.375-.02.139-.138a1.456 1.456 0 00.79-1.462l-.514-.514v-.158zM97.265 188.381l-.118-.119-.119.119a.898.898 0 00-.336.336l-.355.039-.139.138a1.398 1.398 0 00-.711.949 1.342 1.342 0 00.277 1.106l.454.454.237.04.1.098.315-.039.454-.454.139.138.336-.336a1.263 1.263 0 00.654-.724 1.262 1.262 0 00-.062-.975 1.142 1.142 0 00-.533-.533 1.178 1.178 0 00-.593-.237zM73.6 176.45a1.264 1.264 0 001.364.138 1.04 1.04 0 00.355-.356l.158-.158.02-.217a1.31 1.31 0 00-.73-.968 1.434 1.434 0 00-1.324-.02 1.286 1.286 0 00-.632 1.581l.197.197.593-.197zM98.51 187.65l.138-.138a1.398 1.398 0 00-.376-1.245 1.344 1.344 0 00-1.244-.335l-.652.651-.06.257.02.375.12.119.177.375.415.415.276.079.514.04.257-.06.553-.553-.138.02zM127.528 185.971l-.533.533a1.451 1.451 0 00-.395 1.344 1.313 1.313 0 001.244 1.007l.257-.059.415-.178.889-.889.019-.217-.039-.395-.119-.119a1.251 1.251 0 00-.61-.951 1.252 1.252 0 00-1.128-.076zM117.019 199.206l-.553-.553a1.348 1.348 0 00-1.225 0l-.849.849.316.316.276.474.593.593.415-.02.237-.039.158-.158c.339-.142.609-.412.75-.751a1.35 1.35 0 00-.118-.711zM97.285 164.617l.454.455.415.019.237-.237a1.392 1.392 0 00.79-.79 1.508 1.508 0 00-.237-1.225l-.612-.612-.099.099-.336.02-.118.118a1.34 1.34 0 00-.632 1.778l.138.138v.237zM108.782 188.598l-.02-.296-.098-.099a1.464 1.464 0 00-.338-.531 1.468 1.468 0 00-1.113-.424c-.198.013-.39.07-.564.165l-.119.119-.652.651a1.4 1.4 0 00.257 1.087l.534.533.237.04c.246.005.49-.05.711-.158l.79-.791.375-.296zM118.442 182.336a1.404 1.404 0 00-.376-1.205l-.691-.691-.296.059a1.37 1.37 0 00-.771 1.442l.85.85a1.307 1.307 0 001.097-.093 1.315 1.315 0 00.641-.895l-.454.533zM80.771 182.474l.395-.039a1.372 1.372 0 00.81-.81c.056-.3.022-.609-.099-.889a1.34 1.34 0 00-.04-.395l-.928-.928a1.064 1.064 0 00-.612.019l-.928.929c-.075.21-.082.438-.02.652l.237.237a1.456 1.456 0 00.83.829l.118.119.237.276zM86.164 194.03l.474.474c.332.092.687.049.988-.118a1.374 1.374 0 00.75-1.225 1.51 1.51 0 00-.77-1.323l-.237-.237-.178.177a1.517 1.517 0 00-.75.949 1.48 1.48 0 00.276 1.224l-.553.079zM92.366 202.959a1.34 1.34 0 00-.355-1.225l-.494-.494a1.512 1.512 0 00-.909.119c-.194.117-.357.28-.474.474l-.296.296-.02.336.099.099a1.365 1.365 0 00.763.918 1.362 1.362 0 001.192-.029c.205-.119.375-.289.494-.494zM108.861 213.804l-.514-.514-.217-.02-.316.04-1.047 1.047-.02.217.02.375.908.909.297.059.217-.02.395-.078.198-.198a1.375 1.375 0 00.61-1.432 1.368 1.368 0 00-.215-.504l-.316.119zM102.362 170.445a1.4 1.4 0 001.837.652l.81-.81a1.348 1.348 0 00-.157-1.033 1.328 1.328 0 00-.85-.607l-.119-.118-.217.019a1.025 1.025 0 00-.316.119 1.416 1.416 0 00-.867.681 1.433 1.433 0 00-.121 1.097zM87.586 203.275a1.509 1.509 0 00-.415-1.6 1.347 1.347 0 00-.708-.335 1.349 1.349 0 00-.773.118l-.573.573a1.422 1.422 0 00.099 1.126c.108.188.253.353.426.486.173.132.37.228.58.284h.199l.355-.039.79-.791.02.178zM83.24 174.79l.455.455.316-.04.237-.237a1.589 1.589 0 00.889-.889 1.51 1.51 0 00-.237-1.145l-.455-.455h-.237a1.816 1.816 0 00-.73.178.838.838 0 00-.198.197l-.553.554.02.217c.005.359.13.707.355.988l.138.177zM89.383 198.831a1.228 1.228 0 00-.02-.652l-.928-.929a1.45 1.45 0 00-.651.02l-.91.909a1.172 1.172 0 000 .671l.91.909c.221.044.45.044.671 0l.928-.928zM91.32 197.25l.118.119.336.059.118-.118a1.04 1.04 0 00.178-.178 1.43 1.43 0 00.759-.973 1.422 1.422 0 00-.285-1.2l-.652-.652c-.213.023-.42.083-.612.178a1.394 1.394 0 00-.77 1.363l.138.138.197.395.356.356.118.513zM86.599 197.428l1.007-1.008a1.507 1.507 0 00-.415-1.56 1.371 1.371 0 00-1.501-.237l-.474.474-.119.316-.355.356a1.062 1.062 0 00-.02.652l.928.928c.208.053.425.052.632 0l.317.079zM81.897 189.527l.79-.791a1.838 1.838 0 00-.02-.651l-1.007-1.008a1.128 1.128 0 00-.612-.02l-.948.949c-.053.215-.046.44.02.651l.928.929c.209.041.423.041.632 0l.217-.059zM97.423 204.065l-.632-.632-.316.04-.118.118-.376.178-.414.415a1.56 1.56 0 00-.711 2.133l.73.731.316.039.1-.098c.148-.041.293-.094.434-.158.375-.156.673-.455.83-.83a1.379 1.379 0 00.63-.888 1.367 1.367 0 00-.216-1.068l-.257.02zM100.248 200.213l.119.119.138-.139a1.343 1.343 0 00.652-.967 1.481 1.481 0 00-.553-1.423 1.346 1.346 0 00-1.403-.098 1.167 1.167 0 00-.71.711 1.394 1.394 0 00.157 1.106l.139.138a1.148 1.148 0 001.461.553zM96.238 175.897l.297-.139.158-.118.237-.277.118-.197.06-.257.02-.178-.04-.395-.04-.197-.06-.139-.157-.158-.277-.237-.178-.099-.237-.079-.217-.019-.375.019-.178.06-.296.138-.158.118-.257.297-.099.178-.059.256-.04.198.04.395.06.178.058.138.139.178.276.237.178.099.257.059.197.039.376-.019.197-.079zM122.372 202.07l-.316-.316a1.35 1.35 0 00-.632 0l-1.205 1.205a1.402 1.402 0 00.277 1.067l.474.474.237.039.099.099.335-.059.81-.81.06-.257.118-.118-.039-.396-.593-.592.375-.336zM112.871 213.547l-.81.81.039.355.119.119a1.41 1.41 0 00.753.86 1.402 1.402 0 001.143.009l.751-.751.019-.335v-.237a1.445 1.445 0 00-.77-1.087 1.345 1.345 0 00-1.244.257zM136.812 218.762a1.396 1.396 0 00-.414 2.114l.474.474.256.059.099-.099a1.09 1.09 0 00.652-.138 1.228 1.228 0 00.692-.692l.059-.256a1.342 1.342 0 00-1.956-.85l.138-.612zM107.498 175.936l-.691.692-.04.316.119.395.711.711.217.02h.435l.869-.87.079-.276a1.401 1.401 0 00-.711-1.185 1.456 1.456 0 00-.988.197zM128.042 199.561a.98.98 0 00-.218.218l-.454.454-.039.237-.099.099.039.316.909.909c.21.074.438.081.652.019l.711-.711.079-.316.02-.217-.02-.376-.138-.138a1.311 1.311 0 00-1.442-.494zM92.742 192.825l.276-.276.02-.336v-.197l.277-.277-.04-.395-1.027-1.027h-.197l-.356.039-.118.119a1.303 1.303 0 00-.79.79l-.02.257-.395.395-.06.454.04.277-.02.296.04.276.217.415.573.573.494.02.276-.04a1.4 1.4 0 00.77-1.047l.04-.316zM107.221 199.799l-.375-.376-.356.04-.119.118a1.402 1.402 0 00-.79.79 1.543 1.543 0 00.178 1.166l.652.652h.198l.335-.02.119-.119a1.387 1.387 0 00.729-.883 1.387 1.387 0 00-.176-1.131l-.375-.376-.02.139zM99.142 148.538l.118-.119-.059-.256a1.333 1.333 0 00-.77-.771 1.539 1.539 0 00-1.106.079l-.198.198-.73.731.039.276.099.099.177.375.593.593.494.059.276-.079.218-.217a1.341 1.341 0 00.849-.968zM53.057 238.594l.256-.256v-.198a1.338 1.338 0 00-.513-1.185 1.53 1.53 0 00-.711-.237l-.119-.119-.415.02-.296.296a1.46 1.46 0 00-.948.948l-.04.238.02.217.138.138c.136.274.358.496.633.632l.533.534.158-.158a.619.619 0 00.197-.198l.198-.197.909-.475zM66.193 259.474l-.474-.474-.1.098-.315.04-.099.099a1.45 1.45 0 00-.79 1.462l.928.928c.353.113.735.085 1.067-.079a1.315 1.315 0 00.672-.909 1.396 1.396 0 00-.356-1.185l-.533.02zM70.143 208.944l.277-.039.454-.218.415-.414.079-.277.02-.494-.06-.296a3.578 3.578 0 01-.157-1.106l-1.008-1.008-.474-.039-.257.059a1.313 1.313 0 00-.73.968l-.159.158-.059.454-.099.099-.158.514v.316l.237.237a1.291 1.291 0 00.58.944 1.291 1.291 0 001.1.142zM67.338 204.598l.119-.118.04-.316.098-.099a1.533 1.533 0 00-1.224-1.225h-.317l-.553.198-1.145 1.146v.434l.987.988.297.059.098.099.297-.02.553-.197.632-.633.118-.316zM72.198 215.107l-.336-.335-.257-.02-.434.039-.158.158a1.31 1.31 0 00-.652.929 1.368 1.368 0 00.296 1.047l.514.513.256.02.1-.099.355-.079.099-.099a1.255 1.255 0 00.671-1.54l-.158-.158-.296-.376zM82.825 232.688l-.296.297-.119.355-.138.138a1.12 1.12 0 00.237 1.027c.248.262.588.418.948.435l.139.138.395-.039a1.153 1.153 0 00.671-.672 1.259 1.259 0 00-.059-.968 1.423 1.423 0 00-1.778-.711zM84.78 158.316l-.078-.395a1.192 1.192 0 00-.692-.692 1.395 1.395 0 00-1.047.178 1.316 1.316 0 00-.513 1.817l.276.277.692.178.395-.079.79-.79.178-.494zM37.807 182.139l-.276-.277-.692-.178-.395.079-.099.099c-.123.085-.23.193-.316.316l-.296.297-.158.671.06.375a1.173 1.173 0 00.71.712c.338.076.692.02.988-.158a1.427 1.427 0 00.474-1.936zM87.191 146.701l-.533.533a1.259 1.259 0 00.02 1.284l.83.83.316-.079.889-.889-.02-.336-.257-.257a1.34 1.34 0 00-1.758-.612l.513-.474zM71.645 168.529l-.237.237-.198.513v.316c.035.148.095.288.178.415a1.282 1.282 0 00.87.632 1.616 1.616 0 001.224-.355l.553-.553.02-.297-.158-.395a1.37 1.37 0 00-1.798-.415l-.454-.098zM94.737 154.958l-.178.178-.04.434.06.336.158.356a1.375 1.375 0 00.988.632 1.513 1.513 0 001.245-.573l.118-.119.118-.316.099-.098v-.238l-.06-.256-.098-.099a1.476 1.476 0 00-1.403-.691l-.256.059-.475.474-.276-.079zM61.235 175.739l-.178-.178a1.257 1.257 0 00-.138-1.087 1.677 1.677 0 00-1.107-.75l-.335.059-.79.79a1.317 1.317 0 00.098 1.087 1.366 1.366 0 00-.059.967l.909.909h.395l.118-.118a1.656 1.656 0 00.929-1.166l.158-.513zM46.084 175.442l.02-.217-.415-.415a1.337 1.337 0 00-1.403-.652 1.477 1.477 0 00-1.205 1.205v.198l.415.415a1.4 1.4 0 001.403.651 1.478 1.478 0 001.185-1.185zM117.098 159.541l.474.474a1.258 1.258 0 001.087.257l.553-.554c.187-.232.303-.513.336-.809l-.06-.336-.711-.711a1.536 1.536 0 00-1.541.474 1.311 1.311 0 00-.138 1.205zM58.173 154.266l.376-.375v-.237l.098-.099-.079-.316-.118-.118a1.37 1.37 0 00-1.521-.652l-1.008 1.007c-.021.206-.001.415.06.613l.612.612.355.119.119.118a1.331 1.331 0 001.027-.316l.08-.356zM70.895 146.76a1.311 1.311 0 00-.91-.632 1.482 1.482 0 00-1.184.356l-.494.493-.04.237.06.336.928.929h.198l.375-.06.869-.869.06-.257v-.197l-.08-.356.218.02zM68.425 173.368l-.73-.731a1.255 1.255 0 00-.81-.019l-.85.849-.04.355c-.013.262.064.52.218.731l.75.751.277-.04c.374-.158.671-.456.83-.829a1.406 1.406 0 00.355-1.067zM128.279 139.906l-.158.434-.119.119c-.026.29.043.582.198.829.094.141.215.261.355.356l.356.355a1.283 1.283 0 001.462-.592l.414-.415.02-.375-.02-.218-.217-.217a1.48 1.48 0 00-1.541-.672l-.553.554-.197-.158zM63.072 149.209l.257-.059.849-.849.02-.455-.04-.237-.178-.177a1.367 1.367 0 00-1.758-.573l-.572.573a1.368 1.368 0 001.62 1.619l-.198.158zM89.877 133.189a1.2 1.2 0 00.77-.256l.534-.534.06-.257-.02-.217-.08-.355-.138-.139a1.597 1.597 0 00-1.363-.691l-1.047 1.047c-.094.261-.1.545-.02.81l.613.612.276.079.257.02.099-.099.06-.02zM61.511 151.363l-.671-.672a1.396 1.396 0 00-1.324.296 1.482 1.482 0 00-.434 1.423l.79.79a1.48 1.48 0 001.56-.534 1.367 1.367 0 00.08-1.303zM87.172 123.767l-.237.237-.1.296-.078.553.079.316a1.308 1.308 0 00.909.672c.394.061.797-.031 1.126-.257l.572-.573v-.276l-.098-.099-.159-.356a1.315 1.315 0 00-1.62-.513l-.355.355-.04-.355z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M82.233 142.572l-.237-.039h-.198l-.335.059-.731.731a1.369 1.369 0 00.099 1.245 1.425 1.425 0 001.916.493l.533-.533.06-.257v-.197a2.07 2.07 0 00-.238-.751l-.415-.415-.454-.336zM69.59 155.807c.183-.002.363-.035.534-.099.236.031.475.031.711 0 .167.184.389.309.632.356l-.316.316-.059.257-.02.217.04.356.178.414c.15.245.376.433.644.536.267.104.561.117.837.037l1.126-1.126-.06-.414-.67-.672.315-.316a1.312 1.312 0 00-.02-1.363l-.098-.099-.06-.336-.572-.573a.98.98 0 00-.376-.098l-.395-.079-.099.098a1.51 1.51 0 00-.691-.019l-.336-.02-.118.118a1.199 1.199 0 00-.928-.019l-.198.197-.099.099-.632.632-.04.277.02.217.138.415.178.178c.11.164.25.305.415.414.005.034.012.066.02.099zM34.43 127.204l-.297-.296-.375-.099-.138-.138a1.119 1.119 0 00-.988.316 1.227 1.227 0 00-.375.968l-.119.118.06.415a1.283 1.283 0 001.738.435 1.65 1.65 0 00.632-.632 1.286 1.286 0 00-.138-1.087zM58.509 146.385l.256.257a1.308 1.308 0 001.68.019l.553-.553c.127-.36.113-.756-.04-1.106l-.573-.573a1.595 1.595 0 00-1.64.573 1.37 1.37 0 00-.098 1.521l-.138-.138zM67.418 142.059v-.237l.118-.119-.079-.395a1.38 1.38 0 00-.865-.655 1.36 1.36 0 00-1.07.181 1.48 1.48 0 00-.712.909c-.075.363-.02.741.158 1.066l.435.435a1.402 1.402 0 001.126.059 1.377 1.377 0 00.869-.869l.02-.375zM61.551 142.474l.435-.435-.06-.336-.099-.099a1.17 1.17 0 00-.513-.513l-.237-.237-.277-.079a1.398 1.398 0 00-1.334 1.383c0 .228.058.452.169.651a1.189 1.189 0 00.81.573c.106.051.219.084.336.099a1.42 1.42 0 001.165-.415l-.395-.592zM54.973 142.256l.533-.138.553-.553.04-.553-.04-.316-.138-.138a1.396 1.396 0 00-1.857-.593c-.309.064-.588.231-.79.474a1.17 1.17 0 00-.237 1.106l.89.889c.244.027.492.007.73-.059l.316-.119zM38.203 163.847a1.281 1.281 0 00-1.126-.652 1.258 1.258 0 00-1.126.612l-.08.277 1.206 1.205a1.226 1.226 0 001.066-.711c.092-.233.113-.487.06-.731zM79.645 110.236c.237.042.48.029.711-.04l.573-.573.118-.316.119-.118a1.703 1.703 0 00-.06-.652l-.908-.909a1.54 1.54 0 00-.672.04l-.928.928c-.022.212-.008.425.04.632l1.007 1.008zM81.68 125.071l-.316.316c-.022.211-.009.425.04.632l.73.731.296.059.218.02.395-.079.138-.139a1.367 1.367 0 00.474-1.817l-.672-.671-.237-.04-.335.059-.356.356-.375.573zM64.731 124.083l-.612-.612-.652-.139-.395.079a1.315 1.315 0 00-.046 2.286 1.307 1.307 0 001.808-.477c.116-.202.176-.43.174-.663l-.277-.474zM68.623 132.439l-.395.395.04.355.295.297a1.345 1.345 0 001.04.682 1.346 1.346 0 001.153-.465l.81-.81-.099-.099-.059-.375-.85-.849-.256-.06-.099.099-.356.04-.691.691-.533.099zM90.41 156.558l-.118.118a1.426 1.426 0 00-.059 1.482 1.506 1.506 0 001.481.652 1.43 1.43 0 001.146-1.146v-.237l-.059-.375-.672-.672a1.4 1.4 0 00-1.6-.02l-.158.158.04.04zM89.285 95.183l.81-.81a1.37 1.37 0 00-.099-1.244l-.79-.79h-.237a1.37 1.37 0 00-.987.395l-.494.494v.276l.118.119c.045.2.118.393.218.573a1.397 1.397 0 001.461.987zM122.254 192.825a1.255 1.255 0 001.323.613l.494-.494a1.26 1.26 0 00.533-.731 1.218 1.218 0 00-.178-.968l-.375-.375a1.284 1.284 0 00-1.639.178 1.449 1.449 0 00-.158 1.777zM19.041 175.343a1.395 1.395 0 00.376-1.837 1.483 1.483 0 00-.376-.375l-.395-.395a1.281 1.281 0 00-1.264.434l-.197.198-.08.277-.04.474-.315.079c.005.25.073.494.197.711l.534.533.375.099.099.099.355-.356.731.059zM100.94 104.665l-.237.237.019.297.099.098a1.428 1.428 0 001.857.514 1.282 1.282 0 00.612-1.64l-.632-.632a1.337 1.337 0 00-1.758.376l.04.75zM79.013 136.508a1.316 1.316 0 00-1.145-.632l-.158.158-.415.217-.692.692-.039.276-.099.099.099.375.099.099a1.34 1.34 0 001.56.612l.75-.75a1.482 1.482 0 00.04-1.146zM89.68 155.471l-.02-.414-.257-.257a1.59 1.59 0 00-1.126-.81l-.316-.079a1.705 1.705 0 00-1.126-.099 1.981 1.981 0 00-1.441 1.442l.02.217.078.356.87.869h.197a1.29 1.29 0 001.086.296l.119.119.257-.02.099-.099a1.359 1.359 0 00.81-.81l.197-.197.553-.514zM65.976 201.418l.217-.217.04-.237-.099-.099-.06-.336-.276-.276a1.196 1.196 0 00-.928-.573 1.207 1.207 0 00-.988.355l-.178.178a1.258 1.258 0 00-.256 1.363l.81.81.335.02c.368-.002.72-.143.988-.395l.395-.593zM69.334 181.111l.138.139.316-.04a1.537 1.537 0 00.889-.889 1.37 1.37 0 00-.158-1.106l-.593-.592-.237-.04a1.4 1.4 0 00-1.086.336l-.316.316v.237c.003.259.078.512.217.731l.711.711.119.197zM69.156 159.284l-.198-.198-.415-.217-.158-.158h-.316l-.237.039-.948.949a1.503 1.503 0 00-.138.533l.099.296.276.474.237.237.316.119.316.039c.364.058.737-.019 1.047-.217l.593-.592.04-.277-.1-.099a1.62 1.62 0 00-.414-.928zM60.287 164.558l.118.119a1.427 1.427 0 001.577.634c.212-.055.41-.157.576-.299l.494-.493.04-.237.652-.652-.04-.435-.889-.889-.296-.059-.217.02-.395.118a1.455 1.455 0 00-.652.968 1.646 1.646 0 00-.514.751l-.454.454zM76.604 133.861l.316-.079.118-.119a1.506 1.506 0 00.494-1.995l-.75-.75-.218.019-.454.06-.909.908-.02.218c-.048.308.015.623.178.889.119.156.258.296.415.414l.316.316.514.119zM81.68 154.385l-.158-.158a1.338 1.338 0 00-1.54-.632l-.278.079-.869.869c-.028.35.048.7.217 1.007l.87.87.296-.02a1.647 1.647 0 00.928-.929l.534-1.086zM73.561 161.358l-.671-.672a1.79 1.79 0 00-.474-.237l-.672-.671-.217-.02-.336.059-.118.119a1.774 1.774 0 00-1.028 1.027l-.099.099.04.316c-.14.355-.14.75 0 1.106l.77.77c.31.142.65.196.988.158l.178-.178.276-.039.198-.198a1.506 1.506 0 001.165-1.639zM77.65 157.664l-.671-.672-.316.04-.1.099a1.41 1.41 0 00-.8.767 1.424 1.424 0 00-.009 1.109l.652.652a1.45 1.45 0 001.501-.474 1.396 1.396 0 00.08-1.541l-.337.02zM81.284 149.407l-.375.375v.356l.277.277a1.424 1.424 0 001.777.592l.79-.79.02-.336a1.957 1.957 0 00-.217-.731l-.81-.81-.099.099-.335.059-.652.652-.376.257zM48.197 191.028a1.432 1.432 0 00-.928-.652 1.566 1.566 0 00-1.304.395l-.494.494c-.08.371-.016.759.178 1.086a1.454 1.454 0 00.988.711l.257.02.276-.276a1.491 1.491 0 00.909-.909c.14-.267.182-.575.118-.869zM104.891 146.069l.098-.099-.02-.375-.256-.257a1.46 1.46 0 00-1.008-.731 1.317 1.317 0 00-1.126.296l-.217.218a1.28 1.28 0 00-.138 1.481l.553.553.336.099.118.118.237-.039.099-.099c.221-.01.434-.086.612-.217l.593-.593.119-.355zM106.984 139.827l-.375.375.039.355.238.238a1.842 1.842 0 001.679.889l1.007-1.008c.027-.177.02-.358-.02-.533a1.716 1.716 0 00-.652-.652l-.296-.297-.217-.019-.395.039-.711.711-.297-.098zM82.41 160.41l-.177-.178a1.493 1.493 0 00-1.008-.692 1.429 1.429 0 00-1.185.356l-.671.672.02.296.276.277a1.421 1.421 0 00.968.691 1.511 1.511 0 001.225-.356l.572-.572v-.317l-.02-.177zM112.812 152.054l-.119-.119a1.172 1.172 0 00-1.363.06l-.474.474-.059.257-.119.118.06.415a1.264 1.264 0 001.066.632c.249.024.5-.021.726-.129a1.376 1.376 0 00.774-1.19 1.37 1.37 0 00-.176-.716l-.316.198zM139.42 155.274l-.138.138.019.217.158.435a1.4 1.4 0 001.423.672 1.536 1.536 0 001.244-1.245l.119-.118v-.198l-.356-.356a1.49 1.49 0 00-1.146-.711 1.62 1.62 0 00-1.343.751l.02.415zM69.354 165.21l-.573.573.02.296.316.316a1.56 1.56 0 001.086.692 1.63 1.63 0 001.344-.672l.197-.198.08-.474-.02-.257-.929-.928a1.57 1.57 0 00-1.52.652zM114.609 197.428l-.276-.277-.277-.079a1.257 1.257 0 00-.553-.118l-.277.039-.612.613-.079.316-.059.296.059.375a1.202 1.202 0 00.691.692 1.167 1.167 0 00.988-.158 1.344 1.344 0 00.395-1.699zM117.078 167.422a1.51 1.51 0 00-1.817-.632l-.849.85-.02.256.039.435.949.948.276.04h.237l.336-.099-.039-.277.177.178.435-.197.592-.593.02-.494-.158-.158-.178-.257zM91.971 150.217l.988.988.217.019.375-.059 1.087-1.086-.04-.356-.138-.138a1.431 1.431 0 00-.928-.652 1.515 1.515 0 00-1.126.257l-.514.513-.04.237v.198l.12.079zM121.858 146.207a1.283 1.283 0 00-1.481-.415l-.553.553a1.262 1.262 0 00-.059 1.403c.104.144.23.271.375.375l.119.119.296.098.099.099h.237l.276-.079.099-.099a1.34 1.34 0 00.691-1.6l-.158-.158.059-.296zM46.025 123.352l.138-.138.04-.237.098-.099-.04-.356-1.027-1.027h-.197l-.356.04-.118.118a1.454 1.454 0 00-.79 1.422l.079.277.77.77.316.04.257-.02c.281-.029.552-.124.79-.276l-.06.059.1-.573zM60.682 191.225l-1.007-1.007h-.277l-.375.059-.119-.118a1.388 1.388 0 00-.83.829 1.287 1.287 0 00.1 1.047l.552.553c.22.059.45.066.672.021.223-.044.432-.139.612-.278a1.53 1.53 0 00.672-1.106zM55.96 175.008l.139-.138.06-.257-.1-.099c.089-.069.169-.149.238-.237a1.535 1.535 0 00.256-1.758l-.73-.731h-.317a1.706 1.706 0 00-1.106.869 1.48 1.48 0 00-.138.77l-.198.198a1.285 1.285 0 00-.079 1.422l.81.81.316-.079.336-.336.514-.434zM65.501 118.335l.988.987h.356l.118-.118a1.434 1.434 0 00.83-.83 1.535 1.535 0 00-.178-1.165l-.652-.652-.237-.04h-.197l-.336.06-.83.829-.06.257-.019.217.06.376.157.079zM121.147 115.233l.672.672.276.04c.236.057.484.043.711-.04l.791-.79.059-.375a1.287 1.287 0 00-.178-.455 1.392 1.392 0 00-.83-.648 1.39 1.39 0 00-1.046.115 1.366 1.366 0 00-.455 1.481zM113.957 137.989a1.267 1.267 0 00-.217 1.166l.612.612.336.139.237.039a1.394 1.394 0 001.027-.316l.494-.494.04-.276-.119-.119-.098-.375-.099-.099a1.285 1.285 0 00-1.64-.533l-.158.158-.415.098zM127.311 158.81l-.297.296c-.043.224-.029.455.04.672l.691.691.316.079h.198l.375-.059.138-.138a1.292 1.292 0 00.655-.797c.048-.17.061-.349.036-.525a1.273 1.273 0 00-.177-.496l-.81-.81-.099.099-.296.059-.356.356-.414.573zM71.15 141.762l.435.435a1.447 1.447 0 001.166-.731 1.368 1.368 0 00-.02-1.323 1.282 1.282 0 00-.908-.593l-.1-.099h-.197l-.296.06-.099.098a1.347 1.347 0 00-.77.771 1.395 1.395 0 00.158 1.066l.632.316zM42.232 158.355l.257.257.435-.039a2.03 2.03 0 00.71-.198l.238-.237.237-.514.04-.315-.06-.336-.632-.632.119-.119a1.535 1.535 0 00-1.739.039l-.513.514-.06.257.04.355.75.751.178.217zM99.34 120.29l-.139.336-.158.158a1.2 1.2 0 00.04.632l.593.593.335.138.06.296.829.83.415.099.257.02a1.29 1.29 0 001.084-1.358 1.287 1.287 0 00-.195-.598 1.507 1.507 0 00-.85-.652l-.079-.316-.118-.118a1.452 1.452 0 00-1.502-.672l-.454.454-.118.158zM98.648 125.288a1.478 1.478 0 00-.87.869 1.397 1.397 0 00.12 1.185l.71.711a1.62 1.62 0 001.265-.395l.474-.474.039-.237v-.197l-.059-.336-1.165-1.166-.356.04h-.158zM97.68 162.326a1.45 1.45 0 00-1.995-.455c-.3.182-.52.472-.612.81a1.345 1.345 0 00.158 1.028l.276.276.692.178.395-.079.672-.672.177-.691.237-.395zM64.988 183.363l.455-.454.059-.612-.079-.316-.75-.751-.258-.059-.217.02-.395.118-.138.138a1.23 1.23 0 00.77 2.312l.514-.119.04-.277zM96.219 144.192l.671.672a1.507 1.507 0 001.561-.534 1.45 1.45 0 00.079-1.58l-.533-.533-.218-.02a1.507 1.507 0 00-1.126.731 1.563 1.563 0 00-.434 1.264zM106.53 132.379l-.415.415v.356l.297.296a1.538 1.538 0 002.093.474l.929-.928-.059-.375-.81-.81-.395-.119h-.237a3.097 3.097 0 00-.968.296l-.395.396-.04-.001zM59.773 164.123l.534-.533v-.276l-.336-.336a1.514 1.514 0 00-.668-.571 1.518 1.518 0 00-.873-.101l-.118.119-.395.197-.554.553-.059.415c-.014.252.055.502.198.711l.73.731c.456.027.906-.114 1.265-.395l.276-.514zM93.058 143.402l.75-.751c.084-.249.091-.517.02-.77l-.83-.83-.236-.039-.435.039-.889.889-.059.257v.237l.099.415a1.311 1.311 0 001.58.553zM63.704 167.936a1.344 1.344 0 00-.158-1.027l-.277-.277-.355-.079-.158-.158a1.199 1.199 0 00-.712.119l-.394-.395-.297.059a1.563 1.563 0 00-.83.83l-.02.335.514.514a1.31 1.31 0 001.383.632l.375-.375a1.364 1.364 0 001.106-.119 1.582 1.582 0 00-.177-.059zM88.732 116.083c.072-.06.138-.126.198-.198a.809.809 0 00.256-.257 1.197 1.197 0 00.08-1.264l-.988-.988-.218-.019-.375.059-.118.119a1.646 1.646 0 00-.692.691 1.34 1.34 0 00-.296 1.64l.612.612a1.569 1.569 0 001.541-.395zM114.155 124.715l-.376.375a2.25 2.25 0 00.02.613l.612.612.376.138.138.139a1.34 1.34 0 001.047-.336l.494-.494.02-.217-.099-.099-.059-.296-.119-.119a1.455 1.455 0 00-1.561-.652l-.434.435-.059-.099zM36.622 107.036l-.77-.771-.297.02-.098.099a1.422 1.422 0 00-.89.889 1.366 1.366 0 00.139 1.205l.77.77.297-.02a1.506 1.506 0 00.889-.889 1.366 1.366 0 00-.04-1.303zM60.642 187.591l.178-.178a1.203 1.203 0 00.158-1.225l-.928-.928a1.51 1.51 0 00-1.028.316l-.513.513-.04.238.119.118.118.356.1.098a1.255 1.255 0 001.6.573l.375-.375-.139.494zM50.292 146.661l-.75-.75-.337-.099h-.197l-.415.059-.138.138a1.315 1.315 0 00-.474 1.818l.79.79.336-.059a1.437 1.437 0 001.052-.661 1.43 1.43 0 00.133-1.236zM106.392 167.561l-.435-.237a1.256 1.256 0 00-1.521 1.837 1.202 1.202 0 001.106.592l.85-.849.059-.296-.099-.099-.138-.336-.336-.336.514-.276zM89.009 174.395l-.613-.612-.573-.059a1.497 1.497 0 00-1.126.731 1.564 1.564 0 00-.02 1.363l.672.671a1.48 1.48 0 001.56-.533 1.423 1.423 0 00.1-1.561zM50.094 155.392l-.118-.118h-.277a1.308 1.308 0 00-.988.395l-.513.514.02.296.217.217a1.399 1.399 0 001.837.494 1.37 1.37 0 00.533-1.837l-.494-.494-.217.533zM75.695 203.789l.257-.257-.02-.336-.297-.296a1.335 1.335 0 00-.815-.629 1.343 1.343 0 00-1.022.135 1.367 1.367 0 00-.553 1.817l-.059-.059.612.612.218-.02c.289-.004.57-.093.81-.256l.335-.336.534-.375zM62.203 133.841l-.158.158-.08.277v.237l.06.415.158.158a1.513 1.513 0 001.363.691l1.007-1.007.02-.455-.04-.237-.157-.158a1.424 1.424 0 00-1.778-.592l-.455.454.06.059zM54.203 115.648l-.692-.178-.375.06-1.087 1.086.08.395c.183.299.472.517.81.613a1.396 1.396 0 001.52-2.075l-.257.099zM43.18 139.353l-.573-.573-.237-.04h-.197l-.376.059-.869.87c-.075.236-.082.49-.02.731l.1.098.058.257.356.356.415.217.158.158h.316l.118-.119.356-.118.138-.138a1.483 1.483 0 00.257-1.758zM87.882 136.765l-.316.079-1.165 1.165.02.336.276.277a1.348 1.348 0 00.928.691 1.419 1.419 0 001.146-.277l-.04.04.455-.455-.02-.217a1.258 1.258 0 00-.197-.711l-.81-.81-.277-.118zM38.953 148.558l.04-.237-1.107-1.107-.296-.059-.257-.02-.434.119-.178.178a2.063 2.063 0 00-.474.79l.237.237-.336-.02-.02.533.632.633.474.237.297.019a2.206 2.206 0 001.422-.908v-.395zM65.403 99.944l.197-.197.04-.316a1.117 1.117 0 00-.178-.77 1.201 1.201 0 00-1.62-.435l-.296.296a1.285 1.285 0 00-.395 1.501l.75.751c.259.074.532.074.79 0l.198-.198.514-.632zM57.995 132.439l.376-.376.02-.256-.1-.099-.078-.356-.119-.118a1.48 1.48 0 00-1.995-.494l-.573.573-.079.276v.198l.04.355.948.949a1.593 1.593 0 001.224-.395l.336-.257zM71.902 120.389l-.138.138-.06.257-.098.099.059.375 1.106 1.106.375-.059.77-.77.08-.277a1.537 1.537 0 00-.198-.869l.099.099-.73-.099-.732-.178-.316.316-.217-.138zM106.214 180.262l.849.849.375-.019.119-.119a1.398 1.398 0 00.81-1.876l-.83-.83h-.395l-.118.119a1.414 1.414 0 00-.83.829c-.133.338-.125.715.02 1.047zM120.91 211.394a1.475 1.475 0 00-.158-1.185l-.711-.712c-.26.033-.508.13-.721.282a1.582 1.582 0 00-.504.588 1.314 1.314 0 00.06 1.244c.126.281.351.506.632.632l.217.218.395-.001.119-.118a1.443 1.443 0 00.671-.948zM93.196 192.055l-.592-.592-.514-.04-.296.059a1.453 1.453 0 00-.731 1.521l.079.356.632.632.533.099.316-.04a1.425 1.425 0 00.81-1.639l-.237-.356zM98.293 172.538l.079.396a1.236 1.236 0 001.086.612c.253.014.503-.055.711-.198a1.302 1.302 0 00.664-.803 1.32 1.32 0 00-.13-1.034l-.277-.276-.276-.079a1.005 1.005 0 00-.534-.099.416.416 0 00-.118.118 1.765 1.765 0 00-.514.237l-.296.297-.395.829zM80.376 184.094l.178-.178c.07-.176.11-.363.118-.553l-.06-.296-.75-.751-.257-.059-.493-.02-.277.079-.75.751-.06.257-.02.217.06.375.138.139a1.342 1.342 0 001.501.671l.692-.691-.02.059zM95.883 175.225l.099-.099-.139-.138a1.56 1.56 0 00-.355-.356l-.08-.395-.118-.118a1.284 1.284 0 00-.988-.672 1.322 1.322 0 00-1.087.375l-.453.455-.02.217v.198l.079.316.434.434-.02.217.258.257a1.256 1.256 0 001.718.455c.24-.102.432-.293.534-.534.072-.197.118-.403.138-.612zM106.174 150.79a1.29 1.29 0 00-.039 1.383c.087.146.209.268.355.355l.158.158.218.02a1.286 1.286 0 00.869-.869 1.457 1.457 0 00-.079-1.304 1.256 1.256 0 00-1.62-.514l-.197.198.335.573z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M96.673 176.41l.158.158a1.31 1.31 0 001.225-.474 1.421 1.421 0 00.257-1.244l-.85-.85h-.237l-.375.02-.83.83-.079.276.02.494.06.257.552.553.1-.02zM100.347 205.25l-.494-.493a1.414 1.414 0 00-1.343-.316 1.282 1.282 0 00-.949 1.303l.06.257 1.126 1.126.276.079h.237l.376-.059.138-.138a1.281 1.281 0 00.718-1.267 1.29 1.29 0 00-.145-.492zM86.4 195.69l.475-.474a1.316 1.316 0 00-.099-1.205l-.711-.712-.217-.019-.83.829-.099.336-.04.237.02.415.317.316a1.152 1.152 0 00.849.613c.133-.09.247-.204.336-.336zM119.548 173.605l-.435.435.02.415.237.237a1.29 1.29 0 00.909.671 1.523 1.523 0 001.224-.316l.593-.592.02-.218-.04-.316-.138-.138a1.368 1.368 0 00-1.837-.494l-.138.138-.415.178zM96.416 186.741l.316-.039a1.46 1.46 0 00.869-.869 1.259 1.259 0 00-.138-1.126l-.613-.613-.257-.059a1.342 1.342 0 00-1.047.336l-.513.514v.237c.028.254.102.502.217.73l.77.771.396.118zM103.35 195.927a1.41 1.41 0 00.644-.076 1.41 1.41 0 00.541-.359l.119-.118.118-.316.237-.238-.099-.098-.059-.297a1.423 1.423 0 00-1.501-.671l-.929.928a1.375 1.375 0 00.139 1.047c.235.148.512.217.79.198zM100.604 158.336l.04.395a1.37 1.37 0 00.967.652c.304.062.619.013.889-.139.122-.007.242-.034.356-.079l.573-.573.138-.335.119-.119a1.322 1.322 0 00-.04-.593l-.928-.928a1.367 1.367 0 00-.672.04l-.119.118-.217.02c-.359.162-.647.45-.81.81l-.138.138-.158.593zM89.482 164.558l-.395.395c-.047.342.023.69.198.988a1.34 1.34 0 001.264.671 1.475 1.475 0 001.264-.869l.119-.118-.297-.297a1.537 1.537 0 00-1.007-.691 1.5 1.5 0 00-1.185.356l.04-.435zM81.008 171.334a1.364 1.364 0 001.185-.435l.474-.474a1.314 1.314 0 00-.158-.909 1.035 1.035 0 00-.474-.474l-.415-.415-.336.02-.098.099a1.39 1.39 0 00-.85.849 1.343 1.343 0 00.119 1.185c.129.233.32.424.553.554zM71.29 188.519l.374-.375.04-.237-.079-.356-.889-.889-.237-.039-.118-.119-.376.059-.908.909-.04.277.02.256.099.376.197.197a1.38 1.38 0 00.923.545 1.358 1.358 0 001.033-.288l-.04-.316zM114.076 179.077a1.326 1.326 0 00-.408.344 1.325 1.325 0 00-.279 1.008c.023.178.082.35.173.505l.632.632.277.039a1.396 1.396 0 001.6-1.126l-.454-.454a1.373 1.373 0 00-1.541-.948zM80.356 166.573a1.51 1.51 0 001.56-.494 1.366 1.366 0 00.139-1.521l-.553-.553a1.306 1.306 0 00-1.126.138 1.637 1.637 0 00-.711 1.067v.197l.079.356.671.672-.059.138zM108.466 160.311l-.316.316-.02.217.06.297.237.237a1.525 1.525 0 001.027.75 1.359 1.359 0 001.106-.316l.553-.553v-.237a1.566 1.566 0 00-.217-.691l-.731-.731-.612.296a1.456 1.456 0 00-.968.415h-.119zM84.05 167.343l.79.79c.22.034.443.013.653-.059l.573-.573.118-.316.118-.118a1.368 1.368 0 00-.059-.652l-.573-.573-.316-.119-.118-.118a1.283 1.283 0 00-.692.059l-.73.731c-.033.224-.02.453.04.672l.197.276zM86.618 169.911l-.118.119v.355l.296.297a1.396 1.396 0 00.988.711 1.483 1.483 0 001.185-.356l.178-.178.02-.256c-.031-.205-.105-.4-.218-.573a1.372 1.372 0 00-1.422-.672l-.79.79-.119-.237zM86.124 165.19l1.008 1.008a1.51 1.51 0 001.54-.514 1.312 1.312 0 00.12-1.501l-.514-.514-.316-.118-.356-.356a1.374 1.374 0 00-.652.059l-.908.909a1.2 1.2 0 00.039.632l.04.395zM94.54 160.686c.218.023.44.003.651-.059l.929-.928a1.54 1.54 0 00-.06-.613L94.975 158a.973.973 0 00-.632.039l-.593.593-.118.316-.119.119c-.053.209-.04.43.04.632l.987.987zM80.238 176.45l.513-.514-.098-.099-.06-.336-.77-.77a1.542 1.542 0 00-.967-.747 1.524 1.524 0 00-1.206.194l-.711.711-.02.336.099.099c.037.156.104.304.198.434a1.483 1.483 0 00.81.81 1.346 1.346 0 00.925.579 1.343 1.343 0 001.05-.302l.237-.395zM84.268 179.018l-.02.217.138.138a1.37 1.37 0 001.027.593 1.475 1.475 0 001.363-.652 1.34 1.34 0 00.02-1.403 1.287 1.287 0 00-.83-.592 1.372 1.372 0 00-1.086.257l-.553.553a1.115 1.115 0 00-.06.889zM108.268 173.348l.158.277.139.138.316.237.177.099.238.04.217.019.375-.059.198-.079.118-.079.178-.138.217-.297.079-.197.079-.237-.019-.217-.04-.356-.079-.198-.158-.276-.138-.138-.316-.237-.178-.099-.257-.059h-.197l-.376.059-.178.059-.158.079-.138.138-.237.316-.079.198-.059.217-.02.218.059.375.079.197zM83.991 201.26l.296-.296c.034-.198.02-.401-.04-.593l-.849-.849-.237-.04a1.342 1.342 0 00-1.046.336l-.593.593.099.098.079.317.83.829.236.04.119.118.375-.059.375-.375.356-.119zM71.843 192.509l-.691-.691h-.198l-.355.04-.119.118a1.427 1.427 0 00-.73 1.956l.73.731.336.019.217-.02a1.442 1.442 0 001.067-.829 1.481 1.481 0 00-.257-1.324zM68.366 216.885a1.31 1.31 0 00-1.573-.59 1.392 1.392 0 00-.52.314l-.495.493-.02.218.119.118c.01.216.071.425.178.613.136.32.39.575.711.711l.257.019.118-.118a1.418 1.418 0 00.83-.83 1.274 1.274 0 00.395-.948zM108.979 184.608l-.691-.692-.296-.019a.983.983 0 00-.396.158l-1.047 1.047-.039.237.04.434.79.79.276.04a1.449 1.449 0 001.166-.77 1.547 1.547 0 00.197-1.225zM86.816 206.692l-.672-.671-.237-.04-.316.079-1.047 1.047c-.022.219-.002.441.06.652l.71.711.317.079.099.099.375-.059.138-.138a1.345 1.345 0 00.573-1.759zM91.142 170.998l.276.276.336.02.099-.099.257.059.375-.059.889-.889.059-.257-.099-.098-.079-.356-.118-.118a1.357 1.357 0 00-.929-.613l-.276-.039-.395-.395h-.474l-.277.039a1.278 1.278 0 00-.276 0l-.257.059-.731.731-.06.257-.02.494.08.276a1.404 1.404 0 001.086.692l.534.02zM85.136 185.931l.257-.256.02-.218-.06-.336-.118-.118a1.23 1.23 0 00-.889-.652 1.318 1.318 0 00-1.126.257l-.493.494-.04.237.06.375.118.119c.206.331.531.571.909.671.394.061.797-.031 1.126-.257l.138-.138.098-.178zM135.825 174.316l.118.119.257-.059a1.352 1.352 0 00.77-.771 1.278 1.278 0 00-.158-1.066l-.671-.672-.237-.04h-.198l-.276.079-.81.81-.06.257v.514l.455.454c.212.225.502.359.81.375zM42.785 134.533l.257.256.099-.098a1.253 1.253 0 001.145-.633c.13-.206.192-.448.178-.691l.119-.118-.06-.415-.296-.297a1.741 1.741 0 00-1.106-.79l-.257-.02-.257.257c-.28.143-.51.371-.651.652l-.297.296-.02.218.1.099.059.256.493.494.494.534zM22.814 149.052l.336-.336.04-.237-.396-.395a1.4 1.4 0 00-1.52-.652l-.732.731a1.48 1.48 0 00.158 1.066 1.23 1.23 0 00.948.633 1.435 1.435 0 001.166-.455v-.355zM73.502 149.565l.04.277.928.928h.513l.277-.079c.345-.14.714-.214 1.086-.217l.85-.85.06-.256v-.474l-.06-.257a1.342 1.342 0 00-1.027-.672l-.139-.138-.474-.04-.099-.098-.533-.139h-.316l-.237.237h-.197a1.282 1.282 0 00-.73.741 1.282 1.282 0 00.058 1.037zM77.67 146.464l.099.099.355.039.1-.099a1.505 1.505 0 001.224-1.224l-.04-.316-.217-.534-1.106-1.106-.435.04-.987.987-.04.277.119.118.04.316.236.514.652.652v.237zM67.496 152.054l.158-.158.08-.277.02-.256-.08-.435-.158-.158a1.374 1.374 0 00-.968-.573 1.423 1.423 0 00-1.027.356l-.514.513v.435l.119.355.099.099a1.287 1.287 0 001.6.573l.434-.434.237-.04zM50.687 163.807l-.277-.276-.375-.099-.138-.138a1.118 1.118 0 00-.988.316 1.34 1.34 0 00-.395.988l-.119.118.08.395c.17.285.447.491.77.573.328.083.676.034.968-.138a1.426 1.426 0 00.474-1.739zM90.766 117.88l.277-.276a1.312 1.312 0 00-.178-.968 1.455 1.455 0 00-1.412-.635 1.353 1.353 0 00-.876.535 1.339 1.339 0 00-.22.475l.079.395.375.573.513.514.395-.079.178-.178.87-.356zM40.178 102.848l-.08-.395-.394-.593-.277-.276a1.308 1.308 0 00-.474-.04l-.395.079-.573.376-.276.276c-.074.346-.01.707.177 1.007.196.285.493.483.83.554a1.43 1.43 0 001.462-.988zM100.367 111.204h-.356l-.257.257a1.304 1.304 0 00-.909.908c-.027.123-.033.25-.02.375l.1.099.118.316.711.711.296.06h.198l.356-.04.928-.928a1.425 1.425 0 00-.869-1.66l-.296-.098zM74.174 116.261l-.336.059-.494.257-.217.217a1.09 1.09 0 00-.178.415 1.284 1.284 0 00.217 1.047 1.424 1.424 0 001.146.592l.276-.039.672-.672c.07-.115.116-.242.138-.375a1.34 1.34 0 00-1.027-1.541l-.099-.099-.098.139zM100.288 122.187l-.711.711-.138.375a1.284 1.284 0 001.56 1.561l1.008-1.008a1.547 1.547 0 00-.553-1.501l-.159-.158-.434-.119h-.237l-.336.139zM61.67 114.345l.177-.178a1.202 1.202 0 00.633-.869 1.872 1.872 0 00-.277-1.304l-.652-.652h-.197l-.336.059-.119.119a1.346 1.346 0 00-.671.869c-.311.133-.559.38-.692.691l-.138.139v.395l.889.889a1.676 1.676 0 001.481-.218l-.098.06zM50.825 103.737l.593-.593a1.37 1.37 0 00-.573-1.442 1.48 1.48 0 00-1.719.06l-.415.414-.079.277a1.369 1.369 0 00.573 1.442 1.479 1.479 0 001.62-.158zM113.404 140.992l-.119.118.02.376a1.168 1.168 0 00.612.928l.099-.098c.297-.056.576-.185.81-.376l.613-.612.039-.237-.099-.099-.059-.336-.119-.118a1.653 1.653 0 00-1.481-.692 1.28 1.28 0 00-.316 1.146zM74.213 96.665l.336-.099.77-.77a1.452 1.452 0 00-.652-1.521l-.158-.158-.435-.04-.217.02-.415.415a1.006 1.006 0 00-.434.434l-.138.139v.395l.039.237.395.395c.167.257.408.456.692.573l.217-.02zM88.613 99.964a1.368 1.368 0 00-1.343-1.66l-.118.119-.297.099-.77.77.02.218.059.375.988.988.217.019.375-.099.731-.73.138-.1zM68.524 117.643l.1-.098-.02-.376-.317-.316a1.347 1.347 0 00-.573-.573l-.138-.138-.375-.02-.217.02-.554.553a1.596 1.596 0 00-.355.672.754.754 0 00-.02.296l.099.296.691.692c.386.137.81.123 1.186-.04a1.386 1.386 0 00.493-.968zM135.034 134.414l-.415.218-.138.138a1.592 1.592 0 00-.435.75c-.03.17-.03.344 0 .514l.316.316a1.289 1.289 0 001.462.593l.83-.83.099-.336a1.451 1.451 0 00-.652-1.521l-.158-.158-.573-.099-.336.415zM81.226 96.369l.177.177.494.139h.277l.988-.988a1.425 1.425 0 00-.87-1.66l-.138-.138-.138.139a1.398 1.398 0 00-.632.869 1.453 1.453 0 00-.158 1.462zM111.725 103.124c.207.194.469.318.75.356h.198l.356-.119.849-.849a1.513 1.513 0 00-.533-1.442l-.158-.158-.455-.099-.257.02-.375.099-.138.138c-.24.119-.434.313-.553.553l-.06.257v.435l.06.256.513.514-.197.039zM78.717 96.823c.029-.123.035-.25.02-.375l-.02-.218-.237-.237a1.313 1.313 0 00-1.186-.671 1.537 1.537 0 00-1.303.71l-.099.1-.06.256v.198l.06.257.099.099a1.453 1.453 0 001.501.671 1.341 1.341 0 001.225-.79zM116.249 94.413l-.356.04-.81.81a1.318 1.318 0 00.217 1.126 1.493 1.493 0 00.988.592l.296-.06.692-.69.118-.356a1.337 1.337 0 00-.829-1.501l-.237-.04-.079.08zM99.715 104.705l-.81-.81-.217-.02-.316.04-.119.118a1.375 1.375 0 00-.77.968 1.425 1.425 0 001.027 1.659l.257.02.434-.079.554-.553a1.34 1.34 0 00.335-.691l.02-.218-.079-.395-.316-.039zM81.443 105.633c.099.137.219.256.355.356.14.2.314.374.514.513 0 .248.075.489.217.692l-.296.059-.692.691c-.089.124-.15.266-.177.415a1.286 1.286 0 00.71 1.422l.12.119.375.059.217-.02 1.027-1.027-.06-.375-.394-.395.257-.059.118-.119a1.395 1.395 0 00.988-.987l.355-.356-.83-.83a1.425 1.425 0 00-.493-.494l-.435-.434a1.11 1.11 0 00-.652-.652l-.316-.04-.257-.019-.414.098-.968.968c.009.203.049.403.118.593l-.099.099c.259-.023.506-.119.712-.277zM75.556 60.674l-.08-.395-.434-.435a1.117 1.117 0 00-.908-.434 1.286 1.286 0 00-.948.474l-.415.415a1.285 1.285 0 00.968 1.52c.3.036.604.002.888-.098a1.452 1.452 0 00.929-1.047zM79.863 91.174l.059.375a1.313 1.313 0 001.185 1.185l.375-.375a1.423 1.423 0 00.79-.79l.139-.139-.158-.158a1.62 1.62 0 00-1.56-.691 1.342 1.342 0 00-1.087 1.086l.257-.493zM89.304 94.137l.573-.573a1.396 1.396 0 00-1.086-1.68 1.564 1.564 0 00-1.126.178 1.425 1.425 0 00-.613.89l.06.414.158.158a1.2 1.2 0 00.79.79 1.286 1.286 0 001.205-.098l.04-.08zM84.781 90.403l.89-.889c.12-.224.181-.476.177-.73l-.514-.514a1.34 1.34 0 00-1.343-.277 1.256 1.256 0 00-.909.909 1.09 1.09 0 00.158.948c.067.103.154.19.257.257a1.397 1.397 0 001.126.494l.158-.198zM80.12 85.741l.493.257.316.04.316-.04.79-.79c.05-.042.096-.089.139-.138a1.397 1.397 0 00-.949-1.7 1.117 1.117 0 00-.908-.197 1.256 1.256 0 00-.929.613l-.099.098-.098.297.02.217.414.415c.141.211.323.392.534.533l-.04.395zM53.116 89.91a1.314 1.314 0 00-1.6-1.6l-.79.79-.04.236v.277l.395.395a1.341 1.341 0 002.134-.75l-.099.651zM120.022 78.314v.395l.019.217.356.356c.114.196.278.36.474.474l.138.138h.395l.474-.474c.181-.119.336-.273.455-.454l.138-.138v-.396l-.118-.118-.119-.316-.138-.138a1.619 1.619 0 00-.455-.455l-.138-.138h-.395l-.494.494a1.008 1.008 0 00-.434.434l-.158.119zM111.33 91.588l-.316.316a1.09 1.09 0 00-.454.455l-.059.257.059.493.059.257.79.79a1.341 1.341 0 001.6-1.007l.02-.296-.118-.119-.099-.296-.751-.75h-.197l-.534-.1zM99.695 79.242l.158-.158-.099-.098.02-.218-.079-.355-.75-.75a1.314 1.314 0 00-1.526 1.829c.125.261.336.473.597.6a1.257 1.257 0 001.68-.85zM96.83 87.914l-.967.968a1.37 1.37 0 001.481 1.68h.237l.356-.08.988-.988v-.197l-.06-.375-.869-.87-.474-.079-.099-.098-.592.04zM95.981 120.488l-.099.099a1.421 1.421 0 00-1.047 1.047 1.45 1.45 0 00.613 1.481 1.448 1.448 0 001.64.02l.572-.573.04-.237-.02-.217-.099-.099a1.42 1.42 0 00-1.126-1.126l-.099-.099-.375-.296zM137.424 75.094l.238.04.474-.08.296-.296a1.427 1.427 0 00.889-.889l.02-.256-.06-.257-.415-.415a1.396 1.396 0 00-.987-.395l-.277.079-.513.513c-.12.174-.207.368-.257.573a1.372 1.372 0 00.592 1.383zM94.263 168.706a1.259 1.259 0 00.553 1.344l.296.098h.395c.308.034.618-.028.89-.177.287-.205.49-.507.572-.85l-.098-.099v-.197a1.315 1.315 0 00-1.304-.988 1.453 1.453 0 00-1.304.869zM31.288 85.11a1.424 1.424 0 001.56-1.087c.02-.17.02-.343 0-.514l-.394-.395a1.369 1.369 0 00-1.225-.553l-.276.079-.534.533a1.76 1.76 0 00-.355.672v.198l.75.75.356.08.118.236zM139.36 89.988l-.77.77a1.424 1.424 0 001.007 1.64 1.286 1.286 0 001.58-.79l.099-.099.02-.296-.257-.257a1.34 1.34 0 00-1.541-.948l-.138-.02zM101.552 98.087a1.285 1.285 0 00-.415-1.244l-.178-.178-.454-.099-.119-.118-.217.02-.376.098-.83.83a1.34 1.34 0 00.692 1.52l.119.12.336.02.098-.1.237-.04a1.366 1.366 0 001.107-.829zM96.258 119.184l.553-.553a1.594 1.594 0 00-.276-1.383l-.198-.197a1.787 1.787 0 00-.81-.81 1.983 1.983 0 00-2.035.177l-.533.534-.02.257.1.098.098.376.296.296a1.454 1.454 0 00.573.968l.593.593a1.427 1.427 0 001.165-.06l.198-.197.296-.099zM47.467 136.291l.316-.119.75-.751.04-.237a1.253 1.253 0 00-.257-.967 1.247 1.247 0 00-.987-.435l-.1.099a1.281 1.281 0 00-1.105.79l-.119.118.02.376.533.533a1.361 1.361 0 00.988.395l-.08.198zM63.862 123.846l.592.593a1.634 1.634 0 001.264-.079 1.423 1.423 0 00.653-.929.277.277 0 00.098-.099v-.197l-.079-.316-.177-.178a1.293 1.293 0 00-1.008-.494l-.276.079-.475.474c-.17.192-.293.423-.355.672l-.02.296.099.099.079.316-.395-.237zM78.677 107.806v-.277l-.138-.454-.73-.731-.514-.079-.277.04-.316.118a1.37 1.37 0 00-.375.376l-.198.197-.099.573.02.296.139.336.296.296c.23.283.551.479.909.553h.236l.376-.098.513-.514c.096-.198.15-.413.158-.632zM68.682 105.594l-.138.138a1.312 1.312 0 00.217 1.126 1.423 1.423 0 001.106.592l.336-.098.375-.376h.237l.238-.039.968-.968v-.237l-.08-.395-.513-.514-.376-.178-.098-.098a1.366 1.366 0 00-1.126.256 1.592 1.592 0 00-.909.198l-.415.415.178.178zM101.611 94.512l.435.434a1.48 1.48 0 001.718-1.126l.02-.217-.059-.375-1.087-1.087h-.276l-.119.119-.355.158-.158.158c-.234.199-.394.47-.455.77a1.417 1.417 0 00-.019.573l.592.593h-.237zM91.201 112.902a1.55 1.55 0 00.158-.158 1.423 1.423 0 00-.691-1.521l-.316-.316-.257-.019-.217.019-.376.139-.138.138c-.278.229-.46.553-.513.909a.582.582 0 00-.04.276l.08.316.69.692a1.687 1.687 0 001.344 0l.573-.573-.297.098zM80.514 112.369l.08-.276-.04-.553-.158-.159a1.062 1.062 0 00-.178-.493 1.453 1.453 0 00-.099-.573l-.81-.81a1.873 1.873 0 00-1.462.118l-.355.356a1.283 1.283 0 00-.77.77l-.1.099-.019.257.375.375c.11.319.318.595.593.79l.593.593a1.508 1.508 0 002.35-.494zM85.947 112.586l-.1-.098-.078-.316-.613-.613a1.43 1.43 0 00-1.574.332c-.132.137-.234.3-.302.478l-.119.119.139.138a1.43 1.43 0 001.422.672 1.338 1.338 0 001.086-1.087l.138.375zM94.282 109.07l-.79.79a.453.453 0 00-.138.139 1.34 1.34 0 00.85 1.639l.118.119h.355l.633-.632c.162-.203.277-.439.335-.692l.04-.276-.099-.099-.099-.336-.908-.909h-.198l-.099.257zM41.698 116.498a1.37 1.37 0 00-.237-1.107 1.623 1.623 0 00-1.225-.592H40l-.316.316a1.507 1.507 0 00-.632.909 1.563 1.563 0 00.237 1.185l.494.494a1.544 1.544 0 001.284-.099 1.345 1.345 0 00.632-1.106zM113.779 122.799l.652-.652a1.364 1.364 0 00-.679-1.603 1.372 1.372 0 00-.605-.155l-.276.277a1.255 1.255 0 00-1.047 1.047 1.23 1.23 0 00-.02.533l.79.79c.181.134.391.222.613.257l.098.099.336-.06.119-.118.019-.415zM119.587 119.718l-.909.908a1.814 1.814 0 00.613 1.798l.118.118.296.099.218.217.098-.098.297-.099.099-.099a1.374 1.374 0 00.632-.909c.05-.291.023-.591-.079-.869l-1.028-1.027h-.197l-.158-.039zM87.468 117.782l.079-.277a1.369 1.369 0 00-.257-1.165 1.453 1.453 0 00-1.087-.573l-.098.099-.297.059-.71.711a2.164 2.164 0 00-.178.178 1.452 1.452 0 001.362 1.758l.1-.099.296-.099.493-.494.297-.098zM115.419 132.577l-.138-.138a1.201 1.201 0 00-.948-.948l-.297.019-.928.929a1.418 1.418 0 00.336 1.205c.21.204.468.353.75.434a1.365 1.365 0 001.62-1.027l-.395-.474zM132.545 153.219l-.632.633c-.195.299-.265.664-.195 1.014.07.351.275.66.57.862a1.595 1.595 0 001.758.02l.672-.672a1.457 1.457 0 00-.375-1.284 1.501 1.501 0 00-1.462-.395l-.336-.178zM74.805 112.31l-.098.099-.297.059-.671.672a3.006 3.006 0 00-.218.217 1.369 1.369 0 00.316 1.224 1.532 1.532 0 001.442.455l.83-.83.04-.276-.06-.494-.158-.158a1.623 1.623 0 00-1.126-.968z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M85.552 166.79l-.08-.395-.197-.197c-.1-.16-.235-.295-.395-.395l-.197-.198h-.435l-.395.079-.711.711c-.08.337-.023.692.158.988.202.29.505.494.85.573a1.37 1.37 0 001.402-1.166zM107.893 146.701a1.51 1.51 0 00-.869-1.738l-.138-.139-.455-.019-.237.039-.849.849-.079.317.02.217.079.395.731.731.197-.198-.079.277.454.138h.277l.296-.059.731-.731-.079-.079zM101.335 117.07l-.04.237-.098.099.059.375.849.85.237.039.099.099.375-.059.949-.948a1.344 1.344 0 00-.198-1.106 1.483 1.483 0 00-.988-.593l-.355.118-.889.889zM126.027 134.572a1.311 1.311 0 00-.81-1.323l-.257-.02-.513.079a1.313 1.313 0 00-1.008 1.007c-.036.162-.043.33-.02.494l.929.929a1.317 1.317 0 001.6-.692l.138-.138-.059-.336zM86.619 65.85l.83-.83.058-.297v-.197l-.039-.356-.75-.75a1.453 1.453 0 00-1.561.494l-.198.197-.079.514v.276l.119.316.395.395c.207.194.47.318.75.356l.1-.099.375-.02zM50.627 125.229l.198-.198.06-.533-.06-.296-.83-.83a1.362 1.362 0 00-1.185.039 1.146 1.146 0 00-.612.85v.237l.138.138a1.455 1.455 0 001.823.576c.213-.093.401-.237.547-.418l-.079.435zM58.35 110.196l.455-.454a.84.84 0 00.316 0 1.512 1.512 0 001.403-1.086l-.04-.356-.632-.632c-.43-.23-.935-.28-1.402-.138-.23.133-.42.324-.554.553l-.098.098a1.37 1.37 0 00-1.027 1.028.887.887 0 00-.02.335l.098.297.731.731.237.039.218.02.316-.435zM104.238 75.568l-.059.336-.099.099.06.336.81.81a1.423 1.423 0 001.145-.08 1.478 1.478 0 00.692-.968l.039-.236-.099-.1-.098-.335-.79-.79-.257-.06h-.198l-.375.06-.731.73-.04.198zM146.768 111.54l-.04.276.099.454.454.455c.14.2.314.374.514.513l.118.119.336.02.218-.02.73-.731a1.29 1.29 0 00.178-.454 1.37 1.37 0 00-.988-1.66 1.369 1.369 0 00-1.619 1.028zM125.908 123.155a1.313 1.313 0 00-.968.73l-.019.218-.02.415.533.533c.242.276.582.447.948.474l.119-.119.336-.138.711-.711a1.233 1.233 0 00-.81-1.521l-.119-.118-.711.237zM121.325 147.451l-.316.317a1.08 1.08 0 00-.454.454l-.158.158.039.474.948.948a1.34 1.34 0 001.62-.987l.04-.317-.099-.098-.119-.316-.948-.949-.553.316zM92.209 96.487l.474.474a1.425 1.425 0 001.343.277 1.369 1.369 0 00.928-.928 1.23 1.23 0 00-.256-1.047l-.593-.593a1.368 1.368 0 00-1.086.138 1.313 1.313 0 00-.613.89l-.197.79zM59.812 88.704l.613.613c.188.145.403.252.632.316l.336-.02.553-.198.651-.652-.039-.592a1.62 1.62 0 00-1.225-1.225l-.098-.099-.356.08-.83.829-.079.316.1.099.039.355-.297.178zM127.469 100.26l-.336.138-.158.158a1.575 1.575 0 00-.454.455l-.139.138-.02.415.119.118.158.356a1.399 1.399 0 00-.237.237l-.119.118-.019.218-.04.237.04.276.612.613a1.286 1.286 0 001.244.138 1.251 1.251 0 00.85-.849 1.366 1.366 0 00-.178-1.047l.336-.336a1.426 1.426 0 00-.652-1.521l-.138-.138-.435-.04-.434.316zM123.439 103.421a1.455 1.455 0 00-1.225.079 1.645 1.645 0 00-.75.948l-.02.257.375.375c.308.332.733.531 1.186.553h.197l.336-.099.79-.79.099-.375.099-.099-.06-.336-.691-.691-.336.178zM97.364 129.654a1.429 1.429 0 00-.61-1.468 1.41 1.41 0 00-.516-.211 1.277 1.277 0 00-1.007.177 1.289 1.289 0 00-.593.83l.08.395.375.573.513.513.415-.059.573-.375.77-.375zM59.14 122.483l.356-.04 1.008-1.007-.02-.217-.079-.395-.909-.909a1.261 1.261 0 00-1.377.705 1.258 1.258 0 00.35 1.507l.454.257.218.099zM108.683 115.569l-.02.217.158.158a1.506 1.506 0 001.482.652 1.394 1.394 0 001.106-1.106c.042-.141.055-.288.04-.434l-.257-.257a1.366 1.366 0 00-1.304-.237 1.367 1.367 0 00-1.205 1.007zM124.368 113.989l-1.028 1.027a1.54 1.54 0 00.217 1.154c.223.34.571.579.969.663l.138.139.435-.04 1.046-1.047.04-.237-.02-.257-.592-.592a2.813 2.813 0 00-.929-.455l-.276-.355zM68.583 104.942l.296-.06.731-.73.119-.356a1.425 1.425 0 00-.672-1.541l-.099-.099-.434-.158h-.237l-.297.06-.474.474c-.177.187-.3.42-.355.671l-.02.297.237.237c.308.332.733.53 1.185.553l.02.652zM107.103 112.725l.118.118.376-.02.336-.335c.22-.144.409-.333.553-.554l.039-.237.02-.414-.059-.257-.81-.81-.257-.059h-.198l-.414.098-.87.87a1.228 1.228 0 001.166 1.6zM68.82 110.394a1.251 1.251 0 00.593-.83l-.06-.415-.414-.415a1.004 1.004 0 00-.494-.493l-.75-.751a1.534 1.534 0 00-1.206.099l-.83.829a1.34 1.34 0 00.593 1.423l.277.079.257.019a1.349 1.349 0 00.79.791 1.464 1.464 0 001.244-.336zM122.59 89.969c.056-.05.109-.102.158-.158l.395-.04a1.2 1.2 0 00.889-.889l.118-.118-.039-.356-.04-.276-.869-.87a1.65 1.65 0 00-.968.06l-.099.098a1.311 1.311 0 00-1.323 1.008v.197l.138.139a1.54 1.54 0 001.64 1.205zM135.192 113.673l-.335.099-.158.158a1.811 1.811 0 00-.435.434l-.138.139-.02.414.039.237.415.415c.124.142.276.257.446.338.17.082.354.128.542.136l.099-.098.336-.139.632-.632a1.338 1.338 0 00-.652-1.521l-.158-.158-.415-.02-.198.198zM91.003 47.518s.1-.099.02-.257l-.079-.316-.691-.691a1.368 1.368 0 00-1.264.04 1.452 1.452 0 00-.712.948l-.04.276.1.099.079.316.71.711c.407.144.854.123 1.245-.06a1.452 1.452 0 00.632-1.066zM53.116 122.582l.118-.119a1.286 1.286 0 00.968-.77l.02-.218v-.395l-.355-.355.02.02a1.311 1.311 0 00-.948-.475l-.12.119-.355.118-.711.712a1.284 1.284 0 00.79 1.501l.119.118.454-.256zM73.7 85.722l.177-.178v-.514l-.078-.276-.81-.81a1.313 1.313 0 00-1.6 1.007l-.04.316.099.099.099.296.75.75a1.453 1.453 0 001.403-.69zM100.09 139.451a1.668 1.668 0 00-.119-.474 1.341 1.341 0 00-1.284-.731 1.315 1.315 0 00-1.066 1.067 1.202 1.202 0 00.395 1.185l.494.494h.276l.494-.138.454-.455.139-.335.118-.119.099-.494zM82.766 132.478a1.34 1.34 0 00.04-.395l-.573-.573a1.493 1.493 0 00-.636-.267 1.497 1.497 0 00-1.28.406 1.423 1.423 0 00-.356.592l-.02.218.139.138a1.48 1.48 0 001.481.652 1.395 1.395 0 001.205-.771zM67.536 91.924l-.316-.316a1.648 1.648 0 00-1.008-.375l-.276.079-.632.632a2.63 2.63 0 00-.198.197 1.34 1.34 0 001.008 1.6 1.34 1.34 0 001.64-.928l.098-.099-.02-.336-.118-.118-.178-.336zM52.919 144.666l.73-.731c.039-.088.065-.181.08-.276a1.341 1.341 0 00-.988-1.62 1.368 1.368 0 00-1.66.948v.237l.06.415.553.553c.215.182.474.304.75.356l.475.118zM91.161 84.576l-.948.948a1.535 1.535 0 00.514 1.462l.158.158.454.099.257-.02.276-.08.81-.81a1.397 1.397 0 00-.85-1.639l-.236-.04-.435-.078zM97.839 65.85l-.356-.554-.434-.434-.336-.06-.297.02-.81.81a1.369 1.369 0 00.565 1.4c.145.098.31.166.482.2a1.369 1.369 0 001.64-1.047l-.454-.336zM73.561 75.47l.1-.1-.08-.394-.968-.968-.217-.02h-.198l-.296.06-.118.118a1.284 1.284 0 00-.514.513l-.316.316-.04.237.02.257.138.455.89.888a1.453 1.453 0 001.6-1.362zM107.794 104.369l-.336-.336-.316-.079h-.198l-.335.059-.791.79a.457.457 0 00-.138.139 1.433 1.433 0 00.198 1.145 1.54 1.54 0 001.027.593l.79-.79c.17-.182.287-.408.336-.652l.039-.316-.098-.099-.099-.336-.415-.414.336.296zM64.158 79.302l.553-.553.119-.435v-.277l-.79-.79-.415-.217-.158-.158c-.31.005-.613.08-.89.217l.12.356-.238-.237-.573.573-.059.335.02.297.197.474.218.217a2.152 2.152 0 001.679.296l.217-.098zM116.663 62.156l.296-.1.316-.315c.209-.18.355-.423.415-.692a1.256 1.256 0 00-.889-1.442l-.217-.02-.119.12a1.257 1.257 0 00-1.303.829l-.119.118.02.375.316.317c.128.233.32.425.553.553l.296.059.435.198zM89.088 80.684l.336-.098.81-.81a1.481 1.481 0 00-1.107-1.739l-.217-.02-.395.08-.988.987v.237l.139.336.138.139a1.62 1.62 0 001.185.553l.099.335zM107.458 81.475l-.85.849-.059.257v.197l.059.376 1.047 1.047.356-.08.415-.414c.235-.209.4-.485.474-.79l-.533-.534-.396-.632-.138-.138h-.197l-.178-.138zM91.34 148.558l-.12.118.02.376.89.888a1.405 1.405 0 001.114.004 1.397 1.397 0 00.762-.814l.118-.118-.02-.375-.849-.85a1.561 1.561 0 00-1.165.02c-.34.142-.61.411-.75.751zM80.534 181.289a1.563 1.563 0 00.71-.987l.02-.257-.059-.257-.493-.494a1.585 1.585 0 00-1.502-.197 1.311 1.311 0 00-.85.849 1.2 1.2 0 00.14.928l.67.672a1.474 1.474 0 001.364-.257zM73.759 148.281l.02-.257-.08-.316-.75-.75a1.48 1.48 0 00-1.58.592l-.257.257-.06.257.02.336.79.79a1.449 1.449 0 001.739-.632l-.06-.06.1.099.118-.316zM90.806 137.515l-.277.277a1.368 1.368 0 00.376 1.205c.186.178.419.301.671.356a1.346 1.346 0 001.64-.968l-.06-.415-.197-.198a.869.869 0 00-.375-.375l-.02-.02a1.732 1.732 0 00-.533-.178l-.257.06-.198.197-.77.059zM69.848 133.585l.178-.178c.159-.101.294-.236.395-.395l.197-.198.119-.474v-.277l-.08-.316-.73-.73-.316-.08h-.198l-.375.06-.85.849a1.4 1.4 0 00.633 1.501l.158.158.474.04.395.04zM87.191 137.812l.593-.593a.949.949 0 000-.474l.395-.395a1.453 1.453 0 00-.257-1.166 1.277 1.277 0 00-1.047-.454l-.099.099-.296.099-.85.849v.198l-.552.553a1.255 1.255 0 00.948 1.54c.252.034.51.007.75-.079l.415-.177zM111.468 127.125a1.26 1.26 0 00-.988.988c-.03.17-.03.344 0 .513l.573.573a1.26 1.26 0 001.205.059 1.365 1.365 0 00.929-.928 1.256 1.256 0 00-.81-1.481l-.277-.04-.632.316zM86.974 139.214l.138.138a1.454 1.454 0 001.205.534 1.484 1.484 0 001.047-.771l.138-.138v-.435l-.04-.237-.77-.77-.395-.118h-.276l-.297.098-.85.85.1.849zM69.828 162.681l.02-.296-.297-.296a1.391 1.391 0 00-.75-1.146 1.31 1.31 0 00-.852-.129c-.29.053-.554.203-.748.425l-.178.178-.099.454-.099.099-.02.218.08.355 1.047 1.047a1.341 1.341 0 001.896-.909zM66.252 146.148h.198l.296-.06.118-.118a1.368 1.368 0 00.87-.869l.02-.376-.1-.098-.118-.316-.652-.652-.533-.099-.336.059-.83.83a1.52 1.52 0 00-.158.158 1.312 1.312 0 00.178 1.047 1.264 1.264 0 001.047.494zM105.503 152.923l-.988.988a1.362 1.362 0 00.237 1.106 1.514 1.514 0 001.087.612l.098-.098.316-.079.79-.79a1.329 1.329 0 00-.512-1.4 1.338 1.338 0 00-.475-.22l-.119-.119h-.434zM79.685 146.543l.316.316a1.426 1.426 0 001.244-.059 1.457 1.457 0 00.652-.929l.119-.118-.1-.336-.295-.297a1.394 1.394 0 00-.988-.474l-.277.079-.513.514c-.17.182-.287.408-.336.652l-.04.316.08.316.256.257-.118-.237zM78.4 157.98a1.341 1.341 0 001.146.474l1.047-1.047a1.425 1.425 0 00-.652-1.521l-.276-.276h-.198l-.276.276a1.54 1.54 0 00-.613.889 1.257 1.257 0 00-.177 1.205zM102.184 128.824l-.277.276a1.38 1.38 0 00.237 1.146c.188.249.452.429.751.514.067.103.154.19.257.257l.118.118h.396l.217-.02.375-.375c.168-.12.315-.267.435-.435l.118-.118.02-.415-.138-.138-.119-.316-.138-.139a1.609 1.609 0 00-.454-.454l-.257-.257a1.486 1.486 0 00-1.166.02l-.217.217-.158.119zM89.897 125.663H89.7l-.119.119a1.59 1.59 0 00-.553.869 1.423 1.423 0 001.976 1.58l.533-.533a.739.739 0 00.178-.178 1.422 1.422 0 00-.218-1.165 1.538 1.538 0 00-1.125-.573l-.475-.119zM79.052 124.735a1.348 1.348 0 001.186.514l.237-.04.316-.316c.244-.196.418-.466.493-.77a1.175 1.175 0 00-.04-.672l-.75-.751a1.486 1.486 0 00-1.224 0 1.459 1.459 0 00-.712.949 1.258 1.258 0 00.494 1.086zM60.227 130.562l.317-.079.81-.81.059-.296-.099-.099-.06-.375-1.007-1.008h-.237l-.415.099-.79.79-.059.297a1.368 1.368 0 001.244 1.521l.237-.04zM97.779 153.14a1.398 1.398 0 00-1.68 1.008l.02.257.435.434a1.395 1.395 0 001.916.297l.415-.415c.051-.101.091-.207.119-.316a1.399 1.399 0 00-1.225-1.265zM81.877 120.843a1.422 1.422 0 001.462.711 1.34 1.34 0 001.067-1.066v-.198l-.119-.118a1.334 1.334 0 00-.928-.652 1.567 1.567 0 00-1.244.296l-.633.632.08.356.316.039zM106.609 135.619l-.297.099-.711.711a.443.443 0 00-.138.138 1.364 1.364 0 00.237 1.225 1.315 1.315 0 001.007.533l.297-.059.533-.533c.148-.201.256-.429.316-.672a.684.684 0 00.04-.276l-.079-.277-.494-.494a1.398 1.398 0 00-.988-.395h.277zM84.03 123.945l-.019.415.138.138.119.316.138.138c.102.194.26.352.454.454l.139.139h.395l.474-.474c.194-.103.352-.261.454-.455l.138-.138v-.395l-.474-.474a1.313 1.313 0 00-.474-.474l-.138-.138h-.395l-.454.454a1.534 1.534 0 00-.475.474l-.02.02zM84.13 127.56l-.553.553a1.66 1.66 0 00-.159.158 1.45 1.45 0 00.237 1.185c.137.17.308.309.502.407.194.099.407.156.624.166l.237-.04.593-.592c.118-.175.205-.369.257-.573a1.428 1.428 0 00-.593-1.462l-.138-.138-.415-.099-.118-.118-.277.079-.197.474zM87.013 123.806v.395l.099.099-.099.099.158.158a1.48 1.48 0 001.482.691 1.34 1.34 0 001.067-1.066l.078.158-.355-.356-.138-.336-.139-.138a1.085 1.085 0 00-.454-.454l-.119-.119-.414-.02-.494.494a1.806 1.806 0 00-.435.435l-.237-.04zM96.08 125.209l-.02.415.475.474c.107.19.264.347.454.454l.138.138.376.02.138-.138.316-.119.138-.138c.187-.096.339-.248.435-.434l.138-.139v-.395l-.02-.217-.355-.356a1.813 1.813 0 00-.474-.474l-.119-.118h-.395l-.237.039-.355.356c-.181.119-.336.273-.455.454l-.177.178zM75.004 127.915l.335-.059.79-.79.08-.395.02-.218-.08-.316a1.535 1.535 0 00-1.185-1.896l-.099-.099-.316.04-.73.731a3.008 3.008 0 00-.139.454c-.09.388-.034.796.158 1.146a1.369 1.369 0 001.245 1.56l-.08-.158zM76.149 132.577l-.632.632a1.313 1.313 0 00.316 1.146 1.426 1.426 0 001.442.454 1.317 1.317 0 00.987-.987 1.12 1.12 0 00-.197-.988 1.341 1.341 0 00-.968-.573l-.099.099c-.3-.037-.604.04-.85.217zM97.503 144.923l-.08.316v.197l.06.376.079.197.138.218.138.138.317.237.197.079.138.02.218.019.355-.079.198-.079.217-.138.138-.138.237-.316.08-.198.059-.296.019-.217-.079-.356-.079-.197-.138-.218-.138-.138-.296-.217-.198-.079-.158-.04h-.198l-.375.06-.198.079-.217.138-.138.138-.217.296-.08.198zM60.682 148.439l.395-.395c.172-.1.315-.243.415-.415l.138-.138.02-.415-.553-.553a1.315 1.315 0 00-.968-.454l-.099.098-.336.099-.909.909v.197l.08.396.849.849.099-.099.869-.079zM57.857 133.841v-.197l-.06-.376-.83-.829a1.43 1.43 0 00-1.148.036 1.422 1.422 0 00-.748.872l-.098.099.02.336.77.77a1.475 1.475 0 001.343.119 1.342 1.342 0 00.75-.83zM38.597 149.032a1.424 1.424 0 00-.257-1.126 1.449 1.449 0 00-1.086-.534l-.119.119-.316.118-.434.435c-.143.175-.25.376-.316.593-.063.342.008.695.197.987l.415.415a1.481 1.481 0 001.225-.039 1.407 1.407 0 00.691-.968zM90.273 153.615l.099-.099-.02-.296-.593-.593a2.339 2.339 0 00-.395-.158l-.237-.04-.395.04-.889.889v.276l.455.455a1.458 1.458 0 001.363.217c.27-.162.483-.404.612-.691zM59.002 154.425a.76.76 0 00.02-.297l-.099-.099-.118-.316-.731-.73h-.197l-.356.039-.296.296c-.184.115-.34.271-.455.455l-.04.276.04.474.08.277.77.77a1.338 1.338 0 001.382-1.145zM86.638 131.451l1.521 1.521h.237l.395-.079.83-.83a1.283 1.283 0 00-.237-1.106l-.217-.217-.1-.296-1.105-1.107-.455-.098h-.276l-.277.079-.73.731a1.395 1.395 0 00.335 1.244l.08.158zM72.02 138.167l.81-.81c.076-.38.006-.775-.197-1.106a1.48 1.48 0 00-.988-.593l-.099.099-.355.079-.79.79a1.313 1.313 0 00.217 1.126 1.449 1.449 0 00.968.573h.197l.237-.158zM116.723 164.578l.375.375a1.48 1.48 0 001.087-.138c.159-.096.296-.225.401-.379.106-.153.178-.327.211-.51l.059-.296-1.047-1.047-.415-.099h-.276l-.296.099-.593.592a1.148 1.148 0 00.494 1.403zM76.643 71.677l.513.513a1.285 1.285 0 001.245.336c.25-.055.473-.195.632-.395l.71-.711a1.674 1.674 0 00-.236-1.343l-.474-.474a1.62 1.62 0 00-.929-.02l-1.086 1.086a.896.896 0 00-.04.277l.1.099-.435.632zM52.11 68.595l.947-.948a1.48 1.48 0 00-.671-1.54l-.277-.08h-.197l-.277.08-.099.098a1.537 1.537 0 00-.612.889A1.259 1.259 0 0051.2 68.2a1.285 1.285 0 001.146.474l-.237-.079zM88.573 103.796l-.434.435-.119.474v.276l.08.316.75.751c.35.14.677.334.968.573l.158.158.435.079.256-.02.277-.079.691-.691a1.288 1.288 0 00-.296-1.166l-1.245-1.244-.335-.099-.138-.138a1.282 1.282 0 00-1.047.375zM93.63 104.468l.494.494a1.542 1.542 0 001.739-.04l.217-.217.198-.553v-.277c.03-.035.063-.068.098-.099l.04-.237.02-.415-1.067-1.066-.336-.06h-.237l-.415.099-.79.79-.198.514-.02.296.02.297.237.474zM82.51 101.445l.967-.968a1.311 1.311 0 00-.296-1.086 1.256 1.256 0 00-.988-.435l-.118.119-.336.099-.454.454-.158.356a1.285 1.285 0 00.77 1.521l.138.138.474-.198zM62.203 98.443l-.059-.376-.415-.414a1.089 1.089 0 00-.948-.435 1.312 1.312 0 00-.948.435l-.415.414a1.283 1.283 0 00.178.968 1.256 1.256 0 00.79.553 1.397 1.397 0 001.817-1.145zM101.651 65.08l-.395.039c-.311.132-.559.38-.692.691a1.401 1.401 0 00.099 1.086 1.316 1.316 0 001.254.776c.175-.007.346-.05.504-.124l.296-.296.198-.671-.02-.376-.691-.691-.553-.435z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M121.701 113.811l-.277.277-.217.691.04.395c.093.129.206.243.335.336l.376.375.671.198.415-.02a1.294 1.294 0 00.711-.711 1.238 1.238 0 00-.079-.988 1.451 1.451 0 00-1.975-.553zM90.193 61.84l.553.552a1.34 1.34 0 001.284.06c.103-.081.196-.174.277-.277l.573-.573-.099-.098-.04-.316-1.086-1.087H91.3l-.098.099a.363.363 0 00-.158.158 1.368 1.368 0 00-.75 1.699l-.1-.218zM110.777 79.025l.316.316.513.237.316.04c.153-.029.3-.082.435-.158a1.284 1.284 0 00.751-.751 1.463 1.463 0 00-.277-1.264l-.632-.632-.296-.02a.945.945 0 00-.356.118 1.34 1.34 0 00-.593 1.778l.099-.098-.276.434zM98.945 55.025l.178.177h.553l.335-.059a.945.945 0 00.356-.118 1.259 1.259 0 00.711-.949 1.399 1.399 0 00-.474-1.264l-.454-.454-.218-.02-.256.06-.1.098a1.453 1.453 0 00-.829 1.343l.158.158.198.395.375.376-.533.257zM117.138 89.969l-.257.059a1.257 1.257 0 00-1.067.04 1.704 1.704 0 00-.849 1.046l-.099.099.04.356.79.79a1.227 1.227 0 001.086-.02c.304.157.655.199.988.118l.908-.908.04-.395-.138-.138a1.566 1.566 0 00-1.047-1.047h-.395zM115.696 105.08l-.218-.02-.237.04-.178.178c-.242.125-.443.32-.576.559a1.39 1.39 0 00-.174.784 1.508 1.508 0 001.205 1.205l.118.118.139-.138a.994.994 0 00.177-.178 1.31 1.31 0 00.751-1.343 1.515 1.515 0 00-.334-.744 1.517 1.517 0 00-.673-.461zM105.226 33.098l.455-.454a1.256 1.256 0 00.335-1.087l-.513-.513c-.23-.19-.5-.326-.79-.395h-.198l-.336.059-.79.79a1.542 1.542 0 00.376 1.56 1.227 1.227 0 001.461.04zM95.508 91.332l-.376-.376-.237-.04-.099-.098-.316.04-.118.118a1.425 1.425 0 00-.77 1.481l1.027 1.028c.202.043.412.036.612-.02l.968-.968a1.341 1.341 0 00-.237-1.067l-.454-.099zM89.009 78.077a1.37 1.37 0 00-.81.81 1.537 1.537 0 00.276 1.185l.494.494.237.04.1.098.315-.04.948-.948.02-.217-.04-.355-.869-.87-.276-.079-.099-.098-.336.02.04-.04zM115.3 82.68l-.75.75a1.314 1.314 0 00-.079.79l.869.87.336.059c.26.015.519-.04.75-.158l.87-.87-.001-.276-.098-.099a1.452 1.452 0 00-.83-.83 1.368 1.368 0 00-1.067-.236zM86.599 20.357l.414.178.139.138c.29.034.584-.014.85-.138.14-.094.261-.215.355-.356l.355-.355a1.285 1.285 0 00-.454-1.482l-.474-.474-.376-.06-.217.02a1.316 1.316 0 00-.296.1 1.48 1.48 0 00-.79 1.5l.138.139.296.494.06.296zM90.846 86.077l-.06-.257-.217-.415-.533-.533-.435-.079-.257.02-.158.158a1.397 1.397 0 00-.73 1.718l.572.573a1.202 1.202 0 001.067-.04 1.454 1.454 0 00.75-1.145zM76.978 58.145a1.397 1.397 0 00-.197-.79l-.553-.553-.257-.059h-.198l-.355.04-.139.138a1.537 1.537 0 00-.81 1.323l1.048 1.047c.248.082.512.102.77.06l.889-.89.04-.236-.238-.08zM92.742 87.776c-.108.076-.201.17-.277.276l-.395.395a1.37 1.37 0 00.198 1.344 1.593 1.593 0 001.402.533l.909-.909a1.452 1.452 0 00-.415-1.6 1.395 1.395 0 00-1.422-.04zM67.339 60.081l.435.435.553.119.316-.04a1.37 1.37 0 00.81-.81c.09-.384.026-.788-.178-1.126l-.672-.671-.316-.04a.95.95 0 00-.355.119 1.313 1.313 0 00-.672 1.58l.257.257-.178.177zM85.71 66.442l-.277.277.06.335.849.85a1.452 1.452 0 001.244-.02 1.397 1.397 0 00.613-1.837l-.632-.632-.277-.08-.118-.118a1.564 1.564 0 00-.731.178l-.435.435-.296.612zM97.917 80.072l-.059-.494c.057-.239.078-.485.06-.73a1.34 1.34 0 00.414-.613l.474.474.376-.02c.15-.016.296-.056.434-.118a1.285 1.285 0 00.672-1.462l-1.126-1.126-.395.04-.87.869-.217-.218a1.313 1.313 0 00-1.363-.098l-.098.099a1.09 1.09 0 00-.356.039l-.652.652-.118.355-.1.1-.098.374.099.1c-.07.223-.09.459-.06.69l-.02.336.198.198a1.172 1.172 0 00-.118.909l.928.928.297.06h.237l.414-.1.158-.157c.173-.115.32-.262.435-.435.215-.165.374-.393.454-.652zM66.588 112.981l-.276.277-.119.395-.158.158a1.175 1.175 0 00.257 1.008 1.199 1.199 0 00.948.434l.119.119.415-.02a1.262 1.262 0 00.688-1.224 1.259 1.259 0 00-.136-.475 1.231 1.231 0 00-.632-.632 1.342 1.342 0 00-1.106-.04zM87.567 90.502l.256-.257a1.312 1.312 0 00.158-1.66l-.572-.572a1.62 1.62 0 00-1.087-.06l-.593.593a1.648 1.648 0 00.455 1.68 1.313 1.313 0 001.501.197l-.118.079zM84.05 81.178h-.236l-.119-.118-.415.02a1.397 1.397 0 00-.612 1.916 1.313 1.313 0 00.79.79 1.396 1.396 0 001.087-.06l.414-.414a1.256 1.256 0 00.139-1.126 1.23 1.23 0 00-.85-.85l-.197-.158zM84.05 86.986l-.513-.514h-.316l-.119.119a1.285 1.285 0 00-.493.494l-.474.474a1.396 1.396 0 00.651 1.244 1.284 1.284 0 001.284.06 1.23 1.23 0 00.672-.672c.051-.115.091-.234.118-.356a1.396 1.396 0 00-.316-1.185l-.493.336zM83.28 93.603l-.118-.513-.652-.652-.553-.119-.297.06-.118.118a1.397 1.397 0 00-.75 1.817c.021.317.162.613.394.83a1.452 1.452 0 001.087.336l1.007-1.008c.045-.24.052-.487.02-.73l-.02-.139zM103.508 112.013a1.288 1.288 0 00-.217 2.233l.276.276.099-.099.257-.059.968-.968a1.315 1.315 0 00-.632-1.145 1.346 1.346 0 00-.751-.238zM52.405 67.193l.77-.77c.063-.226.07-.463.02-.692l-.928-.928a1.453 1.453 0 00-.672 0l-.908.908c-.05.221-.05.45 0 .672l.928.928c.208.043.423.043.632 0l.158-.118zM68.208 65.652l.316.316c.21.075.438.082.652.02l.711-.711.099-.297.02-.217-.04-.395-.138-.138a1.285 1.285 0 00-1.758-.613l-.692.692-.04.237.04.316.593.592.237.198zM65.917 82.482l-.514.513-.178.652.02.376a1.318 1.318 0 002.509-.81 1.312 1.312 0 00-1.837-.731zM74.53 79.242l.414.415h.355l.277-.276a1.34 1.34 0 00.889-.89 1.397 1.397 0 00-.375-1.204l-.929-.929-.395.04-1.067 1.067.02.375.83.83-.02.572zM100.327 59.45l.118.118a1.428 1.428 0 001.502.158 1.484 1.484 0 00.75-1.422 1.424 1.424 0 00-1.047-1.126l-.217-.02-.395.04-.573.572a1.4 1.4 0 00-.118 1.58l.098.1h-.118zM38.992 55.755l-.81-.81a1.313 1.313 0 00-1.244-.02.808.808 0 00-.178.178l-.533.534v.237c-.007.364.112.72.336 1.007l.513.514.257.02.217-.02c.207-.035.407-.102.593-.198a1.368 1.368 0 00.85-1.442zM138.906 30.47a1.314 1.314 0 00.711-1.264l-.119-.316-.276-.276a1.232 1.232 0 00-.632-.632 1.286 1.286 0 00-1.008.099l-.375.375a1.283 1.283 0 00.059 1.64 1.541 1.541 0 001.64.375zM113.483 131.984a1.398 1.398 0 00-1.817-.513c-.148.1-.276.227-.376.375l-.415.415a1.313 1.313 0 00.336 1.284l.494.494.474.079-.099-.099.06.296c.252.033.508-.016.73-.138l.534-.533.118-.356.119-.118-.356-.356.198-.83zM49.344 44.891l.335.336h.277a1.425 1.425 0 00.672-1.817 1.285 1.285 0 00-1.6-.731l-.633.632a1.34 1.34 0 00.237 1.778l.712-.198zM79.408 69.188a1.286 1.286 0 00-.711 1.106l.04.237.197.435.691.691.257.06h.197l.396-.04a1.313 1.313 0 00.75-1.502l-.73-.73a1.397 1.397 0 00-1.087-.257zM99.142 60.042l-.414-.02-.237.04a.39.39 0 00-.119.118 1.593 1.593 0 00-.988.988 1.142 1.142 0 00-.079.276c-.196.341-.26.742-.178 1.126a1.982 1.982 0 001.443 1.443l.375-.06.869-.869.02-.217a1.388 1.388 0 00.375-1.047l.119-.119v-.276a1.348 1.348 0 00-.83-.83l-.198-.197-.158-.356zM143.114 87.223l-.474-.474-.356.04-.178.177a1.152 1.152 0 00-.612.85 1.234 1.234 0 00.277 1.027l.177.177a1.288 1.288 0 001.324.376l.83-.83.039-.316a1.399 1.399 0 00-.316-.988l-.711-.04zM123.123 82.284l.217-.217v-.316l-.099-.099a1.537 1.537 0 00-.889-.889 1.481 1.481 0 00-1.126.06l-.138.138-.612.612a1.36 1.36 0 00.237 1.106l.434.435h.237c.254-.008.503-.068.731-.178l.533-.533.475-.119zM101.334 80.803l-.671.672-.059.296.039.237.079.276.237.435.435.435c.148.097.317.158.494.177l.335-.059.494-.257.455-.454.039-.316a1.336 1.336 0 00-.118-1.067l-.751-.75h-.198c-.3.043-.582.174-.81.375zM105.898 90.028l.118-.119a1.4 1.4 0 00.791-.79 1.545 1.545 0 00-.218-1.244l-1.244-1.245-.455.02-1.145 1.146-.079.276.118.119.06.415.098.098a1.343 1.343 0 00.81.81c.143.281.371.51.652.652l.099.099.395-.237zM76.603 71.4l-.098-.099-.06-.296-.118-.118a1.48 1.48 0 00-1.956-.613l-.73.731-.02.217.02.455 1.244 1.244c.295.057.6.015.869-.118.163-.111.304-.252.415-.415l.434-.435V71.4zM97.443 67.943a.56.56 0 00-.158.158 1.37 1.37 0 00-.731 1.482l.06.296.868.87c.35.048.708-.007 1.028-.159.074-.056.14-.123.197-.197l.692-.692-.02-.296a1.678 1.678 0 00-.929-.928l-.118-.119-.89-.415zM103.725 76.556l-.751.75c-.134.116-.26.242-.375.376-.15.112-.283.245-.395.395l-.395.395v.356l.118.118a1.79 1.79 0 001.027 1.027l.099.099h.316a1.371 1.371 0 001.087.099l.79-.79c.148-.301.222-.633.217-.968l-.079-.277-.02-.296-.098-.297a1.535 1.535 0 00-1.541-.987zM100.407 72.21l-.692.692.02.296.099.099a1.404 1.404 0 00.7.85 1.397 1.397 0 001.097.078l.573-.573a1.477 1.477 0 00-.395-1.54 1.313 1.313 0 00-1.521-.178.527.527 0 00.119.276zM92.446 67.924l.395.395h.356l.256-.257a1.397 1.397 0 00.751-1.699l-.81-.81-.336-.06c-.26-.008-.517.046-.75.159a.838.838 0 00-.198.197l-.474.474-.04.238-.098.098.04.316.651.652.257.297zM131.36 104.152a1.39 1.39 0 00-.79.79 1.614 1.614 0 00.277 1.343l.671.672a1.495 1.495 0 001.152-.106 1.475 1.475 0 00.725-.902l.02-.257.098-.098-.059-.257a1.416 1.416 0 00-.79-.91 1.416 1.416 0 00-1.205.021l-.099-.296zM90.964 44.16l-.119-.118h-.355l-.257.256a1.34 1.34 0 00-.87.87 1.342 1.342 0 00.198 1.145l.336.336a1.228 1.228 0 001.462.237c.145-.104.272-.23.375-.375l.454-.455.02-.217-.118-.119a2.32 2.32 0 00-.178-.651l-.514-.514-.434-.395zM84.88 41.553l.395.395h.356l.118-.119a.39.39 0 00.119-.118 1.789 1.789 0 001.027-1.58l-.099-.1-.099-.296-.71-.71a1.425 1.425 0 00-1.72.73l-.177.178-.04.237.02.375.692.691.118.317zM103.567 67.746l-.178.178a1.396 1.396 0 00-.85.85 1.519 1.519 0 00.237 1.225l.692.69.296.02c.067-.052.126-.111.178-.178a1.394 1.394 0 00.774-.942 1.392 1.392 0 00-.261-1.191l-.612-.613h-.316l.04-.04zM97.522 36.733l-.119.118a1.09 1.09 0 00-.04 1.343l.554.554.276.079.139.138.375-.02a1.452 1.452 0 00.75-1.027 1.48 1.48 0 00-.157-.83 1.313 1.313 0 00-1.778-.632v.277zM102.796 10.44l.237.238.218.02a1.401 1.401 0 001.224-1.462 1.563 1.563 0 00-.849-1.442h-.198l-.237.04-.197.197a1.363 1.363 0 00-.79 1.067 1.65 1.65 0 00.632 1.422l-.04-.08zM107.28 81.06l.672.671h.277l.316-.118a1.345 1.345 0 00.75-1.027 1.535 1.535 0 00-.533-1.402l-.198-.198-.454-.139h-.276l-.929.929a1.61 1.61 0 00.046.683c.065.222.178.427.329.6zM142.896 38.451l-.355.356-.079.276a1.06 1.06 0 00-.178.534.441.441 0 00.04.276l.711.711.336.1.197.197.375-.02a1.25 1.25 0 00.672-.754 1.255 1.255 0 00-.099-1.004 1.313 1.313 0 00-1.62-.672zM113.246 33.73a1.478 1.478 0 00-.77 1.758l.849.85.237.04.454.019 1.126-1.126v-.237l-.059-.336-.296.02.178-.178-.178-.454-.593-.593-.474-.079-.276.04-.198.276zM94.046 57.474l1.106-1.106-.02-.376-1.087-1.086-.375.02-.118.118a1.369 1.369 0 00-.81.81 1.426 1.426 0 00.177 1.126l.79.79.337-.296zM92.406 27.231a1.229 1.229 0 00-.534 1.442l.593.593a1.174 1.174 0 001.284.257c.14-.095.26-.216.356-.356l.434-.434.02-.218-.198-.197a1.313 1.313 0 00-1.54-.79l-.158.158-.257-.455zM63.724 101.07l-.296-.296h-.198l-.355.039-1.126 1.126v.356l.118.118a1.478 1.478 0 001.343.909l.297-.059.77-.771.08-.276-.02-.257a1.34 1.34 0 00-.198-.79l-.138-.138-.277.039zM132.525 91.727l-.276.079-.83.83-.02.256.04.395-.119.119a1.374 1.374 0 00.83.83 1.23 1.23 0 001.047-.06l.632-.632a1.474 1.474 0 00-.204-1.316 1.476 1.476 0 00-1.179-.62l.079.119zM115.972 95.203l-.158-.158-.257-.06-.099.1-.217-.217a1.509 1.509 0 00-1.718-.415l-.751.75-.04.316.099.1a1.79 1.79 0 00.77 1.204c.24.119.504.18.771.178l.217.217a1.315 1.315 0 001.403.178c.087-.07.167-.15.237-.237l.355-.356.04-.237-.04-.316-.434-.434-.178-.613zM60.247 81.237l.988-.987v-.356l-.099-.099a1.37 1.37 0 00-.81-.81 1.427 1.427 0 00-1.205.06.529.529 0 01-.138.138l-.513.514-.04.237-.118.118.04.356.829.83.257.059.118.118.375-.02.316-.158zM61.452 25.552l.672-.671.078-.316c.05-.235.05-.477 0-.712l-.81-.81-.335-.059-.218-.02a1.48 1.48 0 00-.454.139 1.425 1.425 0 00-.671 1.817 1.369 1.369 0 001.738.632zM83.616 34.48a1.283 1.283 0 001.165.297l.988-.988a1.286 1.286 0 00-.237-1.027L85 32.23l-.237-.04h-.198l-.395.08-.099.098a1.285 1.285 0 00-.651 1.6l.138.138.06.376zM105.365 22.767l.316.316c.216.032.437.025.652-.02l.691-.691.079-.316.118-.119-.039-.395-.138-.138a1.293 1.293 0 00-.751-.705 1.285 1.285 0 00-1.027.073 1.018 1.018 0 00-.198.197l-.474.474-.039.237-.099.1.039.315.593.593.277.079zM84.03 77.445l.553-.553a1.424 1.424 0 00-.652-1.205 1.286 1.286 0 00-1.303-.119 1.23 1.23 0 00-.652.89l-.218.216.04.316.099.1a1.342 1.342 0 00.77.77 1.258 1.258 0 001.067-.08.838.838 0 00.296-.335zM98.332 107.549l.277-.276-.04-.435a1.512 1.512 0 00-.138-.731l-.217-.217-.494-.296-.316-.079-.356.039-.632.632a1.537 1.537 0 00-.099 1.719l.692.691.375-.02.73-.73.218-.297zM64.81 47.676l.395.395c.212.063.437.07.652.02l.83-.83.316-.04 1.027-1.026.138-.415.02-.218a1.312 1.312 0 00-.652-1.046 1.285 1.285 0 00-1.225 0c-.331.153-.597.419-.75.75l-.316.04-.119.118a1.34 1.34 0 00-.77 1.442l.434.435.02.375zM69.69 48.92c.174.387.483.696.869.87.39.14.82.133 1.205-.02l.73-.73a1.591 1.591 0 00-.295-1.285l-.83-.83-.356.04-.118.118-.356.159-.79.79v.355l-.06.534zM106.589 52.595a1.43 1.43 0 00-.755 1.4c.018.188.073.37.163.536a1.14 1.14 0 00.711.71 1.175 1.175 0 001.027-.078l.277-.277.217-.652-.02-.415-.711-.71-.672-.238-.237-.276zM124.545 86.354l-.572-.099-.316.04-.85.849-.059.257-.02.217.079.395.138.138a1.259 1.259 0 002.371-.592l-.079-.514-.692-.691zM88.377 52.634l.592-.592a1.48 1.48 0 00-.434-1.58 1.453 1.453 0 00-1.56-.178 2.155 2.155 0 00-.297.296l-.553.553a1.51 1.51 0 00.632 1.185 1.509 1.509 0 001.62.316zM77.927 41.948l.355.04.296-.297a1.536 1.536 0 00.633-2.055l-.81-.81-.237-.039-.376.02-.296.099-.514.513-.158.395-.02.218c.01.346.084.688.218 1.007l.909.909zM105.443 90.522l-.454-.455-.297-.02-.316.12a1.507 1.507 0 00-.81 1.48l.099.1.198.434.533.533.435.119c.259.015.519-.04.75-.158a.823.823 0 00.198-.198l.553-.553a1.655 1.655 0 00-.316-1.264l-.573-.138zM87.329 55.696l-.75-.75a1.872 1.872 0 00-.771-.06l-.968.968v.435l.909.908.256.06.218.02.395-.08.118-.118a1.256 1.256 0 00.593-1.383zM109.552 86.927a1.232 1.232 0 00-1.027.04l-.277.276-.138.375-.138.138a.977.977 0 00.039.711l-.514.514.02.296.099.1c.161.382.466.687.849.849l.099.098.336.02.454-.454a1.312 1.312 0 00.751-1.344l-.375-.375a1.286 1.286 0 00-.04-1.106 1.441 1.441 0 00-.138-.138zM59.793 57.948l-.474-.474a1.173 1.173 0 00-1.244-.178l-.929.929-.02.217v.395l.119.119c.155.295.396.536.691.69a1.285 1.285 0 001.62.396l.612-.612a1.481 1.481 0 00-.375-1.482zM70.322 33.197l.395.395a.95.95 0 00.612.02l.988-.988a1.284 1.284 0 00-.237-1.067l-.652-.652-.099.1-.316.039-.098.098a1.48 1.48 0 00-.81 1.482l.454.454-.237.119zM46.716 109.189a.778.778 0 00-.178.178l-.671.671-.02.296a1.531 1.531 0 00.889.889 1.427 1.427 0 001.205-.059l.869-.869v-.316l-.099-.099a1.56 1.56 0 00-.869-.869 1.424 1.424 0 00-1.126.178zM128.91 91.47l-.177-.178a1.204 1.204 0 00-1.225-.237l-.929.928a1.279 1.279 0 00.218 1.008l.691.691h.198l.395-.079a1.316 1.316 0 00.691-1.56l-.138-.139.276-.434zM87.31 98.62l-.751.751-.119.316-.118.119.02.415.138.138a1.282 1.282 0 001.778.632 1.22 1.22 0 00.217-.217l.593-.593-.04-.316-.119-.118a1.48 1.48 0 00-1.6-1.126zM112.476 44.298c-.123.1-.236.213-.336.336a1.261 1.261 0 00.237 1.462 1.311 1.311 0 001.481.217 1.26 1.26 0 00.672-1.067.277.277 0 00.099-.098l-.04-.237-.691-.692-.277-.079h-.197l-.099.099-.336.099-.336.335-.177-.375zM117.968 62.156a2.116 2.116 0 00-.297.296l-.415.415-.039.237a1.487 1.487 0 00.652 1.165 1.45 1.45 0 001.343.119l.573-.573a1.424 1.424 0 00-.415-1.56 1.37 1.37 0 00-1.402-.1zM95.982 99.51l-.119.118-.02.296c.004.372.129.733.356 1.028l.415.414.296.02.297-.099a1.396 1.396 0 00.327-2.244 1.342 1.342 0 00-1.473-.245l-.494.494.415.218zM146.215 77.721l-.356-.355-.335-.02-.099.099a1.002 1.002 0 00-.178.178 1.369 1.369 0 00-.652 1.797 1.37 1.37 0 001.778.672l.79-.79v-.237a1.531 1.531 0 00-.178-.81l-.355-.356-.415-.178zM75.438 85.761l.335.336.237.04.435-.04.138-.138a1.537 1.537 0 00.81-1.324l-.138-.138-.218-.415-.533-.533-.434-.08-.257.02-.158.158a1.397 1.397 0 00-.731 1.719l.158.158.356.237zM56.692 92.339l-.237.632.02.415 1.046 1.047.415-.02a1.34 1.34 0 00.73-.73 1.396 1.396 0 00-1.955-1.64l-.02.296zM79.409 105.179l-.731.731v.197l.02.376.869.869c.224.094.471.122.71.079h.198l.257-.06.83-.829.04-.316-.02-.218-.12-.355-.137-.138a1.452 1.452 0 00-1.916-.336zM80.356 60.378l.04.316 1.126 1.126.335.02.119-.119a.529.529 0 00.138-.138 1.425 1.425 0 00.83-.83 1.34 1.34 0 00-.198-1.146l-.671-.671-.218-.02c-.246-.005-.49.05-.71.158l-.692.691-.04.237-.059.376zM88.277 110.058l-.158-.158-.099.099-.237.039-.434.198-.534.533-.079.277-.02.256.08.435.177.178c.157.258.374.475.633.632l.256-.257-.02.336.514.039.277-.039.434-.435.277-.474.079-.276a2.347 2.347 0 00-.83-1.502l-.316.119zM41.956 80.013l-.198-.198-.316-.079a1.312 1.312 0 00-.79.119 1.229 1.229 0 00-.553 1.58l.316.316a1.257 1.257 0 001.442.494l.75-.75a1.62 1.62 0 00.08-.79l-.218-.218-.514-.474zM73.7 89.83l-.514-.513h-.198l-.355.04-.139.138a1.453 1.453 0 00-.632 1.935l.75.751.218.02.376-.02.948-.948a1.704 1.704 0 00-.297-1.284l-.158-.119zM62.776 75.035l.316.316h.197l.376-.02 1.106-1.106-.02-.375-.948-.948a1.341 1.341 0 00-.909.118l-.177.73-.237.673.572.572-.276.04zM125.177 45.503l.85-.85.019-.374-.019-.218a1.393 1.393 0 00-1.798-.928l-.849.85-.02.375.118.118a1.453 1.453 0 00.83.83c.26.16.565.23.869.197zM157.257 33.177a1.51 1.51 0 00-1.205.06.822.822 0 01-.178.177l-.573.573-.039.237a1.61 1.61 0 00.79 1.304 1.281 1.281 0 001.224.04c.281-.143.51-.372.652-.653l.257-.257.02-.375-.119-.118a1.453 1.453 0 00-.829-.988zM135.923 59.33l-.613.613-.059.494.04.277a1.365 1.365 0 001.442.849l.375-.099.612-.612.139-.534-.02-.296a1.372 1.372 0 00-1.561-.928l.099.098-.118-.118-.336.257zM116.9 52.674h.395a1.363 1.363 0 00.711-1.067 1.497 1.497 0 00-.138-.73 1.369 1.369 0 00-1.817-.672l-.277.276-.079.277a.897.897 0 00-.158.513l.099.099c.018.186.087.363.198.514l.197.197.869.593zM126.876 71.46l-.177-.178a1.628 1.628 0 00-.573-.178l-.277.04-.454.217-.593.592-.02.494.04.277.909.908.217.02.375-.02.138-.138a1.42 1.42 0 00.81-1.442l-.513-.514.118-.078zM119.271 55.36l-.119-.118-.138.138a.98.98 0 00-.336.336l-.395.04-.119.118a1.368 1.368 0 00-.75.948 1.306 1.306 0 00.296 1.086l.691.692.317-.04.355-.355.138.138h.198l.158-.158a1.25 1.25 0 00.654-.725 1.272 1.272 0 00-.061-.974 1.405 1.405 0 00-.534-.533 1.062 1.062 0 00-.355-.593zM95.725 43.35a1.229 1.229 0 001.363.138c.138-.097.258-.217.355-.355l.257-.257.02-.217a1.34 1.34 0 00-.692-1.008 1.48 1.48 0 00-1.323-.02 1.286 1.286 0 00-.672 1.58l.415.416.277-.277z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M120.555 54.669l.138-.138a1.32 1.32 0 00-.335-1.245 1.427 1.427 0 00-1.245-.336l-.83.83-.059.257.02.375.118.119.178.375.593.593.513.039.277-.04.553-.553.079-.276zM149.613 53.266a1.289 1.289 0 00-.218.218l-.336.335a1.46 1.46 0 00-.395 1.304 1.315 1.315 0 001.126 1.126l.277-.04.415-.217 1.185-1.185.02-.217-.02-.376-.138-.138a1.284 1.284 0 00-1.916-.81zM138.985 66.422l-.573-.573a1.342 1.342 0 00-1.225 0 .87.87 0 00-.257.257l-.691.692 1.185 1.185h.435l.296-.297a1.4 1.4 0 00.751-.75 1.38 1.38 0 00.079-.514zM119.568 31.636l.454.454.415.02.237-.237a1.394 1.394 0 00.79-.79 1.452 1.452 0 00-.237-1.225l-.672-.671h-.355l-.119.118a1.4 1.4 0 00-.652 1.798l.139.138v.395zM130.787 55.696l-.02-.296-.099-.1a1.448 1.448 0 00-.869-.868 1.504 1.504 0 00-1.165.059l-.119.119-.474.474-.079.276a1.403 1.403 0 00.257 1.087l.533.533.257.02a1.48 1.48 0 00.731-.178.79.79 0 00.197-.198l.593-.592.257-.336zM140.546 49.533a1.366 1.366 0 00-.356-1.185l-.474-.474-.237-.04-.277.04-.118.118a1.367 1.367 0 00-.77 1.442l.849.85a1.51 1.51 0 001.067-.08c.165-.19.274-.422.316-.671zM102.856 49.375l.375-.02a1.367 1.367 0 00.81-.81 1.419 1.419 0 00-.079-.908.868.868 0 00-.039-.395l-.929-.929a.948.948 0 00-.592 0l-.948.948c-.033.217-.026.438.019.652l.119.119v.197c.158.374.456.672.83.83l.118.119.316.197zM108.17 60.93l.474.475c.332.083.684.04.988-.119a1.397 1.397 0 00.79-1.224 1.45 1.45 0 00-.79-1.304l-.119-.119-.277.277a1.401 1.401 0 00-.849.85 1.586 1.586 0 00.237 1.224l-.454-.06zM114.332 69.86a1.404 1.404 0 00-.335-1.245l-.475-.474a1.283 1.283 0 00-.908.079c-.194.117-.357.28-.474.474l-.395.395-.06.336.099.099a1.403 1.403 0 00.85.849 1.284 1.284 0 001.165-.02c.221-.11.406-.282.533-.493zM130.649 80.922l-.375-.376-.237-.04-.356.04-1.047 1.047v.198l.02.375.909.909.276.04h.237l.395-.08.198-.197a1.37 1.37 0 00.395-1.936l-.415.02zM124.605 37.503a1.365 1.365 0 001.817.672l.533-.534.079-.276a1.397 1.397 0 00-1.007-1.68l-.119-.118-.118.119a1.672 1.672 0 00-.336.099 1.424 1.424 0 00-.849 1.718zM109.493 70.195a1.592 1.592 0 00-.375-1.6 1.371 1.371 0 00-1.502-.237l-.572.573a1.275 1.275 0 00.079 1.106 1.502 1.502 0 00.987.79l.099-.099.356-.039.79-.79.138.296zM105.365 41.612l.553.553.316-.04.237-.237a1.395 1.395 0 00.889-.889 1.37 1.37 0 00-.218-1.126l-.553-.553-.217-.02c-.246-.005-.49.05-.711.159l-.652.651-.02.218c-.019.366.101.726.336 1.007l.04.277zM111.33 65.75a1.45 1.45 0 00-.02-.651l-.909-.909a1.059 1.059 0 00-.651-.02l-.889.89a1.173 1.173 0 00-.02.69l.909.91c.214.042.435.049.651.02l.929-.93zM113.265 64.21l.139.138.375.02.217-.02.158-.158a1.397 1.397 0 00.85-.85 1.519 1.519 0 00-.257-1.244l-.494-.494-.257-.02a1.564 1.564 0 00-.612.178 1.367 1.367 0 00-.77 1.363l.138.139.178.375.355.356-.02.217zM108.544 64.348l.929-.928a1.567 1.567 0 00-.376-1.6 1.364 1.364 0 00-1.501-.237l-.513.513-.139.336-.355.356a1.253 1.253 0 000 .632l.928.928c.206.064.427.064.632 0h.395zM103.942 56.388l.771-.77c.052-.222.052-.452 0-.672l-.929-.929a1.284 1.284 0 00-.612.02l-.929.928c-.059.22-.059.452.001.672l.829.83c.21.031.423.031.632 0l.237-.08zM119.33 71.104l-.375-.376-.237-.039-.119-.119-.336.06-.79.79a1.533 1.533 0 00-.819.912 1.526 1.526 0 00.108 1.221l.711.711.316.04.099.099a2.78 2.78 0 00.435-.158c.375-.156.674-.455.829-.83a1.366 1.366 0 00.415-1.956l-.237-.355zM122.194 67.252l.119.119.118-.119a1.312 1.312 0 00.672-.948 1.502 1.502 0 00-.533-1.442 1.427 1.427 0 00-1.423-.119 1.252 1.252 0 00-.711.711 1.51 1.51 0 00.178 1.126l.119.119a1.228 1.228 0 001.461.553zM118.402 42.916l.296-.139.178-.138.237-.277.099-.177.079-.237.02-.218-.02-.375-.059-.217-.079-.119-.119-.158-.296-.257-.178-.098-.237-.08-.198-.039-.375.02-.197.079-.297.138-.158.119-.237.276-.118.198-.079.237-.02.217.039.356.04.197.079.158.118.158.297.257.178.099.237.079.217.02.355-.04.198-.04zM144.259 69.326l-.316-.316a1.312 1.312 0 00-.612.02l-.988.987a1.392 1.392 0 00.257 1.047l.83.83.336-.06.987-.987.02-.217-.04-.356-.612-.612.138-.336zM134.699 80.704l-.712.711v.553l.119.119a1.403 1.403 0 00.74.862 1.398 1.398 0 001.137.027l.75-.75.059-.337-.118-.118a1.423 1.423 0 00-.751-1.106 1.314 1.314 0 00-1.224.04zM158.6 86.117a1.376 1.376 0 00-.711.908 1.426 1.426 0 00.277 1.185l.494.494.237.04.098-.099c.227.023.455-.025.652-.138a1.31 1.31 0 00.711-.711l.06-.257a1.263 1.263 0 00-.83-.83 1.336 1.336 0 00-.988-.592zM129.681 43.054l-.691.691-.04.316c.009.14.05.274.119.395l.849.85.415-.02 1.047-1.047a1.477 1.477 0 00-.692-1.205 1.507 1.507 0 00-1.007.02zM149.968 66.857c-.083.06-.156.134-.217.217l-.691.692.059.335.889.89c.208.042.423.042.632 0l.731-.731.099-.297v-.197l-.02-.376-.139-.138a1.306 1.306 0 00-1.343-.395zM114.807 59.825l.256-.257.06-.336.493-.494-.039-.355-.869-.87-.277-.078-.099-.1-.355.04-.119.119a1.321 1.321 0 00-.79.79.447.447 0 00-.039.276l-.376.376-.059.494.04.276-.02.296.039.277.889.889.494.06.158-.159a1.456 1.456 0 00.81-1.047l-.197-.197zM128.792 66.54l-.375.02-.119.119a1.537 1.537 0 00-.79.79c-.095.388-.039.798.158 1.146l.771.77.355-.04.119-.118a1.394 1.394 0 00.79-.79 1.563 1.563 0 00-.158-1.146l-.751-.75zM121.582 15.597l.02-.218-.04-.276a1.48 1.48 0 00-.77-.77 1.452 1.452 0 00-1.106.078l-.198.198-.474.474-.039.237-.099.099.02.296.098.099.178.415.593.592.494.02.276-.04.316-.316a1.414 1.414 0 00.731-.888zM74.608 105.199l.356-.356a1.286 1.286 0 00-.514-1.185 1.226 1.226 0 00-.691-.257l-.119-.119-.414.02-.297.297a1.672 1.672 0 00-.948.948l-.02.257v.197l.139.138c.142.281.37.51.652.652l.572.573.158-.158c.072-.06.138-.126.198-.198l.197-.197.731-.612zM87.25 125.742h-.197l-.297.059-.098.099a1.427 1.427 0 00-.77 1.482l.73.731a1.345 1.345 0 001.059-.594 1.34 1.34 0 00.146-1.204l-.573-.573zM92.01 75.706l.277-.04.455-.216.592-.593.02-.494-.06-.296a4.169 4.169 0 01-.157-1.106l-.04-.237-.85-.85-.473-.04-.277.04a1.313 1.313 0 00-.73.968l-.159.158-.06.454-.098.1-.197.513v.316l.118.316-.118.118a1.341 1.341 0 001.758.89zM89.206 71.321l.119-.118.04-.356.098-.099a1.593 1.593 0 00-1.225-1.224h-.316l-.553.197-1.145 1.146v.434l.059.257.83.83.296.06h.197l.297-.02.533-.178.652-.652.118-.277zM94.006 81.89l-.434-.435-.237-.04-.455.06-.138.138a1.285 1.285 0 00-.672.908 1.397 1.397 0 00.277 1.067l.514.514.237.04h.197l.375-.1a1.34 1.34 0 00.692-1.56l-.158-.158-.198-.435zM104.475 99.51l-.296.296-.119.355-.138.139a1.033 1.033 0 00.217 1.007c.118.142.266.257.434.335.167.079.349.12.534.12l.119.118.395-.04c.311-.132.558-.38.691-.691a1.394 1.394 0 00-1.936-1.62l.099-.02zM133.889 32.21l-.277-.277a1.254 1.254 0 00-.909-.356c-.373.01-.728.166-.987.434a1.343 1.343 0 00-.02 1.877l.375.138h.711l.356-.158.494-.493.178-.376-.02-.257.099-.533zM94.52 67.193l-.376-.139-.691.02-.356.158-.099.099a2.46 2.46 0 00-.335.336l-.139.375v.711l.158.356a1.34 1.34 0 00.948.395c.35-.02.681-.167.929-.415a1.453 1.453 0 00-.04-1.896zM133.256 20.357l-.375.375-.02.218a1.307 1.307 0 00.356 1.225l.395.394.296.06.237-.04 1.126-1.126.02-.257-.099-.098-.099-.336-.276-.277a1.4 1.4 0 00-1.877-.138h.316zM123.775 45.444l-.198.198-.059.572.079.317.316.316a1.286 1.286 0 001.008.375 1.507 1.507 0 001.106-.672l.099-.098.079-.277.118-.119-.099-.098-.039-.277-.395-.395a1.371 1.371 0 00-1.877.06l-.138.098zM142.561 26.382l-.02.257.079.434.494.494a1.39 1.39 0 001.145.356 1.481 1.481 0 00.949-.949l.098-.098.06-.336-.613-.613a1.45 1.45 0 00-1.56-.335l-.652.652.02.138zM115.597 55.084l-.178-.178a1.336 1.336 0 00-.395-1.027 1.873 1.873 0 00-1.264-.435l-.119.119-.336.138-.77.77a1.397 1.397 0 00.395 1.028c-.055.33.016.667.198.948l.592.592.277.04.217-.02.375-.099.158-.158a1.611 1.611 0 00.573-1.363l.277-.355zM100.802 58.66l-.593-.593a1.395 1.395 0 00-1.521-.257 1.48 1.48 0 00-.85 1.481l.573.573a1.397 1.397 0 001.541.277 1.51 1.51 0 00.85-1.482zM165.475 25.157l.395.395a1.143 1.143 0 001.126-.02l.612-.612c.109-.276.149-.574.118-.87l-.809-.809-.316-.119-.119-.118a1.652 1.652 0 00-1.383.87 1.316 1.316 0 00.376 1.283zM107.162 35.034l.079-.356-.04-.237-.454-.454a1.481 1.481 0 00-1.659-.237l-.613.612-.059.375v.237c.007.213.075.42.198.593l.573.573.375.06.217-.02a1.366 1.366 0 00.909-.554l.474-.592zM117.513 24.564a1.254 1.254 0 00-1.047-.375 1.391 1.391 0 00-1.047.652l-.118.118-.06.336-.098.1.849.849.356.118.118.119.257.02.138-.139.336-.138.454-.455.158-.395.02-.217-.02-.257-.355-.355.059.02zM121.957 50.975l-.533-.533-.296-.02a1.867 1.867 0 00-.79.198l-.553.553-.06.256v.198l.06.336c.068.255.204.488.395.671l.316.316.296.06.099.098.217-.02.099-.098.276-.08.119-.118a1.457 1.457 0 00.612-1.007 1.332 1.332 0 00-.257-.81zM171.282 3.35l-.039.434v.237c.065.29.208.556.414.77.098.126.211.238.336.336l.218.217a1.34 1.34 0 001.283-.928l.119-.119.04-.237.098-.099-.059-.375-.375-.375a1.484 1.484 0 00-1.66-.237l-.572.573.197-.198zM110.58 28.93l.671-.671.079-.277.02-.257-.079-.434-.336-.336a1.369 1.369 0 00-1.837-.138l-.553.553a1.308 1.308 0 00.395.987 1.45 1.45 0 001.64.573zM132.427 6.648c.275-.092.52-.255.711-.474l.138-.138.119-.356v-.198l-.02-.256-.534-.534a1.53 1.53 0 00-1.501-.316l-.671.672-.119.395v.198c-.024.265.046.53.198.75l.651.652.297.02.256-.06.475-.355zM109.651 31.32l-.415-.415-.217-.02a1.366 1.366 0 00-1.185.633 1.597 1.597 0 00-.059 1.481l.711.711a1.533 1.533 0 001.402-.928 1.364 1.364 0 00-.237-1.462zM127.39-1.826l-.218.217-.039.316.079.553.217.217a1.373 1.373 0 001.047.415 1.456 1.456 0 001.027-.553l.514-.514-.099-.098-.079-.277-.257-.257a1.312 1.312 0 00-1.718-.098l-.356.355-.118-.276zM127.43 17.611l-.257.02-1.186 1.185a1.35 1.35 0 00.415 1.166 1.427 1.427 0 001.976 0l.158-.158.177-.415.04-.237-.02-.257-.118-.119a1.652 1.652 0 00-.415-.651l-.138-.139-.376-.138-.256-.257zM77.255 14.925L77 14.668h-.395l-.218-.02a1.09 1.09 0 00-.869.553 1.285 1.285 0 00-.118 1.028l.02.217.138.375a1.285 1.285 0 001.797-.02c.22-.225.359-.517.395-.83a1.312 1.312 0 00-.494-1.046zM105.444 27.35l.335.138a1.343 1.343 0 001.64-.415l.138-.138.04-.237.098-.099a1.533 1.533 0 00-.296-1.047l-.573-.573a1.594 1.594 0 00-1.422.949 1.29 1.29 0 00.296 1.481l-.256-.06zM112.871 20.99l-.237-.238-.139-.375a1.428 1.428 0 00-2.015.04 1.539 1.539 0 00-.434 1.066c.007.364.155.712.415.968l.395.158.217-.02a1.367 1.367 0 001.106-.197 1.364 1.364 0 00.553-1.107l.139-.296zM107.399 22.787l.06-.297-.514-.513a1.927 1.927 0 00-.632-.356l-.138-.138h-.277a1.45 1.45 0 00-.988.988 1.263 1.263 0 00.356 1.225 1.197 1.197 0 00.909.355c.116.03.239.03.355 0a1.399 1.399 0 001.008-.691l-.139-.573zM100.979 24.268l.474-.276.435-.435.039-.316-.059-.533-.356-.356a1.393 1.393 0 00-1.955-.099c-.29.13-.523.362-.652.652a1.256 1.256 0 00.06 1.126l.69.692.297.02c.251-.031.492-.112.711-.238l.316-.237zM90.253 49.414a1.314 1.314 0 00-2.192.534v.276l1.046 1.047h.277l.099-.099a1.313 1.313 0 00.908-.908 1.368 1.368 0 00-.138-.85zM115.774-13.56l.415.415.336.06c.239-.03.468-.11.672-.238l.553-.553.059-.336v-.197a1.855 1.855 0 00-.217-.613l-.573-.572-.316-.04-.099-.099a1.37 1.37 0 00-.632.237l-.553.553-.059.336v.198c.02.21.087.412.197.592l.217.257zM122.313.96l-.019.295c.006.213.075.419.197.593l.198.197.434.238.257.059.336-.02.533-.533a1.288 1.288 0 000-1.857l-.336-.336-.316-.079-.099-.099-.118.474L122.313.96zM105.779 4.218l-.434-.434-.376-.138-.671.04-.257.256a1.31 1.31 0 00-.211 1.802 1.316 1.316 0 001.791.292 1.28 1.28 0 00.457-.876 1.286 1.286 0 00-.299-.942zM111.666 11.29l-.118.119v.237l.098.098.119.356.316.316a1.313 1.313 0 001.185.395 1.34 1.34 0 00.988-.75l.257-.257.118-.356.119-.118v-.238l-.139-.138-.138-.336-.454-.454-.356-.158-.118-.118h-.237l-.119.118-.355.158-.771.77-.395.356zM138.867 29.128l-.119.118a1.399 1.399 0 00.316 1.462 1.453 1.453 0 001.6.257 1.423 1.423 0 00.869-1.383l-.138-.138-.158-.356-.494-.494a1.422 1.422 0 00-1.54.395l-.099.099-.237.04zM122.175-29.995l.178-.177.197-.435.059-.257-.098-.099a1.289 1.289 0 00-.415-1.205l-.534-.533-.335.336c-.35.113-.65.343-.85.652l-.178.178.099.098.059.257.139.138c.113.173.261.321.434.435a1.398 1.398 0 001.245.612zM178.907 56.052a1.343 1.343 0 001.442.257l.494-.494c.207-.242.32-.551.316-.87a1.368 1.368 0 00-.434-.908l-.356-.356a1.287 1.287 0 00-1.521.573 1.484 1.484 0 00.059 1.798zM74.667 65.395a1.34 1.34 0 00-.118-1.857l-.613-.612H73.7a1.341 1.341 0 00-1.086.73l-.099.297-.02.257.099.494.118-.119-.217.217c.068.246.197.47.376.652l.237.237.217.02h.395l.454-.454.494.138zM135.844-23.792l-.119.118.099.099.079.276.296.297a1.397 1.397 0 001.916.02 1.31 1.31 0 00.178-1.759l-.592-.592a1.342 1.342 0 00-1.62.79l-.237.75z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M122.768 12.594a1.258 1.258 0 00-1.265-.316l-.908.909-.119.395-.118.118.019.257.553.553a1.344 1.344 0 001.66.237l.652-.652a1.43 1.43 0 00-.474-1.501zM137.918 28.219l-.138-.415-.277-.276a1.624 1.624 0 00-1.304-.514l-.296.02a1.536 1.536 0 00-1.106.197 1.93 1.93 0 00-.948 1.818l.849.849.375.099.119.118a1.17 1.17 0 00.138-.138 1.425 1.425 0 001.106 0l.218-.02.276-.276a1.426 1.426 0 00.573-1.008l.04-.276.375-.178zM126.679 78.69l.059-.337.099-.098-.751-.75a1.227 1.227 0 00-1.896.276l-.119.118a1.262 1.262 0 00.099 1.363l.514.514.256.06h.198l.316-.08c.342-.108.635-.331.83-.632l.395-.435zM124.762 58.165l.119.119.414-.415a1.417 1.417 0 00.573-1.126 1.37 1.37 0 00-.415-1.047l-.256-.257-.297-.098-.098-.1h-.237a1.39 1.39 0 00-.949.593l-.098.1-.079.276v.197l.138.139c.089.234.223.449.395.632l.316.316.316.079.158.592zM119.014 37.128l-.178-.178-.474-.119-.257-.02-.276.08-1.027 1.027-.06.336c-.05.173-.057.356-.019.533l.829.83.336.059.296-.06c.365-.032.704-.202.949-.473l.158-.159.118-.395.119-.118-.02-.257-.119-.118a1.45 1.45 0 00-.375-.968zM111.804 44.456l.316.316a1.402 1.402 0 001.067.395 1.562 1.562 0 001.067-.632l.118-.118.119-.356v-.237l-.099-.099.316-.316.039-.276-.138-.139-.138-.414-.494-.494-.375-.138-.218-.02-.296.02-.533.533a1.312 1.312 0 00-.376 1.086 1.484 1.484 0 00-.316.87l.119.118-.178-.099zM119.765 10.658l.454-.454a1.48 1.48 0 00-.039-2.055l-.139-.138-.355-.158-.139-.138-.256.02-.218.02-.414.216-.633.633-.059.414.02.218c.041.3.18.577.395.79.106.155.24.289.395.395l.119.118.296.02.099-.099.474.198zM129.899 29.207l-.257-.257a1.37 1.37 0 00-1.64-.217l-.671.671-.099.375.099.1c.058.354.232.68.493.928l.316.316.277.079h.237l.296-.1.099-.098a1.726 1.726 0 00.672-1.146l-.099-.098.277-.553zM123.814 37.977l-.217-.217-.494-.257-.237-.04a1.51 1.51 0 00-.514-.118c-.155-.1-.33-.168-.513-.197l-.119-.119h-.237l-.474.474a1.88 1.88 0 00-.672 1.304l.316.316a1.404 1.404 0 00.277 1.067l.533.533h.237c.33.077.675.05.988-.08l.435-.434.237-.237a1.51 1.51 0 00.454-1.995zM126.817 33.394l-.099-.099-.296-.098-.099-.099-.218.217-.296.099-.257.257a1.41 1.41 0 00-.573.934 1.386 1.386 0 00.277 1.06l.573.574a1.37 1.37 0 001.323-.85 1.425 1.425 0 00-.316-1.5l-.019-.495zM128.219 24.486l-.118.118v.237l.099.099.098.336.297.296a1.369 1.369 0 001.837.138l.513-.513.04-.237v-.198l-.079-.316a1.624 1.624 0 00-.395-.632l-.316-.316-.336-.099h-.198l-.888.889-.554.198zM106.826 73.139a1.482 1.482 0 00-1.067-.395 1.623 1.623 0 00-1.165.73l-.296.297-.04.237c.027.389.196.754.474 1.027a1.52 1.52 0 001.126.415l.257-.06.296-.296a1.502 1.502 0 00.593-1.146 1.43 1.43 0 00-.178-.81zM150.225 15.24l-.119-.118-.118-.316-.297-.296a1.366 1.366 0 00-1.145-.435 1.337 1.337 0 00-1.008.573l-.158.158-.039.237a1.28 1.28 0 00.256 1.482c.1.136.22.256.356.355l.118.119.336.02.376-.376c.19-.107.347-.264.454-.454l.138-.138.138-.336.119-.119.593-.355zM150.679 8.663l-.119.118-.019.257.493.494a1.788 1.788 0 001.857.435l.316-.316.119-.119.099-.296a1.266 1.266 0 00-.435-.988 1.591 1.591 0 00-.81-.415l-.118-.118h-.277l-.217.02-.356.158-.454.454-.079.316zM132.071 34.935l-.138-.138a1.393 1.393 0 00-1.126-.415 1.481 1.481 0 00-1.067.632l-.118.119-.099.296-.099.099v.197l.099.099.099.296.296.297c.31.289.722.445 1.146.434a1.535 1.535 0 001.066-.671l.119-.119.099-.296.099-.099-.02-.217-.099-.297-.257-.217zM159.47 19.113a1.173 1.173 0 00-1.304.395l-.336.335v.277l.02.217.138.376a1.424 1.424 0 001.956-.06 1.304 1.304 0 00.404-.948 1.305 1.305 0 00-.404-.948l-.474.356zM185.939 15.36l-.02.217.237.237a1.482 1.482 0 001.818.671 1.536 1.536 0 00.928-1.481l-.039-.237-.494-.494a1.452 1.452 0 00-1.284-.375 1.536 1.536 0 00-1.087 1.086l-.059.375zM120.713 42.817l-.099.296v.237l.079.277.355.355a1.369 1.369 0 001.205.376 1.566 1.566 0 001.067-1.067l.059-.257-.039-.474-.593-.593-.434-.197-.277-.08a1.647 1.647 0 00-1.323 1.127zM172.685 62.432l-.376-.138h-.276a1.033 1.033 0 00-.553.04l-.731.73-.138.376-.02.335v.316l.158.356a1.204 1.204 0 00.909.355 1.144 1.144 0 00.908-.395 1.336 1.336 0 00.413-1.508 1.338 1.338 0 00-.294-.467zM167.391 32.841a1.481 1.481 0 00-1.896-.158l-.652.652-.04.277.02.256.158.435.533.533.395.158.218.02.276-.04.376-.375-.178-.177.197.197.534-.533.099-.297.019-.256-.099-.494-.177-.178.217-.02zM138.748 22.609l.435.434.355.159.119.118.256-.02.119-.118.336-.138.454-.455.138-.336.119-.118-.02-.257-.118-.118-.119-.356-.217-.217a1.62 1.62 0 00-1.067-.395 1.453 1.453 0 00-1.007.533l-.139.138-.098.336-.119.119.02.256.118.119.435.316zM166.66 11.053a1.26 1.26 0 00-1.521-.02l-.198.198-.197.474a1.28 1.28 0 00.316 1.383c.089.144.211.266.355.355l.099.099.316.04.652-.652a1.31 1.31 0 00.257-1.719l-.158-.158h.079zM87.448 8.169v-.435l-.87-.869-.316-.118-.118-.119-.257-.02-.474.475a1.48 1.48 0 00-.434 1.58l.75.75.336.06.277-.04.177-.178c.269-.084.508-.241.692-.454l.138-.138.099-.494zM118.954 70.175l-.809-.81-.336-.02-.257.06-.356.158h-.197a1.338 1.338 0 00-.573 1.007 1.24 1.24 0 00.395.988l.395.395a1.366 1.366 0 001.166-.573 1.399 1.399 0 00.572-1.204zM110.263 55.676l.02-.217-.02-.257-.118-.118c.088-.07.167-.15.237-.237a1.51 1.51 0 00-.198-1.738l-.592-.593-.218-.02-.316.079-.118.118a1.785 1.785 0 00-.85 1.126c-.074.264-.046.546.079.79l-.118.12a1.308 1.308 0 00.296 1.402l.356.355.296.06h.197l.218-.02.849-.85zM105.068-1.55l.356.356.335.138.119.119h.237l.118-.119.336-.099.119-.118a1.402 1.402 0 00.533-1.047 1.462 1.462 0 00-.454-1.087l-.277-.276-.336-.099-.118-.119-.928.929-.119.355-.119.119v.237l.139.138.138.336-.079.237zM158.067-18.715l.178.177.454.178.237.04.296-.02c.244-.019.481-.093.692-.218l.533-.533.059-.257.099-.098-.04-.356-.118-.119a1.419 1.419 0 00-.336-.335 1.45 1.45 0 00-1.975-.04 1.392 1.392 0 00-.079 1.58zM156.941 5.127a1.316 1.316 0 00.119 1.185l.592.593.376.02h.237a1.285 1.285 0 00.889-.573l.138-.138.079-.356V5.66l-.04-.237-.513-.513a1.258 1.258 0 00-1.719-.099l-.158.158v.158zM175.134 21.858l-.099.099v.276c.018.223.086.44.197.633l.514.513.257.06.296-.02.119-.119.355-.158.158-.158a1.287 1.287 0 00.417-.934 1.281 1.281 0 00-.397-.942l-.336-.336-.316-.08-.237.04-.691.692-.237.434zM116.486 19.666l.138.138.218-.02a1.396 1.396 0 00.967-.968 1.334 1.334 0 00-.355-1.264 1.232 1.232 0 00-1.047-.336l-.435.435a1.314 1.314 0 00-.454.968c.01.373.166.728.435.987l.533.06zM92.762 43.074h.237l.118-.119.415-.138c.229-.08.438-.208.612-.375l.237-.237.1-.573-.04-.316-.455-.455-.493-.296-.06.375.04-.395a1.592 1.592 0 00-1.66.474l-.138.139-.118.355-.119.119.02.256 1.126 1.126.178.06zM138.313-8.286l-.098.376v.237c.006.212.075.418.197.592l.593.593.355.04.119.316.671.671.237.04h.435l.237-.04a1.447 1.447 0 00.77-.968 1.343 1.343 0 00-.414-1.165 1.376 1.376 0 00-.988-.395l-.336-.336a1.398 1.398 0 00-1.62-.237l-.474.474.316-.198zM138.847-3.209a1.544 1.544 0 00-.573 1.086c-.021.42.12.83.395 1.146l.474.474h.237a1.68 1.68 0 001.126-.73l.119-.12.099-.335-.04-.237-.889-.889-.316-.119-.118-.118h-.277l-.099.099-.336.138.198-.395zM147.381 32.782a1.454 1.454 0 00-2.035.099c-.124.12-.222.264-.287.424a1.24 1.24 0 00-.088.504 1.331 1.331 0 00.395.948l.375.139h.711l.356-.159.77-.77-.02-.691-.177-.494zM121.088 61.444l.238-.237.059-.336-.119-.592-.533-.533-.395-.119-.237-.04-.257.02-.573.573c-.711.948-.178 1.6 0 1.778a1.313 1.313 0 001.324.257l.454-.257.039-.514zM141.355 15.616l.435.435.217.02a1.567 1.567 0 001.363-.929 1.456 1.456 0 00-.355-1.54l-.435-.435-.237.04a1.509 1.509 0 00-.948.947 1.477 1.477 0 00-.04 1.462zM148.328 1.61l-.118.12-.02.256.099.099.079.356.316.316c.573.572 1.363.572 2.153-.06l.592-.592.02-.297-.02-.217-.158-.355-.711-.712h-.395l-.237.04c-.31.13-.597.31-.849.533l-.395.395-.356.119zM111.192 44.2l.098-.297.02-.217-.099-.099-.079-.276-.098-.1a.81.81 0 00-.257-.256 1.398 1.398 0 00-1.64-.257l-.79.79-.02.257.02.455c.072.242.209.46.395.632l.395.395a1.731 1.731 0 001.126-.692l.929-.335zM138.136 15.596l.533-.533.02-.296a1.251 1.251 0 00-.178-.731l-.632-.632-.276-.04-.238.04-.414.138-.494.494-.138.375-.119.119.02.256.553.553a1.284 1.284 0 001.363.257zM115.913 46.866a1.15 1.15 0 00-.395-.948l-.356-.158h-.395l-.118-.118c-.247.02-.482.117-.672.276l-.296-.06-.217.02-.415.415a1.534 1.534 0 00-.514 1.067l.099.099.079.316.533.533a1.373 1.373 0 001.521.257l.395-.395a1.288 1.288 0 001.047-.375 1.451 1.451 0 00-.296-.929zM126.955-9.649c.038-.088.065-.181.079-.276a.923.923 0 00.257-.257 1.17 1.17 0 00-.237-1.225l-.711-.711-.257-.02-.118.119-.376.138-.138.138c-.252.217-.413.52-.454.85a1.342 1.342 0 00.158 1.66l.414.414a1.536 1.536 0 001.383-.83zM153.761-7.772l-.079.355-.02.218c.027.208.094.41.197.592l.593.593.375.06.119-.12a1.287 1.287 0 00.948-.592l.119-.118.079-.356.098-.099-.019-.217-.455-.454a1.367 1.367 0 00-1.639-.257l-.455.454.139-.06zM74.252-5.145l-.296-.296-.296-.099-.1-.099h-.197l-.098.1-.277.078-.118.119a1.565 1.565 0 00-.633 1.106 1.482 1.482 0 00.455 1.126l.296.296.296.099.1.099.098-.1.316-.118.099-.098a1.509 1.509 0 00.612-1.087 1.481 1.481 0 00-.257-1.126zM117.987 66.64l.118-.119a1.227 1.227 0 00-.158-1.225l-.572-.572-.336-.02h-.237c-.356.077-.672.28-.889.573l-.139.138-.059.375-.099.1.02.256.494.494a1.257 1.257 0 001.679.138l.138-.138h.04zM97.542 29.68l-.553-.552-.277-.08h-.316l-.237.04-.356.158-.157.158a1.314 1.314 0 00-.02 1.877l.237.237.316.079h.197l.218-.02.434-.435a1.453 1.453 0 00.514-1.461zM157.139 35.627l-.475-.119a1.34 1.34 0 00-1.303.711 1.318 1.318 0 00.276 1.462 1.203 1.203 0 001.225.316l.711-.711.119-.514v-.276l-.731-.731.178-.139zM142.086 46.688l-.434-.434-.218.02a1.425 1.425 0 00-.948.948 1.488 1.488 0 00.316 1.303l.553.554a1.426 1.426 0 001.344-.91 1.396 1.396 0 00-.613-1.48zM99.596 38.214l-.099-.099-.276.08a1.314 1.314 0 00-.85.651l-.355.356.928.928a1.397 1.397 0 001.976-1.975l-.474-.474-.85.533zM136.674 78.492l.019-.217-.098-.1-.079-.315-.316-.316a1.398 1.398 0 00-1.976 1.975l.158.158.376.178.237.04.276-.04.138-.138c.276-.067.521-.228.692-.455l.118-.118.455-.652zM105.819 14.293l-.138.138v.277l.138.138.178.375.158.158a1.508 1.508 0 001.521.336l.652-.652.079-.316v-.237l-.079-.435-.336-.335a1.37 1.37 0 00-1.857-.119l-.158.158-.158.514zM93.453-1.273h-.71l-.356.158-.672.671v.277l.276.277a1.257 1.257 0 00.968.375c.35-.02.68-.167.929-.415a1.37 1.37 0 00-.02-1.956l-.415.613zM88.712 24.486l-.139-.139-.375-.138-.118-.119-.257.02-1.146 1.146c-.009.243.053.483.178.691l.632.633.454.138h.277l.296-.06.553-.553a1.481 1.481 0 00-.355-1.62zM130.294 11.705v.237l.098.336.297.296a1.395 1.395 0 002.113-.138l.395-.395.02-.217-.138-.139a1.528 1.528 0 00-.376-.652l-.474-.474-.316-.079-.098-.098-.238.039-1.283 1.284zM87.093 34.441l-.02-.256-.909-.91-.217-.019-.297.02-.158.158-.414.217-.178.178a2.15 2.15 0 00-.277.87l.356.078-.336.06.138.533.632.632.514.079.296-.02a2.095 2.095 0 001.205-1.205l-.335-.415zM100.268-19.269l.059-.296-.039-.316a1.396 1.396 0 00-.376-.691 1.23 1.23 0 00-1.678-.02l-.297.296a1.229 1.229 0 00-.02 1.521l.553.553.277.04c.262-.005.52-.073.75-.198l.218-.217.553-.672zM101.394 13.976l.099-.335-.02-.257-.513-.514a1.48 1.48 0 00-2.055.04l-.158.158-.138.375-.138.138.02.257.118.119.158.355.612.613.336.059.119.118a1.845 1.845 0 001.106-.71l.454-.415zM111.784-1.194l-.118.118-.001.277.119.118.138.336.455.455.355.158.119.118.257-.02 1.106-1.106.02-.296a1.623 1.623 0 00-.435-.83l-.731.06-.731.02-.316.315-.237.277zM160.24 47.893l.553.554.237.039.218-.02.355-.118.138-.139a1.425 1.425 0 00.316-2.014l-.533-.534-.237-.04-.217.02-.376.1-.138.138a1.563 1.563 0 00-.612 1.007c-.031.361.075.72.296 1.007zM182.344 74.343a1.364 1.364 0 00-.474-1.106l-.434-.434-.771.177a1.68 1.68 0 00-.968 1.166 1.288 1.288 0 00.376 1.166 1.169 1.169 0 00.829.355l.099.099.217-.02.376-.099.118-.118a1.486 1.486 0 00.632-1.186zM150.461 62.53l-.296-.098-.296-.02-.494.1-.178.177a1.4 1.4 0 00-.316 1.66l.692.69.296.06.513-.04.297-.098a1.427 1.427 0 00.355-1.857l-.573-.573zM150.581 42.5l.158.396a1.365 1.365 0 001.224.316c.244-.07.463-.206.632-.395a1.329 1.329 0 00.433-.946 1.353 1.353 0 00-.373-.97l-.376-.139h-.276a1.033 1.033 0 00-.553.04l-.139.138c-.16.1-.295.235-.395.395l-.177.178-.158.988zM136.16 58.225l.02-.257c.04-.196.04-.397 0-.593l-.731-.73-.297-.1h-.276l-.494.139-.553.553-.119.355-.118.119-.02.257.533.533a1.4 1.4 0 001.62.277l.514-.514-.079-.04zM148.921 45.74l-.494-.494a1.311 1.311 0 00-.336-.335l-.138-.336-.158-.158a1.287 1.287 0 00-1.126-.376 1.23 1.23 0 00-.948.633l-.099.098-.059.336.019.218.712.71.019.218.474.474a1.314 1.314 0 001.818 0 1.43 1.43 0 00.336-.691 2.666 2.666 0 00-.02-.297zM152.675 19.508a1.273 1.273 0 00-.053.725c.052.242.173.463.349.638.099.136.219.256.355.355l.139.138.138-.138a1.337 1.337 0 00.691-1.007 1.482 1.482 0 00-.414-1.245 1.259 1.259 0 00-1.699-.079l-.178.178.672.435zM150.007 46.669l.218.02a1.286 1.286 0 001.047-.771c.094-.2.136-.421.123-.642a1.323 1.323 0 00-.202-.622l-.633-.632-.276-.04-.237.04-.375.138-.613.612-.079.316v.277l.119.474.553.553.355.277zM160.891 73.632l-.335-.335-.297-.06a1.48 1.48 0 00-1.382.04 1.28 1.28 0 00-.553 1.501l1.125 1.126h.554l.138-.138.355-.158.139-.139a1.314 1.314 0 00.256-1.837zM144.97 67.943l.138-.138.099-.296a1.316 1.316 0 00-.375-1.166l-.376-.375-.335-.06-.237.04-.968.968-.04.356.02.217.158.395.336.336a1.343 1.343 0 00.987.356c.268-.132.479-.357.593-.633zM171.44 38.155l-.138.138-.02.257v.237l.099.376.277.276a1.343 1.343 0 001.066.395 1.478 1.478 0 001.087-.612l.118-.119.099-.296.099-.099v-.237l-.553-.553a1.37 1.37 0 00-1.916-.02l-.139.138-.079.119zM152.477 56.644l.296-.296a1.616 1.616 0 00.613-1.086 1.54 1.54 0 00-.435-1.067l-.257-.257-.296-.099-.099-.098h-.276a1.2 1.2 0 00-.929.572l-.099.1-.079.276v.197l.139.139c.089.234.223.449.395.632l.316.316.316.079h.237l.158.592zM161.425 63.854a1.338 1.338 0 001.007-.73l.119-.119.059-.336-.099-.099-.039-.237-.415-.414a1.482 1.482 0 00-1.64-.257l-.652.652-.059.256-.099.1c.018.37.173.722.435.987a1.256 1.256 0 001.383.197zM149.198 28.2l.138.375a1.421 1.421 0 001.086.375c.31-.017.604-.143.83-.356.125-.036.245-.089.355-.158l.554-.553.059-.336.02-.217a1.291 1.291 0 00-.218-.573l-.573-.573-.335-.059-.099.099c-.22.016-.432.091-.613.217l-.355.356c-.32.255-.532.622-.593 1.027l.099.099-.355.276zM140.012 37.029l-.118.118.099.1c.036.343.189.664.434.908a1.424 1.424 0 001.403.336 1.456 1.456 0 00.987-1.185v-.198l-.494-.494a1.393 1.393 0 00-1.745-.225 1.389 1.389 0 00-.447.442l-.119.198zM133.612 45.681a1.344 1.344 0 001.067-.75l.276-.277v-.277a1.45 1.45 0 00-.395-.83 1.256 1.256 0 00-.592-.315l-.119-.119h-.237l-.355.079-.119.119a1.486 1.486 0 00-.632 1.027 1.39 1.39 0 00.405 1.096 1.385 1.385 0 001.096.405l-.395-.158zM128.535 64.822l.099-.336-.02-.217-.849-.85-.376-.137-.118-.119-.257.02-.118.118-.336.139-.474.474-.158.395v.237l.019.296.613.612a1.336 1.336 0 001.553.119c.159-.096.296-.224.402-.375l.02-.376zM167.548 44.851a1.397 1.397 0 00-.039 1.936l.415.415.296-.02a1.371 1.371 0 001.244-1.481l-.592-.593a1.4 1.4 0 00-1.324-.257zM131.716 41.296a1.454 1.454 0 001.383-.87c.106-.244.138-.515.093-.779a1.378 1.378 0 00-.35-.702l-.415-.415a1.398 1.398 0 00-1.027.435 1.616 1.616 0 00-.415 1.205l1.027 1.027-.296.099zM157.317 28.1l-.079.316.019.218.099.099.099.296.276.276a1.481 1.481 0 001.166.455 1.31 1.31 0 00.987-.593l.119-.118.079-.277.099-.099v-.197l-.237-.237a1.424 1.424 0 00-.376-.652l-.335-.336-.277-.079-.099-.099h-.197l-.237.237a1.48 1.48 0 00-.85.652l-.256.138zM135.488 41.118l.415.415.336.059h.197c.271-.01.535-.085.771-.217l.553-.553.059-.336-.099-.099a1.36 1.36 0 00-.217-.612l-.553-.553-.336-.06-.099.1c-.233.007-.46.082-.651.216l-.554.553-.039.316-.099.1c.025.22.092.435.198.631l.118.04zM138.689 42.856l.118.119.079.355.336.336c.307.28.711.429 1.126.415.434-.067.824-.3 1.086-.652l.257-.257-.039-.237-.099-.098a1.258 1.258 0 00-.435-.435 1.34 1.34 0 00-1.521-.296l-.829.83-.079-.08zM136.93 38.53l.632.632.257.06a1.509 1.509 0 001.363-.89 1.392 1.392 0 00-.257-1.48l-.138-.14-.336-.137-.118-.119-.336-.02-.336-.059h-.197a1.4 1.4 0 00-.613.217l-.573.573-.059.336.099.099c.018.223.086.438.198.632l.414.296zM143.41 32.051l.355.04.099.098c.215-.036.422-.11.612-.217l.573-.573.04-.355.098-.099a1.429 1.429 0 00-.177-.573l-.81-.81-.356-.04a1.479 1.479 0 00-.632.198l-.573.573-.039.356v.197c.006.213.075.419.197.593l.613.612zM134.106 50.876l.099-.336.098-.098-.02-.257-.967-.968-.316-.119a1.545 1.545 0 00-1.121-.47 1.535 1.535 0 00-1.112.49l-.454.454-.02.218.099.099.059.296.455.454c.285.283.665.452 1.066.474a1.315 1.315 0 001.956-.217l.178-.02zM138.649 52.338l.553.553a1.291 1.291 0 001.107.316 1.48 1.48 0 001.185-.988 1.318 1.318 0 00-.655-1.54 1.313 1.313 0 00-1.637.355l-.118.119a1.365 1.365 0 00-.435 1.185zM160.417 40.762l.237.237.178.1.336.137.198.04h.276l.198-.04.355-.158.158-.118.119-.119.118-.158.119-.355.039-.198v-.276l-.039-.198-.138-.336-.139-.177-.217-.218-.178-.098-.355-.158-.198-.04-.257.02-.197.04-.356.157-.158.119-.099.099-.138.177-.118.356-.04.197-.02.257.079.198.139.336.098.177zM143.963 73.89l.119-.12.019-.256a1.062 1.062 0 00-.177-.573l-.593-.593-.355-.04-.257.02a1.42 1.42 0 00-.929.573l-.118.119-.079.316-.099.099.04.237.77.77.356.119.118.118.257.02.138-.138.336-.139.395-.395.059-.138zM130.076 68.576l-.158-.158-.375-.139-.119-.118h-.237l-.118.118-.336.139-.138.138a1.392 1.392 0 00-.52 1.55 1.4 1.4 0 00.282.504l.475.474.237.04.098-.099.316-.04.158-.158a1.365 1.365 0 00.785-1.044 1.366 1.366 0 00-.449-1.227l.099.02zM132.802 92.951a1.282 1.282 0 00-1.086-.375 1.374 1.374 0 00-1.007.652l-.119.119-.079.355v.198l.04.237.118.118c.06.217.176.415.336.573.269.219.602.343.948.356l.257-.02.118-.119a1.364 1.364 0 00.508-1.61 1.365 1.365 0 00-.33-.503l.296.02zM163.973 51.469l-.553-.553-.099.099-.435.237-.098.098a.73.73 0 00-.297.297l-.612.612-.02.217.02.218.158.395.77.77.257-.02.099-.098a1.4 1.4 0 00.968-.968 1.458 1.458 0 00-.158-1.304zM148.17 78.452l-.336-.335-.296-.06-.099.099-.217.02-1.047 1.047-.02.257c.03.216.104.425.218.612l.197.197.415.218.277.079.296-.02.138-.138.356-.158.138-.139a1.37 1.37 0 00-.02-1.679zM143.272 42.856l.138.139.099-.1.355-.078.099-.1.257-.019.119-.118.335-.139.474-.474.139-.375.019-.217.001-.277-.494-.494a1.406 1.406 0 00-1.067-.355.615.615 0 00-.257.06l-.197-.198h-.277l-.474.079-.198.197a.67.67 0 00-.296.06l-.672.67-.098.297v.277l.099.494.197.197a1.478 1.478 0 001.245.375l.454.1zM141.277 58.837l.118-.119-.02-.217-.118-.118-.138-.336-.139-.139a1.364 1.364 0 00-1.047-.414 1.452 1.452 0 00-1.027.553l-.118.118-.099.336-.119.119v.237l.099.098.138.336.139.139a1.366 1.366 0 001.047.414 1.595 1.595 0 001.047-.533l.118-.118.119-.356zM187.262 34.777l.297-.296a1.394 1.394 0 00.474-.988 1.336 1.336 0 00-.415-1.007l-.336-.336-.316-.08-.099-.098-1.047 1.047-.099.296v.277l.159.474.513.514c.254.175.565.245.869.197zM87.211 19.903l.099-.1.217-.019.198-.197a1.2 1.2 0 00.908-.91 1.256 1.256 0 000-.71l-.02-.217-.157-.396-.316-.316a1.565 1.565 0 00-1.245-.494l-.257.06-.276.276a1.648 1.648 0 00-.494.77l-.138.139-.02.257.889.889.612.968zM71.625 39.004l.06-.335.098-.1-.04-.236-.434-.435a1.425 1.425 0 00-1.66-.237l-.631.632-.06.257c0 .377.149.74.415 1.007a1.34 1.34 0 001.682.1c.178-.124.326-.286.432-.475l.138-.178zM120.752 26.619l.889.889.316.079h.277l.494-.139.197-.197c.315-.204.654-.37 1.008-.494l.651-.652.079-.276.02-.257-.118-.474-.178-.178a1.428 1.428 0 00-1.146-.356l-.257-.02-.434.12-.099-.1-.573.02-.296.099-.652.652a1.426 1.426 0 00-.178 1.284zM123.992 22.55l.099.098.335-.059.119-.118a1.51 1.51 0 00.889-1.482l-1.561-1.56-.276-.04-.237.04-.415.138-.553.553-.138.415-.119.118.04.277 1.382 1.383.435.236zM115.577 30.53l.138-.138v-.277l-.039-.237-.178-.415-.158-.158a1.407 1.407 0 00-1.107-.316 1.31 1.31 0 00-.888.613l-.139.138-.059.336v.197l.612.613a1.343 1.343 0 001.719.138l.158-.158-.059-.336zM102.323 46.195l-.376-.139h-.395l-.118-.118a1.06 1.06 0 00-.889.573 1.21 1.21 0 00-.119 1.027l.119.119.158.355a1.197 1.197 0 00.909.356c.333-.005.652-.14.888-.376a1.452 1.452 0 00-.177-1.797zM164.507-1.727l-.257-.257a1.289 1.289 0 00-.928.336 1.366 1.366 0 00-.455.967 1.313 1.313 0 001.205 1.442l.375-.138.771-.77.02-.217-.139-.376-.158-.158-.434-.83zM157.85 50.481l-.376.139-.75.75c-.007.16.006.319.039.474l.138.376.771.77a1.283 1.283 0 00.948-.356 1.261 1.261 0 00.434-.908 1.421 1.421 0 00-1.204-1.245zM156.23-10.28l.079.355.276.276a1.348 1.348 0 001.107.672.781.781 0 00.375-.02l.731-.73.099-.297.098-.099.02-.336-.099-.098-.098-.336-.968-.968a1.398 1.398 0 00-1.482 1.126l-.138.454zM165.593 14.846l.81.81c.141.045.287.072.435.079a1.23 1.23 0 001.007-.376 1.485 1.485 0 00.395-1.224l-.849-.85-.395-.079a1.34 1.34 0 00-1.324 1.324l-.079.316zM167.193-11.88l.731.73.376.06a1.287 1.287 0 001.066-.475 1.454 1.454 0 00.218-1.323l-1.008-1.008a1.507 1.507 0 00-1.383.79l-.138.139-.059.454.039.237.158.395zM165.692 27.508l-.257-.257a1.29 1.29 0 00-.948-.514c-.463.003-.907.18-1.245.494l-.474.474-.02.257.435.435a1.275 1.275 0 00.968.533c.188.282.466.492.79.593l.217.02.415-.06.909-.908a1.572 1.572 0 00-.455-1.403l-.335.336zM156.961 39.913l-.415-.415-.257-.02a1.37 1.37 0 00-1.343.79 1.507 1.507 0 00.336 1.68l.395.395.296.059a1.4 1.4 0 001.324-.81 1.507 1.507 0 00-.336-1.679zM183.628-27.842l.118-.118.356-.08a1.195 1.195 0 00.79-.79l.099-.098-.099-.099a1.817 1.817 0 00-.474-.75l-.138-.139-.316-.118-.119-.119h-.237l-.119.119-.335.098-.139.139a1.627 1.627 0 00-.454 1.56 1.343 1.343 0 001.067.395zM146.215 17.986l-1.027-1.027a1.371 1.371 0 00-1.403.89l-.138.137.02.415.572.573c.144.15.33.254.534.296l.237.04.395-.079.553-.553a1.252 1.252 0 00.316-1.027l-.059.335zM147.143 3.23a1.313 1.313 0 00-1.047.415 1.392 1.392 0 00-.356 1.186l.988.987.119-.118.355-.119.85-.85v-.236l-.02-.217-.158-.356-.435-.435-.336-.138.04-.118zM167.845 20.219l-.099.098-.375.06-.336.335a1.647 1.647 0 00-.553.554l-.02.217.02.375.73.731c.215.136.459.217.712.237l.296-.02.988-.987c.08-.393.01-.803-.198-1.146a1.365 1.365 0 00-1.165-.454zM173.652-48.09l.455.455c.217.196.497.308.79.316.179.007.359-.013.533-.059l.336-.336a1.316 1.316 0 00.356-1.54l-.85-.85-.316-.04a1.367 1.367 0 00-1.402.89l-.139.138v.592l.237.435zM144.772 11.092l.159-.158.019-.454-.059-.257-.849-.85-.257-.019a1.37 1.37 0 00-1.462 1.106l.02.218.336.335a1.334 1.334 0 00.948.475 1.37 1.37 0 001.145-.396zM146.55-20.079c.146-.244.222-.524.218-.81l-.83-.83-.336-.137-.118-.119a1.513 1.513 0 00-1.363.77l-.139.139-.02.454.06.257.158.355.138.139c.177.202.401.358.652.454l.257.02.454-.099.652-.652.217.06zM145.642 13.502a1.813 1.813 0 00-.395.04l-.375.375a1.394 1.394 0 00-.514 1.264 1.514 1.514 0 00.948 1.146l.099.099.257.02.474-.474a1.448 1.448 0 00.434-1.58 1.342 1.342 0 00-.928-.89zM137.207-23.14l.099.335.612.613.297.099a1.342 1.342 0 001.047-.376c.277-.285.433-.668.434-1.066l-.987-.988-.356-.08a1.313 1.313 0 00-1.343 1.068l.02.256.177.139zM150.027-8.503l-.316.316-.118.316-.139.139v.237l.099.098.099.297.099.098a1.475 1.475 0 001.106.632 1.425 1.425 0 001.383-1.382l-.02-.257-.138-.415-.494-.494a1.426 1.426 0 00-.731-.217l-.217.02-.356.158-.257.454zM110.421 22.49l-.375.139-.434.434a1.095 1.095 0 00-.297.968 1.283 1.283 0 00.613.85l.158.157.375.178a1.26 1.26 0 001.17-.78c.063-.153.095-.318.094-.484.027-.3-.049-.6-.217-.85a1.398 1.398 0 00-1.087-.612zM139.874 13.265l.335-.098a1.282 1.282 0 00.968-1.363l-.395-.395a1.508 1.508 0 00-.928-.613h-.237l-.158.158a1.564 1.564 0 00-.435 1.66 1.37 1.37 0 001.205.928l-.355-.277zM141.297 3.468L141 3.17l-.355-.158a1.366 1.366 0 00-1.305.86 1.378 1.378 0 00-.098.543c-.02.397.115.786.375 1.086a1.377 1.377 0 00.968.455l.376-.139.099-.098a1.344 1.344 0 00.671-.909 1.253 1.253 0 00-.296-1.165l-.138-.178zM138.373 8.604l-.889-.89a1.597 1.597 0 00-.692.139l-.75.75a1.4 1.4 0 00-.059 1.364 1.338 1.338 0 001.106.71 1.15 1.15 0 00.889-.335.807.807 0 00.256-.257 1.394 1.394 0 00.317-1.185l-.178-.296z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M134.462 13.898l.177-.534.02-.296-.099-.336-.908-.909a1.369 1.369 0 00-1.521 1.205 1.288 1.288 0 00-.005 1.017 1.285 1.285 0 00.736.702l.098.099.297.02.553-.554c.155-.198.282-.417.375-.651l.277.237zM142.916 39.893a1.368 1.368 0 00-1.166.573 1.28 1.28 0 00-.098 1.284l.77.77h.237l.257-.059.513-.513a1.307 1.307 0 00.013-1.304 1.317 1.317 0 00-1.119-.672l.593-.079zM120.733-24.325l.414-.06.494-.493c.145-.185.259-.392.336-.613l.02-.217-.06-.376-.513-.513a1.284 1.284 0 00-.573-.336l-.099-.099-.415.06-.493.493a1.28 1.28 0 00-.336.573l-.02.218.079.395.494.493c.148.153.331.268.533.336l.139.139zM135.232-17.846l.395.395c.148.152.331.268.533.335h.277l.474-.118.474-.474.158-.356.119-.118a1.32 1.32 0 00-1.225-1.422l-.336.02-1.047 1.046.119.119.059.573zM124.921-4.374l-.218.217-.138-.138-.119.118-.375.138-.731.731a1.355 1.355 0 00.393.95 1.344 1.344 0 00.951.393 1.31 1.31 0 001.343-.908 1.34 1.34 0 00-1.106-1.501zM134.047-2.873l.79.79.277.04a1.367 1.367 0 001.456-1.068 1.365 1.365 0 00-.014-.651l-.119-.119-.118-.355-.83-.83-.257-.02-.118.119-.356.118-.849.85v.237l-.02.257.118.118.04.514zM166.185-7.357l.099.099a1.422 1.422 0 001.225.869 1.451 1.451 0 001.363-.85 1.422 1.422 0 00-.257-1.6l-.158-.158-.356-.158-.217-.02-.257-.02-.237.238a1.425 1.425 0 00-.908 1.304l-.297.296zM114.728-40.919v-.276l-.138-.454-.316-.316a1.344 1.344 0 00-1.087-.613h-.237l-.79.79c-.198.311-.275.683-.217 1.047l.928.929c.198.07.404.117.612.138a1.367 1.367 0 001.245-1.245zM214.088-13.422a1.257 1.257 0 001.225-.75l.059-.336-.079-.356a1.26 1.26 0 00-.779-1.17 1.258 1.258 0 00-.485-.094l-.237.237a1.256 1.256 0 00-.751 1.422 1.424 1.424 0 001.047 1.047zM141.711 62.195a1.37 1.37 0 00-1.324-1.323 1.198 1.198 0 00-.513.04l-.435.434a1.309 1.309 0 00-.355 1.264l.73.73.218-.217-.119.317c.208.13.446.205.692.217l1.007-1.008v-.355l.099-.099zM129.069-45.304l.751.75a1.424 1.424 0 001.442-1.244 1.286 1.286 0 00-1.008-1.442l-.099-.099-.316.04-.276.277a1.313 1.313 0 00-.692 1.64l.198.078zM143.212-9.234a1.227 1.227 0 00-1.165.613l-.158.158-.059.454-.099.099.02.217.158.355.493.494.336.139.099.098a1.34 1.34 0 001.403-.928l.098-.099-.039-.355-.454-.455a1.4 1.4 0 00-.633-.79zM164.902-7.416l-.454-.455a1.535 1.535 0 00-1.324.494 1.024 1.024 0 00-.217.217c-.333.2-.579.517-.692.89a1.877 1.877 0 00.494 1.955l.138.138.04.119.119.118h.276l.514-.513.158-.158a1.365 1.365 0 00.79-.79l.434-.435a1.42 1.42 0 00-.237-1.146l-.197-.197.158-.237zM189.614 37.997l-.968-.968h-.237a1.256 1.256 0 00-.968.454 1.224 1.224 0 00-.277 1.027l.099.1a1.339 1.339 0 00.968.967h.197l.376-.059.671-.672c.103-.144.175-.308.212-.481a1.33 1.33 0 00.006-.526l-.079.158zM174.621 23.715l.572-.573a1.393 1.393 0 00-.276-1.225 1.37 1.37 0 00-1.008-.493l-.691.691a1.25 1.25 0 00-.316 1.067l.711.71c.207.134.446.21.692.218h.316l.217-.217-.217-.178zM156.506 11.725l-.237.04-.454.216-.711.712.019.494.079.276.218.217c.106.155.24.289.395.395l.296.1.573.019.276-.08.237-.236-.039.04c.257-.268.411-.618.434-.988l-.138-.139-.158-.395-.474-.474a1.229 1.229 0 00-.316-.197zM155.914 21.996l.119.119a1.397 1.397 0 001.066-.395 1.482 1.482 0 00.395-1.185l-.731-.731-.039-.237-.514-.514-.395-.197-.276-.277-.237.04-.376.138-.513.513-.139.376-.098.099a1.284 1.284 0 00.434 1.067c.02.313.13.615.316.868l.415.415.573-.099zM139.657-8.72l.118-.119.099-.296.099-.099a1.513 1.513 0 00-1.462-1.462l-.118.119-.395.118-.791.79-.059.257v.198l.04.276.553.553a1.31 1.31 0 001.402.257l.593-.592h-.079zM159.489-1.412l-.237-.04a1.395 1.395 0 00-1.382.95l-.079.276-.119.118.02.257.118.119.178.375.138.138c.271.232.612.364.968.375.091.02.186.02.277 0l.909-.908a1.597 1.597 0 00-.237-1.304l-.435-.435-.119.08zM160.694 9.196h-.316l-.513.119-.158.158a1.672 1.672 0 00-.494.257 1.37 1.37 0 00-.514.197l-.79.79a1.65 1.65 0 00.356 1.423l.355.355a1.456 1.456 0 00.929.613l.276-.04.435-.434c.285-.169.523-.407.691-.692l.059-.296.198-.198.079-.316a1.51 1.51 0 00-.593-1.936zM159.944 3.784l-.909.908a1.397 1.397 0 001.086 1.719l.218.02.138-.139a1.372 1.372 0 00.435-1.5 1.398 1.398 0 00-1.225-.949l.257-.06zM155.163-3.88l.79.79.217.02a1.398 1.398 0 001.502-1.107l.098-.099-.079-.355-.652-.652a1.954 1.954 0 00-.73-.217h-.277l-.988.987-.039.237.118.119.04.276zM171.124 46.965a1.421 1.421 0 00-1.403 1.009 1.537 1.537 0 00-.059.69l.06.257.316.316c.275.274.639.442 1.027.474a1.425 1.425 0 001.106-.435l.474-.474a1.618 1.618 0 00-.277-1.264 1.557 1.557 0 00-1.244-.573zM165.633-25.294l-.494-.493-.217-.02a1.513 1.513 0 00-1.185.434 1.327 1.327 0 00-.304.513c-.062.191-.08.394-.052.594l.316.316a1.174 1.174 0 001.146.908c.18.02.362 0 .533-.059l.77-.77c.092-.207.152-.427.178-.652l-.118-.119-.099-.335-.316-.317h-.158zM161.682-30.508l.909.908a1.787 1.787 0 001.679-.889l.118-.118.059-.296-.098-.1-.02-.216-.415-.415a1.402 1.402 0 00-.988-.474 1.648 1.648 0 00-.869.237l-.869.869v.237l.119.118.375.139zM164.902 1.473l-.257-.02a1.426 1.426 0 00-1.126.415 1.545 1.545 0 00-.375 1.205l1.007 1.007.257.02a1.398 1.398 0 001.126-.415 1.561 1.561 0 00.395-1.185l-1.027-1.027zM175.015-28.474l-.138.139a1.178 1.178 0 00-.711 1.145l.039.277.692.691.355.158a1.342 1.342 0 001.146-.513c.178-.234.282-.517.296-.81a1.335 1.335 0 00-.829-1.242 1.337 1.337 0 00-.514-.102l-.336.257zM192.635-48.701l.553.553c.144.063.298.096.455.098a1.366 1.366 0 001.323-.77 1.588 1.588 0 00-.257-1.758l-.454-.454-.316-.04a1.312 1.312 0 00-1.185.553c-.161.22-.26.478-.287.75-.028.27.016.543.129.791l.039.277zM161.543 14.866l.988.987c.103.03.209.044.316.04a1.285 1.285 0 001.166-.494 1.619 1.619 0 00.237-1.501l-.712-.711-.296-.1h-.276l-.474.12-.198.197a1.591 1.591 0 00-.751 1.462zM213.594-4.513l-.375.138-.198.198a1.284 1.284 0 00-.395.395l-.177.178.059.454.118.395.494.494.356.158a1.199 1.199 0 00.948-.316c.241-.236.388-.552.415-.889a1.344 1.344 0 00-1.245-1.205zM190.108-23.377a1.51 1.51 0 00-1.541 1.145l-.04.237.04.435 1.007 1.007h.316l.218-.02.395-.157.73-.731-.197-.198.277.04.059-.494-.02-.257-.928-.928-.316-.08zM161.939-12.078l.257.02.118-.12.356-.118.829-.83.02-.256-.118-.356-.83-.83a1.366 1.366 0 00-1.379.887c-.068.184-.097.38-.083.575l.119.12.138.335.573.573zM175.193-39.16a1.284 1.284 0 00-1.185.987l.02.297.158.474a1.283 1.283 0 001.126.85 1.62 1.62 0 00.513-.08l.711-.711.178-.296a1.314 1.314 0 00-.928-1.442l-.237-.04-.356-.04zM113.799 10.698l-.454-.455-.336-.138-.119-.119-.316-.04-.118.12-.336.098-.83.83a1.46 1.46 0 00.731 1.442l.277.079.533.02.257-.06.612-.612c.145-.237.214-.513.198-.79l-.099-.099v-.276zM178.195 36.654l-.276-.08-.553.04-1.008 1.008a1.396 1.396 0 00.218 1.165 1.333 1.333 0 00.948.474l.276-.04.139-.138a1.462 1.462 0 00.533-1.205 1.457 1.457 0 00-.277-1.224zM162.136 31.459l-.474-.475-.039-.316a1.481 1.481 0 00-1.245-1.244l-.099.099-.316.079-.75.75a1.621 1.621 0 00.098 1.403c.121.238.315.432.553.553l.198.197a1.367 1.367 0 00.849.968l.336-.02.731-.73.119-.316v-.237l-.02-.218.059-.494zM120.574-8.246l.336.02.099-.1.355-.118.791-.79a1.407 1.407 0 00-.297-1.126 1.413 1.413 0 00-1.047-.533l-.217.02-1.027 1.027v.434l.138.376.375.375.356.158.138.257zM149.217-56.05l.257-.02.435-.118.395-.395a1.43 1.43 0 00.375-.613l.099-.098-.02-.376-.79-.79a1.515 1.515 0 00-.474-.118 1.424 1.424 0 00-1.501 1.264 1.339 1.339 0 001.224 1.264zM164.032-37.323a1.348 1.348 0 00.83.83h.237l.395-.04.573-.573a1.45 1.45 0 00.316-1.027l-.652-.652-.355-.119-.099-.098a1.284 1.284 0 00-1.363 1.047v.236l.118.396zM188.744-36.711l.395.395c.155.165.354.281.573.336l.257.02.474-.119.533-.533.158-.356.139-.138a1.333 1.333 0 00-.737-1.304 1.345 1.345 0 00-.508-.138l-.296.02-.751.75-.118.316-.099.1.099.098-.119.553zM143.153.228l.573-.573a1.373 1.373 0 00.072-1.334 1.365 1.365 0 00-1.099-.76 1.232 1.232 0 00-1.027.435l-.514.514a1.365 1.365 0 00.297 1.047c.256.278.609.448.987.474l.711.197zM140.664 33.453l.533-.533c.107-.218.181-.451.217-.691l-.059-.336-.849-.85-.336-.02-.573.1-.099-.1a1.509 1.509 0 00-.948 1.423l.119.119.118.355.83.83h.316l.099-.099.355-.119.277-.078zM141.177-35.17l.415.415c.155.159.344.28.553.355h.237l.376-.06.395-.394a.59.59 0 00.217.217l.119.118.217-.02.257-.019.256-.06.632-.631a1.253 1.253 0 00-.059-1.245 1.227 1.227 0 00-1.047-.612 1.537 1.537 0 00-1.027.355l-.296-.098-.119-.119a1.423 1.423 0 00-1.382.87l-.04.236v.435l.296.257zM144.931-31.694a1.522 1.522 0 00.76 1.6c.18.098.379.159.583.178l.237-.04.375-.375a1.642 1.642 0 00.356-1.264l-1.106-1.106-.376-.02-.118.119-.316.118-.692.692.297.098zM175.035-10.202a1.453 1.453 0 00-1.442 1.442 1.32 1.32 0 00.385.94 1.314 1.314 0 00.939.384l.375-.139.494-.493.178-.376.296-.296-.118-.395-.494-.494-.613-.573zM174.126 28.673l-.099-.335-.651-.652-.297-.1-.257-.019-.118.119-.395.158-.494.493-.138.376-.139.138a1.258 1.258 0 002.351.652l.178-.494.059-.336zM159.233-19.17l.217-.02.257-.257a1.544 1.544 0 00.415-1.6 1.372 1.372 0 00-1.264-.908.81.81 0 00-.435.04l-.672.67a1.453 1.453 0 000 1.344 1.454 1.454 0 001.482.731zM155.223-34.32l.81.809.256.06a1.565 1.565 0 001.541-1.541v-.238l-.099-.414-.77-.77-.316-.04h-.237l-.257.059-.612.612c-.157.302-.258.63-.297.968l.099.099-.118.395zM155.282 22.194l-.869-.87-.356-.039a1.423 1.423 0 00-1.422.87l-.099.098-.059.455.02.256.098.297.474.474c.21.127.447.202.692.217.105.01.211.01.316 0l.632-.632a1.628 1.628 0 00.375-1.244l.198.118zM156.783-17.076l-.02-.217-.079-.356-.336-.336a1.502 1.502 0 00-.632-.434l-.237-.04-.434.08-.968.967v.237l.039.237.139.376.493.494.415.138.099.099a1.312 1.312 0 001.521-1.245zM160.615 21.088a1.294 1.294 0 00-.928-.454l-.376.138-.414.415c-.17.172-.28.393-.317.632l-.948.948c-.031.2-.023.404.024.6.048.197.134.382.253.546l.612.612a1.372 1.372 0 001.284-.81l.04-.276-.02-.257a1.32 1.32 0 00.533-1.008c.214-.319.305-.705.257-1.086zM131.735-28.73l-.178-.178-.098-.376a1.233 1.233 0 00-1.008-.77l-.099.099-.355.079-1.027 1.027c-.021.333.062.664.237.948l.098.099a1.287 1.287 0 001.166 1.165l.355-.355a1.534 1.534 0 00.909-1.739zM153.168-44.948l.396.395c.16.126.34.226.533.296l.237.04.415-.06.573-.572a1.281 1.281 0 00.296-1.047l-.771-.77-.316-.12-.098-.098a1.428 1.428 0 00-1.423.909l-.019.217.019.415.158.395zM94.993 9.315l-.256.02-1.027 1.027a1.426 1.426 0 00.217 1.244 1.453 1.453 0 001.106.553l.257-.02 1.007-1.007a1.45 1.45 0 00-.276-1.225 1.397 1.397 0 00-1.028-.592zM175.174 34.62l-.119-.12a1.276 1.276 0 00-.869-.869h-.237l-.395.08-.533.533a1.454 1.454 0 00-.297 1.007l.85.85.355.118a1.339 1.339 0 001.383-1.027l-.099-.099-.039-.474zM135.469 20.238l-.277-.039-.494.099-.573.573-.177.375-.119.119a1.317 1.317 0 00.825 1.249c.164.066.341.098.518.094.107-.005.212-.018.316-.04l.929-.928a1.369 1.369 0 00-.948-1.502zM184.162-14.39a3.001 3.001 0 00-.336.336 1.23 1.23 0 00-.514 1.383 1.288 1.288 0 001.764.807c.224-.101.414-.265.547-.471l.119-.119.257-.257-.04-.276-.217-.454-.474-.474-.336-.06h-.197l-.573-.415zM180.191 3.764a1.454 1.454 0 00-.415.02l-.672.671a1.482 1.482 0 00-.019 1.324 1.567 1.567 0 001.106.79l.217-.02.257-.257a1.513 1.513 0 00.415-1.56 1.404 1.404 0 00-.889-.968zM142.6 25.315l-.336.336a1.372 1.372 0 00-.218 1.047l.909.909c.1.04.209.054.316.039a1.369 1.369 0 001.442-1.245 1.343 1.343 0 00-1.205-1.442l-.099-.098-.335.098-.119.119-.355.237zM197.021 31.261l-.731-.73-.276-.04a1.34 1.34 0 00-1.462 1.264 1.34 1.34 0 001.205 1.442l.217-.02.415-.138.514-.514c.162-.23.245-.508.237-.79l-.099-.099-.02-.375zM131.538 3.191l.534.534.375.177.237.04a1.505 1.505 0 001.343-.75l.158-.159.02-.454-1.007-1.008-.237-.04a1.396 1.396 0 00-1.482 1.127v.237l.059.296zM111.528.09l-.178.375-.119.119-.02.336.81.81.356.158a1.286 1.286 0 00.968-.336 1.306 1.306 0 00.435-.949 1.397 1.397 0 00-1.383-1.382l-.869.869zM125.395 22.016l-.139.139-.355.118-.474.474-.139.336-.138.138v.198l.119.118.099.297.118.118c.16.194.373.338.612.415l.297.296h.237l.257-.059.434-.198.533-.533.08-.356.019-.217a1.478 1.478 0 00-1.56-1.284zM148.388-16.424l-.119.316-.099.099-.019.335.099.1.118.355.79.79.217.02a1.371 1.371 0 001.502-1.502l-.099-.098-.079-.277-.553-.553a1.203 1.203 0 00-.711-.237.817.817 0 00-.297.02l-.651.652-.099-.02zM130.688 30.668l-.533-.533-.454-.06-.257.06-.83.83-.158.434-.138.139c.053.307.19.594.395.83l.573.572h.316l.296-.06.573-.572s.454-1.008.039-1.699l.178.06zM105.207-18.281l-.514-.514a1.312 1.312 0 00-.751-.276 1.23 1.23 0 00-1.185 1.185l.02.217.138.139a1.285 1.285 0 001.087 1.086l.395-.079.336-.336c.235-.125.427-.318.553-.553v-.277l-.079-.592zM128.022 5.878l-.672-.672-.355-.119-.119-.118a1.483 1.483 0 00-1.442 1.442l.02.217.118.395.494.494.336.138.138.139h.198l.652-.652a1.657 1.657 0 00.355-1.264h.277zM125.869-12.394l.849.85.218.019.118-.119.356-.118.849-.85v-.197l-.119-.316-.434-.435a1.426 1.426 0 00-.85-.336l-1.066 1.067v.237l.118.119-.039.079zM194.67-7.278l.099-.099.395-.079.849-.85a1.399 1.399 0 00-1.086-1.719l-.198.001-.395.079-.849.85c-.084.401-.005.82.217 1.165a1.369 1.369 0 00.968.652zM228.705-1.885a1.391 1.391 0 00-1.086-.534h-.237l-.751.75a1.596 1.596 0 00.04 1.502 1.276 1.276 0 00.872.629 1.285 1.285 0 001.044-.254l.691-.691a1.451 1.451 0 00-.573-1.402zM197.199 10.184l-.296.02-.949.948a1.398 1.398 0 00.85 1.442l.375.099.257.02.316-.08.79-.79a1.396 1.396 0 00-.908-1.62l-.139.139.099-.1-.395-.078zM183.846-4.987l.256.257a1.256 1.256 0 001.146-.553c.131-.208.206-.446.217-.692a1.335 1.335 0 00-.699-1.328 1.346 1.346 0 00-.506-.153l-.395.118-.197.198a.951.951 0 00-.395.395.551.551 0 00-.139.138c-.081.166-.115.35-.098.534l.079.276.494.494.237.316zM183.312 16.367l-.178-.178a1.296 1.296 0 00-.415-.415l-.178-.178-.513-.04-.257.06-.929.928v.316l.02.218.119.355.829.83a1.425 1.425 0 001.403-.85l.039-.276-.039-.474.099-.296zM184.714-1.47l-.573-.573a1.26 1.26 0 00-.474.079l-.336-.139-.138-.138a1.314 1.314 0 00-1.086.454A1.29 1.29 0 00181.81-.7l1.067 1.066.119-.118.533.533a1.283 1.283 0 001.383-1.185 1.424 1.424 0 00-.218-.73l.02-.337zM170.275-23.694a1.224 1.224 0 001.106.83c.174.002.347-.025.513-.079l.573-.573a1.314 1.314 0 00-.138-1.205 1.448 1.448 0 00-1.146-.671 1.287 1.287 0 00-1.323 1.047l.039.276.376.375zM186.136-1.47l.316-.317a1.34 1.34 0 00.297-1.284 1.257 1.257 0 00-.889-.889l-.257-.02-.415.06-.928.928-.04.395.06.297.928.928.237-.04.691-.059zM212.073 11.665l-.316.04-.296.296a1.45 1.45 0 00-.988.988 1.26 1.26 0 00.553 1.501l.158.158.474.04.099.098.119-.118.355-.119.79-.79.178-.375.119-.119a1.324 1.324 0 00-.245-1.093 1.311 1.311 0 00-1-.507zM196.31 17.829l-.533-.534a1.291 1.291 0 00-1.067-.592.78.78 0 00-.375.02l-.968.967v.553l.079.317.83.83a.502.502 0 00.237.039 1.283 1.283 0 00.987-.356 1.338 1.338 0 00.81-1.244zM196.646-22.014l.888.889a1.4 1.4 0 001.067-.395 1.502 1.502 0 00.395-1.186l-1.086-1.086a1.399 1.399 0 00-1.462 1.225l.02.217.178.336zM194.552 4.534l.296-.296a1.484 1.484 0 00-.277-1.225 1.34 1.34 0 00-1.027-.513l-.118.118-.316.119-.395.395c-.234.296-.34.672-.297 1.047l.692.691c.222.13.473.205.731.217a.53.53 0 00.296-.02l.691-.69-.276.157zM206.127 4.06a1.463 1.463 0 00.297-1.244l-1.047-1.047a1.397 1.397 0 00-1.383.87l-.099.098-.02.296.494.494a1.285 1.285 0 00.968.454c.247.113.525.141.79.08zM173.376-14.864l.276.277a1.316 1.316 0 001.067-.435c.22-.225.358-.517.395-.83a.873.873 0 00.138-.335l.02-.217-.059-.376-.534-.533a1.257 1.257 0 00-.513-.277l-.218-.02-.414.06-.376.375.158-.158a1.54 1.54 0 00-.316.553v.198l-.138.138a1.428 1.428 0 00.217 1.126l.474.474-.177-.02zM172.29-2.162l.454.454c.256.236.582.382.929.415a1.537 1.537 0 001.284-.691 1.485 1.485 0 00-.04-1.541l-.494-.494-.276-.04a1.398 1.398 0 00-1.126.415 1.51 1.51 0 00-.376 1.205l-.355.277zM173.139 8.683a1.507 1.507 0 00.316-1.264l-.474-.475a1.23 1.23 0 00-.83-.395c-.225-.01-.45.038-.652.139l-.77.77a1.363 1.363 0 001.221 1.755 1.372 1.372 0 001.09-.432l.099-.098zM181.909 26.342l-.138-.336-.434-.434-.317-.119-.118-.118-.336-.02-.118.119-.356.118-.849.85v.237l.138.138.158.395.513.514.376.138h.276a1.398 1.398 0 001.403-1.403l-.198-.079zM198.147-14.39a1.395 1.395 0 001.264 1.502l.296-.06.455-.454a1.396 1.396 0 00-.02-1.955l-.395-.396-.356-.039a1.397 1.397 0 00-1.244 1.402zM168.832 6.51a1.451 1.451 0 00.455-1.56 1.263 1.263 0 00-1.047-1.048l-.237.237a1.37 1.37 0 00-.474 1.027 1.515 1.515 0 00.454 1.166l.138.138.336.138.118.119h.238l.335-.099-.316-.118zM179.441-20.256l1.027 1.027a1.51 1.51 0 001.185-.435 1.394 1.394 0 00.355-1.106l-.711-.711a1.422 1.422 0 00-.711-.237l-.296.02-.691.691a1.372 1.372 0 00-.218 1.008l.06-.257zM171.559 3.902l.395-.079.513-.513a1.28 1.28 0 00.336-.573l-.099-.099-.059-.415-.494-.494a1.223 1.223 0 00-.573-.335l-.217-.02-.375.06-.494.493a1.537 1.537 0 00-.336.612l-.02.218.06.375.493.494c.156.158.345.279.554.355l.316-.079zM175.095 3.211l.533.533.237.04a1.48 1.48 0 001.146-.435 1.59 1.59 0 00.375-1.205l-.75-.75a1.569 1.569 0 00-.632-.158 1.4 1.4 0 00-1.324.81l-.138.138-.02.415.04.237.177.177.356.198zM170.927.979l.414-.06.376-.375a1.534 1.534 0 00.454-1.56 1.346 1.346 0 00-1.185-.948l-.119.118-.355.08-.731.73c-.16.158-.276.356-.336.573v.197l.04.396.533.533c.151.155.343.265.553.316h.356zM170.926-8.147l.395-.04.514-.514c.143-.16.25-.348.316-.553l.02-.217-.079-.395-.514-.514a1.422 1.422 0 00-.533-.296l-.218-.02-.395.04-.513.514c-.16.158-.276.355-.336.572l-.02.218.059.375.514.514c.169.13.355.237.553.316h.237zM176.932 12.16l-.139-.337-.77-.77-.375-.02-.218.02-.316.119a1.542 1.542 0 00-1.131.449 1.531 1.531 0 00-.449 1.13l.099.1.079.316.632.632c.145.056.299.089.454.099.4.017.792-.11 1.106-.356a1.34 1.34 0 001.403-1.402l-.375.02zM181.238 10.342l.573.573a1.202 1.202 0 001.047-.494 1.457 1.457 0 00.237-1.501 1.28 1.28 0 00-1.146-.79 1.144 1.144 0 00-.948.355 1.35 1.35 0 00-.415 1.047l.119.118c.092.286.28.53.533.692zM190.127-12.77l.297.02.217-.02.355-.118.158-.119.218-.178.118-.158.138-.335.06-.218.019-.138-.039-.197-.119-.356-.098-.178-.336-.336-.356-.158-.197-.04-.316-.039-.198.04-.355.118-.198.119-.178.178-.118.158-.178.335-.039.198v.158l.019.217.119.356.118.158.158.197.178.139.356.158.197.04zM199.49 22.984l-.414-.415a1.176 1.176 0 00-.534-.296l-.217-.02-.415.06-.573.573a1.48 1.48 0 00-.296 1.046l.751.751.316.119.118.118h.237l.218-.02.355-.118.85-.85-.119-.118-.277-.83zM185.525 28.14l-.119.118-.355.119-.81.81a1.357 1.357 0 00.631 1.516c.17.1.358.162.554.183l.099-.1.335-.059.791-.79a1.465 1.465 0 00-.099-1.323 1.425 1.425 0 00-1.027-.474zM203.678 44.634a1.255 1.255 0 00-1.067.435 1.363 1.363 0 00-.336 1.165l.731.731c.179.13.391.205.612.217.348-.006.683-.131.948-.355l.455-.455a1.348 1.348 0 00-.678-1.55 1.342 1.342 0 00-.586-.149l-.079-.039zM199.826-7.061l-.099.099-.276.079-.691.691a1.061 1.061 0 00-.08.435l.119.118.079.395 1.007 1.008.257-.06.534-.533a1.312 1.312 0 000-1.343 1.518 1.518 0 00-.85-.889zM205.673 23.675l-.296.02-1.067 1.067.119.316.395.395c.168.131.355.237.553.316h.276l.494-.099.553-.553.158-.355.119-.119a1.34 1.34 0 00-1.304-.988zM178.532.05l1.066-1.066.158-.356.119-.118v-.237l-.02-.218-.158-.395-.81-.81a1.23 1.23 0 00-1.047.415.808.808 0 00-.178.178l-.908.909a.867.867 0 00-.218.217l-.158.158-.039.474.039.237.929.928a1.395 1.395 0 001.185-.513l.04.197zM186.769 12.633a1.368 1.368 0 00-1.462 1.462l.099.099.138.336.435.434.336.139.118.118a1.398 1.398 0 001.387-.874c.072-.18.104-.374.095-.568l-1.146-1.146zM206.404-34.933l.316-.316a1.383 1.383 0 00-.297-1.047 1.28 1.28 0 00-.987-.474h-.277l-1.047 1.047-.039.434 1.086 1.087a1.26 1.26 0 001.245-.731zM121.227 19.666l.414-.415a1.344 1.344 0 00.158-1.304 1.282 1.282 0 00-.513-.514l-.553-.553-.257-.02a1.647 1.647 0 00-1.284.455l-.454.454c-.048.304.001.614.138.89l.988.987.296-.02.118-.118.949.158zM122.056 44.318l-.929-.928a1.426 1.426 0 00-1.422.908l-.02.297.494.494a1.377 1.377 0 00.968.454 1.2 1.2 0 001.027-.435 1.373 1.373 0 00.297-1.205l-.415.415zM150.916 2.658l.178.178.514.039.256-.06.751-.75.395-.197a3.66 3.66 0 01.435-1.028l.059-.256.02-.455-1.027-1.027a1.372 1.372 0 00-1.107.474L150.185.781l-.039.356-.119.118a1.34 1.34 0 00.889 1.403zM150.679-2.518l.494-.494a1.51 1.51 0 00-.316-1.698l-.218-.218-.572-.098-.257.02-.099-.1-.217-.02-.455.06-.513.514-.198.395-.158.158-.019.296.039.237.178.415.691.691.573.1.277-.04.276-.08.099-.098.395-.04zM149.593 8.999l-.632-.632-.395-.158-.237-.04a1.316 1.316 0 00-1.027.474 1.36 1.36 0 00-.297 1.047l.83.83.336.098.099.1a1.284 1.284 0 001.343-1.028l.02-.217-.04-.474zM149.909 29.543l-.375.138-.415.415a1.14 1.14 0 00-.296 1.007 1.28 1.28 0 00.612.85l.415.414a1.288 1.288 0 00.928-.335c.236-.23.377-.54.395-.87a1.399 1.399 0 00-.715-1.457 1.393 1.393 0 00-.549-.162zM202.078-26.973l-.376-.138a1.226 1.226 0 00-.908.395 1.407 1.407 0 00-.376 1.008 1.29 1.29 0 001.324 1.324l.355-.159.494-.494.139-.375.098-.099-.019-.217-.139-.375-.177-.178-.415-.692zM199.53 25.631l-.376.178-.494.494-.138.375c.007.16.034.32.079.474l.178.375.75.751a1.318 1.318 0 001.226-.815c.066-.161.099-.334.098-.508a1.4 1.4 0 00-1.323-1.324zM193.189-34.834l.098.375.277.276a1.311 1.311 0 001.146.593c.127 0 .254-.02.375-.06l.711-.71.099-.297.099-.098v-.316l-.119-.119-.138-.336-.435-.434-.316-.119-.118-.118-.218-.02a1.37 1.37 0 00-1.323 1.323l-.138.06zM204.448-10.498l.83.83c.141.041.289.055.435.04a1.281 1.281 0 00.987-.435 1.631 1.631 0 00.297-1.245l-.85-.85-.395-.039a1.344 1.344 0 00-1.323 1.324l.019.375z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M203.994-37.264l.514.513.296.1.395.039a1.282 1.282 0 001.027-.553 1.51 1.51 0 00.119-1.344l-.81-.81-.257-.059a1.395 1.395 0 00-1.304.87l-.158.157.02.455.04.237.118.395zM205.554 2.105l-.256-.06a1.205 1.205 0 00-.988-.434 1.986 1.986 0 00-1.225.593l-.118.118-.119.356-.098.098v.237l.474.475a1.338 1.338 0 001.007.414c.206.273.498.468.83.553l.118-.118.395-.08.909-.908a1.648 1.648 0 00-.553-1.383l-.376.139zM197.87 15.122l-.395-.395-.296-.02a1.37 1.37 0 00-1.265.909 1.51 1.51 0 00.455 1.64l.395.395a.752.752 0 00.296.02 1.371 1.371 0 001.245-.89 1.452 1.452 0 00-.435-1.66zM219.145-54.47l.118-.118.336-.099a1.283 1.283 0 00.81-.81l.099-.099-.099-.098a1.599 1.599 0 00-.612-.613l-.119-.118-.336-.099-.099-.099-.256.02-.119.119-.336.138-.118.118a1.679 1.679 0 00-.336 1.6 1.227 1.227 0 001.067.159zM185.406-5.876l-.592-.592-.317-.08-.098-.098a1.455 1.455 0 00-1.363 1.007v.237l.059.415.573.573c.167.132.363.22.573.257l.217.02.395-.119.553-.553a1.368 1.368 0 00.237-1.067h-.237zM185.169-20.651a1.337 1.337 0 00-1.29 1.094c-.037.2-.029.407.026.604l.751.751.316.119.118.118.119-.119.355-.157.455-.455.138-.375.118-.119v-.237l-.138-.138-.178-.375-.474-.474-.316-.119v-.118zM207.135-5.323l-.217.02-.356.119-.335.335c-.201.192-.339.44-.395.711l-.02.218.059.375.691.691c.227.135.487.203.751.198l.277-.04.987-.987a1.365 1.365 0 00-.662-1.467 1.373 1.373 0 00-.543-.173h-.237zM207.628-73.887l.474.474c.239.164.521.254.81.257.175-.022.347-.062.514-.119l.355-.356a1.308 1.308 0 00.237-1.54l-.829-.83a1.093 1.093 0 00-.356-.04 1.427 1.427 0 00-1.323 1.008l-.04.237.04.553.118.356zM183.371-12.69l.138-.14v-.473l-.987-.988-.257-.02a1.368 1.368 0 00-1.304 1.304l.02.217.257.257a1.342 1.342 0 00.968.415 1.426 1.426 0 001.165-.573zM182.838-43.763c.147-.243.21-.528.178-.81l-.85-.85-.355-.118-.099-.099a1.4 1.4 0 00-1.284.85l-.059.257.019.454.554.553c.183.19.416.327.671.395h.277l.434-.118.672-.672-.158.158zM184.399-10.202l-.376.06-.395.395a1.365 1.365 0 00-.395 1.303 1.567 1.567 0 001.047 1.047l.296.02.396-.395a1.534 1.534 0 00.335-1.64 1.37 1.37 0 00-.908-.79zM173.317-46.094l.099.336.612.612.296.099a1.397 1.397 0 001.028-.474 1.37 1.37 0 00.316-1.106l-.968-.968-.395-.04a1.314 1.314 0 00-1.205 1.205v.237l.217.099zM187.144-32.583l-.316.316-.138.336-.118.119.019.217.415.415a1.508 1.508 0 001.146.514 1.425 1.425 0 001.383-1.383l-.04-.237-.178-.415-.493-.494a1.54 1.54 0 00-.731-.178l-.139.139-.375.177-.435.474zM192.379-15.021c.076-.158.136-.324.177-.494.148-.193.268-.406.356-.633.241-.05.458-.182.612-.375l.632.632.336.139c.138.056.286.083.435.079a1.284 1.284 0 001.244-1.008v-.197l-.039-.395-.731-.731-.356-.158-.237-.04-.118.119-.336.138-.395.395-.296-.296a1.373 1.373 0 00-1.205-.652l-.336-.099-.336.336a.93.93 0 00-.277.276l-.395.395c-.159.17-.281.372-.355.593l-.237.237a1.147 1.147 0 00-.494.77l.02.297.039.237.198.395.533.533.415.099h.237a3.33 3.33 0 00.553-.237c.14-.095.261-.215.356-.355zM150.106 1.374l-.375.138-.514.514a1.084 1.084 0 00-.197.987 1.275 1.275 0 00.731.731l.138.138.375.178a1.285 1.285 0 001.284-1.284 1.423 1.423 0 00-.316-.83 1.315 1.315 0 00-1.126-.572zM178.71-10.044l.256-.256a1.311 1.311 0 00.85-1.442l-.415-.415a1.563 1.563 0 00-.948-.514l-.218.02-.256.257a1.619 1.619 0 00-.316 1.698 1.42 1.42 0 001.284.85l-.237-.198zM179.361-19.94l-.296-.297-.356-.158a1.394 1.394 0 00-1.422 1.423c.008.395.171.77.454 1.047.266.254.62.395.988.395l.375-.178.158-.158a1.318 1.318 0 00.566-1.61 1.319 1.319 0 00-.348-.504l-.119.04zM176.793-14.646l-.889-.89c-.245.011-.484.086-.691.218l-.533.533a1.397 1.397 0 00.039 1.383 1.26 1.26 0 001.126.612 1.23 1.23 0 00.889-.375l.277-.276a1.398 1.398 0 00.177-1.205h-.395zM173.356-9.036l.138-.534-.02-.296-.118-.316-.593-.593-.296-.098h-.198a1.369 1.369 0 00-1.382 1.382 1.42 1.42 0 00.019.929 1.428 1.428 0 00.791.79l.098.099h.316l.672-.672c.15-.2.257-.429.316-.672l.257-.02zM183.806 16.228a1.366 1.366 0 00-1.126.652 1.34 1.34 0 000 1.304l.612.612.257.06h.237l1.008-1.008a1.35 1.35 0 00-.06-1.284 1.369 1.369 0 00-.928-.336zM156.704-46.074l.395-.08.514-.513c.143-.18.244-.389.296-.612l-.02-.218-.079-.395-.513-.513a1.15 1.15 0 00-.593-.277l-.217.02-.395.08-.494.493a1.176 1.176 0 00-.277.593l-.118.118.099.375.513.514c.182.123.382.216.593.276l.296.139zM171.657-40.721l.395.395c.174.12.368.206.573.257h.277l.474-.158.533-.534.138-.375.119-.118a1.344 1.344 0 00-1.324-1.324l-.316.04-.75.75-.099.296-.099.1.119.118-.04.553zM162.413-26.498l-.099-.1-.119.12-.217.02-.751.75-.118.355a1.333 1.333 0 001.363 1.363 1.315 1.315 0 001.144-1.83 1.314 1.314 0 00-.552-.62l-.651-.058zM171.578-25.748l.455.454.335.139.119.118h.276a1.341 1.341 0 001.087-.573 1.315 1.315 0 00.178-1.244l-.139-.138-.138-.336-.474-.474-.336-.139-.217-.02h-.237l-.119.12-.355.157-.455.454-.158.356-.118.119.02.256v.237l.118.119.158.395zM203.362-32.681l.118.118a1.396 1.396 0 001.284.77 1.452 1.452 0 00.889-2.548l-.138-.138-.375-.138-.217-.02-.257.02-.237.237a1.395 1.395 0 00-.81 1.363l-.257.336zM149.435-62.154l-.02-.256-.178-.455-.316-.316a1.402 1.402 0 00-1.106-.553l-.257.02-.81.81a1.337 1.337 0 00-.138 1.047l.928.928c.198.065.405.099.613.099a1.37 1.37 0 001.284-1.324zM250.612-42.44a1.313 1.313 0 001.186-.869l.019-.296-.098-.375a1.432 1.432 0 00-.395-.83 1.29 1.29 0 00-.949-.356l-.335.336a1.254 1.254 0 00-.652 1.482 1.51 1.51 0 001.224.908zM184.319 38.55a1.396 1.396 0 00-1.343-1.343l-.494.099-.434.434a1.36 1.36 0 00-.237 1.304l.731.73.217-.217-.099.336c.231.104.479.164.731.178l.909-.909-.02-.375.039-.237zM163.44-67.586l.751.75a1.453 1.453 0 001.363-1.362 1.26 1.26 0 00-1.106-1.344l-.099-.098-.297.059-.276.276a1.37 1.37 0 00-.573 1.72h.237zM180.27-32.78a1.312 1.312 0 00-1.107.711l-.039.237-.02.454v.198l.138.138.158.356.494.494.356.118a1.316 1.316 0 001.304-1.027l.098-.099-.039-.355-.474-.474a1.423 1.423 0 00-.869-.751zM201.604-33.057l-.099.099a1.62 1.62 0 00-1.264.593l-.198.197c-.311.236-.534.57-.632.948a1.901 1.901 0 00.632 1.936l.139.139.375.098.217.02.672-.672.138-.138a1.48 1.48 0 00.81-.81l.415-.415a1.32 1.32 0 00-.316-1.106l-.889-.889zM230.207 10.717l-.632-.632-.316-.118-.099-.1h-.237a1.395 1.395 0 00-.928.534 1.219 1.219 0 00-.178 1.047l.099.099a1.312 1.312 0 00.967.968l.218-.02.336-.099.671-.671a1.341 1.341 0 00.138-1.047l-.039.04zM214.246-2.28l.474-.475a1.477 1.477 0 00-.395-1.185 1.364 1.364 0 00-1.047-.415l-.81.81a1.318 1.318 0 00-.237 1.067l.297.296-.08-.079.158.158c.23.106.479.167.731.178l.297-.06.316-.315.296.02zM195.164-12.908l-.158.158-.415.217-.553.553-.04.277.04.513.415.415c.091.155.22.284.375.376l.316.079.573-.02.296-.099.218-.217c.225-.293.337-.658.316-1.027l-.889-.89a1.517 1.517 0 00-.494-.335zM195.381-2.616l.119.118a1.373 1.373 0 001.027-.474 1.487 1.487 0 00.336-1.205l-.672-.671-.039-.237-.534-.534-.415-.178-.237-.04h-.237l-.908.91-.119.395a1.286 1.286 0 00.514 1.027c.036.312.175.604.395.83l.336.335.434-.276zM176.793-31.99l.099-.099.079-.316a1.42 1.42 0 00-1.442-1.442l-.217.02-.356.158-.494.494-.138.415-.059.256.02.218.059.257.573.572c.241.177.531.274.829.277.202-.012.402-.052.593-.119l.454-.454v-.237zM197.139-26.222l-.257-.02a1.4 1.4 0 00-1.303 1.028l-.04.276-.118.119.059.256.553.554c.289.203.634.314.988.316l.276-.04 1.008-1.007a1.846 1.846 0 00-.316-1.304l-.376-.375-.474.197zM199.134-15.733h-.276l-.534.178-.158.158a1.832 1.832 0 00-.375.375 1.51 1.51 0 00-.514.237l-.454.455-.138.335a1.757 1.757 0 00.434 1.383l.336.336a1.429 1.429 0 00.948.553l.297-.06.454-.453c.29-.162.53-.402.691-.692l.02-.296.178-.178.039-.316a1.47 1.47 0 00-.017-1.222 1.477 1.477 0 00-.931-.793zM197.969-21.086l-1.007 1.008a1.374 1.374 0 00.252 1.081 1.38 1.38 0 00.972.538l.218-.02.237-.237a1.396 1.396 0 00.296-1.56 1.314 1.314 0 00-1.284-.81h.316zM192.596-28.355l.79.79h.237a1.428 1.428 0 001.324-1.324l-.099-.098-.079-.356-.672-.672a1.367 1.367 0 00-.731-.177l-.296.02-.612.612-.099.296-.099.099v.237l.119.118.217.455zM212.369 21.009a1.253 1.253 0 00-1.007.494 1.422 1.422 0 00-.296 1.323l.474.474a1.282 1.282 0 001.047.376 1.51 1.51 0 001.086-.494l.316-.316.059-.257a1.534 1.534 0 00-.395-1.225 1.446 1.446 0 00-1.284-.375zM201.426-50.5l-.493-.493h-.198a1.45 1.45 0 00-1.146.514 1.402 1.402 0 00-.296 1.126l.336.335a1.23 1.23 0 001.205.85c.177-.01.35-.05.513-.119l.415-.415a2.05 2.05 0 00.119-.671l-.494-.494.039-.632zM197.001-55.437l.79.79a1.76 1.76 0 001.6-1.008l.119-.118.039-.316-.572-.573a1.347 1.347 0 00-1.008-.376 1.479 1.479 0 00-.849.297l-.139.375-.118.119v.237l.118.118.02.455zM202.749-23.772l-.355-.04a1.316 1.316 0 00-1.067.514 1.362 1.362 0 00-.296 1.205l1.007 1.007.257.02a1.484 1.484 0 001.106-.514c.252-.348.359-.78.297-1.205l-.929-.928-.02-.06zM210.493-54.43l-.138.138a1.113 1.113 0 00-.613 1.205l.771.77.375.139a1.342 1.342 0 001.106-.593c.139-.247.214-.526.218-.81a1.354 1.354 0 00-.393-.95 1.344 1.344 0 00-.951-.393l-.375.494zM226.493-75.961l.553.553c.147.047.301.067.454.059a1.423 1.423 0 001.284-.889 1.508 1.508 0 00-.395-1.699l-.454-.454-.296-.02a1.37 1.37 0 00-1.166.652 1.51 1.51 0 00-.02 1.521l.04.277zM200.419-9.984l.889.889a.67.67 0 00.336.02 1.338 1.338 0 001.106-.593 1.597 1.597 0 00.118-1.502l-.711-.71-.336-.06-.256.02-.455.178-.197.197a1.502 1.502 0 00-.494 1.56zM250.829-33.53l-.375.177-.197.198a.971.971 0 00-.395.395.97.97 0 00-.178.177l.118.435.158.395.494.494.375.138a1.37 1.37 0 00.909-.395 1.396 1.396 0 00-.988-2.291l.079.276zM225.98-50.48a1.563 1.563 0 00-1.462 1.265v.237l.079.434.553.554.396.197.237.04h.316l.908-.909.079-.356-.177-.177.256.02.04-.475-.04-.276-.928-.929-.257.376zM198.601-37.165l.218.02h.237l.118-.12.356-.157.454-.455.138-.335.04-.237-.02-.218-1.165-1.165a1.342 1.342 0 00-1.047.454 1.359 1.359 0 00-.316 1.106l.81.81.177.297zM209.92-65.196a1.337 1.337 0 00-1.087 1.087l.04.276.198.474a1.23 1.23 0 001.185.751c.17.005.339-.029.494-.099l.73-.73.06-.297a1.286 1.286 0 00-1.027-1.383h-.237l-.356-.079zM152.437-10.557l-.671-.672-.218-.02h-.316l-.79.79-.138.336-.119.119a1.426 1.426 0 00.81 1.402l.257.06.533-.02.79-.79c.133-.241.188-.517.158-.79l-.138-.139-.158-.276zM218.73 10.263l-.296-.06-.553.08-.613.612-.138.375-.138.139a1.4 1.4 0 00.336 1.126 1.196 1.196 0 00.968.414l.375-.375a1.343 1.343 0 00.395-1.225 1.464 1.464 0 00-.336-1.086zM202.315 6.332l-.474-.474-.079-.316a1.452 1.452 0 00-1.363-1.087l-.119.119-.296.099-.751.75a1.802 1.802 0 00.218 1.403c.121.238.314.432.553.553l.197.197a1.31 1.31 0 001.225.751c.115 0 .228-.02.336-.06l.968-.967v-.237l-.139-.138-.276-.593zM157.81-30.034h.316l.119-.119.316-.118.79-.79a1.396 1.396 0 00-.356-1.107 1.443 1.443 0 00-1.106-.434l-.217.02-.79.79-.119.316-.118.118v.237l.138.138.138.336.474.474.415.139zM182.66-79.892l.237-.04.415-.178.395-.395c.159-.18.268-.397.316-.632v-.197l-.04-.356-.81-.81a1.365 1.365 0 00-.493-.059 1.403 1.403 0 10-.119 2.805l.099-.138zM198.878-62.37a1.226 1.226 0 00.83.829h.237l.395-.08.573-.572a1.38 1.38 0 00.237-1.027l-.83-.83-.375-.099a1.285 1.285 0 00-1.245 1.245l.02.217.158.316zM223.55-63.694l.415.414c.182.123.382.217.593.277l.256-.02.455-.138.493-.494.139-.375.118-.119a1.317 1.317 0 00-.824-1.25 1.314 1.314 0 00-.519-.093l-.277.04-.75.75-.119.316-.098.099.118.119v.474zM180.962-23.318l.671-.672c.108-.213.159-.45.148-.689a1.388 1.388 0 00-.207-.674 1.29 1.29 0 00-1.146-.632 1.202 1.202 0 00-.968.494l-.256.257-.099.296-.099.099a1.285 1.285 0 001.383 1.383l.573.138zM180.981 10.085l.316-.316.158-.395c.093-.226.147-.467.158-.711l-.079-.316-.632-.632-.316-.119h-.356l-.533.178-.118-.119a1.567 1.567 0 00-.81 1.522l.81.81.355.118.119.118h.316l.355-.355.257.217zM176.24-58.46l.435.435c.156.141.347.237.553.276h.237l.395-.079.395-.395.297.1.217.019.119-.119.256-.02.81-.81a1.368 1.368 0 00-.178-1.244 1.284 1.284 0 00-1.086-.533c-.37.023-.72.177-.988.434l-.296-.296a1.452 1.452 0 00-1.323.968l-.02.217.059.455.118.592zM180.25-55.28a1.477 1.477 0 00.395 1.146 1.456 1.456 0 001.086.494l.257-.06.474-.473a1.556 1.556 0 00.277-1.264l-.79-.79-.316-.12h-.395l-1.047 1.048.059.02zM211.935-36.197a1.454 1.454 0 00-1.442 1.442 1.293 1.293 0 00.395.908 1.256 1.256 0 00.968.376l.415-.218.494-.494.138-.375-.099-.099v-.197l-.177-.376-.494-.493-.198-.474zM214.029 2.638l-.099-.336-.652-.652-.296-.098-.257.02-.118.118-.336.178-.494.493-.099.415-.019.218a1.283 1.283 0 001.264 1.264 1.394 1.394 0 001.126-.77l.158-.514-.178-.336zM195.48-43.921l.218-.02.355-.355a1.422 1.422 0 00.299-1.57 1.42 1.42 0 00-1.346-.86l-.434.079-.573.573a1.45 1.45 0 00.079 1.343 1.534 1.534 0 001.402.81zM190.325-58.717l.809.81c.092.01.185.01.277 0a1.536 1.536 0 001.521-1.52l-.02-.257-.118-.396-.514-.513-.336-.138-.098-.1-.316-.039-.909.909a3.058 3.058 0 00-.217 1.008l.098.098-.177.138zM194.75-2.38l-.87-.869a.944.944 0 00-.335-.02 1.37 1.37 0 00-1.344.988l-.098.1-.02.454.039.276.672.672c.22.13.475.192.731.177a.978.978 0 00.316-.039l.533-.533a1.693 1.693 0 00.257-1.284l.119.079zM193.209-41.65l-.119-.118-.098-.336-.356-.355a1.314 1.314 0 00-.652-.376l-.257-.02-.434.119-.514.514-.178.414-.138.139v.237l.138.138.178.375.494.494.395.119.099.098a1.37 1.37 0 001.442-1.442zM167.351-51.309l-.197-.197-.099-.376a1.286 1.286 0 00-1.067-.71l-.118.118-.336.098-1.027 1.028c-.009.334.095.66.296.928l.099.099a1.284 1.284 0 001.284 1.047l.355-.356a1.594 1.594 0 00.81-1.679zM187.44-69.146l.395.395c.156.14.347.237.553.276h.238l.395-.079.573-.573a1.227 1.227 0 00.236-1.066l-.79-.79-.316-.08-.098-.098a1.452 1.452 0 00-1.344 1.027l-.019.217.059.415.118.356zM133.731-10.459l-.257.02-.909.909a1.417 1.417 0 00.321 1.183 1.43 1.43 0 001.121.496l.277-.04.849-.849a1.365 1.365 0 00-.356-1.185 1.42 1.42 0 00-1.046-.534zM215.55 8.465l-.118-.118a1.404 1.404 0 00-.988-.75l-.217.019-.395.118-.534.534c-.2.302-.284.667-.237 1.027l.85.85.375.098a1.286 1.286 0 001.205-1.205l-.099-.099.158-.474zM174.877-2.774l-.296-.02-.494.138-.573.573-.138.375-.138.139a1.284 1.284 0 001.323 1.323l.316-.04.731-.73.099-.297.119-.118a1.456 1.456 0 00-.949-1.343zM220.804-41.195a3.803 3.803 0 00-.336.336 1.257 1.257 0 00.83 2.252 1.2 1.2 0 001.086-.652l.138-.139.139-.138-.02-.217-.178-.178-.237-.434-.454-.455-.356-.04h-.197l-.415-.335zM218.158-22.666a.756.756 0 00-.415.059l-.692.691a1.539 1.539 0 00.538 1.81c.181.127.389.21.608.245l.217-.02.336-.336c.194-.212.325-.475.378-.757a1.543 1.543 0 00-.081-.843 1.342 1.342 0 00-.889-.85zM182.344 1.73l-.316.315c-.166.33-.208.709-.118 1.067l.928.928.316.04a1.394 1.394 0 001.363-1.363 1.34 1.34 0 00-1.323-1.324l-.336.1-.119.118-.395.118zM237.061 3.428l-.711-.71-.276-.04a1.37 1.37 0 00-.04 2.726l.138-.139.376-.177.513-.514c.147-.243.21-.528.178-.81l-.099-.099-.079-.237zM169.623-19.486l.533.533.395.158.218.02a1.45 1.45 0 001.284-.85l.059-.256-.02-.454-1.007-1.008h-.237a1.372 1.372 0 00-1.304 1.304l.02.257.059.296zM149.85-21.52l-.85.849v.316l.119.316.711.711.375.138a1.31 1.31 0 00.929-.414c.245-.263.379-.61.375-.968a1.425 1.425 0 00-1.402-1.403l-.257.455zM164.961-.206l-.869.869-.118.355-.139.139.02.217.454.454c.171.173.381.302.613.376l.118.118.257.06h.237l.257-.06.869-.87.04-.355-.02-.217a1.452 1.452 0 00-1.719-1.086zM184.893-40.326l-.099.296v.316l.118.119.139.336.592.592.336.139.119.118h.197a1.48 1.48 0 001.087-.494 1.37 1.37 0 00.316-1.106l-.85-.85a1.284 1.284 0 00-.731-.177l-.276.04-.632.632-.316.04zM170.946 7.952l-.276-.277-.257-.06h-.474l-.968.969-.138.454v.237c.074.298.216.575.415.81l.256-.257-.237.237.455.257.177.178.336-.02.296-.099.573-.573s.356-1.027-.079-1.698l-.079-.158zM141.256-39.477a1.313 1.313 0 00-.731-.217 1.256 1.256 0 00-1.205 1.205l.257.257a1.259 1.259 0 001.087 1.086l.217-.02.375-.098.415-.415c.193-.187.324-.428.375-.691l.02-.297-.81-.81zM166.284-16.464l-.691-.691-.336-.099-.118-.118a1.51 1.51 0 00-1.462 1.461l.118.119.158.395.731.73.217.02.099.1.237-.04.514-.514a1.627 1.627 0 00.296-1.284l.237-.079zM162.749-34.578l.474.474.355.119.119.118h.237l.118-.118.356-.158.454-.454.158-.356.119-.119-.119-.118-.138-.336-.435-.434a1.396 1.396 0 00-.889-.257l-1.066 1.066.02.257.118.119.119.197zM231.728-34.815l.217-.02.376-.098.849-.85a1.386 1.386 0 00-.687-1.446 1.4 1.4 0 00-.537-.173h-.198l-.375.098-.85.85a1.379 1.379 0 00.297 1.126 1.301 1.301 0 00.908.513zM266.02-32.128a1.43 1.43 0 00-1.106-.474l-.257.02-.751.75a1.654 1.654 0 00.138 1.521 1.4 1.4 0 001.107.553c.314-.004.617-.123.849-.336l.691-.691a1.337 1.337 0 00-.671-1.343zM235.639-17.55l-.276.04-.929.928a1.398 1.398 0 00.929 1.402l.375.099.257-.02.316-.079.79-.79a1.398 1.398 0 00-1.007-1.52l-.139.137.099-.098-.415-.1zM221.12-31.674l.376.138a1.314 1.314 0 001.086-.651c.136-.219.198-.475.178-.731a1.372 1.372 0 00-1.363-1.363l-.356.158-.217.217a.868.868 0 00-.375.375l-.138.139a1.031 1.031 0 00-.04.553l.83.83.019.335zM222.286-10.4l-.178-.177a2.218 2.218 0 00-.415-.415l-.276-.079-.494.02-.257.059-.415.415-.237.434-.079.277v.316l.119.119.158.355.454.454.356.158.217.02a1.4 1.4 0 001.323-.968l.02-.257-.079-.474-.217-.256zM222.266-28.237l-.237-.237a1.673 1.673 0 00-.474.119l-.356-.119-.118-.118a1.456 1.456 0 00-1.087.533 1.373 1.373 0 00-.197 1.106l.75.75.297.1.118.118.119-.118.415.414a1.258 1.258 0 001.264-1.264 1.623 1.623 0 00-.257-.73l-.237-.554zM206.147-49.294a1.256 1.256 0 001.165.73 1.37 1.37 0 00.494-.098l.573-.573c.06-.204.071-.419.033-.628a1.362 1.362 0 00-.25-.577 1.54 1.54 0 00-1.186-.592 1.286 1.286 0 00-1.205 1.205l.04.276.336.257zM223.688-28.355l.336-.336a1.261 1.261 0 00.197-1.264 1.304 1.304 0 00-.908-.909h-.237l-.435.119-.908.908-.02.415.592.593.435.237.276.079.139-.139.533.297zM250.553-17.254l-.296.02-.336.336a1.4 1.4 0 00-.968.968 1.31 1.31 0 00.672 1.461l.256.06h.474l.119-.119.336-.138.671-.672.139-.375.118-.119a1.31 1.31 0 00-1.185-1.422zM235.343-9.906l-.533-.533a1.4 1.4 0 00-1.107-.553l-.375.06-.889.888.02.534.612.612.395.158.139.138.257.02a1.314 1.314 0 00.948-.435 1.256 1.256 0 00.533-.889zM232.597-49.63l.988.988a1.398 1.398 0 001.027-.474 1.537 1.537 0 00.316-1.225l-.632-.632-.296-.494-.119-.119a1.367 1.367 0 00-1.343 1.344l.02.217.039.395zM232.538-23.022l.297-.296a1.568 1.568 0 00-.376-1.205 1.592 1.592 0 00-1.066-.435l-.83.83a1.367 1.367 0 00-.237 1.067l.81.81c.218.116.464.17.711.158.099.013.2.006.296-.02l.711-.711-.316-.198zM243.955-24.484a1.394 1.394 0 00.198-1.224l-.771-.77-.276-.08-.099-.099a1.45 1.45 0 00-1.324.968l-.098.1v.276l.533.533c.266.254.62.396.988.395.286.063.585.028.849-.099zM209.92-40.701l.375.138a1.453 1.453 0 001.047-.494c.196-.24.301-.54.297-.85a.873.873 0 00.138-.375v-.197l-.099-.376-.533-.533a1.682 1.682 0 00-.553-.237l-.218-.02-.395.08-.533.533c-.146.162-.248.36-.296.573l.098.098-.138.139c-.05.4.057.802.297 1.126l.474.474.039-.08zM209.822-28l.533.534c.271.225.615.345.968.335a1.344 1.344 0 001.205-.77 1.505 1.505 0 00-.139-1.521l-.592-.593-.277-.04a1.452 1.452 0 00-1.382 1.74l-.316.315zM211.579-17.195a1.337 1.337 0 00.217-1.244l-.493-.494a1.345 1.345 0 00-.87-.316c-.224.01-.441.079-.632.197l-.77.77a1.362 1.362 0 00.289 1.147 1.37 1.37 0 001.074.494 1.37 1.37 0 001.185-.554zM221.634-.305l-.672-.672-.336-.098-.118-.119h-.316l-.83.83-.138.375-.119.118v.238l.948.948.395.079h.277A1.373 1.373 0 00221.889.36c.044-.18.051-.365.021-.548l-.276-.118zM234.671-42.183a1.364 1.364 0 00.401.982 1.37 1.37 0 00.982.4l.276-.078.554-.553a1.362 1.362 0 00.305-1.01 1.386 1.386 0 00-.503-.926l-.375-.375a2.053 2.053 0 00-.336-.02 1.37 1.37 0 00-1.304 1.58zM207.115-19.09a1.567 1.567 0 00.336-1.6 1.311 1.311 0 00-1.245-.85l-.474.474a1.337 1.337 0 00-.395 1.066 1.573 1.573 0 00.553 1.146l.119.119.355.118.099.099h.237l.356-.356.059-.217zM215.55-46.568l.593.593.296.098.118.119a1.482 1.482 0 001.146-.514 1.394 1.394 0 00.494-1.244l-.711-.711a1.37 1.37 0 00-.711-.158l-.297.02-.79.79a1.397 1.397 0 00-.138 1.046v-.04zM209.565-21.896l.395-.079.513-.513c.131-.188.231-.394.297-.613l-.02-.217-.099-.375-.514-.514a2.053 2.053 0 00-.592-.276l-.217.02-.376.059-.513.513a1.202 1.202 0 00-.297.612l-.098.1.079.394.572.573c.169.153.372.262.593.316h.277zM213.041-22.844l.494.494h.277a1.395 1.395 0 001.086-.494 1.59 1.59 0 00.296-1.245l-.77-.77a1.284 1.284 0 00-.612-.099 1.37 1.37 0 00-1.284.89l-.139.137.02.415.316.316.316.356zM208.715-24.72l.376-.1.533-.533a1.507 1.507 0 00.316-1.58 1.315 1.315 0 00-1.245-.85l-.098.1-.356.118-.83.83a1.154 1.154 0 00-.276.592l-.099.099.079.395.514.514c.164.15.371.247.592.276l.494.138zM207.925-33.886l.395-.08.513-.513c.151-.165.248-.371.277-.593l.02-.217-.099-.375-.534-.533a1.055 1.055 0 00-.533-.257l-.118-.119-.415.099-.514.513c-.148.162-.25.36-.296.573l.02.218.079.395.533.533c.16.135.349.23.553.277l.119.079zM215.57-14.073l-.652-.652-.336-.139-.099-.098-.415.02-.335.335a1.563 1.563 0 00-1.6 1.6l.098.099.099.296.731.731.494.06a1.65 1.65 0 001.067-.435 1.368 1.368 0 001.244-1.56l-.296-.257zM219.797-16.286l.355.356a1.286 1.286 0 001.027-.593 1.391 1.391 0 00.137-1.447 1.393 1.393 0 00-1.223-.785c-.36.02-.698.175-.948.434a1.43 1.43 0 00-.316 1.067l.138.139a1.004 1.004 0 00.83.83zM226.77-39.93h.316l.198-.04.355-.158.158-.119.178-.178.099-.177.158-.356.04-.198-.02-.177-.04-.198-.138-.336-.138-.177-.178-.178-.178-.099-.336-.138-.217-.02h-.316l-.198.04-.355.157-.158.12-.178.177-.099.178-.158.355-.039.198v.158l.059.217.158.316.099.178.217.178.158.118.356.119.197.04zM238.898-5.007l-.415-.415a1.422 1.422 0 00-.572-.256h-.198l-.415.098-.573.573a1.381 1.381 0 00-.237 1.067l.79.79.336.099.099.099h.237l.119-.119.355-.158.454-.454.139-.336.02-.217-.099-.1-.04-.67zM225.387 1.196l-.138.138-.336.139-.454.454-.119.355-.118.119a1.45 1.45 0 001.343 1.62l.098-.099.435-.079.77-.77a1.35 1.35 0 00-.197-1.304 1.317 1.317 0 00-1.284-.573zM244.726 16.308a1.395 1.395 0 00-1.067.513 1.406 1.406 0 00-.237 1.185l.791.79c.197.118.422.18.651.178a1.48 1.48 0 00.909-.434l.435-.435a1.432 1.432 0 00-.316-1.146 1.408 1.408 0 00-1.166-.651zM236.884-34.992l-.099.098-.277.08-.711.71-.039.435.019.217.119.395.573.573.395.158.138.139.277-.08.632-.632a1.45 1.45 0 00-.099-1.363 1.51 1.51 0 00-.928-.73zM245.121-4.79a.614.614 0 00-.316.04l-1.047 1.047.118.119.119.316.395.395c.161.148.359.25.573.296l.296-.02.454-.138.553-.553.139-.375.118-.119a1.31 1.31 0 00-1.402-1.007zM216.261-26.222l1.067-1.067.119-.355.118-.119V-28l-.138-.138-.158-.356-.474-.474-.336-.138-.099-.099a1.398 1.398 0 00-1.027.474l-.198.198-.731.73-.079.277-.197.198-.059.257v.474l.889.889.276.079a1.424 1.424 0 001.146-.632l-.119.039zM226.257-13.62l-.435-.434-.336-.138a1.368 1.368 0 00-1.027.474 1.567 1.567 0 00-.336 1.087l1.126 1.126a1.37 1.37 0 00.85-.455c.244-.307.364-.695.335-1.086l-.118-.119-.059-.454zM241.269-63.28l.138-.138.079-.276a1.314 1.314 0 00-1.403-1.403l-.276.04-1.047 1.047.02.414.059.257.751.75.276.08a1.2 1.2 0 001.403-.77zM160.674-2.32l.514-.514a1.342 1.342 0 00.039-1.303 1.282 1.282 0 00-.513-.514l-.138-.138-.376-.178-.217-.02h-.237a1.593 1.593 0 00-1.245.534l-.296.296-.039.237c-.03.316.039.633.197.909l.968.968c.099.014.2.008.296-.02l.139-.138.908-.119zM163.361 22.253l-.633-.632-.296-.099a1.397 1.397 0 00-1.343.988v.277l.514.513c.27.247.621.388.987.395a1.343 1.343 0 001.259-1.15 1.334 1.334 0 00-.054-.608l-.434.316zM188.902-21.5l.277.078h.513l.277-.079.415-.415.237-.434.178-.178c.051-.378.164-.745.335-1.086l.139-.139-.04-.434-.059-.257-.711-.711-.257-.06a1.285 1.285 0 00-1.086.573l-1.205 1.205-.02.336-.119.119a1.323 1.323 0 00.256.974 1.308 1.308 0 00.87.507zM188.251-26.656l.494-.494a1.51 1.51 0 00-.435-1.66l-.217-.217-.573-.06-.257.06h-.237l-.415.099-.533.533-.198.395-.039.237v.317l.138.138.197.395.692.691.573.06.276-.04.277-.08.118-.118.139-.256zM188.073-15.061l-.632-.632-.435-.158h-.237a1.228 1.228 0 00-.988.553 1.312 1.312 0 00-.197 1.066l.81.81.375.1a1.344 1.344 0 001.205-1.206l-.138-.138.237-.395zM189.969 5.364l-.355.158-.435.435a1.115 1.115 0 00-.198 1.027 1.29 1.29 0 00.731.73l.139.139.355.158a1.286 1.286 0 00.909-.395c.214-.256.339-.575.355-.909a1.452 1.452 0 00-1.501-1.343zM248.736-37.047l-.139-.375a1.243 1.243 0 00-.888-.415 1.456 1.456 0 00-1.028.395 1.315 1.315 0 00-.098 1.877l.256.256.711.04.376-.138.355-.356.178-.375v-.237l.277-.672z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M207.629-4.118l-.257-.256-.711-.04-.376.138-.098.1c-.132.09-.246.203-.336.335l-.158.355-.04.712.139.375a1.312 1.312 0 00.928.415c.35.007.689-.12.948-.356a1.398 1.398 0 00-.039-1.778zM248.736-48.899l-.395.395-.02.217a1.291 1.291 0 00.297 1.245l.395.395.296.06.217-.02 1.146-1.146.02-.218-.02-.217-.079-.316-.296-.296a1.372 1.372 0 00-1.837-.218l.276.12zM237.97-24.345l-.217.217-.079.553.059.296c.085.124.192.23.316.316a1.201 1.201 0 00.988.435 1.534 1.534 0 001.126-.612l.395-.395v-.198l-.04-.276-.276-.277a1.342 1.342 0 00-1.877-.06h-.395zM257.704-42.42l-.02.257.02.454.494.494a1.284 1.284 0 001.106.435 1.542 1.542 0 001.067-.83l.118-.118.04-.316v-.198l-.04-.237-.296-.296a1.51 1.51 0 00-1.521-.376l-.652.652-.316.08zM229.278-15.14l-.177-.178a1.343 1.343 0 00-.336-1.047 1.817 1.817 0 00-1.245-.494l-.118.119-.316.118-.79.79a1.326 1.326 0 00.335 1.047c-.058.325-.002.66.158.949l.593.592.277.04.217.02.395-.08.158-.158a1.595 1.595 0 00.672-1.303l.177-.415zM214.345-12.335l-.573-.573a1.397 1.397 0 00-1.501-.356 1.509 1.509 0 00-.929 1.442l.573.573a1.4 1.4 0 001.521.336 1.532 1.532 0 00.909-1.422zM280.638-42.44l.395.396a1.227 1.227 0 001.145.04l.613-.613c.116-.275.17-.572.158-.87l-.119-.118-.118-.316-.474-.474-.336-.138-.119-.119a1.589 1.589 0 00-1.402.81 1.278 1.278 0 00.257 1.402zM221.97-35.486l.098-.336.099-.099v-.237l-.434-.435a1.425 1.425 0 00-1.64-.335l-.612.612-.099.375-.02.257c.023.2.084.395.178.573l.593.593.355.079h.237a1.405 1.405 0 00.929-.534l.316-.513zM232.795-45.521a1.316 1.316 0 00-1.047-.415 1.42 1.42 0 00-1.067.593l-.118.118-.079.316-.119.119.02.217.711.711.336.138.138.139h.237l.217-.02.336-.138.474-.474.158-.356.04-.237v-.277l-.119-.118-.118-.316zM235.856-18.913l-.533-.534-.277-.039a1.258 1.258 0 00-.79.158l-.711.711-.02.218.04.355c.06.25.183.481.355.672l.514.513h.198l.395-.395a1.452 1.452 0 00.948-.632 1.283 1.283 0 00-.119-1.027zM287.552-63.971l-.06.454v.237c.035.282.159.546.356.751.099.136.219.256.356.356l.237.237a1.287 1.287 0 001.323-.85l.119-.118.059-.257.099-.099-.04-.355-.375-.376a1.451 1.451 0 00-1.64-.336l-.573.573.139-.217zM225.644-41.531l.849-.85.02-.256-.079-.435-.415-.415a1.396 1.396 0 00-1.837-.217l-.573.573a1.327 1.327 0 00.356 1.027 1.397 1.397 0 001.679.573zM248.598-62.667a1.37 1.37 0 00.711-.435l.138-.138.139-.336.019-.217v-.237l-.138-.138-.138-.336-.139-.138a1.503 1.503 0 00-1.501-.396l-.671.672-.099.375-.119.119c-.025.26.03.522.158.75l.672.672.296.02.277-.04.099-.098.296-.1zM224.578-39.16a.983.983 0 00-.277-.277l-.257-.257a1.372 1.372 0 00-1.244.573 1.566 1.566 0 00-.138 1.481l.79.79a1.506 1.506 0 001.442-.849 1.311 1.311 0 00-.316-1.462zM244.015-71.359l-.217.218-.079.316.059.573.197.197a1.376 1.376 0 001.028.474c.398-.03.77-.206 1.047-.494l.316-.316.019-.217-.039-.277-.375-.375a1.315 1.315 0 00-1.699-.197l-.356.355.099-.257zM243.047-51.941h-.237l-.138.138-.316.119-.85.85a1.48 1.48 0 00.376 1.204 1.423 1.423 0 001.995.138l.592-.592-.019-.296-.119-.119a1.701 1.701 0 00-.375-.691l-.138-.139-.396-.158-.375-.454zM233.308-36.355c.174-.03.341-.09.494-.178a2.37 2.37 0 00.691-.139c.215.129.461.197.711.198l-.296.296-.02.257.02.217.553.553a1.314 1.314 0 001.58.238l.81-.81.02-.257-.02-.218-.118-.395-.395-.395-.356-.158-.118-.118.296-.296a1.338 1.338 0 00-.277-1.344l-.967-.968-.376-.02h-.395l-.118.12a1.315 1.315 0 00-.692.137l-.336.1-.118.118a1.256 1.256 0 00-.928.138l-.534.533-.178.415-.039.237.039.277.139.138.197.395-.454-.454c.165.118.354.199.553.237a1.23 1.23 0 00.632 1.145zM193.09-57.215l-.375-.178h-.395l-.139-.138a1.171 1.171 0 00-.889.533 1.254 1.254 0 00-.158 1.027v.198l.139.375a1.246 1.246 0 00.872.408 1.255 1.255 0 00.905-.329c.22-.215.371-.49.435-.79a1.391 1.391 0 00-.395-1.106zM220.587-43.348l.336.138a1.285 1.285 0 001.639-.336l.395-.395c.041-.368-.05-.74-.257-1.047l-.572-.573a1.567 1.567 0 00-1.462.87 1.34 1.34 0 00.217 1.52l-.296-.177zM228.429-49.413l-.139-.138v-.197l-.138-.376a1.397 1.397 0 00-1.995-.059 1.368 1.368 0 00-.474 1.027 1.28 1.28 0 00.375 1.008l.376.177.098-.098a1.428 1.428 0 001.126-.178 1.283 1.283 0 00.613-1.047l.158-.119zM222.76-47.813l.079-.276.099-.099v-.197l-.099-.1-.099-.296-.217-.217c-.172-.17-.382-.3-.612-.375l-.218-.217h-.276a1.393 1.393 0 00-.988.987 1.28 1.28 0 00.277 1.264 1.165 1.165 0 00.888.376.786.786 0 00.376.02 1.336 1.336 0 001.027-.633l-.237-.237zM216.281-46.667l.494-.257.435-.434.079-.316-.06-.534-.336-.336a1.366 1.366 0 00-1.935-.197 1.175 1.175 0 00-.652.652c-.091.173-.14.365-.143.56-.004.196.038.39.123.566l.691.691.297.06c.246-.027.486-.093.711-.198l.296-.257zM204.192-22.015a1.423 1.423 0 00-1.245-.414 1.316 1.316 0 00-.908.908l-.099.1-.02.256 1.284 1.284h.277a1.316 1.316 0 00.908-.909 1.365 1.365 0 00-.197-1.225zM232.993-83.685l.434.434.316.08.099-.1c.229.002.455-.06.652-.177l.573-.573.059-.336a1.29 1.29 0 00-.198-.632l-.572-.573-.317-.079-.098-.099c-.216.01-.425.072-.613.178l-.572.573-.079.316v.198c.008.215.069.426.177.612l.139.178zM238.879-68.949l-.099.099-.039.276c.007.207.061.41.158.593l.533.533.277.08h.316l.118-.12.356-.157.158-.158a1.278 1.278 0 00.436-.912 1.282 1.282 0 00-.357-.945l-.336-.336-.316-.079h-.237l-.691.692-.277.434zM222.108-66.44l-.257-.257-.375-.138h-.672l-.257.257a1.352 1.352 0 00-.453.923 1.33 1.33 0 00.335.973 1.312 1.312 0 001.6.257 1.257 1.257 0 00.079-2.015zM227.619-59.072l-.118.118-.02.218v.197l.119.356.316.316a1.309 1.309 0 001.165.454 1.253 1.253 0 001.027-.711l.613-.612.019-.257-.118-.119-.138-.375-.455-.454-.335-.139-.139-.138h-.237l-.217.02-.336.138-.79.79-.356.198zM253.931-39.792l-.119.118a1.46 1.46 0 00.257 1.481 1.479 1.479 0 001.6.336 1.483 1.483 0 00.909-1.343l-.119-.118-.138-.376-.593-.592a1.424 1.424 0 00-1.56.296l-.217.217-.02-.02zM240.242-99.784l.494-.494.059-.257v-.197a1.421 1.421 0 00-.336-1.205l-.553-.553-.237.039a1.405 1.405 0 00-.889.612l-.356.356v.198l.277.276c.089.191.202.37.336.533a1.397 1.397 0 001.205.692zM292.49-10.913a1.337 1.337 0 001.422.316l.494-.494c.213-.226.339-.52.356-.83a1.17 1.17 0 00-.376-.928l-.375-.375a1.257 1.257 0 00-1.521.494 1.45 1.45 0 000 1.817zM187.895-6.942a1.369 1.369 0 00-.02-1.877l-.731-.73a1.287 1.287 0 00-1.126.69l-.197.198-.04.277.079.474.119-.119-.218.217c.059.241.174.465.336.652l.514.514h.395l.751-.75.138.454zM253.576-92.89l-.138.138.098.296.178.178a1.425 1.425 0 001.936.119 1.31 1.31 0 00.276-1.738l.376.375-.494-.494a1.314 1.314 0 00-1.64.691l-.592.435zM238.642-57.215a1.31 1.31 0 00-1.245-.376l-.809.81-.139.376-.039.237.019.256.534.534a1.367 1.367 0 001.659.316l.652-.652a1.392 1.392 0 00-.632-1.501zM252.943-40.82l-.099-.415-.276-.276a1.535 1.535 0 00-1.264-.553.758.758 0 00-.297-.02 1.593 1.593 0 00-1.126.138 1.9 1.9 0 00-1.007 1.758l.889.89.375.138.099.098.118-.118a1.427 1.427 0 001.146.04h.198l.178-.178a1.342 1.342 0 00.612-.968l.059-.297.395-.237zM239.155 8.999l.079-.316-.02-.218-.632-.632a1.226 1.226 0 00-.987-.355 1.172 1.172 0 00-.909.553l-.118.118a1.252 1.252 0 00.019 1.363l.672.672h.197l.336-.06c.35-.09.657-.3.869-.592l.494-.533zM238.326-11.585h.197l.296-.098.119-.119a1.458 1.458 0 00.632-1.066 1.405 1.405 0 00-.355-1.067l-.573-.573h-.237a1.401 1.401 0 00-.988.553l-.079.277-.099.098.099.1.039.236c.047.255.171.49.356.672l.316.316.296.099-.019.572zM233.644-32.899l-.178-.178-.474-.158-.257-.02-.296.06-.909.909-.079.316c-.03.176-.037.355-.02.533l.83.83.336.059.316-.04a1.73 1.73 0 00.987-.434l.159-.158.138-.375-.198-.159-.019-.256-.099-.1a1.707 1.707 0 00-.237-.829zM226.059-25.945l.296.296a1.454 1.454 0 001.047.454 1.507 1.507 0 001.126-.572l.119-.12.138-.335.099-.099v-.237l.316-.316.059-.257-.02-.256-.138-.415-.751-.75-.237-.04-.257.02-.138.138-.375.178-.119.118a1.399 1.399 0 00-.434 1.067c-.206.23-.33.522-.356.83l.099.098-.474.198zM235.738-59.29l.435-.434a1.484 1.484 0 00.079-2.054l.079.079-.395-.395h-.277l-.217.02-.415.177-.652.652-.079.356v.237c.029.306.154.596.356.83.114.16.255.3.414.414l.317.04.355.079zM244.903-40.247l-.237-.237a1.337 1.337 0 00-1.639-.296l-.672.671-.099.376-.019.217c.039.353.2.68.454.928l.612.613h.237l.317-.08.118-.118a1.706 1.706 0 00.731-1.086l.099-.099-.099-.099.197-.79zM238.385-31.793l-.691-.69a1.567 1.567 0 00-.514-.159 2.027 2.027 0 00-.514-.237l-.118-.119h-.237l-.218.02-.355.356a1.76 1.76 0 00-.731 1.244l.099.1.099.295c-.05.373.034.75.237 1.067l.671.672a1.9 1.9 0 00.968-.06l.593-.592a1.48 1.48 0 00.711-1.897zM241.21-36.632h-.198l-.099.099-.296.099-.119.118a1.436 1.436 0 00-.627.91 1.415 1.415 0 00.213 1.085l.474.474a1.45 1.45 0 001.363-.81 1.285 1.285 0 00-.237-1.501l-.474-.474zM243.442-45.106l-.118.118-.02.218.02.217.079.316.296.296a1.396 1.396 0 001.857.237l.691-.691-.059-.336.099-.099a1.706 1.706 0 00-.356-.671l-.336-.336-.316-.08-.099-.098h-.237l-.118.119-.316.118-.415.415-.652.257zM219.599 2.44a1.335 1.335 0 00-1.046-.454 1.597 1.597 0 00-1.205.652l-.297.296-.059.257c.006.388.154.76.415 1.047a1.503 1.503 0 001.106.474l.257-.06.296-.295a1.486 1.486 0 00.652-1.087 1.308 1.308 0 00-.119-.83zM265.941-53.146l-.118-.118-.099-.336-.277-.277a1.59 1.59 0 00-1.145-.513 1.48 1.48 0 00-1.047.533l-.178.178-.02.217a1.26 1.26 0 00.178 1.482l.474.474.356.04.375-.376c.209-.077.403-.19.573-.336l.494-.494.434-.474zM266.711-59.684l-.118.118-.02.257.119.355.256.257a1.734 1.734 0 001.798.534l.711-.712.099-.296.099-.099a1.398 1.398 0 00-.395-1.027 1.456 1.456 0 00-.771-.454l-.118-.119h-.277l-.118.119-.376.138-.454.454-.435.475zM246.78-34.42l-.178-.177a1.46 1.46 0 00-1.106-.474 1.394 1.394 0 00-1.106.592l-.119.119-.098.296-.099.099-.02.217.099.099.079.316.197.197a1.428 1.428 0 002.213-.118l.415-.415.019-.217-.296-.296v-.238zM274.889-48.9l-.197.001a1.146 1.146 0 00-1.304.316l-.454.454v.277l-.02.217.158.395a1.431 1.431 0 001.185.395c.287-.05.554-.18.771-.375a1.308 1.308 0 00.444-.931 1.3 1.3 0 00-.365-.965l-.218.217zM301.517-51.111l-.119.118.435.435a1.48 1.48 0 001.521.375 1.587 1.587 0 001.027-1.422l-.039-.237-.514-.514a1.312 1.312 0 00-1.225-.434 1.533 1.533 0 00-1.086 1.086v.593zM235.067-27.13l-.218.217-.098.296-.099.099.099.099.079.276.335.336a1.343 1.343 0 001.205.454 1.593 1.593 0 001.186-.908l.059-.257v-.514l-.573-.573-.435-.237-.256-.059a1.54 1.54 0 00-1.284.77zM285.931-4.868l-.355-.158h-.277a1.506 1.506 0 00-.592 0 .97.97 0 00-.178.177l-.435.435-.177.375v.316l-.02.297.138.375a1.23 1.23 0 00.889.415 1.254 1.254 0 00.948-.356 1.339 1.339 0 00.059-1.876zM282.179-34.696a1.537 1.537 0 00-1.916-.257l-.652.652-.04.276.02.257.119.435.533.533.395.158.119.119h.276l.375-.376-.079-.276.198.197.751-.75.019-.257-.059-.494-.178-.178.119-.04zM254.089-46.39l.83.83h.237l.119-.12.335-.137.81-.81.02-.257-.119-.119-.118-.355-.138-.138a1.404 1.404 0 00-1.613-.331 1.332 1.332 0 00-.461.35l-.494.494.02.257.098.099.474.237zM282.514-56.524a1.284 1.284 0 00-1.521-.099l-.197.198-.218.454a1.234 1.234 0 00.237 1.383c.088.146.21.268.356.355l.118.119.316.04.652-.652a1.284 1.284 0 00.336-1.68l-.158-.157.079.04zM203.658-63.358l.119-.12v-.236l-.119-.119-.138-.335-.849-.85-.257-.02-.099.1-.356.118-.118.118a1.428 1.428 0 00-.514 1.541l.652.652.316.079.297-.02.158-.158c.275-.047.525-.186.711-.395l.138-.138.059-.218zM231.886.11l-.81-.81-.316-.04-.276.04-.376.138-.098-.099a1.42 1.42 0 00-.632.988 1.362 1.362 0 00.335 1.007l.455.455a1.424 1.424 0 001.185-.514 1.482 1.482 0 00.533-1.166zM223.945-14.824l.02-.217v-.277l-.099-.099c.088-.069.168-.149.237-.237a1.54 1.54 0 00-.099-1.758l-.513-.514-.237-.039-.336.06-.119.118a1.816 1.816 0 00-.987.988 1.34 1.34 0 00.039.79l-.118.118a1.278 1.278 0 00.217 1.403l.355.355.297.099h.197l.771-.77.375-.02zM221.713-72.248l.85.85.098-.1.356-.078.138-.138a1.426 1.426 0 00.573-1.008 1.481 1.481 0 00-.395-1.106l-.296-.296-.316-.119-.119-.118h-.237l-.118.118-.336.138-.81.81-.02.257.119.119.118.355.395.316zM275.502-86.648l.178.178.434.197.158.158.297-.02c.242.01.483-.052.691-.177l.77-.77-.019-.336-.119-.119a1.417 1.417 0 00-.336-.336 1.368 1.368 0 00-1.936-.158 1.404 1.404 0 00-.118 1.383zM273.111-62.924a1.309 1.309 0 00.04 1.185l.592.593.356.079h.237a1.48 1.48 0 00.928-.533l.139-.139.099-.336.019-.217-.039-.237-.494-.494a1.342 1.342 0 00-1.738-.197l-.158.158.019.138zM290.475-45.284l-.119.118-.039.277c.008.216.069.426.178.612l.671.672h.316l.218-.02.395-.395a1.312 1.312 0 00.118-1.896l-.336-.336-.316-.08-.098-.098h-.237l-.692.692-.059.454zM232.004-50.46l.257.257h.237a1.393 1.393 0 00.988-.987 1.307 1.307 0 00-.277-1.265 1.336 1.336 0 00-1.027-.395l-.434.435a1.343 1.343 0 00-.475.948c-.021.376.106.745.356 1.027l.375-.02zM207.115-28.296l.257.02.099-.1.414-.137c.23-.082.443-.202.632-.356l.237-.237.119-.553-.02-.336-.118-.316-.237-.237-.494-.296a1.62 1.62 0 00-1.699.395l-.494.494v.237l.119.118.118.356.494.494.356.157.217.297zM255.234-77.245l-.079.355-.039.237c.008.216.069.426.178.613l.592.592.336.06 1.027 1.027.415.02.138-.139a1.203 1.203 0 00.87-.869 1.234 1.234 0 00-.336-1.165 1.284 1.284 0 00-.968-.455l-.336-.336a1.51 1.51 0 00-1.62-.355l-.099.099-.079.316zM255.511-72.149a1.574 1.574 0 00-.632 1.067c-.035.413.092.825.356 1.145l.474.475.237.04a1.734 1.734 0 001.146-.672l.118-.119.099-.336.118-.118-.019-.218-.119-.118-.118-.316-.909-.909h-.237l-.099.099-.355.118-.06-.138zM262.188-35.763a1.433 1.433 0 00-1.013-.437 1.421 1.421 0 00-1.022.417 1.293 1.293 0 00-.436.912 1.283 1.283 0 00.357.945l.277.277.691.02.375-.139.771-.77.02-.692-.02-.533zM234.473-8.483l.237-.237.08-.316-.08-.593-.533-.533-.375-.178-.217-.02h-.277l-.138.138-.376.178-.158.158c-.829.83-.217 1.6-.039 1.778a1.226 1.226 0 001.284.336l.474-.237.118-.474zM257.052-53.225l.573.573a1.479 1.479 0 001.402-.85 1.4 1.4 0 00-.276-1.54 1.096 1.096 0 00-.297-.297l-.237-.237-.237.04a1.587 1.587 0 00-.948.948 1.513 1.513 0 00.02 1.363zM264.736-66.875l-.276.277.099.099.079.355.316.316c.573.573 1.343.632 2.153.06l.434-.435.04-.277-.02-.217-.118-.355-.712-.712-.434-.04-.217.02a4.278 4.278 0 00-.889.495l-.158.157-.297.257zM225.466-26.222l.099-.296.217-.217-.059-.297-.336-.335a1.482 1.482 0 00-1.659-.356l-.81.81-.04.237v.434c.06.25.183.481.356.672l.395.395a1.648 1.648 0 001.145-.632l.692-.415zM253.852-53.423l.514-.513.059-.297c.025-.26-.03-.522-.158-.75l-.632-.632-.277-.04h-.237l-.415.139-.474.474-.177.375-.119.119v.276l.138.138.158.356-.059-.06a1.286 1.286 0 001.679.415zM230.069-23.318a1.303 1.303 0 00-.336-.968l-.375-.178-.376.02-.138-.138a1.254 1.254 0 00-.671.237l-.099-.1-.277-.039-.217.02-.395.395a1.541 1.541 0 00-.593 1.027l.099.1.059.335.534.533a1.335 1.335 0 001.501.316l.375-.375a1.344 1.344 0 001.087-.336 1.513 1.513 0 00-.178-.85zM243.976-79.181l.078-.277a.983.983 0 00.277-.276 1.261 1.261 0 00-.178-1.245l-.77-.77-.257-.02-.217.02-.356.118-.158.158a1.453 1.453 0 00-.474.83 1.29 1.29 0 00-.265.844c.015.304.137.593.344.815l.494.494a1.618 1.618 0 001.482-.691zM270.642-75.942l-.099.336-.019.217c.007.207.061.41.158.593l.612.612.375.06.218.02a1.344 1.344 0 00.948-.554l.138-.138.099-.336.099-.099-.02-.256-.336-.336a1.426 1.426 0 00-1.639-.336l-.455.454-.079-.237zM191.114-77.403l-.296-.297-.296-.099-.099-.098-.217-.02-.099.099-.277.079-.118.118a1.566 1.566 0 00-.691 1.047 1.37 1.37 0 00.395 1.146l.316.316.296.099h.198l.098-.099.316-.08.119-.118a1.48 1.48 0 00.632-1.066 1.374 1.374 0 00-.277-1.027zM231.116-3.466l.119-.118a1.23 1.23 0 00-.08-1.225l-.592-.593-.356-.04-.217-.019c-.356.053-.68.236-.909.514l-.138.138-.099.375-.019.218.019.256.553.554a1.317 1.317 0 001.719.217l-.099.099.099-.376zM212.528-41.333l-.81-.81h-.316l-.217.02-.395.157-.139.139a1.315 1.315 0 00-.099 1.877l.336.335.297.099.098.099.218-.02.434-.435a1.42 1.42 0 00.593-1.461zM271.788-32.405l-.474-.158a1.23 1.23 0 00-1.303.632 1.254 1.254 0 00.177 1.482 1.287 1.287 0 001.205.375l.711-.711.158-.474v-.277l-.711-.71.237-.159zM256.104-22.212l-.533-.533-.218.02a1.596 1.596 0 00-.948.948 1.539 1.539 0 00.257 1.323l.573.573a1.483 1.483 0 001.403-.85 1.368 1.368 0 00-.534-1.48zM214.167-32.82l-.099.1-.276.039a1.372 1.372 0 00-.889.612l-.277.277v.197l.079.277.573.573a1.402 1.402 0 002.454-.821 1.403 1.403 0 00-.34-1.017l-.494-.493-.731.257zM249.15 9.315l.099-.1.02-.216-.099-.1-.059-.335-.296-.296a1.344 1.344 0 00-1.916-.139 1.391 1.391 0 00-.452.918c-.03.351.075.701.294.979l.158.158.395.197.138.139h.276l.139-.139c.275-.072.527-.215.731-.415l.118-.118.454-.533zM221.614-56.386l-.138.139-.02.296.139.138.158.395.197.198a1.454 1.454 0 001.482.375l.849-.85v-.236l-.059-.455-.336-.335a1.37 1.37 0 00-1.837-.218l-.158.158-.277.395zM210.078-72.544l-.692-.06-.375.139-.691.691v.277l.138.375a1.314 1.314 0 00.948.435c.35.007.689-.12.948-.356a1.423 1.423 0 00.079-1.975l-.355.474zM204.113-47.14l-.158-.159-.356-.158-.118-.118-.257-.02-.929.928-.118.316c-.028.24.021.481.138.692l.652.651.454.139.257.02.316-.04.534-.533a1.508 1.508 0 00-.415-1.719zM247.373-58.756l-1.047 1.047-.02.257.099.098.099.336.276.277a1.254 1.254 0 001.067.474 1.397 1.397 0 001.047-.494l.415-.415v-.197l-.04-.237a1.423 1.423 0 00-.375-.652l-.336-.336-.316-.08-.099-.098h-.237l-.533.02zM201.9-37.225v-.237l-1.066-1.066-.297.02-.237.039-.415.217-.177.178c-.186.245-.29.542-.297.85l.317.078-.317.08.099.533.632.632.514.118.296-.02a2.124 2.124 0 001.185-1.184l-.237-.238zM217.782-90.125l.059-.296-.019-.336a1.403 1.403 0 00-.316-.71 1.286 1.286 0 00-1.699-.08l-.296.296a1.285 1.285 0 00-.099 1.521l.553.553.296.06c.265.035.534-.013.771-.138l.217-.218.533-.652zM217.229-56.9l.118-.355.099-.099-.02-.256-.494-.494a1.422 1.422 0 00-2.034-.06l-.158.159-.158.355-.139.138v.277l.119.118.118.356.593.593.355.079a1.707 1.707 0 001.166-.652l.435-.158zM228.39-71.537l-.139.139v.237l.119.118.138.336.514.514.355.158.119.118.257.02.118-.118.356-.119.75-.75.02-.297a1.564 1.564 0 00-.375-.85h-.751l-.75-.039-.316.316-.415.217zM274.258-20.02l.691.692.217-.02.356-.079.138-.138a1.397 1.397 0 00.415-1.995l-.533-.533-.257-.06-.099.099-.375.099-.138.138a1.449 1.449 0 00-.652.968c-.015.295.068.587.237.83zM294.959 7.517a1.38 1.38 0 00-.395-1.106l-.455-.454-.237.04a1.68 1.68 0 00-1.066 1.066 1.279 1.279 0 00.316 1.185c.208.235.497.383.81.415l.098.099.218-.02.375-.06.138-.138c.182-.31.252-.672.198-1.027zM263.532-6.133l-.277-.039-.513.079-.178.178a1.37 1.37 0 00-.435 1.62l.672.671.316.08.533-.02.296-.1a1.396 1.396 0 00.435-1.777l-.119.118.04-.04-.77-.77zM264.874-25.886l.138.375a1.458 1.458 0 001.225.395c.245-.07.469-.199.652-.375a1.342 1.342 0 00.158-1.896l-.276-.277h-.277a.893.893 0 00-.553 0l-.138.138a.945.945 0 00-.395.395l-.178.178-.356 1.067zM249.704-10.913l.039-.277c.049-.187.056-.382.02-.572l-.928-.929-.297-.02-.474.119-.553.553-.138.336-.119.118-.019.257.019.217.139.336.138.139a1.422 1.422 0 001.62.355l.513-.513.04-.119zM263.057-22.765l-.355-.356a1.586 1.586 0 00-.356-.355l-.375-.375a1.374 1.374 0 00-1.106-.475 1.425 1.425 0 00-.988.593l-.099.099-.099.336-.098.098.019.218.613.612.02.217.375.376a1.252 1.252 0 00.878.428 1.258 1.258 0 00.919-.33c.2-.17.327-.41.356-.671a.972.972 0 00.296-.415zM268.233-48.84a1.282 1.282 0 00.217 1.363l.494.494.138-.138a1.46 1.46 0 00.771-.968 1.401 1.401 0 00-.356-1.264 1.287 1.287 0 00-1.699-.198l-.197.198.632.513zM264.085-21.777l.237.04a1.31 1.31 0 001.086-.692 1.34 1.34 0 00.198-1.383l-.455-.454-.296-.06-.257.02-.355.119-.613.612-.118.316-.02.257.118.514.336.335.139.376zM273.566 5.72l-.336-.336-.277-.08a1.45 1.45 0 00-1.362-.019 1.31 1.31 0 00-.672 1.462l1.146 1.145.217.02h.316l.217-.02.356-.158.138-.138a1.302 1.302 0 00.528-.883 1.306 1.306 0 00-.271-.993zM257.961-.799l.454-.454a1.305 1.305 0 00-.335-1.166l-.376-.375-.316-.08-.099-.098-.217.02-.968.968-.039.355v.237l.118.396.336.335a1.234 1.234 0 00.948.435c.222-.133.395-.334.494-.573zM285.931-29.165l-.138.138-.039.237v.237l.079.395.256.257a1.465 1.465 0 001.047.454 1.43 1.43 0 001.107-.553l.118-.118.119-.316.098-.1v-.236l-.098-.099-.119-.355-.118-.12a1.4 1.4 0 00-1.917-.137l-.138.138-.257.178zM265.961-11.604l.296-.1a1.392 1.392 0 00.584-1.63 1.378 1.378 0 00-.307-.502l-.257-.257-.316-.119h-.277a1.34 1.34 0 00-.968.534l-.415.414.04.237c.071.247.192.475.356.672l.316.316.296.099h.197l.455.336zM274.613-4.019a1.458 1.458 0 001.066-.672l.119-.118.079-.316.099-.099-.04-.237-.513-.514a1.427 1.427 0 00-1.601-.375l-.651.652-.079.277-.099.098c.011.368.143.722.375 1.008a1.337 1.337 0 001.245.296zM264.223-40.247l.138.375a1.485 1.485 0 001.067.435c.313-.01.614-.128.849-.336a.87.87 0 00.336-.138l.573-.573.059-.336.02-.217a1.747 1.747 0 00-.178-.573l-.593-.593-.336-.059-.098-.099a1.062 1.062 0 00-.613.178l-.257.257a1.623 1.623 0 00-.632.988v.197l-.335.494zM254.622-31.91l-.237.236c.011.344.144.673.375.928a1.395 1.395 0 001.383.396 1.452 1.452 0 001.087-1.087l-.475-.474a1.365 1.365 0 00-1.106-.474 1.392 1.392 0 00-1.106.553l.079-.079zM247.708-23.535a1.398 1.398 0 001.087-.652l.296-.296.02-.257a1.479 1.479 0 00-.356-.87 1.695 1.695 0 00-.553-.355l-.118-.119-.237-.039-.336.06-.119.118a1.389 1.389 0 00-.691 1.007 1.514 1.514 0 00.375 1.126c.181.15.4.245.632.277zM241.704-4.73l.336-.336v-.237l-.119-.118-.118-.317-.87-.869-.256-.02-.218.02-.731.731-.158.395-.019.218.019.296.613.612a1.371 1.371 0 001.55.191 1.37 1.37 0 00.425-.349l-.454-.217zM281.705-22.686a1.395 1.395 0 00-.119 1.936l.415.415h.277a1.341 1.341 0 001.363-1.363v-.198l-.474-.474a1.402 1.402 0 00-1.462-.316zM246.129-28.04a1.449 1.449 0 001.402-.81 1.338 1.338 0 00-.158-1.5l-.494-.494a1.34 1.34 0 00-1.066.395 1.475 1.475 0 00-.474 1.185l1.007 1.008-.217.217zM272.361-39.93l-.099.295-.098.1.019.216.099.1.099.296.276.276a1.484 1.484 0 001.146.514 1.368 1.368 0 001.008-.534l.434-.434-.138-.139a1.432 1.432 0 00-.375-.651l-.771-.77h-.197l-.139.137a1.37 1.37 0 00-.869.593h-.395zM250-27.96l.415.415.316.079.217.02c.224-.01.442-.078.632-.198l.553-.553.079-.316.02-.218a1.309 1.309 0 00-.178-.612l-.75-.75-.316-.08-.099-.098a1.62 1.62 0 00-.672.197l-.553.553-.079.316c.006.229.067.452.178.652l.237.593zM252.903-26.084l.001.198.098.375.316.316a1.369 1.369 0 001.107.474 1.487 1.487 0 001.106-.592l.257-.257-.02-.257-.099-.098a1.284 1.284 0 00-.316-.554 1.367 1.367 0 00-1.521-.335l-.79.79-.139-.06zM251.442-30.587l.75.75a1.48 1.48 0 001.423-.83c.117-.237.162-.504.13-.767a1.362 1.362 0 00-.308-.714l-.494-.494-.356-.04-.335-.059-.099-.098a1.677 1.677 0 00-.652.177l-.573.573-.059.336v.198c.008.215.069.425.178.612l.395.356zM257.802-37.225l.415.415.336.06.099.098a1.37 1.37 0 00.632-.197l.553-.553.079-.356v-.197a1.364 1.364 0 00-.158-.593l-.593-.593-.316-.079h-.197a1.233 1.233 0 00-.613.178l-.572.573-.06.336-.019.217c.008.216.069.426.177.612l.237.08zM248.005-18.38l.119-.316.118-.118-.039-.238-.099-.098-.138-.336-.771-.77a1.531 1.531 0 00-1.727-.425 1.544 1.544 0 00-.525.346l-.434.434-.04.237.059.336.435.435a1.62 1.62 0 001.047.493 1.348 1.348 0 001.014.376 1.342 1.342 0 00.961-.494l.02.138zM252.469-16.68l.257.256a1.253 1.253 0 001.106.395 1.425 1.425 0 001.205-.928 1.339 1.339 0 00-.257-1.363 1.196 1.196 0 00-.928-.415 1.337 1.337 0 00-1.027.474l-.119.118a1.372 1.372 0 00-.237 1.462zM274.811-27.13l.217.217.178.138.316.158.197.04h.277l.178-.02.375-.138.178-.1.118-.118.119-.158.138-.336.059-.217.02-.257-.039-.197-.139-.336-.098-.178-.238-.237-.158-.118-.335-.178-.198-.04-.257-.02-.197.04-.356.158-.178.099-.138.099-.099.177-.177.336-.04.198.02.257v.197l.158.356.099.177zM256.795 4.989l.04-.277a1.769 1.769 0 00-.158-.593l-.593-.592-.375-.06-.257-.02a1.406 1.406 0 00-.929.534l-.118.119-.099.336-.099.098v.237l.119.119.118.316.81.81.257.02.119-.119.375-.138.79-.79zM243.067-.918l-.178-.177-.356-.158-.118-.119-.257-.02-.119.119-.355.118-.138.139a1.363 1.363 0 00-.63.94 1.377 1.377 0 00.294 1.094l.711.711.316-.04.138-.137a1.372 1.372 0 00.823-1.673 1.37 1.37 0 00-.349-.58l.218-.217zM244.548 23.557a1.412 1.412 0 00-1.066-.434 1.4 1.4 0 00-1.047.612l-.119.118-.099.336v.435l.119.118a1.312 1.312 0 001.244 1.008l.257-.02.119-.119a1.344 1.344 0 00.652-1.007 1.396 1.396 0 00-.06-1.047zM277.813-16.266l-.435-.435-.217-.02-.099-.098-.276.04-.099.098c-.117.08-.217.18-.297.296l-.73.731.019.218.139.414.77.77h.277a1.396 1.396 0 00.968-.967 1.46 1.46 0 00-.02-1.047zM260.628 9.887l-.336-.335-.316-.08-.099-.098-.217.02-.929.928-.039.277c.008.215.069.426.178.612l.533.533.257.06h.316l.118-.119.376-.138.138-.139a1.315 1.315 0 00.02-1.52zM257.565-25.926l.237.04.099.099.356-.08.098-.098.257.02.119-.119.355-.119.474-.474.139-.375.138-.138-.02-.257-.118-.119-.119-.316-.138-.138a1.317 1.317 0 00-1.047-.415l-.257.02-.197-.197h-.277l-.474.079-.178.177-.276.04-.692.691-.098.297-.04.276.099.494.197.198a1.313 1.313 0 001.205.454l.158-.04zM254.86-10.182v-.237l-.099-.099-.119-.355-.138-.139a1.312 1.312 0 00-1.007-.454 1.505 1.505 0 00-1.067.474l-.454.454-.02.257.099.099.138.336.138.138a1.351 1.351 0 001.027.474 1.62 1.62 0 001.047-.494l.455-.454zM301.913-31.713l.237-.04.178-.178a1.29 1.29 0 00.513-.948 1.28 1.28 0 00-.375-1.007l-.336-.336-.336-.099-.217.02-.909.909-.079.316-.019.257.098.493.514.514c.228.107.483.141.731.099zM202.71-51.783l.099.099h.237l.197-.198a1.282 1.282 0 00.909-.909c.072-.23.085-.474.039-.71l-.019-.218-.139-.375-.316-.316a1.7 1.7 0 00-1.244-.573l-.257.06-.277.276a1.399 1.399 0 00-.632.632l-.138.138-.02.257.87.869.691.968zM186.275-33.768l.099-.099-.04-.237-.415-.415a1.423 1.423 0 00-1.62-.355l-.553.553-.079.277c-.008.371.126.732.376 1.007a1.304 1.304 0 001.047.415 1.459 1.459 0 001.066-.672l.119-.474zM235.916-43.309l.731.731.296.099.296.02.494-.139.178-.177c.322-.208.676-.361 1.047-.455l.652-.652.079-.276.02-.257-.079-.474-.178-.178a1.402 1.402 0 00-1.126-.415l-.257-.02-.435.08-.098-.1-.553-.039-.297.1-.336.335a1.294 1.294 0 00-.609.815 1.297 1.297 0 00.175 1.002zM239.353-47.18l.098-.1.336-.019.119-.118a1.56 1.56 0 00.968-1.442l-.099-.297-1.363-1.363-.277-.04-.256.02-.396.119-.553.553-.177.415-.02.217.02.297.138.138.099.296 1.027 1.027.336.297zM230.563-39.654l.138-.139.02-.296-.02-.217-.178-.415-.158-.158a1.365 1.365 0 00-1.086-.375 1.32 1.32 0 00-.929.572l-.138.139-.079.355v.198l.04.237.474.474a1.286 1.286 0 001.679.217l.138-.138.099-.454zM216.498-24.681l-.376-.178-.375-.02-.118-.118a1.116 1.116 0 00-.909.513 1.305 1.305 0 00-.178 1.047v.198l.139.375c.236.241.552.389.888.415.335-.01.656-.136.909-.356a1.397 1.397 0 00.02-1.876zM286.841-33.748l.493.06c.311-.133.559-.381.692-.692a1.253 1.253 0 00.118-1.067 1.314 1.314 0 00-1.6-.968l-.296.297-.336.612-.039.395.77.77.198.593z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M275.996-85.285l.296-.296.336-.613.039-.395a1.45 1.45 0 00-.197-.434l-.277-.277-.612-.336-.395-.04a1.206 1.206 0 00-.692.692 1.23 1.23 0 00-.118.988 1.392 1.392 0 001.62.711zM297.428-28.415l-.533-.533a1.313 1.313 0 00-1.245-.296c-.13.047-.256.107-.375.178l-.692.691.001.316.098.099.079.316.79.79.356.079.217-.217a1.393 1.393 0 001.047-1.56l.257.137zM280.381-49.097l-.612-.612-.296-.099-.455.06a1.345 1.345 0 00-.77.77 1.536 1.536 0 00.02 1.284l.553.553.276.08.099.098c.128 0 .254-.02.376-.06a1.315 1.315 0 00.888-1.639l-.079-.435zM287.591-23.318l-.178-.178-.434-.158-.336-.02a2.098 2.098 0 00-.375.06 1.257 1.257 0 00-.83.83 1.375 1.375 0 00.237 1.303l.632.632h.277a1.394 1.394 0 001.106-1.106l.02-.257-.119-.434-.158-.158.158-.514zM276.134-61.008l.257-.02a1.261 1.261 0 001.087.178 1.714 1.714 0 00.928-.928l.118-.119.04-.355-.77-.77a1.43 1.43 0 00-1.087-.178 1.256 1.256 0 00-.928-.297l-.889.889-.099.415v.197a1.592 1.592 0 00.889 1.205l.454-.217zM280.302-75.606l.237.237h.237a.62.62 0 00.158-.158 1.315 1.315 0 001.086-1.086 1.479 1.479 0 00-.849-1.482l-.217-.217-.218-.02-.296.06a1.337 1.337 0 00-1.086 1.086 1.51 1.51 0 00.948 1.58zM277.3-2.834l-.395.395a1.17 1.17 0 00-.534.968l.119.119c.148.253.359.464.612.612l.119.119h.355l.771-.77a1.707 1.707 0 00-.099-1.64 1.258 1.258 0 00-.948.197zM297.645-58.38l.652.651h.316a1.428 1.428 0 001.047-1.323l-.019-.218-.889-.888a3.14 3.14 0 00-.613-.1l-.987.988a1.334 1.334 0 00.039 1.067l.454-.178zM301.676-44.08a1.343 1.343 0 00.809-1.955l-.651-.652-.336-.02-1.047 1.048-.04.355.02.217 1.027 1.028.356.04-.138-.06zM276.49-53.423l.73-.73a1.43 1.43 0 00.218-.81l-.85-.85-.336-.098a1.674 1.674 0 00-.77.02.84.84 0 00-.197.197l-.593.592-.04.277a1.514 1.514 0 00.573 1.047 1.34 1.34 0 001.265.355zM293.359 13.028l-.454-.454a1.427 1.427 0 00-.85-.02c-.165.04-.319.114-.454.218l-.375.375a1.28 1.28 0 00.178 1.56l.493.494.356.08.119.118c.114 0 .228-.02.335-.06a1.423 1.423 0 001.028-1.303v-.237l-.198-.553-.178-.218zM301.26-52.356l.02.257.118.434.554.554.434.158.217.02a.972.972 0 00.257-.06 1.338 1.338 0 001.027-1.54l-.572-.574a1.319 1.319 0 00-1.047-.138 1.43 1.43 0 00-1.008.889zM309.794-22.33c-.065.282-.044.578.06.849l.829.83.376.02.118-.12a1.478 1.478 0 001.087-1.086v-.237l-.139-.454-.592-.593a1.593 1.593 0 00-.751-.197l-.257.06-.829.829.098.098zM299.661-54.35c.121-.047.24-.1.355-.159l.395-.395a1.399 1.399 0 00.04-1.383 1.593 1.593 0 00-1.264-.79l-.257.06-.612.612a1.502 1.502 0 00.118 1.659 1.344 1.344 0 001.225.395zM319.612-22.508l-.454-.454-.554-.198-.296-.02a1.305 1.305 0 00-.79.79 1.36 1.36 0 00-.039 1.146l.869.87a.78.78 0 00.375-.06 1.286 1.286 0 00.948-1.422l-.158-.158.099-.494zM302.702-32.108l.297-.297.02-.375-.02-.218-.573-.572a1.37 1.37 0 00-1.245-.218 1.398 1.398 0 00-.948 1.7l.948.947c.255.054.519.047.771-.02l.573-.572.177-.375zM293.122-47.654a1.146 1.146 0 00-.04.513c-.102.22-.175.453-.217.691-.21.127-.387.304-.513.514l-.652-.652-.356-.04a1.622 1.622 0 00-.454.06 1.284 1.284 0 00-.929 1.284l.119.118.138.376.652.652.415.059.217-.02.77-.77.356.355a1.285 1.285 0 001.323.336c.116.033.236.046.356.04l.533-.534a1.93 1.93 0 00.277-.276l.158-.356.119-.118a1.68 1.68 0 00.177-.692l.079-.316.099-.099a1.156 1.156 0 00.277-.869l-.099-.296-.968-.968-.454.02-.158.158a1.99 1.99 0 00-.415.415 1.201 1.201 0 00-.81.415zM329.963-74.302l.711-.711a1.06 1.06 0 00-.04-1.027 1.28 1.28 0 00-.849-.613l-.138-.138-.395-.04a1.224 1.224 0 00-.927 1.082c-.016.168.004.338.057.499.088.287.26.542.494.73a1.398 1.398 0 001.087.218zM305.172-56.03l-.336.336a1.258 1.258 0 00-.454 1.6l.572.573c.304.216.678.308 1.047.256l.652-.651a1.592 1.592 0 00-.138-1.72 1.366 1.366 0 00-1.442-.493l.099.099zM307.048-46.292l.238.04.118.118.415.06a1.428 1.428 0 001.003-1.225c.018-.186 0-.374-.055-.553a1.537 1.537 0 00-.79-.79 1.258 1.258 0 00-1.067-.119l-.434.435a1.23 1.23 0 00-.356 1.066 1.341 1.341 0 00.672 1.027l.256-.059zM308.174-52.08l.573.573.316.04.099-.099c.233-.07.444-.199.612-.375l.593-.593a1.486 1.486 0 00-.415-1.323 1.313 1.313 0 00-1.245-.296 1.202 1.202 0 00-.691.69 2.3 2.3 0 00-.257.258 1.396 1.396 0 00.139 1.205l.276-.08zM310.071-58.36v.552l.099.297.494.493.513.198.297.02.138-.139a1.4 1.4 0 001.047-1.64 1.312 1.312 0 00-.257-.888 1.238 1.238 0 00-.988-.514l-.098.099-.277.08-.711.71a2.31 2.31 0 00-.138.731h-.119zM293.576-80.189a1.252 1.252 0 00.909-.908 1.317 1.317 0 00-.316-1.264l-.395-.395h-.277l-1.047 1.046a1.284 1.284 0 001.679 1.482l-.553.04zM335.652-26.696l-.376.178-.513.513a1.313 1.313 0 00-.138.692l.908.908c.198.09.415.131.632.119l.929-.928c.088-.2.128-.416.118-.633l-.928-.928a1.31 1.31 0 00-.612-.099l-.02.178zM319.769-28.158l-.316-.316a1.06 1.06 0 00-.612-.099l-.257.06-.889.889-.02.375.02.217a1.399 1.399 0 001.916.81l.731-.73.02-.336-.217-.218-.376-.652zM325.103-44.237l.119-.119.217.218.415-.415.296-.613.04-.395a1.331 1.331 0 00-.508-.87 1.318 1.318 0 00-1.844.252 1.317 1.317 0 00-.256.974 1.312 1.312 0 001.521.968zM316.056-42.657l-.435-.434-.375-.06h-.197a1.546 1.546 0 00-.257.06 1.278 1.278 0 00-.869.869 1.334 1.334 0 00.138 1.245l.849.849.395.04.099.098 1.008-1.008.019-.375.099-.099-.158-.158-.059-.256-.119-.119-.138-.652zM287.058-27.92l-.119-.12a1.42 1.42 0 00-1.422-.434 1.481 1.481 0 00-.257 2.706l.139.139.375.02.217-.02.435-.435a1.396 1.396 0 00.434-1.54l-.098-.1.296-.217zM346.674-13.027l.81.81a1.344 1.344 0 001.086.257.55.55 0 00.158-.158l.79-.79a1.562 1.562 0 00-.138-1.047l-.632-.632-.257-.06-.099-.098a1.957 1.957 0 00-.632.079 1.373 1.373 0 00-1.095 1.095 1.37 1.37 0 00.009.544zM243.817-6.488a1.317 1.317 0 00-1.027 1.027l.079.316.257.257c.131.289.363.52.652.652.306.154.66.182.987.079l.376-.376a1.26 1.26 0 00.256-1.6 1.477 1.477 0 00-1.58-.355zM287.433-101.621a1.371 1.371 0 001.699.829.947.947 0 00.454-.217l.435-.434a1.314 1.314 0 00-.059-1.324l-.554-.553-.454-.138-.099.098v-.316a1.894 1.894 0 00-.75 0l-.869.87.118.118.079.356v.711zM334.506-4.256l-.435-.434-.296-.06a1.427 1.427 0 00-.948 1.66 1.258 1.258 0 001.422 1.027l.316-.316-.059.059a1.313 1.313 0 00.099-1.798l-.099-.138zM309.418-33.65a1.362 1.362 0 00.929-.928v-.237l-.119-.474-1.066-1.066-.376-.02a1.312 1.312 0 00-1.027 1.343l.119.119.118.316.474.474a1.35 1.35 0 00.948.474zM288.322-28.276l.415.099h.237l.099-.1a1.618 1.618 0 001.086-.888l.218-.218c.24-.302.365-.68.355-1.066a1.926 1.926 0 00-1.106-1.699l-.118-.118-.376-.02-.968.968c-.303.247-.501.6-.553.987l-.237.237-.02.257a1.402 1.402 0 00.613 1.008l.197.197.158.356zM250.06-63.062l.612.612.336.02.237-.237a1.346 1.346 0 00.77-.77 1.274 1.274 0 00-.099-1.048l-.177-.177a1.283 1.283 0 00-1.245-.613l-.829.83-.079.316c-.077.354-.028.725.138 1.047l.336.02zM268.766-54.628l-.257.257-.02.297a1.4 1.4 0 00.672 1.066 1.372 1.372 0 001.106.158l.77-.77a1.432 1.432 0 00-.039-1.106l-.79-.79a1.79 1.79 0 00-.751 0 2.7 2.7 0 00-.197.197l-.455.454-.039.237zM289.981-49.096l.87-.87.019-.256-.019-.297-.178-.454-.435-.435a1.339 1.339 0 00-.375-.375h-.316l-.533.178-.455.454-.118.316a1.35 1.35 0 00-.06 1.047l.889.89c.249-.014.492-.081.711-.198zM287.176-59.013h-.198a1.366 1.366 0 00-.916.776 1.369 1.369 0 00.047 1.2l1.284 1.284.415.098.237-.04 1.067-1.066v-.395a1.346 1.346 0 00-.83-.83 1.256 1.256 0 00-.632-.632l-.119-.118-.355-.277zM312.619-35.27l-.099.1v.316l.099.099a1.48 1.48 0 001.797 1.007l.889-.889.059-.454-.019-.257-.889-.89a1.314 1.314 0 00-.889-.058c-.187.059-.361.153-.514.276l-.454.455.02.296zM291.443-35.743a.985.985 0 00.257-.06 1.373 1.373 0 00.987-1.343v-.276l-.869-.87a1.704 1.704 0 00-1.047-.019 1.071 1.071 0 00-.177.178l-.712.71-.059.297a1.679 1.679 0 00.652 1.166l.138.138.83.079zM286.841-45.363l.296-.099.533-.533c.154-.094.283-.222.376-.376.16-.1.295-.234.395-.395l.533-.533.059-.336-.118-.118a1.626 1.626 0 00-.79-1.225l-.099-.099-.316-.079a1.482 1.482 0 00-1.067-.316l-.79.79c-.208.259-.35.564-.415.89l.059.295-.039.277.039.316a1.536 1.536 0 001.344 1.54zM289.33-40.464l.711-.711.059-.297-.099-.099a1.365 1.365 0 00-1.015-1.235 1.373 1.373 0 00-.565-.029l-.553.553a1.421 1.421 0 00.079 1.58 1.344 1.344 0 001.462.475s-.158-.198-.079-.237zM296.361-34.815l-.415-.415-.335-.098-.099-.1c-.042.05-.089.097-.138.14a1.398 1.398 0 00-1.047 1.56l.79.79.316.118c.249.057.509.044.75-.04a.69.69 0 00.178-.177l.751-.75.02-.337-.376-.375-.395-.316zM264.756-77.561a1.316 1.316 0 00.79-.79 1.616 1.616 0 00-.039-1.383l-.494-.494a1.536 1.536 0 00-1.106-.119 1.591 1.591 0 00-.849.85l-.08.276-.098.1v.276a1.61 1.61 0 00.671 1.066 1.374 1.374 0 001.205.218zM293.596-11.031l.356.078.217-.217a1.368 1.368 0 00.968-1.955l-.336-.336a1.259 1.259 0 00-1.383-.514c-.17.051-.331.131-.474.237l-.632.632v.198a1.69 1.69 0 00.059.652l1.225 1.225zM298.949-7.495l-.434-.435-.336-.06h-.198l-.118.12a1.726 1.726 0 00-1.343 1.342l.039.317.711.71a1.482 1.482 0 001.818-.394l.553-.553.039-.396-.02-.217-.276-.276-.435-.158zM285.398-36.691a.969.969 0 00.257-.06 1.398 1.398 0 00.85-.849 1.521 1.521 0 00-.02-1.244l-.613-.613-.296-.06-.099-.098a1.14 1.14 0 00-.177.178 1.338 1.338 0 00-.83.83 1.507 1.507 0 00.02 1.244l.731.731.296.06-.119-.12zM285.656-5.066l.118-.118a1.169 1.169 0 00.296-1.324l-.908-.909-.395-.04a1.42 1.42 0 00-.889.89 1.623 1.623 0 000 .869 1.307 1.307 0 001.119.973c.175.02.352.005.52-.045l.139-.296zM275.64 19.804l-.376-.375a1.146 1.146 0 00-.454.059 1.34 1.34 0 00-1.086 1.086 1.504 1.504 0 00.81 1.561l.256.257.257.02.296-.06a1.423 1.423 0 00.949-.948 1.624 1.624 0 00-.356-1.501l-.296-.099zM284.253-50.36l-.692-.692-.296-.06-.316.04a1.477 1.477 0 00-.909.909 1.626 1.626 0 00.297 1.481l.197.198.435.197.257.06.335-.02.534-.534a1.536 1.536 0 00.158-1.58zM241.348-15.08l.474-.475c.118-.15.199-.326.237-.513l.02-.257-1.047-1.047-.395-.04c-.311.133-.558.38-.691.692a1.23 1.23 0 00-.119.987 1.374 1.374 0 001.521.652zM269.675-4.967a1.532 1.532 0 001.086-1.6l-.138-.138-.178-.415-.553-.553-.435-.08-.256.02-.297.1-.908.908v.355l.296.02-.296.06.118.474.593.592.474.158.257.02.237.079zM292.786-24.76l-.217.02-.988.987-.059.375.118.119.04.237.829.83.376.059.217-.02a1.393 1.393 0 00.909-.71 1.54 1.54 0 00.039-1.147l-.829-.83-.435.08zM288.934 5.206a1.31 1.31 0 00.77-1.324l-.553-.553a1.255 1.255 0 00-1.323-.414 1.357 1.357 0 00-.454.217l-.613.612v.316l.099.1a1.257 1.257 0 001.343 1.066l.139-.139.592.119zM330.653-62.114l.435.434.356.04.098-.099.317-.079.829-.83.059-.335v-.198a1.484 1.484 0 00-1.165-1.165l-.296.02-.494.138-.632.632-.02.257c-.065.27-.044.552.059.81l.119.118.335.257zM261.319-65.551l.296-.02.494-.217.632-.633.02-.414.118-.119a1.454 1.454 0 00-.592-1.027 1.23 1.23 0 00-1.047-.138l-.553.553a1.394 1.394 0 00-.079 1.304c.142.315.395.568.711.71zM278.88-65.255a1.42 1.42 0 001.6.691l.75-.75.08-.316a1.704 1.704 0 00-.554-1.304 1.447 1.447 0 00-.711-.316l-.217-.217a1.373 1.373 0 00-1.383-.435 1.01 1.01 0 00-.217.217l-.711.712-.02.335 1.383 1.383zM330.417-41.965l-.297.098-.81.81-.059.336-.099.099a1.436 1.436 0 00.632.987 1.428 1.428 0 001.166.139l.928-.929.04-.355-1.008-1.008-.375-.02-.118-.157zM319.038 12.515l-.849.85a1.66 1.66 0 00-.138.69l1.145 1.146c.154.011.309-.009.455-.059a1.368 1.368 0 001.027-1.66 1.397 1.397 0 00-1.64-.967zM298.91-.305a1.304 1.304 0 00-1.067-.474l-.138.138-.375.178-.652.652a1.286 1.286 0 00.04 1.066l.809.81h.396a1.314 1.314 0 00.948-1.462l-.139-.138.178-.77zM275.363 7.201l-.316-.316a1.257 1.257 0 00-.652-.138l-1.007 1.007-.04.395v.198a1.353 1.353 0 00.619.825 1.338 1.338 0 001.021.143c.073-.058.14-.124.198-.198l.75-.75V8.05l-.336-.335-.237-.514zM306.357-42.637l-.652.652a1.391 1.391 0 00.395 1.304 1.31 1.31 0 001.264.315 1.172 1.172 0 00.77-.77l.218-.217.019-.336a1.316 1.316 0 00-.612-.889 1.338 1.338 0 00-1.067-.118 2.575 2.575 0 00-.335.059zM297.824-74.835l-.297.296-.059.415c-.042.242-.036.49.02.73l.849.85.336.02.316-.08.514-.315a1.567 1.567 0 00.434-1.66l-.829-.83-.396-.039-.098.1-.316.078-.534.533.06-.098zM319.809-9.826l-.415-.415a1.253 1.253 0 00-.651-.139l-.139.139-.375.177-.395.395-.336-.02-1.225 1.225a1.32 1.32 0 00.455 1.166 1.364 1.364 0 001.205.217c.337-.144.606-.413.75-.75l.297.02a1.397 1.397 0 001.047-1.285l-.04-.237-.158-.395-.02-.098zM315.127-11.861a1.482 1.482 0 00-1.837-1.244l-.711.71a1.623 1.623 0 00.059 1.324l.771.77.336.02.118-.118.375-.099.81-.81.079-.355v-.198zM279.65-22.33a1.42 1.42 0 00.993-1.243 1.411 1.411 0 00-.064-.555 1.295 1.295 0 00-.613-.81 1.535 1.535 0 00-1.027-.118l-.276.276-.336.613-.059.414.513.514.613.336.256.573zM267.778-59.368l.474.474.553.237h.316l1.067-1.067.02-.415-.139-.138a1.257 1.257 0 00-1.56-.85 1.228 1.228 0 00-.948.949v.513l.217.297zM297.527-19.032l-.651.652a1.475 1.475 0 00.118 1.62 1.424 1.424 0 001.501.474c.138-.054.27-.12.395-.197l.652-.652a1.51 1.51 0 00-.395-1.264 1.45 1.45 0 00-1.62-.633zM305.824-6.468l-.356-.08-.257.06a1.512 1.512 0 00-1.087 1.303c-.02.2.001.401.06.593l.948.948.376.06.296-.06.928-.928a3.403 3.403 0 00-.02-1.007l-.888-.89zM287.709-59.388l.553.553.297.06.099.098a1.48 1.48 0 001.382-1.383l-.098-.454-.534-.533-.415-.178a1.286 1.286 0 00-.731.02l-.809.81c-.142.428-.114.895.079 1.303l.177-.296zM299.186-21.837l.751.751c.224.14.486.209.75.197l.139-.138.395-.197.533-.534.079-.434v-.237l-1.047-1.047h-.434a1.344 1.344 0 00-1.176 1.075c-.037.187-.033.38.01.564zM283.028-56.603a1.225 1.225 0 001.027.119l.692-.692c.094-.224.122-.471.079-.71l.553-.554.02-.296-.099-.099a1.593 1.593 0 00-.652-.968l-.119-.118-.316-.08-.098-.098-.218.02-.217.217a1.29 1.29 0 00-.948 1.185l.02.257.197.197a1.23 1.23 0 00-.178 1.087c.049.193.136.375.257.533zM326.605-18.992l.197.197c.083.102.175.194.277.277a1.2 1.2 0 001.185.395l1.027-1.027.079-.395-.02-.217a1.764 1.764 0 00-.513-.79l-.099-.1a1.315 1.315 0 00-1.521-.69l-.612.611a1.534 1.534 0 000 1.739zM311.69 3.35l-.415-.416a1.336 1.336 0 00-.592-.118l-.138.138-.376.178-.592.592a1.307 1.307 0 00.059 1.087l.672.672.335.02.099-.1a1.428 1.428 0 001.047-1.323l-.138-.138-.158-.395.197-.198zM348.866-66.973l.869-.87.06-.296.098-.099a1.54 1.54 0 00-.651-1.086 1.456 1.456 0 00-1.186-.158 1.071 1.071 0 00-.177.178l-.712.71-.059.297.099.099a1.51 1.51 0 00.632 1.067c.311.177.677.233 1.027.158zM264.835-64.643l.178.178a1.277 1.277 0 001.146.474l.948-.948a1.352 1.352 0 00-.04-1.067l-.829-.83-.376.02a1.286 1.286 0 00-.968 1.403l.119.118-.178.652zM307.048-64.03l.276-.08.475-.236.474-.474.039-.395-.02-.218a1.27 1.27 0 00-.604-.825 1.284 1.284 0 00-1.015-.123 1.346 1.346 0 00-.218.218l-.711.71-.039.317.098.098a1.479 1.479 0 001.245 1.008zM272.322-15.278c.131-.091.245-.205.336-.336a1.257 1.257 0 00-1.363-1.956 1.2 1.2 0 00-.889.89l-.119.118v.237l1.087 1.086.355-.04.316-.315.277.316zM270.208-33.827c.132-.043.258-.103.375-.178l.652-.652a1.508 1.508 0 00-.414-1.284 1.564 1.564 0 00-1.304-.355l-.652.652a1.423 1.423 0 00.783 2.078c.271.089.563.094.837.016l-.277-.277zM298.673-66.52l.119-.118.079-.276a1.535 1.535 0 00-.158-1.067l-.79-.79-.316.04a1.367 1.367 0 00-.312 2.45c.291.187.642.258.983.196l.652-.652-.257.218zM245.299-54.311l.395.395.316.079a.584.584 0 00.276-.04 1.37 1.37 0 00-.632-2.647l-.79.79c-.075.274-.061.565.04.83l.118.118.277.475zM316.352-49.235l-.494-.494-.434-.04-.435.277a1.508 1.508 0 00-1.086 1.087l.019.257.119.434.553.553.415.139.257.02.138-.139a1.4 1.4 0 001.027-1.54l-.158-.159.079-.395zM335.967-52.218l.336-.612.06-.375.098-.1-.098-.295-.929-.929-.415-.06a1.454 1.454 0 00-.731.732 1.509 1.509 0 00-.079 1.027 1.376 1.376 0 00.652.828 1.37 1.37 0 001.047.12l.059-.336zM315.996-69.028l.87-.869.019-.375v-.198l-.731-.73a1.395 1.395 0 00-.691-.178l-.118-.119-.257.02-.257.059-.691.691-.079.277-.02.217.039.395.119.119a1.48 1.48 0 001.797.691zM306.851-25.155l.019-.336-.078-.316-.81-.81-.336-.06h-.198l-.118.12a1.48 1.48 0 00-.83.829 1.332 1.332 0 00-.02 1.165l.692.692.237.04c.249.054.508.04.75-.04.103-.021.202-.054.297-.099l.632-.632-.237-.553zM308.154-75.487l.356.355.257.02.434-.118.751-.751.059-.257v-.434l-.059-.257a1.484 1.484 0 00-.652-.652l-.237.237.079-.316-.514-.158-.296.02-.296.098-.771.77a2.06 2.06 0 00.514 1.62l.375-.177zM348.195-37.422l.217.217.297.099c.25.075.516.081.77.02a1.23 1.23 0 00.869-1.462l-.296-.296a1.34 1.34 0 00-1.324-.77l-.75.75c-.148.23-.224.497-.218.77l.079.277.356.395zM318.802-52.909l.652.652.375.02.099-.1a1.483 1.483 0 001.007-1.797l-.908-.908-.356-.04-.928.928a1.708 1.708 0 00.059 1.324v-.079zM326.822-36.356l-.455-.454-.375-.02-.099-.098-1.007 1.007-.04.356.02.217.81.81c.291.137.623.159.928.059l.317-.672.355-.632-.296-.296-.158-.277zM260.134-18.656l-.929.928-.099.375v.198a1.429 1.429 0 001.033 1.242c.185.052.377.066.567.042l.85-.85.099-.375v-.197a1.485 1.485 0 00-.554-1.027 1.394 1.394 0 00-.967-.336zM226.217-12.592a1.568 1.568 0 001.205.138l.83-.83c.033-.26.002-.523-.092-.768a1.584 1.584 0 00-.442-.634 1.254 1.254 0 00-1.185-.277 1.318 1.318 0 00-.751.554l-.276.276-.079.356-.02.217a1.505 1.505 0 00.81.968zM252.035-34.34l.612-.613.158-.474v-.277a1.396 1.396 0 00-1.185-1.185l-.375.02-.277.079-.435.434-.237.475-.039.316a1.452 1.452 0 001.284 1.284l.118-.119.099.099.277-.04zM269.616-24.385l-.395-.04a1.255 1.255 0 00-.909.91 1.419 1.419 0 00.389 1.29 1.422 1.422 0 001.29.388l.474-.474a.927.927 0 00.257-.494l.02-.217a1.457 1.457 0 00-.099-.573l-.652-.652-.375-.138zM263.096-44.691l.198.197c.157.117.33.21.514.277h.276l.494-.139.593-.592.138-.494v-.277l-1.047-1.047-.395-.04h-.198a1.486 1.486 0 00-1.047 1.285l.158.158.198.434.118.238zM267.64-27.427l.118.119.218-.02a.833.833 0 00.414-.217l.376.059.118-.119a1.315 1.315 0 00.85-.85 1.206 1.206 0 00-.099-1.125l-.711-.711-.336-.02-.316.316-.336-.336-.118.119a1.223 1.223 0 00-.787.574 1.232 1.232 0 00-.122.967c.125.223.31.407.533.533-.019.253.051.504.198.711zM288.52-11.17a1.343 1.343 0 00-1.324-.415c-.169.05-.324.138-.454.257l-.514.514a1.38 1.38 0 00.514 1.106 1.474 1.474 0 001.304.277 1.314 1.314 0 00.928-1.443l-.178-.177-.276-.119zM266.356-27.012l-.336.336a1.421 1.421 0 00.138 1.284 1.314 1.314 0 001.146.553l.237-.04.395-.197.533-.534.06-.375v-.197l-.099-.415-.593-.593-.513-.118h-.277l-.435.434-.256-.138zM237.477-30.884c.102-.02.201-.054.296-.098l.336-.336a1.452 1.452 0 00.632-1.225 1.256 1.256 0 00-1.008-1.244l-.257.02-.454.098-1.126 1.126-.039.395.019.217a1.289 1.289 0 001.601 1.047zM250.336-41.847l.573.573a1.31 1.31 0 001.205.217c.086-.071.166-.15.237-.237l.731-.73v-.198l-.099-.1-.178-.493-.632-.632-.395-.08h-.237l-.257.06a1.4 1.4 0 00-.751.75 1.31 1.31 0 00-.197.87zM263.057-4.118l-.474-.474-.375-.099-.217-.02-.119.12a1.305 1.305 0 00-.79.79 1.45 1.45 0 00-.02 1.244l.751.75.355.04h.198a1.372 1.372 0 00.968-1.64l-.138-.138-.139-.573zM256.38-29.876l-.039.316a1.501 1.501 0 00.671 1.066 1.403 1.403 0 001.126.139l.79-.79a1.308 1.308 0 00-.059-1.086l-.79-.791a1.286 1.286 0 00-.751 0 2.7 2.7 0 00-.197.198l-.593.592-.158.356zM245.693-25.55a1.342 1.342 0 00.139 1.244l.731.731h.316a1.397 1.397 0 001.047-1.284l-.04-.277-.652-.651a1.34 1.34 0 00-1.067-.119c-.183.08-.345.202-.474.356zM282.771-18.498l-.414-.06a1.426 1.426 0 00-.968 1.68l-.04.355.948.948c.179.083.377.117.573.099l.139-.138.375-.178.513-.514c.092-.205.139-.427.139-.651l-.119-.12.02-.216a1.483 1.483 0 00-.593-.988l-.138-.138-.435-.08zM279.611-30.825l-.534-.533a1.536 1.536 0 00-1.027-.079 1.314 1.314 0 00-1.007 1.008 1.396 1.396 0 00.494 1.441l.335.336.158-.158a1.398 1.398 0 00.85-.85 1.506 1.506 0 00-.02-1.244l.751.08zM275.265-40.78a1.371 1.371 0 00.098 1.284l.494.494c.289.14.617.175.929.098.212-.067.403-.19.553-.355l.533-.534.079-.316a1.384 1.384 0 00-.174-.601 1.388 1.388 0 00-.418-.465 1.538 1.538 0 00-1.146-.198 1.34 1.34 0 00-.948.593zM261.181-54.588l.651.652.356.04.118-.12.297-.098.849-.85.02-.375v-.197l-1.087-1.087-.414.02-.277.08a1.344 1.344 0 00-.873 1.282c.004.185.045.367.123.534l.237.119zM259.165-10.814a1.396 1.396 0 00-1.659-.988l-.83.83a1.426 1.426 0 00.692 1.837l.217.217.217.02.336-.06a1.39 1.39 0 00.941-.691 1.393 1.393 0 00.086-1.165zM280.026-40.168a1.538 1.538 0 00.079 1.66 1.368 1.368 0 001.422.513l.573-.573a1.427 1.427 0 00.138-1.126 1.477 1.477 0 00-.908-.908l-.119-.119h-.356l-.75.75-.079-.197zM278.781-11.387l-.573-.573-.316-.04-.217.218a1.368 1.368 0 00-.968 1.956l.81.81c.24.064.494.057.73-.02.07-.063.136-.129.198-.198l.77-.77a1.562 1.562 0 00-.138-1.047l-.296-.336zM278.326-36.77l-.375.178-.513.513a1.834 1.834 0 00-.119.672l.909.908c.207.08.43.115.652.1l.118-.12.375-.177.494-.494c.09-.22.137-.454.139-.691l-.909-.91a1.566 1.566 0 00-.632-.118l-.139.139zM275.186-34.973l-.139-.138-.375-.099h-.197a.969.969 0 00-.257.06 1.398 1.398 0 00-.85.849 1.522 1.522 0 00.02 1.244l.771.77c.211.018.424.004.632-.039a1.424 1.424 0 00.987-1.185v-.237l-.118-.395-.356-.356-.118-.474z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M279.927-34.183l-1.067 1.067a1.448 1.448 0 00.099 1.64 1.308 1.308 0 001.402.493l1.205-1.204c.083-.2.124-.416.119-.633l-.928-.928a1.373 1.373 0 00-.613-.138l-.217-.297zM282.949-25.59l-.376.178-.513.514a1.26 1.26 0 00-.139.651l.929.929c.183.092.388.133.592.118l.139-.138.375-.178.514-.513c.088-.213.128-.442.118-.672l-.928-.928a1.203 1.203 0 00-.613-.1l-.098.14zM270.524-42.835l.652.652.355.04.099-.099.395-.119.356-.355a1.593 1.593 0 001.106-1.975l-.731-.731-.316-.119c-.166 0-.332.02-.494.06a1.51 1.51 0 00-.909.71 1.343 1.343 0 00-.77 1.838l.257.098zM266.988-39.615l-.138-.138-.218.02a1.254 1.254 0 00-.829.83 1.373 1.373 0 00.276 1.5 1.315 1.315 0 001.324.376 1.265 1.265 0 00.711-.711 1.36 1.36 0 00.039-1.146l-.118-.118a1.23 1.23 0 00-1.047-.613zM266.277-14.962l-.316.079-.177.098-.297.218-.138.178-.099.217-.079.197-.059.376.039.197.04.158.099.178.217.296.178.139.217.098.198.08.375.059.197-.04.317-.079.177-.099.297-.217.138-.178.118-.197.06-.218.059-.375-.04-.197-.039-.158-.099-.178-.217-.296-.178-.139-.217-.138-.198-.04-.375-.059-.198.04zM245.674-45.68l.316.317c.192.09.401.136.612.138l.988-.987a1.322 1.322 0 00-.059-1.087l-.771-.77-.335-.02-1.008 1.007-.059.376.336.335-.02.692zM257.171-55.121l.83-.83.059-.375a1.43 1.43 0 00-1.097-1.233 1.423 1.423 0 00-.582-.012l-.771.77-.098.297-.04.237a1.368 1.368 0 00.533 1.205 1.29 1.29 0 001.166-.06zM234.652-64.84a1.376 1.376 0 00.81-.81 1.37 1.37 0 00-.059-1.205l-.672-.672c-.22-.06-.452-.06-.672 0a1.428 1.428 0 00-.711.712l-.158.157a1.362 1.362 0 00.591 1.023 1.372 1.372 0 001.167.183l-.296.612zM255.116-17.274l1.008-1.007a.758.758 0 00-.06-.415l-.928-.928-.435-.08-.217.02-.889.89a1.398 1.398 0 00.454 1.323c.318.198.699.269 1.067.197zM239.61-44.316a.838.838 0 00.197-.198l.731-.73.02-.336-.889-.89a1.23 1.23 0 00-.632-.118l-.277.08-.869.868-.02.376.02.217a1.34 1.34 0 001.719.73zM272.934-30.904l-.296.297-.099.336-.415.414-.039.356.019.217 1.028 1.027.355.04.099-.099a1.376 1.376 0 00.81-.81l.592-.592.494-.02v-.277a.865.865 0 00.099-.296v-.237l-.118-.474-.573-.573-.474-.158h-.277a1.51 1.51 0 00-.909.909l-.296-.06zM260.075-40.563l.414.415.356.04.099.098a1.428 1.428 0 00.79-.79 1.36 1.36 0 00.039-1.146l-.79-.79-.355-.04a1.317 1.317 0 00-.79.79 1.4 1.4 0 00-.06 1.166l.356.356-.059-.1zM258.139 11.31l-.119.118-.02.257a1.236 1.236 0 00.613.89 1.366 1.366 0 001.086.137c.067-.051.126-.11.178-.177l.77-.77.02-.297-.099-.415-.612-.612-.474-.119h-.277l-.099.099a1.229 1.229 0 00-.967.889zM320.698-68.159l-.395.395a1.253 1.253 0 00.276 1.265c.169.18.39.305.632.355l.139.138.414.06.218-.02.138-.138a1.543 1.543 0 001.067-.83l.059-.257.119-.118v-.237a1.573 1.573 0 00-.435-.79l-.533-.534.059.06-.119-.12-.276.001a1.706 1.706 0 00-.178.178l-.138.138-1.047.454zM311.849-91.191l.513.513h.316l.099-.099a1.453 1.453 0 001.027-1.303l-.039-.277-.652-.652a1.425 1.425 0 00-1.067-.118 1.304 1.304 0 00-.79.79 1.341 1.341 0 00.138 1.245l.455-.1zM298.218-42.36h-.276l-.474.118-.613.612-.138.494.02.257c.04.376.02.756-.06 1.126l.02.257.138.415.573.572.435.119h.276a1.317 1.317 0 00.85-.85l.178-.177.138-.415.118-.119.277-.474.059-.296-.079-.356.119-.118a1.325 1.325 0 00-.522-.93 1.31 1.31 0 00-1.039-.236zM300.134-37.56l-.118.118-.119.316-.118.119a1.621 1.621 0 00.869 1.501l.316.04.573-.06.316-.316.395-.197.513-.514.08-.435v-.237l-.099-.296-.929-.928-.316-.04-.573.099-.75.75-.04.08zM297.389-48.8l.612.612.435.04.237-.04a1.479 1.479 0 00.81-.81 1.32 1.32 0 00-.099-1.086l-.79-.79-.376.02a1.314 1.314 0 00-.968 1.402l.159.158-.02.494zM290.376-68.1l.691-.69a1.094 1.094 0 00-.059-1.048 1.2 1.2 0 00-.849-.612l-.139-.138-.395-.04a1.255 1.255 0 00-.671.672 1.284 1.284 0 00-.099.968 1.398 1.398 0 001.521.888zM338.99-37.343l-.297-.296a1.424 1.424 0 00-.908-.396 1.508 1.508 0 00-.988.435 1.304 1.304 0 00-.425.94 1.33 1.33 0 00.385.956l.376.139h.671l.395-.159.356-.355.158-.356v-.237l.277-.671zM299.226-2.636l-.375-.138h-.711l-.356.158a2.8 2.8 0 00-.336.335l-.138.376v.71l.158.356a1.23 1.23 0 00.928.376c.347.002.681-.133.929-.376a1.424 1.424 0 00-.099-1.797zM338.358-49.156l-.514.514a1.369 1.369 0 00.375 1.244l.376.376.316.079.237-.04 1.027-1.027.138-.336.099-.099.02-.256-.119-.119-.099-.336-.276-.276a1.37 1.37 0 00-1.857-.158l.277.434zM328.6-24.03l-.198.198-.059.573.059.297c.086.122.193.229.316.316a1.341 1.341 0 001.007.414 1.627 1.627 0 001.107-.671l.296-.297.02-.217-.06-.296a1.006 1.006 0 00-.276-.277 1.341 1.341 0 00-1.877.02l.099.099-.434-.158zM347.642-42.993l-.02.257.059.454.494.494a1.233 1.233 0 001.106.356 1.422 1.422 0 00.949-.948l.118-.119.02-.336-.435-.434a1.424 1.424 0 00-1.54-.316l-.652.652-.099-.06zM320.342-14.587l-.177-.178a1.269 1.269 0 00-.376-1.007 1.73 1.73 0 00-1.244-.455l-.79.79-.139.336a1.344 1.344 0 00.376 1.047c-.055.33.016.668.197.949l-.039-.04.434.435.257.059.217-.02.396-.118.138-.139a1.645 1.645 0 00.612-1.363l.138-.296zM305.567-11.11l-.02-.218-.533-.533a1.369 1.369 0 00-1.521-.296 1.51 1.51 0 00-.87 1.462l.613.612a1.34 1.34 0 001.501.276 1.455 1.455 0 00.83-1.303zM370.536-44l.415.415a1.257 1.257 0 001.324.02l.553-.554a1.59 1.59 0 00.118-.869l-.79-.79-.316-.119-.119-.118a1.73 1.73 0 00-1.382.87 1.318 1.318 0 00.197 1.146zM312.066-34.716l.079-.355v-.198l-.02-.257-.434-.434a1.425 1.425 0 00-1.66-.277l-.612.613-.079.355.02.257c.006.203.067.401.177.573l.593.592.375.06.218-.02a1.42 1.42 0 00.928-.573l.415-.336zM322.634-45.047a1.4 1.4 0 00-1.067-.395 1.516 1.516 0 00-1.047.652l-.118.118-.079.316-.099.1.04.236.711.711.355.158.119.119h.237l.138-.138.336-.139.474-.474.138-.375.02-.217-.02-.257-.355-.356.217-.059zM326.802-18.597l-.533-.533-.277-.04a1.284 1.284 0 00-.77.178l-.553.553-.059.257-.02.217.059.336c.059.25.189.476.375.652l.514.513.217-.02.316-.315c.322-.25.54-.608.613-1.008.119-.245.16-.52.118-.79zM376.542-65.769l-.04.435.02.217c.061.282.198.541.395.751.09.144.211.266.355.356l.317.316a1.283 1.283 0 001.264-.909l.138-.138.04-.237v-.198l-.06-.375-.375-.376a1.48 1.48 0 00-1.659-.276l-.573.573.178-.139zM315.641-40.74l.672-.672.099-.297v-.237l-.079-.434-.277-.277a1.368 1.368 0 00-1.837-.138l-.454.454a1.354 1.354 0 00.395.988 1.424 1.424 0 001.481.612zM337.686-62.865a1.23 1.23 0 00.692-.454l.138-.138.118-.356.119-.118-.04-.237-.572-.573a1.484 1.484 0 00-1.482-.336l-.672.671-.118.396-.099.098c-.023.27.039.539.178.77l.77.771.297.02.276-.04.099-.098.296-.376zM314.693-38.37l-.533-.534a1.4 1.4 0 00-1.205.613 1.597 1.597 0 00-.059 1.481l.612.612.276.08a1.512 1.512 0 001.383-.91 1.34 1.34 0 00-.474-1.342zM332.768-71.22l-.237.236-.04.317.079.553.198.197a1.425 1.425 0 001.047.415 1.476 1.476 0 001.027-.514l.533-.533-.059-.296-.375-.376a1.315 1.315 0 00-1.699-.118l-.296.296-.178-.177zM332.609-51.902l-.256.02-1.186 1.185a1.427 1.427 0 00.396 1.185 1.423 1.423 0 001.975 0l.454-.454.04-.237-.02-.296-.119-.119a1.306 1.306 0 00-.375-.652l-.158-.158-.375-.138-.376-.336zM282.455-55.082l-.376-.138h-.395l-.138-.138a1.032 1.032 0 00-.829.553 1.254 1.254 0 00-.139 1.047v.197l.257.257a1.238 1.238 0 00.889.38 1.228 1.228 0 00.889-.38c.225-.216.371-.501.415-.81a1.29 1.29 0 00-.573-.968zM310.525-42.38l.336.138a1.286 1.286 0 001.62-.395l.355-.356a1.48 1.48 0 00-.296-1.047l-.573-.572a1.594 1.594 0 00-1.422.948 1.364 1.364 0 00.277 1.501l-.297-.217zM318.149-48.702l-.138-.138.099-.099-.158-.355a1.366 1.366 0 00-1.995.02 1.505 1.505 0 00-.454 1.047c.012.372.168.726.434.987l.375.139.218.02a1.255 1.255 0 001.106-.238 1.229 1.229 0 00.553-1.066l-.04-.317zM312.461-46.963l.059-.297-.218-.217-.098-.296-.218-.217c-.172-.171-.381-.3-.612-.376l-.118-.118h-.277a1.4 1.4 0 00-.968.968 1.257 1.257 0 001.225 1.62c.118.01.237.01.355 0a1.397 1.397 0 001.008-.692l-.138-.375zM306.08-45.521l.475-.277.355-.355.04-.316-.06-.534-.355-.355a1.368 1.368 0 00-1.936-.119 1.37 1.37 0 00-.672.672 1.253 1.253 0 00.06 1.126l.691.691.296.06c.237-.036.465-.116.672-.238l.434-.355zM295.117-20.454a1.25 1.25 0 00-1.244-.375 1.283 1.283 0 00-.929.928l-.019.257 1.145 1.146h.277a1.316 1.316 0 00.947-1.447 1.34 1.34 0 00-.177-.509zM321.646-82.776l.316.079.099.099c.231-.02.455-.095.652-.218l.573-.573.039-.316v-.197a1.503 1.503 0 00-.197-.632l-.573-.573-.316-.04-.099-.098a1.479 1.479 0 00-.632.197l-.573.573-.04.316-.118.119c.02.21.088.412.198.592l.671.672zM327.73-68.712l-.118.119v.276c.008.216.069.426.178.613l.197.197.435.237.257.06h.316l.533-.534a1.343 1.343 0 00.02-1.876l-.356-.356-.296-.06-.099-.098-.237.04-.691.69-.139.692zM311.157-65.413l-.257-.257-.375-.099-.672.04-.356.158a1.319 1.319 0 00-.33 1.457c.067.165.165.314.291.44a1.34 1.34 0 001.62.197 1.252 1.252 0 00.495-.95 1.264 1.264 0 00-.416-.986zM316.885-58.38l-.118.118v.237l.098.099.119.355.316.316a1.312 1.312 0 001.185.395 1.391 1.391 0 001.027-.75l.238-.237.138-.336.118-.119v-.276l-1.066-1.067-.218-.02h-.237l-.118.119-.356.158-.77.77-.356.237zM343.928-40.306l-.118.118a1.39 1.39 0 00.316 1.462 1.422 1.422 0 001.58.276 1.51 1.51 0 00.909-1.382l-.889-.89a1.425 1.425 0 00-1.561.376l-.197.198-.04-.158zM327.77-99.567l.177-.178.198-.434.04-.237v-.198a1.425 1.425 0 00-.415-1.205l-.435-.434-.237.237a1.395 1.395 0 00-.83.632l-.098.099-.079.276v.198l.059.256.138.139c.114.173.262.321.435.434a1.389 1.389 0 001.047.415zM383.712-13.007a1.28 1.28 0 001.422.277l.494-.494a1.287 1.287 0 00-.079-1.778l-.376-.375a1.31 1.31 0 00-1.521.573 1.45 1.45 0 00.06 1.797zM279.393-4.612a1.395 1.395 0 00-.099-1.877l-.612-.611-.217-.02a1.4 1.4 0 00-1.126.73l-.099.297-.02.257.079.474.198-.198-.218.218c.058.25.181.482.356.671l.494.494h.395l.75-.75.119.315zM341.4-93.246l-.138.138.77.77a1.423 1.423 0 001.936.04 1.342 1.342 0 00.178-1.758l-.455-.454-.237-.04a1.313 1.313 0 00-1.619.751l-.435.553zM327.967-56.978a1.34 1.34 0 00-1.244-.336l-.83.83-.118.355-.139.138.04.277.533.533a1.341 1.341 0 001.679.257l.632-.632a1.31 1.31 0 00-.553-1.422zM343-41.235l-.099-.414-.296-.297a1.535 1.535 0 00-1.304-.513h-.276a1.677 1.677 0 00-1.146.197 1.93 1.93 0 00-.929 1.798l.85.85.375.098.119.118c.049-.042.096-.088.138-.138a1.367 1.367 0 001.126.02l.375-.375a1.319 1.319 0 00.553-.988l.04-.277.474-.079zM331.326 9.236l.079-.316.098-.1-.039-.236-.612-.613a1.261 1.261 0 00-1.532-.132c-.155.1-.286.233-.384.39l-.119.118a1.34 1.34 0 00.079 1.383l.514.513.237.04.217.02.316-.08c.348-.095.645-.322.83-.632l.316-.355zM329.468-11.308h.238l.414-.415a1.538 1.538 0 00.593-1.106 1.362 1.362 0 00-.415-1.047l-.257-.257-.296-.099-.099-.098h-.237a1.4 1.4 0 00-.968.573l-.098.098-.079.277-.02.217.138.138c.059.25.189.476.375.652l.316.316.316.08.079.671zM324.036-32.543l-.177-.178-.455-.138-.256.02-.297.059-1.027 1.027-.079.316c-.021.178-.015.358.02.534l.829.83.336.058.297-.059c.364-.033.703-.202.948-.474l.138-.138.158-.395.02-.218-.04-.276-.099-.099a1.866 1.866 0 00-.316-.87zM316.708-25.215l.316.317a1.4 1.4 0 001.067.395 1.449 1.449 0 001.067-.593l.138-.138.099-.336.118-.119-.02-.217.316-.316.04-.276-.138-.139-.139-.415-.474-.474-.395-.158-.217-.02h-.277l-.553.554a1.259 1.259 0 00-.375 1.086c-.2.24-.318.538-.336.85l.099.098-.336-.099zM324.984-58.934l.455-.454a1.48 1.48 0 00-.04-2.055l-.533-.533h-.277l-.118.119-.435.197-.632.632-.079.395.02.218a1.2 1.2 0 00.395.79c.106.155.24.289.395.395l.118.118.277.04h.198l.256.138zM334.94-40.286l-.237-.237a1.398 1.398 0 00-1.659-.237l-.672.671-.098.375.098.1c.052.358.218.691.474.947l.316.317.297.098h.237l.395-.395a1.592 1.592 0 00.691-1.126v-.197l.158-.316zM328.718-31.615l-.217-.217-.494-.257-.138-.138a1.175 1.175 0 00-.534-.138c-.155-.101-.33-.168-.513-.198l-.119-.118h-.237l-.118.118-.336.138-.119.119a1.852 1.852 0 00-.671 1.304l.296.296a1.39 1.39 0 00.277 1.067l.533.533h.198c.325.057.66.023.967-.099l.593-.592a1.48 1.48 0 00.632-1.818zM331.819-36.138l-.098-.1-.297-.098-.197-.197-.415.415a1.394 1.394 0 00-.316 2.014l.553.553a1.398 1.398 0 001.324-.849 1.342 1.342 0 00-.297-1.521l-.257-.217zM333.34-45.047l-.118.118-.02.218.119.118.118.316.297.297a1.369 1.369 0 001.837.138l.513-.514.04-.237.099-.098-.06-.336a1.336 1.336 0 00-.375-.652l-.316-.316-.336-.06h-.237l-.77.77-.791.238zM311.454 3.428a1.405 1.405 0 00-1.067-.395 1.531 1.531 0 00-1.146.711l-.296.296-.02.257c.008.386.164.753.435 1.027a1.503 1.503 0 001.126.415l.553.08.296-.297a1.596 1.596 0 00.593-1.146 1.429 1.429 0 00-.474-.948zM355.405-54.074l-.098-.1-.139-.335-.276-.276a1.43 1.43 0 00-1.146-.475 1.508 1.508 0 00-1.027.593l-.158.158-.02.217a1.261 1.261 0 00.217 1.482c.1.136.22.256.356.355l.118.119.336.02h.198l.276-.277c.205-.084.392-.205.553-.355l.139-.139.138-.335.118-.119.415-.533zM355.919-60.633l-.119.119v.237l.099.099.118.355.257.257a1.818 1.818 0 001.837.454l1.008-1.007a1.507 1.507 0 00-.435-.988 1.7 1.7 0 00-.79-.434l-.118-.119h-.277l-.119.119-.375.138-.454.454-.632.316zM337.054-34.538l-.197-.198a1.403 1.403 0 00-1.126-.415 1.367 1.367 0 00-1.067.633l-.415.414.02.218.099.099.099.296.296.296a1.42 1.42 0 001.148.41 1.43 1.43 0 001.044-.627l.317-.316-.02-.218-.099-.098-.099-.297v-.197zM364.531-50.203h-.197a1.147 1.147 0 00-1.285.376l-.434.434-.02.297-.02.217.277.276a1.31 1.31 0 001.185.356 1.453 1.453 0 001.053-.83c.071-.164.109-.34.11-.52a1.327 1.327 0 00-.392-.961l-.277.355zM391.139-53.62l-.138.138.434.435a1.486 1.486 0 001.541.316 1.617 1.617 0 001.027-.711l-.039-.237-.514-.514a1.426 1.426 0 00-1.264-.395 1.508 1.508 0 00-1.086 1.086l.039-.118zM325.656-26.775l-.316.316-.02.217.079.277.336.336a1.344 1.344 0 001.225.395 1.587 1.587 0 001.146-.948l.039-.277-.02-.494-.592-.592-.435-.198-.158-.158a1.564 1.564 0 00-1.284 1.126zM377.509-6.725l-.395-.158h-.276a1.317 1.317 0 00-.554.04 1.26 1.26 0 00-.177.177l-.692.692.02.335.02.297.138.375a1.266 1.266 0 00.928.375 1.199 1.199 0 00.909-.395 1.342 1.342 0 00.079-1.738zM372.433-36.296a1.539 1.539 0 00-1.936-.198l-.652.652-.04.277.02.256.138.415.771.77.395.12.118.118.297-.02.375-.375.731-.731.039-.277-.098-.494-.178-.177.02-.336zM343.889-46.785l.435.434.355.158.218.02h.237l.118-.118.356-.159.79-.79.02-.256-.119-.119-.138-.336-.158-.158a1.373 1.373 0 00-1.047-.415 1.486 1.486 0 00-1.027.514l-.139.138-.098.336-.119.119.02.256.118.119.178.257zM371.899-58.084a1.255 1.255 0 00-1.54-.04l-.198.198-.197.474a1.282 1.282 0 00.296 1.363c.099.136.219.256.356.355l.098.1.336.019.652-.652a1.312 1.312 0 00.277-1.699l-.158-.158.078.04zM292.766-61.64l.02-.217v-.237l-1.205-1.205-.256-.02-.099.099-.336.138-.138.138a1.51 1.51 0 00-.435 1.58l.691.692.297.06.316-.04.158-.158c.268-.084.508-.242.691-.455l.119-.118.177-.257zM323.641.564l-.809-.81-.336-.02-.257.06-.356.158-.098.098a1.513 1.513 0 00-.593 1.028 1.374 1.374 0 00.395.987l.395.395a1.45 1.45 0 001.185-.553 1.557 1.557 0 00.474-1.343zM314.989-13.935l.02-.217-.02-.257-.119-.119a1.36 1.36 0 00.237-.237 1.538 1.538 0 00-.177-1.758l-.455-.454-.217-.02-.099.099-.316.079-.099.099a1.827 1.827 0 00-.888 1.126c-.053.265-.025.54.079.79l-.119.118a1.23 1.23 0 00.277 1.383l.355.356.316.079.889-.89.336-.177zM310.426-71.28l.356.356.336.138.118.119h.237l.119-.119.336-.099.138-.138a1.392 1.392 0 00.535-1.048 1.401 1.401 0 00-.456-1.085l-.277-.277-.335-.098-.099-.1h-.237l-.79.79-.139.376-.118.119v.237l.02.217.158.355.098.257zM363.524-87.873l.178.178.415.178.276.04.297-.02c.24-.019.472-.1.671-.237l.514-.514.039-.237.099-.099-.02-.375-.118-.119a1.327 1.327 0 00-.336-.335 1.453 1.453 0 00-1.956-.06 1.397 1.397 0 00-.059 1.6zM362.201-64.11a1.308 1.308 0 00.079 1.186l.592.593.356.039.257-.02a1.45 1.45 0 00.908-.553l.119-.118.099-.376-.06-.256-.493-.494a1.289 1.289 0 00-1.719-.139l-.158.158.02-.02zM380.255-47.24l-.118.119-.02.257c.027.23.101.451.217.651l.178.178.415.217.158.158h.316l.138-.138.356-.158.138-.138a1.307 1.307 0 00.33-1.457 1.316 1.316 0 00-.29-.44l-.316-.315-.336-.1-.237.04-.692.692-.237.434zM321.646-49.946l.237.237.118-.118a1.454 1.454 0 00.988-.988 1.364 1.364 0 00-.355-1.264 1.285 1.285 0 00-1.028-.356l-.533.534a1.39 1.39 0 00-.454.967 1.356 1.356 0 00.415 1.008l.612-.02zM297.685-26.775h.237l.217-.02.396-.158c.23-.076.44-.204.612-.375l.217-.217.138-.573-.039-.316-.474-.474-.514-.277-.059.375.04-.395a1.539 1.539 0 00-1.64.455l-.138.138-.119.355-.118.119.019.257.119.118.138.336.494.494.336.138.138.02zM343.711-77.7l-.079.356-.02.257c.02.21.088.412.198.592l.592.593.356.04 1.027 1.027h.435l.237-.04a1.29 1.29 0 00.77-.968 1.317 1.317 0 00-.395-1.146 1.336 1.336 0 00-.968-.414l-.336-.336a1.422 1.422 0 00-1.6-.296l-.474.474.257-.139zM344.205-72.623a1.535 1.535 0 00-.593 1.106c-.013.415.135.82.415 1.126l.435.435a1.678 1.678 0 001.126-.692l.118-.118.099-.336.099-.099-.02-.217-.889-.889-.336-.138-.118-.119h-.237l-.099.099-.356.118.356-.276zM352.403-36.533a1.452 1.452 0 00-2.035.059 1.426 1.426 0 00-.414.928 1.331 1.331 0 00.395.949l.395.158.691-.02.375-.138.514-.514.158-.356v-.71l-.079-.356zM325.873-8.108l.237-.237.059-.336-.098-.573-.553-.553-.376-.138-.217-.02h-.277l-.138.139-.395.197-.158.158a1.257 1.257 0 00.63 2.094c.232.05.473.029.694-.06l.474-.236.118-.435zM346.555-53.758l.553.553a1.478 1.478 0 001.344-.91 1.4 1.4 0 00-.317-1.54l-.434-.434-.217.02a1.51 1.51 0 00-.949.948 1.539 1.539 0 00.02 1.363zM353.647-67.724l-.138.138v.237l.098.099.099.336.316.316c.573.573 1.363.573 2.134-.04l.612-.612.02-.296-.119-.119-.138-.375-.711-.711-.415-.02-.138.138a3.119 3.119 0 00-.889.534l-.138.138-.593.237zM316.096-25.471l.098-.296.099-.1-.099-.296-.237-.237a1.423 1.423 0 00-1.659-.276l-.79.79-.04.237.02.454c.069.245.198.47.375.652l.336.336h.198c.448-.084.849-.33 1.126-.691l.573-.573zM343.335-53.818l.514-.513.039-.316a1.479 1.479 0 00-.177-.731l-.632-.632-.297-.02h-.237l-.434.158-.474.474-.139.375-.039.237.019.257.554.553a1.308 1.308 0 001.303.158zM320.836-22.745a1.37 1.37 0 00-.395-.948l-.356-.159h-.395l-.237-.039a1.09 1.09 0 00-.632.276l-.276-.039-.218.02-.592.592a1.84 1.84 0 00-.553 1.067l.098.099.079.316.554.553a1.28 1.28 0 001.501.237l.296-.296a1.366 1.366 0 001.047-.376 1.43 1.43 0 00.079-1.303zM332.373-79.161c.037-.089.064-.182.079-.277a.92.92 0 00.256-.257 1.204 1.204 0 00-.217-1.244l-.711-.711h-.277l-.118.118-.375.138-.139.139c-.233.236-.385.54-.434.869a1.283 1.283 0 00.138 1.64l.514.513a1.505 1.505 0 001.284-.928zM359.139-77.048l-.079.356v.237c.03.198.09.391.178.573l.612.612.355.04h.237a1.341 1.341 0 00.929-.573l.138-.139.059-.335.099-.1-.02-.256-.454-.454a1.367 1.367 0 00-1.639-.257l-.455.454.04-.158zM279.631-75.171l-.494-.494h-.237l-.297.099-.118.118a1.451 1.451 0 00-.652 1.087 1.488 1.488 0 00.454 1.126l.593.592.217-.217.296-.099.119-.118c.334-.272.553-.66.612-1.087a1.48 1.48 0 00-.493-1.007zM322.694-2.952l.138-.138a1.321 1.321 0 00-.158-1.225l-.553-.553-.356-.04-.118.119a1.315 1.315 0 00-.889.533l-.139.138-.118.356.039.276.494.494a1.315 1.315 0 001.719.139l-.316.316.257-.415z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M302.624-40.089l-.751-.75-.336.02-.118.118-.395.158-.395.395a1.286 1.286 0 00-.02 1.877l.336.335.316.08.237-.04.691-.692a1.423 1.423 0 00.435-1.5zM362.22-33.51a1.417 1.417 0 00-.493-.14 1.257 1.257 0 00-1.284.692 1.313 1.313 0 00.256 1.482 1.35 1.35 0 001.225.316l.711-.711.139-.494-.001-.277-.711-.71.158-.159zM346.97-22.686l-.434-.435-.139.139a1.514 1.514 0 00-.948.948 1.475 1.475 0 00.316 1.303 1.483 1.483 0 001.363-.888 1.393 1.393 0 00-.158-1.067zM304.54-31.575l-.297.06a1.507 1.507 0 00-.75.75l-.099.099-.079.276v.198l.079.276.316.316a1.37 1.37 0 001.995-1.876l-.474-.474-.691.375zM341.262 9.058l.098-.099.02-.217-.099-.099-.079-.316-.316-.316a1.396 1.396 0 10-1.975 1.975l.158.158.395.158.138.139.297-.02.138-.138a1.37 1.37 0 00.711-.435l.118-.118.396-.672zM311.019-55.437l-.139.138-.019.296.138.138.158.396.158.157a1.508 1.508 0 001.521.336l.652-.651.099-.297v-.237l-.079-.434-.336-.336a1.398 1.398 0 00-1.857-.158l-.178.178-.118.474zM298.791-71.102l-.691-.02-.356.158-.79.79v.277l.158.395a1.266 1.266 0 00.968.375 1.346 1.346 0 00.948-.395 1.393 1.393 0 00-.019-1.955l-.218.375zM293.892-45.462l-.158-.158-.355-.158-.217-.02-.257.02-.889.889-.119.316-.098.099c-.001.246.053.489.158.71l.652.653.454.138h.276l.297-.06.533-.533a1.452 1.452 0 00-.277-1.896zM336.639-58.894l-.711.71-.138.337-.099.098-.02.257.119.119.098.335.277.277a1.423 1.423 0 001.106.435 1.396 1.396 0 001.027-.553l.395-.396-.335-.335a1.425 1.425 0 00-.376-.652l-.336-.336-.316-.08-.098-.098-.218.02-.375-.138zM292.095-35.486v-.237l-.889-.89-.218-.019-.296.02-.257.059-.493.494a2.14 2.14 0 00-.277.869l.356.079-.336.06.138.533.632.632.514.118.276-.04a2.172 2.172 0 001.205-1.204l-.355-.474zM305.744-88.999l.06-.296-.04-.316a1.34 1.34 0 00-.356-.711 1.254 1.254 0 00-1.679-.02l-.296.296a1.314 1.314 0 00-.02 1.56l.534.534.296.06c.264.007.525-.061.751-.198l.217-.217.533-.692zM306.613-55.773l.079-.356.099-.099-.02-.256-.493-.494a1.481 1.481 0 00-2.055 0l-.434.434-.02.217v.277l.118.119.139.335.592.593.356.08.099.098a1.734 1.734 0 001.126-.691l.414-.257zM317.122-70.865l-.118.118v.277l1.067 1.067.217.02h.237l.118-.12.356-.157.731-.731.02-.296a1.288 1.288 0 00-.415-.81l-.751.079h-.75l-.316.316-.396.237zM365.124-21.323l.533.533.277.04h.197l.356-.119.138-.138a1.395 1.395 0 00.316-2.015l-.533-.533-.237-.04-.218.02-.375.099-.138.138a1.444 1.444 0 00-.632.988 1.347 1.347 0 00.316 1.027zM386.971 5.344a1.488 1.488 0 00-.454-1.126l-.534-.533-.256.02a1.587 1.587 0 00-1.067 1.066 1.31 1.31 0 00.356 1.186 1.17 1.17 0 00.829.356l.099.098.217-.02.356-.079.138-.138c.182-.241.292-.529.316-.83zM355.405-6.587l-.395-.395-.296-.02-.514.08-.177.177a1.453 1.453 0 00-.356 1.66l.672.671.296.06.533-.02.198-.198a1.371 1.371 0 00.395-1.778l-.356-.237zM355.504-26.795l.158.356a1.312 1.312 0 001.224.355 1.64 1.64 0 00.652-.375 1.328 1.328 0 00.433-.946 1.353 1.353 0 00-.373-.97l-.356-.158h-.316a.976.976 0 00-.533.02l-.139.138a1.373 1.373 0 00-.395.395l-.197.198-.158.987zM340.965-11.19l.02-.256a1.443 1.443 0 000-.593l-.731-.73-.316-.08h-.277l-.474.119-.553.553-.158.355-.118.119v.237l.118.118.138.376.158.158a1.4 1.4 0 001.62.276l.514-.513.059-.139zM353.825-23.575l-.257-.257a.98.98 0 00-.336-.335l-.395-.395a1.287 1.287 0 00-1.126-.376 1.313 1.313 0 00-.968.613l-.099.098-.079.316-.098.1.829.829.02.217.474.474a1.23 1.23 0 001.778.04 1.09 1.09 0 00.336-.692 1.613 1.613 0 00-.079-.632zM357.795-49.788a1.208 1.208 0 00.277 1.343c.089.145.211.267.355.356l.119.118.118-.118a1.39 1.39 0 00.712-.988 1.45 1.45 0 00-.395-1.264 1.288 1.288 0 00-1.699-.119l-.198.198.711.474zM354.892-22.627l.237.04a1.315 1.315 0 001.086-.731 1.42 1.42 0 00-.099-1.284l-.632-.632-.276-.04-.237.04-.356.118-.632.633-.099.296v.276l.119.514.553.553.336.217zM365.519 4.436l-.336-.336-.296-.06a1.423 1.423 0 00-1.383.04 1.287 1.287 0 00-.592 1.501l.77.77.336.139.118.119.257.02h.316l.139-.139.355-.158.138-.138a1.31 1.31 0 00.178-1.758zM349.657-1.431l.118-.119.119-.316.099-.099a1.45 1.45 0 00-.376-1.165l-.276-.277-.336-.059-.099-.099-.987.988-.04.356.02.217.138.415.336.336c.264.242.609.376.968.375.15-.155.259-.345.316-.553zM376.403-30.943l-.118.118-.06.257.119.119.099.415.256.256a1.477 1.477 0 001.067.435 1.674 1.674 0 001.086-.612l.119-.119.119-.316.197-.197-.118-.119-.139-.336-.118-.118a1.398 1.398 0 00-1.916-.06l-.139.139-.454.138zM357.124-12.612l.316-.316a1.427 1.427 0 00.573-1.086 1.363 1.363 0 00-.415-1.087l-.237-.237-.316-.118-.099-.099h-.277a1.433 1.433 0 00-.928.573l-.395.395.138.138c.056.24.179.46.356.632l.336.336.316.08.217-.02.415.81zM366.151-5.323a1.342 1.342 0 001.008-.73l.118-.119.06-.336v-.197l-.04-.237-.415-.415a1.421 1.421 0 00-1.6-.296l-.652.651-.079.277a1.429 1.429 0 00.96 1.348c.206.069.426.087.64.054zM354.26-41.116l.138.375a1.317 1.317 0 001.086.376 1.148 1.148 0 00.83-.356.863.863 0 00.336-.138l.573-.573.059-.336v-.197a1.638 1.638 0 00-.197-.593l-.573-.573-.336-.06a1.086 1.086 0 00-.612.218l-.257.257a1.426 1.426 0 00-.593.988v.197l-.454.415zM344.995-32.365l-.217.217a1.261 1.261 0 00.415.889 1.365 1.365 0 001.382.356 1.508 1.508 0 001.087-1.087v-.198l-.395-.395a1.347 1.347 0 00-1.126-.414 1.457 1.457 0 00-1.087.612l-.059.02zM338.437-23.713a1.373 1.373 0 001.066-.711l.277-.277.02-.257a1.338 1.338 0 00-.376-.849 1.257 1.257 0 00-.592-.316l-.237-.237h-.198l-.316.079-.118.118a1.39 1.39 0 00-.672 1.028 1.445 1.445 0 00.435 1.106c.198.176.447.287.711.316zM333.32-4.612l.099-.335.099-.1-.02-.256-.118-.118-.119-.316-.513-.514-.336-.138-.119-.119-.257-.02-.138.139-.336.138-.474.474-.158.395v.514l.632.632a1.4 1.4 0 001.976-.237l-.218-.139zM372.452-24.306a1.396 1.396 0 00-.059 1.956l.513.513h.277a1.368 1.368 0 001.363-1.363l-.593-.592a1.393 1.393 0 00-1.501-.514zM336.659-28.177a1.453 1.453 0 001.383-.87 1.345 1.345 0 00-.237-1.461l-.396-.395a1.368 1.368 0 00-1.066.434 1.729 1.729 0 00-.415 1.205l.79.79.316.119-.375.178zM362.378-41.136l-.079.316.02.218.573.572a1.59 1.59 0 001.185.474 1.428 1.428 0 001.008-.572l.118-.119.079-.276.099-.1v-.197l-.158-.158a2.106 2.106 0 00-.395-.632l-.494-.494-.316.316c-.34.112-.632.335-.83.632l-.81.02zM340.432-28.316l.415.415.336.06h.197c.222-.026.436-.093.632-.198l.573-.573.04-.316.02-.217a1.401 1.401 0 00-.218-.613l-.573-.573-.316-.04-.098-.098a1.201 1.201 0 00-.652.217l-.553.553-.06.336a.925.925 0 00.198.632l.059.415zM343.533-26.479l.099.099.098.375.336.336a1.428 1.428 0 001.741.214c.181-.109.335-.256.452-.431l.355-.356-.059-.256a1.68 1.68 0 00-.434-.435 1.341 1.341 0 00-1.521-.296l-.791.79-.276-.04zM341.894-30.923l.79.79a1.453 1.453 0 001.383-.87 1.37 1.37 0 00-.257-1.482l-.494-.492-.336-.02-.336-.06h-.197a1.4 1.4 0 00-.613.218l-.572.573-.06.335v.198c.021.21.088.412.198.593l.494.217zM347.958-37.817l.415.414.336.06.099-.1c.224-.009.442-.077.632-.197l.553-.553.059-.336-.099-.098a1.265 1.265 0 00-.197-.593l-.573-.573-.356-.04-.098-.098a1.707 1.707 0 00-.633.198l-.572.572-.06.336v.198c.024.218.098.428.218.612l.276.198zM338.951-18.577l.098-.336v-.237l-.118-.119-.139-.336-.77-.77a1.569 1.569 0 00-1.738-.354 1.57 1.57 0 00-.514.354l-.434.435-.04.237.079.316.435.434c.286.283.665.452 1.066.474a1.344 1.344 0 001.976-.237l.099.139zM343.474-17.056l.019.217.237.237a1.231 1.231 0 001.126.336 1.423 1.423 0 001.166-.968 1.338 1.338 0 00-.336-1.363 1.314 1.314 0 00-.928-.375 1.368 1.368 0 00-1.008.533l-.118.119a1.36 1.36 0 00-.158 1.264zM365.38-28.454l.218.217.177.138.336.139.198.04h.276l.198-.04.355-.158.159-.119.098-.099.139-.177.118-.356.04-.197.019-.257-.039-.198-.158-.355-.119-.158-.237-.237-.158-.119-.355-.118-.198-.04-.257-.02-.197.04-.356.158-.158.118-.118.119-.119.158-.118.356-.06.217v.237l.04.197.158.356.118.158zM348.61 4.515l.098-.1.02-.295a1.334 1.334 0 00-.178-.573l-.592-.593-.376-.06h-.237a1.347 1.347 0 00-.908.594l-.119.118-.099.336-.098.099.039.237.771.77.335.138.119.119h.276l.119-.119.356-.158.375-.375.099-.138zM334.762-.918l-.158-.158-.355-.118-.119-.119h-.237l-.118.119-.336.138-.138.138a1.392 1.392 0 00-.574.975 1.4 1.4 0 00.337 1.08l.474.474.217.02.099.098.336-.059.138-.138a1.478 1.478 0 00.81-1.087 1.308 1.308 0 00-.376-1.363zM337.272 23.478a1.378 1.378 0 00-1.107-.395 1.363 1.363 0 00-1.007.652l-.118.118-.079.356.019.217.139.138c.057.208.166.399.316.553.254.248.593.39.948.396l.257-.257a1.434 1.434 0 00.632-1.027c.07-.246.07-.506 0-.751zM368.818-17.708l-.554-.553-.296.059a2.164 2.164 0 00-.296.296l-1.067 1.067.099.099v.237l.158.395.77.77.257-.02a1.45 1.45 0 00.988-.987 1.592 1.592 0 00-.059-1.363zM352.758 9.137l-.335-.336-.316-.079-.099-.099-.237.04-1.027 1.027-.02.257c.009.224.077.442.197.632l.514.513.257.06h.316l.138-.138.356-.159.138-.138a1.311 1.311 0 00.118-1.58zM348.195-26.518l.237.04.099-.1.316-.079.119-.118.256.02.119-.119.355-.158.455-.454.138-.375.039-.238-.019-.256-.494-.494a1.345 1.345 0 00-1.027-.395l-.277.04-.197-.198-.257-.02-.494.099-.178.178a1.191 1.191 0 00-.178.177l-.691.692-.099.296v.277l.119.474.178.177a1.45 1.45 0 001.244.415l.237.119zM346.022-10.557l.099-.1v-.236l-.475-.474a1.39 1.39 0 00-1.62-.265 1.4 1.4 0 00-.454.364l-.138.138-.118.316-.099.099v.237l.118.118.119.316.138.138a1.484 1.484 0 001.067.435 1.452 1.452 0 001.007-.533l.139-.139.217-.414zM392.265-34.163l.237-.04.178-.177a1.393 1.393 0 00.367-1.524 1.336 1.336 0 00-.308-.471l-.335-.336-.316-.08-.099-.098-.257.02-.909.908-.079.316v.277l.119.474.514.514c.263.17.576.247.888.217zM292.352-50.005l.119.118.217-.02.217-.217a1.368 1.368 0 00.929-.928 1.475 1.475 0 000-.711l-.02-.218-.138-.375-.336-.336a1.93 1.93 0 00-1.264-.513l-.455.454c-.251.19-.427.463-.494.77l-.118.119v.276l.02.218.829.83.494.533zM276.687-30.943l.059-.336-.039-.237-.336-.336a1.398 1.398 0 00-1.64-.256l-.81.81c.013.363.161.708.415.967a1.257 1.257 0 001.087.376 1.422 1.422 0 001.027-.712l.237-.276zM325.834-42.953l.889.889.316.079h.277l.474-.119.197-.197a4.33 4.33 0 011.008-.494l.474-.474.079-.277-.02-.257-.099-.454-.177-.178a1.313 1.313 0 00-1.126-.375l-.257-.02-.454.099h-.554l-.316.079-.395.395a1.366 1.366 0 00-.316 1.304zM329.093-47.003h.198l.335-.059.119-.118a1.537 1.537 0 00.889-1.482l-.099-.296-1.225-1.225-.276-.04-.237.04-.415.138-.553.553-.138.415-.02.218.059.296 1.363 1.363v.197zM320.639-39.101l.138-.139v-.276l-.138-.138-.198-.395-.178-.178a1.314 1.314 0 00-1.086-.297 1.282 1.282 0 00-.909.593l-.118.119-.079.355v.198l.039.237.474.474a1.343 1.343 0 001.719.138l.158-.158.178-.533zM307.226-23.555l-.395-.158h-.356l-.237-.04a1.17 1.17 0 00-.869.553 1.238 1.238 0 00-.184.509c-.024.18-.009.364.046.538v.198l.178.375a1.224 1.224 0 00.888.375c.333-.01.65-.144.889-.375a1.422 1.422 0 00.04-1.975zM378.872-27.21l-.375-.098a1.174 1.174 0 00-.849.494 1.396 1.396 0 00-.257 1.047 1.326 1.326 0 00.481.9 1.31 1.31 0 00.981.285l.77-.77.099-.376-.79-.79-.06-.691zM382.31 25.414l-.771.77-.079.395c.011.16.058.316.139.455l.77.77.375.099a1.34 1.34 0 00.889-.494c.198-.289.282-.64.237-.988a1.393 1.393 0 00-1.56-1.007zM369.174-34.025l.158.356.276.276a1.284 1.284 0 001.205.455.983.983 0 00.376-.099l.612-.612.039-.316.099-.1-.039-.315-.791-.79-.335-.1-.119-.118-.118.119a1.372 1.372 0 00-1.304 1.304l-.059-.06zM383.179-11.012l.829.83h.474a1.372 1.372 0 00.909-.553 1.618 1.618 0 00.178-1.284l-.968-.968a.894.894 0 00-.395 0 1.342 1.342 0 00-1.126 1.482l-.099.098.198.395zM379.643-37.66l.494.495.316.079.415-.02a1.315 1.315 0 00.968-.652 1.606 1.606 0 00-.06-1.363l-.632-.632-.257-.02a1.453 1.453 0 00-1.205 1.008l-.019.256.039.435.158.158-.217.257zM385.569 1.334l-.257-.02-.099-.098A1.203 1.203 0 00384.186.9c-.456.088-.86.35-1.126.73l-.118.12-.08.355v.197l.02.218.494.493a1.399 1.399 0 001.047.336 1.227 1.227 0 00.869.435l.217-.02.376-.138.494-.494.177-.376.119-.118a1.628 1.628 0 00-.711-1.304h-.395zM379.346 15.142l-.237-.237-.237-.04-.256.02a1.312 1.312 0 00-1.087 1.087 1.481 1.481 0 00.632 1.58l.376.375.296-.02a1.398 1.398 0 001.086-1.086 1.447 1.447 0 00-.573-1.679zM392.779-56.445l.355-.355a1.224 1.224 0 00.79-.79l-.118-.12a1.486 1.486 0 00-.612-.611l-.119-.119-.375-.06-.099.1-.237.04-.494.493a1.654 1.654 0 00-.158 1.62 1.285 1.285 0 001.067-.198zM364.689-4.375l-.553-.553-.336-.059a1.425 1.425 0 00-1.185 1.185l.02.218.099.414.375.376c.183.103.384.17.593.197l.217-.02.395-.157.573-.573c.17-.337.205-.725.099-1.087l-.297.06zM362.793-19.032a1.447 1.447 0 00-.968.613 1.53 1.53 0 00-.119 1.225l.771.77.316.079.099.099.849-.85.099-.375.02-.217-.04-.237-.81-.81-.355-.08.138-.216zM386.32-6.251l-.771.77c-.191.2-.309.457-.336.731l.02.217.119.356.691.691c.24.114.51.148.77.1a.977.977 0 00.297-.06l.691-.691.099-.297a1.481 1.481 0 00-.415-1.086 1.315 1.315 0 00-1.165-.73zM379.129-74.44l.474.474c.265.122.559.17.849.138.167-.022.328-.076.474-.158l.356-.356a1.34 1.34 0 00.079-1.58l-.83-.83-.335.02a1.426 1.426 0 00-1.186 1.186v.237l.099.572.02.297zM361.944-10.873l.039-.237-.059-.455-.553-.553-.395-.197-.158-.158a.5.5 0 00-.237.04 1.369 1.369 0 00-1.304 1.303l.553.553a1.34 1.34 0 001.027.316 1.37 1.37 0 001.087-.612z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M357.815-41.788a1.45 1.45 0 00.059-.85l-.849-.849-.356-.079h-.197a1.567 1.567 0 00-1.186.988l-.019.257.059.454.573.573c.199.19.456.31.731.336l.237-.04.434-.158.672-.671-.158.04zM363.267-8.641s-.118.118-.375.099l-.395.395a1.335 1.335 0 00-.237 1.343 1.514 1.514 0 001.047 1.047l.257-.02.414-.415a1.456 1.456 0 00.139-1.64 1.345 1.345 0 00-.85-.81zM348.057-43.052l.434.435.514.276.296.06a1.362 1.362 0 001.2-1.126 1.42 1.42 0 00-.015-.573l-.888-.89a1.064 1.064 0 00-.356 0 1.312 1.312 0 00-1.225 1.225l.139.139-.099.454zM363.405-31.101l-.296.296-.099.336-.019.217.553.553a1.503 1.503 0 001.224.396 1.399 1.399 0 001.166-1.561l-.929-.928a1.312 1.312 0 00-.75-.08l-.415.415-.435.356zM370.596-14.232c.076-.16.116-.335.118-.513.131-.201.231-.421.297-.652.212-.125.388-.302.513-.514l.652.652.375.099a.922.922 0 00.474 0 1.257 1.257 0 001.107-1.106v-.198l-.099-.375-.85-.85-.395-.118-.217-.02-.77.77-.356-.355a1.423 1.423 0 00-1.284-.533l-.336-.06-.217.02-.237.237a1.881 1.881 0 00-.277.277l-.395.395a1.458 1.458 0 00-.276.632l-.316.316a1.055 1.055 0 00-.395.83l.059.296.988.988.434.04.119-.12c.178-.093.35-.199.513-.315.23.144.501.207.771.177zM330.397 6.767l-.711.71a1.17 1.17 0 00-.079 1.028 1.224 1.224 0 00.731.73l.138.14.395.118a1.285 1.285 0 001.086-1.442 1.42 1.42 0 00-.395-.79 1.311 1.311 0 00-1.165-.494zM357.558-7.792l.257-.257a1.29 1.29 0 00.672-1.54l-.514-.514a1.676 1.676 0 00-1.027-.395l-.237.04-.435.434a1.536 1.536 0 00-.118 1.699 1.31 1.31 0 001.363.691l.039-.158zM357.104-17.688l-.276-.277-.415-.099a1.397 1.397 0 00-1.205 1.6c.074.39.275.745.573 1.008a1.394 1.394 0 001.047.256l.434-.434a1.229 1.229 0 00.514-.988 1.337 1.337 0 00-.534-1.126l-.138.06zM355.148-12.137l-.573-.573-.316-.08-.098-.098a1.7 1.7 0 00-.672.276l-.573.573a1.369 1.369 0 00.217 1.363 1.232 1.232 0 002.015 0c.095-.075.182-.162.257-.257a1.476 1.476 0 00.079-1.225l-.336.02zM352.383-6.172l.059-.533-.059-.297-.435-.434-.474-.277-.296-.059-.099.099a1.398 1.398 0 00-1.383 1.383c-.07.311-.035.638.099.928a1.307 1.307 0 00.928.652l.316-.04.692-.691c.11-.217.183-.45.217-.691l.435-.04zM365.578 17.77a1.338 1.338 0 00-1.047.77 1.31 1.31 0 00.138 1.284l.514.513.276.04.929-.929a1.405 1.405 0 00-.218-1.284 1.34 1.34 0 00-.592-.395zM331.661-41.116l.396-.118.513-.514a1.47 1.47 0 00.217-.652l-.019-.217-.139-.376-.513-.513a1.26 1.26 0 00-.613-.217l-.217.02-.375.138-.514.513c-.113.193-.18.41-.197.632l.019.218.139.375.513.513c.181.11.383.177.593.198h.197zM347.109-37.481l.414.414c.166.128.365.203.573.218l.277-.04.454-.217.553-.553.099-.376.099-.098a1.308 1.308 0 00-.961-1.131 1.314 1.314 0 00-.521-.035l-.316.08-.75.75-.06.296-.098.099.217.217.02.376zM339.524-22.31l-.218.217-.217.02-.751.75-.059.375a1.341 1.341 0 001.482 1.205 1.4 1.4 0 001.145-1.146 1.33 1.33 0 00-.377-1.016 1.34 1.34 0 00-1.005-.406zM348.728-22.587l.435.434.335.139.099.098h.277a1.455 1.455 0 001.066-.71 1.471 1.471 0 000-1.265l-.849-.85-.375-.098h-.198l-.257.02-.849.849-.099.375v.198l.02.217.257.257.138.336zM379.505-33.017l.118.118a1.363 1.363 0 002.102.266 1.457 1.457 0 00.38-1.568 1.454 1.454 0 00-.507-.673l-.138-.138-.375-.099h-.237l-.257.06-.138.138a1.45 1.45 0 00-.652 1.481l-.296.415zM322.654-56.287l-.158-.158-.217-.415-.316-.316a1.398 1.398 0 00-1.205-.415l-.237.04-.81.81a1.451 1.451 0 00-.04 1.066l1.027 1.028c.215.039.435.046.652.02a1.366 1.366 0 001.3-1.062c.046-.197.047-.4.004-.598zM425.392-48.01a1.316 1.316 0 001.027-1.027v-.316l-.158-.356a1.312 1.312 0 00-.474-.79 1.313 1.313 0 00-.968-.217l-.336.336a1.312 1.312 0 00-.493 1.56 1.506 1.506 0 001.402.81zM368.581 39.893a1.373 1.373 0 00-1.561-1.047c-.171.02-.338.066-.494.139l-.434.434a1.36 1.36 0 00-.099 1.324l.395.395.434.237.099-.1-.039.317c.234.083.483.116.731.099l.889-.89-.079-.355.158-.553zM335.967-63.24l.455.454.276.08.099.098a1.398 1.398 0 001.185-1.501 1.281 1.281 0 00-1.244-1.245l-.593.593a1.34 1.34 0 00-.336 1.758l.158-.237zM356.571-30.548a1.371 1.371 0 00-1.028.83l-.019.257.059.454.889.889.375.059a1.343 1.343 0 001.205-1.205l-.079-.316-.474-.474a1.453 1.453 0 00-.928-.494zM378.221-32.84l-.396-.157-.217-.02-.099.099a1.535 1.535 0 00-1.185.75.937.937 0 00-.197.198 1.535 1.535 0 00-.534 1.007 1.875 1.875 0 00.85 1.837l.138.139.375.059.099-.099.257-.06.533-.532c.049-.044.095-.09.139-.139.344-.201.598-.526.711-.908l.395-.395a1.463 1.463 0 00-.435-1.067l-.434-.435v-.276zM411.07 7.083l-.632-.632-.336-.06-.256.02c-.362.09-.673.316-.869.632-.087.16-.137.336-.147.517-.011.18.019.362.087.53l.119.119a1.227 1.227 0 001.126.77l.118-.119.336-.138.672-.671a1.426 1.426 0 000-1.067l-.218.099zM393.747-4.039l.257-.256.059-.297.099-.098a1.46 1.46 0 00-.514-1.146 1.314 1.314 0 00-1.086-.297l-.83.83a1.283 1.283 0 00-.119 1.106l.297.297-.02-.02.257.257c.241.083.497.11.75.079l.297-.06.454-.454.099.06zM373.598-12.454l-.869.87-.059.256-.02.297.099.494.435.434c.103.145.23.272.375.375l.316.04.553-.079.672-.672c.186-.315.263-.683.217-1.047l-.909-.908a1.535 1.535 0 00-.81-.06zM374.941-2.26l.119.118a1.37 1.37 0 00.948-.593 1.455 1.455 0 00.197-1.225l-1.363-1.363-.415-.138-.256-.02-.237.04-.909.909-.059.414a1.407 1.407 0 00.612.968c.077.304.251.574.494.77l.118.12h.751zM353.193-29.343l.118-.119.04-.316a1.507 1.507 0 00-1.64-1.244l-.889.889-.098.454-.02.257.869.87c.26.158.566.228.869.197.191-.048.376-.114.553-.198l.455-.454-.257-.336zM374.092-25.945h-.237a1.398 1.398 0 00-1.185 1.185l-.04.276.119.119.059.257.553.553a1.62 1.62 0 001.047.177.615.615 0 00.257-.059l1.007-1.007a1.71 1.71 0 00-.474-1.225l-.296-.296-.81.02zM377.253-15.713l-.316.04-.514.237-.158.158c-.145.103-.271.23-.375.375-.166.077-.325.17-.474.277l-.514.513-.099.336-.118.118a1.79 1.79 0 00.612 1.324l.119.118.296.1a1.425 1.425 0 001.008.454l.098-.1.277-.078.533-.534c.261-.217.453-.505.553-.83l-.019-.296.059-.256.02-.336a1.511 1.511 0 00-.988-1.62zM375.455-20.908l-.929.928a1.375 1.375 0 00.878 1.354c.18.069.372.099.564.088l.573-.573a1.484 1.484 0 00.118-1.58 1.39 1.39 0 00-1.382-.672l.178.455zM369.312-27.526l.434.435.316.119.119.118.217-.02a1.396 1.396 0 001.323-1.323l-.118-.119-.119-.316-.671-.671a1.901 1.901 0 00-.751-.08l-.276.08-.771.77-.079.316.04.237.118.119.218.335zM394.498 19.31a1.337 1.337 0 00-.968.612 1.61 1.61 0 00-.158 1.344l.494.494c.32.228.715.326 1.106.276a1.453 1.453 0 00.987-.632l.317-.316.019-.257a1.505 1.505 0 00-.533-1.165 1.342 1.342 0 00-1.264-.356zM375.455-50.657l-.355-.119-.099-.098-.119.118a1.458 1.458 0 00-1.066.632 1.47 1.47 0 00-.178 1.166l.336.336a1.23 1.23 0 001.303.71c.174-.029.342-.089.494-.177l.771-.77c.07-.21.09-.433.059-.652l-1.146-1.146zM370.674-54.924l.435.435.336.099.118.118.099-.099a1.761 1.761 0 001.462-1.185l.099-.099v-.316l-.672-.671a1.537 1.537 0 00-1.047-.297 1.794 1.794 0 00-.83.395l-.553.553-.098.376-.119.118.257.257.513.316zM379.899-24.089l-.257.02a1.373 1.373 0 00-1.027.632 1.543 1.543 0 00-.158 1.225l.731.73.277.08.098.099h.277a1.482 1.482 0 001.027-.633 1.51 1.51 0 00.138-1.244l-.928-.928-.178.02zM383.988-55.28a1.2 1.2 0 00-.474 1.265l.909.909.375.098a1.338 1.338 0 001.027-.75 1.4 1.4 0 00-.107-1.39 1.392 1.392 0 00-1.256-.605l-.474.474zM397.658-78.628l.572.573c.156.04.319.04.475 0a1.398 1.398 0 001.086-1.087c.076-.304.06-.624-.046-.919a1.584 1.584 0 00-.547-.74l-.276-.276-.257-.06-.316.04a1.257 1.257 0 00-1.067.75 1.566 1.566 0 00.119 1.541l.257.178zM379.149-10.3l.573.572.296.1h.316a1.37 1.37 0 001.047-.732 1.562 1.562 0 00-.04-1.501l-.533-.533-.277-.08-.316-.039-.256.06-.455.217-.178.177a1.56 1.56 0 00-.177 1.758zM426.597-39.18l-.474.474a1.27 1.27 0 00-.395.395.985.985 0 00-.059.257l.158.434.79.79.395.12c.332-.062.632-.236.85-.495a1.376 1.376 0 00.088-1.481 1.367 1.367 0 00-1.313-.692l-.04.198zM400.028-53.225a1.508 1.508 0 00-1.363 1.363l.02.257.138.415.573.572.435.159.256.02.317-.04.908-.909.04-.355-.257-.06.257-.02-.02-.493-.573-.573-.435-.237-.276-.04-.02-.059zM374.309-36.948h.198l.237-.04.849-.849.079-.355.02-.218-.04-.237-.81-.81-.355-.118-.119-.119a1.478 1.478 0 00-.987.593c-.203.33-.274.726-.198 1.106l.83.83.296.217zM382.408-66.065a1.287 1.287 0 00-.968 1.205l.079.276.336.336a1.262 1.262 0 001.264.593c.167-.023.328-.076.475-.158l.711-.711.039-.316a1.285 1.285 0 00-1.205-1.205l-.237.04-.494-.06zM331.425-5.402l-.435-.434-.336-.099h-.197l-.316.04-.81.81-.119.355-.118.118a1.474 1.474 0 00.987 1.304l.297.02.513-.08.613-.611.058-.257c.084-.262.105-.54.06-.81l-.138-.138-.059-.218zM399.712 8.011h-.316l-.533.138-.613.613-.098.415-.139.138a1.495 1.495 0 00.455 1.086 1.337 1.337 0 001.007.297l.553-.553a1.404 1.404 0 00.277-1.265 1.568 1.568 0 00-.593-.869zM382.823 5.858l-.454-.454-.099-.297a1.534 1.534 0 00-1.501-.948l-1.067 1.067a1.703 1.703 0 00.336 1.363c.179.205.411.355.671.434l.218.218a1.286 1.286 0 001.284.612.76.76 0 00.316-.08l.75-.75.06-.296.098-.099-.039-.237-.138-.138-.435-.395zM334.565-25.294l.316-.04.79-.79.139-.335.098-.099a1.36 1.36 0 00-.493-1.047 1.424 1.424 0 00-1.146-.316l-.929.929-.059.335v.435l.869.87.356.078.059-.02zM353.667-77.62l.158-.159.415-.217.375-.375c.131-.203.212-.433.237-.672v-.197l-.098-.336-.81-.81a1.926 1.926 0 00-.474-.04 1.453 1.453 0 00-1.245 1.521 1.424 1.424 0 001.442 1.284zM371.741-62.035a1.287 1.287 0 00.988.632l.217-.02.395-.158.573-.573a1.344 1.344 0 00.099-1.046l-.83-.83-.395-.04a1.287 1.287 0 00-1.225 1.225l.119.118.059.692zM396.117-66.104l.415.415c.176.136.391.212.613.217l.256-.06.415-.177.553-.553.099-.376-.099-.098a1.338 1.338 0 00-1.481-1.205.876.876 0 00-.198.197l-.75.75-.06.297v.198l.119.118.118.277zM358.309-21.224l.573-.573a1.368 1.368 0 00-.217-1.363 1.346 1.346 0 00-1.205-.494 1.197 1.197 0 00-.889.612l-.257.257-.079.277-.099.098c.033.375.21.722.494.968.309.221.689.32 1.067.277l.612-.06zM362.141 11.883l.296-.296.138-.415a2.11 2.11 0 00.079-.712l-.118-.316-.632-.632-.336-.098-.316.04-.553.236a1.596 1.596 0 00-.652 1.6l.83.83.355.079.217.02.317-.04.375-.375v.079zM349.716-55.615l.415.415c.188.111.396.185.612.217l.119-.119.395-.158.454-.454.316.08.217-.02 1.107-1.107a1.376 1.376 0 00-.297-1.205 1.334 1.334 0 00-1.145-.395 1.315 1.315 0 00-.929.533l-.296-.059h-.198a1.4 1.4 0 00-1.145 1.146v.237l.079.434.296.455zM354.003-52.949c.01.426.187.83.494 1.127a1.537 1.537 0 001.145.355l.632-.632c.215-.407.265-.881.139-1.324l-.79-.79-.316-.079h-.198l-.395.04-.652.652-.118.355.059.296zM387.643-37.481a1.481 1.481 0 00-1.185 1.659 1.366 1.366 0 001.501 1.146l.77-.77.119-.396-.02-.217-.751-.75-.434-.672zM394.083.88l-.139-.336-.237-.237-.513-.276-.316-.08-.237.04-.929.928-.059.376v.237a1.286 1.286 0 001.422 1.067 1.228 1.228 0 00.948-.949l.099-.533-.039-.237zM370.437-43.309l.554-.553c.179-.22.288-.49.312-.773a1.422 1.422 0 00-1.596-1.538 2.78 2.78 0 00-.435.119l-.652.651a1.484 1.484 0 00.217 1.324 1.54 1.54 0 001.6.77zM363.662-57.433l.435.435.335.138.119.119.257-.02a1.513 1.513 0 001.255-1.127c.049-.193.058-.394.029-.592l-.06-.256-.158-.395-.513-.514-.336-.099-.099-.099h-.316l-.928.929c-.096.333-.136.68-.119 1.027l.138.138-.039.316zM374.368-1.965l-.849-.849h-.356a1.478 1.478 0 00-1.185 1.185l-.099.1.02.453.988.988c.239.092.497.12.75.08l.297-.06.533-.534a1.602 1.602 0 00.138-1.284l-.237-.079zM368.423-40.78l-.119-.119-.158-.355-.435-.435a1.338 1.338 0 00-.691-.296l-.257.02-.395.158-.533.533-.138.415-.02.217.039.237.909.909.415.059a1.255 1.255 0 001.383-1.343zM379.406-4.039a1.37 1.37 0 00-.988-.276l-.711.71a1.178 1.178 0 00-.178.692l-.651.652-.06.296a1.477 1.477 0 00.514 1.067l.434.435.218.02.197-.198a1.287 1.287 0 001.067-1.067l.02-.257-.079-.276a1.258 1.258 0 00.316-1.067c.034-.248 0-.5-.099-.73zM341.657-47.516l-.198-.198-.158-.355a1.166 1.166 0 00-1.126-.573l-.118.118-.336.139-.652.651-.098.376-.02.217c.025.327.165.635.395.87l.039.039a1.286 1.286 0 001.403.889l.513-.514a1.562 1.562 0 00.356-1.66zM359.613-67.487l.513.514c.183.103.384.17.593.197l.217-.02.395-.118.593-.593a1.406 1.406 0 00.099-1.086l-.791-.79-.316-.08a1.426 1.426 0 00-1.185 1.186v.237l.099.415-.217.138zM312.757-3.21l-.276.04-.711.712-.099.296a1.447 1.447 0 00.474 1.185 1.423 1.423 0 001.165.336l.257-.06 1.008-1.007a1.51 1.51 0 00-.475-1.146 1.447 1.447 0 00-1.343-.355zM396.236 6.51l-.119-.119a1.256 1.256 0 00-1.066-.632l-.218.02-.355.158-.553.553a1.276 1.276 0 00-.099 1.047l.83.83.375.059a1.31 1.31 0 001.205-1.205l-.099-.099.099-.612zM354.556-.127l-.297.02-.454.177-.593.593-.098.375v.237a1.276 1.276 0 00.486.885 1.277 1.277 0 00.975.26.752.752 0 00.316-.078l.731-.731.079-.316a1.396 1.396 0 00-1.145-1.422zM395.88-43.467a1.56 1.56 0 00-.356.356 1.232 1.232 0 00-.256 1.442 1.312 1.312 0 001.343.671 1.282 1.282 0 001.007-.77l.119-.119.039-.237-.019-.217-.968-.968h-.356l-.118.119-.435-.277zM395.347-24.76a2.317 2.317 0 00-.395.118l-.672.672a1.4 1.4 0 00.773 1.754c.217.092.454.127.689.103l.553-.553a1.475 1.475 0 00.119-1.62 1.423 1.423 0 00-1.067-.474zM362.497 3.507l-.316.316a1.535 1.535 0 00-.02 1.087l.869.869c.105.02.212.02.316 0a1.367 1.367 0 001.174-.97 1.368 1.368 0 00-1.45-1.756l-.474.474-.099-.02zM417.056-.918l-.711-.71-.257.019a1.336 1.336 0 00-1.158.956 1.37 1.37 0 001.415 1.75l.928-.928c.085-.268.112-.55.079-.83l-.118-.118-.178-.139zM347.484-16.128l.513.514.415.099.217.02a1.48 1.48 0 001.186-.988l.019-.257-.059-.454-.553-.554-.395-.197-.138-.138h-.237a1.398 1.398 0 00-1.324 1.323l.158.158.198.474zM327.592-15.95l-.474.474-.099.375-.119.119.04.316.928.928.376.1a1.288 1.288 0 001.128-.976 1.28 1.28 0 00.017-.526 1.367 1.367 0 00-1.56-1.205l-.237.395zM344.955 3.507l-.869.87-.079.355-.02.217.613.613c.192.154.426.25.671.276l.119.119.257.02.237-.04.494-.494-.079.079.336-.336.019-.375-.118-.119a1.456 1.456 0 00-1.581-1.185zM360.323-38.568l-.059.336-.099.099.04.316.79.79.336.138.099.099.217-.02a1.347 1.347 0 001.007-.612 1.367 1.367 0 00.198-1.146l-.83-.83a1.313 1.313 0 00-.731-.098l-.296.059-.652.652-.02.217zM351.81 11.033l-.296-.296-.237-.04-.455.06-.987.987-.079.435.019.257c.108.299.292.564.534.77l.257-.257-.237.237.493.218.297.02.316-.04.77-.77a2.017 2.017 0 00-.257-1.679l-.138.098zM317.616-32.365l-.533-.533a1.424 1.424 0 00-.77-.139 1.23 1.23 0 00-1.047 1.324l.276.276a1.284 1.284 0 001.225.948l.118-.118.376-.138.434-.435a1.7 1.7 0 00.316-.711l-.059-.296-.336-.178zM344.54-12.829l-.691-.691-.336-.06-.099-.098a1.452 1.452 0 00-1.244 1.64l.889.888.375.099.217.02.791-.79a1.7 1.7 0 00.138-1.324l-.04.316zM338.951-30.35l.454.454.375.099.218.02.237-.04.849-.85.099-.375v-.197l-.81-.81a1.509 1.509 0 00-.909-.158l-.395.632-.513.513.256.257.139.454zM407.476-38.311l.138-.138.336-.139.474-.474.158-.355.118-.119a1.42 1.42 0 00-1.442-1.442l-.118.119-.376.138-.454.454-.138.376-.119.118a1.482 1.482 0 00.415 1.087 1.397 1.397 0 001.008.375zM441.926-39.457a1.428 1.428 0 00-1.146-.316.618.618 0 00-.257.06l-.77.77a1.67 1.67 0 00.316 1.501 1.37 1.37 0 001.146.434c.314-.054.599-.214.81-.454l.296-.296.158-.356.118-.118a1.565 1.565 0 00-.671-1.225zM413.264-21.659l-.277.08-.415.414-.237.474-.158.158a1.425 1.425 0 001.185 1.185l.376.06.276-.04.81-.81.079-.276a1.428 1.428 0 00-1.185-1.423l-.138.139-.316.04zM397.302-34.025l.376.1a1.257 1.257 0 001.007-.771c.093-.232.127-.483.099-.731a1.372 1.372 0 00-.969-1.174 1.366 1.366 0 00-.532-.05l-.474.473c-.152.096-.28.224-.376.376l-.138.138c-.039.182-.039.37 0 .553l.632.632.375.454zM400.897-12.928l-.197-.197a1.321 1.321 0 00-.415-.415l-.257-.06-.494.06-.592.592-.198.475-.039.276.039.316.692.692.355.118.217.02a1.427 1.427 0 001.166-1.166l-.02-.256-.138-.455h-.119zM398.883-30.666l-.336-.336a.923.923 0 00-.455.177l-.355-.079-.217-.02a1.402 1.402 0 00-.988.672 1.257 1.257 0 00-.099 1.126l.751.75.316.04.099.1.118-.12.356.356.237.04a1.258 1.258 0 001.126-1.403 1.21 1.21 0 00-.336-.691l-.217-.613zM380.433-49.827a1.251 1.251 0 001.244.572.95.95 0 00.474-.237l.553-.553a1.389 1.389 0 00-.355-1.185 1.569 1.569 0 00-1.265-.435 1.256 1.256 0 00-1.086 1.285l.059.256.376.297zM400.206-31.002l.336-.336a1.343 1.343 0 00.059-1.284 1.31 1.31 0 00-1.067-.711l-.217.02-.435.158-.533.533-.099.375-.118.119.039.395.613.612.474.198.276.04.336-.337.336.218zM428.157-23.002a1.324 1.324 0 00-.296.099l-.336.335a1.376 1.376 0 00-.79 1.107 1.286 1.286 0 00.83 1.382l.237.04.474-.08 1.126-1.125.099-.375.059-.218a1.343 1.343 0 00-1.403-1.165zM413.757-13.935l-.553-.553a1.258 1.258 0 00-1.165-.415 1.152 1.152 0 00-.356.079l-.751.75-.019.218.098.533.613.613.395.158.217.02.257-.02a1.257 1.257 0 00.909-.553 1.23 1.23 0 00.355-.83zM406.685-53.126l.475.474.375.138.138.138a1.394 1.394 0 00.968-.573 1.556 1.556 0 00.178-1.244l-.751-.75-.336-.1-.217-.02a1.398 1.398 0 00-1.205 1.482l.119.119.256.336zM409.609-26.696l.079-.277.099-.098a1.46 1.46 0 00-.514-1.146 1.372 1.372 0 00-1.086-.296l-.751.75a1.318 1.318 0 00-.099 1.087l.79.79c.239.092.498.12.751.079l.296-.06.593-.592-.158-.237zM420.79-29.422a1.457 1.457 0 00.059-1.244l-.751-.751-.296-.06a1.42 1.42 0 00-1.165 1.166v.277l.632.632c.296.216.662.315 1.027.276.186-.058.355-.16.494-.296zM385.154-41.748l.395.118a1.46 1.46 0 00.968-.612c.168-.264.244-.577.217-.889.045-.114.072-.234.079-.355l-.119-.12-.138-.374-.533-.534a1.064 1.064 0 00-.573-.217l-.217.02-.376.138-.533.533c-.114.193-.181.41-.198.633l.099.098-.118.119a1.481 1.481 0 00.414 1.086l.139.139.494.217zM386.477-29.086l.533.533c.291.193.641.277.988.237a1.4 1.4 0 001.106-.908 1.484 1.484 0 00-.276-1.502l-.514-.513-.257.02a1.451 1.451 0 00-1.027.632 1.525 1.525 0 00-.201.607c-.024.215-.003.431.063.637l-.415.257zM389.342-18.597a1.373 1.373 0 00.099-1.284l-.494-.494a1.567 1.567 0 00-.929-.217c-.22.03-.427.127-.592.276l-.415.415-.138.336-.099.099a1.359 1.359 0 00.435 1.146 1.393 1.393 0 001.106.355 1.313 1.313 0 001.027-.632zM401.312-2.913l-.671-.671-.316-.08h-.198l-.316.04-.849.85-.099.375.099.099.039.237.948.948.415.02.277-.04a1.341 1.341 0 00.987-1.699l-.316-.079zM409.49-45.857a1.374 1.374 0 00.969 1.174c.173.052.354.07.533.05l.731-.73a1.373 1.373 0 00-.395-1.896l-.356-.356a.56.56 0 00-.356 0 1.375 1.375 0 00-.97.618 1.37 1.37 0 00-.156 1.14zM384.719-19.9a1.478 1.478 0 00.158-1.62 1.367 1.367 0 00-1.323-.731l-.514.513a1.307 1.307 0 00-.296 1.087 1.634 1.634 0 00.691 1.047l.139.138.335.06.237-.04.356-.356.217-.099zM390.073-48.208l.612.613.277.079.217-.02a1.45 1.45 0 001.067-.632 1.485 1.485 0 00.178-1.166l-.83-.83a1.48 1.48 0 00-.711-.078l-.297.059-.691.691a1.426 1.426 0 000 1.067l.178.217zM386.912-23.002l.355-.119.534-.533c.113-.193.181-.409.197-.632l-.118-.119-.119-.355-.533-.534a1.034 1.034 0 00-.612-.217l-.119.119-.375.098-.514.514a1 1 0 00-.217.652l.118.118.119.356.513.514c.188.111.396.185.613.217l.158-.08zM390.251-24.326l.138.139.375.138.119.119.276-.04a1.343 1.343 0 001.008-.613 1.492 1.492 0 00.158-1.264l-.771-.77a1.702 1.702 0 00-.632-.04 1.422 1.422 0 00-1.106 1.107l-.02.217.06.415.335.335.06.257zM385.727-25.728l.375-.06.711-.71a1.479 1.479 0 00.139-1.6 1.345 1.345 0 00-1.324-.731l-1.126 1.126a1.692 1.692 0 00-.217.612l.02.217.138.375.533.534c.183.104.384.17.593.197l.158.04zM383.929-34.716l.375-.138.534-.534c.136-.175.212-.39.217-.612l-.118-.118-.119-.396-.533-.533a1.594 1.594 0 00-.573-.178l-.217.02-.395.119-.514.513a1.59 1.59 0 00-.217.652l.118.118.119.396.533.533a1.9 1.9 0 00.593.198l.197-.04zM393.747-15.89l-.672-.672-.316-.08-.099-.098-.395.04-.355.355a1.502 1.502 0 00-1.069.579 1.515 1.515 0 00-.294 1.179l1.047 1.047h.474a1.513 1.513 0 001.007-.573 1.374 1.374 0 00.9-.63 1.368 1.368 0 00.147-1.088l-.375-.06zM397.697-18.577l.138.138h.198a1.258 1.258 0 00.948-.672 1.395 1.395 0 00-.059-1.52 1.26 1.26 0 00-1.284-.574 1.232 1.232 0 00-.869.514c-.203.33-.274.726-.198 1.106l.119.119a1.146 1.146 0 001.007.889zM401.984-42.854l.316-.04.178-.06.336-.177.158-.158.158-.197.079-.198.099-.336.019-.217-.039-.158-.06-.178-.177-.336-.158-.158-.198-.158-.178-.099-.375-.098h-.197l-.317.04-.197.078-.336.178-.138.138-.158.198-.079.197-.099.376-.02.178.02.177.079.198.198.316.118.158.198.158.197.079.375.099h.198zM417.945-9.51l-.356-.356a1.453 1.453 0 00-.593-.198l-.118.119-.375.138-.593.593a1.39 1.39 0 00-.099 1.086l.771.77.316.08h.197l.217-.02.85-.85.099-.375.098-.099-.217-.217-.197-.671zM405.204-1.826l-.849.85-.079.355-.099.098a1.435 1.435 0 00.416 1.066 1.426 1.426 0 001.065.416l1.087-1.087a1.397 1.397 0 00-.9-1.647 1.338 1.338 0 00-.641-.051zM426.103 11.014a1.43 1.43 0 00-.987.632 1.466 1.466 0 00-.119 1.224l.79.79c.205.09.43.125.652.1.345-.055.655-.239.869-.514l.297-.297a1.322 1.322 0 00-.435-1.106 1.51 1.51 0 00-1.067-.83zM412.592-39.081l-1.008 1.007a.908.908 0 000 .395l.138.138.158.356.573.573.415.138.217.02.929-.929a1.422 1.422 0 00-.257-1.363 1.592 1.592 0 00-1.165-.335zM424.128-10.004a.87.87 0 00-.297.099l-.632.632-.059.336-.099.098.771.77c.175.137.39.213.612.218l.257-.06.434-.197.553-.553.079-.395a1.303 1.303 0 00-.61-.815 1.315 1.315 0 00-1.009-.133zM393.055-28.04l.968-.967-.02-.731-.098-.099-.02-.257-.869-.869-.356-.079-.099-.099a1.34 1.34 0 00-.967.613.864.864 0 00-.178.177l-.415.415-.198.435-.059.257a2.699 2.699 0 00-.197.197l-.04.237.04.474.592.593.435.198.296.059a1.45 1.45 0 001.067-.711l.118.158zM404.513-16.543l-.435-.434-.336-.099h-.197a1.256 1.256 0 00-.948.553 1.306 1.306 0 00-.218 1.126l.632.632.356.08.099.098a1.287 1.287 0 00.968-.573 1.318 1.318 0 00.217-1.126l-.119-.118-.019-.139zM413.777-67.685l.158-.158.039-.237a1.35 1.35 0 00-.474-.987 1.404 1.404 0 00-1.067-.277.974.974 0 00-.296.06l-.77.77-.04.276.04.435.612.612.454.178.277.04a1.285 1.285 0 001.067-.712zM340.471 1.986l.494-.494a1.282 1.282 0 00-.119-1.303 1.616 1.616 0 00-.513-.514l-.138-.138-.395-.119-.218-.02-.237.04a1.673 1.673 0 00-1.165.691l-.297.297-.019.217c.004.308.108.606.296.85l.79.79.277.04a.418.418 0 00.256-.06l.139-.139.849-.138zM346.022 26.145l-.632-.632-.316-.04a1.395 1.395 0 00-1.166 1.166l.02.296.612.612c.294.222.662.32 1.028.277a1.253 1.253 0 00.948-.632 1.372 1.372 0 00.059-1.245l-.553.198zM366.408-20.316l.276.04.494-.06.612-.612.158-.474.04-.276c.029-.378.109-.75.237-1.107v-.236l-.059-.455-.553-.553-.435-.198-.257-.059a1.394 1.394 0 00-1.007.692l-.158.158-.198.434-.474.474-.118.316v.356l-.119.119a1.314 1.314 0 001.561 1.441zM365.183-25.333l.494-.494a1.506 1.506 0 00-.633-1.58l-.316-.119-.572.02-.257.06-.218.02-.414.137-.534.534-.138.415-.02.256.04.316.928.929.296.098.573-.02.731-.73.04.158zM366.349-13.935l-.632-.632-.435-.119h-.237a1.34 1.34 0 00-.908.672 1.31 1.31 0 00-.06 1.086l.81.81.356.04a1.313 1.313 0 001.205-1.205l-.139-.139.04-.513zM370.477 6.273l-.692.691a1.084 1.084 0 00-.098 1.047 1.306 1.306 0 00.73.73l.159.159.395.079a1.233 1.233 0 00.849-.494c.191-.269.27-.602.217-.928a1.415 1.415 0 00-.513-.967 1.422 1.422 0 00-1.047-.317z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M417.096-2.103l-.356-.118a1.314 1.314 0 00-.889.454 1.374 1.374 0 00-.296 1.047 1.34 1.34 0 001.323 1.324l.771-.771.118-.395-.217-.217-.178-.376-.296-.296.02-.652zM418.083 50.56l-.751.75-.118.396c.002.156.036.31.098.454l.771.77.395.12a1.34 1.34 0 00.889-.495c.109-.13.19-.282.238-.445.047-.163.06-.334.038-.503a1.367 1.367 0 00-1.56-1.047zM407.732-9.333l.118.356.277.277a1.369 1.369 0 001.165.533l.415-.099.711-.711.06-.296-.04-.316-.79-.79-.316-.12-.119-.118a.333.333 0 00-.197 0 1.34 1.34 0 00-1.324 1.324l.04-.04zM420.612 14.174l.81.81c.149.036.303.042.454.02a1.288 1.288 0 00.948-.514 1.501 1.501 0 00.218-1.244l-.949-.948a1.311 1.311 0 00-1.58 1.422l-.099.099.198.355zM418.321-12.493l.513.513.316.08.376.02a1.286 1.286 0 001.007-.653 1.446 1.446 0 00.02-1.323l-.751-.75-.257-.06a1.45 1.45 0 00-1.264.948l-.04.237.02.454.158.159-.098.375zM422.527 26.678l-.256-.02a1.482 1.482 0 00-1.047-.375 1.843 1.843 0 00-1.166.692l-.118.118-.079.356-.099.098.02.257.474.474a1.48 1.48 0 001.047.376c.22.247.521.408.849.454l.119-.119.395-.118.888-.89a1.762 1.762 0 00-.651-1.363l-.376.06zM415.693 40.23l-.395-.396h-.277a1.342 1.342 0 00-1.185.988 1.476 1.476 0 00.553 1.62l.375.375.297.02a1.339 1.339 0 001.185-.988 1.48 1.48 0 00-.553-1.62zM432.306-30.666l.118-.119.336-.138a1.225 1.225 0 00.79-.79l.099-.1-.119-.118a1.81 1.81 0 00-.632-.632l-.118-.118-.356-.08h-.197l-.237.04-.494.494a1.642 1.642 0 00-.217 1.6c.334.12.702.107 1.027-.04zM401.886 20.06l-.672-.67-.296-.06-.099-.099a1.478 1.478 0 00-1.284 1.086v.238l.099.414.592.593c.166.127.365.203.573.217l.217-.02.396-.118.572-.573a1.334 1.334 0 00.158-1.067l-.256.06zM400.661 5.364a1.405 1.405 0 00-.988.553 1.476 1.476 0 00-.178 1.205l.83.83.336.098.849-.849.099-.375.099-.099-.02-.257-.889-.889-.336-.099.198-.118zM423.614 19.152l-.119.118-.355.119-.415.415a1.48 1.48 0 00-.375.73l.019.218.08.356.711.71c.226.113.479.161.731.139l.296-.06.711-.71.079-.277.099-.099a1.453 1.453 0 00-.356-1.106 1.47 1.47 0 00-1.106-.553zM419.505-49.274l.474.474c.241.153.528.216.81.177a.977.977 0 00.494-.138l.356-.355a1.311 1.311 0 00.118-1.58l-.849-.85h-.316a1.426 1.426 0 00-1.284 1.086l-.02.257.079.553.138.376zM399.475 13.463l.02-.257-.02-.454-1.007-1.008-.257.02a1.37 1.37 0 00-1.304 1.304l.573.573a1.343 1.343 0 002.193-.652l-.198.474zM396.71-17.63c.125-.257.166-.547.118-.829l-.849-.85-.356-.078-.217-.02a1.541 1.541 0 00-1.225.948l-.039.237.059.455.573.572c.198.184.445.307.711.356l.257-.02.434-.158.672-.671-.138.059zM400.681 15.735c-.13.012-.257.045-.376.099l-.395.395a1.37 1.37 0 00-.296 1.323 1.458 1.458 0 001.047 1.047h.276l.494-.494a1.512 1.512 0 00.218-1.64 1.312 1.312 0 00-.968-.73zM387.05-19.308l.83.83.316.078a1.48 1.48 0 00.987-.513 1.48 1.48 0 00.257-1.126l-.889-.89-.375-.019a1.315 1.315 0 00-1.205 1.205l.039.237.04.198zM401.53-6.37l-.079.356-.119.119.553.553a1.45 1.45 0 001.166.454 1.453 1.453 0 001.264-1.501l-.158-.158-.178-.376-.513-.513a1.259 1.259 0 00-.751-.119L401.53-6.37zM408.246 10.46a1.42 1.42 0 00.139-.493c.133-.21.239-.436.316-.672.209-.112.381-.284.493-.494l.652.652.356.119c.15.02.302.026.454.02a1.228 1.228 0 001.126-1.127v-.197l-.059-.375-.652-.652-.375-.139-.237-.04-.771.771-.355-.355a1.344 1.344 0 00-1.245-.573l-.336-.099-.217.02-.257.257a1.646 1.646 0 00-.257.256l-.493.494c-.142.19-.249.405-.316.632l-.336.336c-.247.21-.402.508-.435.83l.04.276.987.988.435.04.257-.02c.185-.065.358-.158.513-.277a1.34 1.34 0 00.573-.178zM367.158 29.66l-.711.712a1.065 1.065 0 00-.138 1.007 1.198 1.198 0 00.731.731l.138.139.375.138a1.263 1.263 0 00.864-.45 1.26 1.26 0 00.282-.933c-.022-.299-.14-.583-.336-.81a1.315 1.315 0 00-1.205-.533zM394.932 16.347l.256-.257a1.285 1.285 0 00.751-1.501l-.494-.494a1.425 1.425 0 00-1.007-.454l-.217.02-.415.414a1.562 1.562 0 00-.218 1.719 1.375 1.375 0 001.324.73l.02-.177zM394.932 6.43l-.296-.296-.395-.118a1.366 1.366 0 00-1.211 1.006c-.049.18-.06.37-.034.554.027.388.204.75.494 1.008.29.233.657.346 1.027.316l.435-.435a1.393 1.393 0 00.553-.987 1.418 1.418 0 00-.474-1.146l-.099.099zM392.72 11.883l-.534-.534-.296-.098-.099-.1a1.785 1.785 0 00-.671.238l-.534.533a1.454 1.454 0 00.158 1.383 1.286 1.286 0 001.166.533 1.118 1.118 0 00.849-.454.93.93 0 00.277-.277 1.373 1.373 0 00.118-1.185l-.434-.04zM389.677 17.73l.079-.553-.02-.297-.434-.434-.494-.297-.277-.079-.098.1a1.398 1.398 0 00-1.383 1.382 1.313 1.313 0 00.988 1.62l.296-.02.691-.692c.125-.213.212-.447.257-.69l.395-.04zM401.807 42.224a1.372 1.372 0 00-1.087.731 1.289 1.289 0 00.079 1.304l.514.514.257.059.217-.02.889-.889a1.286 1.286 0 00-.178-1.284 1.345 1.345 0 00-.691-.415zM370.556-18.143l.375-.098.514-.514c.148-.187.237-.414.256-.652l-.019-.217-.119-.356-.514-.513a1.448 1.448 0 00-.632-.237l-.217.02-.355.118-.514.514c-.12.193-.2.407-.237.632v.197l.138.375.514.514c.177.122.38.203.592.237l.218-.02zM385.885-13.718l.395.395c.178.122.38.203.593.237l.276-.04.455-.177.553-.553.098-.376.02-.217a1.309 1.309 0 00-.924-1.154 1.317 1.317 0 00-.518-.05l-.296.058-.751.751-.079.316.119.119.059.691zM377.47 1.137l-.218.217-.138-.138-.869.869-.099.375a1.314 1.314 0 002.608.158 1.325 1.325 0 00-.749-1.36 1.312 1.312 0 00-.535-.121zM386.794 1.176l.454.455.316.118.119.119.256.02a1.537 1.537 0 001.087-.652 1.478 1.478 0 00.079-1.264l-.83-.83-.375-.1-.119-.118-.256.02-.85.85-.118.355v.435l.019.257.119.118.099.217zM418.004-7.87l.118.118a1.374 1.374 0 001.324.691 1.486 1.486 0 001.228-1.058 1.477 1.477 0 00-.478-1.55l-.138-.138-.395-.118h-.197l-.277.04-.237.236a1.452 1.452 0 00-.691 1.442l-.257.336zM362.299-33.59l-.039-.276-.218-.415-.316-.316a1.366 1.366 0 00-1.165-.494l-.949.948a1.35 1.35 0 00-.059 1.047l.909.908c.201.046.406.066.612.06a1.37 1.37 0 001.225-1.462zM464.504-20.79a1.31 1.31 0 001.126-.928v-.316l-.138-.375a1.23 1.23 0 00-.435-.79 1.34 1.34 0 00-.967-.297l-.336.336a1.286 1.286 0 00-.553 1.54 1.475 1.475 0 001.303.83zM403.821 64.467a1.394 1.394 0 00-1.501-1.146c-.17.026-.335.072-.494.138l-.434.435a1.287 1.287 0 00-.158 1.304l.395.395.434.237.099-.099-.059.336c.228.08.469.12.711.118l.889-.889-.04-.355.158-.474zM375.83-40.01l.692.691a1.453 1.453 0 001.363-1.363 1.286 1.286 0 00-1.245-1.244l-.098.099-.297.059-.296.296a1.366 1.366 0 00-.435 1.739l.316-.277zM394.971-6.409a1.257 1.257 0 00-1.066.75l-.04.238.04.474.987.987.375.06.099.098a1.316 1.316 0 001.186-1.185l-.099-.099-.079-.355-.455-.454a1.39 1.39 0 00-.948-.514zM416.7-7.752l-.395-.158-.138-.139-.099.1a1.57 1.57 0 00-1.225.67 1.247 1.247 0 00-.217.218c-.304.24-.509.585-.573.968a1.959 1.959 0 00.771 1.877l.138.138.355.079.218.02.197-.08.533-.533.139-.138c.356-.167.643-.453.81-.81l.414-.415a1.392 1.392 0 00-.395-1.067l-.434-.434-.099-.296zM447.714 33.612l-.633-.633-.316-.078-.237.04a1.312 1.312 0 00-.908.592 1.18 1.18 0 00-.119 1.027l.119.118a1.227 1.227 0 001.106.79h.197l.356-.118.672-.672a1.368 1.368 0 00.059-1.047l-.296-.02zM430.923 21.72l.553-.553a1.626 1.626 0 00-.454-1.166 1.344 1.344 0 00-1.087-.335l-.711.71a1.334 1.334 0 00-.178 1.087l.494.494c.229.105.48.153.731.138l.296-.059.336-.336.02.02zM411.15 12.377l-1.028 1.027-.039.276.079.514.415.415c.105.155.239.29.395.395l.296.06.573-.06.652-.652c.211-.305.303-.678.257-1.047l-.889-.889a1.285 1.285 0 00-.711-.04zM412.058 22.648l.119.119a1.422 1.422 0 001.25-1.17 1.43 1.43 0 00-.025-.608l-1.344-1.343-.415-.178-.217-.02-.257.02-.908.909-.099.415a1.458 1.458 0 00.573 1.007c.07.307.236.584.474.79l.178.178.671-.119zM391.554-5.402l.059-.336a1.48 1.48 0 00-1.58-1.303l-.138.138-.356.158-.257.257-.118.434-.02.257.02.218.731.73c.248.17.549.247.849.218.198-.006.394-.053.573-.139l.454-.454-.217-.178zM412.236-1.036l-.257.02a1.367 1.367 0 00-1.165 1.165l-.04.277v.197l.731.731c.304.18.657.256 1.008.217l.276-.04.731-.73a1.762 1.762 0 00-.434-1.264l-.474-.474-.376-.1zM414.962 9.315l-.336.02-.494.217-.158.158c-.145.103-.271.23-.375.375-.179.054-.347.14-.494.257l-.533.533-.119.356-.098.099a1.732 1.732 0 00.553 1.343l.118.118.296.1a1.32 1.32 0 00.988.473l.257-.059.553-.553c.254-.22.45-.498.573-.81v-.277l.079-.276.02-.336a1.486 1.486 0 00-.83-1.738zM413.441 4.04l-.711.712-.119.316a1.429 1.429 0 00.796 1.398c.171.084.357.132.548.142l.395-.395a1.397 1.397 0 00.177-1.56 1.287 1.287 0 00-1.323-.731l.237.119zM407.574-2.853l.435.434.336.138.099.1h.237a1.37 1.37 0 001.303-1.305l-.019-.217-.119-.316-.672-.672a1.56 1.56 0 00-.73-.098l-.297.059-.612.612-.079.316v.237l.118.119v.593zM430.607 45.069a1.289 1.289 0 00-.968.573 1.538 1.538 0 00-.237 1.343l.494.494c.31.221.686.333 1.067.316a1.5 1.5 0 001.066-.593l.296-.296.06-.257a1.651 1.651 0 00-.474-1.185 1.483 1.483 0 00-1.304-.395zM414.863-25.55l-.138-.139-.316-.118-.119-.119-.217.02a1.398 1.398 0 00-1.106.593 1.368 1.368 0 00-.198 1.145l.316.316a1.2 1.2 0 001.284.77c.173-.008.341-.055.494-.137l.77-.77a1.75 1.75 0 00.06-.653l-.672-.671-.158-.237zM410.181-30.113l.435.434.336.139.118.118.099-.099a1.76 1.76 0 001.521-1.126l.099-.098.02-.297-.672-.671a1.253 1.253 0 00-1.027-.316c-.309.03-.604.146-.85.336l-.474.474-.138.375-.02.217.04.237.118.119.395.158zM418.004 1.058a1.48 1.48 0 00-1.324.573 1.589 1.589 0 00-.217 1.244l1.007 1.008c.092.02.186.02.277 0a1.45 1.45 0 001.047-.573 1.472 1.472 0 00.197-1.225l-.908-.909-.079-.118zM423.673-30.054l-.138.138a1.116 1.116 0 00-.514 1.225l.079.276.613.613.395.118a1.393 1.393 0 001.047-.691c.139-.253.194-.543.158-.83a1.32 1.32 0 00-.385-.939 1.303 1.303 0 00-.939-.384l-.316.474zM438.804-52.1c.156.034.315.047.474.04a1.397 1.397 0 001.205-.968 1.592 1.592 0 00-.513-1.699l-.277-.276-.257-.06c-.105-.01-.21-.01-.316 0a1.425 1.425 0 00-1.106.712 1.617 1.617 0 00.079 1.54l.711.712zM416.622 14.806l.889.89h.316a1.344 1.344 0 001.086-.653 1.621 1.621 0 000-1.501l-.711-.711-.316-.04-.277.04-.434.197-.198.198a1.588 1.588 0 00-.355 1.58zM465.334-11.9l-.376.177-.197.198a.95.95 0 00-.395.395.616.616 0 00-.059.257l.138.454.869.87.375.098a1.257 1.257 0 001.166-1.403 1.341 1.341 0 00-1.521-1.047zM439.417-27.15a1.506 1.506 0 00-1.363 1.363l-.02.256.119.435.553.553.415.138.257.06.316-.04.908-.909.06-.335-.277-.08.296-.02-.019-.493-.06-.257-.75-.75-.257-.06-.178.139zM412.967-12.039l.218.02.217-.02.849-.85.099-.375.119-.118-.949-.948-.355-.119-.217-.02a1.31 1.31 0 00-1.008.534c-.116.16-.198.343-.243.537a1.498 1.498 0 00-.014.589l.81.81.474-.04zM422.429-40.82a1.258 1.258 0 00-1.087 1.087l.059.256.237.475a1.288 1.288 0 001.225.671c.181-.016.356-.07.514-.158l.711-.711.039-.316a1.288 1.288 0 00-1.106-1.304l-.257.02-.335-.02zM368.304 17.098l-.336-.1-.118-.118-.316.04-.81.81-.138.335-.119.12a1.51 1.51 0 00.948 1.342l.297.06.513-.08.79-.79c.104-.256.139-.536.099-.81l-.81-.81zM436.257 33.928l-.317-.04-.553.119-.632.632-.099.375-.118.119a1.31 1.31 0 00.851 1.39c.17.06.351.085.532.071l.533-.533a1.447 1.447 0 00.336-1.284 1.542 1.542 0 00-.533-.85zM419.565 31.103l-.652-.652a1.483 1.483 0 00-1.442-1.007l-.119.118-.316.119-.731.73a1.734 1.734 0 00.277 1.383c.13.232.322.423.553.553l.119.119a1.335 1.335 0 001.264.672.751.751 0 00.316-.08l.751-.75.059-.296.099-.1-.02-.216-.138-.139-.02-.454zM372.729-2.182l.336-.02.79-.79.138-.336.099-.098a1.309 1.309 0 00-.434-1.067 1.34 1.34 0 00-1.107-.356l-.217.02-.81.81-.099.336-.098.099.039.237.85.85.375.098.138.217zM394.162-53.6l.257-.06.434-.197.474-.474c.126-.2.213-.42.257-.652l-.099-.099-.098-.336-.81-.81a1.37 1.37 0 00-.494-.02 1.394 1.394 0 00-1.363 1.364 1.37 1.37 0 001.442 1.284zM411.544-37.205a1.35 1.35 0 00.968.692l.218-.02.395-.119.592-.592a1.362 1.362 0 00.139-1.047l-.83-.83-.375-.06a1.285 1.285 0 00-1.225 1.226l.118.118v.632zM436.059-40.188l.395.395a1.2 1.2 0 00.612.257l.277-.04.415-.177.553-.553.118-.356.02-.217a1.305 1.305 0 00-.454-.932 1.303 1.303 0 00-.988-.313l-.296.06-.751.75-.079.316-.099.1.119.118.158.592zM396.275 2.954l.652-.652a1.4 1.4 0 00-.138-1.363 1.259 1.259 0 00-1.185-.553 1.225 1.225 0 00-.929.573l-.237.237-.098.296-.099.099a1.372 1.372 0 00.454.968 1.2 1.2 0 001.027.316l.553.079zM398.626 36.2l.296-.297.138-.415c.096-.231.143-.48.138-.73l-.849-.85-.356-.08-.296.02-.553.198a1.567 1.567 0 00-.731 1.56l.83.83.355.119h.514l.375-.375.139.02zM389.243-31.773l.415.415c.17.135.376.217.593.237l.217-.02.395-.118.474-.474.297.099.217-.02 1.086-1.087a1.31 1.31 0 00-.257-1.205 1.223 1.223 0 00-1.126-.454 1.289 1.289 0 00-.928.494l-.316-.08-.099-.098a1.429 1.429 0 00-1.264 1.067v.237l.079.434.217.573zM393.45-28.908a1.576 1.576 0 00.455 1.165c.306.28.71.429 1.125.415l.692-.692a1.682 1.682 0 00.197-1.303l-.79-.79-.316-.08-.217-.02-.395.04-1.027 1.027.276.238zM426.34-11.98a1.453 1.453 0 00-1.264 1.62 1.326 1.326 0 00.467.906 1.317 1.317 0 00.975.3l.771-.771.118-.395-.099-.1v-.197l-.77-.77-.198-.593zM431.041 26.619l-.454-.454-.514-.316-.316-.08-.217.02-.928.929-.079.395.118.118a1.23 1.23 0 002.41.316l.099-.493-.119-.435zM409.392-18.577l.533-.533a1.51 1.51 0 00.178-1.64 1.424 1.424 0 00-1.383-.75c-.149.02-.295.06-.434.118l-.593.592a1.531 1.531 0 00.178 1.323 1.396 1.396 0 001.521.89zM403.248-32.978l.81.81s.099.099.237 0a1.536 1.536 0 001.363-1.679l-.039-.237-.139-.415-.513-.513-.356-.119-.316-.04-.928.93a3.01 3.01 0 00-.178 1.007l.118.118-.059.138z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M411.466 22.925l-.87-.87h-.355a1.484 1.484 0 00-1.185 1.186v.474l.829.83c.222.103.468.144.711.118.107-.004.212-.018.316-.04l.554-.552a1.597 1.597 0 00.177-1.284l-.177.138zM407.278-16.148l-.118-.118-.139-.336-.434-.435a1.285 1.285 0 00-.692-.335l-.217.02-.435.118-.533.533-.158.395-.02.257.02.217.909.91.395.078.099.099a1.315 1.315 0 001.323-1.403zM416.602 21.068a1.32 1.32 0 00-.988-.316l-.691.691a1.283 1.283 0 00-.218.692l-.671.671-.079.277a1.7 1.7 0 00.454 1.086l.454.455.218.02h.197a1.337 1.337 0 001.087-1.087v-.277l-.06-.256a1.285 1.285 0 00.376-1.047 1.4 1.4 0 00-.079-.909zM380.828-24.069l-.197-.197-.139-.336a1.2 1.2 0 00-1.106-.632l-1.007 1.007-.119.356-.02.217c.015.324.133.636.336.889l.099.099a1.366 1.366 0 001.363.968l.533-.534a1.588 1.588 0 00.257-1.837zM399.613-43.25l.415.415c.164.12.353.201.553.237l.218-.02.414-.098.593-.593a1.366 1.366 0 00.138-1.086l-.77-.77-.316-.08-.099-.098a1.479 1.479 0 00-1.284 1.086v.237l.099.415.039.356zM349.953 18.895l-.257.06-.889.888a1.418 1.418 0 00.408 1.173 1.423 1.423 0 001.172.408.981.981 0 00.158.197l1.028-1.027a1.52 1.52 0 00-.435-1.146 1.425 1.425 0 00-1.185-.553zM432.958 32.367l-.138-.138a1.26 1.26 0 00-1.028-.672l-.256-.06-.376.14-.553.552a1.344 1.344 0 00-.158 1.028l.85.849.375.06a1.343 1.343 0 001.205-1.206l-.099-.099.178-.454zM391.672 23.952h-.276l-.494.138-.593.593-.118.395-.02.217a1.326 1.326 0 00.467.907 1.317 1.317 0 00.975.298.67.67 0 00.296-.059l.731-.73.079-.317a1.396 1.396 0 00-1.047-1.442zM434.854-17.53a1.534 1.534 0 00-.336.335 1.284 1.284 0 00-.336 1.442 1.307 1.307 0 001.304.751 1.374 1.374 0 001.047-.73.391.391 0 01.118-.12l.04-.236-.02-.218-.968-.968-.356-.04-.118.12-.375-.336zM433.451 1.097a1.602 1.602 0 00-.415.1l-.592.592a1.531 1.531 0 00.178 1.324 1.452 1.452 0 001.224.592l.553-.553A1.423 1.423 0 00433.194.8l.257.296zM399.376 27.863l-.316.316a1.397 1.397 0 00-.079 1.067l.83.83.296.02a1.37 1.37 0 001.225-1.462 1.321 1.321 0 00-.384-.94 1.324 1.324 0 00-.94-.384l-.315.119-.297.296-.02.138zM454.094 25.888l-.731-.73h-.277a1.37 1.37 0 00.139 2.705l.237-.04.79-.79c.119-.245.16-.52.118-.79l-.118-.118-.158-.237zM385.233 7.557l.533.533.395.119h.238a1.422 1.422 0 001.205-.929l.039-.237-.039-.474-.85-.85-.237-.039h-.237a1.367 1.367 0 00-1.304 1.304l.04.237.217.336zM365.361 6.806l-.494.494-.098.375-.119.119.02.336.928.928.356.119a1.317 1.317 0 00.928-.494 1.277 1.277 0 00.297-.968 1.369 1.369 0 00-1.502-1.265l-.316.356zM381.875 27.073l-.869.87-.119.355-.02.217.04.237.454.455c.172.169.393.28.632.316l.119.118.276.04h.237l1.028-1.027.019-.376-.019-.217a1.483 1.483 0 00-1.778-.988zM399.081-14.271l-.079.316.039.316.119.119.138.335.434.435.317.118.118.12.217-.02a1.312 1.312 0 001.028-.554 1.42 1.42 0 00.256-1.126l-.849-.85a1.23 1.23 0 00-.731-.137l-.296.059-.652.652-.059.217zM388.374 34.876l-.375-.375-.257-.06-.454.02-.988.988-.099.454.02.257c.095.295.265.56.494.77l.237-.237-.218.218.475.237.296.059.316-.04.77-.77a2.201 2.201 0 00-.197-1.699l-.02.178zM356.175-10.004l-.533-.534a1.341 1.341 0 00-.77-.177 1.255 1.255 0 00-1.186 1.185l.277.276a1.252 1.252 0 001.086 1.087l.218-.02.355-.118.277-.277c.183-.198.307-.445.355-.711l-.039-.277-.04-.434zM382.151 10.717l-.612-.612-.336-.06-.217-.02a1.482 1.482 0 00-1.304 1.581l.889.889.356.119.118.118h.198l.691-.691a1.79 1.79 0 00.178-1.284l.039-.04zM377.272-6.962l.474.474.376.099.118.118.237-.04.85-.849.118-.355-.889-.89a1.367 1.367 0 00-.889-.216l-1.047 1.046.02.218.139.138.493.257zM446.193-11.92l.118-.119.356-.118.474-.474.138-.336.138-.138a1.378 1.378 0 00-.327-1.052 1.372 1.372 0 00-.996-.47l-.119.12-.336.137-.454.455-.158.355-.138.139a1.445 1.445 0 00.375 1.086c.245.252.577.4.929.415zM480.642-11.486a1.421 1.421 0 00-1.145-.395 1.556 1.556 0 00-.257.06l-.751.75a1.655 1.655 0 00.277 1.501 1.285 1.285 0 001.936.08l.296-.297.158-.355.118-.119a1.523 1.523 0 00-.171-.687 1.524 1.524 0 00-.461-.538zM451.19 5.008l-.573.573-.257.455-.059.256a1.4 1.4 0 001.007 1.324l.099-.099.336.06.296-.02.316-.119.593-.593.099-.296a1.426 1.426 0 00-1.126-1.442l-.119.119-.612-.218zM435.802-8.088l.395.118a1.285 1.285 0 001.027-.71 1.34 1.34 0 00.138-.732 1.367 1.367 0 00-1.363-1.363l-.493.494a1.145 1.145 0 00-.376.375l-.138.139a.901.901 0 000 .553l.83.83-.02.296zM438.39 13.107l-.158-.157a2.533 2.533 0 00-.415-.415l-.277-.04-.494.02-.592.592-.218.455-.059.257.02.335.83.83.375.138h.197a1.422 1.422 0 001.265-1.066v-.277l-.099-.454-.375-.217zM437.185-4.73l-.316-.316c-.161.01-.316.058-.455.138l-.375-.098-.099-.1a1.286 1.286 0 00-1.224 1.14c-.024.195-.004.393.059.58l.75.75.316.079h.198l.118-.119.376.376.237.04a1.26 1.26 0 001.264-1.265 1.172 1.172 0 00-.316-.711l-.533-.494zM419.683-24.642a1.255 1.255 0 001.205.652c.172-.016.339-.063.494-.138l.652-.652a1.32 1.32 0 00-.297-1.165 1.359 1.359 0 00-1.224-.514 1.284 1.284 0 00-1.205 1.205l.079.277.296.335zM438.567-4.928l.316-.316a1.309 1.309 0 00-.291-1.804 1.31 1.31 0 00-.617-.25l-.218.02-.434.118-.494.494-.138.375-.119.119.02.415.592.592.475.237.276.04.138-.138.494.098zM466.163 4.337c-.114 0-.228.02-.336.06l-.316.315a1.477 1.477 0 00-.968.968 1.343 1.343 0 00.771 1.442l.256.02.455-.02 1.047-1.047.098-.375.02-.217a1.282 1.282 0 00-1.027-1.146zM451.467 12.712l-.573-.572a1.342 1.342 0 00-1.126-.455 1.152 1.152 0 00-.356.08l-.869.868v.198l.079.513.692.692.395.158.237.04h.237a1.277 1.277 0 00.927-.487 1.276 1.276 0 00.258-1.015l.099-.02zM446.055-26.755l.869.869h.197a1.405 1.405 0 00.988-.553 1.454 1.454 0 00.237-1.225l-.751-.75-.335-.1-.119-.118a1.426 1.426 0 00-1.245 1.442l.119.119.04.316zM447.753-.206l.198-.198a1.46 1.46 0 00-.435-1.146 1.311 1.311 0 00-1.106-.355l-.632.632a1.48 1.48 0 00-.138 1.086l.79.79c.237.099.494.14.75.119l.297-.06.592-.592-.316-.276zM459.072-2.439a1.314 1.314 0 00.098-1.244l-.75-.75-.297-.06a1.398 1.398 0 00-1.244 1.047l-.099.099v.276l.711.711c.296.222.659.333 1.028.316.215-.082.405-.218.553-.395zM424.049-16.325l.375.138a1.513 1.513 0 001.008-.573c.184-.259.274-.572.256-.889a.676.676 0 00.099-.336l-.02-.217-.098-.375-.554-.553a1.06 1.06 0 00-.533-.218h-.237l-.395.119-.533.533a1.675 1.675 0 00-.218.613l.099.098-.138.139c-.009.404.132.798.395 1.106l.217.217.277.198zM424.78-3.624l.533.534c.279.204.625.296.968.256a1.397 1.397 0 001.166-.849 1.487 1.487 0 00-.237-1.501l-.534-.534h-.276a1.402 1.402 0 00-1.266 1.183c-.035.212-.02.429.041.635l-.395.276zM427.19 6.964a1.405 1.405 0 00.118-1.264l-.494-.494a1.312 1.312 0 00-.889-.257 1.174 1.174 0 00-.632.237l-.75.75a1.54 1.54 0 00.375 1.166 1.485 1.485 0 001.106.395 1.399 1.399 0 001.166-.533zM438.429 23.182l-.672-.672-.335-.099-.099-.098-.296.02-.85.849-.099.375-.118.119.039.237.948.948.396.04.296-.02a1.371 1.371 0 001.084-1.113 1.38 1.38 0 00-.017-.547l-.277-.04zM448.622-19.446a1.372 1.372 0 001.363 1.363l.731-.731a1.367 1.367 0 00-.336-1.916l-.375-.376-.336-.02a1.4 1.4 0 00-1.047 1.68zM422.626 5.483a1.56 1.56 0 00.218-1.64 1.311 1.311 0 00-1.284-.77l-.395.395a1.343 1.343 0 00-.336 1.086 1.47 1.47 0 00.652 1.087l.178.177.256-.02.336-.335.375.02zM429.224-22.567l.652.651.296.1.099.098.217-.02a1.53 1.53 0 001.106-.592 1.455 1.455 0 00.198-1.146l-.83-.83a1.868 1.868 0 00-.75-.118l-.277.04-.79.79a1.333 1.333 0 00-.059 1.046l.138-.02zM424.977 2.5l.375-.099.534-.533c.125-.182.2-.393.217-.613l-.02-.217-.098-.375-.514-.514a1.453 1.453 0 00-.632-.237l-.099.099-.395.118-.494.494a1.154 1.154 0 00-.257.652v.198l.119.395.494.493c.178.138.389.227.612.257l.158-.118zM428.335 1.275l.533.533.257-.02a1.343 1.343 0 001.047-.572 1.501 1.501 0 00.217-1.245l-.691-.691a1.817 1.817 0 00-.652-.06 1.341 1.341 0 00-1.185.988l-.04.237.04.395.336.336.138.099zM423.871-.345l.376-.098.513-.514a1.475 1.475 0 00.237-1.62 1.34 1.34 0 00-1.303-.75l-1.107 1.106a1.12 1.12 0 00-.237.592l.02.218.119.395.513.513c.17.135.376.218.593.237l.276-.079zM422.488-9.392l.395-.118.514-.514c.111-.188.185-.396.217-.612l-.119-.119-.099-.375-.533-.534a2.018 2.018 0 00-.573-.217l-.118.119-.395.118-.514.514a1.21 1.21 0 00-.257.612l.02.217.119.396.513.513c.178.122.38.203.593.237l.237-.237zM431.456 9.888l-.671-.672-.336-.099-.099-.099-.395.04-.355.356a1.538 1.538 0 00-1.442 1.718l1.046 1.047c.157.03.318.03.475 0a1.447 1.447 0 001.047-.494 1.34 1.34 0 001.126-1.64l-.396-.157zM435.506 7.379l.395.395a1.285 1.285 0 00.968-.652c.153-.223.237-.487.241-.757a1.377 1.377 0 00-.221-.764 1.316 1.316 0 00-1.245-.612 1.26 1.26 0 00-.908.474c-.22.324-.305.72-.237 1.106l.118.119a1.344 1.344 0 00.889.69zM440.879-16.68l.336-.02.178-.06.336-.178.177-.138.158-.197.099-.178.099-.376.02-.177v-.158l-.06-.218-.177-.335-.316-.317-.198-.079-.356-.118-.197-.04-.316.04-.217.06-.336.177-.158.119-.158.197-.099.178-.099.375-.039.198.039.158.04.197.177.336.119.158.197.158.198.119.375.098.178.02zM455.319 17.354l-.415-.414a1.333 1.333 0 00-.573-.218l-.118.119-.395.118-.632.633a1.234 1.234 0 00-.158 1.066l.77.77.316.08.099.098.257-.02.849-.849.119-.355.098-.1-.118-.118-.099-.81zM442.262 24.446l-.83.83-.118.355a1.418 1.418 0 00.905 1.385c.183.072.379.105.576.097l.118-.119.317-.118.77-.77a1.52 1.52 0 00-.296-1.324 1.429 1.429 0 00-1.442-.336zM462.548 38.214a1.344 1.344 0 00-.988.593 1.425 1.425 0 00-.177 1.205l.869.869c.204.095.427.142.652.138.346-.052.661-.227.889-.494l.296-.296a1.436 1.436 0 00-.395-1.106 1.373 1.373 0 00-1.146-.909zM451.328-12.434l-.098.099-.297.099-.691.691a1.12 1.12 0 00-.04.435l.119.118.138.375.593.593.395.119.237.04.849-.85a1.458 1.458 0 00-.197-1.383 1.54 1.54 0 00-1.008-.336zM461.62 17.216l-.296.06-.751.75-.079.316v.198l.751.75c.17.135.376.218.592.237l.277-.039.454-.178.257-.257.099-.375-.099-.099a1.316 1.316 0 00-1.205-1.363zM431.318-2.3l1.126-1.126.119-.356.019-.217-.019-.218-.889-.888-.336-.1h-.198a1.395 1.395 0 00-.987.554.863.863 0 00-.178.178l-.415.414-.217.415-.079.277a1.79 1.79 0 00-.198.197l-.158.158.04.475.573.572.434.237.277.04a1.422 1.422 0 001.106-.672l-.02.06zM442.143 9.591l-.415-.415-.316-.118-.118-.118a1.51 1.51 0 00-.988.513 1.42 1.42 0 00-.257 1.126l.791.79.316.119.217.02a1.31 1.31 0 001.007-.534 1.412 1.412 0 00.237-1.106l-.118-.119-.356-.158zM453.837-41.017l.138-.139.059-.256a1.277 1.277 0 00-.434-.988 1.313 1.313 0 00-1.047-.336l-.277.04-.79.79-.039.276-.099.1.039.434.593.592.435.198.276.079.099-.099a1.229 1.229 0 001.047-.691zM377.41 25.335l.395-.395a1.25 1.25 0 00-.039-1.304 1.318 1.318 0 00-.514-.514l-.138-.138-.375-.138-.218-.02h-.237a1.699 1.699 0 00-1.205.652l-.276.276-.04.238c-.011.316.079.627.257.889l1.008 1.007.296-.06.118-.118.968-.375zM381.796 49.632l-.83-.83a1.425 1.425 0 00-1.284 1.087l.02.296.533.533c.28.238.641.358 1.008.336a1.338 1.338 0 001.169-1.211 1.391 1.391 0 00-.083-.626l-.533.415zM404.295 4.258l.277.079.513-.04.593-.592.217-.455.04-.276a4.35 4.35 0 01.276-1.106l.02-.218-.059-.454-.554-.553-.414-.217-.257-.06A1.283 1.283 0 00403.92 1l-.178.177-.198.395-.711.712-.02.375-.059.06a1.315 1.315 0 001.541 1.54zM403.228-.76l.494-.493a1.54 1.54 0 00-.553-1.62l-.316-.119-.573-.02-.257.06-.257.02-.395.118-.533.533-.178.415-.019.218.019.335 1.146 1.146.573.02.276-.08.553-.552.02.02zM403.94 10.697l-.632-.632-.415-.098-.257-.02a1.403 1.403 0 00-.928.612 1.314 1.314 0 00-.139 1.087l.83.83.356.039a1.286 1.286 0 001.205-1.205l-.02-.217v-.396zM407.258 30.925l-.711.711a1.03 1.03 0 00-.138 1.008 1.198 1.198 0 00.731.73l.158.159.375.098a1.233 1.233 0 00.869-.434c.21-.27.309-.609.277-.948a1.429 1.429 0 00-.5-.993 1.43 1.43 0 00-1.061-.33zM445.62 28.397l-.297.296c-.143.31-.171.66-.079.988a1.4 1.4 0 00.751.75 1.282 1.282 0 001.466-.096c.137-.108.249-.244.332-.398l.039-.395-.197-.671-.652-.652-.415-.06-.158.159-.79.079zM490.085 56.585l-.02.415.198.672.375.375c.136.075.282.129.435.158l.395.04.672-.198.296-.296a1.224 1.224 0 00.079-.988 1.399 1.399 0 00-.711-.711 1.424 1.424 0 00-1.719.533zM434.38 32.249l.375.098.099.1.217-.02a1.285 1.285 0 001.067-.712c.069-.115.116-.242.138-.375l-.02-.296-1.047-1.047-.336-.06h-.197l-.652.652a.666.666 0 00-.158.158 1.395 1.395 0 00.395 1.818l.119-.316zM461.106 34.48l.297.02.572-.098.218-.218c.116-.094.221-.2.316-.316a1.253 1.253 0 00.079-1.067 1.493 1.493 0 00-.909-.908l-.099-.099-.296-.02-.514.514a2.165 2.165 0 00-.296.297 1.311 1.311 0 00.573 1.758l.059.138zM437.561 21.72l.237-.04.434-.197.237-.237c.097-.074.183-.16.257-.257a1.402 1.402 0 00.04-1.185 1.545 1.545 0 00-1.126-.77l-.099.098-.336.06-.672.671a1.43 1.43 0 00.119 1.54l.652.653.257-.336zM472.603 39.735l-.178.178a1.282 1.282 0 00-.75.75c-.201.421-.229.903-.079 1.344l.671.672.336.059.099-.099a1.29 1.29 0 00.889-.652c.329-.04.632-.203.849-.454l.158-.158.119-.395.118-.119-.019-.256-.573-.573a1.629 1.629 0 00-1.482-.218l-.158-.079zM480.149 52.93l-.415.415-.198.198a1.372 1.372 0 00.158 1.54 1.51 1.51 0 001.679.416l.396-.396c.083-.06.156-.134.217-.217a1.425 1.425 0 00-.158-1.54 1.512 1.512 0 00-1.679-.415z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M430.133-.009l.138-.138.099-.375a1.179 1.179 0 00-.336-1.087l-.573-.573a1.816 1.816 0 00-.869.119l-.79.79-.02.336a1.673 1.673 0 001.165 1.165 1.372 1.372 0 001.186-.237zM455.714 53.306l-.356.04-.75.75a1.399 1.399 0 00.217 1.64l-.059-.06.395.158.237.04.375-.02.158-.158c.177-.108.326-.257.435-.434l.158-.159.118-.395v-.237l-.059-.335-.138-.139a1.347 1.347 0 00-.771-.77l.04.079zM442.755 46.234a1.264 1.264 0 00-.059 1.126 1.403 1.403 0 00.849.85l.119.118.336-.02.987-.987.04-.395-.099-.1-.059-.256-.849-.85-.396-.039-.237.04-.691.691.059-.178zM466.933 34.698l-.138.138-.079.356-.02.217.06.297c.058.27.204.512.414.69l.139.14.335.137.218.02.257-.02.355-.355c.233-.128.425-.32.553-.553.063-.07.129-.135.198-.198v-.316l-.652-.652a1.677 1.677 0 00-1.146-.316 1.517 1.517 0 00-.494.415zM407.554.406l.435-.08.138-.137c.263-.136.477-.35.613-.613a1.37 1.37 0 00.158-.474l-.099-.099a1.318 1.318 0 00-1.245-.968l-.098-.098-.257.02-.534.533-.217.217a1.451 1.451 0 00.217 1.68l.158.157.514.237.217-.375zM448.879 51.686l-.494-.494-.237-.04-.296-.019-.158.158-.395.198-.158.158a.676.676 0 00-.178.177 1.395 1.395 0 00.395 1.818l.138.138h.198a1.4 1.4 0 00.751-.75 1.457 1.457 0 00.434-1.344zM421.402 36.851a1.51 1.51 0 00-.593-.592l-.118-.119-.376.02-.829.83a1.566 1.566 0 00.118 1.54l.178.178.395.198.257.059h.395c.26-.081.497-.223.691-.415l.178-.178.119-.434-.158-.514-.06-.296-.296-.297.099.02zM451.427 51.943l-.118.355-.119.119.001.198.118.118a1.346 1.346 0 00.948.948 1.565 1.565 0 001.462-.355l.395-.395.02-.257a1.479 1.479 0 00-1.166-1.166 1.312 1.312 0 00-1.541.435zM414.686 44.002l.336.06.296-.02.514-.238.296-.296a1.347 1.347 0 00.099-1.126 1.493 1.493 0 00-.81-.81l-.297-.02-.592.593a1.647 1.647 0 00-.257.257 1.313 1.313 0 00.395 1.66l.178.177-.158-.237zM433.353 38.61l1.066 1.066.316.04a1.457 1.457 0 001.028-.751 1.396 1.396 0 00-.553-1.857l-.159-.158-.414-.06-.257.02-.316.316a1.475 1.475 0 00-.553.553l-.119.119-.039.395v.316zM451.21 42.718a1.642 1.642 0 00-.376-.375 2.112 2.112 0 00-.375-.613 1.123 1.123 0 00-.02-.73l.257.02.217-.02.534-.534c.119-.09.226-.196.316-.316a1.314 1.314 0 00-.277-1.58l-.514-.514h-.237l-.276.04-.83.83-.02.375v.197l-.276-.04-.099.1a1.341 1.341 0 00-1.185.71 1.487 1.487 0 00-.257.257l-.237.238-.02.217c.028.122.068.241.119.355l.138.336.138.138c.08.23.208.438.376.613l.454.454c.043.318.205.608.454.81l.336.336.435.04.237-.04.829-.83c.061-.198.081-.406.06-.612.03.048.063.094.099.138zM444.593 87.579l-.06.414.119.356.039.237a1.24 1.24 0 00.731.73 1.256 1.256 0 001.027-.197l.415-.415a1.287 1.287 0 00-.533-1.718 1.622 1.622 0 00-.869-.158 1.247 1.247 0 00-.869.75zM448.78 57.079l.02-.375a1.284 1.284 0 00-.85-1.442l-.118-.119-.494.06-.119.118a1.56 1.56 0 00-.908.593l-.237.236v.198a1.586 1.586 0 001.323 1.086 1.342 1.342 0 001.363-.69l.02.335zM440.503 51.607l-.572.573a1.387 1.387 0 00-.118 1.079 1.39 1.39 0 00.69.837 1.65 1.65 0 001.146.119 1.4 1.4 0 00.751-.751l.039-.435-.138-.138a1.208 1.208 0 00-.514-.988 1.223 1.223 0 00-1.185-.237l-.099-.059zM443.842 56.447l-.296.02-.771.77c-.111.213-.172.45-.177.691l.098.099.079.276.099.1a1.453 1.453 0 001.225.631 1.256 1.256 0 001.106-.671c.159-.297.194-.645.099-.968l-.099-.375a1.39 1.39 0 00-.948-.751l-.415.178zM447.042 62.215l-.593-.593-.335-.06-.297.02-.494.258-.217.217a.695.695 0 00-.138.138 1.399 1.399 0 00.435 1.896c.211.23.499.378.809.415a1.233 1.233 0 001.067-.355l.553-.553-.099-.1-.019-.295-.099-.1a2.145 2.145 0 00-.415-.612l-.158-.276zM474.164 65.593a1.34 1.34 0 00-.001 1.303 1.26 1.26 0 001.127.652l1.185-1.185v-.198l-.079-.276-.119-.118a1.25 1.25 0 00-1.146-.554 1.285 1.285 0 00-.967.376zM406.586 58.422l.119-.395.02-.217-.02-.336-.119-.118a1.499 1.499 0 00-.474-.475l-.138-.138-.395-.118-.217-.02-.316.04-.119.118a1.45 1.45 0 00-.454.454l-.158.158-.099.376-.02.217.04.316.118.118c.071.212.193.401.356.553l.158.159.355.118.218.02.336-.06.118-.118a1.32 1.32 0 00.435-.434l.256-.218zM418.616 48.051l.099.1.158-.159c.171-.116.318-.264.435-.434l.178-.178.098-.494-.019-.257-.079-.316-.573-.573a1.284 1.284 0 00-1.778.514 1.38 1.38 0 00-.217.217l-.099.099.039.316.949.948.809.217zM426.419 63.123l-.237.237-.019.376.217.652.276.276a1.371 1.371 0 001.857-.513 1.367 1.367 0 00-.296-1.6 1.287 1.287 0 00-1.798.572zM431.614 55.518l.77-.77a.823.823 0 01.178-.178 1.368 1.368 0 00.04-1.264 1.404 1.404 0 00-.988-.75l-.118-.12-.395-.039-1.107 1.107-.019.375.019.217 1.245 1.245.375.177zM441.294 24.466l.099.099a1.48 1.48 0 001.323-.731 1.48 1.48 0 00-1.837-2.114l-.869.87a1.428 1.428 0 00.81 1.362l.099.099.375.415zM389.025 56.783l-.158-.159-.474-.079-.256.02-.139.139a1.396 1.396 0 00-1.027.71l-.257.257v.277l.395.395c.193.303.487.527.83.632h.277l.533-.533c.175-.126.328-.28.454-.455a1.397 1.397 0 00-.178-1.205zM456.128-21.461a1.255 1.255 0 00-.158-1.423l-.237-.237-.355-.079a1.259 1.259 0 00-.929-.059 1.18 1.18 0 00-.691.692l-.217.217-.02.217a1.283 1.283 0 00.968 1.284 1.51 1.51 0 001.639-.612zM493.838 76.18a1.397 1.397 0 00-1.778.633l-.158.474v.237l.139.138a1.305 1.305 0 00.928.928l.296.02.277-.04.435-.236c.223-.126.407-.31.533-.534l.118-.118-.118-.119-.099-.375-.632-.632.059-.375zM391.238 41.928l.099.099.711-.712a1.45 1.45 0 00-.514-1.856 1.285 1.285 0 00-1.738.316l-.119.118-.098.297v.237a1.34 1.34 0 001.284 1.284l.375.217zM429.797 44.496a1.259 1.259 0 00.06 1.284l.77.77.375.02.237-.04.751-.75a1.285 1.285 0 00-.257-1.64l-.138-.138-.316-.119-.119-.118h-.237a1.453 1.453 0 00-1.126.73zM440.681 25.631l-.454.454c-.049.044-.095.09-.139.139a1.54 1.54 0 00-.118 1.383.844.844 0 00.099.296c.039.384.214.741.493 1.007a1.962 1.962 0 001.995.415l.85-.85v-.394l-.257-.257a1.424 1.424 0 00-.296-1.087l-.534-.533a1.366 1.366 0 00-1.125-.257l-.277.04-.237-.356zM492.278 22.55h-.316l-.909.908a1.317 1.317 0 00-.02 1.087 1.24 1.24 0 00.731.73l.099.1a1.287 1.287 0 001.284-.494l.138-.139.099-.336v-.434l-.356-.356a1.395 1.395 0 00-.849-.612l.099-.454zM473.077 30.017l-.612-.613a1.591 1.591 0 00-1.244-.257 1.317 1.317 0 00-.791.79.391.391 0 00-.118.12l-.119.118v.316l.257.256a1.43 1.43 0 00.79.79l.297.02.375-.375a1.76 1.76 0 00.533-.533c.067-.052.126-.112.178-.178v-.316l.454-.139zM454.43 41.355l-.158.158.02.494.849.85.454.217.277.04.336-.02c.185-.035.365-.094.533-.178l.217-.218.257-.493.04-.316-.04-.356-.197-.198a1.509 1.509 0 00-.751-.75l-.138-.139-.415-.02-.237.04-.296.297c-.309.109-.573.317-.751.592zM463.477 46.254l.296-.296a1.332 1.332 0 00.099-1.126 1.49 1.49 0 00-.889-.89l-.118-.118h-.356l-.355.356-.257-.257-.277.04-1.224 1.224-.02.415.869.87a1.314 1.314 0 001.146.039c.289.112.604.14.908.079l.356-.356-.178.02zM428.79 47.913l-.474-.474a1.481 1.481 0 00-1.956.612l-.138.139-.04.395.02.217.059.257.949.948.217.02.395-.04.158-.158c.269-.127.485-.343.612-.612.078-.185.138-.377.178-.573l-.099-.099-.039-.316-.099-.099.257-.217zM443.802 33.059l-.177.177a1.393 1.393 0 00.276 1.66l.672.671.375-.02.138-.138c.316-.159.572-.415.731-.73a.89.89 0 00.178-.178l.02-.297-.692-.691a1.786 1.786 0 00-1.284-.336l-.335.336.098-.454zM453.936 36.476l-.217.217-.119.553.04.237a5.06 5.06 0 01-.02 1.087l.77.77a1.704 1.704 0 001.442.297l.336-.336a1.369 1.369 0 00.929-.573l.098-.099.079-.276.198-.198-.118-.119a1.901 1.901 0 00-.376-.928l-.672-.671a1.537 1.537 0 00-2.37.039zM448.701 34.876l-.099.099v.316l.672.671a1.365 1.365 0 001.597.147c.164-.098.307-.228.418-.384l.158-.158v-.237a1.428 1.428 0 00-1.126-1.126 1.37 1.37 0 00-1.344.711.997.997 0 00-.276-.04zM439.733 35.943h.198l.632-.633a.822.822 0 00.178-.177 1.37 1.37 0 00-.396-1.818l-.118-.118-.356-.08-.118-.118-.474.474-.119-.118a1.723 1.723 0 00-.533.533 1.002 1.002 0 00-.178.178l-.02.336 1.304 1.304v.237zM492.416 43.152a1.336 1.336 0 00-.079 1.107 1.584 1.584 0 00.968.968l.237.04.099.098.276-.04a1.53 1.53 0 00.79-.79 1.512 1.512 0 00.079-1.185l-.493-.494a1.534 1.534 0 00-1.265-.237 1.232 1.232 0 00-.612.533zM424.8 17.374l-.514.514a.53.53 0 00-.138.138 1.366 1.366 0 00.731 2.035l.237.04.118-.12a1.285 1.285 0 001.343-.671.97.97 0 00.158-.474v-.198l-.098-.336-.494-.493a1.96 1.96 0 00-.455-.455l-.138-.138-.355-.04-.099.1-.296.098zM418.36 18.757l.77-.77.119-.119a1.73 1.73 0 00-.079-1.896l.118.118-.572-.573h-.316l-.099.1c-.346.149-.621.424-.771.77-.143.265-.198.57-.158.869l1.126 1.126-.138.375zM448.721 29.404l-.178.178c-.19.373-.226.806-.099 1.205a1.538 1.538 0 00.889.889l.099.099h.316l.672-.672a.673.673 0 00.177-.178 1.363 1.363 0 00.099-1.205 1.56 1.56 0 00-.869-.869l-.099-.099-.336-.02-.671.672zM425.906 7.497l.118.119a1.205 1.205 0 00.771 1.126l.276.079h.395l.573-.573a1.37 1.37 0 00.02-1.244 1.538 1.538 0 00-.612-.613 1.318 1.318 0 00-1.483.111c-.138.11-.251.247-.335.403l.277.592zM415.061-17.017l.218-.02.237-.236c.121-.103.233-.215.335-.336a1.365 1.365 0 00-.158-1.541 1.533 1.533 0 00-1.679-.494l-.474.474a.836.836 0 00-.197.198 1.375 1.375 0 00-.02 1.323 1.535 1.535 0 001.304.79l.434-.158zM459.428 38.135l.098.099h.316l.455-.454a1.05 1.05 0 00.237-.238 1.367 1.367 0 00.039-1.264 1.561 1.561 0 00-1.264-.83l-.257.02-.474.158-.592.593-.079.474.019.296a1.598 1.598 0 001.502 1.146zM463.99-17.234l-.039.395.079.277c.029.186.089.366.178.533l.217.217.434.119.119.118.415.06.316-.119.474-.474c.15-.3.185-.644.099-.968a1.399 1.399 0 00-.711-.711 1.37 1.37 0 00-1.581.553zM436.987-4.019a1.534 1.534 0 00.376 1.916l.158.158.434.158.218.02.316-.04.987-.987.02-.415-.02-.217-.592-.593-.218.217.198-.197-.494-.494-.336-.02-.256.06-.455.217-.178.178c-.053.01-.106.022-.158.04zM435.032 26.382l.434-.435.04-.355-.02-.217-.059-.257-1.008-1.008-.355-.04-.119.12-.237.039-.612.612a1.508 1.508 0 00-.099 1.126 1.376 1.376 0 00.81.81l.119.119.355.039.415-.415.336-.138zM416.246 2.697a1.259 1.259 0 00.395 1.462l.198.198.514.04a1.227 1.227 0 001.244-.653c.076-.155.122-.322.138-.494l-.059-.335-.691-.692a1.342 1.342 0 00-1.719.218l-.158.158.138.098zM435.328 79.578l-1.126 1.126-.039.356v.197l.672.672a1.423 1.423 0 001.619-.04l.218-.217.217-.454.079-.277-.04-.316-.098-.296-.178-.178a1.397 1.397 0 00-.573-.573l-.118-.118-.633.118zM486.134 32.21l-.217.216-.198.514-.039.316.079.316.592.593a1.405 1.405 0 001.146.276 1.4 1.4 0 00.751-.75l.158-.158v-.198a1.446 1.446 0 00-.929-.928 1.422 1.422 0 00-1.343-.198zM474.677 44.654l-.217.02-.297.296-.336-.099a1.481 1.481 0 00-1.619.672l-.119.118-.039.316.039.237.454.455a1.62 1.62 0 001.304.513c.271-.02.531-.116.751-.276l.099.099a1.254 1.254 0 001.244-.692c.051-.1.091-.206.119-.316l.098-.099-.039-.316-1.008-1.007-.434.079zM421.086 65.375l.336-.335.039-.356-.098-.099-.04-.237-.513-.513a1.594 1.594 0 00-1.146-.237 1.48 1.48 0 00-.85.849.696.696 0 00-.138.138l-.118.119v.355l1.066 1.067.356.04.217-.02.711-.711.178-.06zM389.994 19.152l.158-.158.04-.474-.02-.257-.356-.356a1.555 1.555 0 00-.395-.592l-.474-.474h-.237l-.119.118-.335.139-.139.138a.976.976 0 00-.335.336 1.423 1.423 0 00.493 1.876 1.37 1.37 0 001.719-.296zM413.223 13.7a1.287 1.287 0 001.107-.434l.138-.139.118-.395.02-.217-.059-.376-.158-.157a1.37 1.37 0 00-.731-.731l-.118-.119h-.356l-.81.81a1.315 1.315 0 00.356 1.699l.138.138.355-.079zM424.286-8.424h.197l.257-.06c.187-.11.344-.266.454-.454l.178-.177.099-.455.138-.414-.099-.297-.573-.573a1.306 1.306 0 00-1.482.111c-.138.11-.251.248-.335.403a.838.838 0 00-.197.197l-.099.1.02.335.869.87.573.414zM438.35 48.585l.118-.119-.138-.138a1.371 1.371 0 00-1.224-.632 1.399 1.399 0 00-1.107.672 1.24 1.24 0 00-.039 1.106l.573.573a1.392 1.392 0 001.066.158 1.236 1.236 0 00.771-.77.892.892 0 00-.02-.85zM467.388 64.94l-.592-.592a2.257 2.257 0 00-.514-.513l-.316-.08-.573.06-.316.119-.474.474-.119.592.633.08-.395.078a1.534 1.534 0 00.908 1.462l.119.118.355.04 1.146-1.146.039-.355.099-.336zM405.461 35.29l.375-.02.138-.137c.213-.068.404-.19.553-.356l.158-.158.119-.355v-.238l-.04-.355a1.03 1.03 0 00.218-.217l.454-.455.04-.276v-.237l-.099-.415-.178-.178a1.368 1.368 0 00-1.146-.474 1.316 1.316 0 00-.987.711 1.45 1.45 0 00-.119 1.067l-.375.375a1.427 1.427 0 00.178 1.64l.158.158.414.178.139-.257zM410.202 33.394c.362.234.801.319 1.224.237a1.534 1.534 0 00.948-.71l.257-.258-.039-.237a1.79 1.79 0 00-.929-.928l-.118-.119h-.356l-1.145 1.146-.04.356v.197l.533.534-.335-.218zM442.439 15.26a1.483 1.483 0 00.652 1.956c.311.16.672.196 1.008.099a1.304 1.304 0 00.731-.731l.039-.395-.197-.672-.692-.691-.395-.04-.671.198-.475.277zM477.245 32.604l-.336-.06-.316.04-.534.257-.533.533-.039.396.039.237.771.77a1.256 1.256 0 001.698-.514 1.2 1.2 0 00-.118-1.343l-.178-.178-.454-.138zM427.624 25.77l.139-.139-.02-.217a1.485 1.485 0 00-1.245-1.047 1.426 1.426 0 00-1.382.75 1.311 1.311 0 00-.178.415v.198l.257.257a1.425 1.425 0 002.39-.099l.039-.119zM412.196 22.945h.198l.237-.04.513-.513a.662.662 0 00.178-.178 1.515 1.515 0 00.114-1.16 1.513 1.513 0 00-.746-.895l-.158-.158-.415-.098h-.237l-.276.079-.889.889-.04.276v.237l.099.375.158.159c.213.272.458.518.731.73l.118.119.415.178zM463.457 46.985l-.296-.02-.534.533c-.096.075-.182.161-.256.257a1.427 1.427 0 00.237 1.66l.592.592.296.02.237-.04.415-.177c.217-.119.396-.297.514-.514a.737.737 0 00.198-.197l.098-.1-.099-.098a1.785 1.785 0 00-.928-.928l-.474-.988zM428.473 28.87l-.138-.137-.356-.08h-.197l-.316.04a1.563 1.563 0 00-.672.395l-.158.158-.138.415v.237l.039.277.929.928.395.04.237-.04.731-.73a1.31 1.31 0 00-.356-1.502zM464.642 41.612a1.252 1.252 0 00-.711.71l-.039.396.098.375.139.139c.11.21.283.383.494.493l.039.277.652.652a1.59 1.59 0 001.185.237l.632-.632a.197.197 0 00.099-.099 1.339 1.339 0 00-.178-1.521l-.375-.375a1.233 1.233 0 00-.652-.89 1.397 1.397 0 00-1.383.238zM407.199 46.669h-.277a1.034 1.034 0 00-.375-.1 1.255 1.255 0 00-1.126.574l-.138.138-.079.356-.04.276.02.218.711.71c.287.165.619.234.948.198a1.314 1.314 0 001.541-.632l.138-.138-.118-.119a1.54 1.54 0 00-1.205-1.481zM401.707 20.317l.356-.039.138-.138c.174-.098.317-.242.415-.415l.158-.158.138-.415.02-.217-.04-.356-.158-.158a1.33 1.33 0 00-.77-.77l-.099-.099h-.355l-.751.75a1.42 1.42 0 00.237 1.66l.158.158.395.158.158.04zM426.202 96.092l-.178.178-.019.296.474.474a1.557 1.557 0 001.224.316 1.4 1.4 0 00.948-.75l.178-.178v-.316l-.592-.593a1.514 1.514 0 00-1.205-.296 1.388 1.388 0 00-.83.87zM483.092 34.184l-.099-.098a1.257 1.257 0 00-1.126.494l-.138.138-.099.415-.099.098.06.336.138.139c.142.339.411.608.751.75l.118.119.375-.02.81-.81a1.312 1.312 0 00-.335-1.679l-.119-.119-.237.237zM453.205 63.993l-.198.197-.138.494-.02.296.099.297.593.592a1.282 1.282 0 001.797-.533.842.842 0 00.198-.198l.098-.098-.019-.297-.731-.73a1.398 1.398 0 00-1.679-.02zM442.498 5.088c-.014.164-.007.33.02.493a1.256 1.256 0 001.047 1.047 1.283 1.283 0 001.323-.691 1.173 1.173 0 00-.039-1.264l-.593-.593-.276-.079h-.514l-.909.909-.059.178zM457.274 16.545a1.072 1.072 0 00-.138.375l-.02.217.158.158a1.622 1.622 0 001.205.613 1.537 1.537 0 001.166-.692l.237-.237-.02-.217a1.45 1.45 0 00-1.264-1.027 1.4 1.4 0 00-1.324.81zM461.126 60.022c.201.312.511.538.869.632h.277l.434-.434c.082-.064.155-.137.218-.218a1.372 1.372 0 00-.573-1.72 1.368 1.368 0 00-1.778.357l-.118.119-.079.316-.001.198.751.75zM489.334 12.97l-.632.631-.198.198a1.372 1.372 0 002.371 1.383l.138-.139.059-.415-.039-.237-.079-.276-.158-.158a1.425 1.425 0 00-.593-.593l-.099-.099-.77-.296zM436.098 60.319h.238l.276-.08.612-.612a1.505 1.505 0 00-.079-1.54l-.177-.179-.415-.217-.237-.04-.297-.019-.158.158-.395.198-.177.177-.158.158a1.396 1.396 0 00.375 1.837l.138.139.454.02zM424.562 76.477l.198.671.652.652.316.04.553-.553a1.348 1.348 0 00.118-1.027 1.32 1.32 0 00-.731-.731 1.425 1.425 0 00-1.876.533l.77.415zM450.538 73.89l-.138.137-.04.395 1.265 1.265.296.02.099-.1c.241-.053.461-.177.632-.355l.474-.474.059-.257v-.474l-.059-.257-.751-.75a1.508 1.508 0 00-1.837.85zM425.51 36.733l.731.73.356.04.099.099.651-.652.158-.158a1.333 1.333 0 00.099-1.166 1.418 1.418 0 00-.829-.83l-.099-.098-.296-.02-.396.395c-.223.126-.407.31-.533.534-.07.06-.136.127-.197.197l-.099.1.02.335.256.257.079.237zM460.632 72.763l-.474.474-.217.415-.158.158.039.237.968.968c.297.08.607.093.909.04l-.02-.376.217.218.474-.277.435-.435.04-.276-.04-.553-.198-.198a2.12 2.12 0 00-1.54-.75l-.435.355zM405.421 74.916l-.297.02-.296.296c-.233.128-.425.32-.553.553a1.203 1.203 0 00.435 1.62l.276.277a1.343 1.343 0 001.501-.435l.139-.138.099-.375v-.198l-.04-.276a1.455 1.455 0 00-.395-.672l-.198-.198-.671-.474zM437.027 64.644h-.356l-.79.79a1.48 1.48 0 00.593 1.976l.138.138.415.06.217-.02 1.205-1.205-.019-.376-.119-.118a1.566 1.566 0 00-.928-.928l-.356-.317zM419.486 58.955l.118-.118.257-.06.83-.829.039-.395-1.047-1.047-.355-.04-.099-.098-.276.079-.099.098c-.283.124-.508.35-.632.633l.276.71.237.712.277.276.474.08zM453.442-1.273l.139-.138.079-.356.118-.118-.039-.238-.554-.553a1.39 1.39 0 00-1.052-.295 1.394 1.394 0 00-.943.552l-.138.138-.099.376v.197l.04.277.533.533a1.419 1.419 0 001.126.296 1.308 1.308 0 00.79-.671zM472.78-29.778a1.396 1.396 0 00-.948.751.454.454 0 00-.158.158l-.039.237.276.277a1.652 1.652 0 001.403.612 1.344 1.344 0 001.027-.672c.158-.275.201-.602.119-.908l-.633-.633a1.427 1.427 0 00-1.047.178zM470.311 3.863l-.178.178-.019.335.059.257.237.474.197.198a1.4 1.4 0 001.66-.119l.454-.454.059-.336-.019-.296-.198-.514-.197-.197a1.398 1.398 0 00-1.838.138l-.217.336zM450.914 9.512l.276-.276a1.254 1.254 0 000-1.264 1.395 1.395 0 00-.533-.534 1.338 1.338 0 00-1.857.474l-.039.395.079.277c.023.188.084.37.178.533l.138.139c.1.16.235.294.395.395l.276.04.099-.1.988-.079zM470.015 19.014l-.277.04a1.347 1.347 0 00-.553.197l-.533.533-.04.277v.316l.06.257.889.889.355.04.217-.02.712-.712a1.424 1.424 0 00-.178-1.64l-.178-.177-.454-.178-.02.178zM454.47 10.223l-.593.593a1.419 1.419 0 00-.138.494l-.455.454a1.368 1.368 0 00-.059 1.205 1.303 1.303 0 00.79.79l.099.1.336-.02.968-.969a.394.394 0 00.138-.138l.494-.494a1.285 1.285 0 00-.494-1.718 1.118 1.118 0 00-.751-.119.809.809 0 00-.335-.178zM428.217 13.838a1.258 1.258 0 001.205-.652 1.44 1.44 0 00.138-.493l-.02-.218-.316-.316a1.342 1.342 0 00-1.165-.415 1.429 1.429 0 00-1.067.751 1.254 1.254 0 00.355 1.66l.178.177.692-.494zM455.062 8.92l-.039-.237a1.371 1.371 0 00-1.047-.81 1.341 1.341 0 00-1.185.434l-.158.158-.119.435-.04.237.06.296.81.81.276.08h.316l.968-.969.158-.434zM477.995-8.957a1.198 1.198 0 00-.197.197l.02.297a1.448 1.448 0 00.434 1.303 1.258 1.258 0 001.6.139l.04-.04.217-.415.237-.237.04-.355-.237-.237-.079-.316-.573-.573a1.287 1.287 0 00-1.502.237zM476.89 7.912l-.119-.118-.296-.02-.197.198a1.316 1.316 0 00-1.008.69.935.935 0 00-.118.356l-.099.1.039.315.652.652.474.237.316.04.336-.06.612-.612a.822.822 0 01.178-.178 1.343 1.343 0 00.079-1.066 1.253 1.253 0 00-.849-.534zM440.998-9.333l.098.1.257-.02.573-.573.119-.119a1.346 1.346 0 00.098-1.126 1.49 1.49 0 00-.889-.889l-.098-.099-.336-.02-.751.751a1.4 1.4 0 00.534 1.837l.138.139.257.02zM464.07 3.863l-.317-.316a1.587 1.587 0 00-1.224-.277 1.477 1.477 0 00-.909.711l-.237.237v.316l.079.277a1.333 1.333 0 00.77.77h.277l.474-.474c.198-.142.372-.315.514-.513.051-.1.091-.207.118-.316v-.316l-.237-.237.692.138zM468.375-6.725a1.477 1.477 0 00-.967-.77l-.099.098-.336.02-.711.711a1.427 1.427 0 00.178 1.64l.533.533h.277l.098-.099c.34-.141.609-.411.751-.75a1.228 1.228 0 00.276-1.383zM437.541 14.787l.276-.277a1.306 1.306 0 00.079-1.146 1.224 1.224 0 00-.632-.632l-.415-.415-.375-.098-.118-.119-.336.02-.138.138c-.17.104-.312.246-.415.415l-.158.158-.119.395-.02.217.04.316.138.139c.113.185.269.34.455.454l.256.257c.325.237.727.343 1.126.296l.139-.138.217.02zM448.543 21.167l.099.099.119-.119c.308-.152.558-.402.711-.71a1.314 1.314 0 00-.079-1.423 1.451 1.451 0 00-1.403-.652l-.533.533a1.05 1.05 0 01-.178.178 1.503 1.503 0 00-.099 1.205 1.537 1.537 0 00.889.889h.474zM458.736 24.999a1.345 1.345 0 00-.909-.909l-.237-.04-.276.277a1.227 1.227 0 00-.652.652c-.103.202-.15.427-.138.652l-.099.099.039.237.474.474a1.426 1.426 0 001.166.336 1.459 1.459 0 00.948-.711 1.422 1.422 0 00-.316-1.067zM478.45 24.565h-.356l-1.106 1.106-.02.375.02.217 1.007 1.008h.435l.217-.02.83-.83a1.37 1.37 0 00-.79-1.817l-.237-.04zM448.483-7.456a1.373 1.373 0 001.523-.092c.143-.11.264-.246.354-.402l.099-.099.059-.256v-.198l-.099-.296a1.37 1.37 0 00-1.778-.79l-.395.395a1.367 1.367 0 00.376 2.153l-.139-.415zM454.884 27.903a1.478 1.478 0 00-1.166-1.166 1.34 1.34 0 00-1.343.632l-.119.119v.198a1.428 1.428 0 00.791.79 1.563 1.563 0 001.264.079l.81-.81.039-.356-.276.514zM435.209 7.004h.316l.692-.692a.647.647 0 01.158-.158 1.513 1.513 0 00.098-1.244 1.333 1.333 0 00-.829-.83l-.099-.099h-.277l-.395.396c-.223.125-.407.31-.533.533a1.037 1.037 0 00-.178.178v.276l.375.375c.15.323.409.581.731.731l-.059.534zM453.738 24.387l.099-.376.02-.217-.02-.336-.118-.118a1.52 1.52 0 00-.474-.474l-.139-.139-.375-.098-.119-.119-.355.04-.119.118a1.369 1.369 0 00-.454.454l-.138.139-.119.395-.019.217.039.316.099.099c.067.234.205.44.395.593l.138.138.356.118.118.119.316-.04.119-.118c.196-.1.355-.259.454-.454l.296-.257zM454.627 20.89l.533-.533.178-.178c.19-.373.226-.806.099-1.205a1.537 1.537 0 00-.889-.889l-.257-.02-.395.396a1.48 1.48 0 00-.435.434 1.318 1.318 0 00.159 1.541l.592.593h.277l.098.098.04-.237zM450.834 23.735l.119-.395.217-.217.02-.257a1.423 1.423 0 00-1.146-1.146 1.34 1.34 0 00-1.363.652l-.118.118-.04.356-.099.099.06.336.039.316.119.118c.115.184.271.339.454.454l.158.159.375.098.218.02.355-.04.119-.118c.177-.108.326-.257.434-.435l.079-.118zM442.498 19.903l.119-.395.118-.119-.059-.336-.118-.118a1.062 1.062 0 00-.455-.454l-.138-.139-.355-.118-.119-.119-.356.04-.118.118a1.035 1.035 0 00-.435.435l-.158.158-.138.375-.02.217.06.336.118.119c.12.18.275.334.454.454l.158.158.376.099.217.02.336-.06.118-.118c.178-.109.327-.258.435-.435l-.04-.118zM463.595 23.142h-.355l-1.166 1.166-.039.316a1.548 1.548 0 00-.199.594 1.55 1.55 0 00.356 1.174c.137.161.305.292.495.385l.118.118.296.06.099.099.573-.573c.131-.09.245-.205.336-.336a1.475 1.475 0 00.158-1.146 1.31 1.31 0 00-.751-1.817l.079-.04zM463.655 18.263l.632-.632a1.332 1.332 0 00.02-1.166 1.545 1.545 0 00-1.284-.849 1.34 1.34 0 00-1.205.692 1.283 1.283 0 00-.099 1.007 1.474 1.474 0 00.81.81l.099.099a1.281 1.281 0 001.027.04zM446.568.544l.118-.276.079-.198.02-.375v-.198l-.099-.257-.099-.177-.217-.297-.158-.118-.158-.08-.178-.059-.395-.039-.178.02-.256.059-.178.099-.297.257-.118.158-.158.276-.059.178-.04.395v.197l.079.238.119.197.237.277.138.138.158.079.178.06.395.039.197-.04.257-.06.178-.098.277-.237.158-.158zM482.894 7.24h-.197l-.257.06c-.18.105-.33.254-.434.434l-.139.139-.118.395-.139.138.079.356.158.158a1.288 1.288 0 00.751.75l.119.119.335-.02 1.028-1.027.059-.375-.02-.218-.928-.928-.297.02zM481.61 22.036l-.138.138-.04.356.791.79a1.393 1.393 0 001.648.065 1.39 1.39 0 00.406-.42l.119-.119.059-.336-.02-.217-.494-.494a1.394 1.394 0 00-1.264-.474 1.371 1.371 0 00-1.067.711zM504.308 12.693a1.222 1.222 0 00-.06 1.165 1.404 1.404 0 00.85.85l.098.098.356-.04.415-.414c.189-.109.346-.265.454-.454a1.32 1.32 0 00.099-1.008l-.514-.513a1.42 1.42 0 00-1.145-.277c-.24.137-.433.344-.553.593zM455.832-5.836l-.099.099-.039.316.968.968.375.098.119-.118.217-.02.691-.691.079-.277.099-.099-.257-.257a1.477 1.477 0 00-1.963-.38 1.479 1.479 0 00-.506.519l.316-.158zM486.154 1.927a1.38 1.38 0 00-.217.217l.02.336.948.948.355.04h.198l.257-.06a1.46 1.46 0 00.454-.454l.198-.197.079-.474v-.277l-.099-.296-.573-.573a1.286 1.286 0 00-1.62.79zM453.264 16.446l-.039-.237-1.521-1.521-.415-.02-.81.81a1.336 1.336 0 00-.079 1.106l.178.178v.316l.059.257.237.434 1.067 1.067h.316l.256-.06.929-.928a1.451 1.451 0 00.039-1.303l-.217-.1zM469.245 13.898l-.731.73a1.431 1.431 0 00-.098 1.127 1.36 1.36 0 00.81.81l.118.118h.356l.829-.83c.178-.348.213-.752.099-1.126a1.472 1.472 0 00-.81-.81l-.118-.118-.455.099zM433.392-23.615l-.336-.335a1.339 1.339 0 00-1.066-.158 1.234 1.234 0 00-.771.77.914.914 0 00-.178.178l-.098.099v.355l1.007 1.008.257.059.336-.02.257-.06.533-.532a1.204 1.204 0 00.059-1.364zM446.627 76.655l-.494-.494a1.315 1.315 0 00-1.106-.672 1.147 1.147 0 00-.712.198l-.73.73a1.675 1.675 0 00-.139 1.364l.534.533c.246.195.555.294.869.277h.237l.257-.06 1.047-1.047.237-.83zM469.304 86.373l-.355.04-.711.711a1.42 1.42 0 00.197 1.66l.514.513h.276l.099-.099a1.4 1.4 0 00.751-.75 1.307 1.307 0 00.059-1.126 1.526 1.526 0 00-.988-.79l.158-.159zM443.861 42.56l.435-.435.257-.454.059-.257v-.316l-.079-.276-.553-.553a2.978 2.978 0 01-.79-.79l-.178-.178-.415-.178-.158-.158-.296.02-.731.73a1.348 1.348 0 00-.04 1.225l.04.237.395.395.158.514.612.612a1.4 1.4 0 001.284-.138zM439.041 40.585l-.395-.395a1.507 1.507 0 00-1.679-.415l-.612.612-.079.277-.297.296-.138.415v.237l.04.276.987.988h.435l.237-.04 1.58-1.58.06-.296-.139-.375zM449.037 46.471l-.217-.02-.277.08-.632.632c-.091.173-.14.365-.143.56a1.27 1.27 0 00.123.566 1.348 1.348 0 00.771.77l.217.02.355-.04.771-.77a1.286 1.286 0 00-.336-1.68l-.138-.138-.494.02zM467.764 54.886l-.04.395.099.376.02.217a1.119 1.119 0 00.73.731 1.342 1.342 0 001.047-.178l.415-.415c.141-.302.176-.643.099-.968a1.316 1.316 0 00-.691-.691 1.4 1.4 0 00-1.679.533zM478.055 62.57l-.395.04c-.311.133-.559.38-.691.691-.084.367-.035.752.138 1.087a1.37 1.37 0 001.797.573l.297-.297.197-.671-.039-.395-.711-.712-.593-.316zM500.219 110.374l-.277.277-.197.671.059.415a.925.925 0 00.336.336l.474.474.691.178.395-.04c.317-.14.571-.394.712-.711a1.456 1.456 0 00-.119-.988 1.402 1.402 0 00-.892-.797 1.391 1.391 0 00-1.182.185zM466.519 59.726l.553.553a1.313 1.313 0 001.264 0 .895.895 0 00.277-.277l.415-.414.039-.237-.059-.297-1.304-1.303h-.356l-.256.256a1.429 1.429 0 00-.672 1.739l.099-.02zM487.892 76.003l.217.217.534.217.296.02c.149-.04.294-.093.435-.158a1.282 1.282 0 00.651-.85 1.662 1.662 0 00-.316-1.264l-.533-.533-.296-.02c-.133.022-.26.07-.376.139a1.316 1.316 0 00-.657 1.309c.022.172.077.338.164.488l-.376.376.257.059zM475.013 52.575l.158.158.454.06.336-.06c.117-.034.229-.08.336-.138a1.23 1.23 0 00.691-.968 1.51 1.51 0 00-.533-1.245l-.573-.573-.276.08a1.598 1.598 0 00-.771 1.402l.632.632-.454.652zM494.747 86.808l-.158.158a1.344 1.344 0 00-1.107.08 1.842 1.842 0 00-.77 1.086l-.099.098.059.376.771.77a1.342 1.342 0 001.106-.08c.306.155.661.184.988.08l.77-.77.04-.237.019-.376-.039-.237a1.561 1.561 0 00-1.146-.948h-.434zM493.838 101.9l-.118-.119-.099.099-.237.04-.178.177a1.313 1.313 0 00-.691 1.363 1.453 1.453 0 001.205 1.205l.217.02.217-.217a1.16 1.16 0 00.198-.198 1.31 1.31 0 00.691-1.363 1.42 1.42 0 00-1.205-1.007zM480.445 30.313l.475-.474a1.207 1.207 0 00.316-1.067l-.534-.533a1.726 1.726 0 00-.81-.376h-.197l-.316.08-.79.79a1.584 1.584 0 00-.005.836c.072.275.217.524.419.724a1.372 1.372 0 001.442.02zM473.156 88.922l-.513-.514-.099.099-.296.06-.119.118a1.425 1.425 0 00-.711 1.5l1.007 1.008c.203.043.413.036.613-.02l.612-.612.138-.336.139-.138a1.425 1.425 0 00-.297-1.047l-.474-.118zM466.025 76.023a1.31 1.31 0 00-.644 1.555c.061.194.166.372.308.519l.612.612h.198l.335-.06.929-.928-.02-.375-.889-.889-.257-.06h-.197l-.356.04-.019-.414zM492.593 79.4l-.75.751a1.515 1.515 0 00-.04.79l.869.87.336.059a1.76 1.76 0 00.751-.198.86.86 0 00.198-.197l.572-.573-.019-.297-.099-.098a1.453 1.453 0 00-.83-.83 1.37 1.37 0 00-.988-.277z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M461.225 18.44l.415.179.138.138c.294.014.586-.048.85-.178.139-.096.259-.216.355-.355l.336-.336a1.286 1.286 0 00-.494-1.482l-.474-.474-.395-.04-.099.1c-.116.033-.229.08-.336.138a1.507 1.507 0 00-.711 1.501l.553.553-.138.257zM468.178 83.944l-.059-.257-.849-.85-.435-.039-.237.04a.446.446 0 00-.158.158 1.368 1.368 0 00-.672 1.738l.573.573a1.34 1.34 0 001.758-1.482l.079.119zM453.146 56.585a1.373 1.373 0 00-.237-.79l-.692-.692-.099.1-.375.059-.138.138a1.48 1.48 0 00-.731 1.363l.158.158.198.395.592.592c.251.075.516.082.771.02l.869-.869.039-.237-.099-.099-.256-.138zM470.232 85.524c-.106.078-.199.17-.276.277l-.395.395a1.337 1.337 0 00.256 1.323 1.54 1.54 0 001.403.494l.79-.79a1.447 1.447 0 00-.474-1.58 1.312 1.312 0 00-1.304-.119zM443.605 58.936l.434.434.573.099.296-.06a1.307 1.307 0 00.692-.888 1.383 1.383 0 00-.217-1.126l-.672-.672-.296-.02a1.45 1.45 0 00-.376.139 1.316 1.316 0 00-.573 1.6l.257.257-.118.237zM462.252 64.546l-.158.158v.197l.059.336.849.85a1.452 1.452 0 001.245-.06 1.424 1.424 0 00.533-1.876l-.632-.633-.276-.079h-.198a1.456 1.456 0 00-.751.198l-.434.434-.237.475zM475.033 77.642a1.628 1.628 0 00-.079-.513 3.03 3.03 0 00.019-.731c.195-.153.328-.37.376-.612l.494.493.395-.04c.152-.028.299-.082.434-.157a1.314 1.314 0 00.593-1.462l-.81-.81-.257-.06-.138-.137-.395.04-.79.79-.316-.317a1.342 1.342 0 00-1.383-.04 1.149 1.149 0 00-.356.08l-.651.652a2.033 2.033 0 00-.099.375l-.099.375v.198c-.06.232-.074.474-.04.71v.356l.099.1a1.113 1.113 0 00-.079.908l.988.988.276.04.257-.02.415-.1.178-.177c.156-.12.296-.259.415-.415.23-.113.423-.292.553-.514zM445.165 111.777l-.276.276-.099.375-.158.158a1.145 1.145 0 00.276.988 1.266 1.266 0 00.949.395l.138.138.395-.039a1.26 1.26 0 00.494-1.719 1.401 1.401 0 00-.632-.632 1.456 1.456 0 00-1.087.06zM465.215 88.408l.257-.257a1.258 1.258 0 00.079-1.659l-.573-.573a1.535 1.535 0 00-1.086-.02l-.593.593a1.644 1.644 0 00.534 1.64 1.338 1.338 0 001.501.158l-.119.118zM461.205 79.341h-.237l-.119-.118-.395.039a1.425 1.425 0 00-.553 1.936c.163.36.451.648.81.81.367.094.755.045 1.087-.138l.414-.415a1.302 1.302 0 00.099-1.126 1.286 1.286 0 00-.869-.87l-.237-.118zM461.482 85.149l-.434-.435-.316.04-.099.098a1.507 1.507 0 00-.514.514l-.276.276-.079.277a1.398 1.398 0 00.691 1.205 1.254 1.254 0 001.284.02 1.12 1.12 0 00.672-.672.64.64 0 00.118-.355 1.45 1.45 0 00-.375-1.166l-.672.198zM460.968 91.786l-.138-.533-.652-.652-.533-.099-.297.06a.53.53 0 00-.138.138 1.335 1.335 0 00-.805 1.296c.009.181.054.358.134.52.039.32.202.61.454.81a1.343 1.343 0 001.106.277l.889-.888c.03-.251.016-.505-.04-.751l.02-.178zM481.946 109.327a1.288 1.288 0 00-.691 1.086 1.26 1.26 0 00.593 1.146l.355.356.257-.06 1.027-1.027.079-.276a1.394 1.394 0 00-.671-1.106 1.286 1.286 0 00-.949-.119zM429.007 66.699l.79-.79a1.61 1.61 0 00-.237-.83l-.928-.928a1.2 1.2 0 00-.632.04l-.573.572-.119.316-.118.119c-.053.215-.046.44.019.652l.929.928c.202.044.413.037.612-.02l.257-.06zM444.75 64.467l.317.316c.208.041.423.041.632 0l.928-.929v-.197l-.059-.415-.139-.138a1.34 1.34 0 00-1.797-.534l-.217.218-.455.454-.039.237-.099.099.079.316.593.593.256-.02zM443.15 81.395l-.513.514-.158.672.059.375a1.312 1.312 0 002.449-.909 1.261 1.261 0 00-1.313-.828 1.26 1.26 0 00-.524.177zM451.644 77.8l.395.396.356-.04.296-.296a1.368 1.368 0 00.731-1.008 1.462 1.462 0 00-.435-1.185l-.928-.928-.375.059-.988.987-.02.218.04.355.849.85.079.593zM476.632 56.842l.139.138a1.425 1.425 0 001.501.08 1.451 1.451 0 00.691-1.443 1.422 1.422 0 00-1.145-1.146l-.119-.118-.395.079-.672.672a1.426 1.426 0 00-.059 1.6l.198.197-.139-.06zM415.12 55.815l-.81-.81a1.369 1.369 0 00-1.245.06.794.794 0 00-.177.177l-.771.77.02.218a1.43 1.43 0 00.375 1.007l.395.395h.277l.217-.02c.201-.044.393-.118.573-.217a1.363 1.363 0 00.904-.55 1.373 1.373 0 00.242-1.03zM513.888 26.342a1.34 1.34 0 00.671-1.303l-.494-.494a1.256 1.256 0 00-.731-.534 1.256 1.256 0 00-.987.119l-.376.375a1.255 1.255 0 00.119 1.62 1.425 1.425 0 001.798.217zM492.751 128.863a1.424 1.424 0 00-1.837-.454 2.606 2.606 0 00-.375.375l-.415.415a1.345 1.345 0 00.395 1.264l.494.494.474.04-.099-.099.04.316c.255.009.508-.053.731-.178l.533-.533.099-.375.02-.218-.455-.454.395-.593zM424.918 44.634l.237.237.296-.02.099-.098a1.423 1.423 0 00.573-1.837 1.285 1.285 0 00-1.62-.672l-.632.632a1.343 1.343 0 00.296 1.758h.751zM456.07 67.528a1.364 1.364 0 00-.672 1.146l.158.158.198.435.711.71.257.06.217-.02.355-.079.099-.098a1.34 1.34 0 00.672-1.541l-.751-.75a1.34 1.34 0 00-1.244-.02zM475.428 57.573h-.435l-.237.04a.383.383 0 01-.119.118 1.704 1.704 0 00-.849 1.126 1.151 1.151 0 00-.178.178 1.68 1.68 0 00-.138 1.126 1.98 1.98 0 001.442 1.442l.217-.02.376-.06.849-.849.02-.217c.126-.15.221-.322.278-.51.058-.186.078-.382.058-.577l.118-.118-.039-.277a1.493 1.493 0 00-.81-.81l-.178-.177-.375-.415zM520.486 82.916l-.336-.335-.237-.04-.099-.099-.316.04-.277.276a1.28 1.28 0 00-.592.91 1.253 1.253 0 00.316 1.027l.178.177a1.25 1.25 0 001.343.316l.83-.83.039-.316a1.282 1.282 0 00-.355-.987l-.494-.139zM500.298 78.808l.138-.139-.02-.296-.098-.099a1.463 1.463 0 00-.889-.889 1.344 1.344 0 00-1.126.1l-.119.118-.612.612a1.23 1.23 0 00.276 1.067l.652.652h.237c.257-.017.507-.09.731-.218a.842.842 0 00.198-.197l.336-.336.296-.375zM478.371 78.334l-.494.494-.019.296.039.237.079.277.771.77c.158.088.333.142.513.158l.316-.08.474-.276.099-.098.119-.316.059-.297a1.427 1.427 0 00-.158-1.066l-.751-.751-.098.099a1.539 1.539 0 00-.949.553zM483.389 87.302l.118-.119a1.396 1.396 0 00.691-.928 1.411 1.411 0 00-.276-1.185l-1.165-1.166-.238-.04h-.434l-.909.91-.059.256v.237l.079.395.118.119a1.29 1.29 0 00.909.671c.194.25.454.442.751.553l.099.1.316.197zM453.363 69.88l-.099-.1-.079-.316a1.508 1.508 0 00-1.995-.573l-.731.731-.02.218.06.454 1.145 1.146c.3.047.606-.009.869-.158.165-.11.306-.25.415-.415l.415-.415.02-.573zM474.045 65.514l-.158.158a1.396 1.396 0 00-.671 1.54l.078.277.87.87c.35.038.704-.038 1.007-.218.073-.058.14-.125.198-.198l.592-.592v-.316l-.099-.1a1.868 1.868 0 00-.948-.947l-.118-.119-.751-.355zM480.683 73.89l-.751.75a1.031 1.031 0 00-.257.454 1.285 1.285 0 00-.395.395l-.474.474.02.376.118.118a1.742 1.742 0 001.027 1.027l.297-.02a1.364 1.364 0 001.106.04l.751-.75c.157-.305.226-.647.197-.988l-.178-.178-.039-.277-.079-.316a1.51 1.51 0 00-1.343-1.106zM477.186 69.681l-.672.672v.316l.099.099a1.389 1.389 0 00.738.84 1.401 1.401 0 001.119.03l.572-.574a1.45 1.45 0 00-.454-1.52 1.368 1.368 0 00-1.541-.12s.198.198.139.257zM469.047 65.731l.395.395.336-.02.277-.276a1.373 1.373 0 00.651-1.719l-.612-.612-.336-.02a1.647 1.647 0 00-.711.198.989.989 0 00-.197.197l-.494.494-.02.217-.099.099.06.336.651.652.099.059zM509.464 100.3a1.31 1.31 0 00-.672.908 1.59 1.59 0 00.336 1.324l.474.474c.379.094.78.044 1.126-.138a1.57 1.57 0 00.731-.968l.019-.257-.019-.217-.139-.139a1.482 1.482 0 00-.908-.908 1.39 1.39 0 00-.948-.079zM466.558 42.026l-.217-.02-.336.02-.118.12c-.044.048-.09.094-.139.137a1.566 1.566 0 00-.79.988 1.407 1.407 0 00.296 1.126l.336.336a1.255 1.255 0 001.462.158c.153-.093.282-.222.375-.375l.119-.119.118-.316.119-.119v-.237l-.119-.118a1.76 1.76 0 00-.197-.632l-.534-.534-.375-.415zM460.375 39.715l.277.277.217.02.336-.02.118-.118.119-.119a1.73 1.73 0 00.948-1.62l-1.027-1.027a1.426 1.426 0 00-1.067.119 2.08 2.08 0 00-.652.652l-.395.395-.02.217.04.395.691.691.415.139zM480.149 65.099l-.178.178a1.512 1.512 0 00-.731.968 1.49 1.49 0 00.029.626c.055.205.153.395.287.559l.691.691.297-.02.098-.098a.97.97 0 00.178-.178 1.513 1.513 0 00.731-.968 1.676 1.676 0 00-.316-1.225l-.672-.671h-.316l-.098.138zM472.82 34.343l-.118.118a1.144 1.144 0 00.019 1.363l.553.553.277.08.138.137.415-.059a1.338 1.338 0 00.672-1.066 1.511 1.511 0 00-.178-.81 1.348 1.348 0 00-.794-.654 1.341 1.341 0 00-1.023.1l.039.238zM476.968 7.873l.139.138.217-.02c.149-.04.294-.093.434-.158a1.422 1.422 0 00.712-1.382 1.59 1.59 0 00-1.245-1.244l-.118-.12-.218.02-.237.04-.197.197c-.205.111-.38.27-.51.462a1.444 1.444 0 00-.241.645 1.565 1.565 0 00.711 1.382l.553.04zM484.436 78.235l.652.652h.276l.099-.099c.078-.067.15-.14.217-.217a1.335 1.335 0 00.731-1.047 1.6 1.6 0 00-.14-.772 1.601 1.601 0 00-.492-.61l-.178-.179-.474-.118-.257.02-.296.098-.533.534a1.535 1.535 0 00.395 1.738zM518.234 34.165l-.277.276-.079.277a.952.952 0 00-.118.553l.039.276.672.672.632.158.395-.04c.311-.132.559-.38.692-.69a1.239 1.239 0 00-.119-.988 1.322 1.322 0 00-.813-.641 1.31 1.31 0 00-1.024.147zM488.366 30.649a1.54 1.54 0 00-.711 1.817l.988.988.454-.02.968-.968.04-.277v-.237l-.079-.316-.297.02.178-.178-.178-.454-.592-.593-.514-.04-.257.06v.198zM470.272 55.143l.83-.83.059-.256.119-.119-.04-.355-1.086-1.087-.376.02-.138.138a1.432 1.432 0 00-.691.889c-.079.39-.001.794.217 1.126l.672.672.098-.1.336-.098zM467.309 25.078a1.256 1.256 0 00-.454 1.482l.553.553a1.26 1.26 0 001.383.118c.145-.103.272-.23.375-.375l.119-.119.098-.296.218-.217-.099-.099-.099-.296a1.284 1.284 0 00-1.56-.731l-.158.158-.376-.178zM441.728 100.043l-.277-.277h-.197l-.356.04-1.126 1.126.02.375.119.119a1.397 1.397 0 001.363.849l.296-.059.79-.79.059-.297-.158-.158a1.366 1.366 0 00-.237-.79l-.138-.138h-.158zM510.095 87.816l-.79.79-.119.316-.019.257.039.394-.118.119a1.45 1.45 0 00.829.83 1.231 1.231 0 001.047-.1l.553-.552a1.31 1.31 0 00-.217-1.284 1.567 1.567 0 00-1.205-.77zM493.72 91.983l-.158-.158-.277-.04-.237-.236a1.509 1.509 0 00-1.738-.316l-.731.73-.02.337.099.098a1.781 1.781 0 00.83 1.146c.236.125.505.173.77.138l.237.237a1.286 1.286 0 001.422.119 1.35 1.35 0 00.237-.237l.573-.573-.099-.099-.039-.316-.435-.434-.434-.396zM437.442 80.388l.987-.988-.019-.335-.119-.119a1.252 1.252 0 00-.81-.81 1.339 1.339 0 00-1.185.119c-.049.043-.095.09-.138.138l-.514.514-.039.237v.197l.039.356 1.087 1.086.395-.04.316-.355zM436.315 24.703l.672-.672.059-.296c.069-.232.069-.48 0-.711l-.81-.81-.335-.06h-.198a1.14 1.14 0 00-.434.159 1.403 1.403 0 00-.692 1.857 1.402 1.402 0 001.857.69l-.119-.157zM458.795 32.683a1.322 1.322 0 001.166.257l.592-.593.139-.335.158-.158a1.596 1.596 0 00-.277-1.028l-.513-.513-.277-.04-.099-.099-.375.1-.099.098a1.312 1.312 0 00-.612 1.64l.434.434-.237.237zM480.05 20.08l.316.317c.223.054.457.04.672-.04l.691-.691.079-.277.02-.217-.059-.415-.119-.119a1.311 1.311 0 00-1.817-.553 1.021 1.021 0 00-.198.198l-.592.592v.198l.039.316.593.593.375.099zM461.067 75.608l.434-.435a1.342 1.342 0 00-.691-1.165 1.231 1.231 0 00-1.304-.04 1.2 1.2 0 00-.612.89l-.119.118.04.316.099.099a1.272 1.272 0 00.75.75 1.397 1.397 0 001.087-.099c.077-.163.184-.31.316-.434zM476.534 105.021l.277-.277-.04-.434a2.18 2.18 0 00-.178-.692l-.237-.237-.513-.276-.316-.04-.356.04-.632.632a1.535 1.535 0 00-.039 1.699l.691.691h.197l.356-.04.731-.73.059-.336zM440.603 46.649l.395.395c.216.032.437.026.652-.02l.967-.968a.87.87 0 00.317-.04l.829-.829.119-.395.019-.257a1.277 1.277 0 00-.697-1 1.285 1.285 0 00-1.219.033 1.377 1.377 0 00-.75.75l-.316.04-.119.118a1.426 1.426 0 00-.584.62c-.127.26-.172.555-.127.842l.454.454.06.257zM445.6 47.617c.167.391.478.702.869.87a1.482 1.482 0 001.185-.08l.731-.73a1.643 1.643 0 00-.355-1.265l-.494-.494-.237-.04-.099-.098-.336.06-1.165 1.165.02.375-.119.237zM482.539 49.83a1.434 1.434 0 00-.674.879 1.431 1.431 0 00.16 1.096c.141.317.394.57.711.71a1.23 1.23 0 001.008-.138l.276-.276.198-.672-.059-.415-.751-.75-.672-.158-.197-.277zM502.333 83.173l-.475-.474-.592-.079-.336.06-.751.75-.059.257v.237l.079.395.138.138a1.258 1.258 0 001.699.514 1.196 1.196 0 00.632-1.185l-.098-.534-.237-.079zM464.366 50.62l.573-.573a1.51 1.51 0 00-.494-1.56 1.34 1.34 0 00-1.541-.12c-.123.086-.23.193-.316.317l-.237.237-.02.217a1.399 1.399 0 00.672 1.146 1.425 1.425 0 001.363.335zM452.948 39.913l.395.395.375.02.297-.297a1.536 1.536 0 00.572-2.074l-.829-.83-.218-.019-.355.04-.296.098-.514.514-.138.415-.119.118c.032.335.119.662.257.968l.415.415.158.237zM483.013 87.756l-.553-.553h-.276a.863.863 0 00-.336.138 1.428 1.428 0 00-.731 1.521l.198.395.553.553.434.08a1.48 1.48 0 00.731-.178.988.988 0 00.217-.218l.475-.474a1.819 1.819 0 00-.356-1.264h-.356zM463.398 53.681l-.731-.73a1.342 1.342 0 00-.751-.04l-.829.83-.04.237.02.454.889.889.276.04.218.02.414-.1.099-.098a1.26 1.26 0 00.435-1.502zM486.944 83.983a1.17 1.17 0 00-1.027.119l-.277.276-.099.376-.158.158c-.046.235-.011.478.099.691l-.395.395-.02.218.04.276.099.099c.168.378.471.68.849.85l.099.098.336.02.355-.356.099-.099a1.31 1.31 0 00.672-1.382l-.178-.178.335.336a1.4 1.4 0 00-.059-1.126 1.562 1.562 0 00-.77-.77zM436.02 57.118l-.474-.474a1.26 1.26 0 00-1.265-.118l-1.126 1.126.04.395.118.118c.163.29.402.529.692.692l.099.099a1.34 1.34 0 001.619.316l.613-.613a1.63 1.63 0 00-.316-1.54zM445.521 32.012l.395.395c.195.043.397.043.592 0l.613-.613.158-.355.138-.139a1.29 1.29 0 00-.296-1.047l-.494-.493-.257-.02-.316.04-.119.118a1.45 1.45 0 00-.73 1.52l.454.455-.138.139zM425.175 108.972a.557.557 0 00-.178.177l-.474.475.019.296.099.099a1.432 1.432 0 00.889.888 1.37 1.37 0 001.205-.098.865.865 0 00.198-.198l.671-.671v-.317l-.099-.098a1.446 1.446 0 00-.888-.889 1.339 1.339 0 00-1.442.336zM506.461 87.737l-.178-.178a1.26 1.26 0 00-1.225-.198l-.572.573-.139.336-.118.118a1.316 1.316 0 00.257 1.008l.533.533.276.04.099-.1.376-.098a1.287 1.287 0 00.632-1.58l-.376-.376.435-.078zM465.117 96.685l-.968.968-.02.217.059.415.139.138a1.288 1.288 0 001.284.709 1.28 1.28 0 00.493-.156c.089-.07.168-.149.238-.237l.335-.336.04-.237-.04-.316-.118-.118a1.454 1.454 0 00-1.442-1.047zM488.07 41.296a2.35 2.35 0 00-.336.336 1.254 1.254 0 00.924 1.83c.266.04.537-.006.775-.132a1.314 1.314 0 00.652-1.086v-.198l-1.008-1.007h-.197l-.099.099-.336.138-.336.336-.039-.316zM494.312 58.916c-.117.079-.217.18-.296.296l-.257.257-.039.237a1.478 1.478 0 00.691 1.126 1.454 1.454 0 001.343.08l.593-.593a1.506 1.506 0 00-.474-1.541 1.453 1.453 0 00-1.561.138zM473.907 97.14l-.119.118-.02.256c-.011.372.124.734.376 1.008l.415.415.296.02.099-.1c.069-.062.135-.128.197-.197a1.367 1.367 0 00-1.244-2.43l-.494.494.494.415zM523.172 73.277l-.356-.356-.335-.02a1.002 1.002 0 00-.178.178 1.37 1.37 0 101.225 2.45l.612-.613.059-.296.119-.119c0-.28-.082-.555-.237-.79l-.119-.118-.79-.316zM452.751 84.319l.336.336.256.02.415-.06.138-.138a1.48 1.48 0 00.751-1.343l-.059-.257-.85-.85-.434-.039-.277.04-.138.138a1.4 1.4 0 00-.652 1.758l.158.158.356.237zM434.34 91.628l-.178.652.02.375.889.889.296.099.395-.08a1.224 1.224 0 00.731-.73 1.397 1.397 0 00-2.034-1.56l-.119.355zM457.59 103.5l-.573.573-.039.237-.119.118.04.356.869.869c.237.076.49.083.731.02l.276-.079.83-.83.02-.296-.02-.218-.119-.355-.138-.138a1.48 1.48 0 00-1.758-.257zM456.642 58.699l.059.336 1.008 1.007h.355l.158-.158a1.53 1.53 0 00.731-.928 1.372 1.372 0 00-.257-1.127l-.671-.671h-.237c-.248.019-.49.086-.712.197a.838.838 0 00-.197.198l-.474.474-.04.237.277.435zM466.677 108.004l-.138-.139-.257.06-.395.197-.731.731-.059.296-.02.257.119.435.158.158c.19.245.452.425.75.513l.237-.237-.019.336.533.02.277-.079.434-.435.237-.474.079-.276a2.377 2.377 0 00-.889-1.482l-.316.119zM419.13 79.934l-.198-.198-.316-.079a1.396 1.396 0 00-.79.158 1.228 1.228 0 00-.474 1.62l.296.296a1.284 1.284 0 001.482.455l.75-.75c.084-.256.097-.529.04-.791l-.198-.198-.592-.513zM451.249 88.428l-.513-.514h-.198l-.375.06-.118.118a1.452 1.452 0 00-.573 1.956l.592.592.257.06.217.02.356-.04.968-.968a1.733 1.733 0 00-.336-1.245l-.277-.04zM439.713 74.087l.138.138.257.06.119.118.375-.06.83-.83.059-.256.02-.217-.04-.356-.77-.77-.277-.08a1.398 1.398 0 00-.889.179l-.138.73-.217.731.316.316.217.297zM500.712 41.849l.711-.711.04-.237v-.356l-.119-.118a1.43 1.43 0 00-.731-.832 1.423 1.423 0 00-1.106-.057l-.77.77v.356l.019.217c.159.374.456.671.83.83a1.339 1.339 0 001.126.138zM532.417 28.357a1.45 1.45 0 00-1.205.099.822.822 0 01-.178.178l-.395.395-.02.256a1.535 1.535 0 00.83 1.265 1.227 1.227 0 001.225 0 1.31 1.31 0 00.651-.652l.238-.237.019-.376-.118-.118a1.519 1.519 0 00-1.047-.81zM512.13 55.32l-.593.593-.059.534.059.256a1.34 1.34 0 001.462.79l.355-.078.632-.633.099-.533-.02-.296a1.426 1.426 0 00-1.6-.889v.197l-.118-.118-.217.178zM492.752 49.572l.415-.059a1.35 1.35 0 00.651-1.086 1.781 1.781 0 00-.158-.711 1.358 1.358 0 00-.791-.687 1.37 1.37 0 00-1.046.074l-.474.474a.949.949 0 00-.118.514l.197.198c.024.184.092.36.198.513l.296.297.83.474zM503.577 67.983l-.178-.178a1.31 1.31 0 00-.553-.158l-.277.079-.454.217-.415.415-.059.257-.02.494.079.276.751.75.257.06.118.119.395-.04.138-.138a1.425 1.425 0 00.731-1.482l-.513-.513v-.158zM495.379 52.081l-.217-.02-.217.218c-.132.09-.246.204-.336.335l-.376.06-.118.118a1.348 1.348 0 00-.691.946 1.34 1.34 0 00.316 1.128l.454.455.237.04.336-.06.434-.435.237.04.237-.237a1.25 1.25 0 00.629-.747 1.259 1.259 0 00-.095-.972 1.49 1.49 0 00-.553-.553l-.277-.316zM471.339 40.92a1.23 1.23 0 001.383.119c.137-.098.257-.218.355-.356l.158-.158v-.197a1.304 1.304 0 00-.731-.968 1.535 1.535 0 00-1.343.04 1.259 1.259 0 00-.573 1.6l.198.197.553-.277z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M496.584 51.31l.138-.137a1.343 1.343 0 00-.395-1.225 1.368 1.368 0 00-1.264-.277l-.83.83-.039.237.039.395.119.119.178.375.592.592.494.02.296-.059.553-.553.119-.316zM525.523 48.684l-.217.217-.317.316a1.487 1.487 0 00-.355 1.343 1.282 1.282 0 001.284.968l.138-.138.435-.237.79-.79.079-.316.02-.218-.06-.375-.138-.138a1.285 1.285 0 00-1.659-.632zM515.409 62.195l-.554-.553a1.283 1.283 0 00-1.224.04 1.642 1.642 0 00-.257.256l-.454.454-.04.238.316.316.277.474.355.355.316.119.237.04.415-.02.296-.297a1.275 1.275 0 00.652-.85 1.229 1.229 0 00-.335-.572zM494.628 28.337l.434.435.415.02.119-.119a.391.391 0 00.118-.118 1.473 1.473 0 00.711-.909 1.527 1.527 0 00-.296-1.205l-.711-.711-.336.02-.138.138a1.369 1.369 0 00-.573 1.837l.138.138.119.474zM506.895 51.943l-.039-.316-.099-.099a1.507 1.507 0 00-.869-.869 1.368 1.368 0 00-1.146.119l-.118.118-.474.474-.079.277a1.308 1.308 0 00.276 1.066l.435.435h.237c.255.005.506-.057.731-.178.083-.06.156-.134.217-.217l.553-.553.375-.257zM516.475 45.326a1.45 1.45 0 00-.395-1.186l-.474-.474-.217-.02-.296.06-.099.098a1.423 1.423 0 00-.731 1.482l.928.928c.358.102.741.059 1.067-.118.151-.228.227-.497.217-.77zM478.648 46.768l.414-.06a1.308 1.308 0 00.672-.908 1.202 1.202 0 00-.119-.909c.007-.12-.006-.24-.039-.356l-.593-.592-.316-.119-.138-.138a1.68 1.68 0 00-.593.04l-.948.948c-.032.21-.019.426.04.632l.138.138v.198c.17.366.464.66.83.83l.118.118.534.178zM484.475 58.086l.454.455c.342.074.698.025 1.007-.139a1.372 1.372 0 00.712-1.264 1.517 1.517 0 00-.81-1.284l-.119-.118-.118.118a1.323 1.323 0 01-.178.178 1.512 1.512 0 00-.731.968 1.451 1.451 0 00.296 1.205l-.513-.119zM490.994 66.857a1.473 1.473 0 00-.396-1.225l-.474-.474a1.34 1.34 0 00-.928.138c-.204.105-.37.27-.474.474l-.395.395-.02.336.119.119a1.37 1.37 0 002.666-.138l-.098.375zM507.764 77.11l-.355-.356-.237-.04-.356.04-.889.888-.059.257-.119.119.06.375.908.909.277.04.217-.02.415-.1.178-.177a1.372 1.372 0 00.569-1.457 1.362 1.362 0 00-.233-.499l-.376.02zM499.883 33.987a1.358 1.358 0 00.805.684 1.363 1.363 0 001.052-.091l.632-.633.059-.296a1.376 1.376 0 00-1.066-1.62l-.218-.02-.118.12a1.034 1.034 0 00-.316.118 1.421 1.421 0 00-.83 1.738zM486.154 67.272a1.506 1.506 0 00-.434-1.58 1.316 1.316 0 00-1.482-.178l-.593.592a1.375 1.375 0 00.119 1.146 1.565 1.565 0 001.047.73l.355-.039.672-.671h.316zM480.919 38.965l.534.533.316-.04.099-.098a.685.685 0 00.158-.158 1.402 1.402 0 00.869-.87 1.425 1.425 0 00-.277-1.145l-.454-.454-.217.02a1.312 1.312 0 00-.731.177l-.198.198-.533.533.02.218c.015.367.146.72.375 1.007l.039.079zM487.062 63.598l.79-.79a1.45 1.45 0 00-.019-.652l-.909-.91a1.23 1.23 0 00-.652.02l-.968.969a1.34 1.34 0 00.02.69l.909.91c.214.055.44.048.651-.02l.178-.217zM489.709 61.148l.119.119.375.02.119-.12c.066-.05.126-.11.178-.177a1.513 1.513 0 00.73-.968 1.671 1.671 0 00-.316-1.225l-.395-.395-.237-.04a1.73 1.73 0 00-.592.198 1.367 1.367 0 00-.711 1.383l.79.79-.06.415zM484.988 61.484l.929-.928a1.43 1.43 0 00-.455-1.561 1.28 1.28 0 00-1.481-.178l-.514.514-.118.316-.336.336c-.052.215-.046.44.02.652l.928.928c.211.032.427.02.632-.04l.395-.04zM480.05 53.74l.79-.79a1.484 1.484 0 00-.039-.632l-.929-.928a.92.92 0 00-.612.02l-.928.928a1.45 1.45 0 00.019.652l.929.928c.216.043.439.037.652-.02l.118-.157zM496.031 67.746l-.514-.514h-.197l-.336.06-.751.75a1.559 1.559 0 00-.831 1.578c.025.21.093.412.199.595l.711.711.316.04.099-.1c.158-.04.31-.1.454-.177.36-.162.648-.45.81-.81a1.378 1.378 0 00.61-.897 1.376 1.376 0 00-.234-1.058l-.336-.178zM498.737 63.815l.118.118.237-.237a1.23 1.23 0 00.632-.987 1.428 1.428 0 00-.612-1.403 1.286 1.286 0 00-1.774.261 1.373 1.373 0 00-.023 1.576l.138.138a1.147 1.147 0 001.284.534zM493.936 39.656l.277-.158.178-.138.217-.296.099-.178.059-.257.04-.197-.06-.376-.059-.178-.079-.158-.118-.158-.316-.237-.178-.098-.257-.06-.178-.02-.395.04-.197.08-.257.137-.178.139-.217.296-.099.178-.059.256-.04.198.059.375.04.198.099.138.118.158.316.237.178.099.237.079h.198l.395-.04.177-.059zM520.88 64.94l-.296-.296a1.089 1.089 0 00-.612.02l-.593.593-.138.336-.059.256a1.365 1.365 0 00.296 1.047l.474.475.237.039.099-.099.316-.04.987-.987.02-.217-.039-.356-.632-.632-.06-.138zM511.794 76.714l-.711.71v.198l.039.356.119.119a1.399 1.399 0 001.916.81l.751-.751.039-.316-.039-.237a1.368 1.368 0 00-1.416-1.178c-.225.01-.445.075-.639.19l-.059.099zM535.953 81.158a1.283 1.283 0 00-.692.929 1.353 1.353 0 00.336 1.165l.494.494.217.02h.198c.22.001.438-.053.632-.158.314-.145.566-.397.711-.711l.059-.257-.098-.099a1.347 1.347 0 00-.83-.83 1.315 1.315 0 00-1.027-.553zM505.196 39.3l-.691.692-.06.296.099.1a.873.873 0 00.138.374l.692.692.237.04.434-.04.771-.77.059-.258a1.396 1.396 0 00-.276-1.027 1.564 1.564 0 00-1.403-.098zM526.49 62.254c-.083.061-.156.135-.217.218l-.494.493.04.316.908.91a1.2 1.2 0 00.652-.02l.711-.712.079-.316v-.197l-.059-.376-.138-.138a1.342 1.342 0 00-1.482-.178zM490.994 56.664l.276-.276.02-.336.158-.158.02-.218-.04-.355-.869-.87-.296-.059-.099-.098-.336.06-.118.118a1.308 1.308 0 00-.672.908.594.594 0 00-.04.277l-.395.395-.039.474.039.276-.019.297.059.257.889.889.494.02.296-.06a1.457 1.457 0 00.731-1.087l-.059-.454zM505.69 63.183l-.277-.277-.098.099-.356.04-.119.118a1.538 1.538 0 00-.79.79 1.428 1.428 0 00.218 1.126l.75.75.356-.039.118-.118a1.443 1.443 0 00.711-.91 1.564 1.564 0 00-.217-1.125l-.138-.138-.158-.316zM495.971 12.219l.02-.218-.04-.237a1.33 1.33 0 00-.77-.77 1.397 1.397 0 00-1.086.099l-.218.217-.474.474-.039.237v.198l.257.256.197.395.593.593h.513l.257-.06.316-.315a1.34 1.34 0 00.474-.87zM452.81 103.697l.356-.355a1.29 1.29 0 00-.554-1.185 1.17 1.17 0 00-.691-.178l-.118-.119-.415.02-.297.296a1.644 1.644 0 00-.869 1.067v.237l.02.217.138.139c.146.278.373.506.652.651l.435.435h.197l.158-.158a1.16 1.16 0 00.198-.198l.612-.612.178-.257zM466.617 124.142l-.335-.336-.238-.039-.098.099-.297.059-.098.099a1.337 1.337 0 00-.731 1.481l.829.83c.171.049.35.064.526.044.176-.021.347-.076.502-.163a1.23 1.23 0 00.651-.928 1.255 1.255 0 00-.375-1.166l-.336.02zM468.929 73.514l.257-.06.454-.217.593-.592v-.514l-.04-.276a2.79 2.79 0 01-.197-1.106l-1.008-1.008-.474-.04-.257.06a1.254 1.254 0 00-.671.987l-.158.158-.04.474-.098.1-.159.552.001.316.138.336-.119.119a1.328 1.328 0 00.735.74 1.316 1.316 0 001.043-.03zM465.966 69.247l.118-.118.04-.316v-.198a1.566 1.566 0 00-1.225-1.225l-.336.02-.533.217-.988.988-.039.237.039.435.988.987.276.04.218-.02.316-.04.533-.217.632-.632-.039-.158zM471.181 79.598l-.435-.435h-.237l-.435.04-.158.158a1.4 1.4 0 00-.632.948 1.343 1.343 0 00.336 1.047l.494.494.237.04.217-.02.336-.1a1.283 1.283 0 00.632-1.58l-.158-.158-.197-.434zM482.362 96.823l-.297.296-.098.336-.159.158a1.093 1.093 0 00.297 1.008 1.147 1.147 0 00.948.395l.138.138.395-.079a1.144 1.144 0 00.672-.672 1.2 1.2 0 00-.099-.968 1.424 1.424 0 00-1.797-.612zM508.752 97.89l-.395-.04a1.286 1.286 0 00-.791.593 1.51 1.51 0 00-.118 1.067 1.33 1.33 0 00.605.829 1.34 1.34 0 001.015.158l.276-.276.356-.593.059-.415-.77-.77-.237-.553zM518.826 149.644l-.296.296-.356.593-.039.395.177.454.297.297.612.335.375.06c.338-.096.627-.314.81-.613a1.34 1.34 0 00.119-.987 1.37 1.37 0 00-1.699-.83zM498.263 92.379l.533.533a1.311 1.311 0 001.244.336 1.38 1.38 0 00.257-.257l.672-.672.039-.316-.098-.099-.099-.296-.79-.79-.356-.079-.118.119a1.397 1.397 0 00-1.067 1.54l-.217-.02zM514.994 113.337l.613.612.316.079c.146.007.292-.006.434-.039a1.331 1.331 0 00.771-.77 1.538 1.538 0 00-.02-1.284l-.77-.771c-.128 0-.254.02-.376.059a1.314 1.314 0 00-.463.226 1.3 1.3 0 00-.494.881c-.019.173-.002.347.049.513l-.099.099.039.395zM508.179 87.44l.178.178.454.178.297.02c.139-.01.278-.029.415-.06a1.342 1.342 0 00.829-.83 1.47 1.47 0 00-.217-1.323l-.632-.632h-.277a1.485 1.485 0 00-1.126 1.126v.237l.099.415.178.178-.198.513zM519.044 125.327h-.277a1.416 1.416 0 00-.525-.2 1.377 1.377 0 00-.561.023 1.79 1.79 0 00-.949.948l-.118.118-.02.336v.198l.652.652a1.333 1.333 0 001.086.177c.26.215.593.321.929.297l.889-.889.118-.395.02-.218a1.592 1.592 0 00-.889-1.205l-.355.158zM514.618 139.827l-.217-.218h-.237l-.276.04a1.346 1.346 0 00-1.087 1.086 1.514 1.514 0 00.83 1.502l.237.237.099-.099.316-.04a1.344 1.344 0 001.086-1.086 1.484 1.484 0 00-.751-1.422z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M518.826 67.153l.395-.395a1.223 1.223 0 00.553-.988l-.217-.217a1.766 1.766 0 00-.632-.632l-.099-.099-.336-.02-.77.77a1.642 1.642 0 00.039 1.62 1.371 1.371 0 001.067-.039zM497.571 122.345l-.651-.652-.336-.02-.099.099a1.42 1.42 0 00-1.047 1.284l.04.237.158.395.592.593c.188.079.389.119.593.118l.988-.988a1.365 1.365 0 00-.02-1.086l-.218.02zM493.838 108.063a1.312 1.312 0 00-.83 1.936l.751.75.336.02h.197l.948-.948.06-.375-1.146-1.146-.336-.02.02-.217zM518.827 117.722l-.81.81c-.144.24-.225.511-.237.79l.138.139.138.335.474.474.316.119c.246.064.505.064.751 0 .069-.063.135-.128.198-.198l.691-.691.039-.276a1.754 1.754 0 00-.572-1.047 1.436 1.436 0 00-1.126-.455zM502.944 50.955l.474.474c.276.087.57.093.85.02a.949.949 0 00.454-.217l.356-.356a1.258 1.258 0 00-.138-1.56l-.494-.494-.356-.119-.118-.118a2.362 2.362 0 00-.336.059 1.452 1.452 0 00-1.047 1.284l.02.257.177.533.158.237zM494.055 116.261l-.02-.257-.118-.435-.553-.553-.415-.138-.237-.04-.237.04a1.428 1.428 0 00-1.067 1.541l.692.691a1.385 1.385 0 001.066.158 1.392 1.392 0 00.889-1.007zM486.075 86.037a1.502 1.502 0 00-.04-.83l-.849-.849-.375-.02-.119.12a1.454 1.454 0 00-1.086 1.086v.237l.138.454.573.573c.232.142.498.217.77.217l.257-.059.415-.217.553-.553-.237-.159zM495.635 118.275a1.709 1.709 0 00-.355.158l-.415.415a1.368 1.368 0 00-.079 1.344 1.515 1.515 0 001.244.809l.257-.059.514-.513c.156-.248.234-.537.224-.829a1.464 1.464 0 00-.283-.811 1.345 1.345 0 00-1.107-.514zM476.178 86.097l.454.454.534.218.316.04a1.338 1.338 0 00.889-.692c.176-.357.204-.769.079-1.146l-.692-.691-.296-.1a.78.78 0 00-.375.06 1.315 1.315 0 00-.968 1.403l.158.158-.099.296zM492.93 96.013l-.277.277-.039.355-.02.218.671.671a1.513 1.513 0 001.245.257 1.455 1.455 0 00.988-1.699l-.949-.948a1.51 1.51 0 00-.731.02l-.592.593-.296.256zM465.018 137.753l-.731.73a1.12 1.12 0 00.059 1.008 1.254 1.254 0 00.731.731l.138.138.415.059a1.285 1.285 0 00.889-1.56 1.504 1.504 0 00-.494-.731 1.32 1.32 0 00-1.007-.375zM490.085 119.875l.257-.256a1.227 1.227 0 00.474-1.581l-.514-.513a1.568 1.568 0 00-1.047-.297l-.652.652a1.597 1.597 0 00.099 1.719 1.34 1.34 0 001.422.514l-.039-.238zM488.366 110.098l-.237-.04-.118-.119-.415-.059a1.433 1.433 0 00-1.008 1.758c.108.385.363.712.712.909a1.315 1.315 0 001.046.138l.455-.454a1.372 1.372 0 00.355-1.067 1.256 1.256 0 00-.671-1.027l-.119-.039zM487.141 115.865l-.573-.572-.316-.04c-.229.057-.44.173-.612.336l-.593.593a1.424 1.424 0 00.376 1.323 1.304 1.304 0 001.079.361 1.32 1.32 0 00.955-.618c.094-.077.18-.163.257-.257a1.26 1.26 0 00-.099-1.205l-.474.079zM485.146 122.127l-.019-.533-.08-.316-.533-.533-.513-.198-.297-.02-.118.119a1.33 1.33 0 00-.88.582 1.33 1.33 0 00-.187 1.038c-.031.309.053.618.237.869a1.303 1.303 0 00.988.553l.098-.099.297-.059.711-.711c.089-.228.143-.468.158-.711l.138.019zM501.285 144.192a1.32 1.32 0 00-.935.892 1.316 1.316 0 00.283 1.261l.514.514.256-.02 1.205-1.205a1.31 1.31 0 00-.355-1.225 1.37 1.37 0 00-.968-.217zM460.158 90.107l.375-.178.514-.514c.104-.2.152-.426.138-.652l-.138-.138-.158-.355-.533-.534a1.4 1.4 0 00-.633-.118l-.908.909c-.099.195-.14.414-.119.632l.119.118.178.376.513.513c.191.093.4.14.612.138l.04-.197zM475.961 91.727l.317.316c.19.093.4.14.612.138l.276-.079.889-.889.02-.375-.02-.217a1.303 1.303 0 00-.61-.815 1.315 1.315 0 00-1.009-.134.823.823 0 00-.198.198l-.731.73-.039.317.335.336.158.474zM470.351 107.767l-.612.612-.336.612-.02.376a1.337 1.337 0 001.64 1.007 1.37 1.37 0 00.968-1.284 1.324 1.324 0 00-.116-.595 1.321 1.321 0 00-1.524-.728zM479.438 106.305l.434.434.356.079h.197a.98.98 0 00.257-.059 1.362 1.362 0 00.938-.829 1.373 1.373 0 00-.128-1.245l-.849-.849-.356-.04-1.106 1.106-.059.376.059.256.276.277-.019.494zM508.653 92.063l.118.118a1.37 1.37 0 001.423.435 1.455 1.455 0 001.027-1.225 1.426 1.426 0 00-.731-1.442l-.138-.138-.395-.04-.218.02-.414.415a1.42 1.42 0 00-.455 1.52l.119.12-.336.216zM449.274 76.22l-.81-.81a1.343 1.343 0 00-1.225-.276l-.158.158-.79.79c-.087.358-.045.736.119 1.067l.513.513.257.06.119.118a3.17 3.17 0 00.632-.08 1.315 1.315 0 001.343-1.54zM552.25 71.34a1.31 1.31 0 001.027-1.026l-.079-.316-.257-.257a1.311 1.311 0 00-.652-.652 1.173 1.173 0 00-.968-.099l-.257.257a1.315 1.315 0 00-.276 1.62 1.455 1.455 0 001.462.474zM507.093 165.763a1.398 1.398 0 00-1.66-.869 1.712 1.712 0 00-.493.217l-.435.435a1.32 1.32 0 00-.15.669c.011.233.083.459.209.654l.494.494.454.138.099-.099-.02.336c.248.051.503.051.751 0l.869-.869-.118-.118-.079-.356.079-.632zM461.6 67.607l.455.455.296.06a1.424 1.424 0 001.007-1.64 1.31 1.31 0 00-1.422-1.067l-.217.217a1.366 1.366 0 00-.138 1.798l.019.177zM486.213 97.436a1.376 1.376 0 00-.928.928l.019.257.099.454.968.968.375.02a1.312 1.312 0 001.028-1.304l-.119-.118-.118-.317-.494-.493a1.398 1.398 0 00-.83-.395zM507.369 92.398l-.395-.118h-.237l-.118.118a1.676 1.676 0 00-1.087.85l-.217.217a1.76 1.76 0 00-.395 1.067 1.984 1.984 0 001.086 1.718l.119.119.375.02.217-.02.889-.89c.309-.236.514-.583.573-.967l.138-.138.04-.277a1.372 1.372 0 00-.593-.988l-.276-.276-.119-.435zM545.039 127.816l-.612-.612-.316-.04-.257.257a1.174 1.174 0 00-.751.751 1.152 1.152 0 00.06 1.047l.019.02a1.261 1.261 0 001.225.632l.83-.83.079-.316a1.563 1.563 0 00-.119-1.027l-.158.118zM526.451 118.987l.237-.237.059-.297-.099-.099a1.458 1.458 0 00-.632-1.066 1.356 1.356 0 00-1.106-.158l-.849.849a1.31 1.31 0 00.02 1.086l.79.791c.244.074.506.074.75 0l.297-.099.454-.454.079-.316zM505.354 113.219l-.691.691-.099.296-.019.218.039.276.158.474.573.573c.139.121.3.215.474.277h.316l.514-.158.671-.672a1.401 1.401 0 00.079-1.067l-.889-.889a1.536 1.536 0 00-1.126-.019zM508.041 123.135h.197a1.318 1.318 0 00.79-.79 1.446 1.446 0 00.02-1.245l-1.343-1.343-.415-.099-.257.02-.237.039-.908.909-.02.415.099.099c.07.184.178.352.318.491.139.14.307.249.492.319.131.289.362.52.651.651l.119.119.494.415zM483.033 99.035l.099-.098v-.316l-.099-.1a1.479 1.479 0 00-1.778-1.026l-.889.889-.039.474v.237l.889.889c.28.122.594.143.889.059.191-.053.372-.14.533-.257l.454-.454-.059-.297zM504.129 99.786a.978.978 0 00-.256.06 1.398 1.398 0 00-1.028 1.303l.02.296.85.85c.334.127.7.148 1.047.059l.889-.889.059-.296a1.764 1.764 0 00-.632-1.146l-.237-.237h-.712zM508.594 109.505l-.316.079-.534.533a1.71 1.71 0 00-.375.375 3.508 3.508 0 00-.395.395l-.533.534-.06.336v.197c.043.247.136.482.271.693.136.21.313.391.519.532l.119.118.316.04a1.369 1.369 0 001.067.316l.731-.731c.219-.251.363-.559.414-.889l-.039-.276.039-.277-.019-.336a1.507 1.507 0 00-1.205-1.639zM506.184 104.566l-.612.613-.06.296-.098.099a1.434 1.434 0 00.513.987c.148.121.318.212.501.267.184.054.376.071.566.05l.652-.652a1.43 1.43 0 00-.079-1.581 1.371 1.371 0 00-1.442-.493s.138.375.059.414zM499.231 98.799l.435.434.335.06h.198l.138-.139a1.37 1.37 0 001.047-1.521l-1.007-1.007a1.705 1.705 0 00-.771.02 1.036 1.036 0 00-.178.177l-.75.75-.02.337.099.098.039.237.139.139.296.415zM530.303 142.118a1.306 1.306 0 00-.79.79 1.514 1.514 0 00.02 1.363l.494.494a1.449 1.449 0 001.106.118 1.51 1.51 0 00.928-.73l.178-.178.02-.217v-.238a1.564 1.564 0 00-.672-1.106 1.415 1.415 0 00-.606-.305 1.414 1.414 0 00-.678.009zM502.47 75.134l-.118-.119-.356-.079-.217.217a1.54 1.54 0 00-1.008.77 1.457 1.457 0 00.001 1.186l.316.316a1.227 1.227 0 001.402.533c.164-.043.318-.117.454-.217l.494-.494.06-.257.098-.098a1.67 1.67 0 00-.059-.652l-.514-.514-.553-.592zM497.078 71.44l.434.434.356.08.098.098.119-.119a1.735 1.735 0 001.323-1.323l.099-.099-.039-.316-.692-.691a1.392 1.392 0 00-1.086-.139 1.481 1.481 0 00-.751.475l-.553.553-.039.395v.197l.059.257.118.119.554.079zM510.174 100.853a.984.984 0 00-.158.158 1.392 1.392 0 00-.948.75 1.464 1.464 0 00.02 1.245l.711.711.296.059.277-.039a1.404 1.404 0 00.849-.85 1.47 1.47 0 000-1.224l-.731-.731-.296-.06-.02-.019zM510.431 69.227l-.119.119a1.172 1.172 0 00-.336 1.323l.909.909.375.06a1.42 1.42 0 00.889-.89c.085-.276.092-.57.02-.849a1.316 1.316 0 00-1.112-1.008 1.313 1.313 0 00-.527.04l-.099.296zM520.841 44.516l.455.454c.152-.009.304-.028.454-.06a1.451 1.451 0 001.106-1.105 1.646 1.646 0 00-.79-1.58l-.276-.277h-.237l-.317.04a1.463 1.463 0 00-.928.928 1.59 1.59 0 00.336 1.52l.197.08zM511.182 114.621l.671.672.297.059.098.099.317-.04a1.398 1.398 0 00.908-.909 1.616 1.616 0 00-.257-1.481l-.177-.178-.455-.217-.158-.158h-.316l-.711.711a1.539 1.539 0 00-.217 1.442zM554.56 79.973l-.474.474a.982.982 0 00-.256.494l-.04.277.217.414.83.83.395.04a1.34 1.34 0 00.79-.593 1.215 1.215 0 00.119-.988 1.311 1.311 0 00-1.581-.948zM526.431 69.385a1.566 1.566 0 00-1.145 1.58l.039.238.198.395.553.553.415.099.257-.02.316-.08.77-.77.039-.236.02-.336-.276-.04.256-.06-.079-.473-.533-.534-.474-.158-.277-.04s-.039-.118-.079-.118zM502.984 88.803h.197l1.047-1.047.04-.355-.02-.218-.968-.968-.375-.059-.218.02a1.393 1.393 0 00-.81.81 1.322 1.322 0 00-.059 1.126l.81.81.356-.119zM507.31 58.916a1.282 1.282 0 00-.79 1.304l.533.533a1.31 1.31 0 001.343.435 1.37 1.37 0 00.455-.218l.612-.612v-.316a1.37 1.37 0 00-1.343-1.106l-.158.158-.652-.178zM464.485 125.525l-.415-.415-.356-.039h-.197l-1.107 1.106-.059.336v.197c.059.282.199.54.402.744.204.203.462.343.744.402h.276l.514-.158.77-.77a1.204 1.204 0 00-.039-.83l-.138-.138-.395-.435zM533.78 130.108l-.297.02-.513.197-.613.612-.059.415-.118.119a1.308 1.308 0 00.592 1.027 1.12 1.12 0 001.027.158l.554-.553a1.462 1.462 0 00.118-1.304 1.475 1.475 0 00-.691-.691zM516.891 130.207l-.672-.672a1.479 1.479 0 00-1.6-.731l-.908.909-.119.316-.099.099c.002.493.193.966.534 1.323.198.176.447.287.711.316l.237.237a1.253 1.253 0 001.343.435c.11-.028.215-.068.316-.119l.731-.731.02-.296-.395-.395-.099-.691zM465.038 105.416l.296-.059.81-.81.079-.356v-.197a1.366 1.366 0 00-1.188-1.175 1.447 1.447 0 00-.61.049c-.05.042-.096.088-.138.138l-.79.79-.04.356 1.008 1.007.355.039.218.218zM477.324 51.093l.929-.928c.101-.21.155-.439.158-.671l-.119-.119-.138-.336-.83-.83a1.926 1.926 0 00-.474.04 1.396 1.396 0 00-1.027 1.66 1.37 1.37 0 001.501 1.184zM497.235 64.25a1.258 1.258 0 001.067.513l.138-.138.376-.178.572-.573a1.25 1.25 0 00-.019-1.047l-.83-.83-.375.02a1.312 1.312 0 00-1.008 1.442l.138.139-.059.651zM520.92 57.118l.316.316c.201.104.426.152.652.139l1.007-1.008.04-.395-.02-.217a1.312 1.312 0 00-1.62-.988l-.928.929-.02.336.237.237.336.651zM489.117 106.463l.652-.652a1.423 1.423 0 00-.376-1.324 1.311 1.311 0 00-1.264-.355 1.168 1.168 0 00-.77.77l-.138.139-.04.316.099.098a1.262 1.262 0 00.592.87 1.383 1.383 0 001.087.138c.053.002.105.002.158 0zM497.117 138.799l.297-.296.079-.435a2.07 2.07 0 000-.711l-.632-.632-.316-.118-.336-.06-.316.079-.514.317a1.51 1.51 0 00-.434 1.659l.829.829.356.04.217-.02.296-.099.376-.375.098-.178zM476.198 73.435l.415.415c.195.101.412.156.632.158l.138-.139.376-.177.474-.474.316.039 1.225-1.225a1.289 1.289 0 00-.455-1.165 1.224 1.224 0 00-1.165-.257 1.45 1.45 0 00-.869.632l-.336-.02a1.392 1.392 0 00-1.047 1.245l.02.257.158.395.118.316zM480.841 75.549a1.539 1.539 0 001.213 1.285c.206.042.418.042.624-.002l.652-.651a1.602 1.602 0 00-.04-1.304l-.79-.79-.336-.02-.099.099-.375.099-.83.83-.039.355.02.099zM516.16 86.57a1.45 1.45 0 00-.968 1.799 1.167 1.167 0 00.711.71 1.286 1.286 0 001.007.139l.277-.277.336-.612.079-.395-.514-.514-.612-.335-.316-.514zM527.4 123.806l-.474-.474-.534-.217-.336-.02-1.066 1.067-.02.415.138.138a1.287 1.287 0 001.561.889c.226-.051.434-.165.598-.33.165-.164.279-.372.33-.598l.02-.534-.217-.336zM498.322 82.996l.573-.573a1.43 1.43 0 00-.119-1.62 1.393 1.393 0 00-1.481-.494l-.415.178-.415.415a1.51 1.51 0 00.395 1.264 1.483 1.483 0 001.462.83zM489.729 69.939l.435.434.336.099.217.02.138-.139a1.565 1.565 0 001.067-1.896l-.948-.948-.376-.06-.316.04-.948.948c-.037.343-.023.689.04 1.027l.118.119.237.356zM507.488 123.53l-.652-.652-.355.039a1.452 1.452 0 00-1.067 1.304l.099.455.533.533.415.178c.246.064.504.064.751 0a2.83 2.83 0 00.197-.198l.612-.612a1.978 1.978 0 00-.039-1.304l-.494.257zM496.663 85.761l-.83-.83a1.59 1.59 0 00-.73-.217l-.968.968-.079.435v.237l.889.889h.395l.098-.1a1.313 1.313 0 001.225-1.382zM512.17 120.745a1.315 1.315 0 00-1.008-.139l-.711.712a1.096 1.096 0 00-.079.711l-.652.652-.019.296a1.527 1.527 0 00.651 1.007l.119.119.296.099h.198l.217-.02.198-.198a1.362 1.362 0 00.987-1.185l-.335-.336a1.289 1.289 0 00.197-1.106 1.395 1.395 0 00-.059-.612zM469.245 82.482l-.198-.198a.859.859 0 00-.257-.256 1.199 1.199 0 00-1.185-.435l-1.027 1.027-.059.376v.237c.078.313.25.596.493.81l.099.098a1.34 1.34 0 001.502.711l.612-.612a1.56 1.56 0 00.02-1.758zM484.534 60.397l.415.415c.188.077.389.117.593.119l.138-.139.395-.158.573-.572a1.43 1.43 0 00-.04-1.106l-.77-.771-.336-.02a1.421 1.421 0 00-1.066 1.304l.039.237.158.395-.099.296zM446.232 130.108l-.277.079-.711.711-.079.277a1.48 1.48 0 00.632 1.106 1.31 1.31 0 001.185.197l.889-.889.04-.316a1.517 1.517 0 00-.573-1.007 1.414 1.414 0 00-1.106-.158zM530.264 129.12l-.159-.158a1.277 1.277 0 00-1.165-.494l-.948.948c-.133.338-.126.715.02 1.047l.829.83.376.02.098-.099a1.28 1.28 0 00.968-1.403l-.118-.118.099-.573zM488.07 127.856l-.276.04-.455.256-.573.573-.039.395.02.218c.042.169.118.329.223.468a1.305 1.305 0 001.396.48c.101-.027.2-.06.297-.099l.612-.612.02-.336a1.428 1.428 0 00-1.225-1.383zM523.587 79.657a1.34 1.34 0 00-.336.336 1.256 1.256 0 00-.079 1.462 1.283 1.283 0 002.311-.376l.118-.118v-.237l-1.007-1.008h-.198l-.355.04-.316.316-.138-.415zM525.404 98.225c-.1.084-.193.177-.276.277l-.593.593a1.448 1.448 0 00.395 1.303 1.532 1.532 0 001.284.376l.652-.652a1.503 1.503 0 00-.099-1.64 1.42 1.42 0 00-1.363-.257zM496.425 130.444l-.138.138-.079.277c-.09.353-.04.728.138 1.047l.474.474.277.079.316-.04a1.368 1.368 0 00.968-1.64 1.341 1.341 0 00-1.6-1.007l-.632.632.276.04zM549.978 119.125l-.435-.435-.296-.099-.099-.098-.296.059a1.365 1.365 0 00-.87 1.607 1.367 1.367 0 001.502 1.04l.79-.79.04-.237c.073-.28.066-.574-.02-.85l-.119-.118-.197-.079zM479.536 113.396l.415.02h.237a1.48 1.48 0 001.087-1.086v-.238l-.119-.434-.553-.553-.434-.158-.257-.02a.457.457 0 00-.139.138 1.343 1.343 0 00-1.066 1.502l.829.829zM459.329 115.589l-.336.573-.079.395-.119.118.968.968.395.04c.346-.101.641-.326.83-.632a1.65 1.65 0 00.119-1.027 1.398 1.398 0 00-1.719-.968l-.059.533zM479.043 132.696l-.87.869-.019.375v.197l.731.731c.202.125.434.193.671.198l.119.118h.276l.83-.829.099-.297.118-.118-.02-.375-.138-.139a1.452 1.452 0 00-1.797-.73zM488.959 88.922l-.02.335.099.1.079.315.79.79.336.1.099.098a.46.46 0 00.138-.138 1.28 1.28 0 00.909-.711 1.352 1.352 0 00.059-1.166l-.692-.691-.237-.04a1.785 1.785 0 00-.75 0 2.93 2.93 0 00-.198.198l-.632.632.02.178zM486.786 139.254l-.276-.277h-.237l-.454.099-.909.909v.474l.039.237c.167.263.389.485.652.651l.237-.237-.079.317.514.158h.276l.336-.099.771-.771a2.195 2.195 0 00-.474-1.619l-.396.158zM447.358 100.576l-.415-.415a1.286 1.286 0 00-.79-.039 1.257 1.257 0 00-.87 1.462l.297.296a1.281 1.281 0 001.303.79l.396-.395-.139.138.198-.197c.142-.232.217-.499.217-.771l-.079-.276-.118-.593zM476.514 116.537l-.672-.671-.375-.02-.099.099a1.504 1.504 0 00-.862.687c-.191.329-.25.719-.165 1.09l.909.909.355.04.218-.02.81-.81a1.689 1.689 0 00-.04-1.304h-.079zM468.751 99.845l.454.455.375.059.218-.02.987-.988.06-.375.197-.316-.908-.909a1.286 1.286 0 00-.909-.04l-.316.673-.375.651.276.277-.059.533zM535.715 83.252l.87-.869.098-.375v-.198a1.396 1.396 0 00-1.56-1.284l-.85.85-.098.375-.02.217a1.505 1.505 0 00.573 1.007c.282.215.635.314.987.277zM569.731 77.721a1.452 1.452 0 00-1.185-.157.562.562 0 00-.158.158l-.75.75a1.646 1.646 0 00.493 1.442 1.37 1.37 0 001.205.257c.285-.122.511-.348.632-.632l.297-.296.118-.356v-.198a1.535 1.535 0 00-.652-.968zM543.558 99.035l-.692.692-.177.494-.04.276a1.425 1.425 0 001.185 1.185l.099-.098.375-.02.257-.059.455-.455.217-.493.059-.297a1.48 1.48 0 00-1.303-1.303l-.119.118-.099-.099-.217.06zM526.155 88.783l.375.06a1.31 1.31 0 00.909-.91 1.935 1.935 0 000-.75 1.343 1.343 0 00-1.639-1.007l-.475.474a.982.982 0 00-.256.494l-.02.217c-.013.188.014.376.079.553l.553.553.474.316zM532.318 109.209l-.178-.178a1.03 1.03 0 00-.513-.277l-.257-.02-.514.119-.592.593-.139.493v.277l1.028 1.027.395.04h.197a1.395 1.395 0 001.047-1.245l-.02-.257-.197-.434-.257-.138zM528.09 91.865l-.118-.119a1.453 1.453 0 00-.435.198l-.375-.06-.119.12a1.433 1.433 0 00-.935 1.43c.013.198.07.39.165.564l.731.731.336.02.415-.415c.034.068.09.123.158.158l.197.198.138-.139a1.258 1.258 0 00.929-1.52 1.205 1.205 0 00-.415-.653l-.672-.513zM507.409 75.252a1.285 1.285 0 001.303.435 1.34 1.34 0 00.475-.237l.533-.534a1.458 1.458 0 00-.494-1.126 1.543 1.543 0 00-1.304-.276 1.344 1.344 0 00-.948 1.422l.178.178.257.138zM529.414 91.41l.336-.335a1.373 1.373 0 00-.119-1.304 1.339 1.339 0 00-1.145-.553l-.238.04-.395.197-.533.533-.059.376v.197l.099.415.592.593.494.138h.277l.335-.336.356.04zM558.156 95.816a2.928 2.928 0 00-.198.197l-.336.336a1.504 1.504 0 00-.651 1.205 1.34 1.34 0 001.007 1.284l.257-.02.454-.099 1.126-1.125.04-.396-.02-.217a1.313 1.313 0 00-1.679-1.165zM545.138 106.581l-.474-.474a1.256 1.256 0 00-1.166-.257 2.085 2.085 0 00-.375.139l-.731.731.197.197.159.514.632.632.414.059.119.119.158-.159a1.219 1.219 0 00.731-.73c.193-.213.311-.484.336-.771zM533.029 68.635l.494.493.375.1.217.019.119-.118a1.314 1.314 0 00.889-.692 1.446 1.446 0 00.019-1.245l-.75-.75-.336-.06h-.198a1.371 1.371 0 00-.987 1.62l.118.12.04.513zM539.291 94.492l.059-.296a1.4 1.4 0 00-.672-1.067 1.369 1.369 0 00-1.126-.178l-.118.119-.632.632c-.132.352-.125.74.02 1.087l.711.71c.242.043.491.036.73-.02.073-.058.139-.124.198-.197l.612-.612.218-.178zM550.057 90.364a1.342 1.342 0 00-.138-1.245l-.731-.73h-.316a1.422 1.422 0 00-1.067 1.263l.039.277.652.652c.328.174.709.216 1.067.118.187-.074.356-.189.494-.335zM513.137 82.68l.375.059a1.276 1.276 0 00.81-.81c.158-.276.201-.602.119-.909.032-.115.046-.236.039-.355l-.138-.139-.158-.355-.553-.553a1.343 1.343 0 00-.593-.119l-.948.948c-.09.198-.131.415-.119.632l.119.119-.099.099a1.422 1.422 0 00.553 1.027l.139.138.454.217zM516.041 95.065l.553.553a1.369 1.369 0 001.857-.909 1.396 1.396 0 00-.494-1.442l-.237-.237a.617.617 0 00-.257.06 1.342 1.342 0 00-.928.73 1.531 1.531 0 000 1.265l-.494-.02zM520.248 105.12a1.525 1.525 0 00-.079-1.304l-.513-.514a1.454 1.454 0 00-.909-.079c-.217.06-.414.176-.573.336l-.414.415-.119.316a1.393 1.393 0 00.593 1.067 1.318 1.318 0 001.145.197 1.28 1.28 0 00.869-.434zM534.096 119.164l-.573-.572-.336-.02h-.197l-.296.099-.85.849-.059.375-.099.099.968.968.217.02.415.02.178-.178a1.362 1.362 0 00.784-.74 1.367 1.367 0 00.006-1.078l-.158.158zM536.822 75.39a1.399 1.399 0 001.659 1.028l.791-.79a1.37 1.37 0 00-.652-1.838l-.257-.256a1.702 1.702 0 00-.336.059 1.364 1.364 0 00-1.043.603 1.372 1.372 0 00-.162 1.195zM515.488 104.389a1.511 1.511 0 00-.04-1.62 1.365 1.365 0 00-1.422-.553l-.573.573a1.42 1.42 0 00-.158 1.106c.173.41.499.736.909.909l.118.118.336.02.099.098.533-.533.198-.118zM517.207 75.647l.415.415.316.04.099.098a.67.67 0 01.138-.138 1.451 1.451 0 00.968-.77 1.39 1.39 0 00.02-1.166l-.83-.83a1.482 1.482 0 00-.751 0l-.968.968c-.076.355-.027.725.139 1.047l.454.336zM517.246 101.05l.376-.177.513-.514a1.26 1.26 0 00.139-.652l-.909-.908a1.396 1.396 0 00-.632-.159l-.909.91a1.458 1.458 0 00-.138.69l.711.712c.195.101.412.156.632.158l.217-.06zM520.407 99.312l.138.138.376.099.098-.099a.615.615 0 00.257-.06 1.343 1.343 0 001.126-.73 1.486 1.486 0 000-1.264l-.77-.77a1.702 1.702 0 00-.632.039 1.373 1.373 0 00-1.087 1.086v.237l.099.415.355.356.04.553zM515.745 98.482l.355-.158.731-.73a1.48 1.48 0 00-.079-1.62 1.285 1.285 0 00-1.402-.534l-1.205 1.205a2.164 2.164 0 00-.119.632l.138.139.178.375.514.514c.2.082.415.123.632.118l.257.06zM512.822 89.79l.79-.79a1.22 1.22 0 00.118-.631l-.118-.119-.158-.356-.534-.533c-.19-.093-.4-.14-.612-.138l-.909.909c-.102.2-.15.426-.138.651l.928.929c.191.093.401.14.613.138l.02-.06zM524.93 107.233l-.652-.652-.355-.039-.099.098-.395.119-.356.356a1.543 1.543 0 00-.99.731 1.529 1.529 0 00-.136 1.224l.731.731.296.099.099.099c.154-.001.306-.02.455-.06.391-.091.732-.332.948-.671a1.356 1.356 0 00.808-.729 1.375 1.375 0 00.021-1.089l-.375-.217zM528.525 104.073l.237.237.217-.02a1.314 1.314 0 00.83-.83 1.511 1.511 0 00-.257-1.521 1.366 1.366 0 00-1.343-.395 1.266 1.266 0 00-.711.711 1.386 1.386 0 00-.059 1.126l.118.119a1.293 1.293 0 00.968.573zM529.671 79.46l.316-.08.178-.098.316-.237.119-.158.138-.218.059-.177.04-.395v-.198l-.04-.158-.099-.178-.237-.316-.158-.118-.217-.139-.178-.059-.375-.06-.217.02-.316.08-.178.098-.277.237-.158.159-.138.217-.04.197-.059.376v.197l.04.158.098.178.238.316.158.119.217.138.217.06.356.039h.197zM549.76 110.492l-.335-.335a.98.98 0 00-.593-.119l-.988.988a1.2 1.2 0 00.04 1.067l.573.572.335.02 1.087-1.086.059-.376-.02-.217-.118-.118-.04-.396zM538.086 119.717l-.849.85-.04.355v.198a1.413 1.413 0 00.56 1 1.407 1.407 0 001.12.244l.75-.75.119-.316.019-.218a1.454 1.454 0 00-.513-1.224 1.419 1.419 0 00-1.166-.139zM560.467 129.812a1.334 1.334 0 00-.829.829 1.373 1.373 0 00.059 1.205l.711.711c.22.06.452.06.672 0a1.22 1.22 0 00.691-.691l.277-.277a1.48 1.48 0 00-.573-1.046 1.345 1.345 0 00-1.008-.731zM540.694 81.83l-1.008 1.008c.009.14.029.278.059.414l1.166 1.166.415.06h.237l.81-.81a1.335 1.335 0 00-.435-1.304 1.477 1.477 0 00-1.244-.534zM555.824 109.208a2.83 2.83 0 00-.197.198l-.731.731-.02.336.771.77c.201.104.426.152.651.138l1.028-1.027.039-.395a1.345 1.345 0 00-1.541-.751zM522.718 95.263l.237-.238.098-.335.494-.494.059-.376-.019-.217-1.028-1.027-.355-.04a1.303 1.303 0 00-.79.79l-.613.613-.138.454v.277l-.079.276v.277l.099.454.593.593.493.138h.277a1.396 1.396 0 00.909-.908l-.237-.237zM535.419 105.12l-.415-.415-.336-.059-.217.019a1.352 1.352 0 00-.77.771 1.285 1.285 0 00-.079 1.145l.612.613.336.059.217-.02a1.366 1.366 0 00.81-.81 1.386 1.386 0 00.059-1.126l-.118-.118-.099-.059zM538.244 53.306l.139-.139.019-.256a1.328 1.328 0 00-.622-.89 1.339 1.339 0 00-1.076-.137c-.096.013-.189.04-.277.079l-.77.77v.277l.098.414.613.613.474.118.257.02.098-.099a1.282 1.282 0 001.047-.77zM474.361 131.728l.375-.376a1.226 1.226 0 00-.276-1.264 1.229 1.229 0 00-.613-.375l-.138-.139-.395-.079-.217.02-.158.158a1.536 1.536 0 00-1.087.81l-.059.257v.237c.052.308.189.595.395.83l.632.632.257.019.198-.197.138-.138.948-.395zM482.835 154.899l-.513-.514h-.316a1.423 1.423 0 00-1.087 1.284l.06.296.553.553c.317.182.693.231 1.047.139a1.378 1.378 0 00.809-.81 1.443 1.443 0 00-.098-1.245l-.455.297zM497.255 106.344l.257.02.494-.138.593-.593.118-.513v-.277a3.649 3.649 0 01.099-1.126l-.04-.237-.118-.435-.375-.375-.455-.138h-.276a1.316 1.316 0 00-.85.849l-.158.158-.158.435-.118.118-.257.494-.059.297.059.335-.118.119a1.39 1.39 0 001.362 1.007zM495.399 101.485l.138-.138.118-.316.099-.099a1.59 1.59 0 00-.83-1.54l-.315-.04-.573.059-1.146 1.146-.099.414.02.257.079.316.81.81.217.02.296.059.573-.059.672-.672-.059-.217zM497.986 112.764l-.612-.612-.435-.04h-.237a1.49 1.49 0 00-.81.81 1.396 1.396 0 00.059 1.087l.692.691.217.02h.356a1.286 1.286 0 00.967-1.403l-.138-.138-.059-.415zM504.624 132.241l-.692.692a1.093 1.093 0 00.02 1.047 1.194 1.194 0 00.85.612l.138.138.375.059a1.34 1.34 0 00.691-.691c.159-.297.194-.645.099-.968a1.368 1.368 0 00-1.481-.889zM534.036 133.525l-.276.277a1.313 1.313 0 00-.237.948c.08.365.291.688.593.909a1.342 1.342 0 001.876-.257l.079-.395-.118-.672-.909-.909-.395-.118-.237.039-.376.178zM574.571 167.146l-.079.395.099.691.355.356c.121.102.262.176.415.217l.415.099.671-.119.277-.276a1.21 1.21 0 00.237-.988 1.309 1.309 0 00-.533-.849 1.397 1.397 0 00-1.857.474zM522.441 135.915l.336.139.118.118.217-.02a1.224 1.224 0 001.146-.553c.109-.073.203-.167.277-.276l.098-.099v-.316l-1.046-1.047-.317-.079-.118-.119-.237.04-.494.494a.478.478 0 00-.138.138 1.396 1.396 0 00.118 1.857l.04-.277zM548.655 141.506l.296.098.573-.059.197-.197c.116-.095.222-.201.316-.316a1.268 1.268 0 00.218-1.047 1.587 1.587 0 00-.296-.592 1.583 1.583 0 00-.514-.416l-.257-.059-.099-.099-.395.395a.923.923 0 00-.276.277 1.31 1.31 0 00.316 1.817l-.079.198z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M526.906 125.762l.256-.02.455-.138.217-.217a1.77 1.77 0 00.277-.277 1.256 1.256 0 00.177-1.165 1.365 1.365 0 00-.948-.948l-.099-.099-.335.019-.652.652a1.563 1.563 0 00-.079 1.581l.652.652.079-.04zM559.38 148.202l-.197.198a1.27 1.27 0 00-.948.513 1.897 1.897 0 00-.237 1.343l.79.791.316.079.118.118a1.302 1.302 0 00.948-.553 1.204 1.204 0 00.909-.356l.158-.158.158-.355.138-.138v-.277l-.118-.119-.178-.375-.138-.138a1.711 1.711 0 00-1.462-.395l-.257-.178zM565.168 162.247l-.395.395a.842.842 0 00-.197.198 1.311 1.311 0 00-.04 1.54 1.453 1.453 0 001.58.632l.396-.395c.082-.061.156-.134.217-.217a1.342 1.342 0 00.039-1.541 1.504 1.504 0 00-1.6-.612zM522.323 103.361l.138-.138.119-.355a1.334 1.334 0 00-.198-1.146l-.533-.533a1.8 1.8 0 00-.869 0l-.791.79-.079.355v.198a1.658 1.658 0 001.146 1.146 1.264 1.264 0 001.067-.317zM540.891 159.541l-.356-.04-.099.099-.256.059-.455.455a1.482 1.482 0 000 1.698l.593.593.375.02.237-.04c.2-.057.387-.151.553-.276l.158-.158.178-.376.138-.138-.019-.375-.139-.139a1.395 1.395 0 00-.75-.75l-.158-.632zM528.96 150.849a1.202 1.202 0 00-.217 1.087 1.46 1.46 0 00.869.869l.098.098.336.06.968-.968.099-.375v-.198l-.02-.257-.849-.849-.395-.079-.119-.119-.257.06-.375.375-.138.296zM554.383 142.493l-.138.139-.139.335-.118.119.02.296c.011.271.116.529.296.731l.533.533.257.02.217-.02.316-.118c.217-.134.4-.317.534-.533a.535.535 0 00.197-.198l.099-.099.059-.296-.533-.533a1.474 1.474 0 00-1.086-.455 1.322 1.322 0 00-.514.079zM499.882 100.912l.435-.039.118-.119c.25-.137.455-.343.593-.592.11-.141.19-.302.237-.475l.099-.098-.099-.1a1.34 1.34 0 00-1.126-1.125l-.118-.118-.257-.02-.119.118-.355.119-.139.138a1.37 1.37 0 00-.237.237 1.512 1.512 0 000 1.699l.158.158.494.296.316-.079zM534.313 157.032l-.652-.652-.316-.039-.237.039-.415.138-.178.178-.158.158a1.365 1.365 0 00-.285.96c.029.344.187.665.443.897l.158.158.218.02a1.403 1.403 0 00.928-.534 1.46 1.46 0 00.419-.597 1.46 1.46 0 00.075-.726zM508.969 138.839a1.232 1.232 0 00-.592-.593l-.139-.138-.375-.059-.217.019-.692.692a1.485 1.485 0 00-.079 1.541l.652.652.395.039h.198c.264-.037.515-.139.731-.296l.177-.178.198-.435.02-.256-.02-.297-.257-.257v-.434zM536.822 157.605l-.395.395v.197a1.444 1.444 0 00.81 1.087 1.57 1.57 0 001.501-.158l.454-.455.04-.276a1.476 1.476 0 00-1.166-1.166 1.399 1.399 0 00-1.244.376zM501.364 145.061l.336.099h.316l.534-.178.217-.217a1.46 1.46 0 00.257-1.126 1.59 1.59 0 00-.692-.928l-.276-.277-.217.217-.297.099-.355.356a1.366 1.366 0 00.158 1.738l.178.178-.159.039zM520.604 142.098l.04.237.909.909.296.059a1.393 1.393 0 001.305-.98 1.394 1.394 0 00-.515-1.548l-.158-.158-.435-.119h-.237l-.276.079-.119.119a1.28 1.28 0 00-.533.533l-.158.158-.079.395-.04.316zM537.77 148.439l-.355-.355a2.817 2.817 0 00-.277-.672 1.22 1.22 0 00.059-.731l.277.079.099-.099.257-.019.395-.395c.122-.102.234-.214.336-.336a1.321 1.321 0 00-.079-1.581l-.534-.533-.237-.039-.099-.099-.276.039-.889.889-.079.356.099.099-.158-.158h-.198a1.37 1.37 0 00-1.264.553.309.309 0 01-.099.098c-.088.07-.168.149-.237.237l-.474.475a.789.789 0 00.059.375l.099.375.138.138c.058.225.151.438.277.633l.454.454c-.001.316.119.62.336.849l.513.514.415.099.257-.02.277-.079.632-.632c.064-.186.111-.378.138-.573l.099-.099c.01.054.023.106.039.158zM525.484 192.095l-.079.395.059.375v.237a1.057 1.057 0 00.711.711 1.221 1.221 0 001.047-.02l.395-.395a1.258 1.258 0 00.064-1.414 1.243 1.243 0 00-.341-.363 1.422 1.422 0 00-.849-.257 1.277 1.277 0 00-1.007.731zM533.523 162.365l.098-.375a1.34 1.34 0 00-.651-1.56l-.119-.119h-.435c-.376.031-.727.2-.987.474l-.376.375a1.566 1.566 0 001.205 1.205 1.313 1.313 0 001.442-.533l-.177.533zM526.017 155.886l-.553.553a1.389 1.389 0 00-.035 1.58 1.4 1.4 0 00.39.396 1.49 1.49 0 001.107.276 1.418 1.418 0 00.928-.573l.079-.395-.039-.237a1.364 1.364 0 00-.376-1.047 1.416 1.416 0 00-1.185-.395l-.316-.158zM528.722 161.121l-.296-.02-.751.751a1.57 1.57 0 00-.256.652l-.099.098.039.277.099.099a1.37 1.37 0 00.869.869 1.341 1.341 0 001.186-.514 1.29 1.29 0 00.217-.968l-.059-.375a1.404 1.404 0 00-.85-.849l-.099-.02zM531.133 167.245l-.593-.593-.336-.099-.296.02-.514.158-.355.356a1.392 1.392 0 00.197 1.935c.172.272.44.47.751.553a1.253 1.253 0 001.086-.217l.593-.592v-.316l-.099-.099a1.644 1.644 0 00-.336-.652l-.098-.454zM557.622 174.06a1.327 1.327 0 00-.249.631c-.028.228.003.46.091.672a1.342 1.342 0 001.027.79l.257-.019 1.205-1.205-.158-.158a1.277 1.277 0 00-1.067-.711 1.333 1.333 0 00-1.106 0zM491.507 158.316l.277-.277.019-.217.02-.336-.099-.098a1.37 1.37 0 00-.316-.633l-.138-.138-.375-.178-.119-.118-.336.02-.098.098a1.146 1.146 0 00-.593.316l-.138.139-.158.355-.139.138.02.336.099.099c.053.221.162.425.316.593l.138.138.356.158.138.138.336-.019h.197c.207-.066.4-.167.573-.297l.02-.217zM504.781 149.585l.099.099.277-.04c.213-.048.41-.15.572-.296l.178-.178.158-.474.02-.257-.039-.316-.573-.573a1.288 1.288 0 00-1.837.297.638.638 0 00-.218.217l-.098.099-.02.335.948.949.533.138zM510.609 165.526l-.138.138-.079.395.158.672.256.257a1.349 1.349 0 001.007.26 1.334 1.334 0 00.89-.537 1.286 1.286 0 00-.059-1.6 1.262 1.262 0 00-1.673-.07c-.159.13-.283.296-.362.485zM516.713 158.632l.099.099.257-.02.494-.494a1.05 1.05 0 01.197-.197 1.436 1.436 0 00.217-1.245 1.342 1.342 0 00-.889-.889l-.138-.138-.355-.079-.119-.119-.257.06-.829.829-.099.376v.197l.02.257 1.086 1.086.316.277zM530.264 129.081l.098.098a1.425 1.425 0 001.383-.553 1.396 1.396 0 00.02-1.6 1.478 1.478 0 00-1.501-.671l-.889.888a1.425 1.425 0 00.612 1.482l.119.119.158.237zM474.4 154.563l-.177-.178-.435-.119-.257-.019-.118.118a1.515 1.515 0 00-1.126.573.858.858 0 00-.178.178l-.099.099-.039.276.375.375c.149.335.416.602.751.751l.276.04h.198l.316-.316a1.39 1.39 0 00.513-1.778zM550.827 85.406a1.283 1.283 0 00.02-1.443l-.217-.217-.356-.158a1.31 1.31 0 00-.888-.178c-.343.059-.648.25-.85.534l-.355.355a1.252 1.252 0 00.81 1.403 1.452 1.452 0 001.836-.296zM575.795 187.057a1.4 1.4 0 00-1.837.376c-.094.154-.167.32-.217.494l-.119.118v.198l.119.118a1.316 1.316 0 00.928.929l.158.158.297-.02.454-.178-.217-.217.237.237c.213-.123.39-.3.513-.514l.119-.118.02-.218-.079-.395-.633-.632.257-.336zM478.371 140.004l.218.02.592-.592a1.395 1.395 0 00-.276-1.897 1.29 1.29 0 00-1.758.059l-.435.435-.02.257c-.045.328.036.661.227.932.19.272.476.461.8.53l.652.256zM516.318 147.451a1.288 1.288 0 00-.119 1.304l.83.83.395.079.217-.02.257-.059.573-.573a1.288 1.288 0 00-.059-1.679l-.475-.474-.237-.04a1.42 1.42 0 00-1.382.632zM529.513 130.147l-.415.178-.158.158a.373.373 0 01-.118.119 1.562 1.562 0 00-.297 1.363l.04.316c-.007.392.134.773.395 1.066a1.922 1.922 0 001.916.652l.849-.849.06-.375-.02-.218-.119-.118a1.41 1.41 0 00-.177-1.126l-.396-.395a1.387 1.387 0 00-1.086-.415l-.296.02-.178-.376zM581.089 133.644l-.316-.04-.889.889a1.254 1.254 0 00-.118 1.067 1.113 1.113 0 00.731.731l.098.098a1.256 1.256 0 001.344-.276l.138-.138.138-.336.119-.119v-.237l-.02-.217-.217-.217a1.422 1.422 0 00-.751-.751l-.257-.454zM561.099 138.622l-.455-.455a1.45 1.45 0 00-1.204-.415 1.375 1.375 0 00-.968.573l-.119.119-.118.118-.04.316-.099.099.04.237a1.253 1.253 0 00.77.771l.099.098.296.06h.632l.158-.158c.217-.135.4-.317.534-.534a.618.618 0 00.197-.197l.099-.099.04-.316.138-.217zM541.148 147.471l-.178.178-.06.454.04.277 1.047 1.047.257.059.336.02c.182.001.363-.032.533-.099l.612-.612.079-.316v-.316l-.099-.297a1.403 1.403 0 00-.612-.889l-.138-.138-.415-.059-.217.02-.277.276a1.537 1.537 0 00-.908.395zM549.543 153.457l.296-.297a1.369 1.369 0 00.237-1.106 1.425 1.425 0 00-.869-.869l-.138-.138-.336-.06-.099.099-.257.059-.158-.158-.237-.039h-.276l-.929.928-.059.415v.237l.059.257.435.434-.178-.177a1.287 1.287 0 001.146.197c.274.162.592.231.908.198l.218-.218.237.238zM514.856 150.731l-.534-.534a1.536 1.536 0 00-2.054.356l-.138.138-.099.375-.02.218.059.256.948.949.237.039.376.02.138-.138c.295-.094.552-.281.731-.534.115-.152.196-.327.237-.513v-.316l-.099-.099.218-.217zM531.667 137.91a.524.524 0 00-.159.159 1.366 1.366 0 00.02 1.679l.652.651.395.04.119-.119a1.67 1.67 0 00.849-.612c.072-.06.138-.126.198-.197l.098-.099.04-.316-.02-.218-.434-.434a1.82 1.82 0 00-1.225-.514l-.277.277-.256-.297zM541.266 142.612l-.217.217-.178.494v.237a1.236 1.236 0 00-.059.533 1.753 1.753 0 00-.118.553v.198l.651.652a1.852 1.852 0 001.383.434l.316-.316a1.394 1.394 0 001.027-.434l.396-.395.019-.218-.118-.118a1.878 1.878 0 00-.237-.948l-.692-.692a1.508 1.508 0 00-2.173-.197zM536.348 140.261l-.099.099-.059.296.099.099.019.217.435.435a1.392 1.392 0 001.013.449 1.388 1.388 0 001.022-.429l.158-.158v-.198a1.396 1.396 0 00-1.107-1.106 1.341 1.341 0 00-1.461.553.888.888 0 00-.02-.257zM527.261 140.261l.099.099.237-.04.651-.652a1.422 1.422 0 00-.158-1.856l-.138-.139-.336-.138-.118-.118-.257.019-.316.119-.099-.099a1.738 1.738 0 00-.533.533l-.296.297-.04.355 1.007 1.008.297.612zM578.601 154.108a1.375 1.375 0 00-.237 1.107 1.608 1.608 0 00.967.967l.257.257h.277c.377-.075.71-.295.928-.612a1.62 1.62 0 00.257-1.166l-.494-.493a1.556 1.556 0 00-1.205-.415 1.283 1.283 0 00-.75.355zM514.816 119.915l-.513.514a.478.478 0 00-.138.138 1.484 1.484 0 00-.277 1.225 1.376 1.376 0 00.81.809l.158.158h.237a1.223 1.223 0 001.403-.493c.099-.137.173-.291.217-.455v-.197l-.02-.336-.553-.553a1.564 1.564 0 00-.474-.474l-.138-.138-.356-.08-.099-.098-.257-.02zM508.238 120.488h.198l.138-.099.494-.494.138-.138a1.708 1.708 0 00.139-1.877l-.712-.711-.296-.02-.099-.098a1.588 1.588 0 00-.908.592 1.677 1.677 0 00-.297.85l-.098.098.039.277.869.869.395.751zM537 134.908l-.178.178a1.421 1.421 0 00-.237 1.185 1.456 1.456 0 00.869.869l.119.119.316.039.237-.039.514-.514a1.477 1.477 0 00.237-1.185 1.615 1.615 0 00-.889-.889l-.099-.099-.296-.059-.099.099-.217.019-.317.316.04-.039zM517.167 110.295l.099.099a1.203 1.203 0 00.593 1.225l.197.197.217.02.099.099.297-.06.395-.395a1.395 1.395 0 00.177-1.244 1.553 1.553 0 00-.612-.612 1.329 1.329 0 00-1-.242 1.342 1.342 0 00-.876.538l.414.375zM509.542 84.576l.217.02.237-.237c.116-.095.222-.2.317-.316a1.372 1.372 0 00.078-1.541 1.62 1.62 0 00-1.6-.731l-.474.474a1.16 1.16 0 00-.217.217 1.388 1.388 0 00-.197 1.304 1.61 1.61 0 001.224.948l.415-.138zM546.501 144.923l.118.119.277.039.099.099.533-.533a1.307 1.307 0 00.435-1.462 1.62 1.62 0 00-1.067-1.067h-.277l-.494.099-.592.592-.158.475v.276a1.562 1.562 0 001.126 1.363zM558.097 90.62l-.099.376.039.276c-.003.191.038.38.119.553l.198.198.414.178.119.118.395.08.632-.08.277-.276a1.288 1.288 0 00.217-.968 1.343 1.343 0 00-1.773-1.027c-.245.088-.46.246-.617.454l.079.118zM529.632 100.28a1.505 1.505 0 00.118 1.936l.474.474.237.039.296.02.159-.158.395-.197.533-.534.079-.395v-.237l-.059-.257-.415-.414-.514-.514-.296-.06h-.277l-.474.198-.177.178a.2.2 0 00-.079-.079zM523.824 130.147l.435-.434.079-.356.019-.217-.039-.237-.849-.849-.336-.099h-.198l-.237.039-.592.593a1.246 1.246 0 00-.237 1.106 1.308 1.308 0 00.671.909l.119.118.355.079h.198l.276-.276.336-.376zM508.239 104.25a1.254 1.254 0 00.217 1.521l.197.198.514.119a1.233 1.233 0 001.304-.514c.102-.145.182-.305.237-.474l-.02-.336-.672-.672a1.37 1.37 0 00-1.738 0l-.158.158h.119zM517.344 182.968h-.197l-.257.06-.849.849-.079.356-.099.098.652.652a1.477 1.477 0 001.62.158l-.06.059.534-.533.019-.336-.098-.296-.158-.158a1.406 1.406 0 00-.573-.573l-.119-.118-.336-.218zM573.82 142.572l-.217.217-.257.494-.059.297.02.335.553.554a1.341 1.341 0 001.633.252c.157-.089.293-.208.401-.351l.178-.178.02-.217a1.427 1.427 0 00-2.094-1.067l-.178-.336zM560.803 153.338l-.218.02-.256.059-.119.119-.237-.237a1.482 1.482 0 00-1.699.474l-.118.118-.099.296-.099.099v.237l.099.099.118.316.119.119a1.676 1.676 0 001.244.691 1.37 1.37 0 00.751-.198l.119.119a1.29 1.29 0 001.343-.514l.336-.335v-.317l-1.008-1.007-.276-.158zM505.019 167.086l.355-.355.099-.336v-.197l-.02-.218-.494-.494a1.375 1.375 0 00-1.086-.375 1.455 1.455 0 00-1.027.632l-.139.139-.118.118-.04.356-.098.098.039.237.81.81.375.099h.198l.257-.059.395-.395.494-.06zM480.069 117.288l.178-.178.099-.454.02-.257-.079-.277-.099-.098a1.718 1.718 0 00-.316-.632l-.494-.494-.257-.02-.098.099-.356.079-.119.118a1.519 1.519 0 00-.355.356 1.393 1.393 0 00.257 1.916 1.372 1.372 0 001.619-.158zM503.814 114.799a1.205 1.205 0 001.145-.277l.158-.158.158-.355.139-.139-.02-.375-.04-.237a1.306 1.306 0 00-.75-.751l-.119-.118-.355-.04-.218.02-.671.672a1.282 1.282 0 00.158 1.738l.138.138.277-.118zM517.602 94.294l.099.1.276-.04c.225-.044.431-.154.593-.316l.177-.178.158-.435.02-.256-.059-.297-.553-.553a1.28 1.28 0 00-.995-.284 1.288 1.288 0 00-.882.541l-.316.316-.02.336.949.948.553.118zM524.278 152.607l.118-.118-.138-.139a1.334 1.334 0 00-1.146-.79 1.31 1.31 0 00-1.185.514 1.255 1.255 0 00-.178 1.086l.573.573a1.228 1.228 0 001.008.296 1.3 1.3 0 00.928-.573.995.995 0 00.02-.849zM550.966 172.538l-.02-.256-.415-.415a1.773 1.773 0 00-.514-.514l-.217-.217h-.593l-.296.099-.454.454-.217.533a1.564 1.564 0 00.711 1.581l.118.118.356.079h.197l1.027-1.027.08-.355.237-.08zM493.364 135.244l.375.02.218-.02c.213-.049.41-.151.572-.297l.158-.158.158-.355-.098-.217v-.356c.081-.063.154-.136.217-.217l.652-.652v-.237l-.04-.435-.158-.158a1.257 1.257 0 00-1.066-.592 1.198 1.198 0 00-1.067.553 1.34 1.34 0 00-.277 1.027l-.335.336a1.39 1.39 0 00-.02 1.639l.454.455.257-.336zM498.362 133.881a1.612 1.612 0 001.185.395c.41-.048.785-.253 1.047-.573l.355-.356-.019-.256a1.762 1.762 0 00-.929-.929l-.118-.118-.336-.02h-.198l-1.027 1.027-.099.336.099.099.02.257.355.355-.335-.217zM532.595 120.073a1.451 1.451 0 00.395 2.015 1.279 1.279 0 00.988.237c.341-.061.646-.252.849-.533l.099-.415-.119-.672-.829-.83-.395-.079-.692.099-.296.178zM564.892 141.743l-.316-.119h-.356l-.533.178-.553.553-.079.395v.237l.059.257.612.612a1.252 1.252 0 001.521-.198 1.257 1.257 0 00.198-1.52l-.356-.356-.197-.039zM516.535 128.626l.256-.256.02-.218a1.532 1.532 0 00-1.165-1.165 1.368 1.368 0 00-1.442.573.917.917 0 00-.198.395l-.118.118.138.139a1.51 1.51 0 001.126.73 1.453 1.453 0 001.383-.316zM501.601 123.846l.119.119.257-.02.493-.494a.752.752 0 00.178-.178c.573-.573.356-1.422-.375-2.114l-.178-.177-.375-.139-.257-.019-.257.019-.889.889-.079.277-.02.217.06.415.138.138c.203.281.45.528.731.731l.138.138.316.198zM549.326 154.148l-.297-.059-.217.019-.415.415a1.399 1.399 0 00-.237.237 1.511 1.511 0 000 1.699l.593.593.296.059h.237l.435-.119c.209-.128.385-.304.513-.513a.646.646 0 00.218-.217l.118-.119-.118-.118a1.794 1.794 0 00-.929-.929l-.197-.948zM516.949 131.767l-.138-.138-.336-.138-.217-.02h-.316a1.4 1.4 0 00-.691.296l-.178.178-.178.375-.039.237.019.297.04.237.77.77.415.099.099-.099.276-.079.554-.553a1.287 1.287 0 00-.08-1.462zM551.262 149.051a1.312 1.312 0 00-.889.534l-.099.375.059.375v.237c.072.232.209.438.395.593l-.098.099.019.296.554.553c.311.268.716.403 1.126.376l.316-.119.414-.415a.363.363 0 00.119-.118 1.316 1.316 0 00.02-1.521l-.376-.376a1.327 1.327 0 00-.553-.987 1.422 1.422 0 00-1.007.098zM493.719 146.661l-.276-.039-.356-.119a1.235 1.235 0 00-1.205.415l-.138.138-.118.356-.06.257-.118.118.039.277.632.632c.267.206.592.323.929.336a1.315 1.315 0 001.6-.415l.138-.138v-.198a1.564 1.564 0 00-1.067-1.62zM491.527 119.895l.375.02.118-.119a1.48 1.48 0 00.554-.276l.158-.158.158-.395.138-.138v-.356l-.04-.237a1.335 1.335 0 00-.77-.771l-.119-.118-.355-.039-.731.73a1.401 1.401 0 000 1.66l.454.454.06-.257zM506.026 198.001l-.296.296-.04.316.554.553a1.42 1.42 0 001.165.455 1.347 1.347 0 001.047-.613.808.808 0 00.178-.177l.118-.119.04-.316-.573-.573a1.453 1.453 0 00-1.166-.415 1.516 1.516 0 00-1.027.593zM570.482 144.054l-.119-.119a1.227 1.227 0 00-1.165.336l-.158.158-.138.375-.119.119.039.355.119.119c.142.339.411.609.751.751l.118.118.375.02h.198l.672-.672a1.284 1.284 0 00-.119-1.698l-.138-.139-.316.277zM537.039 169.793l-.198.197-.197.474-.04.277.04.356.573.572a1.272 1.272 0 001.479.085c.149-.093.278-.216.378-.361l.316-.316.019-.336-.691-.691a1.42 1.42 0 00-1.679-.257zM533.839 110.097a2.357 2.357 0 00-.059.494 1.285 1.285 0 001.679 1.164 1.252 1.252 0 00.855-1.089 1.322 1.322 0 00-.104-.648l-.79-.79-.494-.059-.277.04-.75.75-.06.138zM547.054 123.313c-.103.08-.196.173-.276.276l-.119.119.257.257a1.541 1.541 0 001.106.75 1.568 1.568 0 001.245-.533l.256-.257v-.237a1.424 1.424 0 00-1.145-1.146 1.373 1.373 0 00-1.324.771zM545.138 166.514l.296.296c.145.349.422.626.771.77l.276.04.514-.514c.082-.062.155-.135.217-.217a1.354 1.354 0 00.242-1 1.343 1.343 0 00-.538-.877 1.37 1.37 0 00-1.877.218l-.118.118-.119.316-.099.099.435.751zM579.371 123.787l-.217.019-.454.455a.973.973 0 00-.198.197 1.37 1.37 0 00.277 1.897 1.396 1.396 0 001.896-.237l.138-.139.099-.414v-.238l-.039-.276-.158-.158a1.567 1.567 0 00-.573-.573l-.119-.118-.652-.415zM520.545 163.965l.237.04.296-.059.593-.593a1.485 1.485 0 00.079-1.541l-.652-.652-.277-.039-.237.039-.434.159-.158.158a.767.767 0 00-.178.177 1.367 1.367 0 00.138 1.837l.178.178.415.296zM507.034 178.543l.118.672.85.85h.197l.099-.099.277-.04.276-.276c.21-.292.301-.652.257-1.008a1.43 1.43 0 00-1.087-1.053 1.432 1.432 0 00-1.422.52l.435.434zM533.128 179.275l-.139.138-.079.395-.098.099.059.256.928.929.316.079.099.099c.238-.045.462-.147.652-.297l.652-.652.059-.454-.02-.257-.75-.75a1.482 1.482 0 00-1.679.415zM513.038 139.234l.712.711.355.079.099.099.237-.04.494-.493a.641.641 0 00.158-.158 1.323 1.323 0 00.257-1.126 1.432 1.432 0 00-.711-.949l-.297-.059h-.197l-.257.257a1.559 1.559 0 00-.514.514.532.532 0 00-.197.197l-.119.119-.02.335.139.139-.139.375zM543.282 179.432l-1.047 1.047.098.099.06.296.671.672a2.1 2.1 0 00.909.119l.02-.336.217.217.474-.197.435-.435.098-.296.02-.534-.079-.276a2.15 2.15 0 00-1.422-.948l-.454.572zM488.267 174.514l-.296.02-.316.079a1.14 1.14 0 00-.553.553 1.2 1.2 0 00.237 1.659l.276.277a1.254 1.254 0 001.521-.218l.139-.138.138-.375.119-.119-.02-.296a1.402 1.402 0 00-.316-.711l-.218-.218-.711-.513zM520.959 168.331l-.355-.04-.257.06-.514.513a1.51 1.51 0 00.336 2.035l.158.158.376.099.217.02.257-.06 1.047-1.047.039-.355-.118-.119a1.764 1.764 0 00-.929-.928l-.257-.336zM504.347 160.41h.197l.277-.04.829-.829.099-.376-.099-.099-.059-.256-.77-.771-.336-.098h-.198l-.296.019a1.425 1.425 0 00-.731.534l-.02.019.198.711.099.731.296.297.514.158zM545.593 105.1l.138-.138.138-.376.119-.118-.02-.257-.514-.514a1.4 1.4 0 00-1.574-.31c-.175.078-.331.19-.46.33l-.139.138-.118.356-.119.118-.019.257.138.138.138.336.158.158a1.517 1.517 0 001.087.415 1.393 1.393 0 001.047-.533zM568.408 79.282a1.402 1.402 0 00-1.027.593l-.178.177-.099.1-.039.236.256.257a1.543 1.543 0 001.304.79 1.204 1.204 0 001.087-.533c.195-.253.287-.571.256-.89l.099-.098-.138-.138-.138-.336-.139-.138a1.478 1.478 0 00-1.244-.02zM561.672 112.33l-.197.197-.04.316v.277l.198.474.197.197a1.397 1.397 0 001.659.079l.672-.671v-.316l-.118-.514-.218-.217a1.483 1.483 0 00-1.837-.099l.138.138-.118-.118-.336.257zM541.721 115.451l.277-.277a1.306 1.306 0 00.138-1.244 1.709 1.709 0 00-.533-.534 1.369 1.369 0 00-1.917.257l-.079.395.04.277c.001.19.042.378.119.553l.158.158c.099.16.234.295.395.395l.276.079 1.126-.059zM559.44 127.323l-.257.019a1.1 1.1 0 00-.573.099l-.73.731-.04.316.04.277.217.454.533.533.375.099h.198l.257-.059.533-.533a1.421 1.421 0 00.02-1.64l-.198-.198-.414-.217.039.119zM545.158 116.596l-.534.534a.982.982 0 00-.197.434l-.415.415a1.425 1.425 0 00-.197 1.185 1.273 1.273 0 00.79.79l.098.099.336.02 1.482-1.482a1.256 1.256 0 00-.257-1.758 1.404 1.404 0 00-.751-.237 1.506 1.506 0 00-.355 0zM518.649 116.873a1.256 1.256 0 001.284-.494.992.992 0 00.197-.474l.099-.099-.138-.138a1.334 1.334 0 00-1.087-.533 1.39 1.39 0 00-1.185.592 1.283 1.283 0 00.178 1.679l.198.198.454-.731zM545.908 115.411l-.02-.217a1.343 1.343 0 00-.928-.929 1.321 1.321 0 00-1.244.297l-.158.158-.198.395-.04.237.02.296 1.027 1.027.316.04.277-.04.474-.197.533-.534-.059-.533zM570.936 100.576l-.316.316-.02.297a1.4 1.4 0 00.257 1.363 1.312 1.312 0 001.58.355l.771-.77.079-.356.019-.217-.019-.217-.06-.297-.553-.553a1.317 1.317 0 00-1.738.079zM567.696 117.169l-.118-.118-.316-.079a1.377 1.377 0 00-1.087.533c-.096.074-.182.16-.256.257l-.099.099-.02.335.988.988.336.099.335-.02.593-.593a.808.808 0 00.178-.177 1.33 1.33 0 00.223-.49c.041-.176.046-.359.014-.538a1.433 1.433 0 00-.771-.296zM534.254 95.5l.119.118h.276l.138-.138.356-.158.158-.158.118-.119a1.254 1.254 0 00.238-1.106 1.483 1.483 0 00-.791-.988l-.098-.098-.336-.06-.237.04-.573.573a1.369 1.369 0 00.276 1.896l.139.138.217.06zM555.509 111.54l-.336-.336a1.457 1.457 0 00-1.166-.415 1.434 1.434 0 00-.987.593l-.119.118-.118.119-.04.316v.197l.04.277a1.26 1.26 0 00.77.77l.099.099.276.039.277-.276c.225-.123.41-.308.533-.534l.316-.316.02-.296-.217-.217.652-.138zM561.178 101.524a1.292 1.292 0 00-.869-.869l-.119-.118-.257.099-.691.691a1.37 1.37 0 00-.02 1.639l.632.633.296.019a1.37 1.37 0 00.909-.553 1.26 1.26 0 00.119-1.541zM527.775 119.006l.277-.276a1.53 1.53 0 00.234-.549c.04-.197.041-.4.003-.597a1.62 1.62 0 00-.494-.77l-.79-.79h-.356l-.118.118c-.205.045-.395.14-.553.277l-.159.158-.157.355-.139.139.02.335.02.218c.066.206.166.4.296.572l.356.356a1.48 1.48 0 001.066.435l.119-.119.375.138zM537.849 126.671l.119.118.099-.098c.336-.074.632-.271.829-.554a1.396 1.396 0 00.099-1.442 1.449 1.449 0 00-1.284-.809l-.217.019-.237.237a.7.7 0 00-.178.178 1.314 1.314 0 00-.257 1.166 1.504 1.504 0 00.79.987l.237.198zM547.568 131.728a1.49 1.49 0 00-.909-.909l-.138-.138h-.237l-.139.138a1.171 1.171 0 00-.77.494 1.4 1.4 0 00-.237.632l-.099.099v.237l.099.098.119.316.138.139a1.4 1.4 0 001.106.474 1.408 1.408 0 001.014-.587 1.406 1.406 0 00.211-1.151l-.158.158zM567.104 133.881l-.355-.04-1.087 1.087-.098.375.098.099.02.257.869.869.395.079.257-.02.81-.81a1.426 1.426 0 00-.573-1.916l-.336.02zM541.464 98.324a1.338 1.338 0 001.916-.256l.375-.376-.098-.098-.06-.257a1.371 1.371 0 00-1.659-.988l-.375.375a1.35 1.35 0 00-.237.237 1.4 1.4 0 00.138 1.363zM543.38 134.296a1.54 1.54 0 00-1.165-1.166 1.399 1.399 0 00-1.423.474l-.138.139-.02.217a1.349 1.349 0 00.613.968 1.53 1.53 0 001.244.217l.356-.356-.059.06.296-.297.079-.355.217.099zM526.451 110.967l.316.039.573-.573a.481.481 0 00.139-.138 1.391 1.391 0 00.038-1.735 1.388 1.388 0 00-.473-.398l-.099-.099-.296-.059-.356.355c-.213.138-.395.32-.533.533a1.324 1.324 0 00-.198.198l-.098.099-.04.276.257.257c.156.318.413.575.731.731l.039.514zM542.531 130.641l.395-.395.019-.336-.118-.118a1.032 1.032 0 00-.296-.573l-.158-.158-.356-.158-.138-.138h-.316l-.119.118c-.216.04-.415.143-.573.296l-.158.158-.158.356-.138.138v.316l.099.099c.052.232.168.444.336.612l.158.158.355.158.119.119h.316l.099-.099c.226-.044.437-.146.612-.296l.02-.257zM544.447 126.71a1.05 1.05 0 01.197-.197 1.37 1.37 0 00.217-1.166 1.502 1.502 0 00-.79-.988l-.257-.059-.355.356a1.512 1.512 0 00-.435.434 1.397 1.397 0 00-.059 1.561l.612.612.257.06.613-.613zM539.825 129.555l.158-.356.336-.336-.139-.572a1.532 1.532 0 00-1.165-1.166 1.283 1.283 0 00-1.403.494l-.138.138-.099.336-.098.099.019.336v.316l.02.217c.065.205.173.393.316.553l.158.158.356.158.138.138.336-.019.099-.099c.211-.053.407-.154.572-.296l.534-.099zM531.982 124.636l.277-.276.02-.218.019-.336-.118-.118a1.228 1.228 0 00-.316-.553l-.158-.158-.356-.158-.118-.119-.336-.019-.119.118c-.204.045-.394.14-.553.277l-.158.158-.178.375-.118.118v.356c.053.221.162.425.316.593l.158.158.356.158.138.138.336-.02.118-.118c.206-.04.398-.136.553-.277l.237-.079zM552.486 130.483l-.355-.039h-.198l-1.047 1.047-.079.316a1.546 1.546 0 00-.301 1.183 1.53 1.53 0 00.657 1.029l.118.119.297.098.217.02.316-.118.119-.119c.121-.102.233-.215.335-.336.24-.323.346-.726.297-1.126a1.365 1.365 0 00-.086-1.578 1.381 1.381 0 00-.428-.358l.138-.138zM553.237 125.782l.533-.534a1.28 1.28 0 00.178-1.165 1.506 1.506 0 00-1.087-1.086 1.363 1.363 0 00-1.284.533 1.232 1.232 0 00-.217 1.007 1.406 1.406 0 00.672.909l.099.099c.322.222.72.307 1.106.237zM538.52 106.009l.178-.257.099-.178.099-.375-.02-.218-.039-.237-.079-.197-.198-.316-.119-.158-.138-.099-.197-.079-.356-.079-.217-.02-.257.059-.178.06-.316.197-.158.158-.178.257-.098.178-.079.355-.02.218.059.256.079.198.198.316.138.138.138.099.178.099.356.079.217.02.257-.06.197-.079.316-.197.138-.138zM573.721 117.268l-.098-.099-.277.04c-.2.052-.383.154-.533.296l-.158.158-.158.356-.139.138.02.375.04.237c.149.346.425.621.77.771l.119.118.335.02 1.028-1.027.098-.376v-.197l-.059-.257-.77-.77-.218.217zM570.561 131.807l-.138.138-.079.355v.198l.019.257.514.513c.909.711 1.679.415 2.074-.098l.119-.119.118-.316.099-.099v-.237l-.119-.316-.158-.158a1.424 1.424 0 00-1.185-.632 1.484 1.484 0 00-1.264.514zM594.324 125.505a1.404 1.404 0 00-.197 1.146 1.498 1.498 0 00.849.849l.119.119.335.02.297-.297c.197-.097.356-.257.454-.454a1.283 1.283 0 00.237-.988l-.415-.415a1.367 1.367 0 00-1.106-.434 1.4 1.4 0 00-.573.454zM548.535 100.853l-.118.118-.079.277-.099.099-.02.217.81.81.376.138.118.119.909-.909.079-.277.099-.098-.099-.099-.059-.257-.099-.099a1.395 1.395 0 00-1.166-.731 1.538 1.538 0 00-.652.692zM577.613 112.428l-.316.316-.02.336 1.028 1.027.335.099.099.099.277-.04c.217-.035.418-.139.573-.296l.197-.197.158-.435.02-.296-.059-.297-.553-.553a1.317 1.317 0 00-1.739.237zM543.163 122.621v-.237l-.099-.099-.138-.335-.119-.119-.059-.257-.849-.849-.395-.079h-.198l-.276.079-.356.355.099-.098a1.362 1.362 0 00-.237 1.106l.178.178-.04.316.02.256.178.455 1.066 1.067.316.039.277-.039.454-.178.257-.257a1.476 1.476 0 00.198-1.264l-.277-.04zM559.262 122.206h-.197l-.238.04-.572.573a1.43 1.43 0 00-.257 1.086 1.37 1.37 0 00.691.929l.119.118.355.079.099-.099.237-.039.573-.573a1.373 1.373 0 00.237-1.106 1.393 1.393 0 00-.672-.909l-.118-.118-.257.019zM528.565 80.368l-.316-.316a1.345 1.345 0 00-1.571-.131c-.167.099-.311.23-.424.388l-.296.296-.04.356 1.166 1.165.316.04.257-.06.474-.197.513-.514a1.336 1.336 0 00-.079-1.027zM528.921 181.507l-.455-.455a1.28 1.28 0 00-1.027-.79 1.146 1.146 0 00-.711.119l-.711.711a1.741 1.741 0 00-.316 1.343l.474.474c.233.202.523.326.83.356l.237.039.256-.059.633-.632a.664.664 0 00.197-.198l.119-.118.474-.79zM550.155 194.05l-.335-.02-.692.692a1.451 1.451 0 00-.02 1.679l.534.533.256.02a1.287 1.287 0 00.889-.573 1.228 1.228 0 00.218-1.126 1.506 1.506 0 00-.889-.889l.039-.316zM530.501 147.333l.889-.889.039-.316-.039-.277-.217-.454-.198-.198a3.903 3.903 0 01-.652-.928l-.632-.632-.276-.04-.277.04-.435.158-.335.336a1.305 1.305 0 00-.218 1.205l.04.237.158.434.079.553.593.593a1.369 1.369 0 001.481.178zM525.997 144.765l-.118-.118-.119-.317-.118-.118a1.566 1.566 0 00-1.6-.652l-1.107 1.106-.197.395-.138.139.019.296.139.138.197.395.553.553.415.06h.237l1.501-1.502.336-.375zM534.925 151.837l-.296.019-.632.633a1.31 1.31 0 00.146 1.592c.13.14.289.251.466.324l.099.098.376.02.77-.77a1.313 1.313 0 00-.119-1.699l-.158-.158-.652-.059zM552.625 162.622l-.079.395.059.375v.198a1.179 1.179 0 00.751.751 1.323 1.323 0 001.047-.06l.395-.395a1.201 1.201 0 00.217-.928 1.29 1.29 0 00-.514-.83 1.427 1.427 0 00-1.876.494zM571.766 175.245l.02-.415a1.198 1.198 0 00-.692-.691 1.336 1.336 0 00-1.066-.079 1.287 1.287 0 00-.968 1.088 1.3 1.3 0 00.039.512l.277.276.612.336.395.039.672-.671.711-.395zM520.288 186.425l-.277-.276-.612-.336-.415-.059a1.45 1.45 0 00-.435.197l-.276.277-.336.612-.039.395a1.298 1.298 0 00.634.786 1.308 1.308 0 001.005.103 1.396 1.396 0 00.751-1.699zM577.02 164.617l-.533.534a1.31 1.31 0 00-.297 1.244l.178.375.692.692h.316l1.284-1.284.059-.336v-.197l-.119-.119a1.393 1.393 0 00-1.56-1.047l-.02.138zM556.477 181.803l-.613.612-.098.297c-.003.153.017.306.059.454a1.253 1.253 0 00.77.77 1.387 1.387 0 001.264-.039l.652-.652.079-.277-.079-.395a1.302 1.302 0 00-.634-.786 1.327 1.327 0 00-1.005-.103l-.395.119zM582.275 174.336l-.178.178-.158.434-.02.336.06.376a1.31 1.31 0 00.849.849 1.513 1.513 0 001.304-.237l.533-.533v-.277a1.528 1.528 0 00-.402-.704 1.523 1.523 0 00-.704-.402l-.257-.02-.434.119-.159.158-.434-.277zM544.565 186.129l-.019-.257.098-.098a1.368 1.368 0 00.139-1.087 1.843 1.843 0 00-.929-.928l-.118-.119-.356-.039-.75.75a1.45 1.45 0 00-.159 1.106 1.461 1.461 0 00-.296.929l.909.909.375.098.217.02a1.59 1.59 0 001.205-.889l-.316-.395z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M529.928 182.001l.217-.218v-.237l-.158-.158a1.344 1.344 0 00-1.086-1.086 1.506 1.506 0 00-1.482.849l-.237.237.119.119.039.276a1.34 1.34 0 001.086 1.087 1.487 1.487 0 001.502-.869zM602.74 184.608l.375.375a1.313 1.313 0 001.008.534l.217-.218c.261-.137.475-.35.612-.612l.119-.119v-.355l-.771-.771a1.59 1.59 0 00-1.619.079 1.31 1.31 0 00.059 1.087zM547.054 164.598l.652-.652v-.316a1.422 1.422 0 00-1.323-1.047l-.139.138-.395.158-.592.593a1.145 1.145 0 00-.119.592l.988.988a1.323 1.323 0 001.086-.059l-.158-.395zM561.257 160.548a1.306 1.306 0 00-.79-.79 1.459 1.459 0 00-1.245.059l-.75.751-.02.336 1.067 1.066.355.04.217-.02 1.028-1.027.039-.356.099-.059zM552.151 185.734l-.731-.731a1.42 1.42 0 00-.81-.217l-.83.829-.118.316a1.47 1.47 0 000 .751l.79.79.296.059a1.504 1.504 0 001.008-.572 1.346 1.346 0 00.395-1.225zM618.542 168.45l-.474.474a1.348 1.348 0 00-.02.849c.092.131.205.244.336.336l.356.355a1.286 1.286 0 001.56-.177l.494-.494.099-.376-.059-.335a1.485 1.485 0 00-1.324-1.047l-.237.039-.514.198-.217.178zM553.079 160.943h.237l.455-.138.553-.553.138-.415.02-.257-.04-.237a1.424 1.424 0 00-1.58-1.027l-.652.652a1.333 1.333 0 00-.118 1.066 1.398 1.398 0 00.987.909zM583.026 152.212c.281.075.58.054.849-.059l.83-.83.019-.375-.118-.119a1.565 1.565 0 00-1.106-1.106l-.237.04-.435.118-.652.652a1.476 1.476 0 00-.217.77l1.027 1.028.04-.119zM551.084 162.583l-.158-.356-.415-.415a1.43 1.43 0 00-1.383-.039 1.582 1.582 0 00-.545.546 1.577 1.577 0 00-.225.738l.059.257.612.612a1.569 1.569 0 001.66-.119 1.399 1.399 0 00.395-1.224zM582.808 142.395l-.434.434-.198.553-.019.297a1.433 1.433 0 00.79.79c.362.16.773.174 1.145.039l.711-.711.08-.276a.996.996 0 00-.04-.356 1.313 1.313 0 00-1.442-.928l-.158.158h-.435zM573.306 159.363l-.296-.296-.356-.04h-.197l-.77.771a1.555 1.555 0 00-.218 1.244 1.481 1.481 0 001.739.948l.948-.948a1.236 1.236 0 00-.04-.751l-.573-.572-.237-.356zM557.702 168.963a.926.926 0 00.513.04c.227.097.467.163.711.197.116.219.295.398.514.514l-.652.651-.04.356a1.65 1.65 0 00.06.454 1.283 1.283 0 001.303.909l.119-.118.336-.139.79-.79.039-.395-1.007-1.007.356-.356a1.336 1.336 0 00.335-1.323c.033-.116.046-.236.04-.356l-1.284-1.284a1.856 1.856 0 00-.672-.158l-.316-.079h-.197a1.312 1.312 0 00-.889-.257l-.296.099-.83.83-.04.237v.434l.158.158c.117.158.257.298.415.415a1.345 1.345 0 00.534.968zM530.915 132.38l-.711-.712a1.134 1.134 0 00-.511-.09 1.1 1.1 0 00-.496.15 1.275 1.275 0 00-.613.849l-.138.138-.04.395a1.23 1.23 0 001.581.87c.287-.089.542-.26.731-.494a1.434 1.434 0 00.197-1.106zM549.287 157.15l.256.257a1.287 1.287 0 001.601.455l.592-.593c.194-.32.277-.695.237-1.067l-.573-.573a1.59 1.59 0 00-1.718.139 1.318 1.318 0 00-.455 1.442l.06-.06zM559.084 155.096l.277-.277.039-.395a1.388 1.388 0 00-.686-.852 1.391 1.391 0 00-1.091-.096 1.423 1.423 0 00-.791.79 1.254 1.254 0 00-.118 1.067l.434.435a1.315 1.315 0 001.067.355 1.26 1.26 0 001.008-.691l-.139-.336zM553.296 154.01l.514-.514.04-.316-.099-.099a1.397 1.397 0 00-.376-.612l-.572-.573a1.344 1.344 0 00-1.324.415 1.177 1.177 0 00-.296 1.244c.127.303.368.545.671.672l.257.257a1.37 1.37 0 001.205-.099l-.02-.375zM546.995 152.173l.534-.02.296-.099.454-.454.198-.514v-.316l-.119-.118a1.392 1.392 0 00-1.639-1.047 1.395 1.395 0 00-.889.256 1.434 1.434 0 00-.534 1.008l.099.099.099.296.593.592c.232.094.48.141.731.139l.177.178zM525.345 168.766a1.373 1.373 0 00-.928-.929 1.293 1.293 0 00-1.245.336l-.415.415v.276l1.047 1.047a1.28 1.28 0 001.225-.434c.175-.2.285-.448.316-.711zM578.442 126.453l.791.791c.211.093.441.134.671.118l.929-.928c.088-.199.128-.415.118-.632l-.908-.909a1.482 1.482 0 00-.652-.138l-.139.138-.375.178-.513.513a1.233 1.233 0 00-.119.632l.197.237zM577.158 142.276l-.336.336c-.093.19-.14.4-.138.612l1.047 1.047.395.04.217-.02a1.317 1.317 0 00.938-1.127 1.31 1.31 0 00-.049-.513l-.928-.928-.336-.02-.336.336-.474.237zM561.02 137.041l-.968-.968-.612-.296-.375-.02a1.312 1.312 0 00.375 2.588 1.28 1.28 0 001.114-.259 1.29 1.29 0 00.466-1.045zM562.66 146.108l-.415.415-.06.375-.098.099.059.257a1.342 1.342 0 00.889.889 1.404 1.404 0 001.244-.138l.949-.949.059-.375-.02-.217-1.007-1.008-.376-.019-.098-.099-.435.434-.691.336zM577.593 174.988l-.119.118a1.397 1.397 0 00-.414 1.442 1.456 1.456 0 002.081.84c.257-.137.467-.347.605-.603l.138-.138.02-.415-.632-.632a1.425 1.425 0 00-1.541-.395l-.099.099-.039-.316zM592.072 115.273l.81-.81a1.374 1.374 0 00.257-1.244l-.948-.949a1.541 1.541 0 00-1.067.158l-.869.87c-.017.213.01.429.079.632a1.345 1.345 0 00.52 1.124c.17.132.37.22.582.258.212.039.431.025.636-.039zM599.302 218.11a1.254 1.254 0 001.007 1.007l.336-.059.257-.257c.289-.131.521-.363.652-.651.154-.307.183-.661.079-.988l-.375-.375a1.285 1.285 0 00-1.62-.237 1.452 1.452 0 00-.336 1.56zM503.952 175.027a1.397 1.397 0 00.829-1.698l-.237-.474-.434-.435a1.375 1.375 0 00-1.324.099l-.415.415-.138.454.198.197-.336-.019c-.052.241-.046.492.02.731l.948.948.355-.119.534-.099zM600.962 127.402l-.435.434-.059.297a1.424 1.424 0 001.679.928 1.289 1.289 0 001.027-1.422l-.612-.613a1.368 1.368 0 00-1.798-.098l.198.474zM571.707 152.666a1.26 1.26 0 00-.909-.908l-.257-.02-.454.138a.61.61 0 00-.119.119l-.869.869-.02.375.099.099a1.31 1.31 0 001.324 1.007l.81-.809a1.46 1.46 0 00.395-.87zM577.218 173.744l.098-.415v-.237l-.118-.119a1.615 1.615 0 00-.889-1.086l-.197-.198a1.51 1.51 0 00-1.067-.355 1.928 1.928 0 00-1.699 1.106l-.118.118-.02.376.987.987a1.345 1.345 0 00.988.553l.217.218.257.059a1.278 1.278 0 00.988-.593l.257-.257.316-.157zM542.669 212.204l.612-.613.02-.336-.257-.256a1.165 1.165 0 00-.75-.751 1.12 1.12 0 00-1.028.079l-.197.198a1.224 1.224 0 00-.593 1.224l.83.83.316.079c.354.077.725.028 1.047-.138v-.316zM551.064 193.378l.257.257.316.04a1.457 1.457 0 001.047-.692 1.317 1.317 0 00.138-1.086.273.273 0 00-.055-.083.24.24 0 00-.083-.055l-.613-.613a1.283 1.283 0 00-1.106.04l-.711.711a1.43 1.43 0 00.039.751.547.547 0 00.178.177l.455.455.138.098zM556.418 172.183l-.613-.612-.276-.079-.257-.02h-.277l-.474.197-.434.435a1.227 1.227 0 00-.257.454v.316l.158.553.454.455.297.099c.341.156.729.178 1.086.059l.889-.889a1.459 1.459 0 00-.296-.968zM546.501 175.067l.02.217a1.372 1.372 0 001.381 1.032c.208-.009.411-.064.594-.163l1.363-1.363.079-.434-.019-.217-1.067-1.067h-.435l-.098.099a1.386 1.386 0 00-.81.809c-.281.143-.51.371-.652.652l-.119.119-.237.316zM570.027 149.525l.099.099h.316l.099-.099a1.508 1.508 0 00.988-1.817l-.889-.889-.435-.039-.257.019-.889.889a1.32 1.32 0 00-.059.889c.059.187.153.362.277.514l.454.454.296-.02zM569.751 170.662l-.138-.138a1.365 1.365 0 00-1.344-.988l-.296.02-.849.849a1.481 1.481 0 00-.02 1.047.523.523 0 00.178.178l.592.592.297.06a1.76 1.76 0 001.145-.632l.139-.139.296-.849zM560.23 175.403l-.099-.297-.533-.533a1.668 1.668 0 00-.375-.375 1.896 1.896 0 00-.395-.395l-.534-.534-.335-.059-.099.099a1.672 1.672 0 00-1.225.79l-.119.119-.039.316a1.422 1.422 0 00-.277 1.066l.731.731a1.7 1.7 0 00.889.415l.297-.059.276.039.336-.059a1.48 1.48 0 001.501-1.264zM564.97 172.835l-.691-.692-.296-.059-.099-.099a1.43 1.43 0 00-1.231 1.051 1.419 1.419 0 00-.033.569l.652.652a1.422 1.422 0 001.58-.119c.215-.164.374-.39.455-.649a1.306 1.306 0 00-.001-.792l-.336.138zM570.64 165.743l-.415.415-.059.336v.197l.039.237a1.4 1.4 0 001.561 1.008l.79-.79.099-.336a1.545 1.545 0 00-.02-.731l-.948-.948-.316-.04-.376.376-.355.276zM528.091 197.625a1.336 1.336 0 00-.307-.483 1.336 1.336 0 00-.483-.307 1.512 1.512 0 00-1.363.06l-.494.494a1.474 1.474 0 00.21 1.68c.135.15.298.27.481.354l.158.158.217.02h.237a1.588 1.588 0 001.087-.691 1.32 1.32 0 00.257-1.285zM594.324 168.489l.139-.138.039-.356.099-.099-.119-.118a1.471 1.471 0 00-.888-.889 1.434 1.434 0 00-1.166.02l-.336.336a1.23 1.23 0 00-.494 1.402c.047.172.128.334.238.474l.474.474.256.06c.232.034.469.013.692-.06l.513-.513.553-.593zM597.92 162.958l-.415.415-.079.355.217.218c.075.327.241.627.479.864.237.238.537.404.864.479l.317-.04.77-.77c.177-.327.219-.709.118-1.067a1.563 1.563 0 00-.513-.75l-.534-.534-.414-.059-.395.395-.415.494zM568.843 176.706l-.06-.256a1.443 1.443 0 00-.811-.917 1.443 1.443 0 00-1.223.028l-.712.711-.059.296a.671.671 0 00.04.277 1.512 1.512 0 00.849.849 1.483 1.483 0 001.225-.039l.711-.712.059-.296-.019.059zM600.468 176.252l-.139-.138a1.17 1.17 0 00-1.323-.257l-.889.889-.04.395a1.346 1.346 0 00.889.889c.285.079.585.079.869 0 .341-.1.627-.331.797-.642.171-.311.211-.677.112-1.017l-.276-.119zM625.397 186.09l-.455.454.06.454a1.398 1.398 0 001.185.988 1.51 1.51 0 001.56-.81l.158-.158.02-.217v-.237a.675.675 0 00-.059-.297 1.373 1.373 0 00-.929-.928 1.562 1.562 0 00-1.501.356l-.039.395zM555.094 178.01l-.612.612-.059.297.059.336a1.342 1.342 0 00.889.889 1.485 1.485 0 001.461-.277l.198-.198.217-.454.02-.257-.02-.335-.691-.692a1.538 1.538 0 00-1.462.079zM590.71 220.619l-.474-.474a1.306 1.306 0 00-.494-.257l-.277-.04-1.145 1.146-.04.395c.133.311.381.559.692.691.31.161.671.196 1.007.099a1.31 1.31 0 00.731-1.56zM600.646 192.233a1.508 1.508 0 00-1.58-1.067l-.237.04-.415.177-.534.534-.098.454v.237l.098.296.929.929.336-.02.039-.276.059.256.494-.098.573-.573.158-.474.04-.277.138-.138zM580.714 169.259l1.106 1.107.356.039 1.086-1.086.06-.376-.02-.217a1.404 1.404 0 00-.81-.81 1.437 1.437 0 00-1.146-.039l-.81.81.178.572zM610.7 172.894a1.29 1.29 0 00-1.343-.751l-.534.534c-.19.165-.33.38-.403.622a1.352 1.352 0 00-.012.741l.238.434-.04-.039.356.355.217.02.099.099.296-.02.099-.099a1.338 1.338 0 001.066-1.343l-.158-.158.119-.395zM543.103 131.609l.494-.494.02-.375-.099-.099-.079-.316-.968-.968-.375-.059h-.198a1.483 1.483 0 00-1.146 1.146v.276l.178.494.632.632.257.02c.283.065.579.045.85-.059l.375-.376.059.178zM540.141 200.984l-.06-.297-.197-.513-.632-.632-.395-.04-.139-.138a1.307 1.307 0 00-1.007.612c-.1.153-.164.327-.187.508-.024.181-.008.365.049.539l.553.553a1.453 1.453 0 001.303.079c.302-.143.551-.378.712-.671zM539.667 184.114l.454-.454a.997.997 0 00.217-.218 1.48 1.48 0 00.672-1.619l-.751-.751-.296-.099-.099-.098a1.814 1.814 0 00-1.304.553 1.765 1.765 0 00-.335.73l-.198.198a1.285 1.285 0 00-.415 1.363l.929.928.335.02.395-.395.396-.158zM563.272 131.629l.099.296.81.81.335.059h.198a1.419 1.419 0 00.988-.632 1.449 1.449 0 00.118-1.185l-.928-.928-.336-.02-.217.02-.988.987-.04.395-.039.198zM617.831 142.73l.869.869c.215.106.453.154.692.139l.79-.79.138-.336.119-.119a2.008 2.008 0 00-.079-.474 1.395 1.395 0 00-1.66-.987 1.367 1.367 0 00-.869 1.698zM605.11 162.958a1.26 1.26 0 00-.494 1.086l.987.988a1.29 1.29 0 001.067-.039l.81-.81v-.395l-.099-.099a1.311 1.311 0 00-1.462-.948l-.138.138-.671.079zM612.794 186.445l-.316.316a1.26 1.26 0 00-.138.652l1.027 1.027.395.04.119-.119a1.288 1.288 0 00.928-1.64l-.928-.928h-.316l-.316.316-.455.336zM562.758 155.768l.652.651a1.343 1.343 0 001.324-.414 1.378 1.378 0 00.316-1.265 1.263 1.263 0 00-.771-.77l-.237-.237-.296-.02-.099-.099a1.43 1.43 0 00-.869.633 1.315 1.315 0 00-.099 1.086.707.707 0 00.079.435zM530.599 164.538l.277.277.454.059c.243.042.491.036.731-.02l.85-.849.019-.336-.079-.316-.296-.494a1.627 1.627 0 00-1.679-.415l-.83.83-.039.356.019.217.099.296.534.533-.06-.138zM595.47 142.138l-.415.414a1.224 1.224 0 00-.118.633l.81.81-.02.335 1.225 1.225a1.306 1.306 0 001.401-1.032 1.304 1.304 0 00-.019-.627 1.424 1.424 0 00-.75-.751v-.316l-.099-.099a1.4 1.4 0 00-1.264-1.027l-.237.04-.415.138-.099.257zM593.456 146.8a1.673 1.673 0 00-1.047.691 1.472 1.472 0 00-.178 1.165l.75.751c.429.142.896.114 1.304-.079l.771-.77.039-.356-.099-.099-.098-.375-.81-.81-.356-.079-.276-.039zM583.242 182.376a1.456 1.456 0 00-.712-.85 1.454 1.454 0 00-1.105-.098 1.334 1.334 0 00-.731.73 1.288 1.288 0 00-.099 1.008l.277.276.612.336.415.02.514-.514.336-.612.493-.296zM546.264 194.465l.474-.474.218-.533.019-.336-1.066-1.067-.415.02-.217.02a1.282 1.282 0 00-.87 1.58 1.311 1.311 0 00.949.948l.533.02.375-.178zM586.423 164.459l.593.593a1.48 1.48 0 001.62-.158 1.42 1.42 0 00.474-1.501l-.198-.395-.375-.376a1.472 1.472 0 00-1.264.435 1.397 1.397 0 00-.85 1.402zM599.263 155.768l-.297.296-.079.356v.197l.06.257a1.502 1.502 0 001.896.988l.948-.949.059-.375-.039-.316-.928-.928a3.544 3.544 0 00-1.028.039l-.138.139-.454.296zM546.106 174.553l.651-.652.06-.296-.079-.355a1.395 1.395 0 00-1.304-1.028l-.454.099-.534.534-.177.414c-.065.241-.058.495.019.731l.751.751a1.575 1.575 0 001.323-.059l-.256-.139zM583.638 162.859l.75-.75c.134-.228.203-.487.198-.751l-.138-.138-.198-.395-.533-.534-.415-.059-.257.02-1.047 1.047v.434l.099.099a1.285 1.285 0 001.541 1.027zM548.931 179.195a1.229 1.229 0 00.119-.987l-.712-.711a1.245 1.245 0 00-.711-.079l-.632-.633-.296-.019-.099.099a1.648 1.648 0 00-.988.632l-.118.118-.079.316-.099.099.336.336a1.312 1.312 0 001.185.948l.277-.04.178-.177a1.334 1.334 0 001.099.18 1.349 1.349 0 00.856-.714l-.316.632zM586.265 135.382l.533-.534a1.259 1.259 0 00.376-1.204l-1.067-1.067-.395-.04h-.198a1.401 1.401 0 00-.81.534l-.118.118a1.284 1.284 0 00-.672 1.502l.613.612a1.594 1.594 0 001.738.079zM608.685 150.138l-.414.415a1.145 1.145 0 00-.119.592l.138.139.198.395.573.573a1.232 1.232 0 001.086-.06l.771-.77v-.316l-.099-.099a1.396 1.396 0 00-1.304-1.027l-.237.039-.375.178-.218-.059zM538.126 113.436l-.869-.869-.297-.06a1.458 1.458 0 00-1.106.633 1.425 1.425 0 00-.138 1.205.566.566 0 00.158.158l.711.711.336.059a1.474 1.474 0 001.047-.652 1.32 1.32 0 00.158-1.185zM541.029 197.448l.178-.178a1.17 1.17 0 00.454-1.126l-.948-.948a1.293 1.293 0 00-1.047.059l-.83.83.02.375a1.29 1.29 0 001.422.948l.119-.118.632.158zM541.326 155.234l-.04-.276-.257-.455-.573-.572-.395-.04-.217.02a1.314 1.314 0 00-.811.623 1.322 1.322 0 00-.117 1.016l.928.929.336.02.099-.099a1.396 1.396 0 001.047-1.166zM590.354 189.645l-.356-.356a1.256 1.256 0 00-1.461-.039 1.287 1.287 0 00-.475 1.422 1.255 1.255 0 00.889.889l.119.119h.237l1.106-1.107-.059-.335-.217-.218.217-.375zM571.806 191.877l-.869-.869a1.511 1.511 0 00-1.265.395 1.573 1.573 0 00-.375 1.324l.632.632a1.453 1.453 0 002.212-1.739l-.335.257zM538.936 163.63l-.119-.119-.276-.079a1.344 1.344 0 00-1.067.158l-.533.534-.079.276.059.296a1.366 1.366 0 002.647-.671l-.652-.652.02.257zM551.459 216.925l.395-.395.079-.316-.059-.257a1.346 1.346 0 00-.615-.833 1.354 1.354 0 00-2.012 1.485l.928.928c.27.055.55.034.81-.059l.118-.119.356-.434zM556.101 145.832l-.513.513-.02.415.02.217a1.506 1.506 0 001.086 1.087l.257-.02.434-.118.554-.553.118-.435.04-.237-.04-.237a1.424 1.424 0 00-1.58-1.027l-.139.138-.217.257zM552.961 126.216l-.613-.335-.395-.04-.118-.119-.296.099-.909.909-.04.395a1.3 1.3 0 00.291.44 1.3 1.3 0 00.44.291c.325.162.698.198 1.047.099a1.373 1.373 0 00.929-1.719l-.336-.02zM536.289 146.325l-.869-.869-.376-.019-.829.829a1.702 1.702 0 00-.178.692l-.119.118.02.257.83.83.296.098.218.02.355-.039.119-.119a1.483 1.483 0 00.533-1.798zM580.22 155.175l-.335-.02-.099.099-.297.099-.809.81-.06.336.218.217a1.368 1.368 0 00.829.829 1.348 1.348 0 001.166.02l.632-.632a1.819 1.819 0 00-.02-.731l-.849-.849-.376-.178zM529.987 154.286l.277-.276-.02-.257-.118-.435-.751-.75-.237-.04-.454-.02-.257.06a2.092 2.092 0 00-.652.652l.257.256-.336-.098-.138.533v.276l.987.988a2.154 2.154 0 001.62-.513l-.178-.376zM567.677 113.91l.513-.514a1.256 1.256 0 000-.79 1.227 1.227 0 00-1.442-.849l-.296.296a1.259 1.259 0 00-.771 1.324l.751.75c.232.142.499.217.77.217l.277-.079.198-.355zM552.388 143.422l.671-.672v-.355l-.118-.119a1.456 1.456 0 00-1.778-.988l-.909.909-.039.356.02.217.829.83c.429.141.896.113 1.304-.079l.02-.099zM568.823 135.204l-.455.454-.019.376.019.217.988.988.375.059.949-.948c.112-.291.126-.61.039-.909l-.691-.296-.652-.375-.296.296-.257.138zM587.035 201.853l.85.849.375.099.217-.02a1.407 1.407 0 00.967-.532c.116-.15.201-.322.249-.505.047-.184.057-.375.029-.563l-.849-.849-.376-.099-.099-.099a1.598 1.598 0 00-1.027.593 1.35 1.35 0 00-.336 1.126zM593.297 235.73a1.448 1.448 0 00.158-1.185l-1.007-1.007a1.67 1.67 0 00-1.403.533 1.287 1.287 0 00-.256 1.205c.086.309.283.576.553.751l.276.276.375.099.099-.099a1.544 1.544 0 001.205-.573zM571.411 210.051l-.613-.613-.474-.158-.296-.02a1.4 1.4 0 00-1.067 1.265l.02.375.079.276.434.435.514.237.296.02a1.369 1.369 0 001.284-1.284l-.118-.119.099-.098-.158-.316zM581.248 192.391l-.02.415a1.266 1.266 0 001.62.869 1.345 1.345 0 001.034-1.108 1.326 1.326 0 00-.027-.532l-.493-.494a1.128 1.128 0 00-.494-.217l-.119-.118a1.676 1.676 0 00-.553.079l-.494.493-.454.613zM560.981 199.068l.197-.198a1.36 1.36 0 00.277-.514v-.276l-.119-.474-.612-.613-.494-.098h-.276l-.317.079-.829.829-.04.395-.098.099a1.394 1.394 0 001.264 1.027l.276-.039.435-.198.336-.019zM578.206 194.406l.217-.218-.02-.217a.919.919 0 00-.316-.316l.039-.395-.098-.099a1.422 1.422 0 00-.85-.849 1.46 1.46 0 00-1.145.079l-.632.632-.02.336.237.237-.257.257.02.256a1.26 1.26 0 001.065.929c.166.02.334.006.495-.04.254-.07.476-.223.632-.434a1.01 1.01 0 00.633-.158zM594.364 173.388a1.282 1.282 0 00-.375 1.323c.043.164.117.318.217.455l.553.553a1.368 1.368 0 001.106-.514 1.447 1.447 0 00.257-1.323 1.309 1.309 0 00-.546-.713 1.306 1.306 0 00-.876-.196l-.198.198-.138.217zM578.719 195.749l.158.158a1.32 1.32 0 001.279-.127 1.315 1.315 0 00.558-1.157l-.987-.988-.395-.039-.376.098-.612.613-.118.513v.277l.454.454.039.198zM574.966 224.589l-.534-.533a1.446 1.446 0 00-1.224-.672 1.338 1.338 0 00-1.245 1.047l.02.257.099.454 1.027 1.027.395.04.217-.02a1.279 1.279 0 001.013-.496 1.287 1.287 0 00.232-1.104zM563.884 211.809l.573-.573a1.376 1.376 0 00.237-1.225l-.869-.869h-.198l-.099.099-.494.177-.632.632-.059.415v.237l.04.237c.141.34.411.609.75.751.233.114.494.156.751.119zM601.554 198.831l-.474.474-.118.395v.197l.118.119a1.315 1.315 0 00.79.79 1.516 1.516 0 001.264 0l.731-.731.04-.356a1.366 1.366 0 00-1.64-.967l-.138.138-.573-.059zM575.855 205.685l.296.02a1.415 1.415 0 001.047-.652 1.36 1.36 0 00.158-1.146l-.79-.79a1.32 1.32 0 00-1.086.059l-.79.791c-.054.241-.047.492.019.73l.099.297.593.592.454.099zM580.221 216.332a1.4 1.4 0 001.244-.138l.731-.731v-.316a1.396 1.396 0 00-1.264-1.027l-.277.079-.632.632a1.455 1.455 0 00-.118 1.067c.073.165.181.313.316.434zM587.134 179.195l-.04.396a1.268 1.268 0 00.81.809c.277.132.59.167.889.099l.375.02.929-.928c.092-.184.133-.388.118-.593l-.138-.138-.178-.376-.533-.533a1.572 1.572 0 00-.632-.119l-.119.119-.217-.02a1.484 1.484 0 00-.988.593l-.138.138-.138.533zM574.669 182.534l-.553.553a1.479 1.479 0 00-.059 1.007 1.312 1.312 0 001.007 1.008 1.398 1.398 0 001.423-.514l.237-.237-.099-.099-.079-.276a1.363 1.363 0 00-.319-.511 1.35 1.35 0 00-.511-.319 1.479 1.479 0 00-1.264.04l.217-.652zM564.812 186.86a1.376 1.376 0 001.284-.099l.494-.494a1.373 1.373 0 00-.276-1.501l-.435-.435-.336-.098-.099-.099a1.397 1.397 0 00-1.046.612 1.232 1.232 0 00-.178 1.166 1.287 1.287 0 00.592.948zM551.104 201.043l.652-.652.02-.336-.079-.316-.85-.849-.395-.04-.099.099-.256.059-.929.929.02.415.079.276a1.365 1.365 0 001.302.854c.185-.004.367-.046.535-.123v-.316zM594.897 202.762a1.397 1.397 0 00-.988 1.698l.751.751a1.395 1.395 0 001.837-.691l.316-.316-.039-.356a1.437 1.437 0 00-.688-.976 1.412 1.412 0 00-1.189-.11zM565.386 182.099a1.508 1.508 0 001.639-.099 1.37 1.37 0 00.514-1.422l-.573-.573a1.462 1.462 0 00-1.146-.118 1.504 1.504 0 00-.889.889l-.118.118v.553l.671.672-.098-.02zM594.186 183.166l-.573.573-.039.316.138.138c.074.204.192.39.346.543.153.154.339.272.543.346a1.434 1.434 0 001.166-.02l.81-.81a1.52 1.52 0 00-.02-.731l-.099-.296-.652-.652a1.347 1.347 0 00-1.027.158l-.593.435zM568.803 183.778l.177.376.514.513c.201.104.426.152.652.139l.138-.139.356-.158.513-.513c.104-.201.152-.427.138-.652l-.138-.139-.158-.355-.513-.514a1.175 1.175 0 00-.672-.118l-.889.889a1.25 1.25 0 00-.138.652l.02.019zM570.62 186.899l-.138.139-.099.375v.197l.04.277a1.46 1.46 0 00.824.915 1.448 1.448 0 001.23-.046l.751-.751a2.2 2.2 0 00-.04-.632 1.391 1.391 0 00-1.224-.987l-.218.019-.414.099-.297.296-.415.099zM571.311 182.218l.929.928a1.506 1.506 0 001.619-.118 1.343 1.343 0 00.514-1.423l-1.205-1.205a1.347 1.347 0 00-.632-.118l-.928.928a1.084 1.084 0 00-.119.632l-.178.376zM579.964 179.097l.79.79c.2.083.415.123.632.118l.928-.928c.092-.184.133-.388.119-.593l-.138-.138-.178-.375-.395-.395.02.019a1.351 1.351 0 00-.672-.118l-.929.928c-.093.19-.127.403-.098.613l-.079.079zM562.778 191.62l.553-.553.04-.355-.119-.119-.099-.375-.395-.395a1.557 1.557 0 00-.751-.976 1.56 1.56 0 00-1.224-.13l-.731.73-.099.297-.099.099a1.28 1.28 0 00.079.474c.113.381.358.71.692.928a1.33 1.33 0 00.747.794 1.335 1.335 0 001.09-.024l.316-.395zM566.037 195.117l-.138.138v.198a1.306 1.306 0 00.829.829 1.426 1.426 0 001.482-.296 1.364 1.364 0 00.395-1.343 1.339 1.339 0 00-.731-.731 1.428 1.428 0 00-1.106-.04l-.138.139a1.356 1.356 0 00-.593 1.106zM590.69 195.709l.099.297.079.197.237.277.177.138.218.138.197.04.376.059.197-.039.158-.04.178-.099.296-.217.139-.178.098-.217.079-.198.02-.375v-.197l-.099-.297-.079-.197-.257-.297-.138-.138-.237-.118-.197-.04-.376-.059-.197.039-.158.04-.178.099-.296.217-.119.158-.138.217-.059.218-.04.355.02.217zM560.091 216.51l.336-.336a1.06 1.06 0 00.099-.612l-.118-.119-.178-.375-.593-.593a1.406 1.406 0 00-1.086.06l-.652.651-.02.336.099.099.059.257.909.908.375.06.237-.237.533-.099zM550.59 205.033l-.83-.829-.355-.04h-.198a1.43 1.43 0 00-1.222 1.115 1.436 1.436 0 00-.003.584l.968.968.237.039a1.508 1.508 0 001.225-.553 1.426 1.426 0 00.178-1.284zM541.029 227.631a1.339 1.339 0 00-.83-.83 1.257 1.257 0 00-1.185.079l-.77.771c-.06.22-.06.452 0 .671.147.313.398.565.711.712l.158.158h.197a1.344 1.344 0 001.028-.593 1.42 1.42 0 00.691-.968zM588.537 206.791l-.712-.711-.296-.099c-.14-.002-.28.018-.415.059l-.987.988-.06.415v.237l.751.751a1.512 1.512 0 001.688-1.029c.073-.229.09-.472.05-.71l-.019.099zM561.513 222.555l-.829-.83-.119-.119h-.316l-.869.87a1.09 1.09 0 00-.119.632l.08.276.869.869.375.02.217-.02a1.342 1.342 0 00.711-1.698zM574.709 189.132l.297.296.316.079.493.494.376.059 1.323-1.324.04-.355a1.306 1.306 0 00-.79-.79l-.712-.711-.454-.139-.276-.039-.277-.079h-.277l-.454.138-.573.573-.138.494v.276a1.391 1.391 0 00.928.929l.178.099zM565.563 201.675l.079-.356-.118-.118a1.396 1.396 0 00-.79-.79 1.375 1.375 0 00-1.146-.04l-.79.79-.04.356.099.099a1.323 1.323 0 00.79.79 1.374 1.374 0 001.146.039l.77-.77zM617.001 203.69l.138.138.257.02a1.37 1.37 0 00.889-.652 1.351 1.351 0 00.119-1.067l-.948-.948-.297-.02-.099.099-.414.099-.593.593-.099.493v.277l.099.099a1.308 1.308 0 00.948.869zM537.138 141.604l.395.395a1.252 1.252 0 001.264-.276c.176-.178.293-.406.336-.652l.138-.138.079-.395-.257-.257a1.722 1.722 0 00-.869-1.067l-.158-.158h-.434a1.287 1.287 0 00-.791.435l-.632.632.02.257.573.572.336.652zM514.164 150.592l.593-.592v-.316l-.099-.099a1.392 1.392 0 00-1.304-1.027l-.276.079-.632.632a1.531 1.531 0 00-.119 1.066 1.37 1.37 0 00.81.81 1.47 1.47 0 001.244-.138l-.217-.415zM563.054 163.926v.276l.119.514.612.612.474.119.296-.02a3.636 3.636 0 011.126.059l.178.336.415-.138.553-.553.138-.454v-.277a1.389 1.389 0 00-.849-.849l-.178-.178-.415-.138-.118-.119-.474-.276-.297-.06-.335.06-.119-.119a1.344 1.344 0 00-1.126 1.205zM567.875 161.951l.138.138.336.099.099.098a1.556 1.556 0 001.501-.869l.059-.336-.099-.572-1.145-1.146-.415-.059h-.237l-.316.079-.435.434-.02.218-.059.296.099.573.751.75-.257.297zM556.635 164.775l.494-.494.039-.434-.039-.237a1.32 1.32 0 00-.791-.79 1.3 1.3 0 00-1.086.099l-.79.79.02.375a1.311 1.311 0 001.422.948l.138-.138.593-.119zM537.395 171.926l-.672-.671a1.06 1.06 0 00-1.027.039 1.29 1.29 0 00-.612.849l-.139.139-.059.415c.133.31.381.558.692.691a1.182 1.182 0 00.967.099 1.394 1.394 0 00.85-1.561zM598.631 214.041l.276-.277c.155-.306.183-.661.079-.987a1.352 1.352 0 00-.77-.771 1.31 1.31 0 00-1.778.553l-.039.395.237.672.474.474.415.02.138-.138.968.059z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M553.336 187.156l.039-.395-.237-.671-.375-.376a.982.982 0 00-.435-.158l-.395-.039-.671.237-.277.276a1.238 1.238 0 00-.079.988 1.294 1.294 0 00.711.711 1.4 1.4 0 001.719-.573zM609.811 209.774l-.375-.059h-.198l-.118.118a1.288 1.288 0 00-1.067.711c-.063.118-.11.245-.138.375l.099.099.019.297 1.047 1.047.356.039.099-.099.237-.039.513-.514.139-.138a1.368 1.368 0 00-.435-1.817l-.178-.02zM582.946 208.431l-.296-.02-.553.119-.217.217a1.132 1.132 0 00-.316.316 1.197 1.197 0 00-.04 1.067 1.506 1.506 0 00.889.889l.099.098.296.02.514-.514c.102-.08.195-.173.276-.276a1.368 1.368 0 00-.592-1.778l-.099.099.039-.237zM606.809 220.559l-.237.04-.415.217-.494.494a1.25 1.25 0 000 1.185 1.42 1.42 0 001.146.711l.316-.079.671-.671a1.537 1.537 0 00-.158-1.581l-.177-.177-.396-.198-.158-.158-.098.217zM571.312 203.473l.178-.178a1.352 1.352 0 00.77-.771 1.76 1.76 0 00.04-1.343l-.79-.79-.336-.02-.099.099a1.415 1.415 0 00-.79.79 1.376 1.376 0 00-.83.474l-.158.158-.118.395-.02.218.731.731a1.71 1.71 0 001.501.158l-.079.079zM563.391 190.534l.415-.415c.058-.073.124-.139.197-.197a1.425 1.425 0 00-.197-1.541 1.54 1.54 0 00-1.679-.376l-.613.613a1.31 1.31 0 00.198 1.541 1.481 1.481 0 001.679.375zM614.927 241.992l-.138.138-.059.376a1.223 1.223 0 00.355 1.066l.257.257c.298.024.598-.031.869-.158l.77-.77v-.553a1.639 1.639 0 00-1.264-1.027 1.266 1.266 0 00-.79.671zM587.805 189.428l.356-.04.751-.75a1.456 1.456 0 00-.277-1.66l-.158-.158-.375-.138-.257-.02-.356.04-.158.158a1.305 1.305 0 00-.434.434l-.138.139-.119.395v.237l.059.336.158.158a1.373 1.373 0 00.771.77l.177.099zM600.942 196.124a1.236 1.236 0 00.039-1.106 1.365 1.365 0 00-.869-.869l-.336.02-.987.987-.04.395 1.008 1.008.395.039.237-.039.691-.692-.138.257zM577.119 208.372l.138-.139.079-.355.119-.119-.079-.276a1.462 1.462 0 00-.435-.712l-.138-.138-.356-.118-.217-.02-.257.02-.355.355a1.37 1.37 0 00-.534.534 2.502 2.502 0 00-.098.296v.316l.572.573c.332.238.743.337 1.146.276.177-.128.319-.298.415-.493zM637.467 240.906l-.415.098-.158.158a1.148 1.148 0 00-.593.593 1.486 1.486 0 00-.138.494l-.099.099.119.118a1.31 1.31 0 001.264.909l.217.02.237-.04.514-.514c.088-.069.168-.148.237-.237a1.427 1.427 0 00-.277-1.659l-.158-.158-.533-.217-.217.336zM594.759 190.771l.494.494.237.039h.316l.632-.632.158-.158a1.422 1.422 0 00-.435-1.817l-.256-.257-.218.02a1.317 1.317 0 00-.75.751 1.445 1.445 0 00-.178 1.56zM622.572 204.875c.129.256.337.463.592.593l.119.118.375-.02.83-.829a1.51 1.51 0 00-.158-1.541l-.474-.474-.257-.059-.395.039-.119.119a1.445 1.445 0 00-.671.395l-.178.177-.099.455v.237l.079.316.277.276.079.198zM592.112 190.692a1.94 1.94 0 00.138-.375l.02-.218-.217-.217a1.345 1.345 0 00-.949-.948 1.482 1.482 0 00-1.422.395l-.395.395.02.257a1.481 1.481 0 001.304 1.027 1.311 1.311 0 001.501-.316zM629.071 197.428l-.336-.059-.336.059-.494.257-.276.276a1.452 1.452 0 00-.079 1.146 1.475 1.475 0 00.81.81l.098.099.297.019.553-.553c.096-.074.182-.16.257-.256a1.29 1.29 0 00-.435-1.66l-.158-.158.099.02zM610.582 203.473l-1.067-1.067-.296-.02a1.445 1.445 0 00-.592.27 1.441 1.441 0 00-.416.501 1.423 1.423 0 00.593 1.856l.158.158.435.04.237-.04.316-.316c.21-.141.392-.322.533-.533l.138-.138v-.395l-.039-.316zM592.625 199.897l.356.356c.083.227.211.435.375.612-.06.249-.046.51.04.751l-.297-.02-.256.059-.395.395c-.12.09-.227.196-.317.316a1.262 1.262 0 00.336 1.561l.139.138.355.119.119.118.256-.02.257-.059.81-.81.04-.355-.119-.119h.277l.098-.099a1.31 1.31 0 001.146-.75l.099-.099.474-.474.02-.217-.119-.356-.138-.375a1.66 1.66 0 00-.395-.593l-.454-.454a1.149 1.149 0 00-.455-.81l-.355-.356-.455-.019-.968.967a2.808 2.808 0 00-.039.593 1.27 1.27 0 00-.435 0zM597.92 154.859l.039-.395-.118-.395-.139-.138a1.067 1.067 0 00-.711-.712 1.316 1.316 0 00-1.027.198l-.415.415a1.256 1.256 0 00.573 1.718c.268.138.575.18.869.119a1.334 1.334 0 00.929-.81zM594.64 185.477l-.039.356a1.31 1.31 0 00.869 1.422l.217.02.257-.02.119-.119c.366-.078.689-.289.908-.592l.237-.237-.019-.218a1.623 1.623 0 00-1.363-1.086 1.365 1.365 0 00-1.324.731l.138-.257zM603.055 190.692l.573-.573a1.39 1.39 0 00-.192-1.561 1.388 1.388 0 00-.44-.335 1.533 1.533 0 00-1.146-.079 1.394 1.394 0 00-.75.751l-.02.414.138.139a1.367 1.367 0 00.553.987 1.288 1.288 0 001.205.178l.079.079zM599.579 185.951l.277-.039.77-.771c.109-.221.163-.465.158-.711v-.197l-.277-.277a1.393 1.393 0 00-1.264-.593 1.284 1.284 0 00-1.187 1.087c-.03.191-.015.387.041.573l.119.355a1.404 1.404 0 00.849.85l.514-.277zM596.162 180.321l.592.593.336.059.296-.059.514-.237.197-.198c.049-.043.095-.089.139-.138a1.428 1.428 0 00-.474-1.896 1.165 1.165 0 00-.81-.415 1.295 1.295 0 00-1.047.375l-.553.553.039.316c.145.209.325.389.534.534l.237.513zM569.001 177.694a1.287 1.287 0 00-.04-1.303 1.307 1.307 0 00-1.146-.633l-.869.87-.039.276.019.217.277.277a1.281 1.281 0 001.185.514c.219-.022.429-.096.613-.218zM636.755 182.909l-.118.395v.198l.039.316.119.118c.117.207.287.377.494.494l.138.138.375.099.218.02.335-.059.119-.119c.19-.107.347-.264.454-.454l.158-.158.099-.375v-.238l-.04-.316-.118-.118a1.282 1.282 0 00-.454-.454l-.159-.158-.375-.099-.217-.02-.336.059-.118.119a1.482 1.482 0 00-.455.454l-.158.158zM625.021 193.576l-.118.119-.257.059a1.343 1.343 0 00-.434.434l-.178.178-.079.474v.277l.118.316.573.573a1.317 1.317 0 001.778-.593 2.83 2.83 0 00.198-.197l-.001-.198-.039-.316-.849-.849-.712-.277zM616.785 178.781l.098-.099-.098-.099.138-.138v-.395l-.237-.632-.277-.277a1.308 1.308 0 00-.926-.125 1.31 1.31 0 00-.575 2.258 1.26 1.26 0 001.877-.493zM611.826 186.544h-.198l-.671.671-.178.178a1.43 1.43 0 000 1.264 1.392 1.392 0 001.007.731l.119.119.355.039.218-.019 1.007-1.008.02-.375-1.363-1.363-.316-.237zM603.075 217.853a1.372 1.372 0 00-1.303.751 1.476 1.476 0 00.257 1.6 1.427 1.427 0 001.58.395l.869-.869a1.484 1.484 0 00-.849-1.363l-.099-.099-.455-.415zM654.356 183.996l.158.158.474.039.256-.02.139-.138a1.453 1.453 0 001.027-.751c.056-.049.109-.101.158-.158v-.276l-.395-.395a1.4 1.4 0 00-.85-.613h-.276l-.553.554a3.639 3.639 0 00-.435.434 1.344 1.344 0 00.297 1.166zM589.584 264.175a1.284 1.284 0 00.178 1.442l.217.218.375.098c.289.121.611.135.909.04.319-.138.573-.393.711-.711l.217-.217a1.31 1.31 0 00-1.007-1.284 1.452 1.452 0 00-1.6.414zM549.029 167.679a1.374 1.374 0 001.877-1.165l.02-.217-.237-.238a1.367 1.367 0 00-1.047-.849l-.297-.02-.158.158-.434.237c-.224.126-.408.31-.534.534l-.118.118.02.218.118.355.652.652.138.217zM652.598 198.949l-.83.83a1.425 1.425 0 00.573 1.837 1.259 1.259 0 001.718-.376l.099-.098.099-.297v-.237a1.34 1.34 0 00-1.264-1.264l-.395-.395zM613.959 197.487a1.258 1.258 0 00-.099-1.284l-.75-.75h-.395l-.869.869a1.337 1.337 0 00.296 1.679l.454.454.218-.02a1.36 1.36 0 001.145-.948zM603.648 216.648l.455-.454.118-.119a1.56 1.56 0 00.079-1.382l-.099-.297a1.645 1.645 0 00-.533-1.007 1.922 1.922 0 00-1.995-.336l-.85.849.02.376.277.276a1.405 1.405 0 00.316 1.067l.533.533a1.304 1.304 0 001.146.237l.276-.039.257.296zM552.151 221.251l.335-.02.968-.968a1.333 1.333 0 00-.039-1.067 1.296 1.296 0 00-.751-.75 1.197 1.197 0 00-1.264.513l-.138.139-.079.355-.099.099.494.494c.214.303.529.52.889.612l-.316.593zM571.114 213.231l.593.593a1.454 1.454 0 001.205.217c.178-.072.34-.179.475-.315.136-.135.243-.297.315-.475l.237-.237v-.316l-.257-.257a1.306 1.306 0 00-.79-.79h-.316l-.375.375c-.23.117-.416.304-.534.533a1.07 1.07 0 00-.178.178l-.098.099v.316l-.277.079zM589.564 201.379l.158-.158v-.474l-.869-.87-.474-.197-.277-.04-.336.06c-.18.018-.354.072-.513.158l-.218.217-.237.514-.059.296.059.336.218.217c.141.339.411.609.75.751l.138.138h.435l.435-.435c.312-.08.589-.261.79-.513zM580.24 196.717l-.218.217a1.325 1.325 0 00-.059 1.126 1.456 1.456 0 00.869.869l.119.119h.356l.375-.376.138.139.257.019.257-.059.928-.928.02-.415-.869-.869a1.336 1.336 0 00-1.166-.02 1.504 1.504 0 00-.928-.059l-.356.355.277-.118zM614.888 194.03l.474.474a1.478 1.478 0 001.936-.671l.139-.139.039-.395-1.086-1.086-.257-.059-.099-.099-.395.039-.158.159a1.166 1.166 0 00-.612.612 1.882 1.882 0 00-.158.592l.059.297.118.118v.158zM600.29 209.339a.507.507 0 00.158-.158 1.343 1.343 0 00-.296-1.639l-.336-.336-.276-.079-.119-.119-.395.04-.119.118a1.656 1.656 0 00-.731.731 1.19 1.19 0 00-.197.198v.276l.691.692a1.714 1.714 0 001.284.296l.494-.494-.158.474zM590.078 206.198l.197-.197.119-.553-.02-.218c.01-.179-.01-.36-.059-.533.052-.179.065-.368.039-.553l-.02-.217-.652-.652a1.648 1.648 0 00-1.442-.257l-.355.356a1.357 1.357 0 00-.929.573l-.375.375.119.118c.042.325.172.632.375.889l.672.672a1.538 1.538 0 002.331.197zM595.352 207.68l.099-.099v-.316l-.692-.691a1.375 1.375 0 00-1.597-.066 1.374 1.374 0 00-.398.402l-.138.138.118.119a1.46 1.46 0 001.225.987c.273.022.546-.04.783-.177a1.35 1.35 0 00.541-.593l.059.296zM604.339 206.357l-.217.019-.632.633a.808.808 0 00-.178.177 1.369 1.369 0 00.454 1.798l.119.118.355.079h.198l.257-.019.336-.336c.216-.135.398-.318.533-.534.062-.069.128-.135.197-.197l.001-.198-.02-.335-1.027-1.028-.376-.177zM551.419 200.687a1.366 1.366 0 00.04-1.145 1.628 1.628 0 00-.968-.968l-.138-.139h-.198l-.158.158a1.433 1.433 0 00-.79.79 1.48 1.48 0 00-.059 1.205l.494.494a1.527 1.527 0 001.264.237c.229-.156.407-.375.513-.632zM619.767 224.47l.513-.513a.536.536 0 01.139-.138 1.459 1.459 0 00.059-1.245 1.346 1.346 0 00-.83-.83l-.217-.019-.237.039a1.23 1.23 0 00-1.324.692 1.837 1.837 0 00-.138.493l.099.336.593.593c.106.191.263.348.454.454l.118.119.356.039.217-.02h.198zM626.147 222.91l-.77.771a.383.383 0 00-.119.118 1.753 1.753 0 00-.202.958c.024.333.142.651.341.919l.572.572h.316l.099-.098c.346-.15.621-.425.771-.771.129-.277.177-.585.138-.889l-.099-.098-.079-.277-.79-.79-.178-.415zM595.47 213.132a.56.56 0 00.178-.178 1.37 1.37 0 00.059-1.205 1.432 1.432 0 00-.889-.889h-.316l-.869.869a1.317 1.317 0 00-.04 1.186 1.454 1.454 0 00.87.869l.098.099h.316l.672-.672-.079-.079zM618.937 234.347l-.118-.118a1.172 1.172 0 00-.79-1.106l-.277-.079-.711.711a1.315 1.315 0 00.04 1.264c.122.261.331.471.592.593a1.287 1.287 0 001.798-.573l-.534-.692zM630.493 258.545l-.237.04-.256.257a1.217 1.217 0 00-.316.316 1.345 1.345 0 00.177 1.521c.203.24.474.414.777.499.303.085.624.076.922-.025l.474-.474a1.02 1.02 0 00.218-.217 1.486 1.486 0 00-.02-1.324 1.74 1.74 0 00-1.363-.77l-.376.177zM584.527 204.717l-.099-.099-.296.02-.771.77a1.392 1.392 0 00-.02 1.284 1.544 1.544 0 001.284.771l.257-.02.474-.198.573-.572.059-.494v-.277a1.642 1.642 0 00-1.461-1.185zM581.563 260.185l.04-.395-.079-.277a1.166 1.166 0 00-.198-.513l-.217-.217-.454-.099-.099-.099-.395-.039-.474.118-.474.474a1.309 1.309 0 00-.079.988c.138.319.392.573.711.711a1.31 1.31 0 001.718-.652zM608.191 246.2a1.537 1.537 0 00-.434-1.897l-.178-.178-.415-.138h-.237l-.276.04-1.008 1.007-.02.415.712.711.197-.198-.178.178.514.514.474.158.178-.178.434-.237.277-.277-.04.08zM609.258 215.779l-.414.415-.02.375 1.106 1.106.356.04.889-.889a1.385 1.385 0 00.079-1.106 1.622 1.622 0 00-.81-.81l-.119-.119-.375-.019-.415.414-.277.593zM628.716 238.871a1.263 1.263 0 00-.455-1.482l-.177-.177-.514-.04a1.228 1.228 0 00-1.205.692 1.434 1.434 0 00-.138.493l.099.099.059.296.711.712a1.346 1.346 0 001.699-.277l.158-.158-.237-.158zM607.401 162.603l1.146-1.146.019-.376-.79-.79a1.543 1.543 0 00-1.639.099l-.198.198-.237.474-.039.276.039.316.395.395c.147.231.343.426.573.573l.119.119.612-.138zM557.919 211.335l.217-.218.178-.533.019-.296-.079-.316-.612-.613a1.4 1.4 0 00-1.146-.237 1.243 1.243 0 00-.731.731l-.158.158v.198a1.39 1.39 0 00.909.908 1.418 1.418 0 001.403.218zM569.099 198.653l.434-.435a.587.587 0 00.317.079 1.453 1.453 0 001.6-.691l.118-.119.059-.336-.612-.612a1.813 1.813 0 00-1.323-.454 1.503 1.503 0 00-.731.296l-.099-.099a1.257 1.257 0 00-1.225.711 1.505 1.505 0 00-.118.316l.059.336 1.007 1.008h.514zM622.059 176.351l-.336.336-.04.355.771.771a1.372 1.372 0 002.034-.573l.257-.257v-.356l-1.067-1.066-.375-.02-.217.02-.711.711-.316.079zM654.494 221.666l-.158.158-.06.454.139.138.098.296.099.099c.114.209.285.381.494.494l.356.356.217.019h.237l.118-.118.336-.138.119-.119c.131-.091.245-.204.336-.336a1.386 1.386 0 00.126-1.05 1.383 1.383 0 00-.66-.826 1.363 1.363 0 00-1.797.573zM631.422 227.789a1.256 1.256 0 00-1.087.454l-.158.158-.099.376-.019.217.059.375.158.158a1.311 1.311 0 00.75.751l.099.099.376-.02.809-.81a1.28 1.28 0 00-.414-1.679l-.139-.138-.335.059zM621.011 250.209l-.276.277a1.254 1.254 0 00-.474.474l-.178.178-.059.454v.277l.77.77a1.32 1.32 0 001.477-.16 1.32 1.32 0 00.321-.413c.073-.057.14-.124.197-.197l-.098-.099-.04-.316-.751-.751-.889-.494zM605.229 193.615v.237l.138.139a1.314 1.314 0 001.225.592 1.29 1.29 0 001.106-.711 1.203 1.203 0 000-1.066l-.593-.593a1.396 1.396 0 00-1.047-.138c-.345.149-.62.425-.77.77l-.059.77zM575.815 178.148l.593.593a2.5 2.5 0 00.513.514l.317.039.572-.059.692-.692.099-.572.098-.099a1.568 1.568 0 00-.948-1.462l-.118-.118h-.356l-1.146 1.145-.039.356-.277.355zM638.533 205.981l-.356.04-.138.138a1.369 1.369 0 00-.454.454l-.138.139-.119.395-.02.217.079.356a1.115 1.115 0 00-.217.217l-.454.454-.04.277.02.217.118.435.158.158a1.35 1.35 0 001.166.454 1.312 1.312 0 00.968-.731 1.403 1.403 0 00.098-1.086l.277-.277a1.392 1.392 0 00-.257-1.639l-.158-.158-.415-.139-.118.079zM633.871 208.036a1.53 1.53 0 00-1.767.044 1.487 1.487 0 00-.406.469l-.158.158.059.257a1.672 1.672 0 001.028.83l.098.098.356.001 1.047-1.047.039-.356-.098-.099-.06-.257-.375-.375.237.277zM602.167 227.098a1.464 1.464 0 00.07-1.174 1.455 1.455 0 00-.821-.841 1.22 1.22 0 00-.988-.079 1.35 1.35 0 00-.731.731l-.019.415.217.652.81.809h.395l.671-.197.396-.316zM566.887 210.742l.336.059.316-.039.533-.257.534-.534.019-.414-.908-.909a1.254 1.254 0 00-1.679.573 1.288 1.288 0 00.138 1.323l.355.356.356-.158zM616.685 216.135l-.138.138.04.237a1.532 1.532 0 001.284 1.007 1.45 1.45 0 001.382-.79c.058-.134.104-.273.139-.415l-.02-.217-.237-.237a1.451 1.451 0 00-1.205-.533 1.398 1.398 0 00-1.245.81zM632.192 218.525h-.198l-.257.059-.691.691a1.5 1.5 0 00.711 2.015l.158.158.415.099h.237l.277-.079.889-.889.039-.276v-.237l-.118-.395-.159-.159a3 3 0 00-.711-.711l-.138-.138-.454-.138zM580.3 196.065l.296.02.652-.652c.088-.069.168-.149.237-.237a1.447 1.447 0 00-.277-1.659l-.572-.573-.297-.02-.237.04-.415.177c-.223.126-.407.31-.533.534a2.507 2.507 0 00-.099.296l-.098.099.118.118a1.675 1.675 0 001.027.83l.198 1.027zM615.757 213.073l.119.118.356.079h.197l.316-.039c.239-.101.459-.241.652-.415l.178-.178.099-.415.019-.217-.059-.296-.928-.929h-.395l-.237.04-.633.632a1.282 1.282 0 00.316 1.62zM579.194 201.399a1.337 1.337 0 00.73-.731l.04-.395-.119-.356-.138-.138a.998.998 0 00-.494-.494l-.059-.296-.652-.652a1.537 1.537 0 00-1.185-.198l-.731.731a1.156 1.156 0 00-.099.099 1.312 1.312 0 00.218 1.521l.197.197.277.079a1.138 1.138 0 00.671.87 1.43 1.43 0 001.344-.237zM636.399 194.761l.296-.019a1.178 1.178 0 001.005-.069c.199-.108.364-.269.477-.465l.119-.118.079-.356.019-.257v-.197l-.079-.277-.553-.553a1.453 1.453 0 00-.948-.158 1.282 1.282 0 00-1.521.652l-.119.119.119.118a1.593 1.593 0 001.106 1.58zM642.741 220.856l-.356.039-.138.139a1.436 1.436 0 00-.415.414l-.158.158-.119.396-.02.256.06.336.158.158a1.335 1.335 0 00.77.771l.099.098.355-.039.751-.751a1.4 1.4 0 00-.257-1.639l-.177-.178-.376-.138-.177-.02zM616.132 145.891a.968.968 0 00.178-.178l.099-.099-.02-.296-.573-.573a1.557 1.557 0 00-1.244-.257 1.477 1.477 0 00-.849.85.968.968 0 00-.178.178l-.099.098v.317l.691.691a1.574 1.574 0 001.225.237 1.387 1.387 0 00.77-.968zM561.001 209.359a1.23 1.23 0 001.126-.533l.138-.139.079-.395v-.237l-.079-.316-.138-.138a1.402 1.402 0 00-.751-.751l-.099-.098-.375.019-.85.85a1.311 1.311 0 00.376 1.679l.118.118.455-.059zM589.998 178.662l.178-.178.119-.513v-.277l-.099-.336-.573-.573a1.287 1.287 0 00-1.797.573.827.827 0 00-.198.198l-.099.099.04.316.849.849a1.4 1.4 0 001.58-.158zM602.403 237.251a1.368 1.368 0 00-.019-.494 1.259 1.259 0 00-2.371-.316 1.2 1.2 0 00.079 1.264l.613.613.276.039.494-.019.277-.079.651-.652v-.356zM587.312 226.229c.069-.115.116-.243.138-.375l-.02-.218-.257-.257a1.536 1.536 0 00-1.224-.553 1.452 1.452 0 00-1.146.711l-.138.139v.237a1.475 1.475 0 001.284 1.007 1.431 1.431 0 001.363-.691zM582.512 183.186l-.316-.316a1.396 1.396 0 00-.85-.613h-.276l-.534.534a1.173 1.173 0 00-.217.217 1.37 1.37 0 00.593 1.817 1.338 1.338 0 001.817-.513l.118-.119.06-.336-.395-.671zM555.39 230.733l.613-.613a.767.767 0 00.177-.178 1.369 1.369 0 00-.572-1.837 1.305 1.305 0 00-1.469.132 1.304 1.304 0 00-.329.402l-.138.138-.04.435.474.474c.138.249.343.455.593.592l.099.099.592.356zM607.184 181.862l-.118.119-.277.079-.612.612a1.5 1.5 0 00.158 1.541l.474.474.257.059.296-.02.138-.138.415-.217.178-.178a.659.659 0 00.158-.158c.165-.303.212-.657.131-.992a1.388 1.388 0 00-.566-.825l-.138-.139-.494-.217zM618.265 165.368l-.197-.672-.652-.651-.316-.04-.099.099-.296.099-.277.276a1.33 1.33 0 00-.066 1.026 1.327 1.327 0 00.679.772 1.398 1.398 0 001.876-.613l-.652-.296zM592.409 168.667l.138-.138.02-.376-.02-.217-1.086-1.086h-.316c-.233.07-.444.199-.613.375l-.474.474-.059.257.02.454.059.257.691.691a1.453 1.453 0 001.64-.691zM618.483 205.132l-.691-.691-.355-.04-.652.652a.699.699 0 01-.139.138 1.395 1.395 0 00-.079 1.185 1.345 1.345 0 00.83.83l.296.02.494-.494c.211-.141.393-.322.534-.533l.197-.198v-.197l-.02-.336-.158-.158-.257-.178zM582.334 170.129l.474-.475.198-.395.256-.256-1.086-1.087a1.75 1.75 0 00-.928.02l.059.336-.237-.237-.79.79-.04.276.059.534.198.197a2.099 2.099 0 001.541.711l.296-.414zM637.486 166.336l.296-.059.217-.218c.241-.134.439-.332.573-.573a1.253 1.253 0 00-.514-1.619l-.138-.139-.217-.019a1.255 1.255 0 00-1.462.474l-.138.138-.079.395.099.099.059.296c.073.254.209.485.395.672l.198.197.711.356zM606.137 177.576l.356-.04.79-.79a1.478 1.478 0 00-.652-1.956l-.138-.138-.415-.02-.217.02-1.186 1.185.02.376.119.118c.083.21.208.401.367.561.16.159.351.285.561.368l.395.316zM623.817 182.929l-1.008 1.007-.02.376.02.217.928.928.356.04h.198l.276-.079c.266-.147.485-.366.632-.632l-.296-.692-.237-.711-.277-.276-.572-.178zM591.658 243.928l-.138.138-.079.356.02.217.691.692a1.425 1.425 0 002.015-.317l.138-.138.079-.355-.02-.218-.019-.256-.554-.554a1.522 1.522 0 00-1.145-.237 1.344 1.344 0 00-.988.672zM573.208 273.044c.382-.163.686-.467.85-.849l.177-.178v-.237l-.296-.296a1.675 1.675 0 00-1.402-.573 1.257 1.257 0 00-1.008.691 1.369 1.369 0 00-.099.929l.632.632c.384.105.792.063 1.146-.119zM574.65 239.286l.079-.277.02-.296-.059-.296-.257-.455-.198-.197a1.426 1.426 0 00-1.659.197l-.454.455-.079.316.039.316.218.494.197.197a1.424 1.424 0 001.837-.178l-.138-.138.118.119.336-.257zM593.87 233.063l-.296.297a1.36 1.36 0 00.079 1.264c.136.215.318.397.533.533a1.366 1.366 0 001.857-.553l.02-.415-.178-.177a1.033 1.033 0 00-.198-.514l-.158-.158c-.1-.16-.235-.295-.395-.395h-.276l-.099.099-.889.019zM574.611 224.234l.256-.06c.197-.024.386-.091.554-.197l.533-.534-.217-.335v-.316l-1.047-1.047-.356-.04-.83.83a1.4 1.4 0 00.237 1.62l.198.197.454.138.218-.256zM590.295 232.491l.592-.593c.071-.143.117-.296.139-.454l.632-.632a1.336 1.336 0 00.02-1.166 1.23 1.23 0 00-.79-.79l-.099-.099-.336.02-1.541 1.541a1.263 1.263 0 00-.095.971c.049.16.131.309.239.437.108.129.24.234.389.31.23.102.483.137.731.099l.119.356zM616.349 228.165a1.224 1.224 0 00-1.165.691 1.169 1.169 0 00-.138.494l.019.217.257.257a1.318 1.318 0 001.166.375 1.45 1.45 0 001.066-.79 1.287 1.287 0 00-.415-1.64l-.177-.177-.613.573zM589.741 233.794l.139.139a1.31 1.31 0 001.698.68 1.38 1.38 0 00.534-.384l.158-.158.138-.415v-.237l-.059-.296-.849-.85-.297-.059h-.316l-.968.968-.178.612zM567.341 252.323c.07-.061.136-.127.197-.198l.099-.098-.039-.277a1.43 1.43 0 00-.475-1.304 1.255 1.255 0 00-1.6-.098l-.711.711-.059.375.257.257.118.316.731.731a1.285 1.285 0 001.482-.415zM567.954 235.434l.098.099.336.02.099-.099c.212-.039.412-.127.584-.257.171-.131.31-.3.404-.494a.948.948 0 00.118-.356l-.02-.335-.513-.514-.474-.237-.336-.02-.336.06-.77.77a1.25 1.25 0 00.632 1.778l.178-.415zM604.3 251.671l-.256.02-.573.573c-.049.043-.095.09-.138.138a1.387 1.387 0 00-.06 1.126 1.537 1.537 0 00.889.889l.099.099h.316l.83-.83a1.373 1.373 0 00-.553-1.817l-.139-.138-.415-.06zM580.872 239.108l.336.336a1.487 1.487 0 001.225.197 1.304 1.304 0 00.79-.79l.118-.118.099-.099.02-.336-.178-.178a1.335 1.335 0 00-.77-.77l-.296.02-.376.375c-.22.13-.403.313-.533.533a1.85 1.85 0 00-.198.198l-.098.099.019.296.237.237h-.395zM576.803 249.893c.1.192.243.358.418.484.174.127.376.212.589.247l.099.099.336-.059.711-.711a1.395 1.395 0 00-.237-1.62l-.534-.533-.296.019a1.427 1.427 0 00-.75.751 1.333 1.333 0 00-.336 1.323zM607.045 227.355l-.296.296a1.23 1.23 0 00-.039 1.146c.135.274.357.496.632.632l.414.415.376.098.118.119.336-.059.138-.139c.168-.106.309-.247.415-.415l.158-.158.099-.375.02-.217-.04-.356-.118-.118a1.212 1.212 0 00-.455-.455l-.276-.276a1.566 1.566 0 00-1.126-.257l-.138.138-.218-.019zM595.885 221.369l-.099-.098-.119.118a1.569 1.569 0 00-.711.711 1.425 1.425 0 00.119 1.422 1.476 1.476 0 001.422.593l.435-.434.177-.178a1.418 1.418 0 00.06-1.205 1.448 1.448 0 00-.889-.889l-.395-.04zM585.613 217.814a1.367 1.367 0 001.027.79l.237.039.277-.276a1.57 1.57 0 00.652-.652 1.29 1.29 0 00.138-.652l.099-.099-.613-.612a1.422 1.422 0 00-1.185-.277 1.38 1.38 0 00-.909.711c-.044.366.056.734.277 1.028zM565.899 218.841h.355l1.126-1.126.02-.375-1.146-1.146h-.434l-.237.039-.83.83a1.395 1.395 0 00.377 1.517c.137.124.298.219.473.281l.296-.02zM596.794 249.972a1.361 1.361 0 00-.511-.167 1.344 1.344 0 00-1.346.72l-.099.099-.059.257.099.099.079.276a1.37 1.37 0 001.797.731l.395-.395c.088-.069.168-.149.237-.237a1.392 1.392 0 00-.592-1.383zM589.287 214.732a1.537 1.537 0 001.264 1.067 1.428 1.428 0 001.344-.711l.138-.139-.099-.098a1.475 1.475 0 00-.81-.81 1.703 1.703 0 00-1.284-.02l-.79.79-.039.356.276-.435zM609.634 235.118l-.297.02-.612.612a.521.521 0 01-.139.138 1.381 1.381 0 00-.103 1.22 1.358 1.358 0 00.913.815l.297.02.494-.494c.222-.111.402-.291.513-.514l.198-.197v-.277l-.376-.375a1.392 1.392 0 00-.849-.612l-.039-.356zM590.65 218.268l-.099.375-.019.218.059.335.118.119c.119.181.274.336.455.454l.158.158.375.099.217.02.336-.06.119-.118c.181-.118.336-.273.454-.454l.138-.139.119-.395.02-.217-.079-.316a1.316 1.316 0 00-.494-.494l-.138-.138-.376-.099-.217-.02-.336.06-.099.098c-.19.108-.347.265-.454.455l-.257.059zM589.307 222.337a.865.865 0 00-.198.198 1.397 1.397 0 00-.039 1.185 1.424 1.424 0 00.889.889h.276l.395-.395c.175-.112.323-.26.435-.435a1.365 1.365 0 00-.217-1.56l-.158-.158-.376-.178-.138-.138h-.276l-.593.592zM593.535 218.861l-.099.375.099.099-.218.217v.237a1.485 1.485 0 001.245 1.047 1.344 1.344 0 001.343-.711l.119-.119.039-.355-.099-.099-.079-.316-.059-.336-.119-.118a1.562 1.562 0 00-.454-.455l-.138-.138-.395-.118h-.237l-.316.039-.139.138c-.176.11-.325.258-.434.435l-.059.178zM601.989 222.456l-.098.375-.02.217.059.336.119.119c.115.183.27.338.454.454l.158.158.375.099.218.02.335-.06.119-.118c.175-.112.323-.26.435-.435l.138-.138.118-.395.02-.217-.059-.336-.119-.119a1.064 1.064 0 00-.454-.454l-.158-.158-.395-.079-.218-.02-.335.059-.119.119c-.175.111-.323.26-.434.435l-.139.138zM580.892 219.927h.355l.79-.79.159-.355.118-.119.04-.316a1.548 1.548 0 00-.186-1.776 1.525 1.525 0 00-.506-.377l-.316-.04-.237.04-.454.454a1.43 1.43 0 00-.336.336 1.652 1.652 0 00-.119 1.146 1.374 1.374 0 00.33 1.55c.139.128.302.225.48.287l-.118-.04zM580.872 224.688l-.632.632a1.221 1.221 0 00.02 1.166 1.485 1.485 0 001.284.809 1.362 1.362 0 001.205-.73 1.293 1.293 0 00-.257-1.502 1.393 1.393 0 00-.494-.316l-.099-.099a1.488 1.488 0 00-1.027.04zM598.532 241.913l-.138.296-.059.178-.02.375.02.218.079.237.098.178.257.296.158.118.158.079.178.06.375.019.218-.019.237-.079.178-.099.296-.257.118-.158.139-.296.059-.178.039-.395-.039-.198-.079-.237-.119-.197-.237-.277-.158-.118-.158-.079-.178-.06-.375-.019-.217.019-.237.079-.178.099-.296.257-.119.158zM561.968 236.283l.099-.099.276-.079c.164-.11.305-.251.415-.414l.158-.158.099-.415.118-.119-.079-.355-.158-.158a1.331 1.331 0 00-.77-.771l-.336.02-.928.928-.04.356.02.217.948.948.178.099zM562.818 221.448l.138-.138.039-.356-.809-.809a1.39 1.39 0 00-1.11-.246 1.397 1.397 0 00-.925.66l-.119.119-.079.316.573.573.158.158a1.43 1.43 0 001.265.434 1.364 1.364 0 00.869-.711zM540.417 231.444a1.293 1.293 0 00.02-1.166 1.266 1.266 0 00-.83-.829l-.119-.119-.335.059-.415.415a1.01 1.01 0 00-.455.454 1.24 1.24 0 00-.059 1.008l.514.513c.158.128.343.218.541.263a1.341 1.341 0 001.513-.736l-.375.138zM589.367 248.59l.098-.099.04-.316-.968-.968-.395-.079-.217-.02-.85.85-.059.256-.099.099.277.277a1.5 1.5 0 001.264.553c.357-.075.678-.27.909-.553zM558.827 241.755a.842.842 0 00.099-.296l.099-.099-.02-.336-.968-.968-.336-.02h-.197l-.277.079a1.46 1.46 0 00-.454.455l-.178.177-.079.514.02.257.099.296.573.573a1.34 1.34 0 001.619-.632zM591.322 226.209l.039.237 1.501 1.501.395.04.83-.83a1.282 1.282 0 00.04-1.106l-.178-.178v-.316l-.06-.257-.335-.336.039.04-.375-.375-.415-.218-.257-.059h-.316l-.257.059-.592.593a1.475 1.475 0 000 1.304l-.059-.099zM575.5 229.152l.75-.75a1.429 1.429 0 00.079-1.107 1.373 1.373 0 00-.81-.81l-.118-.118-.336-.02-.849.85a1.25 1.25 0 00-.06 1.126 1.366 1.366 0 00.81.809l.099.099.435-.079zM612.36 265.696l.336.336a1.278 1.278 0 001.066.158 1.253 1.253 0 00.771-.77c.073-.058.14-.124.197-.198l.099-.099-.02-.335-1.165-1.166-.277-.039h-.316l-.829.829a1.254 1.254 0 00.138 1.284zM596.221 165.842l.474.474a1.34 1.34 0 001.145.632 1.06 1.06 0 00.672-.237l.731-.731a1.556 1.556 0 00.099-1.363l-.474-.474a1.427 1.427 0 00-.869-.237l-.218-.02-.928.929a1.474 1.474 0 00-.198.197l-.098.099-.336.731zM573.247 156.775l.356-.039.711-.711c.132-.269.181-.571.139-.867a1.49 1.49 0 00-.376-.793l-.514-.513h-.276c-.34.141-.609.411-.751.75a1.255 1.255 0 00-.02 1.126 1.4 1.4 0 00.988.751l-.257.296zM599.934 199.897l-.375.375-.237.435-.059.296v.316l.731.731c.301.222.568.489.79.79l.178.178.395.198.237.039h.316l.691-.691a1.373 1.373 0 00-.02-1.205l-.039-.237-.415-.415-.158-.514-.612-.612a1.317 1.317 0 00-1.423.316zM604.833 201.675l.475.474c.208.226.48.383.78.451.3.067.614.04.899-.076l.632-.632.059-.257.296-.296.119-.434.118-.119-.059-.296-.988-.988h-.434l-1.146 1.146-.079.276-.099.099-.059.257-.514.395zM594.759 196.183l.218-.019.81-.81a1.321 1.321 0 00-.01-1.112 1.313 1.313 0 00-.86-.705h-.197l-.356.039-.79.79a1.314 1.314 0 00.395 1.66l.158.157h.632zM575.717 188.203l.039-.395-.02-.257-.039-.237a1.066 1.066 0 00-.731-.731 1.213 1.213 0 00-.536-.019 1.214 1.214 0 00-.491.217l-.415.415a1.258 1.258 0 00-.059.968c.133.311.38.558.691.691a1.37 1.37 0 001.561-.652zM621.624 239.128l-.297.296a1.236 1.236 0 00-.177.968 1.386 1.386 0 00.632.869 1.302 1.302 0 00.989.216 1.31 1.31 0 00.848-.552l.079-.395-.158-.671-.652-.652-.395-.079-.158.158-.711-.158zM663.344 271.208l-.06.414.139.692.276.276c.125.096.265.169.415.218l.395.079.672-.158.276-.277a1.45 1.45 0 00.198-.988 1.252 1.252 0 00-.593-.829 1.42 1.42 0 00-1.718.573zM610.107 241.992l.356.119.217-.02a1.373 1.373 0 001.146-.593 1.39 1.39 0 00.257-.257l.098-.098v-.316l-.948-.949-.355-.079h-.198l-.237.04-.494.494-.138.138a1.393 1.393 0 00.198 1.857l.098-.336zM636.518 246.516l.317.079.553-.04.217-.217c.115-.095.221-.201.316-.316a1.265 1.265 0 00.178-1.047 1.507 1.507 0 00-.909-.909l-.099-.099-.296-.059-.514.514a1.847 1.847 0 00-.276.276 1.312 1.312 0 00.395 1.818l.099-.099.098.098-.079.001z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M614.236 231.7l.257-.019.434-.158.218-.218c.104-.079.197-.172.276-.276a1.146 1.146 0 00.138-1.166 1.42 1.42 0 00-1.047-.849l-.335.02-.652.652a1.513 1.513 0 00-.02 1.6l.652.651.079-.237zM647.501 252.797l-.177.178a1.315 1.315 0 00-.909.593c-.239.397-.316.87-.217 1.323l.79.79.336.099h.197a1.367 1.367 0 00.929-.573c.333-.005.652-.139.889-.375l.158-.158.138-.375.039-.238-.019-.256-.553-.553a1.72 1.72 0 00-1.442-.336l-.159-.119zM653.862 266.585l-.415.415a.865.865 0 00-.198.198 1.317 1.317 0 00.04 1.54c.176.249.423.437.71.539.286.103.597.115.89.034l.415-.415a.92.92 0 00.197-.197 1.397 1.397 0 00-.019-1.561 1.54 1.54 0 00-1.62-.553zM608.705 209.458l.139-.138.098-.376a1.305 1.305 0 00-.237-1.106l-.533-.533a1.676 1.676 0 00-.869.039l-.79.791-.06.335v.198a1.706 1.706 0 001.166 1.165 1.337 1.337 0 001.086-.375zM629.407 264.926l-.355-.04-.731.731a1.448 1.448 0 00.059 1.679l.178.178.375.178.138.138h.356l.276-.039c.168-.105.31-.247.415-.415l.158-.158.139-.375.039-.237-.039-.356-.02-.217a1.343 1.343 0 00-.771-.771l-.217-.296zM617.219 256.689a1.236 1.236 0 00-.158 1.106 1.404 1.404 0 00.85.849l.118.119.336.02.869-.87.079-.355-.02-.217-.039-.237-.85-.85-.414-.059-.218.02-.257.059-.375.375.079.04zM642.365 247.345l-.138.139-.119.355-.118.119.019.296c.054.276.176.534.356.751l.514.513.256.02.119-.119.355-.118c.22-.13.404-.313.534-.533.081-.064.154-.137.217-.218l.099-.099.02-.296-.652-.652a1.485 1.485 0 00-1.106-.395c-.13.061-.25.141-.356.237zM586.166 207.897l.455-.019.138-.139c.258-.125.467-.334.593-.592.096-.144.163-.305.197-.474l.118-.119.001-.197a1.318 1.318 0 00-1.107-1.107l-.118-.118-.257.02-.494.494a1.363 1.363 0 00-.237.237 1.423 1.423 0 00.059 1.679l.158.158.514.276-.02-.099zM622.809 262.634l-.652-.651-.296-.02-.257.02-.395.158-.178.177a.808.808 0 00-.177.178 1.396 1.396 0 00.217 1.837l.257.257h.237a1.51 1.51 0 00.889-.573 1.429 1.429 0 00.355-1.383zM596.814 245.488a1.482 1.482 0 00-.593-.592l-.118-.119-.376-.059-.81.81a1.483 1.483 0 00-.039 1.541l.494.494.256.059.396.039.098-.098c.273-.033.529-.151.731-.336l.178-.178.158-.435.039-.237-.059-.296-.178-.178-.177-.415zM625.357 263.089c-.1.07-.186.157-.257.257l-.138.138.02.217a1.365 1.365 0 00.948.948 1.528 1.528 0 001.481-.217l.593-.593.02-.256a1.476 1.476 0 00-1.166-1.166 1.284 1.284 0 00-1.501.672zM589.426 251.968l.316.079.336-.02.533-.178.296-.296a1.404 1.404 0 00.174-1.116 1.391 1.391 0 00-.707-.879l-.119-.119-.296-.059-.217.02-.415.414a.809.809 0 00-.257.257 1.26 1.26 0 00.217 1.679l.257.257-.118-.039zM608.528 248.234l.039.237.909.909.296.059.099.099c.217-.025.426-.096.613-.209.188-.112.349-.263.473-.443a1.427 1.427 0 00-.415-1.916l-.158-.158-.434-.079h-.237l-.277.079-.138.138a1.475 1.475 0 00-.533.534l-.139.138-.059.415-.039.197zM625.93 253.903a1.755 1.755 0 00-.375-.375 4.078 4.078 0 00-.297-.652c.085-.235.091-.491.02-.731l.277.04.098.099.237-.04.395-.395c.132-.09.246-.204.336-.336a1.286 1.286 0 00-.158-1.58l-.513-.514-.257-.019-.099-.099-.257.02-.829.829-.06.375-.098.099-.237-.039h-.198a1.28 1.28 0 00-.697.121c-.217.104-.4.268-.528.471-.088.07-.168.149-.237.237l-.454.455.197.75.139.139c.063.229.171.443.316.632l.454.454a1.148 1.148 0 00.375.849l.356.356.435.079.217-.02.83-.829c.062-.193.101-.391.118-.593.147.108.315.182.494.217zM615.401 298.013l-.079.395.099.376v.237a1.206 1.206 0 00.731.731 1.43 1.43 0 001.047-.099l.395-.395a1.285 1.285 0 00-.375-1.759 1.488 1.488 0 00-.87-.237 1.322 1.322 0 00-.948.751zM622.256 267.968l.059-.375a1.316 1.316 0 00-.691-1.521l-.119-.119-.217-.02-.119.119a1.537 1.537 0 00-.968.494l-.158.158-.019.217a1.564 1.564 0 001.205 1.205 1.31 1.31 0 001.402-.573l-.375.415zM614.454 261.824l-.553.554a1.415 1.415 0 00-.195 1.078 1.437 1.437 0 00.629.897c.332.218.737.296 1.126.217a1.42 1.42 0 00.889-.612l.059-.415-.118-.118a1.421 1.421 0 00-.435-1.067 1.368 1.368 0 00-1.185-.316l-.217-.218zM617.377 266.921h-.276l-.751.751a1.223 1.223 0 00-.237.671l-.119.119.06.296a1.423 1.423 0 001.185.751 1.29 1.29 0 001.146-.593 1.13 1.13 0 00.192-.444c.03-.161.025-.326-.015-.484a1.366 1.366 0 00-.079-.395 1.314 1.314 0 00-.849-.85l-.257.178zM620.043 272.946l-.592-.593-.316-.079h-.316l-.514.198-.217.217-.138.138a1.395 1.395 0 00.276 1.936c.181.255.447.438.751.514a1.261 1.261 0 001.106-.277l.652-.652-.02-.296-.099-.099a3.445 3.445 0 00-.355-.632l-.218-.375zM646.79 278.714a1.284 1.284 0 00-.119 1.304 1.314 1.314 0 001.067.711l1.027-1.027.079-.277.099-.099-.099-.098-.059-.257-.099-.099a1.283 1.283 0 00-2.232-.02l.336-.138zM580.102 265.598l.178-.376.02-.217v-.316l-.119-.119a1.14 1.14 0 00-.316-.592l-.158-.158-.375-.139-.119-.118-.336.02-.118.118a1.072 1.072 0 00-.553.316l-.158.158-.138.376-.02.217v.316l.118.119c.058.208.166.398.316.553l.158.158.376.138.118.118.336-.019.217-.02a2.16 2.16 0 00.554-.316l.019-.217zM593.08 256.254l.099.099.277-.04c.2-.051.384-.153.533-.296l.198-.198.138-.454v-.277l-.079-.316-.553-.553a1.344 1.344 0 00-1.857.356c-.08.064-.153.137-.217.217v.316l.948.948.513.198zM599.46 272.037l-.237.237-.059.376.178.651.276.277a1.37 1.37 0 001.897-.356 1.313 1.313 0 00-.139-1.6 1.29 1.29 0 00-1.054-.236 1.288 1.288 0 00-.862.651zM605.288 264.906h.198l.236-.039.494-.494a.886.886 0 00.198-.198 1.376 1.376 0 00.158-1.224 1.335 1.335 0 00-.889-.889l-.237-.237-.356-.079-.118.118-.237.04-.85.849-.059.375v.198l.059.257 1.008 1.007.395.316zM617.752 234.94a1.365 1.365 0 001.363-.612 1.456 1.456 0 00-.04-1.62 1.423 1.423 0 00-1.521-.573l-.908.909a1.372 1.372 0 00.671 1.461l.435.435zM562.758 262.437l-.178-.178-.454-.099h-.277l-.118.119a1.283 1.283 0 00-1.087.612.808.808 0 00-.177.178l-.04.277.356.355c.141.339.411.609.75.751l.099.099.257.019.533-.533a2.42 2.42 0 00.435-.435 1.376 1.376 0 00-.099-1.165zM636.459 190.396a1.255 1.255 0 00-.02-1.442l-.237-.237-.356-.119a1.259 1.259 0 00-1.718.415l-.356.355a1.285 1.285 0 00.85 1.363 1.452 1.452 0 001.837-.335zM665.437 290.961a1.4 1.4 0 00-1.837.455 1.594 1.594 0 00-.198.474l-.039.237.099.099.019.217a1.343 1.343 0 00.949.948l.276.039.296-.019.435-.198-.217-.217.237.237c.203-.136.377-.311.513-.514l.218-.217-.02-.217-.059-.376-.632-.632-.04-.316zM566.255 247.839l.217-.02.593-.592a1.424 1.424 0 00-.336-1.877 1.287 1.287 0 00-1.758.139l-.119.118-.118.316-.04.237a1.34 1.34 0 001.107 1.422l.454.257zM604.399 253.706a1.34 1.34 0 00-.08 1.304l.83.829.395.04h.198l.276-.079.573-.573a1.339 1.339 0 00-.099-1.679l-.474-.474h-.237a1.338 1.338 0 00-1.382.632zM616.942 235.947l-.375.178-.178.178-.118.118a1.68 1.68 0 00-.237 1.383l.059.296c.006.388.154.761.415 1.047a1.864 1.864 0 001.955.573l.85-.849.02-.375.098-.099-.138-.138a1.534 1.534 0 00-.217-1.126l-.514-.514a1.39 1.39 0 00-1.106-.356h-.277l-.237-.316zM668.598 237.389h-.316l-.889.889a1.232 1.232 0 00-.099 1.047c.057.176.155.335.286.465.13.131.289.229.465.286l.098.099a1.378 1.378 0 001.344-.356l.138-.138.138-.336.099-.099v-.276l-.336-.336a1.478 1.478 0 00-.731-.731l-.197-.514zM648.845 243.158l-.573-.573a1.538 1.538 0 00-1.205-.376 1.38 1.38 0 00-.929.613 1.797 1.797 0 00-.138.138l-.118.119-.04.316.099.098.039.237c.066.182.171.347.307.484.137.136.302.241.483.307l.099.098h.277l.276-.276c.221-.129.405-.313.534-.534l.296-.296v-.316l.593-.039zM629.19 252.856l-.158.158-.04.475.059.256 1.047 1.047.277.04.336.02c.184-.015.365-.062.533-.139l.593-.592.079-.316-.02-.336-.099-.296a1.674 1.674 0 00-.652-.85l-.138-.138-.415-.059-.237.039-.277.277a1.975 1.975 0 00-.888.414zM637.841 258.467l.297-.297a1.334 1.334 0 00.177-1.126 1.54 1.54 0 00-.869-.869l-.118-.118-.336-.06-.119.119-.237.039-.098.099-.158-.158-.237-.039-.297.019-.928.929-.059.415.019.217.079.277.395.395a1.343 1.343 0 001.146.158c.285.128.597.183.909.158l.296-.297.138.139zM603.115 257.084l-.454-.454a1.506 1.506 0 00-2.015.434l-.138.138-.099.376v.237l.059.257 1.067 1.066h.395l.158-.158a1.392 1.392 0 00.988-.988l.098-.098-.019-.297-.099-.098.059-.415zM619.432 243.612a.752.752 0 00-.178.178 1.37 1.37 0 00.119 1.659l.671.671.395.04.119-.119a1.73 1.73 0 00.75-.75l.297-.296.019-.297-.454-.454a1.639 1.639 0 00-1.244-.454l-.396.395-.098-.573zM629.189 247.918l-.217.217-.138.534v.237a1.477 1.477 0 00-.04.513 2.538 2.538 0 00-.099.573v.198l.652.652a1.655 1.655 0 001.403.414l.336-.335a1.308 1.308 0 00.987-.474l.395-.396-.118-.118a1.646 1.646 0 00-.277-.948l-.671-.672a1.52 1.52 0 00-1.018-.692 1.516 1.516 0 00-1.195.297zM624.113 245.844l-.099.099-.02.296.04.237.454.455a1.408 1.408 0 001.024.39 1.406 1.406 0 00.991-.47l.237-.237.02-.217a1.43 1.43 0 00-1.126-1.126 1.341 1.341 0 00-1.403.613.713.713 0 00-.118-.04zM615.085 246.14l.099-.098.237-.04.474-.474a.752.752 0 00.178-.178 1.372 1.372 0 00-.217-1.837l-.139-.138-.335-.139-.099-.098-.257.019-.316.317a1.716 1.716 0 00-.533.533l-.316.316-.02.336 1.007 1.007.237.474zM666.939 257.933a1.224 1.224 0 00-.158 1.106 1.561 1.561 0 00.948.948l.257.06.118.118.257-.02c.374-.082.701-.309.909-.632a1.51 1.51 0 00.197-1.185l-.494-.494a1.73 1.73 0 00-1.244-.375 1.421 1.421 0 00-.79.474zM601.85 226.308l-.513.513a.469.469 0 00-.138.139 1.485 1.485 0 00-.251.586c-.041.212-.037.429.014.638.155.376.454.674.829.83l.237.039.217.02a1.287 1.287 0 001.403-.533 1.42 1.42 0 00.197-.474l.001-.198-.04-.355-.474-.474a1.316 1.316 0 00-.474-.475l-.119-.118-.355-.079-.099.099-.435-.158zM595.371 227.177l.099-.099.237-.04.494-.493c.048-.044.094-.09.138-.139a1.764 1.764 0 00.099-1.876l-.711-.711-.297-.02-.098.099a1.313 1.313 0 00-.889.612 1.53 1.53 0 00-.257.849v.198l.039.237.869.869.277.514zM624.627 240.392a1.76 1.76 0 00-.178.178 1.436 1.436 0 00.107 1.748c.14.157.312.282.505.366l.119.118.296.02.79-.79.198-.198a1.37 1.37 0 00.178-1.165 1.456 1.456 0 00-.869-.869l-.119-.119-.296-.02-.731.731zM603.826 216.589l.119.118a1.115 1.115 0 00.632 1.186l.177.177.218.02.296-.059.415-.415a1.388 1.388 0 00.099-1.244 1.425 1.425 0 00-.593-.593 1.317 1.317 0 00-1.004-.217 1.307 1.307 0 00-.853.573l.494.454zM595.194 191.206l.237-.237a1.32 1.32 0 00.335-.336 1.368 1.368 0 000-1.541 1.535 1.535 0 00-1.639-.652l-.454.454a1.048 1.048 0 00-.218.218 1.503 1.503 0 00-.138 1.323 1.6 1.6 0 00.513.62 1.6 1.6 0 00.751.289l.613-.138zM634.523 250.052l.099.098.296.02.533-.533c.092-.066.172-.146.237-.237a1.363 1.363 0 00.158-1.265 1.586 1.586 0 00-1.165-.928h-.277l-.474.119-.592.592-.139.494v.277a1.621 1.621 0 001.324 1.363zM643.926 195.295l-.079.395.079.276c-.01.192.031.383.118.553l.218.218.415.138.118.118.395.079.316-.079.296-.059.277-.276c.175-.291.238-.635.178-.968a1.34 1.34 0 00-.553-.83 1.363 1.363 0 00-.972-.123c-.329.08-.616.279-.806.558zM615.875 206.08a1.539 1.539 0 00.198 1.936l.178.178.415.178.138.138.296-.02.138-.138.415-.217.534-.534.079-.395-.119-.118-.079-.277-.395-.395-.198.198.514-.514-.533-.533-.297-.02-.256.02-.474.197-.178.178a.305.305 0 01-.162.138.303.303 0 01-.214 0zM611.254 236.184l.434-.434.099-.375-.02-.218-.039-.237-.85-.849-.336-.059-.098.098-.237.04-.613.612a1.371 1.371 0 00.064 1.634c.123.157.276.286.45.381l.118.119.356.039.395-.395.277-.356zM594.621 210.94a1.312 1.312 0 00.257 1.521l.197.197.494.099a1.294 1.294 0 001.304-.553c.089-.147.156-.307.198-.474l-.02-.336-.692-.692a1.311 1.311 0 00-1.718.06l-.178.178h.158zM606.868 289.203l-.237.04-.87.869-.079.355-.098.099.039.237.593.593a1.479 1.479 0 001.639.099l.573-.573.079-.277-.02-.336-.079-.276-.177-.178a1.536 1.536 0 00-.593-.593l-.119-.118-.651.059zM661.704 246.575l-.218.217-.237.474-.059.297.04.355.454.454a1.384 1.384 0 001.126.376 1.243 1.243 0 00.889-.573l.256-.257-.098-.099a1.433 1.433 0 00-.771-1.047 1.485 1.485 0 00-1.382-.197zM649.141 257.874l-.218.02-.257.059-.118.119-.316-.119a1.455 1.455 0 00-1.679.533l-.119.119-.079.316-.098.099.019.217.119.118.099.297.138.138a1.804 1.804 0 001.264.632c.268-.005.528-.088.751-.237l.118.119a1.345 1.345 0 001.324-.573.997.997 0 00.217-.218l.099-.098.019-.336-1.007-1.007-.276-.198zM593.929 273.815l.356-.356.059-.335.099-.099-.731-.731a1.287 1.287 0 00-1.106-.316 1.45 1.45 0 00-.968.652.677.677 0 00-.158.158l-.118.118-.02.336v.198l.968.968.375.059.099-.099.257-.059.395-.395.493-.099zM567.025 225.024l.158-.158.099-.455.02-.256-.257-.257a1.545 1.545 0 00-.356-.632l-.118-.119-.336-.138-.119-.119-.256-.019-.099.098-.336.099-.138.138a1.375 1.375 0 00-.336.336 1.42 1.42 0 00.336 1.916 1.34 1.34 0 001.738-.434zM590.65 221.626c.201.038.408.027.604-.032.196-.058.375-.162.522-.304l.158-.158.139-.375.039-.237-.02-.375-.138-.139a1.282 1.282 0 00-.751-.75l-.118-.119-.375-.02-.791.791a1.288 1.288 0 00.218 1.718l.158.158.355-.158zM603.609 200.569l.277-.04c.224-.044.431-.154.592-.316l.178-.178.138-.454.02-.257-.059-.296-.494-.494a1.325 1.325 0 00-1.004-.217 1.324 1.324 0 00-.853.573l-.197.197-.02.336 1.185 1.185.237-.039zM612.596 258.585l.119-.118-.138-.139a1.403 1.403 0 00-1.166-.731 1.373 1.373 0 00-1.185.593 1.205 1.205 0 00-.119 1.067l.573.573c.308.207.679.297 1.047.256a1.42 1.42 0 00.889-.612.536.536 0 00.231-.45.53.53 0 00-.251-.439zM640.015 277.351l-.573-.573a1.86 1.86 0 00-.514-.513l-.296-.099h-.593l-.296.099-.474.474-.178.533c-.031.306.027.614.168.887.14.274.356.501.622.654l.119.118.375.06h.198l1.027-1.028.079-.355.336-.257zM581.03 242.466h.356l.237-.039c.215-.042.414-.145.573-.297l.158-.158.138-.375.139-.138-.04-.356c.081-.063.154-.136.217-.217l.455-.454.079-.277.039-.237-.079-.435-.158-.158a1.373 1.373 0 00-1.106-.553 1.282 1.282 0 00-1.067.593c-.203.309-.28.683-.217 1.047l-.336.336a1.428 1.428 0 00.02 1.639l-.138-.138.296.296.434-.079zM585.811 241.083c.332.271.758.399 1.185.356.41-.073.777-.299 1.027-.632l.257-.257-.04-.237a1.755 1.755 0 00-.928-.928l-.119-.119h-.355l-1.027 1.027-.079.356-.099.099.039.237.376.375-.237-.277zM619.629 225.755a1.452 1.452 0 00.454 1.995 1.329 1.329 0 00.998.198 1.318 1.318 0 00.839-.574l.079-.395-.138-.691-.83-.83-.375-.059-.691.138-.336.218zM652.756 246.12l-.317-.118-.335.02-.553.197-.554.553-.079.395.02.218.079.276.593.593a1.253 1.253 0 001.738-.356 1.23 1.23 0 000-1.343l-.375-.375-.217-.06zM603.925 234.94l.256-.257a1.428 1.428 0 00-1.114-1.139 1.422 1.422 0 00-1.473.606 1.307 1.307 0 00-.178.415l-.119.118.139.139a1.62 1.62 0 001.145.711 1.534 1.534 0 001.344-.593zM588.932 230.851l.256-.02.494-.494a.767.767 0 00.178-.177c.454-.692.296-1.442-.454-2.114l-.178-.178-.395-.118-.138-.139-.277.04-.889.889-.079.276-.02.218.079.395.158.158c.207.265.446.504.712.711l.553.553zM637.684 259.138l-.297-.059-.434.434a1.62 1.62 0 00-.139 1.956l.593.593.296.059h.237l.415-.138c.214-.138.396-.32.534-.534.073-.057.139-.124.197-.197l.099-.099a1.788 1.788 0 00-.929-.928l-.572-1.087zM604.498 238.081l-.139-.138-.336-.099-.217-.02-.296.02a1.72 1.72 0 00-.691.336l-.178.177-.158.395-.139.139.02.296.909.909.415.059h.237l.731-.731a1.339 1.339 0 00-.158-1.343zM639.422 253.963a1.275 1.275 0 00-.849.573l-.079.395.079.355v.237c.102.217.277.392.494.494l-.099.099.02.296.553.553c.333.235.738.347 1.145.316l.889-.889a1.424 1.424 0 00-.079-1.54l-.355-.356a1.374 1.374 0 00-.593-.948 1.45 1.45 0 00-1.126.415zM581.84 253.864l-.296-.02a.93.93 0 00-.356-.118 1.306 1.306 0 00-1.185.474l-.138.138-.099.336-.059.257-.119.118.06.257.533.533c.273.189.596.292.928.297a1.319 1.319 0 001.581-.474l.118-.119v-.237a1.54 1.54 0 00-.968-1.442zM578.581 227.217h.355l.139-.139c.203.032.411.025.612-.02l.158-.158.158-.395.04-.237-.04-.355-.039-.237a1.334 1.334 0 00-.771-.771l-.118-.118-.336-.02-.672.672a1.458 1.458 0 00.059 1.679l.159.158.395.197-.099-.256zM596.26 304.749l-.296.296-.02.297.454.454c.15.16.335.283.541.359.206.075.427.101.644.076a1.508 1.508 0 001.028-.672c.065-.053.124-.113.178-.178l.098-.098.04-.317-.573-.572a1.567 1.567 0 00-1.185-.396 1.454 1.454 0 00-.909.751zM658.445 248.175a1.198 1.198 0 00-1.166.375l-.158.158-.138.376-.02.217.04.355.118.119c.142.339.412.609.751.751l.118.118.376.02.79-.79a1.257 1.257 0 00-.198-1.699l-.118-.119-.395.119zM626.069 275.257l-.217.217-.158.474-.04.277.059.336.573.573a1.278 1.278 0 001.472.033c.146-.097.27-.223.365-.369.083-.061.157-.134.218-.218l-.001-.316-.612-.612a1.393 1.393 0 00-1.659-.395zM620.577 215.641a1.364 1.364 0 00.889 1.639c.264.061.541.039.792-.063a1.32 1.32 0 00.61-.509 1.228 1.228 0 00.06-1.245l-.791-.79-.513-.04-.257.06-.751.75-.039.198zM634.286 228.323a1.632 1.632 0 00-.178.375l-.02.217.139.138a1.611 1.611 0 001.145.712 1.505 1.505 0 001.225-.593l.237-.237a1.422 1.422 0 00-1.114-1.14 1.425 1.425 0 00-1.474.607l.04-.079zM633.93 271.583l.297.296c.141.339.411.609.75.751l.277.039.513-.513c.083-.062.157-.135.218-.218a1.415 1.415 0 00-2.292-1.659l-.118.119-.099.335-.099.099.553.751zM666.524 227.611l-.099-.098-.573.572a.881.881 0 00-.197.198 1.416 1.416 0 00.316 1.975 1.412 1.412 0 001.975-.316l.158-.158.079-.395v-.237l-.059-.257-.158-.158a1.37 1.37 0 00-.573-.573l-.118-.118-.751-.435zM609.337 270.101h.237l.277-.039.612-.613a1.53 1.53 0 00.04-1.54l-.652-.652-.296-.02-.257.02-.415.178-.336.335a.54.54 0 00-.158.158 1.372 1.372 0 00.217 1.838l.158.158.573.177zM596.517 285.292l.119.672.652.651.316.079h.197l.099.099.099-.099.277-.039.296-.296a1.367 1.367 0 00-1.445-2.061 1.4 1.4 0 00-.886.599l.276.395zM622.533 284.877l-.138.138-.08.395.119.119.059.257.929.928.336.059c.237-.045.461-.146.651-.296l.652-.652.04-.474-.059-.257-.751-.75a1.48 1.48 0 00-1.758.533zM600.863 245.666l.691.692.356.079h.197l.237-.04.494-.494.158-.158a1.39 1.39 0 00.198-1.155 1.384 1.384 0 00-.277-.532 1.384 1.384 0 00-.474-.367l-.316-.04-.375.376a1.774 1.774 0 00-.534.533l-.316.316v.316l.257.257-.296.217zM632.685 284.679l-.236.04-.889.889.217.217.059.296.672.672c.295.113.615.147.928.099v-.356l.217.218.475-.198.434-.435.099-.296-.02-.533-.079-.277a2.179 2.179 0 00-1.442-.889l-.435.553zM577.494 281.914l-.296.02-.297.098a1.264 1.264 0 00-.79.98 1.257 1.257 0 00.455 1.174l.276.276a1.258 1.258 0 001.501-.276l.139-.139.138-.375.118-.118-.019-.297a1.293 1.293 0 00-.316-.711l-.218-.217-.691-.415zM609.89 274.487l-.355-.04-.79.79a1.48 1.48 0 00.414 1.995l.139.139.415.098h.197l.277-.039 1.047-1.047v-.356l-.099-.098a1.652 1.652 0 00-.929-.929l-.316-.513zM593.021 267.178l.098.099.257-.06.83-.829.098-.376-.019-.217-.04-.237-.79-.79-.336-.059h-.197l-.297.019a1.481 1.481 0 00-.651.652l.197.711.158.711.296.297.396.079zM632.034 210.268l.138-.139.099-.375.118-.118-.019-.257-.139-.138-.158-.356-.138-.138a1.418 1.418 0 00-1.038-.379 1.42 1.42 0 00-.996.478l-.139.138-.118.355-.119.119.02.257.533.533a1.483 1.483 0 001.106.395c.315-.036.611-.167.85-.375zM653.802 183.561a1.449 1.449 0 00-1.007.652.812.812 0 00-.178.178l-.099.098-.039.237.257.257a1.674 1.674 0 001.323.731 1.31 1.31 0 001.067-.592c.179-.26.256-.576.217-.889l-.514-.514a1.422 1.422 0 00-1.027-.158zM648.45 216.806l-.198.198-.039.316.039.276.198.474.197.198a1.45 1.45 0 001.679-.02l.454-.454.099-.336.02-.296-.158-.514-.217-.217a1.394 1.394 0 00-1.818-.04l.139.139-.119-.119-.276.395zM628.636 220.698l.277-.277a1.363 1.363 0 00.079-1.264 1.395 1.395 0 00-.534-.533 1.365 1.365 0 00-1.896.316l-.079.395.059.296c.009.186.056.367.139.533l.217.218c.1.16.235.295.395.395l.296.059 1.047-.138zM646.75 231.938h-.276a1.559 1.559 0 00-.573.138l-.533.533-.079.277-.04.316.06.257.217.454.553.553.375.059h.198l.237-.039.553-.553a1.424 1.424 0 00-.059-1.64l-.198-.197-.434-.198-.001.04zM632.054 221.784l-.514.514a2.16 2.16 0 00-.177.454l-.415.415a1.307 1.307 0 00-.178 1.165 1.37 1.37 0 00.81.81l.099.099.336.02 1.343-1.344a1.257 1.257 0 00-.336-1.758 1.198 1.198 0 00-.731-.177.98.98 0 00-.237-.198zM605.564 223.108c.242.045.492.016.717-.083.226-.098.417-.262.548-.47a1.42 1.42 0 00.197-.474l.02-.218-.138-.138a1.289 1.289 0 00-1.107-.514 1.539 1.539 0 00-1.165.652 1.255 1.255 0 00.257 1.679l.178.178.493-.612zM632.765 220.56l-.039-.238a1.4 1.4 0 00-.929-.928 1.365 1.365 0 00-1.224.316l-.178.178-.158.395-.138.138.019.297 1.027 1.027.316.039.277-.039.474-.237.534-.534.019-.414zM657.24 204.816a1.35 1.35 0 00-.217.217v.316a1.385 1.385 0 00.296 1.324 1.313 1.313 0 001.6.296l.77-.771.079-.355v-.198l-.019-.217-.06-.296-.573-.573a1.31 1.31 0 00-1.876.257zM654.573 221.468l-.119-.118-.316-.04-.098-.099-.099.099a1.407 1.407 0 00-1.047.612c-.094.077-.18.163-.257.257l-.099.099v.316l.652.652.455.257.316.079.355-.04.593-.592.178-.178a1.4 1.4 0 00.177-1.047 1.256 1.256 0 00-.691-.257zM620.32 201.122l.119.118h.276l.553-.553a.408.408 0 00.139-.138 1.371 1.371 0 00.197-1.106 1.516 1.516 0 00-.347-.542 1.521 1.521 0 00-.542-.347l-.099-.099-.335-.059-.237.04-.593.592a1.4 1.4 0 00.336 1.877l.138.138.395.079zM642.187 216.312l-.335-.335a1.424 1.424 0 00-1.719-.178 1.336 1.336 0 00-.415.415 1.797 1.797 0 01-.138.138l-.099.099-.059.335.099.099.059.257a1.35 1.35 0 00.77.77l.277.04.276-.277c.22-.129.404-.313.534-.533l.316-.316v-.316l-.138-.138.572-.06zM647.482 206.08a1.432 1.432 0 00-.889-.889h-.356l-.751.751a1.42 1.42 0 00.02 1.639l.553.554.297.019c.359-.067.679-.272.889-.573a1.31 1.31 0 00.237-1.501zM614.769 224.846l.277-.277a1.418 1.418 0 00.178-1.126 1.655 1.655 0 00-.514-.75l-.415-.415-.355-.158-.218-.02-.335.02-.218.02a1.45 1.45 0 00-.513.316l-.158.158-.139.375-.118.119v.355l.118.119c.065.205.173.393.316.553l.277.276c.3.261.689.396 1.086.376l.119-.119.612.178zM625.18 232.174l.118.119.099-.099c.327-.094.612-.296.81-.573a1.373 1.373 0 00.059-1.442 1.532 1.532 0 00-1.323-.77l-.514.513-.178.178a1.482 1.482 0 00-.197 1.185 1.532 1.532 0 00.889.889h.237zM634.997 236.896a1.451 1.451 0 00-.928-.929l-.237-.039h-.198l-.158.158c-.308.066-.577.25-.751.513a1.04 1.04 0 00-.197.633l-.119.118.02.217.474.474a1.369 1.369 0 001.146.395 1.348 1.348 0 00.988-.592c.1-.311.086-.647-.04-.948zM654.652 238.18l-.336-.02h-.198l-1.007 1.007-.059.376-.001.197.06.257.849.849.415.06.257-.02.81-.81a1.37 1.37 0 00-.652-1.877l-.138-.019zM627.649 203.67a1.37 1.37 0 001.896-.316l.474-.474-.098-.099-.04-.276a1.397 1.397 0 00-1.699-.948l-.395.395a1.363 1.363 0 00.198 2.173l-.336-.455zM630.967 239.543a1.532 1.532 0 00-1.165-1.166 1.37 1.37 0 00-1.403.534l-.237.237a1.43 1.43 0 00.633.948 1.625 1.625 0 001.244.178l.81-.81.059-.336.059.415zM613.071 216.984l.296.02.573-.573.158-.158a1.53 1.53 0 00.217-1.245 1.306 1.306 0 00-.83-.829l-.098-.099-.297-.02-.375.375a1.34 1.34 0 00-.513.514.523.523 0 00-.198.198l-.099.098-.039.277.257.257c.167.321.429.583.75.75l.198.435zM630.098 235.908l.158-.356.138-.138-.02-.336-.118-.118a1.2 1.2 0 00-.336-.573l-.158-.158-.356-.119-.138-.138-.336.02-.118.118c-.209.055-.4.164-.553.316l-.158.158-.158.356-.119.119.02.335.099.099a1.7 1.7 0 00.355.593l.158.158.376.138.118.119.336-.02.099-.099c.224-.044.431-.154.592-.316l.119-.158zM631.264 232.451l.118-.118.356-.159.138-.138.198-.197a1.4 1.4 0 00.158-1.186 1.382 1.382 0 00-.87-.869l-.098-.099-.257-.019-.277.276c-.17.131-.323.284-.454.455a1.364 1.364 0 00.02 1.56l-.099-.099.454.455.277.039.099.099h.237zM627.096 234.96l.138-.375.237-.238.02-.256a1.542 1.542 0 00-1.166-1.166 1.37 1.37 0 00-1.402.534l-.139.138-.098.336v.197l.039.672.119.118c.046.213.156.406.316.553l.158.158.375.139.119.118.335-.02.119-.118a1.146 1.146 0 00.83-.79zM619.254 230.397l.158-.395.02-.218v-.316l-.119-.118a1.53 1.53 0 00-.336-.573l-.513-.514h-.356l-.118.119a2.137 2.137 0 00-.553.316l-.139.138-.177.376-.02.217v.355l.099.099c.053.221.162.425.316.593l.158.158.355.158.217.02.336-.02.119-.119c.212-.046.405-.156.553-.316v.04zM639.896 235.355l-.335-.02-1.166 1.166-.039.316a1.545 1.545 0 00-.249.573 1.548 1.548 0 00.236 1.202c.118.174.27.323.447.437l.119.119.316.079.098.098.218-.019.434-.435c.131-.092.244-.206.336-.336.226-.329.318-.731.257-1.126a1.369 1.369 0 00-.632-1.896l-.04-.158zM640.488 230.614l.553-.553a1.193 1.193 0 00.099-1.166 1.403 1.403 0 00-1.185-.948 1.31 1.31 0 00-1.264.593 1.162 1.162 0 00-.158.988 1.39 1.39 0 00.79.79l.099.098c.147.119.322.199.508.233.186.035.378.023.558-.035zM625.002 211.453l.178-.257.079-.197.059-.376v-.197l-.059-.257-.06-.178-.237-.316-.138-.138-.138-.099-.178-.059-.395-.079h-.198l-.237.039-.197.079-.297.218-.158.158-.177.257-.079.197-.06.375v.198l.06.257.079.197.217.297.138.138.139.099.177.059.376.099.217-.02.237-.04.197-.079.297-.217.158-.158zM660.558 221.251l-.099.098-.277.04a1.228 1.228 0 00-.533.296l-.158.158-.158.356-.138.138.039.356.04.237a1.35 1.35 0 00.77.77l.119.119h.316l1.047-1.047.079-.356v-.197l-.06-.257-.77-.771-.217.06zM658.049 235.967l-.138.138-.079.356.099.099.059.257.514.513a1.416 1.416 0 001.075.361 1.417 1.417 0 00.999-.539l.118-.118-.019-.099v-.237l-.119-.119-.119-.316-.158-.158a1.37 1.37 0 00-1.205-.573 1.324 1.324 0 00-1.027.435zM681.458 228.639a1.364 1.364 0 00-.178 1.165 1.391 1.391 0 00.849.85l.119.118h.355l.395-.395c.189-.11.345-.266.455-.454.175-.305.237-.662.178-1.008l-.415-.415a1.42 1.42 0 00-1.146-.395 1.346 1.346 0 00-.612.534zM634.799 205.922l-.098.099-.079.316v.237l.395.395c.078.117.179.218.296.296l.158.158.375.099.218.02.731-.731.079-.277.098-.098-.079-.277-.098-.099a1.345 1.345 0 00-1.166-.691c-.331.085-.624.28-.83.553zM664.311 216.312c-.08.064-.153.137-.217.218v.316l.85.849.335.099h.198l.277-.04c.204-.064.393-.172.553-.316l.197-.197.119-.474.019-.257-.079-.316-.553-.553a1.368 1.368 0 00-1.699.671zM630.256 227.849v-.237l-.474-.475-.059-.256-.849-.85-.396-.079-.217.02-.276.079-.494.494a1.336 1.336 0 00-.178 1.126l.178.178-.04.316.04.276.197.435 1.087 1.086.316.04.257-.06.454-.177.257-.257a1.367 1.367 0 00.138-1.284l.059-.375zM646.335 226.821l-.81.81a1.371 1.371 0 00.065 1.634c.122.157.275.286.449.381l.119.119.355.039.731-.731a1.373 1.373 0 00.198-1.106 1.35 1.35 0 00-.81-.81l-.119-.118-.178-.218zM613.959 186.267l-.316-.316a1.394 1.394 0 00-1.066-.276 1.476 1.476 0 00-.929.612l-.296.296-.02.336 1.067 1.067.316.039.276-.079.455-.217.494-.494c.132-.308.139-.655.019-.968zM618.305 287.208v-.197l-.355-.356a1.21 1.21 0 00-1.047-.77 1.254 1.254 0 00-.711.118l-.632.632a1.781 1.781 0 00-.257 1.363l.474.474c.247.197.553.308.869.316h.237l.79-.79c.072-.059.138-.125.198-.197l.118-.119.316-.474zM640.113 298.981l-.336-.02-.711.711a1.398 1.398 0 00.04 1.66l.533.533.277.04a1.316 1.316 0 00.888-.613 1.224 1.224 0 00.158-1.106 1.424 1.424 0 00-.888-.889l.039-.316zM618.621 253.074l.711-.711.079-.277.04-.316-.158-.158-.237-.474-.119-.119a3.856 3.856 0 01-.671-.908l-.633-.632-.296-.02-.257.059-.434.198-.257.257a1.317 1.317 0 00-.158 1.185l.039.237.178.415.099.098.099.534.592.592a1.312 1.312 0 001.383.04zM614.019 250.684l-.118-.119-.119-.316-.118-.118a1.565 1.565 0 00-1.62-.593l-.81.81a.248.248 0 00-.099.099l-.158.158-.178.415-.118.118.019.296.988.988.415.059.138-.138.277-.079 1.047-1.047.059-.257.395-.276zM623.442 257.42l-.218-.02-.276.039-.632.633a1.415 1.415 0 00-.099 1.125 1.223 1.223 0 00.77.771l.119.118h.355l.771-.77a1.283 1.283 0 00-.178-1.679l-.158-.158-.454-.059zM641.338 267.454l-.079.396.099.375v.237a1.117 1.117 0 00.731.731 1.32 1.32 0 001.047-.099l.395-.395c.174-.29.238-.634.177-.968a1.235 1.235 0 00-.553-.79 1.393 1.393 0 00-1.817.513zM651.709 285.134l.395.079a1.179 1.179 0 00.849-.494 1.33 1.33 0 00.237-1.067 1.277 1.277 0 00-.486-.885 1.281 1.281 0 00-.975-.26l-.771.77-.098.375.197.198.02.217.454.455.178.612z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M647.659 232.609l.77-.77.079-.395-.138-.455-.77-.77-.376-.099a1.264 1.264 0 00-.869.514 1.312 1.312 0 00-.237.987c.1.32.309.594.591.774.282.181.618.257.95.214zM661.526 291.791l-.138-.336-.297-.296a1.345 1.345 0 00-1.205-.455 1.488 1.488 0 00-.395.119l-.711.711-.039.316.039.316 1.027 1.027.218-.019a1.398 1.398 0 001.323-1.324l.178-.059zM647.304 269.074l-.83-.829a1.728 1.728 0 00-.454.019 1.177 1.177 0 00-.909.553 1.572 1.572 0 00-.138 1.284l.849.85h.395a1.339 1.339 0 001.087-1.521l.099-.099-.099-.257zM651.116 295.564l-.494-.494-.335-.059a1.157 1.157 0 00-.376.019 1.297 1.297 0 00-.988.672 1.45 1.45 0 00.079 1.343l.731.731.277.04a1.508 1.508 0 001.126-1.126l.138-.139-.059-.454-.158-.158.059-.375zM644.637 256.689h.237l.098.098a1.26 1.26 0 001.028.317c.453-.095.85-.365 1.106-.751l.138-.138.059-.336.099-.099-.039-.237-.672-.672a1.28 1.28 0 00-1.047-.296 1.4 1.4 0 00-.889-.415l-.217.02-.376.138-.493.494-.158.395-.119.119a1.618 1.618 0 00.711 1.303l.534.06zM650.8 242.861l.375.375.257-.019a1.396 1.396 0 001.106-1.107 1.448 1.448 0 00-.652-1.56l-.276-.277-.297.02a1.344 1.344 0 00-1.086 1.087 1.502 1.502 0 00.573 1.481zM638.296 314.626l-.118.118-.336.139a1.22 1.22 0 00-.672.908l.217.218c.156.259.373.476.633.632l.118.118.356.04.099-.099.236-.04.494-.493a1.53 1.53 0 00.139-1.6 1.198 1.198 0 00-1.166.059zM665.615 262.101l.672.672.316.039.098.099a1.48 1.48 0 001.185-1.185v-.237l-.118-.395-.612-.613a1.178 1.178 0 00-.573-.177l-.217.019-.376.139-.573.573a1.224 1.224 0 00-.098 1.086l.296-.02zM667.709 276.719a1.37 1.37 0 001.137-1.211c.02-.213-.01-.428-.09-.626l-.77-.771-.336-.059-.099-.099-1.047 1.047-.099.375.099.099.04.237.889.889.355.079-.079.04zM644.005 264.274l.119-.118.335-.139.415-.415a1.58 1.58 0 00.316-.75l-.019-.218-.119-.355-.711-.711a1.367 1.367 0 00-.751-.079l-.296.059-.691.691-.079.277-.099.099a1.462 1.462 0 00.454 1.086 1.362 1.362 0 001.126.573zM652.143 332.384l-.474-.474a1.263 1.263 0 00-.85-.138 2.275 2.275 0 00-.474.158l-.355.356a1.305 1.305 0 00-.284.783c-.008.285.078.565.244.797l.356.355.336.139.138.138.336-.02a1.478 1.478 0 001.185-1.185v-.237l-.119-.553-.039-.119zM668.48 268.64v.237l.079.474.533.533.395.198.257.059a.533.533 0 00.138-.138 1.37 1.37 0 001.304-1.304l-.553-.553a1.485 1.485 0 00-1.047-.297 1.509 1.509 0 00-1.106.791zM672.964 299.435a1.624 1.624 0 00-.04.83l.849.849.376.06h.197a1.536 1.536 0 001.087-1.087l.02-.257-.06-.454-.573-.573a1.6 1.6 0 00-.75-.316l-.237.04-.455.178-.651.651.237.079zM667.077 266.368l.375-.099.415-.415a1.401 1.401 0 00.237-1.343 1.56 1.56 0 00-1.047-1.047l-.257.02-.592.592a1.487 1.487 0 00-.315.812c-.021.296.048.591.196.848a1.368 1.368 0 00.988.632zM682.742 300.561l-.455-.454-.513-.277-.296-.059a1.26 1.26 0 00-.949.593 1.371 1.371 0 00-.197 1.106l.988.988h.355a1.317 1.317 0 001.205-1.205l-.138-.139v-.553zM667.235 288.808l.296-.296.079-.356.02-.217-.474-.474a1.508 1.508 0 00-1.186-.395 1.479 1.479 0 00-1.185 1.58l.948.948c.24.087.498.114.751.079l.415-.415.336-.454zM659.847 272.057c-.07.163-.11.337-.119.514a2.43 2.43 0 00-.276.671c-.214.123-.391.3-.514.514l-.652-.652-.375-.099a.807.807 0 00-.454-.02 1.258 1.258 0 00-1.126 1.126v.198l.098.375.692.691.395.119.118.119.87-.87.375.376a1.32 1.32 0 001.284.494l.336.059.217-.02.237-.237c.101-.082.194-.175.276-.277l.396-.395c.128-.192.222-.406.276-.632l.119-.316.118-.118a1.146 1.146 0 00.395-.83l-.059-.296-.988-.988-.434-.039-.237.039c-.18.091-.352.197-.514.316a1.329 1.329 0 00-.454.178zM699.73 250.506l.711-.711a1.07 1.07 0 00.079-1.027 1.234 1.234 0 00-.731-.731l-.138-.139-.395-.079a1.257 1.257 0 00-1.067 1.423c.053.305.198.588.415.81a1.347 1.347 0 001.126.454zM672.766 265.42l-.237.237a1.258 1.258 0 00-.652 1.521l.494.494c.288.243.65.382 1.027.395l.573-.573a1.627 1.627 0 00.099-1.719 1.402 1.402 0 00-1.383-.671l.079.316zM673.379 275.316l.276.277.375.099a1.403 1.403 0 001.169-1.064c.043-.183.048-.372.017-.556a1.54 1.54 0 00-.573-.968 1.338 1.338 0 00-1.047-.257l-.435.435a1.258 1.258 0 00-.494 1.007 1.286 1.286 0 00.534 1.087l.178-.06zM675.275 269.746l.572.573.317.079a1.89 1.89 0 00.671-.277l.612-.612a1.305 1.305 0 00-.237-1.343 1.3 1.3 0 00-.537-.407 1.303 1.303 0 00-.668-.087c-.333.055-.631.24-.829.513a1.644 1.644 0 00-.257.257 1.427 1.427 0 00-.059 1.205l.415.099zM677.941 263.721l-.039.553.039.316.435.435.474.276.296.06.217-.02a1.372 1.372 0 001.245-1.482 1.254 1.254 0 00-.509-1.34 1.323 1.323 0 00-.538-.22l-.099.099-.296.019-.692.692a2.471 2.471 0 00-.237.711l-.296-.099zM664.43 239.977a1.335 1.335 0 001.027-.79 1.362 1.362 0 00-.138-1.284l-.513-.513-.257-.02-.237.039-.889.889a1.31 1.31 0 00.237 1.264c.204.219.475.365.77.415zM699.117 298.408l-.395.119-.514.513a1.488 1.488 0 00-.217.652l.119.119.138.375.513.514c.193.113.409.181.632.197l.119-.118.375-.139.395-.395c.113-.193.181-.409.198-.632l-.119-.118-.138-.376-.533-.533a1.055 1.055 0 00-.593-.197l.02.019zM683.63 294.971l-.415-.415a1.63 1.63 0 00-.592-.197l-.158.158-.455.178-.553.553-.079.395-.019.217c.022.175.08.344.17.496a1.297 1.297 0 00.826.612 1.326 1.326 0 00.801-.061l.751-.751.079-.316-.218-.217-.138-.652zM691.018 279.682l.099.099.514-.514.375-.573.099-.375a1.338 1.338 0 00-1.324-1.126 1.34 1.34 0 00-1.323 1.126 1.28 1.28 0 00.435 1.077 1.295 1.295 0 001.125.286zM681.813 280.077l-.435-.435-.336-.098-.118-.119-.277.04a1.365 1.365 0 00-1.185 1.33c-.004.216.044.431.138.625l.85.85.375.098h.198l.237-.039.849-.849.099-.376-.079-.474-.119-.118-.197-.435zM651.155 290.941l-.118-.118a1.395 1.395 0 00-1.363-.613 1.425 1.425 0 00-1.146 1.146 1.48 1.48 0 00.573 1.521l.138.138.395.08.119-.119.257-.059.237-.237a1.405 1.405 0 00.632-1.462l-.099-.099.375-.178zM708.342 313.441l.06.256.632.633a1.4 1.4 0 001.205.414c.05-.042.096-.088.138-.138l.81-.81a1.336 1.336 0 00.02-1.047l-.83-.829a1.905 1.905 0 00-.612-.02 1.361 1.361 0 00-1.369.951 1.37 1.37 0 00-.054.59zM605.486 306.547a1.316 1.316 0 00-1.028 1.027l.04.316.138.375c.077.304.251.574.494.771.282.196.629.274.968.217l.355-.356a1.253 1.253 0 00.455-1.56 1.453 1.453 0 00-1.422-.79zM661.111 217.893a1.4 1.4 0 002.035.889l.454-.455a1.342 1.342 0 00.079-1.303l-.514-.514-.434-.198-.099.099.059-.336a1.98 1.98 0 00-.75-.079l-.79.79.059.376-.099.731zM695.127 320.572l-.731-.731a1.425 1.425 0 00-1.185 1.501 1.29 1.29 0 001.244 1.245l.415-.415-.158.158a1.312 1.312 0 00.336-1.758h.079zM674.069 288.136a1.248 1.248 0 00.909-.908l.02-.257-.04-.474-.987-.988-.376-.059a1.338 1.338 0 00-1.185 1.185l.119.119.079.316.474.474a1.31 1.31 0 00.987.592zM652.459 290.744l.395.158.218.02a1.573 1.573 0 001.185-.751c.082-.062.155-.135.217-.217.29-.258.468-.62.494-1.008a1.925 1.925 0 00-.849-1.837l-.139-.138-.375-.059-.217.02-.692.691a.677.677 0 00-.158.158 1.403 1.403 0 00-.671.909l-.237.237-.06.256a1.282 1.282 0 00.455 1.047l.434.435v.079zM619.096 251.256l.612.613.316.079h.198l.237-.04c.361-.089.673-.316.869-.632a1.188 1.188 0 00.039-1.067l-.118-.118a1.284 1.284 0 00-1.146-.751l-.118.119-.356.118-.671.672a1.341 1.341 0 00-.02 1.047l.158-.04zM636.538 262.141l-.257.257-.079.276-.099.099a1.486 1.486 0 00.514 1.146 1.465 1.465 0 001.106.276l.731-.731c.262-.214.438-.516.494-.849l-.79-.79a1.367 1.367 0 00-.751-.079.968.968 0 00-.296.059l-.336.336h-.237zM656.805 270.319l.889-.889.039-.237.04-.277-.139-.494-.395-.395a1.742 1.742 0 00-.395-.395l-.316-.039-.553.118-.652.652a1.361 1.361 0 00-.217 1.047l.889.889c.263.079.543.086.81.02zM655.304 260.126l-.099-.099a1.339 1.339 0 00-.968.612 1.479 1.479 0 00-.158 1.225l1.442 1.442.415.138h.237l.237-.039.909-.909.059-.415a1.338 1.338 0 00-.652-.968 1.453 1.453 0 00-.494-.77l-.118-.119-.81-.098zM677.428 286.912l-.118.118-.02.296v.198a1.453 1.453 0 001.659 1.185l.889-.889.079-.434.02-.257-.02-.217-.751-.751a1.28 1.28 0 00-.869-.158c-.194.035-.38.102-.553.198l-.454.454.138.257zM656.529 283.751h.237a1.426 1.426 0 001.185-1.185l.04-.277-.02-.217-.059-.257-.554-.553a1.619 1.619 0 00-1.046-.178 1.573 1.573 0 00-.257.059l-.711.712-.099.296-.099.099c.003.462.179.906.494 1.244l.276.277.613-.02zM653.21 273.598l.316-.04.494-.217.158-.158c.142-.106.269-.233.375-.375.173-.084.338-.184.494-.297l.533-.533.099-.336v-.197a1.706 1.706 0 00-.613-1.324l-.118-.118-.316-.079a1.364 1.364 0 00-1.008-.415l-.809.81a1.841 1.841 0 00-.534.849v.277l-.059.296v.316a1.48 1.48 0 00.988 1.541zM655.086 278.753l.494-.493.099-.297.098-.098a1.399 1.399 0 00-1.442-1.442l-.553.553a1.455 1.455 0 00-.118 1.58 1.37 1.37 0 001.402.652l.02-.455zM661.289 285.312l-.435-.435-.316-.118-.118-.119-.237.039a1.372 1.372 0 00-1.304 1.304l1.007 1.008c.24.087.498.114.751.079.056-.049.109-.102.158-.158l.77-.771.06-.336.098-.098-.039-.237-.119-.119-.276-.039zM635.491 238.812a1.466 1.466 0 00.968-.613 1.645 1.645 0 00.119-1.343l-.494-.494a1.427 1.427 0 00-1.087-.257 1.523 1.523 0 00-1.027.633l-.296.296-.02.257a1.516 1.516 0 00.533 1.165 1.37 1.37 0 001.304.356zM655.343 308.404l.119.118.356.119.098.098.119-.118a1.589 1.589 0 001.066-.672 1.384 1.384 0 00.139-1.165l-.316-.316a1.286 1.286 0 00-1.324-.692c-.174.03-.341.09-.494.178l-.612.612c-.051.22-.065.448-.039.672l.691.691.197.475zM660.302 312.73l-.435-.435-.336-.099-.217-.02a1.767 1.767 0 00-1.442 1.205l-.099.099v.316l.672.672a1.402 1.402 0 001.067.276c.295-.053.57-.19.79-.395l.553-.553.118-.395.119-.118-.04-.237-.118-.119-.632-.197zM650.543 282.111h.277a1.495 1.495 0 001.007-.651c.117-.184.19-.391.213-.607a1.413 1.413 0 00-.075-.638l-1.007-1.007-.276.039a1.363 1.363 0 00-1.008.652 1.477 1.477 0 00-.138 1.245l.908.908.099.059zM646.79 313.401l.118-.118a1.147 1.147 0 00.474-1.265l-.968-.968-.395-.079a1.377 1.377 0 00-1.007.731c-.115.268-.162.56-.138.85a1.323 1.323 0 00.529.897 1.343 1.343 0 001.011.249l.376-.297zM633.614 336.77l-.553-.553a1.928 1.928 0 00-.474-.001 1.394 1.394 0 00-1.086 1.087 1.586 1.586 0 00.592 1.659l.297.297.237.039.296-.02a1.471 1.471 0 001.086-.77 1.588 1.588 0 00-.158-1.541l-.237-.197zM651.234 268.225l-.869-.869-.335.019a1.375 1.375 0 00-1.047.731 1.62 1.62 0 00.079 1.501l.533.534.277.079.296.02.276-.04.435-.237.197-.197a1.652 1.652 0 00.158-1.541zM604.162 297.717l.474-.474a.952.952 0 00.395-.395c.012-.088.032-.173.059-.257l-.178-.415-.829-.83-.376-.098a1.451 1.451 0 00-.849.493c-.105.14-.18.3-.22.47-.041.17-.047.346-.017.518a1.344 1.344 0 001.541.988zM630.928 311.446a1.54 1.54 0 001.363-1.363l-.02-.257-.118-.435-.534-.533-.434-.158-.257-.02-.296.059-.929.929-.039.355.296.06-.296.019.039.474.395.396.455.217.256.059.119.198zM656.41 294.813l-.099-.099-.988.988-.098.375v.198l.039.237.711.711.356.119.217.019a1.319 1.319 0 00.948-.592 1.452 1.452 0 00.198-1.146l-.81-.81h-.474zM648.686 323.989a1.254 1.254 0 00.949-1.225l-.06-.257-.355-.355a1.282 1.282 0 00-1.758-.415l-.633.632-.019.297c.021.318.156.617.382.842.225.226.524.361.842.382l.218-.02.434.119zM698.84 262.694l.455.454.336.059.217.02.296-.059.81-.81.119-.356.098-.098a1.507 1.507 0 00-.987-1.265l-.297-.019-.493.098-.633.633-.059.256a1.315 1.315 0 00-.059.81l.138.138.059.139zM630.533 250.249h.316l.533-.138.612-.613.099-.415.139-.138a1.423 1.423 0 00-.475-1.066 1.332 1.332 0 00-1.007-.297l-.553.553a1.453 1.453 0 00-.237 1.304c.094.327.296.612.573.81zM647.323 252.106l.455.454.118.316a1.42 1.42 0 001.501.909l1.067-1.067a1.737 1.737 0 00-.375-1.363 1.548 1.548 0 00-.692-.415l-.217-.217a1.314 1.314 0 00-1.304-.592l-.335.098-.731.731-.06.297v.197l.04.237.138.138.395.277zM696.016 282.625l-.316.04-.79.79-.138.336.098.098a1.468 1.468 0 00.494 1.047 1.45 1.45 0 001.146.277l.928-.929.06-.335.098-.099-.039-.237-.849-.85-.376-.098-.316-.04zM676.756 336.058a1.42 1.42 0 00-.217.692l.098.336.81.809c.156.04.319.04.474 0a1.406 1.406 0 10-.197-2.805l-.968.968zM659.334 319.861a1.224 1.224 0 00-1.008-.613l-.217.02-.375.138-.593.593a1.342 1.342 0 00-.099 1.047l.83.829.375.06a1.286 1.286 0 001.225-1.225l-.138-.138v-.711zM634.997 324.246l-.395-.395a1.005 1.005 0 00-.632-.237l-.257.059-.435.198-.474.474-.099.375v.237a1.308 1.308 0 00.508.891 1.315 1.315 0 00.994.255c.065-.053.125-.112.178-.178l.612-.612.079-.316.099-.099-.218-.218.04-.434zM672.213 278.852l-.613.613a1.365 1.365 0 00.768 1.76c.215.083.447.109.674.077.183-.034.357-.105.511-.211.153-.105.282-.242.378-.402l.139-.138.079-.316.098-.099a1.561 1.561 0 00-.513-.948 1.396 1.396 0 00-1.047-.257l-.474-.079zM667.946 245.824l-.296.296-.138.415a1.772 1.772 0 00-.06.731l.83.83.336.099.316-.04.553-.237-.099-.099.119.119a1.566 1.566 0 00.652-1.6l-.85-.85-.355-.079-.099.099-.316.04-.375.375-.218-.099zM681.279 313.125l-.414-.415a1.654 1.654 0 00-.613-.217l-.237.039-.375.138-.474.474a.935.935 0 00-.316-.078l-.218.019-1.106 1.106a1.349 1.349 0 00.336 1.205 1.291 1.291 0 001.165.376 1.372 1.372 0 00.909-.553l.316.039a1.393 1.393 0 001.166-1.165v-.238l-.099-.414-.04-.316zM676.914 310.498a1.5 1.5 0 00-.533-1.127 1.541 1.541 0 00-.537-.294c-.197-.061-.404-.082-.609-.061l-.869.869a1.787 1.787 0 00-.099 1.324l.771.77.335.099.099-.099.395-.039.81-.81.119-.356.118-.276zM643.076 295.504a1.453 1.453 0 001.185-1.659 1.299 1.299 0 00-.513-.869 1.426 1.426 0 00-1.008-.257l-.77.77-.079.396.869.869.316.75zM636.143 257.242l.474.474.514.276.335.06 1.067-1.067.04-.395v-.237a1.27 1.27 0 00-.504-.823 1.269 1.269 0 00-.938-.224 1.228 1.228 0 00-.948.948l-.099.533.059.455zM660.381 301.114l-.553.553a1.51 1.51 0 00-.099 1.64 1.43 1.43 0 001.442.652 1.73 1.73 0 00.415-.138l.652-.652a1.573 1.573 0 00-.237-1.344 1.456 1.456 0 00-1.62-.711zM667.334 315.14l-.434-.435-.336-.138-.119-.119-.276.04a1.533 1.533 0 00-1.245 1.758l.138.138.158.395.514.514.375.098h.316l.929-.928c.094-.327.127-.668.099-1.007l-.119-.119v-.197zM655.896 259.81l.573.573.297.098h.355a1.483 1.483 0 001.185-1.185l-.019-.454-.85-.849a1.223 1.223 0 00-.75-.08 1 1 0 00-.297.06l-.612.612a1.712 1.712 0 00-.099 1.324l.217-.099zM662.316 298.507l.118.118.139.336.434.435c.206.158.453.253.711.276h.237l.415-.177.534-.534.118-.395.02-.257-.04-.237-.908-.908-.415-.06a1.26 1.26 0 00-1.306.85c-.06.178-.08.367-.057.553zM650.899 261.963a1.254 1.254 0 00.968.257l.711-.711c.125-.209.187-.449.178-.692l.553-.553.059-.296a1.729 1.729 0 00-.513-1.067l-.336-.336-.218-.02a1.342 1.342 0 00-1.086 1.087l.02.257.079.276a1.335 1.335 0 00-.316 1.067 1.44 1.44 0 00-.099.731zM689.201 304.967l.197.197.138.336a1.229 1.229 0 001.146.553l1.008-1.007.138-.376.099-.099a1.51 1.51 0 00-.415-.888 1.262 1.262 0 00-1.403-.85l-.513.514a1.563 1.563 0 00-.395 1.62zM671.521 325.135l-.415-.415a1.058 1.058 0 00-.573-.217l-.118.118-.395.158-.593.593a1.305 1.305 0 00-.059 1.086l.77.771.316.079.099-.099a1.368 1.368 0 001.185-1.185l-.02-.218-.098-.415-.099-.256zM717.527 260.244a1.31 1.31 0 00.158-.158l.712-.711.078-.276.099-.099a1.444 1.444 0 00-.953-1.5c-.2-.073-.415-.1-.627-.08l-.257.059-.711.711-.079.316-.099.099a1.458 1.458 0 00.514 1.145 1.358 1.358 0 001.165.494zM634.01 251.75a1.283 1.283 0 001.086.613l.119-.119.395-.158.494-.494a1.29 1.29 0 00.098-1.047l-.79-.79-.375-.02a1.286 1.286 0 00-1.205 1.205l.099.099.079.711zM675.69 257.716l.276-.04.474-.158.593-.592.099-.376v-.237a1.309 1.309 0 00-.507-.87 1.316 1.316 0 00-.975-.256c-.109.01-.215.037-.316.079l-.731.731-.059.336a1.427 1.427 0 001.146 1.383zM634.918 301.569c.12-.103.233-.216.336-.336a1.254 1.254 0 00.235-1.475 1.253 1.253 0 00-1.341-.658 1.202 1.202 0 00-.988.79l-.119.118v.237l.02.218.968.967.356.001.098-.099.435.237zM635.234 282.882c.137-.021.27-.061.395-.119l.672-.671a1.622 1.622 0 00-.257-1.324 1.569 1.569 0 00-1.264-.513l-.553.553a1.503 1.503 0 00-.099 1.639 1.42 1.42 0 001.106.435zM667.808 254.062l.197-.198a1.413 1.413 0 00-.02-1.087l-.81-.809a1.668 1.668 0 00-.316 0 1.368 1.368 0 00-1.165 1.521 1.395 1.395 0 001.481 1.205l.633-.632zM613.189 259.336l.395.395.316.118.099.099.296-.02a1.37 1.37 0 00-.296-2.706l-.948.948c-.103.264-.13.551-.079.83l.118.118.099.218zM682.978 273.617l-.513-.513-.415-.099h-.237a1.488 1.488 0 00-1.087 1.087l-.019.256.059.455.553.553.415.178.237.039h.237a1.342 1.342 0 001.205-1.402l-.04-.238-.395-.316zM702.85 273.183l.474-.474.099-.376.119-.118-.04-.316-.928-.929-.395-.079a1.297 1.297 0 00-.889.534c-.198.289-.282.64-.237.988a1.395 1.395 0 001.58 1.185l.217-.415zM685.211 253.923l.869-.869.079-.356v-.197l-.593-.593a1.421 1.421 0 00-.671-.276l-.119-.119-.276-.039-.218.019-1.027 1.028.02.375v.197a1.51 1.51 0 001.936.83zM670.435 296.235l.059-.335-.059-.336-.79-.79-.336-.099-.118-.119-.218.02a1.456 1.456 0 00-1.185 1.778l.751.751c.224.096.471.123.711.079l.296-.06.889-.889zM678.218 246.575l.356.355.256.02.455-.059.908-.909.079-.434-.039-.237a1.817 1.817 0 00-.533-.771l-.257.257.217-.217-.494-.218-.276-.039-.336.059-.771.771a2.096 2.096 0 00.297 1.679l.138-.257zM713.103 289.361l.435.435c.238.117.505.165.77.138a1.257 1.257 0 001.027-1.343l-.276-.277a1.288 1.288 0 00-1.225-.948l-.119.118-.355.158-.435.435c-.17.202-.28.449-.316.711l.059.296.435.277zM685.883 270.279l.632.632.355.079h.198a1.48 1.48 0 001.224-1.659l-.908-.909-.356-.079-.118-.118-.79.79c-.209.402-.252.87-.119 1.304l-.118-.04zM691.709 287.722l-.474-.475-.355-.079-.217-.019-.237.039-.85.85-.099.375-.098.099.809.81c.269.156.582.219.889.177l.376-.652.533-.533-.04-.237-.118-.118-.119-.237zM623.283 296.611l-.139.138-.335.138-.474.474-.139.376-.118.118a1.453 1.453 0 001.442 1.442l.118-.118.356-.158.454-.455.139-.375.118-.119a1.57 1.57 0 00-.434-1.106 1.484 1.484 0 00-.988-.355zM588.853 298.211a1.448 1.448 0 001.145.316l.257-.06.652-.651a1.531 1.531 0 00-.356-1.462 1.281 1.281 0 00-1.145-.435 1.26 1.26 0 00-.81.454l-.296.297-.139.375-.118.119a1.426 1.426 0 00.81 1.047zM617.278 280.018l.277-.079.415-.415.217-.455.039-.276a1.362 1.362 0 00-.39-.795 1.362 1.362 0 00-.795-.39l-.375-.02-.277.039-.79.791-.099.296a1.428 1.428 0 001.304 1.304l.119-.119.355.119zM633.397 292.186l-.375.059a1.279 1.279 0 00-1.007.77c-.088.24-.115.498-.079.751a1.34 1.34 0 001.501 1.185l.474-.474a.915.915 0 00.375-.375l.119-.119a1.12 1.12 0 00-.02-.572l-.612-.613-.376-.612zM629.585 271.188l.197.197c.105.168.247.31.415.415l.237.04.514-.079.592-.593.178-.454.04-.277-.04-.316-.849-.849-.356-.079-.118-.119a1.395 1.395 0 00-1.146 1.146v.276l.138.455.198.237zM631.856 288.907l.237.237h.198c.161-.03.315-.09.454-.178l.355.079h.198a1.317 1.317 0 001.007-.652 1.365 1.365 0 00.079-1.146l-.434-.434-.336-.059-.217.217-.356-.356-.217-.019a1.232 1.232 0 00-1.06.911c-.042.16-.051.327-.026.491.036.27.161.52.355.711l-.237.198zM650.484 307.771a1.315 1.315 0 00-1.284-.572 2.13 2.13 0 00-.474.158l-.652.651a1.252 1.252 0 00.336 1.166 1.528 1.528 0 001.264.434 1.281 1.281 0 001.106-1.303l-.079-.277-.217-.257zM630.453 289.203l-.335.336a1.344 1.344 0 00-.06 1.284 1.376 1.376 0 001.087.691h.237l.415-.178.533-.533.119-.356.118-.118-.039-.395-.593-.593-.474-.197-.277-.04-.335.336-.396-.237zM602.384 281.558l.316-.079.336-.335a1.372 1.372 0 00.79-1.107 1.282 1.282 0 00-.85-1.363l-.256-.059-.474.079-1.126 1.126-.079.395-.02.217a1.31 1.31 0 001.363 1.126zM616.567 272.353l.553.553a1.345 1.345 0 001.146.395c.128-.021.253-.054.375-.098l.948-.949-.118-.513-.909-.909-.395-.118-.237-.04-.237.04a1.256 1.256 0 00-1.086 1.56l-.04.079zM624.271 311.426l-.494-.494-.375-.138-.119-.119-.098.099a1.334 1.334 0 00-.968.612 1.42 1.42 0 00-.158 1.225l.75.751.336.098.099.099a1.393 1.393 0 001.165-1.521l-.118-.118-.02-.494zM621.011 285.015l-.099.297a1.42 1.42 0 00.494 1.126 1.405 1.405 0 001.126.296l.711-.711a1.306 1.306 0 00.099-1.087l-.81-.81a1.769 1.769 0 00-.731-.059l-.316.079-.592.593.118.276zM609.791 287.82a1.434 1.434 0 00-.039 1.264l.751.751.316.04a1.397 1.397 0 001.145-1.146l.099-.099-.039-.276-.534-.534a1.337 1.337 0 00-1.027-.276 1.26 1.26 0 00-.672.276zM645.644 299.732l-.375-.099a1.42 1.42 0 00-.988.632 1.655 1.655 0 00-.198.869 1.231 1.231 0 00-.059.376l.02.217.138.375.534.534c.175.1.371.161.573.177l.217-.019.375-.139.534-.533c.124-.182.199-.393.217-.612l-.099-.099.119-.119a1.649 1.649 0 00-.435-1.066l-.158-.158-.415-.336zM644.163 287.109l-.553-.553a1.452 1.452 0 00-.988-.198 1.401 1.401 0 00-1.106.909 1.476 1.476 0 00.316 1.501l.435.435a.518.518 0 00.256-.02 1.365 1.365 0 001.028-.632 1.636 1.636 0 00.138-1.244l.474-.198zM641.121 276.64a1.423 1.423 0 00-.04 1.304l.494.493c.262.175.576.252.889.218.16-.047.308-.128.434-.237l.751-.751a1.432 1.432 0 00-.454-1.126 1.318 1.318 0 00-1.106-.316c-.364.01-.711.158-.968.415zM628.972 261.133l.652.652.356.079.336-.059.849-.849.079-.356.119-.118-.04-.238-.948-.948-.415-.059-.158.158a1.35 1.35 0 00-.864.634 1.333 1.333 0 00-.124 1.065l.158.039zM621.308 304.235a1.376 1.376 0 00-.987-1.162 1.376 1.376 0 00-.534-.043l-.731.731a1.4 1.4 0 00.434 1.897l.277.276a.672.672 0 00.336-.019 1.375 1.375 0 001.216-1.099c.038-.192.034-.39-.011-.581zM645.762 277.884a1.512 1.512 0 00-.118 1.62 1.339 1.339 0 001.343.711l.474-.474a1.31 1.31 0 00.277-1.106 1.577 1.577 0 00-.692-1.047l-.138-.138-.336-.06h-.197l-.494.494h-.119zM640.804 306.27l-.493-.494-.297-.098-.099-.099-.217.02a1.62 1.62 0 00-1.067.671 1.353 1.353 0 00-.138 1.166l.81.81c.241.083.498.11.751.079l.276-.079.79-.791a1.337 1.337 0 00.02-1.047l-.336-.138zM643.649 281.025l-.395.119-.513.513c-.128.195-.203.42-.218.652l.119.119.138.375.514.513c.188.112.396.186.612.218l.118-.119.376-.138.513-.514c.123-.197.197-.42.218-.652l-.02-.217-.158-.355-.494-.494a1.481 1.481 0 00-.652-.218l-.158.198zM640.31 282.388l-.138-.138-.355-.158-.119-.119-.257.02a1.474 1.474 0 00-1.007.652 1.426 1.426 0 00-.138 1.244l.671.672a1.434 1.434 0 001.108-.136 1.411 1.411 0 00.67-.891l.119-.119-.06-.415-.335-.336-.159-.276zM644.854 283.731l-.375.139-.711.711a1.449 1.449 0 00-.119 1.62 1.37 1.37 0 001.324.691l1.224-1.225c.125-.181.2-.393.218-.612l-.119-.119-.138-.375-.533-.533a.9.9 0 00-.593-.198l-.178-.099zM646.77 292.719l-.375.138-.455.455a1.266 1.266 0 00-.217.612l.02.217.138.376.534.533c.182.104.384.171.592.198l.119-.119.395-.158.513-.514c.119-.184.193-.394.218-.612l-.139-.138-.138-.376-.514-.513a1.429 1.429 0 00-.592-.198l-.099.099zM636.715 274.012l.652.652.356.079.099.099.395-.079.355-.355a1.533 1.533 0 001.075-.601 1.546 1.546 0 00.288-1.197l-1.047-1.047a1.302 1.302 0 00-.474 0 1.511 1.511 0 00-1.007.573c-.182.042-.353.12-.504.23a1.358 1.358 0 00-.548.936c-.023.185-.008.373.044.552l.316.158zM632.765 276.699l-.257-.257a1.174 1.174 0 00-.929.692 1.453 1.453 0 00.06 1.521c.119.174.279.317.465.417a1.323 1.323 0 001.215.011c.188-.096.35-.236.473-.409a1.367 1.367 0 00.197-1.106l-.138-.138a1.543 1.543 0 00-1.086-.731zM628.873 301.055l-.316.079-.198.04-.316.197-.158.158-.158.198-.079.197-.079.356-.019.217.019.139.079.197.198.316.118.158.218.138.177.099.376.099h.197l.316-.04.198-.079.316-.197.158-.119.158-.237.079-.158.099-.375v-.198l-.02-.177-.059-.178-.198-.316-.158-.158-.197-.158-.178-.099-.356-.079-.217-.02zM612.418 267.928l.435.435c.176.1.371.161.573.178l.118-.119.395-.158.573-.573a1.365 1.365 0 00.079-1.106l-.75-.75-.336-.06-.099-.098-.237.039-.849.85-.099.375.217.217-.02.77zM625.081 260.086l.829-.829.079-.356.119-.119a1.438 1.438 0 00-.096-.576 1.435 1.435 0 00-.809-.809 1.438 1.438 0 00-.576-.096l-1.107 1.106a1.517 1.517 0 00.395 1.304 1.434 1.434 0 001.166.375zM604.043 247.464a1.321 1.321 0 00.968-.652 1.233 1.233 0 00.099-1.205l-.791-.79a1.286 1.286 0 00-1.501.434l-.296.297a1.332 1.332 0 00.382 1.119 1.338 1.338 0 001.119.382l.02.415zM618.187 297.44l.987-.987c.03-.144.031-.292.001-.435l-1.067-1.067-.395-.118h-.237l-.81.81a1.453 1.453 0 00.276 1.343 1.625 1.625 0 001.245.454zM606.235 268.501a.902.902 0 00.316-.079l.731-.731.079-.316-.869-.869a1.036 1.036 0 00-.612-.217l-.257.059-.454.217-.534.534-.098.375.019.217c.054.166.14.32.254.452.115.131.255.238.412.314a1.273 1.273 0 001.013.044zM637.545 286.141l-1.046 1.047-.079.356-.02.217.039.237.889.889.336.059.099-.099a1.25 1.25 0 00.948-.592l.592-.593.198-.434.158-.158c.072-.06.138-.126.198-.198l.019-.257-.039-.474-.356-.355-.454-.218-.277-.039a1.362 1.362 0 00-1.066.751l-.139-.139zM626.069 274.902l.434.434.336.099.119.118c.381-.066.721-.278.948-.592a1.465 1.465 0 00.178-1.126l-.79-.79-.336-.099-.099.099a1.337 1.337 0 00-.968.612 1.428 1.428 0 00-.198 1.106l.119.119.257.02zM617.377 326.063l-.158.158-.04.237a1.352 1.352 0 00.503.971 1.344 1.344 0 001.058.274.542.542 0 00.277-.079l.77-.771.059-.257-.039-.434-.613-.612-.454-.178-.296-.02a1.09 1.09 0 00-1.067.711zM689.675 255.345l-.494.494a1.194 1.194 0 00-.118.662c.024.228.113.444.256.622.108.225.289.406.514.514l.138.138.375.099.218.02.256-.02a1.592 1.592 0 001.146-.711l.296-.297.02-.217a1.392 1.392 0 00-.316-.869l-.751-.751-.276-.039a.54.54 0 00-.158.158l-.138.138-.968.059zM683.966 231.424l.633.632.316.04a1.426 1.426 0 001.185-1.186l-.04-.276-.612-.613a1.571 1.571 0 00-1.047-.256 1.341 1.341 0 00-.948.632 1.473 1.473 0 00-.04 1.264l.553-.237zM664.114 278.062l-.276-.039-.494.059-.593.592-.197.474-.02.297a3.537 3.537 0 01-.217 1.086l-.02.257.099.454.533.534.434.197.277.04a1.354 1.354 0 00.988-.672l.178-.178.177-.415.395-.395.099-.296-.039-.356.138-.138a1.307 1.307 0 00-.357-1.114 1.32 1.32 0 00-1.105-.387zM665.378 283.079l-.474.475a1.56 1.56 0 00.652 1.6l.316.079h.592l.158-.158.237-.04.415-.138.533-.534.119-.395.138-.138-.059-.336-.928-.928-.297-.099-.573.02-.75.75-.079-.158zM664.133 271.563l.632.632.435.079h.237a1.343 1.343 0 00.909-.671 1.4 1.4 0 00.256-1.087l-.809-.81-.376-.059a1.318 1.318 0 00-1.205 1.205l.139.138-.218.573zM659.57 251.632l.711-.711a1.037 1.037 0 00.079-1.028 1.225 1.225 0 00-.731-.731l-.138-.138-.395-.118c-.333.055-.631.24-.83.513a1.253 1.253 0 00-.237.949 1.426 1.426 0 001.541 1.264zM687.443 310.754l-.296-.296a1.363 1.363 0 00-.968.02c-.338.16-.61.432-.77.77-.131.33-.125.698.016 1.024.14.326.404.583.734.714l.375-.019.652-.297.514-.513-.02-.415-.158-.158-.079-.83zM665.457 358.578l-.395.039-.632.277-.277.276c-.069.149-.11.31-.118.474l.019.415.277.632.277.277a1.2 1.2 0 001.007-.02 1.266 1.266 0 00.711-.711 1.397 1.397 0 00-.869-1.659z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M682.109 300.166l-.059.375.237.238a1.317 1.317 0 00.908.908l.376.099.098-.099.297-.059 1.047-1.047v-.356l-.929-.928a1.396 1.396 0 00-1.758.612l-.217.257zM683.512 326.932l.02.297.197.553.514.513a1.307 1.307 0 001.086-.059 1.444 1.444 0 00.731-1.047l-.019-.296-.613-.612a1.337 1.337 0 00-1.679.809l-.099.099-.138-.257zM693.033 301.925l.711.711.356.158c.192.079.4.11.607.09.207-.021.405-.093.578-.209a1.448 1.448 0 00.573-1.205l-.79-.79a1.444 1.444 0 00-.817-.068 1.447 1.447 0 00-.724.384l-.198.198-.158.434-.138.138v.159zM679.838 339.061l-.178-.178a1.236 1.236 0 00-.77-.77 1.792 1.792 0 00-1.324.098l-.77.771v.355l.118.119a1.236 1.236 0 00.771.77c.094.327.296.613.573.81l.138.139.415.059.217-.02.751-.751a1.631 1.631 0 00-.02-1.481l.079.079zM667.827 348.306l-.592-.593a1.342 1.342 0 00-1.502.356 1.449 1.449 0 00-.197 1.698l.415.415c.054.077.12.144.197.198a1.375 1.375 0 001.502-.356 1.486 1.486 0 00.177-1.718zM713.497 291.573l.139.139.355.039a1.141 1.141 0 001.027-.474l.218-.217c0-.293-.082-.581-.237-.83l-1.106-1.106-.336.02-.119.118a1.702 1.702 0 00-.889 1.363 1.261 1.261 0 00.948.948zM664.114 324.127l-.059-.336-.77-.77a1.344 1.344 0 00-1.581.435l-.158.158-.099.415v.237l.08.355.158.158c.119.168.266.315.434.435l.158.158.395.079.237-.04.336-.099.158-.158a1.328 1.328 0 00.593-.908l.118-.119zM669.388 310.359a1.369 1.369 0 00-1.145.079 1.311 1.311 0 00-.652 1.008l-.099.098.059.336.869.869.218.02.375-.02.988-.987.019-.218-.019-.414-.711-.712.098-.059zM684.085 332.72l-.119-.118-.375-.06-.118.119-.277.079c-.24.135-.438.333-.573.573l-.138.138-.079.356-.119.118.06.257.375.375c.121.215.298.393.514.514a.902.902 0 00.316.079l.296-.059.573-.573a1.54 1.54 0 00.138-1.166 1.402 1.402 0 00-.474-.632zM710.001 269.272l.138.415.159.158c.129.255.336.463.592.592l.514.079.118-.118a1.283 1.283 0 00.81-1.363l-.02-.218-.039-.237-.771-.77a1.447 1.447 0 00-1.619.435l-.158.158-.158.553.434.316zM664.786 317.135l.177-.178.158-.435.02-.257-.039-.276-.81-.81a1.396 1.396 0 00-1.758.612l-.139.139.02.217a1.296 1.296 0 00.751.75 1.367 1.367 0 001.62.238zM675.788 287.919c.212-.185.369-.424.454-.691v-.198l-.039-.395-.85-.849a1.53 1.53 0 00-1.501.316l-.178.178-.158.434-.019.257.059.375.118.119c.147.23.343.426.573.573l.158.158.455.059.257-.02.493-.494.178.178zM664.865 319.663l-.395-.079-.316.316a1.488 1.488 0 00-.85 1.087c-.025.258.012.518.109.758.096.241.248.455.444.624l.395.395.257-.019a1.48 1.48 0 00.889-1.403 1.316 1.316 0 00-.533-1.679zM667.788 282.21l-.02.336.079.316.672.672a1.456 1.456 0 001.145-.04c.349-.196.604-.523.712-.908v-.316l-.83-.83a1.288 1.288 0 00-1.58.593l-.099.098-.059.257-.02-.178zM675.67 300.008l-.968.968v.316a1.51 1.51 0 00.889.889 1.455 1.455 0 001.778-.83l.138-.138-.02-.454-.533-.533a1.555 1.555 0 00-.632-.435l-.217-.02-.415.02-.02.217zM674.09 318.34c.144-.105.27-.232.375-.376.189-.152.361-.324.514-.513.243.04.493-.002.711-.119l.019.296.692.692c.085.123.192.23.316.316a1.341 1.341 0 001.521-.494l.138-.138.099-.336.118-.119-1.047-1.047-.375.02-.118.119-.02-.257a1.448 1.448 0 00-.889-1.086l-.375-.376-.218.02a1.191 1.191 0 00-.355.158l-.277.277c-.193.133-.36.3-.494.493l-.987.988a1.179 1.179 0 00-.751.553l-.079.277-.099.099-.039.237.039.434.968.968c.196.039.397.039.593 0a1.228 1.228 0 00.02-1.086zM628.656 317.747l-.414.02-.356.158-.138.138a1.172 1.172 0 00-.553.869 1.275 1.275 0 00.316.988l.415.415a1.283 1.283 0 001.639-.731 1.483 1.483 0 00-.79-1.817l-.119-.04zM659.452 317.767h.356a1.258 1.258 0 001.303-1.027v-.198l-.256-.257a1.65 1.65 0 00-.771-.77l-.138-.138-.237.039a1.592 1.592 0 00-.909 1.462 1.37 1.37 0 00.889 1.244l-.237-.355zM663.778 308.897l-.573-.572a1.455 1.455 0 00-1.876 2.074 1.335 1.335 0 00.77.77l.415-.02.138-.138a1.3 1.3 0 00.538-.221 1.3 1.3 0 00.39-.431 1.262 1.262 0 00.06-1.205l.138-.257zM659.393 312.808l-.059-.296-.711-.711a1.46 1.46 0 00-.711-.079l-.415.415a1.313 1.313 0 00-.455 1.284 1.258 1.258 0 001.301 1.082c.166-.01.329-.056.477-.134a.716.716 0 00.237-.237 1.307 1.307 0 00.632-1.027l-.296-.297zM654.118 316.74l.375-.376.099-.296.02-.336-.099-.296-.276-.474-.376-.375a1.394 1.394 0 00-1.817.711 1.16 1.16 0 00-.247.402 1.159 1.159 0 00-.069.467 1.262 1.262 0 00.494 1.007l.118.119.297.098.098.099h.198l.099-.099.276-.078.119-.119c.207-.131.383-.306.513-.514l.178.06zM654.435 344.079a1.336 1.336 0 00-1.284.177 1.313 1.313 0 00-.494 1.205l.731.731.256.02.494-.494a1.286 1.286 0 00.415-1.205 1.365 1.365 0 00-.118-.434zM652.459 276.126l.395.079.218-.02.316-.079.118-.118c.189-.139.355-.305.494-.494l.138-.138.06-.376-.02-.217-.079-.316-.139-.138a1.38 1.38 0 00-.454-.455l-.138-.138-.395-.079-.119.119-.296.098-.119.119a1.81 1.81 0 00-.474.474l-.138.138-.04.395v.198l.099.336.119.118c.114.185.269.341.454.455v.039zM664.312 286.675l.276.276c.114.185.27.34.454.454l.158.158.494.06.257-.06.79-.79a1.304 1.304 0 00-.022-1.022 1.328 1.328 0 00-.303-.424 1.325 1.325 0 00-.445-.272l-.316-.079-.316.079-.85.849-.177.771zM650.424 296.453l-.237-.237-.395.039-.592.316-.257.257a1.317 1.317 0 00-.012 1.036 1.304 1.304 0 00.743.722 1.284 1.284 0 001.54-.474 1.288 1.288 0 00-.351-1.403 1.268 1.268 0 00-.439-.256zM658.682 300.561l.968.968a1.454 1.454 0 001.244-.099 1.397 1.397 0 00.613-1.126l.118-.118-.02-.375-1.106-1.107h-.395l-1.146 1.146-.276.711zM690.742 305.954a1.369 1.369 0 00.889 1.205 1.51 1.51 0 001.56-.415 1.456 1.456 0 00.237-1.62l-.572-.572-.257-.06-.218-.02a1.45 1.45 0 00-1.284.968l-.355.514zM651.629 258.546l.06-.257.039-.474-.296-.297a1.342 1.342 0 00-.889-.889l-.158-.158-.277.04-.494.494a1.508 1.508 0 00-.533.928l.02.257.553.553c.163.139.351.247.553.316a1.31 1.31 0 001.422-.513zM738.21 314.468a1.313 1.313 0 001.422-.356l.217-.217.02-.375a1.203 1.203 0 00-.039-.909 1.27 1.27 0 00-.712-.711h-.197a1.256 1.256 0 00-1.146 1.146 1.448 1.448 0 00.435 1.422zM646.573 365.057a1.397 1.397 0 00-.85-1.679 1.717 1.717 0 00-.513-.079l-.376.375a1.286 1.286 0 00-.691 1.126l.02.296.513.514.218-.217-.237.237c.128.209.304.385.513.513l.119.119.138-.138.356-.159.395-.395.118-.316.277-.197zM666.603 258.664l.04.237.632.632a1.425 1.425 0 001.758-.77 1.316 1.316 0 00-.534-1.679l-.098-.099-.316-.079-.099.099-.237.039a1.343 1.343 0 00-1.166 1.363l.02.257zM669.349 297.282a1.278 1.278 0 00-1.284.218l-.178.177-.178.415-.118.119-.02.217.04.395.731.731a1.343 1.343 0 001.619-.474l.119-.119.079-.316-.099-.099-.02-.217-.099-.099a1.315 1.315 0 00-.592-.948zM689.517 305.48l-.573-.573a1.566 1.566 0 00-1.403.059 2.83 2.83 0 00-.197.198c-.38.093-.712.326-.929.652a1.901 1.901 0 00-.118 2.015l.711.711.099-.099.375-.02.138-.138a.481.481 0 00.139-.138 1.538 1.538 0 001.047-.455l.533-.533a1.445 1.445 0 00.118-1.146l-.059-.256.119-.277zM699.512 356.128l-.079-.316-.81-.81a1.343 1.343 0 00-1.066.158 1.172 1.172 0 00-.553.869l-.099.099a1.25 1.25 0 00.652 1.205l.118.119.356.039h.197l.514-.514c.267-.238.447-.557.513-.908l.257.059zM689.556 338.192l.632-.632a1.517 1.517 0 00.079-1.264 1.433 1.433 0 00-.79-.791l-.217-.217-.316.04-.277.276a1.26 1.26 0 00-.632.909l.02.296.395.395c.171.195.389.345.632.435l.277.079.336-.059-.139.533zM675.808 321.243l-.257-.019-.474.079-.771.77-.138.494v.276l.079.316c.043.177.117.344.218.494l.217.217.533.178.316.04.336-.099.217-.217c.333-.151.6-.418.751-.751l.02-.217-.02-.415-.494-.494a1.785 1.785 0 00-.533-.652zM672.173 330.883l.218.217a1.36 1.36 0 001.185-.086 1.374 1.374 0 00.671-.98l-.039-.356-.296-.296.019-.218v-.237l-.987-.987-.218-.02-.414.02-.889.889a1.29 1.29 0 00.118 1.145 1.43 1.43 0 00.04.909l.434.435.158-.435zM665.812 296.71l.455-.455a1.484 1.484 0 00-.87-1.857l-.138-.138h-.395l-1.086 1.087-.04.237v.237l.099.375.158.158c.125.27.342.487.612.612.189.074.39.114.593.119l.296-.296.316-.079zM682.583 309.608l-.158-.158a1.34 1.34 0 00-1.6.494l-.336.336-.059.257-.02.217.079.395.139.138c.148.324.407.583.731.731l.256.06.297-.02.711-.711c.216-.399.272-.865.158-1.304l-.494-.494.296.059zM680.588 320.058l-.296-.099-.553-.039-.138.138a1.1 1.1 0 00-.514.079 1.867 1.867 0 00-.573.02l-.889.889a1.728 1.728 0 00-.079 1.461l.138.139c.145.348.422.626.771.77l.276.079.336-.336c.323-.087.621-.249.869-.474l.395-.395c.075-.083.154-.163.237-.237a1.508 1.508 0 00.02-1.995zM681.457 314.685l-.316.04-.711.711a1.393 1.393 0 00.101 1.594c.118.149.265.272.433.361l.138.139.118-.119a1.427 1.427 0 00.85-1.323 1.31 1.31 0 00-.889-1.245l.276-.158zM679.166 305.954l.929.929a1.37 1.37 0 001.738-.633l.119-.118.059-.375-.099-.099-.059-.257-.336-.336a1.813 1.813 0 00-.632-.434l-.257-.06-.099-.098-.335.059-1.008 1.007-.02.415zM679.166 359.131a1.387 1.387 0 00-1.126.059 1.607 1.607 0 00-.771 1.126l-.019.257.276.276c.155.353.437.635.79.79a1.56 1.56 0 001.225-.079l.494-.494a1.587 1.587 0 00.079-1.303 1.416 1.416 0 00-.948-.632zM695.562 288.65l-.672-.672a1.519 1.519 0 00-.629-.105 1.519 1.519 0 00-.615.165 1.356 1.356 0 00-.692.968v.237l.138.138c-.013.266.061.53.21.751.15.221.368.387.62.474.155.07.324.103.494.098l.118-.118.317-.119.493-.493c.169-.159.297-.356.376-.573l.098-.099v-.356l-.118-.118-.138-.178zM693.31 282.447l.908.909a1.814 1.814 0 001.877-.336l.553-.553-.039-.316-.099-.099a1.452 1.452 0 00-.771-.77 1.76 1.76 0 00-.908-.04l-1.146 1.146-.02.217-.355-.158zM686.87 313.974l-.178-.178a1.308 1.308 0 00-1.185.079 1.508 1.508 0 00-.731 1.008l.039.316.771.77a1.576 1.576 0 001.225-.079 1.523 1.523 0 00.711-1.027l-.04-.316-.691-.691.079.118zM705.458 288.393l-.138.138a1.121 1.121 0 00-.948.949l-.04.276.02.217.711.712a1.29 1.29 0 001.244-.178c.26-.124.469-.333.593-.593a1.283 1.283 0 00-.77-1.718l-.672.197zM728.293 274.329l.455.454c.122.099.263.173.415.217a1.335 1.335 0 001.481-.336 1.534 1.534 0 00.277-1.738l-.672-.672a1.453 1.453 0 00-1.343.158 1.62 1.62 0 00-.593 1.423l-.02.494zM679.66 325.767l.02.296.888.889a1.428 1.428 0 001.265-.118c.308-.254.512-.613.573-1.008l-.158-.158-.238-.434-.572-.573-.494-.02-.158.158a1.62 1.62 0 00-1.126.968zM735.128 322.804h-.396l-.197.197a1.495 1.495 0 00-.494.257.791.791 0 00-.178.178l-.059.454-.099.099.02.415.731.731a1.242 1.242 0 00.989-.025 1.224 1.224 0 00.65-.746 1.272 1.272 0 00-.156-.983 1.28 1.28 0 00-.811-.577zM718.338 297.796a1.482 1.482 0 00-1.838.612l-.158.158-.079.435v.237l.079.316.968.968.415-.02.731-.731-.197-.197.177.177.218-.454.158-.158-.04-.316-.711-.711.277-.316zM688.095 299.91l.514.513.375-.02 1.007-1.007-.019-.375-.83-.83a1.396 1.396 0 00-1.126.059 1.338 1.338 0 00-.79.79l-.119.119.04.356.672.671.276-.276zM708.915 278.18a1.234 1.234 0 00-1.402.613l-.06.257v.513a1.32 1.32 0 00.81 1.126c.163.065.339.092.514.079l.099-.099.296-.098.711-.711a1.26 1.26 0 00-.454-1.64l-.158-.158-.356.118zM635.333 307.514l-1.126-1.125h-.356l-.81.809a1.454 1.454 0 00.257 1.6l.198.198.513.158.257.02.336-.059.395-.396c.205-.178.355-.411.434-.671l.02-.218-.118-.316zM689.082 351.506l-.217-.217-.533-.139-.297.02-.316.118-.592.593a1.397 1.397 0 00-.099 1.166 1.17 1.17 0 00.751.75l.256.02.119-.119a1.39 1.39 0 00.909-.908 1.452 1.452 0 00.019-1.284zM675.275 341.767l-.454-.454c.038-.108.058-.221.059-.336a1.505 1.505 0 00-.889-1.521l-.099-.099-.336-.019-.671.671a1.85 1.85 0 00-.316 1.383c.051.262.19.499.395.672l-.099.098a1.252 1.252 0 00.83 1.146.674.674 0 00.335.099l.317-.079 1.007-1.008-.079-.553zM647.461 291.475l.316.316h.356l.79-.79a1.47 1.47 0 00.158-.565 1.47 1.47 0 00-.079-.581 1.512 1.512 0 00-.849-.849l-.119-.119-.355.04-1.067 1.066v.395l.849.85v.237zM689.043 254.397l.257.06h.474l.454-.455a1.2 1.2 0 00.494-.494l.138-.138.06-.336.099-.098-.04-.237-.514-.514a1.282 1.282 0 00-.316-.316 1.425 1.425 0 00-1.817.75 1.366 1.366 0 00.711 1.778zM697.616 276.679a1.347 1.347 0 00.592 1.027l.139.139.395.079.118-.119.376-.099.158-.158a1.252 1.252 0 00.592-.869l.02-.217.178-.02-.711-.711a1.282 1.282 0 00-1.64.612l-.138.139-.079.197zM721.024 284.64l.177.178c.109.201.274.365.475.474l.158.158.474.039.928-.928a1.352 1.352 0 00-.015-1.041 1.338 1.338 0 00-.755-.717l-.277-.079-.099.099-.335.059-.968.968.237.79zM666.425 306.25l.217.02.257-.257a1.447 1.447 0 00.474-1.303 1.37 1.37 0 00-.928-.929 1.189 1.189 0 00-1.087.099l-.592.593a1.287 1.287 0 00-.02 1.047 1.335 1.335 0 00.77.77l.909-.04zM654.257 337.185l.593-.593c.2-.14.374-.313.513-.514l.02-.335-.119-.554-.691-.691-.593-.039a1.65 1.65 0 00-1.343 1.106v.197l.02.376 1.126 1.126h.356l.118-.079zM675.156 271.919l.079.355.158.158c.11.188.267.344.454.455l.138.138.395.079.218-.02.375-.099.632.633.296.019.139-.138.415-.178.158-.158c.145-.157.249-.347.305-.553.055-.207.058-.424.011-.632a1.28 1.28 0 00-.869-.869 1.338 1.338 0 00-1.067.039l-.395-.395a1.426 1.426 0 00-1.58.395l-.159.158-.118.435.415.178zM677.704 276.363a1.596 1.596 0 00-.079 1.225c.156.375.454.674.83.83l.098.098.238.04.256-.257c.189-.123.351-.282.476-.469a1.65 1.65 0 00.255-.617v-.198l-.059-.336-.77-.77-.376-.138-.099-.099h-.355l-.81.81.395-.119zM700.026 305.835a1.425 1.425 0 00-1.818.909 1.298 1.298 0 00.02 1.008 1.335 1.335 0 00.731.73l.395-.039.632-.277.652-.652-.039-.395-.277-.632-.296-.652zM687.482 342.755v-.356l-.079-.316-.909-.908-.414.02-.929.928a1.258 1.258 0 00.731 1.6 1.252 1.252 0 001.304-.277l.355-.355-.059-.336zM687.601 292.581l.138.138.118-.118a1.484 1.484 0 00.85-1.403 1.395 1.395 0 00-.909-1.264 1.693 1.693 0 00-.435-.079l-.474.474a1.574 1.574 0 00-.434 1.264 1.643 1.643 0 001.146.988zM688.312 276.936l.948.948a1.496 1.496 0 001.174-.05 1.504 1.504 0 00.782-.878l.138-.139.04-.434-1.047-1.047-.237-.039-.257-.02-.237.039-.395.158-.158.158a2.792 2.792 0 00-.573.81l-.138.139-.04.355z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M671.581 330.804l-.04-.316-.81-.81a1.45 1.45 0 00-1.62.435l-.118.118-.178.415-.158.158.02.296.454.455c.131.207.306.382.514.513l.296.099.118-.118a1.644 1.644 0 00.712-1.107l.81-.138zM684.637 293.845l.139-.138.039-.356-.118-.118-.079-.316a1.117 1.117 0 00-.534-.534l-.158-.158-.434-.079-.237.04-.277.079-.909.909.02.414.889.889a1.28 1.28 0 001.659-.632zM676.874 331.357a1.334 1.334 0 00-.73-.731h-.395l-.396.395a1.093 1.093 0 00-.513.514l-.277.079-.671.672a1.7 1.7 0 00-.079 1.185l.869.869a1.31 1.31 0 001.461-.395l.178-.178.079-.277c.182-.065.347-.17.484-.306.136-.137.241-.302.306-.484a1.42 1.42 0 00-.316-1.343zM664.232 275.257s0-.198-.039-.277l.039-.395a1.347 1.347 0 00-.73-1.047l-.119-.118-.356-.04-.256-.019-.87.869a1.543 1.543 0 00-.059.968c-.051.297.002.602.15.864.147.263.38.467.66.578l.119.118.118-.118a1.481 1.481 0 001.343-1.383zM689.497 266.151l.079.355.158.158c.1.172.243.315.415.415l.158.158.395.079h.237l.355-.118.158-.159a1.278 1.278 0 00.593-.908l-.079-.356-.79-.79a1.478 1.478 0 00-1.62.435l-.158.158-.099.415.198.158zM617.753 300.68l-.178-.178-.316.04-.692.691a1.507 1.507 0 00-.118 1.264 1.391 1.391 0 00.849.85l.257.257.316-.04.711-.711a1.65 1.65 0 00.119-1.225 1.396 1.396 0 00-.948-.948zM686.771 348.681l.099-.099a1.25 1.25 0 00-.632-1.066l-.138-.139-.395-.039-.218.02-.355.355a1.488 1.488 0 00-.593.869v.198l.079.395.81.81a1.285 1.285 0 001.6-.573l.119-.119-.376-.612zM653.19 323.12l-.197-.198-.514-.079-.277.04-.809.81a1.301 1.301 0 00.005 1.029 1.318 1.318 0 00.745.709l.316.079.297-.059.81-.81a1.423 1.423 0 00-.376-1.521zM710.1 304.532a1.37 1.37 0 00-1.304 1.304 1.316 1.316 0 00.849 1.244 1.233 1.233 0 001.245-.217.481.481 0 00.138-.139l.296-.296.04-.277-.079-.513-.81-.81-.375-.296zM700.757 320.71l-.376-.099-.237.04-.355.355a1.585 1.585 0 00-.415 1.284 1.564 1.564 0 00.83 1.067l.138.138.138-.138a1.483 1.483 0 00.869-1.383 1.394 1.394 0 00-.592-1.264zM658.444 330.034l-.335.335a1.424 1.424 0 00-.514.949l.039.276.85.85a1.4 1.4 0 001.758-.771 1.367 1.367 0 00-.731-1.758l-.099-.099-.336-.019h-.197l-.435.237zM708.599 352.02l-.81-.81a1.37 1.37 0 00-1.797.731 1.37 1.37 0 00.73 1.797l.237.039.435.001.533-.534c.212-.184.369-.423.455-.691l.099-.099.118-.434zM654.435 305.757l.039.237.79.79a1.537 1.537 0 001.521-.297l.178-.177.139-.415.019-.257-.019-.296-.652-.652a.308.308 0 00-.158-.158 1.396 1.396 0 00-1.758.612l-.04.237-.059.376zM636.953 296.413l-.632.277-.652.651v.317l.593.592a1.428 1.428 0 001.466-.332 1.35 1.35 0 00.292-.438 1.425 1.425 0 00-.81-1.798l-.257.731zM642.898 321.875l-.138-.138-.395.04-1.225 1.224.039.316.099.099c.13.208.306.384.514.514l.474.474.257.02.454-.06.928-.928a1.478 1.478 0 00-1.007-1.561zM676.46 291.949l-.375.375-.099.296-.118.119.019.336.909.908a1.363 1.363 0 001.185-.039 1.51 1.51 0 00.711-.909l.099-.098-.02-.297-.414-.415a1.752 1.752 0 00-.534-.533l-.296-.059-.336.059-.276.277-.455-.02zM645.506 331.634l-.553-.554-.435-.157-.138-.139-1.205 1.205c-.05.3-.016.608.099.889l.335-.059-.237.237.573.573.296.098.297.02.513-.118.198-.198a2.093 2.093 0 00.553-1.62l-.296-.177zM635.925 277.252l-.098-.296-.317-.316a1.115 1.115 0 00-.533-.534 1.254 1.254 0 00-1.56.652l-.139.138a1.32 1.32 0 00.068.823c.11.26.301.477.545.619l.118.119.395.039h.198l.276-.079c.225-.139.414-.329.553-.553l.198-.197.296-.415zM650.365 307.179l-.039-.356-.83-.829a1.479 1.479 0 00-1.857.869l-.118.118v.395l.039.237.948.949.237.039.099-.099.356-.079.118-.118c.376-.253.639-.643.731-1.087l.316-.039zM653.665 289.183l1.106 1.107.375-.02 1.067-1.067v-.355l-.316-.316a1.51 1.51 0 00-.652-.652l-.672.355-.671.316-.257.257.02.375zM717.883 314.488l.138.138.356.039.217-.019.711-.712c.099-.162.163-.342.189-.53a1.384 1.384 0 00-.29-1.064 1.392 1.392 0 00-.432-.361l-.138-.138-.376-.02h-.197l-.711.711a1.374 1.374 0 00-.139 1.165 1.208 1.208 0 00.672.791zM748.758 329.797a1.505 1.505 0 00-.85-.85l-.257-.059-.256.02-.376.375a1.68 1.68 0 00-.414 1.482 1.305 1.305 0 00.849.849 1.182 1.182 0 00.928-.02l.672-.671a1.422 1.422 0 00-.296-1.126zM715.078 331.91l-.257-.059-.336.02-.731.731a1.4 1.4 0 00.356 1.619l.434.435.336.02.297-.059.572-.573a1.422 1.422 0 00-.375-1.798l-.138.138.118-.118-.276-.356zM706.86 313.46l.277.277a1.284 1.284 0 001.264-.198c.217-.118.395-.296.514-.513a1.4 1.4 0 00-.731-1.798l-.395.04-.198.197a.842.842 0 00-.474.237l-.158.158a1.01 1.01 0 00-.277.514l.02.257.158.829zM700.026 333.649l-.158-.158a1.177 1.177 0 00-.257-.534l-.178-.178-.474-.197-.276-.04-.296.06-1.047 1.047v.355l.869.869a1.418 1.418 0 001.58-.395l.178-.177.118-.475-.059-.177zM706.643 317.075l-.592-.592a1.24 1.24 0 00-.474-.079l-.455-.455a1.261 1.261 0 00-1.165.099 1.262 1.262 0 00-.632.948l.098.099.06.336 1.422 1.422a1.22 1.22 0 00.968.006 1.23 1.23 0 00.672-.697c.105-.245.112-.521.019-.771a.96.96 0 00.079-.316zM699.493 291.554a1.281 1.281 0 00.81 1.126c.167.048.339.075.513.079l.553-.553a1.34 1.34 0 00.237-1.186 1.451 1.451 0 00-.928-.928 1.228 1.228 0 00-1.58.593l-.158.158.553.711zM708.006 317.49l.139-.138a1.317 1.317 0 00.651-1.126 1.283 1.283 0 00-.572-1.126l-.158-.158-.435-.079h-.237l-1.047 1.047-.04.276.04.316.988.988h.671zM728.847 337.777l-.297-.059-.099-.099-.296.059a1.54 1.54 0 00-1.244.613 1.338 1.338 0 00.079 1.62l.177.177.435.158.119.119.375-.02 1.027-1.027a1.316 1.316 0 00-.276-1.541zM711.977 338.962l-.02-.336-.099-.098a1.365 1.365 0 00-.869-.869 1.11 1.11 0 00-.355-.079l-.099.098-.316.079-.672.672-.158.514v.316l.099.335.77.771a1.2 1.2 0 001.067-.04c.252-.114.455-.313.575-.562.119-.25.146-.533.077-.801zM724.204 301.055l.02.218.059.256.691.692a1.256 1.256 0 001.126-.059 1.478 1.478 0 00.751-.988l.099-.099-.04-.316-.849-.849a1.425 1.425 0 00-1.778.75l-.138.138.059.257zM714.169 325.668l.336-.336a1.398 1.398 0 00.117-1.19 1.389 1.389 0 00-.848-.844l-.118-.119-.316.04-.296.296c-.31.213-.523.54-.593.909l.04.276.395.395c.125.224.31.408.533.534l.296.059.316-.039.257-.257-.119.276zM725.389 328.631a1.344 1.344 0 00.652-1.086l-.099-.099-.079-.316-.652-.652a1.477 1.477 0 00-1.6.415l-.395.395.04.276.099.099c.141.339.411.609.75.751a1.254 1.254 0 001.284.217zM699.809 300.917l.296.296a1.454 1.454 0 001.778-.711.926.926 0 00.277-.277l.138-.138.059-.375v-.237l-.079-.316-.138-.138a1.33 1.33 0 00-.435-.435l-.138-.138-.395-.079-.119.118-.355.079-.119.119a1.17 1.17 0 00-.474.474l-.257.256a1.462 1.462 0 00-.118 1.146l.138.139-.059.217zM694.989 312.69l.118.118c.139.319.393.573.712.712a1.338 1.338 0 001.402-.257 1.457 1.457 0 00.435-1.462l-.791-.79a1.306 1.306 0 00-1.185.079 1.507 1.507 0 00-.731 1.007l.04.593zM692.599 323.298a1.571 1.571 0 00.671-1.107l-.039-.237-.277-.276a1.425 1.425 0 00-.671-.672 1.314 1.314 0 00-.652-.059l-.731.731a1.396 1.396 0 101.975 1.975l-.276-.355zM695.68 342.775l-.039-.356-.79-.79-.297-.099-.118-.118-.376.02-1.126 1.125.04.435.968.968a1.38 1.38 0 001.07-.149 1.38 1.38 0 00.629-.878l.039-.158zM723.256 308.68a1.368 1.368 0 00.751 1.778l.098.099.277.039.415-.415c.158-.085.298-.201.411-.341a1.373 1.373 0 00.142-1.515l-.632-.633a1.401 1.401 0 00-1.462.988zM689.122 319.979a1.514 1.514 0 00.909-1.383 1.367 1.367 0 00-.83-1.264l-.119-.118-.098.098a1.373 1.373 0 00-.81.81 1.57 1.57 0 00.118 1.265l.771.77h.355l-.296-.178zM707.236 297.559l.04.316.75.75a1.418 1.418 0 001.245-.059 1.364 1.364 0 00.691-.968l-.02-.296-.414-.415a1.656 1.656 0 00-.514-.513l-.296-.06-.257.02-.395.395a1.625 1.625 0 00-.514.909l-.316-.079zM692.5 318.261l.395.079.118-.119.336-.099.119-.118a1.49 1.49 0 00.474-.474l.138-.138.059-.376.099-.099-.099-.335-.118-.119a1.49 1.49 0 00-.474-.474l-.138-.138-.376-.06-.118.119-.316.079-.119.119a1.253 1.253 0 00-.494.493l-.138.139-.04.395.08.316.118.118c.132.183.292.343.474.474l.02.218zM696.095 318.656l.731.73a1.39 1.39 0 001.185-.079 1.51 1.51 0 00.731-1.007v-.237l-.276-.277a1.565 1.565 0 00-.553-.316 1.32 1.32 0 00-1.502.356l-.158.158-.158.395-.138.138.039.277.099-.138zM692.855 315.377l.376.059.099-.099.098.099.139-.138a1.486 1.486 0 00.888-1.364 1.396 1.396 0 00-.829-1.264l-.119-.118h-.355l-.336.336-.316.079-.138.138a1.39 1.39 0 00-.455.454l-.138.138-.079.396.119.118.079.316.138.138c.101.195.259.354.454.455l.375.257zM695.443 306.507l.395.079.217-.02.316-.079.139-.138c.193-.102.352-.261.454-.454l.138-.139.079-.395-.019-.217-.079-.316-.139-.138a1.317 1.317 0 00-.434-.435l-.139-.138-.395-.079h-.237l-.316.079-.138.138a1.005 1.005 0 00-.454.455l-.139.138-.079.395.02.217.099.336.118.118c.118.182.273.337.455.455l.138.138zM695.167 327.762l-.04-.356-.79-.79-.375-.138-.218-.02h-.316a1.542 1.542 0 00-1.217.025 1.521 1.521 0 00-.502.368 1.513 1.513 0 00-.315.536l-.099.098-.02.336.593.593c.094.129.207.242.336.336.369.139.776.139 1.145 0a1.373 1.373 0 001.719-1.008l.099.02zM699.927 327.268l.553.553a1.253 1.253 0 001.145-.158 1.39 1.39 0 00.692-1.363 1.34 1.34 0 00-.869-1.106 1.203 1.203 0 00-1.008.059 1.306 1.306 0 00-.79.79l.099.099a1.223 1.223 0 00.178 1.126zM715.176 307.91l.297.098.217.02h.356l.217-.059.217-.099.158-.118.277-.277.099-.178.059-.138.039-.198-.019-.375-.06-.217-.098-.218-.099-.177-.297-.257-.158-.119-.296-.098-.197-.04h-.395l-.218.059-.217.099-.158.119-.257.296-.118.158-.04.158-.039.198.019.375.04.197.099.257.118.158.296.257.158.119zM713.36 344.928l-.198-.198a1.061 1.061 0 00-.434-.434l-.139-.139-.395-.079-.158-.474-.375.099-.158.158a1.396 1.396 0 00-.593.909l.059.336 1.028 1.027.375-.02.948-.948.04-.237zM698.604 345.58l-.119-.119-.375.02-.81.81a1.4 1.4 0 00.165 1.632c.13.146.289.263.467.343l.119.119.336.019.73-.73a1.452 1.452 0 00.316-1.304 1.366 1.366 0 00-.829-.79zM710.949 366.775a1.35 1.35 0 00-1.165.099 1.37 1.37 0 00-.672.988v.197l.079.356.415.415a1.205 1.205 0 001.541.355l.513-.513a1.527 1.527 0 00.138-1.205 1.362 1.362 0 00-.849-.692zM722.743 316.226l-.297-.02-.553.553a.782.782 0 00-.296.297l-.119.118-.039.395.019.217.85.85.257.02.494-.494a1.45 1.45 0 00.414-1.324 1.68 1.68 0 00-.73-.612zM719.167 347.338a.583.583 0 00-.316-.079l-.336.059-.869.869.02.336.276.277c.103.193.261.352.455.454l.177.178.475.039.276-.079.77-.77a1.312 1.312 0 00-.928-1.284zM700.322 316.72l1.521-1.521.02-.218-.039-.395-.83-.829a1.332 1.332 0 00-1.126.099.808.808 0 00-.178.177l-.316.04-.711.711a1.224 1.224 0 00-.197.197l-.178.178-.198.435-.019.257.039.316.77.77a1.4 1.4 0 001.284-.138l.158-.079zM704.924 332.147l-.711-.711a1.485 1.485 0 00-1.126.059c-.184.07-.352.179-.492.318-.139.14-.248.308-.318.492l-.098.099.019.336.869.869c.369.129.773.108 1.126-.059a1.51 1.51 0 00.711-.909l.02-.494zM737.32 291.613l.376-.375c.129-.309.136-.655.019-.968a1.352 1.352 0 00-.77-.771l-.178-.178-.098.099-.336.059-.712.712-.158.395-.039.276.039.316.83.83a1.225 1.225 0 001.027-.395zM639.757 318.3l.494-.494a1.366 1.366 0 00.533-1.205 1.493 1.493 0 00-.316-.671l-.73-.731a1.703 1.703 0 00-1.364.059l-.474.474c-.15.278-.205.597-.158.909l1.146 1.145a.644.644 0 00.296.06h.198l.375.454zM633.2 342.083l-.079-.316-.731-.731a1.455 1.455 0 00-1.62.435l-.375.375v.198l.039.276c.148.347.424.624.771.771a1.228 1.228 0 001.106-.119 1.418 1.418 0 00.672-1.066l.217.177zM673.161 310.932l.454.454.455.178.296.02.316-.04.731-.731c.224-.3.49-.566.79-.79l.178-.178.119-.434.039-.237-.059-.297-.692-.691a1.248 1.248 0 00-1.185.119l-.474.474-.494.217-.237.237-.138.336-.119.118a1.32 1.32 0 00.02 1.245zM674.504 306.073l.474-.474a1.508 1.508 0 00.218-1.719l-.237-.237-.494-.257-.455-.454-.415-.06-.256.02-1.008 1.008-.098.296-.04.237.04.434.79.791.494.256.276.04.099.099h.276l.336.02zM670.02 316.601l-.04-.237-.829-.829a1.417 1.417 0 00-1.146.158 1.48 1.48 0 00-.573.928l.099.099.079.355.711.711a1.31 1.31 0 001.62-.553l.138-.138-.059-.494zM664.114 336.216h-.395l-.356.158-.138.139a1.092 1.092 0 00-.573.849 1.318 1.318 0 00.316 1.027l.415.415a1.272 1.272 0 00.984-.036 1.265 1.265 0 00.656-.734 1.403 1.403 0 00-.909-1.818z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M709.981 344.355l-.276.277a1.229 1.229 0 00-.277.948c.046.369.23.708.514.948a1.307 1.307 0 00.967.327 1.314 1.314 0 00.91-.465l.118-.395-.079-.672-.178-.375-.494-.494-.395-.119-.217.02h-.593zM748.086 380.682l-.118.395.059.691.454.455a.923.923 0 00.336.335l.376.099.691-.059.277-.277c.198-.275.295-.609.276-.948a1.364 1.364 0 00-.474-.908 1.454 1.454 0 00-1.877.217zM698.208 345.935l.494.494a1.335 1.335 0 001.205-.494 2.49 2.49 0 00.277-.276l.098-.099v-.316l-.829-.83-.356-.118h-.237l-.474.474a.54.54 0 00-.158.158 1.338 1.338 0 000 1.857l-.02-.85zM723.987 353.343l.316.079h.553l.197-.198c.125-.084.232-.192.316-.316a1.279 1.279 0 00.297-1.047 1.483 1.483 0 00-.751-1.027l-.276-.079-.218-.02-.276.277a2.13 2.13 0 00-.297.296 1.313 1.313 0 00.218 1.837l-.079.198zM703.443 336.157l.256.02.455-.099.494-.494a1.285 1.285 0 00.256-1.165 1.447 1.447 0 00-.948-.948h-.355l-.435.434a1.511 1.511 0 00-.197 1.581l.651.651-.177.02zM734.318 360.83l-.177.177a1.294 1.294 0 00-.988.474c-.273.37-.393.829-.336 1.284l.79.791.316.118.119.119a1.482 1.482 0 001.007-.494c.328.03.654-.069.909-.277l.573-.573.039-.276-.138-.138-.119-.395-.138-.139a1.56 1.56 0 00-1.402-.494l-.455-.177zM739.039 375.151l-.592.592a1.309 1.309 0 00-.139 1.521 1.48 1.48 0 001.541.751l.395-.395a.478.478 0 00.198-.198 1.369 1.369 0 00.158-1.54 1.475 1.475 0 00-1.561-.731zM700.362 313.441l.375-.376a1.193 1.193 0 00-.099-1.126l-.533-.533a2.008 2.008 0 00-.869-.039l-.79.79-.099.336v.197a1.59 1.59 0 00.968 1.284 1.225 1.225 0 001.047-.533zM715.019 370.765l-.356-.039-.099.099-.237.039-.454.455a1.393 1.393 0 00-.119 1.659l.613.612.375.02h.237c.202-.059.394-.145.573-.257l.573-.573.019-.375-.039-.237a1.293 1.293 0 00-.652-.849l-.434-.554zM703.72 361.284a1.2 1.2 0 00-.297 1.086 1.281 1.281 0 00.731.968l.119.119.335.059.87-.869.118-.356v-.197l-.02-.257-.829-.83-.395-.118-.217-.02-.277.04-.375.375h.237zM729.656 354.706l-.513.513v.277c.02.274.108.539.257.77l.533.534.277.039.335-.098c.221-.129.405-.313.534-.534.081-.063.154-.136.217-.217l.099-.099.059-.296-.02-.217-.316-.317a1.5 1.5 0 00-1.047-.533 1.375 1.375 0 00-.415.178zM678.139 309.47h.434l.218-.019a1.4 1.4 0 00.711-.475 1.52 1.52 0 00.256-.454l.198-.197-.099-.099a1.255 1.255 0 00-1.106-1.106l-.119-.119-.237-.039-.217.019-.355.119-.139.138a.673.673 0 00-.237.237 1.422 1.422 0 00-.118 1.659l.573.573.237-.237zM708.619 367.842l-.711-.711-.296-.06-.257.02-.415.139-.336.335a1.423 1.423 0 00.04 1.857l.355.356.217.019a1.342 1.342 0 00.968-.493 1.395 1.395 0 00.435-1.462zM684.658 347.832a1.343 1.343 0 00-.573-.573l-.138-.138-.356-.08-.119-.118-.256.059-.534.534a1.488 1.488 0 00-.301.735c-.031.268.012.54.124.786l.651.651.395.079.099.099c.27-.025.529-.121.751-.276l.178-.178.197-.435.138-.138v-.316l-.256-.257v-.434zM711.087 368.573a1.443 1.443 0 00-.276.276l-.217.218-.02.217a1.45 1.45 0 00.75 1.146 1.457 1.457 0 001.462-.079l.771-.771a1.484 1.484 0 00-1.028-1.303 1.343 1.343 0 00-1.442.296zM676.598 353.6l.217.217.336.02.534-.139.217-.217a1.404 1.404 0 00.116-1.655 1.407 1.407 0 00-.412-.419l-.297-.099-.098-.098-.396.395a1.55 1.55 0 00-.256.256 1.284 1.284 0 00.039 1.699l.277.277-.277-.237zM695.957 351.98l.039.237.909.909.296.098.099.099c.216 0 .429-.047.623-.14.195-.092.367-.226.503-.393a1.397 1.397 0 00-.178-1.956l-.158-.158-.415-.138-.217-.02-.296.059-.119.119a1.176 1.176 0 00-.533.533l-.138.139-.119.395-.296.217zM712.668 359.486a1.615 1.615 0 00-.355-.355 2.842 2.842 0 00-.218-.692c.099-.223.14-.468.119-.711l.276.079.257-.02.376-.375a2.01 2.01 0 00.335-.336 1.286 1.286 0 000-1.58l-.533-.533-.237-.04-.257.02-.81.81-.099.336-.118.118-.257-.059-.099-.099a1.308 1.308 0 00-1.284.454.766.766 0 00-.237.237l-.573.573.02.376.059.375.119.118c.011.243.093.476.237.672l.356.356a1.122 1.122 0 00.276.869l.514.513.434.119h.237l.277-.04.632-.632a1.91 1.91 0 00.178-.573c.123.025.25.031.375.02zM697.399 402.174l-.119.395.04.395v.237a1.063 1.063 0 00.612.81 1.286 1.286 0 001.047.02l.375-.376c.109-.128.191-.277.241-.438.051-.161.068-.331.051-.498a1.252 1.252 0 00-.469-.861 1.532 1.532 0 00-.83-.316 1.371 1.371 0 00-.948.632zM707.473 373.096l.119-.355a1.283 1.283 0 00-.534-1.561l-.256-.256-.099.098a1.683 1.683 0 00-1.027.395l-.435.435a1.566 1.566 0 001.067 1.343 1.336 1.336 0 001.481-.415l-.316.316zM700.42 366.123l-.158.158-.098-.098-.277.276a1.426 1.426 0 00.217 1.995 1.458 1.458 0 001.087.336 1.257 1.257 0 00.948-.513l.119-.396-.02-.217a1.337 1.337 0 00-.316-1.066 1.286 1.286 0 00-1.126-.455l-.376-.02zM702.752 371.536l-.277-.04-.217.02-.533.533c-.15.186-.258.401-.316.632l-.119.119.039.277c.088.226.235.425.426.575.192.149.42.244.661.274a1.317 1.317 0 001.225-.435c.109-.125.189-.274.233-.435a1.11 1.11 0 00.024-.493.78.78 0 00-.02-.375 1.374 1.374 0 00-.85-.85l-.276.198zM704.747 377.798l-.81-.81-.297-.02-.553.119-.513.513a1.375 1.375 0 00-.362.985c.007.179.05.356.126.519.075.164.182.31.315.432.135.286.366.517.652.652a1.309 1.309 0 001.106-.158l.573-.573.039-.316a2.099 2.099 0 00-.296-.691l.02-.652zM730.703 386.41a1.398 1.398 0 00.089 1.867c.162.162.363.281.583.346l.296-.02 1.047-1.047-.059-.257a1.31 1.31 0 00-1.027-.79 1.368 1.368 0 00-.929-.099zM665.812 366.123l.415-.414.04-.317a1.29 1.29 0 00-.277-.632l-.553-.553-.336-.019h-.197a1.37 1.37 0 00-.593.276l-.553.553-.039.316c.043.217.137.421.276.593l.553.553.336.02h.198c.21-.034.408-.122.572-.257l.158-.119zM679.66 358.36l.099.099h.276c.207-.019.405-.094.573-.217l.198-.198.177-.454.04-.277-.04-.316-.553-.553a1.338 1.338 0 00-1.876.178 1.719 1.719 0 00-.218.217l-.019.336-.099.099.849.849.593.237zM684.381 374.657l-.356.356-.099.375.119.672.257.256a1.364 1.364 0 001.916-.138 1.343 1.343 0 00.039-1.62 1.331 1.331 0 00-.444-.27 1.327 1.327 0 00-.515-.076 1.328 1.328 0 00-.917.445zM690.959 368.198l.119.118h.237l.118-.118.336-.139.316-.316a1.422 1.422 0 00.316-1.225 1.342 1.342 0 00-.889-.889l-.138-.138-.375-.099-.119-.118-.257.02-.849.849-.119.356.02.256 1.166 1.166.118.277zM706.505 339.654a1.372 1.372 0 001.422-.435 1.449 1.449 0 00.119-1.62 1.43 1.43 0 00-1.442-.731l-.889.889a1.447 1.447 0 00.513 1.502l.119.118.158.277zM648.943 361.146l-.198-.198-.434-.158-.257-.02-.118.119a1.462 1.462 0 00-1.166.494 1.028 1.028 0 00-.178.177l-.177.178.019.217.139.139a1.34 1.34 0 00.75.75l.257.06.217-.02.297-.296a2.2 2.2 0 00.454-.455 1.338 1.338 0 00.395-.987zM730.031 297.519a1.28 1.28 0 00.139-1.442l-.237-.237-.356-.158a1.31 1.31 0 00-.889-.256 1.218 1.218 0 00-.849.493l-.376.376a1.317 1.317 0 00.692 1.481 1.481 1.481 0 001.876-.257zM747.928 400.633a1.417 1.417 0 00-1.234-.147 1.43 1.43 0 00-.88.878l-.237.237.02.217a1.311 1.311 0 00.83 1.027l.177.178h.277l.454-.138-.118-.119.217.218a1.68 1.68 0 00.514-.514l.335-.336-.039-.395-.612-.612.296-.494zM654 347.002l.217.02.593-.593c.214-.281.318-.631.292-.984a1.447 1.447 0 00-.431-.932 1.312 1.312 0 00-1.777-.039l-.494.493a1.337 1.337 0 00.928 1.521l.672.514zM691.334 357.037a1.26 1.26 0 00-.217 1.284l.849.849.376.099.118-.119.277-.039.553-.553a1.315 1.315 0 00.099-1.679l-.633-.632a1.368 1.368 0 00-1.422.79zM705.675 340.681l-.414.138-.277.277a1.647 1.647 0 00-.395 1.343l.039.316c-.042.387.064.775.297 1.086a1.841 1.841 0 001.876.771l.85-.85.059-.375.02-.217-.139-.138a1.314 1.314 0 00-.079-1.107l-.019-.217-.178-.178a1.425 1.425 0 00-1.047-.494l-.257-.019-.336-.336zM756.877 347.713l-.316-.039-.869.869a1.264 1.264 0 00-.217 1.047c.064.171.164.327.294.456.129.13.285.23.456.294a1.223 1.223 0 001.363-.217l.731-.731-.119-.118-.079-.316a1.477 1.477 0 00-.75-.751l-.494-.494zM736.61 351.308l-.02-.217-.435-.435a1.48 1.48 0 00-1.145-.474 1.335 1.335 0 00-1.008.494l-.138.138-.119.119-.059.296-.099.099.02.217a1.31 1.31 0 00.672.909l.276.039.218-.019.138-.139c.211-.14.393-.321.533-.533.073-.058.14-.124.198-.198l.118-.118.06-.296.79.118zM716.085 358.755l-.178.178-.098.455.02.256 1.224 1.225.336.02c.18.012.36-.008.533-.06l.81-.81.02-.335-.059-.297a1.51 1.51 0 00-.534-.928l-.158-.158-.395-.079-.217-.02-.277.04-.118.118a1.343 1.343 0 00-.909.395zM724.007 365.333l.296-.296a1.396 1.396 0 00-.395-2.133l-.118-.119-.356-.079-.237.04-.296-.297-.297-.019-.138.138-.395.197-.494.494-.099.376v.237l.04.276.553.553a1.373 1.373 0 001.106.277c.266.168.574.258.889.257l.277-.277-.336.375zM689.635 360.197l-.355-.355a1.454 1.454 0 00-2.035.217l-.138.138-.138.376-.02.217.02.257 1.086 1.086.375.02.139-.138c.301-.059.573-.219.77-.455a2.27 2.27 0 00.277-.513l.098-.099v-.316l-.079-.435zM707.295 348.582a.791.791 0 00-.178.178 1.398 1.398 0 00-.079 1.659l.711.711.395.04h.198a1.88 1.88 0 00.889-.573l.197-.198.099-.098.079-.316-.02-.218-.414-.415a1.656 1.656 0 00-1.185-.592l-.218.217-.474-.395zM716.56 353.896l-.217.217-.218.534-.118.118a2.555 2.555 0 00-.099.533 1.171 1.171 0 00-.138.534l-.119.118.04.237.118.119.139.336.138.138a1.852 1.852 0 001.363.533l.316-.316a1.34 1.34 0 001.027-.355l.533-.534.001-.197a2.026 2.026 0 00-.178-.968l-.672-.672a1.537 1.537 0 00-1.916-.375zM711.78 351.249l-.099.099-.079.316.02.217.414.415a1.403 1.403 0 00.988.51 1.394 1.394 0 001.047-.372l.454-.454a1.446 1.446 0 00-.968-1.244 1.39 1.39 0 00-1.481.414s-.316.079-.296.099zM702.732 350.617l.099.099h.237l.652-.652a1.429 1.429 0 00-.04-1.857l-.494-.494-.257-.02-.098.099-.316.079a1.578 1.578 0 00-.514.514l-.197.197-.119.119-.059.336v.197l.909.909.197.474zM753.005 367.941a1.41 1.41 0 00-.316 1.106 1.563 1.563 0 00.809 1.087l.297.296h.237c.382-.039.736-.223.987-.514.256-.332.376-.748.336-1.165l-.494-.494a1.655 1.655 0 00-1.165-.494 1.34 1.34 0 00-.691.178zM691.729 329.461l-.632.632a1.51 1.51 0 00-.375 1.205 1.423 1.423 0 00.691.968l.138.138.237.04a1.257 1.257 0 001.442-.376c.102-.145.182-.305.237-.474l.099-.098v-.356l-.355-.356a1.381 1.381 0 00-.475-.474l-.138-.138-.336-.099-.118-.118-.415-.494zM685.132 329.54l.118.119.237-.001.119-.118.336-.138.257-.257a1.812 1.812 0 00.276-1.896l-.711-.711-.296-.06a1.322 1.322 0 00-.948.514 1.563 1.563 0 00-.356.829l-.099.099.02.297.85.849.197.474zM712.826 345.935a.7.7 0 00-.178.178 1.433 1.433 0 00-.335 1.165 1.543 1.543 0 00.731 1.008l.316.079.632-.632a1.398 1.398 0 00.296-1.166 1.447 1.447 0 00-.711-1.027l-.316-.079-.415.415-.02.059zM694.831 320.117a1.142 1.142 0 00.513 1.265l.435.434.276-.039.376-.376a1.313 1.313 0 00.276-1.224 1.724 1.724 0 00-.474-.711 1.332 1.332 0 00-.981-.316 1.347 1.347 0 00-.915.474l.494.493zM688.904 293.845l.218.02.138-.138c.124-.085.231-.192.316-.317a1.338 1.338 0 00.158-1.54 1.531 1.531 0 00-1.541-.83l-.671.672a1.365 1.365 0 00-.257 1.284 1.56 1.56 0 001.086 1.086l.553-.237zM721.617 356.582l.118.119.296.059h.198l.632-.632a1.453 1.453 0 00.296-1.244 1.594 1.594 0 00-1.066-1.067l-.277-.04-.494.099-.573.573-.197.474-.02.257a1.565 1.565 0 001.087 1.402zM736.925 303.189l-.118.395.039.276a.954.954 0 00.059.573l.198.198.415.178.118.118.376.099.335-.02.297-.02.276-.276c.214-.26.32-.593.296-.929a1.364 1.364 0 00-.474-.908 1.337 1.337 0 00-1.817.316zM707.848 310.892a1.538 1.538 0 000 1.936l.632.632h.277l.276-.039.415-.178.533-.533.119-.395.118-.119-.059-.257-.395-.395-.79-.79-.316-.079-.257.02-.474.118-.198.198a.215.215 0 01.046-.073.215.215 0 01.073-.046zM700.461 339.851l.118-.356.099-.098-.02-.257-.829-.83-.336-.098-.119-.119-.257.02-.118.118-.356.158-.217.218a1.452 1.452 0 00-.296 1.086 1.306 1.306 0 00.612.968l.119.119.355.078.217.02 1.028-1.027zM686.198 313.421a1.258 1.258 0 00.118 1.541l.178.178.494.177a1.228 1.228 0 001.343-.395c.136-.1.256-.22.356-.355v-.316l-.672-.672a1.256 1.256 0 00-1.699-.118l-.177.177.059-.217zM689.873 392.554l-.238.039-.869.869-.138.336v.237l.514.514a1.397 1.397 0 001.58.276l.77-.77.04-.316-.059-.296-.178-.178a1.4 1.4 0 00-.573-.573l-.138-.138h-.711zM749.034 356.108l-.592.593-.099.296.02.336.059.257.375.375a1.285 1.285 0 001.575.365c.164-.083.308-.2.42-.345l.395-.395a1.367 1.367 0 00-.652-1.126 1.45 1.45 0 00-1.501-.356zM735.306 365.965l-.218-.019-.276.276-.217-.217a1.564 1.564 0 00-1.758.336l-.435.434v.237l.099.297.118.118a1.85 1.85 0 001.205.771c.265.023.531-.024.771-.139l.118.119a1.291 1.291 0 001.363-.415.745.745 0 00.237-.237l.119-.119.039-.316-.889-.889-.276-.237zM678.711 375.822l.336-.335.119-.356.099-.099v-.237l-.494-.494a1.422 1.422 0 00-1.067-.434 1.523 1.523 0 00-1.067.553l-.138.138-.138.139-.06.335-.098.099.039.237.79.79.356.119.217.02.257-.02.375-.376.474-.079zM657.259 324.384l.178-.178.158-.434.02-.257-.06-.296a1.786 1.786 0 00-.276-.672l-.731-.731-.356.079-.118.119a1.442 1.442 0 00-.607 1.721c.073.208.193.396.35.55a1.368 1.368 0 001.442.099zM681.102 323.614a1.23 1.23 0 001.166-.218l.553-.553.019-.375-.019-.257a1.507 1.507 0 00-.751-.75l-.119-.119-.375-.059h-.197l-.672.672a1.256 1.256 0 00.04 1.738l.138.138.217-.217zM696.233 304.097l.119.119.256-.02a1.34 1.34 0 00.593-.277l.198-.197.177-.415.04-.276-.02-.297-.553-.553a1.34 1.34 0 00-1.896.158 1.004 1.004 0 00-.198.198l-.118.118-.04.316v.198l.83.83.612.098zM698.9 362.706l.139-.138-.02-.217a1.404 1.404 0 00-1.067-.87 1.347 1.347 0 00-1.225.474 1.192 1.192 0 00-.256 1.047l.454.455c.131.131.289.233.462.298.174.064.36.091.545.077a1.253 1.253 0 00.948-.514.498.498 0 00.02-.612zM724.185 384.435l-.02-.257-.118-.118-.158-.395a3.086 3.086 0 00-.514-.514l-.316-.119-.573-.059-.296.059-.474.474-.237.514a1.595 1.595 0 00.612 1.64l.119.118.375.099.099-.099.256-.02.87-.869.098-.336.277-.118zM669.27 343.269l.355.039h.237c.213-.035.415-.116.593-.237l.573-.573.02-.375.869-.869.039-.237-.019-.415-.159-.158a1.282 1.282 0 00-1.027-.672 1.205 1.205 0 00-1.106.474c-.237.28-.358.641-.336 1.008l-.335.336a1.423 1.423 0 00-.119 1.659l.454.454-.039-.434zM674.287 342.32c.299.302.702.478 1.126.494a1.504 1.504 0 001.086-.533l.277-.277.02-.256a1.594 1.594 0 00-.771-1.047l-.118-.119-.356-.079h-.197l-1.363 1.363.019.257.119.118.138.336.02-.257zM709.448 330.863a1.484 1.484 0 00.237 2.015c.275.217.62.322.968.296.353-.034.679-.204.909-.474l.118-.395-.059-.691-.83-.83-.375-.098-.691.059-.277.118zM740.185 354.686l-.237-.237-.336-.02-.573.139-.553.553-.099.375-.118.118.039.277.593.593a1.287 1.287 0 001.778-.158 1.265 1.265 0 00.158-1.344l-.356-.355-.296.059zM692.835 338.271l.336-.336.02-.217a1.424 1.424 0 00-1.028-1.264 1.343 1.343 0 00-1.481.454c-.116.08-.216.18-.296.296l-.119.119.138.138a1.378 1.378 0 001.047.81 1.433 1.433 0 001.383 0zM678.278 332.483l.118.119h.237l.119-.119.335-.138.316-.316a1.534 1.534 0 00-.237-2.134l-.177-.177-.395-.159-.139-.138-.296.02-1.185 1.185.02.415.158.158c.156.303.364.577.612.81l.138.138.376.336zM723.789 365.985l-.276-.079-.573.573a1.452 1.452 0 00-.119 1.659l.277.277.296.099h.237l.435-.079c.223-.126.408-.31.533-.534l.198-.197.118-.119-.098-.098a1.676 1.676 0 00-.791-1.028l-.237-.474zM693.072 341.471l-.138-.138-.356-.158-.118-.119-.296-.02a1.232 1.232 0 00-.731.257l-.158.158-.198.395-.158.158v.277l.04.237.197.395.474.474.395.119.218.019.711-.711a1.285 1.285 0 00.118-1.343zM726.14 361.106a1.364 1.364 0 00-.908.474l-.099.375.02.376v.237c.036.238.146.46.316.632l-.02.296.533.534c.298.289.692.458 1.106.474l.099-.099.316-.119.534-.533a1.343 1.343 0 00.098-1.521l-.375-.375a1.352 1.352 0 00-.123-.567 1.354 1.354 0 00-.351-.461 1.49 1.49 0 00-1.146.277zM668.835 354.686l-.277-.039a.809.809 0 00-.355-.158 1.206 1.206 0 00-1.205.335l-.553.554-.02.217v.237l.138.138.158.356.158.158c.258.212.576.337.909.355a1.313 1.313 0 001.64-.296l.237-.237a1.57 1.57 0 00-.83-1.62zM668.578 327.801l.356.04.119-.119a1.14 1.14 0 00.572-.217l.593-.593.02-.375-.02-.217a1.364 1.364 0 00-.672-.869l-.118-.119-.336-.059-.099-.099-.592.593a1.424 1.424 0 00-.139 1.679l.455.454-.139-.099zM677.665 406.855a.812.812 0 00-.178.178l-.395.395.02.217.316.316a1.526 1.526 0 001.165.534 1.35 1.35 0 001.047-.534c.075-.056.141-.123.198-.197l.118-.119.06-.296-.534-.533a1.598 1.598 0 00-1.145-.514 1.396 1.396 0 00-.672.553zM745.617 357.432l-.118-.119a1.23 1.23 0 00-1.205.257l-.158.158-.158.356-.139.138-.019.336.138.138a1.343 1.343 0 00.612.85l.119.118.375.059.771-.77a1.26 1.26 0 000-1.699l-.139-.138-.079.316zM710.456 380.741l-.198.198-.237.474-.158.158.02.336.573.572a1.282 1.282 0 001.856-.158c.082-.062.155-.136.218-.217l.099-.099.039-.316.099-.099-.593-.592a1.37 1.37 0 00-1.718-.257zM711.483 320.888a1.368 1.368 0 00-.079.474 1.256 1.256 0 00.81 1.244 1.288 1.288 0 001.336-.303 1.288 1.288 0 00.303-1.336l-.711-.711-.494-.099-.276.039-.633.632-.256.06zM723.73 334.992c-.1.083-.193.176-.276.276l-.138.139.138.138a1.592 1.592 0 001.067.83 1.511 1.511 0 001.283-.455l.396-.395a1.513 1.513 0 00-1.028-1.264 1.393 1.393 0 00-1.442.731zM718.791 378.015l.099.099.079.276c.147.336.415.604.751.751l.257.059.375-.375a.559.559 0 00.217-.217 1.399 1.399 0 10-2.094-1.857l-.118.118-.119.316-.118.119.671.711zM755.869 337.777h-.237l-.099.099-.316.118-.316.316a1.398 1.398 0 001.976 1.976l.158-.158.138-.415v-.237l-.02-.257-.158-.158a1.234 1.234 0 00-.573-.573l-.118-.118-.435-.593zM694.416 373.788l.237.039h.276l.593-.592a1.481 1.481 0 00.198-1.541l.039.039-.494-.493-.296-.06h-.237l-.435.119-.335.336a1.396 1.396 0 00.019 1.876l.158.158.277.119zM680.015 387.477l.059.691.632.632.297.099.098-.099.297-.019.257-.257a1.315 1.315 0 00.316-.988 1.399 1.399 0 00-.494-.928 1.45 1.45 0 00-1.976.197l.514.672zM705.932 389.926l-.158.159-.099.375-.019.217.039.237.948.948.316.079c.24-.025.47-.106.672-.237l.652-.652.099-.454-.02-.257-.751-.75a1.455 1.455 0 00-1.679.335zM688.628 348.602l.711.711.316.119.218.02.75-.751a1.495 1.495 0 00.336-1.126 1.472 1.472 0 00-.652-.968l-.118-.118-.277-.08h-.197l-.257.257c-.241.076-.453.22-.613.415l-.217.217-.099.099-.059.336v.198l.138.138.02.533zM716.105 390.894h-.237l-.908.909v.198l.019.296.672.671c.278.13.583.191.889.178l.039-.355.218.217.513-.158.632-.632.04-.514-.059-.296a2.129 2.129 0 00-1.324-1.047l-.494.533zM661.585 382.203l-.296-.02-.336.059a1.09 1.09 0 00-.652.415 1.227 1.227 0 00.119 1.699l.276.277a1.287 1.287 0 001.541-.119l.533-.533-.02-.297a1.705 1.705 0 00-.256-.77l-.198-.198-.711-.513zM694.495 378.213l-.376-.06-.75.751a1.481 1.481 0 00.177 2.035l.158.158.395.118.218.02.257-.02 1.046-1.047.04-.355.099-.099a1.761 1.761 0 00-.81-1.047l-.454-.454zM678.454 369.284l.218.02.256-.02.83-.83.099-.375.118-.118-.019-.257-.771-.771-.355-.118-.099-.099-.296.02a1.425 1.425 0 00-.771.494l-.098.099.138.73.079.751.296.296.375.178zM723.454 316.779l.138-.138.158-.356.118-.118.02-.257-.118-.119-.139-.375-.138-.138a1.424 1.424 0 00-2.035-.139l-.138.139-.138.335-.138.139v.237l.019.217.119.356.158.158a1.476 1.476 0 001.027.513 1.28 1.28 0 001.087-.454zM747.968 292.62a1.405 1.405 0 00-1.067.554.878.878 0 00-.178.177l-.257.257.158.158a1.682 1.682 0 001.265.869 1.286 1.286 0 001.307-.836c.064-.175.09-.361.075-.546l.099-.099-.118-.119-.119-.355-.138-.139a1.457 1.457 0 00-1.027.079zM738.98 325.115l-.178.178-.079.316v.276l.138.494.198.198a1.451 1.451 0 001.679.178l.671-.672.04-.316-.099-.533-.198-.198a1.454 1.454 0 00-1.837-.217l.139.138-.119-.119-.355.277zM718.93 326.794l.277-.276a1.315 1.315 0 00.217-1.245 1.282 1.282 0 00-.514-.513 1.369 1.369 0 00-1.916.098l-.118.395.039.277a.925.925 0 00.079.553l.633.632.276.079h1.027zM735.74 339.93l-.257-.02a1.153 1.153 0 00-.592.079l-.711.712-.06.335.02.257.158.474.553.553.356.119.118.118.257-.019.534-.534a1.422 1.422 0 00.138-1.639l-.514-.514v.079zM722.229 328.236l-.534.534a1.187 1.187 0 00-.316.316l-.395.395a1.374 1.374 0 00-.296 1.165c.042.19.126.369.245.523.12.153.272.278.446.366l.119.119.316.039.099-.099.237-.039 1.402-1.403a1.217 1.217 0 00.304-.923 1.241 1.241 0 00-.462-.855 1.004 1.004 0 00-.711-.276c-.16.011-.315.058-.454.138zM695.759 326.656c.237.065.488.06.722-.014.235-.073.443-.212.601-.401.097-.144.183-.296.257-.454l.119-.119-.139-.138a1.335 1.335 0 00-1.047-.652 1.49 1.49 0 00-1.205.533 1.256 1.256 0 00.04 1.699l.197.198.455-.652zM723.118 327.031v-.237a1.323 1.323 0 00-.278-.614 1.32 1.32 0 00-1.218-.467 1.32 1.32 0 00-.618.271l-.474.474-.039.237-.02.297.039.237.87.869.316.079h.276l.474-.198.534-.533.138-.415zM749.034 314.033c-.08.065-.153.138-.217.218l-.099.098-.039.316a1.4 1.4 0 00.158 1.344 1.28 1.28 0 001.54.474l.771-.771.099-.336.118-.118-.02-.257-.039-.316-.534-.533a1.314 1.314 0 00-1.738-.119zM744.669 330.369l-.119-.118-.316-.079a1.235 1.235 0 00-1.106.474.789.789 0 00-.257.257l-.118.118-.02.336 1.027 1.027.296.099.336.02.751-.751a1.347 1.347 0 00.316-1.027 1.342 1.342 0 00-.79-.356zM712.806 306.448l.119.118.257.02.237-.039.375-.138.277-.277a1.397 1.397 0 00-.415-2.114l-.119-.118-.316-.079-.217.019-.613.613a1.396 1.396 0 00.139 1.916l.138.138.138-.059zM732.896 323.89l-.316-.316a1.58 1.58 0 00-1.146-.513 1.397 1.397 0 00-1.007.533.363.363 0 00-.119.118l-.118.119-.079.316v.474a1.257 1.257 0 00.651.889l.297.059.217-.02.158-.158c.217-.118.395-.296.514-.513.078-.066.151-.139.217-.217l.099-.099.039-.316-.02-.218.613-.138zM739.256 314.29a1.365 1.365 0 00-.869-.869l-.118-.119-.316-.039-.099.099-.257.059-.435.435a1.369 1.369 0 00-.118 1.619l.632.632.296.06c.375-.034.722-.211.968-.494a1.342 1.342 0 00.316-1.383zM704.727 329.421l.257-.256a1.38 1.38 0 00.316-1.107 1.26 1.26 0 00-.454-.81l-.791-.79-.355-.039-.217.02c-.196.042-.378.13-.534.256l-.553.553-.039.356c.043.219.13.428.256.612l.692.692c.261.3.63.484 1.027.513l.119-.118.276.118zM714.209 337.876l.118.118.099-.098c.338-.071.644-.251.869-.514a1.417 1.417 0 00-.281-2.06 1.489 1.489 0 00-.746-.271l-.356.356a.943.943 0 00-.197.197 1.422 1.422 0 00-.316 1.146 1.476 1.476 0 00.731 1.007l.079.119zM723.513 343.585a1.374 1.374 0 00-.909-.909l-.276-.277-.257.02c-.314.055-.6.215-.81.455a1.15 1.15 0 00-.257.612l-.118.119v.237l.336.335a1.362 1.362 0 001.667.403 1.36 1.36 0 00.742-.894c.05-.195.057-.398.02-.595l-.138.494zM743.01 346.923l-.356-.079-1.007 1.007-.119.356-.099.099.02.257.751.75.415.099h.237l.276-.04.632-.632a1.385 1.385 0 00.205-1.065 1.387 1.387 0 00-.62-.89l-.335.138zM719.8 309.767a1.424 1.424 0 001.955-.139l.474-.474-.059-.296a1.398 1.398 0 00-1.6-1.126l-.375.375a.673.673 0 00-.237.237 1.39 1.39 0 00-.158 1.423zM719.226 345.777a1.51 1.51 0 00-.988-1.303 1.315 1.315 0 00-1.461.355l-.336.336a1.436 1.436 0 00.533 1.007c.353.259.791.372 1.225.316l.375-.375-.039.04.138-.139.099-.335.454.098zM703.957 321.342l.316.04.671-.672a.718.718 0 01.158-.158 1.458 1.458 0 00.356-1.185 1.39 1.39 0 00-.652-.968l-.118-.119-.277-.079-.336.336a1.17 1.17 0 00-.513.514 1.21 1.21 0 00-.218.217l-.355.356.237.237c.159.315.415.572.731.73v.751zM718.733 342.004l.395-.395.02-.336.098-.098a1.46 1.46 0 00-.256-.613l-.554-.553-.335-.02-.099.099c-.215.05-.417.144-.593.277l-.77.77-.02.336.099.099c.021.235.118.457.276.632l.534.533.316.04.099-.099a1.06 1.06 0 00.612-.257l.178-.415zM720.273 338.765l.119-.119.355-.118.336-.336a1.393 1.393 0 00.316-1.146 1.549 1.549 0 00-.251-.603 1.555 1.555 0 00-.48-.444l-.178-.177-.256.059a1.739 1.739 0 00-.455.454 1.396 1.396 0 00-.138 1.561l.593.592.276.079-.237.198zM716.006 340.779l.83-.829a1.482 1.482 0 00-.988-1.304 1.372 1.372 0 00-1.461.395l-.474.474.019.336-.02.336.099.099c.036.219.132.424.277.592l.158.158.355.158.139.139.336.019.217-.019a1.92 1.92 0 00.573-.257l-.06-.297zM708.579 335.407l.415-.415.019-.336a1.645 1.645 0 00-.256-.612l-.553-.553-.336-.02a2.27 2.27 0 00-.573.257l-.573.572-.02.336.099.099c.036.222.124.431.257.613l.573.572.336.02.098-.099c.211-.033.408-.122.573-.257l-.059-.177zM728.57 342.597l-.356-.079-1.047 1.047-.098.336a1.532 1.532 0 00.217 2.232l.119.118.296.099.099.099h.237l.099-.099.296-.099.099-.098c.137-.099.257-.219.355-.356.256-.303.39-.69.375-1.086a1.34 1.34 0 00-.375-1.956l-.316-.158zM729.597 337.896l.533-.534a1.257 1.257 0 00.237-1.145 1.51 1.51 0 00-1.086-1.087 1.429 1.429 0 00-1.324.455 1.34 1.34 0 00.316 1.935l.119.119a1.264 1.264 0 001.205.257zM716.343 317.214l.197-.237.099-.178.099-.375.039-.198-.039-.277-.06-.177-.177-.336-.119-.158-.138-.099-.178-.099-.355-.118-.218-.02-.237.04-.197.039-.336.178-.158.158-.198.237-.098.178-.119.355-.02.218.04.237.039.197.178.336.138.178.119.079.178.098.375.099.197.04.257-.02.198-.079.336-.178.158-.118zM750.674 330.883l-.118-.118-.257.019a1.062 1.062 0 00-.553.237l-.573.573-.02.376.04.237a1.356 1.356 0 00.77.77l.118.118.316.04.237-.04.791-.79.118-.355.099-.099-.02-.257-.77-.77-.178.059zM746.525 345.145l-.138.138-.099.376-.118.118.652.652a1.425 1.425 0 002.074.059l.454-.454.02-.257-.099-.099-.099-.296-.158-.158a1.57 1.57 0 00-1.145-.711 1.37 1.37 0 00-1.344.632zM770.566 340.385a1.303 1.303 0 00-.297 1.126 1.493 1.493 0 00.731.968l.119.118.335.02h.198l.277-.277c.188-.109.344-.266.454-.454.214-.275.319-.62.296-.968l-.059-.257-.119-.118a1.416 1.416 0 00-1.106-.514c-.31.014-.605.141-.829.356zM726.673 312.769l-.514.514.395.395a.736.736 0 00.297.296l.197.197.375.178.119.119.099-.099.237-.04 1.027-1.027-.039-.276a1.345 1.345 0 00-1.107-.83 1.504 1.504 0 00-1.086.573zM754.901 326.3a1.21 1.21 0 00-.217.218l-.119.118-.039.316.04.237 1.224 1.225h.277c.214-.028.418-.109.592-.237l.198-.198.178-.454.039-.277-.039-.316-.534-.533a1.344 1.344 0 00-1.6-.099zM719.819 334.123l.02-.257-.336-.336-.039-.237-.85-.849-.375-.099-.217-.02-.277.04-.494.493a1.291 1.291 0 00-.296 1.087l.158.158-.079.316.02.257.138.454.198.198.079.276.691.692.296.059h.277l.474-.158.198-.198a1.475 1.475 0 00.296-1.284l.118-.592zM735.898 334.794l-.691.692a1.36 1.36 0 00-.113 1.616c.105.167.245.309.41.418l.118.119.336.099h.197l.692-.692a1.342 1.342 0 00.316-1.066 1.309 1.309 0 00-.613-.968l-.118-.119-.534-.099zM708.164 290.941l-.316-.316a1.294 1.294 0 00-1.007-.375 1.365 1.365 0 00-.968.494l-.198.197-.118.119-.059.336v.197l1.047 1.047.335.059.257-.019.474-.198.514-.513a1.257 1.257 0 00.039-1.028zM701.547 391.862l.099-.098-.237-.237a1.198 1.198 0 00-.909-.909 1.536 1.536 0 00-.731.059l-.711.711a1.727 1.727 0 00-.395 1.304l.454.454c.217.232.5.39.81.455l.237.039.257-.02.612-.612a1.61 1.61 0 01.198-.198l.118-.118.198-.83zM721.893 405.848l-.356-.04-.671.672a1.397 1.397 0 00-.119 1.659l.632.632.277.04c.36-.043.691-.219.928-.494a1.34 1.34 0 00.277-1.106 1.519 1.519 0 00-.889-.889l-.079-.474zM705.557 357.985l.909-.909.039-.316-.02-.296-.178-.454-.197-.198a3.46 3.46 0 01-.593-.948l-.652-.652-.296-.059-.257.019-.454.139-.197.197a1.235 1.235 0 00-.277 1.186v.237l.118.434.04.553.573.573a1.339 1.339 0 001.442.494zM701.152 355.002l-.059-.335-.119-.119a1.561 1.561 0 00-1.56-.77l-.632.632a.252.252 0 01-.099.098l-.178.178-.178.376-.158.158v.276l.04.237.197.395.534.534.434.118h.237l.277-.079 1.304-1.304-.04-.395zM709.824 362.726l-.237-.04-.296.02-.613.612a1.295 1.295 0 00-.243 1.088 1.287 1.287 0 00.698.868l.118.119.356.039.217-.02.632-.632a1.255 1.255 0 000-1.699l-.158-.158-.474-.197zM726.516 374.637l-.119.395.04.395-.119.119a1.117 1.117 0 00.731.731 1.207 1.207 0 001.047.019h.198l.276-.276a1.237 1.237 0 00.277-.948 1.237 1.237 0 00-.455-.85 1.424 1.424 0 00-1.876.415z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M147.135 359.009a231.321 231.321 0 01-70.372-164.303 231.333 231.333 0 0167.729-165.411 231.313 231.313 0 01254.616-49.1 231.311 231.311 0 0175.086 51.684L905.089 462.73 577.97 789.86 147.135 359.009z'
                    fill={theme.centerChannelBg}
                />
                <Path
                    d='M147.135 359.009a231.321 231.321 0 01-70.372-164.303 231.333 231.333 0 0167.729-165.411 231.313 231.313 0 01254.616-49.1 231.311 231.311 0 0175.086 51.684L905.089 462.73 577.97 789.86 147.135 359.009z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.06}
                />
                <Path
                    d='M147.135 359.009a231.321 231.321 0 01-70.372-164.303 231.333 231.333 0 0167.729-165.411 231.313 231.313 0 01254.616-49.1 231.311 231.311 0 0175.086 51.684L905.089 462.73 577.97 789.86 147.135 359.009z'
                    fill='url(#paint0_linear_2472:110589)'
                />
            </G>
            {isTablet &&
            <G>
                <Path
                    d='M-31.58 253.269l-.118.395a1.285 1.285 0 00.474.869 1.45 1.45 0 001.529.181 1.32 1.32 0 00.664-.771c.054-.167.074-.342.06-.516l-.752-.751-.395-.118-.652.652-.81.059zM21.084 252.123l.75.751.395.118c.163-.016.323-.056.475-.118l.77-.77.099-.376a1.367 1.367 0 00-.474-.908 1.222 1.222 0 00-.948-.277 1.393 1.393 0 00-1.067 1.58zM-38.789 262.731l.336-.138.277-.277a1.39 1.39 0 00.553-1.185l-.119-.395-.612-.613-.297-.059-.098-.099-.316.04-.79.79-.119.316-.217.217a1.398 1.398 0 001.323 1.324l.08.079zM-15.302 249.753l.83-.83c.026-.15.033-.303.02-.454a1.254 1.254 0 00-.534-.929 1.486 1.486 0 00-1.245-.217l-.849.85a1.448 1.448 0 00-.02.414 1.34 1.34 0 001.443 1.166l.098.099.257-.099zM-41.97 252.084l.534-.534.08-.316-.001-.395a1.286 1.286 0 00-1.956-1.007l-.651.651-.04.277a1.421 1.421 0 00.968 1.245l.158.158.435-.04.138-.138.336.099zM10.772 254.573l-.256.256-.04.238v.276a1.34 1.34 0 00.988 1.185 1.505 1.505 0 001.6-.573l.395-.395c.01-.092.01-.184 0-.276a1.338 1.338 0 00-1.086-1.087 1.43 1.43 0 00-1.6.376zM-60.202 238.157l-.119-.118-.138-.336a1.225 1.225 0 00-.79-.79l-.119.118a1.79 1.79 0 00-.632.633l-.118.118-.08.316v.198l.04.237.494.494a1.596 1.596 0 001.6.217 1.37 1.37 0 00-.138-1.087zM-9.376 268.44l-.671.671-.08.316-.098.099a1.422 1.422 0 001.165 1.165h.237l.435-.079.593-.592c.12-.164.2-.353.237-.553l-.04-.237-.118-.395-.554-.553a1.333 1.333 0 00-1.086-.139l-.02.297zM-23.994 269.625a1.307 1.307 0 00.573.968 1.426 1.426 0 001.205.178l.77-.771.08-.316.098-.099-.849-.849-.356-.119-.217-.019-1.007 1.007-.1.336-.197-.316zM-10.324 246.711l.119.118.138.336.415.415c.195.188.443.312.71.356l.12-.119.355-.079.711-.711c.106-.229.153-.48.138-.731l-.059-.296-.553-.553-.296-.099a1.537 1.537 0 00-1.107.395 1.309 1.309 0 00-.69.968zM-78.75 251.057l.474-.474c.144-.259.206-.555.178-.85a1.847 1.847 0 00-.119-.474l-.494-.494a1.319 1.319 0 00-1.56-.099l-.83.83v.316a1.426 1.426 0 001.087 1.284h.237l.553-.079.474.04zM-15.954 270.869l-.237-.039-.454.059-.988.988v.276a1.397 1.397 0 001.303 1.304l.553-.553a1.316 1.316 0 00.336-1.007 1.396 1.396 0 00-.513-1.028zM-47.026 273.734a1.366 1.366 0 00-.83-.119l-.85.85-.098.375.099.099a1.537 1.537 0 00.948 1.224l.237.04.454-.059.573-.573c.184-.199.307-.446.355-.712l-.02-.256-.157-.435-.573-.573-.138.139zM-13.682 269.664l.079.356.415.415a1.37 1.37 0 001.323.296 1.677 1.677 0 001.067-1.067l-.02-.256-.415-.415a1.452 1.452 0 00-1.64-.218 1.372 1.372 0 00-.81.889zM-48.685 283.413l1.027-1.027a1.311 1.311 0 00-1.062-1.241 1.423 1.423 0 00-.578-.004l-.987.988-.02.375a1.315 1.315 0 001.205 1.205l.257-.02.158-.276zM-36.102 268.578l.296.296.336.099.118.118.218-.019.434-.435a1.454 1.454 0 00.435-1.185 1.424 1.424 0 00-1.502-1.264l-.138.138-.395.197-.514.514a1.258 1.258 0 00-.118.751l.415.414.415.376zM.343 303.127l.71.711a1.034 1.034 0 001.007.138 1.26 1.26 0 00.75-.75l.14-.139.119-.395a1.264 1.264 0 00-.465-.858 1.257 1.257 0 00-.938-.268c-.299.023-.583.14-.81.336a1.372 1.372 0 00-.513 1.225zM-13.05 275.393l-.257-.257a1.31 1.31 0 00-1.501-.751l-.494.494a1.51 1.51 0 00-.434.988l.039.237.435.435a1.62 1.62 0 001.718.177 1.31 1.31 0 00.731-1.323h-.237zM-22.967 275.432l-.296.296-.119.396a1.4 1.4 0 001.01 1.202c.18.05.366.065.55.042a1.435 1.435 0 001.007-.533 1.34 1.34 0 00.317-1.027l-.434-.435a1.375 1.375 0 00-.968-.533 1.415 1.415 0 00-1.146.474l.08.118zM-17.495 277.625l-.869.869c.02.241.102.472.237.672l.593.592a1.392 1.392 0 001.383-.158 1.369 1.369 0 00.533-1.165 1.312 1.312 0 00-.474-.869l-.257-.257a1.347 1.347 0 00-1.225-.119l.08.435zM-11.747 280.568l-.533-.098-.296.059-.316.118-.593.593-.08.316a1.397 1.397 0 001.384 1.383c.307.089.635.068.928-.059a1.433 1.433 0 00.691-.889l.1-.099-.04-.316-.573-.573a1.868 1.868 0 00-.692-.257l.02-.178zM12.807 268.459a1.34 1.34 0 00.731 1.087 1.409 1.409 0 001.284-.099l.514-.514.059-.256-.02-.218-.889-.889a1.286 1.286 0 00-1.778 1.383l.1-.494zM-47.44 299.867l-.1-.375-.513-.513a1.5 1.5 0 00-.652-.257l-.119.118-.355.119-.395.395c-.135.17-.217.376-.237.593l.02.217.118.395.494.494c.183.13.392.217.612.257l.119-.119.395-.119.514-.513c.134-.17.217-.376.236-.593l-.138-.099zM-43.155 284.598l.395-.395c.135-.17.217-.376.237-.593l-.02-.256-.197-.475-.553-.553-.376-.098-.217-.02a1.341 1.341 0 00-1.205 1.442.586.586 0 00.08.316l.73.731.316.079.099.099.118-.119.593-.158zM-28.379 292.895l.297.296-.1.099.87.869.375.099a1.324 1.324 0 00.932-.454 1.313 1.313 0 00.313-.988 1.315 1.315 0 00-1.126-1.126 1.254 1.254 0 00-1.56 1.205zM-28.201 283.591l.434-.435.139-.336.098-.099.02-.256a1.429 1.429 0 00-.652-1.087 1.478 1.478 0 00-1.264-.079l-.85.85-.118.355v.435l.85.849.355.119.119.118.395-.395.474-.039zM-37.308 252.36l.119-.118a1.402 1.402 0 00.691-1.324 1.453 1.453 0 00-2.608-.711l-.138.138-.099.376-.118.118.06.257.137.138a1.426 1.426 0 001.423.711l.099-.098.434.513zM-63.046 308.164l-.277.04-.553.553a1.428 1.428 0 00-.474 1.185l.04.237.71.711c.33.147.703.168 1.048.059l.83-.829a1.71 1.71 0 00.059-.613 1.37 1.37 0 00-1.383-1.343zM-50.424 205.939a1.31 1.31 0 00-.928-1.125l-.336.019-.356.119a1.48 1.48 0 00-.79.474 1.2 1.2 0 00-.276.948l.237.237a1.286 1.286 0 001.54.553 1.51 1.51 0 00.909-1.225zM35.05 266.365a1.4 1.4 0 00-1.126 1.522c.019.171.066.337.138.493l.435.435a1.308 1.308 0 001.323.138l.83-.829-.119-.119.336.059c.1-.23.147-.48.138-.731l-.889-.889-.355.04-.711-.119zM-69.328 294.692l.652-.652a1.398 1.398 0 00-1.442-1.244 1.281 1.281 0 00-1.245 1.244l.474.474a1.369 1.369 0 001.758.455l-.197-.277zM-35.806 275.432a1.258 1.258 0 00.75 1.067l.258.02.454-.02 1.027-1.027.099-.376a1.315 1.315 0 00-1.185-1.185l-.1.099-.355.079-.276.277.04-.04a1.372 1.372 0 00-.712 1.106zM-37.209 253.723l-.414.415a1.59 1.59 0 00.671 1.225l.217.217c.237.309.584.514.968.573a1.844 1.844 0 001.877-.77l.138-.139.1-.375.098-.099-.04-.237-.691-.691a1.45 1.45 0 00-.79-.79l-.514-.514a1.39 1.39 0 00-1.086.415l-.257.257-.277.513zM4.017 222.572l-.632.632-.08.316v.435a1.311 1.311 0 00.613.888 1.287 1.287 0 001.047.139l.198-.198a1.28 1.28 0 00.81-1.086l-.119-.119-.099-.336-.691-.691a1.37 1.37 0 00-1.047-.059v.079zM-7.796 239.402l-.158-.158-.296-.099-.1-.099a1.626 1.626 0 00-1.165.455 1.39 1.39 0 00-.355 1.066l.83.83a1.293 1.293 0 001.106.158l.79-.79c.108-.235.149-.494.118-.751l-.04-.276-.335-.336h-.395zM-17.06 259.175l1.027 1.027.296.02.494-.059.435-.435c.148-.099.275-.227.375-.375l.06-.296-.06-.573-.652-.652a1.45 1.45 0 00-1.047-.257l-.889.889c-.073.23-.087.474-.04.711zM-6.808 258.247l.119-.119a1.395 1.395 0 00-.553-.987 1.54 1.54 0 00-1.225-.237l-.375.375.079-.079-.83.83-.138.414-.139.139 1.047 1.047.415.059.099.099a1.432 1.432 0 00.988-.593c.306-.075.586-.232.81-.454l.256-.257-.553-.237zM-34.8 278.85l-.098-.099-.316-.039-.099-.099a1.48 1.48 0 00-1.284 1.6l.889.889.435.118.256.02.218-.02.75-.75c.166-.258.235-.566.198-.87a1.847 1.847 0 00-.158-.553l-.455-.454-.335.257zM-30.493 258.148v.237a1.396 1.396 0 001.185 1.185l.277.04.118-.119.257-.059.553-.553c.18-.303.257-.657.218-1.008l-.04-.276-.909-.909a1.65 1.65 0 00-1.244.415l-.277.277-.138.77zM-20.161 255.402l.04.317.197.513.158.158c.105.143.232.27.375.375l.909.909.355.119a1.74 1.74 0 001.343-.554l.317-.316a1.3 1.3 0 00.474-.987l-.613-.613a1.822 1.822 0 00-.83-.553l-.276-.039-.178-.178-.335-.02a1.509 1.509 0 00-1.936.869zM-25.416 256.943l1.008 1.007a1.388 1.388 0 001.345-.882 1.4 1.4 0 00.097-.56l-.435-.434a1.399 1.399 0 00-1.56-.217 1.37 1.37 0 00-.731 1.362.49.49 0 00.276-.276zM-32.29 262.83l.77-.771a.95.95 0 000-.237 1.37 1.37 0 00-1.323-1.323l-.099.098-.355.119-.652.652c-.09.24-.13.495-.119.75l.06.257.592.593.316.079.257-.02.099-.099.454-.098zM15.572 239.619a1.314 1.314 0 00.553.988 1.541 1.541 0 001.344.197l.415-.414a1.465 1.465 0 00.335-1.087 1.563 1.563 0 00-.592-1.027l-.316-.316-.237-.04a1.544 1.544 0 00-1.185.474 1.314 1.314 0 00-.317 1.225zM-55.026 255.619l-.455.455.119.118a1.421 1.421 0 00.593 1.106 1.368 1.368 0 001.145.198l.316-.316a1.258 1.258 0 00.77-1.284 1.423 1.423 0 00-.138-.494l-.77-.77c-.2-.079-.418-.106-.632-.079l-.691.691-.257.375zM-59.61 260.321l.77-.771a1.791 1.791 0 00-1.105-1.54l-.1-.099h-.315l-.573.573a1.478 1.478 0 00.06 1.876l.552.553.375.099.139.138.237-.039.118-.119-.158-.671zM-28.418 252.4v.237a1.512 1.512 0 00.592 1.066 1.488 1.488 0 001.225.198l1.007-1.008v-.276a1.507 1.507 0 00-.592-1.067 1.62 1.62 0 00-1.225-.197l-.928.928-.08.119zM-59.55 246.79l.138.138a1.206 1.206 0 001.245.533l.928-.928.099-.375a1.283 1.283 0 00-.691-1.048 1.372 1.372 0 00-.81-.177 1.34 1.34 0 00-1.225 1.462l.316.395zM-82.108 232.409l.553-.553c.026-.15.033-.302.02-.454a1.4 1.4 0 00-.968-1.205 1.643 1.643 0 00-1.7.513l-.434.435v.316a1.344 1.344 0 00.731 1.087 1.481 1.481 0 001.521-.06l.277-.079zM-14.69 253.723l.89-.889-.02-.335a1.317 1.317 0 00-.672-1.067 1.563 1.563 0 00-1.501 0l-.711.711-.04.316.06.257.177.454.277.277a1.54 1.54 0 001.54.276zM-41.535 205.11l.474.474c.102.158.237.293.395.395l.277.079.415-.138.71-.712.139-.375a1.205 1.205 0 00-.474-.869 1.34 1.34 0 00-.968-.296 1.31 1.31 0 00-.968 1.442zM-56.705 231.066a1.564 1.564 0 001.383 1.383h.237l.434-.119.632-.632.158-.435.139-.138-.04-.316-.909-.909-.375-.059-.079.277v-.277h-.474l-.909.909-.059.296-.138.02zM-41.495 257.457l.118-.119-.039-.237-.83-.83-.355-.118-.218-.02-.928.929-.158.355-.118.119a1.457 1.457 0 00.553.987 1.386 1.386 0 001.126.257l.81-.81.039-.513zM-70.217 248.172a1.311 1.311 0 001.086 1.087l.277-.04.474-.237a1.28 1.28 0 00.652-1.244 1.425 1.425 0 00-.139-.494l-.632-.632-.276-.04a1.285 1.285 0 00-1.205 1.205v.237l-.237.158zM-11.786 301.586l-.454.455-.099.335v.198l.02.336.81.81.335.098.119.119a1.401 1.401 0 001.324-.929l.039-.276-.079-.514-.79-.79a1.567 1.567 0 00-.81-.099l-.138.139-.277.118zM4.392 234.049l-.02.296.1.533.632.632.375.099.04.04a1.405 1.405 0 001.086-.415 1.284 1.284 0 00.355-.988l-.553-.553a1.46 1.46 0 00-1.264-.316 1.479 1.479 0 00-.75.672zM1.627 250.721l-.454.454-.297.099a1.423 1.423 0 00-.987 1.462l1.047 1.047a1.744 1.744 0 001.402-.297c.224-.125.408-.31.534-.533l.118-.119a1.313 1.313 0 00.672-1.264l-.08-.355-.73-.731-.316-.079h-.237l-.139.138-.533.178zM-31.52 297.655v-.316l-.81-.81-.316-.118-.118-.119a1.418 1.418 0 00-1.067.435 1.537 1.537 0 00-.355 1.145l.889.889.335.099.119.119.988-.988.118-.356.217.02zM-82.998 276.4l-.157-.158-.198-.434-.375-.376a1.409 1.409 0 00-.652-.296h-.198l-.336.099-.691.691a1.256 1.256 0 00-.04.474 1.408 1.408 0 00.425.986 1.392 1.392 0 00.998.397 1.389 1.389 0 00.986-.425 1.4 1.4 0 00.397-.997l-.159.039zM-66.662 258.958a1.26 1.26 0 00.692-.968l-.119-.119-.118-.395-.593-.592a1.477 1.477 0 00-1.047-.139l-.395.396.079-.079-.296.296-.06.375a1.287 1.287 0 001.225 1.225l.217-.02.415.02zM-69.723 234.444l.415-.415c.138-.178.226-.389.257-.612l-.04-.237-.178-.455-.553-.553-.375-.099-.217-.019a1.314 1.314 0 00-1.323 1.323l.078.435.652.652.316.079.217-.218.751.119zM-26.443 274.089l-.652-.652a1.523 1.523 0 00-.71-.09 1.52 1.52 0 00-.673.248 1.428 1.428 0 00-.553 1.185 1.23 1.23 0 00.573.929l.256.257.317.079a1.294 1.294 0 00.987-.435c.233-.289.346-.657.316-1.027l.139-.494zM6.782 271.64l-.296-.297-.415-.138a1.534 1.534 0 00-.711-.118l-.85.849-.098.336.04.316.197.553-.119.119a1.507 1.507 0 001.561.691l.83-.83.098-.335.119-.119-.04-.316-.375-.375.06-.336zM-61.17 281.24l.415-.415a1.45 1.45 0 00.237-.592v-.237l-.138-.376-.474-.474a.751.751 0 00.079-.316v-.197l-1.087-1.087a1.276 1.276 0 00-1.587.746c-.078.195-.11.407-.092.617a1.309 1.309 0 00.514.948l-.08.316-.098.099a1.396 1.396 0 001.047 1.244l.256.02.415-.099.593-.197zM-58.286 277.052c.43 0 .842-.17 1.145-.474.27-.313.418-.713.415-1.126l-.632-.632a1.764 1.764 0 00-1.304-.198l-.79.791-.099.335-.098.099.04.395 1.026 1.027.297-.217zM-41.554 244.162a1.45 1.45 0 001.62 1.265 1.202 1.202 0 00.868-.474c.231-.275.345-.63.317-.988l-.751-.751-.395-.118-.87.869-.79.197zM-2.936 239.323l-.87.869-.079.316.02.257.909.908.395.079h.237a1.257 1.257 0 001.126-1.363 1.202 1.202 0 00-.83-1.066l-.513-.119-.395.119zM-48.014 261.052l-.454-.455a1.428 1.428 0 00-1.62-.158 1.371 1.371 0 00-.731 1.363c.014.155.047.308.099.455l.651.652a1.437 1.437 0 001.335-.187 1.427 1.427 0 00.582-1.216l.138-.454zM-62.414 267.235l.81-.81-.02-.257a1.536 1.536 0 00-1.679-1.363l-.138.138-.395.158-.514.514-.138.336-.099.098v.317l.04.236.79.791c.325.104.666.15 1.007.138l.119-.119.217-.177z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M-6.532 258.839l-.77.771v.355a1.48 1.48 0 001.185 1.186l.099.098.454-.019.711-.712c.09-.239.131-.494.119-.75l-.06-.296-.552-.554a1.738 1.738 0 00-1.304-.158l.118.079zM-45.643 263.106l-.79.79c-.18.191-.297.432-.336.692l.02.217.118.435.533.533.415.138.237.04 1.027-1.027.1-.415-.1-.099a1.284 1.284 0 00-1.224-1.304zM-8.448 253.565a1.323 1.323 0 00-.316.988l.692.691c.2.148.443.224.69.218l.732.73a1.646 1.646 0 001.086-.454l.455-.454.02-.218A1.31 1.31 0 00-6.157 254l-.257-.02-.296.059a1.38 1.38 0 00-1.047-.375 1.511 1.511 0 00-.692-.099zM-53.407 289.635l-.197.198a1.402 1.402 0 00-.355.158 1.173 1.173 0 00-.613 1.086l.119.119.118.355.652.652.355.119.12.118c.33-.026.647-.149.908-.355l.099-.099a1.285 1.285 0 00.928-1.363l-.415-.415a1.539 1.539 0 00-1.718-.573zM-72.607 270.869l.415-.415c.111-.168.192-.356.237-.553l-.02-.257-.119-.395-.573-.572a1.169 1.169 0 00-1.086-.139l-.77.771-.08.316a1.426 1.426 0 001.186 1.185h.237l.395-.079.178.138zM-10.344 320.372a.504.504 0 00.04.237l1.007 1.007a1.48 1.48 0 001.185-.434 1.432 1.432 0 00.395-1.146l-.059-.257-.711-.711-.296-.099-.1-.099a1.49 1.49 0 00-1.165.415 1.457 1.457 0 00-.296 1.087zM2.91 237.269l-.118.118a1.206 1.206 0 00-.671 1.027l.02.218.138.375.553.553c.152.093.321.155.497.183.177.027.357.018.53-.025l.85-.849.059-.376a1.286 1.286 0 00-1.205-1.205l-.652-.019zM-5.544 278.711l.02.297.158.474.593.592.375.099.217.02a1.291 1.291 0 00.921-.446 1.278 1.278 0 00.304-.976.584.584 0 00-.079-.316l-.612-.613-.336-.098a1.483 1.483 0 00-1.56.967zM-47.066 235.609l.336.336a1.286 1.286 0 001.442.296 1.311 1.311 0 00.75-1.303 1.283 1.283 0 00-.75-1.028l-.118-.118-.218-.02-.987.988v.355l.098.099-.553.395zM-28.438 236.933l.098.415.672.671a1.416 1.416 0 001.778-.722 1.42 1.42 0 00.138-.68l-.652-.652a1.478 1.478 0 00-1.62-.158 1.418 1.418 0 00-.414 1.126zM-1.554 270.929l.316.316c.344.149.73.17 1.087.059l.908-.909v-.316a1.367 1.367 0 00-1.343-1.343 1.396 1.396 0 00-1.343 1.343l.474.474-.099.376zM-3.707 216.231l-.711.711v.277a1.34 1.34 0 001.363 1.363 1.344 1.344 0 001.323-1.324l-.928-.928a1.365 1.365 0 00-.83-.119l-.217.02zM-21.801 285.131l.533-.533.099-.415.119-.118A1.478 1.478 0 00-22 282.84l-.237-.039-.454.059-1.008 1.007.02.217a1.398 1.398 0 001.324 1.324l.138-.138.415-.139zM-22.453 304.984l.474.474.375.138.119.119.316-.04.73-.731.139-.375a1.455 1.455 0 00-.494-.929 1.399 1.399 0 00-.988-.276 1.37 1.37 0 00-1.264 1.501l.593.119zM-2.265 288.43l.87.869.335.099.218.02.237-.04.454-.454c.17-.172.28-.393.316-.632l.119-.119.039-.276v-.237l-1.027-1.027-.356-.04-.217.02a1.455 1.455 0 00-1.043 1.248c-.02.192 0 .385.055.569zM-43.688 271.343l.316.079.099.099.316-.039 1.225-1.225a1.511 1.511 0 00-.573-1.047 1.42 1.42 0 00-1.146-.237l-.73.731a1.23 1.23 0 00-.1.731l.06.296.632.632-.099-.02zM5.518 281.912l-.376.375-.059.257.04.474.987.987.435.079.257-.02c.299-.089.57-.252.79-.474l-.257-.256.217.217.237-.474.06-.296-.04-.316-.77-.771s-1.067-.276-1.699.237l.178-.019zM-39.263 314.248l-.533.534a1.226 1.226 0 00.47 1.709c.195.105.415.156.636.147l.257-.257a1.312 1.312 0 001.086-1.086l-.02-.217-.138-.376-.237-.237a1.285 1.285 0 00-.691-.335l-.297.019-.533.099zM-18.64 288.213l-.672.671-.079.356-.099.099a1.48 1.48 0 001.6 1.323l.89-.889.118-.355v-.237l-.099-.099-.06-.257-.513-.513a1.667 1.667 0 00-1.304-.198l.218.099zM-36.379 293.033l.454-.455.119-.355.02-.217-.968-.968-.376-.099-.118-.118-.89.888a1.428 1.428 0 00-.197.909l1.047 1.047h.237l.139-.138.533-.494zM-41.476 224.251l-.118-.119-.138-.375-.85-.849a1.42 1.42 0 00-1.54 1.343l.118.118.118.356.91.909c.4.027.796-.1 1.105-.356.262-.277.404-.646.395-1.027zM-41.18 189.781a1.403 1.403 0 00-.355 1.146l.02.257.652.651c.25.089.519.114.78.072.263-.041.51-.147.721-.309a1.367 1.367 0 00.474-1.145 1.173 1.173 0 00-.415-.81l-.671-.672a1.507 1.507 0 00-1.205.81zM-24.586 219.174l.573.573.454.257.178.178a1.45 1.45 0 001.323-1.047l.08-.356-.04-.276-.119-.316-.592-.593-.277-.079a1.422 1.422 0 00-1.462 1.106l.139.138-.257.415zM-37.624 234.622l.119-.396a1.343 1.343 0 00-.731-1.047 1.435 1.435 0 00-.711-.118 1.372 1.372 0 00-1.363 1.363l.474.474a.95.95 0 00.395.395l.119.119c.18.049.371.049.553 0l.533-.534.612-.256zM-16.468 231.955l-.178.178a1.566 1.566 0 00-.415.414l-.039.277.02.494.612.612.454.217.159.158.316-.039.849-.849.099-.376.118-.118a1.394 1.394 0 00-1.145-1.146h-.277l-.474.119-.099.059zM-34.265 233.239l-.237.237-.02.217c.021.158.068.311.138.454l-.098.376-.119.118a1.428 1.428 0 00.632 1.027 1.31 1.31 0 001.126.139l.75-.751.08-.316.098-.099-.118-.118a.885.885 0 00.158-.158l.217-.218v-.237a1.256 1.256 0 00-1.363-1.165 1.292 1.292 0 00-.691.296 1.196 1.196 0 00-.553.198zM-54.118 250.8a1.229 1.229 0 00.632-1.225 1.43 1.43 0 00-.138-.494l-.652-.652a1.32 1.32 0 00-1.165.297 1.447 1.447 0 00-.494 1.244 1.287 1.287 0 001.185 1.185l.297-.059.335-.296zM-34.464 231.817l-.335-.336a1.332 1.332 0 00-1.294-.11 1.348 1.348 0 00-.78 1.038l.138.139.118.434.534.533.355.119.119.118.415-.019.592-.593.218-.454.04-.277-.14-.138.02-.454zM-25.297 204.241l.06.296.315.316a1.4 1.4 0 00.968.968 1.34 1.34 0 001.442-.77l.02-.257-.04-.474-1.086-1.087-.356-.118-.217-.02a1.31 1.31 0 00-1.106 1.146zM-16.882 218.898l-.454.454a1.39 1.39 0 00-.435 1.146c.007.122.034.242.08.355l.868.869.514-.079.612-.612.158-.395.139-.138-.02-.257a1.316 1.316 0 00-.514-.909 1.115 1.115 0 00-.948-.434zM-56.33 224.409l.869-.869a1.405 1.405 0 00-.553-.988 1.539 1.539 0 00-1.225-.237l-.988.988v.197a1.368 1.368 0 001.442 1.245l.119-.119.336-.217zM-29.782 222.631l-.197-.197a1.558 1.558 0 00-1.166.454c-.249.304-.37.694-.335 1.086l.671.672a1.347 1.347 0 001.087.138l.69-.691c.105-.222.146-.468.12-.711l-.08-.316-.493-.494-.297.059zM-32.033 211.332a1.433 1.433 0 00-1.264-.119l-.751.751-.08.277-.098.098a1.425 1.425 0 001.166 1.166h.276l.435-.435c.233-.289.346-.657.316-1.027a1.256 1.256 0 000-.711zM-45.822 246.414l.1-.375a1.285 1.285 0 00-.574-1.007 1.53 1.53 0 00-.85-.257l-.374-.099-.218.02-.355.118-.534.534a1.477 1.477 0 00-.217.573l.119.118.118.395.514.514c.176.141.388.23.612.256l.099-.098.118.118a1.439 1.439 0 001.086-.375l.475-.474-.118.039zM-33.14 245.624l.475-.474c.202-.28.294-.624.257-.968a1.312 1.312 0 00-.87-1.145 1.451 1.451 0 00-1.501.237l-.514.513.02.257a1.452 1.452 0 00.573 1.047 1.484 1.484 0 001.245.217l.316.316zM-22.532 243.155a1.424 1.424 0 00-1.284-.099l-.494.494a1.566 1.566 0 00-.257.889c.038.221.126.43.257.612l.691.692a1.436 1.436 0 001.146-.395 1.285 1.285 0 00.395-1.106 1.398 1.398 0 00-.454-1.087zM-6.472 231.836l-.672.672-.098.336-.1.098v.316l.85.85.375.099.119.118 1.086-1.086.04-.395-.02-.297a1.369 1.369 0 00-1.659-1.066l.079.355zM-49.021 221.841a1.328 1.328 0 001.206-.929c.055-.172.075-.353.058-.533l-.632-.632a1.399 1.399 0 00-1.936.316l-.375.376v.355a1.402 1.402 0 001.133 1.065c.181.03.367.024.546-.018zM-24.014 247.758a1.503 1.503 0 00-1.62-.237 1.337 1.337 0 00-.79 1.303l.395.395a1.444 1.444 0 001.087.336 1.496 1.496 0 001.086-.652l.139-.138.079-.356-.04-.236-.355-.356.02-.059zM-52.083 241.219l.593-.592.098-.297-.02-.217a1.395 1.395 0 00-.592-1.106 1.366 1.366 0 00-1.146-.197l-.83.829a1.201 1.201 0 00-.118.711l.06.297.69.691c.33.147.703.168 1.048.059l.217-.178zM-27.035 245.446l-.1-.375-.533-.533a1.443 1.443 0 00-.632-.237l-.118.118-.356.119-.513.513c-.128.19-.21.406-.238.633l-.098.098.138.376.395.395c.185.151.413.241.652.256l.119-.118.375-.099.494-.494a1.2 1.2 0 00.256-.612l.159-.04zM-28.22 242.049l.177-.178.158-.355.119-.119v-.277a1.5 1.5 0 00-.593-1.066 1.426 1.426 0 00-1.244-.178l-.77.77a1.706 1.706 0 00-.06.613 1.392 1.392 0 00.988 1.224l.217.02.434-.039.317-.316.256-.099zM-29.821 246.493l-.099-.375-.415-.415a1.481 1.481 0 00-1.62-.237 1.4 1.4 0 00-.77 1.324l1.185 1.185c.178.138.39.226.613.257l.118-.119.376-.138.513-.514c.112-.188.186-.396.218-.612l-.12-.356zM-38.868 247.916l-.138-.376-.514-.513a1.261 1.261 0 00-.612-.218l-.1.099-.394.119-.534.533a1.23 1.23 0 00-.237.553l.02.218.119.395.513.513c.183.13.392.218.613.257l.118-.118.395-.119.534-.533c.118-.17.193-.367.217-.573v-.237zM-19.628 238.908l-.671.672-.1.335.04.395.356.356a1.506 1.506 0 001.738 1.422l.928-.928.02-.494a1.648 1.648 0 00-.513-1.027 1.346 1.346 0 00-.584-.923 1.337 1.337 0 00-1.076-.183l-.138.375zM-22.157 234.839l.237-.237.02-.217a1.204 1.204 0 00-.632-.988 1.533 1.533 0 00-1.54 0 1.307 1.307 0 00-.593 1.264 1.173 1.173 0 00.474.869 1.45 1.45 0 001.106.277l.415-.415c.229-.121.41-.316.513-.553zM-46.256 229.584l-.02-.335-.06-.178-.177-.336-.139-.178-.197-.158-.178-.099-.375-.098-.198-.04-.158.04-.197.039-.336.178-.316.316-.079.198-.119.355-.02.217.02.297.06.217.177.336.119.158.198.158.197.079.356.118.217.02h.158l.178-.059.336-.178.177-.138.158-.198.1-.177.098-.376.02-.178zM-12.26 214.986l-.415.415a1.2 1.2 0 00-.217.573v.237l.118.395.593.593a1.334 1.334 0 001.066.158l.77-.771.08-.316-.099-.098v-.237l-.85-.85-.355-.118-.217-.02-.119.118-.355-.079zM-5.11 228.063l.83.83.336.099.118.118a1.388 1.388 0 001.055-.407 1.405 1.405 0 00.407-1.054l-1.007-1.008a1.406 1.406 0 00-1.284.296 1.342 1.342 0 00-.454 1.126zM8.58 207.697a1.396 1.396 0 00.612 1.008 1.367 1.367 0 001.186.158l.79-.79c.096-.21.137-.441.118-.672a1.453 1.453 0 00-.493-.889l-.178-.178a1.392 1.392 0 00-1.126.376 1.424 1.424 0 00-.909.987zM-42.029 219.115l.889.889c.142.033.289.046.435.039l.118-.118.376-.138.572-.573.158-.395.04-.237-.711-.712a1.373 1.373 0 00-1.343.198 1.52 1.52 0 00-.534 1.047zM-12.497 208.784l.079.316.77.77.316.079.75-.75c.139-.178.227-.39.258-.613l-.02-.256-.217-.455-.534-.533-.375-.099-.118-.119a1.323 1.323 0 00-.962 1.138c-.018.176 0 .353.053.522zM-31.817 239.086l-.967-.968-.356-.119-.217-.019h-.237l-.89.889-.098.335.099.099a1.405 1.405 0 00.553.988l.909.909.276.078.198.198.256.059.475-.039.572-.573.218-.454.079-.277a1.43 1.43 0 00-.672-1.106h-.198zM-19.964 228.221l-.415.415-.119.356a1.37 1.37 0 00.534.968 1.349 1.349 0 001.126.257l.79-.791.099-.335.118-.119a1.455 1.455 0 00-.553-.988 1.478 1.478 0 00-1.126-.257l-.118.119-.336.375zM-70.572 216.725l-.138-.139-.257-.059a1.293 1.293 0 00-1.263.924 1.4 1.4 0 00-.06.558l.059.296.77.77.296.06.435-.04.592-.593.218-.454.158-.158-.099-.099a1.227 1.227 0 00-.711-1.066zM-4.003 292.895l-.415-.415a1.258 1.258 0 00-1.284.059 1.285 1.285 0 00-.513.514l-.139.138-.138.375-.02.218v.237a1.759 1.759 0 00.652 1.205l.297.296.237.039c.31-.003.613-.1.869-.276l.77-.771.06-.256-.04-.277-.119-.118-.217-.968zM20.254 288.43l-.632.632-.059.297a1.425 1.425 0 001.166 1.165l.098.099.277-.04.533-.533c.221-.287.327-.646.296-1.007a1.204 1.204 0 00-.572-.968 1.418 1.418 0 00-1.245-.139l.138.494zM-25.198 266.05l.059-.297-.02-.494-.612-.612-.454-.217-.277-.04a3.338 3.338 0 01-1.086-.257l-.257-.019-.435.039-.553.553-.198.435-.158.158a1.231 1.231 0 00.652 1.007l.988.988.296.099h.356l.138.138a1.363 1.363 0 001.14-.362 1.362 1.362 0 00.42-1.119zM-30.295 267.037l-.474-.474a1.539 1.539 0 00-1.64.573l-.197.197-.04.593.158.158.02.217.118.435.534.533.414.138.238.04.316-.04 1.126-1.126.02-.572-.653-.652.06-.02zM-18.72 266.405l-.651.652-.119.435v.237a1.255 1.255 0 00.632.948 1.335 1.335 0 001.067.118l.83-.829.059-.336a1.341 1.341 0 00-1.205-1.205l-.138.138-.474-.158zM1.528 263.067l.691.691a1.118 1.118 0 001.028.158 1.268 1.268 0 00.75-.751l.138-.138.1-.375a1.312 1.312 0 00-.455-.889 1.17 1.17 0 00-.928-.257 1.395 1.395 0 00-1.324 1.561zM7.099 288.312l.276.276c.345.024.686-.09.948-.316a1.356 1.356 0 00.455-.968 1.286 1.286 0 00-1.166-1.442l-.395.119-.494.494-.158.355-.02.217.119.356.178.178.257.731zM14.9 236.222l.396-.119.494-.494.158-.355a.949.949 0 000-.474l-.119-.396-.85-.849a1.369 1.369 0 00-.947.316 1.17 1.17 0 00-.435.909 1.364 1.364 0 00.778 1.33c.165.079.343.123.526.132zM15.256 297.023l-.079-.356-.256-.256a1.342 1.342 0 00-1.067-.711l-.395.039-.929.928v.316l.1.336.79.79a.282.282 0 00.118.119 1.34 1.34 0 001.501-1.067l.217-.138zM6.387 271.679l-.099-.296-.612-.612-.434-.119a1.388 1.388 0 00-1.37.936c-.075.213-.1.44-.072.664l.75.751c.126.049.26.075.395.079a1.316 1.316 0 001.304-1.304l.099-.099h.04zM4.155 298.366l-.711-.711-.395-.079a1.206 1.206 0 00-1.087.455 1.45 1.45 0 00-.237 1.303l.929.929a1.43 1.43 0 001.383-.751l.158-.158.04-.435-.02-.256-.06-.297zM6.545 259.037l.257.059a1.265 1.265 0 00.928.534 1.768 1.768 0 001.265-.474l.474-.475.02-.256-.1-.099-.098-.336-.119-.119a1.32 1.32 0 00-.968-.533 1.312 1.312 0 00-.69-.691l-.12-.119-.395.04-.908.908a1.648 1.648 0 00.415 1.442l.039.119zM15.573 246.809l.415.415.276.04a1.34 1.34 0 001.324-.771 1.478 1.478 0 00-.277-1.698l-.395-.396-.296-.059a1.34 1.34 0 00-1.324.77 1.454 1.454 0 00.277 1.699zM-12.636 313.972l-.118-.119-.356.079a1.314 1.314 0 00-.79.79l-.119.119.1.099c.093.285.256.543.473.75l.474.474h.237l.119-.118.316-.119.138-.138a1.652 1.652 0 00.474-1.541 1.288 1.288 0 00-.948-.276zM25.805 269.012l.968.968a1.426 1.426 0 001.442-.849l.02-.217v-.435l-.573-.573a1.944 1.944 0 00-.533-.296l-.217-.02-.415.059-.553.554a1.252 1.252 0 00-.316 1.027l.177-.218zM24.58 283.749a1.294 1.294 0 001.067-.395 1.373 1.373 0 00.376-1.166l-1.087-1.086-.118.118-.376.099-.968.968v.237l.119.118.138.376.711.711.139.02zM4.254 266.267l.099.098.375-.059.336-.336c.246-.127.446-.327.573-.573l.02-.217-.04-.355-.711-.712a1.622 1.622 0 00-.73-.256l-.297.019-.89.889a1.48 1.48 0 00.179 1.166 1.398 1.398 0 001.086.336zM-3.588 333.962a1.367 1.367 0 00-1.284-.257l-.237.237a1.343 1.343 0 00-.395 1.541l.83.83.335.059a1.402 1.402 0 001.423-.869l.138-.138.02-.573-.83-.83zM27.089 275.906l-.158.158-.04.474.988.988a.49.49 0 00.138.138 1.394 1.394 0 001.521-1.086l.02-.217-.257-.257a1.176 1.176 0 00-.928-.494 1.363 1.363 0 00-1.284.296zM24.62 307.018a1.705 1.705 0 00-.257.81l1.205 1.205a1.5 1.5 0 001.363-.731l.158-.158.02-.454-.04-.237-.395-.395a1.614 1.614 0 00-.553-.553l-.256-.02-.455.059-.573.573-.217-.099zM26.28 273.477c.125.011.251.005.375-.02l.395-.395a1.367 1.367 0 00.513-1.264 1.57 1.57 0 00-.889-1.166l-.276-.039-.297.296a1.484 1.484 0 00-.474 1.58 1.364 1.364 0 00.652 1.008zM33.864 310.297l-.059-.335-.83-.83a1.367 1.367 0 00-1.066.355 1.324 1.324 0 00-.435 1.067l.1.296.69.692c.114.045.234.072.356.079a1.316 1.316 0 001.363-1.008v-.237l-.118-.079zM21.459 295.324l.691-.691-.395-.395a1.45 1.45 0 00-1.086-.652 1.426 1.426 0 00-1.482 1.284v.237l.139.415.572.573c.212.141.458.223.712.237l.217-.02.375-.138.257-.85zM17.924 277.408a1.416 1.416 0 00-.198.474 3.04 3.04 0 00-.514.513c-.24.038-.465.141-.651.297l-.89-.889a1.526 1.526 0 00-.434-.119 1.314 1.314 0 00-1.323.889l-.02.217v.356l1.146 1.146.118-.119.336-.099.296-.296.296.296a1.288 1.288 0 001.107.79l.335.336.139-.138.276-.276.494-.494a1.83 1.83 0 00.474-.474l.336-.336c.292-.127.525-.36.652-.652v-.316l-.04-.237-.138-.415-.533-.533-.435-.119-.217-.02c-.208.03-.409.097-.593.198a.34.34 0 00-.02.02zM61.638 265.338l.376-.138.434-.434a1.118 1.118 0 00.316-.988 1.204 1.204 0 00-.573-.85l-.414-.414a1.225 1.225 0 00-1.363 1.165c-.028.3.05.6.217.849a1.391 1.391 0 001.007.81zM32.047 273.832l-.356.119a1.282 1.282 0 00-.987 1.343l.415.415c.212.305.528.522.889.612l.256-.256a1.568 1.568 0 00.474-1.66 1.316 1.316 0 00-1.185-.948l.494.375zM30.408 283.611l.553.553a1.392 1.392 0 001.013-.409 1.4 1.4 0 00.41-1.014 1.506 1.506 0 00-.356-1.066 1.343 1.343 0 00-.948-.475l-.396.119-.158.158a1.234 1.234 0 00-.79.79 1.287 1.287 0 00.277 1.185l.395.159zM33.49 278.593l.888.889a1.78 1.78 0 00.711-.119l.533-.533a1.368 1.368 0 00.08-1.383 1.346 1.346 0 00-1.048-.731 1.31 1.31 0 00-.928.297 1.87 1.87 0 00-.277.276 1.371 1.371 0 00-.296 1.166l.336.138zM37.46 273.358l-.178.494v.316l.059.336.908.908a1.353 1.353 0 001.024-.256 1.378 1.378 0 00.537-.909c.139-.296.167-.633.079-.948a1.398 1.398 0 00-.79-.79L39 272.41h-.316l-.454.454c-.17.184-.305.399-.395.632l-.376-.138zM29.578 247.145a1.31 1.31 0 001.323-1.837l-.691-.691-.237-.039-.257.059-.514.513a1.315 1.315 0 00-.059 1.284c.066.277.218.526.435.711zM50.319 311.858l-.395.039-.494.494c-.17.162-.292.367-.356.593l-.02.217.04.395.514.514c.148.152.331.268.533.336l.119.118.414-.059.494-.494c.143-.16.25-.349.316-.553l.02-.217-.04-.395-.513-.514a1.338 1.338 0 00-.534-.336l-.098-.138zM35.998 305.043l-.395-.395a1.346 1.346 0 00-.435-.435l-.257-.019-.493.098-.553.554-.178.375-.119.118a1.313 1.313 0 001.205 1.442l.297-.019 1.066-1.067-.118-.119-.02-.533zM46.606 291.828l.099.099.217-.218.119.119.355-.158.731-.731a1.344 1.344 0 00-.7-1.329 1.321 1.321 0 00-.505-.152 1.362 1.362 0 00-1.363.888 1.395 1.395 0 001.047 1.482zM37.539 290.149l-.79-.79-.257-.06a1.374 1.374 0 00-1.166.455 1.392 1.392 0 00-.296 1.244l.118.119.119.355.83.83.237.039.118-.118.376-.099.849-.849v-.237l.02-.218-.1-.098-.058-.573zM5.261 293.862l-.118-.118a1.368 1.368 0 00-1.165-.889 1.458 1.458 0 00-1.403.81 1.481 1.481 0 00.217 1.639l.138.139.376.178.118.118.257.02.237-.237a1.373 1.373 0 00.929-1.284l.414-.376zM55.95 328.589l.02.257.098.454.336.336a1.318 1.318 0 001.047.652h.276l.593-.593c.207-.299.292-.667.237-1.027l-.928-.928a2.573 2.573 0 00-.613-.138 1.37 1.37 0 00-1.067.987zM-42.76 298.86a1.312 1.312 0 00-1.244.731l-.06.296.08.395c-.005.31.1.61.296.85a1.261 1.261 0 00.909.434l.217-.217a1.26 1.26 0 00.79-1.422 1.427 1.427 0 00-.988-1.067zM31.316 224.903a1.455 1.455 0 001.343 1.343l.514-.04.434-.434a1.288 1.288 0 00.396-1.264l-.83-.83a1.17 1.17 0 00-.691-.257l-.909.909-.04.356-.217.217zM41.47 332.599l-.77-.77a1.424 1.424 0 00-1.502 1.225 1.286 1.286 0 00.987 1.461l.316-.039.277-.277a1.365 1.365 0 00.711-1.659l-.02.059zM28.215 296.272a1.333 1.333 0 001.165-.573l.158-.158.04-.474v-.197l-1.205-1.205a1.315 1.315 0 00-1.402.889l-.12.118.02.336.455.454a1.346 1.346 0 00.889.81zM6.526 293.902l.553.553a1.59 1.59 0 001.324-.454l.217-.217c.343-.169.621-.447.79-.79a1.902 1.902 0 00-.435-1.976l-.158-.158-.355-.118-.119-.119-.256-.02-.139.139-.336.138-.138.138a.62.62 0 00-.158.158 1.344 1.344 0 00-.79.79l-.316.316c-.079.39 0 .794.217 1.126l.198.198-.099.296zM-17.119 248.014l.988.988c.371-.018.723-.173.987-.435a1.253 1.253 0 00.277-1.027 1.314 1.314 0 00-.988-.987l-.098-.099-.356.039-.672.672a1.424 1.424 0 00-.237 1.027l.1-.178zM-2.522 262.533l-.592.593a1.479 1.479 0 00.256 1.205 1.32 1.32 0 001.008.533l.118-.118.297-.099.375-.375a1.376 1.376 0 00.356-1.067l-.692-.692a1.58 1.58 0 00-.71-.237.76.76 0 00-.297.02l-.297.297.178-.06zM15.375 275.018l.257-.06.434-.197.474-.474.08-.277v-.513l-.08-.277-.217-.217a1.388 1.388 0 00-.395-.395l-.198-.198-.572-.02-.297.06-.237.237a1.48 1.48 0 00-.454.968l.138.138.139.375.474.474c.127.152.28.279.454.376zM16.205 264.746l-.118-.119a1.538 1.538 0 00-1.087.376 1.44 1.44 0 00-.415 1.165l.731.731.04.237 1.086 1.086.237-.039.376-.139.493-.493.158-.395a1.398 1.398 0 00-.414-1.087 1.563 1.563 0 00-.297-.849l-.434-.435-.356-.039zM31.751 295.818l-.118.119-.099.296-.099.099a1.48 1.48 0 001.462 1.461l.099.099.375-.138.474-.474.198-.435.158-.158-.099-.099-.04-.276-.553-.553a1.341 1.341 0 00-.81-.376c-.2.002-.399.029-.592.08l-.494.493.138-.138zM12.076 288.075l.138.138a1.366 1.366 0 001.403-.929l.296-.296-.02-.257-.533-.533a1.596 1.596 0 00-.968-.415h-.276l-.81.81a1.566 1.566 0 00.197 1.304l.455.454.118-.276zM11.148 277.427l.296.02.534-.099.158-.158a1.79 1.79 0 00.474-.237c.186-.029.367-.089.533-.178l.81-.809a1.733 1.733 0 00-.336-1.403l-.355-.356a1.483 1.483 0 00-.889-.612l-.099-.099-.276.04-.514.514a1.616 1.616 0 00-.691.691l-.06.257-.395.395a1.532 1.532 0 00.31 1.694c.142.146.312.261.5.34zM11.78 282.88l.909-.909a1.399 1.399 0 00-1.067-1.738l-.217.019-.138.139a1.42 1.42 0 00-.455 1.521 1.309 1.309 0 001.186.948l-.218.02zM16.383 290.643l-.928-.929a1.398 1.398 0 00-1.502 1.107l.06.375.651.652c.214.136.459.218.712.237l.296-.02.968-.968.02-.257-.1-.099-.177-.098z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M1.627 239.54a1.393 1.393 0 001.066-.395 1.59 1.59 0 00.435-1.304l-.454-.454a1.48 1.48 0 00-2.134-.079l-.474.474a1.51 1.51 0 00.257 1.244 1.394 1.394 0 001.304.514zM5.42 311.799l.631.632a1.521 1.521 0 001.186-.395 1.427 1.427 0 00.395-1.106l-.316-.317a1.257 1.257 0 00-1.047-1.047c-.18-.008-.36.012-.534.06l-.79.79a2.212 2.212 0 00-.158.632l.119.119.098.335.139.139.276.158zM9.29 317.132l-.888-.889a1.728 1.728 0 00-1.699.83l-.119.118-.059.297.02.217.415.415a1.37 1.37 0 00.968.494c.306.018.613-.044.889-.178l.138-.138-.04.039.415-.415v-.237l-.099-.099.06-.454zM6.763 285.052l.158.158a1.485 1.485 0 001.126-.414 1.511 1.511 0 00.415-1.166l-1.008-1.007-.276-.04a1.435 1.435 0 00-1.146.395 1.455 1.455 0 00-.395 1.185l.908.909.218-.02zM-4.043 314.782l.138-.139a1.2 1.2 0 00.731-1.126l-.06-.296-1.105-1.106a1.308 1.308 0 00-1.146.513 1.401 1.401 0 00-.316.791 1.343 1.343 0 001.225 1.461l.533-.098zM-22.157 334.673l-.553-.553a.94.94 0 00-.454-.098 1.362 1.362 0 00-1.363.73 1.593 1.593 0 00.217 1.758l.336.336.296.06a1.372 1.372 0 001.225-.514 1.564 1.564 0 00.198-1.541l.098-.178zM10.417 271.758l-1.066-1.066-.336-.06a1.421 1.421 0 00-1.166.494 1.567 1.567 0 00-.256 1.482l.71.711.297.099h.277l.474-.119.177-.178a1.616 1.616 0 00.889-1.363zM-42.068 289.951l.395-.118.197-.198a.952.952 0 00.395-.395l.178-.178-.079-.474.099-.099-.138-.375-.731-.731a1.263 1.263 0 00-.948.316 1.346 1.346 0 00-.407 1.416 1.345 1.345 0 001.138.935l-.1-.099zM-19.036 309.31a1.566 1.566 0 001.58-1.107l.02-.256-.039-.435-.988-.988-.316-.039-.138.138-.375.138-.514.514-.118.316.197.198-.276-.04-.08.474.04.277.928.928.08-.118zM9.41 298.682l-.257-.019-.099.098-.356.119-.849.849-.04.237.12.119.098.336.83.829a1.311 1.311 0 001.066-.355 1.45 1.45 0 00.435-1.067l-.139-.336-.81-.81zM-4.536 325.547a1.256 1.256 0 001.185-.948v-.276l-.158-.514a1.289 1.289 0 00-1.106-.869c-.167 0-.332.02-.494.059l-.533.533-.1.297a1.312 1.312 0 00.89 1.481l.256.02.06.217zM58.063 277.013l.81.809.296.02.217-.02.336-.098.81-.81a1.425 1.425 0 00-.691-1.482l-.198-.197h-.514l-.296.059-.593.593a1.673 1.673 0 00-.236.79l.118.118-.06.218zM-5.761 249.615l.296.059h.553l.296-.099.435-.435.158-.355.138-.138a1.483 1.483 0 00-.217-1.166 1.311 1.311 0 00-.929-.494l-.276.04-.119.118a1.481 1.481 0 00-.533 1.205 1.482 1.482 0 00.198 1.265zM10.674 255.619l.04.317a1.508 1.508 0 001.244 1.244l.316-.079.75-.751a1.646 1.646 0 00-.059-1.402 1.255 1.255 0 00-.553-.553 1.316 1.316 0 00-1.146-.869l-.335.019-1.047 1.047v.237l.79.79zM50.833 295.779h-.316l-.336.098-.81.81c-.061.395.03.797.257 1.126a1.446 1.446 0 001.066.553l.218-.217.336-.138.632-.632v-.237l-.02-.218-.119-.355-.71-.711-.198-.079zM21.143 342.93h-.276l-.435.119-.474.474a1.509 1.509 0 00-.375.612l-.119.119.04.355.79.79c.146.076.309.117.474.119a1.398 1.398 0 001.521-1.205 1.34 1.34 0 00-1.146-1.383zM6.743 323.868a1.306 1.306 0 00-.85-.849h-.236l-.395.039-.573.573a1.34 1.34 0 00-.336 1.008l.83.829.355.119a1.315 1.315 0 001.422-1.027l-.02-.218-.197-.474zM-17.93 322.703l-.414-.415a1.514 1.514 0 00-.553-.356l-.257.02-.474.079-.929.929a1.339 1.339 0 001.205 1.481l.297-.02 1.066-1.066-.02-.218.08-.434zM28.491 286.791l-.454.454a1.34 1.34 0 00-.079 1.383 1.258 1.258 0 002.074.336l.573-.573a1.367 1.367 0 00-.296-1.047 1.45 1.45 0 00-.968-.494l-.85-.059zM31.711 253.625l-.592.592c-.12.213-.194.449-.218.692l.04.316.277.513.454.455.336.059.572-.138.119.118a1.615 1.615 0 00.988-1.422l-.119-.119-.118-.355-.83-.83-.296-.02-.218.02-.335.099-.06.02zM29.598 322.189l-.415-.415a1.15 1.15 0 00-.553-.316l-.217-.019-.395.039-.395.395-.218-.217-.217-.02-.099-.099h-.237l-.81.81a1.37 1.37 0 00.04 1.264 1.316 1.316 0 001.047.613 1.33 1.33 0 001.027-.316l.316.316a1.371 1.371 0 001.422-.83l.139-.138-.04-.435-.395-.632zM25.983 318.693a1.593 1.593 0 00-.277-1.225 1.595 1.595 0 00-1.027-.593l-.257.02-.375.376a1.731 1.731 0 00-.395 1.264l1.106 1.106.395.039.119-.118.335-.099.139-.138-.02.02.257-.257v-.395zM-3.627 296.509a1.45 1.45 0 001.422-1.422 1.368 1.368 0 00-1.225-1.462l-.375.139-.494.494-.178.375-.316.316.138.375.494.494.534.691zM-1.85 257.674l.1.336.631.632.316.118.138.139.139-.139.375-.138.494-.494.158-.395.119-.118a1.256 1.256 0 00-2.331-.711l-.198.474.06.296zM11.977 305.833l-.217.02-.237.237a1.507 1.507 0 00-.454 1.6 1.426 1.426 0 001.225.948l.454-.02.652-.652a1.448 1.448 0 00.043-1.351 1.458 1.458 0 00-1.09-.802l-.376.02zM15.612 321.083l-.73-.731-.277-.04a1.542 1.542 0 00-1.62 1.423v.237l.08.434.77.771.316.079h.237l.257-.06.612-.612c.146-.306.246-.632.296-.968l.099-.099-.04-.434zM16.837 264.588l.869.869.336.059a1.449 1.449 0 001.442-.849l.079-.474-.02-.218-.671-.671a1.78 1.78 0 00-.712-.237l-.296.019-.474.475a1.585 1.585 0 00-.395 1.224l-.158-.197zM14.467 303.818v.198l.079.355.336.336c.17.197.387.347.632.435l.237.039.415-.059.968-.968.02-.257-.02-.217-.139-.375-.494-.494-.395-.158-.098-.099a1.285 1.285 0 00-1.541 1.264zM11.464 265.575a1.241 1.241 0 00.909.474l.395-.118.415-.415a.86.86 0 00.316-.632l.908-.909a1.84 1.84 0 00-.276-1.145l-.613-.613a1.313 1.313 0 00-1.303.79l-.158.158.04.277a1.279 1.279 0 00-.574.968 1.48 1.48 0 00-.059 1.165zM39.218 316.006l.197.198c.008.122.034.242.08.355a1.254 1.254 0 00.888.889l.1.099.375-.059 1.027-1.027a1.427 1.427 0 00-.198-.949l-.099-.098a1.29 1.29 0 00-1.165-1.166l-.138.139a1.586 1.586 0 00-1.067 1.619zM17.055 331.335a1.54 1.54 0 00-.415-.415l-.139-.138-.414.059-.573.573a1.344 1.344 0 00-.336 1.047l1.086 1.086a1.448 1.448 0 001.442-.849l.04-.237v-.435l-.691-.691zM76.868 278.87h.276l.909-.909a1.595 1.595 0 00-.217-1.245 1.427 1.427 0 00-1.087-.572l-.257.019-.908.909a1.426 1.426 0 00.237 1.225 1.482 1.482 0 001.047.573zM-2.7 251.768l.119.118a1.28 1.28 0 00.869.869l.217.02.395-.079.534-.533a1.347 1.347 0 00.355-.988l-.85-.849-.355-.119-.098-.099a1.312 1.312 0 00-1.383.988l.197.672zM36.571 266.978l.277.039.513-.039.988-.988a1.283 1.283 0 00-1.205-1.442h-.316l-1.047 1.047a1.338 1.338 0 00.79 1.383zM-12.873 300.5c.155-.042.302-.109.435-.198a1.281 1.281 0 00.553-1.383 1.23 1.23 0 00-1.166-.928 1.175 1.175 0 00-1.126.533.427.427 0 00-.138.138l-.138.139-.02.217.04.277.217.454.454.454.356.079.099.099.434.119zM-8.487 282.445c.146.006.292-.007.434-.04l.672-.671a1.428 1.428 0 00-.384-1.863 1.46 1.46 0 00-.643-.271h-.238l-.335.336a1.508 1.508 0 00-.455 1.561 1.456 1.456 0 00.949.948zM29.598 261.743l.335-.336c.217-.303.309-.678.257-1.047l-.909-.908-.316-.079a1.368 1.368 0 00-1.461 1.224 1.344 1.344 0 001.165 1.482l.099-.099.316-.079.119-.118.395-.04zM-24.685 254.553l.75.751.277.039a1.37 1.37 0 001.365-1.22 1.369 1.369 0 00-1.069-1.486l-.118.118-.395.119-.534.533a1.51 1.51 0 00-.257.771l-.02.375zM40.087 284.045l-.514-.514-.395-.197-.138-.138a1.5 1.5 0 00-1.363.73l-.138.139-.04.474 1.126 1.126a1.394 1.394 0 001.521-1.087v-.237l-.059-.296zM59.663 288.173l.889-.888.02-.336-.08-.316-.81-.81a1.453 1.453 0 00-.987.355c-.26.227-.423.545-.455.889a1.424 1.424 0 001.285 1.521l.138-.415zM46.863 265.417l.355-.118.474-.474.139-.336.256-.257-.434-.434a1.479 1.479 0 00-.494-.494l-.316-.316-.218-.02-.256.059-.435.198-.533.533-.099.336-.02.217a1.48 1.48 0 001.58 1.106zM22.842 303.344l.237-.237.04-.316-.099-.099-.099-.336-1.066-1.066a1.563 1.563 0 00-1.126.375 1.455 1.455 0 00-.415 1.086l.711.712c.212.141.457.223.711.237.092.01.185.01.277 0l.75-.751.08.395zM41.608 256.647l.553.553.474.039.237-.039.83-.83.178-.415.02-.256a1.728 1.728 0 00-.356-.83l-.553-.553-.336-.02-.277.079-.474.237-.217.217s-.454 1.008-.06 1.719l-.019.099zM65.885 306.248l.533.533c.194.183.445.295.711.316a1.253 1.253 0 001.324-1.086l-.02-.217-.119-.119a1.337 1.337 0 00-1.086-1.086l-.118-.119-.395.079-.336.336c-.197.17-.347.387-.435.632l-.02.296-.04.435zM43.702 281.497l1.047 1.047a1.482 1.482 0 001.54-1.343l-.02-.218-.118-.395-.987-.987-.238.039-.513.514a1.65 1.65 0 00-.395 1.224l-.316.119zM45.46 299.808l-.968-.968-.118.119-.376.099-.849.849.118.119.1.335.434.435c.235.197.524.321.83.355l1.046-1.046.02-.257-.118-.119-.119.079zM-23.184 293.151h-.237l-.356.079-.85.85a1.42 1.42 0 00.533 1.52c.154.113.33.194.515.238l.395-.079.85-.85a1.647 1.647 0 00-.179-1.165 1.485 1.485 0 00-.671-.593zM-57.1 287.008a1.477 1.477 0 001.067.553h.236l.672-.672a1.617 1.617 0 000-1.501 1.28 1.28 0 00-1.027-.671 1.226 1.226 0 00-.89.256l-.69.692a1.478 1.478 0 00.632 1.343zM-25.317 275.65l.276-.04.929-.928a1.396 1.396 0 00-.81-1.482l-.356-.118-.296-.02-.316.079-.79.79a1.395 1.395 0 00.85 1.6l.138-.138.375.257zM-12.319 291.097l-.276-.277a1.316 1.316 0 00-1.304 1.304 1.367 1.367 0 001.205 1.521l.355-.118.198-.198c.153-.094.281-.222.375-.375a.53.53 0 00.139-.138c.07-.183.104-.378.098-.573l-.75-.751-.04-.395zM-11.352 269.783l.198.197c.104.169.246.312.415.415l.178.178.493.059.257-.059.929-.928.02-.336-.1-.099-.118-.355-.85-.85a1.426 1.426 0 00-1.402.849l-.158.158.04.475.098.296zM-13.13 287.561l.514.514a1.15 1.15 0 00.454-.059l.356.157.118.119a1.405 1.405 0 001.107-.435 1.248 1.248 0 00.336-1.086l-1.067-1.067-.119.119-.434-.435a1.263 1.263 0 00-1.383 1.146c-.021.265.048.53.197.751l-.079.276zM.758 310.179a1.286 1.286 0 00-1.106-.83 1.481 1.481 0 00-.514.04l-.652.652a1.336 1.336 0 00.119 1.224 1.502 1.502 0 001.106.711 1.285 1.285 0 001.363-1.047l-.02-.256-.296-.494zM-14.532 287.541l-.257.257a1.346 1.346 0 00-.336 1.245 1.398 1.398 0 00.909.908l.257.02.415-.059.928-.929.04-.395-.04-.276-.948-.948-.119.118-.85.059zM-40.172 273.793a.84.84 0 00.316 0l.296-.296a1.537 1.537 0 00.988-.988 1.342 1.342 0 00-.533-1.521l-.158-.158-.474-.04h-.198l-.355.119-1.047 1.047a1.285 1.285 0 001.165 1.837zM-24.29 267.985l.098.099.1.296.236.237a1.264 1.264 0 001.028.633.782.782 0 00.375-.02l.968-.968.02-.533-.08-.316-.81-.81-.236-.04a1.29 1.29 0 00-.988.356 1.195 1.195 0 00-.711 1.066zM-25.555 307.809l-.888-.889a.277.277 0 00-.1-.099 1.391 1.391 0 00-1.067.355 1.517 1.517 0 00-.434 1.186l.1.098.118.316.77.771a1.368 1.368 0 001.482-1.205v-.198l.02-.335zM-22.828 281.339l-.297.296a1.453 1.453 0 00.238 1.225 1.435 1.435 0 001.007.533l.099-.099.316-.118.395-.395a1.258 1.258 0 00.316-1.027l-.711-.712a1.424 1.424 0 00-.711-.237.822.822 0 00-.297.02l-.691.691.336-.177zM-34.325 281.655a1.452 1.452 0 00-.316 1.224l.929.929a1.4 1.4 0 001.422-.83l.099-.099.04-.276-.415-.415a1.452 1.452 0 00-.929-.494 1.29 1.29 0 00-.83-.039zM-2.087 301.211l-.276-.277a1.405 1.405 0 00-1.087.415c-.219.21-.358.489-.395.79a1.412 1.412 0 00-.158.356l-.02.217.06.375.513.514c.142.147.319.256.514.316l.119.119.395-.04.474-.474c.15-.166.264-.361.336-.573l.237-.237c.084-.384.02-.786-.178-1.126l-.474-.474-.06.099zM-.724 288.549l-.455-.455a1.48 1.48 0 00-.928-.414 1.387 1.387 0 00-1.284.651 1.482 1.482 0 000 1.541l.593.593.276.039a1.507 1.507 0 001.146-.395 1.48 1.48 0 00.395-1.185l.257-.375zM-1.316 277.684a1.399 1.399 0 00-.336 1.245l.474.474c.215.234.512.375.83.395.23.022.462-.019.671-.119l.77-.77a1.482 1.482 0 00-.177-1.205 1.308 1.308 0 00-1.027-.593 1.455 1.455 0 00-1.205.573zM-9.692 259.827l.138.336.771.77.336.02.118-.118.376-.099.83-.83.019-.257-.138-.138-.158-.395-.514-.514-.375-.138h-.277a1.377 1.377 0 00-1.002.4 1.38 1.38 0 00-.4 1.003l.276-.04zM-26.858 300.184a1.369 1.369 0 00-1.205-1.521l-.692.691a1.333 1.333 0 00-.413.968 1.338 1.338 0 00.413.968l.395.395.336.059a1.372 1.372 0 001.14-1.023c.044-.175.053-.358.026-.537zM2.95 279.936a1.458 1.458 0 00-.514 1.541 1.314 1.314 0 001.067 1.067l.217-.02.119-.119a1.314 1.314 0 00.533-1.007 1.65 1.65 0 00-.454-1.205l-.593-.593-.217.02-.316.119.158.197zM-8.27 306.485l-.81-.81-.217-.02a1.39 1.39 0 00-1.47.926c-.07.197-.093.407-.07.615l.632.632c.207.138.444.226.691.257l.296-.02.672-.671c.193-.298.27-.657.217-1.008l.06.099zM.145 282.504l-.395.04-.513.513c-.16.158-.276.356-.336.573l-.02.217.06.376.493.494c.153.161.343.283.553.355l.119.119.395-.04.494-.494c.167-.164.29-.368.356-.592l.02-.218-.06-.375-.494-.494a1.5 1.5 0 00-.553-.355l-.119-.119zM-3.41 283.097l-.534-.534-.257-.059a1.433 1.433 0 00-1.126.415 1.56 1.56 0 00-.434 1.185l.85.85c.196.082.401.142.612.177a1.392 1.392 0 001.343-.79l.138-.138.04-.435-.297-.296-.336-.375zM.718 285.447l-.395.04-.375.375a1.426 1.426 0 00-.494 1.561 1.316 1.316 0 001.166.968l.355-.079.731-.731a1.51 1.51 0 00.356-.553l.098-.099-.059-.415-.494-.494a1.15 1.15 0 00-.553-.316l-.336-.257zM.58 294.593l-.415.06-.493.493c-.16.155-.281.344-.356.554l-.02.217.06.375.414.415c.149.153.332.268.534.336l.217.02.395-.04.514-.514c.154-.158.275-.346.355-.553l.02-.217-.08-.395-.513-.514a1.347 1.347 0 00-.533-.335l-.099.098zM-5.03 274.129l1.007 1.007.375.02.218-.02.197-.197a1.518 1.518 0 001.175-.353 1.512 1.512 0 00.524-1.109l-.079-.316-.73-.731a1.862 1.862 0 00-.455-.099 1.473 1.473 0 00-1.126.336 1.344 1.344 0 00-1.403 1.402l.297.06zM-9.455 275.907l-.454-.455a1.324 1.324 0 00-1.067.474 1.429 1.429 0 00-.276 1.502 1.345 1.345 0 001.145.829 1.23 1.23 0 00.929-.336 1.335 1.335 0 00.454-1.047l-.118-.118a1.28 1.28 0 00-.613-.849zM-18.779 298.742l-.316-.04-.197.04-.356.118-.178.099-.197.158-.119.158-.177.336-.06.217v.356l.139.375.079.158.197.198.158.118.336.178.178.059.336.02.197-.04.375-.098.159-.119.197-.158.119-.158.177-.336.06-.178.02-.177-.04-.198-.119-.355-.098-.178-.159-.198-.177-.138-.336-.178-.198-.039zM-27.332 262.77l.316.316c.109.165.25.307.415.415l.375-.059.415-.059.573-.573a1.339 1.339 0 00.316-1.027l-.988-.988-.217-.02-.118.119-.356.118-.85.85.1.099.02.809zM-13.268 257.951l.218-.02.355-.119.81-.81a1.422 1.422 0 00-1.146-1.738l-.099-.099-.316.079-.79.791a1.534 1.534 0 00.08 1.343c.21.3.528.506.888.573zM-30.987 240.982a1.333 1.333 0 001.12-.382 1.34 1.34 0 00.382-1.119l-.79-.79a1.148 1.148 0 00-.613-.218 1.365 1.365 0 00-.948.317l-.454.454a1.536 1.536 0 00.197 1.185 1.364 1.364 0 001.106.553zM-28.398 292.875l.889-.889c.051-.133.084-.273.098-.415l.119-.119-.08-.394-1.007-1.008-.691.691a1.43 1.43 0 00-.04 1.383c.144.325.395.59.712.751zM-33.515 261.96l.336-.019 1.047-1.047-.1-.336-.394-.395a1.289 1.289 0 00-.573-.336l-.257-.02-.474.119-.553.553-.158.356-.119.118a1.345 1.345 0 001.245 1.007zM-6.947 286.158l-1.52 1.522.118.118.138.375.83.83a1.283 1.283 0 001.047-.375c.074-.057.14-.124.197-.198l.909-.909.197-.197.178-.178.04-.474-.04-.237-.928-.928a1.415 1.415 0 00-1.205.493l.04.158zM-14.828 273.417a1.312 1.312 0 001.067-.355 1.429 1.429 0 00.454-1.047l-1.126-1.126a1.394 1.394 0 00-1.067.356 1.44 1.44 0 00-.434 1.066l1.106 1.106zM-35.55 320.53l-.336.336a1.339 1.339 0 001.265 1.54h.276l1.047-1.047.04-.434-.04-.277-.928-.928a1.204 1.204 0 00-1.324.81zM50.813 267.867l-.414.415a1.284 1.284 0 00-.178 1.284c.118.217.297.395.514.513l.572.573.218.02a1.702 1.702 0 001.303-.435l.455-.454a1.482 1.482 0 00-.119-.909l-1.205-1.205-.296.02-.118.119-.732.059zM50.556 243.195l.948.948a1.486 1.486 0 001.423-.869l.04-.277-.495-.494a1.37 1.37 0 00-.968-.494 1.232 1.232 0 00-.691-.296 1.396 1.396 0 00-.316 1.185l.06.297zM20.768 284.203l-.178-.178-.513-.039-.277.039-.948.948c-.104.359-.25.704-.435 1.028l-.158.158-.02.454 1.028 1.027a1.39 1.39 0 001.125-.454l.475-.474-.06.059.593-.592.079-.356.138-.138a1.37 1.37 0 00-.85-1.482zM20.886 289.378l-.494.494a1.621 1.621 0 00.277 1.739l.217.217.573.099.257-.02.256.02.415-.06.968-.968.04-.316-.139-.138-.177-.415-.692-.691-.553-.119-.296.02-.178.178-.474-.04zM22.23 277.882l.533.533.415.178.139.138a1.419 1.419 0 001.046-.454 1.284 1.284 0 00.297-1.047l-.83-.83-.336-.099-.099-.098a1.312 1.312 0 00-1.382.987v.237l.217.455zM22.368 257.358l.395-.119.414-.415a1.11 1.11 0 00.336-1.007 1.284 1.284 0 00-.573-.849l-.414-.415a1.314 1.314 0 00-.929.296 1.226 1.226 0 00-.434.869 1.396 1.396 0 001.205 1.64zM49.055 305.576l-.376-.138a1.314 1.314 0 00-.888.455 1.367 1.367 0 00-.297 1.046 1.317 1.317 0 00.385.939 1.301 1.301 0 00.939.385l.75-.751.119-.395-.553-.553-.08-.988zM49.568 358.239l-.77.771-.119.355c.002.157.036.311.1.455l.77.77.395.119a1.34 1.34 0 00.888-.455 1.228 1.228 0 00.297-.968 1.399 1.399 0 00-1.56-1.047zM39.731 298.268l.119.355.276.277a1.37 1.37 0 001.166.533 2.31 2.31 0 00.395-.079l.73-.731.06-.296.099-.099-.02-.296-1.126-1.126h-.237a1.37 1.37 0 00-1.304 1.303l-.158.159zM52.393 321.873l.83.83c.14.041.289.055.435.039a1.34 1.34 0 00.968-.533 1.486 1.486 0 00.217-1.245l-.85-.849-.395-.04a1.373 1.373 0 00-1.205 1.442l-.098.099.098.257zM50.378 295.206l.494.494.297.098h.395a1.258 1.258 0 001.007-.612 1.314 1.314 0 00.02-1.323l-.909-.909a1.54 1.54 0 00-1.284.968l-.04.237.02.454.139.138-.139.455zM54.21 334.397l-.256-.02a1.286 1.286 0 00-1.047-.375 1.636 1.636 0 00-1.146.671l-.138.139-.099.335.613.613a1.48 1.48 0 001.047.375c.217.258.517.433.849.494l.217-.02.395-.118.158-.158.751-.751a1.645 1.645 0 00-.672-1.343l-.671.158zM47.297 347.948l-.257-.257-.217-.02h-.277a1.311 1.311 0 00-1.205.968 1.481 1.481 0 00.553 1.62l.376.375h.276a1.311 1.311 0 001.205-.968 1.48 1.48 0 00-.454-1.718zM64.502 277.131l.218-.02.237-.237a1.222 1.222 0 00.79-.79l.099-.099-.119-.118a1.788 1.788 0 00-.632-.632l-.118-.119-.336-.059-.257.02-.494.494a1.587 1.587 0 00-.237 1.619c.282.08.582.059.85-.059zM33.628 327.601l-.672-.671-.316-.079a1.344 1.344 0 00-1.264 1.066l-.02.218.08.434.592.593c.172.114.368.188.573.217l.118-.118.415-.099.573-.573c.177-.325.233-.703.158-1.067l-.237.079zM32.54 312.885a1.399 1.399 0 00-.987.553 1.484 1.484 0 00-.178 1.205l.77.771.317.079h.197l.85-.85.099-.375.098-.099-.02-.257-.888-.889-.336-.098.079-.04zM55.356 326.89l-.118.119-.356.118-.415.415c-.2.187-.326.439-.355.711l-.02.218.099.375.691.691c.23.122.49.177.75.158l.297-.059.889-.889a1.45 1.45 0 00-.356-1.106 1.372 1.372 0 00-1.106-.751zM51.801 258.504l.474.474c.265.125.556.186.85.177.162-.018.322-.058.474-.118l.375-.375a1.316 1.316 0 00.138-1.561l-.83-.83-.335-.019a1.53 1.53 0 00-1.284 1.086v.237l.06.573.078.356zM31.296 320.964l.04-.276-.04-.435-1.007-1.007-.257.019a1.37 1.37 0 00-1.304 1.304l.04.237.415.415a1.231 1.231 0 00.987.356 1.395 1.395 0 001.126-.613z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M28.808 289.872a1.59 1.59 0 00.118-.829l-.85-.85-.374-.098-.198-.001a1.513 1.513 0 00-1.244.929l-.04.237.04.474.573.573c.19.196.44.321.71.356l.238-.001.434-.158.573-.572.02-.06zM32.482 323.256a1.115 1.115 0 00-.395.079l-.395.395a1.427 1.427 0 00-.316 1.343 1.533 1.533 0 001.047 1.047h.276l.494-.494a1.426 1.426 0 00.237-1.62 1.342 1.342 0 00-.948-.75zM19.148 288.075l.85.849.296.059c.382-.039.736-.222.987-.513a1.428 1.428 0 00.257-1.126l-.987-.988-.376-.02a1.4 1.4 0 00-1.224 1.225l.04.237.157.277zM33.825 300.855l-.297.297-.079.355-.02.218.02.217.317.316a1.396 1.396 0 001.185.474 1.453 1.453 0 001.264-1.501l-.138-.139-.198-.395-.494-.493a1.69 1.69 0 00-.75-.119l-.415.415-.395.355zM40.166 317.982c.081-.159.122-.336.119-.514.153-.193.268-.414.335-.652a1.2 1.2 0 00.593-.395l.87.869c.156.03.314.043.473.04a1.286 1.286 0 001.107-1.106l.02-.218-.08-.355-.652-.652-.375-.139-.237-.039-.691.691-.712-.711a1.368 1.368 0 00-1.244-.612l-.316-.079h-.237l-.138.138-.277.277-.395.395a1.34 1.34 0 00-.296.612l-.119.316-.118.119c-.247.21-.402.507-.435.829l.04.277.968.968.434.079.257-.02c.185-.064.358-.158.514-.276.181.099.385.153.592.158zM-1.178 336.886l-.376.178-.434.434a1.062 1.062 0 00-.139 1.008 1.314 1.314 0 00.731.731l.158.158.376.098a1.248 1.248 0 00.868-.437 1.258 1.258 0 00.297-.926 1.62 1.62 0 00-.355-.829 1.309 1.309 0 00-1.126-.415zM26.714 323.809l.237-.237a1.286 1.286 0 00.77-1.482l-.493-.493a1.502 1.502 0 00-.988-.474l-.237.039-.277.276a1.538 1.538 0 00-.197 1.699 1.31 1.31 0 001.323.771l-.138-.099zM26.793 313.893l-.376-.376-.375-.138a1.415 1.415 0 00-.972.5 1.418 1.418 0 00-.332 1.041c.035.395.219.762.514 1.027a1.407 1.407 0 001.027.316l.376-.178.158-.158a1.367 1.367 0 00.098-2.113l-.118.079zM24.541 319.345l-.573-.573-.296-.099-.099-.099c-.24.02-.472.102-.671.237l-.83.83a1.397 1.397 0 00.138 1.363 1.315 1.315 0 001.165.573 1.232 1.232 0 00.87-.435 1.87 1.87 0 00.276-.276 1.371 1.371 0 00.119-1.186l-.099-.335zM21.46 325.152l.078-.553-.02-.296-.829-.83-.296-.059h-.198a1.396 1.396 0 00-1.383 1.382c-.079.312-.044.641.1.929a1.306 1.306 0 00.79.79l.315-.039.672-.672c.123-.219.222-.451.296-.691l.474.039zM33.351 349.765a1.294 1.294 0 00-1.114.701 1.287 1.287 0 00.087 1.314l.514.514.256.059.257-.02.81-.81a1.257 1.257 0 00-.158-1.264 1.34 1.34 0 00-.652-.494zM2.674 289.141l.375-.098.514-.514c.146-.183.24-.401.276-.632l-.02-.217-.118-.396-.514-.513a1.119 1.119 0 00-.592-.237h-.237l-.356.118-.533.534a1.307 1.307 0 00-.257.612l.02.217.099.376.533.533c.172.114.368.188.573.217h.237zM17.884 293.606l.395.395c.164.12.353.201.553.237l.276-.04.474-.158.553-.553.1-.375.019-.218a1.316 1.316 0 00-.815-1.225 1.321 1.321 0 00-.509-.098.508.508 0 00-.296.059l-.75.751-.08.316-.098.099.118.118.06.692zM9.488 308.362l-.217.217-.217.02-.751.75-.099.376a1.315 1.315 0 002.18 1.08c.263-.229.424-.554.447-.903a1.313 1.313 0 00-1.343-1.54zM19.207 308.875l.336.138.119.119h.276a1.342 1.342 0 001.087-.652 1.37 1.37 0 00.098-1.244l-.849-.85-.355-.118-.218-.02-.237.04-.83.829-.118.356-.02.217v.474l.711.711zM50.003 299.808l.119.119a1.399 1.399 0 001.303.711 1.537 1.537 0 001.265-1.027 1.454 1.454 0 00-.494-1.561l-.138-.138-.395-.119-.218-.019-.257.059-.138.138a1.43 1.43 0 00-.71 1.422l-.337.415zM-5.524 273.516l-.158-.158-.197-.434-.316-.316a1.425 1.425 0 00-1.166-.494l-.237.039-.711.711a1.622 1.622 0 00-.08 1.067l.93.929c.198.056.406.076.612.059a1.398 1.398 0 001.323-1.403zM96.622 287.285a1.23 1.23 0 001.126-.889v-.316l-.139-.376a1.396 1.396 0 00-.414-.809 1.222 1.222 0 00-.948-.277l-.238.237a1.31 1.31 0 00-.573 1.521 1.424 1.424 0 001.186.909zM35.109 371.988a1.397 1.397 0 00-1.995-1.008l-.435.435a1.31 1.31 0 00-.138 1.324l.395.395.435.237.217-.218-.079.316c.228.107.48.154.73.139l1.008-1.008-.04-.355-.098-.257zM8.145 267.274l.652.652a1.426 1.426 0 001.363-1.363 1.284 1.284 0 00-1.244-1.244l-.494.494a1.311 1.311 0 00-.455 1.718l.178-.257zM26.99 301.092a1.375 1.375 0 00-1.067.751l-.04.237.04.474.89.889.355.079.099.099a1.318 1.318 0 001.185-1.185l-.1-.099-.058-.336-.474-.474a1.313 1.313 0 00-.83-.435zM48.7 299.927l-.376-.178-.237-.237a1.597 1.597 0 00-1.225.672 1.023 1.023 0 00-.217.217 1.508 1.508 0 00-.573.968 1.872 1.872 0 00.731 1.876l.139.139.395.079h.197l.237-.04.553-.553c.044-.049.09-.095.139-.138a1.462 1.462 0 00.79-.79l.316-.316a1.341 1.341 0 00-.395-1.107l-.415-.414-.06-.178zM79.456 341.469l-.632-.633-.336-.098-.099-.099-.138.138a1.298 1.298 0 00-.909.593 1.229 1.229 0 00-.118 1.027l.118.119a1.286 1.286 0 001.107.829l.355-.118.672-.672a1.367 1.367 0 00.079-1.067l-.099-.019zM62.646 329.518l.355-.356a1.372 1.372 0 00-.454-1.165 1.302 1.302 0 00-1.067-.356l-.71.711a1.453 1.453 0 00-.198 1.106l.79.79c.23.122.494.164.75.119l.277-.04.336-.335-.08-.474zM42.99 320.016l-1.026 1.027-.02.297.06.493.414.415c.105.156.24.29.395.395l.296.06.573-.06.652-.652c.211-.305.303-.678.257-1.046l-.89-.889a1.285 1.285 0 00-.71-.04zM43.801 330.268l.118.119a1.398 1.398 0 00.988-.553 1.456 1.456 0 00.257-1.205l-1.442-1.442-.395-.158-.237-.04-.257.02-.909.909-.099.414a1.282 1.282 0 00.573 1.008c.069.299.22.572.435.79l.119.119.849.019zM23.533 302.06l.119-.118.059-.297a1.452 1.452 0 00-1.56-1.323l-.87.869-.138.454-.02.257.692.691-.139-.138c.25.17.55.247.85.217.206-.011.408-.065.593-.158l.454-.454h-.04zM44.156 306.603h-.237a1.397 1.397 0 00-1.185 1.186l-.04.276.1.099.059.257.553.553c.31.18.668.263 1.027.237l.276-.039.81-.81a1.873 1.873 0 00-.395-1.265l-.296-.296-.672-.198zM46.744 316.895l-.316.04-.514.197-.158.158a1.18 1.18 0 00-.375.376 1.676 1.676 0 00-.494.256l-.869.87a1.73 1.73 0 00.533 1.363l.336.335c.25.28.595.456.968.494l.277-.039.434-.435a2.06 2.06 0 00.573-.81v-.276l.178-.178.04-.316a1.536 1.536 0 00-.613-2.035zM45.322 311.68l-1.008 1.008a1.393 1.393 0 00.772 1.419c.172.084.36.132.552.141l.217-.02.158-.158a1.397 1.397 0 00.217-1.56 1.312 1.312 0 00-1.343-.751.495.495 0 00.435-.079zM40.304 305.517l.257-.02a1.365 1.365 0 001.304-1.303l-.119-.119-.099-.336-.651-.652a1.675 1.675 0 00-.751-.118l-.277.039-.77.771-.099.296.1.099v.237l1.105 1.106zM62.132 352.866a1.416 1.416 0 00-.552.172c-.17.096-.319.226-.436.382a1.53 1.53 0 00-.237 1.343l.494.494a1.428 1.428 0 001.067.316 1.422 1.422 0 001.047-.573l.454-.454a1.568 1.568 0 00-.494-1.206 1.29 1.29 0 00-1.343-.474zM47.02 282.089l-.474-.474-.118.119a1.4 1.4 0 00-1.107.593 1.314 1.314 0 00-.217 1.125l.336.336a1.309 1.309 0 001.778.632l.671-.671c.079-.215.106-.445.08-.672l-.672-.671-.277-.317zM42.34 277.447l.79.79.098-.098a1.79 1.79 0 001.56-1.087l.02-.296-.691-.691a1.33 1.33 0 00-1.027-.317c-.3.028-.587.137-.83.316l-.474.475-.099.375-.138.138.04.237.118.119.632.039zM49.924 308.737h-.276a1.45 1.45 0 00-1.047.573 1.427 1.427 0 00-.217 1.244l1.007 1.008h.276a1.466 1.466 0 001.047-.573 1.595 1.595 0 00.218-1.245l-1.008-1.007zM55.89 277.684l-.139.139a1.198 1.198 0 00-.533 1.244l.85.85.394.118a1.284 1.284 0 001.047-.691c.156-.24.22-.528.178-.81a1.343 1.343 0 00-.829-1.242 1.335 1.335 0 00-.514-.101l-.454.493zM70.587 255.244l.573.573c.155.033.315.046.474.04a1.426 1.426 0 001.224-.948 1.567 1.567 0 00-.513-1.699l-.435-.435h-.316a1.342 1.342 0 00-1.106.711 1.594 1.594 0 00.04 1.541l.059.217zM48.502 322.367l.968.968.336.019a1.39 1.39 0 001.106-.671 1.621 1.621 0 000-1.501l-.711-.711-.316-.04-.277.039-.435.198-.197.198a1.535 1.535 0 00-.474 1.501zM97.373 296.174l-.474.474c-.16.099-.296.235-.395.395a.55.55 0 00-.159.158l.139.454.177.375.494.494.395.119a1.315 1.315 0 001.166-1.403 1.37 1.37 0 00-1.343-1.066zM71.574 280.726a1.512 1.512 0 00-1.363 1.363v.237l.099.455.533.533.415.178.257.02h.316l.87-.87.059-.335-.178-.178h.276l.02-.494-.909-.908-.276-.08-.119.079zM45.006 295.601l.217.02.968-.968.118-.356.119-.118v-.238l-.81-.809-.375-.139-.119-.118a1.343 1.343 0 00-1.249 1.057c-.04.192-.04.391.005.582l.908.909.218.178zM54.685 266.958a1.313 1.313 0 00-1.087 1.087l.06.296.217.454a1.285 1.285 0 001.224.672 1.44 1.44 0 00.494-.138l.731-.731.04-.277a1.286 1.286 0 00-1.205-1.205h-.237l-.237-.158zM.52 324.757l-.434-.435-.336-.098-.217-.02-.297.02-.81.81-.118.355-.119.119a1.45 1.45 0 00.909 1.343l.296.059.494-.059.79-.79a1.23 1.23 0 00.1-.81l-.12-.119-.138-.375zM67.88 341.745l-.296-.02-.553.119-.613.612-.118.395-.138.139a1.312 1.312 0 00.85 1.389c.17.061.352.086.532.072l.455-.454a1.37 1.37 0 00.335-1.244 1.482 1.482 0 00-.454-1.008zM51.228 338.802l-.454-.454a.835.835 0 00-.1-.336 1.48 1.48 0 00-1.441-1.008l-.099.099-.316.119-.75.75a1.731 1.731 0 00.276 1.383c.173.207.398.364.652.454l.217.218a1.315 1.315 0 001.245.691l.335-.099.731-.731.08-.316v-.237l-.14-.138-.236-.395zM4.688 305.102l.317-.039.79-.79.138-.336.099-.099a1.447 1.447 0 00-.415-1.086 1.568 1.568 0 00-1.126-.376l-.928.929-.1.336.1.098v.237l.849.85.356.118-.08.158zM26.595 253.881l.158-.158.355-.158.396-.395c.153-.188.255-.412.296-.652l-.099-.336-.81-.809a1.935 1.935 0 00-.474-.04 1.424 1.424 0 00.02 2.825l.158-.277zM43.88 270.336a1.293 1.293 0 00.947.672h.238l.395-.119.572-.573a1.372 1.372 0 00.158-1.067l-.83-.829-.374-.06a1.343 1.343 0 00-1.245 1.245l.138.138v.593zM68.335 267.649l.415.415c.178.138.389.227.612.257l.237-.039.454-.178.553-.553.1-.376.019-.217a1.314 1.314 0 00-1.343-1.343l-.277.039-.75.751-.119.316.217.217-.118.711zM28.195 310.436l.652-.652a1.397 1.397 0 00-.118-1.383 1.262 1.262 0 00-1.185-.553 1.237 1.237 0 00-.949.553l-.237.237-.079.316c.023.371.178.721.435.988a1.34 1.34 0 001.047.336l.434.158zM30.27 343.641l.315-.316.158-.395c.088-.226.128-.468.119-.711l-.85-.849-.335-.099h-.316l-.534.217-.118-.118a1.507 1.507 0 00-.75 1.54l.829.83.355.119.1.098.335-.019.355-.356.336.059zM21.499 275.61l.415.415c.17.135.376.217.592.237h.198l.415-.099.474-.474c.103.056.218.09.335.099h.198l.138-.138.237-.04.81-.81a1.284 1.284 0 00-1.363-1.679c-.365.03-.705.2-.948.474l-.316-.079-.099-.098a1.45 1.45 0 00-1.264 1.027l-.02.257.099.414.099.494zM25.627 278.553a1.562 1.562 0 00.455 1.166 1.476 1.476 0 001.126.415l.651-.652a1.68 1.68 0 00.178-1.284l-.79-.79-.336-.099-.099-.099-.395.04-.81.809-.098.336.118.158zM58.379 295.779a1.48 1.48 0 00-1.284 1.6 1.338 1.338 0 00.916 1.148c.17.054.349.074.526.057l.75-.751.139-.375-.968-.968-.079-.711zM62.724 334.416l-.118-.316-.237-.237-.514-.316-.316-.079-.237.04-.968.968-.079.395v.237a1.228 1.228 0 002.41.316l.118-.514-.059-.494zM41.49 289.043l.67-.672a1.48 1.48 0 00.199-1.62 1.421 1.421 0 00-1.383-.75l-.415.098-.573.573a1.396 1.396 0 00.138 1.324 1.483 1.483 0 001.363 1.047zM35.484 274.583l.79.79h.277a1.539 1.539 0 001.363-1.679l-.118-.119-.139-.414-.513-.514-.356-.118-.098-.099-.297-.02-.237.04-.79.79c-.104.325-.15.666-.138 1.007l.118.119.138.217zM43.169 330.545l-.85-.849h-.355a1.417 1.417 0 00-1.185 1.185v.474l.039.237.691.691c.227.113.48.161.731.139l.296-.06.534-.533a1.76 1.76 0 00.197-1.304l-.098.02zM39.356 291.452l-.119-.118-.118-.356-.435-.434a1.286 1.286 0 00-.691-.336h-.237l-.435.119-.533.533-.158.395-.04.237.04.237.908.909.396.079.098.098a1.295 1.295 0 00.954-.398 1.307 1.307 0 00.37-.965zM48.324 328.747a1.277 1.277 0 00-.968-.336l-.691.692a.977.977 0 00-.218.691l-.671.672-.08.276c-.006.412.15.809.435 1.107l.454.454.218.02.099.098a1.335 1.335 0 001.185-.987l.02-.257-.08-.277a1.252 1.252 0 00.415-1.047c.103-.37.06-.766-.118-1.106zM12.965 283.274l-.178-.177-.138-.376a1.283 1.283 0 00-1.106-.632l-.118.119-.336.138-.652.652-.119.355-.118.119c.005.336.131.658.355.909l.1.098a1.314 1.314 0 001.322.968l.415-.414a1.595 1.595 0 00.573-1.759zM32.067 264.41l.415.415c.164.119.353.2.553.237l.217-.02.415-.099.573-.573a1.342 1.342 0 00.177-1.086l-.908-.909-.316-.079a1.344 1.344 0 00-1.264 1.067l-.02.217.079.435.079.395zM-18.265 325.962l-.277.039-.908.909a1.48 1.48 0 00.951 1.517c.195.072.403.1.61.083a1.51 1.51 0 00.256-.059l.988-.988a1.565 1.565 0 00-.455-1.165 1.363 1.363 0 00-1.165-.336zM64.582 340.185l-.139-.139a1.26 1.26 0 00-1.007-.691l-.218.02-.375.138-.573.573a1.452 1.452 0 00-.158 1.027l.85.849.395.079a1.308 1.308 0 001.205-1.205l-.099-.098.119-.553zM23.296 331.296h-.276l-.494.177-.573.573-.119.395-.02.217a1.315 1.315 0 001.423 1.225l.316-.039.73-.731.08-.316.099-.099a1.424 1.424 0 00-1.166-1.402zM66.932 290.307a1.42 1.42 0 00-.336.336 1.227 1.227 0 00-.316 1.422 1.258 1.258 0 001.284.77 1.335 1.335 0 001.047-.731l.119-.118.02-.217-1.087-1.087-.336-.02-.099-.098-.296-.257zM65.371 308.934c-.135.004-.269.03-.395.079l-.592.593a1.564 1.564 0 00.158 1.343 1.537 1.537 0 001.204.613l.574-.573a1.483 1.483 0 00.197-1.62 1.403 1.403 0 00-1.146-.435zM31.04 335.365l-.317.316a1.48 1.48 0 00-.059 1.086l.909.909c.105.01.21.01.316 0a1.374 1.374 0 001.103-1.413 1.367 1.367 0 00-1.241-1.293l-.613.612-.099-.217zM85.777 333.883l-.73-.731h-.278a1.343 1.343 0 00-1.244 1.442 1.309 1.309 0 00.815 1.226c.162.066.334.099.509.098l.138-.139.375-.177.514-.514a1.59 1.59 0 00.118-.829l-.217-.376zM17.093 314.94l.534.533.415.099.217.019a1.563 1.563 0 001.244-.928l.04-.237-.06-.454-.849-.85-.257-.059-.217.02a1.337 1.337 0 00-1.304 1.303l.04.237.197.317zM-2.759 314.051l-.474.474-.138.375-.118.119.02.296.928.928.375.138c.36-.042.691-.219.928-.493a1.291 1.291 0 00.297-.968 1.368 1.368 0 00-1.482-1.284l-.336.415zM13.479 334.357l-.87.869-.118.356-.04.237.04.237.454.454c.172.17.394.28.633.317l.138.138.237.039.257-.019 1.007-1.008.06-.375-.12-.119a1.452 1.452 0 00-1.678-1.126zM31.178 293.191l-.079.316.02.336.79.79.316.118.119.119.217-.02a1.336 1.336 0 001.284-1.679l-.73-.731a1.617 1.617 0 00-.712-.118l-.316.039-.652.652-.256.178zM19.978 342.278l-.376-.375-.236-.039-.455.019-.988.988-.098.454v.237c.102.293.264.562.474.79l.257-.256-.218.217.454.257.277.039.336-.02.79-.79s.296-1.047-.197-1.699l-.02.178zM-11.825 297.122l-.435-.435a1.254 1.254 0 00-1.255-.013 1.26 1.26 0 00-.641 1.08l.04.237.138.138a1.315 1.315 0 001.086 1.087l.217-.02.376-.138.415-.415c.19-.194.315-.443.355-.711l-.02-.297-.276-.513zM13.973 318.061l-.692-.692-.336-.099-.098-.098a1.453 1.453 0 00-1.324 1.56l.04.237.75.751.356.118.217.02.77-.77a1.68 1.68 0 00.198-1.304l.119.277zM9.35 300.282l.474.475.356.118h.197l.257-.02.118-.118.356-.158.454-.455.1-.375.118-.118-.89-.889a1.45 1.45 0 00-.888-.218l-1.047 1.047.02.257.118.119.257.335zM78.211 296.015l.218-.019.355-.119.474-.474.158-.355.119-.119a1.393 1.393 0 00-.763-1.395 1.397 1.397 0 00-.54-.146l-.218.02-.356.119-.85.849c-.016.408.118.809.376 1.126a1.42 1.42 0 001.027.513zM112.661 296.766a1.389 1.389 0 00-1.126-.415l-.257.06-.671.671a1.583 1.583 0 00.237 1.501 1.247 1.247 0 001.126.494c.315-.027.609-.168.829-.395l.297-.296.158-.356.118-.118a1.43 1.43 0 00-.711-1.146zM83.09 312.984l-.296.059-.414.415-.237.435-.08.276a1.424 1.424 0 001.008 1.324l.375.059.277-.04.296-.098.79-.791a1.368 1.368 0 00-1.106-1.461l-.118.118-.1-.099-.394-.197zM67.821 299.769l.376.138a1.332 1.332 0 001.066-.711 1.51 1.51 0 00.139-.731 1.368 1.368 0 00-1.363-1.363l-.376.178-.197.197a.982.982 0 00-.395.395l-.139.139a1.397 1.397 0 00-.02.573l1.048 1.047-.139.138zM70.27 321.024l-.177-.178a1.146 1.146 0 00-.415-.415l-.257-.059-.494.02-.592.592-.217.454-.06.257.02.336.85.85.355.118.217.02a1.43 1.43 0 001.264-1.027v-.277l-.118-.474-.375-.217zM69.145 303.147l-.356-.356a3.124 3.124 0 00-.454.138l-.376-.098-.217-.02a1.37 1.37 0 00-1.027.593 1.31 1.31 0 00-.138 1.125l.75.751.316.079.218-.217.513.513a1.224 1.224 0 001.122-.732 1.227 1.227 0 00-.213-1.322 1.06 1.06 0 00-.138-.454zM51.82 283.057a1.335 1.335 0 001.225.672c.172-.016.34-.063.494-.138l.652-.652a1.342 1.342 0 00-.277-1.185 1.502 1.502 0 00-1.224-.514 1.285 1.285 0 00-1.205 1.205l.04.276.296.336zM70.547 302.929l.336-.335a1.318 1.318 0 00.125-1.284 1.316 1.316 0 00-1.034-.771h-.237l-.434.119-.534.533-.138.375-.118.119.02.415.592.592.454.218.297.059.138-.138.533.098zM98.044 312.451a.584.584 0 00-.316.079l-.316.316a1.39 1.39 0 00-.87 1.066 1.283 1.283 0 00.751 1.423l.158.158.475-.04 1.145-1.146.1-.375.019-.217a1.285 1.285 0 00-1.146-1.264zM83.268 320.708l-.573-.573a1.34 1.34 0 00-1.126-.455 1.173 1.173 0 00-.356.079l-.967.968.079.514.612.612.395.158.138.138.257-.019a1.228 1.228 0 00.928-.494 1.173 1.173 0 00.613-.928zM78.33 281.082l.474.474.375.178.119.118c.064.02.133.02.197 0a1.34 1.34 0 00.968-.533 1.401 1.401 0 00.277-1.225l-.692-.691-.335-.138-.1-.099a1.396 1.396 0 00-1.342 1.343l.118.118-.06.455zM79.653 307.69l.198-.198a1.535 1.535 0 00-.435-1.185c-.3-.261-.689-.395-1.086-.375l-.75.75a1.35 1.35 0 00-.159 1.067l.81.81c.226.113.48.161.73.138l.297-.059.711-.711-.316-.237zM91.012 305.616a1.426 1.426 0 00.138-1.245l-.77-.77-.297-.059-.098-.099a1.452 1.452 0 00-1.264 1.027v.277l.513.513c.28.238.64.358 1.007.336.249.087.518.094.77.02zM56.107 291.413l.375.099a1.326 1.326 0 001.008-.534c.18-.254.276-.557.276-.869.045-.122.078-.247.099-.375l-.119-.119-.118-.355-.533-.534c-.16-.128-.35-.21-.553-.237h-.238l-.395.119-.533.533c-.122.178-.203.38-.237.593l-.138.138c-.02.397.114.787.375 1.087l.474.474.257-.02zM56.72 304.115l.553.553a1.43 1.43 0 001.208.212 1.417 1.417 0 00.925-.805 1.453 1.453 0 00-.237-1.501l-.415-.415h-.276a1.45 1.45 0 00-1.047.573 1.37 1.37 0 00-.237 1.225l-.474.158zM59.05 314.742a1.368 1.368 0 00.139-1.284l-.554-.553a1.371 1.371 0 00-.889-.257c-.22.015-.432.09-.612.217l-.77.771a1.539 1.539 0 00.375 1.165 1.29 1.29 0 001.106.395 1.282 1.282 0 001.205-.454zM70.132 331.039l-.652-.652-.336-.099-.118-.118-.316.039-.83.83-.118.355-.02.218.02.256.928.929.415.059.296-.02a1.397 1.397 0 001.087-1.639l-.356-.158zM80.72 288.489a1.37 1.37 0 001.383 1.383l.73-.731a1.423 1.423 0 00-.315-1.936l-.356-.355a.998.998 0 00-.336-.02 1.361 1.361 0 00-.912.591 1.363 1.363 0 00-.194 1.068zM54.447 313.142a1.511 1.511 0 00.237-1.62 1.342 1.342 0 00-1.303-.79l-.376.375a1.373 1.373 0 00-.375 1.087 1.617 1.617 0 00.652 1.086l.138.139.336.098.099.099.217-.02.356-.355.02-.099zM61.361 285.21l.613.613.296.098.198.198a1.433 1.433 0 001.345-1.116 1.424 1.424 0 00-.002-.622l-.83-.83a1.228 1.228 0 00-.73-.138l-.297.059-.632.632c-.15.33-.17.703-.06 1.047l.1.059zM56.838 310.199l.375-.099.534-.534c.135-.17.217-.376.236-.592l-.02-.217-.098-.376-.514-.513a1.278 1.278 0 00-.612-.257l-.118.118-.376.099-.513.514a1.507 1.507 0 00-.257.652l.118.118.1.376.493.493c.176.142.388.231.612.257l.04-.039zM60.256 309.073l.513.513h.277a1.451 1.451 0 001.047-.572 1.514 1.514 0 00.217-1.245l-.77-.77a1.714 1.714 0 00-.633-.079 1.392 1.392 0 00-1.205.968l-.04.237.02.414.337.336.237.198zM55.81 307.433l.396-.118.73-.731a1.511 1.511 0 00.238-1.62 1.425 1.425 0 00-1.284-.77l-1.106 1.106c-.142.176-.23.388-.257.612l.118.119.1.375.533.533c.17.135.376.218.592.237l-.06.257zM54.507 298.347l.395-.119.533-.533c.138-.178.226-.389.257-.613l-.119-.118-.118-.356-.533-.533a1.226 1.226 0 00-.553-.237l-.218.02-.395.118-.514.514c-.14.176-.23.388-.256.612l.02.217.098.376.494.494c.176.106.37.18.573.217l.336-.059zM63.298 317.685l-.672-.671-.336-.099-.099-.099-.395.04-.335.335a1.5 1.5 0 00-1.11.525A1.507 1.507 0 0060 318.89l.098.099.119.316.731.731c.15.026.303.033.454.02.4-.022.776-.199 1.047-.494a1.375 1.375 0 00.916-.577 1.372 1.372 0 00.21-1.063l-.276-.237zM67.386 315.216l.356.356a1.295 1.295 0 00.967-.652 1.37 1.37 0 00.02-1.521 1.23 1.23 0 00-1.225-.632 1.366 1.366 0 00-.908.474 1.39 1.39 0 00-.257 1.086l.138.138a1.151 1.151 0 00.909.751zM72.937 291.216l.316-.04.178-.059.356-.158.335-.336.1-.178.098-.375.02-.178-.02-.178-.04-.197-.177-.336-.316-.316-.178-.099-.356-.118-.217-.02-.316.039-.198.04-.335.178-.159.118-.157.198-.1.178-.118.355-.04.198.02.177.04.198.178.336.118.158.218.178.177.098.376.099.197.04zM87.08 325.389l-.414-.415a.898.898 0 00-.553-.237h-.237l-.395.119-.573.573a1.335 1.335 0 00-.178 1.086l.77.77.317.079.098.099h.237l.85-.849.098-.375.119-.119-.099-.099-.04-.632zM73.964 332.342l-.83.83-.118.355a1.4 1.4 0 00.891 1.37c.182.07.376.102.57.092l.12-.118.315-.119.77-.77a1.425 1.425 0 00-.276-1.304 1.375 1.375 0 00-1.442-.336zM94.132 346.308a1.28 1.28 0 00-1.007.573 1.424 1.424 0 00-.178 1.205l.79.79c.213.089.442.129.672.119.339-.044.648-.213.869-.474l.276-.277a1.398 1.398 0 00-.375-1.126 1.34 1.34 0 00-1.047-.81zM83.347 295.542l-.889.889a1.614 1.614 0 00-.04.434l.12.119.138.375.592.593.376.138.237.039.81-.81a1.338 1.338 0 00-.198-1.343 1.418 1.418 0 00-1.146-.434zM93.303 325.212l-.296.059-.751.75-.079.316.73.731c.177.142.39.231.613.257l.257-.02.454-.177.553-.554.1-.375.019-.217a1.313 1.313 0 00-1.6-.77zM63.258 305.517l1.047-1.047.099-.375.118-.119-1.007-1.007-.336-.099-.119-.119a1.4 1.4 0 00-.987.554l-.85.849-.04.277-.197.197-.04.277.02.454.573.573.435.237.178.178a1.421 1.421 0 001.126-.692l-.02-.138zM73.45 316.974l-.355-.118a1.31 1.31 0 00-1.007.533 1.398 1.398 0 00-.277 1.106l.79.79.336.099.119.119a1.425 1.425 0 001.007-.534c.233-.318.333-.715.277-1.106l-.89-.889zM86.093 267.037l.158-.158.06-.257a1.337 1.337 0 00-.435-.988 1.405 1.405 0 00-1.047-.335l-.276.039-.79.79-.06.257-.099.099.02.415.632.632.455.217.276.04a1.23 1.23 0 001.106-.751zM9.113 332.639l.494-.494a1.314 1.314 0 00-.04-1.304 1.29 1.29 0 00-.513-.514l-.139-.138-.375-.138-.237-.04-.237.04a1.511 1.511 0 00-1.205.612l-.434.435c-.01.303.08.602.256.849l1.146 1.146a.969.969 0 00.296-.059l.119-.119.869-.276zM13.261 356.975l-.632-.632-.296-.059-.099-.099a1.426 1.426 0 00-1.165 1.165l.02.297.513.513a1.317 1.317 0 001.007.336 1.217 1.217 0 00.968-.573 1.367 1.367 0 00.139-1.244l-.455.296zM36.136 311.7l.257.257.513-.04.277-.079.415-.415.217-.454.04-.277c.042-.379.135-.751.276-1.106l.138-.138-.059-.454-.533-.534-.415-.217-.178-.178a1.286 1.286 0 00-1.027.632l-1.185 1.186-.02.375-.119.118a1.317 1.317 0 00.87 1.25c.17.061.352.086.533.074zM35.267 306.682l.493-.493a1.48 1.48 0 00-.552-1.62l-.198-.198-.593-.039-.178.178h-.236l-.415.138-.534.533-.158.395-.04.237.04.316 1.126 1.126.573.02.277-.079.454-.454-.06-.06zM35.78 318.258l-.632-.632-.434-.119-.218-.019a1.3 1.3 0 00-.954.588 1.31 1.31 0 00-.152 1.111l.83.829.355.04a1.314 1.314 0 001.205-1.205l-.138-.139.139-.454zM38.862 338.565l-.375.178-.415.415a1.113 1.113 0 00-.158 1.027 1.21 1.21 0 00.73.731l.159.158.375.098a1.23 1.23 0 00.87-.434c.213-.26.32-.593.296-.929a1.427 1.427 0 00-1.482-1.244zM81.865 344.589l.395.04a1.34 1.34 0 00.692-.691 1.404 1.404 0 00.099-1.087 1.318 1.318 0 00-1.102-.983 1.313 1.313 0 00-.518.035l-.277.277-.335.612-.06.415.77.77.336.612zM70.567 293.072l.276-.276.336-.613.04-.395a1.225 1.225 0 00-.198-.434l-.296-.297-.613-.335-.395-.04c-.317.141-.57.394-.711.711-.15.314-.185.671-.099 1.008a1.424 1.424 0 001.66.671zM92.493 349.844l-.533-.533a1.283 1.283 0 00-1.245-.297.956.956 0 00-.375.178l-.494.494v.316l.099.099.099.296.79.79.355.079.1.099.118-.118a1.427 1.427 0 001.047-1.561l.04.158zM75.228 329.34l-.612-.613-.316-.079-.435.04a1.48 1.48 0 00-.77.77 1.68 1.68 0 00.02 1.284l.671.672.277.079a1.318 1.318 0 001.062-.529 1.318 1.318 0 00.202-1.17l.099-.099-.198-.355zM82.675 355l-.177-.178-.435-.158h-.316a1.645 1.645 0 00-.375.059 1.314 1.314 0 00-.85.85 1.507 1.507 0 00.257 1.323l.711.711.257-.019a1.456 1.456 0 001.106-1.107v-.276l-.099-.415-.177-.178.098-.612z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M71.08 317.29h.237a1.281 1.281 0 001.067.158 1.656 1.656 0 00.948-.948l.119-.119.039-.355-.77-.771a1.241 1.241 0 00-1.067-.158 1.368 1.368 0 00-.928-.256l-.12.118-.374.178-.494.494-.119.395.119.118a1.56 1.56 0 00.869 1.186l.474-.04zM75.051 302.771l.237.237.217-.019.277-.04a1.31 1.31 0 00.968-1.205 1.482 1.482 0 00-.87-1.462l-.335-.335-.296.059a1.37 1.37 0 00-.968 1.205 1.48 1.48 0 00.77 1.56zM72.523 375.544l-.395.395a1.313 1.313 0 00-.534 1.007l.217.217c.148.254.36.465.613.613l.098.098.336.02.77-.77a1.65 1.65 0 00-.078-1.62 1.343 1.343 0 00-1.027.04zM92.493 319.858l.652.652.316.04.099-.099a1.423 1.423 0 001.046-1.324l-.138-.138-.158-.395-.592-.593a1.394 1.394 0 00-.613-.098l-.968.968a1.31 1.31 0 00.06 1.086l.296-.099zM96.542 334.061a1.314 1.314 0 00.79-.79 1.4 1.4 0 00-.039-1.225l-.75-.751-.336-.019-1.067 1.066-.04.356.02.217 1.027 1.027.376.02.02.099zM71.376 324.974l.79-.79c.148-.229.224-.497.218-.77l-.85-.85-.316-.118a1.429 1.429 0 00-.75.039l-.297.099-.671.672-.06.296a1.482 1.482 0 00.593 1.027 1.345 1.345 0 001.343.395zM88.681 391.307l-.454-.455a1.794 1.794 0 00-.85-.019 3.495 3.495 0 00-.474.237l-.355.355a1.288 1.288 0 00.177 1.561l.494.494.336.099.119.118c.113-.008.226-.028.335-.059a1.486 1.486 0 001.047-1.324l-.04-.237-.177-.533-.158-.237zM96.147 325.883l.02.257.08.513.552.553.415.139.257.019.138-.138a1.367 1.367 0 001.027-1.541l-.71-.711a1.176 1.176 0 00-1.048-.138 1.373 1.373 0 00-.73 1.047zM104.898 355.849c-.086.273-.072.567.039.83l.85.849h.395l.118-.118a1.484 1.484 0 001.087-1.087v-.237l-.158-.434-.593-.593a1.59 1.59 0 00-.75-.198l-.257.06-.81.81.079.118zM94.508 323.888c.127-.031.248-.085.356-.158l.415-.415a1.37 1.37 0 00.04-1.343 1.515 1.515 0 00-1.265-.791l-.77.771a1.507 1.507 0 00.118 1.659 1.306 1.306 0 001.106.277zM114.696 355.612l-.455-.454-.513-.198-.336-.02a1.32 1.32 0 00-.79.791 1.366 1.366 0 00-.04 1.145l.81.81c.128 0 .254-.02.376-.059a1.29 1.29 0 00.908-1.422l-.158-.158.198-.435zM98.024 345.814l.02-.375-.79-.79a1.427 1.427 0 00-2.16.649 1.423 1.423 0 00-.052.852l.948.948c.252.075.522.068.77-.019l1.264-1.265zM88.207 330.466c-.06.171-.08.353-.06.533a2.24 2.24 0 00-.178.691 1.258 1.258 0 00-.513.514l-.652-.652-.375-.059c-.16-.012-.32.001-.475.04a1.342 1.342 0 00-.908 1.303l.118.119.158.355.731.731.415.059.217-.019.534-.534.355.356a1.371 1.371 0 001.344.356l.098-.099.336.019.553-.553a1.38 1.38 0 00.257-.257l.474-.474.158-.355.119-.119c.106-.207.16-.438.158-.671l.059-.336a1.2 1.2 0 00.257-.889l-1.146-1.146-.415.02-.158.158a1.988 1.988 0 00-.415.415 1.2 1.2 0 00-1.046.494zM124.692 303.739l.711-.711a1.11 1.11 0 00-.059-1.007 1.302 1.302 0 00-.85-.613l-.138-.138-.395-.04a1.238 1.238 0 00-.78.601 1.23 1.23 0 00-.089.98c.088.287.26.542.494.731a1.423 1.423 0 001.106.197zM100.039 322.189l-.257.257a1.315 1.315 0 00-.434 1.62l.572.573c.315.207.695.291 1.067.236l.691-.691a1.668 1.668 0 00-.138-1.718 1.338 1.338 0 00-1.462-.474l-.039.197zM101.995 331.888l.256.257.415.059a1.425 1.425 0 00.948-1.778 1.471 1.471 0 00-.81-.81 1.333 1.333 0 00-1.066-.118l-.435.434a1.275 1.275 0 00-.298.503 1.269 1.269 0 00-.038.584 1.253 1.253 0 00.692 1.007l.336-.138zM103.081 326.1l.553.553h.514c.227-.083.435-.21.612-.375l.474-.474a1.34 1.34 0 00-.395-1.304 1.284 1.284 0 00-2.015.277c-.096.074-.183.16-.257.256a1.37 1.37 0 00.099 1.205l.415-.138zM104.918 319.799l.04.553.632.632.513.198h.317l.118-.119a1.397 1.397 0 001.047-1.639 1.31 1.31 0 00-.257-.889 1.262 1.262 0 00-1.007-.494l-.692.691a2.57 2.57 0 00-.138.731l-.573.336zM88.285 298.109a1.31 1.31 0 00.918-1.587 1.308 1.308 0 00-.325-.585l-.494-.494h-.276l-.949.948a1.284 1.284 0 00.396 1.225c.182.24.439.414.73.493zM130.637 351.247l-.375.177-.513.514a1.26 1.26 0 00-.139.652l.929.928c.196.095.414.136.632.119l.928-.929a1.4 1.4 0 00.119-.632l-.929-.928a1.341 1.341 0 00-.592-.119l-.06.218zM114.815 349.963l-.317-.316a1.145 1.145 0 00-.592-.119l-.277.079-.869.869-.039.395.118.119a1.342 1.342 0 001.659.909c.074-.058.14-.125.198-.198l.731-.731.02-.336-.218-.217-.414-.454zM120.05 333.824l.632-.632.296-.613.02-.375a1.374 1.374 0 00-1.146-1.004 1.37 1.37 0 00-.533.036 1.338 1.338 0 00-.929 1.324 1.32 1.32 0 00.518 1.042 1.306 1.306 0 001.142.222zM111.062 335.543l-.435-.435-.355-.079-.099-.099a.609.609 0 01-.158.158 1.429 1.429 0 00-.956 1.54c.027.219.104.429.225.613l.948.949.356.039.217-.02.553-.553-.079.079.316-.316.04-.355-.02-.218-.435-.434-.118-.869zM82.122 350.417l-.138-.138a1.368 1.368 0 00-1.422-.395 1.4 1.4 0 00-1.008 1.244 1.48 1.48 0 00.77 1.442l.139.139.395-.001.217-.019.415-.415a1.371 1.371 0 00.415-1.521l-.099-.099.316-.237zM141.818 364.876l.83.83a1.468 1.468 0 001.244.217c.049-.056.102-.109.158-.158l.79-.79a1.343 1.343 0 00-.138-1.047l-.711-.711-.099.099a1.83 1.83 0 00-.612.059 1.378 1.378 0 00-1.081.41 1.371 1.371 0 00-.381 1.091zM38.961 372.087a1.31 1.31 0 00-.928 1.126l.079.316.257.256c.131.289.362.521.651.652.307.155.661.183.988.079l.375-.375a1.312 1.312 0 00.237-1.62 1.452 1.452 0 00-1.659-.434zM81.905 276.756a1.399 1.399 0 001.699.83c.164-.043.318-.117.454-.218l.435-.434a1.314 1.314 0 00-.099-1.324l-.514-.513-.454-.139-.099.099-.02-.336a1.317 1.317 0 00-.73.02l-.87.869.119.119.099.336-.02.691zM129.709 373.746l-.435-.435-.296-.059a1.396 1.396 0 00-.948 1.659 1.282 1.282 0 001.442 1.008l.513-.514a1.338 1.338 0 00.079-1.778l-.355.119zM104.424 344.511a1.313 1.313 0 00.909-.909v-.277l-.119-.434-1.047-1.047-.375-.02a1.311 1.311 0 00-.988 1.343l.81.81c.2.267.486.456.81.534zM83.367 350.041l.414.099h.237l.119-.118a1.598 1.598 0 00.988-.988.616.616 0 00.197-.197c.25-.292.383-.664.375-1.047a1.953 1.953 0 00-1.125-1.719l-.119-.118-.375-.02-.988.987c-.3.251-.496.602-.553.988l-.119.119-.04.276a1.229 1.229 0 00.593.988l.198.197.198.553zM44.906 315.493l.613.612.335.02.257-.257a1.345 1.345 0 00.77-.77 1.287 1.287 0 00-.098-1.047l-.178-.178a1.347 1.347 0 00-1.264-.593l-.81.81-.118.316c-.09.363-.033.746.158 1.067l.335.02zM63.692 323.888l-.138.138-.06.296.1.099a1.538 1.538 0 00.651 1.047 1.393 1.393 0 001.126.138l.77-.77a1.426 1.426 0 00-.04-1.106l-.79-.79a1.672 1.672 0 00-.77.019c-.069.063-.135.129-.197.198l-.454.454-.198.277zM84.888 329.202l.87-.869.02-.257-.04-.277-.158-.474-.435-.434a1.345 1.345 0 00-.375-.376h-.316l-.534.178-.651.652a1.509 1.509 0 00-.08 1.067l.89.888c.273.046.554.011.81-.098zM82.004 319.325l-.1-.099a1.428 1.428 0 00-.79.79 1.508 1.508 0 000 1.264l1.206 1.205.434.079h.237l.81-.81.04-.236v-.396a1.29 1.29 0 00-.81-.81 1.506 1.506 0 00-.652-.651l-.119-.119-.256-.217zM107.644 342.911v.316l.099.099a1.479 1.479 0 001.817.987l.889-.889.02-.454-.02-.257-.889-.889a1.28 1.28 0 00-.869-.039 1.67 1.67 0 00-.534.257l-.572.572.059.297zM86.448 342.575l.237-.04a1.37 1.37 0 00.988-1.343v-.276l-.87-.87a1.864 1.864 0 00-1.046-.019l-.89.888-.058.297a1.703 1.703 0 00.651 1.165l.119.119.87.079zM81.767 332.955l.316-.079.533-.534a1.11 1.11 0 00.375-.375c.16-.099.296-.235.395-.395l.534-.533.04-.356-.1-.099a1.73 1.73 0 00-.79-1.224l-.118-.119-.316-.039a1.479 1.479 0 00-1.067-.277l-.79.79c-.211.262-.347.576-.395.909l.04.276-.02.257.02.336a1.51 1.51 0 001.343 1.462zM84.296 337.854l.612-.613.04-.316a1.424 1.424 0 00-1.6-1.244l-.652.652a1.367 1.367 0 00.098 1.56c.166.22.395.383.657.467.261.085.542.088.805.007-.014-.172 0-.345.04-.513zM91.327 343.424l-.434-.435-.336-.059-.099-.099a.703.703 0 00-.138.139 1.394 1.394 0 00-1.027 1.541l.81.809.316.079c.24.063.494.056.73-.019a.843.843 0 00.198-.198l.75-.751.02-.335-.375-.376-.415-.296zM59.465 300.934c.36-.162.648-.45.81-.81a1.646 1.646 0 00-.059-1.363l-.494-.493a1.456 1.456 0 00-1.106-.119 1.481 1.481 0 00-.85.849l-.177.178-.02.257a1.584 1.584 0 00.711 1.067 1.426 1.426 0 001.185.434zM88.641 367.128l.119.119.355.079.1-.099a.44.44 0 00.138-.138 1.453 1.453 0 00.968-.771 1.418 1.418 0 00-.02-1.165l-.336-.336a1.202 1.202 0 00-1.403-.494 1.375 1.375 0 00-.454.218l-.632.632.099.098c-.033.225-.02.453.04.672l.533.534.493.651zM94.153 370.743l-.435-.434-.336-.059-.098-.099-.119.118a1.733 1.733 0 00-1.323 1.324l.04.316.69.691c.328.174.71.217 1.067.119.294-.097.56-.267.77-.494l.554-.553.04-.395-.02-.218-.277-.276-.553-.04zM80.384 341.646a.984.984 0 00.257-.059 1.39 1.39 0 00.85-.849 1.569 1.569 0 00-.04-1.265l-.711-.711-.316-.039a.984.984 0 00-.257.059 1.402 1.402 0 00-.85.849 1.454 1.454 0 00.06 1.245l.612.612.297.06.098.098zM80.838 373.272l.139-.138a1.09 1.09 0 00.276-1.304l-.888-.889-.396-.04a1.35 1.35 0 00-.888.889c-.082.279-.075.576.02.85a1.343 1.343 0 001.639.928l.099-.296zM71.021 398.221l-.454-.455c-.153 0-.306.02-.454.059a1.399 1.399 0 00-.988 1.186 1.536 1.536 0 00.81 1.56l.276.277h.237a.672.672 0 00.297-.059 1.376 1.376 0 00.928-.929 1.563 1.563 0 00-.355-1.501l-.297-.138zM79.14 327.957l-.948-.948a1.017 1.017 0 00-.317.079 1.391 1.391 0 00-.908.909 1.619 1.619 0 00.296 1.481l.178.178.434.197.277.04.296-.02.711-.711a1.593 1.593 0 00-.02-1.205zM36.472 363.553l.474-.474c.127-.14.215-.31.257-.494l.02-.257-1.146-1.145-.395-.04c-.311.133-.559.38-.692.691a1.37 1.37 0 00-.098 1.008 1.341 1.341 0 001.58.711zM64.877 373.469a1.506 1.506 0 001.067-1.58l-.04-.237-.177-.415-.553-.553-.435-.079-.257.02-1.106 1.106.02.336.277.039-.257.059.079.475.592.592.474.158.257.02.06.059zM87.85 353.538h-.197l-1.007 1.007-.02.376 1.067 1.066.375.06.217-.02c.366-.153.657-.444.81-.81a1.62 1.62 0 00.04-1.146l-.83-.829-.454.296zM84.217 383.524a1.259 1.259 0 00.77-1.324l-.533-.533a1.23 1.23 0 00-1.343-.395.948.948 0 00-.435.237l-.691.691.02.297a1.343 1.343 0 001.382 1.066l.138-.138.692.099zM125.501 315.927l.415.415.375.02.099-.099.316-.079.81-.81.059-.375-.099-.099a1.477 1.477 0 00-1.145-1.146h-.277l-.513.158-.632.633-.02.256c-.075.274-.061.565.039.83l.395.395.178-.099zM56.107 312.944l.296-.019.514-.237.612-.613.02-.415a1.337 1.337 0 00-.612-1.007 1.205 1.205 0 00-1.028-.158l-.572.573a1.396 1.396 0 00-.08 1.304c.203.289.506.493.85.572zM73.668 313.122a1.48 1.48 0 001.62.711l.75-.75.099-.336a1.762 1.762 0 00-.573-1.284 1.12 1.12 0 00-.711-.316l-.217-.218a1.374 1.374 0 00-1.383-.434c-.087.07-.167.15-.237.237l-.711.711v.316l1.363 1.363zM125.402 336.056l-.316.079-.81.81-.059.336a1.258 1.258 0 00.632.948 1.34 1.34 0 001.166.138.443.443 0 00.138-.138l.79-.79.04-.356-1.107-1.106-.395-.039-.079.118zM114.36 390.635l-.869.869a1.662 1.662 0 00-.138.692l1.145 1.146c.16.006.319-.007.475-.04a1.425 1.425 0 00.987-1.699 1.37 1.37 0 00-1.6-.968zM94.152 377.934a1.225 1.225 0 00-1.066-.474l-.988.987a1.317 1.317 0 00.06 1.047l.69.692.218.019.375-.019a1.26 1.26 0 00.948-1.462l-.138-.138-.099-.652zM70.665 385.618l-.335-.336a1.25 1.25 0 00-.652-.138l-1.027 1.027-.02.375v.198a1.34 1.34 0 001.66.948.823.823 0 00.197-.198l.73-.731.02-.335-.316-.316-.257-.494zM101.263 335.622l-.651.652a1.396 1.396 0 00.414 1.323 1.367 1.367 0 001.265.316 1.26 1.26 0 00.77-.77l.138-.139.04-.316-.099-.098a1.404 1.404 0 00-.632-.87 1.455 1.455 0 00-1.087-.098h-.158zM92.67 303.324l-.296.297-.06.415a1.84 1.84 0 00.02.73l.85.85.336.02.316-.079.395-.396a1.62 1.62 0 00.415-1.679l-.83-.829-.375-.02-.099-.099-.592.593-.08.197zM114.973 368.294l-.415-.415a2.176 2.176 0 00-.652-.138l-.889.889-.336-.02-1.224 1.225a1.286 1.286 0 00.474 1.145 1.174 1.174 0 001.205.218c.357-.099.662-.333.849-.652l.316.039a1.422 1.422 0 001.027-1.303l-.138-.139-.158-.395-.059-.454zM110.291 366.259a1.51 1.51 0 00-1.857-1.224l-.652.651a1.624 1.624 0 00.06 1.324l.77.77.336.02.099-.099.395-.118.81-.81.059-.336-.02-.178zM74.735 356.066a1.482 1.482 0 00.928-1.837 1.315 1.315 0 00-1.64-.889l-.276.277-.336.612-.06.415.514.514.613.336.257.572zM62.626 319.107l.474.474.553.198.336.02 1.066-1.067v-.395l-.138-.138a1.283 1.283 0 00-1.56-.85 1.262 1.262 0 00-.949.949v.513l.218.296zM92.632 359.227l-.652.652a1.451 1.451 0 00.138 1.639 1.395 1.395 0 001.482.455 1.31 1.31 0 00.414-.178l.553-.553a1.396 1.396 0 00-.414-1.245 1.45 1.45 0 00-1.521-.77zM101.461 372.166l-.434-.435-.356-.079h-.198l-.138.138a1.538 1.538 0 00-1.007 1.916l.948.948.355.04.316-.04.949-.948a3.301 3.301 0 00-.04-.987l-.138-.139-.257-.414zM82.557 318.93l.553.553.296.059.099.099.355-.079a1.425 1.425 0 001.028-1.304l-.1-.099-.078-.434-.553-.553-.415-.178a1.514 1.514 0 00-.75 0 2.7 2.7 0 00-.198.197l-.553.553a1.71 1.71 0 00.059 1.324l.257-.138zM94.27 356.284l.751.75c.223.125.476.187.731.178l.237-.039.83-.83.06-.415v-.237l-1.068-1.067-.414.02a1.28 1.28 0 00-.952.553 1.287 1.287 0 00-.174 1.087zM77.876 321.755c.315.166.682.209 1.027.118l.79-.79a.92.92 0 00.06-.691l.651-.652.02-.297a1.646 1.646 0 00-.672-.987l-.118-.119-.316-.079-.1-.099-.216.02-.119.119a1.308 1.308 0 00-.928 1.205l.02.256.197.198a1.232 1.232 0 00-.178 1.086 1.626 1.626 0 00-.118.712zM121.729 359.049l.474.474a1.307 1.307 0 001.225.395l1.027-1.027.059-.375-.119-.119a1.39 1.39 0 00-.533-.809l-.099-.099a1.286 1.286 0 00-1.501-.672l-.612.612a1.507 1.507 0 00.079 1.62zM106.953 381.489l-.395-.395a1.136 1.136 0 00-.593-.118l-1.007 1.007a1.48 1.48 0 00.059 1.087l.771.77.335.02a1.395 1.395 0 001.028-1.304l-.139-.138-.158-.395.099-.534zM143.654 310.85a.962.962 0 00.178-.177l.593-.593.059-.296a1.452 1.452 0 00-.632-1.106 1.416 1.416 0 00-1.205-.139.58.58 0 00-.158.158l-.612.613-.06.336a1.483 1.483 0 00.652 1.046 1.301 1.301 0 001.185.158zM59.683 313.794l.177.178a1.23 1.23 0 001.166.454l.948-.948a1.314 1.314 0 00-.04-1.027l-.83-.83-.374.02-.1.099a1.254 1.254 0 00-.928 1.402l.119.119-.138.533zM101.856 314.169l.277-.079.454-.257.553-.553.04-.434-.119-.119a1.305 1.305 0 00-1.659-.908 2.83 2.83 0 00-.198.197l-.711.711-.039.316a1.48 1.48 0 001.402 1.126zM67.544 363.257a2.82 2.82 0 00.356-.356 1.256 1.256 0 00.04-1.461 1.284 1.284 0 00-1.423-.475 1.204 1.204 0 00-.889.889l-.118.119v.237l1.007 1.007.336-.059.217-.217.475.316zM65.214 344.609a1.09 1.09 0 00.375-.177l.573-.573a1.399 1.399 0 00-.435-1.264 1.479 1.479 0 00-1.303-.356l-.672.672a1.477 1.477 0 00.138 1.6 1.418 1.418 0 001.324.098zM93.461 311.74l.119-.119.079-.276a1.538 1.538 0 00-.158-1.067l-.633-.632-.296-.06-.296.06a1.366 1.366 0 00.672 2.647l.651-.652-.138.099zM40.166 324.303l.395.395.336.059a.448.448 0 00.277-.039 1.372 1.372 0 00.948-1.66 1.369 1.369 0 00-1.62-.987l-.948.948c-.065.275-.051.562.04.829l.118.119.454.336zM111.259 328.905l-.514-.513-.415-.02-.217.02a1.506 1.506 0 00-1.086 1.086l.019.257.119.434.553.554.454.138.237.039c.043-.049.089-.096.139-.138a1.427 1.427 0 001.046-1.56l-.158-.158-.177-.139zM130.894 325.745l.336-.613.059-.375-.099-.099-.099-.296-.928-.928-.395-.04a1.35 1.35 0 00-.814.657 1.334 1.334 0 00-.095 1.042 1.398 1.398 0 001.719.928l.316-.276zM110.765 309.093l.869-.87.04-.355v-.198l-.731-.731a1.338 1.338 0 00-.672-.197l-.118-.119-.257.02-.83.83-.098.296-.119.119.039.355.139.138a1.48 1.48 0 001.738.712zM101.915 353.024l.02-.335-.079-.316-.79-.791-.356-.079h-.197l-.119.119a1.453 1.453 0 00-.83.829 1.402 1.402 0 00-.02 1.166l.83.83c.242.053.493.046.731-.02.101-.027.2-.06.297-.099l.651-.652-.138-.652zM102.883 302.712l.277.277.257-.02.454-.099.672-.671.039-.238.02-.454-.059-.257a1.675 1.675 0 00-.652-.652l-.237.237.079-.316-.494-.138h-.316l-.297.099-.77.77a2.193 2.193 0 00.534 1.64l.493-.178zM143.181 340.481l.217.217.316.119c.247.095.519.102.771.02a1.26 1.26 0 00.869-1.462l-.297-.297a1.278 1.278 0 00-1.343-.75l-.75.75c-.14.225-.209.487-.198.751l.079.277.336.375zM113.59 325.113l.652.651.375.02.118-.118a1.478 1.478 0 00.968-1.798l-.75-.751-.356-.039-.217.02-.83.829a1.602 1.602 0 00.079 1.304l-.039-.118zM121.807 341.706l-.454-.455-.376-.02-.098-.098-1.008 1.007-.02.375-.098.099.829.83c.291.112.61.126.909.039l.296-.691.376-.652-.297-.296-.059-.138zM55.198 359.8l-.75.751-.099.375v.197a1.425 1.425 0 001.62 1.265l.85-.85.078-.355.02-.218a1.59 1.59 0 00-.593-1.027 1.341 1.341 0 00-1.126-.138zM21.3 366.042a1.324 1.324 0 001.185.158l.16-.158.75-.751a1.617 1.617 0 00-.514-1.422 1.374 1.374 0 00-1.205-.257c-.31.087-.576.283-.75.553l-.297.297-.099.335-.02.218a1.4 1.4 0 00.79 1.027zM47.06 344.234l.593-.593.178-.493v-.277a1.425 1.425 0 00-1.186-1.185l-.098.099-.376.019-.256.06-.455.454-.217.494-.04.276a1.426 1.426 0 001.304 1.304l.119-.118.434-.04zM64.68 354.071l-.395-.039a1.339 1.339 0 00-.89.889c-.073.239-.066.496.02.731a1.337 1.337 0 001.108 1.033c.177.027.358.018.532-.026l.474-.474a.922.922 0 00.257-.494l.02-.217a1.451 1.451 0 00-.099-.573l-.533-.533-.494-.297zM58.24 334.002a.946.946 0 00.494.256h.277l.474-.118.592-.593.119-.474v-.276l-1.047-1.047-.395-.04-.119.119a1.425 1.425 0 00-1.027 1.303l.158.159.198.434.276.277zM62.685 351.049l.119.119H63a1.21 1.21 0 00.415-.218l.356.04.217-.02a1.392 1.392 0 00.85-.849 1.233 1.233 0 00-.08-1.107l-.73-.73-.336-.02-.237.237-.238-.237-.237.039a1.263 1.263 0 00-.928 1.066c-.02.166-.006.334.04.495.07.253.223.476.434.632l.158.553zM83.722 367.188a1.256 1.256 0 00-1.323-.376 1.374 1.374 0 00-.455.218l-.553.553a1.344 1.344 0 00.514 1.106 1.371 1.371 0 001.304.237 1.227 1.227 0 00.908-1.422l-.178-.178-.217-.138zM61.342 351.523l-.237.237a1.23 1.23 0 00.138 1.284 1.334 1.334 0 001.146.553l.138-.138.415-.217.474-.474.06-.376v-.197l-.12-.395-.611-.613-.475-.118h-.276l-.454.454h-.198zM32.521 347.79c.073-.058.14-.124.197-.198l.336-.336a1.443 1.443 0 00.633-1.224 1.313 1.313 0 00-1.008-1.245h-.277l-.454.138-1.146 1.146-.059.375.119.119a1.317 1.317 0 00.532 1.015 1.306 1.306 0 001.127.21zM45.301 336.708l.573.573a1.292 1.292 0 001.205.217.725.725 0 00.336-.138l.75-.751-.217-.217-.177-.494-.613-.612-.415-.06h-.237a.946.946 0 00-.256.06 1.225 1.225 0 00-.73.731 1.2 1.2 0 00-.219.691zM58.26 374.398l-.474-.474-.395-.119h-.197a.602.602 0 00-.119.119 1.305 1.305 0 00-.79.79 1.397 1.397 0 00-.02 1.244l.751.751.356.039h.197a1.366 1.366 0 00.948-1.659l-.118-.118-.139-.573zM51.425 348.679l-.04.276-.098.099a1.396 1.396 0 00.672 1.027 1.31 1.31 0 001.125.138l.79-.79a1.31 1.31 0 00-.059-1.086l-.691-.692a1.514 1.514 0 00-.75 0l-.218.218-.593.592-.138.218zM40.778 353.044a1.34 1.34 0 00.158 1.264l.73.731h.317a1.425 1.425 0 001.047-1.284l-.06-.296-.552-.553a1.504 1.504 0 00-1.047-.099 1.282 1.282 0 00-.593.237zM77.836 359.879l-.395-.04a1.375 1.375 0 00-.81.81 1.426 1.426 0 00-.08.909c-.026.123-.033.25-.02.375l.93.929c.183.091.387.132.592.118l.948-.948c.084-.206.118-.43.099-.652l-.099-.099.02-.217a1.483 1.483 0 00-.593-.988l-.138-.138-.454-.059zM74.655 347.572l-.553-.553a1.48 1.48 0 00-1.007-.059 1.372 1.372 0 00-1.008 1.007 1.454 1.454 0 00.534 1.443l.336.335.158-.158a1.387 1.387 0 00.85-.849 1.367 1.367 0 00-.02-1.245l.71.079zM70.211 337.636a1.511 1.511 0 00.138 1.284l.494.494c.294.125.621.146.929.059.213-.066.404-.188.553-.355l.415-.415.079-.316a1.31 1.31 0 00-.593-1.067 1.487 1.487 0 00-1.165-.177 1.42 1.42 0 00-.85.493zM56.048 323.908l.652.651.375.02 1.146-1.146.04-.355-1.206-1.205h-.395l-.177.178a1.425 1.425 0 00-.77 1.837l.335.02zM54.35 367.721a1.397 1.397 0 00-1.7-.988l-.83.83a1.366 1.366 0 00.672 1.817l.336.336c.114-.008.226-.028.336-.059a1.37 1.37 0 001.185-1.936zM74.991 338.229a1.506 1.506 0 00.08 1.62 1.335 1.335 0 001.422.513l.572-.572a1.29 1.29 0 00.119-1.107 1.485 1.485 0 00-.909-.908l-.099-.099-.375-.02-.632.632-.178-.059zM73.945 367.01l-.593-.593-.296-.02-.138.139a1.509 1.509 0 00-.988.79c-.141.377-.134.793.02 1.165l.671.672.237.039c.242.054.493.047.731-.019a.844.844 0 00.198-.198l.671-.672c.079-.35.022-.716-.158-1.027l-.355-.276zM72.562 342.436a1.426 1.426 0 00-.118.672l.908.909c.194.102.414.143.632.118l.909-.908c.093-.211.134-.442.118-.672l-.888-.889a1.608 1.608 0 00-.652-.138l-.909.908zM70.192 343.424l-.139-.138-.355-.079-.217-.02a1.51 1.51 0 01-.257.059 1.403 1.403 0 00-.85.85 1.51 1.51 0 00.02 1.244l.691.692c.206.022.415.001.613-.059a1.372 1.372 0 001.007-1.205l-.118-.119-.119-.395-.355-.356.079-.474zM74.833 344.116l-.908.908a1.481 1.481 0 00.118 1.62 1.283 1.283 0 001.402.494l1.127-1.126c.102-.194.143-.414.118-.632l-.928-.929a1.192 1.192 0 00-.612-.098l-.317-.237zM78.014 352.787l-.79.791a1.227 1.227 0 00-.119.632l.929.928c.183.092.388.133.592.119l.138-.139.376-.178.513-.513c.103-.202.15-.426.139-.652l-.929-.928a1.557 1.557 0 00-.632-.119l-.217.059zM65.588 335.503l.652.652.336.02.099.098.375-.138.395-.395a1.531 1.531 0 001.141-1.355 1.53 1.53 0 00-.074-.62l-.612-.613-.336-.098-.099-.099c-.16.007-.32.033-.474.079a1.508 1.508 0 00-.81.81 1.367 1.367 0 00-.77 1.837l.177-.178zM61.974 338.842l-.119-.119-.217.02a1.248 1.248 0 00-.83.829 1.48 1.48 0 00.277 1.502 1.399 1.399 0 001.363.375 1.259 1.259 0 00.711-.711 1.37 1.37 0 00.04-1.146l-.119-.118a1.282 1.282 0 00-1.106-.632zM61.381 363.494l-.316.079-.177.099-.277.237-.158.158-.119.237-.04.197-.058.376.04.197.039.158.098.178.218.296.158.119.237.118.197.079.376.02h.197l.296-.099.198-.079.296-.257.119-.158.138-.217.04-.198.059-.375-.02-.217-.06-.138-.098-.178-.217-.296-.178-.139-.198-.118-.217-.059-.355-.04-.218.02zM40.6 332.915l.336.336c.187.082.39.122.593.119l1.007-1.008a1.336 1.336 0 00-.079-1.066l-.77-.771-.336-.02-1.086 1.087-.04.395v.197l.237.237.138.494zM52.057 323.433l.83-.829.06-.375-.12-.119a1.393 1.393 0 00-1.101-1.206c-.19-.04-.387-.04-.578.001l-.75.751-.099.335-.02.218a1.396 1.396 0 00.553 1.224 1.284 1.284 0 001.225 0zM29.44 313.853a1.45 1.45 0 00.926-1.438 1.504 1.504 0 00-.156-.596l-.79-.791a1.946 1.946 0 00-.672.04 1.373 1.373 0 00-.69.691l-.179.178a1.426 1.426 0 00.613 1.008 1.26 1.26 0 00.948.908zM50.32 361.36l.71-.711.08-.276-.06-.415-.948-.948-.415-.059h-.237l-.909.908a1.396 1.396 0 00.435 1.304 1.485 1.485 0 001.343.197zM34.556 334.337a.838.838 0 00.197-.197l.731-.731.02-.336-.79-.79a1.575 1.575 0 00-.652-.138l-1.047 1.047-.04.395.119.118a1.312 1.312 0 001.462.632zM67.979 347.533l-.297.296-.099.336-.296.296-.06.375.02.218 1.028 1.027.355.039a1.312 1.312 0 00.79-.79l.613-.612.138-.454.02-.297a.53.53 0 00.079-.276l.02-.257-.139-.454-.572-.573-.474-.158h-.277a1.335 1.335 0 00-.909.908l.06.376zM55.04 337.952l.415.415.356.04.217-.02c.352-.157.633-.438.79-.79a1.4 1.4 0 00.02-1.166l-.77-.77-.356-.039-.099-.099a1.508 1.508 0 00-.81.81 1.48 1.48 0 00-.02 1.126l.139.138.118.355zM53.42 389.825l-.138.138-.02.257c.098.368.33.686.652.889a1.354 1.354 0 001.067.119.84.84 0 00.197-.198l.751-.75.02-.297-.099-.099-.099-.414-.592-.593-.474-.118-.297.019-.098.099a1.311 1.311 0 00-.87.948zM115.486 309.942l-.454.454a1.32 1.32 0 00.276 1.265c.178.175.406.292.652.335l.138.139.415.059.217-.02.139-.138a1.711 1.711 0 00.968-.968l.158-.158-.099-.099v-.237a1.448 1.448 0 00-.435-.79l-.75-.751-.257.02a.655.655 0 00-.178.178l-.138.138-.652.573zM106.479 286.988l.592.593h.316a1.395 1.395 0 001.028-1.304l-.712-.711a1.531 1.531 0 00-1.066-.119 1.494 1.494 0 00-.81.81 1.404 1.404 0 00.138 1.245l.514-.514zM93.184 335.878l-.257-.019-.494.138-.612.612-.099.494v.277c.04.376.02.756-.059 1.126l.02.256.138.415.573.573.435.119h.276a1.316 1.316 0 00.85-.85l.177-.178.119-.434.118-.119.257-.494.06-.296-.08-.355.119-.119a1.311 1.311 0 00-1.54-1.146zM95.14 340.679l-.355.355a1.564 1.564 0 00.869 1.501l.316.04.573-.099.316-.316.395-.197.533-.534.06-.415v-.237l-.08-.316-.81-.81-.217-.019-.296-.06-.573.099-.75.751.02.257zM92.296 329.458l.612.613.454.02.237-.04a1.302 1.302 0 00.79-.79 1.257 1.257 0 00-.079-1.067l-.79-.79-.375.02a1.311 1.311 0 00-.968 1.402l.158.158-.04.474zM85.106 310.14l.69-.692a1.228 1.228 0 00-.058-1.047 1.427 1.427 0 00-.85-.612l-.138-.138-.415-.02a1.254 1.254 0 00-.671.671 1.285 1.285 0 00-.1.968 1.398 1.398 0 001.542.87zM115.862 371.198l.276.276a1.286 1.286 0 00.968-.217 1.386 1.386 0 00.534-.929 1.32 1.32 0 00-.565-1.366 1.316 1.316 0 00-.482-.194l-.376.099-.889.889.099.375.178.178.257.889zM128.03 319.957l.395-.079.771-.77c.036-.162.043-.33.019-.494l-.079-.395-.75-.751a1.208 1.208 0 00-.968.217 1.315 1.315 0 00-.534.85 1.396 1.396 0 001.146 1.422zM123.15 380.541l-.039-.356-.257-.256a1.39 1.39 0 00-.908-.909 1.308 1.308 0 00-.395 0l-.929.928-.039.316.098.099.079.356.79.79.218.02a1.344 1.344 0 001.6-.929l-.218-.059zM116.533 354.565l-.059-.296-.277-.514-.217-.217-.415-.138a1.402 1.402 0 00-1.066.276 1.729 1.729 0 00-.534 1.166l.929.928.395.119a1.34 1.34 0 001.501-1.106l-.158.158-.099-.376zM112.069 380.956l-.711-.711a.98.98 0 00-.376-.099 1.26 1.26 0 00-1.125.336 1.39 1.39 0 00-.356 1.304l.928.928a1.424 1.424 0 001.442-.612l.178-.178.079-.435v-.237l-.059-.296z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M117.777 341.982l.178.178a1.192 1.192 0 00.889.612 1.737 1.737 0 001.284-.375l.494-.494.02-.257.098-.098-.098-.336-.119-.119a1.456 1.456 0 00-.909-.632 1.308 1.308 0 00-.691-.691l-.118-.119-.415.02-.889.889a1.587 1.587 0 00.276 1.462v-.04zM127.852 330.604l.435.435.257.059a1.375 1.375 0 001.402-.652 1.509 1.509 0 00-.138-1.718l-.415-.415-.277-.079a1.39 1.39 0 00-1.402.652 1.509 1.509 0 00.138 1.718zM94.034 395.08l-.217-.02-.375.059a1.238 1.238 0 00-.909.672l-.118.118c.048.298.185.573.394.791l.475.474.237.039.098-.099.356-.079.138-.138a1.564 1.564 0 00.613-1.481 1.286 1.286 0 00-.692-.336zM136.128 353.538l.988.988a1.454 1.454 0 001.501-.751l.139-.138v-.435l-.02-.217-.435-.435a.998.998 0 00-.415-.414l-.138-.139-.415.02-.533.533a1.378 1.378 0 00-.415 1.008l-.257-.02zM133.679 368.037a1.308 1.308 0 001.446-.797 1.43 1.43 0 00.115-.605l-.119-.119-.118-.316-.948-.948-.376.099-.987.987.019.218.119.395.711.711.138.375zM114.874 348.995h.198l.375-.02.316-.316c.243-.13.443-.329.573-.573l.118-.118.02-.376-.711-.711a1.793 1.793 0 00-.691-.296.762.762 0 00-.297-.02l-.79.79c-.094.338-.081.698.04 1.027.193.307.497.527.849.613zM101.797 416.275l-.455-.454a1.773 1.773 0 00-.77-.375h-.514l-.335.335a1.37 1.37 0 00-.514 1.502l.83.829.316.079a1.453 1.453 0 001.501-.75l.158-.158.04-.553-.257-.455zM136.8 360.531l-.158.158-.059.454.02.257.988.988c.334.067.681.007.973-.169.293-.175.509-.453.607-.78l.02-.217-.237-.237a1.537 1.537 0 00-.909-.592 1.47 1.47 0 00-1.245.138zM131.723 391.327c-.171.222-.274.49-.296.77l.02.217.118.356.83.829a1.477 1.477 0 001.422-.592l.158-.158.06-.455-.02-.256-.138-.376-.139-.138a1.317 1.317 0 00-.553-.553l-.138-.138-.474.039-.553.553-.297-.098zM136.228 358.022c.123.029.25.036.375.02l.376-.375a1.434 1.434 0 00.612-1.205 1.566 1.566 0 00-.79-1.265l-.099-.098-.277-.04-.098-.099-.178.178a1.54 1.54 0 00-.613 1.561 1.338 1.338 0 00.692 1.323zM140.692 395.356l-.059-.335-.83-.83a1.339 1.339 0 00-1.106.276 1.463 1.463 0 00-.533 1.008l.948.948a.95.95 0 00.355.119 1.319 1.319 0 001.462-.909v-.237l-.237-.04zM129.491 379.455l.81-.81.02-.218-.059-.296-.119-.118a1.439 1.439 0 00-1.027-.711 1.396 1.396 0 00-1.6 1.125v.237l.079.435.573.573c.197.163.437.265.691.296h.237l.376-.099.019-.414zM127.595 361.242a.978.978 0 00-.257.454 2.97 2.97 0 00-.513.514 1.316 1.316 0 00-.672.276l-.869-.869a1.717 1.717 0 00-.435-.158 1.308 1.308 0 00-.824.162c-.25.142-.446.363-.558.628l-.119.119-.02.375 1.146 1.146.356-.079.138-.138.277.276a1.336 1.336 0 00.967.968l.356.356.119-.119c.094-.076.181-.162.256-.257l.455-.454c.207-.116.378-.287.494-.494l.434-.434a1.177 1.177 0 00.652-.652l.02-.297-.02-.256-.099-.415-.533-.534-.415-.177-.237-.04a2.22 2.22 0 00-.573.138c.159.018.32.004.474-.039zM172.139 352.886l.395-.079.435-.434a1.091 1.091 0 00.375-.968 1.14 1.14 0 00-.494-.889l-.434-.435a1.273 1.273 0 00-.492.018 1.274 1.274 0 00-.781.567 1.25 1.25 0 00-.169.462c-.063.303-.014.619.138.889a1.313 1.313 0 001.027.869zM141.936 358.871l-.356.079a1.228 1.228 0 00-1.165 1.166l.395.395c.147.347.423.623.77.77l.119.119.138-.139a1.542 1.542 0 00.613-1.6 1.309 1.309 0 00-1.067-1.066l.553.276zM139.487 368.472l.573.573a1.427 1.427 0 001.62-1.186 1.597 1.597 0 00-.257-1.126 1.374 1.374 0 00-.909-.553l-.415.099-.138.138a1.335 1.335 0 00-.783.865 1.339 1.339 0 00.19 1.15l.119.04zM142.964 363.731l.888.889c.232.041.471.021.692-.059l.415-.415a1.404 1.404 0 00.197-1.383 1.29 1.29 0 00-1.501-.553 1.902 1.902 0 00-.277.276 1.356 1.356 0 00-.415 1.126l.001.119zM147.369 358.852l-.218.494-.059.296.059.336.929.928a1.424 1.424 0 001.659-1.027c.147-.279.19-.602.119-.909a1.323 1.323 0 00-.672-.908l-.099-.099-.296-.06-.454.455c-.183.171-.324.38-.415.612l-.553-.118zM141.739 332.086a1.348 1.348 0 001.225-.435 1.287 1.287 0 00.237-1.264l-.929-.929-.256.02-.415.415a1.309 1.309 0 00-.139 1.284c-.021.326.077.65.277.909zM156.949 398.3l-.415.019-.474.474a1.04 1.04 0 00-.494.494l-.118.119.039.395.474.474c.112.187.268.343.455.454l.138.139.395-.04.474-.474c.196-.099.355-.259.454-.454l.119-.119v-.395l-.494-.494a1.44 1.44 0 00-.435-.434l-.118-.158zM143.221 390.299l-.317-.316a1.092 1.092 0 00-.434-.434l-.277-.04-.494.059-.948.949a1.278 1.278 0 00.215.995 1.3 1.3 0 00.872.526l.296.02.119-.119.296-.099.75-.75.001-.198-.079-.593zM154.934 378.032l.415.415.217.02.395-.119.731-.731a1.33 1.33 0 00-.204-1.012 1.331 1.331 0 00-.863-.568 1.401 1.401 0 00-1.442.771 1.34 1.34 0 00.751 1.224zM146.065 375.603l-.79-.79-.257-.06a1.35 1.35 0 00-1.205.336 1.376 1.376 0 00-.395 1.185l.02.218.079.355.987.988.218-.02.355-.079.85-.849.079-.474.059-.81zM113.53 376.531l-.118-.118a1.456 1.456 0 00-1.435-.745 1.454 1.454 0 00-1.236 1.041 1.456 1.456 0 00.123 1.106l.572.573.613-.375.217-.02a1.4 1.4 0 001.126-1.126l.138-.336zM161.235 415.347l-.02.257.079.474.317.316a1.508 1.508 0 001.027.75h.237l.276-.078.415-.415c.222-.289.34-.644.336-1.008l-.81-.81a2.149 2.149 0 00-.612-.217 1.393 1.393 0 00-1.245.731zM65.234 377.42a1.288 1.288 0 00-1.324.612l-.079.317.02.375c-.03.318.061.636.257.889a1.234 1.234 0 00.849.494l.099.098.118-.118a1.286 1.286 0 00.909-1.343 1.454 1.454 0 00-.85-1.324zM145.334 310.041a1.395 1.395 0 001.165 1.481h.514l.415-.415a1.34 1.34 0 00.494-1.244l-.06-.296-.592-.593-.237.237.217-.217a1.422 1.422 0 00-.672-.316l-.79.79-.079.355-.375.218zM146.361 418.211l-.751-.751a1.426 1.426 0 00-1.6 1.087 1.285 1.285 0 00.869 1.541l.099.099.297-.02.158-.158a1.372 1.372 0 00.869-1.581l.059-.217zM136.228 380.897a1.31 1.31 0 001.205-.455l.178-.177.078-.474.099-.099-.118-.119-.119-.395-.829-.829a1.28 1.28 0 00-1.482.77l-.118.118.019.336.336.336c.039.212.127.412.257.584.131.171.3.31.494.404zM114.814 376.828l.554.553a1.539 1.539 0 001.343-.356c.081-.063.154-.136.217-.217.353-.155.635-.437.79-.79a1.871 1.871 0 00-.276-2.015l-.534-.534-.256-.019-.119.118-.356.119-.276.276a1.443 1.443 0 00-.929.652l-.434.435a1.462 1.462 0 00.118 1.146l.178.177-.02.455zM95.1 328.984l1.107 1.107a1.23 1.23 0 001.007-.336 1.266 1.266 0 00.395-.988l-.099-.099a1.257 1.257 0 00-.85-1.086h-.552l-.672.672a1.278 1.278 0 00-.316.987l-.02-.257zM108.434 344.688l-.494.494a1.529 1.529 0 00-.063.637c.023.215.092.422.201.608a1.377 1.377 0 00.948.592l.119.119.316-.119.356-.355a1.348 1.348 0 00.434-1.027l-.77-.771a1.496 1.496 0 00-.672-.316l-.296-.02-.296.297.217-.139zM125.225 358.634l.257-.019.454-.139.751-.75.039-.514-.059-.257-.237-.237a1.14 1.14 0 00-.375-.375l-.198-.198-.592-.079-.297.06-.513.513c-.278.245-.46.581-.514.948l.119.119.118.395.514.514c.173.049.356.056.533.019zM126.904 348.501l-.139-.138a1.36 1.36 0 00-1.129.391 1.36 1.36 0 00-.392 1.13l.099.335.514.514.039.237.968.968.217-.02.415-.098.494-.494.178-.376a1.257 1.257 0 00-.316-1.106 1.45 1.45 0 00-.218-.889l-.434-.434-.296-.02zM139.803 380.758l-.336.336a1.509 1.509 0 001.205 1.679l.217-.019.395-.079 1.067-1.067-.02-.257-.138-.138-.158-.356-.158-.158a1.45 1.45 0 00-.79-.434 1.848 1.848 0 00-.593.039l-.474.474-.217-.02zM120.859 371.376l.257.059a1.365 1.365 0 001.462-.79l.316-.317.019-.256-.118-.119-.138-.375-.139-.138a1.837 1.837 0 00-.928-.494l-.296-.02-.099.099-.277.079-.612.612a1.62 1.62 0 00.099 1.324l.434.434.02-.098zM120.8 360.689l.297.059.553-.079.138-.138c.18-.042.352-.108.514-.198.181-.028.36-.075.533-.138l.81-.81a1.714 1.714 0 00-.198-1.422l-.355-.356a1.509 1.509 0 00-.771-.77l-.296-.02-.277.277a2.085 2.085 0 00-.711.711l-.059.257-.197.197a1.695 1.695 0 01-.119.316 1.51 1.51 0 00.138 2.114zM120.958 366.18l.909-.908a1.363 1.363 0 00-.051-1.111 1.375 1.375 0 00-.858-.707l-.217-.019-.158.158a1.366 1.366 0 00-.593 1.461 1.341 1.341 0 001.087 1.087l-.119.039zM124.178 373.548l-.257-.059a1.397 1.397 0 00-1.6.968l.04.356.651.651c.199.162.438.264.692.297l.257.02.118-.119.316-.119.889-.888-1.106-1.107zM114.538 322.15a1.323 1.323 0 001.086-.297 1.554 1.554 0 00.554-1.264l-.435-.435a1.512 1.512 0 00-1.58-.541c-.2.051-.388.141-.553.265l-.494.494c-.07.213-.09.44-.059.662.03.223.111.435.236.622a1.365 1.365 0 001.245.494zM112.187 394.467l.652.652a1.54 1.54 0 001.205-.296 1.344 1.344 0 00.474-1.067l-.296-.296a1.228 1.228 0 00-1.047-1.047 1.616 1.616 0 00-.533.02l-.771.77a1.42 1.42 0 00-.257.612l.099.099.079.356.138.138.257.059zM115.605 400.097l-.909-.909a1.79 1.79 0 00-1.758.692l-.118.118-.099.297-.099.098v.198l.099.099.099.296.118.119a1.429 1.429 0 00.929.572c.295.048.597.007.869-.118l1.027-1.027-.118-.119-.04-.316zM115.94 368.077a1.368 1.368 0 001.166-.297 1.478 1.478 0 00.513-1.145l-.079-.316-.691-.692a.752.752 0 00-.178-.178 1.429 1.429 0 00-1.177.336 1.42 1.42 0 00-.482 1.126l.079.316.849.85zM102.528 396.621l.118-.119a1.117 1.117 0 00.83-1.067v-.276l-.948-.948a1.333 1.333 0 00-1.186.395 1.424 1.424 0 00-.375.77 1.326 1.326 0 00.219 1.006 1.337 1.337 0 00.867.555l.475-.316zM82.873 414.853l-.632-.632a1.477 1.477 0 00-.454-.138 1.39 1.39 0 00-1.403.612 1.537 1.537 0 00.08 1.778l.454.454.296.099a1.366 1.366 0 001.264-.435 1.587 1.587 0 00.316-1.501l.08-.237zM120.583 354.941l-.099-.099-.079-.277-.672-.671-.316-.079a1.453 1.453 0 00-1.224.395 1.562 1.562 0 00-.376 1.442l.929.928.257.02.474-.079.177-.178a1.643 1.643 0 00.929-1.402zM66.734 368.689l.376-.099.197-.197c.16-.101.294-.236.395-.395a1.28 1.28 0 00.198-.198l-.06-.454-.079-.395-.71-.712a1.314 1.314 0 00-.949.238 1.366 1.366 0 00-.142 2.06c.185.185.42.313.675.369l.1-.217zM88.028 389.904a1.538 1.538 0 001.68-.968l.04-.237.019-.454-.988-.988-.296-.059h-.237l-.395.119-.731.731.197.197-.276-.079-.099.494v.237l.08.316.73.731.276-.04zM117.383 381.845l-.118-.119-.238-.039-.217.02-.355.079-.85.849-.039.237.098.099.08.355.849.85a1.318 1.318 0 001.086-.257 1.304 1.304 0 00.534-1.007l.098-.099-.118-.356-.435-.434-.375-.178zM101.125 407.347a1.257 1.257 0 001.284-.85l.02-.296-.099-.494a1.23 1.23 0 00-.987-.988 1.045 1.045 0 00-.494.02l-.929.929a1.288 1.288 0 00.751 1.541l.237.039.217.099zM167.576 364.244l.81.81.336.06.098-.099.356-.079.83-.83a1.54 1.54 0 00-.553-1.541l-.198-.197-.533-.059-.277.039-.632.632a1.398 1.398 0 00-.316.751l.119.118-.04.395zM106.321 331.552l.296.099.534.02.296-.06.83-.829a1.324 1.324 0 00-.099-1.166 1.342 1.342 0 00-.869-.592h-.237l-.237.237a1.314 1.314 0 00-.633 1.145c-.059.189-.079.389-.059.586.021.197.081.387.178.56zM121.768 338.427l.454.454v.316a1.455 1.455 0 001.245 1.244l.098.099.336-.059.751-.751c.204-.441.225-.945.059-1.402a1.59 1.59 0 00-.573-.573l-.099-.099a1.254 1.254 0 00-1.007-1.007.95.95 0 00-.336-.02l-1.047 1.047-.039.237.019.217.139.297zM158.806 382.339l-.316-.04-.356.079-.81.81a1.37 1.37 0 00.158 1.146 1.46 1.46 0 001.008.612l.118.119.336-.099.81-.81.02-.217.098-.099-.079-.395-.711-.711-.276-.395zM125.264 426.784l-.257-.02-.454.099-.474.474a1.45 1.45 0 00-.494.494l-.119.119v.355l.692.692a1.402 1.402 0 001.782-.355c.136-.173.229-.377.272-.594a1.343 1.343 0 00-.948-1.264zM112.504 406.596a1.266 1.266 0 00-.83-.83l-.138-.138-.415.02-.553.553a1.398 1.398 0 00-.415.968l.119.118.138.336.474.474.336.139.099.098a1.315 1.315 0 001.481-.888v-.238l-.296-.612zM88.009 403.356l-.316-.316a1.32 1.32 0 00-.474-.474l-.257-.019-.474.079-.929.928a1.282 1.282 0 00.578 1.361c.149.093.315.154.49.18a.76.76 0 00.295.02l1.067-1.067v-.198l.02-.494zM137.215 371.573l-.454.454a1.368 1.368 0 00-.217 1.363 1.28 1.28 0 001.027.83 1.195 1.195 0 001.027-.316l.593-.593a1.316 1.316 0 00-.198-1.027 1.29 1.29 0 00-.908-.593l-.87-.118zM143.378 338.782l-.612.612a2.049 2.049 0 00-.297.652l.02.336.257.534.454.454.316.079.573-.059.119.118a1.564 1.564 0 001.086-1.323l-.099-.099-.118-.356-.83-.829-.296-.06-.217.02-.356.079v-.158zM135.457 406.912l-.414-.415a1.344 1.344 0 00-.455-.454l-.118-.119-.415.02-.375.376-.336-.336h-.237l-.119-.119-.276.04-.633.632a1.287 1.287 0 00-.079 1.224 1.426 1.426 0 002.035.494l.336.336a1.421 1.421 0 001.481-.691l.158-.158.001-.435-.554-.395zM132.099 403.04a1.52 1.52 0 00-.158-1.224 1.592 1.592 0 00-.988-.672l-.257-.02-.158.158a1.692 1.692 0 00-.494 1.205l.119.119.099.336.513.513.395.079.099-.099.356-.079.671-.671-.197.355zM104.523 378.388a1.41 1.41 0 001.066-.273 1.427 1.427 0 00.554-.952 1.292 1.292 0 00-.237-.987 1.342 1.342 0 00-.85-.534l-.395.079-.988.988.099.415.474.474.277.79zM109.521 339.928l.04.355.849.85.237.039h.237l.375-.099.514-.513.158-.356.138-.138a1.232 1.232 0 00-.737-1.417 1.234 1.234 0 00-1.515.508l-.237.474-.059.297zM119.2 389.114l-.217.02-.138.138a1.512 1.512 0 00-.593 1.541 1.396 1.396 0 001.106 1.106l.455.02.474-.474a1.455 1.455 0 00-.87-2.213l-.217-.138zM121.57 404.562l-.81-.81-.257-.06a1.515 1.515 0 00-1.127.284 1.505 1.505 0 00-.591 1l-.04.237.06.415.948.948.316.079.118.119.277-.04.592-.592c.19-.289.324-.611.396-.948l.098-.099.02-.533zM127.576 348.382l.849.85.336.099a1.45 1.45 0 001.501-.751l.119-.435v-.276l-.652-.652a1.649 1.649 0 00-.692-.296.757.757 0 00-.296-.02l-.336.336a1.678 1.678 0 00-.494 1.205l-.335-.06zM121.886 387.257l.099.099-.059.534.336.335a1.3 1.3 0 00.553.553l.138.139h.435l.967-.968.04-.237v-.237l-.118-.395-.771-.771a1.289 1.289 0 00-1.403.473 1.29 1.29 0 00-.217.475zM122.124 348.896c.208.288.519.486.869.553l.415-.099.414-.415a1.2 1.2 0 00.396-.592l.869-.869a1.543 1.543 0 00-.178-1.166l-.83-.83a1.344 1.344 0 00-1.383.672l-.158.158v.277a1.283 1.283 0 00-.632.908 1.563 1.563 0 00.218 1.403zM145.531 401.5l.178.177.059.376a1.208 1.208 0 00.889.889l.119.118.375-.059 1.027-1.027a1.34 1.34 0 00-.138-.968l-.099-.099a1.285 1.285 0 00-1.165-1.165h-.198l-.138.138a1.564 1.564 0 00-.909 1.62zM122.124 414.952a1.467 1.467 0 00-.435-.435l-.138-.138-.415.02-.553.553a1.228 1.228 0 00-.415 1.007l1.086 1.087a1.402 1.402 0 001.502-.751l.138-.138.02-.415-.79-.79zM186.125 367.583a.592.592 0 00.276.039l.909-.908a1.57 1.57 0 00-.118-1.265 1.517 1.517 0 00-1.008-.651l-.257-.02-.098.099-.297.098-.612.613a1.49 1.49 0 00.118 1.224 1.292 1.292 0 001.087.771zM109.125 333.883l.099.099a1.286 1.286 0 00.869.869l.119.119.395-.04.533-.533a1.45 1.45 0 00.415-.968l-.119-.119-.158-.355-.454-.455-.355-.157a1.314 1.314 0 00-1.482.849l-.118.118.256.573zM147.053 352.412l.157.158.534-.02.968-.968a1.27 1.27 0 00.003-.521 1.276 1.276 0 00-.581-.839 1.28 1.28 0 00-.489-.181l-.296-.019-.119.118-.296.099-.731.731a1.367 1.367 0 00.85 1.442zM94.824 381.726c.162-.027.317-.088.454-.177a1.232 1.232 0 00.672-1.304 1.313 1.313 0 00-1.067-1.067 1.209 1.209 0 00-1.165.454l-.494.494.04.277.177.494.455.454.335.099.12.118.473.158zM100.829 364.027c.137.026.277.033.415.02l.652-.652a1.457 1.457 0 00.178-1.324 1.518 1.518 0 00-.948-.948l-.119-.118-.138.138a1.48 1.48 0 00-.573 1.521 1.398 1.398 0 00.533 1.363zM140.514 346.624l.336-.335c.221-.296.333-.659.316-1.028l-.731-.731a1.371 1.371 0 00-1.751 1.037 1.374 1.374 0 00.922 1.571l.098.099.317-.04.118-.118.375-.455zM87.022 334.891l.73.731a.532.532 0 00.178.177 1.304 1.304 0 001.013-.208 1.318 1.318 0 00.548-.878 1.341 1.341 0 00-1.027-1.58h-.237l-.415.098-.395.395c-.18.212-.291.474-.316.751v.198l-.08.316zM149.126 369.776l-1.027-1.028a1.455 1.455 0 00-1.422.593l-.158.158-.059.454.019.257.988.988a1.342 1.342 0 001.58-.948l.079-.474zM168.189 375.445l.889-.889.039-.316-.039-.316-1.067-1.067a1.352 1.352 0 00-1.353.639 1.279 1.279 0 00-.168.487 1.369 1.369 0 001.126 1.6l.573-.138zM157.265 351.721l.118-.119.376-.099.849-.849.02-.217-.099-.099-.079-.316-.119-.119a1.591 1.591 0 00-.513-.513l-.435-.435-.256.02-.435.158-.533.533-.119.356-.119.118a1.485 1.485 0 001.344 1.581zM130.598 387.119l.059-.296-.099-.099-.079-.356-.928-.928a1.534 1.534 0 00-1.146.277 1.458 1.458 0 00-.494 1.046l.079.316.534.534c.186.172.42.282.671.316l.297.02.118-.119.218.059.77-.77zM152.919 342.555l.553.553.454.099.257-.02.632-.632.218-.415.039-.237a1.863 1.863 0 00-.277-.869l-.553-.553-.336-.059-.276.039-.494.217-.197.198a2.15 2.15 0 00-.237 1.699l.217-.02zM172.91 394.033l.533.533c.18.188.416.313.672.356a1.257 1.257 0 001.422-.949l-.118-.118-.02-.217a1.286 1.286 0 00-.869-1.264l-.217-.02-.376.02-.336.335a1.317 1.317 0 00-.553.553l-.039.317-.099.454zM152.9 367.504l1.007 1.007a1.452 1.452 0 001.64-1.205l.02-.217-.079-.395-1.008-1.007h-.237l-.118.118-.336.138-.139.139a1.553 1.553 0 00-.493 1.205l-.257.217zM153.117 385.894l-.988-.987h-.197l-.376.098-.948.949.099.098.079.316.435.435c.218.221.502.367.81.415l1.047-1.047.039-.237-.02-.217.02.177zM85.263 373.469l-.217-.019-.376.019-.869.87a1.418 1.418 0 00.074 1.099 1.433 1.433 0 00.835.718l.217.02.356-.04.85-.849a1.453 1.453 0 00-.08-1.186 1.346 1.346 0 00-.79-.632zM51.998 364.462a1.393 1.393 0 001.007.652c.083.027.171.034.257.019l.237-.039.573-.573A1.565 1.565 0 0054.21 363a1.308 1.308 0 00-.968-.731c-.312-.056-.634 0-.908.158l-.672.672a1.421 1.421 0 00.336 1.363zM84.611 355.829h.277l.316-.079.75-.75a1.4 1.4 0 00-.691-1.521l-.257-.257-.276-.04-.316.04-.79.79a1.397 1.397 0 00.73 1.679l.119-.118.138.256zM96.266 372.343l-.277-.276a1.285 1.285 0 00-1.185.434 1.508 1.508 0 00-.32 1.207 1.377 1.377 0 00.577.877c.15.101.317.172.494.208l.375-.099.197-.197c.16-.102.294-.236.395-.396a.53.53 0 00.139-.138c.1-.167.154-.358.158-.553l-.435-.434-.118-.633zM99.052 351.187l.177.178c.092.166.229.303.395.395l.198.198.494.098.276-.039.297-.099.73-.731.06-.296-.119-.119-.079-.355-.849-.85a1.451 1.451 0 00-1.482.731l-.06.257.001.474-.04.158zM95.772 368.729l.514.513c.159.007.318-.007.474-.039l.395.395a1.348 1.348 0 001.145-.356 1.26 1.26 0 00.415-1.047l-.098-.099-.119-.316-1.383-1.382a1.221 1.221 0 00-.952.198 1.222 1.222 0 00-.51.829c-.01.396.092.787.297 1.126l-.178.178zM107.763 392.373a1.288 1.288 0 00-.968-.968 1.616 1.616 0 00-.533.02l-.652.652a1.335 1.335 0 000 1.225 1.394 1.394 0 001.066.79c.308.058.627-.001.894-.165a1.26 1.26 0 00.548-.724v-.276l-.355-.554zM94.35 368.61l-.138.138a1.404 1.404 0 00-.454 1.205 1.395 1.395 0 00.908.909l.139.138.454-.02.928-.928.08-.395v-.277l-1.047-1.047-.218.02-.652.257zM70.073 352.827l.297.02.197-.198a1.417 1.417 0 001.106-.83 1.312 1.312 0 00-.375-1.56l-.178-.178-.454-.099h-.198l-.355.079-1.067 1.067a1.316 1.316 0 001.027 1.699zM86.37 348.382l.099.099.059.296.237.238a1.314 1.314 0 00.968.73h.395l.869-.869.06-.533-.06-.336-.85-.849-.236-.04a1.343 1.343 0 00-1.027.237 1.254 1.254 0 00-.514 1.027zM81.688 387.87l-1.008-1.008a1.362 1.362 0 00-1.086.297 1.448 1.448 0 00-.533 1.126l.098.098.08.316.79.791a1.4 1.4 0 001.58-1.067l.119-.119-.04-.434zM86.646 361.736l-.316.316a1.59 1.59 0 00.139 1.244 1.484 1.484 0 00.968.613l.098.098.316-.118.316-.316a1.341 1.341 0 00.415-1.008l-.691-.691a1.47 1.47 0 00-.672-.316h-.316l-.474.474.217-.296zM75.248 360.965a1.4 1.4 0 00-.434 1.185l.948.949a1.424 1.424 0 001.481-.692l.08-.276-.416-.415a1.507 1.507 0 00-.908-.553 1.253 1.253 0 00-.75-.198zM105.53 383.346l-.276-.276a1.358 1.358 0 00-1.107.316 1.346 1.346 0 00-.474.75 1.416 1.416 0 00-.256.257l-.119.119.02.414.494.494c.11.176.259.324.434.435l.119.118h.395l.494-.493c.188-.11.345-.267.454-.455l.257-.257a1.507 1.507 0 00-.079-1.145l-.217-.218-.139-.059zM108.059 370.783l-.375-.376a1.435 1.435 0 00-1.116-.507 1.423 1.423 0 00-1.097.547 1.458 1.458 0 00-.138 1.521l.632.632a1.419 1.419 0 001.146-.316 1.434 1.434 0 00.514-1.146l.434-.355zM108.356 359.899a1.471 1.471 0 00-.435 1.224l.454.455c.207.243.495.404.81.454.218.049.446.029.652-.059l.77-.771a1.311 1.311 0 00-.079-1.185 1.337 1.337 0 00-.968-.691 1.364 1.364 0 00-1.204.573zM101.521 341.409l.118.316.791.79.296.06.217-.02.356-.079.987-.988v-.237l-.138-.415-.514-.513-.375-.138-.276-.04a1.398 1.398 0 00-1.403 1.403l-.059-.139zM81.016 380.166a1.367 1.367 0 00-1.066-1.62l-.257.02-.533.533a1.397 1.397 0 00-.178 1.956l.395.395.336.099a1.397 1.397 0 001.303-1.383zM112.365 362.526a1.538 1.538 0 00-.612 1.521 1.339 1.339 0 001.066 1.066l.099.099.119-.118a1.284 1.284 0 00.592-.948 1.573 1.573 0 00-.355-1.225l-.632-.632-.356.079.079.158zM99.012 388.008l-.099-.296-.85-.85a1.55 1.55 0 00-1.224.316 1.46 1.46 0 00-.494 1.047l.77.771c.205.15.442.252.692.296l.296.02.672-.672a1.322 1.322 0 00.316-.988l-.079.356zM109.421 364.837h-.395l-.493.494c-.197.114-.36.277-.474.474l-.02.217.019.375.494.494c.108.19.265.347.455.455l.118.118h.395l.474-.474c.194-.118.356-.281.474-.474l.139-.138-.04-.395-.454-.455a1.752 1.752 0 00-.474-.474l-.218-.217zM105.827 365.114l-.533-.534a.752.752 0 00-.178-.178 1.422 1.422 0 00-1.659 1.462l.651.652c.183.112.383.193.593.237.277.044.561.003.815-.116a1.4 1.4 0 00.607-.556l.139-.138.079-.395v-.237l-.178-.178-.336-.019zM109.738 367.8l-.415.02-.276.276a1.481 1.481 0 00-.613 1.521 1.343 1.343 0 001.067 1.067h.198l.335-.059.751-.751c.19-.107.347-.264.454-.454l.119-.119v-.395l-.494-.494a1.847 1.847 0 00-.454-.454l-.672-.158zM108.829 376.926h-.395l-.494.494a1.324 1.324 0 00-.454.454l-.119.119v.395l.514.514c.112.174.26.322.435.434l.217.02h.395l.494-.494a1.25 1.25 0 00.454-.454l.04-.237-.02-.376-.494-.493a1.202 1.202 0 00-.454-.455l-.119.079zM104.958 356.027l.099.336.849.849.395.079.099.099.316-.079a1.568 1.568 0 001.628-.731c.109-.181.18-.383.209-.593l-.039-.316-.889-.889-.474-.158a1.479 1.479 0 00-1.126.257 1.342 1.342 0 00-1.403 1.403l.336-.257zM100.394 357.429l-.632-.632a1.288 1.288 0 00-1.126.376 1.476 1.476 0 00-.375 1.481 1.395 1.395 0 00.988.988 1.281 1.281 0 00.987-.277 1.283 1.283 0 00.514-.987l-.119-.119a1.392 1.392 0 00-.237-.83zM89.214 379.395l-.316-.039-.217.02-.375.059-.178.099-.217.138-.119.158-.197.316-.08.198-.039.158.02.217.079.355.079.198.158.198.138.138.316.197.218.06.296.059h.237l.355-.079.178-.099.218-.138.118-.158.198-.316.079-.198.04-.158-.02-.217-.08-.356-.099-.177-.138-.218-.138-.138-.336-.217-.178-.06zM83.683 342.851l.414.415c.115.16.255.301.415.415l.139.138.415-.02.553-.553a1.363 1.363 0 00.414-1.007l-.098-.099-.119-.316-.909-.909-.217.02-.355.079-.85.85v.197l.198.79zM98.123 339.236l.118-.118.356-.079.83-.83a1.395 1.395 0 00-.988-1.817l-.218.138-.316.04-.79.79a1.397 1.397 0 00-.02 1.323 1.315 1.315 0 001.028.553zM81.885 320.826a1.416 1.416 0 001.146-.316 1.447 1.447 0 00.434-1.106l-.118-.119-.118-.316-.435-.434a1.756 1.756 0 00-.612-.297c-.344-.042-.691.05-.968.257l-.455.454a1.4 1.4 0 00.08 1.186 1.395 1.395 0 001.046.691zM80.147 372.699l.099-.099.296-.02.692-.691c.063-.132.11-.271.138-.415l.02-.217-.04-.395-1.106-1.106-.277.039-.513.514a1.392 1.392 0 00-.158 1.383 1.48 1.48 0 00.85 1.007zM77.54 341.409l.316.04 1.224-1.225-.099-.099-.059-.335-.316-.316a1.207 1.207 0 00-.454-.455l-.257-.02-.494.06-.928.928a1.284 1.284 0 001.067 1.422zM101.975 367.938l-1.462 1.462-.039.237.019.217.099.376.85.849a1.363 1.363 0 001.086-.296l1.324-1.324.197-.197.178-.178.079-.474-.02-.257-.928-.928a1.42 1.42 0 00-1.225.395l-.158.118zM94.547 353.637l.79.79a1.342 1.342 0 001.087-.257 1.503 1.503 0 00.553-1.027l-.118-.119-.08-.316-.81-.81a1.46 1.46 0 00-1.105.277c-.3.261-.485.63-.514 1.027l.099.099.099.336zM70.567 399.741l-.336.336c-.082.372-.019.761.178 1.087.223.304.556.51.928.573l.297.019.098-.098.336-.139.711-.711.08-.434-.04-.277-.099-.296-.73-.731a1.174 1.174 0 00-1.423.671zM161.136 354.526l-.513.513a1.282 1.282 0 00-.277 1.265c.112.209.284.381.494.493l.593.593a1.676 1.676 0 001.343-.316l.474-.474c.077-.3.063-.617-.039-.909l-1.166-1.165h-.909zM162.954 329.913l.889.889a1.372 1.372 0 001.482-.731l.099-.099.059-.257.099-.099-.593-.592a1.454 1.454 0 00-.889-.573 1.265 1.265 0 00-1.086.336 1.396 1.396 0 00-.435 1.185l.375-.059zM129.828 368.136l-.356-.356-.494-.099h-.276l-.949.949c-.145.349-.33.681-.553.987l-.158.158-.039.435.02.257.869.869a1.29 1.29 0 001.146-.356l1.204-1.205.099-.335.138-.139a1.369 1.369 0 00-.651-1.165zM129.511 373.43l-.514.514a1.566 1.566 0 00.139 1.718l.217.217.553.158h.277l.237.04h.434l.968-.968.04-.316v-.237l-.139-.415-.73-.731-.554-.158h-.276l-.277.04-.099.098-.276.04zM131.803 362.21l1.066 1.067a1.373 1.373 0 001.087-.376 1.36 1.36 0 00.293-.464c.064-.173.092-.358.082-.543l-.118-.119-.139-.336-.454-.454-.336-.138a1.337 1.337 0 00-1.481.849l-.02.218.02.296zM133.679 341.646l.415-.098.415-.415a1.073 1.073 0 00.395-.948 1.314 1.314 0 00-.494-.929l-.415-.415a1.227 1.227 0 00-.948.198 1.453 1.453 0 00-.494.849 1.405 1.405 0 00.168 1.132 1.396 1.396 0 00.958.626zM149.995 395.178l.099.376a1.171 1.171 0 00.85.494c.374.037.748-.069 1.047-.297a1.32 1.32 0 00.528-.87 1.316 1.316 0 00-.252-.986l-.257-.257-.691-.099-.375.099-.534.533-.02.257-.395.75zM193.967 366.18l.277.277.691.099.376-.099c.131-.091.244-.205.336-.336l.276-.277.099-.691-.119-.395a1.368 1.368 0 00-.869-.514 1.342 1.342 0 00-.968.297 1.372 1.372 0 00-.099 1.639zM148.909 406.971l.514-.513a1.317 1.317 0 00-.178-1.284l-.711-.711-1.422 1.422-.04.237v.197l.06.336.276.277a1.341 1.341 0 001.798.415l-.297-.376zM162.085 383.287l.139-.534-.02-.335-.296-.297a1.264 1.264 0 00-.949-.513 1.418 1.418 0 00-1.185.474l-.415.414.02.297.099.099c.081.103.173.195.276.276a1.313 1.313 0 001.837.217l.198.198.296-.296zM140.554 399.683l.059-.257v-.435l-.099-.336-.257-.256a1.252 1.252 0 00-1.066-.514 1.423 1.423 0 00-1.146.711l-.118.118-.079.316.098.099.02.218.277.276a1.506 1.506 0 001.481.533l.652-.651.178.178zM171.428 375.208l.158.158c-.063.373.022.755.237 1.066a1.706 1.706 0 001.205.613l.336-.099.77-.77a1.333 1.333 0 00-.237-1.067 1.37 1.37 0 00-.059-.968l-.889-.889-.395.04-.138.138c-.225.13-.415.313-.552.534-.138.22-.22.471-.238.73l-.198.514zM186.52 373.746l.553.553a1.371 1.371 0 001.462.474 1.453 1.453 0 001.066-1.343v-.198l-.454-.454a1.34 1.34 0 00-1.462-.474 1.536 1.536 0 00-1.165 1.442zM117.718 397.588l-.395-.395a1.402 1.402 0 00-1.126-.138l-.533.533c-.172.25-.262.547-.257.85l.139.336.77.77a1.587 1.587 0 001.481-.652 1.279 1.279 0 00-.079-1.304zM176.861 396.166l-.119.356-.118.118v.237l.434.435a1.457 1.457 0 001.6.494l.455-.455.118-.355.138-.138a1.136 1.136 0 00-.118-.593l-.613-.612-.335-.099-.119-.119a1.321 1.321 0 00-1.007.455l-.316.276zM165.068 405.075a1.367 1.367 0 001.007.533 1.416 1.416 0 001.126-.493l.455-.455v-.237l-.099-.099-.099-.296-.77-.77-.277-.04-.375.099-.929.928-.019.257.019.217.099.336-.138.02zM164.495 378.309l.711.711c.279.044.564.017.83-.079l.711-.711.039-.237-.019-.336a1.346 1.346 0 00-.277-.711l-.632-.632-.217-.02-.297.059a1.528 1.528 0 00-.731.929 1.222 1.222 0 00-.118 1.027zM108.908 418.33l.079-.435.02-.217a1.315 1.315 0 00-.296-.81 1.082 1.082 0 00-.356-.356l-.316-.316a1.287 1.287 0 00-1.402.731l-.415.415.019.375.139.139c.05.091.126.166.217.217a1.457 1.457 0 001.6.494l.573-.573.138.336zM172.574 401.756l-.809.81-.04.237v.435l.336.336a1.426 1.426 0 001.817.395l.672-.672a1.396 1.396 0 00-.257-1.047 1.428 1.428 0 00-1.719-.494zM147.784 420.7a1.395 1.395 0 00-.791.356l-.513.513-.02.257.119.119.118.355.138.138a1.6 1.6 0 001.442.534l.672-.672.138-.375.119-.119c.054-.265.026-.54-.079-.79l-.85-.849-.256.02-.237.513zM173.858 399.524l.652.652a1.393 1.393 0 001.284-.454 1.537 1.537 0 00.237-1.462l-.81-.81a1.43 1.43 0 00-1.501.711 1.29 1.29 0 00.138 1.363zM151.517 429.767l.237-.237.098-.296-.019-.573-.198-.198a1.51 1.51 0 00-.988-.553 1.427 1.427 0 00-1.086.376l-.454.454.039.276.356.356a1.312 1.312 0 001.679.336l.355-.356-.019.415zM154.283 410.527l.256.02.099-.099.356-.119.928-.928a1.45 1.45 0 00-.257-1.244 1.42 1.42 0 00-1.936-.237l-.612.612-.039.276.118.119c.044.26.153.504.316.711l.415.415.356.474zM203.547 420.404l.277.276.395.04.138.138a1.12 1.12 0 00.929-.415 1.308 1.308 0 00.256-1.007l.119-.119-.099-.375a1.234 1.234 0 00-.837-.504 1.232 1.232 0 00-.941.267c-.239.184-.412.44-.494.731-.039.343.053.689.257.968zM177.512 403.969l-.257-.257a1.283 1.283 0 00-1.659.158l-.296.296a1.424 1.424 0 00-.04 1.502l.672.671a1.615 1.615 0 001.56-.731 1.335 1.335 0 00-.098-1.521l.118-.118zM169.078 409.362l.118.118-.118.119.118.395a1.425 1.425 0 001.995.257c.16-.12.293-.271.391-.445a1.44 1.44 0 00.182-.563 1.482 1.482 0 00-.276-1.027l-.396-.395a1.252 1.252 0 00-1.126.059 1.31 1.31 0 00-.711.988l-.177.494zM174.846 408.295l-.099.296-.118.119.415.414c.11.211.283.384.494.494l.217.218.276.039a1.395 1.395 0 001.067-.869 1.252 1.252 0 00-.158-1.264 1.284 1.284 0 00-.849-.494l-.376-.059a1.431 1.431 0 00-1.066.553l.197.553zM181.364 407.821l-.513.197-.454.455-.08.316v.553l.316.316a1.376 1.376 0 001.42.586c.178-.036.347-.108.497-.211a1.43 1.43 0 00.652-.652 1.293 1.293 0 00.118-1.106l-.711-.711-.316-.079c-.249.013-.495.06-.731.138l-.198.198zM195.646 384.413a1.315 1.315 0 001.801.288 1.31 1.31 0 00.431-.506l.099-.098.059-.257-.968-.968-.257-.02a1.31 1.31 0 00-.982.839 1.31 1.31 0 00.212 1.275l-.395-.553zM161.315 443.081l-.435-.435-.296-.098-.119-.119a1.47 1.47 0 00-.671.119l-.593.592-.079.316-.118.119c-.002.225.045.447.138.652l1.086 1.086c.225.004.448-.043.652-.138l.573-.573.079-.316.119-.119a1.439 1.439 0 00-.099-.612l-.237-.474zM156.83 427.851l.119-.119.059-.257a1.47 1.47 0 00-.118-.592l-.711-.711-.316-.04-.119.119-.375.138-.138.138a1.34 1.34 0 00-.257 1.877l.652.652h.237l.316-.079.375-.376.276-.75zM173.779 426.903l.633.632.691.059.356-.118a1.316 1.316 0 00.422-1.728 1.31 1.31 0 00-1.687-.564 1.288 1.288 0 00-.415 1.719zM168.94 419.1l.256-.257v-.197l-.059-.336-.316-.316a1.42 1.42 0 00-1.126-.573 1.398 1.398 0 00-1.106.593l-.632.632-.04.276.099.099.099.375.83.83.256.059.119-.118.355-.119.692-.691.573-.257zM144.643 397.49l.118-.119a1.337 1.337 0 00-.118-1.462 1.5 1.5 0 00-1.541-.513 1.535 1.535 0 00-1.067 1.264l.119.119.138.375.573.573a1.422 1.422 0 001.6-.178l.217-.217-.039.158zM152.642 458.43l-.671.671v.198a1.335 1.335 0 00.217 1.244l.672.672.237-.04a1.328 1.328 0 00.928-.494l.375-.375-.019-.257-.119-.118a2 2 0 00-.296-.573 1.338 1.338 0 00-1.324-.928z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M108.849 365.094a1.318 1.318 0 00-1.383-.435l-.435.435a1.4 1.4 0 00-.414.81 1.161 1.161 0 00.276.948l.395.395a1.284 1.284 0 001.58-.356 1.48 1.48 0 00-.019-1.797zM213.345 370.822a1.4 1.4 0 00-.177 1.877c.094.152.223.28.375.375l.395.395a1.31 1.31 0 001.205-.573l.197-.197.08-.277-.02-.494-.099.099.296-.099a1.503 1.503 0 00-.257-.691l-.533-.533-.395-.04-.217.02-.514.514-.336-.376zM139.981 450.311l.138-.139-.059-.296-.553-.553a1.422 1.422 0 00-1.897-.276 1.285 1.285 0 00-.434 1.698l.632.632a1.34 1.34 0 001.699-.553l.474-.513zM158.154 416.216a1.373 1.373 0 001.205.494l.494-.494.138-.138.533-.534v-.276l-.118-.119-.119-.355-.158-.158a1.312 1.312 0 00-1.62-.474l-.651.652a1.452 1.452 0 00.296 1.402zM145.452 398.537l.059.415.277.276a1.59 1.59 0 001.225.672l.276.039c.376.115.779.101 1.146-.039a1.955 1.955 0 001.185-1.66l-.118-.118-.119-.356-.474-.474-.356-.158-.118-.118-.119.118a1.59 1.59 0 00-1.126-.177h-.197l-.277.039a1.389 1.389 0 00-.691.928l-.178.178-.395.435zM163.804 350.2l-.119.315v.238l.612.612a1.319 1.319 0 00.968.454 1.29 1.29 0 00.524-.137c.164-.081.308-.196.425-.337l.118-.118a1.2 1.2 0 00.099-1.363l-.79-.79-.336.019a1.37 1.37 0 00-.909.514l-.592.593zM162.737 370.783l-.119-.119-.296.06-.118.118a1.478 1.478 0 00-.731 1.008 1.33 1.33 0 00.257 1.086l.493.494.257.02a1.423 1.423 0 001.027-.435l.297-.296-.119-.119a1.737 1.737 0 00-.296-.691l-.533-.534-.119-.592zM165.423 392.433l.178.178.435.197.256.059.277-.039 1.007-1.007.099-.336c.063-.164.09-.339.079-.514l-.829-.83-.317-.079h-.316a1.483 1.483 0 00-1.027.316l-.158.159-.178.375-.138.138v.277l.099.098a1.086 1.086 0 00.533 1.008zM173.601 386.191l-.118-.119a1.374 1.374 0 00-.988-.553 1.433 1.433 0 00-1.166.454l.02-.019-.336.335-.039.237-.316.317-.079.276v.237l.079.435.889.889.276.039.139-.138.375-.138.118-.119a1.402 1.402 0 00.534-1.047c.219-.215.37-.49.434-.79l-.098-.099.276-.197zM160.86 418.527l-.415.415a1.483 1.483 0 00-.257 2.035l-.118-.119.415.415.256.02h.238l.414-.139.652-.651.119-.356.039-.237a1.503 1.503 0 00-.296-.849 1.944 1.944 0 00-.415-.415l-.099-.099-.316-.079-.217.059zM153.493 398.714l.158.158a1.368 1.368 0 001.619.435l.652-.652.158-.356.02-.217a1.585 1.585 0 00-.356-.987l-.73-.731-.297.059-.098.099a1.642 1.642 0 00-.929.928l.099.099-.296 1.165zM160.801 390.892l.751.751c.145.094.305.16.474.197.158.115.331.208.514.277l.217.217.237.039.099.099.336-.099.118-.118a1.722 1.722 0 00.849-1.205l-.079-.316a1.334 1.334 0 00-.138-1.087l-.652-.651a1.695 1.695 0 00-.987-.04l-.178.178-.277.079-.217.217a1.452 1.452 0 00-1.067 1.462zM157.166 395.001l.435.434.316-.039.118-.119a1.412 1.412 0 00.7-.844 1.436 1.436 0 00-.107-1.092l-.573-.572a1.446 1.446 0 00-1.442.651 1.422 1.422 0 00.118 1.541l.435.04zM154.46 403.633l.118-.118.04-.237-.04-.356-.296-.296a1.394 1.394 0 00-1.817-.395l-.771.77v.316c.028.255.131.496.297.691l.652.652h.237l.316-.118.414-.415.85-.494zM182.648 358.536a1.31 1.31 0 001.008.533 1.544 1.544 0 001.244-.533l.317-.316.059-.257a1.625 1.625 0 00-.316-1.067 1.52 1.52 0 00-1.047-.573l-.257.02-.296.296a1.544 1.544 0 00-.731 1.047 1.29 1.29 0 00.019.85zM131.388 409.598l.099.099.079.356.276.276a1.463 1.463 0 001.087.613 1.42 1.42 0 001.066-.435l.317-.316a1.29 1.29 0 00-.04-1.501 1.523 1.523 0 00-.356-.356l-.118-.118-.356-.079-.237.039-.118.119a1.654 1.654 0 00-.613.296l-.493.494-.593.513zM129.985 416.058l.139-.138.039-.237.099-.099-.059-.336-.257-.257a1.733 1.733 0 00-1.738-.711l-1.028 1.027c-.037.374.069.749.297 1.047.151.263.369.481.632.632l.118.119.257.059h.237l.376-.098.454-.455.434-.553zM152.189 392.749l.177.178a1.425 1.425 0 001.067.553 1.487 1.487 0 001.146-.474l.493-.494-.098-.099-.059-.296-.297-.297a1.518 1.518 0 00-1.047-.572 1.556 1.556 0 00-1.165.493l-.435.435-.019.217.059.297.178.059zM122.815 404.581l.118.119a1.177 1.177 0 001.324-.218l.454-.454.02-.296-.099-.415a1.374 1.374 0 00-1.146-.514 1.642 1.642 0 00-.79.316 1.285 1.285 0 00-.276 1.857l.395-.395zM96.049 404.364l.138-.138-.02-.218a1.332 1.332 0 00-.316-.316 1.365 1.365 0 00-1.462-.513 1.614 1.614 0 00-1.244 1.244l.02.217.493.494a1.418 1.418 0 001.225.553 1.529 1.529 0 001.245-.889l-.08-.434zM164.535 386.566l.415-.415.019-.217-.019-.296-.336-.336a1.365 1.365 0 00-1.146-.553 1.649 1.649 0 00-1.284.81l-.158.158-.039.513 1.066 1.067a1.57 1.57 0 001.482-.731zM115.941 359.662l.277.276.296.02c.169.069.353.09.533.059l.889-.889.04-.316.059-.296-.119-.395a1.453 1.453 0 00-.849-.494 1.37 1.37 0 00-.988.277c-.231.226-.373.53-.399.853-.025.323.068.644.261.905zM116.889 389.727a1.534 1.534 0 001.896.434l.83-.829.019-.257-.078-.435-.949-.948-.296-.02-.217.02-.336.138-.731.731-.059.257.02.494.079.276-.178.139zM143.813 403.969l-.83-.83-.276-.039-.217.019-.336.099-.83.83-.059.257v.197l.099.375.118.119a1.337 1.337 0 00.99.56 1.35 1.35 0 001.065-.402l.138-.138.138-.336.119-.119.019-.256-.138-.336zM114.498 411.396a1.345 1.345 0 001.541.237l.553-.553a1.259 1.259 0 00-.118-1.383 1.034 1.034 0 00-.356-.355l-.118-.119-.316-.079-.415.415a1.309 1.309 0 00-.494 1.64l.158.158-.435.039zM192.447 425.54l-.119.118-.02.257-.099.099.119.355 1.007 1.008.099-.099.375-.059.139-.138a1.45 1.45 0 00.632-1.502l-.751-.75-.316-.079-.296-.02-.257.059c-.28.041-.541.165-.751.356l-.138.138.376.257zM170.224 359.741l.79.79.355.079.257-.02.376-.099.118.119a1.486 1.486 0 00.711-.949 1.285 1.285 0 00-.237-1.027l-.533-.533a1.433 1.433 0 00-1.225.395 1.452 1.452 0 00-.612 1.245zM176.742 375.346l-.138.138-.02.257.098.099a1.273 1.273 0 00-.217.217 1.486 1.486 0 00-.059 1.758l.612.612.316-.039.119-.118a1.734 1.734 0 00.988-.988 1.34 1.34 0 00.039-.79l.119-.119a1.289 1.289 0 00-.079-1.422l-.218-.217-.296-.099-.217-.217-.099.098-.296.099-.455.454-.197.277zM173.661 432.71l-.731-.731-.237-.039-.099-.099-.336.059-.118.119a1.332 1.332 0 00-.672.948 1.422 1.422 0 00.277 1.146l.651.652h.238l.335-.099.83-.83.059-.257-.118-.118-.099-.376.02-.375zM118.726 442.093l-.494-.494-.257-.059-.316-.039a1.255 1.255 0 00-.711.118l-.79.79v.356l.118.118c.045.151.119.292.218.415a1.396 1.396 0 001.916.336 1.368 1.368 0 00.316-1.541zM123.289 418.646a1.222 1.222 0 00.06-1.166l-.593-.592-.356-.119-.256-.02c-.36.02-.699.175-.949.435l-.138.138-.118.356-.119.118-.02.257.119.119.158.355.099.099a1.285 1.285 0 001.698.356l.158-.159.257-.177zM107.684 399.485l.118-.119.06-.257a1.348 1.348 0 00-.119-.632l-.691-.691-.316-.04-.119.119-.355.118-.139.139a1.337 1.337 0 00-.508 1.419c.045.173.124.335.232.477l.336.336.217-.02.099-.099.296-.098.356-.356.533-.296zM165.404 410.073l-.277-.277a1.484 1.484 0 00-1.086.85 1.343 1.343 0 00.158 1.303 1.169 1.169 0 00.987.474l.317-.079.118-.118c.296-.214.5-.531.573-.889a1.384 1.384 0 00-.277-1.067l-.513-.197zM192.229 390.319l-.139-.138-.414.099a1.542 1.542 0 00-.672.276l-.237.237-.178.534-.02.335.099.336.632.632a1.595 1.595 0 001.719-.217l.514-.513.019-.257-.118-.356-.731-.731-.474-.237zM139.803 434.607l.118-.356.04-.237a1.336 1.336 0 00-.119-.632l-.592-.593-.336-.098-.099-.297-.81-.81-.434-.079h-.237a1.429 1.429 0 00-.889.889 1.31 1.31 0 00.237 1.185c.222.293.547.491.909.553l.098.297.119.118a1.4 1.4 0 001.56.494l.632-.632-.197.198zM139.922 429.589a1.585 1.585 0 00.711-1.027 1.418 1.418 0 00-.237-1.146l-.711-.711a1.661 1.661 0 00-1.185.553l-.475.474v.237l.099.099.099.336.909.909.237.039h.197l.356-.079v.316zM136.721 392.65a1.451 1.451 0 002.015.198 1.314 1.314 0 00.257-1.838l-.277-.276-.711-.079-.375.099-.771.77-.098.692-.04.434zM166.826 368.037l-.237.237-.099.336v.593l.949.948h.276l.217-.02.376-.138.158-.158a1.287 1.287 0 00.237-1.778 1.31 1.31 0 00-1.284-.454l-.494.177-.099.257zM140.198 410.507l-.573-.573a1.535 1.535 0 00-1.462.712 1.454 1.454 0 00.119 1.58l.612.612h.237a1.53 1.53 0 001.047-.849 1.566 1.566 0 00.02-1.482zM131.309 423.387l.276-.277v-.197l-.039-.356-.316-.316c-.553-.553-1.265-.751-2.134-.237l-.612.612-.079.277.02.217.079.356.829.829.415.06.217-.02c.336-.078.657-.212.949-.395l.138-.139.257-.414zM174.174 386.566l-.415.415.02.296.336.336a1.449 1.449 0 001.62.474l.849-.85.02-.454a1.523 1.523 0 00-.277-.671l-.513-.514a1.646 1.646 0 00-1.166.533l-.474.435zM143.379 411.001l-.534.534-.079.276c-.044.266-.017.539.079.79l.632.632.277.079h.237l.435-.079.888-.889.02-.256-.039-.237-.139-.376-.118-.118a1.286 1.286 0 00-1.659-.356zM169.848 383.228c-.044.356.047.716.257 1.007l.257.257.375.059.138.138c.244.014.485-.048.692-.177l.098.098.277.079.099.099.375-.375c.338-.236.577-.587.672-.988l.098-.099-.039-.316-.533-.533a1.312 1.312 0 00-1.462-.435l-.178.178.296-.296a1.31 1.31 0 00-1.086.257 1.387 1.387 0 00-.336 1.047zM150.825 437.57a1.028 1.028 0 00-.178.177.959.959 0 00-.276.277 1.231 1.231 0 00.059 1.244l.731.731.237.04h.237l.356-.079.138-.138c.262-.212.461-.492.573-.81a1.287 1.287 0 00.098-1.64l-.612-.612a1.566 1.566 0 00-1.363.81zM124.592 431.881l.099-.336.139-.138a1.443 1.443 0 00-.099-.613l-.613-.612-.355-.119-.119-.118a1.345 1.345 0 00-1.007.454l-.474.474v.237l.099.099.098.296.119.119a1.393 1.393 0 001.6.494l.632-.632-.119.395zM203.548 440.592l.731.731.098-.099.297-.059a1.627 1.627 0 00.77-1.008 1.434 1.434 0 00-.296-1.165l-.316-.316-.218-.02-.316.04-.118.118a1.411 1.411 0 00-.488.396 1.41 1.41 0 00-.252 1.198c.051.206.149.397.285.559l-.177-.375zM170.658 363.375l-.158.158a1.266 1.266 0 00-.02 1.245l.573.573.355.079.119.118a1.45 1.45 0 00.968-.415l.138-.138.138-.336.119-.118v-.277l-.474-.474a1.314 1.314 0 00-1.679-.375l-.376.375.297-.415zM185.572 402.863l.751.75.316.04h.237l.395-.119.138-.138a1.313 1.313 0 00.277-1.857l-.296-.296-.317-.119h-.237l-.098.099-.297.099-.118.118a1.453 1.453 0 00-.751 1.423zM127.477 388.423c.137.082.282.148.434.197a1.28 1.28 0 001.383-.513 1.288 1.288 0 00-.519-1.856 1.292 1.292 0 00-.706-.12l-.573.573-.197.474-.04.277.81.81-.592.158zM143.951 379.652l.533.534h.237a1.454 1.454 0 00.929-.929 1.535 1.535 0 00-.139-1.363l-.651-.652a1.425 1.425 0 00-1.462.711 1.364 1.364 0 00.553 1.699zM184.802 394.151l.099.099.296-.02c.366-.053.698-.243.928-.533l.178-.178-.099-.098-.059-.297-.217-.217a1.413 1.413 0 00-2.543.606c-.059.371.031.749.251 1.053l.494.494.672-.909zM153.887 348.975l-.237.237v.197l.04.316.296.297a1.337 1.337 0 001.419.508c.173-.045.335-.123.477-.232a1.317 1.317 0 00.552-.848 1.318 1.318 0 00-.216-.989l-.612-.612-.257-.02-.237.04a1.508 1.508 0 00-.771.335l-.118.119-.336.652zM175.201 416.947l.158-.158.04-.277-.02-.217-.099-.415-.158-.158a1.48 1.48 0 00-1.442-.533l-1.007 1.007.02.455.316.316a1.367 1.367 0 001.817.395l.138-.139.237-.276zM185.196 434.093l.692.099.375-.099.77-.77.04-.317-.099-.375a1.347 1.347 0 00-1.896-.237 1.426 1.426 0 00-.257 1.956l.375-.257zM193.493 409.361l.553.554.237.039.119-.118.612-.218.869-.869c.055-.239.027-.49-.079-.711l-.632-.632-.434-.198-.139-.138-.335.02-.514.513a1.48 1.48 0 00-.257 1.758zM150.391 415.88l.059-.257-.059-.375-.277-.277a1.315 1.315 0 00-1.027-.553 1.426 1.426 0 00-1.087.376l-.553.553.139.138c.024.246.12.48.276.672l.672.671.316.119.118.118h.237l1.186-1.185zM196.614 399.682l-.019.257.928.929.277.039.256-.02.435-.158.178-.177a1.71 1.71 0 00.395-.83l-.217-.218.316-.039-.04-.514-.632-.632-.513-.158-.297-.02a2.069 2.069 0 00-1.304 1.067l.237.474zM175.893 450.903l-.079.277-.02.335c.025.27.121.529.276.751a1.23 1.23 0 001.679.257l.297-.296a1.229 1.229 0 00.217-1.521l-.731-.731a1.502 1.502 0 00-.77.099l-.218.217-.651.612zM179.172 418.191v.237l.099.099.118.356.139.138a1.48 1.48 0 002.034.257l.573-.573.04-.277-.099-.098-.119-.356-.612-.612-.316-.119-.119-.118a1.782 1.782 0 00-1.204.533l-.534.533zM167.063 431.407l.138-.139.02-.256-.02-.218-.079-.355-.849-.85-.257-.02h-.197l-.356.119-.75.751-.06.296c-.005.312.092.618.277.869l.75.04.731.098.316-.316.336-.019zM126.113 375.8l-.809-.81-.376.06-.138.138a1.423 1.423 0 00-.573 1.955l.81.81.356-.039.138-.138a1.534 1.534 0 00.731-.929 1.397 1.397 0 00-.139-1.047zM108.099 346.407a1.418 1.418 0 00.296 1.165l.573.573.257.02a1.62 1.62 0 001.066-1.067 1.286 1.286 0 00-.987-1.699l-.218-.217-.355.04-.139.138a1.563 1.563 0 00-.493 1.047zM137.769 362.565l.414.415.277.04h.514l.177-.178a1.45 1.45 0 00.573-1.6l-.158-.356-.612-.612-.553-.04-.277.079a1.428 1.428 0 00-.652 1.719l.297.533zM134.944 382.576l-.119-.395a1.308 1.308 0 00-1.165-.494 1.425 1.425 0 00-.672.276 1.337 1.337 0 00-.558.882 1.325 1.325 0 00.242 1.015l.277.276.296.02c.169.069.353.09.533.059l.217-.217c.177-.056.339-.151.475-.277l.197-.197.277-.948zM151.458 369.064l-.06.257c-.073.188-.1.392-.079.593l.929.928.276.04.494-.06 1.067-1.066-.119-.119-.099-.375-.138-.139a1.421 1.421 0 00-1.58-.513l-.514.513-.177-.059zM137.038 379.613l.335.335c.092.131.206.245.336.336l.099.376.138.138a1.425 1.425 0 001.067.553 1.286 1.286 0 001.027-.514l.119-.118.098-.297.119-.118v-.198l-.099-.098-.099-.336-.454-.455v-.197l-.375-.375a1.246 1.246 0 00-.831-.513 1.26 1.26 0 00-.947.236 1.206 1.206 0 00-.415.652.98.98 0 00-.118.593zM129.57 405.055a1.28 1.28 0 00-.118-1.382 1.523 1.523 0 00-.356-.356l-.138-.138-.217.019a1.379 1.379 0 00-.87.87 1.538 1.538 0 00.238 1.303 1.28 1.28 0 001.679.336l.197-.197-.415-.455z'
                    fill={theme.centerChannelColor}
                    fillOpacity={0.08}
                />
                <Path
                    d='M136.07 378.329a1.31 1.31 0 00-1.369 1.219c-.016.226.027.453.124.657l.81.81.257.02.356-.079.829-.83.04-.276-.04-.514-1.007-1.007zM129.215 350.298l.533.534a1.4 1.4 0 001.363.138 1.284 1.284 0 00.77-1.402l-1.264-1.265-.316-.039-.099-.099-.395.118-.138.139a1.28 1.28 0 00-.665 1.377c.034.18.106.35.211.499zM144.148 358.239l-.454.455a1.376 1.376 0 00.217 1.205l.692.691h.237l.948-.948.099-.336v-.237l-.079-.395-.336-.336a1.305 1.305 0 00-.909-.513c-.171.1-.314.243-.415.414zM113.689 383.88l.138-.139.059-.256.02-.218-.04-.395-.256-.257a1.358 1.358 0 00-.433-.386 1.42 1.42 0 00-1.72.288l-.573.572v.198l.098.336.119.118a1.425 1.425 0 001.916.297l.415-.415.257.257zM135.2 368.254l-.296.06a1.596 1.596 0 00-.751.987 1.313 1.313 0 00.297 1.126l.573.573.276.04a1.351 1.351 0 001.008-.455l.414-.414-.019-.257a1.544 1.544 0 00-.277-.672l-.751-.751-.474-.237zM127.299 359.879a1.41 1.41 0 00-.63.155 1.41 1.41 0 00-.496.418l-.119.118-.118.316-.099.099v.237l.099.099.098.296.119.119a1.518 1.518 0 001.561.494l.849-.85a1.538 1.538 0 00-.277-1.067 1.35 1.35 0 00-.987-.434zM134.252 396.937l-.079-.395a1.563 1.563 0 00-1.027-.553 1.5 1.5 0 00-.869.276.985.985 0 00-.376.099l-.573.573-.079.316-.039.237c-.002.203.039.405.118.592l.573.573.336.099.119.119c.217.012.434-.029.632-.119l.257-.257a1.456 1.456 0 00.809-.81l.02-.217.178-.533zM144.603 389.509l.197-.197a1.477 1.477 0 00-.296-.968 1.422 1.422 0 00-1.343-.514 1.483 1.483 0 00-1.067 1.067l-.099.099.297.296a1.391 1.391 0 001.047.573 1.38 1.38 0 001.145-.474l.119.118zM152.268 381.806a1.475 1.475 0 00-1.166.572l-.296.297-.039.237c-.022.326.076.649.276.908.113.186.269.341.455.455l.375.375.336-.059.098-.099a1.284 1.284 0 00.85-.85 1.286 1.286 0 00-.237-1.145 1.368 1.368 0 00-.652-.691zM159.971 363.632l-.138.336-.118.118v.237l.098.099.119.316.869.869.257.02h.197l.356-.118.474-.474.178-.376.158-.158v-.276l-.119-.119-.158-.395-.197-.197a1.403 1.403 0 00-1.536-.326 1.394 1.394 0 00-.459.306l.019.138zM118.489 377.815a1.368 1.368 0 00.316-1.896l-.573-.573-.296-.02a1.33 1.33 0 00-.97.393 1.33 1.33 0 00-.393.97l.454.455a1.372 1.372 0 001.462.671zM153.452 386.467a1.513 1.513 0 00-1.501.672 1.31 1.31 0 00.039 1.501l.514.514a1.398 1.398 0 001.106-.277c.331-.286.536-.69.573-1.126l-.118-.118-.099-.336-.672-.672.158-.158zM126.627 395.435l-.553-.553a1.454 1.454 0 00-1.086-.612 1.31 1.31 0 00-1.087.454l-.316.316.099.099.02.217c.037.243.132.473.276.672.063.082.136.155.217.217l.494.494.218-.02c.359-.045.689-.221.928-.494l.79-.79zM149.66 386.112l-.415-.415-.316-.119-.118-.118a1.335 1.335 0 00-.632.118l-.573.573-.099.336c-.023.218.018.437.119.632l.573.573.316.079.118.118c.231.023.463-.018.672-.118l.573-.573.079-.316.098-.099c.036-.222.001-.45-.098-.652l-.297-.019zM146.855 383.86l-.099-.099-.02-.375-.316-.316a1.413 1.413 0 00-1.087-.567 1.426 1.426 0 00-1.125.488l-.395.395.02.256c.058.21.159.405.296.573a1.336 1.336 0 001.462.474l.79-.79.474-.039zM1