/*
 * Copyright (c) 2024 Christians Martínez Alvarado
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.mardous.booming.coil.util

import android.content.Context
import android.graphics.Bitmap
import coil3.Image
import coil3.SingletonImageLoader
import coil3.asImage
import coil3.request.ImageRequest
import coil3.request.transformations
import coil3.size.Precision
import coil3.toBitmap
import coil3.transform.RoundedCornersTransformation
import com.mardous.booming.data.model.Song
import com.mardous.booming.extensions.media.asAlbumCoverUri

object AutoGeneratedImageUtil {

    suspend fun getImage(context: Context, songs: List<Song>): Image? {
        val bitmap = if (songs.size > 1) {
            MergedImageUtils.joinImages(
                songs.distinctBy { it.albumId }.take(9).mapNotNull {
                    getBitmapWithAlbumId(context, it.albumId, true)
                }
            )
        } else if (songs.size == 1) {
            getBitmapWithAlbumId(context, songs[0].albumId)
        } else {
            null
        }
        return bitmap?.asImage()
    }

    private suspend fun getBitmapWithAlbumId(
        context: Context,
        id: Long,
        roundedCorner: Boolean = false
    ): Bitmap? {
        return try {
            val request = ImageRequest.Builder(context)
                .data(id.asAlbumCoverUri())
                .memoryCacheKey("playlist-preview::id=$id")
                .precision(Precision.EXACT)
                .size(200)
            if (roundedCorner) {
                request.transformations(RoundedCornersTransformation(20f))
            }
            val result = SingletonImageLoader.get(context).execute(request.build())
            result.image?.toBitmap()
        } catch (_: Exception) {
            null
        }
    }
}