package com.mardous.booming.coil.store

import android.content.SharedPreferences
import coil3.map.Mapper
import coil3.request.Options
import com.mardous.booming.coil.model.ArtistImage
import com.mardous.booming.coil.model.AudioCover
import com.mardous.booming.coil.model.AutoGeneratedImage
import com.mardous.booming.coil.model.PlaylistImage
import com.mardous.booming.core.model.filesystem.FileSystemItem
import com.mardous.booming.data.local.room.PlaylistWithSongs
import com.mardous.booming.data.mapper.toSongs
import com.mardous.booming.data.model.*
import com.mardous.booming.extensions.media.albumArtistName
import com.mardous.booming.util.IGNORE_MEDIA_STORE
import com.mardous.booming.util.USE_FOLDER_ART

class SongMapper(private val preferences: SharedPreferences) : Mapper<Song, AudioCover> {
    override fun map(data: Song, options: Options): AudioCover? {
        return AudioCover(
            albumId = data.albumId,
            uri = data.uri,
            path = data.data,
            artistName = data.albumArtistName(),
            albumName = data.albumName,
            title = data.title,
            lastModified = data.rawDateModified,
            isAlbum = false,
            isIgnoreMediaStore = preferences.getBoolean(IGNORE_MEDIA_STORE, true),
            isUseFolderArt = preferences.getBoolean(USE_FOLDER_ART, false)
        )
    }
}

class AlbumMapper(private val preferences: SharedPreferences) : Mapper<Album, AudioCover> {
    override fun map(
        data: Album,
        options: Options
    ): AudioCover? {
        return data.safeGetFirstSong().let {
            AudioCover(
                albumId = data.id,
                uri = it.uri,
                path = it.data,
                artistName = it.albumArtistName(),
                albumName = it.albumName,
                title = it.title,
                lastModified = it.rawDateModified,
                isAlbum = true,
                isIgnoreMediaStore = preferences.getBoolean(IGNORE_MEDIA_STORE, true),
                isUseFolderArt = preferences.getBoolean(USE_FOLDER_ART, false)
            )
        }
    }
}

class ArtistMapper : Mapper<Artist, ArtistImage> {
    override fun map(data: Artist, options: Options): ArtistImage? {
        return ArtistImage(
            id = data.id,
            coverUri = data.safeGetFirstAlbum().albumCover,
            name = data.name
        )
    }
}

class PlaylistMapper : Mapper<PlaylistWithSongs, PlaylistImage> {
    override fun map(data: PlaylistWithSongs, options: Options): PlaylistImage? {
        return PlaylistImage(data.playlistEntity, data.songs.toSongs())
    }
}

class GenreMapper : Mapper<Genre, AutoGeneratedImage> {
    override fun map(data: Genre, options: Options): AutoGeneratedImage? {
        return AutoGeneratedImage(AutoGeneratedImage.TYPE_GENRE, data.id, data.songCount)
    }
}

class YearMapper : Mapper<ReleaseYear, AutoGeneratedImage> {
    override fun map(data: ReleaseYear, options: Options): AutoGeneratedImage? {
        return AutoGeneratedImage(AutoGeneratedImage.TYPE_YEAR, data.year.toLong(), data.songCount)
    }
}

class FileMapper(private val preferences: SharedPreferences) : Mapper<FileSystemItem, AudioCover> {
    override fun map(
        data: FileSystemItem,
        options: Options
    ): AudioCover? {
        val song = data as? Song
        if (song != null) {
            return AudioCover(
                albumId = data.albumId,
                uri = data.uri,
                path = data.data,
                artistName = data.albumArtistName(),
                albumName = data.albumName,
                title = data.title,
                lastModified = data.rawDateModified,
                isAlbum = false,
                isIgnoreMediaStore = preferences.getBoolean(IGNORE_MEDIA_STORE, true),
                isUseFolderArt = preferences.getBoolean(USE_FOLDER_ART, false)
            )
        }
        return null
    }
}