package com.mardous.booming.coil

import android.content.*
import android.database.Cursor
import android.graphics.Bitmap
import android.net.Uri
import android.os.ParcelFileDescriptor
import android.util.Log
import coil3.SingletonImageLoader
import coil3.request.CachePolicy
import coil3.request.ImageRequest
import coil3.size.Precision
import coil3.toBitmap
import com.mardous.booming.BuildConfig
import com.mardous.booming.coil.model.AutoGeneratedImage
import com.mardous.booming.data.local.repository.Repository
import kotlinx.coroutines.runBlocking
import kotlinx.io.IOException
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject
import java.io.File

class CoverProvider : ContentProvider(), KoinComponent {

    private val repository by inject<Repository>()

    override fun onCreate(): Boolean = true

    override fun openFile(uri: Uri, mode: String): ParcelFileDescriptor? {
        val matchCode = uriMatcher.match(uri)
        if (mode != "r" || matchCode == -1) {
            return null
        }

        val id = uri.lastPathSegment ?: return null
        val idAsLong = id.toLongOrNull()
        if (idAsLong == null && matchCode != ALBUM_ARTIST_COVER_CODE) return null

        val cacheDir = context?.cacheDir?.resolve(CACHE_DIR_NAME) ?: return null
        if (!cacheDir.isDirectory && !cacheDir.mkdirs()) return null

        val cacheFile = File(cacheDir, "${matchCode}_${id}.jpg")
        val cacheFileExpired = (System.currentTimeMillis() - cacheFile.lastModified()) > CACHE_FILE_EXPIRES
        if (cacheFile.isFile && cacheFile.length() > 0 && !cacheFileExpired) {
            try {
                return ParcelFileDescriptor.open(cacheFile, ParcelFileDescriptor.MODE_READ_ONLY)
            } catch (e: IOException) {
                Log.e(TAG, "ParcelFileDescriptor.open($cacheFile): failed", e)
            }
        }

        val result = runBlocking {
            val data = when (matchCode) {
                SONG_COVER_CODE -> repository.songById(idAsLong!!)
                ALBUM_COVER_CODE -> repository.albumById(idAsLong!!)
                ARTIST_COVER_CODE -> repository.artistById(idAsLong!!)
                ALBUM_ARTIST_COVER_CODE -> repository.albumArtistByName(id)
                PLAYLIST_COVER_CODE -> repository.playlistWithSongs(idAsLong!!)
                GENRE_COVER_CODE -> {
                    AutoGeneratedImage(
                        type = AutoGeneratedImage.TYPE_GENRE,
                        id = idAsLong!!,
                        songCount = repository.songsByGenre(idAsLong).size
                    )
                }
                else -> null
            }
            if (data != null) {
                context?.let {
                    SingletonImageLoader.get(it).execute(
                        ImageRequest.Builder(it)
                            .data(data)
                            .memoryCachePolicy(CachePolicy.DISABLED)
                            .precision(Precision.EXACT)
                            .size(MAX_BITMAP_DIMENSION)
                            .build()
                    )
                }
            } else null
        }

        if (result != null) {
            val bitmap = result.image?.toBitmap()
            if (bitmap != null && (cacheFile.exists() || cacheFile.createNewFile())) {
                val compressed = try {
                    cacheFile.outputStream().use {
                        bitmap.compress(Bitmap.CompressFormat.JPEG, 100, it)
                    }
                } catch (e: IOException) {
                    Log.d(TAG, "Couldn't compress bitmap data to cache file $cacheFile", e)
                    cacheFile.delete()
                    false
                }
                if (compressed) {
                    return ParcelFileDescriptor.open(cacheFile, ParcelFileDescriptor.MODE_READ_ONLY)
                }
            }
        }

        return null
    }

    override fun getType(uri: Uri): String {
        check(uriMatcher.match(uri) == 1) { "Unknown URI: $uri" }
        return "image/*"
    }

    override fun query(
        uri: Uri,
        projection: Array<out String>?,
        selection: String?,
        selectionArgs: Array<out String>?,
        sortOrder: String?
    ): Cursor = throw UnsupportedOperationException()

    override fun insert(uri: Uri, values: ContentValues?): Uri? = null

    override fun delete(uri: Uri, selection: String?, selectionArgs: Array<out String>?): Int = 0

    override fun update(
        uri: Uri,
        values: ContentValues?,
        selection: String?,
        selectionArgs: Array<out String>?
    ): Int = 0

    companion object {
        private const val TAG = "CoilImageProvider"

        private const val AUTHORITY = "${BuildConfig.APPLICATION_ID}.cover"
        private const val CACHE_DIR_NAME = "covers"
        private const val CACHE_FILE_EXPIRES = (60 * 60 * 1000) * 24 * 7 // 7 days
        private const val MAX_BITMAP_DIMENSION = 1000

        const val SONG_COVER_PATH = "song"
        private const val SONG_COVER_CODE = 1
        const val ALBUM_COVER_PATH = "album"
        private const val ALBUM_COVER_CODE = 2
        const val ARTIST_COVER_PATH = "artist"
        private const val ARTIST_COVER_CODE = 3
        const val ALBUM_ARTIST_COVER_PATH = "album_artist"
        private const val ALBUM_ARTIST_COVER_CODE = 4
        const val PLAYLIST_COVER_PATH = "playlist"
        private const val PLAYLIST_COVER_CODE = 5
        const val GENRE_COVER_PATH = "genre"
        private const val GENRE_COVER_CODE = 6

        private val uriMatcher = UriMatcher(UriMatcher.NO_MATCH)
            .apply {
                addURI(AUTHORITY, "$SONG_COVER_PATH/*", SONG_COVER_CODE)
                addURI(AUTHORITY, "$ALBUM_COVER_PATH/*", ALBUM_COVER_CODE)
                addURI(AUTHORITY, "$ARTIST_COVER_PATH/*", ARTIST_COVER_CODE)
                addURI(AUTHORITY, "$ALBUM_ARTIST_COVER_PATH/*", ALBUM_ARTIST_COVER_CODE)
                addURI(AUTHORITY, "$PLAYLIST_COVER_PATH/*", PLAYLIST_COVER_CODE)
                addURI(AUTHORITY, "$GENRE_COVER_PATH/*", GENRE_COVER_CODE)
            }

        fun getImageUri(path: String, id: Long) = getImageUri(path, id.toString())
        fun getImageUri(path: String, id: String): Uri? {
            val uri = Uri.Builder()
                .scheme(ContentResolver.SCHEME_CONTENT)
                .authority(AUTHORITY)
                .appendPath(path)
                .appendPath(id)
                .build()
            if (uriMatcher.match(uri) > 0) {
                return uri
            }
            return null
        }

        fun clearCache(context: Context) {
            val cacheDir = context.cacheDir?.resolve(CACHE_DIR_NAME)
            if (cacheDir == null || !cacheDir.isDirectory) return

            val cacheFiles = cacheDir.listFiles()
            if (!cacheFiles.isNullOrEmpty()) for (file in cacheFiles) {
                if (!file.delete()) {
                    Log.d(TAG, "Failed to delete cache file $file")
                }
            }
        }
    }
}