/*
 * Copyright (c) 2024 Christians Martínez Alvarado
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.mardous.booming

import android.app.Application
import android.content.Context
import android.os.StrictMode
import android.os.StrictMode.ThreadPolicy
import android.os.StrictMode.VmPolicy
import android.util.Log
import androidx.appcompat.app.AppCompatDelegate
import androidx.core.content.edit
import androidx.preference.PreferenceManager
import cat.ereza.customactivityoncrash.config.CaocConfig
import coil3.ImageLoader
import coil3.PlatformContext
import coil3.SingletonImageLoader
import coil3.network.ktor3.KtorNetworkFetcherFactory
import coil3.request.allowHardware
import coil3.request.crossfade
import coil3.util.Logger
import com.mardous.booming.coil.fetcher.ArtistImageFetcher
import com.mardous.booming.coil.fetcher.AudioCoverFetcher
import com.mardous.booming.coil.fetcher.AutoGeneratedImageFetcher
import com.mardous.booming.coil.fetcher.PlaylistImageFetcher
import com.mardous.booming.coil.store.*
import com.mardous.booming.data.local.ReplayGainTagExtractor
import com.mardous.booming.ui.screen.MainActivity
import com.mardous.booming.ui.screen.error.ErrorActivity
import com.mardous.booming.ui.screen.settings.SettingsScreen
import com.mardous.booming.util.EXPERIMENTAL_UPDATES
import com.mardous.booming.util.Preferences.getDayNightMode
import org.koin.android.ext.android.get
import org.koin.android.ext.koin.androidContext
import org.koin.core.context.startKoin

fun appInstance(): App = App.instance
fun appContext(): Context = appInstance().applicationContext

class App : Application(), SingletonImageLoader.Factory {

    override fun onCreate() {
        super.onCreate()
        instance = this

        startKoin {
            androidContext(this@App)
            modules(appModules)
        }

        if (BuildConfig.DEBUG) enableStrictMode()

        val prefs = PreferenceManager.getDefaultSharedPreferences(this)
        // we cannot call setDefaultValues for multiple fragment based XML preference
        // files with readAgain flag set to false, so always check KEY_HAS_SET_DEFAULT_VALUES
        if (!prefs.getBoolean(PreferenceManager.KEY_HAS_SET_DEFAULT_VALUES, false)) {
            for (screen in SettingsScreen.entries) {
                PreferenceManager.setDefaultValues(this, screen.layoutRes, true)
            }
            if (isExperimentalBuild()) {
                prefs.edit { putBoolean(EXPERIMENTAL_UPDATES, true) }
            }
        }

        // setting Error activity
        CaocConfig.Builder.create()
            .errorActivity(ErrorActivity::class.java)
            .restartActivity(MainActivity::class.java)
            .apply()

        AppCompatDelegate.setDefaultNightMode(getDayNightMode())
    }

    override fun newImageLoader(context: PlatformContext): ImageLoader {
        return ImageLoader.Builder(context)
            .crossfade(true)
            .allowHardware(false)
            .components {
                // Song/album
                add(SongMapper(preferences = get()))
                add(FileMapper(preferences = get()))
                add(AlbumMapper(preferences = get()))
                add(AudioCoverKeyer())
                add(
                    AudioCoverFetcher.Factory(
                        preferences = get(),
                        deezerService = get()
                    )
                )

                // Artist
                add(ArtistMapper())
                add(ArtistImageKeyer(customArtistImageManager = get()))
                add(
                    ArtistImageFetcher.Factory(
                        preferences = get(),
                        customImageManager = get(),
                        deezerService = get()
                    )
                )

                // Playlist
                add(PlaylistMapper())
                add(PlaylistImageKeyer())
                add(PlaylistImageFetcher.Factory())

                // Auto-generated image
                add(GenreMapper())
                add(YearMapper())
                add(AutoGeneratedImageKeyer())
                add(AutoGeneratedImageFetcher.Factory(repository = get()))

                // Network
                add(KtorNetworkFetcherFactory())
            }
            .also { builder ->
                if (BuildConfig.DEBUG) {
                    builder.logger(object : Logger {
                        override var minLevel = Logger.Level.Verbose

                        override fun log(
                            tag: String,
                            level: Logger.Level,
                            message: String?,
                            throwable: Throwable?
                        ) {
                            if (level < minLevel) return
                            message?.let { msg -> Log.println(level.ordinal + 2, tag, msg) }
                            throwable?.let { thr ->
                                Log.println(level.ordinal + 2, tag, Log.getStackTraceString(thr))
                            }
                        }
                    })
                }
            }
            .build()
    }

    override fun onTrimMemory(level: Int) {
        super.onTrimMemory(level)
        ReplayGainTagExtractor.clearCache()

        val imageLoader = SingletonImageLoader.get(this)
        imageLoader.memoryCache?.clear()
    }

    private fun enableStrictMode() {
        StrictMode.setVmPolicy(
            VmPolicy.Builder()
                .detectAll()
                .penaltyLog()
                .build()
        )

        StrictMode.setThreadPolicy(
            ThreadPolicy.Builder()
                .detectAll()
                .penaltyLog()
                .penaltyFlashScreen()
                .build()
        )
    }

    companion object {
        @get:Synchronized
        internal lateinit var instance: App
            private set

        fun isExperimentalBuild(): Boolean =
            BuildConfig.VERSION_NAME.contains("(alpha|beta|rc)".toRegex(RegexOption.IGNORE_CASE))
    }
}