package com.mantz_it.rfanalyzer.ui

import android.graphics.Color
import com.mantz_it.rfanalyzer.ui.composable.FftColorMap

/**
 * <h1>RF Analyzer - Color Maps for the Waterfall Plot</h1>
 *
 * Module:      ColorMaps.kt
 * Description: The color maps for the waterfall plot
 *
 * @author Dennis Mantz
 *
 * Copyright (C) 2025 Dennis Mantz
 * License: http://www.gnu.org/licenses/gpl.html GPL version 2 or higher
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */


fun createColorMap(colorMap: FftColorMap): IntArray {
    return when (colorMap) {
        FftColorMap.GQRX -> createGqrxMap()
        FftColorMap.JET -> createJetColorMap()
        FftColorMap.TURBO-> createTurboMap()
    }
}

fun createGqrxMap(): IntArray {
    return IntArray(256) { i ->
        if (i < 20)         Color.argb(0xff, 0, 0, 0) // level 0: black background
        else if (i < 70)    Color.argb(0xff, 0, 0, 140 * (i - 20) / 50) // level 1: black -> blue
        else if (i < 100)   Color.argb( 0xff, 60 * (i - 70) / 30, 125 * (i - 70) / 30, 115 * (i - 70) / 30 + 140 ) // level 2: blue -> light-blue / greenish
        else if (i < 150)   Color.argb( 0xff, 195 * (i - 100) / 50 + 60, 130 * (i - 100) / 50 + 125, 255 - (255 * (i - 100) / 50) ) // level 3: light blue -> yellow
        else if (i < 250)   Color.argb( 0xff, 255, 255 - 255 * (i - 150) / 100, 0 ) // level 4: yellow -> red
        else                Color.argb( 0xff, 255, 255 * (i - 250) / 5, 255 * (i - 250) / 5 ) // level 5: red -> white
    }
}

fun createJetColorMap(): IntArray {
    return IntArray(256) { index ->
        val x = index / 255.0f
        when {
            x < 0.125f -> {
                val y = x / 0.125f
                Color.argb(1f, 0.0f, 0.0f, 0.5f + 0.5f * y)
            }
            x < 0.375f -> {
                val y = (x - 0.125f) / 0.25f
                Color.argb(1f, 0.0f, y, 1.0f)
            }
            x < 0.625f -> {
                val y = (x - 0.375f) / 0.25f
                Color.argb(1f, y, 1.0f, 1.0f - y)
            }
            x < 0.875f -> {
                val y = (x - 0.625f) / 0.25f
                Color.argb(1f, 1.0f, 1.0f - y, 0.0f)
            }
            else -> {
                val y = (x - 0.875f) / 0.125f
                Color.argb(1f, 1.0f - 0.5f * y, 0.0f, 0.0f)
            }
        }
    }
}

fun createTurboMap(): IntArray {
    fun c(r: Float, g: Float, b: Float): Int {
        return Color.argb(1f, r, g, b)
    }
    return intArrayOf(
        c(0.18995f, 0.07176f, 0.23217f),
        c(0.19483f, 0.08339f, 0.26149f),
        c(0.19956f, 0.09498f, 0.29024f),
        c(0.20415f, 0.10652f, 0.31844f),
        c(0.20860f, 0.11802f, 0.34607f),
        c(0.21291f, 0.12947f, 0.37314f),
        c(0.21708f, 0.14087f, 0.39964f),
        c(0.22111f, 0.15223f, 0.42558f),
        c(0.22500f, 0.16354f, 0.45096f),
        c(0.22875f, 0.17481f, 0.47578f),
        c(0.23236f, 0.18603f, 0.50004f),
        c(0.23582f, 0.19720f, 0.52373f),
        c(0.23915f, 0.20833f, 0.54686f),
        c(0.24234f, 0.21941f, 0.56942f),
        c(0.24539f, 0.23044f, 0.59142f),
        c(0.24830f, 0.24143f, 0.61286f),
        c(0.25107f, 0.25237f, 0.63374f),
        c(0.25369f, 0.26327f, 0.65406f),
        c(0.25618f, 0.27412f, 0.67381f),
        c(0.25853f, 0.28492f, 0.69300f),
        c(0.26074f, 0.29568f, 0.71162f),
        c(0.26280f, 0.30639f, 0.72968f),
        c(0.26473f, 0.31706f, 0.74718f),
        c(0.26652f, 0.32768f, 0.76412f),
        c(0.26816f, 0.33825f, 0.78050f),
        c(0.26967f, 0.34878f, 0.79631f),
        c(0.27103f, 0.35926f, 0.81156f),
        c(0.27226f, 0.36970f, 0.82624f),
        c(0.27334f, 0.38008f, 0.84037f),
        c(0.27429f, 0.39043f, 0.85393f),
        c(0.27509f, 0.40072f, 0.86692f),
        c(0.27576f, 0.41097f, 0.87936f),
        c(0.27628f, 0.42118f, 0.89123f),
        c(0.27667f, 0.43134f, 0.90254f),
        c(0.27691f, 0.44145f, 0.91328f),
        c(0.27701f, 0.45152f, 0.92347f),
        c(0.27698f, 0.46153f, 0.93309f),
        c(0.27680f, 0.47151f, 0.94214f),
        c(0.27648f, 0.48144f, 0.95064f),
        c(0.27603f, 0.49132f, 0.95857f),
        c(0.27543f, 0.50115f, 0.96594f),
        c(0.27469f, 0.51094f, 0.97275f),
        c(0.27381f, 0.52069f, 0.97899f),
        c(0.27273f, 0.53040f, 0.98461f),
        c(0.27106f, 0.54015f, 0.98930f),
        c(0.26878f, 0.54995f, 0.99303f),
        c(0.26592f, 0.55979f, 0.99583f),
        c(0.26252f, 0.56967f, 0.99773f),
        c(0.25862f, 0.57958f, 0.99876f),
        c(0.25425f, 0.58950f, 0.99896f),
        c(0.24946f, 0.59943f, 0.99835f),
        c(0.24427f, 0.60937f, 0.99697f),
        c(0.23874f, 0.61931f, 0.99485f),
        c(0.23288f, 0.62923f, 0.99202f),
        c(0.22676f, 0.63913f, 0.98851f),
        c(0.22039f, 0.64901f, 0.98436f),
        c(0.21382f, 0.65886f, 0.97959f),
        c(0.20708f, 0.66866f, 0.97423f),
        c(0.20021f, 0.67842f, 0.96833f),
        c(0.19326f, 0.68812f, 0.96190f),
        c(0.18625f, 0.69775f, 0.95498f),
        c(0.17923f, 0.70732f, 0.94761f),
        c(0.17223f, 0.71680f, 0.93981f),
        c(0.16529f, 0.72620f, 0.93161f),
        c(0.15844f, 0.73551f, 0.92305f),
        c(0.15173f, 0.74472f, 0.91416f),
        c(0.14519f, 0.75381f, 0.90496f),
        c(0.13886f, 0.76279f, 0.89550f),
        c(0.13278f, 0.77165f, 0.88580f),
        c(0.12698f, 0.78037f, 0.87590f),
        c(0.12151f, 0.78896f, 0.86581f),
        c(0.11639f, 0.79740f, 0.85559f),
        c(0.11167f, 0.80569f, 0.84525f),
        c(0.10738f, 0.81381f, 0.83484f),
        c(0.10357f, 0.82177f, 0.82437f),
        c(0.10026f, 0.82955f, 0.81389f),
        c(0.09750f, 0.83714f, 0.80342f),
        c(0.09532f, 0.84455f, 0.79299f),
        c(0.09377f, 0.85175f, 0.78264f),
        c(0.09287f, 0.85875f, 0.77240f),
        c(0.09267f, 0.86554f, 0.76230f),
        c(0.09320f, 0.87211f, 0.75237f),
        c(0.09451f, 0.87844f, 0.74265f),
        c(0.09662f, 0.88454f, 0.73316f),
        c(0.09958f, 0.89040f, 0.72393f),
        c(0.10342f, 0.89600f, 0.71500f),
        c(0.10815f, 0.90142f, 0.70599f),
        c(0.11374f, 0.90673f, 0.69651f),
        c(0.12014f, 0.91193f, 0.68660f),
        c(0.12733f, 0.91701f, 0.67627f),
        c(0.13526f, 0.92197f, 0.66556f),
        c(0.14391f, 0.92680f, 0.65448f),
        c(0.15323f, 0.93151f, 0.64308f),
        c(0.16319f, 0.93609f, 0.63137f),
        c(0.17377f, 0.94053f, 0.61938f),
        c(0.18491f, 0.94484f, 0.60713f),
        c(0.19659f, 0.94901f, 0.59466f),
        c(0.20877f, 0.95304f, 0.58199f),
        c(0.22142f, 0.95692f, 0.56914f),
        c(0.23449f, 0.96065f, 0.55614f),
        c(0.24797f, 0.96423f, 0.54303f),
        c(0.26180f, 0.96765f, 0.52981f),
        c(0.27597f, 0.97092f, 0.51653f),
        c(0.29042f, 0.97403f, 0.50321f),
        c(0.30513f, 0.97697f, 0.48987f),
        c(0.32006f, 0.97974f, 0.47654f),
        c(0.33517f, 0.98234f, 0.46325f),
        c(0.35043f, 0.98477f, 0.45002f),
        c(0.36581f, 0.98702f, 0.43688f),
        c(0.38127f, 0.98909f, 0.42386f),
        c(0.39678f, 0.99098f, 0.41098f),
        c(0.41229f, 0.99268f, 0.39826f),
        c(0.42778f, 0.99419f, 0.38575f),
        c(0.44321f, 0.99551f, 0.37345f),
        c(0.45854f, 0.99663f, 0.36140f),
        c(0.47375f, 0.99755f, 0.34963f),
        c(0.48879f, 0.99828f, 0.33816f),
        c(0.50362f, 0.99879f, 0.32701f),
        c(0.51822f, 0.99910f, 0.31622f),
        c(0.53255f, 0.99919f, 0.30581f),
        c(0.54658f, 0.99907f, 0.29581f),
        c(0.56026f, 0.99873f, 0.28623f),
        c(0.57357f, 0.99817f, 0.27712f),
        c(0.58646f, 0.99739f, 0.26849f),
        c(0.59891f, 0.99638f, 0.26038f),
        c(0.61088f, 0.99514f, 0.25280f),
        c(0.62233f, 0.99366f, 0.24579f),
        c(0.63323f, 0.99195f, 0.23937f),
        c(0.64362f, 0.98999f, 0.23356f),
        c(0.65394f, 0.98775f, 0.22835f),
        c(0.66428f, 0.98524f, 0.22370f),
        c(0.67462f, 0.98246f, 0.21960f),
        c(0.68494f, 0.97941f, 0.21602f),
        c(0.69525f, 0.97610f, 0.21294f),
        c(0.70553f, 0.97255f, 0.21032f),
        c(0.71577f, 0.96875f, 0.20815f),
        c(0.72596f, 0.96470f, 0.20640f),
        c(0.73610f, 0.96043f, 0.20504f),
        c(0.74617f, 0.95593f, 0.20406f),
        c(0.75617f, 0.95121f, 0.20343f),
        c(0.76608f, 0.94627f, 0.20311f),
        c(0.77591f, 0.94113f, 0.20310f),
        c(0.78563f, 0.93579f, 0.20336f),
        c(0.79524f, 0.93025f, 0.20386f),
        c(0.80473f, 0.92452f, 0.20459f),
        c(0.81410f, 0.91861f, 0.20552f),
        c(0.82333f, 0.91253f, 0.20663f),
        c(0.83241f, 0.90627f, 0.20788f),
        c(0.84133f, 0.89986f, 0.20926f),
        c(0.85010f, 0.89328f, 0.21074f),
        c(0.85868f, 0.88655f, 0.21230f),
        c(0.86709f, 0.87968f, 0.21391f),
        c(0.87530f, 0.87267f, 0.21555f),
        c(0.88331f, 0.86553f, 0.21719f),
        c(0.89112f, 0.85826f, 0.21880f),
        c(0.89870f, 0.85087f, 0.22038f),
        c(0.90605f, 0.84337f, 0.22188f),
        c(0.91317f, 0.83576f, 0.22328f),
        c(0.92004f, 0.82806f, 0.22456f),
        c(0.92666f, 0.82025f, 0.22570f),
        c(0.93301f, 0.81236f, 0.22667f),
        c(0.93909f, 0.80439f, 0.22744f),
        c(0.94489f, 0.79634f, 0.22800f),
        c(0.95039f, 0.78823f, 0.22831f),
        c(0.95560f, 0.78005f, 0.22836f),
        c(0.96049f, 0.77181f, 0.22811f),
        c(0.96507f, 0.76352f, 0.22754f),
        c(0.96931f, 0.75519f, 0.22663f),
        c(0.97323f, 0.74682f, 0.22536f),
        c(0.97679f, 0.73842f, 0.22369f),
        c(0.98000f, 0.73000f, 0.22161f),
        c(0.98289f, 0.72140f, 0.21918f),
        c(0.98549f, 0.71250f, 0.21650f),
        c(0.98781f, 0.70330f, 0.21358f),
        c(0.98986f, 0.69382f, 0.21043f),
        c(0.99163f, 0.68408f, 0.20706f),
        c(0.99314f, 0.67408f, 0.20348f),
        c(0.99438f, 0.66386f, 0.19971f),
        c(0.99535f, 0.65341f, 0.19577f),
        c(0.99607f, 0.64277f, 0.19165f),
        c(0.99654f, 0.63193f, 0.18738f),
        c(0.99675f, 0.62093f, 0.18297f),
        c(0.99672f, 0.60977f, 0.17842f),
        c(0.99644f, 0.59846f, 0.17376f),
        c(0.99593f, 0.58703f, 0.16899f),
        c(0.99517f, 0.57549f, 0.16412f),
        c(0.99419f, 0.56386f, 0.15918f),
        c(0.99297f, 0.55214f, 0.15417f),
        c(0.99153f, 0.54036f, 0.14910f),
        c(0.98987f, 0.52854f, 0.14398f),
        c(0.98799f, 0.51667f, 0.13883f),
        c(0.98590f, 0.50479f, 0.13367f),
        c(0.98360f, 0.49291f, 0.12849f),
        c(0.98108f, 0.48104f, 0.12332f),
        c(0.97837f, 0.46920f, 0.11817f),
        c(0.97545f, 0.45740f, 0.11305f),
        c(0.97234f, 0.44565f, 0.10797f),
        c(0.96904f, 0.43399f, 0.10294f),
        c(0.96555f, 0.42241f, 0.09798f),
        c(0.96187f, 0.41093f, 0.09310f),
        c(0.95801f, 0.39958f, 0.08831f),
        c(0.95398f, 0.38836f, 0.08362f),
        c(0.94977f, 0.37729f, 0.07905f),
        c(0.94538f, 0.36638f, 0.07461f),
        c(0.94084f, 0.35566f, 0.07031f),
        c(0.93612f, 0.34513f, 0.06616f),
        c(0.93125f, 0.33482f, 0.06218f),
        c(0.92623f, 0.32473f, 0.05837f),
        c(0.92105f, 0.31489f, 0.05475f),
        c(0.91572f, 0.30530f, 0.05134f),
        c(0.91024f, 0.29599f, 0.04814f),
        c(0.90463f, 0.28696f, 0.04516f),
        c(0.89888f, 0.27824f, 0.04243f),
        c(0.89298f, 0.26981f, 0.03993f),
        c(0.88691f, 0.26152f, 0.03753f),
        c(0.88066f, 0.25334f, 0.03521f),
        c(0.87422f, 0.24526f, 0.03297f),
        c(0.86760f, 0.23730f, 0.03082f),
        c(0.86079f, 0.22945f, 0.02875f),
        c(0.85380f, 0.22170f, 0.02677f),
        c(0.84662f, 0.21407f, 0.02487f),
        c(0.83926f, 0.20654f, 0.02305f),
        c(0.83172f, 0.19912f, 0.02131f),
        c(0.82399f, 0.19182f, 0.01966f),
        c(0.81608f, 0.18462f, 0.01809f),
        c(0.80799f, 0.17753f, 0.01660f),
        c(0.79971f, 0.17055f, 0.01520f),
        c(0.79125f, 0.16368f, 0.01387f),
        c(0.78260f, 0.15693f, 0.01264f),
        c(0.77377f, 0.15028f, 0.01148f),
        c(0.76476f, 0.14374f, 0.01041f),
        c(0.75556f, 0.13731f, 0.00942f),
        c(0.74617f, 0.13098f, 0.00851f),
        c(0.73661f, 0.12477f, 0.00769f),
        c(0.72686f, 0.11867f, 0.00695f),
        c(0.71692f, 0.11268f, 0.00629f),
        c(0.70680f, 0.10680f, 0.00571f),
        c(0.69650f, 0.10102f, 0.00522f),
        c(0.68602f, 0.09536f, 0.00481f),
        c(0.67535f, 0.08980f, 0.00449f),
        c(0.66449f, 0.08436f, 0.00424f),
        c(0.65345f, 0.07902f, 0.00408f),
        c(0.64223f, 0.07380f, 0.00401f),
        c(0.63082f, 0.06868f, 0.00401f),
        c(0.61923f, 0.06367f, 0.00410f),
        c(0.60746f, 0.05878f, 0.00427f),
        c(0.59550f, 0.05399f, 0.00453f),
        c(0.58336f, 0.04931f, 0.00486f),
        c(0.57103f, 0.04474f, 0.00529f),
        c(0.55852f, 0.04028f, 0.00579f),
        c(0.54583f, 0.03593f, 0.00638f),
        c(0.53295f, 0.03169f, 0.00705f),
        c(0.51989f, 0.02756f, 0.00780f),
        c(0.50664f, 0.02354f, 0.00863f),
        c(0.49321f, 0.01963f, 0.00955f),
        c(0.47960f, 0.01583f, 0.01055f)
    )
}