package com.mantz_it.rfanalyzer.source;

import android.content.ContentResolver;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.provider.OpenableColumns;
import android.util.Log;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * <h1>RF Analyzer - File Source of IQ samples</h1>
 *
 * Module:      FileIQSource.java
 * Description: Simple source of IQ sampling by reading from IQ files generated by the
 *              HackRF. Just for testing.
 *
 * @author Dennis Mantz
 *
 * Copyright (C) 2025 Dennis Mantz
 * License: http://www.gnu.org/licenses/gpl.html GPL version 2 or higher
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
public class FileIQSource implements IQSourceInterface {
	private Callback callback = null;
	private boolean repeat = false;
	private int sampleRate = 1000000;
	private long frequency = 1;
	private int packetSize = 0;
	private long startTime = 0;				// timestamp (in ns) when the source was started
	private long bytesRead = 0;             // total bytes read from file since source start
	private long lastPacketAccessTime = 0;  // only for debugging
	private byte[] buffer = null;
	private Uri uri = null;
	private ContentResolver contentResolver = null;  // Content Resolver which can open the uri
	private BufferedInputStream bufferedInputStream = null;
	private IQConverter iqConverter;
	private int fileFormat;
	private static final String LOGTAG = "FileIQSource";
	public static final int FILE_FORMAT_8BIT_SIGNED = 0;
	public static final int FILE_FORMAT_8BIT_UNSIGNED = 1;
	public static final int FILE_FORMAT_16BIT_SIGNED = 2;

	public FileIQSource() {
	}

	public boolean init(Uri uri, ContentResolver contentResolver, int sampleRate, long frequency, int packetSize, boolean repeat, int fileFormat){
		this.uri = uri;
		this.contentResolver = contentResolver;
		this.repeat = repeat;
		this.fileFormat = fileFormat;
		this.sampleRate = sampleRate;
		this.frequency = frequency;
		this.packetSize = packetSize;
		this.buffer = new byte[packetSize];

		switch (fileFormat) {
			case FILE_FORMAT_8BIT_SIGNED:
				iqConverter = new Signed8BitIQConverter();
				break;
			case FILE_FORMAT_8BIT_UNSIGNED:
				iqConverter = new Unsigned8BitIQConverter();
				break;
			case FILE_FORMAT_16BIT_SIGNED:
				iqConverter = new Signed16BitIQConverter();
				break;
			default:
				Log.e(LOGTAG, "constructor: Invalid file format: " + fileFormat);
				break;
		}
		iqConverter.setFrequency(frequency);
		iqConverter.setSampleRate(sampleRate);
		return true;
	}

	private void reportError(String msg) {
		if(callback != null)
			callback.onIQSourceError(this,msg);
		else
			Log.e(LOGTAG,"Callback is null when reporting Error (" + msg + ")");
	}

	private boolean _openFile() {
		try {
			InputStream inputStream;
			if (uri.getScheme() != null && uri.getScheme().equals("content")) {
				inputStream = contentResolver.openInputStream(uri); // Use ContentResolver for content URIs
			} else {
				inputStream = new FileInputStream(uri.getPath()); // Use FileInputStream for file URIs
			}
			this.bufferedInputStream = new BufferedInputStream(inputStream);
			return true;
		}catch (IOException e) {
			Log.e(LOGTAG, "_openFile: Error while opening file: " + e.getMessage());
			reportError("Error while opening file: " + e.getMessage());
			return false;
		}catch (SecurityException e) {
			Log.e(LOGTAG, "_openFile: Security Exception:" + e.getMessage());
			reportError("Error while opening file: " + e.getMessage());
			return false;
		}
	}

	@Override
	public boolean open(Context context, Callback callback) {
		if(uri == null || contentResolver == null) {
			Log.e(LOGTAG, "open: Error: Uri or contentResolver are null");
			reportError("Error: File Source not correctly initialized");
		}
		this.callback = callback;
		// open the file
		if(_openFile()) {
			callback.onIQSourceReady(this);
			return true;
		} else
			return false;
	}

	@Override
	public boolean isOpen() {
		if(bufferedInputStream == null)
			return false;
		try {
			if(bufferedInputStream.available() > 0)
				return true;
		} catch (IOException e) {

		}
		return false;
	}

	@Override
	public boolean close() {
		// close the file
		try {
			if(bufferedInputStream != null)
				bufferedInputStream.close();
			return true;
		} catch (IOException e) {
			Log.e(LOGTAG, "stopSampling: Error while closing file: " + e.getMessage());
			reportError("Unexpected error while closing file: " + e.getMessage());
			return false;
		}
	}

	@Override
	public String getName() {
		String filename = getFilename();
		if (filename != null)
			return "IQ-File: " + filename;
		else
			return "IQ-File (uninitialized)";
	}

	/**
	 * @return the file name of the file
	 */
	public Uri getUri() {
		return uri;
	}

	public String getFilename() {
		if (uri == null) {
			return null;
		}
		// TODO: take clean kotlin code from MainActivity's openIQFileLauncher once refactored to Kotlin
		if (uri.getScheme() != null && uri.getScheme().equals("content") && contentResolver != null) {
			try {
				Cursor cursor = contentResolver.query(uri, null, null, null, null);
				if (cursor == null) {
					return null;
				}
				int nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
				if (nameIndex != -1) {
					cursor.moveToFirst();
					String filename = cursor.getString(nameIndex);
					cursor.close();
					return filename;
				} else {
					return null;
				}
			} catch (SecurityException e) {
				Log.e(LOGTAG, "getFilename: Security Exception:" + e.getMessage());
				return null;
			}
		} else {
			return uri.getLastPathSegment();
		}
	}

	/**
	 * @return true if repeat is enabled; false if not
	 */
	public boolean isRepeat() {
		return repeat;
	}

	/**
	 * @return if set to true, the file will wrap at the end
	 */
	public void setRepeat(boolean repeat) {
		this.repeat = repeat;
	}

	/**
	 * @return the format of the file: FILE_FORMAT_8BIT_SIGNED, ...
	 */
	public int getFileFormat() {
		return fileFormat;
	}

	/**
	 * @return the format of the file: FILE_FORMAT_8BIT_SIGNED, ...
	 */
	public void setFileFormat(int fileFormat) {
		IQConverter newIqConverter = null;
		switch (fileFormat) {
			case FILE_FORMAT_8BIT_SIGNED:
				newIqConverter = new Signed8BitIQConverter();
				break;
			case FILE_FORMAT_8BIT_UNSIGNED:
				newIqConverter = new Unsigned8BitIQConverter();
				break;
			case FILE_FORMAT_16BIT_SIGNED:
				newIqConverter = new Signed16BitIQConverter();
				break;
			default:
				Log.e(LOGTAG, "setFileFormat: Invalid file format: " + fileFormat);
				break;
		}
		if(newIqConverter != null) {
			newIqConverter.setFrequency(frequency);
			newIqConverter.setSampleRate(sampleRate);
			iqConverter = newIqConverter;
		}
	}

	@Override
	public int getSampleRate() {
		return sampleRate;
	}

	@Override
	public void setSampleRate(int sampleRate) {
		Log.e(LOGTAG,"Setting the sample rate is not supported on a file source");
	}

	@Override
	public long getFrequency() {
		return frequency;
	}

	@Override
	public void setFrequency(long frequency) {
		Log.e(LOGTAG,"Setting the frequency is not supported on a file source");
	}

	@Override
	public long getMaxFrequency() {
		return frequency;
	}

	@Override
	public long getMinFrequency() {
		return frequency;
	}

	@Override
	public int getNextHigherOptimalSampleRate(int sampleRate) {
		return this.sampleRate;
	}

	@Override
	public int getNextLowerOptimalSampleRate(int sampleRate) {
		return this.sampleRate;
	}

	@Override
	public int[] getSupportedSampleRates() {
		return new int[] {this.sampleRate};
	}

	@Override
	public int getPacketSize() {
		return packetSize;
	}

	public int getBytesPerSample() {
		switch (fileFormat) {
			case FILE_FORMAT_8BIT_SIGNED:
			case FILE_FORMAT_8BIT_UNSIGNED:
				return 2;
			case FILE_FORMAT_16BIT_SIGNED:
				return 4;
			default:
				Log.e(LOGTAG, "getBytesPerSample: Invalid file format: " + fileFormat);
				return 0;
		}
	}

	@Override
	public byte[] getPacket(int timeout) {
		byte[] retVal = null;
		if(bufferedInputStream == null)
			return null;

		try {
			// Read the samples.
			if(bufferedInputStream.read(buffer, 0 , buffer.length) == buffer.length) {
				bytesRead += buffer.length;
				retVal = buffer;
			} else {
				if (repeat) {
					// rewind and try again:
					Log.i(LOGTAG, "getPacket: End of File. Rewind!");
					bufferedInputStream.close();
					if(!_openFile()) {
						Log.e(LOGTAG, "getPacket: Error while reopening file.");
						reportError("Error while re-openening file");
						return retVal;
					}
					if (bufferedInputStream.read(buffer, 0, buffer.length) == buffer.length) {
						bytesRead += buffer.length;
						retVal = buffer;
					}
				} else {
					Log.i(LOGTAG, "getPacket: End of File");
					reportError("End of File");
				}
			}

			// Simulate sample rate of real hardware:
			double timePerSample = 1_000_000_000.0 / sampleRate;
			long expectedTime = startTime + (long) (timePerSample * bytesRead / getBytesPerSample());
			long sleep = Math.min(expectedTime - System.nanoTime(), timeout*1_000_000L);
			if(sleep > 0) {
				//Log.i(LOGTAG, "getPacket: sleeping " + sleep + " ns");
				long millis = sleep / 1000000;
				int nanos = (int)(sleep % 1000000);
				Thread.sleep(millis, nanos);
			} else
				Log.w(LOGTAG, "getPacket: To slow for actual sample rate! sleep=" + sleep + " (bytesRead: " + bytesRead + ")");

		} catch (IOException e) {
			Log.e(LOGTAG, "getPacket: Error while reading from file: " + e.getMessage());
			reportError("Unexpected error while reading file: " + e.getMessage());
		} catch (InterruptedException e) {
			Log.w(LOGTAG, "getPacket: Interrupted while sleeping!");
		}
		lastPacketAccessTime = System.nanoTime();
		return retVal;
	}

	@Override
	public void returnPacket(byte[] buffer) {
		// do nothing
	}

	@Override
	public void startSampling() {
		startTime = System.nanoTime();
		bytesRead = 0;
	}

	@Override
	public void stopSampling() {
		// nothing to do here...
	}

	@Override
	public int fillPacketIntoSamplePacket(byte[] packet, SamplePacket samplePacket) {
		return this.iqConverter.fillPacketIntoSamplePacket(packet, samplePacket);
	}

	public int mixPacketIntoSamplePacket(byte[] packet, SamplePacket samplePacket, long channelFrequency) {
		return this.iqConverter.mixPacketIntoSamplePacket(packet, samplePacket, channelFrequency);
	}
}
