package com.machiav3lli.fdroid.viewmodels

import android.content.pm.PackageManager
import android.os.Build
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.machiav3lli.fdroid.NeoApp
import com.machiav3lli.fdroid.RELEASE_STATE_INSTALLED
import com.machiav3lli.fdroid.RELEASE_STATE_NONE
import com.machiav3lli.fdroid.RELEASE_STATE_SUGGESTED
import com.machiav3lli.fdroid.STATEFLOW_SUBSCRIBE_BUFFER
import com.machiav3lli.fdroid.data.content.Preferences
import com.machiav3lli.fdroid.data.database.entity.AntiFeatureDetails
import com.machiav3lli.fdroid.data.database.entity.CategoryDetails
import com.machiav3lli.fdroid.data.database.entity.DownloadStats
import com.machiav3lli.fdroid.data.database.entity.EmbeddedProduct
import com.machiav3lli.fdroid.data.database.entity.ExodusInfo
import com.machiav3lli.fdroid.data.database.entity.Extras
import com.machiav3lli.fdroid.data.database.entity.Installed
import com.machiav3lli.fdroid.data.database.entity.MonthlyPackageSum
import com.machiav3lli.fdroid.data.database.entity.RBLog
import com.machiav3lli.fdroid.data.database.entity.Release
import com.machiav3lli.fdroid.data.database.entity.Repository
import com.machiav3lli.fdroid.data.database.entity.Tracker
import com.machiav3lli.fdroid.data.entity.ActionState
import com.machiav3lli.fdroid.data.entity.DownloadState
import com.machiav3lli.fdroid.data.entity.PrivacyData
import com.machiav3lli.fdroid.data.entity.PrivacyNote
import com.machiav3lli.fdroid.data.entity.ProductItem
import com.machiav3lli.fdroid.data.repository.DownloadedRepository
import com.machiav3lli.fdroid.data.repository.ExtrasRepository
import com.machiav3lli.fdroid.data.repository.InstalledRepository
import com.machiav3lli.fdroid.data.repository.PrivacyRepository
import com.machiav3lli.fdroid.data.repository.ProductsRepository
import com.machiav3lli.fdroid.data.repository.RepositoriesRepository
import com.machiav3lli.fdroid.utils.extension.Quadruple
import com.machiav3lli.fdroid.utils.extension.android.Android
import com.machiav3lli.fdroid.utils.extension.combine
import com.machiav3lli.fdroid.utils.extension.grantedPermissions
import com.machiav3lli.fdroid.utils.extension.text.intToIsoDate
import com.machiav3lli.fdroid.utils.extension.text.nullIfEmpty
import com.machiav3lli.fdroid.utils.findSuggestedProduct
import com.machiav3lli.fdroid.utils.generatePermissionGroups
import com.machiav3lli.fdroid.utils.toPrivacyNote
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.mapLatest
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

@OptIn(ExperimentalCoroutinesApi::class)
class AppPageVM(
    downloadedRepo: DownloadedRepository,
    private val productsRepo: ProductsRepository,
    private val extrasRepo: ExtrasRepository,
    installedRepo: InstalledRepository,
    privacyRepo: PrivacyRepository,
    reposRepo: RepositoriesRepository,
) : ViewModel() {
    private val packageName = MutableStateFlow("")

    private val products = packageName
        .flatMapLatest { pn -> productsRepo.getProduct(pn) }
        .distinctUntilChanged()

    private val developer = products
        .mapLatest {
            it.firstOrNull()?.product?.author?.let {
                it.name.nullIfEmpty() ?: it.email.nullIfEmpty() ?: it.web.nullIfEmpty()
            }.orEmpty()
        }.distinctUntilChanged()

    private val repositories = reposRepo.getAll().distinctUntilChanged()

    private val rbLogs = packageName
        .flatMapLatest { pn -> privacyRepo.getRBLogsMap(pn) }
        .distinctUntilChanged()

    private val installedItem = packageName
        .flatMapLatest { packageName -> installedRepo.get(packageName) }
        .stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(STATEFLOW_SUBSCRIBE_BUFFER),
            null
        )

    private val productRepos = combine(
        products,
        repositories,
    ) { prods, repos ->
        val reposMap = repos.associateBy(Repository::id)
        prods.mapNotNull { product ->
            reposMap[product.product.repositoryId]?.let {
                Pair(product, it)
            }
        }
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(STATEFLOW_SUBSCRIBE_BUFFER),
        emptyList()
    )

    private val suggestedProductRepo = combine(
        productRepos,
        installedItem,
    ) { prodRepos, installed ->
        findSuggestedProduct(prodRepos, installed) { it.first }
    }.stateIn(
        viewModelScope,
        SharingStarted.WhileSubscribed(STATEFLOW_SUBSCRIBE_BUFFER),
        null
    )

    private val releaseItems = combine(
        suggestedProductRepo,
        repositories,
        installedItem,
        rbLogs,
    ) { suggestedProductRepo, repos, installed, logs ->
        val includeIncompatible = Preferences[Preferences.Key.IncompatibleVersions]
        val reposMap = repos.associateBy(Repository::id)

        suggestedProductRepo?.first?.releases.orEmpty()
            .filter { includeIncompatible || it.incompatibilities.isEmpty() }
            .mapNotNull { rel -> reposMap[rel.repositoryId]?.let { Pair(rel, it) } }
            .map { (release, repository) ->
                Quadruple(
                    release,
                    repository,
                    when {
                        installed?.versionCode == release.versionCode && release.signature in installed.signatures
                             -> RELEASE_STATE_INSTALLED

                        release.incompatibilities.isEmpty()
                                && release.selected
                                && release.versionCode >= (installed?.versionCode ?: 0)
                                && (installed?.signatures?.contains(release.signature) ?: true || Preferences[Preferences.Key.DisableSignatureCheck])
                             -> RELEASE_STATE_SUGGESTED

                        else -> RELEASE_STATE_NONE
                    },
                    logs[release.hash],
                )
            }
            .sortedByDescending { it.first.versionCode }
            .toList()
    }

    val coreAppState: StateFlow<CoreAppState> = combine(
        suggestedProductRepo,
        installedItem,
        releaseItems,
        productRepos,
    ) { suggested, installed, releases, prodRepos ->
        CoreAppState(
            suggestedProductRepo = suggested,
            releaseItems = releases,
            productRepos = prodRepos,
            installed = installed,
            isInstalled = installed != null,
            canUpdate = suggested?.first?.canUpdate(installed) ?: false,
            installedVersion = installed?.version ?: "",
        )
    }.stateIn(
        viewModelScope,
        SharingStarted.Lazily,
        CoreAppState()
    )

    // Extra state
    private val categoryDetails = combine(
        suggestedProductRepo,
        productsRepo.getAllCategoryDetails(),
    ) { prod, cats ->
        val catsMap = cats.associateBy(CategoryDetails::name)
        prod?.let {
            it.first.product.categories.map { catsMap[it]?.label ?: it }
        }.orEmpty()
    }.distinctUntilChanged()

    private val downloadingState = downloadedRepo.getLatestFlow(packageName)
        .mapLatest { it?.state }
        .stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(STATEFLOW_SUBSCRIBE_BUFFER),
            null
        )

    private val extras = packageName
        .flatMapLatest { pn -> extrasRepo.get(pn) }
        .stateIn(
            viewModelScope,
            SharingStarted.WhileSubscribed(STATEFLOW_SUBSCRIBE_BUFFER),
            null
        )

    private val authorProducts = combine(
        packageName,
        developer,
        developer.flatMapLatest {
            productsRepo.getAuthorList(it)
        },
    ) { pn, dev, prods ->
        if (dev.isNotEmpty()) prods
            .filter { it.product.packageName != pn }
            .groupBy { it.product.packageName }
            .mapNotNull { it.value.maxByOrNull { embdProd -> embdProd.product.added }?.toItem() }
        else emptyList()
    }.distinctUntilChanged()

    // Privacy state
    private val requestedPermissions = combine(
        packageName,
        installedItem,
    ) { pn, inst ->
        runCatching {
            if (inst != null) NeoApp.context.packageManager.getPackageInfo(
                pn,
                PackageManager.GET_PERMISSIONS
            ).grantedPermissions else emptyMap()
        }.fold(
            onSuccess = { it },
            onFailure = { emptyMap() }
        )
    }.distinctUntilChanged()

    private val exodusInfo = packageName
        .flatMapLatest { pn ->
            privacyRepo.getExodusInfos(pn)
        }
        .mapLatest { it.maxByOrNull(ExodusInfo::version_code) }

    private val trackers = combine(exodusInfo, privacyRepo.getAllTrackers()) { info, trackers ->
        trackers.filter { it.key in info?.trackers.orEmpty() }
    }

    private val privacyData = combine(
        suggestedProductRepo,
        trackers,
        reposRepo.getRepoAntiFeaturesMap().distinctUntilChanged(),
    ) { suggestedProduct, trs, afsMap ->
        PrivacyData(
            permissions = suggestedProduct?.first?.displayRelease
                ?.generatePermissionGroups(NeoApp.context) ?: emptyMap(),
            trackers = trs,
            antiFeatures = suggestedProduct?.let {
                it.first.product.antiFeatures.map { af ->
                    afsMap[af] ?: AntiFeatureDetails(af, "")
                }
            }.orEmpty(),
        )
    }.distinctUntilChanged()

    val privacyPanelState: StateFlow<PrivacyPanelState> = combine(
        trackers,
        installedItem,
        requestedPermissions,
        exodusInfo,
        privacyData,
        rbLogs,
    ) { trackers, installed, permissions, exodus, privacy, logs ->
        PrivacyPanelState(
            trackers = trackers,
            isInstalled = installed != null,
            requestedPermissions = permissions,
            exodusInfo = exodus,
            privacyData = privacy,
            privacyNote = privacy.toPrivacyNote(),
            rbLogs = logs,
        )
    }.stateIn(
        viewModelScope,
        SharingStarted.Lazily,
        PrivacyPanelState()
    )

    // Download stats
    private val downloadStatsInfo = packageName
        .flatMapLatest { pn ->
            combine(
                privacyRepo.getClientSumDownloadStats(pn),
                if (Android.sdk(Build.VERSION_CODES.R))
                    privacyRepo.getSumDownloadOrder(pn)
                else privacyRepo.getSumDownloadOrderLegacy(pn)
            ) { stats, order ->
                Triple(
                    stats.find { it.client == "_total" }?.totalCount ?: 0,
                    stats.filterNot { it.client == "_total" || it.client == "_unknown" }
                        .maxByOrNull { it.totalCount }?.client ?: "",
                    order
                )
            }
        }.distinctUntilChanged()

    private val downloadStatsDailyMap = packageName
        .flatMapLatest { pn ->
            privacyRepo.getLatestDownloadStats(pn)
        }.map {
            it.groupBy { stats -> stats.date.intToIsoDate() }
                .mapValues { entry ->
                    entry.value.groupBy(DownloadStats::client)
                        .mapValues { stats -> stats.value.sumOf { stat -> stat.count } }
                }
        }.distinctUntilChanged()

    private val downloadStatsMonthlyMap = packageName
        .flatMapLatest { pn -> privacyRepo.getMonthlyDownloadStats(pn) }
        .map {
            it.groupBy { stats -> stats.yearMonth.intToIsoDate() }
                .mapValues { entry ->
                    entry.value.groupBy(MonthlyPackageSum::client)
                        .mapValues { stats -> stats.value.sumOf { stat -> stat.totalCount } }
                }
        }.distinctUntilChanged()

    val downloadStatsState: StateFlow<DownloadStatsState> = combine(
        downloadStatsInfo,
        downloadStatsDailyMap,
        downloadStatsMonthlyMap,
    ) { info, daily, monthly ->
        DownloadStatsState(
            info = info,
            dailyMap = daily,
            monthlyMap = monthly,
        )
    }.stateIn(
        viewModelScope,
        SharingStarted.Lazily,
        DownloadStatsState()
    )

    private val actions: Flow<Pair<ActionState, Set<ActionState>>> = combine(
        productRepos,
        installedItem,
        suggestedProductRepo,
        downloadingState,
        extras,
    ) { prods, ins, sgst, ds, extras ->
        val product = sgst?.first
        val compatible = product != null && product.selectedReleases.firstOrNull()
            .let { it != null && it.incompatibilities.isEmpty() }
        val canInstall =
            product != null && ins == null && compatible && ds?.isActive != true
        val canUpdate =
            product != null && compatible && product.canUpdate(ins) &&
                    !shouldIgnore(product.versionCode, extras) && ds?.isActive != true
        val canUninstall = product != null && ins != null && !ins.isSystem
        val canLaunch = product != null &&
                ins != null && ins.launcherActivities.isNotEmpty()
        val canShare = product != null && prods.any { it.second.webBaseUrl.isNotBlank() }

        val actions = mutableSetOf<ActionState>()
        synchronized(actions) {
            if (canUpdate) actions += ActionState.Update
            else if (canInstall && !Preferences[Preferences.Key.KidsMode]) actions += ActionState.Install
            if (canLaunch) actions += ActionState.Launch
            if (ins != null) actions += ActionState.Details
            if (canUninstall) actions += ActionState.Uninstall
            if (canShare) actions += ActionState.Share
        }
        val primaryAction = when {
            canUpdate                                            -> ActionState.Update
            canLaunch                                            -> ActionState.Launch
            canInstall && !Preferences[Preferences.Key.KidsMode] -> ActionState.Install
            canShare                                             -> ActionState.Share
            else                                                 -> ActionState.NoAction
        }

        val mA = if (ds != null && ds.isActive)
            ds.toActionState() ?: primaryAction
        else primaryAction
        mA to actions.minus(mA)
    }.distinctUntilChanged()

    val extraAppState: StateFlow<ExtraAppState> = combine(
        repositories,
        authorProducts,
        categoryDetails,
        downloadingState,
        actions,
        extras,
    ) { repos, authorProds, categories, downloading, acts, ext ->
        ExtraAppState(
            repositories = repos,
            authorProducts = authorProds,
            categoryDetails = categories,
            downloadingState = downloading,
            mainAction = acts.first,
            subActions = acts.second,
            extras = ext,
        )
    }.stateIn(
        viewModelScope,
        SharingStarted.Lazily,
        ExtraAppState()
    )

    fun setApp(pn: String) = packageName.update { pn }

    private fun shouldIgnore(appVersionCode: Long, extras: Extras?): Boolean =
        extras?.ignoredVersion == appVersionCode || extras?.ignoreUpdates == true

    fun setIgnoredVersion(packageName: String, versionCode: Long) {
        viewModelScope.launch {
            extrasRepo.setIgnoredVersion(packageName, versionCode)
        }
    }

    fun setIgnoreUpdates(packageName: String, setBoolean: Boolean) {
        viewModelScope.launch {
            extrasRepo.setIgnoreUpdates(packageName, setBoolean)
        }
    }

    fun setIgnoreVulns(packageName: String, setBoolean: Boolean) {
        viewModelScope.launch {
            extrasRepo.setIgnoreVulns(packageName, setBoolean)
        }
    }

    fun setAllowUnstableUpdates(packageName: String, setBoolean: Boolean) {
        viewModelScope.launch {
            extrasRepo.upsertExtra(packageName) {
                if (it != null) updateAllowUnstable(packageName, setBoolean)
                else insert(Extras(packageName, allowUnstable = setBoolean))
                val features = NeoApp.context.packageManager.systemAvailableFeatures
                    .asSequence().map { feat -> feat.name }
                    .toSet() + setOf("android.hardware.touchscreen")
                /* TODO productsRepo.updateReleases(
                    features,
                    setBoolean || Preferences[Preferences.Key.UpdateUnstable],
                    *repos.value.map { EmbeddedProduct(it.first.toV2(),it.first.releases) }.toList().toTypedArray(),
                )*/
                productsRepo.upsertProduct(
                    *productRepos.value.map { prodRepo ->
                        prodRepo.first.apply {
                            refreshReleases(
                                features,
                                setBoolean || Preferences[Preferences.Key.UpdateUnstable]
                            )
                        }.product
                    }.toTypedArray()
                )
            }
        }
    }

    fun setFavorite(packageName: String, setBoolean: Boolean) {
        viewModelScope.launch {
            extrasRepo.setFavorite(packageName, setBoolean)
        }
    }
}

data class PrivacyPanelState(
    val trackers: List<Tracker> = emptyList(),
    val isInstalled: Boolean = false,
    val requestedPermissions: Map<String, Boolean> = emptyMap(),
    val exodusInfo: ExodusInfo? = null,
    val privacyData: PrivacyData = PrivacyData(),
    val privacyNote: PrivacyNote = PrivacyNote(),
    val rbLogs: Map<String, RBLog> = emptyMap(),
)

data class DownloadStatsState(
    val info: Triple<Long, String, Int> = Triple(0L, "", 9999),
    val dailyMap: Map<String, Map<String, Long>> = emptyMap(),
    val monthlyMap: Map<String, Map<String, Long>> = emptyMap(),
)

data class CoreAppState(
    val suggestedProductRepo: Pair<EmbeddedProduct, Repository>? = null,
    val releaseItems: List<Quadruple<Release, Repository, Int, RBLog?>> = emptyList(),
    val productRepos: List<Pair<EmbeddedProduct, Repository>> = emptyList(),
    val installed: Installed? = null,
    val isInstalled: Boolean = false,
    val canUpdate: Boolean = false,
    val installedVersion: String = "",
)

data class ExtraAppState(
    val repositories: List<Repository> = emptyList(),
    val authorProducts: List<ProductItem> = emptyList(),
    val categoryDetails: List<String> = emptyList(),
    val downloadingState: DownloadState? = null,
    val mainAction: ActionState = ActionState.Bookmark,
    val subActions: Set<ActionState> = emptySet(),
    val extras: Extras? = null,
)