package com.ltrademark.hourly

import android.Manifest
import android.annotation.SuppressLint
import android.app.AlarmManager
import android.content.Intent
import android.content.SharedPreferences
import android.content.pm.PackageManager
import android.media.RingtoneManager
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.os.PowerManager
import android.provider.OpenableColumns
import android.provider.Settings
import android.view.View
import android.widget.Button
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.core.content.edit
import androidx.core.net.toUri
import com.google.android.material.switchmaterial.SwitchMaterial

class MainActivity : AppCompatActivity() {

    private lateinit var prefs: SharedPreferences
    private val notificationPermissionsCode = 101

    // Launchers for Native system sounds
    private val pickShortTone = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
        if (result.resultCode == RESULT_OK) {
            val intent = result.data
            val uri: Uri? = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                intent?.getParcelableExtra(RingtoneManager.EXTRA_RINGTONE_PICKED_URI, Uri::class.java)
            } else {
                @Suppress("DEPRECATION")
                intent?.getParcelableExtra(RingtoneManager.EXTRA_RINGTONE_PICKED_URI)
            }

            if (uri != null) {
                saveCustomTone(uri, "custom_tone_short", findViewById(R.id.txtShortToneName), findViewById(R.id.btnClearShort))
            }
        }
    }

    private val pickLongTone = registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
        if (result.resultCode == RESULT_OK) {
            val intent = result.data
            val uri: Uri? = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                intent?.getParcelableExtra(RingtoneManager.EXTRA_RINGTONE_PICKED_URI, Uri::class.java)
            } else {
                @Suppress("DEPRECATION")
                intent?.getParcelableExtra(RingtoneManager.EXTRA_RINGTONE_PICKED_URI)
            }

            if (uri != null) {
                saveCustomTone(uri, "custom_tone_long", findViewById(R.id.txtLongToneName), findViewById(R.id.btnClearLong))
            }
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)

        supportActionBar?.title = "HRLY Settings"

        prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)

        checkAndRequestPermissions()
        setupBatteryWarning()

        setupServiceToggle()
        setupFooterDebug()
        setupCustomSounds()
        setupQuietHours()
    }

    private fun checkAndRequestPermissions() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            if (ContextCompat.checkSelfPermission(this, Manifest.permission.POST_NOTIFICATIONS) != PackageManager.PERMISSION_GRANTED) {
                ActivityCompat.requestPermissions(
                    this,
                    arrayOf(Manifest.permission.POST_NOTIFICATIONS),
                    notificationPermissionsCode
                )
            }
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            val alarmManager = getSystemService(AlarmManager::class.java)
            if (!alarmManager.canScheduleExactAlarms()) {
                val intent = Intent(
                    Settings.ACTION_REQUEST_SCHEDULE_EXACT_ALARM,
                    "package:$packageName".toUri()
                )
                startActivity(intent)
                Toast.makeText(this, "Please allow 'Alarms & Reminders' for HRLY", Toast.LENGTH_LONG).show()
            }
        }
    }

    override fun onRequestPermissionsResult(requestCode: Int, permissions: Array<out String>, grantResults: IntArray) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if (requestCode == notificationPermissionsCode) {
            if (grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                Toast.makeText(this, "Notifications allowed", Toast.LENGTH_SHORT).show()
            } else {
                Toast.makeText(this, "Permission denied. Service status won\'t be visible.", Toast.LENGTH_LONG).show()
            }
        }
    }

    @SuppressLint("BatteryLife")
    private fun setupBatteryWarning() {
        val containerWarning = findViewById<LinearLayout>(R.id.containerBatteryWarning)
        val btnFix = findViewById<Button>(R.id.btnFixBattery)
        val powerManager = getSystemService(POWER_SERVICE) as PowerManager
        val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)

        fun updateWarningVisibility() {
            val isServiceEnabled = prefs.getBoolean("service_enabled", false)
            val isIgnored = powerManager.isIgnoringBatteryOptimizations(packageName)

            if (isServiceEnabled && !isIgnored) {
                containerWarning.visibility = View.VISIBLE
            } else {
                containerWarning.visibility = View.GONE
            }
        }

        updateWarningVisibility()

        btnFix.setOnClickListener {
            try {
                val intent = Intent(Settings.ACTION_REQUEST_IGNORE_BATTERY_OPTIMIZATIONS).apply {
                    data = "package:$packageName".toUri()
                }
                startActivity(intent)
            } catch (e: Exception) {
                try {
                    val intent = Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS).apply {
                        data = "package:$packageName".toUri()
                    }
                    startActivity(intent)
                    Toast.makeText(this, "Go to Battery > Unrestricted", Toast.LENGTH_LONG).show()
                } catch (e: Exception) {
                    Toast.makeText(this, "Could not open settings", Toast.LENGTH_SHORT).show()
                }
            }
        }
    }

    private fun setupServiceToggle() {
        val switchService = findViewById<SwitchMaterial>(R.id.switchService)
        val containerCustomSoundsToggle = findViewById<LinearLayout>(R.id.containerCustomSoundsToggle)
        val containerQuietPickersToggle = findViewById<LinearLayout>(R.id.containerQuietToggle)
        val containerQuietPickers = findViewById<LinearLayout>(R.id.containerQuietPickers)

        switchService.setOnCheckedChangeListener { _, isChecked ->
            prefs.edit {
                putBoolean("service_enabled", isChecked)
            }

            updateVisualContainerState(containerCustomSoundsToggle, isChecked)
            updateVisualContainerState(containerQuietPickersToggle, isChecked)

            val isQuietEnabled = prefs.getBoolean("quiet_enabled", false)
            updateVisualContainerState(containerQuietPickers, isChecked && isQuietEnabled)

            if (isChecked) {
                val intent = Intent(this, ChimeService::class.java)
                startForegroundService(intent)
                if (switchService.isPressed) Toast.makeText(this, "Hourly Chime Enabled", Toast.LENGTH_SHORT).show()
            } else {
                val intent = Intent(this, ChimeService::class.java)
                stopService(intent)
            }
        }

        val isServiceEnabled = prefs.getBoolean("service_enabled", false)
        val isQuietEnabled = prefs.getBoolean("quiet_enabled", false)

        switchService.isChecked = isServiceEnabled

        updateVisualContainerState(containerCustomSoundsToggle, isServiceEnabled)
        updateVisualContainerState(containerQuietPickersToggle, isServiceEnabled)

        if (isServiceEnabled && isQuietEnabled) {
            containerQuietPickers.visibility = View.VISIBLE
        } else {
            containerQuietPickers.visibility = View.GONE
        }
    }

    private fun updateVisualContainerState(container: View, isEnabled: Boolean) {
        if (isEnabled) {
            container.visibility = View.VISIBLE
            container.alpha = 0f
            container.animate().alpha(1f).setDuration(300).start()
        } else {
            container.visibility = View.GONE
        }
    }

    private fun setupCustomSounds() {
        val switchCustomSounds = findViewById<SwitchMaterial>(R.id.switchCustomSounds)
        val containerSoundPickers = findViewById<LinearLayout>(R.id.containerSoundPickers)

        val isCustomEnabled = prefs.getBoolean("custom_sounds_enabled", false)
        switchCustomSounds.isChecked = isCustomEnabled
        updateVisualContainerState(containerSoundPickers, isCustomEnabled)

        switchCustomSounds.setOnCheckedChangeListener { _, isChecked ->
            prefs.edit {
                putBoolean("custom_sounds_enabled", isChecked)
            }
            updateVisualContainerState(containerSoundPickers, isChecked)
        }

        restoreToneState("custom_tone_short", findViewById(R.id.txtShortToneName), findViewById(R.id.btnClearShort))
        restoreToneState("custom_tone_long", findViewById(R.id.txtLongToneName), findViewById(R.id.btnClearLong))

        findViewById<Button>(R.id.btnPickShort).setOnClickListener {
            val currentUriString = prefs.getString("custom_tone_short", null)
            val intent = Intent(RingtoneManager.ACTION_RINGTONE_PICKER).apply {
                putExtra(RingtoneManager.EXTRA_RINGTONE_TYPE, RingtoneManager.TYPE_NOTIFICATION)
                putExtra(RingtoneManager.EXTRA_RINGTONE_TITLE, "Select Short Tone")
                putExtra(RingtoneManager.EXTRA_RINGTONE_SHOW_SILENT, false) // We have a clear button, so hide Silent
                putExtra(RingtoneManager.EXTRA_RINGTONE_SHOW_DEFAULT, true)

                // Pre-select the current tone if one exists
                if (currentUriString != null) {
                    putExtra(RingtoneManager.EXTRA_RINGTONE_EXISTING_URI,
                        currentUriString.toUri())
                }
            }
            pickShortTone.launch(intent)
        }

        findViewById<Button>(R.id.btnPickLong).setOnClickListener {
            val currentUriString = prefs.getString("custom_tone_long", null)
            val intent = Intent(RingtoneManager.ACTION_RINGTONE_PICKER).apply {
                putExtra(RingtoneManager.EXTRA_RINGTONE_TYPE, RingtoneManager.TYPE_NOTIFICATION)
                putExtra(RingtoneManager.EXTRA_RINGTONE_TITLE, "Select Long Tone")
                putExtra(RingtoneManager.EXTRA_RINGTONE_SHOW_SILENT, false)
                putExtra(RingtoneManager.EXTRA_RINGTONE_SHOW_DEFAULT, true)

                if (currentUriString != null) {
                    putExtra(RingtoneManager.EXTRA_RINGTONE_EXISTING_URI,
                        currentUriString.toUri())
                }
            }
            pickLongTone.launch(intent)
        }

        findViewById<ImageView>(R.id.btnClearShort).setOnClickListener {
            clearCustomTone("custom_tone_short", findViewById(R.id.txtShortToneName), it)
        }
        findViewById<ImageView>(R.id.btnClearLong).setOnClickListener {
            clearCustomTone("custom_tone_long", findViewById(R.id.txtLongToneName), it)
        }
    }

    private fun saveCustomTone(uri: Uri, key: String, textView: TextView, clearBtn: View) {
        try {
            contentResolver.takePersistableUriPermission(uri, Intent.FLAG_GRANT_READ_URI_PERMISSION)
        } catch (_: Exception) {
            // This is expected for system ringtones (content://media/internal/...), so we ignore it.
        }

        prefs.edit {
            putString(key, uri.toString())
        }

        val ringtone = RingtoneManager.getRingtone(this, uri)
        val name = ringtone?.getTitle(this) ?: "Custom Audio"

        textView.text = name
        clearBtn.visibility = View.VISIBLE
    }

    private fun restoreToneState(key: String, textView: TextView, clearBtn: View) {
        val uriString = prefs.getString(key, null)
        if (uriString != null) {
            val fileName = getFileName(uriString.toUri()) ?: getString(R.string.custom_audio)
            textView.text = fileName
            clearBtn.visibility = View.VISIBLE
        } else {
            textView.text = getString(R.string.default_tone)
            clearBtn.visibility = View.GONE
        }
    }

    private fun clearCustomTone(key: String, textView: TextView, clearBtn: View) {
        prefs.edit {
            remove(key)
        }
        textView.text = getString(R.string.default_tone)
        clearBtn.visibility = View.GONE
        Toast.makeText(this, getString(R.string.reset_to_default_tone), Toast.LENGTH_SHORT).show()
    }

    private fun getFileName(uri: Uri): String? {
        if (uri.scheme == "content") {
            contentResolver.query(uri, null, null, null, null)?.use { cursor ->
                if (cursor.moveToFirst()) {
                    val index = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)
                    if (index >= 0) return cursor.getString(index)
                }
            }
        }
        return uri.path?.let { path ->
            val cut = path.lastIndexOf('/')
            if (cut != -1) path.substring(cut + 1) else path
        }
    }

    private fun setupQuietHours() {
        val switchQuiet = findViewById<SwitchMaterial>(R.id.switchQuietHours)
        val containerPickers = findViewById<LinearLayout>(R.id.containerQuietPickers)
        val btnStart = findViewById<Button>(R.id.btnQuietStart)
        val btnEnd = findViewById<Button>(R.id.btnQuietEnd)

        val isEnabled = prefs.getBoolean("quiet_enabled", false)
        var startH = prefs.getInt("quiet_start_h", 22)
        var startM = prefs.getInt("quiet_start_m", 0)
        var endH = prefs.getInt("quiet_end_h", 7)
        var endM = prefs.getInt("quiet_end_m", 0)

        fun updateTimeButton(btn: Button, hour: Int, min: Int) {
            val calendar = java.util.Calendar.getInstance()
            calendar.set(java.util.Calendar.HOUR_OF_DAY, hour)
            calendar.set(java.util.Calendar.MINUTE, min)
            val format = java.text.SimpleDateFormat("h:mm a", java.util.Locale.getDefault())
            btn.text = format.format(calendar.time)
        }

        switchQuiet.isChecked = isEnabled
        containerPickers.visibility = if (isEnabled) View.VISIBLE else View.GONE
        updateTimeButton(btnStart, startH, startM)
        updateTimeButton(btnEnd, endH, endM)

        switchQuiet.setOnCheckedChangeListener { _, isChecked ->
            prefs.edit { putBoolean("quiet_enabled", isChecked) }
            if (isChecked) {
                containerPickers.visibility = View.VISIBLE
                containerPickers.alpha = 0f
                containerPickers.animate().alpha(1f).setDuration(300).start()
            } else {
                containerPickers.visibility = View.GONE
            }
        }

        btnStart.setOnClickListener {
            android.app.TimePickerDialog(this, { _, h, m ->
                startH = h; startM = m
                prefs.edit { putInt("quiet_start_h", h); putInt("quiet_start_m", m) }
                updateTimeButton(btnStart, h, m)
            }, startH, startM, false).show()
        }

        btnEnd.setOnClickListener {
            android.app.TimePickerDialog(this, { _, h, m ->
                endH = h; endM = m
                prefs.edit { putInt("quiet_end_h", h); putInt("quiet_end_m", m) }
                updateTimeButton(btnEnd, h, m)
            }, endH, endM, false).show()
        }
    }

    private fun setupFooterDebug() {
        val footerText = findViewById<TextView>(R.id.footerText)
        val debugLayout = findViewById<LinearLayout>(R.id.debugLayout)
        var tapCount = 0
        var isDebugUnlocked = false
        var currentToast: Toast? = null

        footerText.setOnClickListener {
            currentToast?.cancel()

            if (isDebugUnlocked) {
                currentToast = Toast.makeText(this, "Debug mode is already active", Toast.LENGTH_SHORT)
                currentToast?.show()
                return@setOnClickListener
            }

            tapCount++

            if (tapCount >= 5) {
                isDebugUnlocked = true
                debugLayout.visibility = View.VISIBLE

                currentToast = Toast.makeText(this, "Debug Mode Unlocked!", Toast.LENGTH_SHORT)
                currentToast?.show()

            } else if (tapCount > 2) {
                val tapsRemaining = 5 - tapCount
                currentToast = Toast.makeText(this, "You are $tapsRemaining steps away from being a developer", Toast.LENGTH_SHORT)
                currentToast?.show()
            }
        }

        // Debug Buttons
        findViewById<Button>(R.id.btnTest8).setOnClickListener {
            val intent = Intent(this, ChimeService::class.java).apply {
                action = ChimeService.ACTION_PLAY_CHIME
                putExtra(ChimeService.EXTRA_TEST_HOUR, 8)
            }
            startForegroundService(intent)
        }

        findViewById<Button>(R.id.btnTest12).setOnClickListener {
            val intent = Intent(this, ChimeService::class.java).apply {
                action = ChimeService.ACTION_PLAY_CHIME
                putExtra(ChimeService.EXTRA_TEST_HOUR, 12)
            }
            startForegroundService(intent)
        }

        findViewById<Button>(R.id.btnTestNow).setOnClickListener {
            val intent = Intent(this, ChimeService::class.java).apply {
                action = ChimeService.ACTION_PLAY_CHIME
            }
            startForegroundService(intent)
        }

        findViewById<Button>(R.id.btnTestVisual).setOnClickListener {
            if (!Settings.canDrawOverlays(this)) {
                Toast.makeText(this, "Enable 'Passive AOD' first (Permission needed)", Toast.LENGTH_SHORT).show()
                return@setOnClickListener
            }
            val intent = Intent(this, ChimeService::class.java).apply {
                action = ChimeService.ACTION_TEST_VISUAL
            }
            startForegroundService(intent)
        }
    }

    override fun onCreateOptionsMenu(menu: android.view.Menu?): Boolean {
        menuInflater.inflate(R.menu.main_menu, menu)
        return true
    }

    override fun onOptionsItemSelected(item: android.view.MenuItem): Boolean {
        return when (item.itemId) {
            R.id.action_about -> {
                showAboutDialog()
                true
            }
            else -> super.onOptionsItemSelected(item)
        }
    }

    @SuppressLint("SetTextI18n")
    private fun showAboutDialog() {
        val dialogView = layoutInflater.inflate(R.layout.dialog_about, null)

        val dialog = com.google.android.material.dialog.MaterialAlertDialogBuilder(this)
            .setView(dialogView)
            .show()

        dialog.window?.setBackgroundDrawableResource(android.R.color.transparent)

        val txtVersion = dialogView.findViewById<TextView>(R.id.txtVersion)
        txtVersion.text = "Version ${BuildConfig.VERSION_NAME}\n2026 © Ltrademark. All rights reserved."

        val btnOk = dialogView.findViewById<Button>(R.id.btnAboutOk)
        btnOk.setOnClickListener {
            dialog.dismiss()
        }

        val btnSource = dialogView.findViewById<Button>(R.id.btnAboutSource)
        btnSource.setOnClickListener {
            val browserIntent = Intent(Intent.ACTION_VIEW, "https://github.com/ltrademark/HRLY".toUri())
            startActivity(browserIntent)
        }
    }

    override fun onResume() {
        super.onResume()

        val isServiceEnabled = prefs.getBoolean("service_enabled", false)
        val switchService = findViewById<SwitchMaterial>(R.id.switchService)
        val containerCustomSoundsToggle = findViewById<LinearLayout>(R.id.containerCustomSoundsToggle)
        val containerQuietPickersToggle = findViewById<LinearLayout>(R.id.containerQuietToggle)

        setupBatteryWarning()

        if (switchService.isChecked != isServiceEnabled) {
            switchService.isChecked = isServiceEnabled
            updateVisualContainerState(containerCustomSoundsToggle, isServiceEnabled)
            updateVisualContainerState(containerQuietPickersToggle, isServiceEnabled)
        }

        restoreToneState("custom_tone_short", findViewById(R.id.txtShortToneName), findViewById(R.id.btnClearShort))
        restoreToneState("custom_tone_long", findViewById(R.id.txtLongToneName), findViewById(R.id.btnClearLong))
    }
}
