package com.ltrademark.hourly

import android.app.*
import android.content.Intent
import android.graphics.PixelFormat
import android.media.AudioAttributes
import android.media.MediaPlayer
import android.os.Build
import android.os.IBinder
import android.os.PowerManager
import android.provider.Settings
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import android.widget.ImageView
import androidx.annotation.RequiresApi
import androidx.core.app.NotificationCompat
import androidx.core.content.edit
import androidx.core.net.toUri
import kotlinx.coroutines.*
import java.util.*

class ChimeService : Service() {

    companion object {
        const val ACTION_PLAY_CHIME = "com.ltrademark.hourly.ACTION_PLAY_CHIME"
        const val ACTION_TEST_VISUAL = "com.ltrademark.hourly.ACTION_TEST_VISUAL"
        const val ACTION_SKIP_NEXT = "com.ltrademark.hourly.ACTION_SKIP_NEXT"
        const val ACTION_STOP_SERVICE = "com.ltrademark.hourly.ACTION_STOP_SERVICE"
        const val ACTION_TOGGLE_SUSPEND = "com.ltrademark.hourly.ACTION_TOGGLE_SUSPEND"

        const val EXTRA_TEST_HOUR = "com.ltrademark.hourly.EXTRA_TEST_HOUR"
    }

    private val serviceScope = CoroutineScope(Dispatchers.Main + Job())
    private var windowManager: WindowManager? = null
    private var overlayView: View? = null

    // Configuration
    private val longToneDuration = 1500L
    private val shortToneDuration = 500L

    override fun onBind(intent: Intent?): IBinder? = null

    @RequiresApi(Build.VERSION_CODES.S)
    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        startForeground(1, createNotification())

        when (intent?.action) {
            ACTION_TOGGLE_SUSPEND -> {
                val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)
                val currentState = prefs.getBoolean("is_suspended", false)
                prefs.edit { putBoolean("is_suspended", !currentState) }

                val notificationManager = getSystemService(NotificationManager::class.java)
                notificationManager.notify(1, createNotification())
            }

            ACTION_PLAY_CHIME -> {
                val testHour = intent.getIntExtra(EXTRA_TEST_HOUR, -1)
                val hourToPlay = if (testHour != -1) testHour else Calendar.getInstance().get(Calendar.HOUR_OF_DAY)

                if (testHour == -1) {
                    scheduleNextChime()
                }

                val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)
                val isSuspended = prefs.getBoolean("is_suspended", false)

                if (testHour == -1 && (isDndActive() || isQuietTime() || isSuspended)) {
                    if (!isSuspended) {
                        stopSelf()
                    }
                    return START_NOT_STICKY
                }

                playSequenceForHour(hourToPlay)

                if (testHour == -1) {
                    scheduleNextChime()
                }
            }
            ACTION_TEST_VISUAL -> {
                serviceScope.launch(Dispatchers.Main) {
                    showVisualPulse(3000L, forceShow = true)
                }
            }
            ACTION_SKIP_NEXT -> {
                skipNextChime()
            }
            ACTION_STOP_SERVICE -> {
                stopChimeService()
            }
            else -> {
                scheduleNextChime()
            }
        }

        return START_STICKY
    }

    @RequiresApi(Build.VERSION_CODES.S)
    private suspend fun playTone(defaultResId: Int, duration: Long, type: String) {
        val mediaPlayer = MediaPlayer()
        val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)

        val isCustomEnabled = prefs.getBoolean("custom_sounds_enabled", false)

        val customKey = when {
            !isCustomEnabled -> null
            type == "short" -> "custom_tone_short"
            type == "long" -> "custom_tone_long"
            else -> null
        }

        val customUriString = if (customKey != null) prefs.getString(customKey, null) else null

        try {
            if (customUriString != null) {
                mediaPlayer.setDataSource(this, customUriString.toUri())
            } else {
                val afd = resources.openRawResourceFd(defaultResId)
                mediaPlayer.setDataSource(afd.fileDescriptor, afd.startOffset, afd.length)
                afd.close()
            }

            val attributes = AudioAttributes.Builder()
                .setUsage(AudioAttributes.USAGE_NOTIFICATION)
                .setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION)
                .build()
            mediaPlayer.setAudioAttributes(attributes)

            mediaPlayer.prepare()
            showVisualPulse(duration)
            mediaPlayer.start()
            delay(duration)
            mediaPlayer.release()

        } catch (e: Exception) {
            e.printStackTrace()
            if (customUriString != null) {
                mediaPlayer.release()
                playTone(defaultResId, duration, "fallback")
            } else {
                mediaPlayer.release()
            }
        }
    }

    @RequiresApi(Build.VERSION_CODES.S)
    override fun onCreate() {
        super.onCreate()
        windowManager = getSystemService(WINDOW_SERVICE) as WindowManager
        startForeground(1, createNotification())
        scheduleNextChime()
    }

    private fun getNextChimeTime(hourOffset: Int = 1): String {
        val calendar = Calendar.getInstance()
        calendar.add(Calendar.HOUR_OF_DAY, hourOffset)
        calendar.set(Calendar.MINUTE, 0)

        val hour = calendar.get(Calendar.HOUR_OF_DAY)
        val amPm = if (hour >= 12) "PM" else "AM"
        val hour12 = if (hour % 12 == 0) 12 else hour % 12

        return "$hour12:00 $amPm"
    }

    private fun createNotification(): Notification {
        val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)
        val isSuspended = prefs.getBoolean("is_suspended", false)

        val channelId = "chime_channel"
        val importance = NotificationManager.IMPORTANCE_MIN
        val channel = NotificationChannel(channelId, "Hourly Chime Service", importance).apply {
            description = "Background service for hourly chimes"
            setShowBadge(false)
        }
        getSystemService(NotificationManager::class.java).createNotificationChannel(channel)

        val skipIntent = Intent(this, ChimeService::class.java).apply { action = ACTION_SKIP_NEXT }
        val skipPendingIntent = PendingIntent.getService(
            this, 1, skipIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val stopIntent = Intent(this, ChimeService::class.java).apply { action = ACTION_STOP_SERVICE }
        val stopPendingIntent = PendingIntent.getService(
            this, 2, stopIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val mainIntent = Intent(this, MainActivity::class.java)
        val mainPendingIntent = PendingIntent.getActivity(
            this, 0, mainIntent, PendingIntent.FLAG_IMMUTABLE
        )

        val builder = NotificationCompat.Builder(this, channelId)
            .setSmallIcon(R.drawable.ic_stat_chime)
            .setPriority(NotificationCompat.PRIORITY_MIN)
            .setCategory(Notification.CATEGORY_SERVICE)
            .setContentIntent(mainPendingIntent)
            .setOngoing(true)

        if (isSuspended) {
            val resumeIntent = Intent(this, ChimeService::class.java).apply { action = ACTION_TOGGLE_SUSPEND }
            val resumePending = PendingIntent.getService(this, 3, resumeIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE)

            builder.setContentTitle(getString(R.string.status_suspended))
            builder.setContentText(getString(R.string.status_suspended_description))
            builder.addAction(android.R.drawable.ic_media_play, getString(R.string.action_resume), resumePending)

        } else {
            val skipIntent = Intent(this, ChimeService::class.java).apply { action = ACTION_SKIP_NEXT }
            val skipPending = PendingIntent.getService(this, 1, skipIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE)

            val suspendIntent = Intent(this, ChimeService::class.java).apply { action = ACTION_TOGGLE_SUSPEND }
            val suspendPending = PendingIntent.getService(this, 3, suspendIntent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE)

            builder.setContentTitle(getString(R.string.status_active))
            builder.setContentText("Next chime at ${getNextChimeTime()}")
            builder.addAction(android.R.drawable.ic_media_pause, getString(R.string.action_suspend), suspendPending)
            builder.addAction(android.R.drawable.ic_media_next, "Skip Next Chime", skipPending)
        }

        builder.addAction(android.R.drawable.ic_menu_close_clear_cancel, "Disable", stopPendingIntent)

        return builder.build()
    }

    private fun stopChimeService() {
        val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)
        prefs.edit {
            putBoolean("service_enabled", false)
        }

        val alarmManager = getSystemService(ALARM_SERVICE) as AlarmManager
        val intent = Intent(this, ChimeReceiver::class.java)
        val pendingIntent = PendingIntent.getBroadcast(
            this, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        alarmManager.cancel(pendingIntent)

        stopForeground(STOP_FOREGROUND_REMOVE)
        stopSelf()
    }

    @RequiresApi(Build.VERSION_CODES.S)
    private fun skipNextChime() {
        val alarmManager = getSystemService(ALARM_SERVICE) as AlarmManager
        val intent = Intent(this, ChimeReceiver::class.java)
        val pendingIntent = PendingIntent.getBroadcast(
            this, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val calendar = Calendar.getInstance().apply {
            add(Calendar.HOUR_OF_DAY, 2)
            set(Calendar.MINUTE, 0)
            set(Calendar.SECOND, 0)
            set(Calendar.MILLISECOND, 0)
        }

        if (alarmManager.canScheduleExactAlarms()) {
            alarmManager.setExactAndAllowWhileIdle(
                AlarmManager.RTC_WAKEUP,
                calendar.timeInMillis,
                pendingIntent
            )
        }

        val channelId = "chime_channel"
        val notification = NotificationCompat.Builder(this, channelId)
            .setContentTitle("HRLY Active")
            .setContentText("Next chime at ${getNextChimeTime(2)}")
            .setSmallIcon(R.drawable.ic_stat_chime)
            .setPriority(NotificationCompat.PRIORITY_MIN)
            .addAction(android.R.drawable.ic_media_next, "Skip Next",
                PendingIntent.getService(this, 1, Intent(this, ChimeService::class.java).apply { action = ACTION_SKIP_NEXT }, PendingIntent.FLAG_IMMUTABLE))
            .addAction(android.R.drawable.ic_menu_close_clear_cancel, "Disable",
                PendingIntent.getService(this, 2, Intent(this, ChimeService::class.java).apply { action = ACTION_STOP_SERVICE }, PendingIntent.FLAG_IMMUTABLE))
            .build()

        val notificationManager = getSystemService(NotificationManager::class.java)
        notificationManager.notify(1, notification)
    }

    private fun isDndActive(): Boolean {
        val notificationManager = getSystemService(NOTIFICATION_SERVICE) as NotificationManager
        val currentFilter = notificationManager.currentInterruptionFilter

        return currentFilter != NotificationManager.INTERRUPTION_FILTER_ALL
    }

    private fun isQuietTime(): Boolean {
        val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)
        if (!prefs.getBoolean("quiet_enabled", false)) return false

        val startH = prefs.getInt("quiet_start_h", 22)
        val startM = prefs.getInt("quiet_start_m", 0)
        val endH = prefs.getInt("quiet_end_h", 7)
        val endM = prefs.getInt("quiet_end_m", 0)

        val now = Calendar.getInstance()
        val currentH = now.get(Calendar.HOUR_OF_DAY)
        val currentM = now.get(Calendar.MINUTE)

        val nowMinutes = currentH * 60 + currentM
        val startMinutes = startH * 60 + startM
        val endMinutes = endH * 60 + endM

        return if (startMinutes < endMinutes) {
            nowMinutes in startMinutes until endMinutes
        } else {
            nowMinutes >= startMinutes || nowMinutes < endMinutes
        }
    }

    @RequiresApi(Build.VERSION_CODES.S)
    private fun playSequenceForHour(hour24: Int) {
        serviceScope.launch {
            val hour12 = if (hour24 % 12 == 0) 12 else hour24 % 12
            val longTones = hour12 / 5
            val shortTones = hour12 % 5

            repeat(longTones) {
                playTone(R.raw.tone_long, longToneDuration, "long")
                delay(15)
            }
            repeat(shortTones) {
                playTone(R.raw.tone_short, shortToneDuration, "short")
                delay(15)
            }
        }
    }

    private fun showVisualPulse(duration: Long, forceShow: Boolean = false) {
        val prefs = getSharedPreferences("hourly_prefs", MODE_PRIVATE)
        if (!prefs.getBoolean("visual_enabled", false)) return
        if (!Settings.canDrawOverlays(this)) return

        val powerManager = getSystemService(POWER_SERVICE) as PowerManager
        val isScreenOn = powerManager.isInteractive

        if (isScreenOn && !forceShow) return

        val windowManager = getSystemService(WINDOW_SERVICE) as WindowManager

        // 1. Calculate REAL Screen Size (Modern vs Legacy)
        val screenWidth: Int
        val screenHeight: Int

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            // New Way (Android 11+)
            val metrics = windowManager.currentWindowMetrics
            val bounds = metrics.bounds
            screenWidth = bounds.width()
            screenHeight = bounds.height()
        } else {
            // Old Way (Android 10 and below)
            val displayMetrics = android.util.DisplayMetrics()
            @Suppress("DEPRECATION")
            windowManager.defaultDisplay.getRealMetrics(displayMetrics)
            screenWidth = displayMetrics.widthPixels
            screenHeight = displayMetrics.heightPixels
        }

        // 2. Window Params
        @Suppress("DEPRECATION")
        val layoutFlag = (
                WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE or
                        WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED or
                        WindowManager.LayoutParams.FLAG_TURN_SCREEN_ON or
                        WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON or
                        WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or
                        WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS
                )

        val params = WindowManager.LayoutParams(
            screenWidth,
            screenHeight + 200, // Extra height buffer for nav bars
            WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY,
            layoutFlag,
            PixelFormat.TRANSLUCENT
        )

        // 3. Container with Immersive Flags
        val container = android.widget.FrameLayout(this)
        container.setBackgroundColor(android.graphics.Color.BLACK)
        container.alpha = 0f

        // Hide System Bars (Status/Nav)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            container.windowInsetsController?.hide(android.view.WindowInsets.Type.systemBars())
        } else {
            @Suppress("DEPRECATION")
            container.systemUiVisibility = (
                    View.SYSTEM_UI_FLAG_LOW_PROFILE or
                            View.SYSTEM_UI_FLAG_FULLSCREEN or
                            View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                            View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY or
                            View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION or
                            View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                    )
        }

        // 4. Image Setup
        val sizeInDp = 85
        val scale = resources.displayMetrics.density
        val sizeInPx = (sizeInDp * scale + 0.5f).toInt()

        val imageView = ImageView(this)
        imageView.setImageResource(R.drawable.sphere_glow)
        val imageParams = android.widget.FrameLayout.LayoutParams(sizeInPx, sizeInPx).apply {
            gravity = Gravity.CENTER
        }
        container.addView(imageView, imageParams)

        // 5. Add to Window
        try {
            windowManager.addView(container, params)
        } catch (e: Exception) {
            e.printStackTrace()
            return
        }

        // 6. Animation
        val fadeInTime = 500L
        val holdTime = duration - (fadeInTime * 2)

        container.animate()
            .alpha(1f)
            .setDuration(fadeInTime)
            .withEndAction {
                imageView.animate().scaleX(1.1f).scaleY(1.1f).setDuration(holdTime).start()

                container.animate()
                    .alpha(0f)
                    .setDuration(fadeInTime)
                    .setStartDelay(holdTime)
                    .withEndAction {
                        try {
                            windowManager.removeView(container)
                        } catch (e: Exception) {
                            e.printStackTrace()
                        }
                    }
                    .start()
            }
            .start()
    }

    @RequiresApi(Build.VERSION_CODES.S)
    private fun scheduleNextChime() {
        val alarmManager = getSystemService(ALARM_SERVICE) as AlarmManager
        val intent = Intent(this, ChimeReceiver::class.java)
        val pendingIntent = PendingIntent.getBroadcast(
            this, 0, intent, PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )

        val calendar = Calendar.getInstance().apply {
            add(Calendar.HOUR_OF_DAY, 1)
            set(Calendar.MINUTE, 0)
            set(Calendar.SECOND, 0)
            set(Calendar.MILLISECOND, 0)
        }

        if (alarmManager.canScheduleExactAlarms()) {
            alarmManager.setExactAndAllowWhileIdle(
                AlarmManager.RTC_WAKEUP,
                calendar.timeInMillis,
                pendingIntent
            )
        } else {
            val notificationManager = getSystemService(NotificationManager::class.java)
            val errorNotification = NotificationCompat.Builder(this, "chime_channel")
                .setContentTitle("HRLY Error")
                .setContentText("Permission missing! Cannot schedule alarms.")
                .setSmallIcon(android.R.drawable.ic_dialog_alert)
                .setPriority(NotificationCompat.PRIORITY_HIGH)
                .build()
            notificationManager.notify(999, errorNotification)
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        serviceScope.cancel()
        if (overlayView != null) windowManager?.removeView(overlayView)
    }
}
