"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = utils;

var _assert = _interopRequireDefault(require("assert"));

var _helpers = require("./helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

/**
 * Graphology Utils Specs
 * =======================
 *
 * Testing the utils methods.
 */
var PROPERTIES = ['type', 'multi', 'map', 'selfLoops'];

function utils(Graph) {
  return {
    '#.nullCopy': {
      'it should create an null copy of the graph.': function itShouldCreateAnNullCopyOfTheGraph() {
        var graph = new Graph();
        (0, _helpers.addNodesFrom)(graph, ['John', 'Thomas']);
        graph.addEdge('John', 'Thomas');
        var copy = graph.nullCopy();

        _assert["default"].deepStrictEqual(copy.nodes(), []);

        _assert["default"].strictEqual(copy.order, 0);

        _assert["default"].strictEqual(copy.size, 0);

        PROPERTIES.forEach(function (property) {
          _assert["default"].strictEqual(graph[property], copy[property]);
        });
      },
      'it should be possible to pass options to merge.': function itShouldBePossibleToPassOptionsToMerge() {
        var graph = new Graph({
          type: 'directed'
        });
        var copy = graph.nullCopy({
          type: 'undirected'
        });

        _assert["default"].strictEqual(copy.type, 'undirected');

        _assert["default"]["throws"](function () {
          copy.addDirectedEdge('one', 'two');
        }, /addDirectedEdge/);
      }
    },
    '#.emptyCopy': {
      'it should create an empty copy of the graph.': function itShouldCreateAnEmptyCopyOfTheGraph() {
        var graph = new Graph();
        (0, _helpers.addNodesFrom)(graph, ['John', 'Thomas']);
        graph.addEdge('John', 'Thomas');
        var copy = graph.emptyCopy();

        _assert["default"].deepStrictEqual(copy.nodes(), ['John', 'Thomas']);

        _assert["default"].strictEqual(copy.order, 2);

        _assert["default"].strictEqual(copy.size, 0);

        PROPERTIES.forEach(function (property) {
          _assert["default"].strictEqual(graph[property], copy[property]);
        });
        copy.mergeEdge('Mary', 'Thomas');
        copy.setNodeAttribute('John', 'age', 32);

        _assert["default"].strictEqual(copy.order, 3);

        _assert["default"].strictEqual(copy.size, 1);

        _assert["default"].deepStrictEqual(copy.getNodeAttributes('John'), {
          age: 32
        });

        _assert["default"].deepStrictEqual(graph.getNodeAttributes('John'), {});
      },
      'it should be possible to pass options to merge.': function itShouldBePossibleToPassOptionsToMerge() {
        var graph = new Graph({
          type: 'directed'
        });
        var copy = graph.emptyCopy({
          type: 'undirected'
        });

        _assert["default"].strictEqual(copy.type, 'undirected');

        _assert["default"]["throws"](function () {
          copy.addDirectedEdge('one', 'two');
        }, /addDirectedEdge/);
      }
    },
    '#.copy': {
      'it should create a full copy of the graph.': function itShouldCreateAFullCopyOfTheGraph() {
        var graph = new Graph();
        (0, _helpers.addNodesFrom)(graph, ['John', 'Thomas']);
        graph.addEdge('John', 'Thomas');
        var copy = graph.copy();

        _assert["default"].deepStrictEqual(copy.nodes(), graph.nodes());

        _assert["default"].deepStrictEqual(copy.edges(), graph.edges());

        _assert["default"].strictEqual(copy.order, 2);

        _assert["default"].strictEqual(copy.size, 1);

        PROPERTIES.forEach(function (property) {
          _assert["default"].strictEqual(graph[property], graph[property]);
        });
      }
    },
    '#.upgradeToMixed': {
      'it should correctly upgrade the given graph to a mixed one.': function itShouldCorrectlyUpgradeTheGivenGraphToAMixedOne() {
        var graph = new Graph({
          type: 'directed'
        });
        (0, _helpers.addNodesFrom)(graph, [1, 2, 3]);
        graph.addEdgeWithKey('1->2', 1, 2);
        graph.upgradeToMixed();

        _assert["default"].strictEqual(graph.type, 'mixed');

        graph.addUndirectedEdge(2, 3);

        _assert["default"].strictEqual(graph.size, 2);

        _assert["default"].strictEqual(graph.degree(2), 2);
      }
    },
    '#.upgradeToMulti': {
      'it should correctly upgrade the given graph to a multi one.': function itShouldCorrectlyUpgradeTheGivenGraphToAMultiOne() {
        var graph = new Graph();
        (0, _helpers.addNodesFrom)(graph, [1, 2]);
        graph.addEdgeWithKey('dA', 1, 2);
        graph.addUndirectedEdgeWithKey('uA', 1, 2);
        graph.upgradeToMulti();

        _assert["default"].strictEqual(graph.multi, true);

        graph.addEdgeWithKey('dB', 1, 2);
        graph.addUndirectedEdgeWithKey('uB', 1, 2);

        _assert["default"].strictEqual(graph.size, 4);

        _assert["default"].deepStrictEqual(graph.edges(), ['dA', 'uA', 'dB', 'uB']);
      }
    }
  };
}