"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = neighborsIteration;

var _assert = _interopRequireDefault(require("assert"));

var _take = _interopRequireDefault(require("obliterator/take"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

var METHODS = ['neighbors', 'inNeighbors', 'outNeighbors', 'inboundNeighbors', 'outboundNeighbors', 'directedNeighbors', 'undirectedNeighbors'];

function neighborsIteration(Graph, checkers) {
  var invalid = checkers.invalid,
      notFound = checkers.notFound;
  var graph = new Graph({
    multi: true
  });
  (0, _helpers.addNodesFrom)(graph, ['John', 'Thomas', 'Martha', 'Roger', 'Catherine', 'Alone', 'Forever']);
  graph.replaceNodeAttributes('John', {
    age: 34
  });
  graph.replaceNodeAttributes('Martha', {
    age: 35
  });
  graph.addDirectedEdgeWithKey('J->T', 'John', 'Thomas');
  graph.addDirectedEdgeWithKey('J->M', 'John', 'Martha');
  graph.addDirectedEdgeWithKey('C->J', 'Catherine', 'John');
  graph.addUndirectedEdgeWithKey('M<->R', 'Martha', 'Roger');
  graph.addUndirectedEdgeWithKey('M<->J', 'Martha', 'John');
  graph.addUndirectedEdgeWithKey('J<->R', 'John', 'Roger');
  graph.addUndirectedEdgeWithKey('T<->M', 'Thomas', 'Martha');
  var TEST_DATA = {
    neighbors: {
      are: [['John', 'Martha', true], ['Martha', 'Catherine', false]],
      node: {
        key: 'John',
        neighbors: ['Catherine', 'Thomas', 'Martha', 'Roger']
      }
    },
    inNeighbors: {
      are: [['John', 'Martha', false], ['John', 'Roger', false], ['Martha', 'Catherine', false], ['Thomas', 'John', true]],
      node: {
        key: 'John',
        neighbors: ['Catherine']
      }
    },
    outNeighbors: {
      are: [['John', 'Martha', true], ['John', 'Roger', false], ['Martha', 'Catherine', false]],
      node: {
        key: 'John',
        neighbors: ['Thomas', 'Martha']
      }
    },
    inboundNeighbors: {
      are: [['John', 'Thomas', false], ['John', 'Roger', true], ['Martha', 'John', true]],
      node: {
        key: 'John',
        neighbors: ['Catherine', 'Martha', 'Roger']
      }
    },
    outboundNeighbors: {
      are: [['John', 'Thomas', true], ['John', 'Roger', true], ['Martha', 'John', true], ['John', 'Catherine', false]],
      node: {
        key: 'John',
        neighbors: ['Thomas', 'Martha', 'Roger']
      }
    },
    directedNeighbors: {
      are: [['John', 'Martha', true], ['John', 'Roger', false], ['Martha', 'Catherine', false]],
      node: {
        key: 'John',
        neighbors: ['Catherine', 'Thomas', 'Martha']
      }
    },
    undirectedNeighbors: {
      are: [['John', 'Martha', true], ['John', 'Roger', true], ['Martha', 'Catherine', false]],
      node: {
        key: 'John',
        neighbors: ['Martha', 'Roger']
      }
    }
  };

  function commonTests(name) {
    return _defineProperty({}, '#.' + name, {
      'it should throw if too many arguments are provided.': function itShouldThrowIfTooManyArgumentsAreProvided() {
        _assert["default"]["throws"](function () {
          graph[name](1, 2, 3);
        }, invalid());
      },
      'it should throw if too few arguments are provided.': function itShouldThrowIfTooFewArgumentsAreProvided() {
        _assert["default"]["throws"](function () {
          graph[name]();
        }, invalid());
      },
      'it should throw when the node is not found.': function itShouldThrowWhenTheNodeIsNotFound() {
        _assert["default"]["throws"](function () {
          graph[name]('Test');
        }, notFound());

        if (~name.indexOf('count')) return;

        _assert["default"]["throws"](function () {
          graph[name]('Test', 'SecondTest');
        }, notFound());

        _assert["default"]["throws"](function () {
          graph[name]('Forever', 'Test');
        }, notFound());
      }
    });
  }

  function specificTests(name, data) {
    var _ref3;

    var forEachName = 'forEach' + name[0].toUpperCase() + name.slice(1, -1),
        forEachUntilName = forEachName + 'Until',
        iteratorName = name.slice(0, -1) + 'Entries';
    return _ref3 = {}, _defineProperty(_ref3, '#.' + name, {
      'it should correctly return whether two nodes are neighbors.': function itShouldCorrectlyReturnWhetherTwoNodesAreNeighbors() {
        data.are.forEach(function (_ref2) {
          var node1 = _ref2[0],
              node2 = _ref2[1],
              expectation = _ref2[2];

          _assert["default"].strictEqual(graph[name](node1, node2), expectation, "".concat(name, ": ").concat(node1, " / ").concat(node2));

          _assert["default"].strictEqual(graph[name]('Forever', 'Alone'), false);
        });
      },
      'it should return the correct neighbors array.': function itShouldReturnTheCorrectNeighborsArray() {
        var neighbors = graph[name](data.node.key);

        _assert["default"].deepStrictEqual(neighbors, data.node.neighbors);

        _assert["default"].deepStrictEqual(graph[name]('Alone'), []);
      }
    }), _defineProperty(_ref3, '#.' + forEachName, {
      'it should be possible to iterate over neighbors using a callback.': function itShouldBePossibleToIterateOverNeighborsUsingACallback() {
        var neighbors = [];
        graph[forEachName](data.node.key, function (target, attrs) {
          neighbors.push(target);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), attrs);

          _assert["default"].strictEqual(graph[name](data.node.key, target), true);
        });

        _assert["default"].deepStrictEqual(neighbors, data.node.neighbors);
      }
    }), _defineProperty(_ref3, '#.' + forEachUntilName, {
      'it should be possible to iterate over neighbors using a breakable callback.': function itShouldBePossibleToIterateOverNeighborsUsingABreakableCallback() {
        var neighbors = [];
        graph[forEachUntilName](data.node.key, function (target, attrs) {
          neighbors.push(target);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), attrs);

          _assert["default"].strictEqual(graph[name](data.node.key, target), true);

          return true;
        });

        _assert["default"].deepStrictEqual(neighbors, data.node.neighbors.slice(0, 1));
      }
    }), _defineProperty(_ref3, '#.' + iteratorName, {
      'it should be possible to create an iterator over neighbors.': function itShouldBePossibleToCreateAnIteratorOverNeighbors() {
        var iterator = graph[iteratorName](data.node.key);

        _assert["default"].deepStrictEqual((0, _take["default"])(iterator), data.node.neighbors.map(function (neighbor) {
          return [neighbor, graph.getNodeAttributes(neighbor)];
        }));
      }
    }), _ref3;
  }

  var tests = {
    'Miscellaneous': {
      'self loops should appear when using #.inNeighbors and should appear only once with #.neighbors.': function selfLoopsShouldAppearWhenUsingInNeighborsAndShouldAppearOnlyOnceWithNeighbors() {
        var directed = new Graph({
          type: 'directed'
        });
        directed.addNode('Lucy');
        directed.addEdgeWithKey('test', 'Lucy', 'Lucy');

        _assert["default"].deepStrictEqual(directed.inNeighbors('Lucy'), ['Lucy']);

        _assert["default"].deepStrictEqual(Array.from(directed.inNeighborEntries('Lucy')).map(function (x) {
          return x[0];
        }), ['Lucy']);

        var neighbors = [];
        directed.forEachInNeighbor('Lucy', function (neighbor) {
          neighbors.push(neighbor);
        });

        _assert["default"].deepStrictEqual(neighbors, ['Lucy']);

        _assert["default"].deepStrictEqual(directed.neighbors('Lucy'), ['Lucy']);

        neighbors = [];
        directed.forEachNeighbor('Lucy', function (neighbor) {
          neighbors.push(neighbor);
        });

        _assert["default"].deepStrictEqual(neighbors, ['Lucy']);

        _assert["default"].deepStrictEqual(Array.from(directed.neighborEntries('Lucy')).map(function (x) {
          return x[0];
        }), ['Lucy']);
      }
    }
  }; // Common tests

  METHODS.forEach(function (name) {
    return (0, _helpers.deepMerge)(tests, commonTests(name));
  }); // Specific tests

  for (var name in TEST_DATA) {
    (0, _helpers.deepMerge)(tests, specificTests(name, TEST_DATA[name]));
  }

  return tests;
}