"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = iteration;

var _assert = _interopRequireDefault(require("assert"));

var _take = _interopRequireDefault(require("obliterator/take"));

var _nodes = _interopRequireDefault(require("./nodes"));

var _edges = _interopRequireDefault(require("./edges"));

var _neighbors = _interopRequireDefault(require("./neighbors"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

/**
 * Graphology Iteration Specs
 * ===========================
 *
 * Testing the iteration-related methods of the graph.
 */
function iteration(Graph, checkers) {
  return {
    Adjacency: {
      'it should be possible to iterate over the graph\'s adjacency using callbacks.': function itShouldBePossibleToIterateOverTheGraphSAdjacencyUsingCallbacks() {
        var graph = new Graph();
        graph.addNode(1);
        graph.addNode(2);
        graph.addNode(3);
        graph.addEdge(1, 2);
        graph.addEdge(2, 3);
        graph.addEdge(3, 1);
        graph.addUndirectedEdge(1, 2);
        graph.replaceNodeAttributes(2, {
          hello: 'world'
        });
        var adjacency = [];
        graph.forEach(function (s, t, sa, ta, e, ea, u, g) {
          adjacency.push([u, s, t]);

          _assert["default"].deepStrictEqual(sa, graph.getNodeAttributes(s));

          _assert["default"].deepStrictEqual(ta, graph.getNodeAttributes(t));

          _assert["default"].deepStrictEqual(ea, graph.getEdgeAttributes(e));

          _assert["default"].strictEqual(graph.isUndirected(e), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(e), g);
        });

        _assert["default"].deepStrictEqual(adjacency, [[false, '1', '2'], [true, '1', '2'], [false, '2', '3'], [true, '2', '1'], [false, '3', '1']]);
      },
      'it should be possible to iterate over a multi graph\'s adjacency using callbacks.': function itShouldBePossibleToIterateOverAMultiGraphSAdjacencyUsingCallbacks() {
        var graph = new Graph({
          multi: true
        });
        graph.addNode(1);
        graph.addNode(2);
        graph.addNode(3);
        graph.addEdge(1, 2);
        graph.addEdge(2, 3);
        graph.addEdge(3, 1);
        graph.addEdgeWithKey('test', 2, 3);
        graph.addUndirectedEdge(1, 2);
        graph.replaceNodeAttributes(2, {
          hello: 'world'
        });
        var adjacency = [];
        graph.forEach(function (s, t, sa, ta, e, ea, u, g) {
          adjacency.push([u, s, t]);

          _assert["default"].deepStrictEqual(sa, graph.getNodeAttributes(s));

          _assert["default"].deepStrictEqual(ta, graph.getNodeAttributes(t));

          _assert["default"].deepStrictEqual(ea, graph.getEdgeAttributes(e));

          _assert["default"].strictEqual(graph.isUndirected(e), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(e), g);

          if (!g) _assert["default"].strictEqual(e, 'test');
        });

        _assert["default"].deepStrictEqual(adjacency, [[false, '1', '2'], [true, '1', '2'], [false, '2', '3'], [false, '2', '3'], [true, '2', '1'], [false, '3', '1']]);
      },
      'it should be possible to iterate over the graph\'s adjacency using callbacks until returning true.': function itShouldBePossibleToIterateOverTheGraphSAdjacencyUsingCallbacksUntilReturningTrue() {
        var graph = new Graph();
        graph.addNode(1);
        graph.addNode(2);
        graph.addNode(3);
        graph.addEdge(1, 2);
        graph.addEdge(2, 3);
        graph.addEdge(3, 1);
        graph.addUndirectedEdge(1, 2);
        graph.replaceNodeAttributes(2, {
          hello: 'world'
        });
        var adjacency = [];
        graph.forEachUntil(function (s, t, sa, ta, e, ea, u, g) {
          adjacency.push([u, s, t]);

          _assert["default"].deepStrictEqual(sa, graph.getNodeAttributes(s));

          _assert["default"].deepStrictEqual(ta, graph.getNodeAttributes(t));

          _assert["default"].deepStrictEqual(ea, graph.getEdgeAttributes(e));

          _assert["default"].strictEqual(graph.isUndirected(e), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(e), g);

          if (sa.hello === 'world') return true;
        });

        _assert["default"].deepStrictEqual(adjacency, [[false, '1', '2'], [true, '1', '2'], [false, '2', '3']]);
      },
      'it should be possible to create an iterator over the graph\'s adjacency.': function itShouldBePossibleToCreateAnIteratorOverTheGraphSAdjacency() {
        var edgeKeyGenerator = function edgeKeyGenerator(_ref) {
          var undirected = _ref.undirected,
              source = _ref.source,
              target = _ref.target;
          return "".concat(source).concat(undirected ? '--' : '->').concat(target);
        };

        var graph = new Graph({
          edgeKeyGenerator: edgeKeyGenerator
        });
        graph.addNode(1);
        graph.addNode(2);
        graph.addNode(3);
        graph.addEdge(1, 2);
        graph.addEdge(2, 3);
        graph.addEdge(3, 1);
        graph.addUndirectedEdge(1, 2);
        graph.replaceNodeAttributes(2, {
          hello: 'world'
        });
        var adj = (0, _take["default"])(graph.adjacency()).map(function (p) {
          return [p[0], p[1], p[4]];
        });

        _assert["default"].deepStrictEqual(adj, [['1', '2', '1->2'], ['1', '2', '1--2'], ['2', '3', '2->3'], ['2', '1', '1--2'], ['3', '1', '3->1']]);
      },
      'it should be possible to create an iterator over a multi graph\'s adjacency.': function itShouldBePossibleToCreateAnIteratorOverAMultiGraphSAdjacency() {
        var graph = new Graph({
          multi: true
        });
        graph.addNode(1);
        graph.addNode(2);
        graph.addNode(3);
        graph.addEdgeWithKey(0, 1, 2);
        graph.addEdgeWithKey(1, 2, 3);
        graph.addEdgeWithKey(2, 3, 1);
        graph.addEdgeWithKey(3, 2, 3);
        graph.addUndirectedEdgeWithKey(4, 1, 2);
        graph.replaceNodeAttributes(2, {
          hello: 'world'
        });
        var adj = (0, _take["default"])(graph.adjacency()).map(function (p) {
          return [p[0], p[1], p[4]];
        });

        _assert["default"].deepStrictEqual(adj, [['1', '2', '0'], ['1', '2', '4'], ['2', '3', '1'], ['2', '3', '3'], ['2', '1', '4'], ['3', '1', '2']]);
      },
      'it should be possible to iterate via Symbol.iterator.': function itShouldBePossibleToIterateViaSymbolIterator() {
        if (typeof Symbol === 'undefined') return;

        var edgeKeyGenerator = function edgeKeyGenerator(_ref2) {
          var undirected = _ref2.undirected,
              source = _ref2.source,
              target = _ref2.target;
          return "".concat(source).concat(undirected ? '--' : '->').concat(target);
        };

        var graph = new Graph({
          edgeKeyGenerator: edgeKeyGenerator
        });
        graph.addNode(1);
        graph.addNode(2);
        graph.addNode(3);
        graph.addEdge(1, 2);
        graph.addEdge(2, 3);
        graph.addEdge(3, 1);
        graph.addUndirectedEdge(1, 2);
        graph.replaceNodeAttributes(2, {
          hello: 'world'
        });
        var adj = (0, _take["default"])(graph[Symbol.iterator]()).map(function (p) {
          return [p[0], p[1], p[4]];
        });

        _assert["default"].deepStrictEqual(adj, [['1', '2', '1->2'], ['1', '2', '1--2'], ['2', '3', '2->3'], ['2', '1', '1--2'], ['3', '1', '3->1']]);
      }
    },
    Nodes: (0, _nodes["default"])(Graph, checkers),
    Edges: (0, _edges["default"])(Graph, checkers),
    Neighbors: (0, _neighbors["default"])(Graph, checkers)
  };
}