"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = edgesIteration;

var _assert = _interopRequireDefault(require("assert"));

var _take = _interopRequireDefault(require("obliterator/take"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

var METHODS = ['edges', 'inEdges', 'outEdges', 'inboundEdges', 'outboundEdges', 'directedEdges', 'undirectedEdges'];

function edgesIteration(Graph, checkers) {
  var invalid = checkers.invalid,
      notFound = checkers.notFound;
  var graph = new Graph({
    multi: true
  });
  (0, _helpers.addNodesFrom)(graph, ['John', 'Thomas', 'Martha', 'Roger', 'Catherine', 'Alone', 'Forever']);
  graph.replaceNodeAttributes('John', {
    age: 13
  });
  graph.replaceNodeAttributes('Martha', {
    age: 15
  });
  graph.addDirectedEdgeWithKey('J->T', 'John', 'Thomas', {
    weight: 14
  });
  graph.addDirectedEdgeWithKey('J->M', 'John', 'Martha');
  graph.addDirectedEdgeWithKey('C->J', 'Catherine', 'John');
  graph.addUndirectedEdgeWithKey('M<->R', 'Martha', 'Roger');
  graph.addUndirectedEdgeWithKey('M<->J', 'Martha', 'John');
  graph.addUndirectedEdgeWithKey('J<->R', 'John', 'Roger');
  graph.addUndirectedEdgeWithKey('T<->M', 'Thomas', 'Martha');
  var ALL_EDGES = ['J->T', 'J->M', 'C->J', 'M<->R', 'M<->J', 'J<->R', 'T<->M'];
  var ALL_DIRECTED_EDGES = ['J->T', 'J->M', 'C->J'];
  var ALL_UNDIRECTED_EDGES = ['M<->R', 'M<->J', 'J<->R', 'T<->M'];
  var TEST_DATA = {
    edges: {
      all: ALL_EDGES,
      node: {
        key: 'John',
        edges: ['C->J', 'J->T', 'J->M', 'M<->J', 'J<->R']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: ['J->M', 'M<->J']
      }
    },
    inEdges: {
      all: ALL_DIRECTED_EDGES,
      node: {
        key: 'John',
        edges: ['C->J']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: []
      }
    },
    outEdges: {
      all: ALL_DIRECTED_EDGES,
      node: {
        key: 'John',
        edges: ['J->T', 'J->M']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: ['J->M']
      }
    },
    inboundEdges: {
      all: ALL_DIRECTED_EDGES.concat(ALL_UNDIRECTED_EDGES),
      node: {
        key: 'John',
        edges: ['C->J', 'M<->J', 'J<->R']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: ['M<->J']
      }
    },
    outboundEdges: {
      all: ALL_DIRECTED_EDGES.concat(ALL_UNDIRECTED_EDGES),
      node: {
        key: 'John',
        edges: ['J->T', 'J->M', 'M<->J', 'J<->R']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: ['J->M', 'M<->J']
      }
    },
    directedEdges: {
      all: ALL_DIRECTED_EDGES,
      node: {
        key: 'John',
        edges: ['C->J', 'J->T', 'J->M']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: ['J->M']
      }
    },
    undirectedEdges: {
      all: ALL_UNDIRECTED_EDGES,
      node: {
        key: 'John',
        edges: ['M<->J', 'J<->R']
      },
      path: {
        source: 'John',
        target: 'Martha',
        edges: ['M<->J']
      }
    }
  };

  function commonTests(name) {
    return _defineProperty({}, '#.' + name, {
      'it should throw if too many arguments are provided.': function itShouldThrowIfTooManyArgumentsAreProvided() {
        _assert["default"]["throws"](function () {
          graph[name](1, 2, 3);
        }, invalid());
      },
      'it should throw when the node is not found.': function itShouldThrowWhenTheNodeIsNotFound() {
        _assert["default"]["throws"](function () {
          graph[name]('Test');
        }, notFound());
      },
      'it should throw if either source or target is not found.': function itShouldThrowIfEitherSourceOrTargetIsNotFound() {
        _assert["default"]["throws"](function () {
          graph[name]('Test', 'Alone');
        }, notFound());

        _assert["default"]["throws"](function () {
          graph[name]('Alone', 'Test');
        }, notFound());
      }
    });
  }

  function specificTests(name, data) {
    var _ref2;

    var iteratorName = name.slice(0, -1) + 'Entries',
        forEachName = 'forEach' + name[0].toUpperCase() + name.slice(1, -1),
        forEachUntilName = forEachName + 'Until';
    return _ref2 = {}, _defineProperty(_ref2, '#.' + name, {
      'it should return all the relevant edges.': function itShouldReturnAllTheRelevantEdges() {
        var edges = graph[name]().sort();

        _assert["default"].deepStrictEqual(edges, data.all.slice().sort());
      },
      'it should return a node\'s relevant edges.': function itShouldReturnANodeSRelevantEdges() {
        var edges = graph[name](data.node.key);

        _assert["default"].deepStrictEqual(edges, data.node.edges);

        _assert["default"].deepStrictEqual(graph[name]('Alone'), []);
      },
      'it should return all the relevant edges between source & target.': function itShouldReturnAllTheRelevantEdgesBetweenSourceTarget() {
        var edges = graph[name](data.path.source, data.path.target);
        (0, _assert["default"])((0, _helpers.sameMembers)(edges, data.path.edges));

        _assert["default"].deepStrictEqual(graph[name]('Forever', 'Alone'), []);
      }
    }), _defineProperty(_ref2, '#.' + forEachName, {
      'it should possible to use callback iterators.': function itShouldPossibleToUseCallbackIterators() {
        var edges = [];
        graph[forEachName](function (key, attributes, source, target, sA, tA, u, g) {
          edges.push(key);

          _assert["default"].deepStrictEqual(attributes, key === 'J->T' ? {
            weight: 14
          } : {});

          _assert["default"].strictEqual(source, graph.source(key));

          _assert["default"].strictEqual(target, graph.target(key));

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(source), sA);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), tA);

          _assert["default"].strictEqual(graph.isUndirected(key), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(key), g);
        });
        edges.sort();

        _assert["default"].deepStrictEqual(edges, data.all.slice().sort());
      },
      'it should be possible to use callback iterators over a node\'s relevant edges.': function itShouldBePossibleToUseCallbackIteratorsOverANodeSRelevantEdges() {
        var edges = [];
        graph[forEachName](data.node.key, function (key, attributes, source, target, sA, tA, u, g) {
          edges.push(key);

          _assert["default"].deepStrictEqual(attributes, key === 'J->T' ? {
            weight: 14
          } : {});

          _assert["default"].strictEqual(source, graph.source(key));

          _assert["default"].strictEqual(target, graph.target(key));

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(source), sA);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), tA);

          _assert["default"].strictEqual(graph.isUndirected(key), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(key), g);
        });
        edges.sort();

        _assert["default"].deepStrictEqual(edges, data.node.edges.slice().sort());
      },
      'it should be possible to use callback iterators over all the relevant edges between source & target.': function itShouldBePossibleToUseCallbackIteratorsOverAllTheRelevantEdgesBetweenSourceTarget() {
        var edges = [];
        graph[forEachName](data.path.source, data.path.target, function (key, attributes, source, target, sA, tA, u, g) {
          edges.push(key);

          _assert["default"].deepStrictEqual(attributes, key === 'J->T' ? {
            weight: 14
          } : {});

          _assert["default"].strictEqual(source, graph.source(key));

          _assert["default"].strictEqual(target, graph.target(key));

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(source), sA);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), tA);

          _assert["default"].strictEqual(graph.isUndirected(key), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(key), g);
        });
        (0, _assert["default"])((0, _helpers.sameMembers)(edges, data.path.edges));
      }
    }), _defineProperty(_ref2, '#.' + forEachUntilName, {
      'it should possible to use breakable callback iterators.': function itShouldPossibleToUseBreakableCallbackIterators() {
        var edges = [];
        graph[forEachUntilName](function (key, attributes, source, target, sA, tA, u, g) {
          edges.push(key);

          _assert["default"].deepStrictEqual(attributes, key === 'J->T' ? {
            weight: 14
          } : {});

          _assert["default"].strictEqual(source, graph.source(key));

          _assert["default"].strictEqual(target, graph.target(key));

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(source), sA);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), tA);

          _assert["default"].strictEqual(graph.isUndirected(key), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(key), g);

          return true;
        });

        _assert["default"].strictEqual(edges.length, 1);
      },
      'it should be possible to use breakable callback iterators over a node\'s relevant edges.': function itShouldBePossibleToUseBreakableCallbackIteratorsOverANodeSRelevantEdges() {
        var edges = [];
        graph[forEachUntilName](data.node.key, function (key, attributes, source, target, sA, tA, u, g) {
          edges.push(key);

          _assert["default"].deepStrictEqual(attributes, key === 'J->T' ? {
            weight: 14
          } : {});

          _assert["default"].strictEqual(source, graph.source(key));

          _assert["default"].strictEqual(target, graph.target(key));

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(source), sA);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), tA);

          _assert["default"].strictEqual(graph.isUndirected(key), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(key), g);

          return true;
        });

        _assert["default"].strictEqual(edges.length, 1);
      },
      'it should be possible to use breakable callback iterators over all the relevant edges between source & target.': function itShouldBePossibleToUseBreakableCallbackIteratorsOverAllTheRelevantEdgesBetweenSourceTarget() {
        var edges = [];
        graph[forEachUntilName](data.path.source, data.path.target, function (key, attributes, source, target, sA, tA, u, g) {
          edges.push(key);

          _assert["default"].deepStrictEqual(attributes, key === 'J->T' ? {
            weight: 14
          } : {});

          _assert["default"].strictEqual(source, graph.source(key));

          _assert["default"].strictEqual(target, graph.target(key));

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(source), sA);

          _assert["default"].deepStrictEqual(graph.getNodeAttributes(target), tA);

          _assert["default"].strictEqual(graph.isUndirected(key), u);

          _assert["default"].strictEqual(graph.hasGeneratedKey(key), g);

          return true;
        });

        _assert["default"].strictEqual(edges.length, graph[name](data.path.source, data.path.target).length ? 1 : 0);
      }
    }), _defineProperty(_ref2, '#.' + iteratorName, {
      'it should be possible to return an iterator over the relevant edges.': function itShouldBePossibleToReturnAnIteratorOverTheRelevantEdges() {
        var iterator = graph[iteratorName]();

        _assert["default"].deepStrictEqual((0, _take["default"])(iterator), data.all.map(function (edge) {
          var _graph$extremities = graph.extremities(edge),
              source = _graph$extremities[0],
              target = _graph$extremities[1];

          return [edge, graph.getEdgeAttributes(edge), source, target, graph.getNodeAttributes(source), graph.getNodeAttributes(target)];
        }));
      },
      'it should be possible to return an iterator over a node\'s relevant edges.': function itShouldBePossibleToReturnAnIteratorOverANodeSRelevantEdges() {
        var iterator = graph[iteratorName](data.node.key);

        _assert["default"].deepStrictEqual((0, _take["default"])(iterator), data.node.edges.map(function (edge) {
          var _graph$extremities2 = graph.extremities(edge),
              source = _graph$extremities2[0],
              target = _graph$extremities2[1];

          return [edge, graph.getEdgeAttributes(edge), source, target, graph.getNodeAttributes(source), graph.getNodeAttributes(target)];
        }));
      },
      'it should be possible to return an iterator over relevant edges between source & target.': function itShouldBePossibleToReturnAnIteratorOverRelevantEdgesBetweenSourceTarget() {
        var iterator = graph[iteratorName](data.path.source, data.path.target);

        _assert["default"].deepStrictEqual((0, _take["default"])(iterator), data.path.edges.map(function (edge) {
          var _graph$extremities3 = graph.extremities(edge),
              source = _graph$extremities3[0],
              target = _graph$extremities3[1];

          return [edge, graph.getEdgeAttributes(edge), source, target, graph.getNodeAttributes(source), graph.getNodeAttributes(target)];
        }));
      }
    }), _ref2;
  }

  var tests = {
    'Miscellaneous': {
      'simple graph indices should work.': function simpleGraphIndicesShouldWork() {
        var simpleGraph = new Graph();
        (0, _helpers.addNodesFrom)(simpleGraph, [1, 2, 3, 4]);
        simpleGraph.addEdgeWithKey('1->2', 1, 2);
        simpleGraph.addEdgeWithKey('1->3', 1, 3);
        simpleGraph.addEdgeWithKey('1->4', 1, 4);

        _assert["default"].deepStrictEqual(simpleGraph.edges(1), ['1->2', '1->3', '1->4']);
      },
      'it should also work with typed graphs.': function itShouldAlsoWorkWithTypedGraphs() {
        var undirected = new Graph({
          type: 'undirected'
        }),
            directed = new Graph({
          type: 'directed'
        });
        undirected.mergeEdgeWithKey('1--2', 1, 2);
        directed.mergeEdgeWithKey('1->2', 1, 2);

        _assert["default"].deepStrictEqual(undirected.edges(1, 2), ['1--2']);

        _assert["default"].deepStrictEqual(directed.edges(1, 2), ['1->2']);
      },
      'self loops should appear when using #.inEdges and should appear only once with #.edges.': function selfLoopsShouldAppearWhenUsingInEdgesAndShouldAppearOnlyOnceWithEdges() {
        var directed = new Graph({
          type: 'directed'
        });
        directed.addNode('Lucy');
        directed.addEdgeWithKey('Lucy', 'Lucy', 'Lucy');

        _assert["default"].deepStrictEqual(directed.inEdges('Lucy'), ['Lucy']);

        _assert["default"].deepStrictEqual(Array.from(directed.inEdgeEntries('Lucy')).map(function (x) {
          return x[0];
        }), ['Lucy']);

        var edges = [];
        directed.forEachInEdge('Lucy', function (edge) {
          edges.push(edge);
        });

        _assert["default"].deepStrictEqual(edges, ['Lucy']);

        _assert["default"].deepStrictEqual(directed.edges('Lucy'), ['Lucy']);

        edges = [];
        directed.forEachEdge('Lucy', function (edge) {
          edges.push(edge);
        });

        _assert["default"].deepStrictEqual(edges, ['Lucy']);

        _assert["default"].deepStrictEqual(Array.from(directed.edgeEntries('Lucy')).map(function (x) {
          return x[0];
        }), ['Lucy']);
      },
      'it should be possible to retrieve self loops.': function itShouldBePossibleToRetrieveSelfLoops() {
        var loopy = new Graph();
        loopy.addNode('John');
        loopy.addEdgeWithKey('d', 'John', 'John');
        loopy.addUndirectedEdgeWithKey('u', 'John', 'John');

        _assert["default"].deepStrictEqual(new Set(loopy.edges('John', 'John')), new Set(['d', 'u']));

        _assert["default"].deepStrictEqual(loopy.directedEdges('John', 'John'), ['d']);

        _assert["default"].deepStrictEqual(loopy.undirectedEdges('John', 'John'), ['u']);

        var edges = [];
        loopy.forEachDirectedEdge('John', 'John', function (edge) {
          edges.push(edge);
        });

        _assert["default"].deepStrictEqual(edges, ['d']);

        edges = [];
        loopy.forEachUndirectedEdge('John', 'John', function (edge) {
          edges.push(edge);
        });

        _assert["default"].deepStrictEqual(edges, ['u']);
      }
    }
  }; // Common tests

  METHODS.forEach(function (name) {
    return (0, _helpers.deepMerge)(tests, commonTests(name));
  }); // Specific tests

  for (var name in TEST_DATA) {
    (0, _helpers.deepMerge)(tests, specificTests(name, TEST_DATA[name]));
  }

  return tests;
}