/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.widget.Button
import android.widget.TextView
import android.widget.Toast
import androidx.appcompat.app.AlertDialog

object TutorialManager {

    private const val PREF_FIRST_RUN = "first_run_tutorial"

    fun showDialogIfFirstRun(context: Context) {
        val prefs = context.getSharedPreferences("settings", Context.MODE_PRIVATE)
        val isFirstRun = prefs.getBoolean(PREF_FIRST_RUN, true)

        if (isFirstRun) {
            showTutorialDialog(context)
            // Mark as seen so it doesn't show automatically again
            prefs.edit().putBoolean(PREF_FIRST_RUN, false).apply()
        }
    }

    fun showTutorialDialog(context: Context) {
        val dialogView = android.view.LayoutInflater.from(context).inflate(R.layout.dialog_tutorial_popup, null)
        val tvLink = dialogView.findViewById<TextView>(R.id.tvTutorialLink)
        val btnClose = dialogView.findViewById<Button>(R.id.btnCloseTutorial)

        // Read the link from the raw file
        val tutorialLink = try {
            context.resources.openRawResource(R.raw.tutorial_link)
                .bufferedReader()
                .use { it.readText().trim() }
        } catch (e: Exception) {
            "https://www.youtube.com" // Fallback
        }

        tvLink.text = tutorialLink

        // 1. Single Click -> Open URL
        tvLink.setOnClickListener {
            try {
                val intent = Intent(Intent.ACTION_VIEW, Uri.parse(tutorialLink))
                context.startActivity(intent)
            } catch (e: Exception) {
                Toast.makeText(context, "Could not open link", Toast.LENGTH_SHORT).show()
            }
        }

        // 2. Long Click -> Copy to Clipboard
        tvLink.setOnLongClickListener {
            val clipboard = context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
            val clip = ClipData.newPlainText("Tutorial Link", tutorialLink)
            clipboard.setPrimaryClip(clip)
            Toast.makeText(context, "Link copied to clipboard", Toast.LENGTH_SHORT).show()
            true // Return true to indicate the event was handled
        }

        val dialog = AlertDialog.Builder(context)
            .setView(dialogView)
            .setCancelable(false) // User must press Close
            .create()

        // Make background transparent for rounded corners effect if you have custom drawables,
        // otherwise default is fine.

        btnClose.setOnClickListener {
            dialog.dismiss()
        }

        dialog.show()
    }
}