/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.content.Context
import java.time.LocalDate

/**
 * Manages all persistent user statistics and metrics.
 */
class StatsManager(context: Context) {
    private val prefs = context.getSharedPreferences("stats", Context.MODE_PRIVATE)

    var totalAnswers: Int = 0
        private set

    var learnedCount: Int = 0
        private set

    var streakCount: Int = 0
        private set


    var totalScore: Long = 0
        private set
    private var lastSessionDate: Long = 0L

    private var firstLaunchTime: Long = 0L
    private var totalActiveMillis: Long = 0L

    /** Loads all stored statistics from SharedPreferences upon initialization. */
    init {
        totalAnswers = prefs.getInt("totalAnswers", 0)
        learnedCount = prefs.getInt("learnedCount", 0)
        firstLaunchTime = prefs.getLong("firstLaunchTime", 0L)
        totalActiveMillis = prefs.getLong("totalActiveMillis", 0L)
        streakCount = prefs.getInt("streakCount", 0)
        totalScore = prefs.getLong("totalScore", 0L)
        lastSessionDate = prefs.getLong("lastSessionDate", 0L)

        // Set first launch time if not already set
        if (firstLaunchTime == 0L) {
            firstLaunchTime = System.currentTimeMillis()
        }
    }

    /** Saves all current in-memory statistics to SharedPreferences. */
    fun save() {
        prefs.edit().apply {
            putInt("totalAnswers", totalAnswers)
            putInt("learnedCount", learnedCount)
            putLong("firstLaunchTime", firstLaunchTime)
            putLong("totalActiveMillis", totalActiveMillis)

            // Save calculated derived stats
            putFloat("cardsPerMinute", getCardsPerMinute())
            putFloat("cardsPerDay", getCardsPerDay())

            // Save streak properties
            putInt("streakCount", streakCount)
            putLong("totalScore", totalScore)
            putLong("lastSessionDate", lastSessionDate)

            apply()
        }
    }

    /**
     * Checks if the streak is broken (a day was missed). If so, resets it to 0.
     * This does NOT increment the streak or count as a practice session.
     */
    fun checkAndResetStreak() {
        val today = LocalDate.now()
        val lastDate = if (lastSessionDate == 0L) {
            null
        } else {
            LocalDate.ofEpochDay(lastSessionDate)
        }

        if (lastDate != null && !lastDate.isEqual(today) && !lastDate.isEqual(today.minusDays(1))) {
            // A day was missed. Reset the streak.
            streakCount = 0
        }
    }

    /**
     * Records a practice session for today.
     * This will increment the streak if the last session was yesterday, or start a new streak.
     */
    fun recordPracticeSession() {
        val today = LocalDate.now()
        val lastDate = if (lastSessionDate == 0L) {
            null // Never played before
        } else {
            LocalDate.ofEpochDay(lastSessionDate)
        }

        if (lastDate == null) {
            // First session ever
            streakCount = 1
        } else if (lastDate.isEqual(today)) {
            // Already played today, streak is maintained but not incremented.
            return
        } else if (lastDate.isEqual(today.minusDays(1))) {
            // Played yesterday, increment streak.
            streakCount++
        } else {
            // Missed a day, reset streak to 1 (for today's new session).
            streakCount = 1
        }

        // Update the last session date to today's date
        lastSessionDate = today.toEpochDay()
    }


    /** Increments the count of total words learned. */
    fun logWordMastered() {
        learnedCount++
    }

    /** Increments the total number of answers given. */
    fun logAnswer() {
        totalAnswers++
    }

    /** Adds the given time to the total active practice time. */
    fun addActiveTime(milliseconds: Long) {
        totalActiveMillis += milliseconds
    }

    /** Calculates the average number of cards answered per minute of active time. */
    fun getCardsPerMinute(): Float {
        val totalMinutes = totalActiveMillis / 60000f
        return if (totalMinutes > 0) totalAnswers / totalMinutes else 0f
    }

    /** Calculates the average number of cards answered per day since first launch. */
    fun getCardsPerDay(): Float {
        val totalAnswers = this.totalAnswers
        val firstLaunch = this.firstLaunchTime

        val millisPassed = System.currentTimeMillis() - firstLaunch
        val millisInADay = 1000 * 60 * 60 * 24L

        // Return 0f if less than a full day has passed
        if (millisPassed < millisInADay) {
            return 0f
        }

        // Calculate average
        val days = (millisPassed / millisInADay.toFloat()).coerceAtLeast(1f)
        return totalAnswers / days
    }

    fun addScoreBasedOnStreak(currentStreak: Int) {
        val pointsToAdd = when {
            currentStreak >= 20 -> 40
            currentStreak >= 10 -> 20
            currentStreak >= 5 -> 10
            else -> 5
        }
        totalScore += pointsToAdd
    }
}