/*
 * La Et Le — French Gender Classification Trainer
 *
 * Copyright (C) 2025 Seweryn Polec <sewerynpol@protonmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package com.languageapp.laetle

import android.os.Bundle
import android.text.Html
import android.util.Log
import android.widget.ImageButton
import android.widget.TextView
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import com.google.android.material.progressindicator.LinearProgressIndicator
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import java.util.Locale
import androidx.activity.enableEdgeToEdge
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat


/**
 * Activity for displaying user learning and activity statistics.
 */
class StatisticsActivity : AppCompatActivity() {

    // UI elements
    private lateinit var tvStatsTotalScore: TextView
    private lateinit var tvLearned: TextView
    private lateinit var tvCompletion: TextView
    private lateinit var tvCardsPerDay: TextView
    private lateinit var tvTotalAnswers: TextView
    private lateinit var tvStreak: TextView
    private lateinit var progressCompletion: LinearProgressIndicator
    private lateinit var btnTutorialInfo: ImageButton
    private lateinit var tvMastery: TextView
    private lateinit var progressMastery: LinearProgressIndicator

    private val gson = Gson()
    private val tag = "StatsDebug"

    // --- Add an instance of StatsManager ---
    private lateinit var statsManager: StatsManager


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableEdgeToEdge()

        setContentView(R.layout.activity_statistics)

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.rootLayout)) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }

        // --- Initialize the StatsManager ---
        statsManager = StatsManager(this)

        // Initialize UI components
        tvStatsTotalScore = findViewById(R.id.tvStatsTotalScore)
        tvLearned = findViewById(R.id.tvLearned)
        tvCompletion = findViewById(R.id.tvCompletion)
        tvCardsPerDay = findViewById(R.id.tvCardsPerDay)
        tvTotalAnswers = findViewById(R.id.tvTotalAnswers)
        tvStreak = findViewById(R.id.tvStreak)
        progressCompletion = findViewById(R.id.progressCompletion)
        btnTutorialInfo = findViewById(R.id.btnTutorialInfo)

        tvMastery = findViewById(R.id.tvMastery)
        progressMastery = findViewById(R.id.progressMastery)

        // Setup tutorial button
        btnTutorialInfo.setOnClickListener {
            showTutorialDialog("Statistics Screen Tutorial", R.raw.tutorial_stats)
        }
    }

    override fun onResume() {
        super.onResume()

        // 1. Check if the streak needs to be updated (e.g., if a day has passed).
        // 1. Check if the streak needs to be reset (but don't increment it).
        statsManager.checkAndResetStreak()
        // 2. Save the potentially reset streak value.
        statsManager.save()
        // 3. Load and display all statistics.
        loadAndDisplayStatistics()
    }

    /**
     * Loads calculated statistics from the StatsManager and SharedPreferences, then updates the UI.
     */
    private fun loadAndDisplayStatistics() {
        // --- Get values directly from up-to-date statsManager instance ---
        val totalAnswers = statsManager.totalAnswers
        val learnedCount = statsManager.learnedCount
        val cardsPerDay = statsManager.getCardsPerDay() // Use method to get live calculation
        val streakCount = statsManager.streakCount

        // --- Calculate and Display Learning Completion ---
        val totalWords = resources.openRawResource(R.raw.french_nouns_frequency)
            .bufferedReader().readLines().size - 1
        val completionPercent = if (totalWords > 0) {
            (learnedCount.toFloat() / totalWords) * 100f
        } else {
            0f
        }
        val completionFormatted = String.format(Locale.getDefault(), "%.1f", completionPercent)
        tvCompletion.text = getString(R.string.stat_completion, completionFormatted)
        progressCompletion.setProgressCompat(completionPercent.toInt(), true)

        // --- Calculate and Display Review Mastery ---
        calculateAndDisplayMastery(learnedCount)

        // --- Update other TextViews with data from StatsManager ---
        tvStatsTotalScore.text = "${statsManager.totalScore}"
        tvTotalAnswers.text = getString(R.string.stat_answers, totalAnswers)
        tvLearned.text = getString(R.string.stat_learned, learnedCount)
        val cpdFormatted = String.format(Locale.getDefault(), "%.1f", cardsPerDay)
        tvCardsPerDay.text = getString(R.string.stat_cpd, cpdFormatted)
        tvStreak.text = getString(R.string.stat_streak, streakCount)
    }

    private fun calculateAndDisplayMastery(learnedCount: Int) {
        val progressPrefs = getSharedPreferences("progress", MODE_PRIVATE)
        val reviewDeckType = object : TypeToken<MutableList<ReviewItem>>() {}.type

        // --- STEP 1: Load the Master Deck (the source of truth) ---
        val masterDeckJson = progressPrefs.getString(PREF_REVIEW_DECK, "[]")
        val masterReviewList = gson.fromJson<MutableList<ReviewItem>>(masterDeckJson, reviewDeckType)
        // Use a map for efficient merging, with the word as the key
        val mergedMap = masterReviewList.associateBy { it.word.word }.toMutableMap()

        // --- STEP 2: Load the In-Progress Session Deck (if it exists) ---
        val inProgressSessionJson = progressPrefs.getString("session_review_reviewWordsStack", null)
        if (inProgressSessionJson != null) {
            try {
                val inProgressList = gson.fromJson<List<ReviewItem>>(inProgressSessionJson, reviewDeckType)
                // --- STEP 3: Merge by overwriting master data with session data ---
                // Any word in the session deck is more up-to-date.
                inProgressList.forEach { sessionItem ->
                    mergedMap[sessionItem.word.word] = sessionItem
                }
                Log.d(tag, "Successfully merged ${inProgressList.size} in-progress session items.")
            } catch (e: Exception) {
                // In case the saved session JSON is malformed, don't crash.
                Log.e(tag, "Failed to parse in-progress session JSON", e)
            }
        }

        // The mergedMap contains the most "live" version of all review items.
        val finalReviewList = mergedMap.values

        // --- STEP 4: Perform the calculation on the final, merged list ---
        val masteryThresholdDays = 16
        val masteredCount = finalReviewList.count { item ->
            // A word is mastered ONLY IF its interval is high AND it hasn't failed in the current session.
            item.reviewInterval > masteryThresholdDays && !item.failedInSession
        }

        val totalReviewingCount = learnedCount
        val masteryPercent = if (totalReviewingCount > 0) {
            (masteredCount.toFloat() / totalReviewingCount) * 100f
        } else {
            0f
        }

        val masteryFormatted = String.format(Locale.getDefault(), "%.1f", masteryPercent)
        tvMastery.text = getString(R.string.stat_mastery, masteredCount, totalReviewingCount, masteryFormatted)
        progressMastery.setProgressCompat(masteryPercent.toInt(), true)
    }

    /** Displays the tutorial dialog by reading and rendering HTML from a raw resource. */
    private fun showTutorialDialog(title: String, rawResId: Int) {
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)

        // Read the text from the raw resource file
        val tutorialText = try {
            resources.openRawResource(rawResId)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            e.printStackTrace()
            "Could not load tutorial information for $title. (Resource $rawResId is missing in R.raw)"
        }

        // Render HTML content for bolding and lists
        textView.text =
            Html.fromHtml(tutorialText, Html.FROM_HTML_MODE_COMPACT)

        AlertDialog.Builder(this)
            .setTitle(title)
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }
}