package com.languageapp.laetle

import androidx.appcompat.app.AppCompatActivity
import android.os.Bundle
import android.widget.Button
import android.widget.TextView
import androidx.core.content.ContextCompat
import com.google.gson.Gson
import com.google.gson.reflect.TypeToken
import kotlinx.coroutines.*
import android.media.MediaPlayer
import android.text.Html
import android.view.View
import android.widget.ImageButton
import androidx.appcompat.app.AlertDialog
import androidx.lifecycle.lifecycleScope
import android.util.Log
import androidx.core.content.edit
import java.util.Calendar
import androidx.core.net.toUri
import androidx.activity.enableEdgeToEdge
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat

// Keep data class structure for now to avoid breaking existing JSON saves,
// but default values will be empty strings since the dataset no longer has sentences.
data class WordItem(
    val word: String,
    val gender: String,
    val translation: String = "",
    val exampleFr: String = "",
    val exampleEn: String = "",
    var correctCount: Int = 0
)

data class ReviewItem(
    val word: WordItem,
    var nextReviewDate: Long,
    var reviewInterval: Int,
    var failedInSession: Boolean = false,
    var toReview: Boolean = true
)

private val sessionFailedWords = mutableSetOf<String>()

class StartActivity : AppCompatActivity() {

    private var isSessionComplete = false
    private val tag = "SRS_DEBUG"

    private var isFreshStart = false
    private var streakUpdatedThisSession = false

    internal var sessionMode: Int = REVIEW_MODE_NEW
    internal var reviewWordsStack = mutableListOf<ReviewItem>()

    private lateinit var statsManager: StatsManager

    // UI elements
    private lateinit var tvWord: TextView
    private lateinit var btnLa: Button
    private lateinit var btnLe: Button
    private lateinit var tvTranslation: TextView

    private lateinit var tvExampleFr: TextView
    private lateinit var tvExampleEn: TextView

    private lateinit var btnTutorialInfo: ImageButton

    // Settings-related state
    private var showTranslation = true
    private var showExamplesFr = true
    private var showExamplesEn = true
    private var soundEnabled = true
    private var currentSessionThreshold = 1

    // Word stack management
    private var nounsList = mutableListOf<WordItem>()

    internal var stackSize = 10
    internal var stack = mutableListOf<WordItem?>()
    internal var currentStackIndex = 0
    private var nextWordIndex = 0

    private val prefs by lazy { getSharedPreferences("progress", MODE_PRIVATE) }
    private val settingsPrefs by lazy { getSharedPreferences("settings", MODE_PRIVATE) }
    private val gson = Gson()

    private var currentPlayer: MediaPlayer? = null
    private var pausedTime = 0L

    private lateinit var timeProvider: TimeProvider
    private var lastAnswerTime: Long = 0L
    private lateinit var tvStreakPopup: TextView // The new view
    private var currentStreakInRow = 0           // Track current streak
    private var streakPopupJob: Job? = null      // To manage the 2-second timer

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableEdgeToEdge()

        setContentView(R.layout.activity_start)

        ViewCompat.setOnApplyWindowInsetsListener(findViewById(android.R.id.content)) { v, insets ->
            val systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars())
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom)
            insets
        }

        if (savedInstanceState == null) {
            isFreshStart = true
        }

        timeProvider = RealTimeProvider()
        lastAnswerTime = timeProvider.getCurrentTimeMillis()

        // Initialize UI components
        tvWord = findViewById(R.id.tvWord)
        btnLa = findViewById(R.id.btnLa)
        btnLe = findViewById(R.id.btnLe)
        tvTranslation = findViewById(R.id.tvTranslation)

        tvExampleFr = findViewById(R.id.tvExampleFr)
        tvExampleEn = findViewById(R.id.tvExampleEn)

        btnTutorialInfo = findViewById(R.id.btnTutorialInfo)
        tvStreakPopup = findViewById(R.id.tvStreakPopup)
        currentStreakInRow = prefs.getInt("persistent_current_streak", 0)


        // Load initial settings
        showTranslation = prefs.getBoolean(PREF_SHOW_TRANSLATION, true)
        showExamplesFr = prefs.getBoolean(PREF_SHOW_EXAMPLES_FR, true)
        showExamplesEn = prefs.getBoolean(PREF_SHOW_EXAMPLES_EN, true)
        sessionMode = intent.getIntExtra("SESSION_MODE", REVIEW_MODE_NEW)

        // Load vocabulary
        loadTSV()

        statsManager = StatsManager(this)

        btnLa.setOnClickListener { checkAnswer("la") }
        btnLe.setOnClickListener { checkAnswer("le") }
        btnTutorialInfo.setOnClickListener {
            showTutorialDialog("Practice Screen Tutorial", R.raw.tutorial_start)
        }

        tvWord.apply {
            setOnClickListener {
                if (soundEnabled) {
                    stack.getOrNull(currentStackIndex)?.let { currentWord ->
                        playWordAudio(currentWord)
                    }
                }
            }
            setOnLongClickListener {
                val currentWord = stack.getOrNull(currentStackIndex) ?: return@setOnLongClickListener true
                showWordContextMenu(currentWord.word)
                true
            }
        }

        if (savedInstanceState != null) {
            pausedTime = savedInstanceState.getLong("PAUSED_TIME", 0L)
        }
    }

    override fun onResume() {
        super.onResume()
        currentPlayer = MediaPlayer()

        currentSessionThreshold = if (sessionMode == REVIEW_MODE_NEW) {
            prefs.getInt(PREF_LEARN_THRESHOLD, 3)
        } else {
            prefs.getInt(PREF_REVIEW_THRESHOLD, 2)
        }
        showTranslation = prefs.getBoolean(PREF_SHOW_TRANSLATION, true)
        showExamplesFr = prefs.getBoolean(PREF_SHOW_EXAMPLES_FR, true)
        showExamplesEn = prefs.getBoolean(PREF_SHOW_EXAMPLES_EN, true)
        soundEnabled = settingsPrefs.getBoolean("enable_audio", true)
        btnTutorialInfo.visibility =
            if (settingsPrefs.getBoolean("enable_tutorial", true)) View.VISIBLE else View.GONE

        isSessionComplete = false

        val modePrefix = if (sessionMode == REVIEW_MODE_REVIEW) "session_review_" else "session_learn_"
        val reviewStackJson = prefs.getString("${modePrefix}reviewWordsStack", null)
        val failedJson = prefs.getString("${modePrefix}failedWords", null)

        loadProgress(reviewStackJson = reviewStackJson, failedJson = failedJson)

        showCurrentWord(isFreshStart)
    }

    override fun onPause() {
        super.onPause()
        pausedTime = timeProvider.getCurrentTimeMillis()

        val now = timeProvider.getCurrentTimeMillis()
        statsManager.addActiveTime(now - lastAnswerTime)
        statsManager.save()

        if (!isSessionComplete) {
            val modePrefix = if (sessionMode == REVIEW_MODE_REVIEW) "session_review_" else "session_learn_"
            prefs.edit().apply {
                putString("${modePrefix}stack", gson.toJson(stack))
                putInt("${modePrefix}stackSize", stack.size)
                putInt("${modePrefix}currentStackIndex", currentStackIndex)
                putInt("${modePrefix}nextWordIndex", nextWordIndex)
                putString("${modePrefix}failedWords", gson.toJson(sessionFailedWords))
                if (sessionMode == REVIEW_MODE_REVIEW) {
                    putString("${modePrefix}reviewWordsStack", gson.toJson(reviewWordsStack))
                }
                putLong("${modePrefix}paused_time", pausedTime)
                apply()
            }
        }

        if (sessionMode == REVIEW_MODE_REVIEW) saveReviewProgress()

        currentPlayer?.release()
        currentPlayer = null
    }

    override fun onDestroy() {
        super.onDestroy()
        currentPlayer?.release()
        currentPlayer = null
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        outState.putLong("PAUSED_TIME", pausedTime)
    }

    /**
     * Loads word data from the french_nouns_frequency.tsv resource.
     * Note: Ensure the file is named 'french_nouns_frequency.tsv' and placed in the res/raw folder.
     */
    private fun loadTSV() {
        nounsList.clear()

        // The resource ID is based on the filename without the extension.
        val inputStream = resources.openRawResource(R.raw.french_nouns_frequency)
        inputStream.bufferedReader().useLines { lines ->
            // Skip header and process lines
            lines.drop(1).forEach { line ->
                if (line.isBlank()) return@forEach

                // Split the line by tabs for a TSV file
                val parts = line.split('\t')

                if (parts.size >= 6) {
                    // Trimming and removing quotes is kept for robustness, in case the TSV still has them.
                    val word = parts[1].trim().removeSurrounding("\"")
                    val gender = parts[2].trim().removeSurrounding("\"").lowercase()
                    val translation = parts[3].trim().removeSurrounding("\"")
                    val exampleFr = parts[4].trim().removeSurrounding("\"")
                    val exampleEn = parts[5].trim().removeSurrounding("\"")

                    nounsList.add(WordItem(word, gender, translation, exampleFr, exampleEn))
                }
            }
        }
    }

    private fun loadProgress(
        reviewStackJson: String? = null,
        failedJson: String? = null
    ) {
        if (sessionMode == REVIEW_MODE_REVIEW) {
            loadReviewDeckForSession(reviewStackJson, failedJson)
        } else {
            val stackJson = prefs.getString("session_learn_stack", null)
            if (stackJson != null) {
                val type = object : TypeToken<MutableList<WordItem?>>() {}.type
                stack = gson.fromJson(stackJson, type)
                currentStackIndex = prefs.getInt("session_learn_currentStackIndex", 0)
                nextWordIndex = prefs.getInt("session_learn_nextWordIndex", stackSize)
            } else {
                stack = mutableListOf()
                for (i in 0 until stackSize) {
                    stack.add(if (i < nounsList.size) nounsList[i] else null)
                }
                currentStackIndex = 0
                nextWordIndex = stackSize
            }
        }
    }

    private fun showCurrentWord(playAudioOnShow: Boolean = true) {
        val currentWord = stack.getOrNull(currentStackIndex)
        Log.d(tag, "showCurrentWord START. Index: $currentStackIndex, Word: ${currentWord?.word}, Stack All Null: ${stack.all { it == null }}")

        if (currentWord != null) {
            tvWord.text = currentWord.word
            tvTranslation.apply {
                text = currentWord.translation
                visibility = if (showTranslation) TextView.VISIBLE else TextView.GONE
            }


            tvExampleFr.apply {
                text = currentWord.exampleFr
                visibility = if (showExamplesFr) TextView.VISIBLE else TextView.GONE
            }
            tvExampleEn.apply {
                text = currentWord.exampleEn
                visibility = if (showExamplesEn) TextView.VISIBLE else TextView.GONE
            }

            btnLa.isEnabled = true
            btnLe.isEnabled = true

            if (playAudioOnShow) {
                playWordAudio(currentWord)
            }
        } else if (stack.all { it == null }) {
            tvWord.text = getString(R.string.finished_set)
            tvTranslation.text = ""

            tvExampleFr.text = ""
            tvExampleEn.text = ""

            btnLa.isEnabled = false
            btnLe.isEnabled = false
        }
    }

    private fun getButtonForGender(gender: String) = if (gender == "la") btnLa else btnLe

    private fun flashButton(button: Button, flashColor: Int) {
        button.backgroundTintList = ContextCompat.getColorStateList(this, flashColor)
        lifecycleScope.launch {
            delay(300)
            button.backgroundTintList = ContextCompat.getColorStateList(this@StartActivity, R.color.button_default)
        }
    }

    private fun checkAnswer(selectedGender: String) {
        if (sessionMode == REVIEW_MODE_NEW) {
            checkAnswerNewCard(selectedGender)
        } else {
            checkAnswerReviewCard(selectedGender)
        }
    }

    private fun checkAnswerNewCard(selectedGender: String) {
        val currentWord = stack.getOrNull(currentStackIndex) ?: return

        if (!streakUpdatedThisSession) {
            statsManager.recordPracticeSession()
            statsManager.save()
            streakUpdatedThisSession = true
        }

        val now = timeProvider.getCurrentTimeMillis()
        statsManager.addActiveTime(now - lastAnswerTime)
        lastAnswerTime = now

        val correct = selectedGender == currentWord.gender

        handleStreakCheck(correct)

        if (correct) {
            currentWord.correctCount++
            flashButton(getButtonForGender(selectedGender), R.color.button_correct)
            if (currentWord.correctCount >= currentSessionThreshold) {
                pushToReviewDeck(currentWord)
                replaceWordInStack(currentStackIndex)
            }
        } else {
            currentWord.correctCount = 0
            flashButton(getButtonForGender(selectedGender), R.color.button_incorrect)
        }

        statsManager.logAnswer()
        moveToNextWord()
        saveProgress()
    }

    internal fun checkAnswerReviewCard(selectedGender: String) {
        if (stack.getOrNull(currentStackIndex) == null) {
            moveToNextWord()
            return
        }
        val currentWordItem = stack.getOrNull(currentStackIndex) ?: return
        val originalReviewItem = reviewWordsStack.firstOrNull { it.word.word == currentWordItem.word } ?: return

        if (!streakUpdatedThisSession) {
            statsManager.recordPracticeSession()
            statsManager.save()
            streakUpdatedThisSession = true
        }

        val now = timeProvider.getCurrentTimeMillis()
        statsManager.addActiveTime(now - lastAnswerTime)
        lastAnswerTime = now

        val correct = selectedGender == currentWordItem.gender
        handleStreakCheck(correct)
        val currentWordText = currentWordItem.word

        if (correct) {
            originalReviewItem.word.correctCount++
            flashButton(getButtonForGender(selectedGender), R.color.button_correct)

            if (originalReviewItem.word.correctCount >= currentSessionThreshold) {
                originalReviewItem.toReview = false

                if (sessionFailedWords.contains(currentWordText) || originalReviewItem.failedInSession) {
                    originalReviewItem.reviewInterval = 1
                    originalReviewItem.nextReviewDate = timeProvider.getCurrentCalendar().apply {
                        add(Calendar.DAY_OF_YEAR, 1)
                        set(Calendar.HOUR_OF_DAY, 0)
                        set(Calendar.MINUTE, 0)
                        set(Calendar.SECOND, 0)
                        set(Calendar.MILLISECOND, 0)
                    }.timeInMillis
                } else {
                    val currentInterval = originalReviewItem.reviewInterval
                    originalReviewItem.reviewInterval = if (currentInterval == 0) 1 else currentInterval * 2

                    originalReviewItem.nextReviewDate = timeProvider.getCurrentCalendar().apply {
                        add(Calendar.DAY_OF_YEAR, originalReviewItem.reviewInterval)
                        set(Calendar.HOUR_OF_DAY, 0)
                        set(Calendar.MINUTE, 0)
                        set(Calendar.SECOND, 0)
                        set(Calendar.MILLISECOND, 0)
                    }.timeInMillis
                }

                stack[currentStackIndex] = null
            }

        } else {
            originalReviewItem.word.correctCount = 0
            originalReviewItem.failedInSession = true
            sessionFailedWords.add(currentWordText)
            flashButton(getButtonForGender(selectedGender), R.color.button_incorrect)
        }

        saveReviewProgress()
        btnLa.isEnabled = true
        btnLe.isEnabled = true
        statsManager.logAnswer()
        moveToNextWord()
    }

    private fun replaceWordInStack(index: Int) {
        stack[index] = if (nextWordIndex < nounsList.size) nounsList[nextWordIndex++]
        else null
    }

    private fun moveToNextWord() {
        var foundNext = false
        for (i in 1..stackSize) {
            val nextIndex = (currentStackIndex + i) % stackSize
            if (stack[nextIndex] != null) {
                currentStackIndex = nextIndex
                foundNext = true
                break
            }
        }

        if (!foundNext) {
            tvWord.text = getString(R.string.finished_set)
            btnLa.isEnabled = false
            btnLe.isEnabled = false

            tvTranslation.visibility = View.GONE
            tvExampleFr.visibility = View.GONE
            tvExampleEn.visibility = View.GONE

            if (sessionMode == REVIEW_MODE_REVIEW) {
                saveReviewProgress()
            }
            isSessionComplete = true
            val modePrefix = if (sessionMode == REVIEW_MODE_REVIEW) "session_review_" else "session_learn_"
            prefs.edit {
                remove("${modePrefix}stack")
                    .remove("${modePrefix}stackSize")
                    .remove("${modePrefix}currentStackIndex")
                    .remove("${modePrefix}nextWordIndex")
                    .remove("${modePrefix}failedWords")
                    .remove("${modePrefix}reviewWordsStack")
            }
        } else {
            showCurrentWord()
        }
    }

    private fun saveProgress() {
        if (sessionMode == REVIEW_MODE_REVIEW) return
        prefs.edit().apply {
            putString("stack", gson.toJson(stack))
            putInt("currentStackIndex", currentStackIndex)
            putInt("nextWordIndex", nextWordIndex)
            apply()
        }
    }

    private fun playWordAudio(wordItem: WordItem) {
        btnLa.isEnabled = true
        btnLe.isEnabled = true

        if (!soundEnabled) return

        val index = nounsList.indexOfFirst { it.word == wordItem.word }
        if (index == -1) return

        val resName = "word_${index + 1}"
        val resId = resources.getIdentifier(resName, "raw", packageName)
        if (resId == 0) return

        try {
            currentPlayer?.reset()
            currentPlayer?.setOnErrorListener { _, _, _ -> true }
            val afd = resources.openRawResourceFd(resId)
            currentPlayer?.setDataSource(afd.fileDescriptor, afd.startOffset, afd.length)
            afd.close()
            currentPlayer?.setOnPreparedListener { it.start() }
            currentPlayer?.prepareAsync()
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun showTutorialDialog(title: String, rawResId: Int) {
        val dialogView = layoutInflater.inflate(R.layout.dialog_licenses, null)
        val textView = dialogView.findViewById<TextView>(R.id.tvLicenses)

        val tutorialText = try {
            resources.openRawResource(rawResId)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            e.printStackTrace()
            "Could not load tutorial information."
        }

        textView.text = Html.fromHtml(tutorialText, Html.FROM_HTML_MODE_COMPACT)

        AlertDialog.Builder(this)
            .setTitle(title)
            .setView(dialogView)
            .setPositiveButton("OK", null)
            .show()
    }

    private fun loadReviewDeckForSession(
        reviewStackJson: String? = null,
        failedJson: String? = null
    ) {
        val today = timeProvider.getCurrentTimeMillis()
        val finalSessionItems = mutableMapOf<String, ReviewItem>()
        val masterDeckJson = prefs.getString(PREF_REVIEW_DECK, "[]")
        val masterReviewList = gson.fromJson<MutableList<ReviewItem>>(
            masterDeckJson,
            object : TypeToken<MutableList<ReviewItem>>() {}.type
        )

        val dueCards = masterReviewList.filter { it.nextReviewDate <= today }

        if (reviewStackJson != null) {
            val typeReview = object : TypeToken<MutableList<ReviewItem>>() {}.type
            val oldSessionItems = gson.fromJson<MutableList<ReviewItem>>(reviewStackJson, typeReview)

            oldSessionItems.forEach { item ->
                finalSessionItems[item.word.word] = item
            }

            if (failedJson != null) {
                val typeSet = object : TypeToken<MutableSet<String>>() {}.type
                sessionFailedWords.clear()
                sessionFailedWords.addAll(gson.fromJson(failedJson, typeSet))
            }
        }

        dueCards.forEach { card ->
            val existing = finalSessionItems[card.word.word]
            if (existing == null) {
                card.word.correctCount = 0
                card.failedInSession = false
                card.toReview = true
                finalSessionItems[card.word.word] = card
            } else {
                existing.nextReviewDate = card.nextReviewDate
                existing.reviewInterval = card.reviewInterval
            }
        }

        reviewWordsStack = finalSessionItems.values
            .filter { it.toReview }
            .toMutableList()

        stack = reviewWordsStack
            .map { it.word }
            .toMutableList()
        stackSize = stack.size

        currentStackIndex = prefs.getInt("session_review_currentStackIndex", 0)
            .coerceAtMost(stack.lastIndex.coerceAtLeast(0))
        nextWordIndex = stack.size

        if (reviewStackJson == null) stack.shuffle()
    }

    private fun pushToReviewDeck(masteredWord: WordItem) {
        val initialReviewItem = ReviewItem(
            word = masteredWord,
            nextReviewDate = timeProvider.getCurrentCalendar().apply {
                add(Calendar.DAY_OF_YEAR, 1)
                set(Calendar.HOUR_OF_DAY, 0)
                set(Calendar.MINUTE, 0)
                set(Calendar.SECOND, 0)
                set(Calendar.MILLISECOND, 0)
            }.timeInMillis,
            reviewInterval = 1
        )

        val reviewDeckJson = prefs.getString(PREF_REVIEW_DECK, "[]")
        val masterReviewList = gson.fromJson<MutableList<ReviewItem>>(
            reviewDeckJson,
            object : TypeToken<MutableList<ReviewItem>>() {}.type
        )

        val mergedMap = mutableMapOf<String, ReviewItem>()
        masterReviewList.forEach { item ->
            mergedMap[item.word.word] = item
        }
        mergedMap[initialReviewItem.word.word] = initialReviewItem

        prefs.edit { putString(PREF_REVIEW_DECK, gson.toJson(mergedMap.values.toMutableList())) }

        val currentLearnedCount = prefs.getInt("learnedCount", 0)
        prefs.edit { putInt("learnedCount", currentLearnedCount + 1) }

        statsManager.logWordMastered()
    }

    private fun saveReviewProgress() {
        if (sessionMode != REVIEW_MODE_REVIEW) return

        val reviewDeckJson = prefs.getString(PREF_REVIEW_DECK, "[]")
        val masterReviewList = gson.fromJson<MutableList<ReviewItem>>(
            reviewDeckJson,
            object : TypeToken<MutableList<ReviewItem>>() {}.type
        )

        val mergedMap = mutableMapOf<String, ReviewItem>()

        masterReviewList.forEach { item ->
            mergedMap[item.word.word] = item
        }

        reviewWordsStack.forEach { item ->
            mergedMap[item.word.word] = item
        }

        prefs.edit { putString(PREF_REVIEW_DECK, gson.toJson(mergedMap.values.toMutableList())) }
    }

    private fun showWordContextMenu(word: String) {
        val options = arrayOf("Copy word", "Open in Wiktionary")

        AlertDialog.Builder(this)
            .setTitle(word)
            .setItems(options) { _, which ->
                when (which) {
                    0 -> {
                        val clipboard = getSystemService(CLIPBOARD_SERVICE) as android.content.ClipboardManager
                        val clip = android.content.ClipData.newPlainText("word", word)
                        clipboard.setPrimaryClip(clip)
                    }
                    1 -> {
                        val encodedWord = java.net.URLEncoder.encode(word, "UTF-8")
                        val intent = android.content.Intent(
                            android.content.Intent.ACTION_VIEW,
                            "https://fr.wiktionary.org/wiki/$encodedWord".toUri()
                        )
                        startActivity(intent)
                    }
                }
            }
            .show()
    }
    private fun handleStreakCheck(isCorrect: Boolean) {
        if (isCorrect) {
            currentStreakInRow++
            prefs.edit().putInt("persistent_current_streak", currentStreakInRow).apply()

            // 1. Calculate and Add Score
            statsManager.addScoreBasedOnStreak(currentStreakInRow)
            statsManager.save() // Save immediately

            // 2. Prepare Pop-up Message
            var message: String? = when (currentStreakInRow) {
                5 -> "Good, 5 in a row!"
                10 -> "Wow! 10 in a row!"
                20 -> "Amazing, 20 in a row!"
                50 -> "Fantastic 50 in a row!"
                100 -> "Incredible 100 in a row!"
                200 -> "Epic 200 in a row!"
                500 -> "Legendary 500 in a row!"
                else -> null
            }

            // 3. Append Score Multiifier (only if message exists and streak < 50)
            if (message != null && currentStreakInRow < 50) {
                val multiplierText = when {
                    currentStreakInRow >= 20 -> " (Score x8)"
                    currentStreakInRow >= 10 -> " (Score x4)"
                    currentStreakInRow >= 5 -> " (Score x2)"
                    else -> ""
                }
                message += multiplierText
            }

            if (message != null) {
                showStreakPopup(message)
            }
        } else {
            currentStreakInRow = 0
            prefs.edit().putInt("persistent_current_streak", 0).apply()
            // No score deduction, just no points added
        }
    }

    private fun showStreakPopup(message: String) {

        if (!settingsPrefs.getBoolean(PREF_SHOW_STREAK_POPUP, true)) return
        // Cancel any existing timer so they don't overlap
        streakPopupJob?.cancel()

        tvStreakPopup.text = message
        tvStreakPopup.visibility = View.VISIBLE

        // Reset position for animation
        tvStreakPopup.alpha = 0f
        tvStreakPopup.translationY = -100f

        // Animate In (Slide down and fade in)
        tvStreakPopup.animate()
            .alpha(1f)
            .translationY(0f)
            .setDuration(300)
            .start()

        // Wait 2.5 seconds then hide
        streakPopupJob = lifecycleScope.launch {
            delay(2500)
            // Animate Out (Slide up and fade out)
            tvStreakPopup.animate()
                .alpha(0f)
                .translationY(-100f)
                .setDuration(300)
                .withEndAction {
                    tvStreakPopup.visibility = View.GONE
                }
                .start()
        }
    }
}